!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#include "configure.h"
!---+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+----0----+----1----+----2----+----3--
!===================================================================================================================================
 SUBROUTINE CREATE_MOD_PROCESSSUBR(process_chain, t_compo_root)
!===================================================================================================================================

! * Generates gen/mod_processsubr.F (file unit s_unit)
! * Generates gen/mod_processcontrol.F -- replaces setrrp.F
!   This module provides two subroutines called InitProcessParameters and
!   SetProcessParameters. InitProcParameters reads in the values from
!   medusa.rrp, and completes the rest of the 'static' information in the
!   pp_xxx process parameter structures (rate orders, rate constants, indices
!   of variable concentrations, parameters functions etc.); SetProcessParameters
!   sets the variable parts of the pp_* structures (actual concentrations etc.).
!   This module needs to be generated in four steps, with three building
!   blocks being generated in parallel:
!    - intermediate file mod_processcontrol_1.F (unit c1unit) with the
!      declarations for the namelists;
!    - intermediate file mod_processcontrol_1.F (unit c2unit) with the
!      allocation of the %IOdep components of the TYPE(PP_*) variables,
!      their initialisation, the reading of the namelists from medusa.rrp,
!      and the call of the *LOG routines;
!    - intermediate file mod_processcontrol_3.F (unit c3unit), where the
!      variable parameters get their values (SetProcessParameters subroutine);
!   The complete gen/mod_processcontrol.F is created at the end, with
!   the rest of the preamble, the start and end of the loops, the parts
!   in mod_processcontrol_1.F, mod_processcontrol_2.F and
!   mod_processcontrol_3.F, and the postamble.
! * Generates gen/mod_processdata.F (file unit d_unit)
! * Generates gen/medusa.rrp_template (file unit rpunit)
! * Generates gen/rea.F in a two-stage process
!    - intermediate file: tmp/rea-proc.F (file unit reauni)
!    - generation of gen/rea.F from lib/rea-base.F with EXPAND_INCLUDES
! * Generates gen/drea.F
!    - intermediate file: tmp/drea-proc.F (file unit dreuni)
!    - generation of gen/drea.F from lib/drea-base.F with EXPAND_INCLUDES

USE MOD_MEDUSA_COCOGEN
USE MOD_CONFIGURE_TYPES
USE MOD_CONFIGURE
USE MOD_UTILITIES, ONLY: EXPAND_TOKEN, EXPAND_2TOKEN, EXPAND_3TOKEN, &
                         LOWCASE, UPCASE


IMPLICIT NONE



TYPE(PROCESS), INTENT(IN), TARGET :: process_chain
TYPE(COMPOINFO), INTENT(IN), TARGET :: t_compo_root

INTEGER, PARAMETER :: s_unit = CFG_S_UNIT   ! tmp/mod_processsubr.F
INTEGER, PARAMETER :: d_unit = CFG_D_UNIT   ! tmp/mod_processdata.F
INTEGER, PARAMETER :: c1unit = CFG_C1UNIT   ! tmp/mod_processcontrol_1.F (part of mod_processcontrol.F)
INTEGER, PARAMETER :: ctunit = CFG_TMPUNIT  ! tmp/mod_processcontrol_t.F (repeated parts of mod_processcontrol_1.F)
INTEGER, PARAMETER :: c2unit = CFG_C2UNIT   ! tmp/mod_processcontrol_2.F (part of mod_processcontrol.F)
INTEGER, PARAMETER :: c3unit = CFG_C3UNIT   ! tmp/mod_processcontrol_3.F (part of mod_processcontrol.F)
INTEGER, PARAMETER :: rpunit = CFG_RPUNIT   ! gen/medusa.rrp_template
INTEGER, PARAMETER :: reaunit = CFG_REAUNIT ! tmp/rea-proc.F
INTEGER, PARAMETER :: dreunit = CFG_DREUNIT ! tmp/drea-proc.F

TYPE(PROCESS), POINTER :: process_curr
TYPE(PROCESS), POINTER :: process_work
TYPE(PROCESS), POINTER :: process_xref

CHARACTER(LEN=n_lmaxformat) :: fmttmp

CHARACTER(LEN=20)              :: c_procid, c_tmp
CHARACTER(LEN=n_lmaxparfct)    :: c_ppid, c_piid
CHARACTER(LEN=n_lmaxparfct)    :: c_compo, c_compo1, c_compo2, c_compo3
CHARACTER(LEN=n_lmaxparfct)    :: c_param, c_param1, c_param2, c_param3, c_param4
CHARACTER(LEN=n_lmaxcodeline)  :: c_expression, c_expression1
CHARACTER(LEN=n_lxxlcodeline)  :: c_codeline

CHARACTER(LEN=20)              :: c_yratephasefactor
                                                ! max length = LEN('(1.0D+00 - aphi)') = 16
CHARACTER(LEN=n_lmaxexpress)   :: c_yratefactor
CHARACTER(LEN=n_lmaxexpress)   :: c_yrateconv
CHARACTER(LEN=n_lmaxexpress+3) :: c_yratedivis  ! 3 = LEN('/(') + LEN(')')
CHARACTER(LEN=20)              :: c_yrateunit   ! max length = LEN('mol') = 3
CHARACTER(LEN=20)              :: c_yratephase  ! max length = LEN('solid sediment') = 14
CHARACTER(LEN=n_lmaxexpress)   :: c_instruct1, c_instruct2
CHARACTER(LEN=n_lmaxexpress),   &
  DIMENSION(:), ALLOCATABLE    :: c_linebegin, c_lineend
CHARACTER(LEN=n_lmaxshortid+LEN(cp_prefix_io)), &
  DIMENSION(:), ALLOCATABLE    :: c_ioid_eqns
CHARACTER(LEN=n_lmaxshortid+LEN(cp_prefix_io)), &
  DIMENSION(:), ALLOCATABLE    :: c_ioid_ctls
CHARACTER(LEN=20)              :: c_type_ref

INTEGER :: i, j, n
INTEGER :: i_react_ref, i_prod_ref
INTEGER :: n_compo, n_eqns, n_ctls
INTEGER :: iu
INTEGER :: i_pid1, i_pid2, i_pid3

CHARACTER(LEN=n_lmaxshortid+LEN(cp_prefix_io)), &
  DIMENSION(:), POINTER    :: c_ioid_eqns_allprocs, c_ioid_eqns_save
CHARACTER(LEN=20), &
  DIMENSION(:), POINTER    :: c_procid_allprocs, c_procid_save
INTEGER :: n_eqns_allprocs


TYPE(COMPOINFO), POINTER      :: t_compo
CHARACTER(LEN=n_lmaxnamesgen) :: c_paramname
CHARACTER(LEN=n_lmaxshortid)  :: c_shortid, c_shortid1, c_shortid2, c_shortid3
CHARACTER(LEN=n_lmaxphasid)   :: c_phasid,  c_phasid1,  c_phasid2,  c_phasid3
CHARACTER(LEN=n_lmaxexpress)  :: c_stoech
CHARACTER(LEN=n_lmaxnamesgen) :: c_name
CHARACTER(LEN=n_lmaxshortid)  :: c_dummylabel
CHARACTER(LEN=n_lmaxphasid)   :: c_kindofparam
CHARACTER(LEN=6)              :: c_sharp
CHARACTER(LEN=n_lmaxexpress)  :: c_paramcode
CHARACTER(LEN=n_lmaxidentif)  :: c_typecomponame

INTEGER :: i_param, ixref_param, nlen
INTEGER :: n_gk,      n_pf,      n_pc,      i_gk
INTEGER :: n_gk_xref, n_pf_xref, n_pc_xref, i_gk_xref
LOGICAL :: l_gk_xref, l_pf_xref, l_pc_xref

INTEGER :: idx_xxref, ixxref_test_param, ixxref_param

INTEGER, PARAMETER :: jp_stdout = CFG_STDOUT
CHARACTER(LEN=*), PARAMETER  :: c_fmtinf_a = '("[CREATE_MOD_PROCESSSUBR]: ", A)'
INTEGER, PARAMETER :: jp_stderr = CFG_STDERR
CHARACTER(LEN=*), PARAMETER  :: c_fmterr_a = '("[CREATE_MOD_PROCESSSUBR] error: ", A)'

#ifdef CFG_DEBUG
INTEGER, PARAMETER :: jp_stddbg = CFG_STDDBG
CHARACTER(LEN=*), PARAMETER  :: c_dbgfmt  = '("DEBUG [CREATE_MOD_PROCESSSUBR]: ", A)'
CHARACTER(LEN=*), PARAMETER  :: c_dbgfmt1 = '("DEBUG [CREATE_MOD_PROCESSSUBR]: ", A, I0)'
CHARACTER(LEN=*), PARAMETER  :: c_closestatus = 'KEEP'
#else
CHARACTER(LEN=*), PARAMETER  :: c_closestatus = 'DELETE'
#endif


WRITE(jp_stdout, '()')
WRITE(jp_stdout, c_fmtinf_a) 'starting'

#ifdef CFG_DEBUG
WRITE(jp_stddbg, c_dbgfmt) 'CHECKPOINT - starting'
#endif


n_eqns_allprocs = 0
ALLOCATE(c_ioid_eqns_allprocs(0))
ALLOCATE(c_procid_allprocs(0))


iu = s_unit
OPEN(UNIT=iu, FILE = "tmp/mod_processsubr.F")

                !---+----1----+----2----+----3----+----4----+----5----+----6----+----7-!
WRITE(iu,fmt0) '! This module has been automatically generated by CREATE_MOD_PROCESSSUBR'
WRITE(iu,fmt0) '! from the MEDUSA configuration utility medusa-cocogen.'
WRITE(iu,fmt0) '!---+----1----+----2----+----3----+----4----+----5----+----6----+----7--'
WRITE(iu,fmt0) '!======================================================================='
WRITE(iu,fmt6) 'MODULE MOD_PROCESSSUBR'
WRITE(iu,fmt0) '!======================================================================='
WRITE(iu,fmt_)
WRITE(iu,fmt6) 'USE mod_defines_medusa'
WRITE(iu,fmt_)
WRITE(iu,fmt6) 'IMPLICIT NONE'
WRITE(iu,fmt_)
WRITE(iu,fmt_)
WRITE(iu,fmt_)
WRITE(iu,fmt6) 'CONTAINS'
WRITE(iu,fmt_)
WRITE(iu,fmt_)

                                    ! Preparation for creating the
                                    ! mod_processcontrol.F module.
                                    ! This module needs to be created
                                    ! in several stages:
                                    ! - mod_processcontrol_1.F (unit c1unit)
                                    !   takes the declarations of the namelists
                                    ! - mod_processcontrol_2.F (unit c2unit)
                                    !   takes the part that carries out the
                                    !   allocation of the %IOdep components
                                    !   of the TYPE(PP_*) variables and also
                                    !   carries out their initialisation,
                                    !   does the reading of the namelists
                                    !   from medusa.rrp, and calls the
                                    !   respective *LOG routines.
                                    ! - mod_processcontrol_3.F (unit c3unit)
                                    !   takes the part that assigns the
                                    !   variable parameters their values.
                                    ! The complete mod_processcontrol.F is
                                    ! created at the end, with the rest of
                                    ! the preamble, the start and end of the
                                    ! loops, mod_processcontrol_1.F and
                                    ! mod_processcontrol_2.F, and the postamble.

OPEN(UNIT=c1unit, FILE = "tmp/mod_processcontrol_1.F")
OPEN(UNIT=c2unit, FILE = "tmp/mod_processcontrol_2.F")
OPEN(UNIT=c3unit, FILE = "tmp/mod_processcontrol_3.F")


                                    ! Preparation for creating a template
                                    ! medusa.rrp This one will be created
                                    ! along with the rest of the processsubr
                                    ! parts relative to the individual
                                    ! processes.
iu = rpunit
OPEN(UNIT=iu, FILE = "gen/medusa.rrp_template")
WRITE(iu,fmt0) '! Template for medusa.rrp, automatically generated by'
WRITE(iu,fmt0) '! CREATE_MOD_PROCESSSUBR from the MEDUSA configuration'
WRITE(iu,fmt0) '! utility medusa-cocogen.'



iu = d_unit
OPEN(UNIT=iu, FILE = "tmp/mod_processdata.F")


WRITE(iu,fmt0) '! This module has been automatically generated by CREATE_MOD_PROCESSSUBR'
WRITE(iu,fmt0) '! from the MEDUSA configuration utility medusa-cocogen.'
WRITE(iu,fmt0) '!---+----1----+----2----+----3----+----4----+----5----+----6----+----7-!'
WRITE(iu,fmt0) '!======================================================================='
WRITE(iu,fmt6) 'MODULE MOD_PROCESSDATA'
WRITE(iu,fmt0) '!======================================================================='
WRITE(iu,fmt_)

process_curr => process_chain
scan_ratelaws: DO WHILE(ASSOCIATED(process_curr))
  process_work => process_curr%prev
  DO WHILE(ASSOCIATED(process_work))
    IF(process_work%ratelaw%name == process_curr%ratelaw%name) THEN
      process_curr => process_curr%next
      CYCLE scan_ratelaws
    ENDIF
    process_work => process_work%prev
  ENDDO
  WRITE(iu,fmt6) 'USE ' // cp_prefix_modlib // TRIM(LOWCASE(process_curr%ratelaw%name)) // &
                 ', ONLY: ' // TRIM(UPCASE(process_curr%ratelaw%pp_type))
  process_curr => process_curr%next
ENDDO scan_ratelaws

WRITE(iu,fmt6) 'USE mod_indexparam'
WRITE(iu,fmt_)
WRITE(iu,fmt6) 'IMPLICIT NONE'
WRITE(iu,fmt_)
WRITE(iu,fmt_)
WRITE(iu,fmt6) '! Total number of processes considered'
WRITE(iu,fmt_)
WRITE(c_tmp,'(I0)') n_medusaproc
WRITE(iu,fmt6) 'INTEGER, PARAMETER :: nproc         = ' // TRIM(c_tmp)
WRITE(iu,fmt_)
WRITE(iu,fmt6) '! Number of processes per realm'
WRITE(iu,fmt_)
WRITE(c_tmp,'(I0)') n_medusaproc_difblay
WRITE(iu,fmt6) 'INTEGER, PARAMETER :: nproc_difblay = ' // TRIM(c_tmp)
WRITE(c_tmp,'(I0)') n_medusaproc_reaclay
WRITE(iu,fmt6) 'INTEGER, PARAMETER :: nproc_reaclay = ' // TRIM(c_tmp)
WRITE(c_tmp,'(I0)') n_medusaproc_tranlay
WRITE(iu,fmt6) 'INTEGER, PARAMETER :: nproc_tranlay = ' // TRIM(c_tmp)
WRITE(c_tmp,'(I0)') n_medusaproc_corelay
WRITE(iu,fmt6) 'INTEGER, PARAMETER :: nproc_corelay = ' // TRIM(c_tmp)
WRITE(iu,fmt_)
WRITE(iu,fmt_)
WRITE(iu,fmt6) '! Process name information'
WRITE(iu,fmt_)
WRITE(c_tmp,'(I0)') n_lmaxprocname
WRITE(iu,fmt6) 'INTEGER, PARAMETER :: nlen_procname = ' // TRIM(c_tmp)
WRITE(iu,fmt_)
WRITE(iu,fmt6) 'CHARACTER(LEN=nlen_procname), DIMENSION(nproc) :: c_procname'
WRITE(iu,fmt_)
IF (n_medusaproc /= 0) THEN
  WRITE(iu,fmt_)
  WRITE(iu,fmt6) '! ProcessParameter declarations for processes considered'
  WRITE(iu,fmt_)
  WRITE(iu,fmt6) '! For each process (named ''ProcessName'') we declare the following'
  WRITE(iu,fmt6) '!  - pi_ProcessName:  process ID'
  WRITE(iu,fmt6) '!  - pp_ProcessName:  process parameter structure (container)'
  WRITE(iu,fmt6) '!  - nvp_ProcessName: number of components affected by the process'
  WRITE(iu,fmt6) '!  - iop_ProcessName: io indices of the components affected'
  WRITE(iu,fmt6) '!                     by the process'
  WRITE(iu,fmt6) '!  - nvr_ProcessName: number of components that control'
  WRITE(iu,fmt6) '!                     the rate law describing the process'
  WRITE(iu,fmt6) '!  - ior_ProcessName: io indices of the components that control'
  WRITE(iu,fmt6) '!                     the rate law describing the process'
  WRITE(iu,fmt_)
ENDIF

iu = reaunit
OPEN(UNIT=iu, FILE = "tmp/rea-proc.F")


iu = dreunit
OPEN(UNIT=iu, FILE = "tmp/drea-proc.F")


                                    ! For each process in the linked list
                                    ! process_chain, read in the information,
                                    ! translate it into Fortran subroutines
                                    ! and append it to mod_processsubr.F
                                    ! (unit s_unit)
process_curr => process_chain
DO WHILE(ASSOCIATED(process_curr))

# ifdef CFG_DEBUG
  WRITE(jp_stddbg,c_dbgfmt)  'BEGIN processubr ' // TRIM(process_curr%subr)
# endif
  n_compo = n_medusacompo

# ifdef CFG_DEBUG
  WRITE(jp_stddbg,c_dbgfmt1) 'ncompo = ', n_compo
# endif

  NULLIFY(process_xref)
  IF (process_curr%idx_xref /= 0) THEN
    process_xref => process_curr%prev
    DO WHILE (ASSOCIATED(process_xref))
      IF (process_xref%idx == process_curr%idx_xref) EXIT
      process_xref => process_xref%prev
    ENDDO
  ENDIF


                                    ! Temporary file for namelist in
                                    ! mod_processcontrol_1.F
  OPEN(UNIT=ctunit, FILE = "tmp/mod_processcontrol_t.F")


  ALLOCATE(c_linebegin(n_compo))
  ALLOCATE(c_lineend(n_compo))
  ALLOCATE(c_ioid_eqns(n_compo))
  ALLOCATE(c_ioid_ctls(n_compo))

  c_ppid = process_curr%pp_identifier
  c_piid = process_curr%pi_identifier
  WRITE(c_procid,'(i0)') process_curr%idx


  iu = s_unit
  WRITE(iu,fmt_)
  WRITE(iu,fmt0)   '!---+----1----+----2----+----3----+----4----+----5----+----6----+----7--'
  WRITE(iu,fmt_)
  WRITE(iu,fmt_)
  WRITE(iu,fmt0)   '!-----------------------------------------------------------------------'
  WRITE(iu,fmt6)   'SUBROUTINE '// TRIM(process_curr%subr)
  WRITE(iu,fmtcon) '           (ac, azdn, aphi, areac_factor, ac_rate, dac_rate_dac)'
  WRITE(iu,fmt0)   '!-----------------------------------------------------------------------'
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   'USE mod_indexparam'
  WRITE(iu,fmt6)   'USE mod_materialcharas'
  WRITE(iu,fmt6)   'USE ' // cp_prefix_modlib // TRIM(LOWCASE(process_curr%ratelaw%name))
  WRITE(iu,fmt6)   'USE mod_processdata, ONLY: ' // TRIM(c_ppid)
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   'IMPLICIT NONE'
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   '! Dummy variables (argument list)'
  WRITE(iu,fmt6)   '! -------------------------------'
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   'DOUBLE PRECISION, DIMENSION(ncompo),'
  WRITE(iu,fmtcon) '      INTENT(IN)'
  WRITE(iu,fmtcon) '   :: ac'
  WRITE(iu,fmt6)   'DOUBLE PRECISION,'
  WRITE(iu,fmtcon) '      INTENT(IN)'
  WRITE(iu,fmtcon) '   :: azdn'
  WRITE(iu,fmt6)   'DOUBLE PRECISION,'
  WRITE(iu,fmtcon) '      INTENT(IN)'
  WRITE(iu,fmtcon) '   :: aphi'
  WRITE(iu,fmt6)   'DOUBLE PRECISION,'
  WRITE(iu,fmtcon) '      INTENT(IN)'
  WRITE(iu,fmtcon) '   :: areac_factor'
  WRITE(iu,fmt6)   'DOUBLE PRECISION, DIMENSION(ncompo),'
  WRITE(iu,fmtcon) '      INTENT(OUT), OPTIONAL'
  WRITE(iu,fmtcon) '   :: ac_rate'
  WRITE(iu,fmt6)   'DOUBLE PRECISION, DIMENSION(ncompo,ncompo),'
  WRITE(iu,fmtcon) '      INTENT(OUT), OPTIONAL'
  WRITE(iu,fmtcon) '   :: dac_rate_dac'
  WRITE(iu,fmt_)
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   '! Local variables'
  WRITE(iu,fmt6)   '! ---------------'
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   'DOUBLE PRECISION'
  WRITE(iu,fmtcon) '   :: yrate'
  WRITE(iu,fmt6)   'DOUBLE PRECISION, DIMENSION(ncompo)'
  WRITE(iu,fmtcon) '   :: dyrate'
  WRITE(iu,fmt_)

                                    ! Check nature of first (reference)
                                    ! component in the reactant list of
                                    ! the current process: solid or solute.
                                    ! Adopt conversion factors as necessary:
                                    ! - c_yrateunit units (kg or mol)
                                    ! - c_yratephase phase (solid sediment/porewater)
                                    ! - c_yratefactor conversion factor
                                    !   /unit phase -> /unit total sediment

  IF(process_curr%i_ref < 0) THEN
    c_type_ref  = 'reactant'
    i_react_ref = -process_curr%i_ref
    i_prod_ref  = -1
  ELSEIF(process_curr%i_ref > 0) THEN
    c_type_ref  = 'product'
    i_react_ref = -1
    i_prod_ref  =  process_curr%i_ref
  ELSE
    WRITE(jp_stderr,'()')
    WRITE(jp_stderr,c_fmterr_a) 'process%i_ref = 0 -- should not happen -- aborting'
    CALL ABORT()
  ENDIF

  SELECT CASE(c_type_ref)
  CASE('reactant')

    c_name    = process_curr%cr_reacts(i_react_ref)%name
    c_phasid  = process_curr%cr_reacts(i_react_ref)%phasid
    c_shortid = process_curr%cr_reacts(i_react_ref)%shortid
    c_stoech  = process_curr%cr_reacts(i_react_ref)%stoech

    SELECT CASE(c_phasid)
    CASE('if')  ! solid
      c_yrateunit = 'kg'
      c_yratephase = 'solid sediment'
      c_yratephasefactor = '(1.0D+00 - aphi)'
    CASE('ic')   ! solute
      c_yrateunit = 'mol'
      c_yratephase = 'porewater'
      c_yratephasefactor = 'aphi'
    CASE DEFAULT
      WRITE(jp_stderr, c_fmterr_a) 'reference_id refers to a component with phasid="' // TRIM(c_phasid) //'"'
      WRITE(jp_stderr, '(A)')      '  must have phasid="ic" or phasid="if" -- aborting'
      CALL ABORT()
    END SELECT

                                    ! Prepare character string for
                                    ! possible unit conversion
    IF (c_stoech == '1') THEN
      c_yratedivis = ' '
    ELSE
      c_yratedivis = '/(' // TRIM(c_stoech) // ')'
    ENDIF

    IF (c_phasid == 'if') THEN      ! for solids: convert from molar to mass units
      c_yrateconv = '/mol_' // TRIM(c_shortid)
    ELSE
      c_yrateconv = ''
    ENDIF

    i_prod_ref = -1                 ! invalidate i_prod_ref

  CASE('product')

    c_name    = process_curr%cr_prods(i_prod_ref)%name
    c_phasid  = process_curr%cr_prods(i_prod_ref)%phasid
    c_shortid = process_curr%cr_prods(i_prod_ref)%shortid
    c_stoech  = process_curr%cr_prods(i_prod_ref)%stoech

    SELECT CASE(c_phasid)
    CASE('if')   ! solid
      c_yrateunit = 'kg'
      c_yratephase = 'solid sediment'
      c_yratephasefactor = '(1.0D+00 - aphi)'
    CASE('ic')   ! solute
      c_yrateunit = 'mol'
      c_yratephase = 'porewater'
      c_yratephasefactor = 'aphi'
    CASE DEFAULT
      WRITE(jp_stderr, c_fmterr_a) 'reference_id refers to a component with phasid="' // TRIM(c_phasid) //'"'
      WRITE(jp_stderr, '(A)')      '  must have phasid="ic" or phasid="if" -- aborting'
      CALL ABORT()
    END SELECT

                                    ! Prepare character string for
                                    ! possible unit conversion.
    IF (c_stoech == '1') THEN
      c_yratedivis = ' '
    ELSE
      c_yratedivis = '/(' // TRIM(c_stoech) // ')'
    ENDIF

    IF (c_phasid == 'if') THEN      ! for solids: convert from molar to mass units
      c_yrateconv = '/mol_' // TRIM(c_shortid)
    ELSE
      c_yrateconv = ''
    ENDIF

    i_react_ref = -1                ! invalidate i_react_ref

  END SELECT

  iu = s_unit
  WRITE(iu,fmt6)   'IF (PRESENT(ac_rate)) THEN'
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   '   CALL ' // TRIM(process_curr%ratelaw%name)
  WRITE(iu,fmtcon) '        (' // TRIM(c_ppid)//', ac, azdn, ARATE=yrate)'
  WRITE(iu,fmt_)

  WRITE(iu,fmt6)   '   ! yrate in [' // TRIM(c_yrateunit) // ' ' // &
                   TRIM(c_name) // '/{m3 '// TRIM(c_yratephase)//'}/yr]'
  WRITE(iu,fmt6)   '   ! convert to [' // TRIM(c_yrateunit) // ' ' // &
                   TRIM(c_name) // '/{m3 total sediment}/yr],'
  WRITE(iu,fmt6)   '   ! and take rreac_factor into account'
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   '   yrate = yrate * ' // TRIM(c_yratephasefactor) // &
                                   ' * areac_factor'
  WRITE(iu,fmt_)
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   '   ac_rate(:) =  0.0D+00'
  WRITE(iu,fmt_)


                                    ! Now write out instructions for
                                    ! calculating the rates for the
                                    ! components involved in the process.
                                    ! Two lists of component variables
                                    ! need to be prepared:
                                    ! - c_ioid_eqns(1:n_eqns), providing
                                    !   the IO id's of all the component
                                    !   variables that are affected by
                                    !   the current process
                                    ! - c_ioid_ctrls(1:n_ctrls), providing
                                    !   the IO id's of all the component
                                    !   variables that need to be considered
                                    !   for the rate law that controls the
                                    !   current process.


                                    ! 1. Set up the c_ioid_eqns
                                    !    and associated lists
  n_eqns = 0

                                    ! 1.1 Reactants
  DO i = 1, process_curr%n_reactants

    c_stoech  = process_curr%cr_reacts(i)%stoech
    c_phasid  = process_curr%cr_reacts(i)%phasid
    c_shortid = process_curr%cr_reacts(i)%shortid

    IF (c_stoech == '1') THEN
      c_yratefactor = ' '
    ELSE
      c_yratefactor = ' * (' // TRIM(c_stoech) // ')'
    ENDIF


    SELECT CASE(c_phasid)

    CASE('ic') ! solute

      n_eqns = n_eqns+1

      c_ioid_eqns(n_eqns) = cp_prefix_io // TRIM(c_shortid)
      c_linebegin(n_eqns) =  '('//TRIM(c_ioid_eqns(n_eqns)) //'@)'
      IF (i == i_react_ref) THEN
        c_lineend(n_eqns) =  '  -@'
      ELSE
        c_lineend(n_eqns) =  ' -(@' // &
                                    TRIM(c_yrateconv) //')'// &
                                    TRIM(c_yratefactor) // &
                                    TRIM(c_yratedivis)
      ENDIF
    CASE('if') ! solid

      n_eqns = n_eqns+1

      c_ioid_eqns(n_eqns) = cp_prefix_io // TRIM(c_shortid)
      c_linebegin(n_eqns) =  '(' // TRIM(c_ioid_eqns(n_eqns)) // '@)'
      IF (i == i_react_ref) THEN
        c_lineend(n_eqns) =  '  -@'
      ELSE
        c_lineend(n_eqns) =  ' -(@' // &
                                    TRIM(c_yrateconv) //')'// &
                                    '*mol_'//TRIM(c_shortid) // &
                                    TRIM(c_yratefactor) // &
                                    TRIM(c_yratedivis)
      ENDIF
    END SELECT
  ENDDO


                                    ! 1.2 Products
  DO i = 1, process_curr%n_products

    c_stoech  = process_curr%cr_prods(i)%stoech
    c_phasid  = process_curr%cr_prods(i)%phasid
    c_shortid = process_curr%cr_prods(i)%shortid

    IF (c_stoech == '1') THEN
      c_yratefactor = ' '
    ELSE
      c_yratefactor = ' * (' // TRIM(c_stoech) // ')'
    ENDIF


    SELECT CASE(c_phasid)

    CASE('ic') ! solute

      n_eqns = n_eqns+1

      c_ioid_eqns(n_eqns) = cp_prefix_io // TRIM(c_shortid)
      c_linebegin(n_eqns) =  '('//TRIM(c_ioid_eqns(n_eqns)) //'@)'
      IF (i == i_prod_ref) THEN
        c_lineend(n_eqns) =  '   @'
      ELSE
        c_lineend(n_eqns) =  '  (@' // TRIM(c_yrateconv) //')'// &
                                    TRIM(c_yratefactor) // &
                                    TRIM(c_yratedivis)
      ENDIF

    CASE('if')

      n_eqns = n_eqns+1

      c_ioid_eqns(n_eqns) = cp_prefix_io // TRIM(c_shortid)
      c_linebegin(n_eqns) =  '('//TRIM(c_ioid_eqns(n_eqns)) //'@)'
      IF (i == i_prod_ref) THEN
        c_lineend(n_eqns) =  '   @'
      ELSE
        c_lineend(n_eqns) =  '  (@' // &
                                    TRIM(c_yrateconv) //')'// &
                                    '*mol_' // TRIM(c_shortid) // &
                                    TRIM(c_yratefactor) // &
                                    TRIM(c_yratedivis)
      ENDIF

    END SELECT
  ENDDO


                                    ! 2. Set up the c_ioid_ctrls list
  n_ctls = 0

  DO i = 1, process_curr%ratelaw%n_params
    SELECT CASE(process_curr%ratelaw%kindofparam(i))
    CASE('io')
      n_ctls = n_ctls + 1
      c_ioid_ctls(n_ctls) = ADJUSTL(process_curr%ratelaw%paramcode(i))
    CASE DEFAULT
      CONTINUE
    END SELECT
  ENDDO


# ifdef CFG_DEBUG
  DO i = 1, n_eqns
    WRITE(jp_stddbg,c_dbgfmt) 'c_linebegin <' // TRIM(c_linebegin(i)) // '>'
    WRITE(jp_stddbg,c_dbgfmt) 'c_lineend   <' // TRIM(c_lineend(i))   // '>'
  ENDDO
  WRITE(jp_stddbg,c_dbgfmt) 'END processsubr ' // TRIM(process_curr%subr)
# endif

  DO i = 1, n_eqns
    CALL EXPAND_TOKEN(c_linebegin(i),'@',' ',c_instruct1)
    CALL EXPAND_TOKEN(c_lineend(i),'@','yrate',c_instruct2)
    iu = s_unit
    WRITE(iu,fmt6)    '   ac_rate'// TRIM(c_instruct1)
    WRITE(iu,fmtcon) '        = ' // TRIM(c_instruct2)
    WRITE(iu,fmt_)
  ENDDO

  WRITE(iu,fmt_)
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   'ENDIF'
                                    ! Done with the rate calculation
                                    ! instructions


                                    ! Now for the derivatives
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   'IF (PRESENT(dac_rate_dac)) THEN'
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   '   CALL ' // TRIM(process_curr%ratelaw%name)
  WRITE(iu,fmtcon) '        (' // TRIM(c_ppid) // ', ac, azdn, DARATE_DAC=dyrate)'
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   '   dyrate =  dyrate * '// TRIM(c_yratephasefactor) // &
                                    ' * areac_factor'
  WRITE(iu,fmt_)
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   '   dac_rate_dac(:,:)'
  WRITE(iu,fmtcon) '       =  0.0D+00'
  WRITE(iu,fmt_)
  WRITE(iu,fmt_)


  DO j = 1, n_ctls

    DO i = 1, n_eqns

      c_instruct1 = ',' // TRIM(c_ioid_ctls(j))
      CALL EXPAND_TOKEN(c_linebegin(i), '@', c_instruct1, c_instruct2)
      WRITE(iu,fmt6)    '   dac_rate_dac'// TRIM(c_instruct2)

      c_instruct1 = 'dyrate('//TRIM(c_ioid_ctls(j))//')'
      CALL EXPAND_TOKEN(c_lineend(i), '@', c_instruct1, c_instruct2)
      WRITE(iu,fmtcon) '        = ' // TRIM(c_instruct2)
      WRITE(iu,fmt_)

    ENDDO

    WRITE(iu,fmt_)

  ENDDO


  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   'ENDIF'
  WRITE(iu,fmt_)
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   'RETURN'
  WRITE(iu,fmt_)
  WRITE(iu,fmt0)   '!-----------------------------------------------------------------------'
  WRITE(iu,fmt6)   'END SUBROUTINE ' // TRIM(process_curr%subr)
  WRITE(iu,fmt0)   '!-----------------------------------------------------------------------'
  WRITE(iu,fmt_)
  WRITE(iu,fmt_)
  WRITE(iu,fmt0)   '!---+----1----+----2----+----3----+----4----+----5----+----6----+----7--'
  WRITE(iu,fmt_)
  WRITE(iu,fmt_)
  WRITE(iu,fmt0)   '!-----------------------------------------------------------------------'
  WRITE(iu,fmt6)   'SUBROUTINE ' // TRIM(process_curr%subr) // '_LOG'
  WRITE(iu,fmt0)   '!-----------------------------------------------------------------------'
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   'USE ' // cp_prefix_modlib // TRIM(LOWCASE(process_curr%ratelaw%name)) // &
                   ', ONLY: ' // TRIM(UPCASE(process_curr%ratelaw%pp_Type))
  WRITE(iu,fmt6)   'USE mod_processdata, ONLY: ' // TRIM(c_ppid)
  WRITE(iu,fmt6)   'USE mod_chemicalconsts'
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   'IMPLICIT NONE'
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   'WRITE(jp_stdlog,*)'
  WRITE(iu,fmt6)   'WRITE(jp_stdlog,*) ''Process: ' // TRIM(process_curr%name) // ''''

  fmttmp = ''
  WRITE(fmttmp,*) '(6X, A, " ''",', LEN_TRIM(process_curr%name)+9, '("-"),"''")'

  WRITE(iu,fmttmp) 'WRITE(jp_stdlog,*)'
  WRITE(iu,fmt6)   'WRITE(jp_stdlog,*)'
  WRITE(iu,fmt6)   'WRITE(jp_stdlog,*) '' Subroutine: ' // TRIM(process_curr%subr) // ''''
  WRITE(iu,fmt6)   'WRITE(jp_stdlog,*) '' Rate law: ' // TRIM(process_curr%ratelaw%name) // ''''
  WRITE(iu,fmt6)   'WRITE(jp_stdlog,*) '' pp_param: ' // TRIM(c_ppid) // ''''
  WRITE(iu,fmt6)   'WRITE(jp_stdlog,*) '' Expression:'''
  WRITE(iu,fmt6)   'WRITE(jp_stdlog,*)'


  fmttmp = ''
  WRITE(fmttmp,*) '("! ",', LEN_TRIM(c_procid)+LEN_TRIM(process_curr%name)+11, '("="))'

  iu = rpunit

  ! Template medusa_rrp_template:
  ! +------------------------------------------------------------------
  !>|! Process 1 [AragDissolution]
  !>|! ===========================
  !>|! Expression: k_1 * [Aragonite] * (ksp_1 - [Ca][CO3])**n_1
  ! |&nml_AragDissolution
  ! |k_1 = ...
  ! |n_1 = ...
  ! |/
  ! |
  ! +------------------------------------------------------------------

  WRITE(iu,fmt_)
  WRITE(iu,fmt0)   '! Process ' // TRIM(c_procid) // &
                   ' [' // TRIM(process_curr%name) // ']'
  WRITE(iu,fmttmp)


  iu = c1unit                      !

  ! Partial file mod_processcontrol_1.F:
  ! +------------------------------------------------------------------
  !>|
  !>|! Process 1 [AragDissolution]
  !>|! ===========================
  ! |      ! Expression: k_1 * [Aragonite] * (ksp_1 - [Ca][CO3])**n_1
  ! |      DOUBLE PRECISION :: k_1
  ! |      DOUBLE PRECISION :: n_1
  ! |
  ! +------------------------------------------------------------------

  WRITE(iu,fmt_)
  WRITE(iu,fmt0)   '! Process ' // TRIM(c_procid) // &
                   ' [' // TRIM(process_curr%name) // ']'
  WRITE(iu,fmttmp)


  iu = c2unit

  ! Partial file mod_processcontrol_2.F:
  ! +------------------------------------------------------------------
  !>|
  !>|! Process 1 [AragDissolution]
  !>|! ===========================
  ! |
  ! |      READ(UNIT=rrpuni, NML=nml_AragDissolution)
  ! |      pp_AragDissolution%RateConstant = k_1
  ! |!     pp_AragDissolution%SolubilityProduct to be set in SetProcessParameters
  ! |      pp_AragDissolution%RateOrder = n_1
  ! |!     pp_AragDissolution%acConcProductParam to be set in SetProcessParameters
  ! |      pp_AragDissolution%ioConcProductSpecies = io_co3
  ! |      pp_AragDissolution%ioProportional = io_arag
  ! |
  ! |      CALL ARAGDISSOLUTION_LOG
  ! |
  ! |
  ! +------------------------------------------------------------------

  WRITE(iu,fmt_)
  WRITE(iu,fmt0)   '! Process ' // TRIM(c_procid) // &
                   ' [' // TRIM(process_curr%name) // ']'
  WRITE(iu,fmttmp)


  iu = c3unit

  ! Partial file mod_processcontrol_3.F:
  ! +------------------------------------------------------------------
  !>|
  !>|! Process 1 [AragDissolution]
  !>|! ===========================
  ! |      pp_AragDissolution%SolubilityProduct
  ! |     &   = cct_ksp_arag
  ! |      pp_AragDissolution%acConcProductParam
  ! |     &   = cct_ttcc_ca
  ! +------------------------------------------------------------------

  WRITE(iu,fmt_)
  WRITE(iu,fmt0)   '! Process ' // TRIM(c_procid) // &
                   ' [' // TRIM(process_curr%name) // ']'
  WRITE(iu,fmttmp)


  iu = reaunit
  WRITE(iu,fmt_)
  WRITE(iu,fmt0)   '! Process ' // TRIM(c_procid) // &
                   ' [' // TRIM(process_curr%name) // ']'
  WRITE(iu,fmttmp)


  iu = dreunit
  WRITE(iu,fmt_)
  WRITE(iu,fmt0)   '! Process ' // TRIM(c_procid) // &
                   ' [' // TRIM(process_curr%name) // ']'
  WRITE(iu,fmttmp)


  iu = d_unit
  WRITE(iu,fmt_)
  WRITE(iu,fmt0)   '! Process ' // TRIM(c_procid) // &
                   ' [' // TRIM(process_curr%name) // ']'
  WRITE(iu,fmttmp)
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   'INTEGER, PARAMETER'
  WRITE(iu,fmtcon) '   :: '// TRIM(process_curr%pi_identifier) // ' = ' // TRIM(c_procid)
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   'DATA c_procname(' // TRIM(process_curr%pi_identifier) // ') '
  WRITE(iu,fmtcon) '   / ''' // TRIM(process_curr%name) // ''' /'
  WRITE(iu,fmt_)
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   'TYPE(' // TRIM(process_curr%ratelaw%pp_type) // '), SAVE'
  WRITE(iu,fmtcon) '   :: ' // TRIM(process_curr%pp_identifier)

  WRITE(c_tmp,'(I0)') n_eqns
  WRITE(iu,fmt6)   'INTEGER, PARAMETER'
  WRITE(iu,fmtcon) '   :: '// TRIM(process_curr%nvp_identifier) // ' = ' // TRIM(c_tmp)
  WRITE(iu,fmt6)   'INTEGER, PARAMETER, ' // &
                   'DIMENSION(' // TRIM(process_curr%nvp_identifier) // ')'
  WRITE(iu,fmtcon) '   :: ' // TRIM(process_curr%iop_identifier)
  WRITE(iu,fmtcon,ADVANCE='NO') '      = '

  CALL EXPAND_TOKEN(fmtconind, '@', '11X,', fmttmp)
  CALL WRITE_ARRAY_CONSTRUCTOR(iu, fmttmp, c_ioid_eqns, n_eqns, CFG_ITPL)

  WRITE(c_tmp,'(I0)') n_ctls
  WRITE(iu,fmt6)   'INTEGER, PARAMETER'
  WRITE(iu,fmtcon) '   :: '// TRIM(process_curr%nvr_identifier) // ' = ' // TRIM(c_tmp)
  WRITE(iu,fmt6)   'INTEGER, PARAMETER, ' // &
                   'DIMENSION(' // TRIM(process_curr%nvr_identifier) // ')'
  WRITE(iu,fmtcon) '   :: ' // TRIM(process_curr%ior_identifier)
  WRITE(iu,fmtcon,ADVANCE='NO') '      = '

  CALL EXPAND_TOKEN(fmtconind, '@', '11X,', fmttmp)
  CALL WRITE_ARRAY_CONSTRUCTOR(iu, fmttmp, c_ioid_ctls, n_ctls, CFG_ITPL)


  WRITE(iu,fmt_)
  WRITE(iu,fmt_)


                                    ! Resolve the expression of the rate_law
                                    ! in terms of the actually used parameters

  c_expression = process_curr%ratelaw%expression

                                    ! First, dress the list of parameters
  n_gk      = 0
  n_gk_xref = 0
  n_pf      = 0
  n_pf_xref = 0
  n_pc      = 0
  n_pc_xref = 0

  DO i_param = 1, process_curr%ratelaw%n_params

    c_kindofparam = process_curr%ratelaw%kindofparam(i_param)
    WRITE(c_sharp, '("{#", I0, "}")') i_param

    ixref_param = process_curr%ratelaw%iparam_xref(i_param)

    SELECT CASE(c_kindofparam)

    CASE('gk')                      ! Global Konstant
                                    ! ---------------
      IF (ixref_param == 0)  THEN
                                    ! Global constant is proper to the
                                    ! current process: simply use dummy
                                    ! label from the current process and
                                    ! use the idx of the current process
                                    ! as a suffix.
        c_dummylabel = ADJUSTL(process_curr%ratelaw%dummylabel(i_param))
        nlen = LEN_TRIM(c_dummylabel)
        WRITE(c_param, '(A, "_", I0)') c_dummylabel(1:nlen), process_curr%idx

      ELSE
                                    ! Global constant is inherited from
                                    ! a previous process. Check if that
                                    ! process possibly also inherited the
                                    ! same constant and find the earliest
                                    ! xref'd process for that constant.
                                    ! Use the dummy label from that
                                    ! process and its idx as a suffix.
        process_work => process_xref
        ixxref_param = ixref_param
        DO
          ixxref_test_param = process_work%ratelaw%iparam_xref(ixxref_param)
          IF (ixxref_test_param == 0) EXIT

          ixxref_param = ixxref_test_param
          idx_xxref =  process_work%idx_xref

          DO WHILE (process_work%idx /= idx_xxref)
            process_work => process_work%prev
          ENDDO
        ENDDO

        c_dummylabel = ADJUSTL(process_work%ratelaw%dummylabel(ixxref_param))
        nlen = LEN_TRIM(c_dummylabel)
        WRITE(c_param, '(A, "_", I0)') c_dummylabel(1:nlen), process_work%idx

        n_gk_xref = n_gk_xref + 1

      ENDIF

      n_gk = n_gk + 1


    CASE('pf')                      ! Parameter Function
                                    ! ------------------
      IF (ixref_param == 0)  THEN
                                    ! Parameter Function is proper to the
                                    ! current process
        c_dummylabel = ADJUSTL(process_curr%ratelaw%dummylabel(i_param))
        nlen = LEN_TRIM(c_dummylabel)
        IF(nlen > 0) THEN
          WRITE(c_param, '(A, "_", I0)') c_dummylabel(1:nlen), process_curr%idx
        ELSE
          c_param = process_curr%ratelaw%paramname(i_param)
        ENDIF

      ELSE
                                    ! Parameter Function is inherited from
                                    ! a previous process. Check if that process
                                    ! possibly also inherited the same constant
                                    ! and find the earliest xref'd process
                                    ! for that parameter function.
        process_work => process_xref
        ixxref_param = ixref_param
        DO
          ixxref_test_param = process_work%ratelaw%iparam_xref(ixxref_param)
          IF (ixxref_test_param == 0) EXIT

          ixxref_param = ixxref_test_param
          idx_xxref =  process_work%idx_xref

          DO WHILE (process_work%idx /= idx_xxref)
            process_work => process_work%prev
          ENDDO
        ENDDO

        c_dummylabel = ADJUSTL(process_work%ratelaw%dummylabel(ixxref_param))
        nlen = LEN_TRIM(c_dummylabel)

        IF(nlen > 0) THEN
          WRITE(c_param, '(A, "_", I0)') c_dummylabel(1:nlen), process_work%idx
        ELSE
          c_param = process_work%ratelaw%paramname(ixxref_param)
        ENDIF

        n_pf_xref = n_pf_xref + 1

      ENDIF

      n_pf = n_pf + 1


    CASE('pc')                    ! parameterized concentration
      c_param = process_curr%ratelaw%paramname(i_param)

      IF (ixref_param /= 0) n_pc_xref = n_pc_xref + 1
      n_pc = n_pc + 1

    CASE('io')                    ! concentration of a modelled component

       c_param = ADJUSTL(process_curr%ratelaw%paramname(i_param))

    CASE DEFAULT

       c_param = '?1004?'

    END SELECT

    CALL EXPAND_TOKEN(c_expression, c_sharp, c_param, c_expression1)
    c_expression = c_expression1

  ENDDO
                                    ! Write out the ratelaw expression
                                    ! into the relevant files
  ! 1. s_unit
  WRITE(s_unit,fmt6)   'WRITE(jp_stdlog,*) ''  ' // TRIM(c_expression) //  ''''
  WRITE(s_unit,fmt6)   'WRITE(jp_stdlog,*)'


  ! 2. rpunit -- medusa_rrp_template
  ! +------------------------------------------------------------------
  !=|! Process 1 [AragDissolution]
  !=|! ===========================
  !=|! Expression: k_1 * [Aragonite] * (ksp_1 - [Ca][CO3])**n_1
  !>|&nml_AragDissolution
  ! |k_1 = ...
  ! |n_1 = ...
  ! |/
  ! |
  ! +------------------------------------------------------------------

  WRITE(rpunit,fmt0)   '! Expression: ' // TRIM(c_expression)
  IF (n_gk /= 0) THEN
    IF (n_gk_xref < n_gk) THEN
      WRITE(rpunit,fmt0)   '&nml_' // TRIM(process_curr%name)
    ELSE
      WRITE(rpunit,fmt0, ADVANCE='NO')   '! All global parameter values taken from process '
      WRITE(rpunit,'(I0," [",A,"]")') process_curr%idx_xref, TRIM(process_xref%name)
    ENDIF
  ELSE
    WRITE(rpunit,fmt0)   '! No global parameter values required'
  ENDIF


  ! 3.1 c1unit -- mod_processcontrol_1.F (variable declaration part)
  ! +------------------------------------------------------------------
  !=|
  !=|! Process 1 [AragDissolution]
  !=|! ===========================
  !>|      ! Expression: k_1 * [Aragonite] * (ksp_1 - [Ca][CO3])**n_1
  ! |      DOUBLE PRECISION :: k_1
  ! |      DOUBLE PRECISION :: n_1
  ! |
  ! +------------------------------------------------------------------

  WRITE(c1unit,fmt6)    '! Expression: ' // TRIM(c_expression)

  ! 3.2 ctunit -- mod_processcontrol_t.F
  ! (namelist declaration part for mod_processcontrol_1.F)
  ! +------------------------------------------------------------------
  !>|   NAMELIST /nml_AragDissolution/
  !.|  &          k_1, n_1
  ! |
  ! +------------------------------------------------------------------

  IF(n_gk /= 0) THEN
    IF (n_gk_xref < n_gk) THEN
      WRITE(ctunit,fmt6)   'NAMELIST /nml_' // TRIM(process_curr%name) // '/'
      WRITE(ctunit,fmtcon, ADVANCE='NO') '          '
    ELSE
      WRITE(ctunit,fmt6, ADVANCE='NO')   '! All global parameter values taken from process '
      WRITE(ctunit,'(I0," [",A,"]")') process_curr%idx_xref, TRIM(process_xref%name)
    ENDIF
  ELSE
    WRITE(ctunit,fmt6, ADVANCE='NO')   '! No global parameter values required'
  ENDIF


  ! 4. c2unit -- mod_processcontrol_2.F (reading and assigning part)
  ! +------------------------------------------------------------------
  !=|
  !=|! Process 1 [AragDissolution]
  !=|! ===========================
  !>|
  !>|      IF (l_rrpfile_is_mine)
  !>|     &  READ(UNIT=rrpuni, NML=nml_AragDissolution)
  !>|
  ! |      pp_AragDissolution%RateConstant = k_1
  ! |!     pp_AragDissolution%SolubilityProduct to be set in SetProcessParameters
  ! |      pp_AragDissolution%RateOrder = n_1
  ! |!     pp_AragDissolution%acConcProductParam to be set in SetProcessParameters
  ! |      pp_AragDissolution%ioConcProductSpecies = io_co3
  ! |      pp_AragDissolution%ioProportional = io_arag
  ! |
  ! |      CALL ARAGDISSOLUTION_LOG
  ! |
  ! |
  ! +------------------------------------------------------------------

  IF (n_gk /= 0) THEN
    IF (n_gk_xref < n_gk) THEN
      WRITE(c2unit,fmt_)
      WRITE(c2unit,fmt6)   'IF (l_rrpfile_is_mine)'
      WRITE(c2unit,fmtcon) '  READ(UNIT=rrpuni, NML=nml_' // TRIM(process_curr%name) // ')'
      WRITE(c2unit,fmt_)
    ENDIF
  ENDIF


  ! 5. c3unit -- mod_processcontrol_3.F (variable assigning part)
  ! +------------------------------------------------------------------
  !=|
  !=|! Process 1 [AragDissolution]
  !=|! ===========================
  ! |      pp_AragDissolution%SolubilityProduct
  ! |     &   = cct_ksp_arag
  ! |      pp_AragDissolution%acConcProductParam
  ! |     &   = cct_ttcc_ca
  ! +------------------------------------------------------------------

  ! Nothing to be done
                                    ! Now redo the loop and write out
                                    ! the actual parts of the code
                                    ! to the relevant files
  i_gk      = 0
  i_gk_xref = 0

  DO i_param = 1, process_curr%ratelaw%n_params

    c_typecomponame = ADJUSTL(process_curr%ratelaw%typecomponame(i_param))

    c_kindofparam = process_curr%ratelaw%kindofparam(i_param)
    WRITE(c_sharp, '("{#", I0,"}")') i_param

    ixref_param = process_curr%ratelaw%iparam_xref(i_param)

    SELECT CASE(c_kindofparam)

    CASE('gk')                      ! Global Konstant

      i_gk = i_gk + 1

      IF (ixref_param == 0) THEN
                                    ! Global constant is proper to the
                                    ! current process
        c_dummylabel = ADJUSTL(process_curr%ratelaw%dummylabel(i_param))
        nlen = LEN_TRIM(c_dummylabel)
        WRITE(c_param, '(A, "_", I0)') c_dummylabel(1:nlen), process_curr%idx

      ELSE
                                    ! Global constant is inherited from
                                    ! another process. Search for the original.
        i_gk_xref = i_gk_xref + 1

        process_work => process_xref
        ixxref_param = ixref_param
        DO
          ixxref_test_param = process_work%ratelaw%iparam_xref(ixxref_param)
          IF (ixxref_test_param == 0) EXIT

          ixxref_param = ixxref_test_param
          idx_xxref =  process_work%idx_xref

          DO WHILE (process_work%idx /= idx_xxref)
            process_work => process_work%prev
          ENDDO
        ENDDO

        c_dummylabel = ADJUSTL(process_work%ratelaw%dummylabel(ixxref_param))
        nlen = LEN_TRIM(c_dummylabel)
        WRITE(c_param, '(A, "_", I0)') c_dummylabel(1:nlen), process_work%idx

      ENDIF

      ! 1. s_unit
      WRITE(s_unit,fmt6)   'WRITE(jp_stdlog,*) ''  -> ' // &
                            TRIM(c_param) // ' = '','
      WRITE(s_unit,fmtcon) '                   ' // &
                            TRIM(c_ppid) // '%' // TRIM(c_typecomponame)


      ! 2. rpunit -- medusa_rrp_template
      ! +------------------------------------------------------------------
      !=|! Process 1 [AragDissolution]
      !=|! ===========================
      !=|! Expression: k_1 * [Aragonite] * (ksp_1 - [Ca][CO3])**n_1
      !=|&nml_AragDissolution
      !>|k_1 = ...
      !>|n_1 = ...
      ! |/
      ! |
      ! +------------------------------------------------------------------

      IF (process_curr%ratelaw%iparam_xref(i_param) == 0) THEN
        WRITE(rpunit,fmt0)   TRIM(c_param) // ' = ...'
      ENDIF


      ! 3.1 c1unit --  mod_processcontrol_1.F (variable declaration part)
      ! +------------------------------------------------------------------
      !=|
      !=|! Process 1 [AragDissolution]
      !=|! ===========================
      !=|      ! Expression: k_1 * [Aragonite] * (ksp_1 - [Ca][CO3])**n_1
      !>|      DOUBLE PRECISION :: k_1
      !>|      DOUBLE PRECISION :: n_1
      ! |
      ! +------------------------------------------------------------------

      IF (process_curr%ratelaw%iparam_xref(i_param) == 0) THEN
        WRITE(c1unit,fmt6)   'DOUBLE PRECISION :: ' // TRIM(c_param)
      ENDIF


      ! 3.2 ctunit -- mod_processcontrol_t.F
      ! (namelist declaration part for mod_processcontrol_1.F)
      ! +------------------------------------------------------------------
      !=|   NAMELIST /nml_AragDissolution/
      !>|  &          k_1, n_1
      ! |
      ! +------------------------------------------------------------------

        IF (ixref_param == 0) THEN
          IF ((n_gk-i_gk) == (n_gk_xref-i_gk_xref)) THEN
            WRITE(ctunit, '(A)')               TRIM(c_param)
          ELSE
            WRITE(ctunit, '(A)', ADVANCE='NO') TRIM(c_param) // ', '
          ENDIF
        ENDIF


      ! 4. c2unit -- mod_processcontrol_2.F (reading and assigning part)
      ! +------------------------------------------------------------------
      !=|
      !=|! Process 1 [AragDissolution]
      !=|! ===========================
      !=|
      !=|      IF (l_rrpfile_is_mine)
      !=|     &  READ(UNIT=rrpuni, NML=nml_AragDissolution)
      !=|
      !>|      pp_AragDissolution%RateConstant = k_1
      ! |!     pp_AragDissolution%SolubilityProduct to be set in SetProcessParameters
      !>|      pp_AragDissolution%RateOrder = n_1
      ! |!     pp_AragDissolution%acConcProductParam to be set in SetProcessParameters
      ! |      pp_AragDissolution%ioConcProductSpecies = io_co3
      ! |      pp_AragDissolution%ioProportional = io_arag
      ! |
      ! |      CALL ARAGDISSOLUTION_LOG
      ! |
      ! |
      ! +------------------------------------------------------------------

      IF (ixref_param == 0) THEN
        WRITE(c2unit,fmt0)   '#ifdef ALLOW_MPI'
        WRITE(c2unit,fmt6)   'CALL MPI_BCAST(' // TRIM(c_param) // ', 1, MPI_DOUBLE_PRECISION,'
        WRITE(c2unit,fmtcon) '               jp_exeproc_root, i_mycomm, i_flag)'
        WRITE(c2unit,fmt0)   '#endif'
      ENDIF
      WRITE(c2unit,fmt6)   TRIM(c_ppid) // '%' // TRIM(c_typecomponame) // &
                           ' = ' // TRIM(c_param)


      ! 5. c3unit -- mod_processcontrol_3.F (variable assigning part)
      ! +------------------------------------------------------------------
      !=|
      !=|! Process 1 [AragDissolution]
      !=|! ===========================
      ! |      pp_AragDissolution%SolubilityProduct
      ! |     &   = cct_ksp_arag
      ! |      pp_AragDissolution%acConcProductParam
      ! |     &   = cct_ttcc_ca
      ! +------------------------------------------------------------------

      ! Nothing to be done here

    CASE('pf', 'pc')                ! parameterized concentration

      c_paramcode = process_curr%ratelaw%paramcode(i_param)

      IF (ixref_param == 0) THEN
        c_dummylabel = ADJUSTL(process_curr%ratelaw%dummylabel(i_param))
        nlen = LEN_TRIM(c_dummylabel)
        WRITE(c_param, '(A, "_", I0)') c_dummylabel(1:nlen), process_curr%idx
      ELSE
        c_dummylabel = ADJUSTL(process_xref%ratelaw%dummylabel(ixref_param))
        nlen = LEN_TRIM(c_dummylabel)
        WRITE(c_param, '(A, "_", I0)') c_dummylabel(1:nlen), process_curr%idx_xref
      ENDIF

      IF (nlen /= 0) THEN           ! If there is a dummy label,
                                    ! it is a true parameter function

        ! 1. s_unit
        WRITE(s_unit,fmt6)   'WRITE(jp_stdlog,*) ''  -> '  // TRIM(c_param)      // ' = '' //'
        WRITE(s_unit,fmtcon) '                      ''' // TRIM(c_paramcode)      // ''''


        ! 2. rpunit -- medusa_rrp_template
        ! +------------------------------------------------------------------
        !=|! Process 1 [AragDissolution]
        !=|! ===========================
        !=|! Expression: k_1 * [Aragonite] * (ksp_1 - [Ca][CO3])**n_1
        !=|&nml_AragDissolution
        !=|k_1 = ...
        !=|n_1 = ...
        ! |/
        ! |
        ! +------------------------------------------------------------------

        ! Nothing to be done here


        ! 3.1 c1unit -- mod_processcontrol_1.F (variable declaration part)
        ! +------------------------------------------------------------------
        !=|
        !=|! Process 1 [AragDissolution]
        !=|! ===========================
        !=|      ! Expression: k_1 * [Aragonite] * (ksp_1 - [Ca][CO3])**n_1
        !=|      DOUBLE PRECISION :: k_1
        !=|      DOUBLE PRECISION :: n_1
        ! |
        ! +------------------------------------------------------------------

        ! Nothing to be done here


        ! 3.2 ctunit -- mod_processcontrol_t.F
        ! (namelist declaration part for mod_processcontrol_1.F)
        ! +------------------------------------------------------------------
        !=|   NAMELIST /nml_AragDissolution/
        !=|  &          k_1, n_1
        ! |
        ! +------------------------------------------------------------------

        ! Nothing to be done here


        ! 4. c2unit -- mod_processcontrol_2.F (reading and assigning part)
        ! +------------------------------------------------------------------
        !=|
        !=|! Process 1 [AragDissolution]
        !=|! ===========================
        !=|
        !=|      READ(UNIT=rrpuni, NML=nml_AragDissolution)
        !=|      pp_AragDissolution%RateConstant = k_1
        !>|!     pp_AragDissolution%SolubilityProduct to be set in SetProcessParameters
        !=|      pp_AragDissolution%RateOrder = n_1
        ! |!     pp_AragDissolution%acConcProductParam to be set in SetProcessParameters
        ! |      pp_AragDissolution%ioConcProductSpecies = io_co3
        ! |      pp_AragDissolution%ioProportional = io_arag
        ! |
        ! |      CALL ARAGDISSOLUTION_LOG
        ! |
        ! |
        ! +------------------------------------------------------------------

        WRITE(c2unit,fmt0)   '!     ' // TRIM(c_ppid) // '%' // &
                             TRIM(c_typecomponame) // &
                             ' to be set in SetProcessParameters'


        ! 5. c3unit -- mod_processcontrol_3.F (variable assigning part)
        ! +------------------------------------------------------------------
        !=|
        !=|! Process 1 [AragDissolution]
        !=|! ===========================
        !>|      pp_AragDissolution%SolubilityProduct
        !>|     &   = cct_ksp_arag
        ! |      pp_AragDissolution%acConcProductParam
        ! |     &   = cct_ttcc_ca
        ! +------------------------------------------------------------------

        IF (ixref_param == 0) THEN
          WRITE(c3unit,fmt6)   TRIM(c_ppid) // '%' // &
                               TRIM(c_typecomponame)
          WRITE(c3unit,fmtcon) '   = ' // TRIM(c_paramcode)
        ELSE
          WRITE(c3unit,fmt6)   TRIM(c_ppid) // '%' // &
                               TRIM(c_typecomponame)
          WRITE(c3unit,fmtcon) '   = ' // &
                               TRIM(process_xref%pp_identifier)// '%' // &
                               TRIM(process_xref%ratelaw%typecomponame(ixref_param))
        ENDIF

      ELSE                          ! It is a parameterized concentration

        c_compo = ADJUSTL(process_curr%ratelaw%paramname(i_param))

        ! 1. s_unit
        WRITE(s_unit,fmt6)   'WRITE(jp_stdlog,*) ''  -> [' // TRIM(c_compo) // '] = '' //'
        WRITE(s_unit,fmtcon) '                      ''' // TRIM(c_paramcode) // ''''


        ! 2. rpunit -- medusa_rrp_template
        ! +------------------------------------------------------------------
        !=|! Process 1 [AragDissolution]
        !=|! ===========================
        !=|! Expression: k_1 * [Aragonite] * (ksp_1 - [Ca][CO3])**n_1
        !=|&nml_AragDissolution
        !=|k_1 = ...
        !=|n_1 = ...
        ! |/
        ! |
        ! +------------------------------------------------------------------

        ! Nothing to be done here


        ! 3.1 c1unit -- mod_processcontrol_1.F (variable declaration part)
        ! +------------------------------------------------------------------
        !=|
        !=|! Process 1 [AragDissolution]
        !=|! ===========================
        !=|      ! Expression: k_1 * [Aragonite] * (ksp_1 - [Ca][CO3])**n_1
        !=|      DOUBLE PRECISION :: k_1
        !=|      DOUBLE PRECISION :: n_1
        ! |
        ! +------------------------------------------------------------------

        !  Nothing to be done here


        ! 3.2 ctunit -- mod_processcontrol_t.F
        ! (namelist declaration part for mod_processcontrol_1.F)
        ! +------------------------------------------------------------------
        !=|   NAMELIST /nml_AragDissolution/
        !=|  &          k_1, n_1
        ! |
        ! +------------------------------------------------------------------

        ! Nothing to be done here


        ! 4. c2unit -- mod_processcontrol_2.F (reading part)
        ! +------------------------------------------------------------------
        !=|
        !=|! Process 1 [AragDissolution]
        !=|! ===========================
        !=|
        !=|      READ(UNIT=rrpuni, NML=nml_AragDissolution)
        !=|      pp_AragDissolution%RateConstant = k_1
        !=|!     pp_AragDissolution%SolubilityProduct to be set in SetProcessParameters
        !=|      pp_AragDissolution%RateOrder = n_1
        !>|!     pp_AragDissolution%acConcProductParam to be set in SetProcessParameters
        ! |      pp_AragDissolution%ioConcProductSpecies = io_co3
        ! |      pp_AragDissolution%ioProportional = io_arag
        ! |
        ! |      CALL ARAGDISSOLUTION_LOG
        ! |
        ! |
        ! +------------------------------------------------------------------

        WRITE(c2unit,fmt0)   '!     ' // TRIM(c_ppid) // '%' // &
                             TRIM(c_typecomponame) // &
                             ' to be set in SetProcessParameters'


        ! 5. c3unit -- mod_processcontrol_3.F (variable assigning part)
        ! +------------------------------------------------------------------
        !=|
        !=|! Process 1 [AragDissolution]
        !=|! ===========================
        !=|      pp_AragDissolution%SolubilityProduct
        !=|     &   = cct_ksp_arag
        !>|      pp_AragDissolution%acConcProductParam
        !>|     &   = cct_ttcc_ca
        ! +------------------------------------------------------------------

        ixref_param = process_curr%ratelaw%iparam_xref(i_param)
        IF (ixref_param == 0) THEN
          WRITE(c3unit,fmt6)   TRIM(c_ppid) // '%' // &
                               TRIM(c_typecomponame)
          WRITE(c3unit,fmtcon) '   = ' // TRIM(c_paramcode)
        ELSE
          WRITE(c3unit,fmt6)   TRIM(c_ppid) // '%' // &
                               TRIM(c_typecomponame)
          WRITE(c3unit,fmtcon) '   = ' // &
                               TRIM(process_xref%pp_identifier) // '%' // &
                               TRIM(process_xref%ratelaw%typecomponame(ixref_param))
        ENDIF

      ENDIF

    CASE('io')                      ! concentration of a modelled component

      c_paramcode = process_curr%ratelaw%paramcode(i_param)
      ! 1. s_unit
      ! Nothing to be done here


      ! 2. rpunit -- medusa_rrp_template
      ! +------------------------------------------------------------------
      !=|! Process 1 [AragDissolution]
      !=|! ===========================
      !=|! Expression: k_1 * [Aragonite] * (ksp_1 - [Ca][CO3])**n_1
      !=|&nml_AragDissolution
      !=|k_1 = ...
      !=|n_1 = ...
      ! |/
      ! |
      ! +------------------------------------------------------------------

      ! Nothing to be done here


      ! 3.1 c1unit -- mod_processcontrol_1.F (variable declaration part)
      ! +------------------------------------------------------------------
      !=|
      !=|! Process 1 [AragDissolution]
      !=|! ===========================
      !=|      ! Expression: k_1 * [Aragonite] * (ksp_1 - [Ca][CO3])**n_1
      !=|      DOUBLE PRECISION :: k_1
      !=|      DOUBLE PRECISION :: n_1
      ! |
      ! +------------------------------------------------------------------

      ! Nothing to be done here


      ! 3.2 ctunit -- mod_processcontrol_t.F
      ! (namelist declaration part for mod_processcontrol_1.F)
      ! +------------------------------------------------------------------
      !=|   NAMELIST /nml_AragDissolution/
      !=|  &          k_1, n_1
      ! |
      ! +------------------------------------------------------------------

      ! Nothing to be done here


      ! 4. c2unit -- mod_processcontrol_2.F (reading part)
      ! +------------------------------------------------------------------
      !=|
      !=|! Process 1 [AragDissolution]
      !=|! ===========================
      !=|
      !=|      READ(UNIT=rrpuni, NML=nml_AragDissolution)
      !=|      pp_AragDissolution%RateConstant = k_1
      !=|!     pp_AragDissolution%SolubilityProduct to be set in SetProcessParameters
      !=|      pp_AragDissolution%RateOrder = n_1
      !=|!     pp_AragDissolution%acConcProductParam to be set in SetProcessParameters
      !>|      pp_AragDissolution%ioConcProductSpecies = io_co3
      !>|      pp_AragDissolution%ioProportional = io_arag
      ! |
      ! |      CALL ARAGDISSOLUTION_LOG
      ! |
      ! |
      ! +------------------------------------------------------------------

      WRITE(c2unit,fmt6)   TRIM(c_ppid) // '%' // TRIM(c_typecomponame) // &
                            ' = '  // TRIM(c_paramcode)


      ! 5. c3unit -- mod_processcontrol_3.F (variable assigning part)
      ! +------------------------------------------------------------------
      !=|
      !=|! Process 1 [AragDissolution]
      !=|! ===========================
      !=|      pp_AragDissolution%SolubilityProduct
      !=|     &   = cct_ksp_arag
      !=|      pp_AragDissolution%acConcProductParam
      !=|     &   = cct_ttcc_ca
      ! +------------------------------------------------------------------

      ! Nothing to be done here

    CASE DEFAULT

      c_param = '?1429?'

    END SELECT

  ENDDO


  IF (n_gk /= 0) THEN

    ! 1. s_unit
    WRITE(s_unit,fmt6)   'WRITE(jp_stdlog,*)'


    ! 2. rpunit -- medusa_rrp_template
    ! +------------------------------------------------------------------
    !=|! Process 1 [AragDissolution]
    !=|! ===========================
    !=|! Expression: k_1 * [Aragonite] * (ksp_1 - [Ca][CO3])**n_1
    !=|&nml_AragDissolution
    !=|k_1 = ...
    !=|n_1 = ...
    !>|/
    !>|
    ! +------------------------------------------------------------------

    IF (n_gk_xref < n_gk) THEN
      WRITE(rpunit,fmt0)   '/'
    ENDIF
    WRITE(rpunit,fmt0)


    ! 3.1 c1unit -- mod_processcontrol_1 (variable declaration part)
    ! +------------------------------------------------------------------
    !=|
    !=|! Process 1 [AragDissolution]
    !=|! ===========================
    !=|      ! Expression: k_1 * [Aragonite] * (ksp_1 - [Ca][CO3])**n_1
    !=|      DOUBLE PRECISION :: k_1
    !=|      DOUBLE PRECISION :: n_1
    !>|
    ! +------------------------------------------------------------------

    WRITE(c1unit,fmt_)


    ! 3.2 ctunit -- mod_processcontrol_t.F
    ! (namelist declaration part for mod_processcontrol_1.F)
    ! +------------------------------------------------------------------
    !=|   NAMELIST /nml_AragDissolution/
    !=|  &          k_1, n_1
    !>|
    ! +------------------------------------------------------------------

    WRITE(ctunit,fmt_)


    ! 4. c2unit -- mod_processcontrol_2.F (reading part)
    ! +------------------------------------------------------------------
    !=|
    !=|! Process 1 [AragDissolution]
    !=|! ===========================
    !=|
    !=|      READ(UNIT=rrpuni, NML=nml_AragDissolution)
    !=|      pp_AragDissolution%RateConstant = k_1
    !=|!     pp_AragDissolution%SolubilityProduct to be set in SetProcessParameters
    !=|      pp_AragDissolution%RateOrder = n_1
    !=|!     pp_AragDissolution%acConcProductParam to be set in SetProcessParameters
    !=|      pp_AragDissolution%ioConcProductSpecies = io_co3
    !=|      pp_AragDissolution%ioProportional = io_arag
    ! |
    ! |      CALL ARAGDISSOLUTION_LOG
    ! |
    ! |
    ! +------------------------------------------------------------------


    ! 5. c3unit -- mod_processcontrol_3.F (variable assigning part)
    ! +------------------------------------------------------------------
    !=|
    !=|! Process 1 [AragDissolution]
    !=|! ===========================
    !=|      pp_AragDissolution%SolubilityProduct
    !=|     &   = cct_ksp_arag
    !=|      pp_AragDissolution%acConcProductParam
    !=|     &   = cct_ttcc_ca
    ! +------------------------------------------------------------------

  ENDIF

  IF ((n_pf+n_pc) == 0) THEN

    WRITE(c3unit,fmt6)  '! all set'

  ENDIF



  iu = s_unit
  WRITE(iu,fmt_)
  WRITE(iu,fmt6) 'RETURN'
  WRITE(iu,fmt_)
  WRITE(iu,fmt0) '!-----------------------------------------------------------------------'
  WRITE(iu,fmt6) 'END SUBROUTINE ' // TRIM(process_curr%subr) // '_LOG'
  WRITE(iu,fmt0) '!-----------------------------------------------------------------------'
  WRITE(iu,fmt_)
  WRITE(iu,fmt_)


  iu = c2unit

  ! mod_processcontrol_2.F (reading part)
  ! +------------------------------------------------------------------
  !=|
  !=|! Process 1 [AragDissolution]
  !=|! ===========================
  !=|
  !=|      READ(UNIT=rrpuni, NML=nml_AragDissolution)
  !=|
  !=|      pp_AragDissolution%RateConstant = k_1
  !=|!     pp_AragDissolution%SolubilityProduct to be set in SetProcessParameters
  !=|      pp_AragDissolution%RateOrder = n_1
  !=|!     pp_AragDissolution%acConcProductParam to be set in SetProcessParameters
  !=|      pp_AragDissolution%ioConcProductSpecies = io_co3
  !=|      pp_AragDissolution%ioProportional = io_arag
  !>|
  !>|      CALL ARAGDISSOLUTION_LOG
  !>|
  !>|
  ! +------------------------------------------------------------------

  WRITE(iu,fmt_)
  WRITE(iu,fmt6) 'CALL ' // TRIM(process_curr%subr) // '_LOG'
  WRITE(iu,fmt_)
  WRITE(iu,fmt_)


  iu = reaunit
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   'SELECT CASE(krealm)'
  WRITE(iu,fmt6, ADVANCE='NO') 'CASE('
  DO i = 1, process_curr%n_realms
    WRITE(iu,'(A)', ADVANCE='NO') TRIM(process_curr%c_realm_identifiers(i))
    IF (i <  process_curr%n_realms) THEN
      WRITE(iu,'(", ")', ADVANCE='NO')
    ELSE
      WRITE(iu,'(")")')
    ENDIF
  ENDDO


  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   '  areac_factor = rreac_factor(' // TRIM(process_curr%pi_identifier) // ')'
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   '  CALL ' // TRIM(process_curr%subr)
  WRITE(iu,fmtcon) '     (ac=ac(:), azdn=azdn, aphi=aphi, areac_factor=areac_factor,'
  WRITE(iu,fmtcon) '      ac_rate=arate(:))'
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   '  area(' // TRIM(process_curr%iop_identifier) // '(:))'
  WRITE(iu,fmtcon) '   =   area(' // TRIM(process_curr%iop_identifier) // '(:))'
  WRITE(iu,fmtcon) '    + arate(' // TRIM(process_curr%iop_identifier) // '(:))'
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   '  IF (PRESENT(aproc)) THEN'
  WRITE(iu,fmt6)   '    aproc(:, ' // TRIM(process_curr%pi_identifier) // ') = arate(:)'
  WRITE(iu,fmt6)   '  ENDIF'
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   'CASE DEFAULT'
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   '  IF (PRESENT(aproc)) THEN'
  WRITE(iu,fmt6)   '    aproc(:, ' // TRIM(process_curr%pi_identifier) // ') = 0.0D+00'
  WRITE(iu,fmt6)   '  ENDIF'
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   'END SELECT'
  WRITE(iu,fmt_)


  iu = dreunit
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   'SELECT CASE(krealm)'
  WRITE(iu,fmt6, ADVANCE='NO') 'CASE('
  DO i = 1, process_curr%n_realms
    WRITE(iu,'(A)', ADVANCE='NO') TRIM(process_curr%c_realm_identifiers(i))
    IF (i <  process_curr%n_realms) THEN
      WRITE(iu,'(", ")', ADVANCE='NO')
    ELSE
      WRITE(iu,'(")")')
    ENDIF
  ENDDO


  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   '  areac_factor = rreac_factor(' // TRIM(process_curr%pi_identifier) // ')'
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   '  CALL ' // TRIM(process_curr%subr)
  WRITE(iu,fmtcon) '     (ac=ac(:), azdn=azdn, aphi=aphi, areac_factor=areac_factor,'
  WRITE(iu,fmtcon) '      dac_rate_dac=dr_dac(:,:))'
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   '  aread(' // TRIM(process_curr%iop_identifier) // '(:),' &
                              // TRIM(process_curr%ior_identifier) // '(:))'
  WRITE(iu,fmtcon) '   =   aread(' // TRIM(process_curr%iop_identifier) // '(:),' &
                                   // TRIM(process_curr%ior_identifier) // '(:))'
  WRITE(iu,fmtcon) '    + dr_dac(' // TRIM(process_curr%iop_identifier) // '(:),' &
                                   // TRIM(process_curr%ior_identifier) // '(:))'
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   'CASE DEFAULT'
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   '  CONTINUE'
  WRITE(iu,fmt_)
  WRITE(iu,fmt6)   'END SELECT'
  WRITE(iu,fmt_)

                                    ! Transcribe the current c_ioid_eqns
                                    ! into c_ioid_eqns_allprocs
                                    ! and similarly update c_procid_allprocs

  c_ioid_eqns_save => c_ioid_eqns_allprocs
  ALLOCATE(c_ioid_eqns_allprocs(n_eqns_allprocs+n_eqns))
  c_ioid_eqns_allprocs(1:n_eqns_allprocs) = c_ioid_eqns_save(1:n_eqns_allprocs)
  c_ioid_eqns_allprocs(n_eqns_allprocs+1:n_eqns_allprocs+n_eqns) = c_ioid_eqns(1:n_eqns)
  DEALLOCATE(c_ioid_eqns_save)
  NULLIFY(c_ioid_eqns_save)

  c_procid_save    => c_procid_allprocs
  ALLOCATE(c_procid_allprocs(n_eqns_allprocs+n_eqns))
  c_procid_allprocs(1:n_eqns_allprocs) = c_procid_save(1:n_eqns_allprocs)
  c_procid_allprocs(n_eqns_allprocs+1:n_eqns_allprocs+n_eqns) = c_procid
  DEALLOCATE(c_procid_save)
  NULLIFY(c_procid_save)

  n_eqns_allprocs = n_eqns_allprocs + n_eqns


  DEALLOCATE(c_linebegin)
  DEALLOCATE(c_lineend)
  DEALLOCATE(c_ioid_eqns)
  DEALLOCATE(c_ioid_ctls)

  CLOSE(UNIT=ctunit)                ! Close "mod_processcontrol_t.F"
                                    ! and transcribe its contents into
                                    ! "mod_processcontrol_t1.F" and delete it.
  OPEN(UNIT=ctunit, FILE = "tmp/mod_processcontrol_t.F")
  DO
    READ(ctunit,'(A)', END=991) c_codeline
    WRITE(c1unit, '(A)') TRIM(c_codeline)
    CYCLE
    991 EXIT
  ENDDO
  CLOSE(unit=ctunit, STATUS='DELETE')

  process_curr => process_curr%next

ENDDO

iu = s_unit
WRITE(iu,fmt0)   '!---+----1----+----2----+----3----+----4----+----5----+----6----+----7--'
WRITE(iu,fmt_)
WRITE(iu,fmt_)
WRITE(iu,fmt0) '!======================================================================='
WRITE(iu,fmt6) 'END MODULE MOD_PROCESSSUBR'
WRITE(iu,fmt0) '!======================================================================='


CLOSE(UNIT=s_unit)                  ! Done with the module MOD_PROCESSSUBR: close it



iu = d_unit
WRITE(iu,fmt_)
WRITE(iu,fmt6) '! All process contributions'
WRITE(iu,fmt6) '!  - nvp__allprocs: number of components affected by all'
WRITE(iu,fmt6) '!                   the processes, counting each component'
WRITE(iu,fmt6) '!                   as often as there are processes'
WRITE(iu,fmt6) '!                   affecting it'
WRITE(iu,fmt6) '!  - iop__allprocs: io indices of the components affected by'
WRITE(iu,fmt6) '!                   all the process (includes as many entries'
WRITE(iu,fmt6) '!                   for each component as there are processes'
WRITE(iu,fmt6) '!                   affecting it)'
WRITE(iu,fmt6) '!  - pi__allprocs:  IDs of the processes that affect the'
WRITE(iu,fmt6) '!                   components listed in iop__allprocs'
WRITE(iu,fmt_)
WRITE(iu,fmt_)
WRITE(c_tmp,'(I0)') n_eqns_allprocs
WRITE(iu,fmt6) 'INTEGER, PARAMETER :: nvp__allprocs = ' // TRIM(c_tmp)
WRITE(iu,fmt_)
WRITE(iu,fmt6) 'INTEGER, DIMENSION(nvp__allprocs), PARAMETER'
WRITE(iu,fmtcon) '   :: iop__allprocs'
!WRITE(iu,fmtcon,ADVANCE='NO') '      = '
!
!CALL EXPAND_TOKEN(fmtconind, '@', '11X,', fmttmp)
!CALL WRITE_ARRAY_CONSTRUCTOR(iu, fmttmp, c_ioid_eqns_allprocs, n_eqns_allprocs, CFG_ITPL)
CALL WRITE_CONEQ_ARRAY_CONSTRUCTOR(iu, fmtcon, c_ioid_eqns_allprocs, n_eqns_allprocs, 6)
WRITE(iu,fmt_)
WRITE(iu,fmt6) 'INTEGER, DIMENSION(nvp__allprocs), PARAMETER'
WRITE(iu,fmtcon) '   :: pi__allprocs'
!WRITE(iu,fmtcon,ADVANCE='NO') '      = '
!
!CALL EXPAND_TOKEN(fmtconind, '@', '11X,', fmttmp)
!CALL WRITE_ARRAY_CONSTRUCTOR(iu, fmttmp, c_procid_allprocs, n_eqns_allprocs, CFG_ITPL)
CALL WRITE_CONEQ_ARRAY_CONSTRUCTOR(iu, fmtcon, c_procid_allprocs, n_eqns_allprocs, 6)
WRITE(iu,fmt_)
WRITE(iu,fmt_)
WRITE(iu,fmt_)
WRITE(iu,fmt0) '!======================================================================='
WRITE(iu,fmt6) 'END MODULE MOD_PROCESSDATA'
WRITE(iu,fmt0) '!======================================================================='


CLOSE(UNIT=d_unit)                  ! Done with the module MOD_PROCESSDATA: close it

CLOSE(UNIT=rpunit)                  ! Done with the template for medusa.rrp: close it

CLOSE(UNIT=c1unit)                  ! Done with mod_processcontrol_1.F (declaration
                                    ! part of mod_processcontrol.F): close it

CLOSE(UNIT=c2unit)                  ! Done with mod_processcontrol_2.F (reading and
                                    ! assigning part of mod_processcontrol.F): close it

CLOSE(UNIT=c3unit)                  ! Done with mod_processcontrol_2.F (variable
                                    ! assigning part of mod_processcontrol.F): close it

CLOSE(UNIT=reaunit)                 ! Done with rea-proc.F: close it

CLOSE(UNIT=dreunit)                 ! Done with drea-proc.F: close it


                                    ! Expand the code parts into the *base.F files.
CALL EXPAND_INCLUDES("base/mod_processcontrol-base.F", "tmp/mod_processcontrol.F", "tmp")
CALL EXPAND_INCLUDES("base/mod_rreac-base.F", "tmp/mod_rreac.F", "tmp")


#ifdef CFG_DEBUG
WRITE(jp_stddbg, c_dbgfmt) 'Starting code normalization'
#endif

                                    ! Normalize the source code
CALL NORMALIZE_SOURCECODE("tmp/mod_rreac.F",          "gen/mod_rreac.F")
CALL NORMALIZE_SOURCECODE("tmp/mod_processsubr.F",    "gen/mod_processsubr.F")
CALL NORMALIZE_SOURCECODE("tmp/mod_processdata.F",    "gen/mod_processdata.F")
CALL NORMALIZE_SOURCECODE("tmp/mod_processcontrol.F", "gen/mod_processcontrol.F")


WRITE(jp_stdout, c_fmtinf_a) 'completed'

#ifdef CFG_DEBUG
WRITE(jp_stddbg, c_dbgfmt) 'Done'
#endif

RETURN

!===================================================================================================
 END SUBROUTINE CREATE_MOD_PROCESSSUBR
!===================================================================================================
!---+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+----0----+----1----+----2----+----3--
