!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#include "configure.h"
!===================================================================================================
 SUBROUTINE CREATE_MOD_MATERIALCHARAS(t_compo_chain, t_solsys_chain)
!===================================================================================================

USE MOD_MEDUSA_COCOGEN
USE MOD_CONFIGURE_TYPES
USE MOD_CONFIGURE
USE MOD_UTILITIES, ONLY: EXPAND_TOKEN, WRITE_CONDENSED_DBLE, LOWCASE


IMPLICIT NONE



TYPE(COMPOINFO),   INTENT(IN), TARGET  :: t_compo_chain
TYPE(SYSTEMSINFO), INTENT(IN), TARGET  :: t_solsys_chain

TYPE(COMPOINFO),   POINTER :: t_compo_curr
TYPE(SYSTEMSINFO), POINTER :: t_solsys_curr

CHARACTER(LEN=n_lmaxformat) :: fmttmp
CHARACTER(LEN=3)   :: c_tmp

INTEGER, PARAMETER :: iout = CFG_C_UNIT
INTEGER, PARAMETER :: idcl = CFG_C1UNIT
INTEGER, PARAMETER :: iopr = CFG_C2UNIT

! 4 = LEN('psv_')
CHARACTER(LEN=n_lmaxshortid+4), DIMENSION(:), ALLOCATABLE :: c_psv

CHARACTER(LEN=n_lmaxphasid)  :: c_phasid
CHARACTER(LEN=n_lmaxcomptyp) :: c_class, c_class_master
CHARACTER(LEN=n_lmaxshortid) :: c_shortid, c_shortid_master
CHARACTER(LEN=n_lmaxexpress) :: c_vartype
CHARACTER(LEN=n_lmaxidentif) :: c_varname_c, c_varname_n, c_varname_p
CHARACTER(LEN=n_lmaxexpress) :: c_value_c,   c_value_n,   c_value_p
DOUBLE PRECISION             :: d_value_c,   d_value_n,   d_value_p
CHARACTER(LEN=n_lmaxidentif) :: c_varname_o, c_varname_h
CHARACTER(LEN=n_lmaxexpress) :: c_value_o,   c_value_h
DOUBLE PRECISION             :: d_value_h,   d_value_o
DOUBLE PRECISION             :: d_value
CHARACTER(LEN=n_lmaxexpress), DIMENSION(:),   ALLOCATABLE :: c_eq_mol

CHARACTER(LEN=n_lmaxidentif) :: c_param_name, c_param_varname
INTEGER :: n_checom, i_checom
INTEGER, PARAMETER :: nlen_prefix_checom = LEN_TRIM(cp_prefix_checom)
LOGICAL :: l_have_o, l_have_h, lvalidset_omparams

LOGICAL :: l_ismud

INTEGER :: n_params_found   = 0
INTEGER :: n_omparams_found = 0
INTEGER :: n_omcompo_found  = 0

INTEGER, DIMENSION(:), ALLOCATABLE :: jsubclass_omcompo
INTEGER, PARAMETER :: jp_cnp   = 1
INTEGER, PARAMETER :: jp_cnpoh = 2

INTEGER :: i_if, n, i, i_omcompo
INTEGER :: n_members, i_member, j, k
INTEGER :: nlen

CHARACTER(LEN=n_lmaxidentif), DIMENSION(:), ALLOCATABLE :: c_compo_stoech_var
CHARACTER(LEN=n_lmaxidentif), DIMENSION(:), ALLOCATABLE :: c_elmnt_molwgt_var
CHARACTER(LEN=n_lmaxcodeline):: c_codeline
CHARACTER(LEN=n_lmaxcodeline):: c_codeline1


! Standard I/O related data
! -------------------------

INTEGER, PARAMETER :: jp_stdout = CFG_STDOUT
INTEGER, PARAMETER :: jp_stderr = CFG_STDERR
CHARACTER(LEN=*), PARAMETER  :: c_fmtinf_a   = '("[CREATE_MOD_MATERIALCHARAS]: ", A)'
CHARACTER(LEN=*), PARAMETER  :: c_fmtwar_a   = '("[CREATE_MOD_MATERIALCHARAS] warning: ", A)'
CHARACTER(LEN=*), PARAMETER  :: c_fmterr_a   = '("[CREATE_MOD_MATERIALCHARAS] error: ", A)'

#ifdef CFG_DEBUG
INTEGER, PARAMETER :: jp_stddbg = CFG_STDDBG
CHARACTER(LEN=*), PARAMETER  :: c_fmtdbg_a   = '("DEBUG [CREATE_MOD_MATERIALCHARAS]: ", A)'
#endif



!=====================!
! End of declarations !
!=====================!

WRITE(jp_stdout,'()')
WRITE(jp_stdout,c_fmtinf_a) 'starting'

#ifdef CFG_DEBUG
WRITE(jp_stddbg,'()')
WRITE(jp_stddbg,c_fmtdbg_a) 'starting'
#endif

!============!
! Operations !
!============!

NULLIFY(t_compo_curr)

IF (n_modelsolid > 0) THEN
  ALLOCATE(c_psv(n_modelsolid))
ELSE
  WRITE(jp_stdout,'()')
  WRITE(jp_stdout,c_fmtinf_a) 'no solids registered -- returning'
  RETURN
ENDIF


OPEN(UNIT=iout, FILE = "tmp/mod_materialcharas-base.F")
WRITE(iout,fmt0) '! This module has been automatically generated by CREATE_MATERIALCHARAS'
WRITE(iout,fmt0) '! from the MEDUSA configuration and source code generation utility.'
WRITE(iout,fmt0) '!---+----1----+----2----+----3----+----4----+----5----+----6----+----7--'
WRITE(iout,fmt0) '!======================================================================='
WRITE(iout,fmt6) 'MODULE MOD_MATERIALCHARAS'
WRITE(iout,fmt0) '!======================================================================='
WRITE(iout,fmt_)
WRITE(iout,fmt6) 'USE mod_basicdata_medusa'
WRITE(iout,fmt6) 'USE mod_indexparam'
WRITE(iout,fmt_)
WRITE(iout,fmt6) 'IMPLICIT NONE'
WRITE(iout,fmt_)
WRITE(iout,fmt_)
WRITE(iout,fmt6) '! Specific mass and specific volumes of solid components.'
WRITE(iout,fmt6) '! Only for local usage here (PRIVATE attribute) for setting up'
WRITE(iout,fmt6) '! the apsv array.'
WRITE(iout,fmt6) '! - rho_xxxx: densities of solids [kg/m3]'
WRITE(iout,fmt6) '! - psv_xxxx: partial specific (massic) volumes of solids [m3/kg]'
WRITE(iout,fmt_)


! =================
! Solids' densities
! =================

! Scan t_compo_chain
! - to extract density values (each solid *must* have a parameter with
!   %param_name=cp_phyprodensit, where cp_phyprodensit='Density' typically);
! - prepare the information for setting up the apsv(:) array with partial
!   specific volumes

t_compo_curr => t_compo_chain
i_if = 0

DO WHILE(ASSOCIATED(t_compo_curr))

  c_phasid  = t_compo_curr%phasid
  c_class   = t_compo_curr%class
  c_shortid = t_compo_curr%shortid
  n         = t_compo_curr%n_params

  n_params_found = 0
  l_ismud = TRIM(c_shortid) == TRIM(c_shortid_mud)


  IF (c_phasid == 'if') THEN

    SELECT CASE(c_class)
    CASE(cp_classsolid, cp_classorgmcnp)

      scan_params4density: DO i = 1, n

        IF(t_compo_curr%param_name(i) == cp_phyprodensit) THEN

          WRITE(iout,fmt6) &
            TRIM(t_compo_curr%param_vartype(i)) // ', PARAMETER , PRIVATE'
          WRITE(iout,fmtcon) '   :: ' // &
            TRIM(t_compo_curr%param_varname(i)) // ' = ' // &
            TRIM(t_compo_curr%param_values(i))
          WRITE(iout,fmt6) &
            TRIM(t_compo_curr%param_vartype(i)) // ', PARAMETER , PRIVATE'
          IF (.NOT. l_ismud) THEN
            WRITE(iout,fmt0) '#ifdef SOLIDS_VOLUMELESS'
            WRITE(iout,fmtcon) '   :: ' // &
              'psv_' // TRIM(c_shortid) // ' = 0.0D+00'
            WRITE(iout,fmt0) '#else'
          ENDIF
          WRITE(iout,fmtcon) '   :: ' // &
            'psv_' // TRIM(c_shortid) // ' = ' // &
            '1.0D+00/' // TRIM(t_compo_curr%param_varname(i))
          IF (.NOT. l_ismud) THEN
            WRITE(iout,fmt0) '#endif'
          ENDIF

          WRITE(iout,fmt_)

          n_params_found = 1

          EXIT scan_params4density

        ENDIF

      ENDDO scan_params4density

      IF (n_params_found == 0) THEN
        WRITE(jp_stderr, c_fmterr_a) 'no ''' // cp_phyprodensit // ''' parameter found &
          &for component ''' // TRIM(t_compo_curr%name) // ''' -- aborting'
        CALL ABORT()
      ENDIF


    CASE(cp_classsolidcolour)
      WRITE(iout,fmt6) &
        'DOUBLE PRECISION, PARAMETER , PRIVATE'
      WRITE(iout,fmtcon) '   :: ' // &
        'psv_' // TRIM(c_shortid) // ' = 0.0D+00'
      WRITE(iout,fmt_)


    CASE(cp_classsolidpt)
      WRITE(iout,fmt6) &
        'DOUBLE PRECISION, PARAMETER , PRIVATE'
      WRITE(iout,fmtcon) '   :: ' // &
        'psv_' // TRIM(c_shortid) // ' = 0.0D+00'
      WRITE(iout,fmt_)


    CASE DEFAULT
      WRITE(jp_stderr,c_fmtwar_a) 'No rules defined for solids with class="' // &
        TRIM(c_class) // '" regarding density and partial specific volume!'
      WRITE(jp_stderr,'(A)') 'Provisionally setting "psv_' // TRIM(c_shortid) // '" to 0.'

      WRITE(iout,fmt6) &
        'DOUBLE PRECISION, PARAMETER , PRIVATE'
      WRITE(iout,fmtcon) '   :: ' // &
        'psv_' // TRIM(c_shortid) // ' = 0.0D+00'
      WRITE(iout,fmt_)

    END SELECT

    i_if = i_if+1
    c_psv(i_if) = 'psv_' // c_shortid

  ENDIF

  t_compo_curr => t_compo_curr%next

ENDDO


! set up the apsv(:) array with partial specific volumes

WRITE(iout,fmt_)
WRITE(iout,fmt_)
WRITE(iout,fmt6) '! apsv(1:nsolid): partial specific (massic) volumes of solids [m3/kg]'
WRITE(iout,fmt_)
WRITE(iout,fmt6) 'DOUBLE PRECISION, PARAMETER, DIMENSION(nsolid) :: apsv'
WRITE(iout,fmtcon,ADVANCE='NO') '      = '
CALL EXPAND_TOKEN(fmtconind, '@', '11X,', fmttmp)
CALL WRITE_ARRAY_CONSTRUCTOR(iout, fmttmp, c_psv, n_modelsolid, CFG_ITPL)
WRITE(iout,fmt_)
WRITE(iout,fmt6) '! Scaling factor for apsv'
WRITE(iout,fmt_)
WRITE(iout,fmt6) 'DOUBLE PRECISION, DIMENSION(nsolid), SAVE'
WRITE(iout,fmtcon) '  :: apsv_factor     = 1.0D+00'
WRITE(iout,fmt_)
WRITE(iout,fmt_)



! =======================================
! Chemical compositions of the components
! =======================================

IF (n_modelomcompo > 0) THEN
  ALLOCATE(jsubclass_omcompo(n_modelomcompo))
ENDIF


WRITE(iout,fmt_)
WRITE(iout,fmt_)
WRITE(iout,fmt6) '! Chemical composition data'
WRITE(iout,fmt6) '! -------------------------'

t_compo_curr => t_compo_chain
n_omcompo_found = 0

DO WHILE(ASSOCIATED(t_compo_curr))

  c_phasid  = t_compo_curr%phasid
  c_class   = t_compo_curr%class
  c_shortid = t_compo_curr%shortid
  n         = t_compo_curr%n_params
  n_checom  = t_compo_curr%n_checom

  IF (t_compo_curr%idx_xref /= 0) THEN
    c_class_master = t_compo_curr%xref%class
  ELSE
    c_class_master = c_class
  ENDIF


  IF (n_checom > 0) THEN

    SELECT CASE(c_class_master)

    CASE(cp_classsolute)

      CONTINUE


    CASE(cp_classsolid)

      WRITE(iout,fmt_)
      WRITE(iout,fmt_)
      IF (c_class == c_class_master) THEN
        WRITE(iout,fmt6) '! Stoechiometric data for "' // TRIM(t_compo_curr%name) //'" &
          &(class="' // TRIM(c_class) // '")'
      ELSE
        WRITE(iout,fmt6) '! Stoechiometric data for "' // TRIM(t_compo_curr%name) //'"'
        WRITE(iout,fmt6) '! (class="' // TRIM(c_class) // '",&
                           & master="' // cp_classsolid // '")'
      ENDIF
      WRITE(iout,fmt_)

      DO i = 1, n
        IF (INDEX(t_compo_curr%param_name(i), cp_prefix_checom) == 1) THEN
          READ(t_compo_curr%param_values(i),*) d_value
          WRITE(iout,fmt6,ADVANCE='NO') &
            TRIM(t_compo_curr%param_vartype(i)) // ', SAVE :: ' // &
            TRIM(t_compo_curr%param_varname(i)) // ' = '
          CALL WRITE_CONDENSED_DBLE(iout, d_value, l_linefeed=.TRUE.)
        ENDIF
      ENDDO


    CASE(cp_classorgmcnp)
      IF (c_class == c_class_master) THEN
        n_omcompo_found = n_omcompo_found + 1
#       ifdef CFG_DEBUG
        WRITE(jp_stddbg, c_fmtdbg_a) 'compo with %shortid="' // TRIM(c_shortid) // &
          '" has %class="' // cp_classorgmcnp // '"'
      ELSE
        WRITE(jp_stddbg, c_fmtdbg_a) 'compo with %shortid="' // TRIM(c_shortid) // &
          '" has %class="' // TRIM(c_class) // '"'
        WRITE(jp_stddbg, '(A)') 'and its master compo has %class="' // cp_classorgmcnp // '"'
#       endif
      ENDIF

      WRITE(iout,fmt_)
      WRITE(iout,fmt_)
      IF (c_class == c_class_master) THEN
        WRITE(iout,fmt6) '! Stoechiometric data for "' // TRIM(t_compo_curr%name) //'" ' // &
                       '(class="' // cp_classorgmcnp // '")'
      ELSE
        WRITE(iout,fmt6) '! Stoechiometric data for "' // TRIM(t_compo_curr%name) //'"'
        WRITE(iout,fmt6) '! (class="' // TRIM(c_class) // '",&
                           & master="' // cp_classorgmcnp // '")'
      ENDIF
      WRITE(iout,fmt_)


      n_omparams_found = 0
      lvalidset_omparams = .FALSE.

      WRITE(c_value_o,'()')  ! provisionally empty c_value_o
      WRITE(c_value_h,'()')  ! provisionally empty c_value_h


      DO i = 1, n

        IF (INDEX(t_compo_curr%param_name(i), cp_prefix_checom) /= 1) CYCLE

        SELECT CASE(t_compo_curr%param_name(i))
        CASE(cp_cnp_paramname_c)
          c_value_c   = t_compo_curr%param_values(i)
          READ(c_value_c,*) d_value_c
          c_varname_c = t_compo_curr%param_varname(i)
          n_omparams_found = IBSET(n_omparams_found, 0)
        CASE(cp_cnp_paramname_n)
          c_value_n = t_compo_curr%param_values(i)
          READ(c_value_n,*) d_value_n
          c_varname_n = t_compo_curr%param_varname(i)
          n_omparams_found = IBSET(n_omparams_found, 1)
        CASE(cp_cnp_paramname_p)
          c_value_p   = t_compo_curr%param_values(i)
          READ(c_value_p,*) d_value_p
          c_varname_p = t_compo_curr%param_varname(i)
          n_omparams_found = IBSET(n_omparams_found, 2)
        CASE(cp_cnp_paramname_o)
          c_value_o   = t_compo_curr%param_values(i)
          READ(c_value_o,*) d_value_o
          c_varname_o = t_compo_curr%param_varname(i)
          n_omparams_found = IBSET(n_omparams_found, 3)
        CASE(cp_cnp_paramname_h)
          c_value_h   = t_compo_curr%param_values(i)
          READ(c_value_h,*) d_value_h
          c_varname_h = t_compo_curr%param_varname(i)
          n_omparams_found = IBSET(n_omparams_found, 4)
        CASE DEFAULT
          READ(t_compo_curr%param_values(i),*) d_value
          WRITE(iout,fmt6) TRIM(t_compo_curr%param_vartype(i)) // ', SAVE :: ' // &
                           TRIM(t_compo_curr%param_varname(i)) // ' = '
          CALL WRITE_CONDENSED_DBLE(iout, d_value, l_linefeed=.TRUE.)
        END SELECT
      ENDDO

      IF (IAND(n_omparams_found, 31) == 31) THEN
                                   ! All 5 params (C, N, P, O, H) are given
        lvalidset_omparams = .TRUE.
        IF (c_class == c_class_master) jsubclass_omcompo(n_omcompo_found) = jp_cnpoh

        WRITE(iout,fmt6) '! Composition: C_cO_oH_hN_nP_p'
        WRITE(iout,fmt6) '!  <c> is provided by ' // TRIM(c_shortid) // '_c'
        WRITE(iout,fmt6) '!  <n> is provided by ' // TRIM(c_shortid) // '_n'
        WRITE(iout,fmt6) '!  <p> is provided by ' // TRIM(c_shortid) // '_p'
        WRITE(iout,fmt6) '!  <o> is provided by ' // TRIM(c_shortid) // '_o'
        WRITE(iout,fmt6) '!  <h> is provided by ' // TRIM(c_shortid) // '_h'
        WRITE(iout,fmt6) '! Derived parameter:'
        WRITE(iout,fmt6) '!  <remin_o2> prd. by ' // TRIM(c_shortid) // '_remin_o2'
        WRITE(iout,fmt_)

        WRITE(iout,fmt6, ADVANCE='NO') &
          cp_ftntype_dbleprec // ', SAVE :: ' // TRIM(c_varname_c) // ' = '
        CALL WRITE_CONDENSED_DBLE(iout, d_value_c, l_linefeed=.TRUE.)

        WRITE(iout,fmt6, ADVANCE='NO') &
          cp_ftntype_dbleprec // ', SAVE :: ' // TRIM(c_varname_n) // ' = '
        CALL WRITE_CONDENSED_DBLE(iout, d_value_n, l_linefeed=.TRUE.)

        WRITE(iout,fmt6, ADVANCE='NO') &
          cp_ftntype_dbleprec // ', SAVE :: ' // TRIM(c_varname_p) // ' = '
        CALL WRITE_CONDENSED_DBLE(iout, d_value_p, l_linefeed=.TRUE.)

        WRITE(iout,fmt6, ADVANCE='NO') &
          cp_ftntype_dbleprec // ', SAVE :: ' // TRIM(c_varname_o) // ' = '
        CALL WRITE_CONDENSED_DBLE(iout, d_value_o, l_linefeed=.TRUE.)

        WRITE(iout,fmt6, ADVANCE='NO') &
          cp_ftntype_dbleprec // ', SAVE :: ' // TRIM(c_varname_h) // ' = '
        CALL WRITE_CONDENSED_DBLE(iout, d_value_h, l_linefeed=.TRUE.)

        WRITE(iout,fmt_)


        WRITE(iout,fmt6, ADVANCE='NO') &
          cp_ftntype_dbleprec // ', SAVE :: ' // TRIM(c_shortid) // '_remin_o2 = '
        d_value = d_value_c + 0.25D0*d_value_h - 0.5D0*d_value_o + 1.25D0*(d_value_n + d_value_p)
        CALL WRITE_CONDENSED_DBLE(iout, d_value, l_linefeed=.TRUE.)

      ELSEIF (IAND(n_omparams_found, 7) == 7) THEN
                                   ! Not 5, but at least 3 params (C, N, P) are given

        IF (IAND(n_omparams_found, 8) == 8) THEN
                                   ! O given as well
          WRITE(jp_stderr, c_fmtwar_a) &
            'parameter <' // cp_cnp_paramname_o // '> &
            &cannot be given without <'    // cp_cnp_paramname_h // '> &
            &for component "' // TRIM(t_compo_curr%name) // '"'
          WRITE(jp_stderr, '(A)') 'and will be ignored!'
          n_omparams_found = IBCLR(n_omparams_found, 3)


        ELSEIF (IAND(n_omparams_found, 16) == 16) THEN
                                   ! H given as well
          WRITE(jp_stderr, c_fmtwar_a) &
            'parameter <' // cp_cnp_paramname_h // '> &
            &cannot be given without <'    // cp_cnp_paramname_o // '> &
            &for component "' // TRIM(t_compo_curr%name) // '"'
          WRITE(jp_stderr, '(A)') 'and will be ignored!'
          n_omparams_found = IBCLR(n_omparams_found, 4)

        ENDIF

        lvalidset_omparams = .TRUE.
        IF (c_class == c_class_master) jsubclass_omcompo(n_omcompo_found) = jp_cnp

        WRITE(iout,fmt6) '! Composition: (CH_2O)_c(NH_3)_n(H_3PO_4)_p'
        WRITE(iout,fmt6) '!  <c> is provided by ' // TRIM(c_shortid) // '_c'
        WRITE(iout,fmt6) '!  <n> is provided by ' // TRIM(c_shortid) // '_n'
        WRITE(iout,fmt6) '!  <p> is provided by ' // TRIM(c_shortid) // '_p'
        WRITE(iout,fmt6) '! Derived parameters:'
        WRITE(iout,fmt6) '!  <o> is provided by ' // TRIM(c_shortid) // '_o'
        WRITE(iout,fmt6) '!  <h> is provided by ' // TRIM(c_shortid) // '_h'
        WRITE(iout,fmt6) '!  <remin_o2> prd. by ' // TRIM(c_shortid) // '_remin_o2'
        WRITE(iout,fmt_)

        WRITE(iout,fmt6, ADVANCE='NO') &
          cp_ftntype_dbleprec // ', SAVE :: ' // TRIM(c_varname_c) // ' = '
        CALL WRITE_CONDENSED_DBLE(iout, d_value_c, l_linefeed=.TRUE.)

        WRITE(iout,fmt6, ADVANCE='NO') &
          cp_ftntype_dbleprec // ', SAVE :: ' // TRIM(c_varname_n) // ' = '
        CALL WRITE_CONDENSED_DBLE(iout, d_value_n, l_linefeed=.TRUE.)

        WRITE(iout,fmt6, ADVANCE='NO') &
          cp_ftntype_dbleprec // ', SAVE :: ' // TRIM(c_varname_p) // ' = '
        CALL WRITE_CONDENSED_DBLE(iout, d_value_p, l_linefeed=.TRUE.)

        WRITE(iout,fmt_)


        WRITE(iout,fmt6, ADVANCE='NO') cp_ftntype_dbleprec // ', SAVE :: ' // TRIM(c_shortid) // '_o = '
        d_value_o = d_value_c + 4D0*d_value_p
        CALL WRITE_CONDENSED_DBLE(iout, d_value_o, l_linefeed=.TRUE.)

        WRITE(iout,fmt6, ADVANCE='NO') cp_ftntype_dbleprec // ', SAVE :: ' // TRIM(c_shortid) // '_h = '
        d_value_h = 2D0*d_value_c + 3D0*(d_value_n + d_value_p)
        CALL WRITE_CONDENSED_DBLE(iout, d_value_h, l_linefeed=.TRUE.)

        WRITE(iout,fmt6, ADVANCE='NO') cp_ftntype_dbleprec // ', SAVE :: ' // &
                       TRIM(c_shortid) // '_remin_o2 = '
        d_value = d_value_c + 2D0*d_value_n
        CALL WRITE_CONDENSED_DBLE(iout, d_value, l_linefeed=.TRUE.)


      ELSE

                                   ! No usable set has been found
                                   ! One or more parameters are missing!
        lvalidset_omparams = .FALSE.

        IF (IAND(n_omparams_found, 1) == 0) THEN
          WRITE(jp_stderr, c_fmterr_a) &
            'parameter <' // cp_cnp_paramname_c // '> &
            &is missing for component "' // TRIM(t_compo_curr%name) // '"'
        ENDIF
        IF (IAND(n_omparams_found, 2) == 0) THEN
          WRITE(jp_stderr, c_fmterr_a) &
            'parameter <' // cp_cnp_paramname_n // '> &
            &is missing for component "' // TRIM(t_compo_curr%name) // '"'
        ENDIF
        IF (IAND(n_omparams_found, 4) == 0) THEN
          WRITE(jp_stderr, c_fmterr_a) &
            'parameter <' // cp_cnp_paramname_p // '> &
            &is missing for component "' // TRIM(t_compo_curr%name) // '"'
        ENDIF
        IF (IAND(n_omparams_found, 8) == 0) THEN
          WRITE(jp_stderr, c_fmterr_a) &
            'parameter <' // cp_cnp_paramname_o // '> &
            &is missing for component "' // TRIM(t_compo_curr%name) // '"'
        ENDIF
        IF (IAND(n_omparams_found, 16) == 0) THEN
          WRITE(jp_stderr, c_fmterr_a) &
            'parameter <' // cp_cnp_paramname_h // '> &
            &is missing for component "' // TRIM(t_compo_curr%name) // '"'
        ENDIF

      ENDIF

      IF(.NOT. lvalidset_omparams) THEN
        WRITE(jp_stderr, c_fmterr_a) 'Aborting!'
        CALL ABORT()
      ENDIF


    CASE DEFAULT

      WRITE(jp_stderr,c_fmtwar_a) 'No rules defined for solids or solutes with class="' // &
        TRIM(c_class) // '"'
      WRITE(jp_stderr,c_fmtwar_a) ' -- generated code may be incomplete!'

    END SELECT

  ENDIF

  t_compo_curr => t_compo_curr%next

ENDDO


! Now for the molar masses of the solids
WRITE(iout,fmt_)
WRITE(iout,fmt_)
WRITE(iout,fmt6) '! mol_xxxx: molar masses of solids and selected solutes [kg/mol]'
WRITE(iout,fmt6) '! For strict mass conservation, it may be necessary to adjust'
WRITE(iout,fmt6) '! the mol_xxxx values. This can be done with the subroutine'
WRITE(iout,fmt6) '! SOLIDS_STOECHIOMETRY (in this module). The values provided'
WRITE(iout,fmt6) '! here can still be used as fallback values.'

t_compo_curr => t_compo_chain

DO WHILE(ASSOCIATED(t_compo_curr))

  SELECT CASE(t_compo_curr%class)
  CASE(cp_classsolid, cp_classorgmcnp, cp_classsolidcolour)

    n = t_compo_curr%n_params

    scan_params4molwgt: DO i = 1, n

      IF (t_compo_curr%param_name(i) == cp_phypromolwgt) THEN
        WRITE(iout,fmt_)
        READ(t_compo_curr%param_values(i),*) d_value
        WRITE(iout,fmt6, ADVANCE='NO') &
          TRIM(t_compo_curr%param_vartype(i)) // ', SAVE :: ' // &
          TRIM(t_compo_curr%param_varname(i)) // ' = '
        CALL WRITE_CONDENSED_DBLE(iout, d_value, l_linefeed=.TRUE.)

        EXIT scan_params4molwgt
      ENDIF

    ENDDO scan_params4molwgt

    IF (i > n) THEN  ! if he previous loop went until the end without success
      c_shortid = t_compo_curr%shortid
      WRITE(iout,fmt_)
      WRITE(iout,fmt6) cp_ftntype_dbleprec // ', SAVE :: ' // &
        'mol_' // TRIM(c_shortid) // ' = 0.0D+00  ! to be initialised'
    ENDIF

  CASE DEFAULT

    CONTINUE                        ! Amend here if molar masses for other
                                    ! components need to be considered.
                                    ! Currently only "BasicSolid",
                                    ! "OrgMatter_CNP" and "SolidColour"
                                    ! class components require molar masses.

  END SELECT

  t_compo_curr => t_compo_curr%next

ENDDO

WRITE(iout,fmt_)
WRITE(iout,fmt_)
WRITE(iout,fmt6) '! Convenience array with molar masses of all the solids'
WRITE(iout,fmt_)
WRITE(iout,fmt6)   'DOUBLE PRECISION, DIMENSION(nsolid), SAVE :: amol'


j = 1
t_solsys_curr => t_solsys_chain
DO WHILE(ASSOCIATED(t_solsys_curr))

  IF (t_solsys_curr%name == cp_alkalinity) THEN
    n_members = t_solsys_curr%n_members
    ALLOCATE(c_eq_mol(n_members))
    DO i_member = 1, n_members
      t_compo_curr => COMPOINFO_getNodeByName(t_compo_chain, t_solsys_curr%member_name(i_member))
      DO k = 1, t_compo_curr%n_params
        IF (t_compo_curr%param_name(k) /= cp_alkalinity) THEN
          CYCLE
        ELSE
          READ(t_compo_curr%param_values(k),*) d_value
          CALL WRITE_CONDENSED_DBLE(c_eq_mol(i_member), d_value)
          EXIT
        ENDIF
      ENDDO

    ENDDO

    WRITE(c_tmp, '(I0)') t_solsys_curr%idx
    WRITE(iout,fmt_)
    WRITE(iout,fmt_)
    WRITE(iout,fmt_)
    WRITE(iout,fmt6)   '! ' // cp_alkalinity // ' weights (Solute System ' // TRIM(c_tmp) // ')'
    WRITE(iout,fmt_)
    WRITE(iout,fmt_)
    WRITE(iout,fmt6)   'DOUBLE PRECISION, PARAMETER, &
      &DIMENSION(' // TRIM(t_solsys_curr%nvc_identifier) // &
      ')  :: ' //  'eq_mol_' // cp_shortid_alk
    WRITE(iout,fmtcon, ADVANCE='NO') '      = '

    CALL WRITE_ARRAY_CONSTRUCTOR(iout, fmttmp, c_eq_mol, n_members, CFG_ITPL)

    DEALLOCATE(c_eq_mol)

  ENDIF

  j = j + 1
  t_solsys_curr => t_solsys_curr%next
ENDDO


WRITE(iout,fmt_)
WRITE(iout,fmt_)
WRITE(iout,fmt6) 'CONTAINS'
WRITE(iout,fmt_)
WRITE(iout,fmt0) '!-----------------------------------------------------------------------'
WRITE(iout,fmt6) 'SUBROUTINE SOLIDS_STOECHIOMETRY'
WRITE(iout,fmt0) '!-----------------------------------------------------------------------'
WRITE(iout,fmt_)
WRITE(iout,fmt6) 'IMPLICIT NONE'
WRITE(iout,fmt_)
WRITE(iout,fmt_)
WRITE(iout,fmt0) '#include "mod_materialcharas_1.F"'
WRITE(iout,fmt_)
WRITE(iout,fmt_)
WRITE(iout,fmt0) '#include "mod_materialcharas_2.F"'
WRITE(iout,fmt_)
WRITE(iout,fmt0) '!-----------------------------------------------------------------------'
WRITE(iout,fmt6) 'END SUBROUTINE SOLIDS_STOECHIOMETRY'
WRITE(iout,fmt0) '!-----------------------------------------------------------------------'
WRITE(iout,fmt_)
WRITE(iout,fmt0) '!======================================================================='
WRITE(iout,fmt6) 'END MODULE MOD_MATERIALCHARAS'
WRITE(iout,fmt0) '!======================================================================='
WRITE(iout,fmt0) '!---+----1----+----2----+----3----+----4----+----5----+----6----+----7--'

CLOSE(UNIT=iout)



OPEN(UNIT=idcl, FILE = "tmp/mod_materialcharas_1.F")

OPEN(UNIT=iopr, FILE = "tmp/mod_materialcharas_2.F")


                                    ! Set up the code for the <compo>_stoech(:)
                                    ! array and for calculating mol_<compo>
i_omcompo = 0

t_compo_curr => t_compo_chain

DO WHILE(ASSOCIATED(t_compo_curr))

  c_phasid  = t_compo_curr%phasid
  c_class   = t_compo_curr%class
  c_shortid = t_compo_curr%shortid
  n         = t_compo_curr%n_params
  n_checom  = t_compo_curr%n_checom

  i_checom = 0


  IF (n_checom > 0) THEN

    SELECT CASE(c_class)

    CASE(cp_classsolute)

      CONTINUE


    CASE(cp_classsolid)

      ALLOCATE(c_compo_stoech_var(n_checom))
      ALLOCATE(c_elmnt_molwgt_var(n_checom))

      DO i = 1, n
        c_param_name = t_compo_curr%param_name(i)
        IF(INDEX(c_param_name, cp_prefix_checom) == 1) THEN
          i_checom = i_checom + 1
          c_compo_stoech_var(i_checom) = TRIM(t_compo_curr%param_varname(i))
          c_elmnt_molwgt_var(i_checom) = cp_prefix_datamolwgt // &
                                         TRIM(LOWCASE(c_param_name(nlen_prefix_checom+1:)))
        ENDIF
      ENDDO


    CASE(cp_classorgmcnp)

      i_omcompo = i_omcompo + 1

      SELECT CASE(jsubclass_omcompo(i_omcompo))
      CASE(jp_cnp)
                                    ! Code to adjust the OM_o, OM_h and OM_remin_o2 for
                                    ! components of class "OrgMatter_CNP" without O and H data
                                    ! (make them all consistent).
        WRITE(iopr,fmt_)
        WRITE(iopr,fmt_)

        WRITE(iopr,fmtmlc) 'Adjust ' // TRIM(c_shortid) // '_h, ' &
                                     // TRIM(c_shortid) // '_o and ' &
                                     // TRIM(c_shortid) // '_remin_o2'
        WRITE(iopr,fmtmlc) 'for internal consistency.'
        WRITE(iopr,fmt6) TRIM(c_shortid) // '_o = '          // TRIM(c_shortid) // '_c + ' // &
                                                 '4.0D+00*'  // TRIM(c_shortid) // '_p'

        WRITE(iopr,fmt6) TRIM(c_shortid) // '_h = '          // TRIM(c_shortid) // '_c + ' // &
                                                                TRIM(c_shortid) // '_c + ' // &
                                                 '3.0D+00*(' // TRIM(c_shortid) // '_n + ' // &
                                                                TRIM(c_shortid) // '_p)'
        WRITE(iopr,fmt6) TRIM(c_shortid) // '_remin_o2 = '   // TRIM(c_shortid) // '_c + ' // &
                                                                TRIM(c_shortid) // '_n + ' // &
                                                                TRIM(c_shortid) // '_n'
      CASE(jp_cnpoh)
                                    ! Code to adjust the OM_remin_o2 for components
                                    ! of class "OrgMatter_CNP" with O and H data
                                    ! (make it consistent):
                                    ! OM_remin_o2 = OM_c + OM_h/4D0 - OM_o/2D0
                                    !               + 5D0*(OM_n+OM_p)/4D0
        WRITE(iopr,fmt_)
        WRITE(iopr,fmt_)
        WRITE(iopr,fmtmlc) 'Adjust ' // TRIM(c_shortid) // '_remin_o2'
        WRITE(iopr,fmtmlc) 'for internal consistency.'

        WRITE(iopr,fmt6) TRIM(c_shortid) // '_remin_o2 = '
        WRITE(iopr,fmtcon) '    ' // TRIM(c_shortid) // '_c' // &
                   ' + 0.25D+00*' // TRIM(c_shortid) // '_h' // &
                   ' - 0.50D+00*' // TRIM(c_shortid) // '_o'
        WRITE(iopr,fmtcon) '  + 1.25D+00*(' // TRIM(c_shortid) // '_n + ' // TRIM(c_shortid) // '_p)'

      END SELECT


                                    ! Then set up the OM_stoech array. This is more general than
                                    ! the formulation above, as it might include
                                    ! other constituents (S, etc.) as well that need to be taken
                                    ! into account in the calculation of the molar mass.
                                    ! O and H are treated separately, as the order
                                    ! by which the parameters appear in the uXML
                                    ! file are respected. O and H may be present or
                                    ! absent.
      l_have_o = .FALSE.
      l_have_h = .FALSE.
      ALLOCATE(c_compo_stoech_var(n_checom+2))
      ALLOCATE(c_elmnt_molwgt_var(n_checom+2))

      DO i = 1, n

        c_param_name = t_compo_curr%param_name(i)

                                    ! Check if <c_param_name> starts
                                    ! with "ChemicalComposition_"
        IF(INDEX(c_param_name, cp_prefix_checom) == 1) THEN
          i_checom = i_checom + 1
          c_compo_stoech_var(i_checom) = TRIM(t_compo_curr%param_varname(i))
          c_elmnt_molwgt_var(i_checom) = cp_prefix_datamolwgt // &
                                         TRIM(LOWCASE(c_param_name(nlen_prefix_checom+1:)))
          IF(c_param_name == cp_cnp_paramname_o) l_have_o = .TRUE.
          IF(c_param_name == cp_cnp_paramname_h) l_have_h = .TRUE.
        ENDIF

      ENDDO

      IF(.NOT. l_have_o) THEN
        i_checom = i_checom + 1
        c_compo_stoech_var(i_checom) = TRIM(c_shortid) // '_o'
        c_elmnt_molwgt_var(i_checom) = cp_prefix_datamolwgt // 'o'
      ENDIF

      IF(.NOT. l_have_h) THEN
        i_checom = i_checom + 1
        c_compo_stoech_var(i_checom) = TRIM(c_shortid) // '_h'
        c_elmnt_molwgt_var(i_checom) = cp_prefix_datamolwgt // 'h'
      ENDIF


    CASE (cp_classsolidcolour)

      c_class_master   = t_compo_curr%xref%class
      c_shortid_master = t_compo_curr%xref%shortid
      n                = t_compo_curr%n_params
      nlen = LEN_TRIM(t_compo_curr%shortid)

      SELECT CASE(c_class_master)
      CASE(cp_classsolid)

        WRITE(iopr,fmt_)
        WRITE(iopr,fmt_)
        DO i = 1, n
          IF (INDEX(t_compo_curr%param_name(i), cp_prefix_checom) == 1) THEN
            c_param_varname = t_compo_curr%param_varname(i)
            WRITE(iopr,fmt6) TRIM(t_compo_curr%param_varname(i)) // ' = ' // &
                             TRIM(c_shortid_master) // c_param_varname(nlen+1:)
          ENDIF
        ENDDO

        WRITE(iopr,fmt_)
        WRITE(iopr,fmt6) 'mol_' // TRIM(c_shortid) // ' = mol_' // TRIM(c_shortid_master)


      CASE(cp_classorgmcnp)

        WRITE(iopr,fmt_)
        WRITE(iopr,fmt_)
        WRITE(iopr,fmt6) TRIM(c_shortid) // '_c = ' // TRIM(c_shortid_master) // '_c'
        WRITE(iopr,fmt6) TRIM(c_shortid) // '_n = ' // TRIM(c_shortid_master) // '_n'
        WRITE(iopr,fmt6) TRIM(c_shortid) // '_p = ' // TRIM(c_shortid_master) // '_p'
        WRITE(iopr,fmt6) TRIM(c_shortid) // '_o = ' // TRIM(c_shortid_master) // '_o'
        WRITE(iopr,fmt6) TRIM(c_shortid) // '_h = ' // TRIM(c_shortid_master) // '_h'
        WRITE(iopr,fmt_)
        WRITE(iopr,fmt6) TRIM(c_shortid) // '_remin_o2 = ' // TRIM(c_shortid_master) // '_remin_o2'
        WRITE(iopr,fmt_)
        WRITE(iopr,fmt6) 'mol_' // TRIM(c_shortid) // ' = mol_' // TRIM(c_shortid_master)


      END SELECT

    CASE DEFAULT

      CONTINUE                      ! Amend here if molar masses for other
                                    ! components need to be considered.
                                    ! Currently only "BasicSolid" and
                                    ! "OrgMatter_CNP" class components require
                                    ! molar masses.


    END SELECT


    IF(i_checom > 0) THEN

      WRITE(idcl,fmt_)
      WRITE(idcl,fmt6,ADVANCE='NO') cp_ftntype_dbleprec
      WRITE(idcl,'(", DIMENSION(", I0, ") :: ", A)') &
        i_checom, TRIM(c_shortid) // '_stoech'

      WRITE(idcl,fmt6,ADVANCE='NO') cp_ftntype_dbleprec
      WRITE(idcl,'(", DIMENSION(", I0, "), PARAMETER :: ", A)') &
        i_checom, TRIM(c_shortid) // '_eltmolwgt = '
      WRITE(idcl,fmtcon, ADVANCE='NO') '  '
      CALL EXPAND_TOKEN(fmtconind, '@', '11X,', fmttmp)
      CALL WRITE_ARRAY_CONSTRUCTOR(idcl, fmttmp, c_elmnt_molwgt_var, i_checom, CFG_ITPL)



      WRITE(iopr,fmt_)
      WRITE(iopr,fmt_)
      WRITE(iopr,fmt6,ADVANCE='NO') TRIM(c_shortid) // '_stoech(:) = '
      CALL EXPAND_TOKEN(fmtconind, '@', '11X,', fmttmp)
      CALL WRITE_ARRAY_CONSTRUCTOR(iopr, fmttmp, c_compo_stoech_var, i_checom, CFG_ITPL)

      WRITE(iopr,fmt_)
      WRITE(iopr,fmt6) 'mol_' // TRIM(c_shortid) // ' = SUM(' // &
         TRIM(c_shortid) // '_stoech(:) * ' // &
         TRIM(c_shortid) // '_eltmolwgt(:))'

    ENDIF

  ENDIF

  IF (ALLOCATED(c_compo_stoech_var)) DEALLOCATE(c_compo_stoech_var)
  IF (ALLOCATED(c_elmnt_molwgt_var)) DEALLOCATE(c_elmnt_molwgt_var)


  t_compo_curr => t_compo_curr%next

ENDDO

WRITE(idcl,fmt_)
CLOSE(UNIT=idcl)


WRITE(iopr,fmt_)
WRITE(iopr,fmt_)
WRITE(iopr,fmtmlc) 'Transcribe final molar masses'
WRITE(iopr,fmtmlc) 'into the <amol> array'
WRITE(iopr,fmt_)
WRITE(iopr,fmt6) 'amol(:) = 0.0D+00'
WRITE(iopr,fmt_)

t_compo_curr => t_compo_chain

DO WHILE(ASSOCIATED(t_compo_curr))

  IF (t_compo_curr%phasid == 'if') THEN

    SELECT CASE(t_compo_curr%class)
    CASE(cp_classsolid, cp_classorgmcnp, cp_classsolidcolour)

      c_shortid = t_compo_curr%shortid
      WRITE(iopr,fmt6) 'amol(' // cp_prefix_if // TRIM(c_shortid) // ') = &
        &mol_' // TRIM(c_shortid)

    CASE DEFAULT

      CONTINUE                      ! Amend here if molar masses for other
                                    ! components need to be considered.
                                    ! Currently only "BasicSolid" and
                                    ! "OrgMatter_CNP" class components require
                                    ! molar masses.
    END SELECT

  ENDIF

  t_compo_curr => t_compo_curr%next

ENDDO


WRITE(iopr,fmt_)
CLOSE(UNIT=iopr)

DEALLOCATE(c_psv)



OPEN(UNIT=iout, FILE = "tmp/get_materialcharas.F")

WRITE(iout,fmt0)   '! This subroutine has been automatically generated by'
WRITE(iout,fmt0)   '! CREATE_MOD_MATERIALCHARAS from the MEDUSA configuration'
WRITE(iout,fmt0)   '! and source code generation utility MedusaCoCoGen'
WRITE(iout,fmt0)   '!---+----1----+----2----+----3----+----4----+----5----+----6----+----7--'
WRITE(iout,fmt0)   '!======================================================================='
WRITE(iout,fmt6)   'SUBROUTINE GET_MATERIALCHARAS(i_column, iflag)'
WRITE(iout,fmt0)   '!======================================================================='
WRITE(iout,fmt_)
WRITE(iout,fmt0)   '! This subroutine loads fresh chemical composition and molar'
WRITE(iout,fmt0)   '! mass data into MOD_MATERIALCHARAS for components that do not'
WRITE(iout,fmt0)   '! have uniform composition.'
WRITE(iout,fmt_)
WRITE(iout,fmt_)
WRITE(iout,fmt6)   'USE mod_materialcharas'
WRITE(iout,fmt6)   'USE mod_indexparam, ONLY: nomcompo'
WRITE(iout,fmt_)
WRITE(iout,fmt6)   'IMPLICIT NONE'
WRITE(iout,fmt_)
WRITE(iout,fmt6)   'INTEGER, INTENT(IN)  :: i_column'
WRITE(iout,fmt6)   'INTEGER, INTENT(OUT) :: iflag'
WRITE(iout,fmt_)
WRITE(iout,fmt_)
WRITE(iout,fmt6)   'IF(nomcompo == 0) THEN'
WRITE(iout,fmt6)   '  iflag = -1                        ! Trigger warning flag'
WRITE(iout,fmt6)   '  RETURN'
WRITE(iout,fmt6)   'ENDIF'
WRITE(iout,fmt_)
WRITE(iout,fmt_)
WRITE(iout,fmt6)   'IF((i_column >= 1) .AND. (i_column <= n_grid_seafloor)) THEN'

t_compo_curr => t_compo_chain

i_omcompo = 0

DO WHILE(ASSOCIATED(t_compo_curr))

  c_shortid = t_compo_curr%shortid

  SELECT CASE(t_compo_curr%class)

  CASE(cp_classorgmcnp)

    i_omcompo   = i_omcompo + 1

    WRITE(iout,fmt_)
    WRITE(iout,fmtmlc) TRIM(t_compo_curr%name)
    c_codeline1 = '  ' // TRIM(c_shortid) // '_@ &
                  &= seafloor_omcnp_@(' // cp_prefix_ioo // TRIM(c_shortid) // ', i_column)'

    CALL EXPAND_TOKEN(c_codeline1, '@', 'c', c_codeline)
    WRITE(iout,fmt6) TRIM(c_codeline)

    CALL EXPAND_TOKEN(c_codeline1, '@', 'n', c_codeline)
    WRITE(iout,fmt6) TRIM(c_codeline)

    CALL EXPAND_TOKEN(c_codeline1, '@', 'p', c_codeline)
    WRITE(iout,fmt6) TRIM(c_codeline)

    CALL EXPAND_TOKEN(c_codeline1, '@', 'o', c_codeline)
    WRITE(iout,fmt6) TRIM(c_codeline)

    CALL EXPAND_TOKEN(c_codeline1, '@', 'h', c_codeline)
    WRITE(iout,fmt6) TRIM(c_codeline)

    CALL EXPAND_TOKEN(c_codeline1, '@', 'remin_o2', c_codeline)
    WRITE(iout,fmt6) TRIM(c_codeline)

    WRITE(iout,fmt_)

    WRITE(iout,fmt6) '  mol_' // TRIM(c_shortid) // &
                     ' = seafloor_omcnp_mol(' // cp_prefix_ioo // TRIM(c_shortid) // ', i_column)'
    WRITE(iout,fmt_)


  CASE (cp_classsolidcolour)

    c_class_master   = t_compo_curr%xref%class
    c_shortid_master = t_compo_curr%xref%shortid
    n                = t_compo_curr%n_params
    nlen = LEN_TRIM(t_compo_curr%xref%shortid)

    SELECT CASE(c_class_master)

    CASE(cp_classorgmcnp)

      WRITE(iout,fmt_)
      WRITE(iout,fmtmlc) TRIM(t_compo_curr%name)
      WRITE(iout,fmt6) '  ' // TRIM(c_shortid) // '_c = ' // TRIM(c_shortid_master) // '_c'
      WRITE(iout,fmt6) '  ' // TRIM(c_shortid) // '_n = ' // TRIM(c_shortid_master) // '_n'
      WRITE(iout,fmt6) '  ' // TRIM(c_shortid) // '_p = ' // TRIM(c_shortid_master) // '_p'
      WRITE(iout,fmt6) '  ' // TRIM(c_shortid) // '_o = ' // TRIM(c_shortid_master) // '_o'
      WRITE(iout,fmt6) '  ' // TRIM(c_shortid) // '_h = ' // TRIM(c_shortid_master) // '_h'
      WRITE(iout,fmt6) '  ' // TRIM(c_shortid) // '_remin_o2 = ' // TRIM(c_shortid_master) // '_remin_o2'
      WRITE(iout,fmt_)
      WRITE(iout,fmt6) '  mol_' // TRIM(c_shortid) // ' = mol_' // TRIM(c_shortid_master)
      WRITE(iout,fmt_)


    CASE DEFAULT

      CONTINUE                      ! Amend here if other data have to be transferred

    END SELECT


  CASE DEFAULT

    CONTINUE                        ! Amend here if other data have to be transferred

  END SELECT

  t_compo_curr => t_compo_curr%next

ENDDO

IF(i_omcompo == 0) THEN
  WRITE(iout,fmt_)
  WRITE(iout,fmt0) '! No components of class="' // cp_classorgmcnp // '" found'
ENDIF

WRITE(iout,fmt_)
WRITE(iout,fmt6)   '  iflag = 0                         ! No error if we get here.'
WRITE(iout,fmt_)
WRITE(iout,fmt6)   'ELSE'
WRITE(iout,fmt_)
WRITE(iout,fmt6)   '  iflag = 1                         ! Error: illegal i_column value.'
WRITE(iout,fmt_)
WRITE(iout,fmt6)   'ENDIF'
WRITE(iout,fmt_)
WRITE(iout,fmt_)
WRITE(iout,fmt6)   'RETURN'
WRITE(iout,fmt_)
WRITE(iout,fmt0)   '!======================================================================='
WRITE(iout,fmt6)   'END SUBROUTINE GET_MATERIALCHARAS'
WRITE(iout,fmt0)   '!======================================================================='
WRITE(iout,fmt0)   '!---+----1----+----2----+----3----+----4----+----5----+----6----+----7--'

CLOSE(UNIT=iout)




OPEN(UNIT=iout, FILE = "tmp/save_materialcharas.F")

WRITE(iout,fmt0)   '! This subroutine has been automatically generated by'
WRITE(iout,fmt0)   '! CREATE_MOD_MATERIALCHARAS from the MEDUSA configuration'
WRITE(iout,fmt0)   '! and source code generation utility MedusaCoCoGen'
WRITE(iout,fmt0)   '!---+----1----+----2----+----3----+----4----+----5----+----6----+----7--'
WRITE(iout,fmt0)   '!======================================================================='
WRITE(iout,fmt6)   'SUBROUTINE SAVE_MATERIALCHARAS(i_column, iflag)'
WRITE(iout,fmt0)   '!======================================================================='
WRITE(iout,fmt_)
WRITE(iout,fmt0)   '! This subroutine saves the chemical composition and molar'
WRITE(iout,fmt0)   '! mass data for components that do not have uniform composition'
WRITE(iout,fmt0)   '! from MOD_MATERIALCHARAS into MOD_SEAFLOOR_CENTRAL.'
WRITE(iout,fmt_)
WRITE(iout,fmt_)
WRITE(iout,fmt6)   'USE mod_materialcharas'
WRITE(iout,fmt6)   'USE mod_indexparam, ONLY: nomcompo'
WRITE(iout,fmt_)
WRITE(iout,fmt6)   'IMPLICIT NONE'
WRITE(iout,fmt_)
WRITE(iout,fmt6)   'INTEGER, INTENT(IN)  :: i_column'
WRITE(iout,fmt6)   'INTEGER, INTENT(OUT) :: iflag'
WRITE(iout,fmt_)
WRITE(iout,fmt_)
WRITE(iout,fmt6)   'IF(nomcompo == 0) THEN'
WRITE(iout,fmt6)   '  iflag = -1                        ! Trigger warning flag'
WRITE(iout,fmt6)   '  RETURN'
WRITE(iout,fmt6)   'ENDIF'
WRITE(iout,fmt_)
WRITE(iout,fmt_)
WRITE(iout,fmt6)   'IF((i_column >= 1) .AND. (i_column <= n_grid_seafloor)) THEN'

t_compo_curr => t_compo_chain

i_omcompo = 0

DO WHILE(ASSOCIATED(t_compo_curr))

  SELECT CASE(t_compo_curr%class)
  CASE(cp_classorgmcnp)

    i_omcompo   = i_omcompo + 1
    c_shortid = t_compo_curr%shortid

    WRITE(iout,fmt_)
    WRITE(iout,fmtmlc) TRIM(t_compo_curr%name)
    c_codeline1 = '  seafloor_omcnp_@(' // cp_prefix_ioo // TRIM(c_shortid) // ', i_column) &
                  &= ' // TRIM(c_shortid) // '_@'

    CALL EXPAND_TOKEN(c_codeline1, '@', 'c', c_codeline)
    WRITE(iout,fmt6) TRIM(c_codeline)

    CALL EXPAND_TOKEN(c_codeline1, '@', 'n', c_codeline)
    WRITE(iout,fmt6) TRIM(c_codeline)

    CALL EXPAND_TOKEN(c_codeline1, '@', 'p', c_codeline)
    WRITE(iout,fmt6) TRIM(c_codeline)

    CALL EXPAND_TOKEN(c_codeline1, '@', 'o', c_codeline)
    WRITE(iout,fmt6) TRIM(c_codeline)

    CALL EXPAND_TOKEN(c_codeline1, '@', 'h', c_codeline)
    WRITE(iout,fmt6) TRIM(c_codeline)

    CALL EXPAND_TOKEN(c_codeline1, '@', 'remin_o2', c_codeline)
    WRITE(iout,fmt6) TRIM(c_codeline)

    WRITE(iout,fmt_)

    WRITE(iout,fmt6) '  seafloor_omcnp_mol(' // cp_prefix_ioo // TRIM(c_shortid) // ', i_column) &
                     &= mol_' // TRIM(c_shortid)
    WRITE(iout,fmt_)


  CASE DEFAULT

    CONTINUE                        ! Amend here if other data have to be transferred

  END SELECT

  t_compo_curr => t_compo_curr%next

ENDDO

IF(i_omcompo == 0) THEN
  WRITE(iout,fmt_)
  WRITE(iout,fmt0) '! No components of class="' // cp_classorgmcnp // '" found'
  WRITE(iout,fmt_)
  WRITE(iout,fmt_)
ENDIF

WRITE(iout,fmt6)   '  iflag = 0                         ! No error if we get here.'
WRITE(iout,fmt_)
WRITE(iout,fmt6)   'ELSE'
WRITE(iout,fmt_)
WRITE(iout,fmt6)   '  iflag = 1                         ! Error: illegal i_column value.'
WRITE(iout,fmt_)
WRITE(iout,fmt6)   'ENDIF'
WRITE(iout,fmt_)
WRITE(iout,fmt_)
WRITE(iout,fmt6)   'RETURN'
WRITE(iout,fmt_)
WRITE(iout,fmt0)   '!======================================================================='
WRITE(iout,fmt6)   'END SUBROUTINE SAVE_MATERIALCHARAS'
WRITE(iout,fmt0)   '!======================================================================='
WRITE(iout,fmt0)   '!---+----1----+----2----+----3----+----4----+----5----+----6----+----7--'

CLOSE(UNIT=iout)


CALL EXPAND_INCLUDES("tmp/mod_materialcharas-base.F", "tmp/mod_materialcharas.F", "tmp")

! Now normalize the source code
CALL NORMALIZE_SOURCECODE("tmp/mod_materialcharas.F", "gen/mod_materialcharas.F")
CALL NORMALIZE_SOURCECODE("tmp/get_materialcharas.F", "gen/get_materialcharas.F")
CALL NORMALIZE_SOURCECODE("tmp/save_materialcharas.F", "gen/save_materialcharas.F")

IF (n_modelomcompo >0) THEN
  DEALLOCATE(jsubclass_omcompo)
ENDIF


WRITE(jp_stdout,c_fmtinf_a) 'completed'

#ifdef CFG_DEBUG
WRITE(jp_stddbg,c_fmtdbg_a) 'completed'
CALL FLUSH(jp_stddbg)
#endif

RETURN

!===================================================================================================
 END SUBROUTINE CREATE_MOD_MATERIALCHARAS
!===================================================================================================
