!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#include "configure.h"
!===============================================================================
 SUBROUTINE CREATE_MASS_BALANCES(t_compo_chain, t_solsys_chain)
!===============================================================================

USE MOD_MEDUSA_COCOGEN
USE MOD_CONFIGURE_TYPES
USE MOD_CONFIGURE
USE MOD_UTILITIES, ONLY: EXPAND_TOKEN, EXPAND_2TOKEN, LOWCASE


IMPLICIT NONE


! Variables of the dummy argument list
! ------------------------------------

TYPE(COMPOINFO),   INTENT(IN), TARGET  :: t_compo_chain
TYPE(SYSTEMSINFO), INTENT(IN), TARGET  :: t_solsys_chain


! Extra local derived types
! -------------------------

TYPE CMBDATA
  CHARACTER(LEN=n_lmaxshortid) :: shortid
  CHARACTER(LEN=n_lmaxexpress) :: weights
  TYPE(CMBDATA), POINTER       :: prev
  TYPE(CMBDATA), POINTER       :: next
END TYPE

TYPE CMBDATAP
  INTEGER                      :: n_data
  TYPE(CMBDATA), POINTER       :: ptr
END TYPE


! Local parameters
! ----------------

INTEGER, PARAMETER :: iu      = CFG_C_UNIT
INTEGER, PARAMETER :: iu_decl = CFG_C1UNIT
INTEGER, PARAMETER :: iu_init = CFG_C2UNIT
INTEGER, PARAMETER :: iu_maxr = CFG_C1UNIT
INTEGER, PARAMETER :: iu_glor = CFG_C2UNIT
INTEGER, PARAMETER :: iu_grea = CFG_C3UNIT
INTEGER, PARAMETER :: iu_gsum = CFG_C1UNIT
INTEGER, PARAMETER :: iu_gtot = CFG_C2UNIT

INTEGER, PARAMETER :: iu_gmba = CFG_C1UNIT
INTEGER, PARAMETER :: iu_gstk = CFG_C2UNIT


CHARACTER(LEN=*), PARAMETER :: cp_cmb_total       = 'total'
CHARACTER(LEN=*), PARAMETER :: cp_cmbset_bulk     = 'bulk'
CHARACTER(LEN=*), PARAMETER :: cp_cmbset_orgm     = 'orgm'
INTEGER,          PARAMETER :: jp_nknown_sets     = 2

INTEGER,          PARAMETER :: jp_cmbbulk_total   = 1
CHARACTER(LEN=*), PARAMETER :: cp_cmbbulk_total   = cp_cmb_total
INTEGER,          PARAMETER :: jp_cmbbulk_porew   = 2
CHARACTER(LEN=*), PARAMETER :: cp_cmbbulk_porew   = 'porew'
INTEGER,          PARAMETER :: jp_cmbbulk_solid   = 3
CHARACTER(LEN=*), PARAMETER :: cp_cmbbulk_solid   = 'solid'
INTEGER,          PARAMETER :: jp_nsubsets_bulk   = 3

INTEGER,          PARAMETER :: jp_cmborgm_0 = jp_nsubsets_bulk

INTEGER,          PARAMETER :: jp_cmborgm_total    = jp_cmborgm_0 + 1
CHARACTER(LEN=*), PARAMETER :: cp_cmborgm_total    = cp_cmb_total
INTEGER,          PARAMETER :: jp_cmborgm_cnp      = jp_cmborgm_0 + 2
CHARACTER(LEN=*), PARAMETER :: cp_cmborgm_cnp      = 'cnp'
INTEGER,          PARAMETER :: jp_nsubsets_orgm    = 2

INTEGER,          PARAMETER :: jp_nsubsets         = jp_nsubsets_bulk &
                                                    +jp_nsubsets_orgm

CHARACTER(LEN=n_lmaxidentif), DIMENSION(jp_nsubsets), &
                  PARAMETER :: cp_subsets          =  (/ cp_cmbbulk_total, &
                                                         cp_cmbbulk_porew, &
                                                         cp_cmbbulk_solid, &
                                                         cp_cmborgm_total, &
                                                         cp_cmborgm_cnp // '  ' /)
CHARACTER(LEN=n_lmaxidentif), DIMENSION(jp_nsubsets), &
                  PARAMETER :: cp_sets             =  (/ cp_cmbset_bulk, &
                                                         cp_cmbset_bulk, &
                                                         cp_cmbset_bulk, &
                                                         cp_cmbset_orgm, &
                                                         cp_cmbset_orgm    /)

! Variable name construction related parameters
CHARACTER(LEN=*), PARAMETER :: cp_col = 'c_'
CHARACTER(LEN=*), PARAMETER :: cp_glo = 'g_'
CHARACTER(LEN=*), PARAMETER :: cp_max = 'm_'


! Local variables
! ---------------

CHARACTER(LEN=n_lmaxformat)  :: fmttmp
CHARACTER(LEN=3)             :: c_tmp

CHARACTER(LEN=n_lmaxphasid)  :: c_phasid
CHARACTER(LEN=n_lmaxshortid) :: c_shortid
CHARACTER(LEN=n_lmaxexpress) :: c_vartype


CHARACTER(LEN=n_lmaxidentif), DIMENSION(:),   POINTER :: c_what
INTEGER,                      DIMENSION(:),   POINTER :: n_subsets
TYPE(CMBDATAP),               DIMENSION(:,:), POINTER :: t_cmbdata_table


! Standard I/O related data
! -------------------------

INTEGER,          PARAMETER  :: jp_stdout  = CFG_STDOUT
INTEGER,          PARAMETER  :: jp_stderr  = CFG_STDERR
CHARACTER(LEN=*), PARAMETER  :: c_fmtinf_a = '("[CREATE_MASS_BALANCES]: ", A)'
CHARACTER(LEN=*), PARAMETER  :: c_fmtwar_a = '("[CREATE_MASS_BALANCES] warning: ", A)'
CHARACTER(LEN=*), PARAMETER  :: c_fmterr_a = '("[CREATE_MASS_BALANCES] error: ", A)'

#ifdef CFG_DEBUG
INTEGER,          PARAMETER  :: jp_stddbg  = CFG_STDDBG
CHARACTER(LEN=*), PARAMETER  :: c_fmtdbg_a = '("DEBUG [CREATE_MASS_BALANCES]: ", A)'
CHARACTER(LEN=*), PARAMETER  :: c_fmtdbg_ai = '("DEBUG [CREATE_MASS_BALANCES]: ", A, I0)'
#endif


!=====================!
! End of declarations !
!=====================!

WRITE(jp_stdout,'()') 
WRITE(jp_stdout,c_fmtinf_a) 'starting'
#ifdef CFG_DEBUG
WRITE(jp_stddbg,'()') 
WRITE(jp_stddbg,c_fmtdbg_a) 'starting'
#endif

!============!
! Operations !
!============!


#ifdef CFG_DEBUG
WRITE(jp_stddbg, c_fmtdbg_ai) 'jp_cmbbulk_total = ', jp_cmbbulk_total
WRITE(jp_stddbg, c_fmtdbg_ai) 'jp_cmbbulk_porew = ', jp_cmbbulk_porew
WRITE(jp_stddbg, c_fmtdbg_ai) 'jp_cmbbulk_solid = ', jp_cmbbulk_solid
WRITE(jp_stddbg, c_fmtdbg_ai) 'jp_cmborgm_total = ', jp_cmborgm_total
WRITE(jp_stddbg, c_fmtdbg_ai) 'jp_cmborgm_cnp   = ', jp_cmborgm_cnp
#endif


                                    ! First determine which quantities
                                    ! (alkalinity, masses etc.) should
                                    ! be monitored. Also set up basic
                                    ! data about phases and sub-phases
                                    ! that are more specifically to be
                                    ! monitored
NULLIFY(c_what)
NULLIFY(t_cmbdata_table)


CALL CMB_makeInventory(t_compo_chain, c_what, t_cmbdata_table)


                                    ! Set up the declaration and
                                    ! initialisation parts of the code.

OPEN(UNIT=iu_decl, FILE = "tmp/mass_balances_1.F")  ! declarations
OPEN(UNIT=iu_init, FILE = "tmp/mass_balances_2.F")  ! initialisations
  CALL CMB_createDeclareAndInit(iu_decl, iu_init, c_what, t_cmbdata_table)
CLOSE(UNIT=iu_decl)
CLOSE(UNIT=iu_init)


                                    ! Create local working copies of the input data

OPEN(UNIT=iu, FILE = "tmp/mass_balances_3.F")
  CALL CMB_createPrepareWorkingData(iu, t_compo_chain)
  CALL CMB_createSpreadResults(iu, c_what, t_cmbdata_table)
CLOSE(UNIT=iu)


                                    ! Calculate column sums

OPEN(UNIT=iu, FILE = "tmp/mass_balances_4.F")
  CALL CMB_createColumnSums(iu, c_what, t_cmbdata_table)
CLOSE(UNIT=iu)


                                    ! Calculate global sums (inside i_column loop)
                                    ! and global totals (outside i_column loop)

OPEN(UNIT=iu_gsum, FILE = "tmp/mass_balances_5.F")
OPEN(UNIT=iu_gtot, FILE = "tmp/mass_balances_6.F")
  CALL CMB_createGlobalSumsTotals(iu_gsum, iu_gtot, c_what, t_cmbdata_table)
CLOSE(UNIT=iu_gsum)
CLOSE(UNIT=iu_gtot)


                                    ! Produce code for reporting the results'
                                    ! maximum residuals, global residuals,
                                    ! and global reaction rates

OPEN(UNIT=iu_maxr, FILE = "tmp/mass_balances_7.F")   ! max_residu
OPEN(UNIT=iu_glor, FILE = "tmp/mass_balances_8.F")   ! glo_residu
OPEN(UNIT=iu_grea, FILE = "tmp/mass_balances_9.F")   ! glo_rea
  CALL CMB_createPrintResiduals_Rea(iu_maxr, iu_glor, iu_grea, c_what, t_cmbdata_table)
CLOSE(UNIT=iu_maxr)
CLOSE(UNIT=iu_glor)
CLOSE(UNIT=iu_grea)


                                    ! Produce code to print out
                                    ! global mass balances

OPEN(UNIT=iu_gmba, FILE = "tmp/mass_balances_10.F")
OPEN(UNIT=iu_gstk, FILE = "tmp/mass_balances_11.F")
  CALL CMB_createPrintGlobalInfos(iu_gmba, iu_gstk, c_what, t_cmbdata_table)
CLOSE(UNIT=iu_gmba)
CLOSE(UNIT=iu_gstk)


                                    ! Create the SIGMA function subprograms

OPEN(UNIT=iu, FILE = "tmp/mass_balances_contains_1.F")
  CALL CMB_createSigmaFunctions(iu, c_what, t_cmbdata_table)
CLOSE(UNIT=iu)


                                    ! Fit all the subroutine parts into
                                    ! "lib/mass_balances-base.F"

CALL EXPAND_INCLUDES("base/mass_balances-base.F", "tmp/mass_balances.F", "tmp")


                                    ! Now normalize the source code and
                                    ! store the result in "gen/mass_balances.F"

CALL NORMALIZE_SOURCECODE("tmp/mass_balances.F", "gen/mass_balances.F")


CALL CMB_CMBDATAP_delete(t_cmbdata_table)


WRITE(jp_stdout,c_fmtinf_a) 'completed'
#ifdef CFG_DEBUG
WRITE(jp_stddbg,c_fmtdbg_a) 'completed'
#endif

RETURN

CONTAINS

!-------------------------------------------------------------------------------
 SUBROUTINE CMB_makeInventory(t_compo_root, c_what, t_cmbdata_table)
!-------------------------------------------------------------------------------

USE MOD_UTILITIES, ONLY: L_STR_EQ_DBL

IMPLICIT NONE


TYPE(COMPOINFO), INTENT(IN), TARGET                   :: t_compo_root
CHARACTER(LEN=n_lmaxidentif), DIMENSION(:),   POINTER :: c_what
TYPE(CMBDATAP),               DIMENSION(:,:), POINTER :: t_cmbdata_table


CHARACTER(LEN=LEN(c_what)), DIMENSION(:),   POINTER :: ctmp_what

INTEGER :: n_what, i_what, i_subset
INTEGER :: i
INTEGER :: n_consrv, i_consrv
TYPE(COMPOINFO), POINTER :: t_compo_work

CHARACTER(LEN=n_lmaxidentif) :: c_testname
LOGICAL :: l_register

CHARACTER(LEN=n_lmaxshortid)   :: c_shortid
CHARACTER(LEN=n_lmaxexpress)   :: c_weight
#ifdef CFG_DEBUG
CHARACTER(LEN=LEN(cp_subsets)) :: c_set_curr, c_subset_curr
#endif


! Standard I/O related data
! -------------------------

CHARACTER(LEN=*), PARAMETER  :: c_fmtinf_a   = '("[[CMB_makeInventory]]: ", A)'
#ifdef CFG_DEBUG
CHARACTER(LEN=*), PARAMETER  :: c_fmtdbg_a   = '("DEBUG [[CMB_makeInventory]]: ", A)'
#endif

!=====================!
! End of declarations !
!=====================!


#ifdef CFG_VERBOSE
#if (CFG_VERBOSELEVEL == 2)
WRITE(jp_stdout,c_fmtinf_a) 'starting'
#endif
#endif
#ifdef CFG_DEBUG
WRITE(jp_stddbg,c_fmtdbg_a) 'starting'
#endif


!=====================!
! Start of operations !
!=====================!

NULLIFY(c_what)
NULLIFY(t_cmbdata_table)

                                    ! Provisionally allocate as much
                                    ! space as there are components in
                                    ! t_compo, plus one for alkalinity.
ALLOCATE(ctmp_what(n_medusacompo+1))

                                    ! Pre-initialise n_what/ctmp_what

IF (n_solutes_withalk == 0) THEN
  n_what = 0
ELSE
  ctmp_what(1) = cp_shortid_alk
  n_what = 1
ENDIF


                                    ! Scan t_compo_root for
                                    ! required conservation information

t_compo_work => t_compo_root

DO WHILE (ASSOCIATED(t_compo_work))

  n_consrv = t_compo_work%n_consrv

  SELECT CASE(t_compo_work%phasid)
  CASE('ic', 'if')
    IF(n_consrv /= 0) THEN
                                    ! Set up the list of those parameters
                                    ! that have to be actually monitored
      check_consrv: DO i_consrv = 1, n_consrv
        c_testname = t_compo_work%consrv_name(i_consrv)
        DO i_what = 1, n_what
                                    ! If the current conservation-component
                                    ! is already know, skip it and go to the
                                    ! next one
          IF(ctmp_what(i_what) == c_testname) CYCLE check_consrv
        ENDDO
                                    ! Count it, and register its name
        n_what = n_what + 1
        ctmp_what(n_what) = c_testname
      ENDDO check_consrv
    ENDIF
  END SELECT

  t_compo_work => t_compo_work%next

ENDDO


IF (n_what > 0) THEN
  ALLOCATE(c_what(n_what))
  DO i_what = 1, n_what
    c_what(i_what) = ctmp_what(i_what)
  ENDDO
ENDIF


DEALLOCATE(ctmp_what)


#ifdef CFG_DEBUG
WRITE(jp_stddbg,c_fmtdbg_a, ADVANCE='NO') 'c_what has '
WRITE(jp_stddbg,'(I0, A)') n_what, ' item(s)'
DO i = 1, n_what
  WRITE(jp_stddbg,'(I4, ": <", A, ">")') i, TRIM(c_what(i))
ENDDO
WRITE(jp_stddbg,c_fmtdbg_a, ADVANCE='NO') 'cp_subsets has '
WRITE(jp_stddbg,'(I0, A)') jp_nsubsets, ' item(s)'
DO i = 1, jp_nsubsets
  WRITE(jp_stddbg,'(I4, ": <", A, "; ", A,">")') &
    i, TRIM(cp_subsets(i)), TRIM(cp_sets(i))
ENDDO
#endif

                                    ! Re-scan t_compo to
                                    ! register the data: 
t_cmbdata_table => CMB_CMBDATAP_create(n_what, jp_nsubsets)


t_compo_work => t_compo_root

DO WHILE(ASSOCIATED(t_compo_work))

  SELECT CASE(t_compo_work%phasid)

  CASE('ic', 'if')

#   ifdef CFG_DEBUG
    WRITE(jp_stddbg,c_fmtdbg_a) 'Analysing component "' // TRIM(t_compo_work%name) // '"'
#   endif
    DO i_what = 1, n_what

      l_register = .FALSE.

#     ifdef CFG_DEBUG
      WRITE(jp_stddbg,c_fmtdbg_a,ADVANCE='NO') ' - for c_what="' // TRIM(c_what(i_what)) // '" (i_what='
      WRITE(jp_stddbg,'(I0)',ADVANCE='NO') i_what
      WRITE(jp_stddbg,'("): ")',ADVANCE='NO')
#     endif


      IF(c_what(i_what) == cp_shortid_alk) THEN
                                    ! If the sum to calculate concerns
                                    ! alkalinity, get the required 
                                    ! information from %param_values(:)
        i = COMPOINFO_getParamIdxByName(t_compo_work, cp_alkalinity)
        IF(i > 0) THEN
          IF(.NOT. L_STR_EQ_DBL(t_compo_work%param_values(i), 0D0)) THEN
            c_shortid = ADJUSTL(t_compo_work%shortid)
            c_weight = TRIM(ADJUSTL(t_compo_work%param_values(i)))
            l_register = .TRUE.
          ENDIF
           
        ENDIF

      ELSE
                                    ! The sum to calculate does concern
                                    ! some other element/component than
                                    ! alkalinity. Retrieve the information
                                    ! from %consrv_weights(:)
        i = COMPOINFO_getConsrvIdxByName(t_compo_work, c_what(i_what))
        IF (i > 0) THEN
          c_shortid = ADJUSTL(t_compo_work%shortid)
          c_weight  = ADJUSTL(t_compo_work%consrv_weights(i))
          l_register = .TRUE.
        ENDIF
      ENDIF

#     ifdef CFG_DEBUG
      IF(l_register) THEN
        WRITE(jp_stddbg,'("yes")')
      ELSE
        WRITE(jp_stddbg,'("no")')
      ENDIF
#     endif


      IF(l_register) THEN

        DO i_subset = 1, jp_nsubsets

          SELECT CASE(i_subset)
          CASE(jp_cmbbulk_total)
            CALL CMB_CMBDATA_registerData(t_cmbdata_table, i_what, i_subset, &
                                            c_shortid, c_weight)
#           ifdef CFG_DEBUG
            WRITE(jp_stddbg,c_fmtdbg_a,ADVANCE='NO') '   registering into i_subset='
            WRITE(jp_stddbg,'(I0)') i_subset
            WRITE(jp_stddbg,c_fmtdbg_a,ADVANCE='NO') '   now t_cmbdata_table(.,.)%n_data='
            WRITE(jp_stddbg,'(I0)') t_cmbdata_table(i_what, i_subset)%n_data
#           endif
          CASE(jp_cmbbulk_porew)
            IF(t_compo_work%phasid == 'ic') THEN 
              CALL CMB_CMBDATA_registerData(t_cmbdata_table, i_what, i_subset, &
                                            c_shortid, c_weight)
#             ifdef CFG_DEBUG
              WRITE(jp_stddbg,c_fmtdbg_a,ADVANCE='NO') '   registering into i_subset='
              WRITE(jp_stddbg,'(I0)') i_subset
              WRITE(jp_stddbg,c_fmtdbg_a,ADVANCE='NO') '   now t_cmbdata_table(.,.)%n_data='
              WRITE(jp_stddbg,'(I0)') t_cmbdata_table(i_what, i_subset)%n_data
#             endif
            ENDIF
          CASE(jp_cmbbulk_solid)
            IF(t_compo_work%phasid == 'if') THEN
              CALL CMB_CMBDATA_registerData(t_cmbdata_table, i_what, i_subset, &
                                            c_shortid, c_weight)
#             ifdef CFG_DEBUG
              WRITE(jp_stddbg,c_fmtdbg_a,ADVANCE='NO') '   registering into i_subset='
              WRITE(jp_stddbg,'(I0)') i_subset
              WRITE(jp_stddbg,c_fmtdbg_a,ADVANCE='NO') '   now t_cmbdata_table(.,.)%n_data='
              WRITE(jp_stddbg,'(I0)') t_cmbdata_table(i_what, i_subset)%n_data
#             endif
            ENDIF
          CASE(jp_cmborgm_total)
            IF(t_compo_work%class == cp_classorgmcnp) THEN
              CALL CMB_CMBDATA_registerData(t_cmbdata_table, i_what, i_subset, &
                                            c_shortid, c_weight)
#             ifdef CFG_DEBUG
              WRITE(jp_stddbg,c_fmtdbg_a,ADVANCE='NO') '   registering into i_subset='
              WRITE(jp_stddbg,'(I0)') i_subset
              WRITE(jp_stddbg,c_fmtdbg_a,ADVANCE='NO') '   now t_cmbdata_table(.,.)%n_data='
              WRITE(jp_stddbg,'(I0)') t_cmbdata_table(i_what, i_subset)%n_data
#             endif
            ENDIF
          CASE(jp_cmborgm_cnp)
            IF(t_compo_work%class == cp_classorgmcnp) THEN
              CALL CMB_CMBDATA_registerData(t_cmbdata_table, i_what, i_subset, &
                                            c_shortid, c_weight)
#             ifdef CFG_DEBUG
              WRITE(jp_stddbg,c_fmtdbg_a,ADVANCE='NO') '   registering into i_subset='
              WRITE(jp_stddbg,'(I0)') i_subset
              WRITE(jp_stddbg,c_fmtdbg_a,ADVANCE='NO') '   now t_cmbdata_table(.,.)%n_data='
              WRITE(jp_stddbg,'(I0)') t_cmbdata_table(i_what, i_subset)%n_data
#             endif
            ENDIF
          END SELECT

        ENDDO

      ENDIF

    ENDDO

  END SELECT

  t_compo_work => t_compo_work%next

ENDDO

#ifdef CFG_DEBUG
DO i_what = 1, n_what
  DO i_subset = 1, jp_nsubsets
    WRITE(jp_stddbg,c_fmtdbg_a,ADVANCE='NO') '(i_what,i_subset) = '
    WRITE(jp_stddbg,'("(",I0,",",I0,")")') i_what, i_subset
    c_set_curr    = cp_sets(i_subset)
    c_subset_curr = cp_subsets(i_subset)
    WRITE(jp_stddbg,c_fmtdbg_a) &
      'set:subset = ' // TRIM(c_set_curr) // ':' // TRIM(c_subset_curr)
    WRITE(jp_stddbg,c_fmtdbg_a,ADVANCE='NO') '  t_cmbdata_table(.,.)%n_data = '
    WRITE(jp_stddbg,'(I0)') t_cmbdata_table(i_what, i_subset)%n_data
  ENDDO
ENDDO
#endif


#ifdef CFG_VERBOSE
#if (CFG_VERBOSELEVEL == 2)
WRITE(jp_stdout,c_fmtinf_a) 'done'
#endif
#endif
#ifdef CFG_DEBUG
WRITE(jp_stddbg,c_fmtdbg_a) 'done'
#endif

RETURN

!-------------------------------------------------------------------------------
 END SUBROUTINE CMB_makeInventory
!-------------------------------------------------------------------------------



!-------------------------------------------------------------------------------
 SUBROUTINE CMB_getInformationForOneWhat(t_cmbdata_table, c_what, i_what,       &
                                         c_what_work, c_consrvprop)
!-------------------------------------------------------------------------------


IMPLICIT NONE


! Dummy argument list variables
! -----------------------------

TYPE(CMBDATAP),   DIMENSION(:,:), INTENT(IN),  TARGET :: t_cmbdata_table
CHARACTER(LEN=*), DIMENSION(:),   INTENT(IN),  TARGET :: c_what
INTEGER,                          INTENT(IN)          :: i_what
CHARACTER(LEN=*),                 INTENT(OUT)         :: c_what_work
CHARACTER(LEN=*),                 INTENT(OUT)         :: c_consrvprop


! Local variables
! ---------------

INTEGER :: n_porew, n_solid, n_total

c_what_work = ADJUSTL(LOWCASE(c_what(i_what)))

SELECT CASE(c_what_work)
CASE('c')
  c_consrvprop = 'carbon'
CASE('o2')
  c_consrvprop = 'oxygen'
CASE('si')
  c_consrvprop = 'silicon'
CASE('alk')
  c_consrvprop = 'alkalinity'
CASE DEFAULT
  c_consrvprop = 'subphase "' // TRIM(ADJUSTL(c_what_work)) // '"'
END SELECT


RETURN

!-------------------------------------------------------------------------------
 END SUBROUTINE CMB_getInformationForOneWhat
!-------------------------------------------------------------------------------



!-------------------------------------------------------------------------------
 SUBROUTINE CMB_createPrepareWorkingData(iu, t_compo_root)
!-------------------------------------------------------------------------------

IMPLICIT NONE


! Dummy argument list variables
! -----------------------------

INTEGER, INTENT(IN) :: iu
TYPE(COMPOINFO), INTENT(IN), TARGET :: t_compo_root


! Local variables
! ---------------

TYPE(COMPOINFO),                POINTER     :: t_compo_curr
CHARACTER(LEN=n_lmaxshortid)                :: c_shortid
CHARACTER(LEN=n_lmaxcomptyp)                :: c_class
CHARACTER(LEN=n_lmaxshortid+4)              :: c_molvar
CHARACTER(LEN=n_lmaxidentif)                :: c_io

CHARACTER(LEN=*), PARAMETER  :: c_fmtinf_a   = '("[[CMB_createPrepareWorkingData]]: ", A)'
#ifdef CFG_DEBUG
CHARACTER(LEN=*), PARAMETER  :: c_fmtdbg_a   = '("DEBUG [[CMB_createPrepareWorkingData]]: ", A)'
#endif

!=====================!
! End of declarations !
!=====================!


#ifdef CFG_VERBOSE
#if (CFG_VERBOSELEVEL == 2)
WRITE(jp_stdout,c_fmtinf_a) 'starting'
#endif
#endif
#ifdef CFG_DEBUG
WRITE(jp_stddbg,c_fmtdbg_a) 'starting'
#endif


!=====================!
! Begin of Operations !
!=====================!

# ifdef CFG_DEBUG
  WRITE(iu,fmt0, ADVANCE='NO') '! '
  WRITE(iu,c_fmtinf_a) 'Start of creation'
# endif


WRITE(iu,fmt_)
WRITE(iu,fmt0) '! Create local working copies of the input data:'
WRITE(iu,fmt0) '! ---------------------------------------------'
WRITE(iu,fmt0) '!  1. initialize all the compoenents of the column*(:) arrays to 0;'
WRITE(iu,fmt0) '!  2. copy the solutes'' data as is (they are in mol)'
WRITE(iu,fmt0) '!  3. copy the solids'' data and convert units to moles'
WRITE(iu,fmt0) '!  4. calculate column residuals'
WRITE(iu,fmt0) '!  5. calculate maximum residuals'

WRITE(iu,fmt_)
WRITE(iu,fmt0) '                                    ! 1. Initialize all to zero'
WRITE(iu,fmt_)
WRITE(iu,fmt6) 'column_xm_1(:)  = 0D0'
WRITE(iu,fmt6) 'column_xm_0(:)  = 0D0'
WRITE(iu,fmt6) 'column_woflx(:) = 0D0'
WRITE(iu,fmt6) 'column_boflx(:) = 0D0'
WRITE(iu,fmt6) 'column_rea(:)   = 0D0'
WRITE(iu,fmt_)
WRITE(iu,fmt6) 'max_residual(:) = 0D0'

WRITE(iu,fmt_)
WRITE(iu,fmt_)
WRITE(iu,fmt0) '                                    ! 2. copy in solutes'' data'
WRITE(iu,fmt_)
WRITE(iu,fmt6) ' column_xm_0(jc_to_io(:)) =      total_xm_0(jc_to_io(:), i_column)'
WRITE(iu,fmt6) ' column_xm_1(jc_to_io(:)) =      total_xm_1(jc_to_io(:), i_column)'
WRITE(iu,fmt6) 'column_woflx(jc_to_io(:)) =    topflx_xm_01(jc_to_io(:), i_column)'
WRITE(iu,fmt6) 'column_boflx(jc_to_io(:)) = 0D0  ! !!! Check if botflx_xm_01 is 0 for solutes or not !!!'
WRITE(iu,fmt6) '  column_rea(jc_to_io(:)) = total_rea_xm_01(jc_to_io(:), i_column)'
WRITE(iu,fmt_)
WRITE(iu,fmt_)

WRITE(iu,fmt0) '                                    ! 3. copy in and convert solids'' data'
WRITE(iu,fmt_)


t_compo_curr => t_compo_root

DO WHILE(ASSOCIATED(t_compo_curr))

  SELECT CASE(t_compo_curr%phasid)

  CASE('if')

    c_shortid = ADJUSTL(t_compo_curr%shortid)
    c_class   = ADJUSTL(t_compo_curr%class)
    c_io      = cp_prefix_io // c_shortid
    c_molvar  = 'mol_' // TRIM(c_shortid)

    SELECT CASE(c_class)
    CASE(cp_classsolid, cp_classorgmcnp)

      WRITE(iu,fmt6)  'column_xm_1(' // TRIM(c_io) // ')  =   '  //  &
                       'total_xm_1(' // TRIM(c_io) // ', i_column)/'      // c_molvar
      WRITE(iu,fmt6)  'column_xm_0(' // TRIM(c_io) // ')  =   '  //  &
                       'total_xm_0(' // TRIM(c_io) // ', i_column)/'      // c_molvar
      WRITE(iu,fmt6) 'column_woflx(' // TRIM(c_io) // ') =  '   //   &
                      'topflx_xm_01(' // TRIM(c_io) // ', i_column)/'      // c_molvar
      WRITE(iu,fmt6) 'column_boflx(' // TRIM(c_io) // ') =  '   //   &
                      'botflx_xm_01(' // TRIM(c_io) // ', i_column)/'      // c_molvar
      WRITE(iu,fmt6)   'column_rea(' // TRIM(c_io) // ')   = ' // &
                     'total_rea_xm_01(' // TRIM(c_io) // ', i_column)/'      // c_molvar
      WRITE(iu,fmt_)

    CASE DEFAULT

      CONTINUE

    END SELECT



  CASE DEFAULT

    CONTINUE

  END SELECT

  t_compo_curr => t_compo_curr%next

ENDDO
WRITE(iu,fmt_)


WRITE(iu,fmt0) '                                    ! 4. column residuals'
WRITE(iu,fmt_)
WRITE(iu,fmt6)   'column_residual(:)  ='
WRITE(iu,fmtcon) '  column_xm_1(:) - column_xm_0(:)'
WRITE(iu,fmtcon) '  - (column_woflx(:) - column_boflx(:)) - column_rea(:)'
WRITE(iu,fmt_)
WRITE(iu,fmt_)


WRITE(iu,fmt0) '                                    ! 5. Maximum residuals'
WRITE(iu,fmt_)
WRITE(iu,fmt6)   'DO icompo = 1, ncompo'
WRITE(iu,fmt6)   '  max_residual(icompo)  ='
WRITE(iu,fmtcon) '  MAX(max_residual(icompo), ABS(column_residual(icompo)))'
WRITE(iu,fmt6)   'ENDDO'
WRITE(iu,fmt_)
WRITE(iu,fmt_)

NULLIFY(t_compo_curr)


# ifdef CFG_DEBUG
  WRITE(iu,fmt0, ADVANCE='NO') '! '
  WRITE(iu,c_fmtinf_a) 'End of creation'
# endif

!===================!
! End of Operations !
!===================!


#ifdef CFG_VERBOSE
#if (CFG_VERBOSELEVEL == 2)
WRITE(jp_stdout,c_fmtinf_a) 'done'
#endif
#endif
#ifdef CFG_DEBUG
WRITE(jp_stddbg,c_fmtdbg_a) 'done'
#endif

RETURN

!-------------------------------------------------------------------------------
 END SUBROUTINE CMB_createPrepareWorkingData
!-------------------------------------------------------------------------------



!-------------------------------------------------------------------------------
 SUBROUTINE CMB_createSpreadResults(iu, c_what, t_cmbdata_table)
!-------------------------------------------------------------------------------

USE MOD_UTILITIES, ONLY: L_STR_EQ_DBL

IMPLICIT NONE


! Dummy argument list variables
! -----------------------------

INTEGER, INTENT(IN) :: iu
CHARACTER(LEN=*), DIMENSION(:),   INTENT(IN)         :: c_what
TYPE(CMBDATAP),   DIMENSION(:,:), INTENT(IN), TARGET :: t_cmbdata_table


! Local variables
! ---------------

TYPE(CMBDATA), POINTER         :: t_cmbdata_node
CHARACTER(LEN=LEN(c_what))     :: c_what_curr
CHARACTER(LEN=n_lmaxshortid)   :: c_shortid
CHARACTER(LEN=n_lmaxidentif)   :: c_io
CHARACTER(LEN=n_lmaxidentif)   :: c_varnamebase
CHARACTER(LEN=n_lmaxexpress+7) :: c_times_weight
INTEGER                        :: i_what, i_subset
CHARACTER(LEN=20)              :: c_fmttmp


! Standard I/O related data
! -------------------------

CHARACTER(LEN=*), PARAMETER  :: c_fmtinf_a   = '("[[CMB_createSpreadResults]]: ", A)'
#ifdef CFG_DEBUG
CHARACTER(LEN=*), PARAMETER  :: c_fmtdbg_a   = '("DEBUG [[CMB_createSpreadResults]]: ", A)'
#endif

!=====================!
! End of declarations !
!=====================!


#ifdef CFG_VERBOSE
#if (CFG_VERBOSELEVEL == 2)
WRITE(jp_stdout,c_fmtinf_a) 'starting'
#endif
#endif
#ifdef CFG_DEBUG
WRITE(jp_stddbg,c_fmtdbg_a) 'starting'
#endif


!=====================!
! Begin of Operations !
!=====================!

# ifdef CFG_DEBUG
  WRITE(iu,fmt0, ADVANCE='NO') '! '
  WRITE(iu,c_fmtinf_a) 'Start of creation'
# endif


DO i_what = 1, SIZE(c_what)

  c_what_curr = LOWCASE(ADJUSTL(c_what(i_what)))

  WRITE(iu,fmt_)
  WRITE(iu,fmt_)
  WRITE(iu,fmtc)   '! Prepare (distribute, apply weights)'
  WRITE(iu,fmtc)   '! variables related to "' // TRIM(c_what(i_what)) // '":'
  WRITE(iu,fmtc, ADVANCE='NO')   &
                   '! ----------------------'
  WRITE(c_fmttmp,'("(",I0,"(""-"")",")")') LEN_TRIM(c_what_curr) + 1  ! 1 = LEN('"')
  WRITE(iu,c_fmttmp)

  WRITE(iu,fmtc)   '!  - ' // cp_col //  TRIM(c_what_curr) // '_xm_0(:)'
  WRITE(iu,fmtc)   '!  - ' // cp_col //  TRIM(c_what_curr) // '_xm_1(:)'
  WRITE(iu,fmtc)   '!  - ' // cp_col //  TRIM(c_what_curr) // '_woflx(:)'
  WRITE(iu,fmtc)   '!  - ' // cp_col //  TRIM(c_what_curr) // '_boflx(:)'
  WRITE(iu,fmtc)   '!  - ' // cp_col //  TRIM(c_what_curr) // '_rea(:)'
  WRITE(iu,fmtc)   '!  - ' // cp_col //  TRIM(c_what_curr) // '_resid(:)'
  WRITE(iu,fmtc)   '!  - ' // cp_max //  TRIM(c_what_curr) // '_resid(:)'
  WRITE(iu,fmt_)
  WRITE(iu,fmt_)


  WRITE(iu,fmtc)   '! First pre-set all to 0'

  c_varnamebase = cp_col // TRIM(c_what_curr)
  WRITE(iu,fmt6)   TRIM(c_varnamebase) // '_xm_0(:)     = 0D0'
  WRITE(iu,fmt6)   TRIM(c_varnamebase) // '_xm_1(:)     = 0D0'
  WRITE(iu,fmt6)   TRIM(c_varnamebase) // '_woflx(:)    = 0D0'
  WRITE(iu,fmt6)   TRIM(c_varnamebase) // '_boflx(:)    = 0D0'
  WRITE(iu,fmt6)   TRIM(c_varnamebase) // '_rea(:)      = 0D0'
  WRITE(iu,fmt6)   TRIM(c_varnamebase) // '_resid(:)    = 0D0'
  WRITE(iu,fmt_)

  WRITE(iu,fmtc)   '! Copy in content from column_* '
  WRITE(iu,fmtc)   '! and apply appropriate weights'


                                    ! Browse all of the components required,
                                    ! solids and solutes alike.
  t_cmbdata_node => t_cmbdata_table(i_what,jp_cmbbulk_total)%ptr

  DO WHILE(ASSOCIATED(t_cmbdata_node))

    c_shortid = ADJUSTL(t_cmbdata_node%shortid)
    c_io = '(' // cp_prefix_io // TRIM(c_shortid) // ')'

    IF(L_STR_EQ_DBL(t_cmbdata_node%weights, 1D0)) THEN
      c_times_weight = ''
    ELSE
      c_times_weight = '*DBLE(' // TRIM(t_cmbdata_node%weights) // ')'
    ENDIF

    c_varnamebase = cp_col // TRIM(c_what_curr)
    WRITE(iu,fmt6)   TRIM(c_varnamebase) // '_xm_0'  // TRIM(c_io) // &
                                   '  = column_xm_0' // TRIM(c_io) // ' ' // TRIM(c_times_weight)

    WRITE(iu,fmt6)   TRIM(c_varnamebase) //  '_xm_1' // TRIM(c_io) // &
                                   '  = column_xm_1' // TRIM(c_io) // ' ' // TRIM(c_times_weight)

    WRITE(iu,fmt6)   TRIM(c_varnamebase) // '_woflx' // TRIM(c_io) // &
                                   ' = column_woflx' // TRIM(c_io) // TRIM(c_times_weight)

    WRITE(iu,fmt6)   TRIM(c_varnamebase) // '_boflx' // TRIM(c_io) // &
                                   ' = column_boflx' // TRIM(c_io) // TRIM(c_times_weight)

    WRITE(iu,fmt6)   TRIM(c_varnamebase) //   '_rea' // TRIM(c_io) // &
                                   '   = column_rea' // TRIM(c_io) // '  ' // TRIM(c_times_weight)

    WRITE(iu,fmt_)

    t_cmbdata_node => t_cmbdata_node%next

  ENDDO
  WRITE(iu,fmt_)



  WRITE(iu,fmtc)   '! Column residuals'
  WRITE(iu,fmtc)   '! ----------------'
  WRITE(iu,fmt_)

  c_varnamebase = cp_col // TRIM(c_what_curr)
  WRITE(iu,fmt6)                TRIM(c_varnamebase) // '_resid(:)'
  WRITE(iu,fmtcon) '  =   ' //  TRIM(c_varnamebase) // '_xm_1(:) - '   // &
                                TRIM(c_varnamebase) // '_xm_0(:)'
  WRITE(iu,fmtcon) '    - (' // TRIM(c_varnamebase) // '_woflx(:) - ' // &
                                TRIM(c_varnamebase) // '_boflx(:)) - '  // &
                                TRIM(c_varnamebase) // '_rea(:)'
  WRITE(iu,fmt_)
  WRITE(iu,fmt_)

  WRITE(iu,fmtc)   '! Maximum residuals'
  WRITE(iu,fmtc)   '! -----------------'
  WRITE(iu,fmt_)


  c_varnamebase = cp_max // TRIM(c_what_curr) // '_resid'
  WRITE(iu,fmt6)   'DO icompo = 1, ncompo'
  WRITE(iu,fmt6)   '  ' // TRIM(c_varnamebase) // '(icompo)'
  WRITE(iu,fmtcon) '   = MAX(' // TRIM(c_varnamebase) // '(icompo), '  // &
                        'ABS(' // cp_col // TRIM(c_what_curr) // '_resid(icompo)))'
  WRITE(iu,fmt6)   'ENDDO'
  WRITE(iu,fmt_)
  WRITE(iu,fmt_)

ENDDO



# ifdef CFG_DEBUG
  WRITE(iu,fmt0, ADVANCE='NO') '! '
  WRITE(iu,c_fmtinf_a) 'End of creation'
# endif

!===================!
! End of Operations !
!===================!


#ifdef CFG_VERBOSE
#if (CFG_VERBOSELEVEL == 2)
WRITE(jp_stdout,c_fmtinf_a) 'done'
#endif
#endif
#ifdef CFG_DEBUG
WRITE(jp_stddbg,c_fmtdbg_a) 'done'
#endif


RETURN

!-------------------------------------------------------------------------------
 END SUBROUTINE CMB_createSpreadResults
!-------------------------------------------------------------------------------



!-------------------------------------------------------------------------------
 SUBROUTINE CMB_createDeclareAndInit(iu_decl, iu_init, c_what, t_cmbdata_table)
!-------------------------------------------------------------------------------

USE MOD_UTILITIES, ONLY: EXPAND_TOKEN, EXPAND_2TOKEN, EXPAND_3TOKEN, LOWCASE

IMPLICIT NONE


! Dummy argument list variables
! -----------------------------

INTEGER,                          INTENT(IN)         :: iu_decl
INTEGER,                          INTENT(IN)         :: iu_init
CHARACTER(LEN=*), DIMENSION(:),   INTENT(IN), TARGET :: c_what
TYPE(CMBDATAP),   DIMENSION(:,:), INTENT(IN), TARGET :: t_cmbdata_table


! Local variables
! ---------------

CHARACTER(LEN=n_lmaxnamesgen)  :: c_consrvprop
CHARACTER(LEN=LEN(c_what))     :: c_what_curr
CHARACTER(LEN=LEN(cp_subsets)) :: c_set_curr, c_subset_curr
INTEGER                        :: n_data


CHARACTER(LEN=*), PARAMETER  :: cp_doubleprec = 'DOUBLE PRECISION :: '
CHARACTER(LEN=*), PARAMETER  :: cp_doubleprecdim = 'DOUBLE PRECISION, DIMENSION(ncompo) :: '

INTEGER :: i_what, i_subset
CHARACTER(LEN=n_lmaxidentif) :: c_varnamepart
CHARACTER(LEN=n_lxxlcodeline) :: c_linebegin


! Standard I/O related data
! -------------------------

CHARACTER(LEN=*), PARAMETER  :: c_fmtinf_a   = '("[[CMB_createDeclareAndInit]]: ", A)'
#ifdef CFG_DEBUG
CHARACTER(LEN=*), PARAMETER  :: c_fmtdbg_a   = '("DEBUG [[CMB_createDeclareAndInit]]: ", A)'
#endif

!=====================!
! End of declarations !
!=====================!

#ifdef CFG_VERBOSE
#if (CFG_VERBOSELEVEL == 2)
WRITE(jp_stdout,c_fmtinf_a) 'starting'
#endif
#endif
#ifdef CFG_DEBUG
WRITE(jp_stddbg,c_fmtdbg_a) 'starting'
#endif


!=====================!
! Begin of operations !
!=====================!


# ifdef CFG_DEBUG
  WRITE(iu_decl,fmt0, ADVANCE='NO') '! '
  WRITE(iu_decl,c_fmtinf_a) 'Start of creation'
  WRITE(iu_init,fmt0, ADVANCE='NO') '! '
  WRITE(iu_init,c_fmtinf_a) 'Start of creation'
# endif


WRITE(iu_decl,fmt_)
WRITE(iu_decl,fmt_)
WRITE(iu_decl,fmt0)   '! Variables related to the conservation properties'
WRITE(iu_decl,fmt0)   '! ================================================'
WRITE(iu_decl,fmt_)
WRITE(iu_decl,fmt0)   '! Variable name construction'
WRITE(iu_decl,fmt0)   '! --------------------------'
WRITE(iu_decl,fmt0)   '! ' // cp_col // '*      : in the current column'
WRITE(iu_decl,fmt0)   '! ' // cp_glo // '*      : globally integrated'
WRITE(iu_decl,fmt0)   '! ' // cp_max // '*      : maximum over all columns'
WRITE(iu_decl,fmt0)   '! *_rea  : reaction rate (production-minus-destruction)'
WRITE(iu_decl,fmt0)   '! *_woflx: flux across the water-DBL interface (positive downwards)'
WRITE(iu_decl,fmt0)   '! *_boflx: flux across the sediment bottom (positive downwards)'
WRITE(iu_decl,fmt_)
WRITE(iu_decl,fmt_)



DO i_what = 1, SIZE(c_what)

  CALL CMB_getInformationForOneWhat(t_cmbdata_table, c_what, i_what, &
                                    c_what_curr, c_consrvprop)


  WRITE(iu_decl,fmt0)   '! Local variables related to ' // TRIM(c_consrvprop)
  WRITE(iu_decl,fmt0)   '! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -'
  WRITE(iu_decl,fmt_)
  c_linebegin = cp_doubleprecdim // cp_col // TRIM(c_what_curr)
  WRITE(iu_decl,fmt6)   TRIM(c_linebegin) // '_xm_0'
  WRITE(iu_decl,fmt6)   TRIM(c_linebegin) // '_xm_1'
  WRITE(iu_decl,fmt6)   TRIM(c_linebegin) // '_woflx'
  WRITE(iu_decl,fmt6)   TRIM(c_linebegin) // '_boflx'
  WRITE(iu_decl,fmt6)   TRIM(c_linebegin) // '_rea'
  WRITE(iu_decl,fmt6)   TRIM(c_linebegin) // '_resid'
  WRITE(iu_decl,fmt_)

  c_linebegin = cp_doubleprecdim // cp_glo // TRIM(c_what_curr)
  WRITE(iu_decl,fmt6)   TRIM(c_linebegin) // '_xm_0'
  WRITE(iu_decl,fmt6)   TRIM(c_linebegin) // '_xm_1'
  WRITE(iu_decl,fmt6)   TRIM(c_linebegin) // '_woflx'
  WRITE(iu_decl,fmt6)   TRIM(c_linebegin) // '_boflx'
  WRITE(iu_decl,fmt6)   TRIM(c_linebegin) // '_rea'
  WRITE(iu_decl,fmt6)   TRIM(c_linebegin) // '_resid'
  WRITE(iu_decl,fmt_)

  c_linebegin = cp_doubleprecdim // cp_max // TRIM(c_what_curr)
  WRITE(iu_decl,fmt6)   TRIM(c_linebegin) // '_resid'
  WRITE(iu_decl,fmt_)
  WRITE(iu_decl,fmt_)

  c_linebegin = cp_glo // TRIM(c_what_curr) 
  WRITE(iu_init,fmt_)
  WRITE(iu_init,fmt6)   TRIM(c_linebegin) // '_xm_0(:)     = 0D0'
  WRITE(iu_init,fmt6)   TRIM(c_linebegin) // '_xm_1(:)     = 0D0'
  WRITE(iu_init,fmt_)
  WRITE(iu_init,fmt6)   TRIM(c_linebegin) // '_woflx(:)    = 0D0'
  WRITE(iu_init,fmt6)   TRIM(c_linebegin) // '_boflx(:)    = 0D0'
  WRITE(iu_init,fmt6)   TRIM(c_linebegin) // '_rea(:)      = 0D0'
  WRITE(iu_init,fmt6)   TRIM(c_linebegin) // '_resid(:)    = 0D0'
  WRITE(iu_init,fmt_)

  c_linebegin = cp_max // TRIM(c_what_curr) 
  WRITE(iu_init,fmt6)   TRIM(c_linebegin) // '_resid(:)    = 0D0'
  WRITE(iu_init,fmt_)
  WRITE(iu_init,fmt_)


  do_subsets: DO i_subset = 1, jp_nsubsets

    c_subset_curr = cp_subsets(i_subset)
    c_set_curr    = cp_sets(i_subset)

    n_data = t_cmbdata_table(i_what, i_subset)%n_data

    IF(n_data == 0) THEN
#     ifdef CFG_DEBUG
      WRITE(iu_decl,fmt0)  '! *_' // TRIM(c_set_curr) // &
                              '_' // TRIM(c_subset_curr) // &
                              '_' // TRIM(c_what_curr)     // & 
                            ' : ' // TRIM(c_consrvprop)  // &
                          ' in "' // TRIM(c_set_curr) // &
                              ':' // TRIM(c_subset_curr) // '"'
      WRITE(iu_decl,fmt_)
      WRITE(iu_decl,fmt0) '!   Nothing to consider'
      WRITE(iu_decl,fmt_)
      WRITE(iu_decl,fmt_)
#     endif
      CYCLE do_subsets              ! No need to declare or generate
    ENDIF

    WRITE(iu_decl,fmt0)  '! *_' // TRIM(c_set_curr) // &
                            '_' // TRIM(c_subset_curr) // &
                            '_' // TRIM(c_what_curr)     // & 
                          ' : ' // TRIM(c_consrvprop)  // &
                        ' in "' // TRIM(c_set_curr) // &
                            ':' // TRIM(c_subset_curr) // '"'
    WRITE(iu_decl,fmt_)

    c_linebegin =  cp_doubleprec // cp_col // TRIM(c_what_curr) // '_' // TRIM(c_set_curr) // '_' // TRIM(c_subset_curr)
    WRITE(iu_decl,fmt6)   TRIM(c_linebegin) // '_xm_0'
    WRITE(iu_decl,fmt6)   TRIM(c_linebegin) // '_xm_1'
    WRITE(iu_decl,fmt_)
    WRITE(iu_decl,fmt6)   TRIM(c_linebegin) // '_woflx'
    WRITE(iu_decl,fmt6)   TRIM(c_linebegin) // '_boflx'
    WRITE(iu_decl,fmt6)   TRIM(c_linebegin) // '_rea'
    WRITE(iu_decl,fmt6)   TRIM(c_linebegin) // '_resid'
    WRITE(iu_decl,fmt_)

    c_linebegin =  cp_doubleprec // cp_glo // TRIM(c_what_curr) // '_' // TRIM(c_set_curr) // '_' // TRIM(c_subset_curr)
    WRITE(iu_decl,fmt6)   TRIM(c_linebegin) // '_xm_0'
    WRITE(iu_decl,fmt6)   TRIM(c_linebegin) // '_xm_1'
    WRITE(iu_decl,fmt_)
    WRITE(iu_decl,fmt6)   TRIM(c_linebegin) // '_woflx'
    WRITE(iu_decl,fmt6)   TRIM(c_linebegin) // '_boflx'
    WRITE(iu_decl,fmt6)   TRIM(c_linebegin) // '_rea'
    WRITE(iu_decl,fmt6)   TRIM(c_linebegin) // '_resid'
    WRITE(iu_decl,fmt_)

    c_linebegin =  cp_doubleprec // cp_max // TRIM(c_what_curr) // '_' // TRIM(c_set_curr) // '_' // TRIM(c_subset_curr)
    WRITE(iu_decl,fmt6)   TRIM(c_linebegin) // '_resid'
    WRITE(iu_decl,fmt_)
    WRITE(iu_decl,fmt_)

                                    ! Initialisation parts
                                    ! No need to initialise the col_* variables

    c_linebegin =  cp_glo // TRIM(c_what_curr) // '_' // TRIM(c_set_curr) // '_' // TRIM(c_subset_curr)
    WRITE(iu_init,fmt6)   TRIM(c_linebegin) // '_xm_0      = 0D0'
    WRITE(iu_init,fmt6)   TRIM(c_linebegin) // '_xm_1      = 0D0'
    WRITE(iu_init,fmt_)
    WRITE(iu_init,fmt6)   TRIM(c_linebegin) // '_woflx     = 0D0'
    WRITE(iu_init,fmt6)   TRIM(c_linebegin) // '_boflx     = 0D0'
    WRITE(iu_init,fmt6)   TRIM(c_linebegin) // '_rea       = 0D0'
    WRITE(iu_init,fmt6)   TRIM(c_linebegin) // '_resid     = 0D0'
    WRITE(iu_init,fmt_)

    c_linebegin =  cp_max // TRIM(c_what_curr) // '_' // TRIM(c_set_curr) // '_' // TRIM(c_subset_curr)
    WRITE(iu_init,fmt6)   TRIM(c_linebegin) // '_resid     = 0D0'
    WRITE(iu_init,fmt_)
    WRITE(iu_init,fmt_)

  ENDDO do_subsets

  WRITE(iu_init,fmt_)

  WRITE(iu_decl,fmt_)

ENDDO

#ifdef CFG_DEBUG
  WRITE(iu_decl,fmt0, ADVANCE='NO') '! '
  WRITE(iu_decl,c_fmtinf_a) 'End of creation'
  WRITE(iu_init,fmt0, ADVANCE='NO') '! '
  WRITE(iu_init,c_fmtinf_a) 'End of creation'
#endif

!===================!
! End of operations !
!===================!

#ifdef CFG_VERBOSE
#if (CFG_VERBOSELEVEL == 2)
WRITE(jp_stdout,c_fmtinf_a) 'done'
#endif
#endif
#ifdef CFG_DEBUG
WRITE(jp_stddbg,c_fmtdbg_a) 'done'
#endif

RETURN

!-------------------------------------------------------------------------------
 END SUBROUTINE CMB_createDeclareAndInit
!-------------------------------------------------------------------------------



!-------------------------------------------------------------------------------
SUBROUTINE CMB_createSigmaFunctions(iu, c_what, t_cmbdata_table)
!-------------------------------------------------------------------------------

USE MOD_UTILITIES, ONLY: UPCASE, LOWCASE, EXPAND_TOKEN

IMPLICIT NONE


! Configuration parameters for this subroutine
! --------------------------------------------

! n_lmaxtounroll must be greater than 0 
INTEGER, PARAMETER :: n_lmaxtounroll = 5


! Dummy argument list variables
! -----------------------------

INTEGER,                          INTENT(IN)         :: iu
CHARACTER(LEN=*), DIMENSION(:),   INTENT(IN)         :: c_what
TYPE(CMBDATAP),   DIMENSION(:,:), INTENT(IN), TARGET :: t_cmbdata_table


! Local variables
! ---------------

CHARACTER(LEN=n_lmaxidentif)   :: c_funcname
CHARACTER(LEN=n_lmaxidentif)   :: c_ioname
CHARACTER(LEN=n_lmaxformat)    :: c_fmttmp
CHARACTER(LEN=LEN(cp_subsets)) :: c_subset_curr
CHARACTER(LEN=LEN(cp_subsets)) :: c_set_curr
CHARACTER(LEN=LEN(c_what))     :: c_what_curr

TYPE(COMPOINFO),   POINTER :: t_compo_work
INTEGER :: i, n_data, i_what, i_subset
CHARACTER(LEN=n_lmaxnamesgen) :: c_consrvprop

CHARACTER(LEN=n_lmaxshortid+2), DIMENSION(:), ALLOCATABLE :: c_iolist

TYPE(CMBDATA), POINTER :: t_cmbdata_node



CHARACTER(LEN=*), PARAMETER  :: c_fmtinf_a   = '("[[CMB_createSigmaFunctions]]: ", A)'
CHARACTER(LEN=*), PARAMETER  :: c_fmterr_a   = '("[[CMB_createSigmaFunctions]] error: ", A)'
#ifdef CFG_DEBUG
CHARACTER(LEN=*), PARAMETER  :: c_fmtdbg_a   = '("DEBUG [[CMB_createSigmaFunctions]]: ", A)'
#endif

!=====================!
! End of declarations !
!=====================!


#ifdef CFG_VERBOSE
#if (CFG_VERBOSELEVEL == 2)
WRITE(jp_stdout,c_fmtinf_a) 'starting'
#endif
#endif
#ifdef CFG_DEBUG
WRITE(jp_stddbg,c_fmtdbg_a) 'starting'
#endif


!=====================!
! Start of operations !
!=====================!

#ifdef CFG_DEBUG
  WRITE(iu,fmt0, ADVANCE='NO') '! '
  WRITE(iu,c_fmtinf_a) 'Start of creation'
#endif

DO i_what = 1, SIZE(c_what)

  CALL CMB_getInformationForOneWhat(t_cmbdata_table, c_what, i_what, &
                                    c_what_curr, c_consrvprop)

  DO i_subset = 1, jp_nsubsets

    IF(cp_subsets(i_subset) == cp_cmb_total) THEN
                                    ! If the current subphase is the 'total'
                                    ! one, switch to new set
      c_set_curr = LOWCASE(ADJUSTL(cp_sets(i_subset)))
      CYCLE                         ! Never create SIGMA function for 'total'
    ENDIF

    c_subset_curr = LOWCASE(ADJUSTL(cp_subsets(i_subset)))
    c_funcname = 'SIGMA_' // TRIM(c_what_curr) // '_' // TRIM(c_set_curr) // '_' // TRIM(c_subset_curr)

    n_data = t_cmbdata_table(i_what, i_subset)%n_data

    IF(n_data == 0) CYCLE           ! No need to generate a SIGMA function
                                    ! if there are no components
    ALLOCATE(c_iolist(n_data))
    i = 1
    t_cmbdata_node => t_cmbdata_table(i_what, i_subset)%ptr
    DO WHILE(i <= n_data)
      c_iolist(i) = cp_prefix_io // ADJUSTL(t_cmbdata_node%shortid)
      i = i + 1
      t_cmbdata_node => t_cmbdata_node%next
    ENDDO

                                    ! Write out the source code to
                                    ! UNIT=iu
    WRITE(iu,fmt_)
    WRITE(iu,fmt_)
    WRITE(iu,fmt0)   '!======================================================================='
    WRITE(iu,fmt6)   'DOUBLE PRECISION FUNCTION ' // TRIM(UPCASE(c_funcname)) // '(array)'
    WRITE(iu,fmt0)   '!======================================================================='
    WRITE(iu,fmt_)
    WRITE(iu,fmt6)   'USE mod_indexparam'
    WRITE(iu,fmt6)   'USE mod_materialcharas'
    WRITE(iu,fmt_)
    WRITE(iu,fmt6)   'IMPLICIT NONE'
    WRITE(iu,fmt_)
    WRITE(iu,fmt6)   'DOUBLE PRECISION, DIMENSION(ncompo), INTENT(IN) :: array'
    WRITE(iu,fmt_)
    WRITE(iu,fmt_)



    IF(n_data > n_lmaxtounroll) THEN

      WRITE(iu,fmt6,ADVANCE='NO')   'INTEGER, PARAMETER :: n = '
      WRITE(iu,'(I0)') n_data
      WRITE(iu,fmt6)   'INTEGER, PARAMETER, DIMENSION(n) :: iolist'
      WRITE(iu,fmtcon, ADVANCE='NO') '  = '

                                    ! Adjust format string for indenting
                                    ! by an extra 4 blanks after the
                                    ! 6 initial reserved columns
      CALL EXPAND_TOKEN(fmtconind, '@', '4X,', c_fmttmp)
                                    ! Write out array constructor with
                                    ! io_xxxx varnames
      CALL WRITE_ARRAY_CONSTRUCTOR(iu, c_fmttmp, c_iolist, n_data, CFG_ITPL)


    ENDIF


    WRITE(iu,fmt_)
    WRITE(iu,fmt_)


    WRITE(iu,fmt6) TRIM(LOWCASE(c_funcname)) // ' =   '
    IF(n_data <= n_lmaxtounroll) THEN

      WRITE(iu,fmtcon) '    array(' // TRIM(c_iolist(1)) // ')'

      DO i = 2, n_data
        WRITE(iu,fmtcon) '  + array(' // TRIM(c_iolist(i)) // ')'
      ENDDO

    ELSE

      WRITE(iu,'(A)')  'SUM(array(iolist(:)))'

    ENDIF
    WRITE(iu,fmt_)



    WRITE(iu,fmt_)
    WRITE(iu,fmt6)   'RETURN'
    WRITE(iu,fmt_)
    WRITE(iu,fmt0)   '!======================================================================='
    WRITE(iu,fmt6)   'END FUNCTION ' // TRIM(UPCASE(c_funcname))
    WRITE(iu,fmt0)   '!======================================================================='
    WRITE(iu,fmt_)



    DEALLOCATE(c_iolist)

  ENDDO

ENDDO


#ifdef CFG_DEBUG
  WRITE(iu,fmt0, ADVANCE='NO') '! '
  WRITE(iu,c_fmtinf_a) 'End of creation'
#endif

!=====================!
! End of operations !
!=====================!


#ifdef CFG_VERBOSE
#if (CFG_VERBOSELEVEL == 2)
WRITE(jp_stdout,c_fmtinf_a) 'done'
#endif
#endif
#ifdef CFG_DEBUG
WRITE(jp_stddbg,c_fmtdbg_a) 'done'
#endif

RETURN

!-------------------------------------------------------------------------------
 END SUBROUTINE CMB_createSigmaFunctions
!-------------------------------------------------------------------------------



!-------------------------------------------------------------------------------
 SUBROUTINE CMB_createPrintResiduals_Rea(iu_maxr, iu_glor, iu_grea, &
                                         c_what, t_cmbdata_table)
!-------------------------------------------------------------------------------

USE MOD_UTILITIES, ONLY: L_STR_EQ_DBL

IMPLICIT NONE


! Dummy argument list variables
! -----------------------------

INTEGER,                          INTENT(IN)         :: iu_maxr
INTEGER,                          INTENT(IN)         :: iu_glor
INTEGER,                          INTENT(IN)         :: iu_grea
CHARACTER(LEN=*), DIMENSION(:),   INTENT(IN)         :: c_what
TYPE(CMBDATAP),   DIMENSION(:,:), INTENT(IN), TARGET :: t_cmbdata_table


! Local variables
! ---------------

INTEGER :: i_what, i_subset
CHARACTER(LEN=LEN(c_what)) :: c_what_curr
CHARACTER(LEN=n_lmaxnamesgen) :: c_consrvprop

TYPE(CMBDATA), POINTER       :: t_cmbdata_node
TYPE(COMPOINFO), POINTER     :: t_compo_node
CHARACTER(LEN=n_lmaxnamesgen+3) :: c_fmtname
CHARACTER(LEN=n_lmaxnamesgen+3) :: c_fmtline
CHARACTER(LEN=n_lmaxnamesgen)   :: c_name
INTEGER :: i, j
CHARACTER(LEN=n_lmaxidentif) :: c_varname
CHARACTER(LEN=n_lmaxidentif) :: c_varnamebegin
CHARACTER(LEN=n_lmaxidentif) :: c_varnameend
INTEGER :: iu



! Standard I/O related data
! -------------------------

CHARACTER(LEN=*), PARAMETER  :: c_fmtinf_a   = '("[[CMB_createPrintResiduals_Rea]]: ", A)'
CHARACTER(LEN=*), PARAMETER  :: c_fmterr_a   = '("[[CMB_createPrintResiduals_Rea]] error: ", A)'
#ifdef CFG_DEBUG
CHARACTER(LEN=*), PARAMETER  :: c_fmtdbg_a   = '("DEBUG [[CMB_createPrintResiduals_Rea]]: ", A)'
#endif

!=====================!
! End of declarations !
!=====================!


#ifdef CFG_VERBOSE
#if (CFG_VERBOSELEVEL == 2)
WRITE(jp_stdout,c_fmtinf_a) 'starting'
#endif
#endif
#ifdef CFG_DEBUG
WRITE(jp_stddbg,c_fmtdbg_a) 'starting'
#endif


!=====================!
! Start of operations !
!=====================!

#ifdef CFG_DEBUG
  WRITE(iu_maxr,'("!",35(" -"))')
  WRITE(iu_maxr,fmt0, ADVANCE='NO') '! '
  WRITE(iu_maxr,c_fmtinf_a) 'Start of creation'
  WRITE(iu_maxr,'("!",35(" -"))')
  WRITE(iu_glor,'("!",35(" -"))')
  WRITE(iu_glor,fmt0, ADVANCE='NO') '! '
  WRITE(iu_glor,c_fmtinf_a) 'Start of creation'
  WRITE(iu_glor,'("!",35(" -"))')
  WRITE(iu_grea,'("!",35(" -"))')
  WRITE(iu_grea,fmt0, ADVANCE='NO') '! '
  WRITE(iu_grea,c_fmtinf_a) 'Start of creation'
  WRITE(iu_grea,'("!",35(" -"))')
#endif


WRITE(c_fmtname,'("(""''"",3X, A",I0,",""''"")")') n_lmaxnamesgen
WRITE(c_fmtline,'("(""''"",3X,",I0,"(""-""),""''"")")') n_lmaxnamesgen


DO i_what = 1, SIZE(c_what)

  CALL CMB_getInformationForOneWhat(t_cmbdata_table, c_what, i_what,       &
                                    c_what_curr, c_consrvprop)

  WRITE(iu_maxr,fmt_)
  WRITE(iu_maxr,fmt6) 'WRITE(jp_stdlog,''()'')'
  WRITE(iu_maxr,fmt6) 'WRITE(jp_stdlog,''()'')'

  WRITE(iu_glor,fmt_)
  WRITE(iu_glor,fmt6) 'WRITE(jp_stdlog,''()'')'
  WRITE(iu_glor,fmt6) 'WRITE(jp_stdlog,''()'')'

  WRITE(iu_grea,fmt_)
  WRITE(iu_grea,fmt6) 'WRITE(jp_stdlog,''()'')'
  WRITE(iu_grea,fmt6) 'WRITE(jp_stdlog,''()'')'

  SELECT CASE(c_what_curr)
  CASE(cp_shortid_alk)
    WRITE(iu_maxr,fmt6) 'WRITE(jp_stdlog,fmta) '' * Alkalinity (eq/m2)'''
    WRITE(iu_glor,fmt6) 'WRITE(jp_stdlog,fmta) '' * Alkalinity (eq)'''
    WRITE(iu_grea,fmt6) 'WRITE(jp_stdlog,fmta) '' * Alkalinity (eq/dt)'''
  CASE('c')
    WRITE(iu_maxr,fmt6) 'WRITE(jp_stdlog,fmta) '' * Carbon (mol C/m2)'''
    WRITE(iu_glor,fmt6) 'WRITE(jp_stdlog,fmta) '' * Carbon (mol C)'''
    WRITE(iu_grea,fmt6) 'WRITE(jp_stdlog,fmta) '' * Carbon (mol C/dt)'''
  CASE('o2')
    WRITE(iu_maxr,fmt6) 'WRITE(jp_stdlog,fmta) '' * Oxygen (mol O2/m2)'''
    WRITE(iu_glor,fmt6) 'WRITE(jp_stdlog,fmta) '' * Oxygen (mol O2)'''
    WRITE(iu_grea,fmt6) 'WRITE(jp_stdlog,fmta) '' * Oxygen (mol O2/dt)'''
   CASE('si')
    WRITE(iu_maxr,fmt6) 'WRITE(jp_stdlog,fmta) '' * Silicon (mol Si/m2)'''
    WRITE(iu_glor,fmt6) 'WRITE(jp_stdlog,fmta) '' * Silicon (mol Si)'''
    WRITE(iu_grea,fmt6) 'WRITE(jp_stdlog,fmta) '' * Silicon (mol Si/dt)'''
  CASE DEFAULT
    WRITE(jp_stderr,c_fmterr_a) 'Unknown <c_what> value ' // TRIM(c_what_curr) // &
      ' -- please extend me! Aborting'
    CALL ABORT()
  END SELECT

  WRITE(iu_maxr,fmt6) 'WRITE(jp_stdlog,''()'')'
  WRITE(iu_grea,fmt6) 'WRITE(jp_stdlog,''()'')'


  t_cmbdata_node => t_cmbdata_table(i_what, jp_cmbbulk_porew)%ptr
  scan4porew: DO WHILE(ASSOCIATED(t_cmbdata_node))

    c_shortid = t_cmbdata_node%shortid
    t_compo_node => COMPOINFO_getNodeByShortid(t_compo_chain, c_shortid)

    DO j = 1, 3

      SELECT CASE(j)
      CASE(1)
        iu = iu_maxr
        c_varname = cp_max // TRIM(c_what_curr) // '_resid'
      CASE(2)
        iu = iu_glor
        c_varname = cp_glo // TRIM(c_what_curr) // '_resid'
      CASE(3)
        iu = iu_grea
        c_varname = cp_glo // TRIM(c_what_curr) // '_rea'
      END SELECT

      WRITE(iu,fmt6,ADVANCE='NO')   'WRITE(jp_stdlog,fmt1) '
      WRITE(iu,c_fmtname,ADVANCE='NO') ADJUSTL(t_compo_node%name)
      WRITE(iu,'(",")')

      WRITE(iu,fmtcon,ADVANCE='NO')
      WRITE(iu,'(23X,A)') TRIM(c_varname) // '(' // cp_prefix_io // TRIM(c_shortid) // ')'


    ENDDO

    t_cmbdata_node => t_cmbdata_node%next

  ENDDO scan4porew


  DO j = 1, 3

    SELECT CASE(j)
    CASE(1)
      iu = iu_maxr
      c_varnamebegin = cp_max // TRIM(c_what_curr)
      c_varnameend   = '_resid'
    CASE(2)
      iu = iu_glor
      c_varnamebegin = cp_glo // TRIM(c_what_curr)
      c_varnameend   = '_resid'
    CASE(3)
      iu = iu_grea
      c_varnamebegin = cp_glo // TRIM(c_what_curr)
      c_varnameend   = '_rea'
    END SELECT

    IF(t_cmbdata_table(i_what, jp_cmbbulk_porew)%n_data > 1) THEN

      WRITE(iu,fmt6,ADVANCE='NO') 'WRITE(jp_stdlog,fmtl) '
      WRITE(iu,c_fmtline)
      WRITE(iu,fmt6,ADVANCE='NO') 'WRITE(jp_stdlog,fmt1) '
      c_name = 'Sigma porew'
      WRITE(iu,c_fmtname,ADVANCE='NO')  ADJUSTL(c_name)
      WRITE(iu,'(",")')

      WRITE(iu,fmtcon,ADVANCE='NO')
      WRITE(iu,'(23X,A)') TRIM(c_varnamebegin) // '_bulk_porew' // TRIM(c_varnameend)

      WRITE(iu,fmt6) 'WRITE(jp_stdlog,''()'')'
    
    ENDIF

  ENDDO

  t_cmbdata_node => t_cmbdata_table(i_what, jp_cmbbulk_solid)%ptr
  scan4solid: DO WHILE(ASSOCIATED(t_cmbdata_node))

    c_shortid = t_cmbdata_node%shortid
    t_compo_node => COMPOINFO_getNodeByShortid(t_compo_chain, c_shortid)


    DO j = 1, 3

      SELECT CASE(j)
      CASE(1)
        iu = iu_maxr
        c_varname = cp_max // TRIM(c_what_curr) // '_resid'
      CASE(2)
        iu = iu_glor
        c_varname = cp_glo // TRIM(c_what_curr) // '_resid'
      CASE(3)
        iu = iu_grea
        c_varname = cp_glo // TRIM(c_what_curr) // '_rea'
      END SELECT

      WRITE(iu,fmt6,ADVANCE='NO')   'WRITE(jp_stdlog,fmt1) '
      WRITE(iu,c_fmtname,ADVANCE='NO') ADJUSTL(t_compo_node%name)
      WRITE(iu,'(",")')

      WRITE(iu,fmtcon,ADVANCE='NO')
      WRITE(iu,'(23X,A)') TRIM(c_varname) // '(' // cp_prefix_io // TRIM(c_shortid) // ')'

    ENDDO

    t_cmbdata_node => t_cmbdata_node%next

  ENDDO scan4solid


  DO j = 1, 3

    SELECT CASE(j)
    CASE(1)
      iu = iu_maxr
      c_varnamebegin = cp_max // TRIM(c_what_curr)
      c_varnameend   = '_resid'
    CASE(2)
      iu = iu_glor
      c_varnamebegin = cp_glo // TRIM(c_what_curr)
      c_varnameend   = '_resid'
    CASE(3)
      iu = iu_grea
      c_varnamebegin = cp_glo // TRIM(c_what_curr)
      c_varnameend   = '_rea'
    END SELECT


    IF(t_cmbdata_table(i_what, jp_cmbbulk_solid)%n_data > 1) THEN

      WRITE(iu,fmt6,ADVANCE='NO') 'WRITE(jp_stdlog,fmtl) '
      WRITE(iu,c_fmtline)
      WRITE(iu,fmt6,ADVANCE='NO') 'WRITE(jp_stdlog,fmt1) '
      c_name = 'Sigma solid'
      WRITE(iu,c_fmtname,ADVANCE='NO')  c_name
      WRITE(iu,'(",")')

      WRITE(iu,fmtcon,ADVANCE='NO')
      WRITE(iu,'(23X,A)') TRIM(c_varnamebegin) // '_bulk_solid' // TRIM(c_varnameend)
   
      WRITE(iu,fmt6) 'WRITE(jp_stdlog,''()'')'
    
    ENDIF

    WRITE(iu,fmt6,ADVANCE='NO') 'WRITE(jp_stdlog,fmtl) '
    WRITE(iu,c_fmtline)
    WRITE(iu,fmt6,ADVANCE='NO') 'WRITE(jp_stdlog,fmt1) '
    c_name = 'Sigma total'
    WRITE(iu,c_fmtname,ADVANCE='NO')  c_name
    WRITE(iu,'(",")')

    WRITE(iu,fmtcon,ADVANCE='NO')
    WRITE(iu,'(23X,A)') TRIM(c_varnamebegin) // '_bulk_total' // TRIM(c_varnameend)

  ENDDO

ENDDO


#ifdef CFG_DEBUG
WRITE(iu_maxr,'("!",35(" -"))')
WRITE(iu_maxr,fmt0, ADVANCE='NO') '! '
WRITE(iu_maxr,c_fmtinf_a) 'End of creation'
WRITE(iu_maxr,'("!",35(" -"))')
WRITE(iu_glor,'("!",35(" -"))')
WRITE(iu_glor,fmt0, ADVANCE='NO') '! '
WRITE(iu_glor,c_fmtinf_a) 'End of creation'
WRITE(iu_glor,'("!",35(" -"))')
WRITE(iu_grea,'("!",35(" -"))')
WRITE(iu_grea,fmt0, ADVANCE='NO') '! '
WRITE(iu_grea,c_fmtinf_a) 'End of creation'
WRITE(iu_grea,'("!",35(" -"))')
#endif

!=====================!
! End of operations !
!=====================!


#ifdef CFG_VERBOSE
#if (CFG_VERBOSELEVEL == 2)
WRITE(jp_stdout,c_fmtinf_a) 'done'
#endif
#endif
#ifdef CFG_DEBUG
WRITE(jp_stddbg,c_fmtdbg_a) 'done'
#endif

RETURN

!-------------------------------------------------------------------------------
 END SUBROUTINE CMB_createPrintResiduals_Rea
!-------------------------------------------------------------------------------



!-------------------------------------------------------------------------------
 SUBROUTINE CMB_createColumnSums(iu, c_what, t_cmbdata_table)
!-------------------------------------------------------------------------------

USE MOD_UTILITIES, ONLY: UPCASE

IMPLICIT NONE


! Dummy argument list variables
! -----------------------------

INTEGER,                          INTENT(IN)         :: iu
CHARACTER(LEN=*), DIMENSION(:),   INTENT(IN)         :: c_what
TYPE(CMBDATAP),   DIMENSION(:,:), INTENT(IN), TARGET :: t_cmbdata_table


! Local variables
! ---------------

INTEGER :: i_what, i_subset
INTEGER :: n_data
CHARACTER(LEN=LEN(c_what))    :: c_what_curr
CHARACTER(LEN=n_lmaxnamesgen) :: c_consrvprop
CHARACTER(LEN=n_lmaxidentif)  :: c_funcname
CHARACTER(LEN=LEN(cp_subsets)) :: c_set_curr, c_set_prev
CHARACTER(LEN=LEN(cp_subsets)) :: c_subset_curr
TYPE(CMBDATA), POINTER        :: t_cmbdata_node
CHARACTER(LEN=20)             :: c_fmttmp
CHARACTER(LEN=n_lmaxidentif)  :: c_type
CHARACTER(LEN=n_lmaxidentif)  :: c_varname
CHARACTER(LEN=n_lmaxidentif)  :: c_varnameend
CHARACTER(LEN=1)              :: c_0or1

INTEGER :: j
LOGICAL :: l_skip

CHARACTER(LEN=*), PARAMETER  :: c_fmtinf_a   = '("[[CMB_createColumnSums]]: ", A)'
#ifdef CFG_DEBUG
CHARACTER(LEN=*), PARAMETER  :: c_fmtdbg_a   = '("DEBUG [[CMB_createColumnSums]]: ", A)'
#endif

!=====================!
! End of declarations !
!=====================!


#ifdef CFG_VERBOSE
#if (CFG_VERBOSELEVEL == 2)
WRITE(jp_stdout,c_fmtinf_a) 'starting'
#endif
#endif
#ifdef CFG_DEBUG
WRITE(jp_stddbg,c_fmtdbg_a) 'starting'
#endif


!=====================!
! Start of operations !
!=====================!

#ifdef CFG_DEBUG
WRITE(iu,fmt0, ADVANCE='NO') '! '
WRITE(iu,c_fmtinf_a) 'Start of creation'
#endif

WRITE(iu,fmt_)

DO i_what = 1, SIZE(c_what)

  CALL CMB_getInformationForOneWhat(t_cmbdata_table, c_what, i_what,   &
                                    c_what_curr, c_consrvprop)

  WRITE(iu,fmtc)   '! Sums for ' // TRIM(ADJUSTL(c_consrvprop))
  WRITE(iu,fmtc, ADVANCE='NO') &
                   '! ---------'
  WRITE(c_fmttmp,'("(",I0,"(""-"")",")")') LEN_TRIM(ADJUSTL(c_consrvprop))
  WRITE(iu,c_fmttmp)
  WRITE(iu,fmt_)

!~   WRITE(iu,fmtc)   '! Total ' // TRIM(ADJUSTL(c_what(i_what))) // ' contents'
!~ 
!~ 
!~   DO i_subset = 1, jp_nsubsets
!~ 
!~     c_subset_curr = LOWCASE(ADJUSTL(cp_subsets(i_subset)))
!~ 
!~     IF(c_subset_curr == cp_cmb_total) THEN
!~ 
!~       WRITE(iu,fmt_)
!~ 
!~     ELSE
!~ 
!~       c_set_curr    = LOWCASE(ADJUSTL(cp_sets(i_subset)))
!~       c_funcname = 'SIGMA_' // TRIM(c_set_curr)    // &
!~                         '_' // TRIM(c_subset_curr) // &
!~                         '_' // TRIM(c_what_curr)
!~ 
!~       c_varname = cp_col // TRIM(c_what_curr)   // &
!~                      '_' // TRIM(c_set_curr)    // &
!~                      '_' // TRIM(c_subset_curr)
!~       WRITE(iu,fmt6)  '  ' // TRIM(c_varname)   // '_xm_0 = ' // &
!~         TRIM(UPCASE(c_funcname)) // '(' // cp_col // TRIM(c_what_curr) // '_xm_0(:))'
!~ 
!~       WRITE(iu,fmt6)  '  ' // TRIM(c_varname)   // '_xm_1 = ' // &
!~         TRIM(UPCASE(c_funcname)) // '(' // cp_col // TRIM(c_what_curr) // '_xm_1(:))'
!~ 
!~     END IF
!~ 
!~   ENDDO
!~ 
!~   WRITE(iu, fmt_)
!~ 
!~   DO j = 0, 1
!~ 
!~     SELECT CASE(j)
!~     CASE(0)
!~       c_0or1 = '0'
!~     CASE(1)
!~       c_0or1 = '1'
!~     END SELECT
!~ 
!~     i_subset = 1
!~     DO WHILE(i_subset <= jp_nsubsets)
!~ 
!~       c_subset_curr = LOWCASE(ADJUSTL(cp_subsets(i_subset)))
!~       c_set_curr    = LOWCASE(ADJUSTL(cp_sets(i_subset)))
!~ 
!~       c_varname     = cp_col // TRIM(c_what_curr)   // &
!~                       '_'    // TRIM(c_set_curr)    // &
!~                       '_'    // TRIM(c_subset_curr) // &
!~                       '_xm_' // c_0or1
!~ 
!~       IF(c_subset_curr == cp_cmb_total) THEN
!~ 
!~         IF(i_subset < jp_nsubsets) THEN
!~ 
!~           WRITE(iu,fmt6)       '  ' // TRIM(c_varname)
!~ 
!~           i_subset = i_subset + 1
!~           c_subset_curr = LOWCASE(ADJUSTL(cp_subsets(i_subset)))
!~           c_varname     = cp_col // TRIM(c_what_curr)   // &
!~                           '_'    // TRIM(c_set_curr)    // &
!~                           '_'    // TRIM(c_subset_curr) // &
!~                           '_xm_' // c_0or1
!~     
!~           WRITE(iu,fmtcon) '   =   ' // TRIM(c_varname)
!~           i_subset = i_subset + 1
!~ 
!~         ENDIF
!~ 
!~       ELSE
!~ 
!~         WRITE(iu,fmtcon) '     + ' // TRIM(c_varname)
!~         i_subset = i_subset + 1
!~ 
!~       ENDIF
!~ 
!~     ENDDO
!~ 
!~   ENDDO
!~ 
!~ 
!~   WRITE(iu, fmt_)


  DO j = 1, 6

    SELECT CASE(j)
    CASE(1)
      WRITE(iu,fmtc)   '! Total ' // TRIM(ADJUSTL(c_what(i_what))) // ' contents'
      c_type = 'xm_0'
    CASE(2)
      c_type = 'xm_1'
    CASE(3)
      WRITE(iu,fmtc)   '! Total ' // TRIM(ADJUSTL(c_what(i_what))) // ' fluxes'
      c_type = 'woflx'
    CASE(4)
      c_type = 'boflx'
    CASE(5)
      WRITE(iu,fmtc)   '! Total ' // TRIM(ADJUSTL(c_what(i_what))) // ' reacted'
      c_type = 'rea'
    CASE(6)
      WRITE(iu,fmtc)   '! Column and maximum residuals '
      WRITE(iu,fmtc)   '! for ' // TRIM(ADJUSTL(c_what(i_what))) // ' carrying materials'
      c_type = 'resid'
    END SELECT


    do_subsets: DO i_subset = 1, jp_nsubsets

      n_data = t_cmbdata_table(i_what, i_subset)%n_data
      IF (n_data == 0) CYCLE do_subsets

      c_subset_curr = LOWCASE(ADJUSTL(cp_subsets(i_subset)))

      IF(c_subset_curr == cp_cmb_total) THEN

        WRITE(iu,fmt_)              ! print out blank line when new set starts

      ELSE

        c_set_curr    = LOWCASE(ADJUSTL(cp_sets(i_subset)))

        c_funcname = 'SIGMA_' // TRIM(c_what_curr)   // &
                          '_' // TRIM(c_set_curr)    // &
                          '_' // TRIM(c_subset_curr)
                         

        c_varname  = cp_col   // TRIM(c_what_curr)   // &
                     '_'      // TRIM(c_set_curr)    // &
                     '_'      // TRIM(c_subset_curr) // &
                     '_'      // TRIM(c_type)

        WRITE(iu,fmt6)   '  ' // TRIM(c_varname)  // &
                        ' = ' // TRIM(UPCASE(c_funcname)) // &
                          '(' // cp_col // TRIM(c_what_curr) // &
                                    '_' // TRIM(c_type) // '(:))'
      ENDIF

    ENDDO do_subsets

    WRITE(iu,fmt_)

    i_subset = 1
    DO WHILE(i_subset <= jp_nsubsets)

      n_data = t_cmbdata_table(i_what, i_subset)%n_data
      IF (n_data == 0) THEN
        l_skip = .TRUE.
      ELSE
        l_skip = .FALSE.
      ENDIF

      c_subset_curr = LOWCASE(ADJUSTL(cp_subsets(i_subset)))
      c_set_curr    = LOWCASE(ADJUSTL(cp_sets(i_subset)))

      c_varname     = cp_col // TRIM(c_what_curr)   // &
                      '_'    // TRIM(c_set_curr)    // &
                      '_'    // TRIM(c_subset_curr) // &
                      '_'    // TRIM(c_type)

      IF(c_subset_curr == cp_cmb_total) THEN

        IF(i_subset < jp_nsubsets) THEN

          IF(.NOT. l_skip) THEN
            WRITE(iu,fmt_)
            WRITE(iu,fmt6)     '  ' // TRIM(c_varname)
          ENDIF

          i_subset = i_subset + 1
          c_subset_curr = LOWCASE(ADJUSTL(cp_subsets(i_subset)))

          c_varname     = cp_col // TRIM(c_what_curr)   // &
                          '_'    // TRIM(c_set_curr)    // &
                          '_'    // TRIM(c_subset_curr) // &
                          '_'    // TRIM(c_type)

          IF(.NOT. l_skip) WRITE(iu,fmtcon) '    =   ' // TRIM(c_varname)
          i_subset = i_subset + 1

        ENDIF

      ELSE

        IF(.NOT. l_skip) THEN
          WRITE(iu,fmtcon) '      + ' // TRIM(c_varname)
        ENDIF
        i_subset = i_subset + 1

      ENDIF

    ENDDO

    WRITE(iu,fmt_)

  ENDDO


  do_subsets_m: DO i_subset = 1, jp_nsubsets

    n_data = t_cmbdata_table(i_what, i_subset)%n_data
    IF(n_data == 0) CYCLE do_subsets_m

    c_subset_curr = LOWCASE(ADJUSTL(cp_subsets(i_subset)))
    c_set_curr    = LOWCASE(ADJUSTL(cp_sets(i_subset)))

    IF(c_subset_curr == cp_cmb_total) WRITE(iu,fmt_) ! extra blank line

    c_varnameend  =        TRIM(c_what_curr)   // &
                    '_' // TRIM(c_set_curr)    // &
                    '_' // TRIM(c_subset_curr) // &
                    '_resid'

    WRITE(iu,fmt6)           '  ' // cp_max // TRIM(c_varnameend)
    WRITE(iu,fmtcon) '    = MAX(' // cp_max // TRIM(c_varnameend) // ', ' // &
                           'ABS(' // cp_col // TRIM(c_varnameend) // '))'
  ENDDO do_subsets_m

  WRITE(iu,fmt_)
  WRITE(iu,fmt_)

ENDDO


#ifdef CFG_DEBUG
WRITE(iu,fmt0, ADVANCE='NO') '! '
WRITE(iu,c_fmtinf_a) 'End of creation'
#endif



!===================!
! End of operations !
!===================!

#ifdef CFG_VERBOSE
#if (CFG_VERBOSELEVEL == 2)
WRITE(jp_stdout,c_fmtinf_a) 'done'
#endif
#endif
#ifdef CFG_DEBUG
WRITE(jp_stddbg,c_fmtdbg_a) 'done'
#endif

RETURN

!-------------------------------------------------------------------------------
 END SUBROUTINE CMB_createColumnSums
!-------------------------------------------------------------------------------



!-------------------------------------------------------------------------------
 SUBROUTINE CMB_createGlobalSumsTotals(iu_gsum, iu_gtot, c_what, t_cmbdata_table)
!-------------------------------------------------------------------------------

IMPLICIT NONE


! Dummy argument list variables
! -----------------------------

INTEGER,                          INTENT(IN)         :: iu_gsum
INTEGER,                          INTENT(IN)         :: iu_gtot
CHARACTER(LEN=*), DIMENSION(:),   INTENT(IN)         :: c_what
TYPE(CMBDATAP),   DIMENSION(:,:), INTENT(IN), TARGET :: t_cmbdata_table


! Local variables
! ---------------
INTEGER                        :: i_what, i_subset
INTEGER                        :: n_data
CHARACTER(LEN=LEN(c_what))     :: c_what_curr
CHARACTER(LEN=n_lmaxnamesgen)  :: c_consrvprop
CHARACTER(LEN=n_lmaxidentif)   :: c_endofvarname
CHARACTER(LEN=n_lmaxidentif)   :: c_varname
CHARACTER(LEN=LEN(cp_subsets)) :: c_set_curr, c_subset_curr
CHARACTER(LEN=20)              :: c_fmttmp
CHARACTER(LEN=n_lmaxidentif)   :: c_type
INTEGER                        :: j

LOGICAL                        :: l_skip


CHARACTER(LEN=*), PARAMETER    :: c_fmtinf_a   = '("[[CMB_createGlobalSumsTotals]]: ", A)'

#ifdef CFG_DEBUG
CHARACTER(LEN=*), PARAMETER    :: c_fmtdbg_a   = '("DEBUG [[CMB_createGlobalSumsTotals]]: ", A)'
#endif

!=====================!
! End of declarations !
!=====================!


#ifdef CFG_VERBOSE
#if (CFG_VERBOSELEVEL == 2)
WRITE(jp_stdout,c_fmtinf_a) 'starting'
#endif
#endif
#ifdef CFG_DEBUG
WRITE(jp_stddbg,c_fmtdbg_a) 'starting'
#endif


!=====================!
! Start of operations !
!=====================!

#ifdef CFG_DEBUG
WRITE(iu_gsum,fmt0, ADVANCE='NO') '! '
WRITE(iu_gsum,c_fmtinf_a) 'Start of creation'
WRITE(iu_gtot,fmt0, ADVANCE='NO') '! '
WRITE(iu_gtot,c_fmtinf_a) 'Start of creation'
#endif

WRITE(iu_gsum,fmt_)

DO i_what = 1, SIZE(c_what)


  CALL CMB_getInformationForOneWhat(t_cmbdata_table, c_what, i_what,   &
                                    c_what_curr, c_consrvprop)
  WRITE(iu_gsum,fmt_)
  WRITE(iu_gsum,fmt_)
  WRITE(iu_gsum,fmt6)   '  ! Global sums for ' // TRIM(ADJUSTL(c_consrvprop)) // &
    ' carrying materials'
  WRITE(iu_gsum,fmt6, ADVANCE='NO')   '  ! '
  WRITE(c_fmttmp,'("(",I0,"(""-"")",")")') LEN_TRIM(ADJUSTL(c_consrvprop)) + 35
  WRITE(iu_gsum,c_fmttmp)
  WRITE(iu_gsum,fmt_)

  DO j = 1, 6

    SELECT CASE(j)
    CASE(1)
      c_endofvarname =   TRIM(c_what_curr) // '_xm_0(:)'
    CASE(2)
      c_endofvarname =   TRIM(c_what_curr) // '_xm_1(:)'
    CASE(3)
      c_endofvarname =   TRIM(c_what_curr) // '_woflx(:)'
    CASE(4)
      c_endofvarname =   TRIM(c_what_curr) // '_boflx(:)'
    CASE(5)
      c_endofvarname =   TRIM(c_what_curr) // '_rea(:)'
    CASE(6)
      c_endofvarname =   TRIM(c_what_curr) // '_resid(:)'
    END SELECT

    WRITE(iu_gsum,fmt6)       '  ' // cp_glo // TRIM(c_endofvarname)
    WRITE(iu_gsum,fmtcon) '    = ' // cp_glo // TRIM(c_endofvarname) // &
               ' + sfc_area*' // cp_col // TRIM(c_endofvarname)
    WRITE(iu_gsum,fmt_)

  ENDDO



  DO j = 1, 6

    SELECT CASE(j)
    CASE(1)
      c_type = 'xm_0'
    CASE(2)
      c_type = 'xm_1'
    CASE(3)
      c_type = 'woflx'
    CASE(4)
      c_type = 'boflx'
    CASE(5)
      c_type = 'rea'
    CASE(6)
      c_type = 'resid'
    END SELECT

    do_subsets: DO i_subset = 1, jp_nsubsets

      n_data = t_cmbdata_table(i_what,i_subset)%n_data
      IF(n_data == 0) CYCLE do_subsets

      c_subset_curr = LOWCASE(ADJUSTL(cp_subsets(i_subset)))

                                    ! skip 'total' subsets -- they are
                                    ! build later from the sums of subsets
                                    ! Leave a blank line nevertheless
      IF(c_subset_curr == cp_cmb_total) THEN
        WRITE(iu_gsum,fmt_)
        CYCLE do_subsets
      ENDIF

      c_set_curr    = LOWCASE(ADJUSTL(cp_sets(i_subset)))

      c_endofvarname =   TRIM(c_what_curr)   // &
                  '_' // TRIM(c_set_curr)    // &
                  '_' // TRIM(c_subset_curr) // &
                  '_' // TRIM(c_type)


      WRITE(iu_gsum,fmt6)       '  ' // cp_glo // TRIM(c_endofvarname)
      WRITE(iu_gsum,fmtcon) '    = ' // cp_glo // TRIM(c_endofvarname) // &
                 ' + sfc_area*' // cp_col // TRIM(c_endofvarname)

    ENDDO do_subsets

    WRITE(iu_gsum,fmt_)
    WRITE(iu_gsum,fmt_)


                                    ! Now build the 'total subset sums

    WRITE(iu_gtot,fmt_)

    i_subset = 1
    DO WHILE(i_subset <= jp_nsubsets)

      n_data = t_cmbdata_table(i_what, i_subset)%n_data
      IF (n_data == 0) THEN
        l_skip = .TRUE.
      ELSE
        l_skip = .FALSE.
      ENDIF

      c_subset_curr = LOWCASE(ADJUSTL(cp_subsets(i_subset)))
      c_set_curr    = LOWCASE(ADJUSTL(cp_sets(i_subset)))

      c_varname     = cp_glo // TRIM(c_what_curr)   // &
                      '_'    // TRIM(c_set_curr)    // &
                      '_'    // TRIM(c_subset_curr) // &
                      '_'    // TRIM(c_type)

      IF(c_subset_curr == cp_cmb_total) THEN

        IF(i_subset < jp_nsubsets) THEN

          IF(.NOT. l_skip) THEN
            WRITE(iu_gtot,fmt_)
            WRITE(iu_gtot,fmt6)     '  ' // TRIM(c_varname)
          ENDIF

          i_subset = i_subset + 1
          c_subset_curr = LOWCASE(ADJUSTL(cp_subsets(i_subset)))

          c_varname     = cp_glo // TRIM(c_what_curr)   // &
                          '_'    // TRIM(c_set_curr)    // &
                          '_'    // TRIM(c_subset_curr) // &
                          '_'    // TRIM(c_type)

          IF(.NOT. l_skip) THEN
            WRITE(iu_gtot,fmtcon) '    =   ' // TRIM(c_varname)
          ENDIF
          i_subset = i_subset + 1

        ENDIF

      ELSE

        IF(.NOT. l_skip) THEN
          WRITE(iu_gtot,fmtcon) '      + ' // TRIM(c_varname)
        ENDIF
        i_subset = i_subset + 1

      ENDIF

    ENDDO

    WRITE(iu_gtot,fmt_)
    WRITE(iu_gtot,fmt_)

  ENDDO


ENDDO


#ifdef CFG_DEBUG
WRITE(iu_gsum,fmt0, ADVANCE='NO') '! '
WRITE(iu_gsum,c_fmtinf_a) 'End of creation'
WRITE(iu_gtot,fmt0, ADVANCE='NO') '! '
WRITE(iu_gtot,c_fmtinf_a) 'End of creation'
#endif

!===================!
! End of operations !
!===================!

#ifdef CFG_VERBOSE
#if (CFG_VERBOSELEVEL == 2)
WRITE(jp_stdout,c_fmtinf_a) 'done'
#endif
#endif
#ifdef CFG_DEBUG
WRITE(jp_stddbg,c_fmtdbg_a) 'done'
#endif

RETURN

!-------------------------------------------------------------------------------
 END SUBROUTINE CMB_createGlobalSumsTotals
!-------------------------------------------------------------------------------



!-------------------------------------------------------------------------------
 SUBROUTINE CMB_createPrintGlobalInfos(iu_gmba, iu_gstk, c_what, t_cmbdata_table)
!-------------------------------------------------------------------------------

USE MOD_UTILITIES, ONLY : L_STR_EQ_DBL


IMPLICIT NONE


! Dummy argument list variables
! -----------------------------

INTEGER,                          INTENT(IN)         :: iu_gmba
INTEGER,                          INTENT(IN)         :: iu_gstk
CHARACTER(LEN=*), DIMENSION(:),   INTENT(IN)         :: c_what
TYPE(CMBDATAP),   DIMENSION(:,:), INTENT(IN), TARGET :: t_cmbdata_table


! Local variables
! ---------------

INTEGER :: i_what, i_subset
INTEGER :: n_porew, n_solid
CHARACTER(LEN=LEN(c_what)) :: c_what_curr
CHARACTER(LEN=n_lmaxnamesgen) :: c_consrvprop

TYPE(CMBDATA), POINTER       :: t_cmbdata_node
TYPE(COMPOINFO), POINTER     :: t_compo_node
CHARACTER(LEN=n_lmaxnamesgen+3) :: c_fmtname
CHARACTER(LEN=n_lmaxnamesgen+3) :: c_fmtline
CHARACTER(LEN=n_lmaxnamesgen+3) :: c_fmtinda
CHARACTER(LEN=n_lmaxnamesgen)   :: c_name
INTEGER :: i, j
CHARACTER(LEN=n_lmaxidentif) :: c_varnamebase
CHARACTER(LEN=*), PARAMETER :: c_indent = '    '


! Standard I/O related data
! -------------------------

CHARACTER(LEN=*), PARAMETER  :: c_fmtinf_a   = '("[[CMB_createPrintGlobalInfos]]: ", A)'
CHARACTER(LEN=*), PARAMETER  :: c_fmterr_a   = '("[[CMB_createPrintGlobalInfos]] error: ", A)'
#ifdef CFG_DEBUG
CHARACTER(LEN=*), PARAMETER  :: c_fmtdbg_a   = '("DEBUG [[CMB_createPrintGlobalInfos]]: ", A)'
#endif

!=====================!
! End of declarations !
!=====================!


#ifdef CFG_VERBOSE
#if (CFG_VERBOSELEVEL == 2)
WRITE(jp_stdout,c_fmtinf_a) 'starting'
#endif
#endif
#ifdef CFG_DEBUG
WRITE(jp_stddbg,c_fmtdbg_a) 'starting'
#endif


!=====================!
! Start of operations !
!=====================!

#ifdef CFG_DEBUG
WRITE(iu_gmba,'("!",35(" -"))')
WRITE(iu_gmba,fmt0, ADVANCE='NO') '! '
WRITE(iu_gmba,c_fmtinf_a) 'Start of creation'
WRITE(iu_gmba,'("!",35(" -"))')
WRITE(iu_gstk,'("!",35(" -"))')
WRITE(iu_gstk,fmt0, ADVANCE='NO') '! '
WRITE(iu_gstk,c_fmtinf_a) 'Start of creation'
WRITE(iu_gstk,'("!",35(" -"))')
#endif


WRITE(c_fmtname,'("(""''"",3X, A",I0,",""''"")")') n_lmaxnamesgen
WRITE(c_fmtline,'("(""''"",3X,",I0,"(""-""),""''"")")') n_lmaxnamesgen
WRITE(c_fmtinda,'("(", I0, "X, A)")') n_lmaxnamesgen + 3 + 2  ! 3 = LEN(3X);
                                                              ! 2 = LEN(', ')

DO i_what = 1, SIZE(c_what)

  CALL CMB_getInformationForOneWhat(t_cmbdata_table, c_what, i_what,   &
                                    c_what_curr, c_consrvprop)

  WRITE(iu_gmba,fmt_)
  WRITE(iu_gmba,fmt6) 'WRITE(jp_stdlog,''()'')'
  WRITE(iu_gmba,fmt6) 'WRITE(jp_stdlog,''()'')'

  WRITE(iu_gstk,fmt_)
  WRITE(iu_gstk,fmt6) 'WRITE(jp_stdlog,''()'')'
  WRITE(iu_gstk,fmt6) 'WRITE(jp_stdlog,''()'')'

  SELECT CASE(c_what_curr)
  CASE(cp_shortid_alk)
    WRITE(iu_gmba,fmt6) 'WRITE(jp_stdlog,fmta) '' * Alkalinity (eq [eq/eq_t])'''
    WRITE(iu_gstk,fmt6) 'WRITE(jp_stdlog,fmta) '' * Alkalinity (eq)'''
  CASE('c','C')
    WRITE(iu_gmba,fmt6) 'WRITE(jp_stdlog,fmta) '' * Carbon (mol C [mol/mol_t])'''
    WRITE(iu_gstk,fmt6) 'WRITE(jp_stdlog,fmta) '' * Carbon (mol C)'''
  CASE('o2','O2')
    WRITE(iu_gmba,fmt6) 'WRITE(jp_stdlog,fmta) '' * Oxygen (mol O2 [mol/mol_t])'''
    WRITE(iu_gstk,fmt6) 'WRITE(jp_stdlog,fmta) '' * Oxygen (mol O2)'''
  CASE('si','Si')
    WRITE(iu_gmba,fmt6) 'WRITE(jp_stdlog,fmta) '' * Silicon (mol Si [mol/mol_t])'''
    WRITE(iu_gstk,fmt6) 'WRITE(jp_stdlog,fmta) '' * Silicon (mol Si)'''
  CASE DEFAULT
    WRITE(jp_stderr,c_fmterr_a) 'Unknown <c_what> value ' // TRIM(c_what_curr) // &
      ' -- please extend me! Aborting'
    CALL ABORT()
  END SELECT

  WRITE(iu_gmba,fmt6) 'WRITE(jp_stdlog,''()'')'
  WRITE(iu_gstk,fmt6) 'WRITE(jp_stdlog,''()'')'

                                    ! Scan the porew subphase component
                                    ! list for the current c_what (i.e.,
                                    ! c_what(i_what)): if there is only
                                    ! one element in the list, print out
                                    ! the info for that element, else
                                    ! print out the info for the bulk
                                    ! porew phase (no details)
  t_cmbdata_node => t_cmbdata_table(i_what, jp_cmbbulk_porew)%ptr

  n_porew = t_cmbdata_table(i_what, jp_cmbbulk_porew)%n_data

  SELECT CASE(n_porew)
  CASE(1)

    c_shortid = t_cmbdata_node%shortid
    t_compo_node => COMPOINFO_getNodeByShortid(t_compo_chain, c_shortid)

    c_name = ADJUSTL(t_compo_node%name)
    WRITE(iu_gmba,fmt6) 'io = ' // cp_prefix_io // TRIM(c_shortid)
    WRITE(iu_gmba,fmt6,ADVANCE='NO') 'WRITE(jp_stdlog,fmta) '
    WRITE(iu_gmba,c_fmtname) c_name

    c_varnamebase = cp_glo // TRIM(c_what_curr)
    WRITE(iu_gmba,fmt6)   'net =   ' // TRIM(c_varnamebase) // '_woflx(io) - ' // &
                                        TRIM(c_varnamebase) // '_boflx(io)'
    WRITE(iu_gmba,fmtcon) '      + ' // TRIM(c_varnamebase) // '_rea(io)'
    WRITE(iu_gmba,fmt6)   'del =   ' // TRIM(c_varnamebase) // '_xm_1(io) - '  // &
                                        TRIM(c_varnamebase) // '_xm_0(io)'
    WRITE(iu_gmba,fmt6)   'ref =  (' // TRIM(c_varnamebase) // '_xm_1(io) + '  // &
                                        TRIM(c_varnamebase) // '_xm_0(io))/2D0'
    WRITE(iu_gmba,fmt6)   'WRITE(jp_stdlog,fmt1) ''    Top Flux   '', ' // &
                                        TRIM(c_varnamebase) // '_woflx(io)'
    WRITE(iu_gmba,fmt6)   'WRITE(jp_stdlog,fmt1) ''    Bot Flux   '', ' // &
                                        TRIM(c_varnamebase) // '_boflx(io)'
    WRITE(iu_gmba,fmt6)   'WRITE(jp_stdlog,fmt1) ''    Total Rea  '', ' // &
                                        TRIM(c_varnamebase) // '_rea(io)'
    WRITE(iu_gmba,fmt6)   'WRITE(jp_stdlog,fmt1) ''    Net        '', net'
    WRITE(iu_gmba,fmt6)   'WRITE(jp_stdlog,fmt1) ''    Act Delta  '', del'
    WRITE(iu_gmba,fmt6)   'WRITE(jp_stdlog,fmt2) ''    Diff       '', (del - net), (del-net)/ref'
    WRITE(iu_gmba,fmt6)   'WRITE(jp_stdlog,''()'')'
    WRITE(iu_gmba,'()')


    WRITE(iu_gstk,fmt6)   'io = ' // cp_prefix_io // TRIM(c_shortid)
    WRITE(iu_gstk,fmt6,ADVANCE='NO') 'WRITE(jp_stdlog,fmt3) '
    WRITE(iu_gstk,c_fmtname,ADVANCE='NO') ADJUSTL(t_compo_node%name)
    WRITE(iu_gstk,'(",")')

    WRITE(iu_gstk,fmtcon) c_indent // TRIM(c_varnamebase) // '_xm_0(io), ' // &
                                      TRIM(c_varnamebase) // '_xm_1(io)'
    WRITE(iu_gstk,fmt_)


  CASE(2:)

    c_name = 'Sigma porew'
    WRITE(iu_gmba,fmt6,ADVANCE='NO') 'WRITE(jp_stdlog,fmta) '
    WRITE(iu_gmba,c_fmtname) c_name

    c_varnamebase =  cp_glo // TRIM(c_what_curr) // '_bulk_porew'
    WRITE(iu_gmba,fmt6)   'net =   ' // TRIM(c_varnamebase) // '_woflx - ' // &
                                        TRIM(c_varnamebase) // '_boflx'
    WRITE(iu_gmba,fmtcon) '      + ' // TRIM(c_varnamebase) // '_rea'
    WRITE(iu_gmba,fmt6)   'del =   ' // TRIM(c_varnamebase) // '_xm_1 - '  // &
                                        TRIM(c_varnamebase) // '_xm_0'
    WRITE(iu_gmba,fmt6)   'ref =  (' // TRIM(c_varnamebase) // '_xm_1 + '  // &
                                        TRIM(c_varnamebase) // '_xm_0)/2D0'

    WRITE(iu_gmba,fmt6) 'WRITE(jp_stdlog,fmt1) ''    Top Flux   '', ' // &
                                        TRIM(c_varnamebase) // '_woflx'
    WRITE(iu_gmba,fmt6) 'WRITE(jp_stdlog,fmt1) ''    Bot Flux   '', ' // &
                                        TRIM(c_varnamebase) // '_boflx'
    WRITE(iu_gmba,fmt6) 'WRITE(jp_stdlog,fmt1) ''    Total Rea  '', ' // &
                                        TRIM(c_varnamebase) // '_rea'
    WRITE(iu_gmba,fmt6) 'WRITE(jp_stdlog,fmt1) ''    Net        '', net'
    WRITE(iu_gmba,fmt6) 'WRITE(jp_stdlog,fmt1) ''    Act Delta  '', del'
    WRITE(iu_gmba,fmt6) 'WRITE(jp_stdlog,fmt2) ''    Diff       '', del - net, (del-net)/ref'
    WRITE(iu_gmba,fmt6) 'WRITE(jp_stdlog,''()'')'
    WRITE(iu_gmba,fmt_)
    WRITE(iu_gmba,fmt_)


    WRITE(iu_gstk,fmt6,ADVANCE='NO') 'WRITE(jp_stdlog,fmt3) '
    WRITE(iu_gstk,c_fmtname,ADVANCE='NO') ADJUSTL(c_name)
    WRITE(iu_gstk,'(",")')

    WRITE(iu_gstk,fmtcon) c_indent // TRIM(c_varnamebase) // '_xm_0, ' // &
                                      TRIM(c_varnamebase) // '_xm_1'
    WRITE(iu_gstk,fmt_)
    WRITE(iu_gstk,fmt_)


  END SELECT


  t_cmbdata_node => t_cmbdata_table(i_what, jp_cmbbulk_solid)%ptr

  n_solid = t_cmbdata_table(i_what, jp_cmbbulk_solid)%n_data

  DO WHILE(ASSOCIATED(t_cmbdata_node))

    c_shortid = ADJUSTL(t_cmbdata_node%shortid)
    t_compo_node => COMPOINFO_getNodeByShortid(t_compo_chain, c_shortid)
    c_name = ADJUSTL(t_compo_node%name)

    WRITE(iu_gmba,fmt6)   'io = ' // cp_prefix_io // TRIM(c_shortid)
    WRITE(iu_gmba,fmt6,ADVANCE='NO') 'WRITE(jp_stdlog,fmta) '
    WRITE(iu_gmba,c_fmtname) c_name

    c_varnamebase = cp_glo // TRIM(c_what_curr)
    WRITE(iu_gmba,fmt6)   'net =   ' // TRIM(c_varnamebase) // '_woflx(io) - ' // &
                                        TRIM(c_varnamebase) // '_boflx(io)'
    WRITE(iu_gmba,fmtcon) '      + ' // TRIM(c_varnamebase) // '_rea(io)'
    WRITE(iu_gmba,fmt6)   'del =   ' // TRIM(c_varnamebase) // '_xm_1(io) - '  // &
                                        TRIM(c_varnamebase) // '_xm_0(io)'
    WRITE(iu_gmba,fmt6)   'ref =  (' // TRIM(c_varnamebase) // '_xm_1(io) + '  // &
                                        TRIM(c_varnamebase) // '_xm_0(io))/2D0'
    WRITE(iu_gmba,fmt6)   'WRITE(jp_stdlog,fmt1) ''    Top Flux   '', ' // &
                                        TRIM(c_varnamebase) // '_woflx(io)'
    WRITE(iu_gmba,fmt6)   'WRITE(jp_stdlog,fmt1) ''    Bot Flux   '', ' // &
                                        TRIM(c_varnamebase) // '_boflx(io)'
    WRITE(iu_gmba,fmt6)   'WRITE(jp_stdlog,fmt1) ''    Total Rea  '', ' // &
                                        TRIM(c_varnamebase) // '_rea(io)'
    WRITE(iu_gmba,fmt6)   'WRITE(jp_stdlog,fmt1) ''    Net        '', net'
    WRITE(iu_gmba,fmt6)   'WRITE(jp_stdlog,fmt1) ''    Act Delta  '', del'
    WRITE(iu_gmba,fmt6)   'WRITE(jp_stdlog,fmt2) ''    Diff       '', (del - net), (del-net)/ref'
    WRITE(iu_gmba,fmt6)   'WRITE(jp_stdlog,''()'')'
    WRITE(iu_gmba,'()')


    WRITE(iu_gstk,fmt6)   'io = ' // cp_prefix_io // TRIM(c_shortid)
    WRITE(iu_gstk,fmt6,ADVANCE='NO') 'WRITE(jp_stdlog,fmt3) '
    WRITE(iu_gstk,c_fmtname,ADVANCE='NO') ADJUSTL(t_compo_node%name)
    WRITE(iu_gstk,'(",")')

    WRITE(iu_gstk,fmtcon) c_indent // TRIM(c_varnamebase) // '_xm_0(io), ' // &
                                      TRIM(c_varnamebase) // '_xm_1(io)'
    WRITE(iu_gstk,fmt_)


    t_cmbdata_node => t_cmbdata_node%next

  ENDDO


  IF(n_solid > 1) THEN

    c_name = 'Sigma solid'

    WRITE(iu_gmba,fmt6,ADVANCE='NO') 'WRITE(jp_stdlog,fmta) '
    WRITE(iu_gmba,c_fmtname) c_name

    c_varnamebase =  cp_glo // TRIM(c_what_curr) // '_bulk_solid'
    WRITE(iu_gmba,fmt6)   'net =   ' // TRIM(c_varnamebase) // '_woflx - ' // &
                                        TRIM(c_varnamebase) // '_boflx'
    WRITE(iu_gmba,fmtcon) '      + ' // TRIM(c_varnamebase) // '_rea'
    WRITE(iu_gmba,fmt6)   'del =   ' // TRIM(c_varnamebase) // '_xm_1 - '  // &
                                        TRIM(c_varnamebase) // '_xm_0'
    WRITE(iu_gmba,fmt6)   'ref =  (' // TRIM(c_varnamebase) // '_xm_1 + '  // &
                                        TRIM(c_varnamebase) // '_xm_0)/2D0'

    WRITE(iu_gmba,fmt6) 'WRITE(jp_stdlog,fmt1) ''    Top Flux   '', ' // &
                                        TRIM(c_varnamebase) // '_woflx'
    WRITE(iu_gmba,fmt6) 'WRITE(jp_stdlog,fmt1) ''    Bot Flux   '', ' // &
                                        TRIM(c_varnamebase) // '_boflx'
    WRITE(iu_gmba,fmt6) 'WRITE(jp_stdlog,fmt1) ''    Total Rea  '', ' // &
                                        TRIM(c_varnamebase) // '_rea'
    WRITE(iu_gmba,fmt6) 'WRITE(jp_stdlog,fmt1) ''    Net        '', net'
    WRITE(iu_gmba,fmt6) 'WRITE(jp_stdlog,fmt1) ''    Act Delta  '', del'
    WRITE(iu_gmba,fmt6) 'WRITE(jp_stdlog,fmt2) ''    Diff       '', del - net, (del-net)/ref'
    WRITE(iu_gmba,fmt6) 'WRITE(jp_stdlog,''()'')'
    WRITE(iu_gmba,fmt_)
    WRITE(iu_gmba,fmt_)


    WRITE(iu_gstk,fmt6,ADVANCE='NO') 'WRITE(jp_stdlog,fmt3) '
    WRITE(iu_gstk,c_fmtname,ADVANCE='NO') ADJUSTL(c_name)
    WRITE(iu_gstk,'(",")')

    WRITE(iu_gstk,fmtcon) c_indent // TRIM(c_varnamebase) // '_xm_0, ' // &
                                      TRIM(c_varnamebase) // '_xm_1'
    WRITE(iu_gstk,fmt_)
    WRITE(iu_gstk,fmt_)


  ENDIF


  c_name = 'Sigma total'

  WRITE(iu_gmba,fmt6,ADVANCE='NO') 'WRITE(jp_stdlog,fmta) '
  WRITE(iu_gmba,c_fmtname) c_name

  c_varnamebase =  cp_glo // TRIM(c_what_curr) // '_bulk_total'
  WRITE(iu_gmba,fmt6)   'net =   ' // TRIM(c_varnamebase) // '_woflx - ' // &
                                      TRIM(c_varnamebase) // '_boflx'
  WRITE(iu_gmba,fmtcon) '      + ' // TRIM(c_varnamebase) // '_rea'
  WRITE(iu_gmba,fmt6)   'del =   ' // TRIM(c_varnamebase) // '_xm_1 - '  // &
                                      TRIM(c_varnamebase) // '_xm_0'
  WRITE(iu_gmba,fmt6)   'ref =  (' // TRIM(c_varnamebase) // '_xm_1 + '  // &
                                      TRIM(c_varnamebase) // '_xm_0)/2D0'

  WRITE(iu_gmba,fmt6) 'WRITE(jp_stdlog,fmt1) ''    Top Flux   '', ' // &
                                      TRIM(c_varnamebase) // '_woflx'
  WRITE(iu_gmba,fmt6) 'WRITE(jp_stdlog,fmt1) ''    Bot Flux   '', ' // &
                                      TRIM(c_varnamebase) // '_boflx'
  WRITE(iu_gmba,fmt6) 'WRITE(jp_stdlog,fmt1) ''    Total Rea  '', ' // &
                                      TRIM(c_varnamebase) // '_rea'
  WRITE(iu_gmba,fmt6) 'WRITE(jp_stdlog,fmt1) ''    Net        '', net'
  WRITE(iu_gmba,fmt6) 'WRITE(jp_stdlog,fmt1) ''    Act Delta  '', del'
  WRITE(iu_gmba,fmt6) 'WRITE(jp_stdlog,fmt2) ''    Diff       '', del - net, (del-net)/ref'
  WRITE(iu_gmba,fmt6) 'WRITE(jp_stdlog,''()'')'
  WRITE(iu_gmba,fmt_)
  WRITE(iu_gmba,fmt_)


  WRITE(iu_gstk,fmt6,ADVANCE='NO') 'WRITE(jp_stdlog,fmt3) '
  WRITE(iu_gstk,c_fmtname,ADVANCE='NO') ADJUSTL(c_name)
  WRITE(iu_gstk,'(",")')

  WRITE(iu_gstk,fmtcon) c_indent // TRIM(c_varnamebase) // '_xm_0, ' // &
                                    TRIM(c_varnamebase) // '_xm_1'
  WRITE(iu_gstk,fmt_)
  WRITE(iu_gstk,fmt_)


ENDDO

#ifdef CFG_DEBUG
WRITE(iu_gmba,fmt0, ADVANCE='NO') '! '
WRITE(iu_gmba,c_fmtinf_a) 'End of creation'
WRITE(iu_gstk,fmt0, ADVANCE='NO') '! '
WRITE(iu_gstk,c_fmtinf_a) 'End of creation'
#endif

!===================!
! End of operations !
!===================!

#ifdef CFG_VERBOSE
#if (CFG_VERBOSELEVEL == 2)
WRITE(jp_stdout,c_fmtinf_a) 'done'
#endif
#endif
#ifdef CFG_DEBUG
WRITE(jp_stddbg,c_fmtdbg_a) 'done'
#endif

RETURN

!-------------------------------------------------------------------------------
 END SUBROUTINE CMB_createPrintGlobalInfos
!-------------------------------------------------------------------------------



!-------------------------------------------------------------------------------
 SUBROUTINE CMB_CMBDATA_registerData(t_cmbdata_table, i1, i2, c_shortid, c_weight)
!-------------------------------------------------------------------------------

IMPLICIT NONE

TYPE(CMBDATAP), DIMENSION(:,:), TARGET :: t_cmbdata_table
INTEGER, INTENT(IN) :: i1
INTEGER, INTENT(IN) :: i2
CHARACTER(LEN=n_lmaxshortid), INTENT(IN) :: c_shortid
CHARACTER(LEN=n_lmaxexpress), INTENT(IN) :: c_weight


TYPE(CMBDATA), POINTER :: t_cmbdata_node


IF(.NOT.ASSOCIATED(t_cmbdata_table(i1,i2)%ptr)) THEN
  t_cmbdata_table(i1,i2)%ptr => CMB_CMBDATA_create()
  t_cmbdata_node => t_cmbdata_table(i1,i2)%ptr
ELSE
  t_cmbdata_node => CMB_CMBDATA_appendNode(t_cmbdata_table(i1,i2)%ptr)
ENDIF

t_cmbdata_node%shortid = c_shortid
t_cmbdata_node%weights = c_weight
t_cmbdata_table(i1,i2)%n_data = t_cmbdata_table(i1,i2)%n_data + 1

RETURN

!-------------------------------------------------------------------------------
 END SUBROUTINE CMB_CMBDATA_registerData
!-------------------------------------------------------------------------------



!-------------------------------------------------------------------------------
 FUNCTION CMB_CMBDATAP_create(ndim1, ndim2) RESULT(cp)
!-------------------------------------------------------------------------------

IMPLICIT NONE

TYPE(CMBDATAP), DIMENSION(:,:), POINTER :: cp
INTEGER, INTENT(IN) :: ndim1, ndim2

TYPE(CMBDATA), POINTER :: c_wk

INTEGER :: i1, i2

ALLOCATE(cp(ndim1, ndim2))
DO i2 = 1, ndim2
  DO i1 = 1, ndim1
    cp(i1,i2)%n_data = 0
    NULLIFY(cp(i1,i2)%ptr)
  ENDDO
ENDDO

!-------------------------------------------------------------------------------
 END FUNCTION CMB_CMBDATAP_create
!-------------------------------------------------------------------------------



!-------------------------------------------------------------------------------
 SUBROUTINE CMB_CMBDATAP_delete(cp)
!-------------------------------------------------------------------------------

IMPLICIT NONE

TYPE(CMBDATAP), DIMENSION(:,:), POINTER :: cp

INTEGER :: i1, i2

IF(ASSOCIATED(cp)) THEN
  DO i2 = 1, SIZE(cp, DIM=2)
    DO i1 = 1, SIZE(cp, DIM=1)
      CALL CMB_CMBDATA_delete(cp(i1,i2)%ptr)
    ENDDO
  ENDDO
  DEALLOCATE(cp)
  NULLIFY(cp)
ENDIF

!-------------------------------------------------------------------------------
 END SUBROUTINE CMB_CMBDATAP_delete
!-------------------------------------------------------------------------------



!-------------------------------------------------------------------------------
 FUNCTION CMB_CMBDATA_create() RESULT(c)
!-------------------------------------------------------------------------------

IMPLICIT NONE

TYPE(CMBDATA), POINTER :: c

ALLOCATE(c)

c%shortid = ''
c%weights = ''
NULLIFY(c%prev)
NULLIFY(c%next)

RETURN

!-------------------------------------------------------------------------------
 END FUNCTION CMB_CMBDATA_create
!-------------------------------------------------------------------------------



!-------------------------------------------------------------------------------
 SUBROUTINE CMB_CMBDATA_delete(c)
!-------------------------------------------------------------------------------

IMPLICIT NONE

TYPE(CMBDATA), POINTER  :: c


TYPE(CMBDATA), POINTER :: c_wk

IF(ASSOCIATED(c)) THEN

  c_wk => c
  IF(ASSOCIATED(c%prev)) THEN
    NULLIFY(c%prev%next)
    NULLIFY(c%prev)
  ENDIF

  DO WHILE(ASSOCIATED(c_wk%next))
    c_wk => c_wk%next
    DEALLOCATE(c_wk%prev)
  ENDDO
  DEALLOCATE(c_wk)

  NULLIFY(c)

ENDIF

NULLIFY(c_wk)

RETURN

!-------------------------------------------------------------------------------
 END SUBROUTINE CMB_CMBDATA_delete
!-------------------------------------------------------------------------------



!-------------------------------------------------------------------------------
 FUNCTION CMB_CMBDATA_appendNode(c) RESULT(c_end)
!-------------------------------------------------------------------------------

IMPLICIT NONE

TYPE(CMBDATA), TARGET  :: c
TYPE(CMBDATA), POINTER :: c_end


TYPE(CMBDATA), POINTER :: c_wk


c_wk => c
DO WHILE(ASSOCIATED(c_wk%next))
  c_wk => c_wk%next
ENDDO

c_end => CMB_CMBDATA_create()
c_wk%next => c_end
c_end%prev => c_wk

NULLIFY(c_wk)

RETURN

!-------------------------------------------------------------------------------
 END FUNCTION CMB_CMBDATA_appendNode
!-------------------------------------------------------------------------------


!===================================================================================================
 END SUBROUTINE CREATE_MASS_BALANCES
!===================================================================================================
