!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


!-----------------------------------------------------------------------
      SUBROUTINE SEAFLOOR_SETUP(n_columns_host, darea_gridelts_host,
     &                              dcnpoh_c_host, dcnpoh_n_host,
     &                              dcnpoh_p_host, dcnpoh_o_host,
     &                              dcnpoh_h_host, dcnpoh_remin_o2_host)
!-----------------------------------------------------------------------

      ! Uses the following variables from MOD_SEAFLOOR_CENTRAL
      !  n_grid_seafloor  (write, ** gets initialised here **)

      ! **all** others get ALLOCATEd here

      ! The following get initialised:
      !  reaclay_xzdn     (write, ** gets initialised here **)
      !  reaclay_xphi     (write, ** gets initialised here **)
      !  reaclay_xdphi    (write, ** gets initialised here **)
      !  tranlay_yphi     (write, ** gets initialised here **)
      !  reaclay_xvphi    (write, ** gets initialised here **)
      !  reaclay_xvdphi   (write, ** gets initialised here **)
      !  seafloor_omcnp_* (write, ** gets initialised here **)


      USE mod_defines_medusa
      USE mod_execontrol_medusa, ONLY: ABORT_MEDUSA
#ifdef ALLOW_MPI
      USE mpi
      USE mod_execontrol_medusa, ONLY: jp_exeproc_ncio,
     &                              lp_exeproc_singleproc_nc,
     &                              MEDEXE_NPROC,
     &                              MEDEXE_MPI_COMM,
     &                              MEDEXE_MPI_COMM_RANK,
     &                              MEDEXE_MPI_SETPARTITIONING
#ifdef DEBUG
      USE mod_execontrol_medusa, ONLY: MEDEXE_MPI_GETPARTITIONING
#endif
#endif

      USE mod_gridparam,    ONLY: idnw, idnt, idnb, idvw, idvs, idvb,
     &                              jp_grid_dynamic,
     &                              jp_grid_static_local,
     &                              jp_grid_static_global,
     &                              SETUP_GRID, SELECTED_GRIDTYPE,
     &                              GRID_DEF

      USE mod_milieucharas, ONLY: xphi, xdphi, yphi, xvphi, xvdphi,
     &                              jp_phi_dynamic,
     &                              jp_phi_static_local,
     &                              jp_phi_static_global,
     &                              SELECTED_PHITYPE,
     &                              POROTORTUOSITY_DEF


      USE mod_materialcharas


      IMPLICIT NONE

      INTEGER,                     INTENT(IN)  :: n_columns_host

      DOUBLE PRECISION, DIMENSION(:),
     &                             INTENT(IN)  :: darea_gridelts_host

      DOUBLE PRECISION, DIMENSION(:,:), OPTIONAL,
     &                             INTENT(IN)  :: dcnpoh_c_host,
     &                                            dcnpoh_n_host,
     &                                            dcnpoh_p_host,
     &                                            dcnpoh_o_host,
     &                                            dcnpoh_h_host,
     &                                            dcnpoh_remin_o2_host


      INTEGER :: i, j, k
      INTEGER :: iselect_grid, iselect_phi
      INTEGER :: i_column, ngrid_columns, nphi_columns
      INTEGER :: iflag

#ifdef ALLOW_MPI
      INTEGER :: n_cprocs, i_myrank
      CHARACTER(LEN=63) :: c_fmt
#ifdef DEBUG
      INTEGER :: nsedcol_seafloor_global
      INTEGER :: iproc_1stocn
      INTEGER, DIMENSION(:), ALLOCATABLE :: nsedcol_pproc
      INTEGER, DIMENSION(:), ALLOCATABLE :: ioffset_sedcol_pproc
#endif
#endif

      LOGICAL :: l_usedefault_cnpoh = .FALSE.

      INTEGER, DIMENSION(2) :: ishape_dcnpoh
      INTEGER, DIMENSION(2) :: ishape_dcnpoh_x

      CHARACTER(LEN=*), PARAMETER :: cfmt_as
     &  = '(A, "(/ ", I0, ", ", I0, " /)")'

      INTEGER  :: ngrid_cnpoh

      DOUBLE PRECISION, DIMENSION(idnw:idnb) :: xzdn
      DOUBLE PRECISION, DIMENSION(idvw:idvb) :: xzdv

![XXX] Temporary fix (should be among the dummy arguments)
      INTEGER,          DIMENSION(:), ALLOCATABLE  :: imask_ocean_host


      ! Standard I/O related data
      ! -------------------------

      CHARACTER(LEN=*), PARAMETER :: cfmt_modprocname_a = 
     &  '("[MOD_SEAFLOOR_CENTRAL/SEAFLOOR_SETUP (base1D)]: ", A)'

      CHARACTER(LEN=*), PARAMETER :: cfmt_a     = '(" - ", A)'
      CHARACTER(LEN=*), PARAMETER :: cfmt_a_ind = '(3X, A)'



!----------------------------------------------------------------------
! Subroutine Start
!----------------------------------------------------------------------

#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_modprocname_a) 'Start'
#endif

![XXX] Temporary fix
      nix = n_columns_host

      IF (n_columns_host <= 0) THEN
#ifdef ALLOW_MPI
        WRITE(jp_stderr, cfmt_modprocname_a) 'Warning'
        WRITE(jp_stderr, cfmt_a)
     &    '"n_columns_host" is zero or negative in this process!'
        WRITE(jp_stderr, cfmt_a_ind)
     &    '-- Continuing with fingers crossed!'
#else
        WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
        WRITE(jp_stderr, cfmt_a)
     &    '"n_columns_host" must be strictly positive -- aborting!'
        CALL ABORT_MEDUSA()
#endif
      ENDIF
                                    ! Copy the value of n_columns_host
                                    ! into MOD_SEAFLOOR_CENTRAL.
      n_grid_seafloor = n_columns_host
      ngrid           = n_grid_seafloor

![XXX] Temporary fix (next two lines)
      ALLOCATE(imask_ocean_host(n_columns_host))
      imask_ocean_host(:) = 1


#ifdef ALLOW_MPI
      n_cprocs = MEDEXE_NPROC()

      IF (n_cprocs > 0) THEN

#ifdef DEBUG
        i_myrank = MEDEXE_MPI_COMM_RANK()
#endif

        CALL MEDEXE_MPI_SETPARTITIONING(n_grid_seafloor)

      ELSE

        WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
        WRITE(jp_stderr, cfmt_a)
     &     'Execution environment (MOD_EXECONTROL_MEDUSA) not set up!'
        WRITE(jp_stderr, cfmt_a_ind) '-- Aborting'
        CALL ABORT_MEDUSA()

      ENDIF
#endif


#ifdef DEBUG
#ifdef ALLOW_MPI
      WRITE(jp_stddbg, cfmt_a, ADVANCE="NO") 'Process '
      WRITE(jp_stddbg, '(I0, 1X)', ADVANCE="NO") i_myrank
      WRITE(jp_stddbg,
     &  '("(MPI parallel execution with ", I0, " processes)")')
     &  n_cprocs

      WRITE(jp_stddbg, cfmt_a, ADVANCE="NO")
     &  'Number of seafloor sediment grids (this process): '
      WRITE(jp_stddbg, '(I0)') n_grid_seafloor

      ALLOCATE(nsedcol_pproc(0:n_cprocs-1))
      ALLOCATE(ioffset_sedcol_pproc(0:n_cprocs-1))

      CALL MEDEXE_MPI_GETPARTITIONING(nsedcol_seafloor_global,
     &                              iproc_1stocn,
     &                              nsedcol_pproc,
     &                              ioffset_sedcol_pproc)

      WRITE(c_fmt,'("(""(/""",I0,"(1X,I0),1X,""/)"")")') n_cprocs

      WRITE(jp_stddbg, cfmt_a, ADVANCE="NO")
     &  'Number of seafloor sediment grids per process: '
      WRITE(jp_stddbg, c_fmt) nsedcol_pproc

      WRITE(jp_stddbg, cfmt_a, ADVANCE="NO")
     &  'Offsets of seafloor sediment grids per process: '
      WRITE(jp_stddbg, c_fmt) ioffset_sedcol_pproc

      WRITE(jp_stddbg, cfmt_a, ADVANCE="NO")
     &  'Total number of seafloor sediment grids (all processes): '
      WRITE(jp_stddbg, '(I0)') nsedcol_seafloor_global

      WRITE(jp_stddbg, cfmt_a, ADVANCE="NO")
     &  'Rank of first process with valid ocean-seafloor grid points: '
      WRITE(jp_stddbg, '(I0)') iproc_1stocn

      DEALLOCATE(nsedcol_pproc)
      DEALLOCATE(ioffset_sedcol_pproc)
#else
      WRITE(jp_stddbg, cfmt_a, ADVANCE="NO")
     &  'Number of seafloor sediment grids: '
      WRITE(jp_stddbg, '(I0)') n_grid_seafloor
#endif
#endif

      l_usedefault_cnpoh = .FALSE.  ! By default, no default values!


                                    ! Check validity of dcnpoh_x_host data
      ifnomcompo_checkvalidity:
     & IF (nomcompo > 0) THEN
                                    ! Either
                                    ! - dcnpoh_c_host is present, and then
                                    !   all of the dcnpoh_[n,p,o,h,remin_o2]_host
                                    !   dummy args must be present, and have the
                                    !   same shape as dcnpoh_c_host
                                    ! Or
                                    ! - dcnpoh_c_host is not present, and then
                                    !   default values from MOD_MATERIALCHARAS
                                    !   are used; any dcnpoh_[n,p,o,h,remin_o2]_host
                                    !   that would be present will be ignored

                                    ! Start checking with dcnpoh_c_host

        IF (.NOT. PRESENT(dcnpoh_c_host)) THEN

          WRITE(jp_stderr, cfmt_modprocname_a) 'Warning'
          WRITE(jp_stderr, cfmt_a)
     &      'Missing argument "dcnpoh_c_host"' //
     &      ' -- falling back to default values for ALL "dcnpoh_*"!'
          l_usedefault_cnpoh = .TRUE.

        ELSE

          ishape_dcnpoh = SHAPE(dcnpoh_c_host)
          ngrid_cnpoh = ishape_dcnpoh(2)

                                    ! ngrid_cnpoh can be equal to 1 or
                                    ! to nix:
          IF ((ANY(ishape_dcnpoh(:) /= (/ nomcompo, 1 /))) .AND.
     &        (ANY(ishape_dcnpoh(:) /= (/ nomcompo, nix /))))
     &    THEN

            WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
            WRITE(jp_stderr, cfmt_a)
     &        '"dcnpoh_c_host" has an illegal shape'
            WRITE(jp_stderr, cfmt_a_ind, ADVANCE="NO")
            WRITE(jp_stderr, cfmt_as, ADVANCE="NO")
     &                                ' * expected: ', nomcompo, 1
            WRITE(jp_stderr, cfmt_as) ' or ', nomcompo, nix
            WRITE(jp_stderr, cfmt_a_ind, ADVANCE="NO")
            WRITE(jp_stderr, cfmt_as) ' * actually: ', ishape_dcnpoh
            WRITE(jp_stderr, cfmt_a_ind) 'Aborting!'
            CALL ABORT_MEDUSA()

          ENDIF

        ENDIF


        ifdefaultcnpoh_checkvalidity:
     &  IF (l_usedefault_cnpoh) THEN
                                    ! We are using default values:
                                    ! all other dcnpoh_x_host that are
                                    ! present are ignored. Issue warning.

          IF (PRESENT(dcnpoh_n_host)) THEN
            WRITE(jp_stderr, cfmt_modprocname_a) 'Warning'
            WRITE(jp_stderr, cfmt_a)
     &        'Ignoring argument "dcnpoh_n_host"' //
     &        ' -- using default values!'
          ENDIF

          IF (PRESENT(dcnpoh_p_host)) THEN
            WRITE(jp_stderr, cfmt_modprocname_a) 'Warning'
            WRITE(jp_stderr, cfmt_a)
     &        'Ignoring argument "dcnpoh_p_host"' //
     &        ' -- using default values!'
          ENDIF

          IF (PRESENT(dcnpoh_o_host)) THEN
            WRITE(jp_stderr, cfmt_modprocname_a) 'Warning'
            WRITE(jp_stderr, cfmt_a)
     &        'Ignoring argument "dcnpoh_o_host"' //
     &        ' -- using default values!'
          ENDIF

          IF (PRESENT(dcnpoh_h_host)) THEN
            WRITE(jp_stderr, cfmt_modprocname_a) 'Warning'
            WRITE(jp_stderr, cfmt_a)
     &        'Ignoring argument "dcnpoh_h_host"' //
     &        ' -- using default values!'
          ENDIF

          IF (PRESENT(dcnpoh_remin_o2_host)) THEN
            WRITE(jp_stderr, cfmt_modprocname_a) 'Warning'
            WRITE(jp_stderr, cfmt_a)
     &        'Ignoring argument "dcnpoh_remin_o2_host"' //
     &        ' -- using default values!'
          ENDIF


        ELSE ifdefaultcnpoh_checkvalidity
                                    ! Not using default values:
                                    ! all other dcnpoh_x_host must be
                                    ! present and have the same shape
                                    ! as dcnpoh_c_host: check for this!
          IF (.NOT. PRESENT(dcnpoh_n_host)) THEN

            WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
            WRITE(jp_stderr, cfmt_a)
     &        'Missing argument "dcnpoh_n_host" -- aborting!'
            CALL ABORT_MEDUSA()

          ELSE

            ishape_dcnpoh_x = SHAPE(dcnpoh_n_host)

            IF (ANY(ishape_dcnpoh_x(:) /= ishape_dcnpoh(:))) THEN
              WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
              WRITE(jp_stderr, cfmt_a)
     &          'Incompatible shape of "dcnpoh_n_host"'
              WRITE(jp_stderr, cfmt_a_ind, ADVANCE="NO")
              WRITE(jp_stderr, cfmt_as) ' * expected: ',
     &          ishape_dcnpoh
              WRITE(jp_stderr, cfmt_a_ind, ADVANCE="NO")
              WRITE(jp_stderr, cfmt_as) ' * actually: ',
     &          ishape_dcnpoh_x
              WRITE(jp_stderr, cfmt_a_ind) 'Aborting!'
              CALL ABORT_MEDUSA()
            ENDIF

          ENDIF


          IF (.NOT. PRESENT(dcnpoh_p_host)) THEN

            WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
            WRITE(jp_stderr, cfmt_a)
     &        'Missing argument "dcnpoh_p_host" -- aborting!'
            CALL ABORT_MEDUSA()

          ELSE

            ishape_dcnpoh_x = SHAPE(dcnpoh_p_host)

            IF (ANY(ishape_dcnpoh_x(:) /= ishape_dcnpoh(:))) THEN
              WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
              WRITE(jp_stderr, cfmt_a)
     &          'Incompatible shape of "dcnpoh_p_host"'
              WRITE(jp_stderr, cfmt_a_ind, ADVANCE="NO")
              WRITE(jp_stderr, cfmt_as) ' * expected: ',
     &          ishape_dcnpoh
              WRITE(jp_stderr, cfmt_a_ind, ADVANCE="NO")
              WRITE(jp_stderr, cfmt_as) ' * actually: ',
     &          ishape_dcnpoh_x
              WRITE(jp_stderr, cfmt_a_ind) 'Aborting!'
              CALL ABORT_MEDUSA()
            ENDIF

          ENDIF


          IF (.NOT. PRESENT(dcnpoh_o_host)) THEN

            WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
            WRITE(jp_stderr, cfmt_a)
     &        'Missing argument "dcnpoh_o_host" -- aborting!'
            CALL ABORT_MEDUSA()

          ELSE

            ishape_dcnpoh_x = SHAPE(dcnpoh_o_host)

            IF (ANY(ishape_dcnpoh_x(:) /= ishape_dcnpoh(:))) THEN
              WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
              WRITE(jp_stderr, cfmt_a)
     &          'Incompatible shape of "dcnpoh_o_host"'
              WRITE(jp_stderr, cfmt_a_ind, ADVANCE="NO")
              WRITE(jp_stderr, cfmt_as) ' * expected: ',
     &          ishape_dcnpoh
              WRITE(jp_stderr, cfmt_a_ind, ADVANCE="NO")
              WRITE(jp_stderr, cfmt_as) ' * actually: ',
     &          ishape_dcnpoh_x
              WRITE(jp_stderr, cfmt_a_ind) 'Aborting!'
              CALL ABORT_MEDUSA()
            ENDIF

          ENDIF


          IF (.NOT. PRESENT(dcnpoh_h_host)) THEN

            WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
            WRITE(jp_stderr, cfmt_a)
     &        'Missing argument "dcnpoh_h_host" -- aborting!'
            CALL ABORT_MEDUSA()

          ELSE

            ishape_dcnpoh_x = SHAPE(dcnpoh_h_host)

            IF (ANY(ishape_dcnpoh_x(:) /= ishape_dcnpoh(:))) THEN
              WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
              WRITE(jp_stderr, cfmt_a)
     &          'Incompatible shape of "dcnpoh_h_host"'
              WRITE(jp_stderr, cfmt_a_ind, ADVANCE="NO")
              WRITE(jp_stderr, cfmt_as) ' * expected: ',
     &          ishape_dcnpoh
              WRITE(jp_stderr, cfmt_a_ind, ADVANCE="NO")
              WRITE(jp_stderr, cfmt_as) ' * actually: ',
     &          ishape_dcnpoh_x
              WRITE(jp_stderr, cfmt_a_ind) 'Aborting!'
              CALL ABORT_MEDUSA()
            ENDIF

          ENDIF


          IF (.NOT. PRESENT(dcnpoh_remin_o2_host)) THEN

            WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
            WRITE(jp_stderr, cfmt_a)
     &        'Missing argument "dcnpoh_remin_o2_host" -- aborting!'
            CALL ABORT_MEDUSA()

          ELSE

            ishape_dcnpoh_x = SHAPE(dcnpoh_remin_o2_host)

            IF (ANY(ishape_dcnpoh_x(:) /= ishape_dcnpoh(:))) THEN
              WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
              WRITE(jp_stderr, cfmt_a)
     &          'Incompatible shape of "dcnpoh_remin_o2_host"'
              WRITE(jp_stderr, cfmt_a_ind, ADVANCE="NO")
              WRITE(jp_stderr, cfmt_as) ' * expected: ',
     &          ishape_dcnpoh
              WRITE(jp_stderr, cfmt_a_ind, ADVANCE="NO")
              WRITE(jp_stderr, cfmt_as) ' * actually: ',
     &          ishape_dcnpoh_x
              WRITE(jp_stderr, cfmt_a_ind) 'Aborting!'
              CALL ABORT_MEDUSA()
            ENDIF

          ENDIF

        ENDIF ifdefaultcnpoh_checkvalidity

      ELSE ifnomcompo_checkvalidity

        IF (PRESENT(dcnpoh_c_host)) THEN
          WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
          WRITE(jp_stderr, cfmt_a)
     &      'Optional dummy argument "dcnpoh_c_host" ' //
     &      'not allowed with nomcompo = 0 -- aborting!'
          CALL ABORT_MEDUSA()
        ENDIF

        IF (PRESENT(dcnpoh_n_host)) THEN
          WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
          WRITE(jp_stderr, cfmt_a)
     &      'Optional dummy argument "dcnpoh_n_host" ' //
     &      'not allowed with nomcompo = 0 -- aborting!'
          CALL ABORT_MEDUSA()
        ENDIF

        IF (PRESENT(dcnpoh_p_host)) THEN
          WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
          WRITE(jp_stderr, cfmt_a)
     &      'Optional dummy argument "dcnpoh_p_host" ' //
     &      'not allowed with nomcompo = 0 -- aborting!'
          CALL ABORT_MEDUSA()
        ENDIF

        IF (PRESENT(dcnpoh_o_host)) THEN
          WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
          WRITE(jp_stderr, cfmt_a)
     &      'Optional dummy argument "dcnpoh_o_host" ' //
     &      'not allowed with nomcompo = 0 -- aborting!'
          CALL ABORT_MEDUSA()
        ENDIF

        IF (PRESENT(dcnpoh_h_host)) THEN
          WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
          WRITE(jp_stderr, cfmt_a)
     &      'Optional dummy argument "dcnpoh_h_host" ' //
     &      'not allowed with nomcompo = 0 -- aborting!'
          CALL ABORT_MEDUSA()
        ENDIF

        IF (PRESENT(dcnpoh_remin_o2_host)) THEN
          WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
          WRITE(jp_stderr, cfmt_a)
     &      'Optional dummy argument "dcnpoh_remin_o2_host" ' //
     &      'not allowed with nomcompo = 0 -- aborting!'
          CALL ABORT_MEDUSA()
        ENDIF


      ENDIF ifnomcompo_checkvalidity


                                    ! Set up the grid geometry
      CALL SETUP_GRID

                                    ! Retrieve grid type
      iselect_grid = SELECTED_GRIDTYPE()

                                    ! Retrieve porosity profile type
      iselect_phi  = SELECTED_PHITYPE()

                                    ! Allocate space for the variables

      SELECT CASE(iselect_grid)     ! - depending on the type of grid (static, dynamic),
      CASE(jp_grid_static_global)   !   1 or ngrid effective columns are required for
        ngrid_columns = 1           !   the grid-related arrays (xzdn)
#ifdef ALLOW_MPI
        IF (ngrid == 0) ngrid_columns = 0   ! exception for multi-processor environments
#endif

      CASE(jp_grid_static_local, jp_grid_dynamic)
        ngrid_columns = ngrid

      CASE DEFAULT
        WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
        WRITE(jp_stderr, cfmt_a)
     &    'MEDUSA grid not yet set-up -- aborting!'
        CALL ABORT_MEDUSA()

      END SELECT

      SELECT CASE(iselect_phi)      ! - depending on the type of porosity profile (static,
      CASE(jp_phi_static_global)    !   dynamic), 1 or ngrid effective columns are required for
        nphi_columns = 1            !   the phi-related arrays (xphi, xdphi, yphi)
#ifdef ALLOW_MPI
        IF (ngrid == 0) nphi_columns = 0    ! exception for multi-processor environments
#endif
      CASE DEFAULT
        nphi_columns = ngrid
      END SELECT


      ALLOCATE(reaclay_xzdn  (idnw:idnb, ngrid_columns))
      ALLOCATE(reaclay_xzdv  (idvw:idvb, ngrid_columns))

      ALLOCATE(reaclay_xphi  (idnt:idnb, nphi_columns))
      ALLOCATE(reaclay_xdphi (idnt:idnb, nphi_columns))
      ALLOCATE(reaclay_xvphi (idvs:idvb, nphi_columns))
      ALLOCATE(reaclay_xvdphi(idvs:idvb, nphi_columns))
      ALLOCATE(tranlay_yphi  (           nphi_columns))

      ALLOCATE(reaclay_profiles(idnw:idnb, 1:ncompo, ngrid))

      ALLOCATE(reaclay_xwtot(idvs:idvb, ngrid))
      ALLOCATE(reaclay_xutot(idvw:idvb, ngrid))

      ALLOCATE(reaclay_bconc   (1:nsolut, ngrid))
      ALLOCATE(reaclay_wcflx   (1:nsolut, ngrid))
      ALLOCATE(reaclay_wcflx_bi(1:nsolut, ngrid))

      ALLOCATE(reaclay_bfflx (1:nsolid, ngrid))
      ALLOCATE(tranlay_solids(1:nsolid, ngrid))


                                    ! Forcings
                                    ! --------

      ALLOCATE(seafloor_wdata (ngrid))

      ALLOCATE(seafloor_wsolutes(1:nsolut, ngrid))
      ALLOCATE(seafloor_wfflx   (1:nsolid, ngrid))


                                    ! Surface characteristics
                                    ! -----------------------

      ALLOCATE(seafloor_surf(ngrid))


                                    ! Grid characteristics
                                    ! --------------------

      ALLOCATE(maskval_seafloor(ngrid))
      ALLOCATE(ix_seafloor(ngrid))
      ALLOCATE(icolumn_seafloor(nix))


      k = 0


      DO i = 1, nix

        IF (imask_ocean_host(i) > 0) THEN

          k = k + 1
          ix_seafloor(k)      = i
          icolumn_seafloor(i) = k
          seafloor_surf(k)    = darea_gridelts_host(i)
          maskval_seafloor(k) = imask_ocean_host(i)

        ELSE

          icolumn_seafloor(i) = jp_is_not_at_seafloor_idx

        ENDIF

      ENDDO


                                    ! * Geographically distributed
                                    !   OrgMatter Redfield ratios
                                    !   and molar masses

      IF (nomcompo > 0) THEN

        ALLOCATE(seafloor_omcnp_c(nomcompo, ngrid))
        ALLOCATE(seafloor_omcnp_n(nomcompo, ngrid))
        ALLOCATE(seafloor_omcnp_p(nomcompo, ngrid))
        ALLOCATE(seafloor_omcnp_o(nomcompo, ngrid))
        ALLOCATE(seafloor_omcnp_h(nomcompo, ngrid))
        ALLOCATE(seafloor_omcnp_remin_o2(nomcompo, ngrid))
        ALLOCATE(seafloor_omcnp_mol(nomcompo, ngrid))


        DO i_column = 1, ngrid

          IF (.NOT. l_usedefault_cnpoh) THEN
                                    ! If we are not using the default
                                    ! values from MOD_MATERIALCHARAS,
                                    ! then transcribe the requested
                                    ! stoechiometric constants' values
                                    ! into MOD_MATERIALCHARAS now.
            k = MIN(ix_seafloor(i_column), ngrid_cnpoh)

#include "seafloor_setup_1D-cnpoh.F"

          ENDIF
                                    ! Call SOLIDS_STOECHIOMETRY
                                    ! to regularize the data and adjust
                                    ! molar masses accordingly.
          CALL SOLIDS_STOECHIOMETRY

                                    ! Register the complete material
                                    ! data into the seafloor_omcnp_*
                                    ! arrays in MOD_SEAFLOOR_CENTRAL.
          CALL SAVE_MATERIALCHARAS(i_column, iflag)

        ENDDO

      ENDIF



      CALL GRID_DEF(xzdn, xzdv)     ! Initial grid-point and vertex distribution

      CALL POROTORTUOSITY_DEF(xzdn, xzdv)   ! Initializes all the vars in
                                            ! MOD_MILIEUCHARAS (xphi, xdphi, ...)

      DO i_column = 1, ngrid_columns
        CALL SAVE_COLUMN(i_column, iflag, xzdn = xzdn, xzdv = xzdv)
      ENDDO

      DO i_column = 1, nphi_columns
        CALL SAVE_MILIEUCHARAS(i_column, iflag)
      ENDDO

![XXX] Temporary fix
      DEALLOCATE(imask_ocean_host)


#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_modprocname_a) 'End'
      WRITE(jp_stddbg, '()')
#endif


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE SEAFLOOR_SETUP
!-----------------------------------------------------------------------
