!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


!=======================================================================
      MODULE MOD_FVUPWIND_PARAMS
!=======================================================================


      USE MOD_GRIDPARAM,            ONLY: ndn
      USE MOD_INDEXPARAM,           ONLY: ncompo, nsolid,
     &                                    ncat1compo, ncat2compo


      IMPLICIT NONE


      PRIVATE :: nvars_cat1_node, nvars_cat2_node,
     &           neqns_cat1_node, neqns_cat2_node


      INTEGER, PARAMETER :: nvars_cat1_node  = ncat1compo
      INTEGER, PARAMETER :: nvars_cat2_node  = ncat2compo

      INTEGER, PARAMETER
     &  :: nvars_node  = nvars_cat1_node + nvars_cat2_node


      INTEGER, PARAMETER :: neqns_cat1_node  = nvars_cat1_node
      INTEGER, PARAMETER :: neqns_cat2_node  = nvars_cat2_node

      INTEGER, PARAMETER
     &  :: neqns_node = neqns_cat1_node + neqns_cat2_node


      INTEGER, PARAMETER :: nvars_total = ndn * nvars_node
      INTEGER, PARAMETER :: neqns_total = ndn * neqns_node


                                    ! YLambda related stuff
      DOUBLE PRECISION, DIMENSION(nsolid, nsolid), SAVE :: ylambda
      LOGICAL, SAVE :: l_ylambda_initialized = .FALSE.


                                    ! Scaling related stuff
      DOUBLE PRECISION, DIMENSION(ncompo)               :: ac_scale0
      LOGICAL, SAVE :: l_no_scaling = .TRUE.
      LOGICAL, SAVE :: l_scaling_initialized = .FALSE.


      CONTAINS


!-----------------------------------------------------------------------
      SUBROUTINE FVUPWIND_INIT_YLAMBDA
!-----------------------------------------------------------------------

      USE MOD_DEFINES_MEDUSA,       ONLY: jp_realm_tranlay
      USE MOD_INDEXPARAM,           ONLY: ncompo, jf_to_io, jc_to_io
      USE MOD_PROCESSDATA,          ONLY: nproc
      USE MOD_RREAC,                ONLY: DREACRATE,
     &                                    rreac_factor, rreac_factor_max

      IMPLICIT NONE


      DOUBLE PRECISION                           :: azdn
      DOUBLE PRECISION                           :: aphi
      DOUBLE PRECISION, DIMENSION(ncompo)        :: ac
      DOUBLE PRECISION, DIMENSION(ncompo,ncompo) :: aread
      DOUBLE PRECISION, DIMENSION(nproc)         :: rreac_factor_save


      IF (l_ylambda_initialized) RETURN

                                    ! Retrieve the decay matrix for the
                                    ! TRANLAY realm and store its
                                    ! components related to solids into
                                    ! ylambda. This matrix will be set
                                    ! from the results of a call to DREACRATE.
                                    ! Argument values:
      azdn = 0.10D+00               ! - dummy value, actually not used
      aphi = 0.0D+00                ! - must pretend pure solid phase here
      ac(jf_to_io(:)) =  1.0D+00    ! - dummy non-zero concentrations for solids
      ac(jc_to_io(:)) = -1.0D+00    ! - negative concentrations for solutes
                                    !   (these force the reaction rate
                                    !   derivatives to zero where solutes are
                                    !   involved, thus filtering these out).

      rreac_factor_save(:) = rreac_factor(:)  ! Save current rreac_factor
      rreac_factor(:) = rreac_factor_max      ! and set to maximum

      CALL DREACRATE(jp_realm_tranlay, azdn, aphi, ac, aread)
      ylambda(:,:) = aread(jf_to_io(:),jf_to_io(:))

      rreac_factor(:) = rreac_factor_save(:)  ! Restore rreac_factor

      l_ylambda_initialized = .TRUE.


      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE FVUPWIND_INIT_YLAMBDA
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE FVUPWIND_INIT_SCALES
!-----------------------------------------------------------------------


      USE mod_execontrol_medusa,     ONLY: ABORT_MEDUSA
      USE mod_defines_medusa
      USE mod_logunits
      USE mod_gridparam,             ONLY: ndn, idnw, idnt, idnb
      USE mod_indexparam
      USE mod_seafloor_central
#ifdef ALLOW_MPI
      USE mpi
      USE mod_execontrol_medusa,     ONLY: MEDEXE_MPI_COMM,
     &                                     MEDEXE_MPI_COMM_RANK,
     &                                     jp_exeproc_root
#endif


      IMPLICIT NONE


      CHARACTER(LEN=*), PARAMETER ::
     &   cfn_scalesinit = "medusa_compo_scales.nml"

                                    ! Default values for the scales
      DOUBLE PRECISION, PARAMETER :: dp_scale0_solid   = 2.6D+03 ! [kg/m3_solid]
      DOUBLE PRECISION, PARAMETER :: dp_scale0_solut   = 1.0D+00 ! [mol/m3_porew]
      DOUBLE PRECISION, PARAMETER :: dp_scale0_solidpt = 1.0D+06 ! [yr]

      INTEGER :: istatus
      INTEGER :: iuinit

      INTEGER :: i_compo, i_solid
      INTEGER :: i_mcompo, i_msolid
      INTEGER :: i_ptsolid
      INTEGER :: i_flag

      LOGICAL :: l_inifile_is_mine
      LOGICAL :: l_exists


#ifdef ALLOW_MPI
      INTEGER :: i_mycomm, i_myrank
#endif


#include "fvupwind_init_scales-decl.F"


      CHARACTER(LEN=*), PARAMETER :: cfmt_modprocname_a
     &  = '("[MOD_FVUPWIND_PARAMS/fvupwind_init_scales]: ", A)'

      CHARACTER(LEN=*), PARAMETER :: cfmt_a     = '(" - ", A)'
      CHARACTER(LEN=*), PARAMETER :: cfmt_a_ind = '("   ", A)'


!- - - - - - - - - - -
! End of declarations
!- - - - - - - - - - -


!----------------------------------------------------------------------
! Subroutine Start
!----------------------------------------------------------------------


#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_modprocname_a) 'Start'
#endif

      IF (l_scaling_initialized) THEN
#ifdef DEBUG
        WRITE(jp_stddbg, cfmt_modprocname_a) 'Return @ 1'
        WRITE(jp_stddbg, '()')
#endif
        RETURN
      ENDIF


#ifdef ALLOW_MPI
      i_mycomm = MEDEXE_MPI_COMM()
      i_myrank = MEDEXE_MPI_COMM_RANK()
      l_inifile_is_mine = (i_myrank == jp_exeproc_root)
#else
      l_inifile_is_mine = .TRUE.
#endif


      IF (l_inifile_is_mine) THEN

#include "fvupwind_init_scales-start.F"


#ifdef DEBUG
        WRITE(jp_stddbg, cfmt_a, ADVANCE="NO") 'ac_scale0(:) = '
        WRITE(jp_stddbg, *) ac_scale0(:)
#endif

        DO i_ptsolid = 1, nptsolid

          i_solid = jpf_to_if(i_ptsolid)
          i_compo = jf_to_io(i_solid)

          i_msolid = jpf_to_ifm(i_ptsolid)
          i_mcompo = jf_to_io(i_msolid)

#ifdef DEBUG
          WRITE(jp_stddbg, cfmt_a, ADVANCE="NO")
          
          WRITE(jp_stddbg, '("PT scale (", I0, ") adjusted from ")',
     &                              ADVANCE="NO") i_compo
          WRITE(jp_stddbg, '(E13.6)', ADVANCE="NO") ac_scale0(i_compo)
#endif

          ac_scale0(i_compo) = ac_scale0(i_compo)*ac_scale0(i_mcompo)

#ifdef DEBUG
          WRITE(jp_stddbg, '(" to ")', ADVANCE="NO")
          WRITE(jp_stddbg, '(E13.6)') ac_scale0(i_compo)
#endif

        ENDDO

#ifdef DEBUG
        WRITE(jp_stddbg, cfmt_a, ADVANCE="NO") 'l_no_scaling = '
        WRITE(jp_stddbg, '(L1)') l_no_scaling
#endif

      ENDIF


#ifdef ALLOW_MPI
                                    ! Broadcast the data read in to
                                    ! all ranks
      CALL MPI_BCAST(l_no_scaling, 1, MPI_LOGICAL,
     &                              jp_exeproc_root, i_mycomm, i_flag)
      CALL MPI_BCAST(ac_scale0(:), ncompo, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, i_flag)
#endif

      l_scaling_initialized = .TRUE.

#ifdef DEBUG
        WRITE(jp_stddbg, cfmt_modprocname_a) 'End'
        WRITE(jp_stddbg, '()')
#endif


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE FVUPWIND_INIT_SCALES
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE FVUPWIND_SCALES(xzdn, jf_svc, xc_scale, eqn_scale)
!-----------------------------------------------------------------------

      USE mod_indexparam
      USE mod_gridparam,            ONLY: idnw, idnt, idnz, idnb,
     &                                          idvs,       idvb
      USE mod_materialcharas,       ONLY: eq_mol_alk
      USE mod_milieucharas,         ONLY: xvphi, xvtor2
      USE mod_transport,            ONLY: dcf_molion,
     &                                    dcf_biotur, dcf_biointer
      USE mod_equilibsubr


      IMPLICIT NONE

      ! Dummy variables (argument list)
      ! -------------------------------

      DOUBLE PRECISION, DIMENSION(idnw:idnb),
     &                                     INTENT(IN)  :: xzdn
      INTEGER, DIMENSION(idnt:idnb),       INTENT(IN)  :: jf_svc
      DOUBLE PRECISION, DIMENSION(ncompo,idnw:idnb),
     &                                     INTENT(OUT) :: xc_scale
      DOUBLE PRECISION, DIMENSION(neqns_node,idnw:idnb),
     &                                     INTENT(OUT) :: eqn_scale


! Local variables
! ---------------

      INTEGER                             :: inode, ieqn
      INTEGER                             :: io_svc

      DOUBLE PRECISION                    :: aphi_scale
      DOUBLE PRECISION, DIMENSION(ncompo) :: acx_scale
      DOUBLE PRECISION                    :: time_scale_solids
      DOUBLE PRECISION, DIMENSION(nsolut) :: time_scale_solutes

#include "fvupwind_scales-dae_decl.F"



! End of declarations
! -------------------

      IF (.NOT. l_scaling_initialized) CALL FVUPWIND_INIT_SCALES


      IF (l_no_scaling) THEN

        xc_scale(:,:)  = 1.0D+00
        eqn_scale(:,:) = 1.0D+00

        RETURN

      ENDIF


                                    ! Auxiliary variables
                                    ! ===================

                                    ! Time scales
                                    ! -----------

                                    !  - solids (biodiffusion)
      time_scale_solids = (xzdn(idnz)-xzdn(idnt))**2
     &                              / MAXVAL(dcf_biotur(:))

                                    !  - solutes (diffusion)
      time_scale_solutes(:) = (xzdn(idnb)-xzdn(idnt))**2
     &                              / dcf_molion(:)
     &                              * MINVAL(xvtor2(:))


                                    ! Porosity scale value
                                    ! --------------------

      aphi_scale = (xvphi(idvs) + xvphi(idvb)) / 2.0D+00


                                    ! "hat"-concentration scales
                                    ! --------------------------
                                    ! divided by the equation time scale,
                                    ! set to the diffusion time scale

      acx_scale(jf_to_io(:)) = ac_scale0(jf_to_io(:))
     &                              * (1.0D+00 - aphi_scale)
     &                                / time_scale_solids
      acx_scale(jc_to_io(:)) = ac_scale0(jc_to_io(:))
     &                              * aphi_scale
     &                                / time_scale_solutes(:)


                                    ! Component scales
                                    ! ----------------

      DO inode = idnw, idnb
        xc_scale(:, inode) = ac_scale0(:)
      ENDDO


                                    ! Equation scales
                                    ! ---------------

      ! eqn_syst(jc_to_io(:), idnw) = xc(jc_to_io(:), idnw) - wconc(:)
      eqn_scale(jc_to_io(:), idnw) = ac_scale0(jc_to_io(:))

      IF (idnw < idnt) THEN

        ! Node idnw
        ! ---------

        ! eqn_syst(jf_to_io(:), idnw) = xc(jf_to_io(:), idnw)

        eqn_scale(jf_to_io(:), idnw) = ac_scale0(jf_to_io(:))


        ! Nodes idnw+1, ..., idnt-1
        ! -------------------------

        DO inode = idnw+1, idnt-1

          ! eqn_syst(jc_to_io(:), inode) = normal equation
          ! eqn_syst(jf_to_io(:), inode) = xc(jf_to_io(:), inode)
          eqn_scale(jc_to_io(:), inode) = acx_scale(jc_to_io(:))
          eqn_scale(jf_to_io(:), inode) = ac_scale0(jf_to_io(:))

        ENDDO

        ! Node idnt (solutes)
        ! ---------

        ! eqn_syst(jc_to_io(:), idnt) = normal equation
        eqn_scale(jc_to_io(:), inode) = acx_scale(jc_to_io(:))

      ELSE

        ! Node idnt (solutes)
        ! ---------

        ! eqn_syst(jc_to_io(:), idnt) = xc(jc_to_io(:), idnt) - wconc(:)
        eqn_scale(jc_to_io(:), idnt) = ac_scale0(jc_to_io(:))

      ENDIF


      ! Node idnt (solids)
      ! ---------

      ! eqn_syst(jf_to_io(:), idnt) = normal equation
      eqn_scale(jf_to_io(:), idnt) = acx_scale(jf_to_io(:))

#ifndef SOLVSED_NO_SVC_EQUATION
                                    ! Adjust scale for static volume
                                    ! conservation equation (already O(1))
      io_svc = jf_to_io(jf_svc(idnt))
      eqn_scale(io_svc, idnt) = 1.0D+00
#endif

      ! Nodes idnt+1, ..., idnb
      ! -----------------------

      DO inode = idnt+1, idnb

        ! eqn_syst(:, inode) = normal equation
        eqn_scale(:, inode) = acx_scale(:)

#ifndef SOLVSED_NO_SVC_EQUATION
                                    ! Adjust scale for static volume
                                    ! conservation equation (already O(1))
        io_svc = jf_to_io(jf_svc(inode))
        eqn_scale(io_svc, inode) = 1.0D+00
#endif

      ENDDO


#include "fvupwind_scales-dae_substitutions.F"


      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE FVUPWIND_SCALES
!-----------------------------------------------------------------------


!=======================================================================
      END MODULE MOD_FVUPWIND_PARAMS
!=======================================================================

