!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


************************************************************************
      SUBROUTINE STORE_NC_AUX(filename, title_string)
************************************************************************

! Add surface area information
      USE mod_gridparam
      USE mod_indexparam
      USE mod_materialcharas
      USE mod_milieucharas, ONLY : xphi, POROTORTUOSITY_DEF

      USE mod_seafloor_central, ONLY: N_COLUMNS_USED, IJ_COLUMNS_USED,
     &                              COLUMN_IJ2N, COLUMN_N2IJ,
     &                              COLUMN_AREA4N
      USE mod_defines_medusa, ONLY: jp_stderr

      USE mod_netcdfinc

      USE mod_netcdfparam


      IMPLICIT NONE


      CHARACTER(len=*) :: filename, title_string

      INTENT(IN) :: filename, title_string
      OPTIONAL :: title_string

      INTEGER :: n_grid_seafloor, ndprf, nditv
      INTEGER :: iflag

      DOUBLE PRECISION, DIMENSION(idnw:idnb) ::  xzdn
      DOUBLE PRECISION, DIMENSION(idvw:idvb) ::  xzdv


      INTEGER, DIMENSION(3) :: dim, start, count

      INTEGER :: status, k
      INTEGER :: ncid, iccf
      INTEGER :: i, j, n

! Dimensions and dimension variables
!  'lev' (levels)
      INTEGER :: dim_lev      ! dimension ID
      INTEGER ::  id_lev      ! ID of dimension variable
! 'col' (columns)
      INTEGER :: dim_col      ! dimension ID
      INTEGER ::  id_col      ! ID of dimension variable
! 'idprf' (MBM depth profile index)
      INTEGER :: dim_idprf
      INTEGER ::  id_idprf
! 'jditv' (MBM depth interval index)
      INTEGER :: dim_jditv
      INTEGER ::  id_jditv

      DOUBLE PRECISION, PARAMETER :: badsfelt = 1D36

! Data variables
      INTEGER ::  id_xzdn      ! Depth coordinates
      INTEGER ::  id_xphi      ! porosity profiles
      INTEGER ::  id_col4ij    ! column index n for grid element (i,j)
      INTEGER ::  id_i4col     ! index i(n)
      INTEGER ::  id_j4col     ! index j(n)
      INTEGER ::  id_area      ! surface areas of columns (n)

      DOUBLE PRECISION, ALLOCATABLE, DIMENSION(:,:) :: xlon_ij
      DOUBLE PRECISION, ALLOCATABLE, DIMENSION(:)   :: xlat_ij

      DOUBLE PRECISION                         :: sfc_area

! STORE_NC_AUX is outside the module, and so has to use the interface
      CALL N_COLUMNS_USED(n_grid_seafloor)
      CALL IJ_COLUMNS_USED(ndprf, nditv)

! If n_grid_seafloor == -1, the basic setup has not yet been
! done, and it does not make any sense to proceed
      IF(n_grid_seafloor == -1) THEN
        WRITE(jp_stderr,*) 'STORE_NC_AUX: n_grid_seafloor == -1!'
        WRITE(jp_stderr,*) '              Setup not yet done. Aborting!'
        CALL ABORT()
      ENDIF

!-----------------------
! Create the data file
!-----------------------
      status = NF_CREATE(TRIM(filename), nf_share, ncid)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)


!-----------------------
! Define dimensions and dimension variables
!-----------------------
!
! Columns on the string ('col')              dim_col, n_grid_seafloor
! Levels ('lev' )                            dim_lev, ndn
! 'Depth profile' index of MBM ('idprf')     dim_idprf, ndprf
! 'Depth interval' index of MBM ('jditv')    dim_jditv, nditv
!
! (Id's of dimension variables for dim_xxx are id_xxx)

! Columns on the string
      status = NF_DEF_DIM(ncid,
     &                    'col', n_grid_seafloor,
     &                    dim_col)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
      status = NF_DEF_VAR(ncid, 'col', nf_int,
     &                    1, dim_col,
     &                    id_col)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)

! Levels
      status = NF_DEF_DIM(ncid,
     &                    'lev', ndn,
     &                    dim_lev)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
      status = NF_DEF_VAR(ncid, 'lev', nf_int,
     &                    1, dim_lev,
     &                    id_lev)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
      status = NF_PUT_ATT_TEXT(ncid, id_lev, 'positive',
     &                       4, 'down')
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)


! 'Depth profile' index of MBM
      status = NF_DEF_DIM(ncid, 'idprf',
     &                    ndprf, dim_idprf)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)

      status = NF_DEF_VAR(ncid, 'idprf', nf_int,
     &                    1, dim_idprf, id_idprf)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)

      status = NF_PUT_ATT_TEXT(ncid, id_idprf, 'long_name',
     &                         23, 'MBM_Depth_Profile_Index')
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)


! 'Depth interval' index of MBM
      status = NF_DEF_DIM(ncid, 'jditv',
     &                    nditv, dim_jditv)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)

      status = NF_DEF_VAR(ncid, 'jditv', nf_int,
     &                    1, dim_jditv, id_jditv)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)

      status = NF_PUT_ATT_TEXT(ncid, id_jditv, 'long_name',
     &                         24, 'MBM_Depth_Interval_Index')
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)



         
!-----------------------
! Define data variables
!-----------------------

! xzdn(lev)
      status = NF_DEF_VAR(ncid, vsn_xzdn, nf_double,
     &                    1, dim_lev,
     &                    id_xzdn)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
      status = NF_PUT_ATT_TEXT(ncid, id_xzdn,
     &                         'long_name', vll_xzdn, vln_xzdn)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
      status = NF_PUT_ATT_TEXT(ncid, id_xzdn,
     &                         'units', ul_m, un_m)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)


! xphi(lev)
      status = NF_DEF_VAR(ncid, 'poro', nf_double,
     &                    1, dim_lev,
     &                    id_xphi)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
      status = NF_PUT_ATT_TEXT(ncid, id_xphi, 'long_name',
     &                         8, 'Porosity')
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
      status = NF_PUT_ATT_TEXT(ncid, id_xphi, 'units',
     &                         30, 'm3 porewater m-3 bulk sediment')
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)


! col4ij(idprf,jditv)
      dim(1) = dim_idprf
      dim(2) = dim_jditv
      
      status = NF_DEF_VAR(ncid, 'col4ij', nf_int,
     &                    2, dim(1:2),
     &                    id_col4ij)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
      status = NF_PUT_ATT_TEXT(ncid, id_col4ij, 'long_name',
     &                         17, 'Column_index_of_ij')
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
      status = NF_PUT_ATT_INT(ncid, id_col4ij,
     &                        '_FillValue', nf_int,
     &                        1, -1)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
      status = NF_PUT_ATT_INT(ncid, id_col4ij,
     &                        'valid_min', nf_int,
     &                        1, 1)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
      status = NF_PUT_ATT_INT(ncid, id_col4ij,
     &                        'valid_max', nf_int,
     &                        1, n_grid_seafloor)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)


! i4col(col)
      status = NF_DEF_VAR(ncid, 'i4col', nf_int,
     &                    1, dim_col,
     &                    id_i4col)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
      status = NF_PUT_ATT_TEXT(ncid, id_i4col, 'long_name',
     &                  41, 'MBM_Depth_Profile_Index_from_Column_Index')
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
      status = NF_PUT_ATT_INT(ncid, id_i4col,
     &                        '_FillValue', nf_int,
     &                        1, -1)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
      status = NF_PUT_ATT_INT(ncid, id_i4col,
     &                        'valid_min', nf_int,
     &                        1, 1)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
      status = NF_PUT_ATT_INT(ncid, id_i4col,
     &                        'valid_max', nf_int,
     &                        1, ndprf)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)


! j4col(col)
      status = NF_DEF_VAR(ncid, 'j4col', nf_int,
     &                    1, dim_col,
     &                    id_j4col)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
      status = NF_PUT_ATT_TEXT(ncid, id_j4col, 'long_name',
     &                 42, 'MBM_Depth_Interval_Index_from_Column_Index')
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
      status = NF_PUT_ATT_INT(ncid, id_j4col,
     &                        '_FillValue', nf_int,
     &                        1, -1)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
      status = NF_PUT_ATT_INT(ncid, id_j4col,
     &                        'valid_min', nf_int,
     &                        1, 1)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
      status = NF_PUT_ATT_INT(ncid, id_j4col,
     &                        'valid_max', nf_int,
     &                        1, nditv)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)



! area(col)
      status = NF_DEF_VAR(ncid, 'area', nf_double,
     &                    1, dim_col,
     &                    id_area)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
      status = NF_PUT_ATT_TEXT(ncid, id_area, 'long_name',
     &                   12, 'Surface_Area')
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
      status = NF_PUT_ATT_TEXT(ncid, id_area, 'units',
     &                         ul_m2, un_m2)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)


!----------------------
! Put global attributes
!----------------------

      IF(PRESENT(title_string)) THEN
        status=NF_PUT_ATT_TEXT(ncid,nf_global,'title',
     &                          LEN_TRIM(title_string),title_string)
        IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
      ENDIF


!--------------------
! End define mode
!--------------------
      
      status = NF_ENDDEF(ncid)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)


      DO k = 1, n_grid_seafloor
        i = k
        status = NF_PUT_VAR1_INT(ncid, id_col, i, k)
        IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
      ENDDO

      DO k = idnw, idnb
        i = k-idnw+1
        status = NF_PUT_VAR1_INT(ncid, id_lev, i, k)
        IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
      ENDDO

      DO k = 1, ndprf
        i = k
        status = NF_PUT_VAR1_INT(ncid, id_idprf, i, k)
        IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
      ENDDO

      DO k = 1, nditv
        i = k
        status = NF_PUT_VAR1_INT(ncid, id_jditv, i, k)
        IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
      ENDDO


! xzdn(lev)
      CALL GRID_DEF(xzdn, xzdv)
      status = NF_PUT_VAR_DOUBLE(ncid, id_xzdn, xzdn)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)

! xphi(lev)
      CALL POROTORTUOSITY_DEF(xzdn, xzdv)
      status = NF_PUT_VAR_DOUBLE(ncid, id_xphi, xphi)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
        

! col4ij
      DO i = 1, ndprf
        start(1) = i
        DO j = 1, nditv
          start(2) = j
          CALL COLUMN_IJ2N(i, j, iflag, k)
          IF (iflag == 0) THEN
            status = NF_PUT_VAR1_INT(ncid, id_col4ij,
     &                               start(1:2), k)
            IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
          ENDIF
        ENDDO
      ENDDO


! icol, jcol
      DO k = 1, n_grid_seafloor
        CALL COLUMN_N2IJ(k, iflag, i, j)
        IF (iflag == 0) THEN
          status = NF_PUT_VAR1_INT(ncid, id_i4col, k, i)
          IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
          status = NF_PUT_VAR1_INT(ncid, id_j4col, k, j)
          IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
        ELSE
          WRITE(jp_stderr,*)
     &     'STORE_NC_AUX: Inconsistency regarding COLUMN_N2IJ call:'
          WRITE(jp_stderr,*)
     &     '              called with k = ',k
          WRITE(jp_stderr,*)
     &     '              returned iflag = ', iflag
          CALL ABORT()
        ENDIF
      ENDDO




! Surface areas of grid_elements
      DO k = 1, n_grid_seafloor
        CALL COLUMN_AREA4N(k, iflag, sfc_area)
        status = NF_PUT_VAR1_DOUBLE(ncid, id_area, k, sfc_area)
        IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)
      ENDDO


      status = NF_CLOSE(ncid)
      IF (status /= nf_noerr) CALL HANDLE_ERRORS(status)

      RETURN
      END SUBROUTINE STORE_NC_AUX
