!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


!***********************************************************************
      SUBROUTINE SETWFL
     >                 (temps, wflxvn, wflxvm, wflxhn, wflxhm,
     >                  cofvn, cofvm, cofhn, cofhm,
     <                  wflux, wfluxi, wfluxo)
!***********************************************************************

! This subroutine determines the current water flux distribution
! wflux from the read data, as a function of time.

      USE mod_mbm_geometry, ONLY: nro1, nro2, nro3, ro1, ro2, ro3,
     >                            nivmer

      IMPLICIT NONE


! Input variables

      DOUBLE PRECISION temps

! Water flux distribution data.
! These must not be changed here!
      DOUBLE PRECISION, DIMENSION(ro1+1:ro3+nro3, ro1+1:ro3+nro3),
     >   INTENT(IN)
     >   :: wflxvn, wflxvm, wflxhn, wflxhm
      DOUBLE PRECISION, INTENT(IN)
     >   :: cofvn, cofvm, cofhn, cofhm


! Output variables

! Water fluxes actually used in the calculations,
! determined here.
      DOUBLE PRECISION, DIMENSION(ro1+1:ro3+nro3, ro1+1:ro3+nro3),
     >   INTENT(OUT)
     >   ::  wflux
      DOUBLE PRECISION, DIMENSION(ro1+1:ro3+nro3), INTENT(OUT)
     >   :: wfluxi, wfluxo


! Local variables

      DOUBLE PRECISION, DIMENSION(ro1+1:ro3+nro3, ro1+1:ro3+nro3)
     >   :: wflux_mix, wflux_adv
      DOUBLE PRECISION
     >   :: wflux_mixmax


! sverdr: 1 Sv [10^18 m^3/yr] (1Sv = 10^6 m^3/s = 3.15576D+13 m^3/yr)
! sverd1: = sverdr/1000.

      DOUBLE PRECISION, PARAMETER
     &   :: sverdr=3.15576D-05, sverd1=3.15576D-08


      INTEGER i, j


! Perturbations (as a function of sea-level)
! - scaling parameters:
!   notice: nivmer < 0 if lower than at present-day
!   (typically, nivmer = -130m at the Last Glacial Maximum)

      DOUBLE PRECISION, PARAMETER ::  nivmer_high =    0D0
      DOUBLE PRECISION, PARAMETER ::  nivmer_low  = -130D0

! - scaling factor(s):
      DOUBLE PRECISION ::  nivmer_factor
      


! First determine the basic mixing and net advection components
! of the wflux field (Units: 10^18 m^3/yr) from the data read
! in from the DAT file

      wflux_mix(:,:) = (cofvm*wflxvm(:,:) + cofhm*wflxhm(:,:)) * sverd1
      wflux_adv(:,:) = (cofvn*wflxvn(:,:) + cofhn*wflxhn(:,:)) * sverd1


! ----------------------------------------------------------------------
! Changes to individual fluxes to be put below this line ---------------
! ----------------------------------------------------------------------

! nivmer < 0 if lower than at present-day
! - 0 <= nivmer_factor <= 1,
! - nivmer_factor = 0 for nivmer = nivmer_high 
! - nivmer_factor = 1 for nivmer = nivmer_low

!      nivmer_factor = (nivmer - nivmer_high)/(nivmer_low-nivmer_high)
      nivmer_factor = 0D0

! and now for the actual changes

!! Example: reduce the net transport from SANT [ro1+3] to DANT [ro3+2]
!! by up to 8 Sverdrup; transmit changes onto the loop
!! SANT-DANT-DATL-TEATL-SANT

!         wflux_adv(ro1+3,ro3+2)
!     & = wflux_adv(ro1+3,ro3+2) - 8D0*sverdr*nivmer_factor
!         wflux_adv(ro3+1,ro3+2)
!     & = wflux_adv(ro3+1,ro3+2) + 8D0*sverdr*nivmer_factor
!         wflux_adv(ro3+1,ro2+1)
!     & = wflux_adv(ro3+1,ro2+1) - 8D0*sverdr*nivmer_factor
!         wflux_adv(ro1+3,ro2+1)
!     & = wflux_adv(ro1+3,ro2+1) + 8D0*sverdr*nivmer_factor


!! and reduce the mixing SANT-DANT to zero

!         wflux_mix(ro1+3,ro3+2)
!     & = wflux_mix(ro1+3,ro3+2)*(1D0-nivmer_factor)
!         wflux_mix(ro3+2,ro1+3)
!     & = wflux_mix(ro1+3,ro3+2)

! ----------------------------------------------------------------------
! ... and above this one  ----------------------------------------------
! ----------------------------------------------------------------------


! Now regularize the fluxes: we require that
! * wflux_adv(i,j) >= 0  for all i,j
! * if wflux_adv(i,j) > 0  then wflux_adv(j,i) = 0, for all i,j
! * wflux_adv(i,i) = 0, for all i
! * wflux_mix(i,j) >= 0, for all i,j
! * wflux_mix(i,j) = wflux_mix(j,i), for all i,j
! * wflux_mix(i,i) = 0, for all i

! To decide on changes, we use the greatest in absolute value, if
! both have the same sbsolute value, set the one for j>i to zero.

      DO i = ro1+1, ro3+nro3

         ! Set the diagonal terms to zero
         wflux_adv(i,i) = 0.D0
         wflux_mix(i,i) = 0.D0
         
         DO j = ro1+1, i-1


            ! Mixing fluxes first
            ! -------------------

            ! If both fluxes are different from zero then set
            ! set the one with the smallest absolute value to zero

            IF (wflux_mix(i,j) /= wflux_mix(i,j)) THEN
               wflux_mixmax
     &          = MAX(ABS(wflux_mix(i,j)),ABS(wflux_mix(j,i)))
               wflux_mix(i,j) = wflux_mixmax
               wflux_mix(j,i) = wflux_mixmax
            ELSE
               IF(wflux_mix(i,j) <0D0) THEN
                 wflux_mix(i,j) = -wflux_mix(i,j)
                 wflux_mix(j,i) = -wflux_mix(j,i)
               ENDIF
            ENDIF


            ! Advection fluxes
            ! ----------------

            ! If both fluxes are different from zero then
            ! set that with the smallest absolute value to zero
            IF ((wflux_adv(i,j) /= 0) .AND. (wflux_adv(j,i) /= 0)) THEN
               IF(ABS(wflux_adv(i,j)) < ABS(wflux_adv(j,i))) THEN
                  wflux_adv(i,j) = 0D0
               ELSE
                  wflux_adv(j,i) = 0D0
               ENDIF
            ENDIF

            ! If the (i,j) component is strictly negative now, reflect it
            ! onto the (j,i) component and set the (i,j) component to zero.
            IF (wflux_adv(i,j) < 0D0) THEN
               wflux_adv(j,i) = -wflux_adv(i,j)
               wflux_adv(i,j) = 0D0
            ENDIF

            ! If the (j,i) component is strictly negative, reflect it
            ! onto the (i,j) component and set the (j,i) component to zero.
            IF (wflux_adv(j,i) < 0D0) THEN
               wflux_adv(i,j) = -wflux_adv(j,i)
               wflux_adv(j,i) = 0D0
            ENDIF


         ENDDO
      ENDDO

! Now the actual wflx(:,:) can be calculated      

      wflux(:,:) = wflux_mix(:,:) + wflux_adv(:,:)


! Then calculate wfluxi(i) (water flux into reservoir i)
! and wfluxo(i) (water flux out of reservoiri) from wflux(i,j).

      wfluxo(:) = 0.D+00
      wfluxi(:) = 0.D+00

      DO i = ro1+1, ro3+nro3
         DO  j = ro1+1, ro3+nro3
             wfluxo(i) = wfluxo(i)+wflux(i,j)
             wfluxi(i) = wfluxi(i)+wflux(j,i)
         ENDDO
      ENDDO


!      WRITE(31,*) "Time: ", temps
!      WRITE(31,*) "Nivmer: ", nivmer
!      WRITE(31,*) "Nivmer_factor ", nivmer_factor
!      WRITE(31,*) "Mix ro1+3,ro3+2 / ro3+2,ro1+3",
!     >   wflux_mix(ro1+3,ro3+2)/sverdr, wflux_mix(ro3+2,ro1+3)/sverdr
!      WRITE(31,*) "Adv ro1+3,ro3+2 / ro3+2,ro1+3",
!     >   wflux_adv(ro1+3,ro3+2)/sverdr, wflux_adv(ro3+2,ro1+3)/sverdr
!      WRITE(31,*) "Adv ro1+3,ro2+1 / ro2+1,ro1+3",
!     >   wflux_adv(ro1+3,ro2+1)/sverdr, wflux_adv(ro2+1,ro1+3)/sverdr
!      WRITE(31,*) "Adv ro2+1,ro3+1 / ro3+1,ro2+1",
!     >   wflux_adv(ro2+1,ro3+1)/sverdr, wflux_adv(ro3+1,ro2+1)/sverdr
!      WRITE(31,*) "Adv ro3+1,ro3+2 / ro3+2,ro3+1",
!     >   wflux_adv(ro3+1,ro3+2)/sverdr, wflux_adv(ro3+2,ro3+1)/sverdr
!      WRITE(31,*) wfluxi/sverdr
!      WRITE(31,*) wfluxo/sverdr

      RETURN
      END
!...:....1....:....2....:....3....:....4....:....5....:....6....:....7.|
