!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#ifdef FN_THISFILE
#undef FN_THISFILE
#endif
#define FN_THISFILE "setgeo.F"
#ifndef __LINE__
#define __LINE__ 0
#endif
c     ******************************************************************
      SUBROUTINE SETGEO(updtts)
c     ******************************************************************
c
c     This subroutine determines the model geometry from
c     the depth profiles: it calculates the volumes and 
c     surface areas of the different reservoirs.
c     Required variables: temps 
c                         hypsar, hypsvl
c                         tsr87,sr87,tdo18,do18
c                         ttatl,tatl,ttant,tant,ttpac,tpac
c     Useful data, but not truly necessary:
c                         itatl,itant,ipac
c     Calculated things:  temp,salin
c                         k0,k1,k2,kb,borate,ksp,koxy
c                         volum,surf,propor
c                         nivmer
c
      USE mod_mbm_files, ONLY: erruni
      USE mod_mbm_geometry
      USE mod_mbm_tempo
      USE mod_mbm_paleod
      USE mod_mbm_wfluxes
      USE mod_mbm_chimiq

      USE mod_mbm_biocarbonates, ONLY: ARAG_FRACTION_ITP,
     &                                 CARB_RAINRATIO_ITP

      IMPLICIT NONE



      LOGICAL updtts

      DOUBLE PRECISION    hv0(ro1+1:ro1+nro1),   ha0(ro1+1:ro1+nro1)
      DOUBLE PRECISION  hv100(ro1+1:ro1+nro1), ha100(ro1+1:ro1+nro1)
      DOUBLE PRECISION hv1000(ro1+1:ro1+nro1),ha1000(ro1+1:ro1+nro1)


      DOUBLE PRECISION tsurf,tvolum,d1000,d100,d0,rd1000,rd100,rd0
      DOUBLE PRECISION tperio,eperio
      DOUBLE PRECISION :: sealevel_jtp, dsealevel_itp
      DOUBLE PRECISION :: forcingfunc1_jtp
      INTEGER id1000,id100,id0,i,i1,i2,j,j1,j2

      DOUBLE PRECISION rho

! Function-name dummy variables
      DOUBLE PRECISION AK0,AK1,AK2,AKB,AKW,
     >                 AKCALC,TOTBOR,TOTO2,RHOSW

      CHARACTER(LEN=*), PARAMETER ::
     &  fn_thisfile     = FN_THISFILE
      CHARACTER(LEN=*), PARAMETER ::
     &  fmt_info_a      = '("['//fn_thisfile//']: ", A)' ,
     &  fmt_info_ai     = '("['//fn_thisfile//']: ", A, I0)',
     &  fmt_infolin_ia  = '("['//fn_thisfile//':", I0, "]: ", A)'


      IF (updtts) THEN

! ---------
! Sea-level
! ---------
      IF(n_sealevel == 1) THEN      ! If there is only one single
                                    ! datum, use it as a constant.
        sealevel_itp  = sealevel(1,2)
        dsealevel_itp = 0D0

      ELSE

        IF(i_sealevel == -1) i_sealevel = 1

        IF(temps > sealevel(n_sealevel,1)) THEN
          WRITE(erruni,fmt_infolin_ia) (__LINE__),
     &                    'temps > sealevel(n_sealevel,1)'
          WRITE(erruni,*) 'temps = ', temps
          WRITE(erruni,*) 'sealevel(n_sealevel,1) = ', 
     &                     sealevel(n_sealevel,1)
          WRITE(erruni,fmt_infolin_ia) (__LINE__), 'Aborting!'
          CALL ABORT()
        ELSEIF(temps < sealevel(1,1)) THEN
          WRITE(erruni,*) 'temps < sealevel(1,1)'
          WRITE(erruni,*) 'temps = ', temps
          WRITE(erruni,*) 'sealevel(1,1) = ', 
     &                     sealevel(1,1)
          WRITE(erruni,fmt_infolin_ia) (__LINE__), 'Aborting!'
          CALL ABORT()
        ELSE
                                    ! Search for i_sealevel such that
                                    ! sealevel(i_sealevel,1)
                                    !   <= temps < sealevel(i_sealevel+1,1)
          DO WHILE(temps < sealevel(i_sealevel,1))
!            WRITE(STDDBG,*) '[setgeo.F:99] ', i_sealevel,
!     &                      sealevel(i_sealevel,1), temps,
!     &                      i_sealevel-1
            i_sealevel = i_sealevel-1
          ENDDO

          IF (i_sealevel < n_sealevel) THEN
            DO WHILE(temps >= sealevel(i_sealevel+1,1))
!              WRITE(STDDBG,*)  '[setgeo.F:106] ', i_sealevel,
!     &                        sealevel(i_sealevel+1,1), temps,
!     &                        i_sealevel+1
              IF (i_sealevel == (n_sealevel-1)) THEN
                EXIT
              ELSE
                i_sealevel = i_sealevel+1
              ENDIF
            ENDDO
          ENDIF

          i1=i_sealevel
          i2=i_sealevel+1

!          WRITE(STDDBG,*) '[setgeo.F:115] si- ', sealevel(i1,1), temps,
!     &                           sealevel(i2,1)

          sealevel_itp = sealevel(i1,2)
     &         + (sealevel(i2,2) - sealevel(i1,2))*
     &           (         temps - sealevel(i1,1))
     &          /(sealevel(i2,1) - sealevel(i1,1))

                                    ! Calculate variation between
                                    ! 'temps' and 'temps+epoch'
          j = i_sealevel
          DO WHILE((temps+epoch) >= sealevel(j+1,1))
!            WRITE(STDDBG,*) '[setgeo.F:127] ', j,sealevel(j+1,1),
!     &                      temps+epoch, j+1
            IF (j == (n_sealevel-1)) EXIT
            j = j+1
          ENDDO

          j1=j
          j2=j+1

!          WRITE(STDDBG,*) '[setgeo.F:136] sj- ', sealevel(j1,1),
!     &                           temps+epoch, sealevel(j2,1)

          sealevel_jtp = sealevel(j1,2)
     &         + (sealevel(j2,2) - sealevel(j1,2))*
     &           ( (temps+epoch) - sealevel(j1,1))
     &          /(sealevel(j2,1) - sealevel(j1,1))

          dsealevel_itp = (sealevel_jtp - sealevel_itp)/epoch

          sealevel_itp = (sealevel_itp + sealevel_jtp)/2D0

        ENDIF
      ENDIF



! --------------------
! Carbonate Rain Ratio
! --------------------

      CALL CARB_RAINRATIO_ITP(temps)


! --------------------------------
! Aragonite Fraction of Carbonates
! --------------------------------

      CALL ARAG_FRACTION_ITP(temps)


! ------------
! Temperatures
! ------------

      IF(n_temp == 1) THEN
                                    ! If there is only one temperature set
        temp(:) = temp_val(:,1)     ! (single time), use it as a constant.

      ELSE

        IF(i_temp == -1) i_temp = 1

        IF(temps > temp_time(n_temp)) THEN
          WRITE(erruni,fmt_infolin_ia) (__LINE__),
     &                    'temps > temp_time(n_temp)'
          WRITE(erruni,*) 'temps = ', temps
          WRITE(erruni,*) 'temp_time(n_temp) = ', 
     &                                 temp_time(n_temp)
          WRITE(erruni,fmt_infolin_ia) (__LINE__), 'Aborting!'
          CALL ABORT()
        ELSEIF(temps < temp_time(1)) THEN
          WRITE(erruni,*) 'temps < temp_time(1)'
          WRITE(erruni,*) 'temps = ', temps
          WRITE(erruni,*) 'temp_time(1) = ', 
     &                                 temp_time(1)
          WRITE(erruni,fmt_infolin_ia) (__LINE__), 'Aborting!'
          CALL ABORT()
        ELSE
                                    ! Search for i_temp such that
                                    ! temp_time(i_temp) <= temps < temp_time(i_temp+1)
          DO WHILE(temps < temp_time(i_temp))
!            WRITE(STDDBG,*) i_temp,
!     &                      temp_time(i_temp), temps,
!     &                      i_temp-1
            i_temp = i_temp-1
          ENDDO

          IF (i_temp < n_temp) THEN
            DO WHILE(temps >= temp_time(i_temp+1))
!             WRITE(STDDBG,*) i_temp,
!     &                       temp_time(i_temp+1), temps,
!     &                       i_temp+1
              IF (i_temp == (n_temp-1)) THEN
                EXIT
              ELSE
                i_temp = i_temp+1
              ENDIF
            ENDDO
          ENDIF

          i1=i_temp
          i2=i_temp+1

!          WRITE(STDDBG,*) 't--', temp_time(i1), temps,
!     &                           temp_time(i2)

          temp(:) = temp_val(:,i1)
     >                + (temp_val(:,i2) - temp_val(:,i1))*
     >                  (temps          - temp_time( i1))
     >                 /(temp_time( i2) - temp_time( i1))
        ENDIF
      ENDIF

! -------------
! New sea-level
! -------------

      nivmer=  sealevel_itp
      dnimer= dsealevel_itp

! id1000 = index corresponding to 1000m depth
! id100  = index corresponding to  100m depth
! id0    = index corresponding to    0m depth
! hv1000 = hypsvl interpolated to actual 1000m depth below sea-level
! hv100  = hypsvl interpolated to actual  100m depth below sea-level
! hv0    = hypsvl interpolated to actual    0m depth below sea-level
! ha1000 = hypsar interpolated to actual 1000m depth below sea-level
! ha100  = hypsar interpolated to actual  100m depth below sea-level
! ha0    = hypsar interpolated to actual    0m depth below sea-level

! NOTICE: hypsvl(i,j) = total volume of water in the parts of the
!                       ocean where the seafloor is shallower
!                       than i*resol (resol = 100m) when
!                       sea-level = 0m.
! NOTICE: hypsar(i,j) = surface area of the parts of the
!                       ocean where the seafloor is shallower
!                       than i*resol (resol = 100m) when
!                       sea-level = 0m.
! NOTICE: nivmer = sea=level

      d1000  = DBLE(1000) - nivmer
      d100   = DBLE(100)  - nivmer
      d0     =            - nivmer
      rd1000 = d1000/resol
      rd100  =  d100/resol
      rd0    =    d0/resol
      id1000 = INT(rd1000)
      id100  = INT( rd100)
      id0    = INT(   rd0)


         hv0(ro1+1:ro1+nro1) =
     >                           hypsvl(   id0  , ro1+1:ro1+nro1)
     >   + (   rd0 -    id0) * ( hypsvl(   id0+1, ro1+1:ro1+nro1)
     >                          -hypsvl(   id0  , ro1+1:ro1+nro1))
       hv100(ro1+1:ro1+nro1) =
     >                           hypsvl( id100  , ro1+1:ro1+nro1)
     >   + ( rd100 -  id100) * ( hypsvl( id100+1, ro1+1:ro1+nro1)
     >                          -hypsvl( id100  , ro1+1:ro1+nro1))
      hv1000(ro1+1:ro1+nro1) =
     >                           hypsvl(id1000  , ro1+1:ro1+nro1)
     >   + (rd1000 - id1000) * ( hypsvl(id1000+1, ro1+1:ro1+nro1)
     >                          -hypsvl(id1000  , ro1+1:ro1+nro1))

         ha0(ro1+1:ro1+nro1) =
     >                           hypsar(   id0  , ro1+1:ro1+nro1)
     >   + (   rd0 -    id0) * ( hypsar(   id0+1, ro1+1:ro1+nro1)
     >                          -hypsar(   id0  , ro1+1:ro1+nro1))
       ha100(ro1+1:ro1+nro1) =
     >                           hypsar( id100  , ro1+1:ro1+nro1)
     >   + ( rd100 -  id100) * ( hypsar( id100+1, ro1+1:ro1+nro1)
     >                          -hypsar( id100  , ro1+1:ro1+nro1))
      ha1000(ro1+1:ro1+nro1) =
     >                           hypsar(id1000  , ro1+1:ro1+nro1)
     >   + (rd1000 - id1000) * ( hypsar(id1000+1, ro1+1:ro1+nro1)
     >                          -hypsar(id1000  , ro1+1:ro1+nro1))


!--------
! Volumes
!--------

! surface: 1000m - 0m (ATN,ANT,IPN)
!           100m - 0m (ATE,IPE)

      i = ro1+1
      volum(i)=(hv1000(i)-hv0(i)
     >        +hypsar(bottom,i)*1000.d+00
     >        -ha1000(i)*d1000+ha0(i)*d0)*1.D-18
      i = ro1+3
      volum(i)=(hv1000(i)-hv0(i)
     >        +hypsar(bottom,i)*1000.d+00
     >        -ha1000(i)*d1000+ha0(i)*d0)*1.D-18
      i = ro1+5
      volum(i)=(hv1000(i)-hv0(i)
     >        +hypsar(bottom,i)*1000.d+00
     >        -ha1000(i)*d1000+ha0(i)*d0)*1.D-18
      i = ro1+2
      volum(i)=( hv100(i)-hv0(i)
     >        +hypsar(bottom,i)*100.d+00
     >        - ha100(i)*d100 +ha0(i)*d0)*1.D-18
      i = ro1+4
      volum(i)=( hv100(i)-hv0(i)
     >        +hypsar(bottom,i)*100.d+00
     >        - ha100(i)*d100 +ha0(i)*d0)*1.D-18

! thermocline: 1000m - 100m (ATE,IPE)

      volum(ro2+1)=(hv1000(ro1+2)-hv100(ro1+2)
     >        +hypsar(bottom,ro1+2)*900.d+00
     >        -ha1000(ro1+2)*d1000+ha100(ro1+2)*d100)*1.d-18
      volum(ro2+2)=(hv1000(ro1+4)-hv100(ro1+4)
     >        +hypsar(bottom,ro1+4)*900.d+00
     >        -ha1000(ro1+4)*d1000+ha100(ro1+4)*d100)*1.d-18

! deep: deepest sea-floor - 1000m (ATL,ANT,IP)

      volum(ro3+1)=(hypsvl(bottom,ro1+1)-hv1000(ro1+1)
     >         -(hypsar(bottom,ro1+1)-ha1000(ro1+1))*d1000 +
     >              hypsvl(bottom,ro1+2)-hv1000(ro1+2)
     >         -(hypsar(bottom,ro1+2)-ha1000(ro1+2))*d1000  )*1.d-18
      volum(ro3+2)=(hypsvl(bottom,ro1+3)-hv1000(ro1+3)
     >         -(hypsar(bottom,ro1+3)-ha1000(ro1+3))*d1000  )*1.d-18
      volum(ro3+3)=(hypsvl(bottom,ro1+4)-hv1000(ro1+4)
     >         -(hypsar(bottom,ro1+4)-ha1000(ro1+4))*d1000 +
     >              hypsvl(bottom,ro1+5)-hv1000(ro1+5)
     >         -(hypsar(bottom,ro1+5)-ha1000(ro1+5))*d1000  )*1.d-18

      tvolum= SUM(volum(ro1+1:ro3+nro3))



! -------------
! Surface areas
! -------------

! surface: everything that is shallower than the deepest sea-floor,
!          but deeper than 0m actual depth

      surf(ro1+1:ro1+nro1) =
     >    hypsar(bottom,ro1+1:ro1+nro1) - ha0(ro1+1:ro1+nro1)

      tsurf = SUM(surf(ro1+1:ro1+nro1))
      propor(ro1+1:ro1+nro1) = surf(ro1+1:ro1+nro1)/tsurf

      propor(ro1+nro1)=DBLE(1) - SUM(propor(ro1+1:ro1+nro1-1))


! Chemical equilibrium constants (cf. Millero)
! Salinity is kept the same in all of the reservoirs,
! the total amount of salt in the ocean being conserved,
! with 35.4 g/kg for a total volume of 1.34138(18)m3 water

      DO i=ro1+1,ro2+nro2
         salin(i)=35.4d+00*1.34138d+00/tvolum
         rho = RHOSW(temp(i),salin(i),0.0d+00)
! k0 given (mol/kg-SW)/atm_CO2
! k0 required here in (mol/m3-SW)/atm_CO2
         k0(i)=0.331d-03*ak0(temp(i),salin(i),0.0d+00)*rho
         k1(i)=ak1(temp(i),salin(i),0.0d+00)*rho
         k2(i)=ak2(temp(i),salin(i),0.0d+00)*rho
         kb(i)=akb(temp(i),salin(i),0.0d+00)*rho
         kw(i)=akw(temp(i),salin(i),0.0d+00)*(rho**2)
         borate(i)=totbor(temp(i),salin(i),0.0D+00)*rho
! toto2 already in mol/m3-SW
         koxy(i)=TOTO2(temp(i),salin(i),0.0D+00)
         ksp(i)=AKCALC(temp(i),salin(i),0.0d+00)*(rho**2)
      ENDDO

! Deep reservoirs: pressure effect with mean pressure of 300 bars

      DO  i=ro3+1,ro3+nro3
         salin(i)=35.4d+00*1.34138d+00/tvolum
         rho = RHOSW(temp(i),salin(i),3000.0d+00)
         k0(i)=0.331d-03*ak0(temp(i),salin(i),3000.0d+00)*rho
         k1(i)=ak1(temp(i),salin(i),3000.0D+00)*rho
         k2(i)=ak2(temp(i),salin(i),3000.0D+00)*rho
         kb(i)=akb(temp(i),salin(i),3000.0D+00)*rho
         kw(i)=akw(temp(i),salin(i),3000.0D+00)*(rho**2)
         borate(i)=TOTBOR(temp(i),salin(i),3000.0D+00)*rho
         koxy(i)=TOTO2(temp(i),salin(i),3000.0D+00)
      ENDDO
   

! Water flux distribution

      CALL SETWFL
     >                 (temps, wflxvn, wflxvm, wflxhn, wflxhm,
     >                  cofvn, cofvm, cofhn, cofhm,
     <                  wflux, wfluxi, wfluxo)


      ENDIF ! (of IF(updtts)

      lstupd = temps
      updt01 = .TRUE.

! Before leaving, record the current time
! and tell *sediao* to update its chemical constants

      RETURN
      END
!-----*--1---------2---------3---------4---------5---------6---------7-*

!     ******************************************************************
      SUBROUTINE SETUPD(temps, updtts, iasd01, svsd01, biof01)
!     ******************************************************************

!     This subroutine sets the different flags that control the updating

      USE MOD_MBM_OPTIONS

      IMPLICIT NONE


      DOUBLE PRECISION temps
      LOGICAL updtts, iasd01, svsd01, biof01


      IF ((temps .GE. carwb1) .AND. (temps .LT. carwb0)) THEN
         ctcrbw = .TRUE.
      ELSE
         ctcrbw = .FALSE.
      ENDIF


      IF ((temps .GE. carws1) .AND. (temps .LT. carws0)) THEN
         ctcrsw = .TRUE.
      ELSE
         ctcrsw = .FALSE.
      ENDIF


      IF ((temps .GE. crlit1) .AND. (temps .LT. crlit0)) THEN
         ctcrli = .TRUE.
      ELSE
         ctcrli = .FALSE.
      ENDIF


      IF ((temps .GE. sirwt1) .AND. (temps .LT. sirwt0)) THEN
         ctsirw = .TRUE.
      ELSE
         ctsirw = .FALSE.
      ENDIF


      IF ((temps .GE. corlt1) .AND. (temps .LT. corlt0)) THEN
         ctcrlp = .TRUE.
      ELSE
         ctcrlp = .FALSE.
      ENDIF


      IF ((temps .GE. chbnt1) .AND. (temps .LT. chbnt0)) THEN
         ctshbp = .TRUE.
      ELSE
         ctshbp = .FALSE.
      ENDIF


      IF ((temps .GE. vatst1) .AND. (temps .LT. vatst0)) THEN
         updtts = .TRUE.
      ELSE
         updtts = .FALSE.
      ENDIF


      IF ((temps .GE. iasdt1) .AND. (temps .LT. iasdt0)) THEN
         iasd01 = .TRUE.
      ELSE
         iasd01 = .FALSE.
      ENDIF


      IF ((temps .GE. rcsdt1) .AND. (temps .LT. rcsdt0)) THEN
         svsd01 = .TRUE.
      ELSE
         svsd01 = .FALSE.
      ENDIF


      IF ((temps .GE. bioft1) .AND. (temps .LT. bioft0)) THEN
         biof01 = .TRUE.
      ELSE
         biof01 = .FALSE.
      ENDIF


      RETURN
      END
