!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#ifdef FN_THISFILE
#undef FN_THISFILE
#endif
#define FN_THISFILE "mod_mbm_shelffluxes.F90"
#ifndef __LINE__
#define __LINE__ 0
#endif
!=========================
MODULE mod_mbm_shelffluxes
!=========================

USE mod_mbm_geometry, ONLY: nro1

IMPLICIT NONE

PRIVATE

PUBLIC SHELFCARBONATES_FROMFILE

                                    ! n_shflux: number of shelf fluxes
                                    !           currently considered in this module
INTEGER, PARAMETER :: n_shflux  = 2
CHARACTER(LEN=31), DIMENSION(n_shflux), PARAMETER :: shflux_names = (/ 'coralo', 'shbnko' /)
INTEGER, PARAMETER :: i_shflux_coralo = 1
INTEGER, PARAMETER :: i_shflux_shbnko = 2

                                    ! n_shflux_xxxx: number of data-set
                                    !           entries to consider for xxxx
INTEGER, SAVE :: n_shflux_time   = 0
INTEGER, SAVE :: n_shflux_coralo = 0
INTEGER, SAVE :: n_shflux_shbnko = 0

LOGICAL, SAVE :: f_shelfsetup_tbd = .TRUE.



DOUBLE PRECISION, DIMENSION(:),   POINTER, SAVE   :: shflux_time   => NULL()
DOUBLE PRECISION, DIMENSION(:,:), POINTER, SAVE   :: shflux_coralo => NULL(), &
                                                     shflux_shbnko => NULL()
DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE, SAVE :: shflux_coralo_itp, &
                                                     shflux_shbnko_itp
DOUBLE PRECISION, DIMENSION(:),   POINTER, SAVE   :: shflux_coralom => NULL(), &
                                                     shflux_shbnkom => NULL()


INTEGER, SAVE :: i_shflux = -1


! Default values
DOUBLE PRECISION, PARAMETER :: shflux_period_dflt = 120D3

DOUBLE PRECISION, DIMENSION(nro1), PARAMETER :: shflux_coralo_dflt    &
           = (/ 0.00D-6, 1.03D-6, 0.00D-6, 2.81D-6, 0.00D-6 /)
DOUBLE PRECISION, DIMENSION(nro1), PARAMETER :: shflux_shbnko_dflt    &
           = (/ 0.00D-6, 1.87D-6, 0.00D-6, 5.11D-6, 0.00D-6 /)


CONTAINS

!=========================================================
SUBROUTINE SHELFCARBONATES_FROMFILE(temps, coralo, shbnko)
!=========================================================

USE mod_mbm_files, ONLY: dbguni, erruni, resusd, resuni, &
                         shfusd, shffil, HANDLE_NCERRORS
USE mod_mbm_ncforcing

IMPLICIT NONE

DOUBLE PRECISION, INTENT(IN) ::  temps
DOUBLE PRECISION, DIMENSION(nro1), INTENT(OUT) ::  coralo, shbnko


DOUBLE PRECISION :: shflux_factor


INTEGER :: i, j, i1, i2
LOGICAL :: use_some_dflt



TYPE(FORCINGDESC), DIMENSION(n_shflux) :: shflux_descs


CHARACTER(LEN=*), PARAMETER :: fn_thisfile     = FN_THISFILE
CHARACTER(LEN=*), PARAMETER :: fmt_info_a      = '("['//fn_thisfile//']: ", A)' , &
                               fmt_info_ai     = '("['//fn_thisfile//']: ", A, I0)', &
                               fmt_infolin_ia  = '("['//fn_thisfile//':", I0, "]: ", A)', &
                               fmt_infolin_iai = '("['//fn_thisfile//':", I0, "]: ", A, I0)'


shelfsetup_tbd: IF (f_shelfsetup_tbd) THEN
   WRITE(dbguni,*) 
   WRITE(dbguni,'("[MOD_MBM_SHELFFLUXES/SHELFCARBONATES_FROMFILE]:")')
   WRITE(dbguni,'(" Shelf fluxes")')

   IF (n_shflux_time /= 0) THEN     ! Reset everything
      IF(ASSOCIATED(shflux_time))      DEALLOCATE(shflux_time)
      IF(ASSOCIATED(shflux_coralo))    DEALLOCATE(shflux_coralo)
      IF(ASSOCIATED(shflux_shbnko))    DEALLOCATE(shflux_shbnko)
      IF(ASSOCIATED(shflux_coralom))   DEALLOCATE(shflux_coralom)
      IF(ASSOCIATED(shflux_shbnkom))   DEALLOCATE(shflux_shbnkom)

      IF(ALLOCATED(shflux_coralo_itp)) DEALLOCATE(shflux_coralo_itp)
      IF(ALLOCATED(shflux_shbnko_itp)) DEALLOCATE(shflux_shbnko_itp)

      n_shflux_time = 0
      n_shflux_coralo = 0
      n_shflux_shbnko = 0
   ENDIF

   use_some_dflt = .FALSE.          ! We expect to find data for all variables

   ifshfused: IF (shfusd == 1) THEN
                                    ! If we use shelf flux scenarios
                                    ! from a file then read them in now

      IF (resusd /= 0) THEN
         WRITE(resuni,*)
         WRITE(resuni,*)
         WRITE(resuni,'("[Shelf flux scenario info]:")')

         WRITE(resuni,'(A)') ' Reading in shelf flux scenario data from file "' // TRIM(shffil) // '"'
      ENDIF


      CALL READ_NCFORCING(shffil, shflux_period_dflt, &
                          shflux_names, shflux_time, shflux_descs)

      n_shflux_time = SIZE(shflux_time)



      IF (.NOT. shflux_descs(i_shflux_coralo)%use_dflt) THEN
         shflux_coralo  => shflux_descs(i_shflux_coralo)%forcing
         shflux_coralom => shflux_descs(i_shflux_coralo)%forcing_mean

         shflux_coralo(:,:) = shflux_coralo(:,:)*1D-6
         shflux_coralom(:)  = shflux_coralom(:)*1D-6

         n_shflux_coralo = SIZE(shflux_coralo) / nro1

      ELSE
         ALLOCATE(shflux_coralo(nro1, 1))
         ALLOCATE(shflux_coralom(nro1))
         shflux_coralo(:,1) = shflux_coralo_dflt(:)
         shflux_coralom(:)  = shflux_coralo_dflt(:)

         n_shflux_coralo = 1

         use_some_dflt = .TRUE.
      ENDIF

      IF (.NOT. shflux_descs(i_shflux_shbnko)%use_dflt) THEN
         shflux_shbnko  => shflux_descs(i_shflux_shbnko)%forcing
         shflux_shbnkom => shflux_descs(i_shflux_shbnko)%forcing_mean

         shflux_shbnko(:,:)  = shflux_shbnko(:,:)*1D-6
         shflux_shbnkom(:)   = shflux_shbnkom(:)*1D-6

         n_shflux_shbnko = SIZE(shflux_shbnko) / nro1

      ELSE
         ALLOCATE(shflux_shbnko(nro1, 1))
         ALLOCATE(shflux_shbnkom(nro1))
         shflux_shbnko(:,1) = shflux_shbnko_dflt(:)
         shflux_shbnkom(:)  = shflux_shbnko_dflt(:)

         n_shflux_shbnko = 1

         use_some_dflt = .TRUE.
      ENDIF

   ELSE ifshfused

                                    ! No scenario file given:
                                    ! use default values
      IF (resusd /= 0) THEN
         WRITE(resuni,'(" No SHF file given in mbm.cfg.")')
         WRITE(resuni,'(" Using default values")')

         WRITE(dbguni, *)
         WRITE(dbguni,'(" No SHF file given in mbm.cfg.")')
         WRITE(dbguni,'(" Using default values")')
      ENDIF

      ALLOCATE(shflux_coralo(nro1, 1))
      ALLOCATE(shflux_coralom(nro1))

      ALLOCATE(shflux_shbnko(nro1, 1))
      ALLOCATE(shflux_shbnkom(nro1))
 
      shflux_coralo(:,1) = shflux_coralo_dflt(:)
      shflux_shbnko(:,1) = shflux_shbnko_dflt(:)

      shflux_coralom(:)   = shflux_coralo_dflt(:)
      shflux_shbnkom(:)   = shflux_shbnko_dflt(:)

      n_shflux_coralo = 1
      n_shflux_shbnko = 1

      n_shflux_time   = 1

      use_some_dflt = .TRUE.

   ENDIF ifshfused


                                    ! Allocate space for the interpolates.
   ALLOCATE(shflux_coralo_itp(nro1))
   ALLOCATE(shflux_shbnko_itp(nro1))

                                    ! Pre-set all interpolates for
                                    ! fluxes that have only one datum.
   IF(n_shflux_coralo == 1) shflux_coralo_itp(:) = shflux_coralo(:,1)
   IF(n_shflux_shbnko == 1) shflux_shbnko_itp(:) = shflux_shbnko(:,1)

   WRITE(dbguni, *)
   WRITE(dbguni, '(" n_shflux_coralo = ", I0, " (time size: ", I0, ")")') &
     n_shflux_coralo, SIZE(shflux_coralo)/SIZE(shflux_coralo,1)
   WRITE(dbguni, '(" n_shflux_shbnko = ", I0, " (time size: ", I0, ")")') &
     n_shflux_shbnko, SIZE(shflux_shbnko)/SIZE(shflux_shbnko,1)


   IF (use_some_dflt) THEN
      IF (resusd /= 0) THEN
         WRITE(resuni,'(" Warning: one or more default values have been adopted.")')
         WRITE(resuni,'(" Consistency needs to be checked!")')
      ENDIF

      WRITE(dbguni, *)
      WRITE(dbguni, *) 'One or more default flux values adopted.'
   ENDIF


   f_shelfsetup_tbd = .FALSE.       ! Shelf flux setup is done now

   CALL FLUSH(dbguni)

ENDIF shelfsetup_tbd


! Interpolating the fluxes
! ------------------------

IF (n_shflux_time /= 1) THEN        ! We need to recalculate at least one interpolation

   IF (i_shflux == -1) i_shflux = 1


   IF (temps > shflux_time(n_shflux_time)) THEN
      WRITE(erruni,fmt_infolin_ia) (__LINE__), 'temps > shflux_time(n_shflux_time)'
      WRITE(erruni,*) ' temps = ', temps
      WRITE(erruni,*) ' shflux_time(n_shflux_time) = ', shflux_time(n_shflux_time)
      WRITE(erruni,fmt_infolin_ia) (__LINE__), 'Aborting!'
      CALL ABORT()
   ELSEIF(temps < shflux_time(1)) THEN
       WRITE(erruni,fmt_infolin_ia) (__LINE__), ' temps < shflux_time(1)'
       WRITE(erruni,*) ' temps = ', temps
       WRITE(erruni,*) ' shflux_time(1) = ',  shflux_time(1)
       WRITE(erruni,fmt_infolin_ia) (__LINE__), 'Aborting!'
       CALL ABORT()
   ELSE                             ! OK temps is acceptable


                                    ! Search for i_shflux such that
                                    ! shflux_time(i_shflux)
                                    !   <= temps < shflux_time(i_shflux_time+1)
      DO WHILE(temps < shflux_time(i_shflux))
!         WRITE(dbguni,*) i_shflux, shflux_time(i_shflux_teim,1), temps, i_shflux-1
         i_shflux = i_shflux-1
      ENDDO

      IF (i_shflux < n_shflux_time) THEN
         DO WHILE(temps >= shflux_time(i_shflux+1))
!           WRITE(dbguni,*) i_shflux, shflux_time(i_shflux+1,1), temps, i_shflux+1
            IF (i_shflux == (n_shflux_time-1)) THEN
               EXIT
            ELSE
               i_shflux = i_shflux+1
            ENDIF
         ENDDO
      ENDIF

      i1=i_shflux
      i2=i_shflux+1


      shflux_factor = (temps - shflux_time(i1)) / (shflux_time(i2) - shflux_time(i1))

      IF (n_shflux_coralo /= 1)  &
         shflux_coralo_itp(:) = shflux_coralo(:,i1) + (shflux_coralo(:,i2) - shflux_coralo(:,i1))*shflux_factor

      IF (n_shflux_shbnko /= 1)  &
         shflux_shbnko_itp(:) = shflux_shbnko(:,i1) + (shflux_shbnko(:,i2) - shflux_shbnko(:,i1))*shflux_factor
   ENDIF
ENDIF

coralo(:) = shflux_coralo_itp(:)
shbnko(:) = shflux_shbnko_itp(:)

!STOP ! for debugging purposes

RETURN

!======================================
END SUBROUTINE shelfcarbonates_fromfile
!======================================

!*****************************
END MODULE mod_mbm_shelffluxes
!*****************************
