!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#ifdef FN_THISFILE
#undef FN_THISFILE
#endif
#define FN_THISFILE "mod_mbm_ncforcing.F90"
#ifndef __LINE__
#define __LINE__ 0
#endif
!=======================================================================
MODULE MOD_MBM_NCFORCING
!=======================================================================

IMPLICIT NONE

INTEGER, PARAMETER :: ndimmax_forcing = 2

TYPE FORCINGDESC
LOGICAL                                   :: use_dflt
INTEGER, DIMENSION(:), POINTER            :: ishape
DOUBLE PRECISION, DIMENSION(:,:), POINTER :: forcing
DOUBLE PRECISION, DIMENSION(:), POINTER   :: forcing_mean
END TYPE FORCINGDESC




CONTAINS

   !--------------------------------------------------------------------
   SUBROUTINE READ_NCFORCING(forcing_file, period_dflt, &
                             forcing_names, forcing_time, forcing_descs)
   !--------------------------------------------------------------------

   USE MOD_MBM_TEMPO, ONLY: simul_tinf, simul_tmax, simul_t_set
   USE MOD_MBM_FILES, ONLY: erruni, dbguni, resusd, resuni, HANDLE_NCERRORS

   IMPLICIT NONE


   CHARACTER(LEN=*), INTENT(IN) :: forcing_file
   DOUBLE PRECISION, INTENT(IN) :: period_dflt
   CHARACTER(LEN=*), DIMENSION(:), INTENT(IN) :: forcing_names

   TYPE(FORCINGDESC), DIMENSION(:), INTENT(OUT) :: forcing_descs
   DOUBLE PRECISION, DIMENSION(:), POINTER :: forcing_time


   CHARACTER(LEN=*), PARAMETER :: fn_thisfile     = FN_THISFILE
   CHARACTER(LEN=*), PARAMETER :: fmt_info_a      = '("['//fn_thisfile//']: ", A)' , &
                                  fmt_info_ai     = '("['//fn_thisfile//']: ", A, I0)', &
                                  fmt_infolin_ia  = '("['//fn_thisfile//':", I0, "]: ", A)', &
                                  fmt_infolin_iai = '("['//fn_thisfile//':", I0, "]: ", A, I0)'

#include "netcdf.inc"

   INTEGER :: status, nc_id, ll, n_dim1
   INTEGER :: dating_dimid, dating_dimlen, dating_varid
   INTEGER :: anyfrc_varid, anyfrc_irank
   INTEGER, DIMENSION(ndimmax_forcing) :: anyfrc_shape, anyfrc_idims

   DOUBLE PRECISION, DIMENSION(:), POINTER :: ncforc_dates_orig
   DOUBLE PRECISION :: ncforc_tinf, ncforc_tmax, ncforc_period

   INTEGER :: n_ncforc              ! Number of fluxes [= SIZE(forcing_names)]
   INTEGER :: n_ncforc_time         ! NUMBER of time instants with data

   INTEGER :: n_flx_in_file         ! Number of fluxes for which the file
                                    ! actually includes data

   CHARACTER(LEN=80) :: c_title
   CHARACTER(LEN=31) :: c_varname

   INTEGER :: i, j, i1, i2, is, i_first, i_last
   INTEGER :: n_current
   INTEGER :: n_replicbfr, n_replicaft
   LOGICAL :: failed_replication_check


   DOUBLE PRECISION, DIMENSION(:,:), POINTER :: ncforc_anyfrc_orig
   DOUBLE PRECISION, DIMENSION(:,:), POINTER :: ncforc_anyfrc
   DOUBLE PRECISION, DIMENSION(:), POINTER   :: ncforc_anyfrcm

   IF(.NOT.simul_t_set) THEN
     WRITE(erruni,fmt_infolin_ia) (__LINE__), 'Simulation time interval not yet set. Aborting!'
     CALL ABORT()
   ENDIF

   failed_replication_check = .TRUE. ! Pre-set to "Replication not possible"

   n_ncforc = SIZE(forcing_names)

   WRITE(dbguni,*)
   WRITE(dbguni,'("[MOD_MBM_NCFORCING/READ_NCFORCING]:")')
   WRITE(dbguni,'(" Got request to read ",I0, " forcing variables from file ", A)')  n_ncforc, forcing_file


                                    ! Open NetCDF File
   status = NF_OPEN(forcing_file, NF_NOWRITE, nc_id)
   IF (status /= NF_NOERR) CALL HANDLE_NCERRORS(status, fn_thisfile, (__LINE__-1))

   IF (resusd /= 0) THEN

      status = NF_GET_ATT_TEXT(nc_id, NF_GLOBAL, 'title', c_title)
      IF (status /= NF_NOERR) THEN
         IF (status /= NF_ENOTATT) THEN
            CALL HANDLE_NCERRORS(status, fn_thisfile, (__LINE__-3))
         ELSE
            WRITE(resuni,'(" No title attribute found in file.")')
         ENDIF

      ELSE

         status = NF_INQ_ATTLEN(nc_id, NF_GLOBAL, 'title', ll)
         IF (status /= NF_NOERR) CALL HANDLE_NCERRORS(status, fn_thisfile, (__LINE__-1))

         WRITE(resuni,'(" Title from file: ", A)') c_title(1:ll)
      ENDIF
   ENDIF


                                    ! Dimension inquiries:
                                    ! -- of dimension 'age' (dating)
                                    !    This could be changed to allow for
                                    !    arbitrary dating (test for a set
                                    !    of names and units). Currently, we
                                    !    assume that dates are given by ages
                                    !    before 1950.
   status = NF_INQ_DIMID(nc_id, 'age', dating_dimid)
   IF (status /= NF_NOERR) CALL HANDLE_NCERRORS(status, fn_thisfile, (__LINE__-1))

                                    ! Inquire length for dimension 'age'
   status = NF_INQ_DIMLEN(nc_id, dating_dimid, dating_dimlen)
   IF (status /= NF_NOERR) CALL HANDLE_NCERRORS(status, fn_thisfile, (__LINE__-1))


                                    ! Variables' inquiries
                                    ! -- for variable 'age'
   status = NF_INQ_VARID(nc_id, 'age', dating_varid)
   IF (status /= NF_NOERR) CALL HANDLE_NCERRORS(status, fn_thisfile, (__LINE__-1))



                                    ! Allocate space for one 1D array to hold the dating
                                    ! information from the file
   ALLOCATE(ncforc_dates_orig(dating_dimlen))


   only_one_datum: IF (dating_dimlen == 1) THEN
      n_ncforc_time = 1             ! If there are only shelf flux
      n_replicbfr   = 0             ! data for one single instant in time,
      n_replicaft   = 0             ! then, we use that value throughout
                                    ! the whole simulation without changes.
                                    ! No time information required then,
                                    ! no replication will be needed.
   ELSE only_one_datum
                                    ! Allocate for the age "column" from file

                                    ! Read in the original date information
      status = NF_GET_VAR_DOUBLE(nc_id, dating_varid, ncforc_dates_orig(:))
      IF (status /= NF_NOERR) CALL HANDLE_NCERRORS(status, fn_thisfile, (__LINE__-1))

                                    ! get age:replicate_period information into ncforc_period
      status = NF_GET_ATT_DOUBLE(nc_id, dating_varid, 'replicate_period', ncforc_period)

      IF (status /= NF_NOERR) THEN
         IF (status == NF_ENOTATT) THEN
                                    ! if the error is due to the absence of a replicate_period
                                    ! attribute to the 'age' variable, we may ignore it and adopt
                                    ! the default replicate_period <period_dflt>
            ncforc_period = period_dflt

            IF (resusd /= 0) THEN
               WRITE(resuni,'(" age:replicate_period not found.")')
               WRITE(resuni,'(" Using default period of ", F16.4, " years" )') period_dflt
            ENDIF
         ELSE                    ! there was another, unexpected, error: report and abort
            CALL HANDLE_NCERRORS(status, fn_thisfile, (__LINE__-14))
         ENDIF
      ENDIF

      replication_allowed: IF (ncforc_period > 0D0) THEN
                                    ! If replication is allowed, check
                                    ! whether it is required

                                    ! Check age extent:
         IF (ncforc_dates_orig(dating_dimlen) > ncforc_dates_orig(1)) THEN
                                    !  -- if ages are in increasing order
            ncforc_tinf = 1950D0 - ncforc_dates_orig(dating_dimlen)
            ncforc_tmax = 1950D0 - ncforc_dates_orig(1)
            i1          = dating_dimlen
            i2          = 1
            is          = -1
         ELSE
                                    !  -- if ages are in decreasing order.
            ncforc_tinf = 1950D0 - ncforc_dates_orig(1)
            ncforc_tmax = 1950D0 - ncforc_dates_orig(dating_dimlen)
            i1          = 1
            i2          = dating_dimlen
            is          = 1
         ENDIF                                

                                    ! How many replicates do we need
                                    ! before temp_tinf ?
         IF (ncforc_tinf > simul_tinf) THEN

            n_replicbfr = INT((ncforc_tinf - simul_tinf)/ncforc_period)

            IF ((ncforc_tinf - simul_tinf) > DBLE(n_replicbfr)*ncforc_period) THEN
               n_replicbfr = n_replicbfr + 1
            ENDIF

         ELSE

            n_replicbfr = 0

         ENDIF
            
                                    ! How many replicates do we need after temp_tmax ?
         IF (simul_tmax > ncforc_tmax) THEN

            n_replicaft = INT((simul_tmax - ncforc_tmax)/ncforc_period)

            IF ((simul_tmax - ncforc_tmax) > DBLE(n_replicaft)*ncforc_period) THEN

               n_replicaft = n_replicaft + 1
            ENDIF

         ELSE

            n_replicaft = 0

         ENDIF

         IF (resusd /= 0) THEN
            IF ((n_replicbfr > 0) .OR. (n_replicaft > 0)) THEN
               WRITE(resuni,'(" The data in the file only extend from")')
               WRITE(resuni,'(F16.4, " to ", F16.4)') ncforc_tinf, ncforc_tmax
               WRITE(resuni,'(" Extending the temporal range by assuming a periodic evolution.")')
               WRITE(resuni,'(" Adopting a period of ", F16.4, " years.")') ncforc_period
            ENDIF
         ENDIF

                                    ! Total space required for time dimension
         n_ncforc_time = (n_replicbfr+n_replicaft+1) * dating_dimlen
         failed_replication_check = .FALSE.

      ELSE replication_allowed

         IF ((ncforc_tinf > simul_tinf) .OR. (simul_tmax > ncforc_tmax)) THEN
            WRITE(erruni,'(A)') 'The data in the file ' // TRIM(forcing_file)
            WRITE(erruni,'(A)') 'cannot be extended by periodic replication!'
            WRITE(erruni,'(A)') 'It might be necessary to provide a file that '
            WRITE(erruni,'(A)') 'covers the complete simulation time interval.'
            WRITE(erruni,'(A)') 'Need to recheck that later on. '

            failed_replication_check = .TRUE.
            n_ncforc_time = dating_dimlen
         ELSE
            failed_replication_check = .FALSE.
            n_replicbfr = 0
            n_replicaft = 0
            n_ncforc_time = dating_dimlen
         ENDIF

      ENDIF replication_allowed

   ENDIF only_one_datum


   WRITE(dbguni,*) ' n_ncforc_time set to ', n_ncforc_time
   WRITE(dbguni,*) ' n_replicbfr set to ', n_replicbfr
   WRITE(dbguni,*) ' n_replicaft set to ', n_replicaft

   n_flx_in_file = 0                ! number of fluxes that require time
                                    ! interval duplication

   WRITE(dbguni,*)
   WRITE(dbguni,*) ' Starting to initialize the forcings'

   do4ncforc: DO j = 1, n_ncforc+1

      n_current = 0                 ! we have not yet read anything

      getting_fluxdata: IF (j <= n_ncforc) THEN 
                                    ! If we are dealing with fluxes and not time
                                    ! then we are getting the informations from
                                    ! the file, else we have the raw data already
                                    ! and only the replications needs to be done.

         c_varname = forcing_names(j)
         forcing_descs(j)%use_dflt = .FALSE. ! we expect to find a variable for each flux


         WRITE(dbguni,*)
         WRITE(dbguni,*) ' Setting up "' // TRIM(c_varname) // '"'

         status = NF_INQ_VARID(nc_id, c_varname, anyfrc_varid)
         IF (status /= NF_NOERR) THEN
            IF (status == NF_ENOTVAR) THEN
                                    ! There is no variable called <c_varname> in the file.
                                    ! Use default value.
               IF (resusd /= 0) THEN
                  WRITE(resuni,'(A)') ' File does not include "' // TRIM(c_varname) // '" variable'
                  WRITE(resuni,'(A,E12.4)') ' Requesting to use default value(s)'
               ENDIF

               WRITE(dbguni,*) ' "'// TRIM(c_varname) // '" not found in file, requesting to use default value(s)'

               forcing_descs(j)%use_dflt = .TRUE.
               NULLIFY(forcing_descs(j)%ishape)
               NULLIFY(forcing_descs(j)%forcing)
               NULLIFY(forcing_descs(j)%forcing_mean)

               n_current = 1        ! only one datum for variable <c_varname>

               WRITE(dbguni,*) ' set number of values to ', n_current
               WRITE(dbguni,*) ' set use_dflt to ', forcing_descs(j)%use_dflt

               CYCLE do4ncforc  ! nothing else to be done in this case

            ELSE
                                    ! some other error occurred while searching for <c_varname>
               CALL HANDLE_NCERRORS(status, fn_thisfile, (__LINE__-25))

            ENDIF

         ELSE                       ! OK, there is a variable called <c_varname> in the file
                                    ! Inquire number of dimensions for the variable <anyfrc_varid>
            status = NF_INQ_VARNDIMS(nc_id, anyfrc_varid, anyfrc_irank)
            IF (status /= NF_NOERR) CALL HANDLE_NCERRORS(status, fn_thisfile, (__LINE__-1))

            IF (anyfrc_irank > ndimmax_forcing) THEN
               WRITE(erruni,'("Number of dimension for a forcing variable must not exceed ", I0)') ndimmax_forcing
               WRITE(erruni,'("Detected variable with ", I0, " dimensions. Aborting!")') anyfrc_irank
               CALL ABORT()
            ENDIF

                                    ! Inquire the IDs of the dimensions for the variable <anyfrc_varid>
            status = NF_INQ_VARDIMID(nc_id, anyfrc_varid, anyfrc_idims(1:anyfrc_irank))
            IF (status /= NF_NOERR) CALL HANDLE_NCERRORS(status, fn_thisfile, (__LINE__-1))

            IF (anyfrc_idims(anyfrc_irank) /= dating_dimid) THEN
               WRITE(erruni,'("Last dimension of the forcing variable is not the age dimension. Aborting.")')
               CALL ABORT()
            ENDIF


            SELECT CASE(anyfrc_irank)
            CASE(1)
               n_dim1    = 1
               ALLOCATE(forcing_descs(j)%ishape(1))
            CASE(2)
               status = NF_INQ_DIMLEN(nc_id, anyfrc_idims(1), n_dim1)
               IF (status /= NF_NOERR) CALL HANDLE_NCERRORS(status, fn_thisfile, (__LINE__-1))
               ALLOCATE(forcing_descs(j)%ishape(2))
               forcing_descs(j)%ishape(1) = n_dim1
            CASE DEFAULT
               WRITE(erruni,'("Illegal anyfrc_irank = ", I0, ". Aborting.")') anyfrc_irank
               CALL ABORT()
            END SELECT

                                    ! Allocate the appropriate space for readin in the data from the file
            ALLOCATE(ncforc_anyfrc_orig(n_dim1, dating_dimlen))

            n_current = n_ncforc_time


            WRITE(dbguni,*) ' set number of values to ', n_current
            WRITE(dbguni,*) ' set use_dflt to ', forcing_descs(j)%use_dflt
            WRITE(dbguni,*) ' set n_dim1 to ', n_dim1

         ENDIF


         IF (.NOT. forcing_descs(j)%use_dflt) THEN
                                    ! If the variable exists in the file,
                                    ! (i.e., we do not use the default value)
                                    ! then read in the values from the file.

            WRITE(dbguni,*) ' not requesting to use default values for "' // TRIM(c_varname) // '"'

            IF(n_ncforc_time == 1) THEN
                                    ! if only one datum exists in the file, this
                                    ! will be the final forcing data, no replication
                                    ! needs to be done and we can read in the
                                    ! data into the final array directly.

               WRITE(dbguni,*) ' Allocating ncforc_anyfrc(n_dim1,1) and ncforc_anyfrcm(n_dim1)'
               WRITE(dbguni,*) ' for using the single data set from the file.'
               ALLOCATE(ncforc_anyfrc(n_dim1,1))
               ALLOCATE(ncforc_anyfrcm(n_dim1))

               SELECT CASE(anyfrc_irank)
               CASE(1)
                  status = NF_GET_VAR_DOUBLE(nc_id, anyfrc_varid, ncforc_anyfrc(1,:))
                  IF (status /= NF_NOERR) CALL HANDLE_NCERRORS(status, fn_thisfile, (__LINE__-1))
               CASE DEFAULT
                  status = NF_GET_VAR_DOUBLE(nc_id, anyfrc_varid, ncforc_anyfrc(:,:))
                  IF (status /= NF_NOERR) CALL HANDLE_NCERRORS(status, fn_thisfile, (__LINE__-1))
               END SELECT

               WRITE(dbguni,'("  Read 1 value from file.")')

            ELSE

               WRITE(dbguni,*) ' Allocating ncforc_anyfrcm(n_dim1) for using data from the file.'
               ALLOCATE(ncforc_anyfrcm(n_dim1))

               SELECT CASE(anyfrc_irank)
               CASE(1)
                  WRITE(dbguni,'("  Attempting to read ", I0, " values from file.")') dating_dimlen
                  status = NF_GET_VAR_DOUBLE(nc_id, anyfrc_varid, ncforc_anyfrc_orig(1,:))
                  IF (status /= NF_NOERR) CALL HANDLE_NCERRORS(status, fn_thisfile, (__LINE__-1))

                  status = NF_GET_ATT_DOUBLE(nc_id, anyfrc_varid, 'average', ncforc_anyfrcm(1))
                  IF (status /= NF_NOERR) CALL HANDLE_NCERRORS(status, fn_thisfile, (__LINE__-1))
               CASE DEFAULT
                  WRITE(dbguni,'("  Attempting to read ", I0, "x", I0, " values from file.")') &
                               n_dim1, dating_dimlen
                  status = NF_GET_VAR_DOUBLE(nc_id, anyfrc_varid, ncforc_anyfrc_orig(:,:))
                  IF (status /= NF_NOERR) CALL HANDLE_NCERRORS(status, fn_thisfile, (__LINE__-1))

                  status = NF_GET_ATT_DOUBLE(nc_id, anyfrc_varid, 'average', ncforc_anyfrcm(:))
                  IF (status /= NF_NOERR) CALL HANDLE_NCERRORS(status, fn_thisfile, (__LINE__-1))
               END SELECT


                                   ! increment number of vars requiring
                                   ! replication of time intervals.
               n_flx_in_file = n_flx_in_file+1

            ENDIF

         ENDIF

      ELSE getting_fluxdata        ! i.e., if i == n_ncforc + 1,
                                   ! we are setting up time information

         WRITE(dbguni,*)
         WRITE(dbguni,*) ' Setting up ncforc_time'

         IF (n_flx_in_file == 0) THEN
                                   ! If there is no flux variable in the file
                                   ! that needs replication, there is no need
                                   ! to replicate anything! The date information
                                   ! in the file is disregarded in this case.
                                   ! Reset n_ncforc_time to 1.

            WRITE(dbguni,*) ' No flux variable needs replication.'
            WRITE(dbguni,*) ' resetting n_ncforc_time to 1'
            WRITE(dbguni,*) ' setting ncforc_time(1) = 0D0'

            n_ncforc_time = 1
            IF (ASSOCIATED(forcing_time)) DEALLOCATE(forcing_time)
            ALLOCATE(forcing_time(1))
            forcing_time(1) = 0D0
         ENDIF


         IF (n_ncforc_time /= 1) THEN
                                   ! Copy replicates into ncforc_time
                                   ! if allowed
            IF (failed_replication_check) THEN
               WRITE(erruni,'(A)') 'Impossible to recover from failed replication check. Aborting.'
               WRITE(erruni,'(A)') 'Please provide a file that covers the complete simulation time interval.'
               CALL ABORT()
            ENDIF

            WRITE(dbguni,*) ' Allocating forcing_time(n_ncforc_time)'
            IF (ASSOCIATED(forcing_time)) DEALLOCATE(forcing_time)
            ALLOCATE(forcing_time(n_ncforc_time))

            i_first = 1
            i_last  = dating_dimlen

            DO i = n_replicbfr, 1, -1
               forcing_time(i_first:i_last) = 1950D0 - ncforc_dates_orig(i1:i2:is) - DBLE(i)*ncforc_period
               i_first = i_last + 1
               i_last  = i_last + dating_dimlen
            ENDDO

            forcing_time(i_first:i_last) = 1950D0 - ncforc_dates_orig(i1:i2:is)

            DO i = 1, n_replicaft
               i_first = i_last + 1
               i_last  = i_last + dating_dimlen
               forcing_time(i_first:i_last) = 1950D0 - ncforc_dates_orig(i1:i2:is) + DBLE(i)*ncforc_period
            ENDDO

         ELSE                       ! n_ncforc_time = 1
                                    ! ncforc_time normally not required
            IF (ASSOCIATED(forcing_time)) DEALLOCATE(forcing_time)
            ALLOCATE(forcing_time(1))
            forcing_time(1) = 0D0

         ENDIF

         EXIT do4ncforc             ! And EXIT since we have done evrything.
                                    ! The rest of the loop only deals with non-time
                                    ! related variables.

      ENDIF getting_fluxdata


      IF (n_current /= 1) THEN
                                    ! Replication of the flux values if
                                    ! if n_current > 1, i.e., only if
                                    ! the current flux is not constant
                                    ! in time.
           
         IF (failed_replication_check) THEN
            WRITE(erruni,'(A)') 'Impossible to recover from failed replication check. Aborting.'
            WRITE(erruni,'(A)') 'Please provide a file that covers the complete simulation time interval.'
            CALL ABORT()
         ENDIF

         WRITE(dbguni,'("  Allocating ncforc_anyfrc(n_dim1,n_ncforc_time)")')
         ALLOCATE(ncforc_anyfrc(n_dim1, n_ncforc_time))

         i_first = 1                ! Copy replicates into ncforc_anyfrc
         i_last  = dating_dimlen

         DO i = n_replicbfr, 1, -1
            ncforc_anyfrc(:,i_first:i_last) = ncforc_anyfrc_orig(:,i1:i2:is)
            i_first = i_last + 1
            i_last  = i_last + dating_dimlen
         ENDDO

         ncforc_anyfrc(:,i_first:i_last) = ncforc_anyfrc_orig(:,i1:i2:is)

         DO i = 1, n_replicaft
            i_first = i_last + 1
            i_last  = i_last + dating_dimlen
            ncforc_anyfrc(:,i_first:i_last) = ncforc_anyfrc_orig(:,i1:i2:is)
         ENDDO

      ENDIF

      forcing_descs(j)%ishape(:) = (/ n_dim1, n_current /)
      forcing_descs(j)%forcing => ncforc_anyfrc
      forcing_descs(j)%forcing_mean => ncforc_anyfrcm


                                 
                            
                             
      NULLIFY(ncforc_anyfrc_orig)   ! Now that the correct pointer points
      NULLIFY(ncforc_anyfrc)        ! to the relevant information, nullify
      NULLIFY(ncforc_anyfrcm)       ! ncforc_anyfrc for the next round.


   ENDDO do4ncforc


   WRITE(dbguni, '(" dating_dimlen, n_ncforc_time: ", I0, 1X, I0)') dating_dimlen, n_ncforc_time
!      WRITE(dbguni,*) ncforc_dates
!      WRITE(dbguni,*) ncforc_time
   WRITE(dbguni,'(" ncforc_tinf, ncforc_tmax:", 2F16.4)') ncforc_tinf, ncforc_tmax

   IF (n_ncforc_time > 1) WRITE(dbguni,'(" i1, i2, is:", 3(" ", i0))') i1, i2, is
   CALL FLUSH(dbguni)
 
                                    ! Deallocate space, as it is not
                                    ! required anymore
   DEALLOCATE(ncforc_dates_orig)

   !--------------------------------------------------------------------
   END SUBROUTINE READ_NCFORCING
   !--------------------------------------------------------------------

!=======================================================================
END MODULE MOD_MBM_NCFORCING
!=======================================================================
