!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


!=======================================================================
      MODULE MOD_MBM_MEDUSA_O2S
!=======================================================================

      USE mod_mbm_geometry, ONLY: mbm_nro1 => nro1,
     &                            mbm_ndint => n_sflelt


      IMPLICIT NONE

      PRIVATE :: mbm_nro1, mbm_ndint

      INTEGER, SAVE, PRIVATE :: n_grid_seafloor   = -1
      INTEGER, SAVE, PRIVATE :: nix               = -1
      INTEGER, SAVE, PRIVATE :: njy               = -1

      INTEGER, SAVE          :: mbm_o2s_id        = -1
      
                                    ! Surface areas (m2)
                                    ! mbm_surf(i,j), i=1,nix, j=1,njy
      DOUBLE PRECISION, SAVE, DIMENSION(mbm_nro1, mbm_ndint) :: mbm_surf

                                    ! Depth below sea-level of seafloor elements (m)
                                    ! mbm_dept(j), j=1,njy
      DOUBLE PRECISION, SAVE, DIMENSION(          mbm_ndint) :: mbm_dept

                                    ! Fraction of seafloor element at in the
                                    ! three depth intervals
      DOUBLE PRECISION, SAVE, DIMENSION(        mbm_ndint,3) :: mbm_fa

                                    ! Seawater temperature at seafloor elements (Kelvin)
                                    ! mbm_temp(i,j), i=1,nix, j=1,njy
      DOUBLE PRECISION, SAVE, DIMENSION(mbm_nro1, mbm_ndint) :: mbm_temp

                                    ! Salinity of seawater at seafloor elements (-)
                                    ! mbm_sali(i,j), i=1,nix, j=1,njy
      DOUBLE PRECISION, SAVE, DIMENSION(mbm_nro1, mbm_ndint) :: mbm_sali

                                    ! Dissolved Inorganic Carbon at seafloor elements (mol/m3)
                                    ! mbm_dic(i,j), i=1,nix, j=1,njy
      DOUBLE PRECISION, SAVE, DIMENSION(mbm_nro1, mbm_ndint) :: mbm_dic

                                    ! Total Alkalinity at seafloor elements (equivalent/m3)
                                    ! mbm_alk(i,j), i=1,nix, j=1,njy
      DOUBLE PRECISION, SAVE, DIMENSION(mbm_nro1, mbm_ndint) :: mbm_alk

                                    ! Dissolved Oxygen at seafloor elements (mol O2/m3)
                                    ! mbm_oxyg(i,j), i=1,nix, j=1,njy
      DOUBLE PRECISION, SAVE, DIMENSION(mbm_nro1, mbm_ndint) :: mbm_oxyg

                                    ! Organic matter rain rate at sediment top (1E18 mol C / yr)
                                    ! mbm_orgm(i,j), i=1,nix; j=1,njy
      DOUBLE PRECISION, SAVE, DIMENSION(mbm_nro1, mbm_ndint) :: mbm_orgm

                                    ! Calcite rain rate at sediment top (1E18 mol C / m2 / yr)
                                    ! mbm_calc(i,j), i=1,nix, j=1,njy
      DOUBLE PRECISION, SAVE, DIMENSION(mbm_nro1, mbm_ndint) :: mbm_calc

                                    ! Aragonite rain rate at sediment top (1E18 mol C / m2 / yr)
                                    ! mbm_arag(i,j), i=1,nix, j=1,njy 
      DOUBLE PRECISION, SAVE, DIMENSION(mbm_nro1, mbm_ndint) :: mbm_arag

                                    ! Clay rain rate at sediment top (1E18 kg / m2 / yr)
                                    ! mbm_clay(i,j), i=1,nix, j=1,njy
      DOUBLE PRECISION, SAVE, DIMENSION(mbm_nro1, mbm_ndint) :: mbm_clay


      CONTAINS

!-----------------------------------------------------------------------
      SUBROUTINE OCEAN_TO_SEDIMENT(i_request, rbflag)
!-----------------------------------------------------------------------
      ! Loads a new complete set of forcing arrays into the
      !
      !  - seafloor_surf
      !  - seafloor_wdbsl
      !  - seafloor_wtmpc
      !  - seafloor_wsalin
      !  - seafloor_wsolutes (already speciated)
      !  - seafloor_wfflx

      USE mod_seafloor_wdata,       ONLY: WDATA_CONTAINER
      USE mod_defines_medusa,       ONLY: jp_stderr
      USE mod_basicdata_medusa,     ONLY: dp_zero_degc
      USE mod_indexparam
      USE mod_materialcharas,       ONLY: mol_calc, mol_arag,
     &                                    mol_om, om_c

      USE mod_seafloor_central,     ONLY: N_COLUMNS_USED,
     &                                    IJ_COLUMNS_USED,
     &                                    COLUMN_N2IJ, SAVE_AREA4N,
     &                                    SAVE_BOUNDARY_CONDS

#ifdef WITH_CALC_PT
      USE mod_mbm_tempo, ONLY: temps
#endif

      IMPLICIT NONE

      INTEGER, INTENT(IN)  :: i_request
      INTEGER, INTENT(OUT) :: rbflag

      
      INTEGER, SAVE :: n_datasets_read =  0
      INTEGER, SAVE :: mbm_o2s_previd  = -1


      TYPE(WDATA_CONTAINER) :: wdata
      DOUBLE PRECISION :: wtmpc, wsalin, wdbsl
      DOUBLE PRECISION :: wtk
      DOUBLE PRECISION :: dic, alk, co2, hco3, co3
      DOUBLE PRECISION :: bt, boh4, boh3
      DOUBLE PRECISION :: oh,h3o
      INTEGER :: iflag
      INTEGER :: i, j, n

      ! Dummy function names
      DOUBLE PRECISION :: TOTBOR, RHOSW

      DOUBLE PRECISION, DIMENSION(nsolut) :: wconc
      DOUBLE PRECISION, DIMENSION(nsolid) :: wfflx
      DOUBLE PRECISION, DIMENSION(nsolid) :: tsolids


      ! Standard I/O related data
      ! -------------------------

      CHARACTER(LEN=*), PARAMETER :: c_fmterr_a
     &  = '("[MOD_MBM_MEDUSA_O2S/OCEAN_TO_SEDIMENT] error: ", A)'



      IF(n_datasets_read == 0) THEN

        IF(mbm_o2s_id == -1) THEN

          WRITE(jp_stderr, c_fmterr_a)
     &      'Data must be loaded into MOD_MBM_MEDUSA_O2S -- aborting!'
          CALL ABORT()

        ELSE

          CALL N_COLUMNS_USED(n_grid_seafloor)
          CALL IJ_COLUMNS_USED(nix, njy)

          IF (mbm_nro1 /= nix) THEN
            WRITE(jp_stderr,c_fmterr_a)
     &        'incompatible number of surface ocean reservoirs'
            WRITE(jp_stderr,'(" - in MOD_SEAFLOOR_CENTRAL: ", I0)') nix
            WRITE(jp_stderr,'(" - in MBM: ", I0)') mbm_nro1
            WRITE(jp_stderr,'("Aborting!")')
            CALL ABORT()
          ENDIF
    
    
          IF (mbm_ndint /= njy) THEN
            WRITE(jp_stderr,c_fmterr_a)
     &        'incompatible number of depth intervals'
            WRITE(jp_stderr,'(" - in MOD_SEAFLOOR_CENTRAL: ", I0)') njy
            WRITE(jp_stderr,'(" - in MBM: ", I0)') mbm_ndint
            WRITE(jp_stderr,'("Aborting!")')
            CALL ABORT()
          ENDIF

        ENDIF

      ENDIF


      IF(mbm_o2s_id <= mbm_o2s_previd) THEN
                                    ! No new data available
        IF(i_request == 0) THEN
          rbflag = 1
          RETURN
        ELSE                        ! No more *new* data! However, a complete
          rbflag = -1               ! set has previously been stored.
                                    ! Use it, but issue warning!
        ENDIF

      ENDIF


      IF (i_request == -1) THEN

        n_datasets_read =  0
        mbm_o2s_previd  = -1
        mbm_o2s_id      = -1

        rbflag = 0

        RETURN

      ENDIF


      DO n = 1, n_grid_seafloor

        CALL COLUMN_N2IJ(i_column = n, iflag = iflag,
     &                              ix = i, jy = j)

        CALL SAVE_AREA4N(i_column = n, iflag = iflag,
     &                              sfc_area = mbm_surf(i,j))

        wdbsl  = mbm_dept(j)
        wsalin = mbm_sali(i,j)
        wtmpc  = mbm_temp(i,j) - dp_zero_degc

c~ Temporarily commented to mimick old-style behaviour
c~         wtk  = mbm_temp(i,j)
        dic = mbm_dic(i,j)
        alk = mbm_alk(i,j)
        wtk = wtmpc + dp_zero_degc

        bt  = TOTBOR(wtk, wsalin, wdbsl)*RHOSW(wtk, wsalin, wdbsl)

        CALL SPECIA_CB(wtk,wsalin,wdbsl, alk,dic, co3,hco3,co2,
     &                              bt, boh4,boh3, oh,h3o,
     &                              iflag)

        IF(iflag /= 0) THEN
          rbflag = 2
          RETURN
        ENDIF

        wdata%wdbsl  = wdbsl
        wdata%wtmpc  = wtmpc
        wdata%wsalin = wsalin

        wconc(ic_co2)  = co2
        wconc(ic_hco3) = hco3
        wconc(ic_co3)  = co3
        wconc(ic_o2)   = mbm_oxyg(i,j)


        wfflx(if_clay) = 1.0D+18*mbm_clay(i,j)
        wfflx(if_calc) = 1.0D+18*mbm_calc(i,j)*mol_calc
        wfflx(if_arag) = 1.0D+18*mbm_arag(i,j)*mol_arag
        wfflx(if_om)   = 1.0D+18*((mbm_orgm(i,j)/om_c))*mol_om
#ifdef WITH_CALC_PT
        wfflx(if_calc_pt) = wfflx(if_calc)*temps
#endif


        CALL SAVE_BOUNDARY_CONDS(i_column = n, gbcflag = iflag,
     &                              wdata = wdata,
     &                              wconc = wconc(:),
     &                              wfflx = wfflx(:))

      ENDDO

      n_datasets_read = n_datasets_read+1
      mbm_o2s_previd = mbm_o2s_id
      rbflag = 0

      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE OCEAN_TO_SEDIMENT
!-----------------------------------------------------------------------


!=======================================================================
      END MODULE MOD_MBM_MEDUSA_O2S
!=======================================================================
