!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


!     ******************************************************************
      MODULE MOD_MBM_GEOMETRY
!     ******************************************************************
      
      IMPLICIT NONE

! Number of reservoirs
      INTEGER, PARAMETER :: nra1=1
      INTEGER, PARAMETER :: nro1=5
      INTEGER, PARAMETER :: nro2=2
      INTEGER, PARAMETER :: nro3=3

! Reservoir ordering
      INTEGER, PARAMETER :: ra1=0         ! 0
      INTEGER, PARAMETER :: ro1=ra1+nra1  ! 1
      INTEGER, PARAMETER :: ro2=ro1+nro1  ! 6
      INTEGER, PARAMETER :: ro3=ro2+nro2  ! 8

! Number of reservoirs
! - nres : total number of reservoirs
! - nroc : total number of oceanic reservoirs

      INTEGER, PARAMETER :: nres = ro3 + nro3 - ra1 ! 11
      INTEGER, PARAMETER :: nroc = ro3 + nro3 - ro1 ! 10


!DEL      INTEGER, PARAMETER, PRIVATE :: nra1=1, nro1=5, nro2=2, nro3=3
!DEL      INTEGER, PARAMETER, PRIVATE :: ra1=0,
!DEL     >                               ro1=nra1,
!DEL     >                               ro2=nra1+nro1,
!DEL     >                               ro3=nra1+nro1+nro2
!DEL      INTEGER, PARAMETER, PRIVATE :: bottom=80




! Geometric horizons
! ------------------
! These are horizontal planes that define the outlines of the model
! reservoirs. They include all the horizontal planes that
! delimit the vertical extents of the reservoirs.
! currently, these are:
! - the sea-level (0m)
! - the 100m depth horizon (100m below sea-level)
! - the 1000m depth horizon (1000m below sea-level)

! Horizon parameters
! ------------------
! - n_geohor     : number of geometric horizons
! - i_geohor_slv : index of sea-level horizon
! - i_geohor_dps : index of deep-sea horizon

      INTEGER, PARAMETER :: n_geohor     = 3,
     >                      i_geohor_slv = 0, 
     >                      i_geohor_dps = i_geohor_slv + n_geohor - 1


! Geometric levels
! ----------------
! Geometric levels (slices) are delimited by subsequent geometric
! horizons. The index of each level is equal to that of the its upper
! horizon, increased by 1: level numbers start at 1, while horizon
! numbers start at 0.
! - n_geolev     : number of geometric levels
! - i_geolev_top : index of top slice
! - i_geolev_bot : index of bottom slice

      INTEGER, PARAMETER :: n_geolev     = 3,
     >                      i_geolev_top = i_geohor_slv+1,
     >                      i_geolev_bot = i_geolev_top + n_geolev - 1


! Hypsometric profiles
! --------------------
! The following are currently statically linked to the surface
! reservoirs.
! - n_hypspr     : number of hypsometric profiles
! - i_hypspr_1   : index of first hypsometric profile
! - i_hypspr_n   : index of last hypsometric profile

      INTEGER, PARAMETER :: n_hypspr     = nro1,
     >                      i_hypspr_1   = ro1+1,
     >                      i_hypspr_n   = i_hypspr_1 + n_hypspr - 1


! Depths of characteristic geometric horizons below sea-level
! (static parameters)

      DOUBLE PRECISION, DIMENSION(i_geohor_slv:i_geohor_dps),
     >         PARAMETER :: d_geohor = (/ 0D0, 100D0, 1000D0 /)


! - nivmer : sea-level w/r to present-day (m)
!            <nivmer> is negative if sea-level is deeper than
!            at present-day.
      DOUBLE PRECISION, SAVE :: nivmer


! Positions of characteristic geometric horizons within profiles
! (these change in time as sea-level varies)
! initialised at the same values as d_geohor

      DOUBLE PRECISION, DIMENSION(i_geohor_slv:i_geohor_dps),
     >         SAVE      :: z_geohor = d_geohor


! Index of interval where characteristic geometric
! horizons are positioned within profiles
! (these change in time as sea-level varies)
! Initialised at illegal values to force initialisation
!
!      INTEGER, DIMENSION(i_geohor_slv:i_geohor_dps,
!     >                   i_bathyp_1:i_bathyp_n),
!     >         SAVE      :: ip_geohor = (/ (/ -1, -1, -1 /),
!     >                                     (/ -1, -1, -1 /),
!     >                                     (/ -1, -1, -1 /),
!     >                                     (/ -1, -1, -1 /),
!     >                                     (/ -1, -1, -1 /) /)


! Correspondency table geo-level -> box-ID
! ----------------------------------------
! Index of box that a given mosaic element <il,ip> belongs to.
! (Should be either statically set up in a pre-compilation
! configuration stage or dynamically during model initialisation)

      INTEGER, DIMENSION(i_geolev_top:i_geolev_bot,
     >                   i_hypspr_1:i_hypspr_n),
     >         PARAMETER :: i_box4geo
     >        = RESHAPE((/ ro1+1,  ro1+1, ro3+1,
     >                     ro1+2,  ro2+1, ro3+1,
     >                     ro1+3,  ro1+3, ro3+2,
     >                     ro1+4,  ro2+2, ro3+3,
     >                     ro1+5,  ro1+5, ro3+3 /), (/3, 5/))


! Sea-floor nodes and elements
! ----------------------------
! - resol        : distance between two subsequent nodes
! - n_sflnod     : number of nodes in each hypsometric profile
! - i_sflnod_top : index of top node in each hypsometric profile
! - i_sflnod_bot : index of bottom  node in each hypsometric profile
! <n_sflnod> and the other two companion parameters should eventually
! be set dynamically.

      DOUBLE PRECISION, PARAMETER
     >                   :: resol        = 100.D+00

      INTEGER, PARAMETER :: n_sflnod     = 81,
     >                      i_sflnod_top = 0,
     >                      i_sflnod_bot = i_sflnod_top + n_sflnod - 1

! - maxdep(ip)    : greatest significant node in profile <ip>
!                  (adjusted in INITIA)
      INTEGER, DIMENSION(i_hypspr_1:i_hypspr_n),
     >         SAVE      ::  maxdep


! Sea-floor elements
! ------------------
! Sea-floor elements are delimited by subsequent sea-floor nodes.
! They are numbered by the node number of their deepest node:
! Sea-floor element <is> is delimited by sea-floor nodes <is-1> and
! <is>.
!
! - n_sflelt     : number of sea-floor elements (=n_sflnod-1)
! - i_sflelt_1   : index of topmost element
! - i_sflelt_n   : index of bottom element

      INTEGER, PARAMETER :: n_sflelt     = n_sflnod-1,
     >                      i_sflelt_1   = i_sflnod_top+1,
     >                      i_sflelt_n   = i_sflnod_bot


! Hypsometric data
! ----------------
! These are read in from a file (hyspar, hypsvl) or calculated
! (sflelt_sfcarea) in INITIA.
      DOUBLE PRECISION, DIMENSION(i_sflnod_top:i_sflnod_bot,
     >                            i_hypspr_1  :i_hypspr_n),
     >         SAVE      :: hypsar, hypsvl, hyps_z

      DOUBLE PRECISION, DIMENSION(i_sflelt_1:i_sflelt_n,
     >                            i_hypspr_1:i_hypspr_n),
     >         SAVE      ::  sflelt_sfcarea

! Geometric characteristics of the model reservoirs
! calculated in SETGEO.

      DOUBLE PRECISION, DIMENSION(ro1+1:ro3+nro3),
     >         SAVE      ::  volum
      DOUBLE PRECISION, DIMENSION(ro1+1:ro1+nro1),
     >         SAVE      ::  surf, propor


! Transitional variables: to be discarded ASAP
! - bottom

      INTEGER, PARAMETER :: bottom=n_sflelt

      END MODULE MOD_MBM_GEOMETRY
