!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


      MODULE MOD_MBM_FILES

      USE MOD_GENERALPARAMS

      IMPLICIT NONE

      INTEGER, SAVE ::    datusd = 0, datuni = -1
      INTEGER, SAVE ::    iniusd = 0, iniuni = -1
      INTEGER, SAVE ::    elvusd = 0, elvuni = -1
      INTEGER, SAVE ::    resusd = 0, resuni = -1
      INTEGER, SAVE ::    conusd = 0, conuni = -1
      INTEGER, SAVE ::    sdiusd = 0, sdiuni = -1
      INTEGER, SAVE ::    sdousd = 0, sdouni = -1
      INTEGER, SAVE ::    flxusd = 0, flxuni = -1
      INTEGER, SAVE ::    weausd = 0, weauni = -1
      INTEGER, SAVE ::    tprusd = 0
      INTEGER, SAVE ::    cbfusd = 0
      INTEGER, SAVE ::    slvusd = 0
      INTEGER, SAVE ::    ff1usd = 0
      INTEGER, SAVE ::    purusd = 0
      INTEGER, SAVE ::    crrusd = 0
      INTEGER, SAVE ::    afcusd = 0
      INTEGER, SAVE ::    shfusd = 0

      INTEGER, SAVE ::                loguni = -1
      INTEGER, SAVE ::                erruni = -1
      INTEGER, SAVE ::                dbguni = -1

      INTEGER, SAVE ::                tmpuni = -1

      CHARACTER(len=gnrl_maxfilename_len), SAVE ::      datfil
      CHARACTER(len=gnrl_maxfilename_len), SAVE ::      inifil
      CHARACTER(len=gnrl_maxfilename_len), SAVE ::      elvfil
      CHARACTER(len=gnrl_maxfilename_len), SAVE ::      cslfil
      CHARACTER(len=gnrl_maxfilename_len), SAVE ::      cmsfil
      CHARACTER(len=gnrl_maxfilename_len), SAVE ::      resfil
      CHARACTER(len=gnrl_maxfilename_len), SAVE ::      confil
      CHARACTER(len=gnrl_maxfilename_len), SAVE ::      sdifil
      CHARACTER(len=gnrl_maxfilename_len), SAVE ::      sdofil
      CHARACTER(len=gnrl_maxfilename_len), SAVE ::      flxfil
      CHARACTER(len=gnrl_maxfilename_len), SAVE ::      weafil
      CHARACTER(len=gnrl_maxfilename_len), SAVE ::      cbffil
      CHARACTER(len=gnrl_maxfilename_len), SAVE ::      tprfil ! Temperature file

      CHARACTER(len=gnrl_maxfilename_len), SAVE ::      slvfil ! Sealevel file

      CHARACTER(len=gnrl_maxfilename_len), SAVE ::      ff1fil ! Forcing function 1
      CHARACTER(len=gnrl_maxfilename_len), SAVE ::      purfil ! Phosphate Utilization Ratio
      CHARACTER(len=gnrl_maxfilename_len), SAVE ::      crrfil ! Carbonate Rain Ratio
      CHARACTER(len=gnrl_maxfilename_len), SAVE ::      afcfil ! Aragonite Fraction of Carbonates

      CHARACTER(len=gnrl_maxfilename_len), SAVE ::      shffil ! Shelf Flux history file


! NetCDF CSL file data (ancient ConSoLe file)
! ===========================================

      INTEGER, SAVE ::
     &   cslusd =  0                   ! CSL file status (0: not used, else: used) set from mbm.cfg
      INTEGER, SAVE ::
     &   csluni = -1                   ! CSL file logical unit number (dynamically assigned if CSL file used)

#ifdef MBM_WITH_NETCDF
! Dimension definitions and variables
      INTEGER, SAVE ::
     &   nc_cslfilid
      INTEGER, SAVE ::
     &   nc_csldimid_ro1,              ! NC dimension ID for surface ocean reservoirs in CSL file
     &   nc_csldimid_time              ! NC dimension ID for time (unlimited) dimension in CSL file

      INTEGER, SAVE ::
     &   nc_cslvarid_ro1,              ! NC variable ID for the "surface ocean reservoirs" dimension index array in CSL file
     &   nc_cslvarid_time              ! NC variable ID for the "time" dimension in CSL file



! Data variables

      ! Arrays with time dimension only
      INTEGER, PARAMETER :: nc_cslidx_t_0  = 1

      INTEGER, PARAMETER :: nc_cslidx_pco2a = nc_cslidx_t_0        ! - sea-level
      INTEGER, PARAMETER :: nc_cslidx_swc   = nc_cslidx_pco2a  + 1 ! - carbonate (sedimentary) rock weathering: CO2 pathway
      INTEGER, PARAMETER :: nc_cslidx_sws   = nc_cslidx_swc    + 1 ! - carbonate (sedimentary) rock weathering: H2SO4 pathway
      INTEGER, PARAMETER :: nc_cslidx_iwt   = nc_cslidx_sws    + 1 ! - silicate (igneous) rock weathering (only pathway)
      INTEGER, PARAMETER :: nc_cslidx_vot   = nc_cslidx_iwt    + 1 ! - volcanic CO2 input into the atmosphere
      INTEGER, PARAMETER :: nc_cslidx_ort   = nc_cslidx_vot    + 1 ! - organic recycling (loopback from sediment)
      INTEGER, PARAMETER :: nc_cslidx_owt   = nc_cslidx_ort    + 1 ! - organic weathering (input from continent)
      INTEGER, PARAMETER :: nc_cslidx_pwt   = nc_cslidx_owt    + 1 ! - phosphate
      INTEGER, PARAMETER :: nc_cslidx_tic   = nc_cslidx_pwt    + 1 ! - total ocean DIC inventory
      INTEGER, PARAMETER :: nc_cslidx_tta   = nc_cslidx_tic    + 1 ! - total ocean TA inventory
      INTEGER, PARAMETER :: nc_cslidx_tpo   = nc_cslidx_tta    + 1 ! - total ocean PO4 inventory
      INTEGER, PARAMETER :: nc_cslidx_to2   = nc_cslidx_tpo    + 1 ! - total ocean dissolved O2 inventory
      INTEGER, PARAMETER :: nc_cslidx_tc13  = nc_cslidx_to2    + 1 ! - global average DIC delta C-13
      INTEGER, PARAMETER :: nc_cslidx_tc14  = nc_cslidx_tc13   + 1 ! - global average DIC delta C-14
      INTEGER, PARAMETER :: nc_cslidx_cxb   = nc_cslidx_tc14   + 1 ! - carbon exchange flux (from file)
      INTEGER, PARAMETER :: nc_cslidx_pxb   = nc_cslidx_cxb    + 1 ! - PO4 exchange flux (from file)
#ifdef MEDMBM_NC_DICBALANCE
      INTEGER, PARAMETER :: nc_cslidx_c_in  = nc_cslidx_pxb    + 1 ! - Global C input
      INTEGER, PARAMETER :: nc_cslidx_c_out = nc_cslidx_c_in   + 1 ! - Global C output
#else
      INTEGER, PARAMETER, PRIVATE
     &                   :: nc_cslidx_c_out = nc_cslidx_pxb
#endif
#ifdef MEDMBM_NC_ALKBALANCE
      INTEGER, PARAMETER :: nc_cslidx_a_in  = nc_cslidx_c_out  + 1 ! - Global alkalinity output
      INTEGER, PARAMETER :: nc_cslidx_a_out = nc_cslidx_a_in   + 1 ! - Global alkalinity output
#else
      INTEGER, PARAMETER, PRIVATE
     &                   :: nc_cslidx_a_out = nc_cslidx_c_out
#endif

      INTEGER, PARAMETER :: nc_cslidx_t_1   = nc_cslidx_a_out

      ! Arrays with time dimension and box ro1 dimension
      INTEGER, PARAMETER :: nc_cslidx_tr1_0 = nc_cslidx_t_1    + 1
      INTEGER, PARAMETER :: nc_cslidx_crlo  = nc_cslidx_tr1_0      ! - carbonate accumulation by coral reefs
      INTEGER, PARAMETER :: nc_cslidx_shbo  = nc_cslidx_crlo   + 1 ! - carbonate accumulation in banks and on shelves
      INTEGER, PARAMETER :: nc_cslidx_omxp  = nc_cslidx_shbo   + 1 ! - Org Matter export flux
      INTEGER, PARAMETER :: nc_cslidx_ca2s  = nc_cslidx_omxp   + 1 ! - Calcite flux to sediment (deposition flux)
      INTEGER, PARAMETER :: nc_cslidx_ar2s  = nc_cslidx_ca2s   + 1 ! - Aragonite flux to sediment (deposition flux)
      INTEGER, PARAMETER :: nc_cslidx_ot2s  = nc_cslidx_ar2s   + 1 ! - "Other" (clay) flux to sediment (deposition flux)
      INTEGER, PARAMETER :: nc_cslidx_om2s  = nc_cslidx_ot2s   + 1 ! - Org Matter flux to sediment (deposition flux)
      INTEGER, PARAMETER :: nc_cslidx_cabf  = nc_cslidx_om2s   + 1 ! - Calcite sediment bottom flux (flux across 
      INTEGER, PARAMETER :: nc_cslidx_arbf  = nc_cslidx_cabf   + 1 ! - Aragonite sediment bottom flux
      INTEGER, PARAMETER :: nc_cslidx_otbf  = nc_cslidx_arbf   + 1 ! - "Other" (clay) sediment bottom flux
      INTEGER, PARAMETER :: nc_cslidx_ombf  = nc_cslidx_otbf   + 1 ! - Org Matter sediment bottom flux
      INTEGER, PARAMETER :: nc_cslidx_cats  = nc_cslidx_ombf   + 1 ! - total inventory of calcite in the sediment
      INTEGER, PARAMETER :: nc_cslidx_arts  = nc_cslidx_cats   + 1 ! - total inventory of aragonite in the sediment
      INTEGER, PARAMETER :: nc_cslidx_otts  = nc_cslidx_arts   + 1 ! - total inventory of "other" in the sediment
      INTEGER, PARAMETER :: nc_cslidx_omts  = nc_cslidx_otts   + 1 ! - total inventory of organic matter in the sediment
      INTEGER, PARAMETER :: nc_cslidx_citf  = nc_cslidx_omts   + 1 ! - 
      INTEGER, PARAMETER :: nc_cslidx_cotf  = nc_cslidx_citf   + 1 ! - 
      INTEGER, PARAMETER :: nc_cslidx_aitf  = nc_cslidx_cotf   + 1 ! - 
      INTEGER, PARAMETER :: nc_cslidx_aotf  = nc_cslidx_aitf   + 1 ! - 
      INTEGER, PARAMETER :: nc_cslidx_oitf  = nc_cslidx_aotf   + 1 ! - 
      INTEGER, PARAMETER :: nc_cslidx_ootf  = nc_cslidx_oitf   + 1 ! - 

      INTEGER, PARAMETER :: nc_cslidx_tr1_1 = nc_cslidx_ootf
      INTEGER, PARAMETER :: nc_csl_ndata    = nc_cslidx_tr1_1

      INTEGER, DIMENSION(nc_csl_ndata)
     &                   :: nc_cslvarid



! NetCDF CMS file data (CMS: complete model state)
! ================================================
      INTEGER, SAVE ::
     &   cmsusd =  0,
     &   cmsuni = -1


      INTEGER, SAVE ::
     &   nc_cmsfilid                   ! NC file ID for CMS file
      INTEGER, SAVE ::
     &   nc_cmsdimid_ro,               ! NC dimension ID for all ocean reservoirs in CMS file
     &   nc_cmsdimid_time              ! NC dimension ID for time (unlimited) dimension in CMS file
      INTEGER, SAVE ::
     &   nc_cmsvarid_ro,               ! NC variable ID for the "all ocean reservoirs" dimension index array in CMS file
     &   nc_cmsvarid_time              ! NC variable ID for the "time" dimension in CMS file

      ! Arrays with time dimension only
      INTEGER, PARAMETER :: nc_cmsidx_t_0  = 1

      INTEGER, PARAMETER :: nc_cmsidx_slv   = nc_cmsidx_t_0        ! - sea-level
      INTEGER, PARAMETER :: nc_cmsidx_ff1   = nc_cmsidx_slv   + 1  ! - climate forcing 1

      INTEGER, PARAMETER :: nc_cmsidx_pco2a = nc_cmsidx_ff1   + 1  ! - atmospheric pCO2
      INTEGER, PARAMETER :: nc_cmsidx_dc13a = nc_cmsidx_pco2a + 1  ! - atmospheric delta C-13
      INTEGER, PARAMETER :: nc_cmsidx_dc14a = nc_cmsidx_dc13a + 1  ! - atmospheric Delta C-14

      INTEGER, PARAMETER :: nc_cmsidx_t_1  = nc_cmsidx_dc14a


      ! Arrays with time dimension and box ro dimension
      INTEGER, PARAMETER :: nc_cmsidx_tr_0  = nc_cmsidx_t_1   + 1

      INTEGER, PARAMETER :: nc_cmsidx_vols  = nc_cmsidx_tr_0       ! - box volumes
      INTEGER, PARAMETER :: nc_cmsidx_temp  = nc_cmsidx_vols  + 1  ! - box temperatures
      INTEGER, PARAMETER :: nc_cmsidx_salin = nc_cmsidx_temp  + 1  ! - box salinities

      INTEGER, PARAMETER :: nc_cmsidx_po4   = nc_cmsidx_salin + 1  ! - PO4 concentrations
      INTEGER, PARAMETER :: nc_cmsidx_oxy   = nc_cmsidx_po4   + 1  ! - dissolved O2 concentrations
      INTEGER, PARAMETER :: nc_cmsidx_dic   = nc_cmsidx_oxy   + 1  ! - DIC
      INTEGER, PARAMETER :: nc_cmsidx_alk   = nc_cmsidx_dic   + 1  ! - total alkalinity (TA)
      INTEGER, PARAMETER :: nc_cmsidx_dc13  = nc_cmsidx_alk   + 1  ! - DIC delta C-13
      INTEGER, PARAMETER :: nc_cmsidx_dc14  = nc_cmsidx_dc13  + 1  ! - DIC Delta C-14

      INTEGER, PARAMETER :: nc_cmsidx_tr_1  = nc_cmsidx_dc14
      INTEGER, PARAMETER :: nc_cms_ndata    = nc_cmsidx_tr_1

      INTEGER, DIMENSION(nc_cms_ndata)
     &                   :: nc_cmsvarid


      INTEGER, SAVE ::
     &   iccf





      CONTAINS

      !*****************************************************************
       SUBROUTINE HANDLE_ERRORS(status)
      !*****************************************************************

#include "netcdf.inc"

      INTEGER :: status

      IF (STATUS.NE.NF_NOERR) THEN
        PRINT *, NF_STRERROR(STATUS)
        PRINT *, 'NetCDF error detected; aborting.'
        CALL ABORT()
      ENDIF
      !*****************************************************************
      END SUBROUTINE HANDLE_ERRORS
      !*****************************************************************


      !*****************************************************************
       SUBROUTINE HANDLE_NCERRORS(status, whatfile, whatline)
      !*****************************************************************

      IMPLICIT NONE

#include "netcdf.inc"

      INTEGER, INTENT(IN) :: status
      CHARACTER(LEN=*)    :: whatfile
      INTEGER, INTENT(IN) :: whatline

      IF (STATUS /= NF_NOERR) THEN
        IF(whatline > 0) THEN
          WRITE(erruni,"('[',A,':',I0,']: ', A)")
     >      TRIM(whatfile), whatline, TRIM(NF_STRERROR(status))
        ELSE
          WRITE(erruni,"('[',A,':???]: ', A)")
     >      TRIM(whatfile), TRIM(NF_STRERROR(status))
        ENDIF
        PRINT *, 'NetCDF error detected; aborting.'

        CLOSE(erruni)
        CLOSE(loguni)
#  ifdef DEBUG
        CLOSE(dbguni)
#  endif
        
        CALL ABORT()
      ENDIF

      RETURN

      !*****************************************************************
      END SUBROUTINE HANDLE_NCERRORS
      !*****************************************************************


      !*****************************************************************
      SUBROUTINE SETUP_MBM_CSL_NCFILE
      !*****************************************************************
      USE MOD_MBM_INFO
      USE MOD_MBM_GEOMETRY, ONLY: nro1, ro1

      IMPLICIT NONE

      INTEGER :: status, i, ilen
      INTEGER, DIMENSION(2) :: dim, start, count
      CHARACTER(LEN=256) :: string256



      CHARACTER(LEN=gnrl_maxnclongname_len), DIMENSION(nc_csl_ndata)
     &   :: s_longname
      CHARACTER(LEN=gnrl_maxncshrtname_len), DIMENSION(nc_csl_ndata)
     &   :: s_shrtname
      CHARACTER(LEN=gnrl_maxncunitname_len), DIMENSION(nc_csl_ndata)
     &   :: s_unitname


#include "netcdf.inc"


! Initialise the basic information
! ----------------------

        s_longname(nc_cslidx_pco2a)  = 'Atmospheric pCO2'
        s_shrtname(nc_cslidx_pco2a)  = 'pco2'
        s_unitname(nc_cslidx_pco2a)  = 'ppmv'

        s_longname(nc_cslidx_swc)    = 'DIC in by Carb Weath via CO2'
        s_shrtname(nc_cslidx_swc)    = 'swcitc'
        s_unitname(nc_cslidx_swc)    = 'Tmol/yr'

        s_longname(nc_cslidx_sws)    = 'DIC in by Carb Weath via H2SO4'
        s_shrtname(nc_cslidx_sws)    = 'swcits'
        s_unitname(nc_cslidx_sws)    = 'Tmol/yr'

        s_longname(nc_cslidx_iwt)    = 'DIC in by Silic Weath via CO2'
        s_shrtname(nc_cslidx_iwt)    = 'iwcitt'
        s_unitname(nc_cslidx_iwt)    = 'Tmol/yr'

        s_longname(nc_cslidx_vot)    = 'Volcanic CO2 input'
        s_shrtname(nc_cslidx_vot)    = 'volctt'
        s_unitname(nc_cslidx_vot)    = 'Tmol/yr'

        s_longname(nc_cslidx_ort)    = 'CO2 in by OrgM Recycl via O2'
        s_shrtname(nc_cslidx_ort)    = 'orco2t'
        s_unitname(nc_cslidx_ort)    = 'Tmol/yr'

        s_longname(nc_cslidx_owt)    = 'CO2 in by OrgM Weath via O2'
        s_shrtname(nc_cslidx_owt)    = 'owco2t'
        s_unitname(nc_cslidx_owt)    = 'Tmol/yr'

        s_longname(nc_cslidx_pwt)    = 'PO4 in by Rock Weath'
        s_shrtname(nc_cslidx_pwt)    = 'rwpo4t'
        s_unitname(nc_cslidx_pwt)    = 'Tmol/yr'

        s_longname(nc_cslidx_tic)    = 'Global Ocean DIC'
        s_shrtname(nc_cslidx_tic)    = 'tic'
        s_unitname(nc_cslidx_tic)    = 'Petamol_C'

        s_longname(nc_cslidx_tta)    = 'Global Ocean Alkalinity'
        s_shrtname(nc_cslidx_tta)    = 'tta'
        s_unitname(nc_cslidx_tta)    = 'Petaequiv'

        s_longname(nc_cslidx_tpo)    = 'Global Ocean Phosphate'
        s_shrtname(nc_cslidx_tpo)    = 'tpo'
        s_unitname(nc_cslidx_tpo)    = 'Petamol_P'

        s_longname(nc_cslidx_to2)    = 'Global Ocean Oxygen'
        s_shrtname(nc_cslidx_to2)    = 'to2'
        s_unitname(nc_cslidx_to2)    = 'Petamol_O2'

        s_longname(nc_cslidx_tc13)   = 'Average Ocean d13C'
        s_shrtname(nc_cslidx_tc13)   = 'tc13'
        s_unitname(nc_cslidx_tc13)   = 'permil'

        s_longname(nc_cslidx_tc14)   = 'Average Ocean D14C'
        s_shrtname(nc_cslidx_tc14)   = 'tc14'
        s_unitname(nc_cslidx_tc14)   = 'permil'

        s_longname(nc_cslidx_cxb)    = 'CO2 Exchange TerrBio->Atm'
        s_shrtname(nc_cslidx_cxb)    = 'co2xcb'
        s_unitname(nc_cslidx_cxb)    = 'Tmol/yr'

        s_longname(nc_cslidx_pxb)    = 'PO4 Exchange TerrBio->Oce'
        s_shrtname(nc_cslidx_pxb)    = 'po4xcb'
        s_unitname(nc_cslidx_pxb)    = 'Tmol/yr'

#ifdef MEDMBM_NC_DICBALANCE
        s_longname(nc_cslidx_c_in)   = 'Total input of C'
        s_shrtname(nc_cslidx_c_in)   = 'c_in'
        s_unitname(nc_cslidx_c_in)   = 'Tmol/yr'

        s_longname(nc_cslidx_c_out)  = 'Total output of C'
        s_shrtname(nc_cslidx_c_out)  = 'c_out'
        s_unitname(nc_cslidx_c_out)  = 'Tmol/yr'

#endif
#ifdef MEDMBM_NC_ALKBALANCE
        s_longname(nc_cslidx_a_in)   = 'Total input of Alk'
        s_shrtname(nc_cslidx_a_in)   = 'a_in'
        s_unitname(nc_cslidx_a_in)   = 'Teq/yr'

        s_longname(nc_cslidx_a_out)   = 'Total output of Alk'
        s_shrtname(nc_cslidx_a_out)   = 'a_out'
        s_unitname(nc_cslidx_a_out)   = 'Teq/yr'

#endif
        s_longname(nc_cslidx_crlo)   = 'Coral-reef C_in Flux'
        s_shrtname(nc_cslidx_crlo)   = 'coralo'
        s_unitname(nc_cslidx_crlo)   = 'Tmol/yr'

        s_longname(nc_cslidx_shbo)   = 'Shelf_Bank-reef C_in Flu'
        s_shrtname(nc_cslidx_shbo)   = 'shbnko'
        s_unitname(nc_cslidx_shbo)   = 'Tmol/yr'

        s_longname(nc_cslidx_omxp)   = 'OrgM Export Flux'
        s_shrtname(nc_cslidx_omxp)   = 'orgmxp'
        s_unitname(nc_cslidx_omxp)   = 'Tmol/yr'

        s_longname(nc_cslidx_ca2s)   = 'Calcite Sed_in Flux'
        s_shrtname(nc_cslidx_ca2s)   = 'calc2s'
        s_unitname(nc_cslidx_ca2s)   = 'Tmol/yr'

        s_longname(nc_cslidx_ar2s)   = 'Aragonite Sed_in Flux'
        s_shrtname(nc_cslidx_ar2s)   = 'arag2s'
        s_unitname(nc_cslidx_ar2s)   = 'Tmol/yr'

        s_longname(nc_cslidx_ot2s)   = 'Clay Sed_in Flux'
        s_shrtname(nc_cslidx_ot2s)   = 'clay2s'
        s_unitname(nc_cslidx_ot2s)   = 'Pg/yr'

        s_longname(nc_cslidx_om2s)   = 'OrgM Sed_in Flux'
        s_shrtname(nc_cslidx_om2s)   = 'orgm2s'
        s_unitname(nc_cslidx_om2s)   = 'Tmol/yr'

        s_longname(nc_cslidx_cabf)   = 'Calcite Sed_out Flux'
        s_shrtname(nc_cslidx_cabf)   = 'calcbf'
        s_unitname(nc_cslidx_cabf)   = 'Tmol/yr'

        s_longname(nc_cslidx_arbf)   = 'Aragonite Sed_out Flux'
        s_shrtname(nc_cslidx_arbf)   = 'aragbf'
        s_unitname(nc_cslidx_arbf)   = 'Tmol/yr'

        s_longname(nc_cslidx_otbf)   = 'Clay Sed_out Flux'
        s_shrtname(nc_cslidx_otbf)   = 'claybf'
        s_unitname(nc_cslidx_otbf)   = 'Pg/yr'

        s_longname(nc_cslidx_ombf)   = 'OrgM Sed_out Flux'
        s_shrtname(nc_cslidx_ombf)   = 'orgmbf'
        s_unitname(nc_cslidx_ombf)   = 'Tmol/yr'

        s_longname(nc_cslidx_cats)   = 'Total Sedimentary Calcite'
        s_shrtname(nc_cslidx_cats)   = 'calcts'
        s_unitname(nc_cslidx_cats)   = 'Tmol'

        s_longname(nc_cslidx_arts)   = 'Total Sedimentary Aragonite'
        s_shrtname(nc_cslidx_arts)   = 'aragts'
        s_unitname(nc_cslidx_arts)   = 'Tmol'

        s_longname(nc_cslidx_otts)   = 'Total Sedimentary Clay'
        s_shrtname(nc_cslidx_otts)   = 'clayts'
        s_unitname(nc_cslidx_otts)   = 'Pg'

        s_longname(nc_cslidx_omts)   = 'Total Sedimentary OrgMatte'
        s_shrtname(nc_cslidx_omts)   = 'orgmts'
        s_unitname(nc_cslidx_omts)   = 'Tmol'

        s_longname(nc_cslidx_citf)   = 'DIC Sed_to_Oce Flux'
        s_shrtname(nc_cslidx_citf)   = 'citf'
        s_unitname(nc_cslidx_citf)   = 'Tmol/yr'

        s_longname(nc_cslidx_cotf)   = 'DIC Oce_to_Sed Flux'
        s_shrtname(nc_cslidx_cotf)   = 'cotf'
        s_unitname(nc_cslidx_cotf)   = 'Tmol/yr'

        s_longname(nc_cslidx_aitf)   = 'ALK Sed_to_Oce Flux'
        s_shrtname(nc_cslidx_aitf)   = 'aitf'
        s_unitname(nc_cslidx_aitf)   = 'Teq/yr'

        s_longname(nc_cslidx_aotf)   = 'ALK Oce_to_Sed Flux'
        s_shrtname(nc_cslidx_aotf)   = 'aotf'
        s_unitname(nc_cslidx_aotf)   = 'Teq/yr'

        s_longname(nc_cslidx_oitf)   = 'O2 Sed_to_Oce Flux'
        s_shrtname(nc_cslidx_oitf)   = 'oitf'
        s_unitname(nc_cslidx_oitf)   = 'Tmol/yr'

        s_longname(nc_cslidx_ootf)   = 'O2 Oce_to_Sed Flux'
        s_shrtname(nc_cslidx_ootf)   = 'ootf'
        s_unitname(nc_cslidx_ootf)   = 'Tmol/yr'


! Create the data file
!-----------------------

         status = NF_CREATE(cslfil, NF_CLOBBER, nc_cslfilid)
         IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)


! Define and set up dimensions
!-----------------------------

        ! All surface reservoirs: dimension 'ro1'
        ! - define the dimension itself
         status = NF_DEF_DIM(nc_cslfilid, 'ro1',
     &                       nro1,
     &                       nc_csldimid_ro1)
         IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

        ! - define the variable relative to the dimension 'ro1'
         status = NF_DEF_VAR(nc_cslfilid, 'ro1',
     &                       NF_INT, 1, nc_csldimid_ro1,
     &                       nc_cslvarid_ro1)
         IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)


        ! Time: dimension 'time'
        ! - define the dimension itself
         status = NF_DEF_DIM(nc_cslfilid, 'time',
     &                       NF_UNLIMITED,
     &                       nc_csldimid_time)
         IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)
 
        ! - define the variable relative to the dimension 'time'
         status = NF_DEF_VAR(nc_cslfilid, 'time',
     &                       NF_DOUBLE, 1, nc_csldimid_time,
     &                       nc_cslvarid_time)
         IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)
 
        ! - set its units
         status = NF_PUT_ATT_TEXT(nc_cslfilid, nc_cslvarid_time,
     &                            'units',
     &                            3, 'kyr')
         IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)



! Define data variables:
!----------------------

       ! Data variables with time dimension only

       DO i = nc_cslidx_t_0, nc_cslidx_t_1
          status = NF_DEF_VAR(nc_cslfilid, TRIM(s_shrtname(i)),
     &                        NF_DOUBLE, 1, nc_csldimid_time,
     &                        nc_cslvarid(i))
          IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)
       ENDDO


       ! Data variables with time and ro1 dimensions

       dim(1)=nc_csldimid_ro1
       dim(2)=nc_csldimid_time

       DO i = nc_cslidx_tr1_0, nc_cslidx_tr1_1
          status = NF_DEF_VAR(nc_cslfilid, TRIM(s_shrtname(i)),
     &                        NF_DOUBLE, 2, dim,
     &                        nc_cslvarid(i))
         IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)
       ENDDO


! Set data variable attributes
!-----------------------------

       DO i = 1, nc_csl_ndata

          ilen = LEN_TRIM(s_longname(i))
          status = NF_PUT_ATT_TEXT(nc_cslfilid, nc_cslvarid(i),
     &                             'long_name',
     &                              ilen, s_longname(i))
          IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

          ilen = LEN_TRIM(s_unitname(i))
          status = NF_PUT_ATT_TEXT(nc_cslfilid, nc_cslvarid(i),
     &                             'units',
     &                             ilen, s_unitname(i))
          IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)
       ENDDO


! Set global attributes
!----------------------

       string256 = 'MBM model simulations'
       status = NF_PUT_ATT_TEXT(nc_cslfilid,
     &                          NF_GLOBAL, 'title',
     &                          LEN_TRIM(string256), string256)
       IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)


! End define mode
!----------------
   
       status = NF_ENDDEF(nc_cslfilid)
       IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)


! Set 'ro1' coordinate equal to its index value
!----------------------------------------------

        DO i = 1, nro1
          status = NF_PUT_VAR1_INT(nc_cslfilid,
     &                             nc_cslvarid_ro1, i, ro1+i)
          IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)
        ENDDO


      status = NF_SYNC(nc_cslfilid)
      IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)


      RETURN

      !*****************************************************************
      END SUBROUTINE SETUP_MBM_CSL_NCFILE
      !*****************************************************************



      !*****************************************************************
      SUBROUTINE SETUP_MBM_CMS_NCFILE
      !*****************************************************************
      USE MOD_MBM_INFO
      USE MOD_MBM_GEOMETRY, ONLY: nro1, nro2, nro3, ro1

      IMPLICIT NONE

      INTEGER :: status, i, ilen
      INTEGER, DIMENSION(2) :: dim, start, count
      CHARACTER(LEN=256) :: string256


      CHARACTER(LEN=gnrl_maxnclongname_len), DIMENSION(nc_cms_ndata)
     &   :: s_longname
      CHARACTER(LEN=gnrl_maxncshrtname_len), DIMENSION(nc_cms_ndata)
     &   :: s_shrtname
      CHARACTER(LEN=gnrl_maxncunitname_len), DIMENSION(nc_cms_ndata)
     &   :: s_unitname


#include "netcdf.inc"


! Initialise the basic information
! ----------------------

        s_longname(nc_cmsidx_slv)   = 'Sea-level rel. to present-day'
        s_shrtname(nc_cmsidx_slv)   = 'slv'
        s_unitname(nc_cmsidx_slv)   = 'm'

        s_longname(nc_cmsidx_ff1)   = 'Forcing Function 1'
        s_shrtname(nc_cmsidx_ff1)   = 'ff1'
        s_unitname(nc_cmsidx_ff1)   = '-'

        s_longname(nc_cmsidx_pco2a) = 'Atmospheric pCO2'
        s_shrtname(nc_cmsidx_pco2a) = 'pco2'
        s_unitname(nc_cmsidx_pco2a) = 'ppmv'

        s_longname(nc_cmsidx_dc13a) = 'atmospheric delta C-13'
        s_shrtname(nc_cmsidx_dc13a) = 'dc13a'
        s_unitname(nc_cmsidx_dc13a) = 'permil'

        s_longname(nc_cmsidx_dc14a) = 'Atmospheric Delta C-14'
        s_shrtname(nc_cmsidx_dc14a) = 'Dc14a'
        s_unitname(nc_cmsidx_dc14a) = 'permil'

        s_longname(nc_cmsidx_vols)  = 'Reservoir Volume'
        s_shrtname(nc_cmsidx_vols)  = 'volum'
        s_unitname(nc_cmsidx_vols)  = '10^18 m^3'

        s_longname(nc_cmsidx_temp)  = 'Temperature'
        s_shrtname(nc_cmsidx_temp)  = 'temp'
        s_unitname(nc_cmsidx_temp)  = 'K'

        s_longname(nc_cmsidx_salin) = 'Salinity'
        s_shrtname(nc_cmsidx_salin) = 's'
        s_unitname(nc_cmsidx_salin) = '-'

        s_longname(nc_cmsidx_po4)   = 'Phosphate Concentration'
        s_shrtname(nc_cmsidx_po4)   = 'po4'
        s_unitname(nc_cmsidx_po4)   = 'mol/m^3'

        s_longname(nc_cmsidx_oxy)   = 'Dissolved Oxygen Concentration'
        s_shrtname(nc_cmsidx_oxy)   = 'o2'
        s_unitname(nc_cmsidx_oxy)   = 'mol/m^3'

        s_longname(nc_cmsidx_dic)   = 'Dissolved Inorganic Carbon'
        s_shrtname(nc_cmsidx_dic)   = 'dic'
        s_unitname(nc_cmsidx_dic)   = 'mol/m^3'

        s_longname(nc_cmsidx_alk)   = 'Total Alkalinity'
        s_shrtname(nc_cmsidx_alk)   = 'alk'
        s_unitname(nc_cmsidx_alk)   = 'eq/m^3'

        s_longname(nc_cmsidx_dc13)  = 'DIC delta C-13'
        s_shrtname(nc_cmsidx_dc13)  = 'dc13'
        s_unitname(nc_cmsidx_dc13)  = 'permil'

        s_longname(nc_cmsidx_dc14)  = 'DIC Delta C-14'
        s_shrtname(nc_cmsidx_dc14)  = 'Dc14'
        s_unitname(nc_cmsidx_dc14)  = 'permil'


! Create the data file
!-----------------------

        status = NF_CREATE(cmsfil, NF_CLOBBER, nc_cmsfilid)
        IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

! Define dimensions
!----------------------

        ! All Reservoirs: dimension 'ro'
        ! - define the dimension itself
        status = NF_DEF_DIM(nc_cmsfilid,
     &                      'ro', nro1+nro2+nro3,
     &                      nc_cmsdimid_ro)
        IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

        ! - define the variable relative to the dimension 'ro'
        status = NF_DEF_VAR(nc_cmsfilid, 'ro',
     &                      NF_INT, 1, nc_cmsdimid_ro,
     &                      nc_cmsvarid_ro)
        IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)


        ! Time: dimension 'time'
        ! - define the dimension itself
        status = NF_DEF_DIM(nc_cmsfilid,
     &                      'time', NF_UNLIMITED,
     &                      nc_cmsdimid_time)
        IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)
 
        ! - define the variable relative to the dimension 'time'
        status = NF_DEF_VAR(nc_cmsfilid, 'time',
     &                      NF_DOUBLE, 1, nc_cmsdimid_time,
     &                      nc_cmsvarid_time)
        IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

        ! - set its units
        status = NF_PUT_ATT_TEXT(nc_cmsfilid, nc_cmsvarid_time,
     &                           'units',
     &                           3, 'kyr')
        IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)



! Define data variables:
!----------------------

       ! Data variables with time dimension only

       DO i = nc_cmsidx_t_0, nc_cmsidx_t_1
          status = NF_DEF_VAR(nc_cmsfilid, TRIM(s_shrtname(i)),
     &                        NF_DOUBLE, 1, nc_cmsdimid_time,
     &                        nc_cmsvarid(i))
          IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)
       ENDDO


       ! Data variables with time and ro dimensions

       dim(1)=nc_cmsdimid_ro
       dim(2)=nc_cmsdimid_time

       DO i = nc_cmsidx_tr_0, nc_cmsidx_tr_1
          status = NF_DEF_VAR(nc_cmsfilid, TRIM(s_shrtname(i)),
     &                        NF_DOUBLE, 2, dim,
     &                        nc_cmsvarid(i))
          IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)
       ENDDO


! Set data variable attributes
!-----------------------------

       DO i = 1, nc_cms_ndata

          ilen = LEN_TRIM(s_longname(i))
          status = NF_PUT_ATT_TEXT(nc_cmsfilid, nc_cmsvarid(i),
     &                             'long_name',
     &                              ilen, s_longname(i))
          IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

          ilen = LEN_TRIM(s_unitname(i))
          status = NF_PUT_ATT_TEXT(nc_cmsfilid, nc_cmsvarid(i),
     &                             'units',
     &                             ilen, s_unitname(i))
          IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)
       ENDDO


! Set global attributes
!----------------------

       string256 = 'MBM model simulations'
       status = NF_PUT_ATT_TEXT(nc_cmsfilid,
     &                          NF_GLOBAL, 'title',
     &                          LEN_TRIM(string256), string256)
       IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)


! End define mode
!----------------
   
       status = NF_ENDDEF(nc_cmsfilid)
       IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)


! Set 'ro' coordinate equal to its index value
!---------------------------------------------
       DO i = 1, nro1+nro2+nro3
          status = NF_PUT_VAR1_INT(nc_cmsfilid,
     &                             nc_cmsvarid_ro, i, ro1+i)
          IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)
       ENDDO

 
      status = NF_SYNC(nc_cmsfilid)
      IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

      RETURN
      
      !*****************************************************************
      END SUBROUTINE SETUP_MBM_CMS_NCFILE
      !*****************************************************************

#endif

      END MODULE MOD_MBM_FILES
