!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#ifdef FN_THISFILE
#undef FN_THISFILE
#endif
#define FN_THISFILE "mod_mbm_biocarbonates.F90"
#ifndef __LINE__
#define __LINE__ 0
#endif
!===========================
MODULE MOD_MBM_BIOCARBONATES
!===========================

USE mod_mbm_geometry, ONLY: nro1

IMPLICIT NONE

PRIVATE                             ! By default, everything in this module is private


! ====================
! Carbonate Rain Ratio
! ====================

PUBLIC CARB_RAINRATIO_FROMFILE, CARB_RAINRATIO_ITP,        &
       CARB_RAINRATIO_DFLT_RESET, CARB_RAINRATIO_DFLT_GET, &
       CARB_RAINRATIO_MINMAX_GET


! Data storage
! ------------

INTEGER, PARAMETER &                ! Number of carbonate rain ratios
   :: n_rhocac_types  = 1           ! currently considered in this module

CHARACTER(LEN=31), DIMENSION(n_rhocac_types), PARAMETER &
   :: rhocac_names = (/ 'rhocac' /)
INTEGER, PARAMETER &
   :: i_rhocac_bulk = 1             ! Index of "rhocac_bulk" in the array of names

                       
INTEGER, SAVE &
   :: n_rhocac_time = 0             ! Number of data-set entries to consider for rhocac_time
INTEGER, SAVE &
   :: n_rhocac_bulk = 0             ! Number of data-set entries to consider for rhocac_bulk

LOGICAL, SAVE &
   :: f_rhocacsetup_tbd = .TRUE.    ! Flag controlling whether setup still needs To Be Done (TBD) 


DOUBLE PRECISION, DIMENSION(:), POINTER, SAVE &
   :: rhocac_time  => NULL()
DOUBLE PRECISION, DIMENSION(:,:), POINTER, SAVE &
   :: rhocac_bulk  => NULL()
DOUBLE PRECISION, DIMENSION(:), POINTER, SAVE &
   :: rhocac_bulkm => NULL()
DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE, SAVE &
   :: rhocac_bulk_min, rhocac_bulk_max

DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE, SAVE &
   :: rhocac_bulk_itp


INTEGER, SAVE &
   :: i_rhocac_itp = -1             ! Current index into the rhocac_time array


! Default values
! --------------

DOUBLE PRECISION, PARAMETER     &   ! Default replication period
   :: rhocac_period_dflt = 120D3

DOUBLE PRECISION, DIMENSION(nro1) & ! Default carbonate rain ratios (may be overwritten
   :: rhocac_dflt = (/ 0.023D0, 0.31D0, 0.023D0, 0.31D0, 0.023D0 /) ! with DAT values by INITIA)


INTEGER, SAVE &                     ! 0: hard-coded value (previous line)
   :: rhocac_dflt_vers = 0          ! 1: reset to value from DAT file

LOGICAL &
   :: rhocac_dflt_maybereset = .TRUE. ! .TRUE.: the default value may be reset
                                    ! .FALSE.: the default value must not be reset



! ==================
! Aragonite Fraction
! ==================

PUBLIC ARAG_FRACTION, ARAG_FRACTION_DFLT_RESET, ARAG_FRACTION_DFLT_GET, ARAG_FRACTION_ITP


! Data storage
! ------------

INTEGER, PARAMETER &                ! Number of aragonite fraction types
   :: n_frcara_types  = 1           ! currently considered in this module
CHARACTER(LEN=31), DIMENSION(n_frcara_types), PARAMETER &
   :: frcara_names = (/ 'frcara' /)
INTEGER, PARAMETER &
   :: i_frcara_bulk = 1

                         
INTEGER, SAVE &
   :: n_frcara_time   = 0           ! Number of data-set entries in frcara_time
INTEGER, SAVE &
   :: n_frcara_bulk   = 0           ! Number of data-set entries in frcara_bulk


LOGICAL, SAVE &
   :: f_frcarasetup_tbd = .TRUE.    ! Flag controlling whether setup still needs To Be Done (TBD) 


DOUBLE PRECISION, DIMENSION(:), POINTER, SAVE &
   :: frcara_time   => NULL()
DOUBLE PRECISION, DIMENSION(:), POINTER, SAVE &
   :: frcara_bulk   => NULL()

DOUBLE PRECISION, SAVE &            ! Interpolated value
   :: frcara_bulk_itp = 0.35D0      ! CURRENTLY PRESET TO ALLOW CONSISTENCY CHECKS


INTEGER, SAVE &                     ! Index of most recent bracketting interval for time
   :: i_frcara_itp = -1


! Default values
! --------------

DOUBLE PRECISION, PARAMETER &
   :: frcara_period_dflt = 120D3

DOUBLE PRECISION, SAVE &            ! Default value, may be overwritten by
   :: frcara_dflt = 0.35D0          ! INITIA with DAT value!

INTEGER, SAVE &                     ! 0: hard-coded value (previous line)
   :: frcara_dflt_vers = 0          ! 1: reset to value from DAT file

LOGICAL &
   :: frcara_dflt_maybereset = .TRUE. ! .TRUE.: the default value may be reset
                                    ! .FALSE.: the default value must not be reset


! =======================
! Common module variables
! =======================

CHARACTER(LEN=*), PARAMETER                                            &
   :: fn_thisfile     = FN_THISFILE,                                   &
      fmt_info_a      = '("['//fn_thisfile//']: ", A)' ,               &
      fmt_info_ai     = '("['//fn_thisfile//']: ", A, I0)',            &
      fmt_infolin_ia  = '("['//fn_thisfile//':", I0, "]: ", A)',       &
      fmt_infolin_iai = '("['//fn_thisfile//':", I0, "]: ", A, I0)'




CONTAINS


!=================================================
SUBROUTINE CARB_RAINRATIO_FROMFILE(time, rhocac_x)
!=================================================

! This subroutine returns the carbonate rain ratios for each of the
! hypsometric profiles, as a function of time.
! Possibly, a remapping from surface reservoirs to profiles
! may have to be performed (currently not implemented).

 
USE mod_mbm_geometry, ONLY: nro1, i_hypspr_1, i_hypspr_n
USE mod_mbm_files, ONLY: dbguni, resusd, resuni, crrusd, crrfil, &
                        HANDLE_NCERRORS
USE mod_mbm_ncforcing


IMPLICIT NONE


! Dummy argument variables
! ------------------------

DOUBLE PRECISION, INTENT(IN) &
   :: time

DOUBLE PRECISION, DIMENSION(i_hypspr_1:i_hypspr_n), INTENT(OUT) &
   :: rhocac_x



! Local variables
! ---------------

DOUBLE PRECISION :: rhocac_factor


LOGICAL &
   :: use_some_dflt



TYPE(FORCINGDESC), DIMENSION(n_rhocac_types) &
   :: rhocac_descs



                                    ! First check if the setup still needs to be done

rhocacsetup_tbd: IF (f_rhocacsetup_tbd) THEN

   WRITE(dbguni,*) 
   WRITE(dbguni,'("[MOD_MBM_CARB_RAINRATIO/CARB_RAINRATIO_FROMFILE]:")')
   WRITE(dbguni,'(" Carbonate Rain Ratios")')

   IF (n_rhocac_time /= 0) THEN     ! Reset everything if necessary

      IF(ASSOCIATED(rhocac_time))      DEALLOCATE(rhocac_time)
      IF(ASSOCIATED(rhocac_bulk))      DEALLOCATE(rhocac_bulk)
      IF(ASSOCIATED(rhocac_bulkm))     DEALLOCATE(rhocac_bulkm)
      IF(ALLOCATED(rhocac_bulk_min))   DEALLOCATE(rhocac_bulk_min)
      IF(ALLOCATED(rhocac_bulk_max))   DEALLOCATE(rhocac_bulk_max)

      IF(ALLOCATED(rhocac_bulk_itp))   DEALLOCATE(rhocac_bulk_itp)

      n_rhocac_time = 0
      n_rhocac_bulk = 0

   ENDIF

   use_some_dflt = .FALSE.          ! We expect to find data for all variables

   ifcrrused: IF (crrusd == 1) THEN
                                    ! If we use carbonate rain ratio scenarios
                                    ! from a file then read them in now

      IF (resusd /= 0) THEN
         WRITE(resuni,*)
         WRITE(resuni,*)
         WRITE(resuni,'("[Carbonate rain ratio scenario info]:")')

         WRITE(resuni,'(A)') ' Reading in carbonate rain ratio scenario data from file "' // TRIM(crrfil) // '"'
      ENDIF


      CALL READ_NCFORCING(crrfil, rhocac_period_dflt, &
                          rhocac_names, rhocac_time, rhocac_descs)

      n_rhocac_time = SIZE(rhocac_time)


      IF (.NOT. rhocac_descs(i_rhocac_bulk)%use_dflt) THEN

         rhocac_bulk  => rhocac_descs(i_rhocac_bulk)%forcing
         rhocac_bulkm => rhocac_descs(i_rhocac_bulk)%forcing_mean

         n_rhocac_bulk = SIZE(rhocac_bulk) / nro1

      ELSE

         ALLOCATE(rhocac_bulk(nro1, 1))
         ALLOCATE(rhocac_bulkm(nro1))

         rhocac_bulk(:,1) = rhocac_dflt(:)
         rhocac_bulkm(:)  = rhocac_dflt(:)

         n_rhocac_bulk = 1

         rhocac_dflt_maybereset = .FALSE. ! The default value must not be reset from now on
         use_some_dflt = .TRUE.

      ENDIF


   ELSE ifcrrused

                                    ! No scenario file given:
                                    ! use default values
      IF (resusd /= 0) THEN
         WRITE(resuni,'(" No CRR file given in mbm.cfg.")')
         WRITE(resuni,'(" Using default values")')

         WRITE(dbguni, *)
         WRITE(dbguni,'(" No CRR file given in mbm.cfg.")')
         WRITE(dbguni,'(" Using default values")')
      ENDIF

      ALLOCATE(rhocac_bulk(nro1, 1))
      ALLOCATE(rhocac_bulkm(nro1))
 
      rhocac_bulk(:,1) = rhocac_dflt(:)
      rhocac_bulkm(:)  = rhocac_dflt(:)

      n_rhocac_bulk = 1
      n_rhocac_time = 1

      i_rhocac_itp = -1

      rhocac_dflt_maybereset = .FALSE. ! The default value must not be reset from now on
      use_some_dflt = .TRUE.

   ENDIF ifcrrused

   ALLOCATE(rhocac_bulk_min(nro1))  ! Allocate space for the minimum values
   rhocac_bulk_min(:) = MINVAL(rhocac_bulk(:,:), 2)  ! and calculate them.
   
   ALLOCATE(rhocac_bulk_max(nro1))  ! Allocate space for the maximum values
   rhocac_bulk_max(:) = MAXVAL(rhocac_bulk(:,:), 2)  ! and calculate them.

   ALLOCATE(rhocac_bulk_itp(nro1))  ! Allocate space for the interpolates.

                                    ! Pre-set all interpolates for
                                    ! fluxes that have only one datum.
   IF(n_rhocac_bulk == 1) rhocac_bulk_itp(:) = rhocac_bulk(:,1)


   WRITE(dbguni, *)
   WRITE(dbguni, '(" n_rhocac_bulk = ", I0, " (time size: ", I0, ")")') &
     n_rhocac_bulk, SIZE(rhocac_bulk)/SIZE(rhocac_bulk,1)


   IF (use_some_dflt) THEN
      IF (resusd /= 0) THEN
         WRITE(resuni,'(" Warning: one or more default values have been adopted.")')
         WRITE(resuni,'(" Consistency needs to be checked!")')
      ENDIF

      WRITE(dbguni, *)
      WRITE(dbguni, *) 'One or more default flux values adopted.'
   ENDIF


   f_rhocacsetup_tbd = .FALSE.      ! Carbonate rain ratio setup is done now

   CALL FLUSH(dbguni)

ENDIF rhocacsetup_tbd


! CURRENTLY NO INTERPOLATION PERFORMED HERE (for DSR2 compatibility)!
! The actual ierpolation is performed in SETGEO, and here
! we return the latest value produced by SETGEO

rhocac_x(:) = rhocac_bulk_itp(:)

!=====================================
END SUBROUTINE CARB_RAINRATIO_FROMFILE
!=====================================



!==================================
SUBROUTINE CARB_RAINRATIO_ITP(time)
!==================================

USE mod_mbm_paleod, ONLY: epoch
USE mod_mbm_files, ONLY: erruni

IMPLICIT NONE

DOUBLE PRECISION, INTENT(IN) :: time


INTEGER &
   :: i1, i2

DOUBLE PRECISION &
   :: rhocac_factor


IF (f_rhocacsetup_tbd) THEN

   WRITE(erruni,fmt_infolin_ia) (__LINE__), 'Setup is still To Be Done. Aborting!'
   CALL ABORT()

ENDIF


IF (n_rhocac_time == 1) THEN        ! If there is only one single
                                    ! datum, use it as a constant.
   rhocac_bulk_itp(:)  = rhocac_bulk(:,1)

   RETURN

ENDIF


                                    ! We need to recalculate at least one interpolation

                                    ! If first call, preset i_frcara
IF (i_rhocac_itp == -1) i_rhocac_itp = 1


IF (time > rhocac_time(n_rhocac_time)) THEN

   WRITE(erruni,fmt_infolin_ia) (__LINE__), 'time > rhocac_time(n_rhocac_time)'
   WRITE(erruni,*) ' time = ', time
   WRITE(erruni,*) ' rhocac_time(n_rhocac_time) = ', rhocac_time(n_rhocac_time)
   WRITE(erruni,fmt_infolin_ia) (__LINE__), 'Aborting!'

   CALL ABORT()

ELSEIF(time < rhocac_time(1)) THEN

   WRITE(erruni,fmt_infolin_ia) (__LINE__), ' time < rhocac_time(1)'
   WRITE(erruni,*) ' time = ', time
   WRITE(erruni,*) ' rhocac_time(1) = ',  rhocac_time(1)
   WRITE(erruni,fmt_infolin_ia) (__LINE__), 'Aborting!'

   CALL ABORT()

ELSE                                ! OK time is acceptable

                                    ! Search for i_rhocac_itp such that
                                    ! rhocac_time(i_rhocac_itp) <= time < rhocac_time(i_rhocac_itp+1)
   DO WHILE(time < rhocac_time(i_rhocac_itp))
!     WRITE(dbguni,*) i_rhocac_itp, rhocac_time(i_rhocac_teim,1), time, i_rhocac-1
      i_rhocac_itp = i_rhocac_itp-1
   ENDDO

   IF (i_rhocac_itp < n_rhocac_time) THEN
      DO WHILE(time >= rhocac_time(i_rhocac_itp+1))
!        WRITE(dbguni,*) i_rhocac_itp, rhocac_time(i_rhocac_itp+1,1), time, i_rhocac_itp+1
         IF (i_rhocac_itp == (n_rhocac_time-1)) THEN
            EXIT
         ELSE
            i_rhocac_itp = i_rhocac_itp+1
         ENDIF
      ENDDO
   ENDIF

   i1 = i_rhocac_itp
   i2 = i_rhocac_itp + 1

!  WRITE(dbguni,*) 'fi-', rhocac_time(i1), temps, rhocac_time(i2)

   rhocac_factor = (time - rhocac_time(i1)) / (rhocac_time(i2) - rhocac_time(i1))

   rhocac_bulk_itp(:) = rhocac_bulk(:,i1) + (rhocac_bulk(:,i2) - rhocac_bulk(:,i1))*rhocac_factor


ENDIF


RETURN

!================================
END SUBROUTINE CARB_RAINRATIO_ITP
!================================




!==================================================
SUBROUTINE CARB_RAINRATIO_DFLT_RESET(rhocac_newdflt)
!==================================================

! This subroutine resets the default value
! rhocac_dflt. This can only be done once.

USE mod_mbm_files, ONLY: erruni

IMPLICIT NONE


DOUBLE PRECISION, DIMENSION(:), INTENT(IN) &
   :: rhocac_newdflt

IF(rhocac_dflt_maybereset) THEN

   rhocac_dflt(:) = rhocac_newdflt(:)
   rhocac_dflt_vers = 1                ! Reset from DAT file
   rhocac_dflt_maybereset = .FALSE.    ! The default value must not be reset now

ELSE

   WRITE(erruni,'(A)') '[MOD_MBM_BIOCARBONATES/CARB_RAINRATIO_RESETDFLT]: '
   WRITE(erruni,'(A)') ' rhocac_dflt cannot been reset. Aborting.'
   CALL ABORT()

ENDIF

RETURN

!=======================================
END SUBROUTINE CARB_RAINRATIO_DFLT_RESET
!=======================================




!========================================================================
SUBROUTINE CARB_RAINRATIO_DFLT_GET(rhocac_dflt_out, rhocac_dflt_vers_out)
!========================================================================

! This subroutine returns the default value rhocac_dflt and the
! version number of the currently active default values.
! This subroutine is the only interface to get this information
! from outside the module.

IMPLICIT NONE


DOUBLE PRECISION, DIMENSION(:), INTENT(OUT) &
   :: rhocac_dflt_out
INTEGER, INTENT(OUT) &
   :: rhocac_dflt_vers_out


rhocac_dflt_out(:)   = rhocac_dflt(:)
rhocac_dflt_vers_out = rhocac_dflt_vers

RETURN

!=====================================
END SUBROUTINE CARB_RAINRATIO_DFLT_GET
!=====================================




!========================================================================
SUBROUTINE CARB_RAINRATIO_MINMAX_GET(rhocac_min_out, rhocac_max_out)
!========================================================================

! This subroutine returns the arrays with the minimu and maximum value
! for rhocac_bulk.
! This subroutine is the only interface to get this information
! from outside the module.

IMPLICIT NONE


DOUBLE PRECISION, DIMENSION(:), INTENT(OUT) &
   :: rhocac_min_out, rhocac_max_out


rhocac_min_out(:) = rhocac_bulk_min(:)
rhocac_max_out(:) = rhocac_bulk_max(:)

RETURN

!=======================================
END SUBROUTINE CARB_RAINRATIO_MINMAX_GET
!=======================================






!=====================================
SUBROUTINE ARAG_FRACTION(time, frcara)
!=====================================

USE mod_mbm_files, ONLY: dbguni, resusd, resuni, afcusd, afcfil, &
                         HANDLE_NCERRORS
USE mod_mbm_ncforcing

IMPLICIT NONE


! Dummy argument variables
! ------------------------

DOUBLE PRECISION, INTENT(IN) &
   :: time
DOUBLE PRECISION, INTENT(OUT) &
   :: frcara


! Local variables
! ---------------

LOGICAL &
   :: use_some_dflt

TYPE(FORCINGDESC), DIMENSION(n_frcara_types) &
   :: frcara_descs


frcarasetup_tbd: IF(f_frcarasetup_tbd) THEN

                                     ! If the frcara-related parts of
                                     ! this module have not yet been
                                     ! set up, set it up now.
   WRITE(dbguni,*) 
   WRITE(dbguni,'("[MOD_MBM_BIOCARBONATES/ARAG_FRACTION]:")')
   WRITE(dbguni,'(" Setting up the Aragonite Fraction in Carbonates")')

   IF (n_frcara_time /= 0) THEN     ! Reset everything if necessary

      IF(ASSOCIATED(frcara_time)) DEALLOCATE(frcara_time)
      IF(ASSOCIATED(frcara_bulk)) DEALLOCATE(frcara_bulk)

      n_frcara_time = 0
      n_frcara_bulk = 0

      i_frcara_itp = -1

   ENDIF

   use_some_dflt = .FALSE.          ! We expect to find data for all variables

   ifafcused: IF (afcusd == 1) THEN
                                    ! If we use an aragonite fraction scenario
                                    ! from a file then read it in now
      IF (resusd /= 0) THEN
         WRITE(resuni,*)
         WRITE(resuni,*)
         WRITE(resuni,'("[Aragonite fraction scenario info]: ")')
         WRITE(resuni,'(A)') ' Reading in aragonite fraction scenario data from file "' // TRIM(afcfil) // '"'
      ENDIF


      CALL READ_NCFORCING(afcfil, frcara_period_dflt, &
                          frcara_names, frcara_time, frcara_descs)

      n_frcara_time = SIZE(frcara_time)


      IF (.NOT. frcara_descs(i_frcara_bulk)%use_dflt) THEN

         frcara_bulk  => frcara_descs(i_frcara_bulk)%forcing(1,:)

         n_frcara_bulk = SIZE(frcara_bulk)

      ELSE

         ALLOCATE(frcara_bulk(1))
         frcara_bulk(1) = frcara_dflt

         n_frcara_bulk = 1

         frcara_dflt_maybereset = .FALSE. ! The default value must not be reset from now on
         use_some_dflt = .TRUE.

      ENDIF


   ELSE ifafcused

                                    ! No scenario file given:
                                    ! use default values
      IF (resusd /= 0) THEN

         WRITE(resuni,'(" No AFC file given in mbm.cfg.")')
         WRITE(resuni,'(" Using default values")')

         WRITE(dbguni, *)
         WRITE(dbguni,'(" No AFC file given in mbm.cfg.")')
         WRITE(dbguni,'(" Using default values")')
      ENDIF

      ALLOCATE(frcara_bulk(1))

      frcara_bulk(1) = frcara_dflt

      n_frcara_bulk = 1
      n_frcara_time = 1
      frcara_dflt_maybereset = .FALSE. ! The default value must not be reset from now on
      use_some_dflt = .TRUE.

   ENDIF ifafcused

                                    ! Pre-set all interpolates for
                                    ! fluxes that have only one datum.
   IF(n_frcara_bulk == 1) frcara_bulk_itp = frcara_bulk(1)


   WRITE(dbguni, *)
   WRITE(dbguni, '(" n_frcara_bulk = ", I0, " (time size: ", I0, ")")') &
     n_frcara_bulk, SIZE(frcara_bulk)


   IF (use_some_dflt) THEN
      IF (resusd /= 0) THEN
         WRITE(resuni,'(" Warning: one or more default values have been adopted.")')
         WRITE(resuni,'(" Consistency needs to be checked!")')
      ENDIF

      WRITE(dbguni, *)
      WRITE(dbguni, *) 'One or more default flux values adopted.'
   ENDIF


   f_frcarasetup_tbd = .FALSE.       ! Aragonite fraction setup is done now

   CALL FLUSH(dbguni)                ! Make sure all is written to the DBG file

ENDIF frcarasetup_tbd


! CURRENTLY NO INTERPOLATION PERFORMED HERE (for DSR2 compatibility)!
! New interpolates are only calculated whenever SETGEO is called, and
! here we return the latest available value.


frcara = frcara_bulk_itp

RETURN

!===========================
END SUBROUTINE ARAG_FRACTION
!===========================



!==================================
SUBROUTINE ARAG_FRACTION_ITP(temps)
!==================================

USE mod_mbm_paleod, ONLY: epoch
USE mod_mbm_files, ONLY: erruni, dbguni


IMPLICIT NONE

DOUBLE PRECISION, INTENT(IN) &
   :: temps


INTEGER &
   :: i1, i2

DOUBLE PRECISION &
   :: frcara_factor



IF (f_frcarasetup_tbd) THEN

   WRITE(erruni,fmt_infolin_ia) (__LINE__), 'Setup is still To Be Done. Aborting!'
   CALL ABORT()

ENDIF


IF (n_frcara_time == 1) THEN        ! If there is only one single
                                    ! datum, use it as a constant.
   frcara_bulk_itp  = frcara_bulk(1)

   RETURN

ENDIF

                                    ! If first call, preset i_frcara
IF (i_frcara_itp == -1) i_frcara_itp = 1


IF (temps > frcara_time(n_frcara_time)) THEN

   WRITE(erruni,fmt_infolin_ia) (__LINE__), 'temps > frcara_time(n_frcara_time)'
   WRITE(erruni,*) 'temps = ', temps
   WRITE(erruni,*) 'frcara_time(n_frcara_time) = ', frcara_time(n_frcara_time)
   WRITE(erruni,fmt_infolin_ia) (__LINE__), 'Aborting!'

   CALL ABORT()

ELSEIF (temps < frcara_time(1)) THEN

   WRITE(erruni,*) 'temps < frcara_time(1)'
   WRITE(erruni,*) 'temps = ', temps
   WRITE(erruni,*) 'frcara_time(1) = ', frcara_time(1)
   WRITE(erruni,fmt_infolin_ia) (__LINE__), 'Aborting!'

   CALL ABORT()

ELSE
                                    ! Search for i_frcara_itp such that
                                    ! frcara_time(i_frcara_itp) <= temps < frcara_time(i_frcara_itp+1)
   DO WHILE (temps < frcara_time(i_frcara_itp))
!     WRITE(dbguni,*) i_frcara_itp, frcara_time(i_frcara_itp), temps, i_frcara_itp-1
      i_frcara_itp = i_frcara_itp-1
   ENDDO

   IF (i_frcara_itp < n_frcara_time) THEN
      DO WHILE (temps >= frcara_time(i_frcara_itp+1))
!        WRITE(dbguni,*) i_frcara_itp, frcara_time(i_frcara_itp+1), temps, i_frcara_itp+1
         IF (i_frcara_itp == (n_frcara_time-1)) THEN
            EXIT
         ELSE
            i_frcara_itp = i_frcara_itp+1
         ENDIF
      ENDDO
   ENDIF

   i1 = i_frcara_itp
   i2 = i_frcara_itp + 1

!  WRITE(dbguni,*) 'fi-', frcara_time(i1), temps, frcara_time(i2)

   frcara_factor =  (temps - frcara_time(i1)) / (frcara_time(i2) - frcara_time(i1))

   frcara_bulk_itp = frcara_bulk(i1) + (frcara_bulk(i2) - frcara_bulk(i1)) * frcara_factor


ENDIF


RETURN

!===============================
END SUBROUTINE ARAG_FRACTION_ITP
!===============================




!==================================================
SUBROUTINE ARAG_FRACTION_DFLT_RESET(frcara_newdflt)
!==================================================

! This subroutine resets the default value
! frcara_dflt. This can only be done once.

USE mod_mbm_files, ONLY: erruni


IMPLICIT NONE


DOUBLE PRECISION, INTENT(IN) :: frcara_newdflt

IF(frcara_dflt_maybereset) THEN

   frcara_dflt = frcara_newdflt
   frcara_dflt_vers = 1                ! Reset from DAT file
   frcara_dflt_maybereset = .FALSE.    ! The default value must not be reset now

ELSE

   WRITE(erruni,'(A)') '[MOD_MBM_BIOCARBONATES/ARAG_FRACTION_RESETDFLT]: '
   WRITE(erruni,'(A)') ' frcara_dflt cannot been reset. Aborting.'
   CALL ABORT()

ENDIF

RETURN

!======================================
END SUBROUTINE ARAG_FRACTION_DFLT_RESET
!======================================




!=======================================================================
SUBROUTINE ARAG_FRACTION_DFLT_GET(frcara_dflt_out, frcara_dflt_vers_out)
!=======================================================================

! This subroutine returns the default value frcara_dflt and the
! version number of the currently active default values.
! This subroutine is the only interface to get this information
! from outside the module.

IMPLICIT NONE


DOUBLE PRECISION, INTENT(OUT) :: frcara_dflt_out
INTEGER, INTENT(OUT) :: frcara_dflt_vers_out


frcara_dflt_out      = frcara_dflt
frcara_dflt_vers_out = frcara_dflt_vers

RETURN

!====================================
END SUBROUTINE ARAG_FRACTION_DFLT_GET
!====================================




!===============================
END MODULE MOD_MBM_BIOCARBONATES
!===============================

