!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#ifdef FN_THISFILE
#undef FN_THISFILE
#endif
#define FN_THISFILE "initia.F"
#ifndef __LINE__
#define __LINE__ 0
#endif
      !*****************************************************************
      SUBROUTINE INITIA(tinf, tmax, pas, tol, 
     <                  pco2a, c13a, c14a,
     <                  vpo4, vcit, cc14, valk, cc13, voxy, r76)
      !*****************************************************************

      ! Calls: SETGEO

      USE mod_mbm_geometry
      USE mod_mbm_options
      USE mod_mbm_tempo
      USE mod_mbm_files
      USE mod_mbm_biocts
      USE mod_mbm_other
      USE mod_mbm_biocarbonates, ONLY: ARAG_FRACTION_DFLT_RESET,
     &                                 ARAG_FRACTION,
     &                                 CARB_RAINRATIO_DFLT_RESET,
     &                                 CARB_RAINRATIO_FROMFILE
      USE mod_mbm_chimiq
      USE mod_mbm_wfluxes
      USE mod_mbm_paleod
      USE mod_mbm_conbio
      USE mod_mbm_files, ONLY: erruni, dbguni


      IMPLICIT NONE



#include "netcdf.inc"

      DOUBLE PRECISION tinf, tmax, pas, tol
      DOUBLE PRECISION pco2a, c13a, c14a
      DOUBLE PRECISION vpo4(ro1+1:ro3+nro3)
      DOUBLE PRECISION vcit(ro1+1:ro3+nro3)
      DOUBLE PRECISION cc14(ro1+1:ro3+nro3)
      DOUBLE PRECISION valk(ro1+1:ro3+nro3)
      DOUBLE PRECISION cc13(ro1+1:ro3+nro3)
      DOUBLE PRECISION voxy(ro1+1:ro3+nro3)
      DOUBLE PRECISION r76

      INTEGER i, j, k, nmbflx
      INTEGER :: i1, i2, is, i_first, i_last
      INTEGER :: n_replicbfr, n_replicaft

      INTEGER tstatm, tstsur, tstint, tstpro
      INTEGER flxnet, flxech, cofnet, cofech
      DOUBLE PRECISION addcit, addalk, addpo4
      DOUBLE PRECISION tvolum
      LOGICAL updtts

      DOUBLE PRECISION :: frcara
      DOUBLE PRECISION, DIMENSION(ro1+1:ro1+nro1)
     &                 :: rhocac


      ! Format string to read in organic data and calculate variation
      CHARACTER(LEN=255) form08
      CHARACTER(LEN=80) messag
      DOUBLE PRECISION :: time_data, value_data

! NetCDF related variables
! - general
      INTEGER :: nc_id, status, dimid(2)
! - specific for hypsometric data
      INTEGER :: dpi_dimid, dpi_dimlen, dpi_varid
      INTEGER :: dep_dimid, dep_dimlen, depth_varid, i_refsealevel
      INTEGER :: sfa_varid, vol_varid, maxdep_varid
      CHARACTER(LEN=64) :: global_att ! Length according to uti/bathym_nc.f90
      INTEGER :: ll
! - specific for temperature data
      INTEGER :: oresid_dimid, oresid_dimlen
      INTEGER :: dating_dimid, dating_dimlen
      INTEGER :: dating_varid
      INTEGER :: temp_varid, temp_dimid(2)
      DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE
     &        :: temp_dates
      DOUBLE PRECISION, DIMENSION(:,:), ALLOCATABLE
     &        :: temp_valorig
      DOUBLE PRECISION
     &        :: temp_tinf, temp_tmax


      CHARACTER(LEN=*), PARAMETER ::
     &  fn_thisfile     = FN_THISFILE
      CHARACTER(LEN=*), PARAMETER ::
     &  fmt_info_a      = '("['//fn_thisfile//']: ", A)' ,
     &  fmt_info_ai     = '("['//fn_thisfile//']: ", A, I0)',
     &  fmt_infolin_ia  = '("['//fn_thisfile//':", I0, "]: ", A)',
     &  fmt_infolin_iai = '("['//fn_thisfile//':", I0, "]: ", A, I0)'


      OPEN(UNIT=iniuni, FILE=inifil)



      READ(iniuni,*) tstatm
      READ(iniuni,*) tstsur
      READ(iniuni,*) tstint
      READ(iniuni,*) tstpro

      IF (     (tstatm /= nra1).OR.(tstsur /= nro1)
     &     .OR.(tstint /= nro2).OR.(tstpro /= nro3) ) THEN
         WRITE(erruni,fmt_infolin_ia) (__LINE__), 'The .ini file,'
         WRITE(erruni,*) inifil
         WRITE(erruni,*)'is not in agreement'
         WRITE(erruni,*)'with the configuration of ***initia***.'
         WRITE(erruni,*)'The current configuration is:'
         WRITE(erruni,*)
         WRITE(erruni,*)'     nra1 : ',nra1
         WRITE(erruni,*)'     nro1 : ',nro1
         WRITE(erruni,*)'     nro2 : ',nro2
         WRITE(erruni,*)'     nro3 : ',nro3
         STOP
      ENDIF

      READ(iniuni,*) TINF
      READ(iniuni,*) TMAX
      READ(iniuni,*) PAS
      READ(iniuni,*) TOL

      
      simul_tinf = tinf             ! Copy tinf and tmax into mod_mbmtempo
      simul_tmax = tmax
      simul_t_set = .TRUE.          ! and flag that this has been done


      IF(resusd /= 0) THEN
         WRITE(resuni,*)
         WRITE(resuni,*)
         WRITE(resuni,'("[Time stepping and control info]:")')
         WRITE(resuni,'(" TINF: ",F16.4)') TINF
         WRITE(resuni,'(" TMAX: ",F16.4)') TMAX
         WRITE(resuni,'(" PAS:  ",F16.4)') PAS
         WRITE(resuni,'(" TOL:  ",F16.4)') TOL
      ENDIF

      READ(iniuni,*) pco2a
      READ(iniuni,*) c13a
      READ(iniuni,*) c14a

      DO i=ro1+1, ro3+nro3
         READ(iniuni,*) vpo4(i)
      ENDDO

      DO i=ro1+1, ro3+nro3
         READ(iniuni,*) vcit(i)
      ENDDO

      DO i=ro1+1, ro3+nro3
         READ(iniuni,*) cc14(i)
      ENDDO

      DO i=ro1+1, ro3+nro3
         READ(iniuni,*) valk(i)
      ENDDO
      DO i=ro1+1, ro3+nro3
         READ(iniuni,*) cc13(i)
      ENDDO

      DO i=ro1+1, ro3+nro3
         READ(iniuni,*) voxy(i)
      ENDDO

      READ(iniuni,*) r76

! (24FEB1999)
! Read in additional amounts of CIT, ALK, PO4 (10^18 mol).
! These amounts will be added to the oceanic reservoirs,
! in ratios equal to the respective volumes, to make up the
! indicated totals.

      READ(iniuni,*) addcit
      READ(iniuni,*) addalk
      READ(iniuni,*) addpo4

! (03MAR1993) sediments initialised by **sedim**
! (24FEB1999) Ocean sr87/sr86 in the .ini file

      CLOSE(UNIT=iniuni)

      temps=tinf                    ! initialize temps


      OPEN(UNIT=datuni, FILE=datfil)

      READ(datuni,*)TSTATM
      READ(datuni,*)TSTSUR
      READ(datuni,*)TSTINT
      READ(datuni,*)TSTPRO

      IF ( (TSTATM.NE.nra1).OR.(TSTSUR.NE.nro1)
     &     .OR.(TSTINT.NE.nro2).OR.(TSTPRO.NE.nro3) ) THEN
         WRITE(erruni,fmt_infolin_ia) (__LINE__), 'The .dat file,'
         WRITE(erruni,*) datfil
         WRITE(erruni,*)'is not in agreement'
         WRITE(erruni,*)'with the configuration of ***initia***.'
         WRITE(erruni,*)'The current configuration is:'
         WRITE(erruni,*)' '
         WRITE(erruni,*)'     nra1 : ',nra1
         WRITE(erruni,*)'     nro1 : ',nro1
         WRITE(erruni,*)'     nro2 : ',nro2
         WRITE(erruni,*)'     nro3 : ',nro3
         STOP
      ENDIF

! (09JUN1992) EPOCH added to the variables to be initialised by
! the *.dat file
      READ(datuni,*)epoch

!     /BIOCTS/ ***  Redfield ratios (RHOPC = P:C, RHONC = N:C),
!                   remineralisation constants and PO4 fractions used
!
! Before rev. 222:
!               READ(datuni,*) RHOPC
!               READ(datuni,*) RHONC

! From rev. 222 onwards:
               orgm_p = 1.0D+00

               READ(datuni,*) orgm_c
               RHOPC = orgm_p/orgm_c

               READ(datuni,*) orgm_n
               RHONC = orgm_n/orgm_c

               orgm_ro2 = orgm_c + orgm_n + orgm_n

! Unchanged at rev. 222:
               READ(datuni,*)COREM2
               READ(datuni,*)COREM3
               READ(datuni,*)RHOCAC

               CALL CARB_RAINRATIO_DFLT_RESET(rhocac)

               READ(datuni,*)PO4UTI

               !CALL CARB_RAINRATIO_DFLT_RESET(rhocac)

!     /FLUXES/ ***  Water fluxes between the differents reservoirs.
!              *    The flux from reservoir i to reservoir j is given 
!              *    by wflux(i,j). Only fluxes between neighbouring
!              *    reservoirs are initialised. (Units:  10^18 m3/yr).
!              *
!              *    25JAN1999:
!              *    The actual wflux is now updated by *setwfl*,
!              *    called by *setgeo*. The inital flux data
!              *    are stored in wflxvn, wflxvm, wflxhn, wflxhm
!              *    together with the scaling coefficients cofvn,
!              *    cofvm, cofhn, cofhm (module MOD_MBM_WFLUXES).
!              *    The meanings of the endings are:
!              *    'vn' : vertical net
!              *    'vm' : vertical mixing
!              *    'hn' : horizontal net
!              *    'hm' : horizontal mixing
!              *    The initial distribution is obtained by setting
!              *    wflux = cofvn*wflxvn + cofvm*wflxvm +
!              *            cofhn*wflxhn + cofhm*wflxhm
!
               DO i=ro1+1,ro3+nro3
                  DO j=ro1+1,ro3+nro3
                    wflxvn(i,j)=0.D+00
                    wflxvm(i,j)=0.D+00
                    wflxhn(i,j)=0.D+00
                    wflxhm(i,j)=0.D+00
                  ENDDO
               ENDDO
!
!              *** Vertical Fluxes ***
               READ(datuni,*) nmbflx, cofnet, cofech
               cofvn = DBLE(cofnet)
               cofvm = DBLE(cofech)
               DO k = 1, nmbflx
                  READ(datuni,*) i, j, flxnet, flxech
                  wflxvn(ro1+i, ro1+j) = DBLE(flxnet)
                  wflxvm(ro1+i, ro1+j) = DBLE(flxech)
                  wflxvm(ro1+j, ro1+i) = DBLE(flxech)
               ENDDO

!              *** Horizontal Fluxes ***
               READ(datuni,*) nmbflx, cofnet, cofech
               cofhn = DBLE(cofnet)
               cofhm = DBLE(cofech)
               DO k = 1, nmbflx
                  READ(datuni,*) i, j, flxnet, flxech
                  wflxhn(ro1+i, ro1+j) = DBLE(flxnet)
                  wflxhm(ro1+i, ro1+j) = DBLE(flxech)
                  wflxhm(ro1+j, ro1+i) = DBLE(flxech)
               ENDDO

! Calculation of the fields wflux, wfluxi and wfluxo actually used in the equations
! by subroutine SETWFL, CALLED from SETGEO) deferred to after the sea-level
! is known (so that reservoir volumes can be calculated and residence/flushing
! times deduced.

!     /OTHER/  ***  RESIDENCE TIME OF CO2 IN THE ATMOSPHERE
!
               READ(datuni,*)TAU
!
!              ***  AMOUNT OF CO2 IN THE ATMOSPHERE (10^18 MOLES/PAL)
!
               READ(datuni,*)QTECO2(1)
!
!              ***  FRACTIONATION COEFFICIENTS FOR C13 (IN 0/00)
!
               READ(datuni,*)FRC13A
               READ(datuni,*)FRC13B
!
!              ***  CONSTANT CaCO3 INPUT (IN 10^18 MOLES OF Ca++)
!
               READ(datuni,*)bicinp
!
!              ***  CALCITE DISSOLUTION CONSTANTS
!
               READ(datuni,*)neroca
               READ(datuni,*)keroca
!
!              ***  ARAGONITE FRACTION IN THE TOTAL CARBONATE RAIN

               READ(datuni,*)frcara
               IF ((frcara < 0D0).OR.(frcara > 1D0)) THEN
                  WRITE(erruni,*) ' frcara is below 0 or exceeds 1.' 
                  WRITE(erruni,*) ' This is not valid. Aborting!'
                  CALL ABORT()
               ENDIF

                                    ! Reset default frcara value
               CALL ARAG_FRACTION_DFLT_RESET(frcara)
       

!              *** Optional constant bicarbonate weathering fluxes:
!              Cst. Carbonate Rock weathering,
!                   Coral Reef weathering
!                   Silicate Rock weathering

               ! Ct H2CO3 CRk Weath    
               READ(datuni,*)carwbt, carwb1, carwb0   ! Tot HCO3 prod. (10^18 mol HCO3/an); t_on; t_off

               ! Ct H2SO4 CRk Weath
               READ(datuni,*)carwst, carws1, carws0   ! Tot HCO3 prod. (10^18 mol HCO3/an); t_on; t_off

               ! Constant corali
               READ(datuni,*)crlitt, crlit1, crlit0   ! Tot CaCO3 dissol. (10^18 mol CaCO3/an); t_on; t_off 

               ! Const SiRock Weath
               READ(datuni,*)sirwtt, sirwt1, sirwt0   ! Tot HCO3 prod. (10^18 mol HCO3/an); t_on; t_off

               ! Const Cor-Reef acc
               READ(datuni,*)corlo2, corlo4, corlt1, corlt0   ! Atl; Pac (10^18 mol CaCO3/an); t_on; t_off

               ! Const Shlf&Bnk acc
               READ(datuni,*)chbno2, chbno4, chbnt1, chbnt0   ! Atl; Pac (10^18 mol CaCO3/an); t_on; t_off

               ! T/S control
               READ(datuni,*)vatst1, vatst0   ! 

               ! Sediment recording
               READ(datuni,*)rcsdt1, rcsdt0   ! t_on; t_off (T/S variable if t_on<temps<t_off)

               ! Sediment interactive
               READ(datuni,*)iasdt1, iasdt0   ! t_on; t_off (sed. recorded while t_on<temps<t_off)

               ! Cont. biosph. forcing
               READ(datuni,*)bioft1, bioft0   ! t_on; t_off (forcing active t_on<temps<t_off)

               IF(resusd /= 0) THEN
               WRITE(resuni,*)
               WRITE(resuni,*)
               WRITE(resuni,'("[Control options]:")')
               WRITE(resuni,70)"Constant H2CO3 carb rock weath",
     &                    carwbt, carwb1, carwb0
               WRITE(resuni,70)"Constant H2SO4 carb rock weath",
     &                    carwst, carws1, carws0
               WRITE(resuni,70)"Constant coral-reef weathering",
     &                    crlitt, crlit1, crlit0
               WRITE(resuni,70)"Constant silic rock weathering",
     &                    sirwtt, sirwt1, sirwt0
               WRITE(resuni,70)"Constant TEATL coral-reef acc.",
     &                    corlo2, corlt1, corlt0
               WRITE(resuni,70)"Constant TEI-P coral-reef acc.",
     &                    corlo4, corlt1, corlt0
               WRITE(resuni,70)"Constant TEATL bank&shelf acc.",
     &                    chbno2, chbnt1, chbnt0
               WRITE(resuni,70)"Constant TEI-P bank&shelf acc.",
     &                    chbno4, chbnt1, chbnt0
               WRITE(resuni,71)"Temperature/Salinity variable ",
     &                    vatst1, vatst0
               WRITE(resuni,71)"Sediment exch. fluxes recorded",
     &                    rcsdt1, rcsdt0
               WRITE(resuni,71)"Sediment exch. fluxes calcul. ",
     &                    iasdt1, iasdt0
               WRITE(resuni,71)"Contin. biosph. forcing active",
     &                    bioft1, bioft0
   70          FORMAT(1X,A30, " at ", E10.4, 
     &                " from t = ", F8.0, " to t = ", F8.0)
   71          FORMAT(1X,A30, 
     &                " from t = ", F8.0, " to t = ", F8.0)
               ENDIF


      CLOSE(UNIT=datuni)


! ------------------------------------
! Hypsometric data: from a NetCDF file
! ------------------------------------

      IF(resusd /= 0) THEN
        WRITE(resuni,*)
        WRITE(resuni,*)
        WRITE(resuni,'("[Depthprofiles info]:")')
        WRITE(resuni,'(A)') ' Reading in geometric data from'
     &                 // ' file "' // TRIM(elvfil) // '"'
      ENDIF

                                    ! Open NetCDF File
      status = NF_OPEN(elvfil, NF_NOWRITE, nc_id)
      IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)


      IF(resusd /= 0) THEN

        status = NF_GET_ATT_TEXT(nc_id, NF_GLOBAL, 'title', global_att)
        IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

        status = NF_INQ_ATTLEN(nc_id, NF_GLOBAL, 'title', ll)
        IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

        WRITE(resuni,'(" Title from file: ",A)') global_att(1:ll)


        status = NF_GET_ATT_TEXT(nc_id, NF_GLOBAL, 'variant',global_att)
        IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

        status = NF_INQ_ATTLEN(nc_id, NF_GLOBAL, 'variant', ll)
        IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

        WRITE(resuni,'(" Variant: ",A)') global_att(1:ll)

      ENDIF


      IF(resusd /= 0) THEN
      ENDIF


                                    ! Inquire ID of dimension 'dpi'
                                    !    (i.e.: Depth Profile IDs)
      status = NF_INQ_DIMID(nc_id, 'dpi', dpi_dimid)
      IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

                                    ! Inquire length for dimension 'dpi'
      status = NF_INQ_DIMLEN(nc_id, dpi_dimid, dpi_dimlen)
      IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

                                    ! and carry out consistency check
      IF(dpi_dimlen /= nro1) THEN
        WRITE(erruni,*) 'The ELV file,', TRIM(elvfil),
     &                  ' does not have the right number of profiles:'
        WRITE(erruni,*) 'Should be ', nro1
        WRITE(erruni,*) 'Is ', dpi_dimlen
        STOP       
      ENDIF

                                    ! Inquire ID of dimension 'dep' (Depth)
      status = NF_INQ_DIMID(nc_id, 'dep', dep_dimid)
      IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

                                    ! Inquire length for dimension 'dep'
      status = NF_INQ_DIMLEN(nc_id, dep_dimid, dep_dimlen)
      IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

                                    ! and carry out consistency check
      IF(dep_dimlen /= n_sflnod) THEN
        WRITE(erruni,*) 'The profiles in the ELV file ', TRIM(elvfil),
     &                  ' do not have the right number of nodes:'
        WRITE(erruni,*) 'Should be ', n_sflnod
        WRITE(erruni,*) 'Is ', dep_dimlen
        STOP       
      ENDIF


                                    ! Inquire ID for variable 'dpi'
      status = NF_INQ_VARID(nc_id, 'dpi', dpi_varid)
      IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)


                                    ! Inquire ID for variable 'depth'
      status = NF_INQ_VARID(nc_id, 'depth', depth_varid)
      IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

      status = NF_GET_ATT_INT(nc_id, depth_varid,
     &                        'reference_sealevel_index', i_refsealevel)
      IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)


                                    ! and carry out consistency check
      IF(i_refsealevel /= 1) THEN
        WRITE(erruni,*) 'The reference_sealevel_index of the ''depth'''
     &               // ' variable'
        WRITE(erruni,*) 'in the ELV file ', TRIM(elvfil),
     &                  ' is not correct:'
        WRITE(erruni,*) 'Should be ', 1
        WRITE(erruni,*) 'Is ', i_refsealevel
        STOP       
      ENDIF



                                    ! Inquire ID for variable 'sfa'
      status = NF_INQ_VARID(nc_id, 'sfa', sfa_varid)
      IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

                                    ! Inquire dimension IDs for variable 'sfa'
      status = NF_INQ_VARDIMID(nc_id, sfa_varid, dimid(1:2))
      IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

                                    ! and carry out consistency check
      IF((dimid(1) /= dep_dimid) .OR.
     &   (dimid(2) /= dpi_dimid) ) THEN
        WRITE(erruni,*) 'The ''sfa'' variable in the ELV file ',
     &                  TRIM(elvfil),
     &                  'does not have the right dimensions:'
        WRITE(erruni,*) 'Should be ', dep_dimid, dpi_dimid
        WRITE(erruni,*) 'Is ', dimid(1:2)
        STOP       
      ENDIF

                                    ! Inquire ID for variable 'vol'
      status = NF_INQ_VARID(nc_id, 'vol', vol_varid)
      IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)
                                    ! Inquire dimension IDs for variable 'vol'
      status = NF_INQ_VARDIMID(nc_id, vol_varid, dimid(1:2))
      IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

                                    ! and carry out consistency check
      IF((dimid(1) /= dep_dimid) .OR.
     &   (dimid(2) /= dpi_dimid) ) THEN
        WRITE(erruni,*) 'The ''vol'' variable in the ELV file ',
     &                  TRIM(elvfil),
     &                  ' does not have the right dimensions:'
        WRITE(erruni,*) 'Should be ', dep_dimid, dpi_dimid
        WRITE(erruni,*) 'Is ', dimid(1:2)
        STOP       
      ENDIF

                                    ! Inquire ID for variable 'maxdep'
      status = NF_INQ_VARID(nc_id, 'maxdep', maxdep_varid)
      IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)
                                    ! Inquire dimension IDs for variable 'maxdep'
      status = NF_INQ_VARDIMID(nc_id, maxdep_varid, dimid(1))
      IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

                                    ! and carry out consistency check
      IF(dimid(1) /= dpi_dimid) THEN
        WRITE(erruni,*) 'The ''maxdep'' variable in the ELV file ',
     &                  TRIM(elvfil),
     &                  ' does not have the right dimensions:'
        WRITE(erruni,*) 'Should be ', dpi_dimid
        WRITE(erruni,*) 'Is ', dimid(1)
        STOP       
      ENDIF

                                    ! Reading the variables

      status = NF_GET_VAR_DOUBLE(nc_id, sfa_varid, hypsar(:,:))
      IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

      status = NF_GET_VAR_DOUBLE(nc_id, vol_varid, hypsvl(:,:))
      IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

      status = NF_GET_VAR_INT(nc_id, maxdep_varid, maxdep(:))
      IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

      maxdep(:) = maxdep(:) - i_refsealevel + i_sflnod_top
        
                                    ! Done with reading geometry/hyspometry description
                                    ! Close the file
      status = NF_CLOSE(nc_id)
      IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

 
!      WRITE(dbguni,*) 'ELVFIL'
!      WRITE(dbguni,*) hypsar
!      WRITE(dbguni,*) hypsvl
!      WRITE(dbguni,*) maxdep


      DO j = i_hypspr_1, i_hypspr_n
         hypsar(maxdep(j):bottom-1,j)=hypsar(bottom,j)
         hypsvl(maxdep(j):bottom-1,j)=hypsvl(bottom,j)
      ENDDO
      sflelt_sfcarea(:,:) = hypsar(i_sflnod_top+1:i_sflnod_bot  ,:)
     >                     -hypsar(i_sflnod_top  :i_sflnod_bot-1,:)

      DO k = i_sflnod_top, i_sflnod_bot
         hyps_z(k,:) = DBLE(k)*resol
      ENDDO



! ---------
! Sea-level
! ---------
      IF(resusd /= 0) THEN
        WRITE(resuni,*)
        WRITE(resuni,*)
        WRITE(resuni,'("[' // sealevel_name // ' info]: ")')
      ENDIF

      IF(slvusd /= 0) THEN
        CALL INITIA_DATASET_FROM_TXTFILE
     &  (slvfil, sealevel_name, n_sealevel,
     &   sealevel)
      ELSE
        IF(resusd /= 0) WRITE(resuni,'(A)')
     &                  ' Using present-day level (default).'

        IF(ASSOCIATED(sealevel)) THEN
          DEALLOCATE(sealevel)
          NULLIFY(sealevel)
        ENDIF

        ALLOCATE(sealevel(1, 2))

        sealevel(1,1) = 0D0
        sealevel(1,2) = 0D0

        n_sealevel = 1

      ENDIF


! --------------------
! Carbonate Rain Ratio
! --------------------

! Call CARB_RAINRATIO_FROMFILE in MOD_MBM_BIOCARBONATES to initialise rhocac now.

      CALL CARB_RAINRATIO_FROMFILE(temps, rhocac)


! --------------------------------
! Aragonite Fraction in Carbonates
! --------------------------------

! Call ARAG_FRACTION in MOD_MBM_BIOCARBONATES to initialise frcara now.

      CALL ARAG_FRACTION(temps, frcara)


! --------------------------------
! Temperatures: from a NetCDF file
! --------------------------------

      IF(resusd /= 0) THEN
        WRITE(resuni,*)
        WRITE(resuni,*)
        WRITE(resuni,'("[Temperature info]:")')
      ENDIF

      IF(n_temp /= 0) THEN
        IF(ALLOCATED(temp_time)) DEALLOCATE(temp_time)
        IF(ALLOCATED(temp_val)) DEALLOCATE(temp_val)
      ENDIF


      IF (tprusd == 1) THEN         ! If we use temperature evolutions
                                    ! from a file then read them in now
        IF(resusd /= 0) THEN
          WRITE(resuni,'(A)') ' Reading in temperature data from'
     &                 // ' file "' // TRIM(tprfil) // '"'
        ENDIF

                                    ! Open NetCDF File
        status = NF_OPEN(tprfil, NF_NOWRITE, nc_id)
        IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)


                                    ! Dimension inquiries:
                                    ! -- of dimension 'ori'
                                    !    (i.e.: Ocean REServoir IDs)
        status = NF_INQ_DIMID(nc_id, 'ori', oresid_dimid)
        IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

                                    ! -- of dimension 'age' (dating)
                                    !    This could be changed to allow for
                                    !    arbitrary dating (test for a set
                                    !    of names and units). Currently, we
                                    !    assume that dates are given by ages
                                    !    before 1950.
        status = NF_INQ_DIMID(nc_id, 'age', dating_dimid)
        IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

                                    ! Inquire length for dimension 'ori'
        status = NF_INQ_DIMLEN(nc_id, oresid_dimid, oresid_dimlen)
        IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

                                    ! and carry out consistency check
        IF(oresid_dimlen /= (nro1+nro2+nro3)) THEN
          WRITE(erruni,*) 'The TPR file ', TRIM(tprfil),
     &       ' does not have the right number of reservoirs:'
          WRITE(erruni,*) 'Should be ', nro1+nro2+nro3
          WRITE(erruni,*) 'Is ', oresid_dimlen
          STOP       
        ENDIF

                                    ! Inquire length for dimension 'age'
        status = NF_INQ_DIMLEN(nc_id, dating_dimid, dating_dimlen)
        IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)


                                    ! Variables' inquiries
                                    ! -- for variable 'age'
        status = NF_INQ_VARID(nc_id, 'age', dating_varid)
        IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

                                    ! -- for variable 'temp'
        status = NF_INQ_VARID(nc_id, 'temp', temp_varid)
        IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

                                    ! Inquire dimension IDs
                                    ! -- for variable 'temp'
        status = NF_INQ_VARDIMID(nc_id, temp_varid, temp_dimid(1:2))
        IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

                                    !    carry out consistency check
        IF((temp_dimid(1) /= oresid_dimid) .OR.
     &     (temp_dimid(2) /= dating_dimid) ) THEN
          WRITE(erruni,*) 'The TPR file ', TRIM(tprfil),
     &       ' does not have the right temp variable dimensions:'
          WRITE(erruni,*) 'Should be ', oresid_dimid, dating_dimid
          WRITE(erruni,*) 'Is ', temp_dimid(1:2)
          STOP       
        ENDIF


                                    ! Reading the variables

        IF(dating_dimlen == 1) THEN
           n_temp = 1               ! If there are only temperature
                                    ! data for one single instant in time,
                                    ! then, we use that temperature
                                    ! distribution throughout the whole
                                    ! simulation without changes.
          ALLOCATE(temp_val(ro1+1:ro3+nro3, 1))
          status = NF_GET_VAR_DOUBLE(nc_id, temp_varid,
     &                               temp_val(:,1))
          IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)
        ELSE
                                    ! Allocate space for arrays to
                                    ! store the original data.
          ALLOCATE(temp_dates(dating_dimlen))
          ALLOCATE(temp_valorig(ro1+1:ro3+nro3, dating_dimlen))

                                    ! Read in the original data.
          status = NF_GET_VAR_DOUBLE(nc_id, dating_varid,
     &                               temp_dates(:))
          IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

! get replicate_period attribute !!

          status = NF_GET_VAR_DOUBLE(nc_id, temp_varid,
     &                               temp_valorig(:,:))
          IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)


                                    ! Check age extent:
          IF(temp_dates(dating_dimlen) > temp_dates(1)) THEN
                                    !  -- if ages are in increasing order
            temp_tinf = 1950D0 - temp_dates(dating_dimlen)
            temp_tmax = 1950D0 - temp_dates(1)
            i1        = dating_dimlen
            i2        = 1
            is        = -1
          ELSE
                                    !  -- if ages are in decreasing order.
            temp_tinf = 1950D0 - temp_dates(1)
            temp_tmax = 1950D0 - temp_dates(dating_dimlen)
            i1        = 1
            i2        = dating_dimlen
            is        = 1
          ENDIF                                

                                    ! How many replicates do we need:
                                    !  -- before temp_tinf ?
          IF(temp_tinf > tinf) THEN
            n_replicbfr = INT((temp_tinf - tinf)/period)
            IF((temp_tinf - tinf) > DBLE(n_replicbfr)*period) THEN
              n_replicbfr = n_replicbfr + 1
            ENDIF
          ELSE
            n_replicbfr = 0
          ENDIF
            
                                    !  -- after temp_tmax ?
          IF(tmax > temp_tmax) THEN
            n_replicaft = INT((tmax - temp_tmax)/period)
            IF((tmax - temp_tmax) > DBLE(n_replicaft)*period) THEN
              n_replicaft = n_replicaft + 1
            ENDIF
          ELSE
            n_replicaft = 0
          ENDIF

          IF(resusd /= 0) THEN
            IF((n_replicbfr > 0) .OR. (n_replicaft > 0)) THEN
            WRITE(resuni,'(" Reservoir temperatures extend only from")')
            WRITE(resuni,'(F16.4," to ",F16.4)') temp_tinf, temp_tmax
            WRITE(resuni,'(" Extending the temporal range by assuming",
     &                     " a periodic evolution.")')
            WRITE(resuni,'(" Adopting a period of ", F16.4, " years.")')
     &        period
            ENDIF
          ENDIF

                                    ! Total space required for time dimension
          n_temp = (n_replicbfr + n_replicaft + 1) * dating_dimlen

                                    ! Allocate space:
          ALLOCATE(temp_time(n_temp))
          ALLOCATE(temp_val(ro1+1:ro3+nro3, n_temp))


          i_first = 1               ! Copy replicates into temp_time
          i_last  = dating_dimlen   ! and temp_val

          DO i = n_replicbfr, 1, -1
            temp_time(i_first:i_last) = 1950D0 - temp_dates(i1:i2:is)
     &                                  - DBLE(i)*period
            temp_val(:,i_first:i_last) = temp_valorig(:, i1:i2:is)
            i_first = i_last + 1
            i_last  = i_last + dating_dimlen
          ENDDO

          temp_time(i_first:i_last) = 1950D0 - temp_dates(i1:i2:is)
          temp_val(:,i_first:i_last) = temp_valorig(:, i1:i2:is)

          DO i = 1, n_replicaft
            i_first = i_last + 1
            i_last  = i_last + dating_dimlen
            temp_time(i_first:i_last) = 1950D0 - temp_dates(i1:i2:is)
     &                                  + DBLE(i)*period
            temp_val(:,i_first:i_last) = temp_valorig(:, i1:i2:is)
          ENDDO

          WRITE(dbguni,*) 
          WRITE(dbguni,*) '[INITIA]:'
          WRITE(dbguni,*) 'Temperatures'
          WRITE(dbguni,*) dating_dimlen, n_temp
!          WRITE(dbguni,*) temp_dates
!          WRITE(dbguni,*) temp_valorig(ro3+1,:)
!          WRITE(dbguni,*) temp_time
!          WRITE(dbguni,*) temp_val(ro3+1,:)
          WRITE(dbguni,*) temp_tinf, temp_tmax
          WRITE(dbguni,'(3(" ", i0))') i1, i2, is
          CALL FLUSH(dbguni)

                                    ! Deallocate space, as it is not
                                    ! required anymore
          DEALLOCATE(temp_dates)
          DEALLOCATE(temp_valorig)

        ENDIF

                                    ! Close the file
        status = NF_CLOSE(nc_id)
        IF (status /= NF_NOERR) CALL HANDLE_ERRORS(status)

                                    ! Done with reading temperuture distribution
                                    ! and evolution from NetCDF file.


      ELSE                          ! If no temperature file is specified
                                    ! adopte default distribution.
        n_temp = 1
        ALLOCATE(temp_time(1))
        ALLOCATE(temp_val(ro1+1:ro3+nro3, 1))
        temp_time(1) = 0D0
        temp_val(ro1+1, 1) = 275.0000D+00
        temp_val(ro1+2, 1) = 292.0000D+00
        temp_val(ro1+3, 1) = 275.0000D+00
        temp_val(ro1+4, 1) = 292.0000D+00
        temp_val(ro1+5, 1) = 275.0000D+00
        temp_val(ro2+1, 1) = 275.1880D+00
        temp_val(ro2+2, 1) = 274.4570D+00
        temp_val(ro3+1, 1) = 275.1880D+00
        temp_val(ro3+2, 1) = 273.6010D+00
        temp_val(ro3+3, 1) = 274.4570D+00
      ENDIF


! Now call **setgeo** to calculate the geometric properties
! of the different volumes a first time.

      updtts = .TRUE.
      CALL setgeo(updtts)

      IF(resusd /= 0) THEN
         WRITE(resuni,*)
         WRITE(resuni,*)
         WRITE(resuni,'("[Water fluxes and ventilation times:]")')
         WRITE(resuni,*)
     &      'Reservoir: Vol., Water in, Water out, Ventil. time'
         WRITE(resuni,*)
         DO i = ro1+1, ro3+nro3
            WRITE(resuni,'(I4, ":", 3E12.4, F8.3)')
     &         i, volum(i), wfluxi(i), wfluxo(i), volum(i)/wfluxo(i)
         ENDDO
      ENDIF


! Now that the volumes are known, calculate their total
      tvolum=0.0D+00
      DO i = ro1+1, ro3+nro3
         tvolum=tvolum+volum(i)
      ENDDO

! Convert concentrations to total contents where necessary and
! add the additional amounts of CIT, ALK and PO4 to the reservoirs.

!      tvolum=SUM(volum(ro1+1:ro3+nro3))
      addpo4 = addpo4/tvolum
      addcit = addcit/tvolum
      addalk = addalk/tvolum
      DO i = ro1+1, ro3+nro3
         vpo4(i) = (vpo4(i)+addpo4)*volum(i)
         vcit(i) = (vcit(i)+addcit)*volum(i)
         valk(i) = (valk(i)+addalk)*volum(i)
         voxy(i) =  voxy(i)*volum(i)
      ENDDO





! Read in organic data and calculate variation
      IF(cbfusd /= 0) THEN
        OPEN(UNIT=tmpuni, FILE=cbffil)
        READ(tmpuni,*) form08
        READ(tmpuni,form08)tcbio(1), stcbio(1), frcbio(1), stpbio(1)
        tcbio(1)=tcbio(1)+120000.D+00
        DO i = 2, ncbio
           READ(tmpuni,form08)tcbio(i), stcbio(i), frcbio(i), stpbio(i)
           ! Convert [-120000,0] to [0,120000]
           tcbio(i)=tcbio(i)+120000.D+00
           ! Unit conversions:
           !   x/12.011 to convert x from Gt = 10^15 gC to 10^15 molC
           !   y/1000 to convert y from 10^15 mol to 10^18 mol 
           dcbio(i-1)=((stcbio(i)-stcbio(i-1))/
     +                 ( tcbio(i)- tcbio(i-1)))/12011.D+00
           ! Unit conversions:
           !   x/30.973762 to convert x from 10^12 gP to 10^12 molP
           !   y/1000000 to convert y from 10^12 mol to 10^18 mol 
           dpbio(i-1)=((stpbio(i)-stpbio(i-1))/
     +                 ( tcbio(i)- tcbio(i-1)))/30973762.D+00
        END DO
        cindex=1
        CLOSE(UNIT=tmpuni)
      ELSE
        DO i=1, ncbio
           tcbio(i) = (i-1)*120000.D+00/dble(ncbio-1)
        END DO
        stcbio(:) = 0.D+00
        frcbio(:) = 0.D+00
        stpbio(:) = 0.D+00
        dcbio(:)  = 0.D+00
        dpbio(:)  = 0.D+00
        cindex = 1
      ENDIF

      
      RETURN

      CONTAINS
!     ******************************************************************

!       ================================================================
        SUBROUTINE INITIA_DATASET_FROM_TXTFILE
     &     (file_name, dataset_name, n_dataset, dataset)
!       ================================================================

        IMPLICIT NONE
        CHARACTER(LEN=*), INTENT(IN)
     &    :: file_name, dataset_name
        INTEGER
     &    :: n_dataset
        DOUBLE PRECISION, DIMENSION(:,:), POINTER
     &    :: dataset

        INTEGER
     &    :: n_data_in_file
        INTEGER
     &    :: i, i1, i2, is, i_first, i_last
        INTEGER
     &    :: n_replicbfr, n_replicaft
        DOUBLE PRECISION
     &    :: file_tinf, file_tmax, file_period

        DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE
     &    :: file_dates
        DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE
     &    :: file_vals_1d

        CHARACTER(LEN=80)
     &    :: messag

        INTEGER
     &    :: i_error

        WRITE(dbguni,*) 
        WRITE(dbguni,'(A)') '[INITIA/INITIA_DATASET_FROM_TXTFILE]:'
        WRITE(dbguni,'(" Dataset name: ",A)')
     &    '"'//TRIM(dataset_name)//'"'
        WRITE(dbguni,'(" Filename: ",A)')
     &    '"'//TRIM(file_name)//'"'

        IF(ALLOCATED(file_dates)) DEALLOCATE(file_dates)
        IF(ALLOCATED(file_vals_1d)) DEALLOCATE(file_vals_1d)

        IF(resusd /= 0) THEN
          WRITE(resuni,'(A)') ' Reading in ' // dataset_name
     &                 // ' values from file "'
     &                 // TRIM(file_name) // '"'
        ENDIF
        OPEN(UNIT=tmpuni,FILE=file_name)
        READ(tmpuni,*) messag

        IF(resusd /= 0) THEN
          WRITE(resuni,'(" Info from file: ",A)') TRIM(messag)
        ENDIF

        IF(n_dataset /= 0) THEN
          IF(ASSOCIATED(dataset)) THEN
            DEALLOCATE(dataset)
            NULLIFY(dataset)
          ENDIF
        ENDIF
      
        READ(tmpuni,*) n_data_in_file, file_period

        WRITE(dbguni,'(" Nbr of entries in file: ", I0)') n_data_in_file
        WRITE(dbguni,'(" Replication period    : ", F16.4)') file_period

        IF(n_data_in_file == 1) THEN
          n_dataset = 1        ! If there is only one single
                                    ! datum, then we use that
                                    ! throughout the whole
                                    ! simulation without changes.
          ALLOCATE(dataset(1, 2))
          READ(tmpuni,*) dataset(1,:)

          IF(resusd /= 0) THEN
            WRITE(resuni,*) 'Using constant ' // dataset_name
          ENDIF
        ELSE

          ALLOCATE(file_dates(n_data_in_file))
          ALLOCATE(file_vals_1d(n_data_in_file))
          
                                    ! Read in the data
          READ(tmpuni,*) (file_dates(i), file_vals_1d(i),
     &                    i=1, n_data_in_file)

                                    ! Check age extent:
          IF(file_dates(n_data_in_file) > file_dates(1)) THEN
                                    !  -- if ages are in deincreasing order (oldest last)
            file_tinf = 1950D0 - file_dates(n_data_in_file)
            file_tmax = 1950D0 - file_dates(1)
            i1        = n_data_in_file
            i2        = 1
            is        = -1
          ELSE
                                    !  -- if ages are in increasing order (oldest first)
            file_tinf = 1950D0 - file_dates(1)
            file_tmax = 1950D0 - file_dates(n_data_in_file)
            i1        = 1
            i2        = n_data_in_file
            is        = 1
          ENDIF                                

          WRITE(dbguni,'(" Time: from ", F16.4," to ",F16.4)')
     &      file_tinf, file_tmax
          WRITE(dbguni,'(" Triplet: ", 3(" ", I0))') i1, i2, is

          IF(file_period > 0D0) THEN ! if file_period > 0, replication is allowed

                                     ! If replication period shorter than the file
                                     ! extent, there is amibuity for the replication
            IF(file_period < (file_tmax-file_tinf)) THEN
              WRITE(erruni,'(A)')
     &          'The period for replication given in the file '
     &           // TRIM(file_name)
              WRITE(erruni,'(A)')
     &          'is shorter than the time interval covered by '
     &          // 'the data in that file!'
              WRITE(erruni,'(A)')
     &          'Please correct this inconsistency. Aborting.'
              CALL ABORT()
            ENDIF
                                    ! How many replicates do we need:
                                    !  -- before file_tinf ?
            IF(file_tinf > tinf) THEN
              n_replicbfr = INT((file_tinf - tinf)/file_period)
              IF((file_tinf-tinf) > DBLE(n_replicbfr)*file_period) THEN
                n_replicbfr = n_replicbfr + 1
              ENDIF
            ELSE
              n_replicbfr = 0
            ENDIF
            
                                    !  -- after file_tmax ?
            IF(tmax > file_tmax) THEN
              n_replicaft = INT((tmax - file_tmax)/file_period)
              IF((tmax-file_tmax) > DBLE(n_replicaft)*file_period) THEN
                n_replicaft = n_replicaft + 1
              ENDIF
            ELSE
              n_replicaft = 0
            ENDIF

          ELSE                      ! if file_period <= 0, replication is not allowed
            IF((file_tinf > tinf).OR.(tmax > file_tmax)) THEN
              WRITE(erruni,'(A)')
     &          'The data in the file ' // TRIM(file_name)
              WRITE(erruni,'(A)')
     &          'cannot be extended by periodic replication!'
              WRITE(erruni,'(A)')
     &          'Please provide a file that covers the complete ' //
     &          'simulation time interval. Aborting.'
              CALL ABORT()
            ELSE
              n_replicbfr = 0
              n_replicaft = 0
            ENDIF
          ENDIF

          
          IF(resusd /= 0) THEN
            IF((n_replicbfr > 0) .OR. (n_replicaft > 0)) THEN
              WRITE(resuni,'(" ' // dataset_name 
     &                           // ' data extend only from")')
              WRITE(resuni,'(F16.4," to ",F16.4)') file_tinf, file_tmax
              WRITE(resuni,'(" Extending the temporal range by",' //
     &                     ' " assuming a periodic evolution.")')
              WRITE(resuni,'(" Adopting a period of ", F16.4,'
     &                       // '" years.")') file_period
            ENDIF
          ENDIF

                                    ! Total space required for time dimension
          n_dataset = (n_replicbfr + n_replicaft + 1)
     &                     * n_data_in_file


          WRITE(dbguni,'(" Nbr of data, nbr of replicates before, ",'
     &               //' "after):", 3(" ", I0))')
     &                    n_dataset, n_replicbfr, n_replicaft
          CALL FLUSH(dbguni)

                                    ! Allocate space:
          ALLOCATE(dataset(n_dataset, 2), STAT = i_error)
          IF(i_error /= 0) THEN
            WRITE(erruni,fmt_infolin_ia) (__LINE__-2), 
     &        'Error while trying to allocate space for the dataset'
            WRITE(erruni,'(A," read from file ",A)') TRIM(dataset_name),
     &                   '"'// TRIM(file_name) // '")'
            WRITE(erruni,'("Shape required: (/",I0,",",I0,"/)")')
     &        n_dataset, 2
            WRITE(erruni,'("Return error: ",I0)')  i_error
            WRITE(erruni,'("Aborting!")')
            CALL ABORT()
          ENDIF
            

          i_first = 1               ! Copy replicates into temp_time
          i_last  = n_data_in_file  ! and temp_val

          DO i = n_replicbfr, 1, -1
            dataset(i_first:i_last,1)
     &        = 1950D0 - file_dates(i1:i2:is) - DBLE(i)*file_period
            dataset(i_first:i_last,2) = file_vals_1d(i1:i2:is)
            i_first = i_last + 1
            i_last  = i_last + n_data_in_file
          ENDDO

          dataset(i_first:i_last,1) = 1950D0 - file_dates(i1:i2:is)
          dataset(i_first:i_last,2) = file_vals_1d(i1:i2:is)

          DO i = 1, n_replicaft
            i_first = i_last + 1
            i_last  = i_last + n_data_in_file
            dataset(i_first:i_last,1)
     &        = 1950D0 - file_dates(i1:i2:is) + DBLE(i)*file_period
            dataset(i_first:i_last,2) = file_vals_1d(i1:i2:is)
          ENDDO

         !WRITE(dbguni,*) file_dates
         !WRITE(dbguni,*) file_vals_1d(:)
         !WRITE(dbguni,*) dataset(:,1)
         !WRITE(dbguni,*) dataset(:,2)

          DEALLOCATE(file_dates)     ! Deallocate space, as it is not
          DEALLOCATE(file_vals_1d)   ! required anymore

        ENDIF

                     
        CLOSE(tmpuni)               ! Close the file      


!       ================================================================
        END SUBROUTINE INITIA_DATASET_FROM_TXTFILE
!       ================================================================

!     ******************************************************************
      END SUBROUTINE INITIA
!     ******************************************************************
