!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


!     ******************************************************************
      SUBROUTINE EQNOCE_ORGM_POWERLAW
     >           (orgm_x_nodcum, orgm_xs_oocelt,
     <            orgm_r, orgm2s)
!     ******************************************************************

! The state of this subroutine is at best hybrid.
! Large parts of it are still dependent on the 0m-100m-1000m-bottom
! vertical layout.

! Also have treated O2 and PO4 here ?  Probably best to do so.

      USE MOD_MBM_GEOMETRY
      USE MOD_MBM_BIOCTS, ONLY: corem2
      USE MOD_MBM_EQNOCE_TXT,
     &  ONLY: txt_orgm_flx, txt_orgm_sed, txt_orgm_rmn, format7


      IMPLICIT NONE



! Input variables
! ===============

      DOUBLE PRECISION, DIMENSION(i_sflnod_top:i_sflnod_bot,
     &                            i_hypspr_1  :i_hypspr_n,3)
     &                  :: orgm_x_nodcum

      DOUBLE PRECISION, DIMENSION(i_hypspr_1:i_hypspr_n,3)
     &                  :: orgm_xs_oocelt


      INTENT(IN)        :: orgm_x_nodcum, orgm_xs_oocelt



! Output variables
! ================

      DOUBLE PRECISION, DIMENSION(ro1+1:ro3+nro3,3) ::
     &   orgm_r

      DOUBLE PRECISION, DIMENSION(1:bottom,ro1+1:ro1+nro1,3) ::
     &   orgm2s

      INTENT(OUT) :: orgm_r, orgm2s


! Local variables
! ===============

      DOUBLE PRECISION, DIMENSION(3) ::
     &   orgm_xj

      DOUBLE PRECISION, 
     &   DIMENSION(i_geolev_top:i_geolev_bot, ro1+1:ro1+nro1, 3) ::
     &   orgm_rl


      DOUBLE PRECISION,  DIMENSION(i_sflnod_top:i_sflnod_bot,
     &                             i_hypspr_1  :i_hypspr_n)
     &                 :: d_odzk1
      DOUBLE PRECISION,  DIMENSION(i_hypspr_1:i_hypspr_n)
     &                 :: d200_odzk1, d1000_odzk

! - Depth horizons
!   * d0    : Sea Surface
!   * d100  : Top of intermediate water
!   * d200  : Minimum depth of "Open Ocean"
!   * d1000 : Top of Deep Sea

      DOUBLE PRECISION :: d0
      INTEGER          :: i_sflnod_d0
      DOUBLE PRECISION :: d200, d1000
      INTEGER          :: i_sflelt_d200, i_sflelt_d1000


! - Surface areas of depth horizons

      DOUBLE PRECISION, DIMENSION(ro1+1:ro1+nro1)
     &                 :: sfc200,sfc1000


! - Export flux over the ocean surface where the water
!   depth is greater than 1000m

      DOUBLE PRECISION, DIMENSION(i_hypspr_1:i_hypspr_n,3)
     &                  :: orgm_x_sfc1000


! - Organic matter degradation profile parameters
!   * odz : export d.b.s.l. (m)
!   * odk : power-law exponent (Martin exponent), derived from
!           corem2, which is equal to the fraction of organic matter
!           remineralised between 100m and 1000m b.s.l.

      DOUBLE PRECISION, DIMENSION(i_hypspr_1:i_hypspr_n), PARAMETER
     &   :: odz = 100.D0    ! meters B.S.L.
      DOUBLE PRECISION, DIMENSION(i_hypspr_1:i_hypspr_n)
     &   :: odk


! - Loop indices

      INTEGER :: i, j, il, ip, ir

! Calculations
! ============

      odk(:) = LOG10(1D0-corem2(:))

      d0 = -nivmer/resol
      i_sflnod_d0 = INT(d0) ! node above d0 level

      DO j = i_sflnod_top, i_sflnod_d0
        d_odzk1(j,:) = 0D0
      ENDDO

      DO j = i_sflnod_d0+1, i_sflnod_bot
        d_odzk1(j,:) = ((hyps_z(j,:)+nivmer)/odz(:))**(odk(:)+1D0)
      ENDDO

      d200_odzk1(:) = ( 200D0/odz(:))**(odk(:)+1D0)
      d1000_odzk(:) = (1000D0/odz(:))**odk(:)


       d200      = (200.D+00-nivmer)/resol
      i_sflelt_d200 = INT(d200)+1
      sfc200(:)  = hypsar(bottom,:)-
     &              (hypsar(i_sflelt_d200-1,:)
     &               +(d200-(i_sflelt_d200-1))
     &                *sflelt_sfcarea(i_sflelt_d200,:))

       d1000     = (1000.D+00-nivmer)/resol
      i_sflelt_d1000 = INT(d1000)+1
      sfc1000(:) = hypsar(bottom,:)-
     &              (hypsar(i_sflelt_d1000-1,:)
     &               +(d1000-(i_sflelt_d1000-1))
     &                *sflelt_sfcarea(i_sflelt_d1000,:))


! Total export flux over the 1000m+ water surface (10^18 mol/yr)
      DO ip = i_hypspr_1, i_hypspr_n
        j = i_sflelt_d1000
        orgm_x_sfc1000(ip,:) =
     &      (orgm_x_nodcum(i_sflnod_bot,ip,:)-orgm_x_nodcum(j,ip,:))
     &    + (orgm_x_nodcum(j,ip,:)-orgm_x_nodcum(j-1,ip,:))
     &     *(i_sflelt_d1000-d1000)
      ENDDO
  
! Attention: if corem == 0.9, odk = -1 and everything will fail in
! the part below. In this case, the integrals become logarithms.

      orgm_rl(:,:,:) = 0D0
      orgm2s(:,:,:) = 0D0


      DO ip = i_hypspr_1, i_hypspr_n

!       il = i_geolev_top

!       orgm2s(1:i_sflelt_d200-1,ip,:) = 0D0


        il = i_geolev_top+1
        j  = i_sflelt_d200

! Total export in 10^18 mol/yr over column j
        orgm_xj(:) =  orgm_x_nodcum(j,ip,:)-orgm_x_nodcum(j-1,ip,:)

        orgm2s(j,ip,:) =
     &      orgm_xs_oocelt(ip,:)
     &    *(sflelt_sfcarea(j,ip))*(odz(ip)/(odk(ip)+1D0))
     &    *(d_odzk1(j,ip) - d200_odzk1(ip))/resol
        orgm_rl(il,ip,:) = orgm_xj(:) - orgm2s(j,ip,:)

        DO j = i_sflelt_d200+1, i_sflelt_d1000-1
          orgm_xj(:) =  orgm_x_nodcum(j,ip,:)-orgm_x_nodcum(j-1,ip,:)
          orgm2s(j,ip,:) =  orgm_xj(:)
     &                     *(odz(ip)/(odk(ip)+1D0))
     &                     *(d_odzk1(j,ip)-d_odzk1(j-1,ip))/resol
          orgm_rl(il,ip,:) = orgm_rl(il,ip,:)
     &                     + orgm_xj(:) - orgm2s(j,ip,:)
        ENDDO

        j = i_sflelt_d1000
        orgm_xj(:) =  orgm_x_nodcum(j,ip,:)-orgm_x_nodcum(j-1,ip,:)
        orgm2s(j,ip,:) =  orgm_xj(:)
     &                     *(odz(ip)/(odk(ip)+1D0))
     &                     *(d_odzk1(j,ip)-d_odzk1(j-1,ip))/resol

! Add the contribution from [i_sflelt_d1000-1, d1000]
! to orgm_rl(level 2), and then straight away all the stuff that
! remineralises above the 1000m depth horizon.
        orgm_rl(il,ip,:) = orgm_rl(il,ip,:)
     &               + (orgm_xj(:) - orgm2s(j,ip,:))
     &                *(d1000-(i_sflelt_d1000-1))
     &               + orgm_x_sfc1000(ip,:)*(1D0 - d1000_odzk(ip))


        il = i_geolev_top+2 ! = i_geolev_bot

        DO j = i_sflelt_d1000+1, maxdep(ip)

          orgm_xj(:) =  orgm_x_nodcum(j,ip,:)-orgm_x_nodcum(j-1,ip,:)
          orgm2s(j,ip,:) =  orgm_xj(:)
     &                     *(odz(ip)/(odk(ip)+1D0))
     &                     *(d_odzk1(j,ip)-d_odzk1(j-1,ip))/resol

        ENDDO

        orgm2s(maxdep(ip)+1:bottom,ip,:) = 0D0

        DO i = 1, 3 ! for each isotope
          orgm_rl(i_geolev_bot,ip,i)
     &      = orgm_x_nodcum(i_sflnod_bot,ip,i)
     &      - SUM(orgm_rl(i_geolev_top:i_geolev_bot-1,ip,i))
     &      - SUM(orgm2s(:,ip,i))
        ENDDO
        
        
      ENDDO


      orgm_r(:,:) = 0D0

      DO ip = i_hypspr_1, i_hypspr_n
        DO il = i_geolev_top, i_geolev_bot
          ir = i_box4geo(il,ip)
          orgm_r(ir,:) = orgm_r(ir,:) + orgm_rl(il,ip,:)
        ENDDO
      ENDDO


! Write out logging information into MOD_MBM_EQNOCE_TXT
! for retrieval by eqnoce if necessary

      WRITE(txt_orgm_flx(i_geohor_slv+1),format7)
     >  'Organic Flux at  100m:',
     >  orgm_x_nodcum(i_sflnod_bot,:,1)*1.0D+06,
     >  SUM(orgm_x_nodcum(i_sflnod_bot,:,1)) * 1.0D+06

      WRITE(txt_orgm_flx(i_geohor_dps  ),format7)
     >  'Organic Flux at 1000m:',
     >  orgm_x_sfc1000(:,1)*((1D3/odz(:))**odk(:)) *1.0D+06,
     >  SUM(orgm_x_sfc1000(:,1)*((1D3/odz(:))**odk(:)))*1.0D+06

      WRITE(txt_orgm_sed(i_geolev_top+1),format7)
     >  'Org Sedim  100m-1000m:',
     >  SUM(orgm2s(1:i_sflelt_d1000,:,1),1) *1.0D+06,
     >  SUM(orgm2s(1:i_sflelt_d1000,:,1))*1.0D+06

      WRITE(txt_orgm_sed(i_geolev_bot  ),format7)
     >  'Org Sedim 1000m-8000m:',
     >  SUM(orgm2s(i_sflelt_d1000+1:bottom,:,1),1) *1.0D+06,
     >  SUM(orgm2s(i_sflelt_d1000+1:bottom,:,1))*1.0D+06

      WRITE(txt_orgm_rmn(i_geolev_top+1),format7)
     >  'Org Remin  100m-1000m:',
     >  orgm_rl(i_geolev_top+1,:,1)*1.0D+06,
     >  SUM(orgm_rl(i_geolev_top+1,:,1)) * 1.0D+06

      WRITE(txt_orgm_rmn(i_geolev_bot  ),format7)
     >  'Org Remin 1000m-8000m:',
     >  orgm_rl(i_geolev_bot,:,1)*1.0D+06,
     >  SUM(orgm_rl(i_geolev_bot,:,1)) * 1.0D+06

 7    FORMAT(A25,5F8.2,2X,F8.2)

      RETURN
      END SUBROUTINE EQNOCE_ORGM_POWERLAW
