!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


!     ******************************************************************
      SUBROUTINE EQNOCE_ORGM_DISTRIBUTION
     >           (orgm_x, i_oocelt, frac_oocelt,
     <            orgm_x_nodcum, orgm_xs_oocelt)
!     ******************************************************************

! This subroutine distributes the total organic matter export production
! <orgm_x> over the export surface area, using a distribution function
! <orgm_x_xi> which may be prescribed externally via the subroutine
! EQNOCE_ORGM_X_XI. That distribution function needs to be approximately
! normalised such that
! Sum_j orgm_x_xi(j)*ExportArea(j) = Sum_j ExportArea(j)
! An exact renormalisation is performed at the end of this subroutine,
! if necessary.
! This subroutine returns the corresponding cumulative export
! production of the sea-surface nodes, numbered from shallowest to
! greatest depths.
! It also returns the export flux density over the sea-surface element
! that is above the sea-floor element that intersects the "Open-ocean
! horizon". This is the only flux density that is explicitely required
! at later stages in EQNOCE, and, if the fraction of that sea-surface
! element that is actually providing export production becomes small,
! the specific flux may possibly not be accurately recalculated from
! the total export flux from that element.

      USE MOD_MBM_GEOMETRY


      IMPLICIT NONE


! Input variables
! ===============

      DOUBLE PRECISION, DIMENSION(i_hypspr_1:i_hypspr_n,3)
     &                  :: orgm_x

      INTEGER           :: i_oocelt

      DOUBLE PRECISION  :: frac_oocelt


      INTENT(IN)        :: orgm_x, i_oocelt, frac_oocelt


! Output variables
! ================

      DOUBLE PRECISION, DIMENSION(i_sflnod_top:i_sflnod_bot,
     &                            i_hypspr_1  :i_hypspr_n,3)
     &                  :: orgm_x_nodcum

      DOUBLE PRECISION, DIMENSION(i_hypspr_1:i_hypspr_n,3)
     &                  :: orgm_xs_oocelt

      INTENT(OUT)       :: orgm_x_nodcum, orgm_xs_oocelt


! Local variables
! ===============

! Open-ocean Sea-Surface area
      DOUBLE PRECISION, DIMENSION(i_hypspr_1:i_hypspr_n)
     &                  :: sfc_ooc

      DOUBLE PRECISION, DIMENSION(i_hypspr_1:i_hypspr_n,3)
     &                  :: sfcsum

! Weights for <orgm_x> distribution
      DOUBLE PRECISION, DIMENSION(i_sflelt_1:i_sflelt_n,
     &                            i_hypspr_1:i_hypspr_n)
     &                  :: orgm_x_xi=1D0

! Normalisation of weighting factors
      DOUBLE PRECISION  :: xi_norm

! Dummy loop indices
      INTEGER           :: i, ip, k


! Calculations
! ============


      sfc_ooc(:)  = hypsar(i_sflnod_bot,:)- hypsar(i_oocelt,:)
     &               +frac_oocelt*sflelt_sfcarea(i_oocelt,:)

! Insert here weight calculation if non-homogeneous weighting is
! required!  If delicate balancing is required, the returned
! <orgm_x_xi> should be normalised (area weighted) as accurately
! as possible. The finally calculated <orgm_x_nodcum> will be
! renormalised though if necessary (currently not activated)!
!
!      CALL EQNOCE_ORGM_X_XI(orgm_x_xi)


! Initialise the nodes above <i_oocelt>, if any, to 0.
! These values are normally not be used, but to avoid any
! trouble in case they would be, it is better to be sure there
! are predictible and realistic values, ie., zeros.

      IF(i_oocelt > i_sflnod_top) THEN
         orgm_x_nodcum(i_sflnod_top:i_oocelt-1,:,:) = 0D0
      ENDIF


      DO i = 1, 3

        DO ip = i_hypspr_1, i_hypspr_n

          k = i_oocelt
          orgm_x_nodcum(k,ip,i)
     &     =  (frac_oocelt*sflelt_sfcarea(k,ip)/sfc_ooc(ip))
     &       *orgm_x_xi(k,ip)*orgm_x(ip,i)
          sfcsum(ip,i) = frac_oocelt*sflelt_sfcarea(k,ip)
     &                              *orgm_x_xi(k,ip)

          DO k = i_oocelt+1, maxdep(ip)
            orgm_x_nodcum(k,ip,i)
     &       = orgm_x_nodcum(k-1,ip,i)
     &         + (sflelt_sfcarea(k,ip)/sfc_ooc(ip))
     &           *orgm_x_xi(k,ip)*orgm_x(ip,i)
          sfcsum(ip,i) = sfcsum(ip,i)+sflelt_sfcarea(k,ip)
     &                               *orgm_x_xi(k,ip)
          ENDDO

          IF(maxdep(ip).LT.i_sflnod_bot) THEN
            orgm_x_nodcum(maxdep(ip)+1:i_sflnod_bot,ip,i)
     &      = orgm_x_nodcum(maxdep(ip),ip,i)
          ENDIF

          orgm_xs_oocelt(ip,i)
     &     =  (1D0/sfc_ooc(ip))*orgm_x_xi(i_oocelt,ip)*orgm_x(ip,i)


! Fine adjustment to make sure that the total flux is exactly
! conform with the <orgm_x> provided.

          IF(i == 1) THEN
            IF(orgm_x_nodcum(maxdep(ip),ip,i) /= orgm_x(ip,i)) THEN
              xi_norm = orgm_x(ip,i)/orgm_x_nodcum(maxdep(ip),ip,i)
            ELSE
              xi_norm = 1D0
            ENDIF
          ENDIF

!          IF(xi_norm /= 1D0) THEN
!            orgm_x_nodcum(i_sflnod_top:maxdep(ip)-1,ip,i)
!     &       = orgm_x_nodcum(i_sflnod_top:maxdep(ip)-1,ip,i)*xi_norm
!            orgm_x_nodcum(maxdep(ip):i_sflnod_bot,ip,i) = orgm_x(ip,i)
!
!            orgm_xs_oocelt(ip,i) = orgm_xs_oocelt(ip,i)*xi_norm
!          ENDIF

        ENDDO

      ENDDO

      RETURN
      END SUBROUTINE EQNOCE_ORGM_DISTRIBUTION
