!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


/***********************************************************************
   Development notes:
    - include extra option parameter to allow for different
      possibilities in one single subroutine;
    - different cases by SELECT CASE switch;
    - cases selectable by codes (transfer MBMRHOCAC_XXX defines from
      below into a header file);
    - include ampfac as read-in parameter into a config file (namelist?)
      instead of having it as a hardcoded parameter here.
    - include a logging possibility (similar to the chemical reactions
      in medusa) to have printed out the adopted values.
***********************************************************************/

/* Select one option among the four available below.
   Only the first one defined will actually be considered! */
#undef MBMRHOCAC_HOMOGENEOUS /* Default */
#undef MBMRHOCAC_SHALLOW_HAT
#define MBMRHOCAC_MIDDLE_HAT
#undef MBMRHOCAC_DEEP_HAT
/* End of selectable options */
/* Option validity check below */
#define MBMRHOCAC_DEFAULT
#ifdef MBMRHOCAC_HOMOGENEOUS
#undef MBMRHOCAC_SHALLOW_HAT
#undef MBMRHOCAC_MIDDLE_HAT
#undef MBMRHOCAC_DEEP_HAT
#undef MBMRHOCAC_HATS
#undef MBMRHOCAC_DEFAULT
#endif
#ifdef MBMRHOCAC_SHALLOW_HAT
#define MBMRHOCAC_HATS
#undef MBMRHOCAC_HOMOGENEOUS
#undef MBMRHOCAC_MIDDLE_HAT
#undef MBMRHOCAC_DEEP_HAT
#undef MBMRHOCAC_DEFAULT
#endif
#ifdef MBMRHOCAC_MIDDLE_HAT
#define MBMRHOCAC_HATS
#undef MBMRHOCAC_HOMOGENEOUS
#undef MBMRHOCAC_SHALLOW_HAT
#undef MBMRHOCAC_DEEP_HAT
#undef MBMRHOCAC_DEFAULT
#endif
#ifdef MBMRHOCAC_DEEP_HAT
#define MBMRHOCAC_HATS
#undef MBMRHOCAC_HOMOGENEOUS
#undef MBMRHOCAC_SHALLOW_HAT
#undef MBMRHOCAC_MIDDLE_HAT
#undef MBMRHOCAC_DEFAULT
#endif
#ifdef MBMRHOCAC_DEFAULT
#define MBMRHOCAC_HOMOGENEOUS
#undef MBMRHOCAC_HATS
#endif
!     ******************************************************************
      SUBROUTINE EQNOCE_CARB_DISTRIBUTION
     >           (time, orgm_x_nodcum, orgm_xs_oocelt,
     >            i_oocelt, frac_oocelt,
     <            carb_x_nodcum, carb_xs_oocelt)
!     ******************************************************************

!     This is a parameterised version of the EQNOCE_CARB_DISTRIBUTION
!     used before. The <ampfac> parameter allows to choose different
!     combinations 2 times the rain ratio reduction for 1/2 of the
!     production. Here it is now ampfac times the rain ratio reduction
!     for 1/ampfac of the production.
!     Permissible values for ampfac depend on the rain ratio reduction
!     scenario (see CARB_RAINRATIO(rhocac_x, rhocac_bas, rhocac_ext))
!     used. For a standard reduction of 40%, ampfac should be lower
!     than 1/40%=2.5. A value of 2.5 would lead to complete shut down
!     over about 40% of the sea-floor, which is, certainly one
!     special possibility to bring about a 40% reduction!. A value
!     greater than this would lead to negative rain ratios at some
!     times.
!
!     ampfac obviously has to be larger than 1, else the reduction
!     would become an increase, and would have to be spread over a
!     larger surface area than available.
!
! *** No test is currently made to catch these types of anomalies ***

#ifdef MBMRHOCAC_HOMOGENEOUS
      USE mod_mbm_geometry, ONLY : i_sflnod_top, i_sflnod_bot,
     &                             i_hypspr_1,   i_hypspr_n
#endif
#ifdef MBMRHOCAC_HATS
      USE mod_mbm_geometry, ONLY : i_sflnod_top, i_sflnod_bot,
     &                             i_sflelt_1,   i_sflelt_n,
     &                             i_hypspr_1,   i_hypspr_n,
     &                             maxdep
#endif
      USE mod_mbm_biocarbonates, ONLY :
     &   CARB_RAINRATIO => CARB_RAINRATIO_FROMFILE,
     &   CARB_RAINRATIO_MINMAX_GET



      IMPLICIT NONE


! Argument Variables
! ==================

! Input Variables
! ---------------

      DOUBLE PRECISION, INTENT(IN),
     &  :: time

      DOUBLE PRECISION, INTENT(IN),
     &  DIMENSION(i_sflnod_top:i_sflnod_bot, i_hypspr_1:i_hypspr_n, 3)
     &  :: orgm_x_nodcum

      DOUBLE PRECISION, INTENT(IN),
     &  DIMENSION(i_hypspr_1:i_hypspr_n, 3)
     &  :: orgm_xs_oocelt

      INTEGER, INTENT(IN)
     &  :: i_oocelt

      DOUBLE PRECISION, INTENT(IN)
     &  :: frac_oocelt


! Output Variables
! ----------------

      DOUBLE PRECISION, INTENT(OUT),
     &  DIMENSION(i_sflnod_top:i_sflnod_bot, i_hypspr_1:i_hypspr_n,3)
     &  :: carb_x_nodcum

      DOUBLE PRECISION, INTENT(OUT),
     &  DIMENSION(i_hypspr_1:i_hypspr_n,3)
     &  :: carb_xs_oocelt


! Local Variables
! ===============

#ifdef MBMRHOCAC_HOMOGENEOUS
! Export rain ratio
      DOUBLE PRECISION, DIMENSION(i_hypspr_1:i_hypspr_n)
     &                  :: rhocac_x

! Dummy loop indices
      INTEGER           :: ip
#endif
#ifdef MBMRHOCAC_HATS
! Base rain ratio value
      DOUBLE PRECISION,
     &  DIMENSION(i_hypspr_1:i_hypspr_n), SAVE
     &  :: rhocac_bas = 0D0
      LOGICAL
     &  :: f_rhocac_bas_tbinit = .TRUE.

! Distributed rain ratio
      DOUBLE PRECISION,
     &  DIMENSION(i_sflelt_1:i_sflelt_n,   i_hypspr_1:i_hypspr_n)
     &  :: rhocac_i

! Amplification factor for rhocac (must be strictly greater than 1).
! A value of exactly 1 would lead to homogeneous distributions

      DOUBLE PRECISION, PARAMETER
     &  :: rhocac_amp=2D0

! Number of total production fractions to calculate
! fraction 1:             orgm_x/rhocac_amp
! fraction 2: orgm_x/2 - (orgm_x/rhocac_amp)/2
! fraction 3: orgm_x/2 + (orgm_x/rhocac_amp)/2
! fraction 4: orgm_x   - (orgm_x/rhocac_amp)

      INTEGER, PARAMETER
     &  :: n_frac=4

! Dummy loop indices
      INTEGER
     &  :: i, ip

      INTEGER, SAVE,
     &  DIMENSION(i_hypspr_1:i_hypspr_n, n_frac)
     &  :: i_n=-1

      DOUBLE PRECISION,
     &  DIMENSION(i_hypspr_1:i_hypspr_n, n_frac)
     &  :: f_n, orgmx_n

      DOUBLE PRECISION,
     &  DIMENSION(i_hypspr_1:i_hypspr_n)
     &  :: rhocac_x, rhocac_min, rhocac_max

      DOUBLE PRECISION
     &  :: orgm_xj, orgm_nj, orgm_nja, rhocac_0, rhocac_1
#endif


! Calculations
! ============

#ifdef MBMRHOCAC_HOMOGENEOUS

                                   ! Calculate the global average rain
      CALL CARB_RAINRATIO(time, rhocac_x)  ! ratios for each profile
                                   ! CARB_RAINRATIO returns the actual
                                   ! rain ratios, as a function of time
                                   ! in <rhocac_x>.


      DO ip = i_hypspr_1, i_hypspr_n
        carb_x_nodcum(:,ip,1) = rhocac_x(ip)*orgm_x_nodcum(:,ip,1)
      ENDDO

      carb_x_nodcum(:,:,2:3) = 0D0 ! Set C13 and C14 fluxes to zero.
                                   ! They are calculated in the
                                   ! calling EQNOCE routine.

      carb_xs_oocelt(:,1) = rhocac_x(:)*orgm_xs_oocelt(:,1)
      carb_xs_oocelt(:,2:3) = 0D0  ! Set C13 and C14 fluxes to 0
                                   ! See above
#endif  /* MBMRHOCAC_HOMOGENEOUS */

#ifdef MBMRHOCAC_HATS

      IF (f_rhocac_bas_tbinit) THEN

                                   ! Initialisation of <rhocac_bas>.
                                   ! <rhocac_bas> provides the BASic value
                                   ! for peak interglacial (present-day) times.
                                   ! We currently simply set it to the maximum
                                   ! value of rhocac.

         CALL CARB_RAINRATIO_MINMAX_GET(rhocac_min, rhocac_max)
         rhocac_bas(:) = rhocac_max(:)
         f_rhocac_bas_tbinit = .FALSE.  ! No need to repeat this initialisation.

      ENDIF

      CALL CARB_RAINRATIO(time, rhocac_x)

      orgmx_n(:,1) =  orgm_x_nodcum(i_sflnod_bot,:,1)/rhocac_amp
      orgmx_n(:,2) = (orgm_x_nodcum(i_sflnod_bot,:,1)-orgmx_n(:,1))/2D0
      orgmx_n(:,3) = (orgm_x_nodcum(i_sflnod_bot,:,1)+orgmx_n(:,1))/2D0
      orgmx_n(:,4) =  orgm_x_nodcum(i_sflnod_bot,:,1)-orgmx_n(:,1)

! First approximation of localisation
      IF(i_n(i_hypspr_1,1) == -1) THEN
        i_n(:,1) =  MAXDEP(:)/2
        i_n(:,2) =  MAXDEP(:)/4
        i_n(:,3) = (MAXDEP(:)*3)/4
        i_n(:,4) =  MAXDEP(:)/2
      ENDIF

      DO i = 1, n_frac
        DO ip = i_hypspr_1, i_hypspr_n
          ! Make sure i_n(ip,i) is lower than i_sflnod_bot to start.
          ! Else orgm_xj below cannot be defined.
          i_n(ip,i) = MIN(i_n(ip,i), i_sflnod_bot-1)
          search_node: DO
            orgm_nj  =  orgmx_n(ip, i)
     &                 -orgm_x_nodcum(i_n(ip,i),   ip, 1)
            orgm_xj  =  orgm_x_nodcum(i_n(ip,i)+1, ip, 1)
     &                 -orgm_x_nodcum(i_n(ip,i),   ip, 1)
            IF (orgm_nj .LT. 0D0) THEN
              ! We are at least a whole interval to deep
              i_n(ip,i) = i_n(ip,i)-1
              CYCLE search_node
            ELSEIF (orgm_nj .EQ. 0D0) THEN
              ! <orgmx_n> falls exactly onto a node
              ! Check whether the node above does not lead to
              ! the same deviation.
              IF (i_n(ip,i) .EQ. (i_oocelt-1)) THEN
                ! If we are already at the topmost node, we are done:
                ! The remaining orgm_x fraction on the interval is 0.
                f_n(ip,i) = 0D0
                EXIT search_node
              ELSE
                orgm_nja =  orgmx_n(ip, i)
     &                     -orgm_x_nodcum(i_n(ip,i)-1, ip, 1)
                IF (orgm_nja .EQ. 0D0) THEN
                  ! The node above also has the same value as <orgmx_n>:
                  ! decrement i_n(ip,i) by 1, and cycle once more!
                  i_n(ip,i) = i_n(ip,i)-1
                  CYCLE search_node
                ELSE
                  ! The node above has a different (lower) value:
                  ! We are done at the present i_n(ip,i).
                  ! The remaining orgm_x fraction on the interval is 0.
                  f_n(ip,i) = 0D0
                  EXIT search_node
                ENDIF
              ENDIF
            ELSEIF (orgm_nj .LT. orgm_xj) THEN
              ! <orgmx_n> falls exactly between the current i_n(ip,i)
              ! and i_n(ip,i)+1. Notice that we cannot get here when
              ! orgm_xj=0, since then the "IF (orgm_nj .LT. 0D0)"
              ! block takes precedence. So, in fact, we are done:
              f_n(ip,i) = orgm_nj/orgm_xj
              EXIT search_node
            ELSEIF (orgm_nj .EQ. orgm_xj) THEN
              ! <orgmx_n> falls exactly onto the node below the current
              ! i_n(ip,i). Notice that we cannot get here when
              ! orgm_xj=0, since then the "IF (orgm_nj .EQ. 0D0)" block
              ! takes precedence. So, in fact, we are done then.
              ! The node searched for is the one below, and the
              ! remaining orgm_x fraction on the interval is 0.
              ! (This case also covers the pathological case when
              ! orgmx_n=orgm_x_nodcum(i_sflnod_bot, ip, 1)).
              i_n(ip,i) = i_n(ip,i)+1
              f_n(ip,i) = 0D0
              EXIT search_node
            ELSE
              ! In this case, we know that
              ! orgm_nj > orgm_xj and orgm_nj > 0 (even if orgm_xj=0)
              ! We are at least a whole interval to shallow:
              i_n(ip,i) = i_n(ip,i)+1
              CYCLE search_node
            ENDIF
          ENDDO search_node
        ENDDO
      ENDDO

      DO ip = i_hypspr_1, i_hypspr_n

        rhocac_0 = rhocac_bas(ip)
        rhocac_1 = rhocac_bas(ip)+rhocac_amp*(rhocac_x(ip)-rhocac_0)

#ifdef MBMRHOCAC_SHALLOW_HAT
        rhocac_i(i_sflelt_1:i_oocelt-1,  ip) =                  0D0
        IF (i_n(ip,1) .GE. i_oocelt) ! possibly equal to i_oocelt-1
     &    rhocac_i(i_oocelt:i_n(ip,1),     ip) =                rhocac_1
        IF ((i_n(ip,1)+1) .LE. i_sflelt_n)
     &    rhocac_i(i_n(ip,1)+1,            ip) =     f_n(ip,1) *rhocac_1
     &                                         +(1D0-f_n(ip,1))*rhocac_0
        IF ((i_n(ip,1)+2) .LE. i_sflelt_n)
     &    rhocac_i(i_n(ip,1)+2:i_sflelt_n, ip) =                rhocac_0
#endif

#ifdef MBMRHOCAC_MIDDLE_HAT
        rhocac_i(i_sflelt_1:i_oocelt-1,  ip) =                  0D0
        IF (i_n(ip,2) .GE. i_oocelt) ! possibly equal to i_oocelt-1
     &    rhocac_i(i_oocelt:i_n(ip,2),     ip) =                rhocac_0
        IF ((i_n(ip,2)+1) .LE. i_sflelt_n) THEN
          IF (i_n(ip,2) .EQ. i_n(ip,3)) THEN
            rhocac_i(i_n(ip,2)+1,            ip) =   f_n(ip,2) *rhocac_0
     &                                   +(f_n(ip,3)-f_n(ip,2))*rhocac_1
     &                                         +(1D0-f_n(ip,3))*rhocac_0
          ELSE
            rhocac_i(i_n(ip,2)+1,            ip) =   f_n(ip,2) *rhocac_0
     &                                         +(1D0-f_n(ip,2))*rhocac_1
          ENDIF
        ENDIF
        IF (i_n(ip,3) .LE. i_sflelt_n) THEN
          IF (i_n(ip,3) .GE. (i_n(ip,2)+2))
     &      rhocac_i(i_n(ip,2)+2:i_n(ip,3),  ip) =              rhocac_1
        ENDIF
        
        IF ((i_n(ip,3)+1) .LE. i_sflelt_n) THEN
          IF (i_n(ip,2) .NE. i_n(ip,3))
     &      rhocac_i(i_n(ip,3)+1,            ip) =   f_n(ip,3) *rhocac_1
     &                                         +(1D0-f_n(ip,3))*rhocac_0
        ENDIF
        IF ((i_n(ip,3)+2) .LE. i_sflelt_n)
     &    rhocac_i(i_n(ip,3)+2:i_sflelt_n, ip) =                rhocac_0
#endif

#ifdef MBMRHOCAC_DEEP_HAT
        rhocac_i(i_sflelt_1:i_oocelt-1,  ip) =                  0D0
        IF (i_n(ip,4) .GE. i_oocelt) ! possibly equal to i_oocelt-1
     &    rhocac_i(i_oocelt:i_n(ip,4),     ip) =                rhocac_0
        IF ((i_n(ip,4)+1) .LE. i_sflelt_n)
     &    rhocac_i(i_n(ip,4)+1,            ip) =     f_n(ip,4) *rhocac_0
     &                                         +(1D0-f_n(ip,4))*rhocac_1
        IF ((i_n(ip,4)+2) .LE. i_sflelt_n)
     &    rhocac_i(i_n(ip,4)+2:i_sflelt_n, ip) =                rhocac_1
#endif
      ENDDO


! Now determine the cumulated carbonate flux

      carb_xs_oocelt(:, 1) =   orgm_xs_oocelt(:, 1)
     &                       * rhocac_i(i_oocelt, :)
      carb_xs_oocelt(:, 2:3) = 0D0


      carb_x_nodcum(:, :, :) = 0D0

      DO i = i_sflnod_top+1, i_sflnod_bot
        carb_x_nodcum(i, :, 1) =
     &    carb_x_nodcum(i-1, :, 1)
     &    + (orgm_x_nodcum(i, :, 1) - orgm_x_nodcum(i-1, :, 1))
     &     * rhocac_i(i,:)
          
      ENDDO
#endif  /* MBMRHOCAC_HATS */

      RETURN

      END SUBROUTINE EQNOCE_CARB_DISTRIBUTION
