!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#define CLAY_FLUXDENSITY 2D-21 /* 2 g/m2/yr = 2D-21 x 10^18 kg/m2/yr */
C     ******************************************************************
      SUBROUTINE EQNOCE(yoca, ysed, dtyoca,
     >            corali0, coralo0, shbnki0, shbnko0,
     >            sxcini, sxcino, sxcori, sxcoro,
     >            sxalki, sxalko, sxoxyi, sxoxyo,
     <            arag2s, calc2s, othr2s, orgm2s,
     &            iflag, biof01)
C     ******************************************************************
C
C     Calls: ADVECT, ACTIVY, CONTIW
C     iflag : exit status
C             iflag=0  all OK and within allowed limits
C             iflag=1  *activy* caused trouble
C
C     iasd01, svsd01, biof01 : options/flags
C     iasd01=.TRUE.  : fully InterActive SeDiment
C           =.FALSE. : replay of previously recorded in/outputs
C     svsd01=.TRUE.  : SaVe SeDiment (recording) on
C           =.FALSE. : off
C     biof01

      USE mod_mbm_geometry
      USE mod_mbm_options
      USE mod_mbm_tempo, ONLY: temps
      USE mod_mbm_xsetup
      USE mod_mbm_biocts
      USE mod_mbm_materialparams, ONLY: c13rk, c14rk
      USE mod_mbm_other
      USE mod_mbm_chimiq
      USE mod_mbm_eqnoce_txt

      USE mod_mbm_paleod, ONLY: epoch
      USE mod_mbm_biocarbonates, ONLY: ARAG_FRACTION
      USE mod_mbm_conbio


      USE mod_mbm_medusa_o2s

#ifdef LOOP_ORGMATTER
      USE mod_mbm_medusa_s2o, ONLY: mbm_orgm_lb
#endif
#ifdef LOOP_CALCITE
      USE mod_mbm_medusa_s2o, ONLY: mbm_calc_lb
#endif
#ifdef LOOP_ARAGONITE
      USE mod_mbm_medusa_s2o, ONLY: mbm_arag_lb
#endif

#ifdef MBM_WITH_NETCDF
      USE mod_mbm_files
#else
      USE mod_mbm_files, ONLY: erruni, resusd, resuni,
     &                         cbfusd, cslusd, csluni
#endif

      USE mod_mbm_dorpri, ONLY: taumin_oca, taumin_oca_var,
     &                          taumin_oca_ires


      IMPLICIT NONE


      INTENT(IN) :: ysed
      DOUBLE PRECISION yoca(n_xoca), dtyoca(n_xoca)
      DOUBLE PRECISION ysed(n_xsed)
      DOUBLE PRECISION, DIMENSION(ro1+1:ro1+nro1) ::
     &   coralo, corali, shbnko, shbnki
      DOUBLE PRECISION, DIMENSION(ro1+1:ro1+nro1) ::
     &   coralo0, corali0, shbnko0, shbnki0

      DOUBLE PRECISION, DIMENSION(1:bottom,ro1+1:ro1+nro1,3) ::
     &   calc2s, arag2s, orgm2s
      DOUBLE PRECISION, DIMENSION(1:bottom,ro1+1:ro1+nro1  ) ::
     &   othr2s

      DOUBLE PRECISION, DIMENSION(ro1+1:ro1+nro1,3,3) ::
     &   sxcini, sxcino, sxcori, sxcoro
      DOUBLE PRECISION, DIMENSION(ro1+1:ro1+nro1,3) ::
     &   sxalki, sxalko, sxoxyi, sxoxyo

      INTEGER iflag
      LOGICAL biof01
C
      DOUBLE PRECISION, DIMENSION(n_xoca) ::  xoca, dtxoca
#include "eqnoce_x.equivalence"
#include "eqnoce_dtx.equivalence"

      DOUBLE PRECISION xsed(n_xsed)
#include "eqnsed_x.equivalence"


       DOUBLE PRECISION :: tc_in, tc_out, ta_in, ta_out
       COMMON /INOUT/ tc_in, tc_out, ta_in, ta_out

! Carbonate rain ratio
      DOUBLE PRECISION, SAVE, DIMENSION(ro1+1:ro1+nro1)
     &   :: rhocac
     
! Aragonite fraction
      DOUBLE PRECISION
     &   :: frcara



! Local variables
! ===============

! Phosphate consumption and recycling rates
      DOUBLE PRECISION csmpo4(ro1+1:ro1+nro1)
      DOUBLE PRECISION rcypo4(ro1+1:ro3+nro3)

! Oxygen consumption rate
      DOUBLE PRECISION csmoxy(ro1+1:ro3+nro3)

      DOUBLE PRECISION  diffa(ro1+1:ro1+nro1)
      DOUBLE PRECISION tdiffa
      DOUBLE PRECISION  diffs(ro1+1:ro1+nro1)
      DOUBLE PRECISION tdiffs
      DOUBLE PRECISION gfco2i(ro1+1:ro1+nro1)
      DOUBLE PRECISION gfco2o(ro1+1:ro1+nro1)
      DOUBLE PRECISION wfpo4i(ro1+1:ro3+nro3)
      DOUBLE PRECISION wfpo4o(ro1+1:ro3+nro3)
      DOUBLE PRECISION wfciti(ro1+1:ro3+nro3)
      DOUBLE PRECISION wfcito(ro1+1:ro3+nro3)
      DOUBLE PRECISION wfoxyi(ro1+1:ro3+nro3)
      DOUBLE PRECISION wfoxyo(ro1+1:ro3+nro3)
      DOUBLE PRECISION wfalki(ro1+1:ro3+nro3)
      DOUBLE PRECISION wfalko(ro1+1:ro3+nro3)
      DOUBLE PRECISION dic_w(ro1+1:ro1+nro1,3)
      DOUBLE PRECISION dic_i(ro1+1:ro3+nro3,3)
      DOUBLE PRECISION dic_o(ro1+1:ro3+nro3,3)
      DOUBLE PRECISION alk_i(ro1+1:ro3+nro3)
      DOUBLE PRECISION alk_o(ro1+1:ro3+nro3)
      DOUBLE PRECISION oxy_i(ro1+1:ro3+nro3)
      DOUBLE PRECISION oxy_o(ro1+1:ro3+nro3)
      DOUBLE PRECISION po4_r(ro1+1:ro1+nro1)
      DOUBLE PRECISION po4_w(ro1+1:ro1+nro1)
      DOUBLE PRECISION po4_i(ro1+1:ro3+nro3)
      DOUBLE PRECISION po4_o(ro1+1:ro3+nro3)

! Organic matter C fluxes:
! - calc_o1(i,:) : across bottom interface of surface reservoir <i>
! - calc_o2(i,:) : across bottom interface of intermediate reservoir
!                  located below surface reservoir <i>
!                  for i=ro1+1/3/5 (i.e. surface reservoirs
!                  without intermediate ones below)
!                  calc_o2(i,:)=calc_o1(i,:)
! - orgm_x(i,:)  : "export" production (across 100m depth horizon)
!                  in reservoir <i>
! - orgm_r(i,:)  : remineralisation within reservoir <i>
! Similarly for arag_xn, carb_xn, except that
! carb_xn also includes at the end the contributions
! from coral reef and shelf inorganic matter accumulation

      DOUBLE PRECISION, DIMENSION(ro1+1:ro1+nro1,3) ::
     &   orgm_x, carb_o1, calc_o1, arag_o1
      DOUBLE PRECISION, DIMENSION(ro1+1:ro3+nro3,3)
     &                  :: calc_d, arag_d
      DOUBLE PRECISION, DIMENSION(ro1+1:ro3+nro3,3)
     &                  :: orgm_r
      DOUBLE PRECISION, DIMENSION(i_sflnod_top:i_sflnod_bot,
     &                            i_hypspr_1  :i_hypspr_n,3)
     &                  :: orgm_x_nodcum, carb_x_nodcum,
     &                     calc_x_nodcum, arag_x_nodcum
      DOUBLE PRECISION, DIMENSION(i_hypspr_1  :i_hypspr_n,3)
     &                  :: orgm_xs_sflelt_200, carb_xs_sflelt_200

      DOUBLE PRECISION   cort(ro2+1:ro2+nro2,3)
      DOUBLE PRECISION atc13i(ra1+1:ra1+nra1)
      DOUBLE PRECISION atc13o(ra1+1:ra1+nra1)
      DOUBLE PRECISION aqc13i(ro1+1:ro3+nro3)
      DOUBLE PRECISION aqc13o(ro1+1:ro3+nro3)
      DOUBLE PRECISION iwc13i, swc13i
      DOUBLE PRECISION atc14i(ra1+1:ra1+nra1)
      DOUBLE PRECISION atc14o(ra1+1:ra1+nra1)
      DOUBLE PRECISION aqc14i(ro1+1:ro3+nro3)
      DOUBLE PRECISION aqc14o(ro1+1:ro3+nro3)
      DOUBLE PRECISION iwc14i, swc14i

      DOUBLE PRECISION iwci(ro1+1:ro1+nro1)
      DOUBLE PRECISION swci(ro1+1:ro1+nro1)
      DOUBLE PRECISION volcsi, volcca, volctt
      DOUBLE PRECISION iwcitt
      DOUBLE PRECISION swcitc, swcits, swcitt
      DOUBLE PRECISION orco2t, crbict, arbict
      DOUBLE PRECISION owco2t, rwpo4t

      COMMON /FORCINGS/ volcsi, volcca, volctt,
     &                  iwcitt, swcitc, swcits, swcitt,
     &                  coralo, corali, shbnko, shbnki

      DOUBLE PRECISION  po4(ro1+1:ro3+nro3)
      DOUBLE PRECISION  cit(ro1+1:ro3+nro3)
      DOUBLE PRECISION  alk(ro1+1:ro3+nro3)
      DOUBLE PRECISION  oxy(ro1+1:ro3+nro3)
      DOUBLE PRECISION xc13(ro1+1:ro3+nro3)
      DOUBLE PRECISION xc14(ro1+1:ro3+nro3)
      DOUBLE PRECISION  co3(ro1+1:ro3+nro3)
      DOUBLE PRECISION   ph(ro1+1:ro3+nro3)
      DOUBLE PRECISION pco2(ro1+1:ro3+nro3)
      INTEGER i, j, ir, il, ip, iiflag, i2, i3
      INTEGER itemps
      DOUBLE PRECISION ptemps

      DOUBLE PRECISION totcar, totalk, totc13, totc14, totpo4, totoxy
C
C     The following is active only if cbcinp=.TRUE.
C     co2xcb: CO2 exchange with continental biosphere
C             >0 = net gain for atmosphere
C             <0 = net loss for atmosphere
C     c13xcb: deltaC13 exchange with continental biosphere
C             =co2xcb*deltaC13/14
C     co2xcb: PO4 exchange with continental biosphere
C             >0 = net gain for ocean
C             <0 = net loss for ocean
      DOUBLE PRECISION co2xcb, c13xcb, po4xcb

      DOUBLE PRECISION ::  d0,  d100,  d200,  d1000
      DOUBLE PRECISION :: frac_sflelt_d200
      INTEGER          :: id0, id100, id200, id1000
      INTEGER          :: i_sflelt_d0, i_sflelt_d100,
     &                    i_sflelt_d200, i_sflelt_d1000
      DOUBLE PRECISION, DIMENSION(ro1+1:ro1+nro1)
     &                 :: sfc200,sfc1000


      INTEGER, SAVE :: ionce = 0

      DOUBLE PRECISION :: total_out

!      DOUBLE PRECISION, PARAMETER ::
!     &   perma_stock = 1000D0 * 1D-3/12D0 ! PgC to 10^18 mol C
!      DOUBLE PRECISION, PARAMETER ::
!     &   perma_releasedurat = 10000D0       ! years
!      DOUBLE PRECISION, PARAMETER ::
!     &   perma_uptake = -perma_stock/(120000D0-perma_releasedurat)
!      DOUBLE PRECISION, PARAMETER ::
!     &   perma_release = perma_stock/perma_releasedurat
!      DOUBLE PRECISION, PARAMETER ::
!     &   perma_c13 = -30.0D0
!      DOUBLE PRECISION, PARAMETER ::
!     &   perma_c14 = -1000D0
!      DOUBLE PRECISION, PARAMETER ::
!     &   perma_start = 1950D0-17000D0
!      DOUBLE PRECISION, PARAMETER ::
!     &   perma_t3 = perma_start+50D0
!      DOUBLE PRECISION, PARAMETER ::
!     &   perma_t4 = perma_t3 -100D0
!      DOUBLE PRECISION, PARAMETER ::
!     &   perma_t1 = perma_t3 + perma_releasedurat
!      DOUBLE PRECISION, PARAMETER ::
!     &   perma_t2 = perma_t4 + perma_releasedurat
!      DOUBLE PRECISION, PARAMETER ::
!     &   perma_t5 = perma_t1 - 120000D0
!      DOUBLE PRECISION, PARAMETER ::
!     &   perma_t6 = perma_t2 - 120000D0
!      DOUBLE PRECISION, PARAMETER ::
!     &   perma_t7 = perma_t3 - 120000D0
!      DOUBLE PRECISION, PARAMETER ::
!     &   perma_t8 = perma_t4 - 120000D0
!      DOUBLE PRECISION, PARAMETER ::
!     &   perma_releaseslope = (perma_release-perma_uptake)/100D0
!      DOUBLE PRECISION ::
!     &   permafrostco2, permafrostco2_c13, permafrostco2_c14


      xoca = yoca
      xsed = ysed

! Force Delta C14 = 0 in xoca
      c14a=2D0*(c13a+25D0)/(1D0-2D0*(c13a+25D0)/1D3)


       d0        = (-nivmer)/resol
      id0        = INT(d0)
      i_sflelt_d0 = INT(d0)+1

       d100      = (100.D+00-nivmer)/resol
      id100      = INT(d100)
      i_sflelt_d100 = INT(d100)+1


       d200      = (200.D+00-nivmer)/resol
      id200      = INT(d200)
      i_sflelt_d200 = INT(d200)+1
      frac_sflelt_d200 = i_sflelt_d200 - d200
      sfc200(:)  = hypsar(bottom,:)-
     &              (hypsar(id200,:)
     &               +(d200-id200)*sflelt_sfcarea(id200+1,:))

       d1000     = (1000.D+00-nivmer)/resol
      id1000     = INT(d1000)
      i_sflelt_d100 = INT(d1000)+1
      sfc1000(:) = hypsar(bottom,:)-
     &              (hypsar(id1000,:)
     &               +(d1000-id1000)*sflelt_sfcarea(id1000+1,:))


      DO i = i_sflelt_1, i_sflelt_n
        mbm_surf(:,i) = sflelt_sfcarea(i,:)
        mbm_dept(i) = (DBLE(i)-0.5D0)*resol+nivmer
      ENDDO
      mbm_dept(i_sflelt_1:id0) = 0D0

!GM 13MAR2005

C     === FLUXES  ======================================================
C
C     ---  FLUXES OF DISSOLVED PHOSPHATE (10^18 MOLES/YEAR)
C               OF DISSOLVED INORGANIC CARBONE (10^18 MOLES/YEAR)
C               OF DISSOLVED OXYGEN (10^18 MOLES/YEAR)
C               OF DISSOLVED ALKALINITY  (10^18 EQ/YEAR)
c               OF CARBON-13  (/YEAR)
C               OF CARBON-14  (/YEAR)  --------------------------------
C
      ! In case of negative amounts of matter, exit immediately
      iflag = 0

      IF (ANY(vcit(:) < 0D0)) THEN
         WRITE(erruni,*) '[EQNOCE]: Negative vcit provided.'
         iflag = 1
      ENDIF

      IF (ANY(valk(:) < 0D0)) THEN
         WRITE(erruni,*) '[EQNOCE]: Negative valk provided.'
         iflag = 1
      ENDIF

      IF (ANY(vpo4(:) < 0D0)) THEN
         WRITE(erruni,*) '[EQNOCE]: Negative vpo4 provided.'
         iflag = 1
      ENDIF

      IF (ANY(voxy(:) < 0D0)) THEN
         WRITE(erruni,*) '[EQNOCE]: Negative voxy provided.'
         iflag = 1
      ENDIF

      IF(iflag /= 0) THEN
         WRITE(erruni,*)'    Returning to caller.'
         RETURN
      ENDIF
      

      cit(:)=vcit(:)/volum(:)
      alk(:)=valk(:)/volum(:)
      po4(:)=vpo4(:)/volum(:)
      oxy(:)=voxy(:)/volum(:)
      xc13(:)=cit(:)*cc13(:)
      xc14(:)=cit(:)*cc14(:)

      CALL ADVECT(po4,wfpo4i,wfpo4o)
      CALL ADVECT(cit,wfciti,wfcito)
      CALL ADVECT(oxy,wfoxyi,wfoxyo)
      CALL ADVECT(alk,wfalki,wfalko)
      CALL ADVECT(xc13,aqc13i,aqc13o)
      CALL ADVECT(xc14,aqc14i,aqc14o)

      ! Determine pco2 and co3 for all of the boxes
      CALL ACTIVY(borate,k0,k1,k2,kb,kw,
     &            cit,alk,ph,pco2,co3,nro1+nro2+nro3,iiflag)
      ! In case of trouble with activy, exit immediately
      IF (iiflag == 1) THEN
         WRITE(erruni,*) '[EQNOCE]: ACTIVY failed.'
         WRITE(erruni,*)'    Returning to caller.'
         iflag = 1
         RETURN
      ENDIF

      tdiffa=0.D+00
      tdiffs=0.D+00
      atc13i(ra1+1)=0.D+00
      atc14i(ra1+1)=0.D+00

      dic_i(:,:) = 0D0
      dic_o(:,:) = 0D0

      alk_i(:) = 0D0
      alk_o(:) = 0D0

      oxy_i(:) = 0D0
      oxy_o(:) = 0D0

      po4_i(:) = 0D0
      po4_o(:) = 0D0

      iwci(:)   = 0D0
      swci(:)   = 0D0
      

      DO 11 i=ro1+1,ro1+nro1
C
C            (10^18 MOLES/YEAR)  -------------------------------------
C
C
C        ---  DIFFUSION OF CO2 BETWEEN THE ATMOSPHERE AND THE SURFACE
C             xDIFFx IN PAL (PRESENT ATMOSPHERIC LEVEL)
C             GFCO2x IN 10^18 MOLES/YEAR  ------------------------------
C
         diffa(i) = pco2(i)*propor(i)
         diffs(i) = pco2a*propor(i)
         gfco2i(i) = diffs(i)*qteco2(ra1+1)/tau
         gfco2o(i) = diffa(i)*qteco2(ra1+1)/tau
         tdiffa = tdiffa+diffa(i)
         tdiffs = tdiffs+diffs(i)
         atc13i(ra1+1) = atc13i(ra1+1)+diffa(i)*cc13(i)
         atc14i(ra1+1) = atc14i(ra1+1)+diffa(i)*cc14(i)
         aqc13i(i) = aqc13i(i)+gfco2i(i)*c13a
         aqc13o(i) = aqc13o(i)+gfco2o(i)*(cc13(i)-frc13a)
         aqc14i(i) = aqc14i(i)+gfco2i(i)*c14a
         aqc14o(i) = aqc14o(i)+gfco2o(i)*(cc14(i)-2.D+00*frc13a)
C
   11 CONTINUE

      atc13i(ra1+1) = (atc13i(ra1+1)-tdiffa*frc13a)/tau
      atc14i(ra1+1) = (atc14i(ra1+1)-tdiffa*2.D+00*frc13a)/tau
      atc13o(ra1+1) = tdiffs*c13a/tau
      atc14o(ra1+1) = tdiffs*c14a/tau

!1- Organic Matter related Fluxes and Species
!1- Phosphate consumption rate (10^18 moles/year)
      csmpo4(:) = po4uti(:)*wfpo4i(ro1+1:ro1+nro1)

!1- O2 concentration (moles/m3), in equilibrium with atmosphere
      oxy(ro1+1:ro1+nro1) = koxy(ro1+1:ro1+nro1)*0.209D+00

!1- Particulate organic and inorganic carbon fluxes (10^18 moles/year)
!1  

! Total export in 10^18 mol/yr
      orgm_x(:,1) = csmpo4(:)/rhopc
      orgm_x(:,2) = orgm_x(:,1)*(cc13(ro1+1:ro1+nro1)-    frc13b)
      orgm_x(:,3) = orgm_x(:,1)*(cc14(ro1+1:ro1+nro1)-2D0*frc13b)


      CALL EQNOCE_ORGM_DISTRIBUTION
     >           (orgm_x,
     >            i_sflelt_d200, frac_sflelt_d200,
     <            orgm_x_nodcum, orgm_xs_sflelt_200)

      CALL EQNOCE_ORGM_POWERLAW
     >           (orgm_x_nodcum, orgm_xs_sflelt_200,
     <            orgm_r, orgm2s)


!1- Oxygen sink and Phosphate source resulting from organic matter
!   remineralisation (reservoirs ro2+: and ro3+:)
! !!!CORRECT!!! parameterize Redfield ratio below
      csmoxy(:) = orgm_r(:,1)*orgm_ro2/orgm_c
      rcypo4(:) = orgm_r(:,1)*rhopc


!2- Carbonate fluxes from the top to the sediments  *****

      CALL EQNOCE_CARB_DISTRIBUTION
     >           (temps, orgm_x_nodcum, orgm_xs_sflelt_200,
     >            i_sflelt_d200, frac_sflelt_d200,
     <            carb_x_nodcum, carb_xs_sflelt_200)

      DO i = ro1+1, ro1+nro1
        carb_x_nodcum(:,i,2) = carb_x_nodcum(:,i,1)*cc13(i)
        carb_x_nodcum(:,i,3) = carb_x_nodcum(:,i,1)*cc14(i)
      ENDDO

!2- A fraction 'frcara' of the total produced CaCO3 is aragonite:
      CALL ARAG_FRACTION(temps, frcara)
      arag_x_nodcum(:,:,:) = carb_x_nodcum(:,:,:)*frcara
      calc_x_nodcum(:,:,:) = carb_x_nodcum(:,:,:)-arag_x_nodcum(:,:,:)

      carb_o1(:,:) = carb_x_nodcum(i_sflnod_bot,:,:)
      arag_o1(:,:) = arag_x_nodcum(i_sflnod_bot,:,:)
      calc_o1(:,:) = calc_x_nodcum(i_sflnod_bot,:,:)

!3- Ocean-to-sediment fluxes

      i = 3 ! 3 isotopes
      CALL EQNOCE_XPRT_STRAIGHTTOSEAFLOOR
     >           (arag_x_nodcum, i, arag_d, arag2s)

      CALL EQNOCE_XPRT_STRAIGHTTOSEAFLOOR
     >           (calc_x_nodcum, i, calc_d, calc2s)


!3- Clay (othr) proceeds directly from the surface to the sediment
!   0.002 kg/m2/yr over the whole ocean (units here: 10^18 kg/m2/yr)
       othr2s(:,:) = (CLAY_FLUXDENSITY) * sflelt_sfcarea(:,:)
!TEST001! increase clay flux in Southern Ocean to 5 g/m2/yr to mimick
!TEST001! dilution by opal accumulation
!TEST001       othr2s(:,ro1+3) = othr2s(:,ro1+3) * 2.5D0


!4- Sediment-Ocean (Medusa-MBM) interface
!4- This is the outflux from the sediment, as DIC, resulting from
!   the oxidation of Organic matter and the dissolution of
!   carbonate material in the sediment.
      dic_i(ro1+1,:)  =   sxcini(ro1+1,1,:)+sxcini(ro1+1,2,:)
     &                  + sxcori(ro1+1,1,:)+sxcori(ro1+1,2,:)
      dic_i(ro1+2,:)  =   sxcini(ro1+2,1,:)
     &                  + sxcori(ro1+2,1,:)
      dic_i(ro1+3,:)  =   sxcini(ro1+3,1,:)+sxcini(ro1+3,2,:)
     &                  + sxcori(ro1+3,1,:)+sxcori(ro1+3,2,:)
      dic_i(ro1+4,:)  =   sxcini(ro1+4,1,:)
     &                  + sxcori(ro1+4,1,:)
      dic_i(ro1+5,:)  =   sxcini(ro1+5,1,:)+sxcini(ro1+5,2,:)
     &                  + sxcori(ro1+5,1,:)+sxcori(ro1+5,2,:)

      dic_i(ro2+1,:)  =   sxcini(ro1+2,2,:)
     &                  + sxcori(ro1+2,2,:)
      dic_i(ro2+2,:)  =   sxcini(ro1+4,2,:)
     &                  + sxcori(ro1+4,2,:)

      dic_i(ro3+1,:)  =   sxcini(ro1+1,3,:)+sxcini(ro1+2,3,:)
     &                  + sxcori(ro1+1,3,:)+sxcori(ro1+2,3,:)
      dic_i(ro3+2,:)  =   sxcini(ro1+3,3,:)
     &                  + sxcori(ro1+3,3,:)
      dic_i(ro3+3,:)  =   sxcini(ro1+4,3,:)+sxcini(ro1+5,3,:)
     &                  + sxcori(ro1+4,3,:)+sxcori(ro1+5,3,:)

!4- This is a possible uptake of deep-ocean DIC by the sediment,
!   in case the top gradients became negative
!   (This was impossible with MBM-CIPROMILA).
      dic_o(ro1+1,:)  =   sxcino(ro1+1,1,:)+sxcino(ro1+1,2,:)
      dic_o(ro1+2,:)  =   sxcino(ro1+2,1,:)
      dic_o(ro1+3,:)  =   sxcino(ro1+3,1,:)+sxcino(ro1+3,2,:)
      dic_o(ro1+4,:)  =   sxcino(ro1+4,1,:)
      dic_o(ro1+5,:)  =   sxcino(ro1+5,1,:)+sxcino(ro1+5,2,:)

      dic_o(ro2+1,:)  =   sxcino(ro1+2,2,:)
      dic_o(ro2+2,:)  =   sxcino(ro1+4,2,:)

      dic_o(ro3+1,:)  =   sxcino(ro1+1,3,:)+sxcino(ro1+2,3,:)
      dic_o(ro3+2,:)  =   sxcino(ro1+3,3,:)
      dic_o(ro3+3,:)  =   sxcino(ro1+4,3,:)+sxcino(ro1+5,3,:)

!4- This is the outflux of Alk from the sediment resulting from
!   the combined dissolution of carbonate material
!   and oxidation of organic matter there
      alk_i(ro1+1)  =   sxalki(ro1+1,1)+sxalki(ro1+1,2)
      alk_i(ro1+2)  =   sxalki(ro1+2,1)
      alk_i(ro1+3)  =   sxalki(ro1+3,1)+sxalki(ro1+3,2)
      alk_i(ro1+4)  =   sxalki(ro1+4,1)
      alk_i(ro1+5)  =   sxalki(ro1+5,1)+sxalki(ro1+5,2)

      alk_i(ro2+1)  =   sxalki(ro1+2,2)
      alk_i(ro2+2)  =   sxalki(ro1+4,2)

      alk_i(ro3+1)  =   sxalki(ro1+1,3)+sxalki(ro1+2,3)
      alk_i(ro3+2)  =   sxalki(ro1+3,3)
      alk_i(ro3+3)  =   sxalki(ro1+4,3)+sxalki(ro1+5,3)

!4- This is a possible uptake of Alk by the sediment,
!   in case the top gradients became negative
!   (This was impossible with MBM-CIPROMILA).
      alk_o(ro1+1)  =   sxalko(ro1+1,1)+sxalko(ro1+1,2)
      alk_o(ro1+2)  =   sxalko(ro1+2,1)
      alk_o(ro1+3)  =   sxalko(ro1+3,1)+sxalko(ro1+3,2)
      alk_o(ro1+4)  =   sxalko(ro1+4,1)
      alk_o(ro1+5)  =   sxalko(ro1+5,1)+sxalko(ro1+5,2)

      alk_o(ro2+1)  =   sxalko(ro1+2,2)
      alk_o(ro2+2)  =   sxalko(ro1+4,2)

      alk_o(ro3+1)  =   sxalko(ro1+1,3)+sxalko(ro1+2,3)
      alk_o(ro3+2)  =   sxalko(ro1+3,3)
      alk_o(ro3+3)  =   sxalko(ro1+4,3)+sxalko(ro1+5,3)

!4- This is the outflux of O2 from the sediment
!   (should normally be 0 (??))
      oxy_i(ro1+1)  =   sxoxyi(ro1+1,1) + sxoxyi(ro1+1,2)
      oxy_i(ro1+2)  =   sxoxyi(ro1+2,1)
      oxy_i(ro1+3)  =   sxoxyi(ro1+3,1) + sxoxyi(ro1+3,2)
      oxy_i(ro1+4)  =   sxoxyi(ro1+4,1)
      oxy_i(ro1+5)  =   sxoxyi(ro1+5,1) + sxoxyi(ro1+5,2)

      oxy_i(ro2+1)  =                     sxoxyi(ro1+2,2)
      oxy_i(ro2+2)  =                     sxoxyi(ro1+4,2)

      oxy_i(ro3+1)  =   sxoxyi(ro1+1,3)
     &                + sxoxyi(ro1+2,3)
      oxy_i(ro3+2)  =   sxoxyi(ro1+3,3)
      oxy_i(ro3+3)  =   sxoxyi(ro1+4,3)
     &                + sxoxyi(ro1+5,3)

!4- This is the uptake of O2 by the sediment, resulting from
!   the oxidation of Organic matter.
      oxy_o(ro1+1)  =   sxoxyo(ro1+1,1) + sxoxyo(ro1+1,2)
      oxy_o(ro1+2)  =   sxoxyo(ro1+2,1)
      oxy_o(ro1+3)  =   sxoxyo(ro1+3,1) + sxoxyo(ro1+3,2)
      oxy_o(ro1+4)  =   sxoxyo(ro1+4,1)
      oxy_o(ro1+5)  =   sxoxyo(ro1+5,1) + sxoxyo(ro1+5,2)

      oxy_o(ro2+1)  =                     sxoxyo(ro1+2,2)
      oxy_o(ro2+2)  =                     sxoxyo(ro1+4,2)

      oxy_o(ro3+1)  =   sxoxyo(ro1+1,3)
     &                + sxoxyo(ro1+2,3)
      oxy_o(ro3+2)  =   sxoxyo(ro1+3,3)
      oxy_o(ro3+3)  =   sxoxyo(ro1+4,3)
     &                + sxoxyo(ro1+5,3)
      
!4- This is the returning flux of phosphate from the
!   sediment resulting from organic matter degradation there
      po4_i(:) = oxy_o(:)/orgm_ro2


!5- Continental weathering fluxes
!
!   iwcitt: total HCO3 input from silicate weathering
!   swcitc: HCO3 input from carbonate weathered through H2CO3
!   swcits: HCO3 input from carbonate weathered through H2SO4
!   swcitt: = swcitc+swcits
!
!   volcsi: volc./hydroth. CO2 flux related to silicate weathering
!           equal to half the average HCO3 sili
!   volcca: volc./hydroth. CO2 flux related to carb. weath. by H2SO4
!           equal to half the average HCO3 carb from H2SO4
!   volctt: = volcsi-volcca
!
!   For scenario 0:
!        swcitc=2.0D+00*(bicinp+corali(ro1+2)+corali(ro1+4))
!        swcits=0.0D+00
!        corali(ro1+2) and corali(ro1+4) unchanged
!
!   For other scenarii:
!        iwcitt=fct(HCO3 flux present and LGM)
!        swcitc=fct(HCO3 flux present and LGM)
!        swcits=0.0D+00
!        volcca=0.0D+00
!        corali(ro1+2)=corali(ro1+4)=0.0D+0, and not
!        at all taken into account !


      CALL contiw(temps,coralo,corali,
     <            volcsi,volcca,iwcitt,swcitc,swcits,owco2t,rwpo4t)
!
!
!! Permafrost
!
!      IF(temps < perma_t8) THEN
!        permafrostco2     = perma_uptake
!      ELSEIF(temps < perma_t7) THEN
!        permafrostco2 =
!     &    perma_uptake+(temps-perma_t8)*perma_releaseslope
!      ELSEIF(temps < perma_t6) THEN
!        permafrostco2     = perma_release
!      ELSEIF(temps < perma_t5) THEN
!        permafrostco2 =
!     &    perma_release-(temps-perma_t6)*perma_releaseslope
!      ELSEIF(temps < perma_t4) THEN
!        permafrostco2     = perma_uptake
!      ELSEIF(temps < perma_t3) THEN
!        permafrostco2 =
!     &    perma_uptake+(temps-perma_t4)*perma_releaseslope
!      ELSEIF(temps < perma_t2) THEN
!        permafrostco2     = perma_release
!      ELSEIF(temps < perma_t1) THEN
!        permafrostco2 =
!     &    perma_release-(temps-perma_t2)*perma_releaseslope
!      ELSE
!        permafrostco2     = perma_uptake
!      ENDIF
!
!      permafrostco2_c13 = permafrostco2*perma_c13
!      IF(permafrostco2 < 0D0) THEN
!        permafrostco2_c14 = permafrostco2*0D0
!      ELSE
!        permafrostco2_c14 = permafrostco2*perma_c14
!      ENDIF


!5- Organic matter input: C oxidised as CO2 into the atmosphere,
!   as a result of loop-back Recycling
#ifdef LOOP_ORGMATTER
      owco2t = 0D0  ! To avoid having the model run away: C_org
      rwpo4t = 0D0  ! burial will else grow without bounds
      orco2t = SUM(mbm_orgm_lb)
#else
      orco2t = 0D0
#endif
#ifdef LOOP_CALCITE
      crbict = 2D0*(SUM(mbm_calc_lb) + SUM(shbnko))
#else
      crbict = 0D0
#endif
#ifdef LOOP_ARAGONITE
      arbict = 2D0*(SUM(mbm_arag_lb) + SUM(coralo))
#else
      arbict = 0D0
#endif

!5- Constant Carbonate Rock Weathering
      IF (ctcrbw) THEN
         swcitc=carwbt+2.0*(corali(ro1+2)+corali(ro1+4))
      ENDIF
      IF (ctcrsw) THEN
         swcits=carwst
         volcca=carwst/2.0D+00         
      ENDIF

!5- Constant silicate rock Weathering
      IF (ctsirw) THEN
         iwcitt=sirwtt
         volcsi=sirwtt/2.0D+00
      ENDIF

!5- Constant organic rock Weathering - TBD !! owco2t=0 declaration !!
!      IF (ctorrw) THEN
!         owco2t=orrwtt
!      ENDIF

      ! 10APR2005 Added loop-back contributions (only carbonates)
      swcitc = swcitc + crbict + arbict

      swcitt = swcitc+swcits
      volctt = volcsi-volcca
      
!5- Distribute swcitt and iwcitt 1/2 and 1/2 between the Atlantic
!   and Indo-Pacific Oceans
      iwci(ro1+2)=.5d+00*iwcitt
      iwci(ro1+4)=iwcitt-iwci(ro1+2)
      swci(ro1+2)=.5d+00*swcitt
      swci(ro1+4)=swcitt-swci(ro1+2)

      iwc13i = (c13a-20.D+00)
      swc13i = swcitc/swcitt * ((c13a-20.D+00)+c13rk)/2.0 +
     >         swcits/swcitt * c13rk
      iwc14i = (c14a-40.D+00)
      swc14i = swcitc/swcitt * ((c14a-40.D+00)+c14rk)/2.0 +
     >         swcits/swcitt * c14rk

!5- Carry to resulting fluxes (dissolved) into dic_i, alk_i ...)
      dic_w(:    ,:) = 0D0
      dic_w(ro1+2,1) = iwci(ro1+2)        + swci(ro1+2)
      dic_w(ro1+2,2) = iwci(ro1+2)*iwc13i + swci(ro1+2)*swc13i
      dic_w(ro1+2,3) = iwci(ro1+2)*iwc14i + swci(ro1+2)*swc14i
      dic_w(ro1+4,1) = iwci(ro1+4)        + swci(ro1+4)
      dic_w(ro1+4,2) = iwci(ro1+4)*iwc13i + swci(ro1+4)*swc13i
      dic_w(ro1+4,3) = iwci(ro1+4)*iwc14i + swci(ro1+4)*swc14i

      dic_i(ro1+1:ro1+nro1,:) = dic_i(ro1+1:ro1+nro1,:)
     &                       +  dic_w(:             ,:)

!5- Organic matter input: C oxidised as CO2 into the atmosphere,
!   P in Redfield (or appropriate) ratio into surface ocean,
!   ALK input into surface ocean accordingly
      po4_r(:) = 0D0
#ifdef LOOP_ORGMATTER
      po4_r(ro1+2) = .5D+00*orco2t*rhopc
      po4_r(ro1+4) = orco2t*rhopc - po4_r(ro1+2)
#endif

!5- PO4 input by weathering
      po4_w(:) = 0D0
      po4_w(ro1+2) = .5D+00*rwpo4t
      po4_w(ro1+4) = rwpo4t - po4_w(ro1+2)

      po4_i(ro1+1:ro1+nro1) = po4_i(ro1+1:ro1+nro1)
     &                         + po4_r(:) + po4_w(:)

!5- Weathering products bring in as many equivalents of alkalinity
!   as they bring moles of bicarbonate for carbonate and igneous
!   rock weathering
      alk_i(ro1+1:ro1+nro1)
     &    = alk_i(ro1+1:ro1+nro1)
     &      + dic_w(:,1)
     &      + (po4_r(:) + po4_w(:)) * (-1D0-(rhonc/rhopc))

!6- Book coral-reef and shelf and bank outputs into carb_o1
!   Now that the transfer to the seafloor has been taken care of,
!   complete the outputs from ro1+2&4 by coral-reef and shelf parts.

      ir = ro1+2
      carb_o1(ir,1) = carb_o1(ir,1) + (coralo(ir)+shbnko(ir))
      carb_o1(ir,2) = carb_o1(ir,2) + (coralo(ir)+shbnko(ir))*cc13(ir)
      carb_o1(ir,3) = carb_o1(ir,3) + (coralo(ir)+shbnko(ir))*cc14(ir)

      ir = ro1+4
      carb_o1(ir,1) = carb_o1(ir,1) + (coralo(ir)+shbnko(ir))
      carb_o1(ir,2) = carb_o1(ir,2) + (coralo(ir)+shbnko(ir))*cc13(ir)
      carb_o1(ir,3) = carb_o1(ir,3) + (coralo(ir)+shbnko(ir))*cc14(ir)


!7- Continental biospheric exchange forcings
      IF ((cbcinp) .AND. (biof01)) THEN
         CALL XCB(temps, c13a, co2xcb, c13xcb, po4xcb)
      ELSE
         co2xcb=0.0D+00
         c13xcb=0.0D+00
         po4xcb=0.0D+00
      ENDIF

!7- Phosphate:
!  * if released from the continental biosphere, put it 1/2 and 1/2
!    into the surface temperate Atlantic and Indo-Pacific Reservoirs
!    Introduce the correct (Redfield) amount of alkalinity as well
!  * if taken up from the continental biosphere, take it from the
!    lithosphere (no impact on ocean)

      IF(po4xcb >= 0) THEN
        po4_i(ro1+2) = po4_i(ro1+2) + po4xcb/2D0
        po4_i(ro1+4) = po4_i(ro1+4) + (po4xcb - po4xcb/2D0)
        alk_i(ro1+2) = alk_i(ro1+2)
     &                 + (po4xcb/2D0)*(-1D0-rhonc/rhopc)
        alk_i(ro1+4) = alk_i(ro1+4)
     &                 + (po4xcb - po4xcb/2D0)*(-1D0-rhonc/rhopc)
      ELSE
        po4xcb = 0D0  ! to have adequate output to file
      ENDIF

!8- If secmem is called with supvis=8 or 16, then store the calculated
!   fluxes at interfaces into MOD_MBM_O2S
      supvis8: IF((supvis == 8) .OR. (supvis == 16)) THEN

c~ !8- Redfield composition (CH2O)106 (NH3)16 (H3PO4)
c~       mbm_om_c =   106D0
c~       mbm_om_n =    16D0
c~       mbm_om_p =     1D0
c~       mbm_om_o =  (    mbm_om_c + 4D0*mbm_om_p)
c~       mbm_om_h =  (2D0*mbm_om_c + 3D0*mbm_om_n +  3D0*mbm_om_p)
c~       mbm_om_ro2 = 138D0

!8- Temp here in Kelvin; Medusa needs it in degC; conversion done
!   by OCEAN-TO-SEDIMENT (from MOD_MBM_COUPLED from Medusa)
      
      DO i = 1, nro1
        ir = ro1+i
        SELECT CASE(i)
          CASE(1)
            i2 = ir
            i3 = ro3+1
          CASE(2)
            i2 = ro2+1
            i3 = ro3+1
          CASE(3)
            i2 = ir
            i3 = ro3+2
          CASE(4)
            i2 = ro2+2
            i3 = ro3+3
          CASE(5)
            i2 = ir
            i3 = ro3+3
        END SELECT

        DO j = 1, id0
          mbm_fa(j,:) = 0D0
        ENDDO

        j = id0+1
        mbm_fa(j,1) = DBLE(id0+1)-d0
        mbm_fa(j,2) = 0D0
        mbm_fa(j,3) = 0D0

        DO j =id0+2, id100
          mbm_fa(j,1) = 1D0
          mbm_fa(j,2) = 0D0
          mbm_fa(j,3) = 0D0
        ENDDO

      ! Initialize to have values everywhere
        DO j = 1, id100
          mbm_temp(i,j) =  temp(ir)
          mbm_sali(i,j) = salin(ir)
          mbm_dic (i,j) =   cit(ir)
          mbm_alk (i,j) =   alk(ir)
          mbm_oxyg(i,j) =   oxy(ir)
        ENDDO

        j = id100+1
        mbm_fa(j,1) =  d100-DBLE(id100)
        mbm_fa(j,2) =  1D0 - mbm_fa(j,1)
        mbm_fa(j,3) = 0D0
        
        mbm_temp(i,j) =  temp(ir)*mbm_fa(j,1) +  temp(i2)*mbm_fa(j,2)
        mbm_sali(i,j) = salin(ir)*mbm_fa(j,1) + salin(i2)*mbm_fa(j,2)
        mbm_dic (i,j) =   cit(ir)*mbm_fa(j,1) +   cit(i2)*mbm_fa(j,2)
        mbm_alk (i,j) =   alk(ir)*mbm_fa(j,1) +   alk(i2)*mbm_fa(j,2)
        mbm_oxyg(i,j) =   oxy(ir)*mbm_fa(j,1) +   oxy(i2)*mbm_fa(j,2)

        DO j = id100+2, id1000
          mbm_temp(i,j) =  temp(i2)
          mbm_sali(i,j) = salin(i2)
          mbm_dic (i,j) =   cit(i2)
          mbm_alk (i,j) =   alk(i2)
          mbm_oxyg(i,j) =   oxy(i2)
          mbm_fa(j,1)   =  0D0
          mbm_fa(j,2)   =  1D0
          mbm_fa(j,3)   =  0D0
        ENDDO
        
        j = id1000+1
        mbm_fa(j,1) = 0D0
        mbm_fa(j,2) = d1000-DBLE(id1000) 
        mbm_fa(j,3) = 1D0 - mbm_fa(j,2)
        mbm_temp(i,j) =  temp(i2)*mbm_fa(j,2) +  temp(i3)*mbm_fa(j,3)
        mbm_sali(i,j) = salin(i2)*mbm_fa(j,2) + salin(i3)*mbm_fa(j,3)
        mbm_dic (i,j) =   cit(i2)*mbm_fa(j,2) +   cit(i3)*mbm_fa(j,3)
        mbm_alk (i,j) =   alk(i2)*mbm_fa(j,2) +   alk(i3)*mbm_fa(j,3)
        mbm_oxyg(i,j) =   oxy(i2)*mbm_fa(j,2) +   oxy(i3)*mbm_fa(j,3)

        DO j = id1000+2, bottom
          mbm_fa(j,1) = 0D0
          mbm_fa(j,2) = 0D0
          mbm_fa(j,3) = 1D0
          mbm_temp(i,j) =  temp(i3)
          mbm_sali(i,j) = salin(i3)
          mbm_dic (i,j) =   cit(i3)
          mbm_alk (i,j) =   alk(i3)
          mbm_oxyg(i,j) =   oxy(i3)
        ENDDO
      ENDDO

!      mbm_fa(:,1) = 0D0
!      mbm_fa(:,2) = 0D0
!      mbm_fa(:,3) = 1D0


      IF(supvis == 8) THEN
        DO i = 1, nro1
          WHERE(mbm_surf(i,:) /= 0D0)
!DEL        mbm_arag(i,:) = arag2s(  ro1+i, 1)/sfc200(ro1+i)
            mbm_arag(i,:) = arag2s(:, ro1+i, 1)/mbm_surf(i,:)
             ! 10^18 mol C/m2/yr

!DEL            mbm_calc(i,:) = calc2s(  ro1+i, 1)/sfc200(ro1+i)
            mbm_calc(i,:) = calc2s(:, ro1+i, 1)/mbm_surf(i,:)
             ! 10^18 mol C/m2/yr

            mbm_orgm(i,:) = orgm2s(:, ro1+i, 1)/mbm_surf(i,:)
             ! 10^18 mol C/m2/yr

            mbm_clay(i,:) = othr2s(:, ro1+i)/mbm_surf(i,:)
!           ! 10^18 kg/m2/yr
          ELSEWHERE
            mbm_calc(i,:) = 0D0
            mbm_arag(i,:) = 0D0
            mbm_orgm(i,:) = 0D0
            mbm_clay(i,:) = 0D0
          END WHERE
        ENDDO
      ELSE
        DO i = 1, nro1
          WHERE(mbm_surf(i,:) /= 0D0)
!DEL        mbm_arag(i,:) = (st_arag2s(  ro1+i)/epoch)/sfc200(ro1+i)
            mbm_arag(i,:) = (st_arag2s(:, ro1+i)/epoch)/mbm_surf(i,:)
             ! 10^18 mol C/m2/yr

!DEL        mbm_calc(i,:) = (st_calc2s(  ro1+i)/epoch)/sfc200(ro1+i)
            mbm_calc(i,:) = (st_calc2s(:, ro1+i)/epoch)/mbm_surf(i,:)
             ! 10^18 mol C/m2/yr

            mbm_orgm(i,:) = (st_orgm2s(:, ro1+i)/epoch)/mbm_surf(i,:)
            ! 10^18 mol C/m2/yr

            mbm_clay(i,:) = (st_othr2s(:, ro1+i)/epoch)/mbm_surf(i,:)
           ! 10^18 kg/m2/yr
          ELSEWHERE
            mbm_calc(i,:) = 0D0
            mbm_arag(i,:) = 0D0
            mbm_orgm(i,:) = 0D0
            mbm_clay(i,:) = 0D0
          END WHERE
        ENDDO
      ENDIF

      ! Correct for shallow regions
      ! In columns shallower than the one covered by the 200m depth
      ! horizon, the carbonate fluxes are set to zero;
      ! Only the clay flux is kept
      ! For the column covered by the 200m depth horizon,
      ! the fraction of the flux above the 200m depth horizon
      ! is discarded, and the fraction below the 200m depth horizon
      ! is evenly spread
!DEL      mbm_arag(:,1:id200) = 0D0
!DEL      mbm_arag(:,id200+1) = mbm_arag(:,id200+1)*(id200+1-d200)
!DEL      mbm_calc(:,1:id200) = 0D0
!DEL      mbm_calc(:,id200+1) = mbm_calc(:,id200+1)*(id200+1-d200)

      mbm_o2s_id  = mbm_o2s_id+1

! volctt = volcsi - volcca
! Average(iwcitt) =  2*volctt
! Average(swcits) =  2*volcca
      tc_in = volctt + swcitc/2.d+00 + swcits
      tc_out = SUM(   coralo + shbnko 
     &              + SUM(orgm2s(:,:,1),1)
     &              + SUM(calc2s(:,:,1),1)
     &              + SUM(arag2s(:,:,1),1)
     &            )

      ta_in = iwcitt + swcitt
      ta_out =  SUM(  coralo*2D0 + shbnko*2D0
     &              + SUM(orgm2s(:,:,1),1)*(-rhopc-rhonc)
     &              + SUM(calc2s(:,:,1),1)*2D0
     &              + SUM(arag2s(:,:,1),1)*2D0
     &             )
      ! additional contributions from the sediment/ocean exchange
      ! to be added
      ENDIF supvis8

      IF ((supvis.EQ.1) .OR. (supvis.EQ.2)) THEN
         totcar = SUM(vcit) + qteco2(ra1+1)*pco2a
         totalk = SUM(valk)
         totpo4 = SUM(vpo4)
         totoxy = SUM(voxy)
         totc13 = SUM(cc13*vcit) + qteco2(ra1+1)*pco2a*c13a
         totc14 = SUM((cc14-2.0D+00*(cc13+25.D+00)*(1.D+00+cc14*1.D-3))
     >                *vcit) ! Delta_C14(atm) = 0

      IF ((supvis.NE.2) .AND. (resusd .NE. 0)) THEN
         WRITE(resuni,7) 'Total Carbon :',    totcar*1.D+03
         WRITE(resuni,7) 'Total Alkalinity :',totalk*1.D+03
         WRITE(resuni,7) 'Total Phosphate :', totpo4*1.0D+06
         WRITE(resuni,7) 'Total Oxygen :',    totoxy*1.0D+03
         WRITE(resuni,7) 'Mean DC13 :',       totc13/totcar
         WRITE(resuni,*) ' '
         WRITE(resuni,*) ' '
         WRITE(resuni,'(A14,4A8)')
     >      'Atmosphere :','pCO2','DC13','DC14','DDC14'
         WRITE(resuni,'(14X,4F8.2)')
     >   pco2a*331.0D+00,c13a,c14a,
     >            c14a-2.0D+00*(c13a+25.D+00)*(1.D+00+c14a*1.D-3)
         WRITE(resuni,*) ' '
         WRITE(resuni,9) 'SNATL','SEQATL','SANT','SEQI-P ','SNPAC',
     >              'TEQATL','TEQI-P','DATL','DANT','DI-P'
         WRITE(resuni,8) '[CIT]',cit
         WRITE(resuni,8) '[ALK]',alk
         WRITE(resuni,8) '[CO3=]',(co3(i)*1.0d+03,i=ro1+1,ro3+nro3)
         WRITE(resuni,8) '[OXY]',oxy
         WRITE(resuni,8) '[PO4]',(po4(i)*1.0D+03,i=ro1+1,ro3+nro3)
         WRITE(resuni,8) 'DC13',cc13
         WRITE(resuni,8) 'DC14',cc14
         WRITE(resuni,8) 'DDC14',(cc14(i)
     >    -2.0D+00*(cc13(i)+25.0D+00)*(1.0D+00+cc14(i)*1.0D-3),
     >                                         i=ro1+1,ro3+nro3)
         WRITE(resuni,8) 'pH',ph
         WRITE(resuni,8) 'SpCO2',(pco2(i)*331.D+00,i=ro1+1,ro1+nro1)
         WRITE(resuni,*) ' '
         WRITE(resuni,7) 'CO2 exchange Atm->Oc :',
     >          ((gfco2i(i)-gfco2o(i))*1.D+06,i=ro1+1,ro1+nro1),
     >            SUM(gfco2i-gfco2o) * 1.0D+06
         WRITE(resuni,'(A)') txt_orgm_flx(i_geohor_slv+1)
         WRITE(resuni,'(A)') txt_orgm_flx(i_geohor_dps  )
         WRITE(resuni,'(A)') txt_orgm_sed(i_geolev_top+1)
         WRITE(resuni,'(A)') txt_orgm_sed(i_geolev_bot  )
         WRITE(resuni,'(A)') txt_orgm_rmn(i_geolev_top+1)
         WRITE(resuni,7) 'O2 WCResp  100m-1000m:',
     >        csmoxy(ro1+1)*1.0D+06,csmoxy(ro2+1)*1.0D+06,
     >        csmoxy(ro1+3)*1.0D+06,csmoxy(ro2+2)*1.0D+06,
     >        csmoxy(ro1+5)*1.0D+06,
     >       (csmoxy(ro1+1)+csmoxy(ro2+1)+csmoxy(ro1+3)+
     >        csmoxy(ro2+2)+csmoxy(ro1+5))* 1.0D+06
         WRITE(resuni,'(A)') txt_orgm_rmn(i_geolev_bot  )
         WRITE(resuni,7) 'O2 WCResp 1000m-8000m:',
     >        0D0,csmoxy(ro3+1)*1.0D+06,
     >        csmoxy(ro3+2)*1.0D+06,csmoxy(ro3+3)*1.0D+06,
     >        0D0,
     >       (csmoxy(ro3+1)+csmoxy(ro3+2)
     >                     +csmoxy(ro3+3))* 1.0D+06
         WRITE(resuni,7) 'Inorganic Flx at 100m:',
     >        (calc_o1(:,1)+arag_o1(:,1)) * 1.0D+06,
     >        SUM(calc_o1(:,1)+arag_o1(:,1)) * 1.0D+06
         WRITE(resuni,7) 'CORALO :',
     >        coralo(:) * 1.0D+6,
     >        SUM(coralo(:)) * 1.0D+06
         WRITE(resuni,7) 'SHBNKO :',
     >        shbnko(:) * 1.D+6,
     >        SUM(shbnko(:)) * 1.0D+06
         WRITE(resuni,7) 'Riverine HCO3- input :',
     >        dic_w(:,1) * 1.D+6,
     >        SUM(dic_w(:,1)) * 1.0D+06
         WRITE(resuni,7) 'Volc CO2 (Sil, Car, Tot):',
     >              volcsi*1.0D+6, volcca*1.0D+6, volctt*1.0D+6

 7       FORMAT(A25,5F8.2,2X,F8.2)
 8       FORMAT(A7,10F9.3)
 9       FORMAT(7X,10A9)
      ENDIF
      ENDIF

      taumin_oca = 1D+6
C
C     ===  EQUATIONS  ==================================================
C
C
C     ---  ATMOSPHERIC RESERVOIR  --------------------------------------
C
c     Variations in PAL/year
      dpco2a= (tdiffa-pco2a)/tau + 
     &        ( co2xcb
!     &         +permafrostco2
     &         +volctt
     &         +orco2t
     &         +owco2t
     &         -iwcitt-swcitc/2.d+00
     &        )/qteco2(ra1+1)

      total_out = (pco2a/tau+iwcitt+swcitc/2.d+00)/qteco2(ra1+1)
      IF(total_out > pco2a*taumin_oca) THEN
        taumin_oca=pco2a/total_out
        taumin_oca_var  = 'pCO2'
        taumin_oca_ires = ra1+1
      ENDIF

      dc13a= ( atc13i(ra1+1)-atc13o(ra1+1)
     &        +( c13xcb
!     &          +permafrostco2_c13
     &          +volctt*c13rk
     &          +orco2t*(-25.0D0)
     &          +owco2t*(-25.0D0)
     &          -(iwcitt+swcitc/2.d+00)*(c13a-20.0D+00)
     &         )/qteco2(ra1+1)
     &        -dpco2a*c13a
     &       ) / pco2a
! Force Delta C14 = 0:
      dc14a=0.D+00
      c14a=2D0*(c13a+25D0)/(1D0-2D0*(c13a+25D0)/1D3)
C
C
C     ---  SURFACE RESERVOIRS  -----------------------------------------
C
      DO 100 i=ro1+1,ro1+nro1
         dvpo4(i)=  wfpo4i(i) - wfpo4o(i)
     &              - csmpo4(i) + rcypo4(i) !DEL + xcbpo4(i)
     &              + po4_i(i) - po4_o(i)

         total_out = wfpo4o(i)
     &              + (csmpo4(i)-rcypo4(i))
     &              + po4_o(i)
         IF(total_out > vpo4(i)*taumin_oca) THEN
           taumin_oca=vpo4(i)/total_out
           taumin_oca_var  = ' PO4'
           taumin_oca_ires = i
         ENDIF


         dvcit(i)=  wfciti(i) - wfcito(i)
     &              + gfco2i(i) - gfco2o(i)
     &              -(orgm_x(i,1) - orgm_r(i,1))
     &              - carb_o1(i,1)
     &              + dic_i(i,1) - dic_o(i,1)

         total_out = wfcito(i)+gfco2o(i) 
     &              + (orgm_x(i,1) - orgm_r(i,1))
     &              + carb_o1(i,1) + dic_o(i,1)
         IF(total_out > vcit(i)*taumin_oca) THEN
           taumin_oca=vcit(i)/total_out
           taumin_oca_var  = ' DIC'
           taumin_oca_ires = i
         ENDIF


         dvoxy(i)= 0.D+00


         dvalk(i)=  wfalki(i)-wfalko(i)
     &              -(orgm_x(i,1) - orgm_r(i,1))*(-rhopc-rhonc)
     &              - carb_o1(i,1)* 2D0
     &              + alk_i(i) - alk_o(i)

         total_out = wfalko(i)
     &              + carb_o1(i,1)* 2D0 + alk_o(i)
         IF(total_out > valk(i)*taumin_oca) THEN
           taumin_oca=valk(i)/total_out
           taumin_oca_var  = 'TALK'
           taumin_oca_ires = i
         ENDIF


c         dcc13(i)= (aqc13i(i)-aqc13o(i)
c     &              +cori(i,2)-coro(i,2)
c     &              +cini(i,2)-cino(i,2)
c     &               -dvcit(i)*cc13(i)) / vcit(i)
c C13 Equation needs to be reformulated by caller !!!
         dcc13(i)=   aqc13i(i)-aqc13o(i)
     &              -(orgm_x(i,2) - orgm_r(i,2))
     &              - carb_o1(i,2)
     &              + dic_i(i,2) - dic_o(i,2)
c         dcc14(i)= (aqc14i(i)-aqc14o(i)
c     &               +cori(i,3)-coro(i,3)
c     &               +cini(i,3)-cino(i,3)
c     &               -dvcit(i)*cc14(i)) / vcit(i)
c     &                 - lmdc14*(1000.d+00+cc14(i))
c C14 Equation needs to be reformulated by caller !!!
         dcc14(i)=  aqc14i(i)-aqc14o(i)
     &              -(orgm_x(i,3) - orgm_r(i,3))
     &              - carb_o1(i,3)
     &              + dic_i(i,3) - dic_o(i,3)
  100 CONTINUE
C
C     ---  INTERMEDIATE RESERVOIRS AND
C          DEEP RESERVOIRS ---------------------------------------------
C
      DO 200 i=ro2+1,ro2+nro2+nro3
        dvpo4(i)= wfpo4i(i)-wfpo4o(i)
     &            + rcypo4(i)
     &            + po4_i(i) - po4_o(i)

        total_out = wfpo4o(i)+po4_o(i)
        IF(total_out > vpo4(i)*taumin_oca) THEN
           taumin_oca=vpo4(i)/total_out
           taumin_oca_var  = ' PO4'
           taumin_oca_ires = i
         ENDIF


        dvcit(i)= wfciti(i)-wfcito(i)
     &            + orgm_r(i,1)
     &            + calc_d(i,1) + arag_d(i,1)
     &            + dic_i(i,1) - dic_o(i,1)

        total_out = wfcito(i) + dic_o(i,1)
        IF(total_out > vcit(i)*taumin_oca) THEN
           taumin_oca=vcit(i)/total_out
           taumin_oca_var  = ' DIC'
           taumin_oca_ires = i
         ENDIF


        dvoxy(i)= wfoxyi(i)-wfoxyo(i)
     &            - csmoxy(i)
     &            + oxy_i(i) - oxy_o(i)

        total_out = wfoxyo(i) + csmoxy(i)
     &              + oxy_o(i)
        IF(total_out > voxy(i)*taumin_oca) THEN
           taumin_oca=voxy(i)/total_out
           taumin_oca_var  = ' OXY'
           taumin_oca_ires = i
         ENDIF


        dvalk(i)= wfalki(i)-wfalko(i)
     &            + orgm_r(i,1)*(-rhopc-rhonc)
     &            + (calc_d(i,1) + arag_d(i,1))*2D0
     &            + (alk_i(i) - alk_o(i))

        total_out = wfalko(i)
     &            - orgm_r(i,1)*(-rhopc-rhonc)
     &            + alk_o(i)
        IF(total_out > valk(i)*taumin_oca) THEN
           taumin_oca=valk(i)/total_out
           taumin_oca_var  = 'TALK'
           taumin_oca_ires = i
         ENDIF


c        dcc13(i)= (aqc13i(i)-aqc13o(i)
c     &                +cori(i,2)-coro(i,2)
c     &                +cini(i,2)-cino(i,2)
c     &                -dvcit(i)*cc13(i)) / vcit(i)
c C13 Equation needs to be reformulated by caller !!!
        dcc13(i)= aqc13i(i)-aqc13o(i)
     &            + orgm_r(i,2)
     &            + calc_d(i,2) + arag_d(i,2)
     &            + dic_i(i,2) - dic_o(i,2)

c        dcc14(i)= (aqc14i(i)-aqc14o(i)
c     &                +cori(i,3)-coro(i,3)
c     &                +cini(i,3)-cino(i,3)
c     &                -dvcit(i)*cc14(i)) / vcit(i)
c     &                 - lmdc14*(1000.d+00+cc14(i))
c C13 Equation needs to be reformulated by caller !!!
        dcc14(i)=  aqc14i(i)-aqc14o(i)
     &            + orgm_r(i,3)
     &            + calc_d(i,3) + arag_d(i,3)
     &            + dic_i(i,3) - dic_o(i,3)
  200 CONTINUE

      dtyoca = dtxoca


C currently mixed stuff
#ifndef MBM_WITH_NETCDF
      IF ((supvis.EQ.2) .AND. (cslusd .NE. 0)) THEN
         WRITE(csluni,'(31E14.6)') temps, pco2a*331.D+00,
     &    swcitc, swcits, iwcitt, totcar, totalk, totpo4,
     &    totc13/totcar, cit, alk, co2xcb, po4xcb
      ENDIF

      IF ((supvis.EQ.4) .AND. (cslusd.NE. 0)) THEN
         WRITE(csluni,'(8E14.6)') SNGL(temps), SNGL(-1.0d+00),
     &         SNGL(swcitc),SNGL(swcits),SNGL(iwcitt),
     &         SNGL(swcitc+iwcitt), SNGL(volcsi), SNGL(volcca)
      ENDIF
#else
      IF ((supvis.EQ.2) .AND. (cslusd .NE. 0)) THEN
         CALL EQNOCE_WRITE_CSL_NCFILE
      ENDIF

      IF ((supvis.EQ.2) .AND. (cmsusd .NE. 0)) THEN
         CALL EQNOCE_WRITE_CMS_NCFILE
      ENDIF
#endif

c     If we got here, everything is fine!
      iflag=0
      RETURN


      CONTAINS


!        ----------------------------------
         SUBROUTINE EQNOCE_WRITE_CSL_NCFILE
!        ----------------------------------

         IMPLICIT NONE

#ifdef MBM_WITH_NETCDF
#include "netcdf.inc"

         INTEGER :: i_ncvar
         INTEGER :: i_stat
         INTEGER, DIMENSION(2) :: i_start, n_count
         DOUBLE PRECISION                            :: value
         DOUBLE PRECISION, DIMENSION(nro1+nro2+nro3) :: value_ro
         DOUBLE PRECISION, DIMENSION(nro1)           :: value_ro1
         DOUBLE PRECISION, DIMENSION(nro3)           :: value_ro3



         i_start(1) =    1
         i_start(2) = iccf

         i_ncvar = nc_cslvarid(nc_cslidx_pco2a)
         value = pco2a*331.D+00
         i_stat = NF_PUT_VAR1_DOUBLE(nc_cslfilid,
     &                               i_ncvar, i_start(2), value)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)

         i_ncvar = nc_cslvarid(nc_cslidx_tic)
         value = (totcar-qteco2(ra1+1)*pco2a)*1D3
         i_stat = NF_PUT_VAR1_DOUBLE(nc_cslfilid,
     &                               i_ncvar, i_start(2), value)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)

         i_ncvar =  nc_cslvarid(nc_cslidx_tta)
         value = totalk*1D3
         i_stat = NF_PUT_VAR1_DOUBLE(nc_cslfilid,
     &                              i_ncvar, i_start(2), value)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)

         i_ncvar = nc_cslvarid(nc_cslidx_tpo)
         value = totpo4*1D3
         i_stat = NF_PUT_VAR1_DOUBLE(nc_cslfilid,
     &                               i_ncvar, i_start(2), value)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)

         i_ncvar = nc_cslvarid(nc_cslidx_to2)
         value = totoxy*1D3
         i_stat = NF_PUT_VAR1_DOUBLE(nc_cslfilid,
     &                              i_ncvar , i_start(2), value)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)

         i_ncvar = nc_cslvarid(nc_cslidx_tc13)
         value =  (totc13-qteco2(ra1+1)*pco2a*c13a)
     &           /(totcar-qteco2(ra1+1)*pco2a)
         i_stat = NF_PUT_VAR1_DOUBLE(nc_cslfilid,
     &                               i_ncvar, i_start(2), value)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)

         i_ncvar = nc_cslvarid(nc_cslidx_tc14)
         value = totc14/(totcar-qteco2(ra1+1)*pco2a)
         i_stat = NF_PUT_VAR1_DOUBLE(nc_cslfilid,
     &                               i_ncvar, i_start(2), value)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)

         i_ncvar = nc_cslvarid(nc_cslidx_swc)
         value = swcitc*1D6
         i_stat = NF_PUT_VAR1_DOUBLE(nc_cslfilid,
     &                               i_ncvar, i_start(2), value)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)
      
         i_ncvar = nc_cslvarid(nc_cslidx_sws)
         value = swcits*1D6
         i_stat = NF_PUT_VAR1_DOUBLE(nc_cslfilid,
     &                               i_ncvar, i_start(2), value)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)
      
         i_ncvar = nc_cslvarid(nc_cslidx_iwt)
         value = iwcitt*1D6
         i_stat = NF_PUT_VAR1_DOUBLE(nc_cslfilid,
     &                               i_ncvar, i_start(2), value)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)

         i_ncvar = nc_cslvarid(nc_cslidx_vot)
         value = volctt*1D6
         i_stat = NF_PUT_VAR1_DOUBLE(nc_cslfilid,
     &                               i_ncvar, i_start(2), value)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)

         i_ncvar = nc_cslvarid(nc_cslidx_ort)
         value = orco2t*1D6
         i_stat = NF_PUT_VAR1_DOUBLE(nc_cslfilid,
     &                               i_ncvar, i_start(2), value)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)

         i_ncvar = nc_cslvarid(nc_cslidx_owt)
         value = owco2t*1D6
         i_stat = NF_PUT_VAR1_DOUBLE(nc_cslfilid,
     &                               i_ncvar, i_start(2), value)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)

         i_ncvar = nc_cslvarid(nc_cslidx_pwt)
         value = rwpo4t*1D6
         i_stat = NF_PUT_VAR1_DOUBLE(nc_cslfilid,
     &                               i_ncvar, i_start(2), value)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)

         i_ncvar = nc_cslvarid(nc_cslidx_cxb)
         value = co2xcb*1D6
         i_stat = NF_PUT_VAR1_DOUBLE(nc_cslfilid,
     &                               i_ncvar, i_start(2), value)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)

         i_ncvar = nc_cslvarid(nc_cslidx_pxb)
         value = po4xcb*1D6
         i_stat = NF_PUT_VAR1_DOUBLE(nc_cslfilid,
     &                               i_ncvar, i_start(2), value)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)


         n_count(1) = nro1
         n_count(2) =    1

         i_ncvar = nc_cslvarid(nc_cslidx_omxp)
         value_ro1 = orgm_x(:,1)*1D6
         i_stat = NF_PUT_VARA_DOUBLE(nc_cslfilid,
     &                               i_ncvar, i_start, n_count,
     &                               value_ro1)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)
      
         i_ncvar = nc_cslvarid(nc_cslidx_ca2s)
         value_ro1 = SUM(calc2s(:,:,1),1)*1D6
         i_stat = NF_PUT_VARA_DOUBLE(nc_cslfilid,
     &                               i_ncvar, i_start, n_count,
     &                               value_ro1)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)
      
         i_ncvar = nc_cslvarid(nc_cslidx_ar2s)
         value_ro1 = SUM(arag2s(:,:,1),1)*1D6
         i_stat = NF_PUT_VARA_DOUBLE(nc_cslfilid,
     &                               i_ncvar, i_start, n_count,
     &                               value_ro1)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)
      
         i_ncvar = nc_cslvarid(nc_cslidx_om2s)
         value_ro1 = SUM(orgm2s(:,:,1),1)*1D6
         i_stat = NF_PUT_VARA_DOUBLE(nc_cslfilid,
     &                               i_ncvar, i_start, n_count,
     &                               value_ro1)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)
      
         i_ncvar = nc_cslvarid(nc_cslidx_ot2s)
         value_ro1 = SUM(othr2s(:,:),1)*1D6
         i_stat = NF_PUT_VARA_DOUBLE(nc_cslfilid,
     &                               i_ncvar, i_start, n_count,
     &                               value_ro1)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)
      
         i_ncvar = nc_cslvarid(nc_cslidx_crlo)
         value_ro1 = coralo*1D6
         i_stat = NF_PUT_VARA_DOUBLE(nc_cslfilid,
     &                               i_ncvar, i_start, n_count,
     &                               value_ro1)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)
      
         i_ncvar = nc_cslvarid(nc_cslidx_shbo)
         value_ro1 = shbnko*1D6
         i_stat = NF_PUT_VARA_DOUBLE(nc_cslfilid,
     &                               i_ncvar, i_start, n_count,
     &                               value_ro1)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)



         i_stat = NF_SYNC(nc_cslfilid)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)

#endif

         RETURN

!        --------------------------------------
         END SUBROUTINE EQNOCE_WRITE_CSL_NCFILE
!        --------------------------------------


!        ----------------------------------
         SUBROUTINE EQNOCE_WRITE_CMS_NCFILE
!        ----------------------------------

         IMPLICIT NONE

#ifdef MBM_WITH_NETCDF
#include "netcdf.inc"

         INTEGER :: i_ncvar
         INTEGER :: i_stat
         INTEGER, DIMENSION(2) :: i_start, n_count
         DOUBLE PRECISION                            :: value
         DOUBLE PRECISION, DIMENSION(nro1+nro2+nro3) :: value_ro


         i_start(1) =    1
         i_start(2) = iccf


! Sea-level
         i_ncvar = nc_cmsvarid(nc_cmsidx_slv)
         i_stat = NF_PUT_VAR1_DOUBLE(nc_cmsfilid,
     &                               i_ncvar, i_start(2), nivmer)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)

! Aragonite Fraction (previously Climate Forcing 1)
         i_ncvar = nc_cmsvarid(nc_cmsidx_ff1)
         value = frcara
         i_stat = NF_PUT_VAR1_DOUBLE(nc_cmsfilid,
     &                               i_ncvar, i_start(2), value)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)

! Atmospheric pCO2
         i_ncvar = nc_cmsvarid(nc_cmsidx_pco2a)
         value = pco2a*331.D+00
         i_stat = NF_PUT_VAR1_DOUBLE(nc_cmsfilid,
     &                               i_ncvar, i_start(2), value)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)
   
! Atmospheric delta C-13
         i_ncvar = nc_cmsvarid(nc_cmsidx_dc13a)
         i_stat = NF_PUT_VAR1_DOUBLE(nc_cmsfilid,
     &                               i_ncvar, i_start(2), c13a)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)
   
! Atmospheric Delta C-14
         i_ncvar = nc_cmsvarid(nc_cmsidx_dc14a)
         value = c14a-2.0D+00*(c13a+25.D+00)*(1.D+00+c14a*1.D-3)
         i_stat = NF_PUT_VAR1_DOUBLE(nc_cmsfilid,
     &                               i_ncvar, i_start(2), value)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)


         n_count(1) = nro1+nro2+nro3
         n_count(2) = 1

! Box volumes
         i_ncvar = nc_cmsvarid(nc_cmsidx_vols)
         value_ro = volum(:)
         i_stat = NF_PUT_VARA_DOUBLE(nc_cmsfilid,
     &                               i_ncvar,
     &                               i_start, n_count, volum)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)

! Temperatures
         i_ncvar = nc_cmsvarid(nc_cmsidx_temp)
         i_stat = NF_PUT_VARA_DOUBLE(nc_cmsfilid,
     &                               i_ncvar,
     &                               i_start, n_count, temp)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)

! Salinities
         i_ncvar = nc_cmsvarid(nc_cmsidx_salin)
         i_stat = NF_PUT_VARA_DOUBLE(nc_cmsfilid,
     &                               i_ncvar,
     &                               i_start, n_count, salin)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)

! Phosphate
         i_ncvar = nc_cmsvarid(nc_cmsidx_po4)
         i_stat = NF_PUT_VARA_DOUBLE(nc_cmsfilid,
     &                               i_ncvar,
     &                               i_start, n_count, po4)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)

! Oxygen
         i_ncvar = nc_cmsvarid(nc_cmsidx_oxy)
         i_stat = NF_PUT_VARA_DOUBLE(nc_cmsfilid,
     &                               i_ncvar,
     &                               i_start, n_count, oxy)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)

! DIC
         i_ncvar = nc_cmsvarid(nc_cmsidx_dic)
         i_stat = NF_PUT_VARA_DOUBLE(nc_cmsfilid,
     &                               i_ncvar,
     &                               i_start, n_count, cit)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)

! Alkalinity
         i_ncvar = nc_cmsvarid(nc_cmsidx_alk)
         i_stat = NF_PUT_VARA_DOUBLE(nc_cmsfilid,
     &                               i_ncvar,
     &                               i_start, n_count, alk)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)

! DIC delta C-13
         i_ncvar = nc_cmsvarid(nc_cmsidx_dc13)
         i_stat = NF_PUT_VARA_DOUBLE(nc_cmsfilid,
     &                               i_ncvar,
     &                               i_start, n_count, cc13)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)

! DIC Delta C-14
         i_ncvar = nc_cmsvarid(nc_cmsidx_dc14)
         value_ro = cc14(:)
     &    -2.0D+00*(cc13(:)+25.0D+00)*(1.0D+00+cc14(:)*1.0D-3)
         i_stat = NF_PUT_VARA_DOUBLE(nc_cmsfilid,
     &                               i_ncvar,
     &                               i_start, n_count, value_ro)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)


! Synchronize CMS file before returning

         i_stat = NF_SYNC(nc_cmsfilid)
         IF (i_stat /= NF_NOERR) CALL HANDLE_ERRORS(i_stat)

#endif

         RETURN

!        --------------------------------------
         END SUBROUTINE EQNOCE_WRITE_CMS_NCFILE
!        --------------------------------------


!     ******************************************************************
      END SUBROUTINE EQNOCE
!     ******************************************************************



C---+----1----*----2----*----3----*----4----*----5----*----6----*----7--
C
C     ******************************************************************
      SUBROUTINE XCB(temps,c13a,
     <               co2xcb,c13xcb,po4xcb)
C     ******************************************************************
C
C
      USE mod_mbm_geometry, ONLY: nro1, nro3, ro1, ro3
      USE mod_mbm_paleod, ONLY: period
      USE mod_mbm_conbio

      IMPLICIT NONE


      DOUBLE PRECISION temps, c13a
      DOUBLE PRECISION co2xcb, c13xcb, po4xcb
C
C     ptemps: periodical time between 0 120000 after Present 
      DOUBLE PRECISION ptemps, ddc13
C
      ptemps = DMOD(temps-1980.D+00,period)
      IF (ptemps .LT. 0.0D+00) ptemps=ptemps+period
C
      DO
         IF (cindex.EQ.ncbio) cindex=1
         IF(ptemps.LT.tcbio(cindex) .OR.
     &     (ptemps.GE.(tcbio(cindex+1)))) THEN
           cindex=cindex+1
           CYCLE
         ELSE
           EXIT
         ENDIF
      ENDDO

C     Now cindex should contain the index such that
C     tcbio(cindex) <= ptemps < tcbio(cindex+1)
      co2xcb = -dcbio(cindex)
C
      ddc13 = frcbio(cindex)+((         ptemps-tcbio(cindex))/
     &                        (tcbio(cindex+1)-tcbio(cindex)))*
     &                       (frcbio(cindex+1)-frcbio(cindex))
      c13xcb = co2xcb*(c13a+ddc13)
      po4xcb = -dpbio(cindex)

      RETURN
      END

