!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


!=======================================================================
      MODULE MOD_JEASIM_O2S
!=======================================================================

      USE mod_defines_medusa
      USE mod_execontrol_medusa, ONLY: ABORT_MEDUSA

      IMPLICIT NONE

      PRIVATE

      PUBLIC :: OCEAN_TO_SEDIMENT


      INTEGER, PARAMETER, PUBLIC :: jp_err_o2s_noerror =  0
      INTEGER, PARAMETER, PUBLIC :: jp_err_o2s_eof     = -1


      CONTAINS


!-----------------------------------------------------------------------
      SUBROUTINE OCEAN_TO_SEDIMENT(cfn_csvin_jeasim, k_flag)
!-----------------------------------------------------------------------
      ! Loads a new complete set of forcing arrays into the
      !
      !  - seafloor_wdata
      !  - seafloor_wsolutes
      !  - seafloor_wfflx


      USE mod_seafloor_wdata,       ONLY: WDATA_CONTAINER
      USE mod_indexparam
      USE mod_basicdata_medusa
      USE mod_materialcharas
      USE mod_gridparam,            ONLY: idvb
      USE mod_milieucharas,         ONLY: xvphi
      USE mod_logunits

      USE mod_seafloor_central,     ONLY:
     &                              SAVE_BOUNDARY_CONDS, N_COLUMNS_USED


      IMPLICIT NONE


      CHARACTER(LEN=*), INTENT(IN)  :: cfn_csvin_jeasim
      INTEGER,          INTENT(OUT) :: k_flag


                                    ! Variables for reading in data from file
      DOUBLE PRECISION :: bbdry_wtot
      DOUBLE PRECISION :: wbdry_dbsl, wbdry_tmpc, wbdry_sali
      DOUBLE PRECISION :: wconc_co2, wconc_hco3, wconc_co3
      DOUBLE PRECISION :: wconc_boh4, wconc_boh3
      DOUBLE PRECISION :: wconc_o2, wconc_no3, wconc_nh4
      DOUBLE PRECISION :: wconc_mn, wconc_fe
      DOUBLE PRECISION :: wconc_so4, wconc_hs, wconc_h2s
      DOUBLE PRECISION :: wbdry_ph

      DOUBLE PRECISION :: wflux_clay
      DOUBLE PRECISION :: wflux_orgc, wbdry_frac_orgcf
      DOUBLE PRECISION :: wflux_calc, wbdry_omega_calc
      DOUBLE PRECISION :: wflux_mno2, wflux_feoh3
      DOUBLE PRECISION :: dummy


      CHARACTER(LEN=1024) :: c_fullline
      CHARACTER(LEN=12)   :: c_site, c_line_site

      DOUBLE PRECISION :: wflux_orgms, wflux_orgmf

      INTEGER :: i, i_flag

      INTEGER, SAVE :: i_unit = -1

      INTEGER, SAVE :: n_datasets_read = 0




      TYPE(WDATA_CONTAINER)               :: wdata
      DOUBLE PRECISION, DIMENSION(nsolut) :: wconc
      DOUBLE PRECISION, DIMENSION(nsolid) :: wfflx


      ! Standard I/O related data
      ! -------------------------

      CHARACTER(LEN=*), PARAMETER :: c_fmtinf_a =
     &  '("[MOD_JEASIM_O2S/OCEAN_TO_SEDIMENT]: ", A)'

      CHARACTER(LEN=*), PARAMETER :: c_fmterr_a =
     &  '("[MOD_JEASIM_O2S/OCEAN_TO_SEDIMENT] error: ", A)'

#ifdef DEBUG
      CHARACTER(LEN=*), PARAMETER :: c_fmtdbg_a =
     &  '("[MOD_JEASIM_O2S/OCEAN_TO_SEDIMENT] debug: ", A)'
#endif


                                    ! If data have been read in so far,
                                    ! return
      IF (n_datasets_read /= 0) THEN
#ifdef DEBUG
        WRITE(jp_stddbg, c_fmtdbg_a) 'Returning previously read record'
#endif

        k_flag = jp_err_o2s_noerror

        RETURN

      ENDIF

                                    ! No data read in so far:
                                    ! open file and get the site data
      i_flag = RESERVE_LOGUNIT(i_unit)
      IF (i_flag /= 0) THEN
        WRITE(jp_stderr, c_fmterr_a)
     &    'No more free units available -- aborting.'
        CALL ABORT_MEDUSA()
#ifdef DEBUG

      ELSE
        WRITE(jp_stddbg, c_fmtdbg_a, ADVANCE='NO')
     &    'Assigning JEASIM_DATA unit '
        WRITE(jp_stddbg, '(I0)') i_unit
#endif
      ENDIF


      OPEN (i_unit, FILE=cfn_csvin_jeasim, STATUS='OLD')


      READ(i_unit, *) c_site         ! Read ID of site to process
      c_site = ADJUSTL(c_site)

      READ(i_unit, '(A)') c_fullline ! Skip two header lines
      READ(i_unit, '(A)') c_fullline


                                    ! Search for the line with the
                                    ! data for the selected site
      DO

        READ(i_unit, '(A)', END=999) c_fullline

        READ(c_fullline, *) c_line_site
        IF (TRIM(ADJUSTL(c_line_site)) == TRIM(c_site)) EXIT

      ENDDO


      READ(c_fullline, *) c_line_site, dummy, dummy,
     &                    wbdry_dbsl, wbdry_omega_calc, wbdry_tmpc,
     &                    wbdry_sali, dummy, bbdry_wtot,
     &                    (dummy, i = 1, 4),
     &                    wconc_o2, wconc_no3,
     &                    wconc_hco3, wconc_co3, wconc_co2, wbdry_ph,
     &                    dummy, wconc_nh4, wconc_so4,
     &                    wconc_mn, wconc_fe, wconc_h2s, wconc_hs,
     &                    wconc_boh3, wconc_boh4,
     &                    wflux_mno2, wflux_feoh3, wflux_orgc,
     &                    wbdry_frac_orgcf, dummy, wflux_calc,
     &                    dummy, dummy, wflux_clay

#ifdef DEBUG
      WRITE(jp_stddbg, *) 'c_line_site = ', TRIM(c_line_site)
      WRITE(jp_stddbg, *) 'wbdry_dbsl = ', wbdry_dbsl
      WRITE(jp_stddbg, *) 'wbdry_omega_calc = ', wbdry_omega_calc
      WRITE(jp_stddbg, *) 'wbdry_sali = ', wbdry_sali
      WRITE(jp_stddbg, *) 'bbdry_wtot = ', bbdry_wtot
      WRITE(jp_stddbg, *) 'wconc_o2 = ', wconc_o2
      WRITE(jp_stddbg, *) 'wconc_no3 = ', wconc_no3
      WRITE(jp_stddbg, *) 'wbdry_ph = ', wbdry_ph
      WRITE(jp_stddbg, *) 'wflux_orgc = ', wflux_orgc
      WRITE(jp_stddbg, *) 'wbdry_frac_orgcf = ', wbdry_frac_orgcf
      WRITE(jp_stddbg, *) 'wflux_calc = ', wflux_calc
      WRITE(jp_stddbg, *) 'wflux_clay = ', wflux_clay
#endif


                                    ! Concentrations read in from the file
                                    ! are in µmol/L, to be converted to
                                    ! mol/m3 (the factor 1.0D-03 does this).
      wconc(ic_co3)  = wconc_co3  * 1.0D-03
      wconc(ic_hco3) = wconc_hco3 * 1.0D-03
      wconc(ic_co2)  = wconc_co2  * 1.0D-03

      wconc(ic_boh3) = wconc_boh3 * 1.0D-03
      wconc(ic_boh4) = wconc_boh4 * 1.0D-03

      wconc(ic_o2)   = wconc_o2   * 1.0D-03
      wconc(ic_no3)  = wconc_no3  * 1.0D-03
      wconc(ic_nh4)  = wconc_nh4  * 1.0D-03
      wconc(ic_mn)   = wconc_mn   * 1.0D-03
      wconc(ic_fe)   = wconc_fe   * 1.0D-03


                                    ! Derive fast and slow organic matter
                                    ! fluxes in µmol_OrgMatter/cm2/yr
                                    ! (wflux_orgc is in µmol_C/cm2/yr).
      wflux_orgmf = (wflux_orgc * wbdry_frac_orgcf)/omf_c
      wflux_orgms = (wflux_orgc - wflux_orgc * wbdry_frac_orgcf)/oms_c


                                    ! Flux values read in from the file
                                    ! are in µmol/cm2/yr, to be converted
                                    ! to kg/m2/yr: the factor 1.0D-02
                                    ! below converts from µmol/cm2 to
                                    ! mol/m2 (and from µg/cm2 to g/cm2)
                                    ! The mol_xyz are in kg_xyz/mol.)
      wfflx(if_clay)  = wflux_clay  * 1.0D-02 * 1.0D-03
c~       wfflx(if_clay)  = wflux_clay  * 0.0D+00
      wfflx(if_calc)  = wflux_calc  * 1.0D-02 * mol_calc
      wfflx(if_omf)   = wflux_orgmf * 1.0D-02 * mol_omf
      wfflx(if_oms)   = wflux_orgms * 1.0D-02 * mol_oms
      wfflx(if_mno2)  = wflux_mno2  * 1.0D-02 * mol_mno2
      wfflx(if_feoh3) = wflux_feoh3 * 1.0D-02 * mol_feoh3

                                    ! The following only holds when there
                                    ! is no interphase biodiffusion or
                                    ! when there are no porosity gradients
                                    ! (the factor 1.0D-02 converts w from
                                    ! cm/yr to m/yr)
c~       wfflx(if_clay) = (bbdry_wtot * 1.0D-02 * (1.0D+00 - xvphi(idvb)))
c~      &                 / apsv(if_clay)


      wdata%wdbsl       = wbdry_dbsl
      wdata%wtmpc       = wbdry_tmpc
      wdata%wsalin      = wbdry_sali
      wdata%womega_calc = wbdry_omega_calc
      wdata%wph         = wbdry_ph

      CALL SAVE_BOUNDARY_CONDS(i_column = 1, gbcflag = i_flag,
     &                              wdata  = wdata,
     &                              wconc  = wconc(:),
     &                              wfflx  = wfflx(:))


      n_datasets_read = n_datasets_read + 1


 999  CONTINUE

      CLOSE(i_unit)

#ifdef DEBUG
        WRITE(jp_stddbg, c_fmtdbg_a, ADVANCE='NO')
     &    'Releasing JEASIM_DATA unit '
        WRITE(jp_stddbg, '(I0)') i_unit
#endif

      i_flag = FREE_LOGUNIT(i_unit)

      IF (i_flag /= 0) THEN
        WRITE(jp_stderr, c_fmterr_a, ADVANCE='NO')
        WRITE(jp_stderr, '(I0,A)')
     &    i_flag, ' while freeing logical unit "i_unit" ' //
     &    '-- ignored.'
      ENDIF

      i_unit = -1
      SELECT CASE(n_datasets_read)
      CASE(0)
        k_flag = jp_err_o2s_eof
      CASE DEFAULT
        k_flag = jp_err_o2s_noerror
      END SELECT


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE OCEAN_TO_SEDIMENT
!-----------------------------------------------------------------------

!=======================================================================
      END MODULE MOD_JEASIM_O2S
!=======================================================================
