!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#ifdef CFN_THISFILE
#undef CFN_THISFILE
#endif
#define CFN_THISFILE "medusa_jeasim.F"
#ifndef __LINE__
#define __LINE__ 0
#endif
!---+----1----+----2----+----3----+----4----+----5----+----6----+----7--
!=======================================================================
      PROGRAM MEDUSA
!=======================================================================

      USE mod_defines_medusa
      USE mod_execontrol_medusa,    ONLY: ABORT_MEDUSA
      USE mod_basicdata_medusa
      USE mod_logunits
      USE mod_gridparam
      USE mod_indexparam
      USE mod_milieucharas
      USE mod_materialcharas
      USE mod_chemicalconsts
      USE mod_transport
      USE mod_timecont_medusa_tsi
      USE mod_store_ncfiles

      USE mod_equilibdata
      USE mod_equilibcontrol

      USE mod_processdata
      USE mod_processcontrol

      USE mod_seafloor_central
      USE mod_seafloor_init
      USE mod_sedcore
      USE mod_files_medusa
      USE mod_jeasim_medusa_setup,  ONLY: SETUP_MEDUSA_FOR_JEASIM
      USE mod_jeasim_o2s

      USE mod_medinterfaces

      IMPLICIT NONE

!     SI units generally used in this program:
!     kg     for solids mass units
!     moles  for solutes mass units
!     m      for space dimensions
!     yr     for time units (!!exception!!)
!     mol/m3 for solute concentrations
!     kg/m3  for solid concentrations


!     In general, variable names matching ...
!     ... "w*" relate to the top grid node of solid sediment+DBL (i=idnw);
!     ... "t*" relate to the top grid node of the solid sediment (i=idnt);
!              (N.B.: if no DBL is considered, idnt=idnw)
!     ... "a*" relate to any grid node (i=idnt, ..., idnb);
!     ... "b*" relate to the bottom grid node (i=idnb);
!     ... "x*" relate to values at grid nodes in the
!         resolved sediment part;
!     ... "y*" relate to the buffering layer between the
!         resolved sediment layers and the historical part below;
!     ... "z*" relate to the historical layers below the bottom
!         of the represented sediment;
!     ... "dx*" relate to derivatives of values at grid nodes in the
!         resolved sediment layers;
!     ... "vx*" relate to variations of values in layers, i.e.
!         between grid nodes in the resolved sediment layers;
!     ... "*conc" represent a concentration;
!     ... "?fflx" represent a solid FLuX;
!     ... "?cflx" represent a solute FLuX



!     xc(i,j): concentration profiles (i=idnw,idnb)
!            for all components (j=1,ncompo)
!     ysolid(nsolid): concentration profiles of solids in the
!            buffering joint layer
!     apsv(j): partial specific (massic) volumes for solids
!            (j=1,nsolid) <<from mod_materialparams>>
!
!
!
!     xzdn(i): depths of the grid nodes (i=idnw, idnb)
!     vxzdn(i): thickness of the layers, i.e.,
!            variation of xzdn between grid nodes,
!            where vxzdn(i)=xzdn(i)-xzdn(i-1); i=idit, idib
!                             (or equivalently i=idnt+1, idnb)





! xw(i): solid burial rate at grid nodes (i=idnt, idnb);
!                actually equal to w(*)*(1-phi(*)), as this
!                product is most often used
! xu(i): porewater velocity at grid nodes (i=idnw, idnb)
!                (actually equal to u(*)*phi(*), as this
!                product is most often used


! At the top of the sediment + possible DBL ('w' interface):
! wdbsl: Depth Below Sea-Level of the sediment pile
! wtmpdc: TeMPerature in Degrees Centigrade of the sed't pile
! wsalin: SALINity of porewater

! atime: current time
! datime: time step to be applied
! datime_tsi: time step to be applied after nc_file done
!                and there are still steps to carry out

! atime and datime are declared anyway, even for steady-state
! simulations - they will simply not get used. Else the
! subroutine declarations would get too complicated.

      CHARACTER(LEN=*), PARAMETER ::
     &  cfn_thisfile     = CFN_THISFILE
      CHARACTER(LEN=*), PARAMETER ::
     &  cfmt_info_a      = '("['//cfn_thisfile//']: ", A)' ,
     &  cfmt_info_ai     = '("['//cfn_thisfile//']: ", A, I0)',
     &  cfmt_infolin_ia  = '("['//cfn_thisfile//':", I0, "]: ", A)',
     &  cfmt_infolin_iai = '("['//cfn_thisfile//':", I0, "]: ", A, I0)'

      CHARACTER(LEN=*), PARAMETER :: cfmt_a     = '(" - ", A)'
      CHARACTER(LEN=*), PARAMETER :: cfmt_a_ind = '(3X, A)'

      CHARACTER(LEN=*), PARAMETER ::
     &  cfmt_timeto = '("Time step ", I0,' //
     &                ' ", from ", F0.2, " to ", F0.2, ":")'
      CHARACTER(LEN=*), PARAMETER ::
     &  cfmt_timess = '("Steady-state time step ", I0,' //
     &                ' " (with time = ", F0.2, "):")'

      DOUBLE PRECISION :: atime, datime, atime_tsi, datime_tsi
      INTEGER :: nsteps, nst, nstepsini

      INTEGER :: i_flag, n_trouble, n_columns, i_request, istep
      DOUBLE PRECISION :: atime0, atime0_tsi
      INTEGER :: n_datasets_read = 0


! Further variables provided by modules

!   mod_milieuparams:
!
!     xphi(i): sediment porosity at grid nodes (i=idnt, idnb)
!     yphi: sediment porosity in the buffering joint layer (constant)
!     zphi: sediment porosity in the historical layers (constant)
!     dxphi(i): dxphi(i) = (d xphi/d z)|z=xzdn(i),
!            actual value of the derivative if xphi=fct(xzdn)
!            analytically known
!     xtor2(i): sediment tortuosity^2 ($\theta^2$) at
!            grid nodes (i=idnt, idnb)
!     dxtor2(i): dxtor2(i) = (d xtor2/d z)|z=xzdn(i),
!            actual value of the derivative if xtor2=fct(xzdn)
!            analytically known
!     ytor2: sed_t tortuosity^2 in the buffering joint layer (constant)
!     ztor2: sed_t tortuosity^2 in the historical layers (constant)
!
!-----------------------------------------------------------------------

      CHARACTER (LEN=jp_lmaxpathname) :: cfn_medusa_err = "medusa.err"
      CHARACTER (LEN=jp_lmaxpathname) :: cfn_medusa_log = "medusa.log"
#ifdef DEBUG
      CHARACTER (LEN=jp_lmaxpathname) :: cfn_medusa_dbg = "medusa.dbg"
#endif


      OPEN(jp_stderr, FILE=cfn_medusa_err)
      CALL REPORT_COMPILE_OPTIONS(jp_stderr)
      CALL REPORT_EXTRA_COMPILE_OPTIONS(jp_stderr)

      OPEN(jp_stdlog, FILE=cfn_medusa_log)

#ifdef DEBUG
      IF (jp_stddbg /= jp_stderr) OPEN(jp_stddbg, FILE=cfn_medusa_dbg)
      WRITE(jp_stddbg, cfmt_infolin_ia) (__LINE__), 'Starting'
      WRITE(jp_stddbg, '()')
      CALL REPORT_COMPILE_OPTIONS(jp_stddbg)
      CALL REPORT_EXTRA_COMPILE_OPTIONS(jp_stddbg)
      CALL DEBUG_MEDUSA_MAIN_INDICES
#endif


      CALL INIT_FILELIST_MEDUSA()


      CALL PREPARE_NAMELISTS(cfn_csvin_jeasim = cfn_csvin_jeasim)

                                    ! From mod_medana_medusa_setup.F
      CALL SETUP_MEDUSA_FOR_JEASIM(n_columns = n_columns)


                                    ! Must be done here as
                                    ! this also calls CORELAY_SETUP
      CALL SETUP_SEDCORE_SYSTEM(cfn_ncin_sedcore, cfn_ncout_sedcore)


      CALL SETUP_TRANSPORT          ! From mod_transport.F


      CALL InitEquilibParameters    ! From mod_equilibcontrol.F


      CALL InitProcessParameters    ! From mod_processcontrol.F


                                    ! Initialize the sediment state
      IF (cfn_ncin_init /= "/dev/null") THEN
                                    ! From MOD_SEAFLOOR_INIT
        CALL InitSeafloorFromNetCDFFiles(cfn_reaclay = cfn_ncin_init,
     &                                   cfn_flx     = cfn_ncin_flx)

      ELSEIF (cfn_nmlin_init /= "/dev/null") THEN
                                    ! From MOD_SEAFLOOR_INIT
        CALL InitSeafloorFromNamelistFile(cfn_nmlin_init)

      ELSE

        WRITE(jp_stderr, cfmt_info_a)
     &    'Fatal error: no valid initialisation file given'
        WRITE(jp_stderr, cfmt_a, ADVANCE="NO")
        WRITE(jp_stderr, '("cfn_nmlin_init = """, A, """")')
     &                              TRIM(cfn_nmlin_init)
        WRITE(jp_stderr, cfmt_a, ADVANCE="NO")
        WRITE(jp_stderr, '("cfn_ncin_init = """, A, """")')
     &                              TRIM(cfn_ncin_init)
        WRITE(jp_stderr, '("Aborting!")')
        CALL ABORT_MEDUSA()

      ENDIF

      ! Get time stepping information
      i_request = 0  ! Get info from scratch
      CALL GET_TIMESTEP_INFO(i_request, i_flag,
     &               atime0_tsi, datime_tsi, nsteps)
#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_a)   'Time stepping information:'
      WRITE(jp_stddbg, cfmt_a_ind, ADVANCE="NO") '  atime0 = '
      WRITE(jp_stddbg, '(F0.2)') atime0_tsi
      WRITE(jp_stddbg, cfmt_a_ind, ADVANCE="NO") '  datime = '
      WRITE(jp_stddbg, '(F0.2)') datime_tsi
      WRITE(jp_stddbg, cfmt_a_ind, ADVANCE="NO") '  nsteps = '
      WRITE(jp_stddbg, '(I0)') nsteps
      WRITE(jp_stddbg, '()') 
#endif
      i_request = 1 ! Close file
      CALL GET_TIMESTEP_INFO(i_request, i_flag)


      CALL OCEAN_TO_SEDIMENT(cfn_csvin_jeasim = cfn_csvin_jeasim,
     &                              k_flag = i_flag)


      SELECT CASE(i_flag)

        CASE(jp_err_o2s_noerror)
          CONTINUE

        CASE(jp_err_o2s_eof)
          WRITE(jp_stderr, cfmt_infolin_ia) (__LINE__),
     &        'End  of file detected - requested site not found. ' //
     &        'Aborting!'
          CALL ABORT_MEDUSA()

        CASE DEFAULT
          WRITE(jp_stderr, cfmt_infolin_ia) (__LINE__),
     &        'Unknown error in OCEAN_TO_SEDIMENT. Aborting!'
          CALL ABORT_MEDUSA()

      END SELECT

                                    ! Open NetCDF results files and
                                    ! store initial state
      atime = atime0_tsi
#ifdef NCFILES_SAVEALLSTEPS
      CALL OPEN_NCFILES_MEDUSA(atime)
#else
      CALL OPEN_NCFILES_MEDUSA()
#endif


      DO istep = 1, nsteps

        atime = atime0_tsi + (istep-1)*datime_tsi
        datime = datime_tsi


        WRITE(jp_stderr, '()')
        IF (datime == 0.0D+00) THEN
          WRITE(jp_stderr, cfmt_timess) istep, atime
        ELSE
          WRITE(jp_stderr, cfmt_timeto) istep, atime, atime + datime
        ENDIF

        CALL SOLVSED_ONESTEP(atime, datime, n_columns,
     &                       i_flag, n_trouble)

        atime = atime + datime

        CALL REACLAY_X_CORELAY(atime)

#ifdef NCFILES_SAVEALLSTEPS
        CALL WRITERES_NCFILES_MEDUSA(atime)
#else
        IF (istep == nsteps) THEN
          CALL WRITERES_NCFILES_MEDUSA(atime)
        ENDIF
#endif

      ENDDO


      CALL CLOSE_RESFILES_MEDUSA()

      CALL STORE_CSV_SUMMARY(cfn_csvout_summary)


#ifdef DEBUG
      WRITE(jp_stddbg,cfmt_infolin_ia) (__LINE__), 'End.'
      IF (jp_stddbg /= jp_stderr) CLOSE(jp_stddbg)
#endif
      CLOSE(jp_stderr)
      CLOSE(jp_stdlog)



!=======================================================================

      CONTAINS

!-----------------------------------------------------------------------
      SUBROUTINE REPORT_EXTRA_COMPILE_OPTIONS(iunit)
!-----------------------------------------------------------------------

      IMPLICIT NONE

      INTEGER, INTENT(IN) :: iunit


#ifdef NCFILES_SAVEALLSTEPS
      WRITE(iunit, '(" -DNCFILES_SAVEALLSTEPS")')
#endif


#ifdef USE_CA_BTM_FINAL
      WRITE(iunit, '(" -DUSE_CA_BTM_FINAL")')
#endif

#ifdef USE_CORG_BTM_FINAL
      WRITE(iunit, '(" -DUSE_CORG_BTM_FINAL")')
#endif

      WRITE(iunit, '()')


      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE REPORT_EXTRA_COMPILE_OPTIONS
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE PREPARE_NAMELISTS(cfn_csvin_jeasim)
!-----------------------------------------------------------------------

      IMPLICIT NONE

      CHARACTER(LEN=*), INTENT(IN) :: cfn_csvin_jeasim


      CHARACTER(LEN=*), PARAMETER
     &  :: cfn_rrp_base       = "medusa-base.rrp",
     &     cfn_rrp            = "medusa.rrp"
      CHARACTER(LEN=*), PARAMETER
     &  :: cfn_milieu_base    = "medusa_milieu_config-base.nml",
     &     cfn_milieu         = "medusa_milieu_config.nml"
      CHARACTER(LEN=*), PARAMETER
     &  :: cfn_transport_base = "medusa_transport_config-base.nml",
     &     cfn_transport      = "medusa_transport_config.nml"

      CHARACTER(LEN=1024) :: c_fullline
      CHARACTER(LEN=12)   :: c_site, c_line_site

      DOUBLE PRECISION :: dummy
      DOUBLE PRECISION :: db_0
      DOUBLE PRECISION :: phi_0, phi_infty, phi_alpha
      DOUBLE PRECISION :: flx_orgc, ak_orgmf, ak_orgms
      DOUBLE PRECISION :: an_calcdiss, ak_calcdiss

      INTEGER :: istatus, iunit, iunit_nml
      INTEGER :: i



      istatus = RESERVE_LOGUNIT(iunit)
      istatus = RESERVE_LOGUNIT(iunit_nml)


      OPEN(UNIT=iunit, FILE=cfn_csvin_jeasim, STATUS="OLD")

      READ(iunit, *) c_site         ! Read ID of site to process
      c_site = ADJUSTL(c_site)

      READ(iunit, '(A)') c_fullline ! Skip two header lines
      READ(iunit, '(A)') c_fullline


                                    ! Search for the line with the
                                    ! data for the selected site
#ifdef DEBUG
      WRITE(jp_stddbg, *) 'Searching for site "' // TRIM(c_site) // '"'
#endif


      DO

        READ(iunit, '(A)', END=999) c_fullline

        READ(c_fullline, *) c_line_site
        IF (TRIM(ADJUSTL(c_line_site)) == TRIM(c_site)) THEN
#ifdef DEBUG
          WRITE(jp_stddbg, *) ' found line for site "' //
     &                              TRIM(c_site) // '"'
#endif
          EXIT
        ENDIF

      ENDDO


      READ(c_fullline, *) c_line_site, (dummy, i = 1, 8),
     &                    db_0, phi_0, phi_infty, phi_alpha,
     &                    (dummy, i = 1, 17),
     &                    flx_orgc,
     &                    dummy,
     &                    ak_orgmf,
     &                    dummy,
     &                    an_calcdiss, ak_calcdiss,
     &                    dummy

#ifdef DEBUG
      WRITE(jp_stddbg, *) 'db_0 = ', db_0 * dp_cm2_p_yr
      WRITE(jp_stddbg, *) 'phi_0 = ', phi_0
      WRITE(jp_stddbg, *) 'phi_infty = ', phi_infty
      WRITE(jp_stddbg, *) 'phi_scale = ', dp_cm/ABS(phi_alpha)
      WRITE(jp_stddbg, *) 'flx_orgc = ', flx_orgc
      WRITE(jp_stddbg, *) 'ak_orgmf = ', ak_orgmf
      WRITE(jp_stddbg, *) 'an_calc = ', an_calcdiss
      WRITE(jp_stddbg, *) 'ak_calc = ', ak_calcdiss
#endif

      CLOSE(UNIT=iunit)


                                    ! Prepare a "medusa_milieu_config.nml"
                                    ! that includes the characteristics of
                                    ! the porosity profile

      OPEN(UNIT=iunit,     FILE=cfn_milieu_base)
      OPEN(UNIT=iunit_nml, FILE=cfn_milieu)


      DO

        READ(iunit, '(A)', END=991) c_fullline

        IF (c_fullline(1:15) == '&nml_phi_expdec') THEN

          WRITE(iunit_nml, '(A)') TRIM(c_fullline)

          WRITE(iunit_nml, '("phi_0     = ")', ADVANCE="NO")
          WRITE(iunit_nml, *) phi_0, ' ! [m3_porewater/m3_totalsed]'

          WRITE(iunit_nml, '("phi_infty = ")', ADVANCE="NO")
          WRITE(iunit_nml, *) phi_infty, ' ! [m3_porewater/m3_totalsed]'

          WRITE(iunit_nml, '("phi_scale = ")', ADVANCE="NO")
          WRITE(iunit_nml, *) 1.0D-02/ABS(phi_alpha), ' ! [m]'

          DO
            READ(iunit, '(A)') c_fullline
            IF (c_fullline(1:7)  == 'phi_0 =')     CYCLE
            IF (c_fullline(1:11) == 'phi_infty =') CYCLE
            IF (c_fullline(1:11) == 'phi_scale =') CYCLE
            WRITE(iunit_nml, '(A)') TRIM(c_fullline)
            IF (c_fullline(1:1)  == '/') EXIT
          ENDDO

        ELSE

          WRITE(iunit_nml, '(A)') TRIM(c_fullline)

        ENDIF

      ENDDO

  991 CLOSE(UNIT=iunit)
      CLOSE(UNIT=iunit_nml)


                                    ! Prepare a "medusa_transport_config.nml"
                                    ! that includes the characteristics of
                                    ! of the DB profile

      OPEN(UNIT=iunit,     FILE=cfn_transport_base)
      OPEN(UNIT=iunit_nml, FILE=cfn_transport)

      DO

        READ(iunit, '(A)', END=992) c_fullline

        IF (c_fullline(1:17) == '&nml_biodif_erfc') THEN

          WRITE(iunit_nml, '(A)') TRIM(c_fullline)

          WRITE(iunit_nml, '("db_0 = ")', ADVANCE="NO")
          WRITE(iunit_nml, *) db_0*dp_cm2_p_yr, ' ! [m2/yr]'

          DO
            READ(iunit, '(A)') c_fullline
            IF (c_fullline(1:6)  == 'db_0 =')     CYCLE
            WRITE(iunit_nml, '(A)') TRIM(c_fullline)
            IF (c_fullline(1:1)  == '/') EXIT
          ENDDO

        ELSE

          WRITE(iunit_nml, '(A)') TRIM(c_fullline)

        ENDIF

      ENDDO

  992 CLOSE(UNIT=iunit)
      CLOSE(UNIT=iunit_nml)

                                    ! Prepare a "medusa.rrp" that includes
                                    ! the special parameter values for the
                                    ! different rate laws

      OPEN(UNIT=iunit,     FILE=cfn_rrp_base)
      OPEN(UNIT=iunit_nml, FILE=cfn_rrp)

                                    ! k_s/[yr^-1] = 2.2*10^-5 * (FCorg/[µmol/cm^2/yr])**2.1
      ak_orgms = 2.2D-05 * flx_orgc**2.1D+00

      DO

        READ(iunit, '(A)', END=993) c_fullline

        IF (c_fullline(1:20) == '&nml_CalcDissolution') THEN

          WRITE(iunit_nml, '(A)') TRIM(c_fullline)

          WRITE(iunit_nml, '("k_1 = ")', ADVANCE="NO")
                                    ! k_calcdiss from  JEASIM CSV
                                    ! is expressed in %/day
          WRITE(iunit_nml, *) ak_calcdiss /(dp_day*100.0D+00)

          WRITE(iunit_nml, '("n_1 = ")', ADVANCE="NO")
          WRITE(iunit_nml, *) an_calcdiss

          DO
            READ(iunit, '(A)') c_fullline
            IF (c_fullline(1:5) == 'k_1 =') CYCLE
            IF (c_fullline(1:5) == 'n_1 =') CYCLE
            WRITE(iunit_nml, '(A)') TRIM(c_fullline)
            IF (c_fullline(1:1) == '/') EXIT
          ENDDO

        ELSEIF (c_fullline(1:20) == '&nml_OrgMatterF_Oxic') THEN

          WRITE(iunit_nml, '(A)') TRIM(c_fullline)

          WRITE(iunit_nml, '("k_2 = ")', ADVANCE="NO")
          WRITE(iunit_nml, *) ak_orgmf

          DO
            READ(iunit, '(A)') c_fullline
            IF (c_fullline(1:5) == 'k_2 =') CYCLE
            WRITE(iunit_nml, '(A)') TRIM(c_fullline)
            IF (c_fullline(1:1) == '/') EXIT
          ENDDO

        ELSEIF (c_fullline(1:20) == '&nml_OrgMatterS_Oxic') THEN

          WRITE(iunit_nml, '(A)') TRIM(c_fullline)

          WRITE(iunit_nml, '("k_6 = ")', ADVANCE="NO")
          WRITE(iunit_nml, *) ak_orgms

          DO
            READ(iunit, '(A)') c_fullline
            IF (c_fullline(1:5) == 'k_6 =') CYCLE
            WRITE(iunit_nml, '(A)') TRIM(c_fullline)
            IF (c_fullline(1:1) == '/') EXIT
          ENDDO

        ELSE

          WRITE(iunit_nml, '(A)') TRIM(c_fullline)

        ENDIF

      ENDDO

  993 CLOSE(UNIT=iunit)
      CLOSE(UNIT=iunit_nml)

                                    ! All ended well if this point was reached
      istatus = FREE_LOGUNIT(iunit)
      istatus = FREE_LOGUNIT(iunit_nml)

      RETURN


  999 CLOSE(UNIT=iunit)

      istatus = FREE_LOGUNIT(iunit)

      WRITE(jp_stderr, '("Site """, A, """ not found")')
     &                              TRIM(ADJUSTL(c_site))
      WRITE(jp_stderr, '("Impossible to proceed -- aborting!")')

      CALL ABORT_MEDUSA()


      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE PREPARE_NAMELISTS
!-----------------------------------------------------------------------


!=======================================================================
      END
!=======================================================================
