!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#ifdef CFN_THISFILE
#undef CFN_THISFILE
#endif
#define CFN_THISFILE "store_nc_aux.F"
#ifndef __LINE__
#define __LINE__ 0
#endif
!=======================================================================
      SUBROUTINE STORE_NC_AUX(cfn_ncin_bec, cfn_ncout_aux, title_string)
!=======================================================================


      USE mod_defines_medusa
      USE mod_execontrol_medusa,    ONLY: ABORT_MEDUSA

#ifdef ALLOW_MPI
      USE mpi,                      ONLY: MPI_COMM_NULL, MPI_PROC_NULL,
     &                                    MPI_STATUS_SIZE,
     &                                    MPI_DOUBLE_PRECISION
      USE mod_execontrol_medusa,    ONLY: jp_exeproc_ncio,
     &                                    jp_exeproc_root,
     &                                    MEDEXE_MPI_TOPO2D_PP4RANK
#endif

      USE mod_gridparam,            ONLY: idnw, idnb, ndn, GRID_DEF

#ifdef MEDUSA_BASE2DT2D
      USE mod_seafloor_central,     ONLY: N_COLUMNS_USED, COLUMN_AREA4N,
     &                                    COLUMN_IJ_ITJT2N,
     &                                    COLUMN_N2IJ_ITJT,
     &                                    COLUMN_N2XY, COLUMN_N2MASKVAL
#endif

      USE mod_netcdfinc
      USE mod_netcdfparam

      USE mod_coupsim_medusa_setup, ONLY: GET_NLONSLATSZTS_FILE


      IMPLICIT NONE


      CHARACTER(LEN=*), INTENT(IN)           :: cfn_ncin_bec
      CHARACTER(LEN=*), INTENT(IN)           :: cfn_ncout_aux
      CHARACTER(LEN=*), INTENT(IN), OPTIONAL :: title_string


      CHARACTER(LEN=*), PARAMETER :: cfn_thisfile =
     &  CFN_THISFILE

      INTEGER :: ncid

      INTEGER :: ncid_file, ncid_var

      INTEGER :: nsedcol_central = -1
      INTEGER :: nsedcol_ncfile  = -1
#ifdef MEDUSA_BASE2DT2D
      INTEGER :: nitx_ncfile      = -1
      INTEGER :: njty_ncfile      = -1
#endif

      LOGICAL :: l_file_is_mine = .TRUE.

#ifdef ALLOW_MPI
      INTEGER :: n_cprocs = -1
      INTEGER :: i_mycomm = MPI_COMM_NULL
      INTEGER :: i_myrank = MPI_PROC_NULL
#  ifdef MEDUSA_BASE2DT2D
      INTEGER :: i_myip   = -1
      INTEGER :: i_myjp   = -1

                                    ! Number of processes
      INTEGER :: npx                !  - along X-direction
      INTEGER :: npy                !  - along Y-direction
#  endif

      INTEGER, DIMENSION(MPI_STATUS_SIZE) :: impi_status

      INTEGER :: nsedcol_global = -1
      INTEGER :: iproc_1stocn   = -1
      INTEGER, DIMENSION(:), ALLOCATABLE :: nsedcol_pproc
      INTEGER, DIMENSION(:), ALLOCATABLE :: ioffset_sedcol_pproc

      LOGICAL :: l_onewrites4many = .FALSE.

      INTEGER,          DIMENSION(:),   ALLOCATABLE :: iarr_mpirecv_c
      DOUBLE PRECISION, DIMENSION(:),   ALLOCATABLE :: darr_mpirecv_c
#  ifdef MEDUSA_BASE2DT2D
      INTEGER,          DIMENSION(:,:, :,:),
     &                                  ALLOCATABLE :: iarr_mpirecv_4d
      DOUBLE PRECISION, DIMENSION(:,:, :,:),
     &                                  ALLOCATABLE :: darr_mpirecv_4d
#  endif
      INTEGER :: i_destrank
#endif


      INTEGER :: k, n
      INTEGER :: iflag, istatus

#ifdef MEDUSA_BASE2DT2D
      INTEGER :: nix, njy, nitx, njty, n4d
      INTEGER, DIMENSION(4) :: dim, istart
      INTEGER :: i, j
      INTEGER :: ip, jp
      INTEGER :: it, jt
      INTEGER :: ix, jy
      DOUBLE PRECISION :: x, y
#endif

      DOUBLE PRECISION, DIMENSION(idnw:idnb) :: xzdn


! Dimensions and dimension variables
!  'lev' (levels)
      INTEGER            :: dim_lev         ! dimension ID
      INTEGER            ::  id_lev         ! ID of dimension variable

! 'col' (columns)
      INTEGER            :: dim_col         ! dimension ID
      INTEGER            ::  id_col         ! ID of dimension variable

#ifdef MEDUSA_BASE2DT2D
      CHARACTER(LEN=*), PARAMETER   :: c_name_ix     = 'ix'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_ix    = 'X_Index'
      INTEGER, PARAMETER :: nlen_lname_ix = LEN(c_lname_ix)
      INTEGER            :: dim_ix
      INTEGER            ::  id_ix

      CHARACTER(LEN=*), PARAMETER   :: c_name_jy     = 'jy'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_jy    = 'Y_Index'
      INTEGER, PARAMETER :: nlen_lname_jy = LEN(c_lname_jy)
      INTEGER            :: dim_jy
      INTEGER            ::  id_jy

      CHARACTER(LEN=*), PARAMETER   :: c_name_itx     = 'itx'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_itx    = 'Tile_X_Index'
      INTEGER, PARAMETER :: nlen_name_itx  = LEN(c_name_itx)
      INTEGER, PARAMETER :: nlen_lname_itx = LEN(c_lname_itx)
      INTEGER            :: dim_itx
      INTEGER            ::  id_itx

      CHARACTER(LEN=*), PARAMETER   :: c_name_jty     = 'jty'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_jty    = 'Tile_Y_Index'
      INTEGER, PARAMETER :: nlen_name_jty  = LEN(c_name_jty)
      INTEGER, PARAMETER :: nlen_lname_jty = LEN(c_lname_jty)
      INTEGER            :: dim_jty
      INTEGER            ::  id_jty
#endif


! Data variables
!  - general
      INTEGER ::  id_xzdn           ! depth coordinates
      INTEGER ::  id_area           ! surface areas of columns (n)
      INTEGER ::  id_maskval        ! mask value maskval(n)   

      DOUBLE PRECISION   :: sfc_area
      INTEGER            :: mval

      INTEGER,          DIMENSION(:), ALLOCATABLE ::  iarr_c
      DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE ::  darr_c

#ifdef MEDUSA_BASE2DT2D
!  - 2DT2D specific
      INTEGER ::  id_col4ij_itjt    ! column index n for grid element (i,j, it,jt)

                                    ! index i(n)
      CHARACTER(LEN=*), PARAMETER   :: c_name_i4col     = 'i4col'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_i4col    = c_lname_ix //
     &                                              '_from_Column_Index'
      INTEGER, PARAMETER :: nlen_lname_i4col = LEN(c_lname_i4col)
      INTEGER ::  id_i4col

                                    ! index j(n)
      CHARACTER(LEN=*), PARAMETER   :: c_name_j4col     = 'j4col'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_j4col    = c_lname_jy //
     &                                              '_from_Column_Index'
      INTEGER, PARAMETER :: nlen_lname_j4col = LEN(c_lname_j4col)
      INTEGER ::  id_j4col

                                    ! index it(n)
      CHARACTER(LEN=*), PARAMETER   :: c_name_it4col    = 'it4col'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_it4col   = c_lname_itx //
     &                                              '_from_Column_Index'
      INTEGER, PARAMETER :: nlen_lname_it4col = LEN(c_lname_it4col)
      INTEGER ::  id_it4col

                                    ! index jt(n)
      CHARACTER(LEN=*), PARAMETER   :: c_name_jt4col    = 'jt4col'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_jt4col   = c_lname_jty //
     &                                              '_from_Column_Index'
      INTEGER, PARAMETER :: nlen_lname_jt4col = LEN(c_lname_jt4col)
      INTEGER                       ::  id_jt4col

                                    ! xref value of grid element (n)
      CHARACTER(LEN=*), PARAMETER   :: c_name_xref4col  = 'xref4col'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_xref4col = 'Longitude'
      CHARACTER(LEN=*), PARAMETER   :: c_unit_xref4col  = un_degE
      INTEGER, PARAMETER :: nlen_lname_xref4col = LEN(c_lname_xref4col)
      INTEGER, PARAMETER :: nlen_unit_xref4col  = LEN(c_unit_xref4col)
      INTEGER ::  id_xref4col
      REAL,             ALLOCATABLE :: rarr_tlong(:,:)
      DOUBLE PRECISION, ALLOCATABLE :: darrx_6d(:,:, :,:, :,:)
      DOUBLE PRECISION, ALLOCATABLE :: darrx_4d(:,:, :,:)

                                    ! yref value of grid element (n)
      CHARACTER(LEN=*), PARAMETER   :: c_name_yref4col  = 'yref4col'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_yref4col = 'Latitude'
      CHARACTER(LEN=*), PARAMETER   :: c_unit_yref4col  = un_degN
      INTEGER, PARAMETER :: nlen_lname_yref4col = LEN(c_lname_yref4col)
      INTEGER, PARAMETER :: nlen_unit_yref4col  = LEN(c_unit_yref4col)
      INTEGER ::  id_yref4col
      REAL,             ALLOCATABLE :: rarr_tlat(:,:)
      DOUBLE PRECISION, ALLOCATABLE :: darry_6d(:,:, :,:, :,:)
      DOUBLE PRECISION, ALLOCATABLE :: darry_4d(:,:, :,:)


      INTEGER,          ALLOCATABLE :: iarri_c(:)
      INTEGER,          ALLOCATABLE :: iarrj_c(:)
      INTEGER,          ALLOCATABLE :: iarrit_c(:)
      INTEGER,          ALLOCATABLE :: iarrjt_c(:)
      INTEGER,          ALLOCATABLE :: iarr_4d(:,:, :,:)

      DOUBLE PRECISION, ALLOCATABLE :: darrx_c(:)
      DOUBLE PRECISION, ALLOCATABLE :: darry_c(:)
#endif


                                    ! Longitude value of grid
                                    ! element (ix,jy, itx,jty)
      CHARACTER(LEN=*), PARAMETER   :: c_name_lon4ij_itjt =
     &                              'lon4ij_itjt'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_lon4ij_itjt =
     &                              'Array of grid longitudes'
      CHARACTER(LEN=*), PARAMETER   :: c_unit_lon4ij_itjt =
     &                              un_degE
      INTEGER,          PARAMETER   :: nlen_lname_lon4ij_itjt =
     &                              LEN(c_lname_lon4ij_itjt)
      INTEGER,          PARAMETER   :: nlen_unit_lon4ij_itjt =
     &                              LEN(c_unit_lon4ij_itjt)
      INTEGER                       :: id_lon4ij_itjt

                                    ! Latitude value of grid
                                    ! element (ix,jy, itx,jty)
      CHARACTER(LEN=*), PARAMETER   :: c_name_lat4ij_itjt =
     &                              'lat4ij_itjt'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_lat4ij_itjt =
     &                              'Array of grid latitudes'
      CHARACTER(LEN=*), PARAMETER   :: c_unit_lat4ij_itjt =
     &                              un_degN
      INTEGER,          PARAMETER   :: nlen_lname_lat4ij_itjt =
     &                              LEN(c_lname_lat4ij_itjt)
      INTEGER,          PARAMETER   :: nlen_unit_lat4ij_itjt =
     &                              LEN(c_unit_lat4ij_itjt)
      INTEGER                       ::  id_lat4ij_itjt

      INTEGER :: nix_bec = -1
      INTEGER :: njy_bec = -1


#ifdef DEBUG
      CHARACTER(LEN=*), PARAMETER :: cfmt_dbg_a
     &  = '("[STORE_NC_AUX] debug: ", A)'
#endif


! Subroutine start
! ================

      CALL SNCA_SETUP()


                                    ! The root process first reads in
                                    ! the longitude-latitude information
                                    ! for the grid
      IF (i_myrank == jp_exeproc_root) THEN

        CALL GET_NLONSLATSZTS_FILE(nix_bec, njy_bec)

                                    ! Open file <cfn_ncin_bec>
                                    ! to retrieve the complete longitude
                                    ! and latitude arrays
        istatus = NF_OPEN(cfn_ncin_bec, NF_NOWRITE, ncid_file)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

        ALLOCATE(rarr_tlong(nix_bec, njy_bec))
        ALLOCATE(rarr_tlat(nix_bec, njy_bec))

                                    ! Read in longitude array
                                    ! (type: FLOAT; units: degrees_east)
        istatus = NF_INQ_VARID(ncid_file, 'TLONG', ncid_var)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
        istatus = NF_GET_VAR_REAL(ncid_file, ncid_var, rarr_tlong)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

                                    ! Read in latitude array
                                    ! (type: FLOAT; units: degrees_north)
        istatus = NF_INQ_VARID(ncid_file, 'TLAT', ncid_var)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
        istatus = NF_GET_VAR_REAL(ncid_file, ncid_var, rarr_tlat)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

                                    ! Close the file, since all that
                                    ! can be used has been read in now.
        istatus = NF_CLOSE(ncid_file)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))


        ALLOCATE(darrx_6d(nix,njy, nitx,njty, npx,npy))
        ALLOCATE(darry_6d(nix,njy, nitx,njty, npx,npy))


      ENDIF


      ALLOCATE(iarr_c(nsedcol_central))
      ALLOCATE(darr_c(nsedcol_central))

#ifdef MEDUSA_BASE2DT2D
      ALLOCATE(iarr_4d(nix, njy, nitx,njty))
      ALLOCATE(darrx_4d(nix, njy, nitx,njty))
      ALLOCATE(darry_4d(nix, njy, nitx,njty))

      ALLOCATE(iarri_c(nsedcol_central))
      ALLOCATE(iarrj_c(nsedcol_central))
      ALLOCATE(iarrit_c(nsedcol_central))
      ALLOCATE(iarrjt_c(nsedcol_central))

      ALLOCATE(darrx_c(nsedcol_central))
      ALLOCATE(darry_c(nsedcol_central))
#endif


#ifdef ALLOW_MPI
                                    ! The root process now sends the
                                    ! longitude-latitude grid information
                                    ! to each process (only for that part
                                    ! of the grid that belongs to the process)
                                    ! 
      IF (i_myrank == jp_exeproc_root) THEN

        DO jp = 1, npy
        DO ip = 1, npx
        DO jt = 1, njty
        DO it = 1, nitx

          DO jy = 1, njy
            j = (jp - 1) * njty*njy + (jt - 1) * njy + jy

            DO ix = 1, nix
              i = (ip - 1) * nitx*nix + (it - 1) * nix + ix

              darrx_6d(ix,jy, it,jt, ip,jp) = DBLE(rarr_tlong(i,j))
              darry_6d(ix,jy, it,jt, ip,jp) = DBLE(rarr_tlat(i,j))

            ENDDO

          ENDDO

        ENDDO
        ENDDO
        ENDDO
        ENDDO


        DO i_destrank = 0, n_cprocs-1

          CALL MEDEXE_MPI_TOPO2D_PP4RANK(i_destrank, ip, jp)

          IF (i_destrank == jp_exeproc_root) THEN

            darrx_4d(:,:, :,:) = darrx_6d(:,:, :,:, ip,jp)
            darry_4d(:,:, :,:) = darry_6d(:,:, :,:, ip,jp)

          ELSE

#  ifdef DEBUG
            WRITE(jp_stddbg, cfmt_dbg_a, ADVANCE='NO')
     &        'Now sending arrays to process '
            WRITE(jp_stddbg, '(I0, " (ip=",I0,",jp=",I0,")")')
     &        i_destrank, ip, jp
#  endif

            CALL MPI_SEND(darrx_6d(:,:, :,:, ip,jp), n4d,
     &                              MPI_DOUBLE_PRECISION,
     &                              i_destrank, i_destrank,
     &                              i_mycomm, iflag)

            CALL MPI_SEND(darry_6d(:,:, :,:, ip,jp), n4d,
     &                              MPI_DOUBLE_PRECISION,
     &                              i_destrank, i_destrank,
     &                              i_mycomm, iflag)

          ENDIF

        ENDDO

      ELSE

#  ifdef DEBUG
        WRITE(jp_stddbg, cfmt_dbg_a, ADVANCE='NO')
     &    'Now receiving arrays from process '
        WRITE(jp_stddbg, '(I0)') jp_exeproc_root
#  endif

        CALL MPI_RECV(darrx_4d, n4d, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_myrank,
     &                              i_mycomm, impi_status, iflag)
        CALL MPI_RECV(darry_4d, n4d, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_myrank,
     &                              i_mycomm, impi_status, iflag)

      ENDIF
#endif


      IF (l_file_is_mine) THEN

        !-----------------------
        ! Create the data file
        !-----------------------
        istatus = NF_CREATE(TRIM(cfn_ncout_aux), NF_CLOBBER, ncid)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))


        !-------------------------------------------
        ! Define dimensions and dimension variables
        !-------------------------------------------

                                    ! Columns on the string
                                    ! ---------------------
                                    ! - dim id
        istatus = NF_DEF_DIM(ncid, 'col', nsedcol_ncfile, dim_col)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

                                    ! - dim variable id
        istatus = NF_DEF_VAR(ncid, 'col', NF_INT, 1, dim_col, id_col)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

#ifdef ALLOW_MPI
        IF (l_onewrites4many) THEN
          istatus = NF_PUT_ATT_INT(ncid, id_col, 'offset', NF_INT, 1, 0)
          IF (istatus /= NF_NOERR)
     &      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
        ELSE
          istatus = NF_PUT_ATT_INT(ncid, id_col, 'offset', NF_INT, 1,
     &                              ioffset_sedcol_pproc(i_myrank))
          IF (istatus /= NF_NOERR)
     &      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
       ENDIF
#else
        istatus = NF_PUT_ATT_INT(ncid, id_col, 'offset', NF_INT, 1, 0)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
#endif


                                    ! Levels
                                    ! ------
                                    ! - dim id
        istatus = NF_DEF_DIM(ncid, 'lev', ndn, dim_lev)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

                                    ! - dim variable id
        istatus = NF_DEF_VAR(ncid, 'lev', NF_INT, 1, dim_lev, id_lev)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

                                    ! - dim variable attributes
        istatus = NF_PUT_ATT_TEXT(ncid, id_lev, 'positive', 4, 'down')
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))


#ifdef MEDUSA_BASE2DT2D
                                    ! 'X' index
                                    ! ---------
                                    ! - dim id
        istatus = NF_DEF_DIM(ncid, c_name_ix, nix, dim_ix)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

                                    ! - dim variable id
        istatus = NF_DEF_VAR(ncid, c_name_ix, NF_INT,
     &                              1, dim_ix, id_ix)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

                                    ! - dim variable attributes
        istatus = NF_PUT_ATT_TEXT(ncid, id_ix, 'long_name',
     &                              nlen_lname_ix, c_lname_ix)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))


                                    ! 'Y' index
                                    ! ---------
                                    ! - dim id
        istatus = NF_DEF_DIM(ncid, c_name_jy, njy, dim_jy)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

                                    ! - dim variable id
        istatus = NF_DEF_VAR(ncid, c_name_jy, NF_INT,
     &                              1, dim_jy, id_jy)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

                                    ! - dim variable attributes
        istatus = NF_PUT_ATT_TEXT(ncid, id_jy, 'long_name',
     &                         nlen_lname_jy, c_lname_jy)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))


                                    ! 'XT' index
                                    ! ----------
                                    ! - dim id
        istatus = NF_DEF_DIM(ncid, c_name_itx, nitx_ncfile, dim_itx)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                     ! - dim variable id
       istatus = NF_DEF_VAR(ncid, c_name_itx, NF_INT,
     &                              1, dim_itx, id_itx)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    ! - dim variable attributes
        istatus = NF_PUT_ATT_TEXT(ncid, id_itx, 'long_name',
     &                              nlen_lname_itx, c_lname_itx)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

#  ifdef ALLOW_MPI
        IF (l_onewrites4many) THEN
          istatus = NF_PUT_ATT_INT(ncid, id_itx, 'offset', NF_INT, 1, 0)
          IF (istatus /= NF_NOERR)
     &      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
        ELSE
          it = 0                    ! get the offset
          CALL SNCA_CENT2GLOB_ITJT(it=it, ip=i_myip, jp=i_myjp)
          istatus = NF_PUT_ATT_INT(ncid, id_itx, 'offset', NF_INT, 1,
     &                              it)
          IF (istatus /= NF_NOERR)
     &      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
        ENDIF
#  else
        istatus = NF_PUT_ATT_INT(ncid, id_itx, 'offset', NF_INT, 1, 0)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
#  endif


                                    ! 'YT' index
                                    ! ----------
                                    ! - dim id
        istatus = NF_DEF_DIM(ncid, c_name_jty, njty_ncfile, dim_jty)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                     ! - dim variable id
        istatus = NF_DEF_VAR(ncid, c_name_jty, NF_INT,
     &                              1, dim_jty, id_jty)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    ! - dim variable attributes
        istatus = NF_PUT_ATT_TEXT(ncid, id_jty, 'long_name',
     &                         nlen_lname_jty, c_lname_jty)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

#  ifdef ALLOW_MPI
        IF (l_onewrites4many) THEN
          istatus = NF_PUT_ATT_INT(ncid, id_jty, 'offset', NF_INT, 1, 0)
          IF (istatus /= NF_NOERR)
     &      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
        ELSE
          jt = 0                    ! get the offset
          CALL SNCA_CENT2GLOB_ITJT(jt=jt, ip=i_myip, jp=i_myjp)
          istatus = NF_PUT_ATT_INT(ncid, id_jty, 'offset', NF_INT, 1,
     &                              it)
          IF (istatus /= NF_NOERR)
     &      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
        ENDIF
#  else
        istatus = NF_PUT_ATT_INT(ncid, id_jty, 'offset', NF_INT, 1, 0)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
#  endif
#endif


        !-----------------------
        ! Define data variables
        !-----------------------

                                    ! xzdn(lev)
                                    ! ---------
        istatus = NF_DEF_VAR(ncid, vsn_xzdn, NF_DOUBLE,
     &                              1, dim_lev, id_xzdn)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))
        istatus = NF_PUT_ATT_TEXT(ncid, id_xzdn,
     &                              'long_name', vll_xzdn, vln_xzdn)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))
        istatus = NF_PUT_ATT_TEXT(ncid, id_xzdn,
     &                              'units', ul_m, un_m)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))


                                    ! Area(col)
                                    ! ---------
        istatus = NF_DEF_VAR(ncid, 'area', NF_DOUBLE,
     &                              1, dim_col, id_area)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))
        istatus = NF_PUT_ATT_TEXT(ncid, id_area, 'long_name',
     &                              12, 'Surface_Area')
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))
        istatus = NF_PUT_ATT_TEXT(ncid, id_area, 'units', ul_m2, un_m2)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))


#ifdef MEDUSA_BASE2DT2D
                                    ! maskval(col)
                                    ! ------------

        istatus = NF_DEF_VAR(ncid, 'maskval', NF_INT,
     &                              1, dim_col, id_maskval)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_TEXT(ncid, id_maskval, 'long_name',
     &                              10, 'Mask_Value')
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_INT(ncid, id_maskval,
     &                              '_FillValue', NF_INT, 1, -1)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))


                                    ! col4ij_itjt(ix,jy, itx,jty)
                                    ! ---------------------------
        dim(1) = dim_ix
        dim(2) = dim_jy
        dim(3) = dim_itx
        dim(4) = dim_jty
      
        istatus = NF_DEF_VAR(ncid, 'col4ij_itjt', NF_INT,
     &                              4, dim(1:4), id_col4ij_itjt)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_TEXT(ncid, id_col4ij_itjt, 'long_name',
     &                              23, 'Column_index_of_ij_itjt')
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_INT(ncid, id_col4ij_itjt,
     &                              '_FillValue', NF_INT, 1, -1)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_INT(ncid, id_col4ij_itjt,
     &                              'valid_min', NF_INT, 1, 1)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_INT(ncid, id_col4ij_itjt,
     &                              'valid_max', NF_INT,
     &                              1, nsedcol_ncfile)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-4))


                                    ! xref4col(col)
                                    ! -------------
        istatus = NF_DEF_VAR(ncid, c_name_xref4col, NF_DOUBLE,
     &                              1, dim_col, id_xref4col)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_TEXT(ncid, id_xref4col, 'long_name',
     &                              nlen_lname_xref4col,
     &                              c_lname_xref4col)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-4))

        istatus = NF_PUT_ATT_TEXT(ncid, id_xref4col, 'units',
     &                              nlen_unit_xref4col,
     &                              c_unit_xref4col)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-4))


                                    ! yref4col(col)
                                    ! -------------
        istatus = NF_DEF_VAR(ncid, c_name_yref4col, NF_DOUBLE,
     &                              1, dim_col, id_yref4col)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_TEXT(ncid, id_yref4col, 'long_name',
     &                              nlen_lname_yref4col,
     &                              c_lname_yref4col)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-4))

        istatus = NF_PUT_ATT_TEXT(ncid, id_yref4col, 'units',
     &                              nlen_unit_yref4col,
     &                              c_unit_yref4col)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-4))


                                    ! i4col(col)
                                    ! ----------
        istatus = NF_DEF_VAR(ncid, c_name_i4col, NF_INT,
     &                              1, dim_col, id_i4col)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_TEXT(ncid, id_i4col, 'long_name',
     &                              nlen_lname_i4col, c_lname_i4col)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_INT(ncid, id_i4col,
     &                              '_FillValue', NF_INT, 1, -1)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_INT(ncid, id_i4col,
     &                              'valid_min',  NF_INT, 1, 1)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_INT(ncid, id_i4col,
     &                              'valid_max',  NF_INT, 1, nix)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))


                                    ! j4col(col)
                                    ! ----------
        istatus = NF_DEF_VAR(ncid, c_name_j4col, NF_INT,
     &                              1, dim_col, id_j4col)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))
        istatus = NF_PUT_ATT_TEXT(ncid, id_j4col, 'long_name',
     &                              nlen_lname_j4col, c_lname_j4col)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_INT(ncid, id_j4col,
     &                              '_FillValue', NF_INT, 1, -1)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_INT(ncid, id_j4col,
     &                              'valid_min',  NF_INT, 1, 1)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_INT(ncid, id_j4col,
     &                              'valid_max',  NF_INT, 1, njy)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))


                                    ! it4col(col)
                                    ! -----------
        istatus = NF_DEF_VAR(ncid, c_name_it4col, NF_INT,
     &                              1, dim_col, id_it4col)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_TEXT(ncid, id_it4col, 'long_name',
     &                              nlen_lname_it4col, c_lname_it4col)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_INT(ncid, id_it4col,
     &                              '_FillValue', NF_INT, 1, -1)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        it = 1
#  ifdef ALLOW_MPI
                                    ![MPI] if one file per process, then
                                    ! convert the central-based i=1 value
                                    ! to the global based one
        IF (.NOT. l_onewrites4many) THEN
          CALL SNCA_CENT2GLOB_ITJT(it=it, ip=i_myip, jp=i_myjp)
        ENDIF
#  endif
        istatus = NF_PUT_ATT_INT(ncid, id_it4col,
     &                              'valid_min', NF_INT, 1, it)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        it = nitx_ncfile
#  ifdef ALLOW_MPI
                                    ![MPI] if one file per process, then
                                    ! convert the central-based i=nitx_ncfile
                                    ! value to the global based one
        IF (.NOT. l_onewrites4many) THEN
          CALL SNCA_CENT2GLOB_ITJT(it=it, ip=i_myip, jp=i_myjp)
        ENDIF
#  endif
        istatus = NF_PUT_ATT_INT(ncid, id_it4col,
     &                              'valid_max', NF_INT, 1, it)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


                                    ! jt4col(col)
                                    ! -----------
        istatus = NF_DEF_VAR(ncid, c_name_jt4col, NF_INT,
     &                              1, dim_col, id_jt4col)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        istatus = NF_PUT_ATT_TEXT(ncid, id_jt4col, 'long_name',
     &                              nlen_lname_jt4col, c_lname_jt4col)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_INT(ncid, id_jt4col,
     &                              '_FillValue', NF_INT, 1, -1)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        jt = 1
#  ifdef ALLOW_MPI
                                    ![MPI] if one file per process, then
                                    ! convert the central-based j=1 value
                                    ! to the global based one
        IF (.NOT. l_onewrites4many) THEN
          CALL SNCA_CENT2GLOB_ITJT(jt=jt, ip=i_myip, jp=i_myjp)
        ENDIF
#  endif
        istatus = NF_PUT_ATT_INT(ncid, id_jt4col,
     &                              'valid_min', NF_INT, 1, jt)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        jt = njty_ncfile
#  ifdef ALLOW_MPI
                                    ![MPI] if one file per process, then
                                    ! convert the central-based j=njty_ncfile
                                    ! value to the global based one
        IF (.NOT. l_onewrites4many) THEN
          CALL SNCA_CENT2GLOB_ITJT(jt=jt, ip=i_myip, jp=i_myjp)
        ENDIF
#  endif
        istatus = NF_PUT_ATT_INT(ncid, id_jt4col,
     &                              'valid_max', NF_INT, 1, jt)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


                                    ! lon4ij_itjt(ix,jy, itx, jty)
                                    ! ----------------------------
        istatus = NF_DEF_VAR(ncid, c_name_lon4ij_itjt, NF_DOUBLE,
     &                              4, dim(1:4), id_lon4ij_itjt)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_TEXT(ncid, id_lon4ij_itjt, 'long_name',
     &                              nlen_lname_lon4ij_itjt,
     &                              c_lname_lon4ij_itjt)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_TEXT(ncid, id_lon4ij_itjt, 'units',
     &                              nlen_unit_lon4ij_itjt,
     &                              c_unit_lon4ij_itjt)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-4))


                                    ! lat4ij_itjt(ix,jy, itx, jty)
                                    ! ----------------------------
        istatus = NF_DEF_VAR(ncid, c_name_lat4ij_itjt, NF_DOUBLE,
     &                              4, dim(1:4), id_lat4ij_itjt)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_TEXT(ncid, id_lat4ij_itjt, 'long_name',
     &                              nlen_lname_lat4ij_itjt,
     &                              c_lname_lat4ij_itjt)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_TEXT(ncid, id_lat4ij_itjt, 'units',
     &                              nlen_unit_lat4ij_itjt,
     &                              c_unit_lat4ij_itjt)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-4))

#endif


        !----------------------
        ! Put global attributes
        !----------------------

        IF (PRESENT(title_string)) THEN
          istatus = NF_PUT_ATT_TEXT(ncid, NF_GLOBAL, 'title',
     &                          LEN_TRIM(title_string), title_string)
          IF (istatus /= NF_NOERR)
     &      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))
        ENDIF


        !--------------------
        ! End define mode
        !--------------------

        istatus = NF_ENDDEF(ncid)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

      ENDIF

                                    ! Set 'col' coordinate variable
                                    ! equal to its index value
      iarr_c(:) = (/ (n, n=1, nsedcol_central) /)
#ifdef ALLOW_MPI
     &             + ioffset_sedcol_pproc(i_myrank)
#endif

      CALL SNCA_PUT_C_INT(ncid, id_col, iarr_c)


      IF (l_file_is_mine) THEN
                                    ! Set 'lev' coordinate variable
                                    ! equal to its index value
        DO k = 1, idnb-idnw+1
          i = k-1+idnw
          istatus = NF_PUT_VAR1_INT(ncid, id_lev, k, i)
          IF (istatus /= NF_NOERR)
     &      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
        ENDDO

      ENDIF



#ifdef MEDUSA_BASE2DT2D
      IF (l_file_is_mine) THEN
                                    ! Set 'ix' coordinate variable
                                    ! equal to its index value
        DO k = 1, nix

          i = k

          istatus = NF_PUT_VAR1_INT(ncid, id_ix, k, i)
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        ENDDO

                                    ! Set 'jy' coordinate variable
                                    ! equal to its index value
        DO k = 1, njy

          j = k

          istatus = NF_PUT_VAR1_INT(ncid, id_jy, k, j)
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        ENDDO

                                    ! Set 'itx' coordinate variable
                                    ! equal to its index value
        DO k = 1, nitx_ncfile

          it = k

#  ifdef ALLOW_MPI
                                    ![MPI] if one file per process, then
                                    ! convert the central-based itx values
                                    ! to the global based
          IF (.NOT. l_onewrites4many) THEN
            CALL SNCA_CENT2GLOB_ITJT(it=it, ip=i_myip, jp=i_myjp)
          ENDIF
#  endif

          istatus = NF_PUT_VAR1_INT(ncid, id_itx, k, it)
          IF (istatus /= NF_NOERR)
     &      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

        ENDDO

                                    ! Set 'jty' coordinate variable
                                    ! equal to its index value
        DO k = 1, njty_ncfile

          jt = k

#  ifdef ALLOW_MPI
                                    ![MPI] if one file per process, then
                                    ! convert the central-based jty values
                                    ! to the global-based
          IF (.NOT. l_onewrites4many) THEN
            CALL SNCA_CENT2GLOB_ITJT(jt=jt, ip=i_myip, jp=i_myjp)
          ENDIF
#  endif

          istatus = NF_PUT_VAR1_INT(ncid, id_jty, k, jt)
          IF (istatus /= NF_NOERR)
     &      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

        ENDDO

      ENDIF
#endif


      IF (l_file_is_mine) THEN
                                    ! xzdn(lev)
        CALL GRID_DEF(xzdn)
        istatus = NF_PUT_VAR_DOUBLE(ncid, id_xzdn, xzdn)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

      ENDIF


#ifdef MEDUSA_BASE2DT2D
                                    ! col4ij_itjt
      DO i = 1, nix
      DO j  = 1, njy
      DO it = 1, nitx
      DO jt = 1, njty

        CALL COLUMN_IJ_ITJT2N(i, j, it, jt, iflag, k)

        IF (iflag <= 0) THEN        ! Ignore warning for "Not-at-seafloor"

#  ifdef ALLOW_MPI
          IF (k /= jp_is_not_at_seafloor_idx) THEN
            iarr_4d(i,j,it,jt) = k + ioffset_sedcol_pproc(i_myrank)
          ELSE
            iarr_4d(i,j,it,jt) = jp_is_not_at_seafloor_idx
          ENDIF
#  else
          iarr_4d(i,j,it,jt) = k
#  endif

        ELSE

          WRITE(jp_stderr,*) 'Error upon COLUMN_IJ_ITJT2N call:'
          WRITE(jp_stderr,*) ' called with i,j = ', i, j
          WRITE(jp_stderr,*) ' called with it,jt = ', it,jt
          WRITE(jp_stderr,*) ' returned iflag = ', iflag
          WRITE(jp_stderr,*) 'Aborting!'
          CALL ABORT_MEDUSA()

        ENDIF

      ENDDO
      ENDDO
      ENDDO
      ENDDO

      CALL SNCA_PUT_4D_INT(ncid, id_col4ij_itjt, iarr_4d)
#endif


#ifdef MEDUSA_BASE2DT2D
                                    ! i4col, j4col
      DO k = 1, nsedcol_central

        CALL COLUMN_N2IJ_ITJT(k, iflag, i, j, it, jt)

        IF (iflag == 0) THEN

#  ifdef ALLOW_MPI
                                    ![MPI]: need to correct i and j:
                                    ! (i,j,it,jt) central -> (i,j,it,jt) global
          CALL SNCA_CENT2GLOB_ITJT(it,jt, ip=i_myip, jp=i_myjp)
#  endif
          iarri_c(k)  = i
          iarrj_c(k)  = j
          iarrit_c(k) = it
          iarrjt_c(k) = jt

        ELSE

          WRITE(jp_stderr,*) 'Error upon COLUMN_N2IJ_ITJT call:'
          WRITE(jp_stderr,*) ' called with k = ', k
          WRITE(jp_stderr,*) ' returned iflag = ', iflag
          WRITE(jp_stderr,*) 'Aborting!'
          CALL ABORT_MEDUSA()

        ENDIF

      ENDDO

      CALL SNCA_PUT_C_INT(ncid, id_i4col,  iarri_c)
      CALL SNCA_PUT_C_INT(ncid, id_j4col,  iarrj_c)

      CALL SNCA_PUT_C_INT(ncid, id_it4col, iarrit_c)
      CALL SNCA_PUT_C_INT(ncid, id_jt4col, iarrjt_c)
#endif


#ifdef MEDUSA_BASE2DT2D
                                    ! xref4col, yref4jcol
      DO k = 1, nsedcol_central

        CALL COLUMN_N2XY(k, iflag, x, y)

        IF (iflag == 0) THEN

          darrx_c(k) = x
          darry_c(k) = y

        ELSE

          WRITE(jp_stderr,*) 'Error upon COLUMN_N2XY call:'
          WRITE(jp_stderr,*) ' called with k = ', k
          WRITE(jp_stderr,*) ' returned iflag = ', iflag
          WRITE(jp_stderr,*) 'Aborting!'
          CALL ABORT_MEDUSA()

        ENDIF

      ENDDO

      CALL SNCA_PUT_C_DOUBLE(ncid, id_xref4col, darrx_c)
      CALL SNCA_PUT_C_DOUBLE(ncid, id_yref4col, darry_c)


                                    ! lon4ij_itjt, lat4ij_itjt
      CALL SNCA_PUT_4D_DOUBLE(ncid, id_lon4ij_itjt, darrx_4d)
      CALL SNCA_PUT_4D_DOUBLE(ncid, id_lat4ij_itjt, darry_4d)


                                    ! Surface areas of grid_elements
      DO k = 1, nsedcol_central

        CALL COLUMN_AREA4N(k, iflag, sfc_area)

        IF (iflag == 0) THEN

          darr_c(k) = sfc_area

        ELSE

          WRITE(jp_stderr,*) 'Error upon COLUMN_AREA4N call:'
          WRITE(jp_stderr,*) ' called with k = ', k
          WRITE(jp_stderr,*) ' returned iflag = ', iflag
          WRITE(jp_stderr,*) 'Aborting!'
          CALL ABORT_MEDUSA()

        ENDIF

      ENDDO

      CALL SNCA_PUT_C_DOUBLE(ncid, id_area, darr_c)


                                    ! maskval
      DO k = 1, nsedcol_central

        CALL COLUMN_N2MASKVAL(k, iflag, mval)

        IF (iflag == 0) THEN

          iarr_c(k) = mval

        ELSE

          WRITE(jp_stderr,*) 'Error upon COLUMN_N2MASKVAL call:'
          WRITE(jp_stderr,*) ' called with k = ', k
          WRITE(jp_stderr,*) ' returned iflag = ', iflag
          WRITE(jp_stderr,*) 'Aborting!'
          CALL ABORT_MEDUSA()

        ENDIF

      ENDDO

      CALL SNCA_PUT_C_INT(ncid, id_maskval, iarr_c)
#endif


      IF (l_file_is_mine) THEN
        istatus = NF_CLOSE(ncid)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
      ENDIF


      DEALLOCATE(iarr_c)
      DEALLOCATE(darr_c)

#ifdef MEDUSA_BASE2DT2D
      DEALLOCATE(iarr_4d)

      IF (i_myrank == jp_exeproc_root) THEN
        DEALLOCATE(rarr_tlong)
        DEALLOCATE(rarr_tlat)
        DEALLOCATE(darrx_6d)
        DEALLOCATE(darry_6d)
      ENDIF

      DEALLOCATE(iarri_c)
      DEALLOCATE(iarrj_c)

      DEALLOCATE(darrx_c)
      DEALLOCATE(darry_c)
      DEALLOCATE(darrx_4d)
      DEALLOCATE(darry_4d)
#endif


      CALL SNCA_RESET()


      RETURN



      CONTAINS



#ifdef ALLOW_MPI
#  ifdef MEDUSA_BASE2DT2D
!-----------------------------------------------------------------------
      SUBROUTINE SNCA_CENT2GLOB_ITJT(it, jt, ip, jp)
!-----------------------------------------------------------------------

! Adjust the i and j coordinates on the process' subgrid
! to their counterparts on the global grid if
! - running in an MPI environment
! - and l_onewritesformany
! Assumes (npx,npy) array of processes, each process dealing with an
! (nix,njy) array, with no overlapping.


      IMPLICIT NONE

      INTEGER, INTENT(INOUT), OPTIONAL :: it, jt
      INTEGER, INTENT(IN)              :: ip, jp


      IF (PRESENT(it)) it = (ip-1)*nitx + it
      IF (PRESENT(jt)) jt = (jp-1)*njty + jt

      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE SNCA_CENT2GLOB_ITJT
!-----------------------------------------------------------------------
#endif
#endif


!-----------------------------------------------------------------------
      SUBROUTINE SNCA_SETUP()
!-----------------------------------------------------------------------

      USE mod_seafloor_central,     ONLY: N_COLUMNS_USED
#ifdef MEDUSA_BASE2DT2D
      USE mod_seafloor_central,     ONLY: IJ_ITJT_COLUMNS_USED
#endif

#ifdef ALLOW_MPI
      USE mod_execontrol_medusa,    ONLY: MEDEXE_NPROC,
     &                                    MEDEXE_MPI_COMM,
     &                                    MEDEXE_MPI_COMM_RANK,
     &                                    MEDEXE_MPI_GETPARTITIONING,
     &                                    lp_exeproc_singleproc_nc
#  ifdef MEDUSA_BASE2DT2D
      USE mod_execontrol_medusa,    ONLY: MEDEXE_MPI_GETTOPO_DIMLENS,
     &                                    MEDEXE_MPI_TOPO2D_PP4RANK
#  endif
#endif


      IMPLICIT NONE


#ifdef ALLOW_MPI
#  ifdef MEDUSA_BASE2DT2D
      INTEGER, DIMENSION(2) :: ilens_mpitopo
#  endif
#endif

                                    ! Get the number of columns in the
                                    ! mod_seafloor_central of the
                                    ! current process
      CALL N_COLUMNS_USED(nsedcol_central)

                                    ! If nsedcol_central == -1, the
                                    ! basic setup has not yet been
                                    ! done, and it does not make any
                                    ! sense to proceed
      IF (nsedcol_central < 0) THEN
        WRITE(jp_stderr,*) '[STORE_NC_AUX/SNCA_SETUP] error: ' //
     &    'Sea-floor setup not yet done -- aborting!'
        CALL ABORT_MEDUSA()
      ENDIF


#ifdef MEDUSA_BASE2DT2D
                                    ! Get the number of seafloor grid
                                    ! elements along the X and Y directions
                                    ! [MPI] would actually have to check
                                    ! if these are all the same
      CALL IJ_ITJT_COLUMNS_USED(nix, njy, nitx, njty)
      n4d = nix*njy*nitx*njty
#endif


#ifdef ALLOW_MPI
      n_cprocs = MEDEXE_NPROC()     ! Get number of processes

                                    ! Get communicator of Medusa
      i_mycomm = MEDEXE_MPI_COMM()

                                    ! Get rank of process executing this instance
      i_myrank = MEDEXE_MPI_COMM_RANK()

#  ifdef MEDUSA_BASE2DT2D
                                    ! Get the 2D process array shape.
      CALL MEDEXE_MPI_GETTOPO_DIMLENS(ilens_mpitopo)
      npx = ilens_mpitopo(1)
      npy = ilens_mpitopo(2)
                                    ! Get the process coordinates in the 2D topo
                                    ! for the process executing this instance
      CALL MEDEXE_MPI_TOPO2D_PP4RANK(i_myrank, i_myip, i_myjp)
#  endif


                                    ! Now set up the file writing
                                    ! distribution logic:
      l_file_is_mine = .TRUE.       ! - by default, each process writes
                                    !   its own files

      l_onewrites4many = .FALSE.    ! - by default: no process writes data
                                    !   for others; this must be .FALSE.
                                    !   if n_cprocs==1 ("many" means ">1")

      IF (n_cprocs > 1) THEN        ! When running in a multi-processor
                                    ! environment with more than one process,
                                    ! then this may need to be adapted:
        IF (lp_exeproc_singleproc_nc) THEN  ! - if single-processor I/O
                                            !   for NetCDF files is required,
          l_onewrites4many = .TRUE.         !   set flag to indicate that the
                                            !   writing process writes for
                                            !   other (>1) processes than itself
          IF (i_myrank /= jp_exeproc_ncio) THEN     ! - and deactivate writing for other
            l_file_is_mine = .FALSE.                !   processes than the writing one
          ENDIF

        ELSE                        ! Multi-process environment where each
                                    ! process controls its own file. However,
                                    ! do not create a file if there are no
                                    ! data to write. This could happen if one
                                    ! process gets attributed a domain without
                                    ! seafloor points.
          IF (nsedcol_central == 0) l_file_is_mine = .FALSE.

        ENDIF

      ENDIF
                                    ! Get the number of columns in all
                                    ! the mod_seafloor_central's of all
                                    ! concurrent processes and the
                                    ! partitioning information
      ALLOCATE(nsedcol_pproc(0:n_cprocs-1))
      ALLOCATE(ioffset_sedcol_pproc(0:n_cprocs-1))

      CALL MEDEXE_MPI_GETPARTITIONING(nsedcol_global, iproc_1stocn,
     &                              nsedcol_pproc, ioffset_sedcol_pproc)

                                    ! If single-processor NetCDF I/O, and
                                    ! execution environment includes more
                                    ! than one processor, then allocate
                                    ! work-space memory to receive the data
                                    ! by the writing process:
      IF (l_onewrites4many) THEN

        nsedcol_ncfile = nsedcol_global

        IF (i_myrank == jp_exeproc_ncio) THEN
          ALLOCATE(iarr_mpirecv_c(nsedcol_ncfile))
          ALLOCATE(darr_mpirecv_c(nsedcol_ncfile))
        ELSE
          ALLOCATE(iarr_mpirecv_c(0))     ! pro-forma allocation
          ALLOCATE(darr_mpirecv_c(0))     ! pro-forma allocation
        ENDIF

#  ifdef MEDUSA_BASE2DT2D
        nitx_ncfile = nitx*npx
        njty_ncfile = njty*npy

        IF (i_myrank == jp_exeproc_ncio) THEN
          ALLOCATE(iarr_mpirecv_4d(nix, njy, nitx, njty))
          ALLOCATE(darr_mpirecv_4d(nix, njy, nitx, njty))
        ELSE
          ALLOCATE(iarr_mpirecv_4d(0, 0, 0, 0))     ! pro-forma allocation
          ALLOCATE(darr_mpirecv_4d(0, 0, 0, 0))     ! pro-forma allocation
        ENDIF
#  endif

      ELSE

        nsedcol_ncfile = nsedcol_central

#  ifdef MEDUSA_BASE2DT2D
        nitx_ncfile = nitx
        njty_ncfile = njty
#  endif

      ENDIF

#  ifdef DEBUG
      WRITE(jp_stddbg, '(A,I0)')
     &   '[SNCA_SETUP] n_cprocs = ', n_cprocs
      WRITE(jp_stddbg, '(A,I0)')
     &   '[SNCA_SETUP] i_myrank = ', i_myrank
#    ifdef MEDUSA_BASE2DT2D
      WRITE(jp_stddbg, '(A,"("I0,",",I0,")")')
     &   '[SNCA_SETUP] (i_myip, i_myjp) = ', i_myip, i_myjp
#    endif
      WRITE(jp_stddbg, '(A,I0)')
     &   '[SNCA_SETUP] nsedcol_central = ', nsedcol_central
      WRITE(jp_stddbg, '(A,I0)')
     &   '[SNCA_SETUP] nsedcol_global = ', nsedcol_global
      WRITE(jp_stddbg, '(A,I0)')
     &   '[SNCA_SETUP] nsedcol_ncfile = ', nsedcol_ncfile
#    ifdef MEDUSA_BASE2DT2D
      WRITE(jp_stddbg, '(A,I0," * ",I0)')
     &   '[SNCA_SETUP] nix * njy = ', nix, njy
      WRITE(jp_stddbg, '(A,I0," * ",I0)')
     &   '[SNCA_SETUP] nitx * njty = ', nitx, njty
      WRITE(jp_stddbg, '(A,I0," * ",I0)')
     &   '[SNCA_SETUP] nitx_ncfile * njty_ncfile = ',
     &   nitx_ncfile, njty_ncfile
#    endif
      WRITE(jp_stddbg, '(A,L1)')
     &   '[SNCA_SETUP] l_onewrites4many = ', l_onewrites4many
      WRITE(jp_stddbg, '(A,L1)')
     &   '[SNCA_SETUP] l_file_is_mine = ', l_file_is_mine
#  endif
#else
                                    ! In single-processor environments the
                                    ! running process always owns its files
                                    ! and writes only nsedcol_central-long
                                    ! records
      l_file_is_mine = .TRUE.
      nsedcol_ncfile = nsedcol_central

#  ifdef MEDUSA_BASE2DT2D
      nitx_ncfile = nitx
      njty_ncfile = njty
#  endif

#  ifdef DEBUG
      WRITE(jp_stddbg, '(A,I0)')
     &   '[SNCA_SETUP] nsedcol_central = ', nsedcol_central
      WRITE(jp_stddbg, '(A,I0)')
     &   '[SNCA_SETUP] nsedcol_ncfile = ', nsedcol_ncfile
#    ifdef MEDUSA_BASE2DT2D
      WRITE(jp_stddbg, '(A,I0," * ",I0)')
     &   '[SNCA_SETUP] nix * njy = ', nix, njy
      WRITE(jp_stddbg, '(A,I0," * ",I0)')
     &   '[SNCA_SETUP] nitx * njty = ', nitx, njty
      WRITE(jp_stddbg, '(A,I0," * ",I0)')
     &   '[SNCA_SETUP] nitx_ncfile * njty_ncfile = ',
     &   nitx_ncfile, njty_ncfile
#    endif
      WRITE(jp_stddbg, '(A,L1)')
     &   '[SNCA_SETUP] l_file_is_mine = ', l_file_is_mine
#  endif
#endif


      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE SNCA_SETUP
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE SNCA_RESET()
!-----------------------------------------------------------------------


      IMPLICIT NONE


      nsedcol_central = -1
      nsedcol_ncfile  = -1

#ifdef MEDUSA_BASE2DT2D
      nix  = -1
      njy  = -1
      nitx = -1
      njty = -1
      nitx_ncfile = -1
      njty_ncfile = -1
#endif

#ifdef ALLOW_MPI
                                    ! Deallocate the work-space memory reserved
                                    ! to receive the data in the writing process.
                                    ! Deallocate in all processes: in the non
                                    ! writing ones, the arrays are also allocated,
                                    ! with 0 size.
      IF (l_onewrites4many) THEN
        DEALLOCATE(iarr_mpirecv_c)
        DEALLOCATE(darr_mpirecv_c)
#  ifdef MEDUSA_BASE2DT2D
        DEALLOCATE(iarr_mpirecv_4d)
        DEALLOCATE(darr_mpirecv_4d)
#  endif
      ENDIF

      DEALLOCATE(nsedcol_pproc)
      DEALLOCATE(ioffset_sedcol_pproc)

      nsedcol_global = -1
      iproc_1stocn   = -1

      n_cprocs = -1
      i_myrank = MPI_PROC_NULL
#  ifdef MEDUSA_BASE2DT2D
      i_myip   = -1
      i_myjp   = -1
#  endif
#endif


      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE SNCA_RESET
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE SNCA_PUT_C_DOUBLE(ncid, id_var, dvar_c)
!-----------------------------------------------------------------------

! Write out one complete record provided by the DOUBLE PRECISION array
! dvar_c(1:nsedcol_central) into a NetCDF variable with dimension (dim_col)
! Under MPI, stitch them together if l_onewrites4many==.TRUE.

#ifdef ALLOW_MPI
      USE mpi,                      ONLY: MPI_DOUBLE_PRECISION
#endif


      IMPLICIT NONE


      INTEGER,          INTENT(IN)               :: ncid
      INTEGER,          INTENT(IN)               :: id_var
      DOUBLE PRECISION, INTENT(IN), DIMENSION(:) :: dvar_c


      INTEGER :: istatus


#ifdef ALLOW_MPI
      IF (l_onewrites4many) THEN

        CALL MPI_GATHERV(dvar_c(:),
     &    nsedcol_central, MPI_DOUBLE_PRECISION,
     &    darr_mpirecv_c(:),
     &    nsedcol_pproc(:), ioffset_sedcol_pproc(:),
     &    MPI_DOUBLE_PRECISION,
     &    jp_exeproc_ncio, i_mycomm, istatus)

        IF (i_myrank == jp_exeproc_ncio) THEN
          istatus = NF_PUT_VAR_DOUBLE(ncid, id_var, darr_mpirecv_c(:))
          IF (istatus /= NF_NOERR)
     &      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
        ENDIF

      ELSE

        istatus = NF_PUT_VAR_DOUBLE(ncid, id_var, dvar_c(:))
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

      ENDIF
#else
      istatus = NF_PUT_VAR_DOUBLE(ncid, id_var, dvar_c(:))
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
#endif


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE SNCA_PUT_C_DOUBLE
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE SNCA_PUT_C_INT(ncid, id_var, ivar_c)
!-----------------------------------------------------------------------

! Write out one complete record provided by the INTEGER array
! ivar_c(1:nsedcol_central) into a NetCDF variable with dimension (dim_col)
! Under MPI, stitch them together if l_onewrites4many==.TRUE.

#ifdef ALLOW_MPI
      USE mpi,                      ONLY: MPI_INTEGER
#endif


      IMPLICIT NONE


      INTEGER, INTENT(IN)               :: ncid
      INTEGER, INTENT(IN)               :: id_var
      INTEGER, INTENT(IN), DIMENSION(:) :: ivar_c


      INTEGER :: istatus


#ifdef ALLOW_MPI
      IF (l_onewrites4many) THEN

        CALL MPI_GATHERV(ivar_c(:),
     &    nsedcol_central, MPI_INTEGER,
     &    iarr_mpirecv_c(:),
     &    nsedcol_pproc(:), ioffset_sedcol_pproc(:),
     &    MPI_INTEGER,
     &    jp_exeproc_ncio, i_mycomm, istatus)

        IF (i_myrank == jp_exeproc_ncio) THEN
          istatus = NF_PUT_VAR_INT(ncid, id_var, iarr_mpirecv_c(:))
          IF (istatus /= NF_NOERR)
     &      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
        ENDIF

      ELSE

        istatus = NF_PUT_VAR_INT(ncid, id_var, ivar_c(:))
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

      ENDIF
#else
      istatus = NF_PUT_VAR_INT(ncid, id_var, ivar_c(:))
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
#endif


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE SNCA_PUT_C_INT
!-----------------------------------------------------------------------



#ifdef MEDUSA_BASE2DT2D
!-----------------------------------------------------------------------
      SUBROUTINE SNCA_PUT_4D_INT(ncid, id_var, ivar_4d)
!-----------------------------------------------------------------------

! Write out one sub-record provided by the INTEGER array
! ivar_4d(1:nix,1:njy,1:nitx,1:njty) into a NetCDF array with dimension
! (1:nix,1:njy, 1:nitx_ncfile,1:njty_ncfile).
! Under MPI, stitch them together if l_onewrites4many==.TRUE.

#  ifdef ALLOW_MPI
      USE mpi,                      ONLY: MPI_INTEGER, MPI_STATUS_SIZE
      USE mod_execontrol_medusa,    ONLY: MEDEXE_MPI_TOPO2D_PP4RANK
#  endif


      IMPLICIT NONE


      INTEGER, INTENT(IN)                      :: ncid
      INTEGER, INTENT(IN)                      :: id_var
      INTEGER, INTENT(IN), DIMENSION(:,:, :,:) :: ivar_4d


      INTEGER :: istatus
      INTEGER, DIMENSION(4) :: istart, ncount
#  ifdef ALLOW_MPI
      INTEGER :: i_othrank
      INTEGER, DIMENSION(MPI_STATUS_SIZE) :: impi_status
#  endif

#  ifdef DEBUG
      CHARACTER(LEN=*), PARAMETER :: cfmt_dbg_a
     &  = '("[STORE_NC_AUX/SNCA_PUT_4D_INT] debug: ", A)'
#  endif


      IF (ANY(SHAPE(ivar_4d) /= (/ nix, njy, nitx, njty /))) THEN

        WRITE(jp_stderr, '("[SNCA_PUT_4D_INT] error: ")', ADVANCE='NO')
        WRITE(jp_stderr, '("incorrect shape for ivar_4d")')
        WRITE(jp_stderr, '(" expected: (/", 3(I0, ","), I0, "/)")')
     &    nix, njy, nitx, njty
        WRITE(jp_stderr, '(" actually: (/", 3(I0, ","), I0, "/)")')
     &    SHAPE(ivar_4d)
        WRITE(jp_stderr, '("Aborting!")')

        CALL ABORT_MEDUSA()

      ENDIF


#  ifdef ALLOW_MPI
      IF (l_onewrites4many) THEN

        IF (i_myrank == jp_exeproc_ncio) THEN

          DO i_othrank = 0, n_cprocs-1

            CALL MEDEXE_MPI_TOPO2D_PP4RANK(i_othrank, ip, jp)

            it = 1
            jt = 1
            CALL SNCA_CENT2GLOB_ITJT(it=it, jt=jt, ip=ip, jp=jp)

            istart(1:4) = (/   1,   1,   it,   jt /)
            ncount(1:4) = (/ nix, njy, nitx, njty /)

            IF (i_othrank == jp_exeproc_ncio) THEN

              istatus = NF_PUT_VARA_INT(ncid, id_var,
     &                              istart(1:4), ncount(1:4),
     &                              ivar_4d(:,:, :,:))
              IF (istatus /= NF_NOERR)
     &         CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-4))

            ELSE

#    ifdef DEBUG
              WRITE(jp_stddbg, cfmt_dbg_a, ADVANCE='NO')
     &          'Now receiving arrays from process '
              WRITE(jp_stddbg, '(I0, " (ip=",I0,",jp=",I0,")")')
     &          i_othrank, ip, jp
#    endif

              CALL MPI_RECV(iarr_mpirecv_4d, n4d, MPI_INTEGER,
     &                              i_othrank, i_othrank,
     &                              i_mycomm, impi_status, iflag)

              istatus = NF_PUT_VARA_INT(ncid, id_var,
     &                              istart(1:4), ncount(1:4),
     &                              iarr_mpirecv_4d(:,:, :,:))
              IF (istatus /= NF_NOERR)
     &         CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-4))

            ENDIF

          ENDDO

        ELSE

#    ifdef DEBUG
          WRITE(jp_stddbg, cfmt_dbg_a, ADVANCE='NO')
     &      'Now sending arrays to process '
          WRITE(jp_stddbg, '(I0)') jp_exeproc_ncio
#    endif

          CALL MPI_SEND(ivar_4d, n4d, MPI_INTEGER,
     &                              jp_exeproc_ncio, i_myrank,
     &                              i_mycomm, iflag)

        ENDIF


      ELSE

        istatus = NF_PUT_VAR_INT(ncid, id_var, ivar_4d(:,:, :,:))
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

      ENDIF
#  else
      istatus = NF_PUT_VAR_INT(ncid, id_var, ivar_4d(:,:, :,:))
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
#  endif


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE SNCA_PUT_4D_INT
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
      SUBROUTINE SNCA_PUT_4D_DOUBLE(ncid, id_var, dvar_4d)
!-----------------------------------------------------------------------

! Write out one sub-record provided by the DOUBLE PRECISION array
! dvar_4d(1:nix,1:njy,1:nitx,1:njty) into a NetCDF array with dimension
! (1:nix,1:njy, 1:nitx_ncfile,1:njty_ncfile).
! Under MPI, stitch them together if l_onewrites4many==.TRUE.

#  ifdef ALLOW_MPI
      USE mpi,                      ONLY: MPI_DOUBLE_PRECISION,
     &                                    MPI_STATUS_SIZE
      USE mod_execontrol_medusa,    ONLY: MEDEXE_MPI_TOPO2D_PP4RANK
#  endif


      IMPLICIT NONE


      INTEGER, INTENT(IN)                               :: ncid
      INTEGER, INTENT(IN)                               :: id_var
      DOUBLE PRECISION, INTENT(IN), DIMENSION(:,:, :,:) :: dvar_4d


      INTEGER :: istatus
      INTEGER, DIMENSION(4) :: istart, ncount
#  ifdef ALLOW_MPI
      INTEGER :: i_othrank
      INTEGER, DIMENSION(MPI_STATUS_SIZE) :: impi_status
#  endif

#  ifdef DEBUG
      CHARACTER(LEN=*), PARAMETER :: cfmt_dbg_a
     &  = '("[STORE_NC_AUX/SNCA_PUT_4D_DOUBLE] debug: ", A)'
#  endif

      IF (ANY(SHAPE(dvar_4d) /= (/ nix, njy, nitx, njty /))) THEN

        WRITE(jp_stderr, '("[SNCA_PUT_4D_DOUBLE] error: ")',
     &                               ADVANCE='NO')
        WRITE(jp_stderr, '("incorrect shape for dvar_4d")')
        WRITE(jp_stderr, '(" expected: (/", 3(I0, ","), I0, "/)")')
     &    nix, njy, nitx, njty
        WRITE(jp_stderr, '(" actually: (/", 3(I0, ","), I0, "/)")')
     &    SHAPE(dvar_4d)
        WRITE(jp_stderr, '("Aborting!")')

        CALL ABORT_MEDUSA()

      ENDIF


#  ifdef ALLOW_MPI
      IF (l_onewrites4many) THEN

        IF (i_myrank == jp_exeproc_ncio) THEN

          DO i_othrank = 0, n_cprocs-1

            CALL MEDEXE_MPI_TOPO2D_PP4RANK(i_othrank, ip, jp)

            it = 1
            jt = 1
            CALL SNCA_CENT2GLOB_ITJT(it=it, jt=jt, ip=ip, jp=jp)

            istart(1:4) = (/   1,   1,   it,   jt /)
            ncount(1:4) = (/ nix, njy, nitx, njty /)

            IF (i_othrank == jp_exeproc_ncio) THEN

              istatus = NF_PUT_VARA_DOUBLE(ncid, id_var,
     &                              istart(1:4), ncount(1:4),
     &                              dvar_4d(:,:, :,:))
              IF (istatus /= NF_NOERR)
     &         CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-4))

            ELSE

#    ifdef DEBUG
              WRITE(jp_stddbg, cfmt_dbg_a, ADVANCE='NO')
     &          'Now receiving arrays from process '
              WRITE(jp_stddbg, '(I0, " (ip=",I0,",jp=",I0,")")')
     &          i_othrank, ip, jp
#    endif

              CALL MPI_RECV(darr_mpirecv_4d, n4d,
     &                              MPI_DOUBLE_PRECISION,
     &                              i_othrank, i_othrank,
     &                              i_mycomm, impi_status, iflag)

              istatus = NF_PUT_VARA_DOUBLE(ncid, id_var,
     &                              istart(1:4), ncount(1:4),
     &                              darr_mpirecv_4d(:,:, :,:))
              IF (istatus /= NF_NOERR)
     &         CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-4))

            ENDIF

          ENDDO

        ELSE

#    ifdef DEBUG
          WRITE(jp_stddbg, cfmt_dbg_a, ADVANCE='NO')
     &      'Now sending arrays to process '
          WRITE(jp_stddbg, '(I0)') jp_exeproc_ncio
#    endif

          CALL MPI_SEND(dvar_4d, n4d, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_ncio, i_myrank,
     &                              i_mycomm, iflag)

        ENDIF


      ELSE

        istatus = NF_PUT_VAR_DOUBLE(ncid, id_var, dvar_4d(:,:, :,:))
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

      ENDIF
#  else
      istatus = NF_PUT_VAR_DOUBLE(ncid, id_var, dvar_4d(:,:, :,:))
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
#  endif


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE SNCA_PUT_4D_DOUBLE
!-----------------------------------------------------------------------
#endif



!=======================================================================
      END SUBROUTINE STORE_NC_AUX
!=======================================================================
