!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#ifdef CFN_THISFILE
#undef CFN_THISFILE
#endif
#define CFN_THISFILE "mod_coupsim_subr.F"
#ifndef __LINE__
#define __LINE__ 0
#endif
!=======================================================================
      MODULE MOD_COUPSIM_SUBR
!=======================================================================


      IMPLICIT NONE

      PRIVATE

      PUBLIC :: COUPSIM_BIOGEOCHEM_STEP


      CHARACTER(LEN=*), PARAMETER :: cfn_thisfile =
     &  CFN_THISFILE


      CONTAINS


!-----------------------------------------------------------------------
      SUBROUTINE COUPSIM_BIOGEOCHEM_STEP
!-----------------------------------------------------------------------


      USE mod_defines_medusa
      USE mod_execontrol_medusa,    ONLY: ABORT_MEDUSA
      USE mod_logunits

      USE mpi
      USE mod_execontrol_medusa,    ONLY: jp_exeproc_root,
     &                                    MEDEXE_NPROC, MEDEXE_MPI_COMM,
     &                                    MEDEXE_MPI_COMM_RANK,
     &                                    MEDEXE_MPI_GETTOPO_DIMLENS,
     &                                    MEDEXE_MPI_GETPARTITIONING,
     &                                    MEDEXE_MPI_TOPO2D_PP4RANK

      USE mod_netcdfinc,            ONLY: NF_OPEN, NF_CLOSE,
     &                                    NF_NOERR, NF_NOWRITE,
     &                                    NF_INQ_VARID,
     &                                    NF_GET_VAR_DOUBLE,
     &                                    NF_GET_VAR_REAL,
     &                                    HANDLE_NCERRORS

      USE mod_basicdata_medusa
      USE mod_indexparam
      USE mod_materialcharas
      USE mod_logunits

      USE mod_seafloor_wdata

      USE mod_seafloor_central,     ONLY: N_COLUMNS_USED,
     &                                    IJ_ITJT_COLUMNS_USED,
     &                                    COLUMN_N2IJ_ITJT

      USE mod_files_medusa,         ONLY: cfn_ncin_bec

      USE mod_coupsim_medusa_setup, ONLY: GET_NLONSLATSZTS_FILE,
     &                                    GET_IMASK_OCEAN_FILE
      USE mod_coupsim_o2s


      IMPLICIT NONE


      INTEGER       :: i, j, k

      INTEGER, SAVE :: n_datasets_read = 0

      INTEGER       :: istatus, iflag


      INTEGER       :: ncid_file, ncid_var

      INTEGER, SAVE :: nlons_file      = -1
      INTEGER, SAVE :: nlats_file      = -1
      INTEGER, SAVE :: nzts_file       = -1

      INTEGER, DIMENSION(:,:), ALLOCATABLE, SAVE :: imask_ocean_file


      INTEGER, SAVE :: nsedcol_central = -1
      INTEGER, SAVE :: nix             = -1
      INTEGER, SAVE :: njy             = -1
      INTEGER, SAVE :: nijxy_t         = -1


      INTEGER, SAVE :: n_cprocs = 0
      INTEGER, SAVE :: i_mycomm = MPI_COMM_NULL
      INTEGER, SAVE :: i_myrank = MPI_PROC_NULL

      INTEGER, SAVE :: nsedcol_global    = -1
      INTEGER, SAVE :: nsx = -1
      INTEGER, SAVE :: nsy = -1
      INTEGER, SAVE :: npx = -1
      INTEGER, SAVE :: npy = -1

      INTEGER       :: ix, jy
      INTEGER       :: is, js
      INTEGER       :: ip, jp
      INTEGER       :: i_destrank
      INTEGER, DIMENSION(MPI_STATUS_SIZE) :: impi_status

      INTEGER, DIMENSION(2) :: nlens_mpitopo


      DOUBLE PRECISION, DIMENSION(:),       ALLOCATABLE :: dvar1d_zt
      REAL,             DIMENSION(:,:),     ALLOCATABLE :: rvar2d_ht
      DOUBLE PRECISION, DIMENSION(:,: ,:,:, :,:),
     &                                      ALLOCATABLE :: dvar6d_ht



      ! Standard I/O related data
      ! -------------------------

      CHARACTER(LEN=*), PARAMETER :: cfmt_modprocname_a
     &  = '("[MOD_COUPSIM_SUBR/COUPSIM_BIOGEOCHEM_STEP]: ", A)'

      CHARACTER(LEN=*), PARAMETER :: cfmt_a     = '(" - ", A)'
      CHARACTER(LEN=*), PARAMETER :: cfmt_a_ind = '("   ", A)'


#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_modprocname_a) 'Start'
#endif

      ifnodsread: IF (n_datasets_read == 0) THEN

        n_cprocs = MEDEXE_NPROC()
        i_mycomm = MEDEXE_MPI_COMM()
        i_myrank = MEDEXE_MPI_COMM_RANK()

        CALL MEDEXE_MPI_GETTOPO_DIMLENS(nlens_mpitopo)
        npx = nlens_mpitopo(1)
        npy = nlens_mpitopo(2)

        CALL MEDEXE_MPI_GETPARTITIONING(
     &    knsedcol_global = nsedcol_global)


        CALL N_COLUMNS_USED(nsedcol_central)
        IF (nsedcol_central == -1) THEN
          WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
          WRITE(jp_stderr, cfmt_a)
     &       'MOD_SEAFLOOR_CENTRAL not yet set up -- aborting.'
          CALL ABORT_MEDUSA()
        ENDIF

                                    ! X-Y extension per tile
        CALL IJ_ITJT_COLUMNS_USED(nix, njy, nsx, nsy)
        nijxy_t = nsx*nix * nsy*njy


        IF (.NOT. ALLOCATED(seafloor_dept)) THEN
          WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
          WRITE(jp_stderr, cfmt_a) 
     &      'MOD_COUPSIM_O2S not yet set up -- aborting.'
          CALL ABORT_MEDUSA()
        ENDIF


        IF (i_myrank == jp_exeproc_root) THEN

          CALL GET_NLONSLATSZTS_FILE(nlons_file, nlats_file, nzts_file)

          ALLOCATE(imask_ocean_file(nlons_file, nlats_file))

          CALL GET_IMASK_OCEAN_FILE(imask_ocean_file)

        ELSE

          nlons_file = 0
          nlats_file = 0
          nzts_file  = 0
          ALLOCATE(imask_ocean_file(0, 0))

        ENDIF

      ENDIF ifnodsread

      
                                    !===============================!
                                    ! Open annual mean results file !
                                    !===============================!

      IF (i_myrank == jp_exeproc_root) THEN

        istatus = NF_OPEN(cfn_ncin_bec, NF_NOWRITE, ncid_file)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))


                                    ! Initialize DEPTH data.
                                    ! These must be read separately, as
                                    ! they are not on the global 3D grid.
        ALLOCATE(dvar1d_zt(nzts_file))
        ALLOCATE(rvar2d_ht(nlons_file, nlats_file))

                                    ! Read in bottom-of-layer depth distributions
                                    ! z_t (type: DOUBLE; units: cm)

                                    ! double z_t(z_t) ;
                                    !   z_t:long_name = "depth from surface to midpoint of layer" ;
                                    !   z_t:units = "centimeters" ;

        istatus = NF_INQ_VARID(ncid_file, 'z_t', ncid_var)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
        istatus = NF_GET_VAR_DOUBLE(ncid_file, ncid_var, dvar1d_zt)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

                                    ! Read in ocean depth distributions
                                    ! HT (type: REAL; units: cm)

                                    ! float HT(nlat, nlon) ;
                                    !   HT:long_name = "ocean depth at T points" ;
                                    !   HT:units = "centimeter" ;

        istatus = NF_INQ_VARID(ncid_file, 'HT', ncid_var)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
        istatus = NF_GET_VAR_REAL(ncid_file, ncid_var, rvar2d_ht)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))


        ALLOCATE(dvar6d_ht(nix,njy, nsx, nsy, npx,npy))

        DO jp = 1, npy
        DO ip = 1, npx
        DO js = 1, nsy
        DO is = 1, nsx

          DO jy = 1, njy
            j = (jp - 1) * nsy*njy + (js - 1) * njy + jy

            DO ix = 1, nix
              i = (ip - 1) * nsx*nix + (is - 1) * nix + ix

              k = imask_ocean_file(i,j)

              IF (k > 0) THEN
                dvar6d_ht(ix,jy, is,js, ip,jp) = DBLE(rvar2d_ht(i,j))
              ELSE
                dvar6d_ht(ix,jy, is,js, ip,jp) = 0.0D+00
              ENDIF

            ENDDO
          ENDDO

        ENDDO
        ENDDO
        ENDDO
        ENDDO


        DO i_destrank = 0, n_cprocs-1

          CALL MEDEXE_MPI_TOPO2D_PP4RANK(i_destrank, ip, jp)


          IF (i_destrank == jp_exeproc_root) THEN
            seafloor_dept(:,:, :,:) = dvar6d_ht(:,:, :,:, ip,jp)
          ELSE
            CALL MPI_SEND(dvar6d_ht(:,:, :,:, ip,jp),
     &             nijxy_t, MPI_DOUBLE_PRECISION, i_destrank,
     &             i_destrank, i_mycomm, iflag)
          ENDIF

        ENDDO


      ELSE

        CALL MPI_RECV(seafloor_dept,
     &               nijxy_t, MPI_DOUBLE_PRECISION, jp_exeproc_root,
     &               i_myrank, i_mycomm, impi_status, iflag)

      ENDIF


      IF (i_myrank == jp_exeproc_root) THEN
        DEALLOCATE(dvar6d_ht)
      ENDIF


                                    ! Read in the temperature (tmpdc) data
                                    !   float TEMP(z_t, nlat, nlon) ;
                                    !     TEMP:long_name = "Potential Temperature" ;
                                    !     TEMP:units = "degC" ;
                                    !     TEMP:coordinates = "TLONG TLAT z_t " ;
                                    !     TEMP:time_op = "average" ;
                                    !     TEMP:FillValue = 9.96921e+36f ;
                                    !     TEMP:missing_value = 9.96921e+36f ;
                                    !     TEMP:cell_methods = "time: mean" ;

      CALL READNC_SEAFLOOR_2D(ncid_file, 'TEMP',
     &                              seafloor_temp)


                                    ! Read in the salinity (salin) data
                                    !   float SALT(z_t, nlat, nlon) ;
                                    !     SALT:long_name = "Salinity" ;
                                    !     SALT:units = "gram/kilogram" ;
                                    !     SALT:coordinates = "TLONG TLAT z_t " ;
                                    !     SALT:time_op = "average" ;
                                    !     SALT:FillValue = 9.96921e+33f ;
                                    !     SALT:missing_value = 9.96921e+33f ;
                                    !     SALT:cell_methods = "time: mean" ;

      CALL READNC_SEAFLOOR_2D(ncid_file, 'SALT',
     &                              seafloor_sali)


                                    ! Read in DIC data
                                    !   float DIC(z_t, nlat, nlon) ;
                                    !     DIC:long_name = "Dissolved Inorganic Carbon" ;
                                    !     DIC:units = "mmol C/m^3" ;
                                    !     DIC:coordinates = "TLONG TLAT z_t " ;
                                    !     DIC:time_op = "average" ;
                                    !     DIC:FillValue = 9.96921e+36f ;
                                    !     DIC:missing_value = 9.96921e+36f ;
                                    !     DIC:cell_methods = "time: mean" ;

      CALL READNC_SEAFLOOR_2D(ncid_file, 'DIC',
     &                              seafloor_wconc_dic)


                                    ! Read in ALK data
                                    !   float ALK(z_t, nlat, nlon) ;
                                    !     ALK:long_name = "Alkalinity" ;
                                    !     ALK:units = "meq/m^3" ;
                                    !     ALK:coordinates = "TLONG TLAT z_t " ;
                                    !     ALK:time_op = "average" ;
                                    !     ALK:FillValue = 9.96921e+36f ;
                                    !     ALK:missing_value = 9.96921e+36f ;
                                    !     ALK:cell_methods = "time: mean" ;

      CALL READNC_SEAFLOOR_2D(ncid_file, 'ALK',
     &                              seafloor_wconc_alk)


                                    ! Read in Clay/Dust Flux data
                                    !   float dust_FLUX_IN(z_t, nlat, nlon) ;
                                    !     dust_FLUX_IN:long_name = "Incoming Flux of dust" ;
                                    !     dust_FLUX_IN:units = "g/cm^2/sec" ;
                                    !     dust_FLUX_IN:coordinates = "TLONG TLAT z_t " ;
                                    !     dust_FLUX_IN:time_op = "average" ;
                                    !     dust_FLUX_IN:FillValue = 9.96921e+36f ;
                                    !     dust_FLUX_IN:missing_value = 9.96921e+36f ;
                                    !     dust_FLUX_IN:cell_methods = "time: mean" ;

      CALL READNC_SEAFLOOR_FLX2D(ncid_file, 'dust_FLUX_IN',
     &                              seafloor_wfflx_clay)


                                    ! Read in Calcite Flux data
                                    !   float CaCO3_FLUX_IN(z_t, nlat, nlon) ;
                                    !     CaCO3_FLUX_IN:long_name = "Incoming Flux of CaCO3" ;
                                    !     CaCO3_FLUX_IN:units = "mmol CaCO3/m^3 cm/sec" ;
                                    !     CaCO3_FLUX_IN:coordinates = "TLONG TLAT z_t " ;
                                    !     CaCO3_FLUX_IN:time_op = "average" ;
                                    !     CaCO3_FLUX_IN:FillValue = 9.96921e+36f ;
                                    !     CaCO3_FLUX_IN:missing_value = 9.96921e+36f ;
                                    !     CaCO3_FLUX_IN:cell_methods = "time: mean" ;

      CALL READNC_SEAFLOOR_FLX2D(ncid_file, 'CaCO3_FLUX_IN',
     &                              seafloor_wfflx_calc)


                                    ! Read in Corg Flux data
                                    !	  float POC_FLUX_IN(z_t, nlat, nlon) ;
                                    !	    POC_FLUX_IN:long_name = "Incoming Flux of POC" ;
                                    !	    POC_FLUX_IN:units = "mmol POC/m^3 cm/sec" ;
                                    !	    POC_FLUX_IN:coordinates = "TLONG TLAT z_t " ;
                                    !	    POC_FLUX_IN:time_op = "average" ;
                                    !	    POC_FLUX_IN:FillValue = 9.96921e+36f ;
                                    !	    POC_FLUX_IN:missing_value = 9.96921e+36f ;
                                    !	    POC_FLUX_IN:cell_methods = "time: mean" ;

      CALL READNC_SEAFLOOR_FLX2D(ncid_file, 'POC_FLUX_IN',
     &                              seafloor_wfflx_om)


                                    ! Read in Opal Flux data
                                    !   float SiO2_FLUX_IN(z_t, nlat, nlon) ;
                                    !     SiO2_FLUX_IN:long_name = "Incoming Flux of SiO2" ;
                                    !     SiO2_FLUX_IN:units = "mmol SiO2/m^3 cm/sec" ;
                                    !     SiO2_FLUX_IN:coordinates = "TLONG TLAT z_t " ;
                                    !     SiO2_FLUX_IN:time_op = "average" ;
                                    !     SiO2_FLUX_IN:FillValue = 9.96921e+36f ;
                                    !     SiO2_FLUX_IN:missing_value = 9.96921e+36f ;
                                    !     SiO2_FLUX_IN:cell_methods = "time: mean" ;

      CALL READNC_SEAFLOOR_FLX2D(ncid_file, 'SiO2_FLUX_IN',
     &                              seafloor_wfflx_opal)


                                    ! Read in Nitrate data
                                    !   float NO3(z_t, nlat, nlon) ;
                                    !     NO3:long_name = "Dissolved Inorganic Nitrate" ;
                                    !     NO3:units = "mmol N/m^3" ;
                                    !     NO3:coordinates = "TLONG TLAT z_t " ;
                                    !     NO3:time_op = "average" ;
                                    !     NO3:FillValue = 9.96921e+36f ;
                                    !     NO3:missing_value = 9.96921e+36f ;
                                    !     NO3:cell_methods = "time: mean" ;

      CALL READNC_SEAFLOOR_2D(ncid_file, 'NO3',
     &                              seafloor_wconc_no3)


                                    ! Read in Oxygen data
                                    !   float O2(z_t, nlat, nlon) ;
                                    !     O2:long_name = "Dissolved Oxygen" ;
                                    !     O2:units = "mmol O2/m^3" ;
                                    !     O2:coordinates = "TLONG TLAT z_t " ;
                                    !     O2:time_op = "average" ;
                                    !     O2:FillValue = 9.96921e+36f ;
                                    !     O2:missing_value = 9.96921e+36f ;
                                    !     O2:cell_methods = "time: mean" ;

      CALL READNC_SEAFLOOR_2D(ncid_file, 'O2',
     &                              seafloor_wconc_o2)


                                    !   float SiO3(z_t, nlat, nlon) ;
                                    !     SiO3:long_name = "Dissolved Inorganic Silicate" ;
                                    !     SiO3:units = "mmol Si/m^3" ;
                                    !     SiO3:coordinates = "TLONG TLAT z_t " ;
                                    !     SiO3:time_op = "average" ;
                                    !     SiO3:FillValue = 9.96921e+36f ;
                                    !     SiO3:missing_value = 9.96921e+36f ;
                                    !     SiO3:cell_methods = "time: mean" ;

      CALL READNC_SEAFLOOR_2D(ncid_file, 'SiO3',
     &                              seafloor_wconc_sio2)


      IF (i_myrank == jp_exeproc_root) THEN
                                    ! Done - close file
        istatus = NF_CLOSE(ncid_file)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
      ENDIF


      n_datasets_read = 1           ! Increment counter now that the
                                    ! boundary condition set is complete

      CALL VALIDATE_O2S_DATASET     ! Validate the dataset in MOD_COUPSIM_O2S

      IF (i_myrank == jp_exeproc_root) THEN
        DEALLOCATE(dvar1d_zt)
        DEALLOCATE(rvar2d_ht)
      ENDIF

#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_modprocname_a) 'End'
      WRITE(jp_stddbg, '()')
#endif


      RETURN


      CONTAINS


!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      SUBROUTINE READNC_SEAFLOOR_2D(ncid_file, cname_var, data4d_dp)
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

      USE mod_netcdfinc,            ONLY: NF_INQ_VARID, NF_GET_VAR_REAL,
     &                                    NF_NOERR,
     &                                    HANDLE_NCERRORS


      IMPLICIT NONE


      INTEGER,          INTENT(IN)  :: ncid_file
      CHARACTER(LEN=*), INTENT(IN)  :: cname_var
      DOUBLE PRECISION, INTENT(OUT) :: data4d_dp(:,:, :,:)

      INTEGER :: is, js
      INTEGER :: ix, jy
      INTEGER :: i, j, k
      INTEGER :: ncid_var
      INTEGER :: istatus, iflag

      INTEGER :: ip, jp
      INTEGER :: i_destrank
      INTEGER, DIMENSION(MPI_STATUS_SIZE) :: impi_status

      REAL,             ALLOCATABLE :: data3d_r(:,:,:)
      DOUBLE PRECISION, ALLOCATABLE :: data6d_dp(:,:, :,:, :,:)


      IF (n_datasets_read == 0) THEN

        IF (i_myrank == jp_exeproc_root) THEN

          IF ((nlons_file == -1) .OR. (nlats_file == -1)) THEN
            WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
            WRITE(jp_stderr, cfmt_a) 
     &        'MEDUSA_COUPSIM not yet set up -- aborting.'
            CALL ABORT_MEDUSA()
          ENDIF

        ENDIF

      ENDIF


      IF (i_myrank == jp_exeproc_root) THEN

        ALLOCATE(data3d_r(nlons_file, nlats_file, nzts_file))

        istatus = NF_INQ_VARID(ncid_file, cname_var, ncid_var)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

        istatus = NF_GET_VAR_REAL(ncid_file, ncid_var, data3d_r)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

        ALLOCATE(data6d_dp(nix,njy, nsx,nsy, npx,npy))


        DO jp = 1, npy
        DO ip = 1, npx
        DO js = 1, nsy
        DO is = 1, nsx

          DO jy = 1, njy
            j = (jp - 1) * nsy*njy + (js - 1) * njy + jy

            DO ix = 1, nix
              i = (ip - 1) * nsx*nix + (is - 1) * nix + ix

              k = imask_ocean_file(i,j)

              IF (k > 0) THEN

                data6d_dp(ix,jy, is,js, ip,jp) = DBLE(data3d_r(i,j,k))

              ELSE

                data6d_dp(ix,jy, is,js, ip,jp) = 0.0D+00

              ENDIF

            ENDDO

          ENDDO

        ENDDO
        ENDDO
        ENDDO
        ENDDO


        DO i_destrank = 0, n_cprocs-1

          CALL MEDEXE_MPI_TOPO2D_PP4RANK(i_destrank, ip, jp)

          IF (i_destrank == jp_exeproc_root) THEN
            data4d_dp(:,:, :,:) = data6d_dp(:,:, :,:, ip,jp)
          ELSE
            CALL MPI_SEND(data6d_dp(:,:, :,:, ip,jp),
     &             nijxy_t, MPI_DOUBLE_PRECISION, i_destrank,
     &             i_destrank, i_mycomm, iflag)
          ENDIF

        ENDDO

      ELSE

        CALL MPI_RECV(data4d_dp,
     &               nijxy_t, MPI_DOUBLE_PRECISION, jp_exeproc_root,
     &               i_myrank, i_mycomm, impi_status, iflag)

      ENDIF


      IF (i_myrank == jp_exeproc_root) THEN
        DEALLOCATE(data3d_r)
        DEALLOCATE(data6d_dp)
      ENDIF


      RETURN


!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      END SUBROUTINE READNC_SEAFLOOR_2D
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -



!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      SUBROUTINE READNC_SEAFLOOR_FLX2D(ncid_file, cname_var, data4d_dp)
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

      USE mod_netcdfinc,            ONLY: NF_INQ_VARID, NF_GET_VAR_REAL,
     &                                    NF_NOERR,
     &                                    HANDLE_NCERRORS


      IMPLICIT NONE


      INTEGER,          INTENT(IN)  :: ncid_file
      CHARACTER(LEN=*), INTENT(IN)  :: cname_var
      DOUBLE PRECISION, INTENT(OUT) :: data4d_dp(:,:, :,:)

      INTEGER :: is, js
      INTEGER :: ix, jy
      INTEGER :: i, j, k
      INTEGER :: ncid_var
      INTEGER :: istatus, iflag

      INTEGER :: ip, jp
      INTEGER :: i_destrank
      INTEGER, DIMENSION(MPI_STATUS_SIZE) :: impi_status

      REAL,             ALLOCATABLE :: data3d_r(:,:,:)
      DOUBLE PRECISION, ALLOCATABLE :: data6d_dp(:,:, :,:, :,:)

      DOUBLE PRECISION :: dflxratio, dexponent


      IF (n_datasets_read == 0) THEN

        IF (i_myrank == jp_exeproc_root) THEN

          IF ((nlons_file == -1) .OR. (nlats_file == -1)) THEN
            WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
            WRITE(jp_stderr, cfmt_a) 
     &        'MEDUSA_COUPSIM not yet set up -- aborting.'
            CALL ABORT_MEDUSA()
          ENDIF

        ENDIF

      ENDIF


      IF (i_myrank == jp_exeproc_root) THEN

        ALLOCATE(data3d_r(nlons_file, nlats_file, nzts_file))

        istatus = NF_INQ_VARID(ncid_file, cname_var, ncid_var)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

        istatus = NF_GET_VAR_REAL(ncid_file, ncid_var, data3d_r)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

        ALLOCATE(data6d_dp(nix,njy, nsx,nsy, npx,npy))


        DO jp = 1, npy
        DO ip = 1, npx
        DO js = 1, nsy
        DO is = 1, nsx

          DO jy = 1, njy
            j = (jp - 1) * nsy*njy + (js - 1) * njy + jy

            DO ix = 1, nix
              i = (ip - 1) * nsx*nix + (is - 1) * nix + ix

              k = imask_ocean_file(i,j)

              IF (k < 1) THEN

                data6d_dp(ix,jy, is,js, ip,jp) = 0.0D+00

              ELSEIF (k < 3) THEN

                data6d_dp(ix,jy, is,js, ip,jp) = DBLE(data3d_r(i,j,k))

              ELSEIF (data3d_r(i,j,k) == 0.0E+00) THEN

                data6d_dp(ix,jy, is,js, ip,jp) = 0.0D+00

              ELSE
                                    ! exponentially decreasing extrapolation
                                    ! F_depos
                                    !   = F_in(k) / (F_in(k-1)/F_in(k))
                                    !               **((z_t(k-1) - ht(i,j))/
                                    !                  (z_t(k-2) - z_t(k-1)))
                dexponent =   (dvar1d_zt(k-1) - DBLE(rvar2d_ht(i,j)))
     &                      / (dvar1d_zt(k-2) - dvar1d_zt(k-1))
                dflxratio =   DBLE(data3d_r(i,j,k-1))
     &                      / DBLE(data3d_r(i,j,k))
                data6d_dp(ix,jy, is,js, ip,jp)
     &                              =   DBLE(data3d_r(i,j,k))
     &                                / dflxratio**dexponent

              ENDIF

            ENDDO

          ENDDO
 
        ENDDO
        ENDDO
        ENDDO
        ENDDO


        DO i_destrank = 0, n_cprocs-1

          CALL MEDEXE_MPI_TOPO2D_PP4RANK(i_destrank, ip, jp)

          IF (i_destrank == jp_exeproc_root) THEN
            data4d_dp(:,:, :,:) = data6d_dp(:,:, :,:, ip,jp)
          ELSE
            CALL MPI_SEND(data6d_dp(:,:, :,:, ip,jp),
     &             nijxy_t, MPI_DOUBLE_PRECISION, i_destrank,
     &             i_destrank, i_mycomm, iflag)
          ENDIF

        ENDDO

      ELSE

        CALL MPI_RECV(data4d_dp,
     &               nijxy_t, MPI_DOUBLE_PRECISION, jp_exeproc_root,
     &               i_myrank, i_mycomm, impi_status, iflag)

      ENDIF


      IF (i_myrank == jp_exeproc_root) THEN
        DEALLOCATE(data3d_r)
        DEALLOCATE(data6d_dp)
      ENDIF


      RETURN


!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      END SUBROUTINE READNC_SEAFLOOR_FLX2D
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -



!-----------------------------------------------------------------------
      END SUBROUTINE COUPSIM_BIOGEOCHEM_STEP
!-----------------------------------------------------------------------


!=======================================================================
      END MODULE MOD_COUPSIM_SUBR
!=======================================================================
