!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


!---+----1----+----2----+----3----+----4----+----5----+----6----+----7--
!=======================================================================
      MODULE MOD_COUPSIM_O2S
!=======================================================================


      USE mod_defines_medusa,       ONLY: jp_stderr
#ifdef DEBUG
      USE mod_defines_medusa,       ONLY: jp_stddbg
#endif
      USE mod_execontrol_medusa,    ONLY: ABORT_MEDUSA


      IMPLICIT NONE


      PRIVATE

      PUBLIC :: SETUP_MOD_COUPSIM_O2S, CLEAR_O2S_DATASET,
     &                              OCEAN_TO_SEDIMENT,
     &                              VALIDATE_O2S_DATASET

      INTEGER, SAVE :: nsedcol_central     = -1
      INTEGER, SAVE :: nsx                 = -1
      INTEGER, SAVE :: nsy                 = -1
      INTEGER, SAVE :: nix                 = -1
      INTEGER, SAVE :: njy                 = -1

      INTEGER, SAVE :: n_datasets_uploaded =  0
      INTEGER, SAVE :: id_dataset_o2s      = -1
      INTEGER, SAVE :: id_prevdataset_o2s  = -1



      LOGICAL, SAVE :: l_mod_o2s_setup_done = .FALSE.


                                    ! Depth below sea-level of
                                    ! seafloor elements [units @host ...]
      DOUBLE PRECISION, SAVE, DIMENSION(:,:, :,:), ALLOCATABLE, PUBLIC
     &  :: seafloor_dept

                                    ! Seawater temperature at seafloor
                                    ! elements [units @host ...]
      DOUBLE PRECISION, SAVE, DIMENSION(:,:, :,:), ALLOCATABLE, PUBLIC
     &  :: seafloor_temp

                                    ! Salinity of seawater at seafloor
                                    ! elements [-]
      DOUBLE PRECISION, SAVE, DIMENSION(:,:, :,:), ALLOCATABLE, PUBLIC
     &  :: seafloor_sali



                                    ! Boundary concentrations
      DOUBLE PRECISION, SAVE, DIMENSION(:,:, :,:), ALLOCATABLE, PUBLIC
     &  :: seafloor_wconc_dic
      DOUBLE PRECISION, SAVE, DIMENSION(:,:, :,:), ALLOCATABLE, PUBLIC
     &  :: seafloor_wconc_alk
      DOUBLE PRECISION, SAVE, DIMENSION(:,:, :,:), ALLOCATABLE, PUBLIC
     &  :: seafloor_wconc_o2
      DOUBLE PRECISION, SAVE, DIMENSION(:,:, :,:), ALLOCATABLE, PUBLIC
     &  :: seafloor_wconc_no3
      DOUBLE PRECISION, SAVE, DIMENSION(:,:, :,:), ALLOCATABLE, PUBLIC
     &  :: seafloor_wconc_sio2


                                    ! Boundary fluxes
      DOUBLE PRECISION, SAVE, DIMENSION(:,:, :,:), ALLOCATABLE, PUBLIC
     &  :: seafloor_wfflx_clay
      DOUBLE PRECISION, SAVE, DIMENSION(:,:, :,:), ALLOCATABLE, PUBLIC
     &  :: seafloor_wfflx_calc
      DOUBLE PRECISION, SAVE, DIMENSION(:,:, :,:), ALLOCATABLE, PUBLIC
     &  :: seafloor_wfflx_om
      DOUBLE PRECISION, SAVE, DIMENSION(:,:, :,:), ALLOCATABLE, PUBLIC
     &  :: seafloor_wfflx_opal



      ! Dummy function names
      DOUBLE PRECISION :: TOTBOR, RHOSW

      ! SPECIA_CB arguments
      DOUBLE PRECISION :: xdic, xalk, xtk, xwsalin, xwdbsl, xwtmpdc, xbt
      DOUBLE PRECISION :: xco2, xco3, xhco3, xboh4, xboh3, xh3o, xoh

      CONTAINS


!-----------------------------------------------------------------------
      SUBROUTINE SETUP_MOD_COUPSIM_O2S
!-----------------------------------------------------------------------


      USE mod_seafloor_central,     ONLY: N_COLUMNS_USED,
     &                                    IJ_ITJT_COLUMNS_USED


      IMPLICIT NONE


      ! Standard I/O related data
      ! -------------------------

      CHARACTER(LEN=*), PARAMETER :: c_fmterr_a
     &  = '("[MOD_COUPSIM_O2S/SETUP_MOD_COUPSIM_O2S] error: ", A)'


      IF (l_mod_o2s_setup_done) THEN
        WRITE(jp_stderr, c_fmterr_a)
     &      'setup already done -- aborting!'
        CALL ABORT_MEDUSA()
      ENDIF


      CALL N_COLUMNS_USED(nsedcol_central)
      CALL IJ_ITJT_COLUMNS_USED(nix, njy, nsx, nsy)


      ALLOCATE(seafloor_temp(nix,njy, nsx,nsy))
      ALLOCATE(seafloor_sali(nix,njy, nsx,nsy))
      ALLOCATE(seafloor_dept(nix,njy, nsx,nsy))

      ALLOCATE(seafloor_wconc_dic(nix,njy, nsx,nsy))
      ALLOCATE(seafloor_wconc_alk(nix,njy, nsx,nsy))
      ALLOCATE(seafloor_wconc_o2(nix,njy, nsx,nsy))
      ALLOCATE(seafloor_wconc_no3(nix,njy, nsx,nsy))
      ALLOCATE(seafloor_wconc_sio2(nix,njy, nsx,nsy))

      ALLOCATE(seafloor_wfflx_clay(nix,njy, nsx,nsy))
      ALLOCATE(seafloor_wfflx_calc(nix,njy, nsx,nsy))
      ALLOCATE(seafloor_wfflx_om(nix,njy, nsx,nsy))
      ALLOCATE(seafloor_wfflx_opal(nix,njy, nsx,nsy))


      id_dataset_o2s       = 0
      id_prevdataset_o2s   = 0
      l_mod_o2s_setup_done = .TRUE.


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE SETUP_MOD_COUPSIM_O2S
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE CLEAR_O2S_DATASET
!-----------------------------------------------------------------------


      IMPLICIT NONE


      ! Standard I/O related data
      ! -------------------------

      CHARACTER(LEN=*), PARAMETER :: c_fmterr_a
     &  = '("[MOD_COUPSIM_O2S/CLEAR_O2S_DATASET] error: ", A)'


      IF (.NOT. l_mod_o2s_setup_done) THEN
        WRITE(jp_stderr, c_fmterr_a)
     &    'MOD_COUPSIM_O2S not yet set up -- aborting!'
        CALL ABORT_MEDUSA()
      ENDIF


      seafloor_temp(:,:, :,:)       = 0.0D+00
      seafloor_sali(:,:, :,:)       = 0.0D+00
      seafloor_dept(:,:, :,:)       = 0.0D+00

      seafloor_wconc_dic(:,:, :,:)  = 0.0D+00
      seafloor_wconc_alk(:,:, :,:)  = 0.0D+00
      seafloor_wconc_o2(:,:, :,:)   = 0.0D+00
      seafloor_wconc_no3(:,:, :,:)  = 0.0D+00
      seafloor_wconc_sio2(:,:, :,:) = 0.0D+00

      seafloor_wfflx_clay(:,:, :,:) = 0.0D+00
      seafloor_wfflx_calc(:,:, :,:) = 0.0D+00
      seafloor_wfflx_om(:,:, :,:)   = 0.0D+00
      seafloor_wfflx_opal(:,:, :,:) = 0.0D+00


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE CLEAR_O2S_DATASET
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE VALIDATE_O2S_DATASET
!-----------------------------------------------------------------------


      IMPLICIT NONE


      ! Standard I/O related data
      ! -------------------------

      CHARACTER(LEN=*), PARAMETER :: c_fmterr_a
     &  = '("[MOD_COUPSIM_O2S/VALIDATE_O2S_DATASET] error: ", A)'


      IF (.NOT. l_mod_o2s_setup_done) THEN
        WRITE(jp_stderr, c_fmterr_a)
     &    'MOD_COUPSIM_O2S not yet set up -- aborting!'
        CALL ABORT_MEDUSA()
      ENDIF

      id_dataset_o2s = id_dataset_o2s + 1


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE VALIDATE_O2S_DATASET
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE OCEAN_TO_SEDIMENT(rbflag)
!-----------------------------------------------------------------------
      ! Loads a new complete set of forcing arrays into
      ! MOD_SEAFLOOR_CENTRAL


      USE mod_seafloor_wdata,       ONLY: WDATA_CONTAINER
      USE mod_indexparam
      USE mod_seafloor_central,     ONLY: COLUMN_N2IJ_ITJT,
     &                                    SAVE_BOUNDARY_CONDS

      USE mod_basicdata_medusa,     ONLY: dp_zero_degc, dp_molvol_stp

      USE mod_materialcharas        ! for mol weights, molar data, etc.


      IMPLICIT NONE


      INTEGER, INTENT(OUT) :: rbflag


      INTEGER :: is, js, i, j, n, iflag


      TYPE(WDATA_CONTAINER)               :: wdata
      DOUBLE PRECISION, DIMENSION(nsolut) :: wconc
      DOUBLE PRECISION, DIMENSION(nsolid) :: wfflx

      DOUBLE PRECISION :: xwdbsl, xwtmpdc, xwsalin
      DOUBLE PRECISION :: xtk
      DOUBLE PRECISION :: xrho
      DOUBLE PRECISION :: xdic, xalk, xbt
      DOUBLE PRECISION :: xco3, xhco3, xco2, xboh4, xboh3, xoh, xh3o

      DOUBLE PRECISION :: RHOSW, TOTBOR


      ! Standard I/O related data
      ! -------------------------

      CHARACTER(LEN=*), PARAMETER :: cfmt_modprocname_a
     &  = '("[MOD_COUPSIM_O2S/OCEAN_TO_SEDIMENT]: ", A)'

      CHARACTER(LEN=*), PARAMETER :: cfmt_a     = '(" - ", A)'
      CHARACTER(LEN=*), PARAMETER :: cfmt_a_ind = '("   ", A)'


#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_modprocname_a) 'Start'
#endif


      IF (.NOT. l_mod_o2s_setup_done) THEN
        WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
        WRITE(jp_stderr, cfmt_a)
     &    'MOD_COUPSIM_O2S not yet set up -- aborting!'
        CALL ABORT_MEDUSA()
      ENDIF


      IF (n_datasets_uploaded == 0) THEN
                                    ! No data have been uploaded into
                                    ! Medusa's MOD_SEAFLOOR_CENTRAL
                                    ! from this module so far
        IF (id_dataset_o2s == -1) THEN
                                    ! No data have been deposited into
                                    ! this module so far
          WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
          WRITE(jp_stderr, cfmt_a)
     &      'Data must be loaded into MOD_COUPSIM_O2S_SETUP ' //
     &      '-- aborting!'
          CALL ABORT_MEDUSA()

        ENDIF

      ENDIF



      DO n = 1, nsedcol_central

        CALL COLUMN_N2IJ_ITJT(i_column = n, iflag = iflag,
     &                        ix = i, jy = j,
     &                        itx = is, jty = js)


                                    ! Convert units here, if necessary:
                                    ! --------------------------------

                                    !  - wdata%wdbsl [m]
                                    !    (seafloor depth, positive downwards)
                                    !  - seafloor_dept [cm]
        wdata%wdbsl  = seafloor_dept(i,j, is,js)*1.0D-02
                                    !  - wdata%wsalin [-]
                                    !  - seafloor_sali [-]
        wdata%wsalin = seafloor_sali(i,j, is,js)
                                    !  - wdata%wtmpc [degC]
                                    !  - seafloor_temp [degC]
        wdata%wtmpc  = seafloor_temp(i,j, is,js)


        xwdbsl  = wdata%wdbsl
        xwtmpdc = wdata%wtmpc
        xwsalin = wdata%wsalin
  
        xtk     = xwtmpdc + dp_zero_degc

        xrho    = RHOSW(xtk, xwsalin, xwdbsl)             ! [kg_SW/m3_SW]
  
        xdic    = seafloor_wconc_dic(i,j, is,js)*1.0D-03  ! [mmol/m3_SW] --> [mol/m3_SW]
        xalk    = seafloor_wconc_alk(i,j, is,js)*1.0D-03  ! [meq/m3_SW]  --> [mol/m3_SW]
        xbt     = TOTBOR(xtk,xwsalin,xwdbsl)*xrho         ! [mol/kg_SW]  --> [mol/m3_SW]

        CALL SPECIA_CB(xtk,xwsalin,xwdbsl, xalk,
     &                              xdic, xco3,xhco3,xco2,
     &                              xbt, xboh4,xboh3,
     &                              xoh,xh3o,
     &                              iflag)

        IF (iflag /= 0) THEN        ! SPECIA_CB returned an error
          rbflag = 2                ! Raise error flag
#ifdef DEBUG
          WRITE(jp_stddbg, cfmt_modprocname_a) 'Return @ 2'
          WRITE(jp_stddbg, '()')
#endif
          RETURN                    ! and return straight away
        ENDIF


                                    !  - wconc [mol/m3]
                                    !  - xco3, xhco3, xco2 [mol/m3_SW]
        wconc(ic_co3)  = xco3
        wconc(ic_hco3) = xhco3
        wconc(ic_co2)  = xco2
                                    !  - wconc [mol/m3]
                                    !  - seafloor_wconc_o2 [mmol/m3_SW]
        wconc(ic_o2) = seafloor_wconc_o2(i,j, is,js) * 1.0D-03

                                    !  - wconc [mol/m3]
                                    !  - seafloor_wconc_no3 [mmol/m3_SW]

        wconc(ic_no3) = seafloor_wconc_no3(i,j, is,js) * 1.0D-03
                                    !  - wconc [mol/m3]
                                    !  - seafloor_wconc_sio2 [mmol/m3_SW]

        wconc(ic_sio2) = seafloor_wconc_sio2(i,j, is,js) * 1.0D-03

                                    !  - wfflx [kg clay/m2/yr]
                                    !  - seafloor_wfflx_clay [g/cm2/s]
                                    !    J/[kg/m2/yr]
                                    !      = J/[(10^3 g) / (10^4 cm2) / (1/dp_sec) s)]
                                    !      = 10/dp_sec * J/[g/cm2/s]
        wfflx(if_clay) = seafloor_wfflx_clay(i,j, is,js)
     &                              * 1.0D+01 / dp_sec

                                    !  - wfflx [kg CaCO3/m2/yr]
                                    !  - seafloor_wfflx_calc [mmol CaCO3/m3 cm/s]
                                    !    J/[kg CaCO3/m2/yr]
                                    !      = J/[(1/M_CaCO3) mol CaCO3/m3 m/yr]
                                    !      = J/[(1/M_CaCO3) (10^3 mmol) / m3 * (10^2 cm) / (1/dp_sec) s)]
                                    !      = J/[(10^5*dp_sec/M_CaCO3) mmol / m3 cm / s)]
                                    !      = M_CaCO3/(10^5*dp_sec) * J/[mmol/m3 cm/s]
        wfflx(if_calc) = seafloor_wfflx_calc(i,j, is,js)
     &                              * mol_calc/(1.0D+05*dp_sec)

                                    !  - wfflx [kgOM/m2/yr]
                                    !  - seafloor_wfflx_om [mmol POC/m3 cm/s]
                                    !    J/[kg OM/m2/yr]
                                    !      = J/[(1/M_OM) mol OM/m3 m/yr]
                                    !      = J/[(1/M_OM) (om_c mol POC)/m3 m/yr]
                                    !      = J/[(om_c/M_OM) (10^3 mmol) / m3 (10^2 cm) / (1/dp_sec) s)]
                                    !      = J/[(10^5*om_c*dp_sec/M_OM) mmol / m3 cm / s)]
                                    !      = M_OM/(10^5*om_c*dp_sec) * J/[mmol/m3 cm/s]
        wfflx(if_om) = seafloor_wfflx_om(i,j, is,js)
     &                              *mol_om/(om_c*1.0D+05*dp_sec)

                                    !  - wfflx [kg opal/m2/yr]
                                    !  - seafloor_wfflx_opal [mmol SiO2/m3 cm/s]
                                    !    J/[kg opal/m2/yr]
                                    !      = M_opal/(10^5*opal_si*dp_sec) * J/[mmol/m3 cm/s]
        wfflx(if_opal) = seafloor_wfflx_opal(i,j, is,js)
     &                              *mol_opal/(opal_si*1.0D+05*dp_sec)


        CALL SAVE_BOUNDARY_CONDS(i_column = n, gbcflag = iflag,
     &                              wdata = wdata,
     &                              wconc = wconc(:),
     &                              wfflx = wfflx(:))

      ENDDO

      n_datasets_uploaded = n_datasets_uploaded + 1
      id_prevdataset_o2s = id_dataset_o2s
      rbflag = 0


#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_modprocname_a) 'End'
      WRITE(jp_stddbg, '()')
#endif


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE OCEAN_TO_SEDIMENT
!-----------------------------------------------------------------------


!=======================================================================
      END MODULE MOD_COUPSIM_O2S
!=======================================================================
