!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#ifdef CFN_THISFILE
#undef CFN_THISFILE
#endif
#define CFN_THISFILE "store_nc_aux.F"
#ifndef __LINE__
#define __LINE__ 0
#endif
!=======================================================================
      SUBROUTINE STORE_NC_AUX(cfn_ncin_bec, cfn_ncout_aux, title_string)
!=======================================================================


      USE mod_defines_medusa
      USE mod_execontrol_medusa,    ONLY: ABORT_MEDUSA

#ifdef ALLOW_MPI
      USE mpi,                      ONLY: MPI_COMM_NULL, MPI_PROC_NULL,
     &                                    MPI_STATUS_SIZE,
     &                                    MPI_DOUBLE_PRECISION
      USE mod_execontrol_medusa,    ONLY: jp_exeproc_ncio,
     &                                    jp_exeproc_root,
     &                                    MEDEXE_MPI_TOPO2D_PP4RANK
#endif

      USE mod_gridparam,            ONLY: idnw, idnb, ndn, GRID_DEF

#ifdef MEDUSA_BASE2D
      USE mod_seafloor_central,     ONLY: N_COLUMNS_USED, COLUMN_AREA4N,
     &                                    COLUMN_IJ2N, COLUMN_N2IJ,
     &                                    COLUMN_N2XY, COLUMN_N2MASKVAL
#endif

      USE mod_netcdfinc
      USE mod_netcdfparam

      USE mod_coupsim_medusa_setup, ONLY: GET_NLONSLATSZTS_FILE


      IMPLICIT NONE


      CHARACTER(LEN=*), INTENT(IN)           :: cfn_ncin_bec
      CHARACTER(LEN=*), INTENT(IN)           :: cfn_ncout_aux
      CHARACTER(LEN=*), INTENT(IN), OPTIONAL :: title_string


      CHARACTER(LEN=*), PARAMETER :: cfn_thisfile =
     &  CFN_THISFILE

      INTEGER :: ncid

      INTEGER :: ncid_file, ncid_var

      INTEGER :: nsedcol_central = -1
      INTEGER :: nsedcol_ncfile  = -1
#ifdef MEDUSA_BASE2D
      INTEGER :: nix_ncfile      = -1
      INTEGER :: njy_ncfile      = -1
#endif

      LOGICAL :: l_file_is_mine = .TRUE.

#ifdef ALLOW_MPI
      INTEGER :: n_cprocs = -1
      INTEGER :: i_mycomm = MPI_COMM_NULL
      INTEGER :: i_myrank = MPI_PROC_NULL
#  ifdef MEDUSA_BASE2D
      INTEGER :: i_myip   = -1
      INTEGER :: i_myjp   = -1

                                    ! Number of processes
      INTEGER :: npx                !  - along X-direction
      INTEGER :: npy                !  - along Y-direction
#  endif

      INTEGER, DIMENSION(MPI_STATUS_SIZE) :: impi_status

      INTEGER :: nsedcol_global = -1
      INTEGER :: iproc_1stocn   = -1
      INTEGER, DIMENSION(:), ALLOCATABLE :: nsedcol_pproc
      INTEGER, DIMENSION(:), ALLOCATABLE :: ioffset_sedcol_pproc

      LOGICAL :: l_onewrites4many = .FALSE.

      INTEGER,          DIMENSION(:),   ALLOCATABLE :: iarr_mpirecv_c
      DOUBLE PRECISION, DIMENSION(:),   ALLOCATABLE :: darr_mpirecv_c
#  ifdef MEDUSA_BASE2D
      INTEGER,          DIMENSION(:,:), ALLOCATABLE :: iarr_mpirecv_ij
      DOUBLE PRECISION, DIMENSION(:,:), ALLOCATABLE :: darr_mpirecv_ij
#  endif
      INTEGER :: i_destrank
#endif


      INTEGER :: k, n
      INTEGER :: iflag, istatus

#ifdef MEDUSA_BASE2D
      INTEGER :: nix, njy, nijxy
      INTEGER, DIMENSION(2) :: dim, istart
      INTEGER :: i, j
      INTEGER :: ip, jp
      INTEGER :: ix, jy
      DOUBLE PRECISION :: x, y
#endif

      DOUBLE PRECISION, DIMENSION(idnw:idnb) :: xzdn


! Dimensions and dimension variables
!  'lev' (levels)
      INTEGER            :: dim_lev         ! dimension ID
      INTEGER            ::  id_lev         ! ID of dimension variable

! 'col' (columns)
      INTEGER            :: dim_col         ! dimension ID
      INTEGER            ::  id_col         ! ID of dimension variable

#ifdef MEDUSA_BASE2D
      CHARACTER(LEN=*), PARAMETER   :: c_name_ix     = 'ix'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_ix    = 'X_Index'
      INTEGER, PARAMETER :: nlen_lname_ix = LEN(c_lname_ix)
      INTEGER            :: dim_ix
      INTEGER            ::  id_ix

      CHARACTER(LEN=*), PARAMETER   :: c_name_jy     = 'jy'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_jy    = 'Y_Index'
      INTEGER, PARAMETER :: nlen_lname_jy = LEN(c_lname_jy)
      INTEGER            :: dim_jy
      INTEGER            ::  id_jy
#endif


! Data variables
!  - general
      INTEGER ::  id_xzdn           ! depth coordinates
      INTEGER ::  id_area           ! surface areas of columns (n)
      INTEGER ::  id_maskval        ! mask value maskval(n)   

      DOUBLE PRECISION   :: sfc_area
      INTEGER            :: mval

      INTEGER,          DIMENSION(:), ALLOCATABLE ::  iarr_c
      DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE ::  darr_c

#ifdef MEDUSA_BASE2D
!  - 2D specific
      INTEGER ::  id_col4ij         ! column index n for grid element (i,j)

                                    ! index i(n)
      CHARACTER(LEN=*), PARAMETER   :: c_name_i4col     = 'i4col'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_i4col    = c_lname_ix //
     &                                              '_from_Column_Index'
      INTEGER, PARAMETER :: nlen_lname_i4col = LEN(c_lname_i4col)
      INTEGER ::  id_i4col

                                    ! index j(n)
      CHARACTER(LEN=*), PARAMETER   :: c_name_j4col     = 'j4col'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_j4col    = c_lname_jy //
     &                                              '_from_Column_Index'
      INTEGER, PARAMETER :: nlen_lname_j4col = LEN(c_lname_j4col)
      INTEGER ::  id_j4col

                                    ! xref value of grid element (n)
      CHARACTER(LEN=*), PARAMETER   :: c_name_xref4col  = 'xref4col'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_xref4col = 'Longitude'
      CHARACTER(LEN=*), PARAMETER   :: c_unit_xref4col  = un_degE
      INTEGER, PARAMETER :: nlen_lname_xref4col = LEN(c_lname_xref4col)
      INTEGER, PARAMETER :: nlen_unit_xref4col  = LEN(c_unit_xref4col)
      INTEGER ::  id_xref4col
      REAL, ALLOCATABLE, DIMENSION(:,:) :: rarr_tlong
      DOUBLE PRECISION, DIMENSION(:,:, :,:), ALLOCATABLE :: darr_lon4ij
      DOUBLE PRECISION, DIMENSION(:,:),      ALLOCATABLE :: darrx_ij

                                    ! yref value of grid element (n)
      CHARACTER(LEN=*), PARAMETER   :: c_name_yref4col  = 'yref4col'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_yref4col = 'Latitude'
      CHARACTER(LEN=*), PARAMETER   :: c_unit_yref4col  = un_degN
      INTEGER, PARAMETER :: nlen_lname_yref4col = LEN(c_lname_yref4col)
      INTEGER, PARAMETER :: nlen_unit_yref4col  = LEN(c_unit_yref4col)
      INTEGER ::  id_yref4col
      REAL, ALLOCATABLE, DIMENSION(:,:) :: rarr_tlat
      DOUBLE PRECISION, DIMENSION(:,:, :,:), ALLOCATABLE :: darr_lat4ij
      DOUBLE PRECISION, DIMENSION(:,:),      ALLOCATABLE :: darry_ij


      INTEGER,          DIMENSION(:),   ALLOCATABLE :: iarri_c
      INTEGER,          DIMENSION(:),   ALLOCATABLE :: iarrj_c
      INTEGER,          DIMENSION(:,:), ALLOCATABLE :: iarr_ij

      DOUBLE PRECISION, DIMENSION(:),   ALLOCATABLE :: darrx_c
      DOUBLE PRECISION, DIMENSION(:),   ALLOCATABLE :: darry_c
#endif


                                    ! Longitude value of
                                    ! grid element (ix,jy)
      CHARACTER(LEN=*), PARAMETER   :: c_name_lon4ij  = 'lon4ij'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_lon4ij =
     &                              'Array of grid longitudes'
      CHARACTER(LEN=*), PARAMETER   :: c_unit_lon4ij  = un_degE
      INTEGER,          PARAMETER   :: nlen_lname_lon4ij =
     &                              LEN(c_lname_lon4ij)
      INTEGER,          PARAMETER   :: nlen_unit_lon4ij =
     &                              LEN(c_unit_lon4ij)
      INTEGER                       :: id_lon4ij

                                    ! Latitude value of
                                    ! grid element (ix,jy)
      CHARACTER(LEN=*), PARAMETER   :: c_name_lat4ij  = 'lat4ij'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_lat4ij =
     &                              'Array of grid latitudes'
      CHARACTER(LEN=*), PARAMETER   :: c_unit_lat4ij  = un_degN
      INTEGER,          PARAMETER   :: nlen_lname_lat4ij =
     &                              LEN(c_lname_lat4ij)
      INTEGER,          PARAMETER   :: nlen_unit_lat4ij =
     &                              LEN(c_unit_lat4ij)
      INTEGER                       ::  id_lat4ij

      INTEGER :: nix_bec = -1
      INTEGER :: njy_bec = -1


#ifdef DEBUG
      CHARACTER(LEN=*), PARAMETER :: cfmt_dbg_a
     &  = '("[STORE_NC_AUX] debug: ", A)'
#endif


! Subroutine start
! ================

      CALL SNCA_SETUP()


                                    ! The root process first reads in
                                    ! the longitude-latitude information
                                    ! for the grid
      IF (i_myrank == jp_exeproc_root) THEN

        CALL GET_NLONSLATSZTS_FILE(nix_bec, njy_bec)

                                    ! Open file <cfn_ncin_bec>
                                    ! to retrieve the complete longitude
                                    ! and latitude arrays
        istatus = NF_OPEN(cfn_ncin_bec, NF_NOWRITE, ncid_file)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

        ALLOCATE(rarr_tlong(nix_bec, njy_bec))
        ALLOCATE(rarr_tlat(nix_bec, njy_bec))

                                    ! Read in longitude array
                                    ! (type: FLOAT; units: degrees_east)
        istatus = NF_INQ_VARID(ncid_file, 'TLONG', ncid_var)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
        istatus = NF_GET_VAR_REAL(ncid_file, ncid_var, rarr_tlong)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

                                    ! Read in latitude array
                                    ! (type: FLOAT; units: degrees_north)
        istatus = NF_INQ_VARID(ncid_file, 'TLAT', ncid_var)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
        istatus = NF_GET_VAR_REAL(ncid_file, ncid_var, rarr_tlat)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

                                    ! Close the file, since all that
                                    ! can be used has been read in now.
        istatus = NF_CLOSE(ncid_file)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))


        ALLOCATE(darr_lon4ij(nix,njy, npx,npy))
        ALLOCATE(darr_lat4ij(nix,njy, npx,npy))


      ENDIF


      ALLOCATE(iarr_c(nsedcol_central))
      ALLOCATE(darr_c(nsedcol_central))

#ifdef MEDUSA_BASE2D
      ALLOCATE(iarr_ij(nix, njy))
      ALLOCATE(darrx_ij(nix, njy))
      ALLOCATE(darry_ij(nix, njy))

      ALLOCATE(iarri_c(nsedcol_central))
      ALLOCATE(iarrj_c(nsedcol_central))

      ALLOCATE(darrx_c(nsedcol_central))
      ALLOCATE(darry_c(nsedcol_central))
#endif


#ifdef ALLOW_MPI
                                    ! The root process now sends the
                                    ! longitude-latitude grid information
                                    ! to each process (only for that part
                                    ! of the grid that belongs to the process)
                                    ! 
      IF (i_myrank == jp_exeproc_root) THEN

        DO jp = 1, npy
        DO ip = 1, npx

          DO jy = 1, njy
            j = (jp - 1) * njy + jy

            DO ix = 1, nix
              i = (ip - 1) * nix + ix

              darr_lon4ij(ix,jy, ip,jp) = DBLE(rarr_tlong(i,j))
              darr_lat4ij(ix,jy, ip,jp) = DBLE(rarr_tlat(i,j))

            ENDDO

          ENDDO

        ENDDO
        ENDDO


        DO i_destrank = 0, n_cprocs-1

          CALL MEDEXE_MPI_TOPO2D_PP4RANK(i_destrank, ip, jp)

          IF (i_destrank == jp_exeproc_root) THEN

            darrx_ij(:,:) = darr_lon4ij(:,:, ip,jp)
            darry_ij(:,:) = darr_lat4ij(:,:, ip,jp)

          ELSE

#  ifdef DEBUG
            WRITE(jp_stddbg, cfmt_dbg_a, ADVANCE='NO')
     &        'Now sending arrays to process '
            WRITE(jp_stddbg, '(I0, " (ip=",I0,",jp=",I0,")")')
     &        i_destrank, ip, jp
#  endif

            CALL MPI_SEND(darr_lon4ij(:,:, ip,jp), nijxy,
     &                              MPI_DOUBLE_PRECISION,
     &                              i_destrank, i_destrank,
     &                              i_mycomm, iflag)

            CALL MPI_SEND(darr_lat4ij(:,:, ip,jp), nijxy,
     &                              MPI_DOUBLE_PRECISION,
     &                              i_destrank, i_destrank,
     &                              i_mycomm, iflag)

          ENDIF

        ENDDO

      ELSE

#  ifdef DEBUG
        WRITE(jp_stddbg, cfmt_dbg_a, ADVANCE='NO')
     &    'Now receiving arrays from process '
        WRITE(jp_stddbg, '(I0)') jp_exeproc_root
#  endif

        CALL MPI_RECV(darrx_ij, nijxy, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_myrank,
     &                              i_mycomm, impi_status, iflag)
        CALL MPI_RECV(darry_ij, nijxy, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_myrank,
     &                              i_mycomm, impi_status, iflag)

      ENDIF
#endif


      IF (l_file_is_mine) THEN

        !-----------------------
        ! Create the data file
        !-----------------------
        istatus = NF_CREATE(TRIM(cfn_ncout_aux), NF_CLOBBER, ncid)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))


        !-------------------------------------------
        ! Define dimensions and dimension variables
        !-------------------------------------------

                                    ! Columns on the string
                                    ! ---------------------
                                    ! - dim id
        istatus = NF_DEF_DIM(ncid, 'col', nsedcol_ncfile, dim_col)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

                                    ! - dim variable id
        istatus = NF_DEF_VAR(ncid, 'col', NF_INT, 1, dim_col, id_col)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

#ifdef ALLOW_MPI
        IF (l_onewrites4many) THEN
          istatus = NF_PUT_ATT_INT(ncid, id_col, 'offset', NF_INT, 1, 0)
          IF (istatus /= NF_NOERR)
     &      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
        ELSE
          istatus = NF_PUT_ATT_INT(ncid, id_col, 'offset', NF_INT, 1,
     &                              ioffset_sedcol_pproc(i_myrank))
          IF (istatus /= NF_NOERR)
     &      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
       ENDIF
#else
        istatus = NF_PUT_ATT_INT(ncid, id_col, 'offset', NF_INT, 1, 0)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
#endif


                                    ! Levels
                                    ! ------
                                    ! - dim id
        istatus = NF_DEF_DIM(ncid, 'lev', ndn, dim_lev)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

                                    ! - dim variable id
        istatus = NF_DEF_VAR(ncid, 'lev', NF_INT, 1, dim_lev, id_lev)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

                                    ! - dim variable attributes
        istatus = NF_PUT_ATT_TEXT(ncid, id_lev, 'positive', 4, 'down')
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))


#ifdef MEDUSA_BASE2D
                                    ! 'X' index
                                    ! ---------
                                    ! - dim id
        istatus = NF_DEF_DIM(ncid, c_name_ix, nix_ncfile, dim_ix)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

                                    ! - dim variable id
        istatus = NF_DEF_VAR(ncid, c_name_ix, NF_INT,
     &                              1, dim_ix, id_ix)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

                                    ! - dim variable attributes
        istatus = NF_PUT_ATT_TEXT(ncid, id_ix, 'long_name',
     &                              nlen_lname_ix, c_lname_ix)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

#  ifdef ALLOW_MPI
        IF (l_onewrites4many) THEN
          istatus = NF_PUT_ATT_INT(ncid, id_ix, 'offset', NF_INT, 1, 0)
          IF (istatus /= NF_NOERR)
     &      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
        ELSE
          i = 0                     ! get the offset
          CALL SNCA_CENT2GLOB_IJ(i=i, ip=i_myip, jp=i_myjp)
          istatus = NF_PUT_ATT_INT(ncid, id_ix, 'offset', NF_INT, 1, i)
          IF (istatus /= NF_NOERR)
     &      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
        ENDIF
#  else
        istatus = NF_PUT_ATT_INT(ncid, id_ix, 'offset', NF_INT, 1, 0)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
#  endif


                                    ! 'Y' index
                                    ! ---------
                                    ! - dim id
        istatus = NF_DEF_DIM(ncid, c_name_jy, njy_ncfile, dim_jy)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

                                    ! - dim variable id
        istatus = NF_DEF_VAR(ncid, c_name_jy, NF_INT,
     &                              1, dim_jy, id_jy)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

                                    ! - dim variable attributes
        istatus = NF_PUT_ATT_TEXT(ncid, id_jy, 'long_name',
     &                         nlen_lname_jy, c_lname_jy)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

#  ifdef ALLOW_MPI
        IF (l_onewrites4many) THEN
          istatus = NF_PUT_ATT_INT(ncid, id_jy, 'offset', NF_INT, 1, 0)
          IF (istatus /= NF_NOERR)
     &      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
        ELSE
          j = 0                     ! get the offset
          CALL SNCA_CENT2GLOB_IJ(j=j, ip=i_myip, jp=i_myjp)
          istatus = NF_PUT_ATT_INT(ncid, id_jy, 'offset', NF_INT, 1, j)
          IF (istatus /= NF_NOERR)
     &      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
        ENDIF
#  else
        istatus = NF_PUT_ATT_INT(ncid, id_jy, 'offset', NF_INT, 1, 0)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
#  endif
#endif


        !-----------------------
        ! Define data variables
        !-----------------------

                                    ! xzdn(lev)
                                    ! ---------
        istatus = NF_DEF_VAR(ncid, vsn_xzdn, NF_DOUBLE,
     &                              1, dim_lev, id_xzdn)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))
        istatus = NF_PUT_ATT_TEXT(ncid, id_xzdn,
     &                              'long_name', vll_xzdn, vln_xzdn)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))
        istatus = NF_PUT_ATT_TEXT(ncid, id_xzdn,
     &                              'units', ul_m, un_m)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))


                                    ! Area(col)
                                    ! ---------
        istatus = NF_DEF_VAR(ncid, 'area', NF_DOUBLE,
     &                              1, dim_col, id_area)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))
        istatus = NF_PUT_ATT_TEXT(ncid, id_area, 'long_name',
     &                              12, 'Surface_Area')
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))
        istatus = NF_PUT_ATT_TEXT(ncid, id_area, 'units', ul_m2, un_m2)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))


#ifdef MEDUSA_BASE2D
                                    ! maskval(col)
                                    ! ------------

        istatus = NF_DEF_VAR(ncid, 'maskval', NF_INT,
     &                              1, dim_col, id_maskval)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_TEXT(ncid, id_maskval, 'long_name',
     &                              10, 'Mask_Value')
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_INT(ncid, id_maskval,
     &                              '_FillValue', NF_INT, 1, -1)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))


                                    ! col4ij(ix,jy)
                                    ! -------------
        dim(1) = dim_ix
        dim(2) = dim_jy

        istatus = NF_DEF_VAR(ncid, 'col4ij', NF_INT,
     &                              2, dim(1:2), id_col4ij)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_TEXT(ncid, id_col4ij, 'long_name',
     &                              18, 'Column_index_of_ij')
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_INT(ncid, id_col4ij,
     &                              '_FillValue', NF_INT, 1, -1)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_INT(ncid, id_col4ij,
     &                              'valid_min', NF_INT, 1, 1)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_INT(ncid, id_col4ij,
     &                              'valid_max', NF_INT,
     &                              1, nsedcol_ncfile)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-4))


                                    ! xref4col(col)
                                    ! -------------
        istatus = NF_DEF_VAR(ncid, c_name_xref4col, NF_DOUBLE,
     &                              1, dim_col, id_xref4col)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_TEXT(ncid, id_xref4col, 'long_name',
     &                              nlen_lname_xref4col,
     &                              c_lname_xref4col)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-4))

        istatus = NF_PUT_ATT_TEXT(ncid, id_xref4col, 'units',
     &                              nlen_unit_xref4col,
     &                              c_unit_xref4col)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-4))


                                    ! yref4col(col)
                                    ! -------------
        istatus = NF_DEF_VAR(ncid, c_name_yref4col, NF_DOUBLE,
     &                              1, dim_col, id_yref4col)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_TEXT(ncid, id_yref4col, 'long_name',
     &                              nlen_lname_yref4col,
     &                              c_lname_yref4col)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-4))

        istatus = NF_PUT_ATT_TEXT(ncid, id_yref4col, 'units',
     &                              nlen_unit_yref4col,
     &                              c_unit_yref4col)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-4))


                                    ! i4col(col)
                                    ! ----------
        istatus = NF_DEF_VAR(ncid, c_name_i4col, NF_INT,
     &                              1, dim_col, id_i4col)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_TEXT(ncid, id_i4col, 'long_name',
     &                              nlen_lname_i4col, c_lname_i4col)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_INT(ncid, id_i4col,
     &                              '_FillValue', NF_INT, 1, -1)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        i = 1
#  ifdef ALLOW_MPI
                                    ![MPI] if one file per process, then
                                    ! convert the central-based i=1 value
                                    ! to the global based one
        IF (.NOT. l_onewrites4many) THEN
          CALL SNCA_CENT2GLOB_IJ(i=i, ip=i_myip, jp=i_myjp)
        ENDIF
#  endif
        istatus = NF_PUT_ATT_INT(ncid, id_i4col,
     &                              'valid_min',  NF_INT, 1, i)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        i = nix_ncfile
#  ifdef ALLOW_MPI
                                    ![MPI] if one file per process, then
                                    ! convert the central-based i=nix_ncfile value
                                    ! to the global based one
        IF (.NOT. l_onewrites4many) THEN
          CALL SNCA_CENT2GLOB_IJ(i=i, ip=i_myip, jp=i_myjp)
        ENDIF
#  endif
        istatus = NF_PUT_ATT_INT(ncid, id_i4col,
     &                              'valid_max',  NF_INT, 1, i)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))


                                    ! j4col(col)
                                    ! ----------
        istatus = NF_DEF_VAR(ncid, c_name_j4col, NF_INT,
     &                              1, dim_col, id_j4col)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))
        istatus = NF_PUT_ATT_TEXT(ncid, id_j4col, 'long_name',
     &                              nlen_lname_j4col, c_lname_j4col)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_INT(ncid, id_j4col,
     &                              '_FillValue', NF_INT, 1, -1)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        j = 1
#  ifdef ALLOW_MPI
                                    ![MPI] if one file per process, then
                                    ! convert the central-based j=1 value
                                    ! to the global based one
        IF (.NOT. l_onewrites4many) THEN
          CALL SNCA_CENT2GLOB_IJ(j=j, ip=i_myip, jp=i_myjp)
        ENDIF
#  endif
        istatus = NF_PUT_ATT_INT(ncid, id_j4col,
     &                              'valid_min',  NF_INT, 1, j)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        j = njy_ncfile
#  ifdef ALLOW_MPI
                                    ![MPI] if one file per process, then
                                    ! convert the central-based j=njy_ncfile value
                                    ! to the global based one
        IF (.NOT. l_onewrites4many) THEN
          CALL SNCA_CENT2GLOB_IJ(j=j, ip=i_myip, jp=i_myjp)
        ENDIF
#  endif
        istatus = NF_PUT_ATT_INT(ncid, id_j4col,
     &                              'valid_max',  NF_INT, 1, j)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))


                                    ! lon4ij(ix,jy)
                                    ! -------------
        istatus = NF_DEF_VAR(ncid, c_name_lon4ij, NF_DOUBLE,
     &                              2, dim(1:2), id_lon4ij)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_TEXT(ncid, id_lon4ij, 'long_name',
     &                              nlen_lname_lon4ij, c_lname_lon4ij)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_TEXT(ncid, id_lon4ij, 'units',
     &                              nlen_unit_lon4ij,
     &                              c_unit_lon4ij)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-4))


                                    ! lat4ij(ix,jy)
                                    ! -------------
        istatus = NF_DEF_VAR(ncid, c_name_lat4ij, NF_DOUBLE,
     &                              2, dim(1:2), id_lat4ij)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_TEXT(ncid, id_lat4ij, 'long_name',
     &                              nlen_lname_lat4ij, c_lname_lat4ij)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        istatus = NF_PUT_ATT_TEXT(ncid, id_lat4ij, 'units',
     &                              nlen_unit_lat4ij,
     &                              c_unit_lat4ij)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-4))

#endif


        !----------------------
        ! Put global attributes
        !----------------------

        IF (PRESENT(title_string)) THEN
          istatus = NF_PUT_ATT_TEXT(ncid, NF_GLOBAL, 'title',
     &                          LEN_TRIM(title_string), title_string)
          IF (istatus /= NF_NOERR)
     &      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))
        ENDIF


        !--------------------
        ! End define mode
        !--------------------

        istatus = NF_ENDDEF(ncid)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

      ENDIF

                                    ! Set 'col' coordinate variable
                                    ! equal to its index value
      iarr_c(:) = (/ (n, n=1, nsedcol_central) /)
#ifdef ALLOW_MPI
     &             + ioffset_sedcol_pproc(i_myrank)
#endif

      CALL SNCA_PUT_C_INT(ncid, id_col, iarr_c)


      IF (l_file_is_mine) THEN
                                    ! Set 'lev' coordinate variable
                                    ! equal to its index value
        DO k = 1, idnb-idnw+1
          i = k-1+idnw
          istatus = NF_PUT_VAR1_INT(ncid, id_lev, k, i)
          IF (istatus /= NF_NOERR)
     &      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
        ENDDO

      ENDIF



#ifdef MEDUSA_BASE2D
      IF (l_file_is_mine) THEN
                                    ! Set 'ix' coordinate variable
                                    ! equal to its index value
        DO k = 1, nix_ncfile

          i = k

#  ifdef ALLOW_MPI
                                    ![MPI] if one file per process, then
                                    ! convert the central-based ix values
                                    ! to the global based
          IF (.NOT. l_onewrites4many) THEN
            CALL SNCA_CENT2GLOB_IJ(i=i, ip=i_myip, jp=i_myjp)
          ENDIF
#  endif

          istatus = NF_PUT_VAR1_INT(ncid, id_ix, k, i)
          IF (istatus /= NF_NOERR)
     &      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

        ENDDO

                                    ! Set 'jy' coordinate variable
                                    ! equal to its index value
        DO k = 1, njy_ncfile

          j = k

#  ifdef ALLOW_MPI
                                    ![MPI] if one file per process, then
                                    ! convert the central-based jy values
                                    ! to the global-based
          IF (.NOT. l_onewrites4many) THEN
            CALL SNCA_CENT2GLOB_IJ(j=j, ip=i_myip, jp=i_myjp)
          ENDIF
#  endif

          istatus = NF_PUT_VAR1_INT(ncid, id_jy, k, j)
          IF (istatus /= NF_NOERR)
     &      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

        ENDDO

      ENDIF
#endif


      IF (l_file_is_mine) THEN
                                    ! xzdn(lev)
        CALL GRID_DEF(xzdn)
        istatus = NF_PUT_VAR_DOUBLE(ncid, id_xzdn, xzdn)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

      ENDIF


#ifdef MEDUSA_BASE2D
                                    ! col4ij
      DO i = 1, nix
      DO j = 1, njy

        CALL COLUMN_IJ2N(i, j, iflag, k)

        IF (iflag <= 0) THEN        ! Ignore warning for "Not-at-seafloor"

#  ifdef ALLOW_MPI
          IF (k /= jp_is_not_at_seafloor_idx) THEN
            iarr_ij(i, j) = k + ioffset_sedcol_pproc(i_myrank)
          ELSE
            iarr_ij(i, j) = jp_is_not_at_seafloor_idx
          ENDIF
#  else
          iarr_ij(i, j) = k
#  endif

        ELSE

          WRITE(jp_stderr,*) 'Error upon COLUMN_IJ2N call:'
          WRITE(jp_stderr,*) ' called with i,j = ', i, j
          WRITE(jp_stderr,*) ' returned iflag = ', iflag
          WRITE(jp_stderr,*) 'Aborting!'
          CALL ABORT_MEDUSA()

        ENDIF

      ENDDO
      ENDDO

      CALL SNCA_PUT_IJ_INT(ncid, id_col4ij, iarr_ij)
#endif


#ifdef MEDUSA_BASE2D
                                    ! i4col, j4col
      DO k = 1, nsedcol_central

        CALL COLUMN_N2IJ(k, iflag, i, j)

        IF (iflag == 0) THEN

#  ifdef ALLOW_MPI
                                    ![MPI]: need to correct i and j:
                                    ! (i,j) central -> (i,j) global
          CALL SNCA_CENT2GLOB_IJ(i=i, j=j, ip=i_myip, jp=i_myjp)
#  endif
          iarri_c(k) = i
          iarrj_c(k) = j

        ELSE

          WRITE(jp_stderr,*) 'Error upon COLUMN_N2IJ call:'
          WRITE(jp_stderr,*) ' called with k = ', k
          WRITE(jp_stderr,*) ' returned iflag = ', iflag
          WRITE(jp_stderr,*) 'Aborting!'
          CALL ABORT_MEDUSA()

        ENDIF

      ENDDO

      CALL SNCA_PUT_C_INT(ncid, id_i4col, iarri_c)
      CALL SNCA_PUT_C_INT(ncid, id_j4col, iarrj_c)
#endif


#ifdef MEDUSA_BASE2D
                                    ! xref4col, yref4jcol
      DO k = 1, nsedcol_central

        CALL COLUMN_N2XY(k, iflag, x, y)

        IF (iflag == 0) THEN

          darrx_c(k) = x
          darry_c(k) = y

        ELSE

          WRITE(jp_stderr,*) 'Error upon COLUMN_N2XY call:'
          WRITE(jp_stderr,*) ' called with k = ', k
          WRITE(jp_stderr,*) ' returned iflag = ', iflag
          WRITE(jp_stderr,*) 'Aborting!'
          CALL ABORT_MEDUSA()

        ENDIF

      ENDDO

      CALL SNCA_PUT_C_DOUBLE(ncid, id_xref4col, darrx_c)
      CALL SNCA_PUT_C_DOUBLE(ncid, id_yref4col, darry_c)


                                    ! lon4ij, lat4ij
      CALL SNCA_PUT_IJ_DOUBLE(ncid, id_lon4ij, darrx_ij)
      CALL SNCA_PUT_IJ_DOUBLE(ncid, id_lat4ij, darry_ij)
#endif

                                    ! Surface areas of grid_elements
      DO k = 1, nsedcol_central

        CALL COLUMN_AREA4N(k, iflag, sfc_area)

        IF (iflag == 0) THEN

          darr_c(k) = sfc_area

        ELSE

          WRITE(jp_stderr,*) 'Error upon COLUMN_AREA4N call:'
          WRITE(jp_stderr,*) ' called with k = ', k
          WRITE(jp_stderr,*) ' returned iflag = ', iflag
          WRITE(jp_stderr,*) 'Aborting!'
          CALL ABORT_MEDUSA()

        ENDIF

      ENDDO

      CALL SNCA_PUT_C_DOUBLE(ncid, id_area, darr_c)


                                    ! maskval
      DO k = 1, nsedcol_central

        CALL COLUMN_N2MASKVAL(k, iflag, mval)

        IF (iflag == 0) THEN

          iarr_c(k) = mval

        ELSE

          WRITE(jp_stderr,*) 'Error upon COLUMN_N2MASKVAL call:'
          WRITE(jp_stderr,*) ' called with k = ', k
          WRITE(jp_stderr,*) ' returned iflag = ', iflag
          WRITE(jp_stderr,*) 'Aborting!'
          CALL ABORT_MEDUSA()

        ENDIF

      ENDDO

      CALL SNCA_PUT_C_INT(ncid, id_maskval, iarr_c)


      IF (l_file_is_mine) THEN
        istatus = NF_CLOSE(ncid)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
      ENDIF


      DEALLOCATE(iarr_c)
      DEALLOCATE(darr_c)

#ifdef MEDUSA_BASE2D
      DEALLOCATE(iarr_ij)

      IF (i_myrank == jp_exeproc_root) THEN
        DEALLOCATE(rarr_tlong)
        DEALLOCATE(rarr_tlat)
        DEALLOCATE(darr_lon4ij)
        DEALLOCATE(darr_lat4ij)
      ENDIF

      DEALLOCATE(iarri_c)
      DEALLOCATE(iarrj_c)

      DEALLOCATE(darrx_c)
      DEALLOCATE(darry_c)
      DEALLOCATE(darrx_ij)
      DEALLOCATE(darry_ij)
#endif


      CALL SNCA_RESET()


      RETURN



      CONTAINS



#ifdef ALLOW_MPI
#  ifdef MEDUSA_BASE2D
!-----------------------------------------------------------------------
      SUBROUTINE SNCA_CENT2GLOB_IJ(i, j, ip, jp)
!-----------------------------------------------------------------------

! Adjust the i and j coordinates on the process' subgrid
! to their counterparts on the global grid if
! - running in an MPI environment
! - and l_onewritesformany
! Assumes (npx,npy) array of processes, each process dealing with an
! (nix,njy) array, with no overlapping.


      IMPLICIT NONE

      INTEGER, INTENT(INOUT), OPTIONAL :: i,  j
      INTEGER, INTENT(IN)              :: ip, jp


      IF (PRESENT(i)) i = (ip-1)*nix + i
      IF (PRESENT(j)) j = (jp-1)*njy + j

      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE SNCA_CENT2GLOB_IJ
!-----------------------------------------------------------------------
#  endif
#endif


!-----------------------------------------------------------------------
      SUBROUTINE SNCA_SETUP()
!-----------------------------------------------------------------------

      USE mod_seafloor_central,     ONLY: N_COLUMNS_USED
#ifdef MEDUSA_BASE2D
      USE mod_seafloor_central,     ONLY: IJ_COLUMNS_USED
#endif

#ifdef ALLOW_MPI
      USE mod_execontrol_medusa,    ONLY: MEDEXE_NPROC,
     &                                    MEDEXE_MPI_COMM,
     &                                    MEDEXE_MPI_COMM_RANK,
     &                                    MEDEXE_MPI_GETPARTITIONING,
     &                                    lp_exeproc_singleproc_nc
#  ifdef MEDUSA_BASE2D
      USE mod_execontrol_medusa,    ONLY: MEDEXE_MPI_GETTOPO_DIMLENS,
     &                                    MEDEXE_MPI_TOPO2D_PP4RANK
#  endif
#endif


      IMPLICIT NONE


#ifdef ALLOW_MPI
#  ifdef MEDUSA_BASE2D
      INTEGER, DIMENSION(2) :: ilens_mpitopo
#  endif
#endif

                                    ! Get the number of columns in the
                                    ! mod_seafloor_central of the
                                    ! current process
      CALL N_COLUMNS_USED(nsedcol_central)

                                    ! If nsedcol_central == -1, the
                                    ! basic setup has not yet been
                                    ! done, and it does not make any
                                    ! sense to proceed
      IF (nsedcol_central < 0) THEN
        WRITE(jp_stderr,*) '[STORE_NC_AUX/SNCA_SETUP] error: ' //
     &    'Sea-floor setup not yet done -- aborting!'
        CALL ABORT_MEDUSA()
      ENDIF


#ifdef MEDUSA_BASE2D
                                    ! Get the number of seafloor grid
                                    ! elements along the X and Y directions
                                    ! [MPI] would actually have to check
                                    ! if these are all the same
      CALL IJ_COLUMNS_USED(nix, njy)
      nijxy = nix*njy
#endif


#ifdef ALLOW_MPI
      n_cprocs = MEDEXE_NPROC()     ! Get number of processes

                                    ! Get communicator of Medusa
      i_mycomm = MEDEXE_MPI_COMM()

                                    ! Get rank of process executing this instance
      i_myrank = MEDEXE_MPI_COMM_RANK()

#  ifdef MEDUSA_BASE2D
                                    ! Get the 2D process array shape.
      CALL MEDEXE_MPI_GETTOPO_DIMLENS(ilens_mpitopo)
      npx = ilens_mpitopo(1)
      npy = ilens_mpitopo(2)
                                    ! Get the process coordinates in the 2D topo
                                    ! for the process executing this instance
      CALL MEDEXE_MPI_TOPO2D_PP4RANK(i_myrank, i_myip, i_myjp)
#  endif

                                    ! Now set up the file writing
                                    ! distribution logic:
      l_file_is_mine = .TRUE.       ! - by default, each process writes
                                    !   its own files

      l_onewrites4many = .FALSE.    ! - by default: no process writes data
                                    !   for others; this must be .FALSE.
                                    !   if n_cprocs==1 ("many" means ">1")

      IF (n_cprocs > 1) THEN        ! When running in a multi-processor
                                    ! environment with more than one process,
                                    ! then this may need to be adapted:
        IF (lp_exeproc_singleproc_nc) THEN  ! - if single-processor I/O
                                            !   for NetCDF files is required,
          l_onewrites4many = .TRUE.         !   set flag to indicate that the
                                            !   writing process writes for
                                            !   other (>1) processes than itself
          IF (i_myrank /= jp_exeproc_ncio) THEN     ! - and deactivate writing for other
            l_file_is_mine = .FALSE.                !   processes than the writing one
          ENDIF

        ELSE                        ! Multi-process environment where each
                                    ! process controls its own file. However,
                                    ! do not create a file if there are no
                                    ! data to write. This could happen if one
                                    ! process gets attributed a domain without
                                    ! seafloor points.
          IF (nsedcol_central == 0) l_file_is_mine = .FALSE.

        ENDIF

      ENDIF
                                    ! Get the number of columns in all
                                    ! the mod_seafloor_central's of all
                                    ! concurrent processes and the
                                    ! partitioning information
      ALLOCATE(nsedcol_pproc(0:n_cprocs-1))
      ALLOCATE(ioffset_sedcol_pproc(0:n_cprocs-1))

      CALL MEDEXE_MPI_GETPARTITIONING(nsedcol_global, iproc_1stocn,
     &                              nsedcol_pproc, ioffset_sedcol_pproc)

                                    ! If single-processor NetCDF I/O, and
                                    ! execution environment includes more
                                    ! than one processor, then allocate
                                    ! work-space memory to receive the data
                                    ! by the writing process:
      IF (l_onewrites4many) THEN

        nsedcol_ncfile = nsedcol_global

        IF (i_myrank == jp_exeproc_ncio) THEN
          ALLOCATE(iarr_mpirecv_c(nsedcol_ncfile))
          ALLOCATE(darr_mpirecv_c(nsedcol_ncfile))
        ELSE
          ALLOCATE(iarr_mpirecv_c(0))       ! pro-forma allocation
          ALLOCATE(darr_mpirecv_c(0))       ! pro-forma allocation
        ENDIF

#  ifdef MEDUSA_BASE2D
        nix_ncfile = nix*npx
        njy_ncfile = njy*npy

        IF (i_myrank == jp_exeproc_ncio) THEN
          ALLOCATE(iarr_mpirecv_ij(nix, njy))
          ALLOCATE(darr_mpirecv_ij(nix, njy))
        ELSE
          ALLOCATE(iarr_mpirecv_ij(0, 0))   ! pro-forma allocation
          ALLOCATE(darr_mpirecv_ij(0, 0))   ! pro-forma allocation
        ENDIF
#  endif

      ELSE

        nsedcol_ncfile = nsedcol_central

#  ifdef MEDUSA_BASE2D
        nix_ncfile = nix
        njy_ncfile = njy
#  endif

      ENDIF

#  ifdef DEBUG
      WRITE(jp_stddbg, '(A,I0)')
     &   '[SNCA_SETUP] n_cprocs = ', n_cprocs
      WRITE(jp_stddbg, '(A,I0)')
     &   '[SNCA_SETUP] i_myrank = ', i_myrank
#    ifdef MEDUSA_BASE2D
      WRITE(jp_stddbg, '(A,"("I0,",",I0,")")')
     &   '[SNCA_SETUP] (i_myip, i_myjp) = ', i_myip, i_myjp
#    endif
      WRITE(jp_stddbg, '(A,I0)')
     &   '[SNCA_SETUP] nsedcol_central = ', nsedcol_central
      WRITE(jp_stddbg, '(A,I0)')
     &   '[SNCA_SETUP] nsedcol_global = ', nsedcol_global
      WRITE(jp_stddbg, '(A,I0)')
     &   '[SNCA_SETUP] nsedcol_ncfile = ', nsedcol_ncfile
#    ifdef MEDUSA_BASE2D
      WRITE(jp_stddbg, '(A,I0," * ",I0)')
     &   '[SNCA_SETUP] nix * njy = ', nix, njy
      WRITE(jp_stddbg, '(A,I0," * ",I0)')
     &   '[SNCA_SETUP] nix_ncfile * njy_ncfile = ',
     &   nix_ncfile, njy_ncfile
#    endif
      WRITE(jp_stddbg, '(A,L1)')
     &   '[SNCA_SETUP] l_onewrites4many = ', l_onewrites4many
      WRITE(jp_stddbg, '(A,L1)')
     &   '[SNCA_SETUP] l_file_is_mine = ', l_file_is_mine
#  endif
#else
                                    ! In single-processor environments the
                                    ! running process always owns its files
                                    ! and writes only nsedcol_central-long
                                    ! records
      l_file_is_mine = .TRUE.
      nsedcol_ncfile = nsedcol_central

#  ifdef MEDUSA_BASE2D
      nix_ncfile = nix
      njy_ncfile = njy
#  endif

#  ifdef DEBUG
      WRITE(jp_stddbg, '(A,I0)')
     &   '[SNCA_SETUP] nsedcol_central = ', nsedcol_central
      WRITE(jp_stddbg, '(A,I0)')
     &   '[SNCA_SETUP] nsedcol_ncfile = ', nsedcol_ncfile
#    ifdef MEDUSA_BASE2D
      WRITE(jp_stddbg, '(A,I0," * ",I0)')
     &   '[SNCA_SETUP] nix * njy = ', nix, njy
      WRITE(jp_stddbg, '(A,I0," * ",I0)')
     &   '[SNCA_SETUP] nix_ncfile * njy_ncfile = ',
     &   nix_ncfile, njy_ncfile
#    endif
      WRITE(jp_stddbg, '(A,L1)')
     &   '[SNCA_SETUP] l_file_is_mine = ', l_file_is_mine
#  endif
#endif


      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE SNCA_SETUP
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE SNCA_RESET()
!-----------------------------------------------------------------------


      IMPLICIT NONE


      nsedcol_central = -1
      nsedcol_ncfile  = -1

#ifdef MEDUSA_BASE2D
      nix = -1
      njy = -1
      nix_ncfile = -1
      njy_ncfile = -1
#endif

#ifdef ALLOW_MPI
                                    ! Deallocate the work-space memory reserved
                                    ! to receive the data in the writing process.
                                    ! Deallocate in all processes: in the non
                                    ! writing ones, the arrays are also allocated,
                                    ! with 0 size.
      IF (l_onewrites4many) THEN
        DEALLOCATE(iarr_mpirecv_c)
        DEALLOCATE(darr_mpirecv_c)
#  ifdef MEDUSA_BASE2D
        DEALLOCATE(iarr_mpirecv_ij)
        DEALLOCATE(darr_mpirecv_ij)
#  endif
      ENDIF

      DEALLOCATE(nsedcol_pproc)
      DEALLOCATE(ioffset_sedcol_pproc)

      nsedcol_global = -1
      iproc_1stocn   = -1

      n_cprocs = -1
      i_myrank = MPI_PROC_NULL
#  ifdef MEDUSA_BASE2D
      i_myip   = -1
      i_myjp   = -1
#  endif
#endif


      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE SNCA_RESET
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE SNCA_PUT_C_DOUBLE(ncid, id_var, dvar_c)
!-----------------------------------------------------------------------

! Write out one complete record provided by the DOUBLE PRECISION array
! dvar_c(1:nsedcol_central) into a NetCDF variable with dimension (dim_col)
! Under MPI, stitch them together if l_onewrites4many==.TRUE.

#ifdef ALLOW_MPI
      USE mpi,                      ONLY: MPI_DOUBLE_PRECISION
#endif


      IMPLICIT NONE


      INTEGER,          INTENT(IN)               :: ncid
      INTEGER,          INTENT(IN)               :: id_var
      DOUBLE PRECISION, INTENT(IN), DIMENSION(:) :: dvar_c


      INTEGER :: istatus


#ifdef ALLOW_MPI
      IF (l_onewrites4many) THEN

        CALL MPI_GATHERV(dvar_c(:),
     &    nsedcol_central, MPI_DOUBLE_PRECISION,
     &    darr_mpirecv_c(:),
     &    nsedcol_pproc(:), ioffset_sedcol_pproc(:),
     &    MPI_DOUBLE_PRECISION,
     &    jp_exeproc_ncio, i_mycomm, istatus)

        IF (i_myrank == jp_exeproc_ncio) THEN
          istatus = NF_PUT_VAR_DOUBLE(ncid, id_var, darr_mpirecv_c(:))
          IF (istatus /= NF_NOERR)
     &      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
        ENDIF

      ELSE

        istatus = NF_PUT_VAR_DOUBLE(ncid, id_var, dvar_c(:))
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

      ENDIF
#else
      istatus = NF_PUT_VAR_DOUBLE(ncid, id_var, dvar_c(:))
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
#endif


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE SNCA_PUT_C_DOUBLE
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE SNCA_PUT_C_INT(ncid, id_var, ivar_c)
!-----------------------------------------------------------------------

! Write out one complete record provided by the INTEGER array
! ivar_c(1:nsedcol_central) into a NetCDF variable with dimension (dim_col)
! Under MPI, stitch them together if l_onewrites4many==.TRUE.

#ifdef ALLOW_MPI
      USE mpi,                      ONLY: MPI_INTEGER
#endif


      IMPLICIT NONE


      INTEGER, INTENT(IN)               :: ncid
      INTEGER, INTENT(IN)               :: id_var
      INTEGER, INTENT(IN), DIMENSION(:) :: ivar_c


      INTEGER :: istatus


#ifdef ALLOW_MPI
      IF (l_onewrites4many) THEN

        CALL MPI_GATHERV(ivar_c(:),
     &    nsedcol_central, MPI_INTEGER,
     &    iarr_mpirecv_c(:),
     &    nsedcol_pproc(:), ioffset_sedcol_pproc(:),
     &    MPI_INTEGER,
     &    jp_exeproc_ncio, i_mycomm, istatus)

        IF (i_myrank == jp_exeproc_ncio) THEN
          istatus = NF_PUT_VAR_INT(ncid, id_var, iarr_mpirecv_c(:))
          IF (istatus /= NF_NOERR)
     &      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
        ENDIF

      ELSE

        istatus = NF_PUT_VAR_INT(ncid, id_var, ivar_c(:))
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

      ENDIF
#else
      istatus = NF_PUT_VAR_INT(ncid, id_var, ivar_c(:))
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
#endif


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE SNCA_PUT_C_INT
!-----------------------------------------------------------------------



#ifdef MEDUSA_BASE2D
!-----------------------------------------------------------------------
      SUBROUTINE SNCA_PUT_IJ_INT(ncid, id_var, ivar_ij)
!-----------------------------------------------------------------------

! Write out one sub-record provided by the INTEGER array
! ivar_ij(1:nix,1:njy) into a NetCDF array with dimension
! (1:nix_ncfile,1:njy_ncfile).
! Under MPI, stitch them together if l_onewrites4many==.TRUE.

#  ifdef ALLOW_MPI
      USE mpi,                      ONLY: MPI_INTEGER, MPI_STATUS_SIZE
      USE mod_execontrol_medusa,    ONLY: MEDEXE_MPI_TOPO2D_PP4RANK
#  endif


      IMPLICIT NONE


      INTEGER, INTENT(IN)                 :: ncid
      INTEGER, INTENT(IN)                 :: id_var
      INTEGER, INTENT(IN), DIMENSION(:,:) :: ivar_ij


      INTEGER :: istatus
      INTEGER, DIMENSION(2) :: istart, ncount
#  ifdef ALLOW_MPI
      INTEGER :: i_othrank
      INTEGER, DIMENSION(MPI_STATUS_SIZE) :: impi_status
#  endif

#  ifdef DEBUG
      CHARACTER(LEN=*), PARAMETER :: cfmt_dbg_a
     &  = '("[STORE_NC_AUX/SNCA_PUT_IJ_INT] debug: ", A)'
#  endif


      IF (ANY(SHAPE(ivar_ij) /= (/ nix, njy /))) THEN

        WRITE(jp_stderr, '("[SNCA_PUT_IJ_INT] error: ")', ADVANCE='NO')
        WRITE(jp_stderr, '("incorrect shape for ivar_ij")')
        WRITE(jp_stderr, '(" expected: (/", I0, ",", I0, "/)")')
     &    nix, njy
        WRITE(jp_stderr, '(" actually: (/", I0, ",", I0, "/)")')
     &    SHAPE(ivar_ij)
        WRITE(jp_stderr, '("Aborting!")')

        CALL ABORT_MEDUSA()

      ENDIF


#  ifdef ALLOW_MPI
      IF (l_onewrites4many) THEN

        IF (i_myrank == jp_exeproc_ncio) THEN

          DO i_othrank = 0, n_cprocs-1

            CALL MEDEXE_MPI_TOPO2D_PP4RANK(i_othrank, ip, jp)

            i = 1
            j = 1
            CALL SNCA_CENT2GLOB_IJ(i=i, j=j, ip=ip, jp=jp)

            istart(1:2) = (/   i,   j /)
            ncount(1:2) = (/ nix, njy /)

            IF (i_othrank == jp_exeproc_ncio) THEN

              istatus = NF_PUT_VARA_INT(ncid, id_var,
     &                              istart(1:2), ncount(1:2),
     &                              ivar_ij(:,:))
              IF (istatus /= NF_NOERR)
     &         CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-4))

            ELSE

#    ifdef DEBUG
              WRITE(jp_stddbg, cfmt_dbg_a, ADVANCE='NO')
     &          'Now receiving arrays from process '
              WRITE(jp_stddbg, '(I0, " (ip=",I0,",jp=",I0,")")')
     &          i_othrank, ip, jp
#    endif

              CALL MPI_RECV(iarr_mpirecv_ij, nijxy, MPI_INTEGER,
     &                              i_othrank, i_othrank,
     &                              i_mycomm, impi_status, iflag)

              istatus = NF_PUT_VARA_INT(ncid, id_var,
     &                              istart(1:2), ncount(1:2),
     &                              iarr_mpirecv_ij(:,:))
              IF (istatus /= NF_NOERR)
     &         CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-4))

            ENDIF

          ENDDO

        ELSE

#    ifdef DEBUG
          WRITE(jp_stddbg, cfmt_dbg_a, ADVANCE='NO')
     &      'Now sending arrays to process '
          WRITE(jp_stddbg, '(I0)') jp_exeproc_ncio
#    endif

          CALL MPI_SEND(ivar_ij, nijxy, MPI_INTEGER,
     &                              jp_exeproc_ncio, i_myrank,
     &                              i_mycomm, iflag)

        ENDIF


      ELSE

        istatus = NF_PUT_VAR_INT(ncid, id_var, ivar_ij(:,:))
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

      ENDIF
#  else
      istatus = NF_PUT_VAR_INT(ncid, id_var, ivar_ij(:,:))
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
#  endif


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE SNCA_PUT_IJ_INT
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
      SUBROUTINE SNCA_PUT_IJ_DOUBLE(ncid, id_var, dvar_ij)
!-----------------------------------------------------------------------

! Write out one sub-record provided by the DOUBLE PRECISION array
! dvar_ij(1:nix,1:njy) into a NetCDF array with dimension
! (1:nix_ncfile,1:njy_ncfile).
! Under MPI, stitch them together if l_onewrites4many==.TRUE.

#  ifdef ALLOW_MPI
      USE mpi,                      ONLY: MPI_DOUBLE_PRECISION,
     &                                    MPI_STATUS_SIZE
      USE mod_execontrol_medusa,    ONLY: MEDEXE_MPI_TOPO2D_PP4RANK
#  endif


      IMPLICIT NONE


      INTEGER, INTENT(IN)                          :: ncid
      INTEGER, INTENT(IN)                          :: id_var
      DOUBLE PRECISION, INTENT(IN), DIMENSION(:,:) :: dvar_ij


      INTEGER :: istatus
      INTEGER, DIMENSION(2) :: istart, ncount
#  ifdef ALLOW_MPI
      INTEGER :: i_othrank
      INTEGER, DIMENSION(MPI_STATUS_SIZE) :: impi_status
#  endif

#  ifdef DEBUG
      CHARACTER(LEN=*), PARAMETER :: cfmt_dbg_a
     &  = '("[STORE_NC_AUX/SNCA_PUT_IJ_DOUBLE] debug: ", A)'
#  endif

      IF (ANY(SHAPE(dvar_ij) /= (/ nix, njy /))) THEN

        WRITE(jp_stderr, '("[SNCA_PUT_IJ_DOUBLE] error: ")',
     &                               ADVANCE='NO')
        WRITE(jp_stderr, '("incorrect shape for dvar_ij")')
        WRITE(jp_stderr, '(" expected: (/", I0, ",", I0, "/)")')
     &    nix, njy
        WRITE(jp_stderr, '(" actually: (/", I0, ",", I0, "/)")')
     &    SHAPE(dvar_ij)
        WRITE(jp_stderr, '("Aborting!")')

        CALL ABORT_MEDUSA()

      ENDIF


#  ifdef ALLOW_MPI
      IF (l_onewrites4many) THEN

        IF (i_myrank == jp_exeproc_ncio) THEN

          DO i_othrank = 0, n_cprocs-1

            CALL MEDEXE_MPI_TOPO2D_PP4RANK(i_othrank, ip, jp)

            i = 1
            j = 1
            CALL SNCA_CENT2GLOB_IJ(i=i, j=j, ip=ip, jp=jp)

            istart(1:2) = (/   i,   j /)
            ncount(1:2) = (/ nix, njy /)

            IF (i_othrank == jp_exeproc_ncio) THEN

              istatus = NF_PUT_VARA_DOUBLE(ncid, id_var,
     &                              istart(1:2), ncount(1:2),
     &                              dvar_ij(:,:))
              IF (istatus /= NF_NOERR)
     &         CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-4))

            ELSE

#    ifdef DEBUG
              WRITE(jp_stddbg, cfmt_dbg_a, ADVANCE='NO')
     &          'Now receiving arrays from process '
              WRITE(jp_stddbg, '(I0, " (ip=",I0,",jp=",I0,")")')
     &          i_othrank, ip, jp
#    endif

              CALL MPI_RECV(darr_mpirecv_ij, nijxy,
     &                              MPI_DOUBLE_PRECISION,
     &                              i_othrank, i_othrank,
     &                              i_mycomm, impi_status, iflag)

              istatus = NF_PUT_VARA_DOUBLE(ncid, id_var,
     &                              istart(1:2), ncount(1:2),
     &                              darr_mpirecv_ij(:,:))
              IF (istatus /= NF_NOERR)
     &         CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-4))

            ENDIF

          ENDDO

        ELSE

#    ifdef DEBUG
          WRITE(jp_stddbg, cfmt_dbg_a, ADVANCE='NO')
     &      'Now sending arrays to process '
          WRITE(jp_stddbg, '(I0)') jp_exeproc_ncio
#    endif

          CALL MPI_SEND(dvar_ij, nijxy, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_ncio, i_myrank,
     &                              i_mycomm, iflag)

        ENDIF


      ELSE

        istatus = NF_PUT_VAR_DOUBLE(ncid, id_var, dvar_ij(:,:))
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

      ENDIF
#  else
      istatus = NF_PUT_VAR_DOUBLE(ncid, id_var, dvar_ij(:,:))
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
#  endif


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE SNCA_PUT_IJ_DOUBLE
!-----------------------------------------------------------------------
#endif



!=======================================================================
      END SUBROUTINE STORE_NC_AUX
!=======================================================================
