!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


!=======================================================================
      SUBROUTINE STORE_NC_AUX(cfn_ncin_bec, cfn_ncout_aux, title_string)
!=======================================================================


      USE mod_defines_medusa
      USE mod_execontrol_medusa,    ONLY: ABORT_MEDUSA

      USE mpi,                      ONLY: MPI_COMM_NULL, MPI_PROC_NULL
      USE mod_execontrol_medusa,    ONLY: jp_exeproc_root

      USE mod_gridparam,            ONLY: idnw, idnb, ndn, GRID_DEF

      USE mod_seafloor_central,     ONLY: N_COLUMNS_USED, COLUMN_AREA4N

      USE mod_netcdfinc
      USE mod_netcdfparam


      IMPLICIT NONE


      CHARACTER(LEN=*), INTENT(IN)           :: cfn_ncin_bec
      CHARACTER(LEN=*), INTENT(IN)           :: cfn_ncout_aux
      CHARACTER(LEN=*), INTENT(IN), OPTIONAL :: title_string


      INTEGER :: ncid_file
      INTEGER :: ncid_var
      INTEGER :: ncid

      INTEGER :: nsedcol_central = -1
      INTEGER :: nsedcol_ncfile  = -1
      INTEGER :: nix_ncfile      = -1
      INTEGER :: njy_ncfile      = -1

      INTEGER, DIMENSION(:),   ALLOCATABLE :: ix_seafloor_file
      INTEGER, DIMENSION(:),   ALLOCATABLE :: jy_seafloor_file
      INTEGER, DIMENSION(:,:), ALLOCATABLE :: icolumn_seafloor_file


      LOGICAL :: l_file_is_mine = .TRUE.

      INTEGER :: n_cprocs = -1
      INTEGER :: i_mycomm = MPI_COMM_NULL
      INTEGER :: i_myrank = MPI_PROC_NULL
      INTEGER :: nsedcol_global = -1
      INTEGER :: iproc_1stocn   = -1
      INTEGER, DIMENSION(:), ALLOCATABLE :: nsedcol_pproc
      INTEGER, DIMENSION(:), ALLOCATABLE :: ioffset_sedcol_pproc


      INTEGER :: i, j, k, n
      INTEGER :: iflag, istatus

      INTEGER, DIMENSION(2) :: dim, istart

      DOUBLE PRECISION, DIMENSION(idnw:idnb) :: xzdn


! Dimensions and dimension variables
!  'lev' (levels)
      INTEGER            :: dim_lev         ! dimension ID
      INTEGER            ::  id_lev         ! ID of dimension variable

! 'col' (columns)
      INTEGER            :: dim_col         ! dimension ID
      INTEGER            ::  id_col         ! ID of dimension variable


                              ! ... Please adapt the names below
      CHARACTER(LEN=*), PARAMETER   :: c_name_ix     = 'ix'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_ix    = 'X_Index'
      INTEGER, PARAMETER :: nlen_lname_ix = LEN(c_lname_ix)
      INTEGER            :: dim_ix
      INTEGER            ::  id_ix

                              ! ... Please adapt the two names below
      CHARACTER(LEN=*), PARAMETER   :: c_name_jy     = 'jy'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_jy    = 'Y_Index'
      INTEGER, PARAMETER :: nlen_lname_jy = LEN(c_lname_jy)
      INTEGER            :: dim_jy
      INTEGER            ::  id_jy


! Data variables
!  - general
      INTEGER ::  id_xzdn           ! depth coordinates
      INTEGER ::  id_area           ! surface areas of columns (n)
      REAL, ALLOCATABLE, DIMENSION(:,:) :: rarr_tarea


!  - 2D-specific grid information
      INTEGER ::  id_col4ij         ! column index n for grid element (i,j)

                                    ! index i(n)
      CHARACTER(LEN=*), PARAMETER   :: c_name_i4col     = 'i4col'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_i4col    = c_lname_ix //
     &                                              '_from_Column_Index'
      INTEGER, PARAMETER :: nlen_lname_i4col = LEN(c_lname_i4col)
      INTEGER ::  id_i4col

                                    ! index j(n)
      CHARACTER(LEN=*), PARAMETER   :: c_name_j4col     = 'j4col'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_j4col    = c_lname_jy //
     &                                              '_from_Column_Index'
      INTEGER, PARAMETER :: nlen_lname_j4col = LEN(c_lname_j4col)
      INTEGER ::  id_j4col

                                    ! xref value of grid element (n)
      CHARACTER(LEN=*), PARAMETER   :: c_name_xref4col  = 'xref4col'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_xref4col = 'Longitude'
      CHARACTER(LEN=*), PARAMETER   :: c_unit_xref4col  = un_degE
      INTEGER, PARAMETER :: nlen_lname_xref4col = LEN(c_lname_xref4col)
      INTEGER, PARAMETER :: nlen_unit_xref4col  = LEN(c_unit_xref4col)
      INTEGER ::  id_xref4col
      REAL, ALLOCATABLE, DIMENSION(:,:) :: rarr_tlong

                                    ! yref value of grid element (n)
      CHARACTER(LEN=*), PARAMETER   :: c_name_yref4col  = 'yref4col'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_yref4col = 'Latitude'
      CHARACTER(LEN=*), PARAMETER   :: c_unit_yref4col  = un_degN
      INTEGER, PARAMETER :: nlen_lname_yref4col = LEN(c_lname_yref4col)
      INTEGER, PARAMETER :: nlen_unit_yref4col  = LEN(c_unit_yref4col)
      INTEGER ::  id_yref4col
      REAL, ALLOCATABLE, DIMENSION(:,:) :: rarr_tlat

      DOUBLE PRECISION  :: s, x, y



! Subroutine start
! ================

      CALL SNCA_SETUP()



      IF (l_file_is_mine) THEN

                                    ! Open file <cfn_ncin_bec>
                                    ! to retrieve the complete longitude
                                    ! and latitude arrays
        istatus = NF_OPEN(cfn_ncin_bec, NF_NOWRITE, ncid_file)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        ALLOCATE(rarr_tlong(nix_ncfile, njy_ncfile))
        ALLOCATE(rarr_tlat(nix_ncfile, njy_ncfile))
        ALLOCATE(rarr_tarea(nix_ncfile, njy_ncfile))

                                    ! Read in longitude array
                                    ! (type: FLOAT; units: degrees_east)
        istatus = NF_INQ_VARID(ncid_file, 'TLONG', ncid_var)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        istatus = NF_GET_VAR_REAL(ncid_file, ncid_var, rarr_tlong)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    ! Read in latitude array
                                    ! (type: FLOAT; units: degrees_north)
        istatus = NF_INQ_VARID(ncid_file, 'TLAT', ncid_var)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        istatus = NF_GET_VAR_REAL(ncid_file, ncid_var, rarr_tlat)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    ! Read in surface area array
                                    ! (type: FLOAT; units: cm²)
        istatus = NF_INQ_VARID(ncid_file, 'TAREA', ncid_var)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        istatus = NF_GET_VAR_REAL(ncid_file, ncid_var, rarr_tarea)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    ! Close the file, since all that
                                    ! can be used has been read in now.
        istatus = NF_CLOSE(ncid_file)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)




        !-----------------------
        ! Create the data file
        !-----------------------
        istatus = NF_CREATE(TRIM(cfn_ncout_aux), NF_CLOBBER, ncid)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


        !-------------------------------------------
        ! Define dimensions and dimension variables
        !-------------------------------------------

                                    ! Columns on the string
                                    ! ---------------------
                                    ! - dim id
        istatus = NF_DEF_DIM(ncid, 'col', nsedcol_ncfile, dim_col)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    ! - dim variable id
        istatus = NF_DEF_VAR(ncid, 'col', NF_INT, 1, dim_col, id_col)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_INT(ncid, id_col, 'offset', NF_INT, 1, 0)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


                                    ! Levels
                                    ! ------
                                    ! - dim id
        istatus = NF_DEF_DIM(ncid, 'lev', ndn, dim_lev)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    ! - dim variable id
        istatus = NF_DEF_VAR(ncid, 'lev', NF_INT, 1, dim_lev, id_lev)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    ! - dim variable attributes
        istatus = NF_PUT_ATT_TEXT(ncid, id_lev, 'positive', 4, 'down')
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


                                    ! 'X' index
                                    ! ---------
                                    ! - dim id
        istatus = NF_DEF_DIM(ncid, c_name_ix, nix_ncfile, dim_ix)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    ! - dim variable id
        istatus = NF_DEF_VAR(ncid, c_name_ix, NF_INT,
     &                              1, dim_ix, id_ix)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    ! - dim variable attributes
        istatus = NF_PUT_ATT_TEXT(ncid, id_ix, 'long_name',
     &                              nlen_lname_ix, c_lname_ix)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_INT(ncid, id_ix, 'offset', NF_INT, 1, 0)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


                                    ! 'Y' index
                                    ! ---------
                                    ! - dim id
        istatus = NF_DEF_DIM(ncid, c_name_jy, njy_ncfile, dim_jy)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    ! - dim variable id
        istatus = NF_DEF_VAR(ncid, c_name_jy, NF_INT,
     &                              1, dim_jy, id_jy)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    ! - dim variable attributes
        istatus = NF_PUT_ATT_TEXT(ncid, id_jy, 'long_name',
     &                         nlen_lname_jy, c_lname_jy)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_INT(ncid, id_jy, 'offset', NF_INT, 1, 0)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


        !-----------------------
        ! Define data variables
        !-----------------------

                                    ! xzdn(lev)
                                    ! ---------
        istatus = NF_DEF_VAR(ncid, vsn_xzdn, NF_DOUBLE,
     &                              1, dim_lev, id_xzdn)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        istatus = NF_PUT_ATT_TEXT(ncid, id_xzdn,
     &                              'long_name', vll_xzdn, vln_xzdn)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        istatus = NF_PUT_ATT_TEXT(ncid, id_xzdn,
     &                              'units', ul_m, un_m)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


                                    ! Area(col)
                                    ! ---------
        istatus = NF_DEF_VAR(ncid, 'area', NF_DOUBLE,
     &                              1, dim_col, id_area)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        istatus = NF_PUT_ATT_TEXT(ncid, id_area, 'long_name',
     &                              12, 'Surface_Area')
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        istatus = NF_PUT_ATT_TEXT(ncid, id_area, 'units', ul_m2, un_m2)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


                                    ! col4ij(ix,jy)
                                    ! -------------
        dim(1) = dim_ix
        dim(2) = dim_jy

        istatus = NF_DEF_VAR(ncid, 'col4ij', NF_INT,
     &                              2, dim(1:2), id_col4ij)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_TEXT(ncid, id_col4ij, 'long_name',
     &                              18, 'Column_index_of_ij')
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_INT(ncid, id_col4ij,
     &                              '_FillValue', NF_INT, 1, -1)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_INT(ncid, id_col4ij,
     &                              'valid_min', NF_INT, 1, 1)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_INT(ncid, id_col4ij,
     &                              'valid_max', NF_INT,
     &                              1, nsedcol_ncfile)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


                                    ! xref4col(col)
                                    ! -------------
        istatus = NF_DEF_VAR(ncid, c_name_xref4col, NF_DOUBLE,
     &                              1, dim_col, id_xref4col)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_TEXT(ncid, id_xref4col, 'long_name',
     &                              nlen_lname_xref4col,
     &                              c_lname_xref4col)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_TEXT(ncid, id_xref4col, 'units',
     &                              nlen_unit_xref4col,
     &                              c_unit_xref4col)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


                                    ! yref4col(col)
                                    ! -------------
        istatus = NF_DEF_VAR(ncid, c_name_yref4col, NF_DOUBLE,
     &                              1, dim_col, id_yref4col)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_TEXT(ncid, id_yref4col, 'long_name',
     &                              nlen_lname_yref4col,
     &                              c_lname_yref4col)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_TEXT(ncid, id_yref4col, 'units',
     &                              nlen_unit_yref4col,
     &                              c_unit_yref4col)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


                                    ! i4col(col)
                                    ! ----------
        istatus = NF_DEF_VAR(ncid, c_name_i4col, NF_INT,
     &                              1, dim_col, id_i4col)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_TEXT(ncid, id_i4col, 'long_name',
     &                              nlen_lname_i4col, c_lname_i4col)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_INT(ncid, id_i4col,
     &                              '_FillValue', NF_INT, 1, -1)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        istatus = NF_PUT_ATT_INT(ncid, id_i4col,
     &                              'valid_min',  NF_INT, 1, 1)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        istatus = NF_PUT_ATT_INT(ncid, id_i4col,
     &                              'valid_max',  NF_INT, 1, nix_ncfile)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


                                    ! j4col(col)
                                    ! ----------
        istatus = NF_DEF_VAR(ncid, c_name_j4col, NF_INT,
     &                              1, dim_col, id_j4col)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        istatus = NF_PUT_ATT_TEXT(ncid, id_j4col, 'long_name',
     &                              nlen_lname_j4col, c_lname_j4col)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_INT(ncid, id_j4col,
     &                              '_FillValue', NF_INT, 1, -1)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        istatus = NF_PUT_ATT_INT(ncid, id_j4col,
     &                              'valid_min',  NF_INT, 1, 1)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        istatus = NF_PUT_ATT_INT(ncid, id_j4col,
     &                              'valid_max',  NF_INT, 1, njy_ncfile)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)



        !----------------------
        ! Put global attributes
        !----------------------

        IF (PRESENT(title_string)) THEN
          istatus = NF_PUT_ATT_TEXT(ncid, NF_GLOBAL, 'title',
     &                          LEN_TRIM(title_string), title_string)
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        ENDIF


        !--------------------
        ! End define mode
        !--------------------

        istatus = NF_ENDDEF(ncid)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


                                    ! Set 'col' coordinate variable
                                    ! equal to its index value
        DO n = 1, nsedcol_global
          i = n
          istatus = NF_PUT_VAR1_INT(ncid, id_col, n, i)
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        ENDDO

                                    ! Set 'lev' coordinate variable
                                    ! equal to its index value
        DO k = 1, idnb-idnw+1
          i = k - 1 + idnw
          istatus = NF_PUT_VAR1_INT(ncid, id_lev, k, i)
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        ENDDO

                                    ! Set 'ix' coordinate variable
                                    ! equal to its index value
        DO k = 1, nix_ncfile
          i = k
          istatus = NF_PUT_VAR1_INT(ncid, id_ix, k, i)
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        ENDDO

                                    ! Set 'jy' coordinate variable
                                    ! equal to its index value
        DO k = 1, njy_ncfile
          j = k
          istatus = NF_PUT_VAR1_INT(ncid, id_jy, k, j)
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        ENDDO


                                    ! xzdn(lev)
        CALL GRID_DEF(xzdn)
        istatus = NF_PUT_VAR_DOUBLE(ncid, id_xzdn, xzdn)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


        istatus = NF_PUT_VAR_INT(ncid, id_col4ij,
     &                              icolumn_seafloor_file(:,:))
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_VAR_INT(ncid, id_i4col,
     &                              ix_seafloor_file(:))
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_VAR_INT(ncid, id_j4col,
     &                              jy_seafloor_file(:))
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


                                   ! Surface areas of grid_elements
                                   ! and xref4col, yref4jcol

        DO n = 1, nsedcol_global

          i = ix_seafloor_file(n)
          j = jy_seafloor_file(n)

          s = DBLE(rarr_tarea(i,j)) * 1.0D-04       ! cm^2 --> m^2
          istatus = NF_PUT_VAR1_DOUBLE(ncid, id_area, n, s)
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

          x = DBLE(rarr_tlong(i,j))
          istatus = NF_PUT_VAR1_DOUBLE(ncid, id_xref4col, n, x)
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

          y = DBLE(rarr_tlat(i,j))
          istatus = NF_PUT_VAR1_DOUBLE(ncid, id_yref4col, n, y)
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        ENDDO


        istatus = NF_CLOSE(ncid)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


        DEALLOCATE(rarr_tlong)
        DEALLOCATE(rarr_tlat)
        DEALLOCATE(rarr_tarea)


      ENDIF



      CALL SNCA_RESET()


      RETURN



      CONTAINS



!-----------------------------------------------------------------------
      SUBROUTINE SNCA_SETUP()
!-----------------------------------------------------------------------

      USE mod_seafloor_central,     ONLY: N_COLUMNS_USED

      USE mod_execontrol_medusa,    ONLY: MEDEXE_NPROC,
     &                                    MEDEXE_MPI_COMM,
     &                                    MEDEXE_MPI_COMM_RANK,
     &                                    MEDEXE_MPI_GETPARTITIONING,
     &                                    lp_exeproc_singleproc_nc

      USE mod_coupsim_medusa_setup, ONLY: GET_NLONSLATSZTS_FILE,
     &                                    GET_ICOLUMN_SEAFLOOR_FILE,
     &                                    GET_IXJY_SEAFLOOR_FILE


      IMPLICIT NONE



                                    ! Get the number of columns in the
                                    ! mod_seafloor_central of the
                                    ! current process
      CALL N_COLUMNS_USED(nsedcol_central)

                                    ! If nsedcol_central == -1, the
                                    ! basic setup has not yet been
                                    ! done, and it does not make any
                                    ! sense to proceed
      IF (nsedcol_central < 0) THEN
        WRITE(jp_stderr,*) '[STORE_NC_AUX/SNCA_SETUP] error: ' //
     &    'Sea-floor setup not yet done -- aborting!'
        CALL ABORT_MEDUSA()
      ENDIF


      n_cprocs = MEDEXE_NPROC()     ! Get number of processes

                                    ! Get communicator of Medusa
      i_mycomm = MEDEXE_MPI_COMM()

                                    ! Get rank of process executing this instance
      i_myrank = MEDEXE_MPI_COMM_RANK()

                                    ! Now set up the file writing
                                    ! distribution logic:
      l_file_is_mine = .TRUE.       ! - by default, each process writes
                                    !   its own files

      IF (n_cprocs > 1) THEN        ! When running in a multi-processor 
                                    ! environment with more than one process,
                                    ! then this may need to be adapted:
                                    ! - for the NC_AUX file, we require
                                    !   single-processor I/O, by the root process
                                    !   as there is information only available
                                    !   to that process

        IF (i_myrank /= jp_exeproc_root) THEN       ! - and deactivate writing for other
          l_file_is_mine = .FALSE.                  !   processes than the writing one
        ENDIF                                       !   (here exceptionally, the root process)

      ENDIF
                                    ! Get the number of columns in all
                                    ! the mod_seafloor_central's of all
                                    ! concurrent processes and the
                                    ! partitioning information
      ALLOCATE(nsedcol_pproc(0:n_cprocs-1))
      ALLOCATE(ioffset_sedcol_pproc(0:n_cprocs-1))

      CALL MEDEXE_MPI_GETPARTITIONING(nsedcol_global, iproc_1stocn,
     &                              nsedcol_pproc, ioffset_sedcol_pproc)

                                    ! If single-processor NetCDF I/O, and
                                    ! execution environment includes more
                                    ! than one processor, then allocate
                                    ! work-space memory to receive the data
                                    ! by the writing process:
      nsedcol_ncfile = nsedcol_global


#ifdef DEBUG
      WRITE(jp_stddbg, '(A,I0)')
     &   '[SNCA_SETUP] n_cprocs = ', n_cprocs
      WRITE(jp_stddbg, '(A,I0)')
     &   '[SNCA_SETUP] i_myrank = ', i_myrank
      WRITE(jp_stddbg, '(A,I0)')
     &   '[SNCA_SETUP] nsedcol_central = ', nsedcol_central
      WRITE(jp_stddbg, '(A,I0)')
     &   '[SNCA_SETUP] nsedcol_global = ', nsedcol_global
      WRITE(jp_stddbg, '(A,I0)')
     &   '[SNCA_SETUP] nsedcol_ncfile = ', nsedcol_ncfile
      WRITE(jp_stddbg, '(A,I0," * ",I0)')
     &   '[SNCA_SETUP] nix_ncfile * njy_ncfile = ',
     &   nix_ncfile, njy_ncfile
      WRITE(jp_stddbg, '(A,L1)')
     &   '[SNCA_SETUP] l_file_is_mine = ', l_file_is_mine
#endif


      IF (i_myrank == jp_exeproc_root) THEN

        CALL GET_NLONSLATSZTS_FILE(nix_ncfile, njy_ncfile)

        ALLOCATE(ix_seafloor_file(nsedcol_global))
        ALLOCATE(jy_seafloor_file(nsedcol_global))
        ALLOCATE(icolumn_seafloor_file(nix_ncfile, njy_ncfile))

        CALL GET_IXJY_SEAFLOOR_FILE(ix_seafloor_file, jy_seafloor_file)
        CALL GET_ICOLUMN_SEAFLOOR_FILE(icolumn_seafloor_file)

      ELSE

        nix_ncfile = 0
        njy_ncfile = 0
        
        ALLOCATE(ix_seafloor_file(0))
        ALLOCATE(jy_seafloor_file(0))
        ALLOCATE(icolumn_seafloor_file(0, 0))

      ENDIF



      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE SNCA_SETUP
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE SNCA_RESET()
!-----------------------------------------------------------------------


      IMPLICIT NONE


      nsedcol_central = -1
      nsedcol_ncfile  = -1

      nix_ncfile = -1
      njy_ncfile = -1

      DEALLOCATE(nsedcol_pproc)
      DEALLOCATE(ioffset_sedcol_pproc)

      nsedcol_global = -1
      iproc_1stocn   = -1

      n_cprocs = -1
      i_myrank = MPI_PROC_NULL


      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE SNCA_RESET
!-----------------------------------------------------------------------


!=======================================================================
      END SUBROUTINE STORE_NC_AUX
!=======================================================================
