!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#ifdef CFN_THISFILE
#undef CFN_THISFILE
#endif
#define CFN_THISFILE "medusa_coupsim.F"
#ifndef __LINE__
#define __LINE__ 0
#endif
!---+----1----+----2----+----3----+----4----+----5----+----6----+----7--
!=======================================================================
      PROGRAM MEDUSA
!=======================================================================

      USE mod_defines_medusa
      USE mod_execontrol_medusa,    ONLY: ABORT_MEDUSA

      USE mpi
      USE mod_execontrol_medusa,    ONLY: MEDEXE_RANKFILENAME,
     &                                    MEDEXE_MPI_INIT, MEDEXE_NPROC,
     &                                    MEDEXE_MPI_COMM,
     &                                    MEDEXE_MPI_COMM_RANK,
     &                                    MEDEXE_MPI_FINALIZE

      USE mod_basicdata_medusa
      USE mod_logunits
      USE mod_gridparam
      USE mod_indexparam
      USE mod_milieucharas
      USE mod_materialcharas
      USE mod_chemicalconsts
      USE mod_transport
      USE mod_timecont_medusa_tsi,  ONLY: GET_TIMESTEP_INFO
      USE mod_store_ncfiles

      USE mod_equilibdata
      USE mod_equilibcontrol

      USE mod_processdata
      USE mod_processcontrol

      USE mod_seafloor_central
      USE mod_seafloor_init
      USE mod_sedcore
      USE mod_files_medusa

      USE mod_coupsim_medusa_setup, ONLY: SETUP_MEDUSA_FOR_COUPSIM

      USE mod_coupsim_o2s,          ONLY: SETUP_MOD_COUPSIM_O2S,
     &                                    CLEAR_O2S_DATASET,
     &                                    OCEAN_TO_SEDIMENT

      USE mod_coupsim_s2o,          ONLY: SETUP_MOD_COUPSIM_S2O,
     &                                    CLEAR_S2O_DATASET,
     &                                    SEDIMENT_TO_OCEAN

      USE mod_coupsim_subr,         ONLY: COUPSIM_BIOGEOCHEM_STEP

      USE mod_zone_mass_totals

      USE mod_medinterfaces

      IMPLICIT NONE

!     SI units generally used in this program:
!     kg     for solids mass units
!     moles  for solutes mass units
!     m      for space dimensions
!     yr     for time units (!!exception!!)
!     mol/m3 for solute concentrations
!     kg/m3  for solid concentrations


!     In general, variable names matching ...
!     ... "w*" relate to the top grid node of solid sediment+DBL (i=idnw);
!     ... "t*" relate to the top grid node of the solid sediment (i=idnt);
!              (N.B.: if no DBL is considered, idnt=idnw)
!     ... "a*" relate to any grid node (i=idnt, ..., idnb);
!     ... "b*" relate to the bottom grid node (i=idnb);
!     ... "x*" relate to values at grid nodes in the
!         resolved sediment part;
!     ... "y*" relate to the buffering layer between the
!         resolved sediment layers and the historical part below;
!     ... "z*" relate to the historical layers below the bottom
!         of the represented sediment;
!     ... "dx*" relate to derivatives of values at grid nodes in the
!         resolved sediment layers;
!     ... "vx*" relate to variations of values in layers, i.e.
!         between grid nodes in the resolved sediment layers;
!     ... "*conc" represent a concentration;
!     ... "?fflx" represent a solid FLuX;
!     ... "?cflx" represent a solute FLuX



!     xc(i,j): concentration profiles (i=idnw,idnb)
!            for all components (j=1,ncompo)
!     ysolid(nsolid): concentration profiles of solids in the
!            buffering joint layer
!     apsv(j): partial specific (massic) volumes for solids
!            (j=1,nsolid) <<from mod_materialparams>>
!
!
!
!     xzdn(i): depths of the grid nodes (i=idnw, idnb)
!     vxzdn(i): thickness of the layers, i.e.,
!            variation of xzdn between grid nodes,
!            where vxzdn(i)=xzdn(i)-xzdn(i-1); i=idit, idib
!                             (or equivalently i=idnt+1, idnb)





! xw(i): solid burial rate at grid nodes (i=idnt, idnb);
!                actually equal to w(*)*(1-phi(*)), as this
!                product is most often used
! xu(i): porewater velocity at grid nodes (i=idnw, idnb)
!                (actually equal to u(*)*phi(*), as this
!                product is most often used


! At the top of the sediment + possible DBL ('w' interface):
! wdbsl: Depth Below Sea-Level of the sediment pile
! wtmpdc: TeMPerature in Degrees Centigrade of the sed't pile
! wsalin: SALINity of porewater

! atime: current time
! datime: time step to be applied

! atime and datime are declared anyway, even for steady-state
! simulations - they will simply not get used. Else the
! subroutine declarations would get too complicated.

      CHARACTER(LEN=*), PARAMETER ::
     &  cfn_thisfile     = CFN_THISFILE
      CHARACTER(LEN=*), PARAMETER ::
     &  cfmt_info_a      = '("['//cfn_thisfile//']: ", A)' ,
     &  cfmt_info_ai     = '("['//cfn_thisfile//']: ", A, I0)',
     &  cfmt_infolin_ia  = '("['//cfn_thisfile//':", I0, "]: ", A)',
     &  cfmt_infolin_iai = '("['//cfn_thisfile//':", I0, "]: ", A, I0)'

      DOUBLE PRECISION :: atime, datime, duree
      INTEGER :: nsteps, nst, nstepsini

      INTEGER :: iflag, n_trouble, n_columns, i_request, istep
      INTEGER :: i_column
      DOUBLE PRECISION :: atime0


      ! total_xm_ini/0/1:  total masses (1:ncompo, 1:n_grid_seafloor)
      DOUBLE PRECISION, ALLOCATABLE, DIMENSION(:,:)
     &    :: total_xm_ini, total_xm_0, total_xm_1

      DOUBLE PRECISION, DIMENSION(nsolut) :: wcflx
      DOUBLE PRECISION, DIMENSION(nsolid) :: wfflx, bfflx

      ! t_0 to t_1
      DOUBLE PRECISION, ALLOCATABLE, DIMENSION(:,:)
     &    :: topflx_xm_01, botflx_xm_01, total_rea_xm_01
      ! initial to end
      DOUBLE PRECISION, ALLOCATABLE, DIMENSION(:,:)
     &    :: topflx_xm_ie, botflx_xm_ie, total_rea_xm_ie


! Further variables provided by modules

!   mod_milieuparams:
!
!     xphi(:): sediment porosity at grid nodes (i=idnt, idnb)
!     yphi: sediment porosity in the buffering joint layer (constant)
!     zphi: sediment porosity in the historical layers (constant)
!     dxphi(:): dxphi(i) = (d xphi/d z)|z=xzdn(i),
!            actual value of the derivative if xphi=fct(xzdn)
!            analytically known
!     xtor2(i): sediment tortuosity^2 ($\theta^2$) at
!            grid nodes (i=idnt, idnb)
!     dxtor2(i): dxtor2(i) = (d xtor2/d z)|z=xzdn(i),
!            actual value of the derivative if xtor2=fct(xzdn)
!            analytically known
!     ytor2: sed_t tortuosity^2 in the buffering joint layer (constant)
!     ztor2: sed_t tortuosity^2 in the historical layers (constant)
!
!-----------------------------------------------------------------------

      CHARACTER (LEN=jp_lmaxpathname) :: cfn_medusa_err = "medusa.err"
      CHARACTER (LEN=jp_lmaxpathname) :: cfn_medusa_log = "medusa.log"
#ifdef DEBUG
      CHARACTER (LEN=jp_lmaxpathname) :: cfn_medusa_dbg = "medusa.dbg"
#endif

      INTEGER :: istatus

![MPI stuff]
      INTEGER :: n_cprocs, i_mycomm, i_myrank
      INTEGER :: nlen


                                    ! Initialize the MPI environment
      CALL MEDEXE_MPI_INIT

                                    ! Get the basic MPI characteristics
      n_cprocs = MEDEXE_NPROC()
      i_mycomm = MEDEXE_MPI_COMM()
      i_myrank = MEDEXE_MPI_COMM_RANK()


                                    ! Individualize files names
                                    ! (insert process rank) if required
                                    ! and open them
      CALL MEDEXE_RANKFILENAME(cfn_medusa_err)
      OPEN(jp_stderr, FILE=cfn_medusa_err)
      CALL REPORT_COMPILE_OPTIONS(jp_stderr)
      CALL REPORT_EXTRA_COMPILE_OPTIONS(jp_stderr)

      CALL MEDEXE_RANKFILENAME(cfn_medusa_log)
      OPEN(jp_stdlog, FILE=cfn_medusa_log)

#ifdef DEBUG
      CALL MEDEXE_RANKFILENAME(cfn_medusa_dbg)
      IF (jp_stddbg /= jp_stderr) OPEN(jp_stddbg, FILE=cfn_medusa_dbg)
      WRITE(jp_stddbg, cfmt_infolin_ia) (__LINE__),
     &                              'Starting (MPI already initialized)'
      WRITE(jp_stddbg, '()')
      CALL REPORT_COMPILE_OPTIONS(jp_stddbg)
      CALL REPORT_EXTRA_COMPILE_OPTIONS(jp_stddbg)
      CALL DEBUG_MEDUSA_MAIN_INDICES
#endif


                                    ! Get time stepping information
      i_request = 0                 !  - info from scratch, please
      CALL GET_TIMESTEP_INFO(i_request, iflag,
     &                              atime0, datime, nsteps)

#ifdef DEBUG
      WRITE(jp_stddbg,'()')
      WRITE(jp_stddbg,*) 'Time control data from "medusa.tsi":'
      WRITE(jp_stddbg,*) ' - atime0 ', atime0
      WRITE(jp_stddbg,*) ' - datime ', datime
      WRITE(jp_stddbg,*) ' - nsteps ', nsteps
#endif 


      CALL INIT_FILELIST_MEDUSA()

                                    ! From mod_coupsim_medusa_setup.F
      CALL SETUP_MEDUSA_FOR_COUPSIM(n_columns = n_columns)


                                    ! Must be done here as
                                    ! this also calls CORELAY_SETUP
      CALL SETUP_SEDCORE_SYSTEM(cfn_ncin_sedcore, cfn_ncout_sedcore)


      CALL SETUP_TRANSPORT          ! From mod_transport.F


      CALL InitEquilibParameters    ! From mod_equilibcontrol.F


      CALL InitProcessParameters    ! From mod_processcontrol.F


                                    ! Set up the communication modules
      CALL SETUP_MOD_COUPSIM_O2S()  !  - MOD_COUPSIM_O2S
      CALL SETUP_MOD_COUPSIM_S2O()  !  - MOD_COUPSIM_S2O [simulation only here]

      CALL CLEAR_S2O_DATASET        ! Clear the dataset in MOD_COUPSIM_S2O 
                                    ! so that the host model can use
                                    ! sensible values for the
                                    ! sediment-to-ocean fluxes (set to zero
                                    ! to start with, possibly overridden below)
                                    ! [simulation only here].


                                    ! Initialize the sediment state
      IF (cfn_ncin_init /= "/dev/null") THEN
                                    ! From MOD_SEAFLOOR_INIT
        CALL InitSeafloorFromNetCDFFiles(cfn_reaclay = cfn_ncin_init,
     &                                   cfn_flx     = cfn_ncin_flx)

                                    ! Override zero sediment-to-ocean
                                    ! fluxes if flux data have been
                                    ! read into SEAFLOOR_CENTRAL by
                                    ! InitSeafloorFromNetCDFFiles
        IF (cfn_ncin_flx /= "/dev/null") CALL SEDIMENT_TO_OCEAN(iflag)

      ELSEIF (cfn_nmlin_init /= "/dev/null") THEN
                                    ! From MOD_SEAFLOOR_INIT
        CALL InitSeafloorFromNamelistFile(cfn_nmlin_init)

      ELSE

        WRITE(jp_stderr,'()')
        WRITE(jp_stderr,
     &    '("Error: no valid initialisation file given:")')
        WRITE(jp_stderr,'(" - cfn_nmlin_init = """, A, """")')
     &                              TRIM(cfn_nmlin_init)
        WRITE(jp_stderr,'(" - cfn_ncin_init = """, A, """")')
     &                              TRIM(cfn_ncin_init)
        WRITE(jp_stderr,'("Aborting!")')
        CALL ABORT_MEDUSA()

      ENDIF

      ALLOCATE(total_xm_ini(ncompo,n_columns))
      ALLOCATE(total_xm_0(ncompo,n_columns))
      ALLOCATE(total_xm_1(ncompo,n_columns))
      ALLOCATE(topflx_xm_01(ncompo,n_columns))
      ALLOCATE(botflx_xm_01(ncompo,n_columns))
      ALLOCATE(total_rea_xm_01(ncompo,n_columns))
      ALLOCATE(topflx_xm_ie(ncompo,n_columns))
      ALLOCATE(botflx_xm_ie(ncompo,n_columns))
      ALLOCATE(total_rea_xm_ie(ncompo,n_columns))

      topflx_xm_ie(:,:)    = 0.0D+00
      botflx_xm_ie(:,:)    = 0.0D+00
      total_rea_xm_ie(:,:) = 0.0D+00


      duree = nsteps*datime

      atime = atime0

      DO istep = 1, nsteps

        CALL CLEAR_O2S_DATASET      ! Not necessary in this application,
                                    ! but recommended in general.

        CALL COUPSIM_BIOGEOCHEM_STEP


        IF (istep == 1) THEN
          ! Can only be done at this point as several variable
          ! values are only known now

#ifdef NCFILES_SAVEALLSTEPS
          CALL OPEN_NCFILES_MEDUSA(atime)
#else
          CALL OPEN_NCFILES_MEDUSA()
#endif

          CALL ZONE_MASS_TOTALS(total_xm=total_xm_ini)
          total_xm_0(:,:) = total_xm_ini(:,:)

        ELSE

          total_xm_0(:,:) = total_xm_1(:,:)

        ENDIF

        CALL OCEAN_TO_SEDIMENT(iflag)

        SELECT CASE(iflag)
        CASE(0)
          CONTINUE
        CASE(2)
          WRITE(jp_stderr, cfmt_infolin_ia) (__LINE__),
     &      'Error after SPECIA_CB call in OCEAN_TO_SEDIMENT. '
     &      // 'Aborting!'
          CALL ABORT_MEDUSA()
        CASE DEFAULT
          WRITE(jp_stderr, cfmt_infolin_ia) (__LINE__),
     &      'Unknown error in OCEAN_TO_SEDIMENT. Aborting!'
          CALL ABORT_MEDUSA()
        END SELECT


        CALL SOLVSED_ONESTEP(atime, datime, n_columns,
     &                       iflag, n_trouble)

        atime = atime0 + istep*datime


        CALL ZONE_MASS_TOTALS(total_xm=total_xm_1, 
     &                        total_rea_xm=total_rea_xm_01)
        total_rea_xm_01(:,:) = total_rea_xm_01(:,:)*datime

        DO i_column = 1, n_columns
          CALL GET_BOUNDARY_CONDS(i_column=i_column, gbcflag=iflag,
     &          wfflx=wfflx)
          topflx_xm_01(jf_to_io(:),i_column) = wfflx(:)*datime
          
          CALL GET_BOUNDARY_FLUXES(i_column=i_column, gbfflag=iflag,
     &          wcflx=wcflx, bfflx=bfflx)
          topflx_xm_01(jc_to_io(:),i_column) = wcflx(:)*datime
          botflx_xm_01(jf_to_io(:),i_column) = bfflx(:)*datime
          botflx_xm_01(jc_to_io(:),i_column) = 0.0D+00
        ENDDO

        topflx_xm_ie(:,:)    = topflx_xm_ie(:,:) + topflx_xm_01(:,:)
        botflx_xm_ie(:,:)    = botflx_xm_ie(:,:) + botflx_xm_01(:,:)
        total_rea_xm_ie(:,:) = total_rea_xm_ie(:,:)
     &                         + total_rea_xm_01(:,:)


        WRITE(jp_stdlog,*)
        WRITE(jp_stdlog,*)
        WRITE(jp_stdlog,*) '********************************'
        WRITE(jp_stdlog,*) ' Step ', istep,'/',nsteps
        WRITE(jp_stdlog,*) '********************************'
        WRITE(jp_stdlog,*)
        WRITE(jp_stdlog,*) ' --- Mass Balances for this step '
        CALL MASS_BALANCES(total_xm_0, total_xm_1,
     &                     topflx_xm_01, botflx_xm_01,
     &                     total_rea_xm_01, n_columns)
        WRITE(jp_stdlog,*)
        WRITE(jp_stdlog,*)
        WRITE(jp_stdlog,*) ' --- Mass Balances integrated since start'
        CALL MASS_BALANCES(total_xm_ini, total_xm_1,
     &                     topflx_xm_ie, botflx_xm_ie,
     &                     total_rea_xm_ie,
     &                     n_columns)
        WRITE(jp_stdlog,*)


        CALL REACLAY_X_CORELAY(atime)


#ifdef NCFILES_SAVEALLSTEPS
        CALL WRITERES_NCFILES_MEDUSA(atime)
#else
        IF (istep == nsteps) THEN
          CALL WRITERES_NCFILES_MEDUSA(atime)
        ENDIF
#endif

                                    ! Prepare next step [fake simulation here]
        CALL SEDIMENT_TO_OCEAN(iflag)

      ENDDO


      CALL CLOSE_RESFILES_MEDUSA()


#ifdef DEBUG
      WRITE(jp_stddbg,cfmt_infolin_ia) (__LINE__), 'End.'
      IF (jp_stddbg /= jp_stderr) CLOSE(jp_stddbg)
#endif
      CLOSE(jp_stderr)
      CLOSE(jp_stdlog)

      CALL MPI_FINALIZE(istatus)


!=======================================================================

      CONTAINS

!-----------------------------------------------------------------------
      SUBROUTINE REPORT_EXTRA_COMPILE_OPTIONS(iunit)
!-----------------------------------------------------------------------

      IMPLICIT NONE

      INTEGER, INTENT(IN) :: iunit


#ifdef NCFILES_SAVEALLSTEPS
      WRITE(iunit, '(" -DNCFILES_SAVEALLSTEPS")')
#endif

      WRITE(iunit, '()')


      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE REPORT_EXTRA_COMPILE_OPTIONS
!-----------------------------------------------------------------------


!=======================================================================
      END
!=======================================================================
