!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#ifdef CFN_THISFILE
#undef CFN_THISFILE
#endif
#define CFN_THISFILE "store_nc_aux.F"
#ifndef __LINE__
#define __LINE__ 0
#endif
!=======================================================================
      SUBROUTINE STORE_NC_AUX(cfn_ncin_bec, cfn_ncout_aux, title_string)
!=======================================================================


      USE mod_defines_medusa
      USE mod_execontrol_medusa,    ONLY: ABORT_MEDUSA

      USE mod_gridparam,            ONLY: idnw, idnb, ndn, GRID_DEF

      USE mod_seafloor_central,     ONLY: N_COLUMNS_USED, COLUMN_AREA4N
#ifdef MEDUSA_BASE2D
      USE mod_seafloor_central,     ONLY: IJ_COLUMNS_USED,
     &                                    COLUMN_IJ2N, COLUMN_N2IJ,
     &                                    COLUMN_N2XY
#endif

      USE mod_netcdfinc

      USE mod_netcdfparam


      IMPLICIT NONE


      CHARACTER(LEN=*), INTENT(IN)           :: cfn_ncin_bec
      CHARACTER(LEN=*), INTENT(IN)           :: cfn_ncout_aux
      CHARACTER(LEN=*), INTENT(IN), OPTIONAL :: title_string


      CHARACTER(LEN=*), PARAMETER :: cfn_thisfile =
     &  CFN_THISFILE

      INTEGER :: ncid

      INTEGER :: ncid_file, ncid_var

      INTEGER :: nsedcol_central = -1
      INTEGER :: nsedcol_ncfile  = -1
      INTEGER :: nix_ncfile      = -1
      INTEGER :: njy_ncfile      = -1

      LOGICAL :: l_file_is_mine = .TRUE.


      INTEGER :: k, n
      INTEGER :: iflag, istatus

#ifdef MEDUSA_BASE2D
      INTEGER :: nix, njy, nijxy
      INTEGER, DIMENSION(2) :: dim, istart
      INTEGER :: i, j
      INTEGER :: i_ncfile, j_ncfile
      INTEGER :: ip, jp
      DOUBLE PRECISION :: x, y
#endif

      DOUBLE PRECISION, DIMENSION(idnw:idnb) :: xzdn


! Dimensions and dimension variables
!  'lev' (levels)
      INTEGER            :: dim_lev         ! dimension ID
      INTEGER            ::  id_lev         ! ID of dimension variable

! 'col' (columns)
      INTEGER            :: dim_col         ! dimension ID
      INTEGER            ::  id_col         ! ID of dimension variable

#ifdef MEDUSA_BASE2D
                              ! ... Please adapt the names below
      CHARACTER(LEN=*), PARAMETER   :: c_name_ix     = 'ix'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_ix    = 'X_Index'
      INTEGER, PARAMETER :: nlen_lname_ix = LEN(c_lname_ix)
      INTEGER            :: dim_ix
      INTEGER            ::  id_ix

                              ! ... Please adapt the two names below
      CHARACTER(LEN=*), PARAMETER   :: c_name_jy     = 'jy'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_jy    = 'Y_Index'
      INTEGER, PARAMETER :: nlen_lname_jy = LEN(c_lname_jy)
      INTEGER            :: dim_jy
      INTEGER            ::  id_jy
#endif


! Data variables
!  - general
      INTEGER ::  id_xzdn           ! depth coordinates
      INTEGER ::  id_area           ! surface areas of columns (n)

      DOUBLE PRECISION  :: sfc_area

      INTEGER,          DIMENSION(:), ALLOCATABLE ::  iarr_c
      DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE ::  darr_c

#ifdef MEDUSA_BASE2D
!  - 2D specific
      INTEGER ::  id_col4ij         ! column index n for grid element (i,j)

                                    ! index i(n)
      CHARACTER(LEN=*), PARAMETER   :: c_name_i4col     = 'i4col'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_i4col    = c_lname_ix //
     &                                              '_from_Column_Index'
      INTEGER, PARAMETER :: nlen_lname_i4col = LEN(c_lname_i4col)
      INTEGER ::  id_i4col

                                    ! index j(n)
      CHARACTER(LEN=*), PARAMETER   :: c_name_j4col     = 'j4col'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_j4col    = c_lname_jy //
     &                                              '_from_Column_Index'
      INTEGER, PARAMETER :: nlen_lname_j4col = LEN(c_lname_j4col)
      INTEGER ::  id_j4col

                                    ! xref value of grid element (n)
      CHARACTER(LEN=*), PARAMETER   :: c_name_xref4col  = 'xref4col'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_xref4col = 'Longitude'
      CHARACTER(LEN=*), PARAMETER   :: c_unit_xref4col  = un_degE
      INTEGER, PARAMETER :: nlen_lname_xref4col = LEN(c_lname_xref4col)
      INTEGER, PARAMETER :: nlen_unit_xref4col  = LEN(c_unit_xref4col)
      INTEGER ::  id_xref4col

                                    ! yref value of grid element (n)
      CHARACTER(LEN=*), PARAMETER   :: c_name_yref4col  = 'yref4col'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_yref4col = 'Latitude'
      CHARACTER(LEN=*), PARAMETER   :: c_unit_yref4col  = un_degN
      INTEGER, PARAMETER :: nlen_lname_yref4col = LEN(c_lname_yref4col)
      INTEGER, PARAMETER :: nlen_unit_yref4col  = LEN(c_unit_yref4col)
      INTEGER ::  id_yref4col

      INTEGER, DIMENSION(:),   ALLOCATABLE ::  iarri_c
      INTEGER, DIMENSION(:),   ALLOCATABLE ::  iarrj_c
      INTEGER, DIMENSION(:,:), ALLOCATABLE ::  iarr_ij

      DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE ::  darrx_c
      DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE ::  darry_c


                                    ! Longitude value of grid element (ix,jy)
      CHARACTER(LEN=*), PARAMETER   :: c_name_lon4ij  = 'lon4ij'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_lon4ij =
     &                              'Array of grid longitudes'
      CHARACTER(LEN=*), PARAMETER   :: c_unit_lon4ij  = un_degE
      INTEGER, PARAMETER :: nlen_lname_lon4ij = LEN(c_lname_lon4ij)
      INTEGER, PARAMETER :: nlen_unit_lon4ij  = LEN(c_unit_lon4ij)
      INTEGER ::  id_lon4ij

                                    ! Latitude value of grid element (ix,jy)
      CHARACTER(LEN=*), PARAMETER   :: c_name_lat4ij  = 'lat4ij'
      CHARACTER(LEN=*), PARAMETER   :: c_lname_lat4ij =
     &                              'Array of grid latitudes'
      CHARACTER(LEN=*), PARAMETER   :: c_unit_lat4ij  = un_degN
      INTEGER, PARAMETER :: nlen_lname_lat4ij = LEN(c_lname_lat4ij)
      INTEGER, PARAMETER :: nlen_unit_lat4ij  = LEN(c_unit_lat4ij)
      INTEGER ::  id_lat4ij

      REAL, DIMENSION(:,:), ALLOCATABLE ::  rarr_tlong
      REAL, DIMENSION(:,:), ALLOCATABLE ::  rarr_tlat

      DOUBLE PRECISION, DIMENSION(:,:), ALLOCATABLE ::  darr_lon4ij
      DOUBLE PRECISION, DIMENSION(:,:), ALLOCATABLE ::  darr_lat4ij

#endif



! Subroutine start
! ================

      CALL SNCA_SETUP()


                                    ! Open file <cfn_ncin_bec>
                                    ! to retrieve the complete longitude
                                    ! and latitude arrays
      istatus = NF_OPEN(cfn_ncin_bec, NF_NOWRITE, ncid_file)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

      ALLOCATE(rarr_tlong(nix_ncfile, njy_ncfile))
      ALLOCATE(rarr_tlat(nix_ncfile, njy_ncfile))

                                    ! Read in longitude array
                                    ! (type: FLOAT; units: degrees_east)
      istatus = NF_INQ_VARID(ncid_file, 'TLONG', ncid_var)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
      istatus = NF_GET_VAR_REAL(ncid_file, ncid_var, rarr_tlong)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

                                    ! Read in latitude array
                                    ! (type: FLOAT; units: degrees_north)
      istatus = NF_INQ_VARID(ncid_file, 'TLAT', ncid_var)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
      istatus = NF_GET_VAR_REAL(ncid_file, ncid_var, rarr_tlat)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

                                    ! Close the file, since all that
                                    ! can be used has been read in now.
      istatus = NF_CLOSE(ncid_file)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

      ALLOCATE(darr_lon4ij(nix_ncfile, njy_ncfile))
      ALLOCATE(darr_lat4ij(nix_ncfile, njy_ncfile))

      darr_lon4ij(:,:) = DBLE(rarr_tlong(:,:))
      darr_lat4ij(:,:) = DBLE(rarr_tlat(:,:))


      ALLOCATE(iarr_c(nsedcol_central))
      ALLOCATE(darr_c(nsedcol_central))


      ALLOCATE(iarr_ij(nix_ncfile, njy_ncfile))

      ALLOCATE(iarri_c(nsedcol_central))
      ALLOCATE(iarrj_c(nsedcol_central))

      ALLOCATE(darrx_c(nsedcol_central))
      ALLOCATE(darry_c(nsedcol_central))



      IF (l_file_is_mine) THEN

        !-----------------------
        ! Create the data file
        !-----------------------
        istatus = NF_CREATE(TRIM(cfn_ncout_aux), NF_CLOBBER, ncid)
        IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))


        !-------------------------------------------
        ! Define dimensions and dimension variables
        !-------------------------------------------

                                    ! Columns on the string
                                    ! ---------------------
                                    ! - dim id
        istatus = NF_DEF_DIM(ncid, 'col', nsedcol_ncfile, dim_col)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    ! - dim variable id
        istatus = NF_DEF_VAR(ncid, 'col', NF_INT, 1, dim_col, id_col)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_INT(ncid, id_col, 'offset', NF_INT, 1, 0)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


                                    ! Levels
                                    ! ------
                                    ! - dim id
        istatus = NF_DEF_DIM(ncid, 'lev', ndn, dim_lev)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    ! - dim variable id
        istatus = NF_DEF_VAR(ncid, 'lev', NF_INT, 1, dim_lev, id_lev)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    ! - dim variable attributes
        istatus = NF_PUT_ATT_TEXT(ncid, id_lev, 'positive', 4, 'down')
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


#ifdef MEDUSA_BASE2D
                                    ! 'X' index
                                    ! ---------
                                    ! - dim id
        istatus = NF_DEF_DIM(ncid, c_name_ix, nix_ncfile, dim_ix)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    ! - dim variable id
        istatus = NF_DEF_VAR(ncid, c_name_ix, NF_INT,
     &                              1, dim_ix, id_ix)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    ! - dim variable attributes
        istatus = NF_PUT_ATT_TEXT(ncid, id_ix, 'long_name',
     &                              nlen_lname_ix, c_lname_ix)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_INT(ncid, id_ix, 'offset', NF_INT, 1, 0)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


                                    ! 'Y' index
                                    ! ---------
                                    ! - dim id
        istatus = NF_DEF_DIM(ncid, c_name_jy, njy_ncfile, dim_jy)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    ! - dim variable id
        istatus = NF_DEF_VAR(ncid, c_name_jy, NF_INT,
     &                              1, dim_jy, id_jy)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    ! - dim variable attributes
        istatus = NF_PUT_ATT_TEXT(ncid, id_jy, 'long_name',
     &                         nlen_lname_jy, c_lname_jy)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_INT(ncid, id_jy, 'offset', NF_INT, 1, 0)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
#endif


        !-----------------------
        ! Define data variables
        !-----------------------

                                    ! xzdn(lev)
                                    ! ---------
        istatus = NF_DEF_VAR(ncid, vsn_xzdn, NF_DOUBLE,
     &                              1, dim_lev, id_xzdn)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        istatus = NF_PUT_ATT_TEXT(ncid, id_xzdn,
     &                              'long_name', vll_xzdn, vln_xzdn)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        istatus = NF_PUT_ATT_TEXT(ncid, id_xzdn,
     &                              'units', ul_m, un_m)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


                                    ! Area(col)
                                    ! ---------
        istatus = NF_DEF_VAR(ncid, 'area', NF_DOUBLE,
     &                              1, dim_col, id_area)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        istatus = NF_PUT_ATT_TEXT(ncid, id_area, 'long_name',
     &                              12, 'Surface_Area')
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        istatus = NF_PUT_ATT_TEXT(ncid, id_area, 'units', ul_m2, un_m2)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


#ifdef MEDUSA_BASE2D
                                    ! col4ij(ix,jy)
                                    ! -------------
        dim(1) = dim_ix
        dim(2) = dim_jy

        istatus = NF_DEF_VAR(ncid, 'col4ij', NF_INT,
     &                              2, dim(1:2), id_col4ij)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_TEXT(ncid, id_col4ij, 'long_name',
     &                              18, 'Column_index_of_ij')
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_INT(ncid, id_col4ij,
     &                              '_FillValue', NF_INT, 1, -1)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_INT(ncid, id_col4ij,
     &                              'valid_min', NF_INT, 1, 1)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_INT(ncid, id_col4ij,
     &                              'valid_max', NF_INT,
     &                              1, nsedcol_ncfile)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


                                    ! xref4col(col)
                                    ! -------------
        istatus = NF_DEF_VAR(ncid, c_name_xref4col, NF_DOUBLE,
     &                              1, dim_col, id_xref4col)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_TEXT(ncid, id_xref4col, 'long_name',
     &                              nlen_lname_xref4col,
     &                              c_lname_xref4col)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_TEXT(ncid, id_xref4col, 'units',
     &                              nlen_unit_xref4col,
     &                              c_unit_xref4col)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


                                    ! yref4col(col)
                                    ! -------------
        istatus = NF_DEF_VAR(ncid, c_name_yref4col, NF_DOUBLE,
     &                              1, dim_col, id_yref4col)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_TEXT(ncid, id_yref4col, 'long_name',
     &                              nlen_lname_yref4col,
     &                              c_lname_yref4col)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_TEXT(ncid, id_yref4col, 'units',
     &                              nlen_unit_yref4col,
     &                              c_unit_yref4col)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


                                    ! i4col(col)
                                    ! ----------
        istatus = NF_DEF_VAR(ncid, c_name_i4col, NF_INT,
     &                              1, dim_col, id_i4col)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_TEXT(ncid, id_i4col, 'long_name',
     &                              nlen_lname_i4col, c_lname_i4col)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_INT(ncid, id_i4col,
     &                              '_FillValue', NF_INT, 1, -1)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        istatus = NF_PUT_ATT_INT(ncid, id_i4col,
     &                              'valid_min',  NF_INT, 1, 1)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        istatus = NF_PUT_ATT_INT(ncid, id_i4col,
     &                              'valid_max',  NF_INT, 1, nix_ncfile)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


                                    ! j4col(col)
                                    ! ----------
        istatus = NF_DEF_VAR(ncid, c_name_j4col, NF_INT,
     &                              1, dim_col, id_j4col)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        istatus = NF_PUT_ATT_TEXT(ncid, id_j4col, 'long_name',
     &                              nlen_lname_j4col, c_lname_j4col)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_INT(ncid, id_j4col,
     &                              '_FillValue', NF_INT, 1, -1)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        istatus = NF_PUT_ATT_INT(ncid, id_j4col,
     &                              'valid_min',  NF_INT, 1, 1)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        istatus = NF_PUT_ATT_INT(ncid, id_j4col,
     &                              'valid_max',  NF_INT, 1, njy_ncfile)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


                                    ! lon4ij(ix,jy)
                                    ! -------------
        istatus = NF_DEF_VAR(ncid, c_name_lon4ij, NF_DOUBLE,
     &                              2, dim(1:2), id_lon4ij)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_TEXT(ncid, id_lon4ij, 'long_name',
     &                              nlen_lname_lon4ij, c_lname_lon4ij)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_TEXT(ncid, id_lon4ij, 'units',
     &                              nlen_unit_lon4ij,
     &                              c_unit_lon4ij)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


                                    ! lat4ij(ix,jy)
                                    ! -------------
        istatus = NF_DEF_VAR(ncid, c_name_lat4ij, NF_DOUBLE,
     &                              2, dim(1:2), id_lat4ij)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_TEXT(ncid, id_lat4ij, 'long_name',
     &                              nlen_lname_lat4ij, c_lname_lat4ij)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        istatus = NF_PUT_ATT_TEXT(ncid, id_lat4ij, 'units',
     &                              nlen_unit_lat4ij,
     &                              c_unit_lat4ij)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

#endif


        !----------------------
        ! Put global attributes
        !----------------------

        IF (PRESENT(title_string)) THEN
          istatus = NF_PUT_ATT_TEXT(ncid, NF_GLOBAL, 'title',
     &                          LEN_TRIM(title_string), title_string)
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        ENDIF


        !--------------------
        ! End define mode
        !--------------------

        istatus = NF_ENDDEF(ncid)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

      ENDIF

                                    ! Set 'col' coordinate variable
                                    ! equal to its index value
      iarr_c(:) = (/ (n, n=1, nsedcol_central) /)

      CALL SNCA_PUT_C_INT(ncid, id_col, iarr_c)


      IF (l_file_is_mine) THEN
                                    ! Set 'lev' coordinate variable
                                    ! equal to its index value
        DO k = 1, idnb-idnw+1
          i = k-1+idnw
          istatus = NF_PUT_VAR1_INT(ncid, id_lev, k, i)
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        ENDDO

      ENDIF



#ifdef MEDUSA_BASE2D
      IF (l_file_is_mine) THEN
                                    ! Set 'ix' coordinate variable
                                    ! equal to its index value
        DO k = 1, nix_ncfile

          i = k

          istatus = NF_PUT_VAR1_INT(ncid, id_ix, k, i)
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        ENDDO

                                    ! Set 'jy' coordinate variable
                                    ! equal to its index value
        DO k = 1, njy_ncfile

          j = k
          istatus = NF_PUT_VAR1_INT(ncid, id_jy, k, j)
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        ENDDO

      ENDIF
#endif


      IF (l_file_is_mine) THEN
                                    ! xzdn(lev)
        CALL GRID_DEF(xzdn)
        istatus = NF_PUT_VAR_DOUBLE(ncid, id_xzdn, xzdn)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

      ENDIF


#ifdef MEDUSA_BASE2D
                                    ! col4ij
      DO i = 1, nix_ncfile
      DO j = 1, njy_ncfile

        CALL COLUMN_IJ2N(i, j, iflag, k)

        IF (iflag <= 0) THEN        ! Ignore warning for "Not-at-seafloor"

          iarr_ij(i, j) = k

        ELSE

          WRITE(jp_stderr,*) 'Error upon COLUMN_IJ2N call:'
          WRITE(jp_stderr,*) ' called with i,j = ', i, j
          WRITE(jp_stderr,*) ' returned iflag = ', iflag
          WRITE(jp_stderr,*) 'Aborting!'
          CALL ABORT_MEDUSA()

        ENDIF

      ENDDO
      ENDDO

      CALL SNCA_PUT_IJ_INT(ncid, id_col4ij, iarr_ij)
#endif


#ifdef MEDUSA_BASE2D
                                    ! icol, jcol
      DO k = 1, nsedcol_central

        CALL COLUMN_N2IJ(k, iflag, i, j)

        IF (iflag == 0) THEN

          iarri_c(k) = i
          iarrj_c(k) = j

        ELSE

          WRITE(jp_stderr,*) 'Error upon COLUMN_N2IJ call:'
          WRITE(jp_stderr,*) ' called with k = ', k
          WRITE(jp_stderr,*) ' returned iflag = ', iflag
          WRITE(jp_stderr,*) 'Aborting!'
          CALL ABORT_MEDUSA()

        ENDIF

      ENDDO

      CALL SNCA_PUT_C_INT(ncid, id_i4col, iarri_c)
      CALL SNCA_PUT_C_INT(ncid, id_j4col, iarrj_c)

#endif


#ifdef MEDUSA_BASE2D
                                    ! xref4col, yref4jcol
      DO k = 1, nsedcol_central

        CALL COLUMN_N2XY(k, iflag, x, y)

        IF (iflag == 0) THEN
          darrx_c(k) = x
          darry_c(k) = y
        ELSE
          WRITE(jp_stderr,*) 'Error upon COLUMN_N2XY call:'
          WRITE(jp_stderr,*) ' called with k = ', k
          WRITE(jp_stderr,*) ' returned iflag = ', iflag
          WRITE(jp_stderr,*) 'Aborting!'
          CALL ABORT_MEDUSA()
        ENDIF

      ENDDO

      CALL SNCA_PUT_C_DOUBLE(ncid, id_xref4col, darrx_c)
      CALL SNCA_PUT_C_DOUBLE(ncid, id_yref4col, darry_c)
#endif


#ifdef MEDUSA_BASE2D
                                    ! lon4ij, lat4ij
      CALL SNCA_PUT_IJ_DOUBLE(ncid, id_lon4ij, darr_lon4ij)
      CALL SNCA_PUT_IJ_DOUBLE(ncid, id_lat4ij, darr_lat4ij)
#endif

                                    ! Surface areas of grid_elements
      DO k = 1, nsedcol_central

        CALL COLUMN_AREA4N(k, iflag, sfc_area)

        IF (iflag == 0) THEN
          darr_c(k) = sfc_area
        ELSE
          WRITE(jp_stderr,*) 'Error upon COLUMN_AREA4N call:'
          WRITE(jp_stderr,*) ' called with k = ', k
          WRITE(jp_stderr,*) ' returned iflag = ', iflag
          WRITE(jp_stderr,*) 'Aborting!'
          CALL ABORT_MEDUSA()
        ENDIF

      ENDDO

      CALL SNCA_PUT_C_DOUBLE(ncid, id_area, darr_c)


      IF (l_file_is_mine) THEN
        istatus = NF_CLOSE(ncid)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
      ENDIF


      DEALLOCATE(iarr_c)
      DEALLOCATE(darr_c)

      DEALLOCATE(iarr_ij)
      DEALLOCATE(rarr_tlong)
      DEALLOCATE(rarr_tlat)
      DEALLOCATE(darr_lon4ij)
      DEALLOCATE(darr_lat4ij)

      DEALLOCATE(iarri_c)
      DEALLOCATE(iarrj_c)

      DEALLOCATE(darrx_c)
      DEALLOCATE(darry_c)


      CALL SNCA_RESET()


      RETURN



      CONTAINS

!-----------------------------------------------------------------------
      SUBROUTINE SNCA_SETUP()
!-----------------------------------------------------------------------


      USE mod_seafloor_central,     ONLY: N_COLUMNS_USED
#ifdef MEDUSA_BASE2D
      USE mod_seafloor_central,     ONLY: IJ_COLUMNS_USED
#endif


      IMPLICIT NONE

                                    ! Get the number of columns in the
                                    ! mod_seafloor_central of the
                                    ! current process
      CALL N_COLUMNS_USED(nsedcol_central)

                                    ! If nsedcol_central == -1, the
                                    ! basic setup has not yet been
                                    ! done, and it does not make any
                                    ! sense to proceed
      IF (nsedcol_central < 0) THEN
        WRITE(jp_stderr,*) '[STORE_NC_AUX/SNCA_SETUP] error: ' //
     &    'Sea-floor setup not yet done -- aborting!'
        CALL ABORT_MEDUSA()
      ENDIF


#ifdef MEDUSA_BASE2D
                                    ! Get the number of seafloor grid
                                    ! elements along the X and Y directions
                                    ! [MPI] would actually have to check
                                    ! if these are all the same
      CALL IJ_COLUMNS_USED(nix, njy)
      nijxy = nix*njy
#endif

                                    ! In single-processor environments the
                                    ! running process always owns its files
                                    ! and writes only nsedcol_central-long
                                    ! records
      l_file_is_mine = .TRUE.
      nsedcol_ncfile = nsedcol_central

#ifdef MEDUSA_BASE2D
      nix_ncfile = nix
      njy_ncfile = njy
#endif

#ifdef DEBUG
      WRITE(jp_stddbg, '(A,I0)')
     &   '[SNCA_SETUP] nsedcol_central = ', nsedcol_central
      WRITE(jp_stddbg, '(A,I0)')
     &   '[SNCA_SETUP] nsedcol_ncfile = ', nsedcol_ncfile
#  ifdef MEDUSA_BASE2D
      WRITE(jp_stddbg, '(A,I0," * ",I0)')
     &   '[SNCA_SETUP] nix * njy = ', nix, njy
      WRITE(jp_stddbg, '(A,I0," * ",I0)')
     &   '[SNCA_SETUP] nix_ncfile * njy_ncfile = ',
     &   nix_ncfile, njy_ncfile
#  endif
      WRITE(jp_stddbg, '(A,L1)')
     &   '[SNCA_SETUP] l_file_is_mine = ', l_file_is_mine
#endif


      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE SNCA_SETUP
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE SNCA_RESET()
!-----------------------------------------------------------------------


      IMPLICIT NONE


      nsedcol_central = -1
      nsedcol_ncfile  = -1
#ifdef MEDUSA_BASE2D
      nix = -1
      njy = -1
      nix_ncfile = -1
      njy_ncfile = -1
#endif


      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE SNCA_RESET
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE SNCA_PUT_C_DOUBLE(ncid, id_var, dvar_c)
!-----------------------------------------------------------------------

! Write out one complete record provided by the DOUBLE PRECISION array
! dvar_c(1:nsedcol_central) into a NetCDF variable with dimension (dim_col)

      USE mod_netcdfinc,            ONLY: HANDLE_ERRORS


      IMPLICIT NONE


      INTEGER,          INTENT(IN)               :: ncid
      INTEGER,          INTENT(IN)               :: id_var
      DOUBLE PRECISION, INTENT(IN), DIMENSION(:) :: dvar_c


      INTEGER :: istatus


      istatus = NF_PUT_VAR_DOUBLE(ncid, id_var, dvar_c(:))
      IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE SNCA_PUT_C_DOUBLE
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE SNCA_PUT_C_INT(ncid, id_var, ivar_c)
!-----------------------------------------------------------------------

! Write out one complete record provided by the INTEGER array
! ivar_c(1:nsedcol_central) into a NetCDF variable with dimension (dim_col)

      USE mod_netcdfinc,            ONLY: HANDLE_ERRORS


      IMPLICIT NONE


      INTEGER, INTENT(IN)               :: ncid
      INTEGER, INTENT(IN)               :: id_var
      INTEGER, INTENT(IN), DIMENSION(:) :: ivar_c


      INTEGER :: istatus


      istatus = NF_PUT_VAR_INT(ncid, id_var, ivar_c(:))
      IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE SNCA_PUT_C_INT
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE SNCA_PUT_IJ_INT(ncid, id_var, ivar_ij)
!-----------------------------------------------------------------------

! Write out one complete record provided by the INTEGER array
! ivar_c(1:nsedcol_central) into a NetCDF variable with dimension (dim_col)

      USE mod_netcdfinc,            ONLY: HANDLE_ERRORS


      IMPLICIT NONE


      INTEGER, INTENT(IN)                 :: ncid
      INTEGER, INTENT(IN)                 :: id_var
      INTEGER, INTENT(IN), DIMENSION(:,:) :: ivar_ij


      INTEGER :: istatus
      INTEGER, DIMENSION(2) :: istart, ncount

#ifdef DEBUG
      CHARACTER(LEN=*), PARAMETER :: cfmt_dbg_a
     &  = '("[STORE_NC_AUX/SNCA_PUT_IJ_INT] debug: ", A)'
#endif

      IF (ANY(SHAPE(ivar_ij) /= (/ nix_ncfile, njy_ncfile /))) THEN

        WRITE(jp_stderr, '("[SNCA_PUT_IJ_INT] error: ")', ADVANCE='NO')
        WRITE(jp_stderr, '("incorrect shape for ivar_ij")')
        WRITE(jp_stderr, '(" expected: (/", I0, ",", I0, "/)")')
     &    nix, njy
        WRITE(jp_stderr, '(" actually: (/", I0, ",", I0, "/)")')
     &    SHAPE(ivar_ij)
        WRITE(jp_stderr, '("Aborting!")')

        CALL ABORT_MEDUSA()

      ENDIF


      istatus = NF_PUT_VAR_INT(ncid, id_var, ivar_ij(:,:))
      IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE SNCA_PUT_IJ_INT
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
      SUBROUTINE SNCA_PUT_IJ_DOUBLE(ncid, id_var, dvar_ij)
!-----------------------------------------------------------------------

! Write out one complete record provided by the DOUBLE PRECISION array
! dvar_c(1:nsedcol_central) into a NetCDF variable with dimension (dim_ix, dim_jy)

      USE mod_netcdfinc,            ONLY: HANDLE_ERRORS


      IMPLICIT NONE


      INTEGER, INTENT(IN)                          :: ncid
      INTEGER, INTENT(IN)                          :: id_var
      DOUBLE PRECISION, INTENT(IN), DIMENSION(:,:) :: dvar_ij


      INTEGER :: istatus
      INTEGER, DIMENSION(2) :: istart, ncount

#ifdef DEBUG
      CHARACTER(LEN=*), PARAMETER :: cfmt_dbg_a
     &  = '("[STORE_NC_AUX/SNCA_PUT_IJ_DOUBLE] debug: ", A)'
#endif

      IF (ANY(SHAPE(dvar_ij) /= (/ nix_ncfile, njy_ncfile /))) THEN

        WRITE(jp_stderr, '("[SNCA_PUT_IJ_DOUBLE] error: ")',
     &                               ADVANCE='NO')
        WRITE(jp_stderr, '("incorrect shape for dvar_ij")')
        WRITE(jp_stderr, '(" expected: (/", I0, ",", I0, "/)")')
     &    nix, njy
        WRITE(jp_stderr, '(" actually: (/", I0, ",", I0, "/)")')
     &    SHAPE(dvar_ij)
        WRITE(jp_stderr, '("Aborting!")')

        CALL ABORT_MEDUSA()

      ENDIF


      istatus = NF_PUT_VAR_DOUBLE(ncid, id_var, dvar_ij(:,:))
      IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE SNCA_PUT_IJ_DOUBLE
!-----------------------------------------------------------------------


!=======================================================================
      END SUBROUTINE STORE_NC_AUX
!=======================================================================
