!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#ifdef CFN_THISFILE
#undef CFN_THISFILE
#endif
#define CFN_THISFILE "mod_coupsim_subr.F"
#ifndef __LINE__
#define __LINE__ 0
#endif
!=======================================================================
      MODULE MOD_COUPSIM_SUBR
!=======================================================================


      IMPLICIT NONE

      PRIVATE

      PUBLIC :: COUPSIM_BIOGEOCHEM_STEP

      INTEGER, SAVE :: nix             = -1
      INTEGER, SAVE :: njy             = -1

      INTEGER, SAVE :: nlons_file      = -1
      INTEGER, SAVE :: nlats_file      = -1
      INTEGER, SAVE :: nzts_file       = -1

      INTEGER, DIMENSION(:,:), ALLOCATABLE, SAVE :: imask_ocean_file

      CHARACTER(LEN=*), PARAMETER :: cfn_thisfile =
     &  CFN_THISFILE


      CONTAINS


!-----------------------------------------------------------------------
      SUBROUTINE COUPSIM_BIOGEOCHEM_STEP
!-----------------------------------------------------------------------


      USE mod_defines_medusa
      USE mod_execontrol_medusa,    ONLY: ABORT_MEDUSA
      USE mod_logunits

      USE mod_netcdfinc,            ONLY: NF_OPEN, NF_CLOSE,
     &                                    NF_NOERR, NF_NOWRITE,
     &                                    NF_INQ_VARID,
     &                                    NF_GET_VAR_DOUBLE,
     &                                    NF_GET_VAR_REAL,
     &                                    HANDLE_NCERRORS

      USE mod_basicdata_medusa
      USE mod_indexparam
      USE mod_materialcharas
      USE mod_logunits

      USE mod_seafloor_wdata

      USE mod_seafloor_central,     ONLY: N_COLUMNS_USED,
     &                                    IJ_COLUMNS_USED,
     &                                    COLUMN_IJ2N

      USE mod_files_medusa,         ONLY: cfn_ncin_bec

      USE mod_coupsim_medusa_setup, ONLY: GET_NLONSLATSZTS_FILE,
     &                                    GET_IMASK_OCEAN_FILE
      USE mod_coupsim_o2s


      IMPLICIT NONE


      INTEGER       :: i, j, n

      INTEGER, SAVE :: n_datasets_read = 0

      INTEGER       :: istatus, iflag


      INTEGER       :: ncid_file, ncid_var

      INTEGER       :: iu_nml


      INTEGER, SAVE :: nsedcol_central   = -1


      DOUBLE PRECISION, DIMENSION(:),   ALLOCATABLE :: dvar1d_zt
      REAL,             DIMENSION(:,:), ALLOCATABLE :: rvar2d_ht



      ! Standard I/O related data
      ! -------------------------

      CHARACTER(LEN=*), PARAMETER :: cfmt_modprocname_a
     &  = '("[MOD_COUPSIM_SUBR/COUPSIM_BIOGEOCHEM_STEP]: ", A)'

      CHARACTER(LEN=*), PARAMETER :: cfmt_a     = '(" - ", A)'
      CHARACTER(LEN=*), PARAMETER :: cfmt_a_ind = '("   ", A)'


#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_modprocname_a) 'Start'
#endif

      ifnodsread: IF (n_datasets_read == 0) THEN

        CALL N_COLUMNS_USED(nsedcol_central)
        IF (nsedcol_central == -1) THEN
          WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
          WRITE(jp_stderr, cfmt_a)
     &       'MOD_SEAFLOOR_CENTRAL not yet set up -- aborting.'
          CALL ABORT_MEDUSA()
        ENDIF

        CALL IJ_COLUMNS_USED(nix, njy)


        IF (.NOT. ALLOCATED(seafloor_dept)) THEN
          WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
          WRITE(jp_stderr, cfmt_a) 
     &      'MOD_COUPSIM_O2S not yet set up -- aborting.'
          CALL ABORT_MEDUSA()
        ENDIF

        CALL GET_NLONSLATSZTS_FILE(nlons_file, nlats_file, nzts_file)

        ALLOCATE(imask_ocean_file(nlons_file, nlats_file))

        CALL GET_IMASK_OCEAN_FILE(imask_ocean_file)

        
      ENDIF ifnodsread

      
                                    !===============================!
                                    ! Open annual mean results file !
                                    !===============================!

      istatus = NF_OPEN(cfn_ncin_bec, NF_NOWRITE, ncid_file)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))


                                    ! Initialize DEPTH data.
                                    ! These must be read separately, as
                                    ! they are not on the global 3D grid.
      ALLOCATE(dvar1d_zt(nzts_file))
      ALLOCATE(rvar2d_ht(nlons_file, nlats_file))

                                    ! Read in bottom-of-layer depth distributions
                                    ! z_t (type: DOUBLE; units: cm)

                                    ! double z_t(z_t) ;
                                    !   z_t:long_name = "depth from surface to midpoint of layer" ;
                                    !   z_t:units = "centimeters" ;

      istatus = NF_INQ_VARID(ncid_file, 'z_t', ncid_var)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
      istatus = NF_GET_VAR_DOUBLE(ncid_file, ncid_var, dvar1d_zt)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

                                    ! Read in ocean depth distributions
                                    ! HT (type: REAL; units: cm)

                                    ! float HT(nlat, nlon) ;
                                    !   HT:long_name = "ocean depth at T points" ;
                                    !   HT:units = "centimeter" ;

      istatus = NF_INQ_VARID(ncid_file, 'HT', ncid_var)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
      istatus = NF_GET_VAR_REAL(ncid_file, ncid_var, rvar2d_ht)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))



      DO j = 1, njy
      DO i = 1, nix

        CALL COLUMN_IJ2N(i, j, iflag, n)

        IF (n == jp_is_not_at_seafloor_idx) THEN

          seafloor_dept(i,j) = 0.0D+00

        ELSE

          seafloor_dept(i,j) = DBLE(rvar2d_ht(i,j))

        ENDIF

      ENDDO
      ENDDO




                                    ! Read in the temperature (tmpdc) data
                                    !   float TEMP(z_t, nlat, nlon) ;
                                    !     TEMP:long_name = "Potential Temperature" ;
                                    !     TEMP:units = "degC" ;
                                    !     TEMP:coordinates = "TLONG TLAT z_t " ;
                                    !     TEMP:time_op = "average" ;
                                    !     TEMP:FillValue = 9.96921e+36f ;
                                    !     TEMP:missing_value = 9.96921e+36f ;
                                    !     TEMP:cell_methods = "time: mean" ;

      CALL READNC_SEAFLOOR_2D(ncid_file, 'TEMP',
     &                              seafloor_temp)


                                    ! Read in the salinity (salin) data
                                    !   float SALT(z_t, nlat, nlon) ;
                                    !     SALT:long_name = "Salinity" ;
                                    !     SALT:units = "gram/kilogram" ;
                                    !     SALT:coordinates = "TLONG TLAT z_t " ;
                                    !     SALT:time_op = "average" ;
                                    !     SALT:FillValue = 9.96921e+33f ;
                                    !     SALT:missing_value = 9.96921e+33f ;
                                    !     SALT:cell_methods = "time: mean" ;

      CALL READNC_SEAFLOOR_2D(ncid_file, 'SALT',
     &                              seafloor_sali)


                                    ! Read in DIC data
                                    !   float DIC(z_t, nlat, nlon) ;
                                    !     DIC:long_name = "Dissolved Inorganic Carbon" ;
                                    !     DIC:units = "mmol C/m^3" ;
                                    !     DIC:coordinates = "TLONG TLAT z_t " ;
                                    !     DIC:time_op = "average" ;
                                    !     DIC:FillValue = 9.96921e+36f ;
                                    !     DIC:missing_value = 9.96921e+36f ;
                                    !     DIC:cell_methods = "time: mean" ;

      CALL READNC_SEAFLOOR_2D(ncid_file, 'DIC',
     &                              seafloor_wconc_dic)


                                    ! Read in ALK data
                                    !   float ALK(z_t, nlat, nlon) ;
                                    !     ALK:long_name = "Alkalinity" ;
                                    !     ALK:units = "meq/m^3" ;
                                    !     ALK:coordinates = "TLONG TLAT z_t " ;
                                    !     ALK:time_op = "average" ;
                                    !     ALK:FillValue = 9.96921e+36f ;
                                    !     ALK:missing_value = 9.96921e+36f ;
                                    !     ALK:cell_methods = "time: mean" ;

      CALL READNC_SEAFLOOR_2D(ncid_file, 'ALK',
     &                              seafloor_wconc_alk)


                                    ! Read in Clay/Dust Flux data
                                    !   float dust_FLUX_IN(z_t, nlat, nlon) ;
                                    !     dust_FLUX_IN:long_name = "Incoming Flux of dust" ;
                                    !     dust_FLUX_IN:units = "g/cm^2/sec" ;
                                    !     dust_FLUX_IN:coordinates = "TLONG TLAT z_t " ;
                                    !     dust_FLUX_IN:time_op = "average" ;
                                    !     dust_FLUX_IN:FillValue = 9.96921e+36f ;
                                    !     dust_FLUX_IN:missing_value = 9.96921e+36f ;
                                    !     dust_FLUX_IN:cell_methods = "time: mean" ;

      CALL READNC_SEAFLOOR_FLX2D(ncid_file, 'dust_FLUX_IN',
     &                              seafloor_wfflx_clay)


                                    ! Read in Calcite Flux data
                                    !   float CaCO3_FLUX_IN(z_t, nlat, nlon) ;
                                    !     CaCO3_FLUX_IN:long_name = "Incoming Flux of CaCO3" ;
                                    !     CaCO3_FLUX_IN:units = "mmol CaCO3/m^3 cm/sec" ;
                                    !     CaCO3_FLUX_IN:coordinates = "TLONG TLAT z_t " ;
                                    !     CaCO3_FLUX_IN:time_op = "average" ;
                                    !     CaCO3_FLUX_IN:FillValue = 9.96921e+36f ;
                                    !     CaCO3_FLUX_IN:missing_value = 9.96921e+36f ;
                                    !     CaCO3_FLUX_IN:cell_methods = "time: mean" ;

      CALL READNC_SEAFLOOR_FLX2D(ncid_file, 'CaCO3_FLUX_IN',
     &                              seafloor_wfflx_calc)


                                    ! Read in Corg Flux data
                                    !	  float POC_FLUX_IN(z_t, nlat, nlon) ;
                                    !	    POC_FLUX_IN:long_name = "Incoming Flux of POC" ;
                                    !	    POC_FLUX_IN:units = "mmol POC/m^3 cm/sec" ;
                                    !	    POC_FLUX_IN:coordinates = "TLONG TLAT z_t " ;
                                    !	    POC_FLUX_IN:time_op = "average" ;
                                    !	    POC_FLUX_IN:FillValue = 9.96921e+36f ;
                                    !	    POC_FLUX_IN:missing_value = 9.96921e+36f ;
                                    !	    POC_FLUX_IN:cell_methods = "time: mean" ;

      CALL READNC_SEAFLOOR_FLX2D(ncid_file, 'POC_FLUX_IN',
     &                              seafloor_wfflx_om)


                                    ! Read in Opal Flux data
                                    !   float SiO2_FLUX_IN(z_t, nlat, nlon) ;
                                    !     SiO2_FLUX_IN:long_name = "Incoming Flux of SiO2" ;
                                    !     SiO2_FLUX_IN:units = "mmol SiO2/m^3 cm/sec" ;
                                    !     SiO2_FLUX_IN:coordinates = "TLONG TLAT z_t " ;
                                    !     SiO2_FLUX_IN:time_op = "average" ;
                                    !     SiO2_FLUX_IN:FillValue = 9.96921e+36f ;
                                    !     SiO2_FLUX_IN:missing_value = 9.96921e+36f ;
                                    !     SiO2_FLUX_IN:cell_methods = "time: mean" ;

      CALL READNC_SEAFLOOR_FLX2D(ncid_file, 'SiO2_FLUX_IN',
     &                              seafloor_wfflx_opal)


                                    ! Read in Nitrate data
                                    !   float NO3(z_t, nlat, nlon) ;
                                    !     NO3:long_name = "Dissolved Inorganic Nitrate" ;
                                    !     NO3:units = "mmol N/m^3" ;
                                    !     NO3:coordinates = "TLONG TLAT z_t " ;
                                    !     NO3:time_op = "average" ;
                                    !     NO3:FillValue = 9.96921e+36f ;
                                    !     NO3:missing_value = 9.96921e+36f ;
                                    !     NO3:cell_methods = "time: mean" ;

      CALL READNC_SEAFLOOR_2D(ncid_file, 'NO3',
     &                              seafloor_wconc_no3)


                                    ! Read in Oxygen data
                                    !   float O2(z_t, nlat, nlon) ;
                                    !     O2:long_name = "Dissolved Oxygen" ;
                                    !     O2:units = "mmol O2/m^3" ;
                                    !     O2:coordinates = "TLONG TLAT z_t " ;
                                    !     O2:time_op = "average" ;
                                    !     O2:FillValue = 9.96921e+36f ;
                                    !     O2:missing_value = 9.96921e+36f ;
                                    !     O2:cell_methods = "time: mean" ;

      CALL READNC_SEAFLOOR_2D(ncid_file, 'O2',
     &                              seafloor_wconc_o2)


                                    !   float SiO3(z_t, nlat, nlon) ;
                                    !     SiO3:long_name = "Dissolved Inorganic Silicate" ;
                                    !     SiO3:units = "mmol Si/m^3" ;
                                    !     SiO3:coordinates = "TLONG TLAT z_t " ;
                                    !     SiO3:time_op = "average" ;
                                    !     SiO3:FillValue = 9.96921e+36f ;
                                    !     SiO3:missing_value = 9.96921e+36f ;
                                    !     SiO3:cell_methods = "time: mean" ;

      CALL READNC_SEAFLOOR_2D(ncid_file, 'SiO3',
     &                              seafloor_wconc_sio2)


                                    ! Done - close file
      istatus = NF_CLOSE(ncid_file)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))


      n_datasets_read = 1           ! Increment counter now that the
                                    ! boundary condition set is complete

      CALL VALIDATE_O2S_DATASET     ! Validate the dataset in MOD_COUPSIM_O2S

      DEALLOCATE(dvar1d_zt)
      DEALLOCATE(rvar2d_ht)

#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_modprocname_a) 'End'
      WRITE(jp_stddbg, '()')
#endif


      RETURN


      CONTAINS


!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      SUBROUTINE READNC_SEAFLOOR_2D(ncid_file, cname_var, data2d_dp)
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

      USE mod_netcdfinc,            ONLY: NF_INQ_VARID, NF_GET_VAR_REAL,
     &                                    NF_NOERR,
     &                                    HANDLE_NCERRORS


      IMPLICIT NONE


      INTEGER,          INTENT(IN)  :: ncid_file
      CHARACTER(LEN=*), INTENT(IN)  :: cname_var
      DOUBLE PRECISION, INTENT(OUT) :: data2d_dp(:,:)

      INTEGER :: i, j, k
      INTEGER :: n
      INTEGER :: ncid_var
      INTEGER :: istatus, iflag

      REAL, DIMENSION(:,:,:), ALLOCATABLE :: data3d_r


      IF (n_datasets_read == 0) THEN

        IF ((nlons_file == -1) .OR. (nlats_file == -1)) THEN
          WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
          WRITE(jp_stderr, cfmt_a) 
     &      'MEDUSA_COUPSIM not yet set up -- aborting.'
          CALL ABORT_MEDUSA()
        ENDIF

      ENDIF


      ALLOCATE(data3d_r(nlons_file, nlats_file, nzts_file))

      istatus = NF_INQ_VARID(ncid_file, cname_var, ncid_var)
      IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

      istatus = NF_GET_VAR_REAL(ncid_file, ncid_var, data3d_r)
      IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))


      DO j = 1, njy
      DO i = 1, nix

        CALL COLUMN_IJ2N(i, j, iflag, n)

        IF (n == jp_is_not_at_seafloor_idx) THEN

          data2d_dp(i,j) = 0.0D+00

        ELSE

            k = imask_ocean_file(i,j)
            data2d_dp(i,j) = DBLE(data3d_r(i,j,k))

        ENDIF

      ENDDO
      ENDDO


      DEALLOCATE(data3d_r)


      RETURN


!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      END SUBROUTINE READNC_SEAFLOOR_2D
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -



!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      SUBROUTINE READNC_SEAFLOOR_FLX2D(ncid_file, cname_var, data2d_dp)
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

      USE mod_netcdfinc,            ONLY: NF_INQ_VARID, NF_GET_VAR_REAL,
     &                                    NF_NOERR,
     &                                    HANDLE_NCERRORS


      IMPLICIT NONE


      INTEGER,          INTENT(IN)  :: ncid_file
      CHARACTER(LEN=*), INTENT(IN)  :: cname_var
      DOUBLE PRECISION, INTENT(OUT) :: data2d_dp(:,:)

      INTEGER :: i, j, k
      INTEGER :: n
      INTEGER :: ncid_var
      INTEGER :: istatus, iflag

      REAL, DIMENSION(:,:,:), ALLOCATABLE :: data3d_r
      DOUBLE PRECISION :: dflxratio, dexponent


      IF (n_datasets_read == 0) THEN

        IF ((nlons_file == -1) .OR. (nlats_file == -1)) THEN
          WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
          WRITE(jp_stderr, cfmt_a) 
     &      'MEDUSA_COUPSIM not yet set up -- aborting.'
          CALL ABORT_MEDUSA()
        ENDIF

      ENDIF


      ALLOCATE(data3d_r(nlons_file, nlats_file, nzts_file))

      istatus = NF_INQ_VARID(ncid_file, cname_var, ncid_var)
      IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

      istatus = NF_GET_VAR_REAL(ncid_file, ncid_var, data3d_r)
      IF (istatus /= NF_NOERR)
     &    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))


      DO j = 1, njy
      DO i = 1, nix

        CALL COLUMN_IJ2N(i, j, iflag, n)

        IF (n == jp_is_not_at_seafloor_idx) THEN

          data2d_dp(i,j) = 0.0D+00

        ELSE

          k = imask_ocean_file(i,j)

          IF (k < 3) THEN

            data2d_dp(i,j) = DBLE(data3d_r(i,j,k))

          ELSEIF (data3d_r(i,j,k) == 0.0E+00) THEN

            data2d_dp(i,j) = 0.0D+00

          ELSE
                                    ! exponentially decreasing extrapolation
                                    ! F_depos
                                    !   = F_in(k) / (F_in(k-1)/F_in(k))
                                    !               **((z_t(k-1) - ht(i,j))/
                                    !                  (z_t(k-2) - z_t(k-1)))
            dexponent =   (dvar1d_zt(k-1) - DBLE(rvar2d_ht(i,j)))
     &                  / (dvar1d_zt(k-2) - dvar1d_zt(k-1))
            dflxratio = DBLE(data3d_r(i,j,k-1))/DBLE(data3d_r(i,j,k))
            data2d_dp(i,j) = DBLE(data3d_r(i,j,k))/dflxratio**dexponent

          ENDIF

        ENDIF

      ENDDO
      ENDDO


      DEALLOCATE(data3d_r)


      RETURN


!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      END SUBROUTINE READNC_SEAFLOOR_FLX2D
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -



!-----------------------------------------------------------------------
      END SUBROUTINE COUPSIM_BIOGEOCHEM_STEP
!-----------------------------------------------------------------------


!=======================================================================
      END MODULE MOD_COUPSIM_SUBR
!=======================================================================
