!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


!---+----1----+----2----+----3----+----4----+----5----+----6----+----7--
!=======================================================================
      MODULE MOD_COUPSIM_S2O
!=======================================================================


      USE mod_defines_medusa,       ONLY: jp_stderr, jp_stdlog
#ifdef DEBUG
      USE mod_defines_medusa,       ONLY: jp_stddbg
#endif
      USE mod_execontrol_medusa,    ONLY: ABORT_MEDUSA


      IMPLICIT NONE


      PRIVATE

      PUBLIC :: SETUP_MOD_COUPSIM_S2O, CLEAR_S2O_DATASET,
     &                             SEDIMENT_TO_OCEAN

      INTEGER, SAVE :: nsedcol_central   = -1
      INTEGER, SAVE :: nix               = -1
      INTEGER, SAVE :: njy               = -1

      INTEGER, SAVE :: id_dataset_s2o    = -1

      LOGICAL, SAVE :: l_mod_s2o_setup_done = .FALSE.


                                    ! Solute fluxes across sediment top
                                    ! one array per solute component
      DOUBLE PRECISION, SAVE, DIMENSION(:,:), ALLOCATABLE, PUBLIC
     &  :: seafloor_wcflx_dic
      DOUBLE PRECISION, SAVE, DIMENSION(:,:), ALLOCATABLE, PUBLIC
     &  :: seafloor_wcflx_alk
      DOUBLE PRECISION, SAVE, DIMENSION(:,:), ALLOCATABLE, PUBLIC
     &  :: seafloor_wcflx_o2
      DOUBLE PRECISION, SAVE, DIMENSION(:,:), ALLOCATABLE, PUBLIC
     &  :: seafloor_wcflx_no3
      DOUBLE PRECISION, SAVE, DIMENSION(:,:), ALLOCATABLE, PUBLIC
     &  :: seafloor_wcflx_sio2


                                    ! Solid burial fluxes
                                    ! one array per solid component in Medusa.
      DOUBLE PRECISION, SAVE, DIMENSION(:,:), ALLOCATABLE, PUBLIC
     &  :: seafloor_bfflx_clay
      DOUBLE PRECISION, SAVE, DIMENSION(:,:), ALLOCATABLE, PUBLIC
     &  :: seafloor_bfflx_calc
      DOUBLE PRECISION, SAVE, DIMENSION(:,:), ALLOCATABLE, PUBLIC
     &  :: seafloor_bfflx_om
      DOUBLE PRECISION, SAVE, DIMENSION(:,:), ALLOCATABLE, PUBLIC
     &  :: seafloor_bfflx_opal


      CONTAINS


!-----------------------------------------------------------------------
      SUBROUTINE SETUP_MOD_COUPSIM_S2O
!-----------------------------------------------------------------------


      USE mod_seafloor_central,     ONLY: N_COLUMNS_USED,
     &                                    IJ_COLUMNS_USED


      IMPLICIT NONE


      ! Standard I/O related data
      ! -------------------------

      CHARACTER(LEN=*), PARAMETER :: c_fmterr_a
     &  = '("[MOD_COUPSIM_S2O/SETUP_MOD_COUPSIM_S2O] error: ", A)'

      CHARACTER(LEN=*), PARAMETER :: c_fmtinfo_a
     &  = '("[MOD_COUPSIM_S2O/SETUP_MOD_COUPSIM_S2O] info: ", A)'


      IF (l_mod_s2o_setup_done) THEN
        WRITE(jp_stderr, c_fmterr_a)
     &      'setup already done -- aborting!'
        CALL ABORT_MEDUSA()
      ENDIF


      WRITE(jp_stdlog, c_fmtinfo_a)
     &  'COUPling SIMulator -- simulating MOD_COUPSIM_S2O set-up only'

      !CALL N_COLUMNS_USED(nsedcol_central)
      !CALL IJ_COLUMNS_USED(nix, njy)

      nsedcol_central = 0           ! Shortcut actual usage of
      nix = 0                       ! the arrays in this module by
      njy = 0                       ! forcing all dimensions to 0.


      ALLOCATE(seafloor_wcflx_dic(nix, njy))
      ALLOCATE(seafloor_wcflx_alk(nix, njy))
      ALLOCATE(seafloor_wcflx_o2(nix, njy))
      ALLOCATE(seafloor_wcflx_no3(nix, njy))
      ALLOCATE(seafloor_wcflx_sio2(nix, njy))

      ALLOCATE(seafloor_bfflx_clay(nix, njy))
      ALLOCATE(seafloor_bfflx_calc(nix, njy))
      ALLOCATE(seafloor_bfflx_om(nix, njy))
      ALLOCATE(seafloor_bfflx_opal(nix, njy))


      l_mod_s2o_setup_done = .TRUE.


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE SETUP_MOD_COUPSIM_S2O
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE CLEAR_S2O_DATASET
!-----------------------------------------------------------------------


      IMPLICIT NONE


      ! Standard I/O related data
      ! -------------------------

      CHARACTER(LEN=*), PARAMETER :: c_fmterr_a
     &  = '("[MOD_COUPSIM_O2S/CLEAR_S2O_DATASET] error: ", A)'


      IF (.NOT. l_mod_s2o_setup_done) THEN
        WRITE(jp_stderr, c_fmterr_a)
     &    'MOD_COUPSIM_O2S not yet set up -- aborting!'
        CALL ABORT_MEDUSA()
      ENDIF


      seafloor_wcflx_dic(:,:)  = 0.0D+00
      seafloor_wcflx_alk(:,:)  = 0.0D+00
      seafloor_wcflx_o2(:,:)   = 0.0D+00
      seafloor_wcflx_no3(:,:)  = 0.0D+00
      seafloor_wcflx_sio2(:,:) = 0.0D+00

      seafloor_bfflx_clay(:,:) = 0.0D+00
      seafloor_bfflx_calc(:,:) = 0.0D+00
      seafloor_bfflx_om(:,:)   = 0.0D+00
      seafloor_bfflx_opal(:,:) = 0.0D+00



      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE CLEAR_S2O_DATASET
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE SEDIMENT_TO_OCEAN(kflag)
!-----------------------------------------------------------------------
      ! Returns model outcome to the module mod_coupsim_s2o, carrying out
      ! unit conversions appropriate for the COUPSIM model


      USE mod_indexparam
      USE mod_seafloor_central,     ONLY: COLUMN_IJ2N, COLUMN_N2IJ,
     &                                    GET_BOUNDARY_FLUXES

      USE mod_materialcharas    ! for mol weights, molar data, etc.


      IMPLICIT NONE


      INTEGER, INTENT(OUT) :: kflag


      INTEGER :: i, j, n, iflag

      DOUBLE PRECISION, DIMENSION(nsolut) :: wcflx
      DOUBLE PRECISION, DIMENSION(nsolid) :: bfflx

      DOUBLE PRECISION :: sfc_area


      ! Standard I/O related data
      ! -------------------------

      CHARACTER(LEN=*), PARAMETER :: cfmt_modprocname_a
     &  = '("[MOD_COUPSIM_O2S/OCEAN_TO_SEDIMENT]: ", A)'

      CHARACTER(LEN=*), PARAMETER :: cfmt_a     = '(" - ", A)'
      CHARACTER(LEN=*), PARAMETER :: cfmt_a_ind = '("   ", A)'


#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_modprocname_a) 'Start'
#endif


      IF (.NOT. l_mod_s2o_setup_done) THEN
        WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
        WRITE(jp_stderr, cfmt_a)
     &    'MOD_COUPSIM_S2O not yet set up -- aborting!'
        CALL ABORT_MEDUSA()
      ENDIF


      DO n = 1, nsedcol_central

        CALL COLUMN_N2IJ(i_column = n, iflag = iflag,
     &                              ix = i, jy = j)

        CALL GET_BOUNDARY_FLUXES(i_column = n, gbfflag = iflag,
     &                              wcflx = wcflx(:),
     &                              bfflx = bfflx(:))


                                    ! Convert units here, if necessary:
                                    ! --------------------------------

                                    !  - wcflx [mol/m2/yr]
                                    !  - seafloor_wcflx_* [mmol/m3 cm/s]
                                    !    J/[mmol/m3 cm/s]
                                    !      = J/[(10^-3 mol)/m3 (10^-2 m)/(dp_sec yr)]
                                    !      = 10*dp_sec * J/[mol/m2/yr]
                                    !    
      seafloor_wcflx_dic(i,j)  =
     &  (wcflx(ic_co3) + wcflx(ic_hco3) + wcflx(ic_co2))
     &  * 10.0D+00*dp_sec
      seafloor_wcflx_alk(i,j)  =
     &  (wcflx(ic_hco3) + wcflx(ic_co3) + wcflx(ic_co3))
     &  * 10.0D+00*dp_sec
      seafloor_wcflx_o2(i,j)   = wcflx(ic_o2) * 10.0D+00*dp_sec
      seafloor_wcflx_no3(i,j)  = wcflx(ic_no3) * 10.0D+00*dp_sec
      seafloor_wcflx_sio2(i,j) = wcflx(ic_sio2) * 10.0D+00*dp_sec

                                    !  - bfflx [kg/m2/yr]
                                    !  - seafloor_bfflx_clay [g/cm2/s]
                                    !    J/[g/cm2/yr]
                                    !      = J/[(10^-3 kg)/ (10^-4 m2) / (dp_sec yr)]
                                    !      = dp_sec/10 * J/[kg/m2/yr]
      seafloor_bfflx_clay(i,j) = bfflx(if_clay)
     &                              * dp_sec/10.0D+00

                                    !  - bfflx [kg/m2/yr]
                                    !  - seafloor_bfflx_calc [mmol/cm2/s]
      seafloor_bfflx_calc(i,j) = bfflx(if_calc)
     &                              * dp_sec/(mol_calc*10.0D+00)

                                    !  - bfflx [kg/m2/yr]
                                    !  - seafloor_bfflx_om [molC/cm2/s]
      seafloor_bfflx_om(i,j)   = bfflx(if_om)
     &                              * dp_sec/(mol_om*10.0+00) * om_c

                                    !  - bfflx [kg/m2/yr]
                                    !  - seafloor_bfflx__opal [molSi/cm2/s]
      seafloor_bfflx_opal(i,j) = bfflx(if_opal)
     &                              * dp_sec/(mol_opal*10.0D+00)
     &                              * opal_si


      ENDDO


      id_dataset_s2o  = id_dataset_s2o + 1

      kflag = 0


#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_modprocname_a) 'End'
      WRITE(jp_stddbg, '()')
#endif


      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE SEDIMENT_TO_OCEAN
!-----------------------------------------------------------------------

!=======================================================================
      END MODULE MOD_COUPSIM_S2O
!=======================================================================
