!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#ifdef CFN_THISFILE
#undef CFN_THISFILE
#endif
#define CFN_THISFILE "mod_coupsim_medusa_setup.F"
#ifndef __LINE__
#define __LINE__ 0
#endif
!=======================================================================
      MODULE MOD_COUPSIM_MEDUSA_SETUP
!=======================================================================

      IMPLICIT NONE


      PRIVATE

      PUBLIC :: SETUP_MEDUSA_FOR_COUPSIM
      PUBLIC :: GET_NLONSLATSZTS_FILE
      PUBLIC :: GET_IMASK_OCEAN_FILE


      LOGICAL, SAVE :: medusa4coupsim_setup_done = .FALSE.

      CHARACTER(LEN=*), PARAMETER :: cfn_thisfile =
     &  CFN_THISFILE

      INTEGER, SAVE :: nlons_file = -1
      INTEGER, SAVE :: nlats_file = -1
      INTEGER, SAVE :: nzts_file  = -1

      INTEGER, SAVE, DIMENSION(:,:), ALLOCATABLE :: imask_ocean_file


      CONTAINS


!-----------------------------------------------------------------------
      SUBROUTINE SETUP_MEDUSA_FOR_COUPSIM(n_columns)
!-----------------------------------------------------------------------

      USE mod_defines_medusa
      USE mod_execontrol_medusa,    ONLY: ABORT_MEDUSA

      USE mod_gridparam
      USE mod_seafloor_central,     ONLY: SEAFLOOR_SETUP

      USE mod_materialcharas
      USE mod_logunits

      USE mod_files_medusa,         ONLY: cfn_ncin_bec,
     &                                    cfn_nmlin_setup

      USE mod_netcdfinc,            ONLY: NF_OPEN, NF_CLOSE,
     &                                    NF_INQ_DIMID, NF_INQ_DIMLEN,
     &                                    NF_INQ_VARID,
     &                                    NF_GET_VAR_DOUBLE,
     &                                    NF_GET_VAR_REAL,
     &                                    NF_GET_VARA_REAL,
     &                                    NF_GET_VAR_INT,
     &                                    NF_GET_ATT_REAL,
     &                                    NF_NOWRITE, NF_NOERR,
     &                                    HANDLE_NCERRORS

      IMPLICIT NONE


      INTEGER, INTENT(OUT) :: n_columns


      INTEGER :: iu_setup, iu_mask
      LOGICAL :: l_exists


      INTEGER,          DIMENSION(:,:), ALLOCATABLE
     &  :: imask_ocean_coupsim
      DOUBLE PRECISION, DIMENSION(:,:), ALLOCATABLE
     &  :: dlon_gridelts_coupsim
      DOUBLE PRECISION, DIMENSION(:,:), ALLOCATABLE
     &  :: dlat_gridelts_coupsim
      DOUBLE PRECISION, DIMENSION(:,:), ALLOCATABLE
     &  :: darea_gridelts_coupsim

      DOUBLE PRECISION, DIMENSION(:,:,:), ALLOCATABLE
     &  :: domcnp_c, domcnp_n, domcnp_p, domcnp_o, domcnp_h, domcnp_ro2


      INTEGER :: nlons_coupsim, nlats_coupsim
      INTEGER :: i, j, k

      INTEGER :: ncid_file
      INTEGER :: ncid_dim_lon, ncid_dim_lat, ncid_dim_zt, ncid_dim_zw
      INTEGER :: ncid_var
      INTEGER :: istatus
      INTEGER, DIMENSION(3) :: istart, ncount

      INTEGER :: nzws_file

      REAL,             DIMENSION(:,:), ALLOCATABLE :: data2d_r
      REAL,             DIMENSION(:,:), ALLOCATABLE :: rvar2d_lon
      REAL,             DIMENSION(:,:), ALLOCATABLE :: rvar2d_lat
      REAL,             DIMENSION(:,:), ALLOCATABLE :: rvar2d_area
      DOUBLE PRECISION, DIMENSION(:),   ALLOCATABLE :: dvar1d_zw
      INTEGER,          DIMENSION(:,:), ALLOCATABLE :: ivar2d_kmt


                                    ! Redfield ratios:
                                    ! initialize with default values
      DOUBLE PRECISION :: rcp  = 122.0D+00
      DOUBLE PRECISION :: rhp  = 243.0D+00
      DOUBLE PRECISION :: roop =  64.0D+00
      DOUBLE PRECISION :: rnp  =  16.0D+00
      DOUBLE PRECISION :: rpp  =   1.0D+00
      DOUBLE PRECISION :: rop  = 172.0D+00

      NAMELIST /nml_redfield/ rcp, rhp, roop, rnp, rpp, rop


      DOUBLE PRECISION :: dlimlat_north =   90.0D+00
      DOUBLE PRECISION :: dlimlat_south =  -90.0D+00
      DOUBLE PRECISION :: dlimzw_top    = 1000.0D+00

      NAMELIST /nml_regridding/
     &                    dlimlat_north, dlimlat_south, dlimzw_top

      CHARACTER(LEN=32) :: cfmt

      REAL :: rlimlat_north
      REAL :: rlimlat_south

! Standard I/O related data
! -------------------------

      CHARACTER(LEN=*), PARAMETER :: cfmt_modprocname_a
     &  = '("[MOD_COUPSIM_MEDUSA_SETUP/SETUP_MEDUSA_FOR_COUPSIM]: ", A)'

      CHARACTER(LEN=*), PARAMETER :: cfmt_a     = '(" - ", A)'
      CHARACTER(LEN=*), PARAMETER :: cfmt_a_ind = '("   ", A)'


#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_modprocname_a) 'Start'
#endif


                                    ! Setup can only be done once!
                                    ! Abort program if the setup
                                    ! has already been called before.
      IF (medusa4coupsim_setup_done) THEN
        WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
        WRITE(jp_stderr, cfmt_a) 'Setup already done -- aborting.'
        CALL ABORT_MEDUSA()
      ENDIF

                                    ! Read in Redfield ratios and the
                                    ! grid cropping information from the
                                    ! COUPSIM setup namelist file
                                    ! - check if cfn_nmlin_setup is valid
                                    !   * set to "/dev/null" ?
      IF (cfn_nmlin_setup  == "/dev/null") THEN

        WRITE(jp_stderr, '()')
        WRITE(jp_stderr, cfmt_modprocname_a) 'Error'
        WRITE(jp_stderr, cfmt_a, ADVANCE="NO")
        WRITE(jp_stderr, '("cfn_nmlin_setup = ""/dev/null"" '//
     &                              'not valid -- aborting!")')
        CALL ABORT_MEDUSA()

      ENDIF

                                    !   * exists ?
      INQUIRE(FILE=cfn_nmlin_setup, EXIST=l_exists)
      IF (.NOT. l_exists) THEN

        WRITE(jp_stderr, '()')
        WRITE(jp_stderr, cfmt_modprocname_a) 'Error'
        WRITE(jp_stderr, cfmt_a, ADVANCE="NO")
        WRITE(jp_stderr, '("cfn_nmlin_setup = """, A, """: '//
     &     'no such file -- aborting!")') TRIM(cfn_nmlin_setup)
        CALL ABORT_MEDUSA()

      ENDIF

                                    !  - request logical file unit number
      istatus = RESERVE_LOGUNIT(iu_setup)
      IF (istatus /= 0) THEN
        WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
        WRITE(jp_stderr, cfmt_a)
     &    'Unable to reserve a logical unit number' //
     &    ' for the COUPSIM_SETUP unit -- aborting!'
        CALL ABORT_MEDUSA()
#ifdef DEBUG
      ELSE
        WRITE(jp_stddbg, cfmt_a, ADVANCE="NO")
     &    'Assigning logical unit number '
        WRITE(jp_stddbg, '(I0, A)') iu_setup,
     &    ' to the COUPSIM_SETUP unit.'
#endif
      ENDIF

                                    !  - open the file, read in the data
                                    !    and close the file.
      OPEN(iu_setup, FILE=cfn_nmlin_setup, STATUS="OLD")

      READ(UNIT=iu_setup, NML=nml_redfield)
      READ(UNIT=iu_setup, NML=nml_regridding)

      CLOSE(iu_setup)

                                    !  - release the logical file unit number
#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_a, ADVANCE="NO")
     &  'Releasing logical unit number '
      WRITE(jp_stddbg, '(I0, A)', ADVANCE="NO")
     &  iu_setup, ' (attached to the COUPSIM_SETUP unit)'
#endif
      istatus = FREE_LOGUNIT(iu_setup)
      IF (istatus /= 0) THEN
        WRITE(jp_stderr, cfmt_modprocname_a) 'Warning'
        WRITE(jp_stderr, cfmt_a, ADVANCE="NO")
     &    'FREE_LOGUNIT returned error code '
        WRITE(jp_stderr, '(I0, A)') istatus,
     &    ' when trying to release the logical unit number'
     &    //  ' attached to the COUPSIM_SETUP unit -- ignoring.'
#ifdef DEBUG
        WRITE(jp_stddbg, '()')
        WRITE(jp_stddbg, cfmt_a_ind, ADVANCE="NO")
        WRITE(jp_stddbg, '(A, I0, A)')
     &    'FREE_LOGUNIT returned error code ', istatus,
     &    ' when trying to release the logical unit number'
     &    //  ' attached to the COUPSIM_SETUP unit -- ignoring.'
      ELSE
        WRITE(jp_stddbg, '(" - done.")')
#endif
      ENDIF

      CALL FLUSH(jp_stderr)
#ifdef DEBUG
      CALL FLUSH(jp_stddbg)
#endif



!-----------------------------------------------------------
! TBD: Check validity of regridding data
!  - dlimlat_north <=  90
!  - dlimlat_south >= -90
!  - dlimlat_north > dlimlat_south
!-----------------------------------------------------------

      rlimlat_north = REAL(dlimlat_north)
      rlimlat_south = REAL(dlimlat_south)
      dlimzw_top    = dlimzw_top*100.0D+00  ! m --> cm

                                    ! Use the  <cfn_ncin_bec> NetCDF file
                                    ! to determine the geometry.

                                    ! Check if <cfn_ncin_bec> is valid
                                    ! - set to "/dev/null" ?
      IF (cfn_ncin_bec  == "/dev/null") THEN

        WRITE(jp_stderr, '()')
        WRITE(jp_stderr, cfmt_modprocname_a) 'Error'
        WRITE(jp_stderr, cfmt_a, ADVANCE="NO")
        WRITE(jp_stderr, '("cfn_ncin_bec = ""/dev/null"" '//
     &                              'not valid --  aborting!")')
        CALL ABORT_MEDUSA()

      ENDIF

                                    ! - exists ?
      INQUIRE(FILE=cfn_ncin_bec, EXIST=l_exists)
      IF (.NOT. l_exists) THEN

        WRITE(jp_stderr, '()')
        WRITE(jp_stderr, cfmt_modprocname_a) 'Error'
        WRITE(jp_stderr, cfmt_a, ADVANCE="NO")
        WRITE(jp_stderr, '("cfn_ncin_bec = """, A, """: '//
     &      'no such file --  aborting!")') TRIM(cfn_ncin_bec)
        CALL ABORT_MEDUSA()

      ENDIF



                                    ! Open file
      istatus = NF_OPEN(cfn_ncin_bec, NF_NOWRITE, ncid_file)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))


                                    ! Inquire information for dimensions
                                    !   - dimension 'lon'
      istatus = NF_INQ_DIMID(ncid_file, 'nlon', ncid_dim_lon)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

      istatus = NF_INQ_DIMLEN(ncid_file, ncid_dim_lon, nlons_file)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

                                    !   - dimension 'lat'
      istatus = NF_INQ_DIMID(ncid_file, 'nlat', ncid_dim_lat)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

      istatus = NF_INQ_DIMLEN(ncid_file, ncid_dim_lat, nlats_file)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

                                    !   - dimension 'z_t'
      istatus = NF_INQ_DIMID(ncid_file, 'z_t', ncid_dim_zt)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

      istatus = NF_INQ_DIMLEN(ncid_file, ncid_dim_zt, nzts_file)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

                                    !   - dimension 'z_w'
      istatus = NF_INQ_DIMID(ncid_file, 'z_w', ncid_dim_zw)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

      istatus = NF_INQ_DIMLEN(ncid_file, ncid_dim_zw, nzws_file)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))



      nlons_coupsim = nlons_file
      nlats_coupsim = nlats_file


      ALLOCATE(rvar2d_lat(nlons_file, nlats_file))
      ALLOCATE(rvar2d_lon(nlons_file, nlats_file))
      ALLOCATE(dvar1d_zw(nzws_file))

      ALLOCATE(ivar2d_kmt(nlons_file, nlats_file))

      ALLOCATE(rvar2d_area(nlons_file, nlats_file))

      ALLOCATE(imask_ocean_file(nlons_file, nlats_file))


                                    ! Read in longitude array
                                    ! (type: FLOAT; units: degrees_east)
      istatus = NF_INQ_VARID(ncid_file, 'TLONG', ncid_var)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
      istatus = NF_GET_VAR_REAL(ncid_file, ncid_var, rvar2d_lon)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))


                                    ! Read in latitude array
                                    ! (type: FLOAT; units: degrees_north)
      istatus = NF_INQ_VARID(ncid_file, 'TLAT', ncid_var)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
      istatus = NF_GET_VAR_REAL(ncid_file, ncid_var, rvar2d_lat)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))


                                    ! Read in surface area array
                                    ! (type: FLOAT; units: cm^2)
      istatus = NF_INQ_VARID(ncid_file, 'TAREA', ncid_var)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
      istatus = NF_GET_VAR_REAL(ncid_file, ncid_var, rvar2d_area)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))



                                    ! Read data arrays required for
                                    ! the ocean mask
                                    ! - KMT (type: INTEGER).
      istatus = NF_INQ_VARID(ncid_file, 'KMT', ncid_var)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
      istatus = NF_GET_VAR_INT(ncid_file, ncid_var, ivar2d_kmt)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

                                    ! - REGION_MASK (type: INTEGER).
      istatus = NF_INQ_VARID(ncid_file, 'REGION_MASK', ncid_var)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
      istatus = NF_GET_VAR_INT(ncid_file, ncid_var, imask_ocean_file)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

                                    ! - z_w (type: DOUBLE; units: cm)
      istatus = NF_INQ_VARID(ncid_file, 'z_w', ncid_var)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
      istatus = NF_GET_VAR_DOUBLE(ncid_file, ncid_var, dvar1d_zw)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))


                                    ! Close the file, since all that
                                    ! can be used has been read in now.
      istatus = NF_CLOSE(ncid_file)
      IF (istatus /= NF_NOERR)
     &  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))

                                    ! Currently imask_ocean_file
                                    !  * is 0 over the land areas
                                    !  * is -14,..., -1 on lakes,
                                    !    internal seas and semi-enclosed basins
                                    !  * 1,...,10 for ocean basins

      WHERE (imask_ocean_file(:,:) < 0)
        imask_ocean_file(:,:) = 0
      ENDWHERE

                                    ! Mask out all points where
                                    ! - the seafloor depth is shallower than
                                    !   <dlimzw_top>

      DO j = 1, nlats_file
      DO i = 1, nlons_file

        IF (imask_ocean_file(i,j) > 0) THEN

          k = ivar2d_kmt(i,j)
          imask_ocean_file(i,j) = k
          IF (dvar1d_zw(k) < dlimzw_top) imask_ocean_file(i,j) = -101
     &                              ! -101 --> actually ocean, but too shallow
        ENDIF

      ENDDO
      ENDDO


      DO j = 1, nlats_file          ! Ignore points north of a
      DO i = 1, nlons_file          ! NH threshold (default: 60°N)
                                    ! and south of a SH threshold
                                    ! (default: 90°S) -- reset to "land"
        IF (     (rvar2d_lat(i,j) > rlimlat_north)
     &      .OR. (rvar2d_lat(i,j) < rlimlat_south)) THEN
                                    ! -102 --> actually deep ocean,
                                    !          but out of lat bounds
          IF (imask_ocean_file(i,j) > 0)
     &                              imask_ocean_file(i,j) = -102
                                    ! -103 --> actually shallow ocean,
                                    !          but also out of lat bounds
          IF (imask_ocean_file(i,j) == -101)
     &                              imask_ocean_file(i,j) = -103
        ENDIF

      ENDDO
      ENDDO


                                    ! Prepare and initialize the arrays
                                    ! for the COUPSIM grid

      ALLOCATE(imask_ocean_coupsim   (nlons_coupsim, nlats_coupsim))
      ALLOCATE(dlat_gridelts_coupsim (nlons_coupsim, nlats_coupsim))
      ALLOCATE(dlon_gridelts_coupsim (nlons_coupsim, nlats_coupsim))
      ALLOCATE(darea_gridelts_coupsim(nlons_coupsim, nlats_coupsim))


      imask_ocean_coupsim(:,:)    = imask_ocean_file(:,:)

                                    !  - dlat_gridelts_coupsim
      dlat_gridelts_coupsim(:,:)  = DBLE(rvar2d_lat(:,:))

                                    !  - dlon_gridelts_coupsim
      dlon_gridelts_coupsim(:,:)  = DBLE(rvar2d_lon(:,:))


      darea_gridelts_coupsim(:,:) = DBLE(rvar2d_area(:,:)*1.0E+04)      ! cm^2 --> m^2



                                    ! * Geographically distributed
                                    !   Redfield ratios and OrgMatter
                                    !   molar mass
      IF (nomcompo > 0) THEN
                                    ! C:N:P:O:H ratios are uniform here,
                                    ! hence the (/ nomcompo, 1, 1 /) shape.
        ALLOCATE(domcnp_c(nomcompo, 1, 1))
        ALLOCATE(domcnp_n(nomcompo, 1, 1))
        ALLOCATE(domcnp_p(nomcompo, 1, 1))
        ALLOCATE(domcnp_o(nomcompo, 1, 1))
        ALLOCATE(domcnp_h(nomcompo, 1, 1))
        ALLOCATE(domcnp_ro2(nomcompo, 1, 1))


        domcnp_c(ioo_om, 1, 1)   = rcp
        domcnp_n(ioo_om, 1, 1)   = rnp
        domcnp_p(ioo_om, 1, 1)   = rpp
        domcnp_o(ioo_om, 1, 1)   = roop
        domcnp_h(ioo_om, 1, 1)   = rhp
        domcnp_ro2(ioo_om, 1, 1) = rop


        CALL SEAFLOOR_SETUP(imask_ocean_coupsim,
     &                              dlon_gridelts_coupsim,
     &                              dlat_gridelts_coupsim,
     &                              darea_gridelts_coupsim,
     &                              domcnp_c, domcnp_n,
     &                              domcnp_p, domcnp_o,
     &                              domcnp_h, domcnp_ro2,
     &                              n_columns)

        DEALLOCATE(domcnp_c)
        DEALLOCATE(domcnp_n)
        DEALLOCATE(domcnp_p)
        DEALLOCATE(domcnp_o)
        DEALLOCATE(domcnp_h)
        DEALLOCATE(domcnp_ro2)

      ELSE

        CALL SEAFLOOR_SETUP(imask_ocean_coupsim,
     &                              dlon_gridelts_coupsim,
     &                              dlat_gridelts_coupsim,
     &                              darea_gridelts_coupsim,
     &                              n_columns=n_columns)

      ENDIF

                                    ! Finally save the complete mask
                                    ! to a file called "coupsim_mask.csv"

                                    !  - request logical file unit number
      istatus = RESERVE_LOGUNIT(iu_mask)
      IF (istatus /= 0) THEN
        WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
        WRITE(jp_stderr, cfmt_a)
     &    'Unable to reserve a logical unit number' //
     &    ' for the COUPSIM_MASK unit -- aborting!'
        CALL ABORT_MEDUSA()
#ifdef DEBUG
      ELSE
        WRITE(jp_stddbg, cfmt_a, ADVANCE="NO")
     &    'Assigning logical unit number '
        WRITE(jp_stddbg, '(I0, A)') iu_mask,
     &    ' to the COUPSIM_MASK unit.'
#endif
      ENDIF

                                    !  - open the file, write out
                                    !    the data and close the file.
      OPEN(iu_mask, FILE="coupsim_mask.csv")

      WRITE(UNIT=iu_mask, FMT='(I0, " ", I0)')
     &                              nlons_file, nlats_file

      WRITE(cfmt, '("(I4,", I0, "("" "", I4))")') nlons_file-1

      DO j = 1, nlats_file
        WRITE(UNIT=iu_mask, FMT=cfmt) imask_ocean_file(:,j)
      ENDDO

      CLOSE(iu_mask)

                                    !  - release the logical file unit number
#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_a, ADVANCE="NO")
     &  'Releasing logical unit number '
      WRITE(jp_stddbg, '(I0, A)', ADVANCE="NO")
     &  iu_mask, ' (attached to the COUPSIM_MASK unit)'
#endif
      istatus = FREE_LOGUNIT(iu_mask)
      IF (istatus /= 0) THEN
        WRITE(jp_stderr, cfmt_modprocname_a) 'Warning'
        WRITE(jp_stderr, cfmt_a, ADVANCE="NO")
     &    'FREE_LOGUNIT returned error code '
        WRITE(jp_stderr, '(I0, A)') istatus,
     &    ' when trying to release the logical unit number'
     &    //  ' attached to the COUPSIM_MASK unit -- ignoring.'
#ifdef DEBUG
        WRITE(jp_stddbg, '()')
        WRITE(jp_stddbg, cfmt_a_ind, ADVANCE="NO")
        WRITE(jp_stddbg, '(A, I0, A)')
     &    'FREE_LOGUNIT returned error code ', istatus,
     &    ' when trying to release the logical unit number'
     &    //  ' attached to the COUPSIM_MASK unit -- ignoring.'
      ELSE
        WRITE(jp_stddbg, '(" - done.")')
#endif
      ENDIF

      CALL FLUSH(jp_stderr)
#ifdef DEBUG
      CALL FLUSH(jp_stddbg)
#endif


#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_a, ADVANCE="NO")
     &  'Number of seafloor grid elements: '
      WRITE(jp_stddbg, '(I0)') n_columns
#endif



      DEALLOCATE(rvar2d_lat)
      DEALLOCATE(rvar2d_lon)
      DEALLOCATE(rvar2d_area)
      DEALLOCATE(ivar2d_kmt)
      DEALLOCATE(dvar1d_zw)

      DEALLOCATE(imask_ocean_coupsim)
      DEALLOCATE(dlat_gridelts_coupsim)
      DEALLOCATE(dlon_gridelts_coupsim)
      DEALLOCATE(darea_gridelts_coupsim)


      medusa4coupsim_setup_done = .TRUE.


#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_modprocname_a) 'End'
      WRITE(jp_stddbg, '()')
#endif


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE SETUP_MEDUSA_FOR_COUPSIM
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE GET_NLONSLATSZTS_FILE(nlons, nlats, nzts)
!-----------------------------------------------------------------------

      USE mod_defines_medusa,       ONLY: jp_stderr
      USE mod_execontrol_medusa,    ONLY: ABORT_MEDUSA

      IMPLICIT NONE


      INTEGER, INTENT(OUT) :: nlons
      INTEGER, INTENT(OUT) :: nlats
      INTEGER, INTENT(OUT) :: nzts


! Standard I/O related data
! -------------------------

      CHARACTER(LEN=*), PARAMETER :: cfmt_modprocname_a
     &  = '("[MOD_COUPSIM_MEDUSA_SETUP/GET_NLONSLATSZTS_FILE]: ", A)'

      CHARACTER(LEN=*), PARAMETER :: cfmt_a     = '(" - ", A)'

      IF (.NOT. medusa4coupsim_setup_done) THEN
        WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
        WRITE(jp_stderr, cfmt_a) 'Setup not yet done -- aborting.'
        CALL ABORT_MEDUSA()
      ENDIF

      nlons = nlons_file
      nlats = nlats_file
      nzts  = nzts_file

      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE GET_NLONSLATSZTS_FILE
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE GET_IMASK_OCEAN_FILE(imask)
!-----------------------------------------------------------------------

      USE mod_defines_medusa,       ONLY: jp_stderr
      USE mod_execontrol_medusa,    ONLY: ABORT_MEDUSA

      IMPLICIT NONE


      INTEGER, DIMENSION(:,:), INTENT(OUT) :: imask


! Standard I/O related data
! -------------------------

      CHARACTER(LEN=*), PARAMETER :: cfmt_modprocname_a
     &  = '("[MOD_COUPSIM_MEDUSA_SETUP/GET_IMASK_OCEAN_FILE]: ", A)'

      CHARACTER(LEN=*), PARAMETER :: cfmt_a     = '(" - ", A)'

      IF (.NOT. medusa4coupsim_setup_done) THEN
        WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
        WRITE(jp_stderr, cfmt_a) 'Setup not yet done -- aborting.'
        CALL ABORT_MEDUSA()
      ENDIF

      imask(:,:) = imask_ocean_file(:,:)

      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE GET_IMASK_OCEAN_FILE
!-----------------------------------------------------------------------

!=======================================================================
      END MODULE MOD_COUPSIM_MEDUSA_SETUP
!=======================================================================
