;####################################################################
;  SCRIPT FOR CALCULATING MULTIPLE STASTICS FOR
;  MULTIVARIABLE INTEGRATED EVALUATION (MVIE) of model performance
;  References:
;  Zhang et al., 2020, GMDD, https://doi.org/10.5194/gmd-2020-310
;  Xu et al., 2017, GMD, https://doi.org/10.5194/gmd-10-3805-2017
;  Xu et al., 2016, GMD, https://doi.org/10.5194/gmd-9-4365-2016
;  AFTER CALCULATION, VFE DIAGRAM / METRICS TABLE CAN BE PLOTED.
;******************************************************************
; The script assumes all variables derived from one model or observation
; are saved in one data file (str). If not, please merge all variables
; into one data file using the third party sofware, e.g., cdo 
; Users can modify the namelist at the beginning of this script
; based on the application.
;******************************************************************
; Please report bugs to: zhangmengzhuo1996@163.com or xuzhf@tea.ac.cn
;
;####################################################################
load "$NCARG_ROOT/lib/ncarg/nclscripts/csm/gsn_code.ncl"   
load "$NCARG_ROOT/lib/ncarg/nclscripts/csm/gsn_csm.ncl"
load "$NCARG_ROOT/lib/ncarg/nclscripts/csm/contributed.ncl"
load "./function/MVIETool_fun.ncl"
begin
;              **********************************
;              * namelist to be edited by users *
;              **********************************
;============== user's editing starts here ===========================================================
;------ configuration for input and output data --------------------------
  Inputdatadir    =            ;Model and observation data input directory (str)

 Model_filenames =            ;File names of observation/reanalysis data (array of str)

 Obs_filenames   =            ;File names of observation/reanalysis data (array of str) 

 Varname         =            ;List of Variables to be evaluated (str)
												;Names of variables composing a vector should be written together and put in parentheses seperated by comma
												;e.g., "T2m,precip,(u850,v850)"

 MVIE_filename   =            ;Output filename (str) [default: "MVIE_stats.nc"]

;------ configuration for data dimension and coordinates info  -------------------------
 Var_Coords      =  		      ;Evaluating specified part of data (True) or the whole [False: default]?

 isCoords_time   =      	 	;Variable has dimension and coordinate for time (True) or not [False: default]?
				                        ;valid when 'Var_Coords = True'    
 Coords_time     =       		;Name of time dimension (str) [default: "time"], valid when 'isCoords_time = True' 
 Range_time      =       		;Range of time in 'Coords_time'-variable (2-element array of str/num), valid when 'isCoords_time = True' 
                                    ;Specify sub-time to be evaluated
				                        ;if str, must have format 'YYYYMM','YYYYMMDD',or 'YYYYMMDDHH'

 isCoords_geo    = 		      ;Variable has dimensions and coordinates for lat/lon (True) or not [False: default]?
                                    ;valid when 'Var_Coords = True'   
 Coords_geo      =         	;Names of lat/lon dimension names (array of str), valid when 'isCoords_geo = True'  
 Range_geo       =         	;Range of lat/lon in 'Coords_geo'-variables (array of str),  valid when 'isCoords_geo = True'
                                    ;Specify subregion to be evaluated. e.g. (/"lat|0.:45","lon|0:180"/), valid when 'isCoords_geo = True'

 hasLevel        =            ;Variable has level dimension (str)[name of level dimension, e.g.:"lev"] or not [False: default]?
                                    ;The level coordinate is required to read data at specific level.
                                    ;valid when 'Var_Coords = True'   
 VarLev          =            ;Specify level for each variable (array of num conforms to 'Varname'), valid when setting 'hasLevel' with level dimension name
                                    ;if one variable without level dim, 'VarLev' of it will be omitted.

;------ configuration for area weight and variable weight ---------------
 Isarea_wgt      =      		;Considering area weighting (str)[latitude dimension name] or not [default: False] in the statistics?
  
 Wgt_var         = 			   ;Variable Weighting (1/array of num) [default: 1 - no weight]
				                        ;array of num: list of weighting for the corresponding variables listed in "Varname"

;------ configuration for statistical metrics ---------------------------
 ComMask_On     =             ;Create a common mask for datasets of all models and reference [True: default] to deal with
                                    ;missing points or unify missing points of each model-reference pair (False)? 
 Unify_VarMiss  =             ;Unify missing points across all variables of one model [True] or not [False: default]?    

 Typ_stat       = 		      ;Type of calculated statistics ["double"/"float": default] 

 Stats_mode     = 			   ;Mode of statistics: centered [0: default] / uncentered (1) 

 Cal_VME        =    		   ;Calculate VME(ME) [1: dafault] /MEVM (0) in centered mode 

 MISS_F         =       	   ;Parameter F in MISS (num) [default: 2]

 Print_stats_r  = 		      ;Print the range of statistics to screen [True: default] or not (False)? 

;============== user's editing ends here =============================================================

;*****************************************
;* Check for input information from user *
;*****************************************

   Header = (/"########################","#      MVIETool        #", \
             "########################"/)
   print_table([/Header/],"%s")
   
; Number of model 
   if (.not.isvar("Model_filenames")) then
      print("****Calculate_MVIE error: cannot find Model_filenames, and it must be set !")
      exit
   end if

   if ((.not.isstring(Model_filenames)).or.(dimsizes(dimsizes(Model_filenames)).ne.1)) then
      print("****Calculate_MVIE error: Model_filenames should be an 1D array of string !")
      exit
   end if

   Nmodel = dimsizes(Model_filenames)
   if (Nmodel .lt. 2) then
      print("****Calculate_MVIE error: num of Model_filenames should be greater than 1 !")
      exit
   end if  

; Number of observation/reanalysis
   if (.not.isvar("Obs_filenames")) then
      print("****Calculate_MVIE error: cannot find obs_filenames, and it must be set !")
      exit
   end if

   if ((.not.isstring(Obs_filenames)).or.(dimsizes(dimsizes(Obs_filenames)).ne.1)) then
      print("****Calculate_MVIE error: Obs_filenames should be an 1D array of string !")
      exit
   end if

   Nobs = dimsizes(Obs_filenames)

; Input directory
   if ((.not.isvar("Inputdatadir")).or.(.not.isstring(Inputdatadir)).or.(.not.isscalar(Inputdatadir))) then
      print("****Calculate_MVIE error: Inputdatadir must be set, and it should be a string !")
      exit
   else
      dir_split = str_split_by_length(Inputdatadir, (/strlen(Inputdatadir)-1,1/))
      if (dir_split(1).ne."/") then
         Inputdatadir = Inputdatadir + "/"
      end if
   end if
   
; Type of statistics
   if (.not.isvar("Typ_stat")) then
      Typ_stat = "float"
   else
      if ((.not.isstring(Typ_stat)).or.(.not.isscalar(Typ_stat)).or. \
         ((Typ_stat.ne."float").and.(Typ_stat.ne."double"))) then      
         print("****Calculate_MVIE error: Typ_stat is a string, and should be 'float' or 'double' !")
         exit
      end if
   end if

; Variables 
   if (.not.isvar("Varname")) then
      print("****Calculate_MVIE error: cannot find Varname, and it must be set !")
      exit
   end if

   if ((.not.isstring(Varname)).or.(.not.isscalar(Varname))) then
      print("****Calculate_MVIE error: Varname should be a string, and has format as "+ \
            "'var1,var2,(var3,var4),var5,(var6,var7,var8)' !")
      exit
   end if

   if (.not.isvar("Wgt_var")) then
      Wgt_var = 1
   else
      if ((.not.isnumeric(Wgt_var)).or.(dimsizes(dimsizes(Wgt_var)).ne.1)) then   
         print("****Calculate_MVIE error: if there are weights for var, "+ \
               "Wgt_var should be an array of numeric values, otherwise, Wgt_var is 1 !")
         exit
      else
         if (typeof(Wgt_var).ne.Typ_stat) then
            Wgt_var := totype(Wgt_var,Typ_stat)
         end if
      end if
   end if

   R := Get_Mvars( Varname, Wgt_var)
   Var_names = R[0]
   Nvar      = R[1]
   M         = R[2]
   Mvar_D    = R[3]
   M_wgt     = R[4]
   Varwgt_str= R[5]

   Data_D = sum(Mvar_D)
   
; Coordinate for variable 
   if ((.not.isvar("Var_Coords")).or.(.not.islogical(Var_Coords)) \
      .or.(.not.isscalar(Var_Coords))) then
      print("****Calculate_MVIE error: cannot find Var_Coords, or Var_Coords is not a logical value !")
      exit
   end if

   if (Var_Coords) then
      if (.not.isvar("isCoords_time")) then
         isCoords_time = False
      else
         if ((.not.islogical(isCoords_time)).or.(.not.isscalar(isCoords_time))) then
            print("****Calculate_MVIE error: isCoords_time should be a logical value (True/False) !")
            exit
         end if
     
         if (isCoords_time) then
            if (.not.isvar("Coords_time")) then
               isCoords_time@name = "time"
            else
               if ((.not.isstring(Coords_time)).or.(.not.isscalar(Coords_time))) then
                  print("****Calculate_MVIE error: Coords_time should be a string !")
                  exit
               else
                  isCoords_time@name = Coords_time
               end if
            end if

            if ((.not.isvar("Range_time")).or.(dimsizes(Range_time).ne.2)) then
               print("****Calculate_MVIE error: cannot find Range_time, "+ \
                  "or Range_time should be an array of 2 elements !")
               exit
            else
               if ((.not.isstring(Range_time)).and.(.not.isnumeric(Range_time))) then
                  print("****Calculate_MVIE error: Range_time should be two string or two numeric values !")
                  exit
               else
                  if (isstring(Range_time).and.(strlen(Range_time(0)).ne.strlen(Range_time(1)))) then
                     print("****Calculate_MVIE error: two strings in Range_time should have the same length !"+ \
                          "If Range_time is string, it should have format 'YYYYMM','YYYYMMDD',or 'YYYYMMDDHH'.")
                     exit
                  end if
                  isCoords_time@range = Range_time
               end if
            end if
         end if
      end if

      if (.not.isvar("isCoords_geo")) then
         isCoords_geo = False
      else
         if ((.not.islogical(isCoords_geo)).or.(.not.isscalar(isCoords_geo))) then
            print("****Calculate_MVIE error: isCoords_geo should be a logical value (True/False) !")
            exit
         end if
         
         if (isCoords_geo) then
            if (.not.isvar("Coords_geo")) then
               print("****Calculate_MVIE error: cannot find Coords_geo, "\
                     + "and it must be set under 'isCoords_geo=True' !")
               exit
            else
               if (.not.isstring(Coords_geo)) then
                  print("****Calculate_MVIE error: Coords_geo should be string !")
                  exit
               else
                  isCoords_geo@name = Coords_geo
               end if
            end if

            if ((.not.isvar("Range_geo")).or.(.not.isstring(Range_geo)).or. \
                  (dimsizes(Range_geo).ne.dimsizes(Coords_geo))) then
               print("****Calculate_MVIE error: cannot find Range_geo, or Range_geo is not string, "+\
                        "or Range_geo does not match Coords_geo!")
               exit
            end if

            Ngeo = dimsizes(Coords_geo)
            Geo_R = new((/2,Ngeo/),float)

            do i = 0,(Ngeo-1)
               igeo := str_split(Range_geo(i), "|")

               if ((dimsizes(igeo).ne.2).or.(dimsizes(str_split(igeo(1), ":")).ne.2)) then
                  print("****Calculate_MVIE error: Range_geo should have format as: "+\
                        "(/'lat|0:45','lon|0:180'/) !")
                  exit
               end if

               if (ismissing(ind(igeo(0).eq.Coords_geo))) then
                  print("****Calculate_MVIE error: cannot find "+igeo(0)+" of Range_geo in Coords_geo!")
                  exit
               else
                  ind_geo = ind(igeo(0).eq.Coords_geo)
               end if

               Geo_R(:,ind_geo) = tofloat(str_split(igeo(1), ":"))
            end do

            isCoords_geo@range = Geo_R
         end if
      end if

      if ((.not.isCoords_time).and.(.not.isCoords_geo)) then
         print("****Calculate_MVIE error: Under 'Var_Coords=True', "+\
               "isCoords_time and isCoords_geo should not be all False !")
         exit
      end if
   end if

; Levels for variables
   ; Levels for variables
   if (Var_Coords) then      
      if (.not.isvar("hasLevel")) then
         hasLevel = False
      else
         if (isstring(hasLevel).and.isscalar(hasLevel)) then
            if (.not.isvar("VarLev")) then
               print("****Calculate_MVIE error: cannot find VarLev !")
               exit
            else
               if (.not.isnumeric(VarLev) .or. (dimsizes(dimsizes(VarLev)).ne.1) \
                  .or. (dimsizes(VarLev).ne.Nvar)) then
                  print("****Calculate_MVIE error: VarLev should be numeric array, conforming to the dimsizes of 'Varname' !")
                  exit
               end if
               hasLevel@VarLev = VarLev
            end if
         else
            if (islogical(hasLevel)) then
               if ((.not.isscalar(hasLevel)) .or. hasLevel) then
                  print("****Calculate_MVIE error: hasLevel should be False or name of level coordinate for variable !")
                  exit
               end if
            else
               print("****Calculate_MVIE error: hasLevel should be False or name of level coordinate for variable !")
               exit
            end if
         end if
      end if
   end if

; Area weighting
   if ((.not.isvar("Isarea_wgt")).or.(.not.isCoords_geo)) then
      Isarea_wgt := False
   else 
      if ((.not.isscalar(Isarea_wgt)).or.((Isarea_wgt.ne.False).and.(.not.isstring(Isarea_wgt)))) then
         print("****Calculate_MVIE error: Isarea_wgt should be False or a string for name of latitude coordinate !")
         exit
      end if
      if (isstring(Isarea_wgt).and.ismissing(ind(Isarea_wgt.eq.Coords_geo))) then
         print("****Calculate_MVIE error: cannot find "+Isarea_wgt+" in Coords_geo !")
         exit
      end if
   end if

; Centered / Uncentered statistics
   if (.not.isvar("Stats_mode")) then
      Stats_mode = 0
   else
      if ((.not.isnumeric(Stats_mode)).or.(.not.isscalar(Stats_mode)) \
         .or.((Stats_mode.ne.0).and.(Stats_mode.ne.1))) then
         print("****Calculate_MVIE error: Stats_mode should be an integer (0/1) !")
         exit
      end if
   end if

; Use the common / unique mask for missing points across all models
   if (.not.isvar("ComMask_On")) then
      ComMask_On = True
   else
      if (.not.islogical(ComMask_On)) then
         print("****Calculate_MVIE error: ComMask_On should be logical (True/False) !")
         exit
      end if
   end if

; Unify missing points across all variables of one model
   if (.not.isvar("Unify_VarMiss")) then
      Unify_VarMiss = False
   else
      if (.not.islogical(Unify_VarMiss)) then
         print("****Calculate_MVIE error: Unify_VarMiss should be logical (True/False) !")
         exit
      end if
   end if

; Calculate VME(ME) / MEVM 
   if (Stats_mode.eq.0) then
      if (.not.isvar("Cal_VME")) then
         Cal_VME = 1
      else
         if ((.not.isnumeric(Cal_VME)).or.(.not.isscalar(Cal_VME)) \
            .or.((Cal_VME.ne.0).and.(Cal_VME.ne.1))) then
            print("****Calculate_MVIE error: Cal_VME should be an integer (0/1) !")
            exit
         end if
      end if
   end if

; Parameter F in MISS
   if (.not.isvar("MISS_F")) then
      MISS_F = 2.
   else
      if ((.not.isnumeric(MISS_F)).or.(.not.isscalar(MISS_F))) then
         print("****Calculate_MVIE error: MISS_F should be a numeric value !")
         exit
      end if
   end if

; Print range of stats
   if (.not.isvar("Print_stats_r")) then
      Print_stats_r = True
   else
      if ((.not.islogical(Print_stats_r)).or.(.not.isscalar(Print_stats_r))) then
         print("****Calculate_MVIE error: Print_stats_r should be a logical value (True/False) !")  
         exit
      end if
   end if

; Output filename (str)  [default: "MVIE_stats.nc"]
   if (.not.isvar("MVIE_filename")) then
      MVIE_filename = "MVIE_stats.nc"
   else
      if ((.not.isstring(MVIE_filename)).or.(.not.isscalar(MVIE_filename))) then
         print("****Calculate_MVIE error: MVIE_filename should be a string with format as 'MVIE.nc' !")
         exit
      end if
       
      if (strlen(MVIE_filename).lt.4) then
         MVIE_filename = MVIE_filename + ".nc"
      else
         filename_split = str_split_by_length(MVIE_filename, (/strlen(MVIE_filename)-3,3/))
         if (filename_split(1).ne.".nc") then
            MVIE_filename = MVIE_filename + ".nc"
         end if
      end if
   end if

;*****************
;* Read raw data *
;*****************

; Read data of model and obserations 
   Files = Inputdatadir+array_append_record(Model_filenames,Obs_filenames,0)

   if (Var_Coords) then
      do icase = 0,(Nmodel+Nobs-1)
         R := ReadVars_Coord(Files(icase),Var_names,isCoords_time,isCoords_geo,Isarea_wgt,hasLevel,Typ_stat)
         idata := R[0]

         if (icase.eq.0) then
            dim0 = dimsizes(idata)
            All_data = new((/Nmodel+Nobs,Data_D,dim0(1)/),Typ_stat)
         else
            if (any(dimsizes(idata).ne.dim0)) then
               print("****Calculate_MVIE error: dimsize of vars in "+ \
                     Files(icase)+" donot match that in "+Files(0)+" !")
               exit
            end if
         end if

         All_data(icase,:,:) = idata
         
         if (icase.eq.(Nmodel+Nobs-1)) then
            Area_wgt = R[1]
         end if
      end do
   else
      do icase = 0,(Nmodel+Nobs-1)
         idata := ReadVars(Files(icase),Var_names,Typ_stat)

         if (icase.eq.0) then
            dim0 = dimsizes(idata)
            All_data = new((/Nmodel+Nobs,Data_D,dim0(1)/),Typ_stat)
         else
            if (any(dimsizes(idata).ne.dim0)) then
               print("****Calculate_MVIE error: dimsizes of vars in "+ \
                     Files(icase)+" donot match these in "+Files(icase-1)+" !")
               exit
            end if
         end if

         All_data(icase,:,:) = idata
      end do
      Area_wgt = 1
   end if  

;********************************
;* Calculate Reference and Mask *
;********************************
   
   if (Unify_VarMiss) then
      Mask_obs = new(dim0(1),Typ_stat)
      Mask_obs = 0

      do iobs = 0,(Nobs-1)
         do iD = 0,(Data_D-1)
            Mask_obs = Mask_obs + All_data(Nmodel+iobs,iD,:) - All_data(Nmodel+iobs,iD,:)
         end do
      end do
   else
      Mask_obs = new((/Data_D,dim0(1)/),Typ_stat)
      Mask_obs = 0

      do iobs = 0,(Nobs-1)
         Mask_obs = Mask_obs + All_data(Nmodel+iobs,:,:) - All_data(Nmodel+iobs,:,:)
      end do
   end if

   if (ComMask_On) then
      if (Unify_VarMiss) then
         Com_Mask = new(dim0(1),Typ_stat)
         Com_Mask = 0

         do imodel = 0,(Nmodel-1)
            do iD = 0,(Data_D-1)
               Com_Mask = Com_Mask + All_data(imodel,iD,:) - All_data(imodel,iD,:)
            end do
         end do
      else
         Com_Mask = new((/Data_D,dim0(1)/),Typ_stat)
         Com_Mask = 0

         do imodel = 0,(Nmodel-1)
            Com_Mask = Com_Mask + All_data(imodel,:,:) - All_data(imodel,:,:)
         end do
      end if
      Mask_obs = Mask_obs + Com_Mask
   end if

   if (Unify_VarMiss) then
      Mask_obs := conform_dims((/Data_D,dim0(1)/), Mask_obs, 1)
   end if

   if (Nobs.eq.1) then
      Reference = All_data(Nmodel+Nobs-1,:,:) + Mask_obs
   else
      Reference = dim_avg_n(All_data(Nmodel:(Nmodel+Nobs-1),:,:),0) + Mask_obs
   end if
   
;*******************
;* Calsulate stats *
;*******************
         
   Ncase = where(Nobs.eq.1,Nmodel,Nmodel+Nobs)
 
   if (M .eq. 1) then  
      ;single variable evaluation   
      if (Data_D.eq.1) then 
         Header = (/"------------------------","|     Scalar field     |","------------------------"/)
      else
         Header = (/"------------------------","|     Vector field     |","------------------------"/)
      end if
      print_table([/Header/],"%s")

      if (Stats_mode.eq.0) then
         Header := "---( Centered stats )---"
      else
         Header := "--( Uncentered stats )--" 
      end if
      print_table([/Header/],"%s")

      if (Stats_mode.eq.0) then
         if (Data_D.eq.1) then
            Stats_names  = (/"CORR","SD","cRMSD","ME"/)
         else
            Stats_names  = (/"cVSC","cRMSL","cRMSVD",where(Cal_VME.eq.1,"VME","MEVM")/)
         end if
      else
         if (Data_D.eq.1) then
            Stats_names  = (/"uCORR","rms","RMSD"/)
         else
            Stats_names  = (/"VSC","RMSL","RMSVD"/)
         end if
      end if
     
      Nstats = dimsizes(Stats_names)
      Stats  = new((/Nstats,Ncase/),Typ_stat)

      Stats_other  = new((/2,Ncase/),Typ_stat)

      Skillscore_names = (/"S1_cen","S2_cen","S1_uncen","S2_uncen"/) 
      SkillS = new((/4,Ncase/),Typ_stat)

      if ((Stats_mode.eq.0).and.(Cal_VME.eq.0).and.(Data_D.eq.2)) then
         MEVD = new(Ncase,Typ_stat)
      end if
 
      do icase = 0,(Ncase-1)
         ;uniform missing points for each pair of model and reference
         R := unif_missing_p(All_data(icase,:,:),Reference,Unify_VarMiss)

	      Case_data := R[0]
	      Ref_data  := R[1]
      
         ;calculate stats
         Stats(0,icase) = VSCor(Case_data, Ref_data, Stats_mode, Area_wgt)
	      Stats(1,icase) = RMSLength(Case_data, Stats_mode, Area_wgt)/ \
                           RMSLength(Ref_data, Stats_mode, Area_wgt)
	      Stats(2,icase) = RMSVDiff(Case_data, Ref_data, Stats_mode, Area_wgt)/ \
                              RMSLength(Ref_data, Stats_mode, Area_wgt)

         Stats_other(0,icase) = VSCor(Case_data, Ref_data, 1-Stats_mode, Area_wgt)
         Stats_other(1,icase) = RMSLength(Case_data, 1-Stats_mode, Area_wgt)/ \
                                 RMSLength(Ref_data, 1-Stats_mode, Area_wgt)

         if (Stats_mode.eq.0) then
            Stats(3,icase) = VecMError(Case_data,Ref_data,Area_wgt,Cal_VME)/ \
                                 RMSLength(Ref_data, Stats_mode, Area_wgt)
            if ((Cal_VME.eq.0).and.(Data_D.eq.2)) then
               MEVD(icase) = MEVecDir(Case_data, Ref_data, Area_wgt)
            end if
         end if
      end do 

      if (Stats_mode.eq.0) then
         R := Skill_score2(Stats(1,:), Stats(0,:))
         SkillS(0,:) = R[0]
         SkillS(1,:) = R[1]

         R := Skill_score2(Stats_other(1,:), Stats_other(0,:))
         SkillS(2,:) = R[0]
         SkillS(3,:) = R[1]
      else
         R := Skill_score2(Stats(1,:), Stats(0,:))
         SkillS(2,:) = R[0]
         SkillS(3,:) = R[1]

         R := Skill_score2(Stats_other(1,:), Stats_other(0,:))
         SkillS(0,:) = R[0]
         SkillS(1,:) = R[1]
      end if

      ;print range of stats
      if (Print_stats_r) then
         Pri_stat_range1(Stats,SkillS,Stats_names)
      end if
   else
      ;multivariable integrated evaluation    
      Header = (/"------------------------","|  Multivariable field |","------------------------"/)
      print_table([/Header/],"%s")

      if (Stats_mode.eq.0) then
         Header := "---( Centered stats )---"
      else
         Header := "--( Uncentered stats )--"  
      end if
      print_table([/Header/],"%s")

      if (Stats_mode.eq.0) then
         ;centered stats	
         ;names of stats for individual variables
         Stats1_names  = (/"CORR", "SD", "cRMSD", "ME"/)

         ;mean error of vector direction between model and ref
         if (Cal_VME.eq.0) then 
            MEVD = new((/Ncase,M/),Typ_stat)
         end if

         ;names of stats for multivariable field
         Stats2_names  = (/"cVSC", "cRMSL", "SD_std", "cRMSVD", where(Cal_VME.eq.1,"VME","MEVM")/)

         ;rms, VSC for calculating uncentered MISS
         rms = new((/Ncase,M/),Typ_stat)
         VSC = new((/Ncase/),Typ_stat)
      else
         ;uncentered stats
         ;names of stats for individual variables
	      Stats1_names  = (/"uCORR", "rms", "RMSD"/)

         ;names of stats for multivariable field
         Stats2_names  = (/"VSC", "RMSL", "rms_std", "RMSVD"/)

         ;SD, cVSC for calculating centered MISS
         SD   = new((/Ncase,M/),Typ_stat)
         cVSC = new((/Ncase/),Typ_stat)
      end if

      Nstats1 = dimsizes(Stats1_names)
      Nstats2 = dimsizes(Stats2_names)

      Stats1 = new((/Nstats1,Ncase,M/),Typ_stat)
      Stats2 = new((/Nstats2,Ncase/),Typ_stat)

      MISS_names = (/"cMISS","uMISS"/)
      MISS = new((/2,Ncase/),Typ_stat)
      
      do icase = 0,(Ncase-1)
         ;uniform missing points for each pair of model and reference  
         R := unif_missing_p(All_data(icase,:,:),Reference,Unify_VarMiss)
	      Case_data := R[0]
	      Ref_data  := R[1]
       
         ;calculate stats for individual variables
         iD = 0                
         do ivar = 0,(M-1) 
            ;CORR/uCORR
	         Stats1(0,icase,ivar) = VSCor(Case_data(iD:(iD+Mvar_D(ivar)-1),:), \
                                    Ref_data(iD:(iD+Mvar_D(ivar)-1),:),Stats_mode,Area_wgt)
            ;SD/rms
	         Stats1(1,icase,ivar) = RMSLength(Case_data(iD:(iD+Mvar_D(ivar)-1),:),Stats_mode,Area_wgt)/ \
   	                              RMSLength(Ref_data(iD:(iD+Mvar_D(ivar)-1),:),Stats_mode,Area_wgt)
            ;cRMSD/RMSD
	         Stats1(2,icase,ivar) = RMSVDiff(Case_data(iD:(iD+Mvar_D(ivar)-1),:), \
                                       Ref_data(iD:(iD+Mvar_D(ivar)-1),:),Stats_mode,Area_wgt)/ \
	                                    RMSLength(Ref_data(iD:(iD+Mvar_D(ivar)-1),:),Stats_mode,Area_wgt)

            if (Stats_mode.eq.0) then
	            ;ME	
	            Stats1(3,icase,ivar) = VecMError(Case_data(iD:(iD+Mvar_D(ivar)-1),:), \
                                       Ref_data(iD:(iD+Mvar_D(ivar)-1),:),Area_wgt,Cal_VME) / \
	                                    RMSLength(Ref_data(iD:(iD+Mvar_D(ivar)-1),:),Stats_mode,Area_wgt)
               ;MEVD
	            if ((Cal_VME.eq.0).and.(Mvar_D(ivar).eq.2)) then
	               MEVD(icase,ivar) = MEVecDir(Case_data(iD:(iD+Mvar_D(ivar)-1),:), \
                                       Ref_data(iD:(iD+Mvar_D(ivar)-1),:),Area_wgt)
	            end if
               ;rms
               rms(icase,ivar) = RMSLength(Case_data(iD:(iD+Mvar_D(ivar)-1),:),1,Area_wgt)/ \
   	                              RMSLength(Ref_data(iD:(iD+Mvar_D(ivar)-1),:),1,Area_wgt)	      
	         else
               ;SD
               SD(icase,ivar) = RMSLength(Case_data(iD:(iD+Mvar_D(ivar)-1),:),0,Area_wgt)/ \
   	                              RMSLength(Ref_data(iD:(iD+Mvar_D(ivar)-1),:),0,Area_wgt)	
            end if  
            	   
            iD = iD + Mvar_D(ivar)	          
         end do
  
         ;normalization and add variable weighting for multivariable field
         M_nor = Nor_matrix(Ref_data,Mvar_D,M_wgt,Area_wgt)
         
         Case_data = Case_data * M_nor
	      Ref_data  = Ref_data * M_nor
         
         ;calculate stats for mutivariable intergrated field
         ;cVSC/VSC
	      Stats2(0,icase) = VSCor(Case_data,Ref_data,Stats_mode,Area_wgt)

         ;cRMSL/RMSL
	      Stats2(1,icase) = RMSLength(Case_data,Stats_mode,Area_wgt) \
                              / RMSLength(Ref_data,Stats_mode,Area_wgt)
         ;SD_std/rms_std
         if (dimsizes(M_wgt).eq.1) then
	         Stats2(2,icase) = sqrt(dim_avg((Stats1(1,icase,:)-dim_avg(Stats1(1,icase,:)))^2))
         else
            Stats2(2,icase) = Std_varWgt(Stats1(1,icase,:),M_wgt)
         end if

         ;cRMSVD/RMSVD
	      Stats2(3,icase) = RMSVDiff(Case_data,Ref_data,Stats_mode,Area_wgt) \
                              /RMSLength(Ref_data,Stats_mode,Area_wgt)

         if (Stats_mode.eq.0) then
            ;VME/MEVM                                     	      
	         Stats2(4,icase) = VecMError(Case_data,Ref_data,Area_wgt,Cal_VME)/ \
                                 RMSLength(Ref_data,Stats_mode,Area_wgt)	      
            ;VSC
            VSC(icase) = VSCor(Case_data,Ref_data,1,Area_wgt)

            ;centerted MISS
            MISS(0,icase) = MISScore(Stats1(1,icase,:),Stats2(0,icase),M_wgt,MISS_F) 
            ;uncentered MISS
            MISS(1,icase) = MISScore(rms(icase,:),VSC(icase),M_wgt,MISS_F) 
         else
            ;cVSC
            cVSC(icase) = VSCor(Case_data,Ref_data,0,Area_wgt)

            ;uncenterted MISS
            MISS(1,icase) = MISScore(Stats1(1,icase,:),Stats2(0,icase),M_wgt,MISS_F) 
            ;centered MISS
            MISS(0,icase) = MISScore(SD(icase,:),cVSC(icase),M_wgt,MISS_F) 
         end if
      end do

      ;print Var wgt
      if (dimsizes(M_wgt).gt.1) then
         format_Var = "%-"+max(strlen(Varwgt_str))+"s%4.3f"
         Var_list = [/Varwgt_str,M_wgt/]

         print_table([/(/"Var Weight:============="/)/],"%s")
         print_table(Var_list, format_Var)  
      end if

      ;print range of stats
      if (Print_stats_r) then
         if (Stats_mode.eq.1) then
            Pri_stat_range2(Stats1,Stats2,MISS,0)
         else
            Pri_stat_range2(Stats1,Stats2,MISS,Cal_VME)
         end if
      end if     
   end if 

;********************************************
;* Save results of MVIE in MVIE_filename.nc *
;********************************************

; Create MVIE_filename.nc
   if (ismissing(systemfunc("ls " + MVIE_filename ))) then           
      print_table([/"***** Create NetCDF file: <" + MVIE_filename + "> *****"/],"%s")  
   else
      system("rm " + MVIE_filename)
      print_table([/"***** Delete existing <" + MVIE_filename +  \
            "> ,and create <" + MVIE_filename + "> *****"/],"%s")
   end if

   f_MVIE = addfile(MVIE_filename, "c")

; Save results 
   if (M .eq. 1) then
      ;single scalar/vector field evaluation
      filedimdef(f_MVIE,"var",1,False)
      filedimdef(f_MVIE,"case",Ncase,False)
      filevardef(f_MVIE,"case","integer","case")
      caseAtt = 0
      caseAtt@long_name = "Count for models or/and observations evaluated"
      filevarattdef(f_MVIE,"case",caseAtt)
      f_MVIE->case = (/ispan(1,Ncase,1)/)

      Stats_longname = new((/2,4/), "string")
      if (Data_D.eq.1) then
         Stats_longname(0,:) = (/"Correlation Coefficient","Standard Deviation", \
                                 "Centered Root Mean Square Difference","Mean Error"/)
         Stats_longname(1,0:2) =  (/"Uncentered Correlation Coefficient", \
                                    "Root Mean Square","Root Mean Square Difference"/)

         Scores_longname = (/"Centered Skill score1 for scalar field", \
                             "Centered Skill score2 for scalar field", \
                             "Uncentered Skill score1 for scalar field", \
                             "Uncentered Skill score2 for scalar field"/)

         Stats_units = (/"divided by SD of reference","divided by rms of reference"/)
      else                    
         Stats_longname(0,:) = (/"Centered Vector Similarity Coefficient","Centered Root Mean Square Length", \
                                 "Centered Root Mean Square Vector Difference", "Vector Mean Error"/)
         if (Stats_mode.eq.0) then
            if (Cal_VME.eq.1) then
               Stats_longname(0,3) = "Magnitude error of vector mean"
            end if
         end if

         Stats_longname(1,0:2) = (/"Vector Similarity Coefficient","Root Mean Square Length", \
                                   "Root Mean Square Vector Difference"/)

         Scores_longname = (/"Centered Skill Score1 for vector field", \
                             "Centered Skill Score2 for vector field", \
                             "Uncentered Skill Score1 for vector field", \
                             "Uncentered Skill Score2 for vector field"/)

         Stats_units = (/"divided by cRMSL of reference","divided by RMSL of reference"/)
      end if
      
      statAtt = 0
      do istat = 0,(Nstats-1)
         filevardef(f_MVIE,Stats_names(istat),Typ_stat,"case")

         if (istat.gt.0) then
            statAtt@units = Stats_units(Stats_mode)
         end if
         statAtt@long_name = Stats_longname(Stats_mode,istat)
         filevarattdef(f_MVIE,Stats_names(istat),statAtt)

         f_MVIE->$Stats_names(istat)$ = (/Stats(istat,:)/)
      end do

      statAtt := 0
      do istat = 0,3
         filevardef(f_MVIE,Skillscore_names(istat),Typ_stat,"case")

         statAtt@long_name = Scores_longname(istat)
         filevarattdef(f_MVIE,Skillscore_names(istat),statAtt)

         f_MVIE->$Skillscore_names(istat)$ = (/SkillS(istat,:)/)
      end do
   else
      ;MVIE
      filedimdef(f_MVIE,(/"case","var"/),(/Ncase,M/),(/False,False/))

      filevardef(f_MVIE,"case","integer","case")
      caseAtt = 0
      caseAtt@long_name = "Count for models or/and observations evaluated"
      filevarattdef(f_MVIE,"case",caseAtt)
      f_MVIE->case = (/ispan(1,Ncase,1)/)

      filevardef(f_MVIE,"var","integer","var")
      varAtt = 0
      varAtt@long_name = "Count for individual scalar/vector variables evaluated"
      filevarattdef(f_MVIE,"var",varAtt)
      f_MVIE->var = (/ispan(1,M,1)/)
   
      Stats1_longname = new((/2,4/),"string")
      Stats1_longname(0,:) = (/"Correlation Coefficient","Standard Deviation", \
                               "Centered Root Mean Square Difference","Mean Error"/)
      Stats1_longname(1,0:2) = (/"Uncentered Correlation Coefficient", \
                                 "Root Mean Square","Root Mean Square Difference"/)
      Stats1_units = (/"divided by SD of reference","divided by rms of reference"/)

      statAtt = 0 
      do istat = 0,(Nstats1-1)   
         filevardef(f_MVIE,Stats1_names(istat),Typ_stat,(/"case","var"/))

         if (istat.gt.0) then
            statAtt@units = Stats1_units(Stats_mode)
         end if  
         statAtt@long_name = Stats1_longname(Stats_mode,istat)   
         filevarattdef(f_MVIE,Stats1_names(istat),statAtt) 

         f_MVIE->$Stats1_names(istat)$ = (/Stats1(istat,:,:)/)
      end do

      Stats2_longname = new((/2,5/),"string")
      Stats2_longname(0,:)   = (/"Centered Vector Similarity Coefficient","Centered Root Mean Square Length", \
                                "standard deviation of SD","Centered Root Mean Square Vector Difference", \
                                "Vector Mean Error"/)
      if (Stats_mode.eq.0) then
         if (Cal_VME.eq.1) then
            Stats2_longname(0,4) = "Mean error of vector magnitude"
         end if
      end if

      Stats2_longname(1,0:3) = (/"Vector Similarity Coefficient","Root Mean Square Length","standard deviation of rms", \
                                 "Root Mean Square Vector Difference"/) 
      Stats2_units = (/"divided by cRMSL of reference","divided by RMSL of reference"/)                                                            
      
      statAtt := 0 
      do istat = 0,(Nstats2-1)
         filevardef(f_MVIE,Stats2_names(istat),Typ_stat,"case")

         if ((istat.eq.1).or.(istat.gt.2)) then
            statAtt@units = Stats2_units(Stats_mode)
         else
            statAtt@units = " "
         end if
         statAtt@long_name = Stats2_longname(Stats_mode,istat)
         filevarattdef(f_MVIE,Stats2_names(istat),statAtt)

         f_MVIE->$Stats2_names(istat)$ = (/Stats2(istat,:)/)
      end do

      MISS_longname = (/"Centered Multivariable Integrated Skill Score", \
                        "Uncentered Multivariable Integrated Skill Score"/)  
      statAtt := 0
      do istat = 0,1
         filevardef(f_MVIE,MISS_names(istat),Typ_stat,"case")

         statAtt@long_name = MISS_longname(istat)
         filevarattdef(f_MVIE,MISS_names(istat),statAtt)

         f_MVIE->$MISS_names(istat)$ = (/MISS(istat,:)/)
      end do
   end if

   ;MEVD
   if (isvar("MEVD")) then
      if (M.eq.1) then
         filevardef(f_MVIE,"MEVD",Typ_stat,"case")
      else
         filevardef(f_MVIE,"MEVD",Typ_stat,(/"case","var"/))
      end if

      statAtt@long_name = "Mean error of vector direction between model and ref"
      filevarattdef(f_MVIE,"MEVD",statAtt)

      f_MVIE->MEVD = (/MEVD/)
   end if 
   
;####################################################################
end



