#################################################################
import Ngl
import Nio
import os
import math
import numpy as np
#################################################################
# Function to plot VFE diagram
#################################################################
# Arguments:
#  wks       : graphic workstation
#  RATIO_raw : ratio of amplitude of model/observation/reanalysis
#              to that of reference
#  Rv_raw    : similarity cofficient between reference and
#              model/observation/reanalysis
#  nmodel    : number of model
#  nCase     : number of model and obs/reanalysis
#  opt_VFE   : Ngl.Resource()
#################################################################
# plot attribution in opt_VFE and defalut:
# opt_VFE.xyMax                     = 1.65             ## max value of axis (int/float) 
# opt_VFE.xyFontHeightF             = 0.0175           ## font height for axis (int/float)) 
# opt_VFE.OneX                      = "REF"            ## X-axis label at one point (str) 
# opt_VFE.tiYAxisString             = "RMSL"           ## Y-axis string (str)  
# opt_VFE.CorLabel                  = "Similarity"     ## Arc label (str)
# opt_VFE.stnRad                    = (not draw)       ## standard radius for RMSL (list/tuple consisting of int/float)
# opt_VFE.ccRays                    = (not draw)       ## radial lines for VSC (list/tuple consisting of int/float)
# opt_VFE.centerDiffRMS             = False            ## whether to draw radii from REF (bool) 
# opt_VFE.MarkersOn                 = False            ## whether to draw markers representing model or/and obs (bool)
# opt_VFE.MarkersIndex              = [4,8]            ## marker types for models and observations/reanalysis (list/tuple consisting of 2 int/float)
# opt_VFE.gsMarkerThicknessF        = 1.0              ## marker thinckness (int/float)
# opt_VFE.gsMarkerSizeF             = 0.0085           ## marker size (int/float)
# opt_VFE.Colors                    =                  ## colors of points (list/tuple)
#                                                      ### [1] of named colors (str) [2] of NhlTColorIndex (int:-1~255) [3] of RGB/RGBA map 
# opt_VFE.CountTxOn                 = True             ## whether to add count text for points (bool)
# opt_VFE.CountFontHeightF          = 0.0085           ## textfont height for count of points (int/float)
# opt_VFE.CountTxXOffset            = 0.015            ## X distance between point markers and point text (int/float)
# opt_VFE.CountTxYOffset            = 0.015            ## Y distance between point markers and point text (int/float)
# opt_VFE.DatatypeLabelsOn          = False            ## whether to draw data type labels (logical)
# opt_VFE.DatatypeLabels            = ["Models","Renalysis"]
#                                                      ## labels of data type (list/tuple of 1/2 str) 
# opt_VFE.DatatypeLegendFontHeightF = 0.05             ## data type legend font height (int/float) 
# opt_VFE.DatatypeLegendWidth       = 0.1              ## width of data type legend (int/float)
# opt_VFE.DatatypeLegendHeight      = 0.1              ## height of data type legend (int/float)
# opt_VFE.DatatypeLegendXL          = 0.8              ## X strat location of data type legend (int/float)
# opt_VFE.DatatypeLegendYL          = 0.9              ## Y strat location of data type legend (int/float)
# opt_VFE.caseLabels                = (not draw)       ## case names for labels (list/tuple of str  or False) 
# opt_VFE.caseLabelsFontHeightF     = 0.013            ## font height of caselabels (int/float)
# opt_VFE.caseLabelsYinterval       = 0.06             ## leading of caselabels (int/float)
# opt_VFE.caseLabelsXloc            = 0.05             ## X strat location of caselabels (int/float)
# opt_VFE.caseLabelsYloc            = 0.3              ## Y strat location of caselabels (int/float)
# opt_VFE.stdline                   = (not draw)       ## whether to plot rms_std/SD_std line or not 
# opt_VFE.draw_grid                 = False            ## whether to draw the grids on the wks (bool)
# opt_VFE.obsUncer                  = (no value)       ## a value to measure observational uncertainty (int/float) 
                                                       ### which is as the length of red horizontal bar centered at (1,0)
#################################################################
def VFE_diagram(wks, RATIO_raw, Rv_raw, nmodel, nCase, opt_VFE):
    try:
        RATIO = np.array(RATIO_raw,float)
        Rv = np.array(Rv_raw,float)
    except:
        print("VFE diagram error: error in input RATIO_raw and Rv_raw (cannot translate numpy.array)!")
        os._exit(0)

    # check error for input data
    if (not isinstance(nmodel,int)) or (not isinstance(nCase,int)):
        print("VFE diagram error: 'nmodel' and 'nCase' should be int!")
        os._exit(0)
    if nmodel > nCase:
        print("VFE diagram error: 'nmodel' cannot be greater than 'nCase'!")
        os._exit(0)
    if (RATIO.ndim!=1) or (Rv.ndim!=1):
        print("VFE diagram error: 'Ratio' and 'Rv' should be 1D array!")
        os._exit(0)
    if (Rv.shape[0]<nCase) or (RATIO.shape[0]<nCase):
        print("VFE diagram error: the shape of 'Ratio' and/or 'Rv' does not match 'nCase'!")
        os._exit(0)
    if type(opt_VFE) != type(Ngl.Resources()):
        print("VFE diagram error: 'opt_VFE' should be Ngl.Resources()!")
        os._exit(0)
    
    RATIO = RATIO[0:nCase]
    Rv = Rv[0:nCase]

    if any(np.isnan(RATIO)) or any(np.isnan(Rv)):
        print("VFE diagram error: there is Nan-value in 'Ratio' or/and 'Rv' of which index is form 0 to (Ncase-1)!")
        os._exit(0)

    # compute location for points
    rad = np.arccos(Rv)
    X   = RATIO * np.cos(rad)     
    Y   = RATIO * np.sin(rad)

    # compute location for std line
    if hasattr(opt_VFE,"stdline"):
        try:
            STD = np.array(opt_VFE.stdline,float)
        except:
            print("VFE diagram error: 'stdline' of opt_VFE cannot translate to numpy.array!")
            os._exit(0)
        if (STD.ndim!=1) or (STD.shape[0]<nCase):
            print("VFE diagram error: 'stdline' of opt_VFE is not 1-D array or its length is less than 'nCase'!")
            os._exit(0)
        STD = STD[0:nCase]

        X1 = (RATIO - STD) * np.cos(rad)  
        Y1 = (RATIO - STD) * np.sin(rad)   
        X2 = (RATIO + STD) * np.cos(rad)  
        Y2 = (RATIO + STD) * np.sin(rad)   

    # compute max for abscissa
    xyMin  = 0.  
    xyOne  = 1.00

    # set max of coordinate
    if hasattr(opt_VFE,"xyMax"):
        if not isinstance(opt_VFE.xyMax,(int,float)):
            print("VFE diagram error: 'xyMax' of opt_VFE should be int/float!")
            os._exit(0)
        xyMax = np.float(opt_VFE.xyMax)
        
        if xyMax < 1.65:
            xyMax = 1.65
        else:
            if xyMax < 2.:
                xyMax = 2.
    else:
        if np.max(RATIO) < 1.65:
            xyMax = 1.65
        else:
            if np.max(RATIO) < 2.:
                xyMax = 2.
            else:
                xyMax = np.ceil(np.max(RATIO)*10.)/10.

    if xyMax == 1.65:
        XB_V  = np.array((0.,0.25,0.5,0.75,1.,1.25,1.5),float)
    elif xyMax == 2:
        XB_V  = np.array((0.,0.25,0.5,0.75,1.,1.25,1.5,1.75),float)
    else:
        XB_V  = np.array((0.,0.25,0.5,0.75,1.,1.25,1.5,1.75,2.),float)

    # paneling purposes
    xyMax_Panel = xyMax+ 0.10

    # specify size of font
    if hasattr(opt_VFE, "xyFontHeightF"):
        if not isinstance(opt_VFE.xyFontHeightF,(int,float)):
            print("VFE diagram error: 'xyFontHeightF' of opt_VFE should be int/float!")
            os._exit(0)
        FontHeightF  = np.float(opt_VFE.xyFontHeightF)
    else:
        FontHeightF  = 0.0175
    
    #=============================================Part 1:base plot
    # set attr of radii (x/y)
    rxy                   = Ngl.Resources()       
    rxy.nglDraw           = False
    rxy.nglFrame          = False          # Don't advance the frame.
  
    rxy.vpHeightF         = 0.65
    rxy.vpWidthF          = 0.65

    rxy.tmYLBorderOn      = False
    rxy.tmXBBorderOn      = False
    rxy.tmYRBorderOn      = False
    rxy.tmYROn            = False         # Turn off right tick marks.
    rxy.tmXTBorderOn      = False
    rxy.tmXTOn            = False         # Turn off right tick marks.
    rxy.tmXBMode          = "Explicit"
  
    rxy.tmXBValues        = XB_V                                    
    XB_L                  = [str(x) for x in XB_V.tolist()]     
    XB_L[0]               = "   "
    XB_L[4]               = "REF"
    if hasattr(opt_VFE,"OneX"):
        if not isinstance(opt_VFE.OneX,str):
            print("VFE diagram error: 'OneX' of opt_VFE should be str!")
            os._exit(0)
        XB_L[4] = opt_VFE.OneX
  
    rxy.tmXBLabels           = XB_L
    rxy.tmXBMajorLengthF     = 0.015                                   
    rxy.tmXBLabelFontHeightF = FontHeightF
    rxy.tmXBMinorOn          =  False
    rxy.tmYLMode             = "Manual"
    rxy.tmYLMinorOn          = False
    rxy.tmYLMajorLengthF     = rxy.tmXBMajorLengthF
    rxy.tmYLLabelFontHeightF = FontHeightF
    rxy.tmYLMode             = "Explicit" 
    rxy.tmYLValues           = XB_V 
    rxy.tmYLLabels           = ["{:.2f}".format(x) for x in XB_V.tolist()]  

    if hasattr(opt_VFE,"tiYAxisString"):
        if not isinstance(opt_VFE.tiYAxisString,str):
            print("VFE diagram error: 'tiYAxisString' of opt_VFE should be str!")
            os._exit(0)
        rxy.tiYAxisString     = opt_VFE.tiYAxisString
    else:
        rxy.tiYAxisString     = "RMSL"
  
    rxy.tiYAxisFontHeightF  = FontHeightF                                           
    rxy.trXMaxF             = xyMax_Panel
    rxy.trYMaxF             = xyMax_Panel
    rxy.xyDashPatterns      = 0.                   # line characteristics (dash,solid)  
    rxy.xyLineThicknesses   = 2.                   # choose line thickness

    if hasattr(opt_VFE, "tiMainString"):
        if not isinstance(opt_VFE.tiMainString,str):
            print("VFE diagram error: 'tiMainString' of opt_VFE should be str!")
            os._exit(0)
        rxy.tiMainString      = opt_VFE.tiMainString
    
        if hasattr(opt_VFE, "tiMainFontHeightF"):
            if not isinstance(opt_VFE.tiMainFontHeightF,(int,float)):
                print("VFE diagram error: 'tiMainFontHeightF' of opt_VFE should be int/float!")
                os._exit(0)
            rxy.tiMainFontHeightF = np.float(opt_VFE.tiMainFontHeightF)
        else:
            rxy.tiMainFontHeightF = 0.0225              
                                                 
    # create outer 'correlation axis'
    npts    = 200                                            # arbitrary
    xx      = np.linspace(xyMin,xyMax,npts) 
    yy      = np.sqrt(xyMax**2 - xx**2)                      # outer correlation line (xyMax)

    # create and draw XY plot.
    VFE  = Ngl.xy(wks,xx,yy,rxy)
    
    # draw x and y to xyMax
    rsrRes  = Ngl.Resources()
    rsrRes.gsLineThicknessF  = rxy.xyLineThicknesses         # line thickness
    rsrRes.gsLineDashPattern = 0                             # solid line pattern
  
    dum0 = Ngl.add_polyline(wks, VFE, np.array((0.,0.)), np.array((0.,xyMax)), rsrRes)
    dum1 = Ngl.add_polyline(wks, VFE, np.array((0.,xyMax)), np.array((0.,0.)), rsrRes)

    # draw stn Rad
    xx = Ngl.fspan(xyMin, xyOne ,npts)                       # draw 1.0 standard radius
    yy = np.sqrt(xyOne - xx**2)
  
    rsrRes.gsLineDashPattern = 1.                            # dashed line pattern  
    rsrRes.gsLineThicknessF  = rxy.xyLineThicknesses         # line thickness
  
    dum2 = Ngl.add_polyline(wks, VFE, xx, yy, rsrRes)
  
    del xx,yy

    rsrRes.gsLineThicknessF  = 1.

    if hasattr(opt_VFE, "stnRad"):
        stnRad = opt_VFE.stnRad
        if (not isinstance(stnRad,(list,tuple))) or \
            (not all(isinstance(x,(int,float)) for x in stnRad)):
            print("VFE diagram error: 'stnRad' of opt_VFE should be list/tuple of which elments are int/float!")
            os._exit(0)

        rsrRes.gsLineThicknessF  = 1 
        dum3 = []

        for i in range(len(stnRad)):
            if stnRad[i]<xyMax:
                rr = np.float(stnRad[i])
                xx = Ngl.fspan(xyMin, rr, npts) 
                yy = np.sqrt(rr**2-xx**2)

                dum3.append(Ngl.add_polyline(wks, VFE, xx, yy, rsrRes))
        del xx,yy

    #====================================Part 2:Correlation labels
    # plot Rv Arc label
    txRes               = Ngl.Resources()                     
    txRes.txFontHeightF = FontHeightF                
    txRes.txAngleF      = -45.

    if hasattr(opt_VFE,"CorLabel"):
        if not isinstance(opt_VFE.CorLabel,str):
            print("VFE diagram error: 'CorLabel' of opt_VFE should be str!")
            os._exit(0)
        dum4 = Ngl.add_text(wks,VFE,opt_VFE.CorLabel,xyMax_Panel/np.sqrt(2.)+0.1,\
            xyMax_Panel/np.sqrt(2.)+0.1,txRes)
    else:
        dum4 = Ngl.add_text(wks,VFE,"Similarity",xyMax_Panel/np.sqrt(2.)+0.1,\
            xyMax_Panel/np.sqrt(2.)+0.1,txRes)

    # plot correlation labels
    sLabels = ["0.0","0.1","0.2","0.3","0.4","0.5","0.6",\
        "0.7","0.8","0.9","0.95","0.99","1.0"]
           
    cLabels = np.array([float(x) for x in sLabels],float)

    radC    = np.arccos(cLabels)
    angC    = np.degrees(radC)
    rad_max = xyMax                                 

    xC      = rad_max * np.cos(radC)
    yC      = rad_max * np.sin(radC)
    xC      = xC + 0.020 * np.cos(radC)
    yC      = yC + 0.060 * np.sin(radC)

    txRes.txAngleF      = 0.0 
    txRes.txFontHeightF = FontHeightF*0.50          
    txRes.txJust        = "CenterLeft"              
    txRes.txFontHeightF = FontHeightF               

    plRes  = Ngl.Resources()
    plRes.gsLineThicknessF = 2.
 
    tmEnd = 0.975
    radTM = xyMax*tmEnd 

    dum5 = []
    dum6 = []                            

    for i in range(len(sLabels)):
        txRes.txAngleF = angC[i]

        dum5.append(Ngl.add_text(wks, VFE, sLabels[i], xC[i], yC[i], txRes))

        xTM = (xyMax*math.cos(radC[i]), radTM*math.cos(radC[i]))
        yTM = (xyMax*math.sin(radC[i]), radTM*math.sin(radC[i]))

        dum6.append(Ngl.add_polyline(wks, VFE, xTM, yTM, plRes))
                                                  
    mTM = np.array((0.05,0.15,0.25,0.35,0.45,0.55,0.65,0.75,0.85,0.91,0.92,0.93,0.94,0.96,0.97,0.98),float)
             
    rad_mTM  = np.arccos(mTM)                        # angles: correlation labels
    end_mTM  = xyMax * (1. - (1. - tmEnd) * 0.5)     # radius end: minor TM 

    dum7 = []

    for i in range(len(mTM)):
        xTM = (xyMax*math.cos(rad_mTM[i]), end_mTM*math.cos(rad_mTM[i]))
        yTM = (xyMax*math.sin(rad_mTM[i]), end_mTM*math.sin(rad_mTM[i]))

        dum7.append(Ngl.add_polyline(wks, VFE, xTM, yTM, plRes))

    if hasattr(opt_VFE, "ccRays"):
        if (not isinstance(opt_VFE.ccRays,(list,tuple))) or \
            (not all(isinstance(x,(int,float)) for x in opt_VFE.ccRays)):
            print("VFE diagram error: 'ccRays' of opt_VFE should be list/tuple of which elments are int/float!")
            os._exit(0)
        ccRays = np.array(opt_VFE.ccRays)
        radRL = np.arccos(ccRays[np.logical_and(ccRays>0.,ccRays<1.)])

        rlRes = Ngl.Resources()
        rlRes.gsLineDashPattern = 2.                  # line pattern
        rlRes.gsLineThicknessF  = 1.                  # choose line thickness      
        rlRes.gsLineColor       =  "Gray58"

        dum8 = []

        for i in range(len(radRL)):
            xRL = xyMax * math.cos(radRL[i])
            yRL = xyMax * math.sin(radRL[i])

            dum8.append(Ngl.add_polyline(wks, VFE, (0,xRL), (0,yRL), rlRes))

    #=======================Part 3:Concentric about 1.0 on XB axis
    # draw radii from 1.0 in x-axis
    if hasattr(opt_VFE,"centerDiffRMS"):
        if not isinstance(opt_VFE.centerDiffRMS,bool):
            print("VFE diagram error: 'centerDiffRMS' of opt_VFE should be bool!")
            os._exit(0)

        if opt_VFE.centerDiffRMS:
            respl                    = Ngl.Resources()     # polyline mods desired
            respl.gsLineThicknessF   = 1.0                 # line thickness
            respl.gsLineDashPattern  = 2                   # short dash lines
            respl.gsLineColor        = "Black"             # line color    

            if hasattr(opt_VFE,"centerDiffRMScolor"):
                respl.gsLineColor    =  "LightGray" 
          
            dx   = 0.25                                     
            npts = 100                                     
            radR = np.radians(Ngl.fspan(180,360,npts))

            dum9 = []

            if xyMax == 1.65:
                ncon = 4

                for i in range(ncon):
                    rr = i*dx  
                    xx = 1.+rr*np.cos(radR)
                    yy = np.abs(rr*np.sin(radR))

                    if i < 2:
                        dum9.append(Ngl.add_polyline(wks,VFE,xx,yy,respl))
                    if i == 3:
                        n3 = np.int32(0.77*npts)
                        dum9.append(Ngl.add_polyline(wks,VFE,xx[0:(n3+1)],yy[0:(n3+1)],respl))
                    if i == 4:
                        n4 = np.int32(0.61*npts)
                        dum9.append(Ngl.add_polyline(wks,VFE,xx[0:(n4+1)],yy[0:(n4+1)],respl))

            elif (xyMax>=2.) and (xyMax<2.25):
                ncon = 4

                for i in range(ncon):
                    rr = i*dx
                    xx = 1.+rr*np.cos(radR)
                    yy = abs(rr*np.sin(radR))

                    dum9.append(Ngl.add_polyline(wks,VFE,xx,yy,respl))

            elif xyMax > 2.25:
                ncon = np.int32((np.max(XB_V) - 1.)/0.25)

                for i in range(ncon):
                    rr = i*dx
                    xx = 1.+rr*np.cos(radR)
                    yy = np.abs(rr*np.sin(radR))

                    if i < 4:
                        dum9.append(Ngl.add_polyline(wks,VFE,xx,yy,respl))
                    else:
                        n_s = np.int32(np.arccos(1./rr)*npts/math.pi)
                        dum9.append(Ngl.add_polyline(wks,VFE,xx[(n_s-1):100],yy[(n_s-1):100],respl))
            del xx,yy,radR

    #======================================Part 4:Plot data points / count-text
    # set markers for points of models/obs
    if hasattr(opt_VFE, "MarkersOn"):
        if not isinstance(opt_VFE.MarkersOn,bool):
            print("VFE diagram error: MarkersOn' of opt_VFE should be bool!")
            os._exit(0)
        MarkerOn = opt_VFE.MarkersOn
    else:
        MarkerOn = False

    if MarkerOn:
        if hasattr(opt_VFE,"MarkersIndex"):
            Markers = opt_VFE.MarkersIndex
            if (not isinstance(Markers,(list,tuple))) or (not all(isinstance(x,int) for x in Markers)):
                print("VFE diagram error: 'MarkersIndex' of opt_VFE should be list/tuple, and its elements are int/float up to 2!")
                os._exit(0)
            if not all(x>=0 and x<=16 for x in Markers):
                print("VFE diagram error: 'MarkersIndex' of opt_VFE is invalid, it should be int raning from 0 to 16!")
                os._exit(0)
            if len(Markers) > 2:
                Markers = Markers[0:2]
            if (nCase>nmodel) and (len(Markers)==1):
                Markers = [Markers[0],Markers[0]]
        else:
            Markers = [4,16]

        if hasattr(opt_VFE,"gsMarkerThicknessF"):
            if not isinstance(opt_VFE.gsMarkerThicknessF,(int,float)):
                print("VFE_diagram error: 'gsMarkerThicknessF' of opt_VFE should be int/float!")
                os._exit(0)
            gsMarkerThicknessF = np.float(opt_VFE.gsMarkerThicknessF)
        else:
            gsMarkerThicknessF = 1.0

        if hasattr(opt_VFE,"gsMarkerSizeF"):
            if not isinstance(opt_VFE.gsMarkerSizeF,(int,float)):
                print("VFE_diagram error: 'gsMarkerSizeF' of opt_VFE should be int/float!")
                os._exit(0)
            gsMarkerSizeF = np.float(opt_VFE.gsMarkerSizeF)
        else:
            gsMarkerSizeF = 0.0085

        if hasattr(opt_VFE, "CountTxOn"):
            CountTxOn = opt_VFE.CountTxOn
            if not isinstance(CountTxOn, bool):
                print("VFE_diagram error: 'CountTxOn' of opt_VFE should be bool!")
                os._exit(0)
        else:
            CountTxOn = True

        if CountTxOn:
            if hasattr(opt_VFE, "CountTxYOffset"):
                if not isinstance(opt_VFE.CountTxYOffset,(int,float)):
                    print("VFE_diagram error: 'CountTxYOffset' of opt_VFE should be int/float!")
                    os._exit(0)
                CountTxYOffset    = np.float(opt_VFE.CountTxYOffset)             
            else:
                CountTxYOffset    = 0.015                    

            if hasattr(opt_VFE, "CountTxXOffset"):
                if not isinstance(opt_VFE.CountTxXOffset,(int,float)):
                    print("VFE_diagram error: 'CountTxXOffset' of opt_VFE should be int/float!")
                    os._exit(0)
                CountTxXOffset    = np.float(opt_VFE.CountTxXOffset)             
            else:
                CountTxXOffset    = 0.015 
    else:
        CountTxXOffset    = 0.0
        CountTxYOffset    = 0.0

    if hasattr(opt_VFE, "CountFontHeightF"):
        if not isinstance(opt_VFE.CountFontHeightF,(int,float)):
            print("VFE_diagram error: 'CountFontHeightF' of opt_VFE should be int/float!")
            os._exit(0)
        CountFontHeightF     = np.float(opt_VFE.CountFontHeightF)
    else:
        CountFontHeightF     = 0.0085         

    # set colors for points of models/obs & text
    if hasattr(opt_VFE, "Colors"):
        Colors0  = opt_VFE.Colors

        if isinstance(Colors0,(list,tuple)):
            if all(isinstance(x,str) for x in Colors0):
                for x in Colors0:
                    if Ngl.get_named_color_index(wks,x) < 0:
                        print("VFE_diagram error: named color-",x," in 'Colors' of opt_VFE is invalid!")
                        os._exit(0)
                Color_isRGB = False
            elif all(isinstance(x,int) for x in Colors0):
                for x in Colors0:
                    if x<0:
                        print("VFE_diagram error: when 'Colors' of opt_VFE is set by NhlTColorIndex,",\
                            "it should be list of int greater than 0! (refer to:",\
                                "http://www.ncl.ucar.edu/Document/HLUs/Classes/Workstation.shtml#NhlTColorIndex )")
                        os._exit(0)
                Color_isRGB = False
            elif all(isinstance(x,list) for x in Colors0) or all(isinstance(x,tuple) for x in Colors0):
                try:
                    Color0_array = np.array(Colors0)
                except:
                    print("VFE_diagram error: when 'Colors' of opt_VFE is set by RGB or RGBA map,",\
                        "it should be list(tuple) of Ncolor list(tuple) of which lengths are same!")
                    os._exit(0)
                if (Color0_array.ndim!=2) or ((Color0_array.shape[1]!=3) and (Color0_array.shape[1]!=4)):
                    print("VFE_diagram error: when 'Colors' of opt_VFE is set by RGB or RGBA map,",\
                        "it should be list(tuple) of Ncolor list(tuple) of which len is 3 or 4 and values are numeric!")
                    os._exit(0)
                if np.min(Color0_array) < 0.:
                    print("VFE_diagram error: when 'Colors' of opt_VFE is set by RGB or RGBA map,",\
                        "it should be greater than 0!")
                    os._exit(0)

                del Color0_array
                Color_isRGB = True
            else:
                print("VFE_diagram error: 'Colors' of opt_VFE should be set to list of named colors,",\
                    "NhlTColorIndex or RGB/RGBA map ! (refer to readme.namelist or User Guide)")
                os._exit(0)
        else:
            print("VFE_diagram error: 'Colors' of opt_VFE should be set to list of named colors,",\
                "NhlTColorIndex or RGB/RGBA map ! (refer to readme.namelist or User Guide)")
            os._exit(0)       
    else:
        Colors0  = ["darkolivegreen", "darkorange", "brown4", "rosybrown1","orangered",\
                    "blue2", "chocolate", "cadetblue4", "cornflowerblue",  "dimgrey", \
		            "gold3", "honeydew4", "hotpink3", "indianred", "lemonchiffon3", \
		            "mediumpurple", "mistyrose2", "navy", "mediumpurple4", "orangered", \
		            "palevioletred","peru", "peachpuff2", "royalblue", "salmon", "seagreen",\
		            "sienna", "steelblue","darkslateblue","darkgreen","antiquewhite4", "aquamarine4"]

    if len(Colors0) > (nCase-1):
        Colors = Colors0[0:nCase]
    elif len(Colors0) == 1:
        Colors = Colors0*nCase
    else:
        Colors = [Colors0[0]]*nmodel
        if nCase > nmodel:
            Colors = Colors+([Colors0[1]]*(nCase-nmodel))

    gsRes = Ngl.Resources()
    gsRes2 = Ngl.Resources()

    if MarkerOn:    
        gsRes.gsMarkerThicknessF = gsMarkerThicknessF      
        gsRes.gsMarkerSizeF      = gsMarkerSizeF 
    else:
        Colors = [Colors0[0]]*nmodel
        if nCase > nmodel:
            Colors = Colors+([Colors0[1]]*(nCase-nmodel))         

    ptRes = Ngl.Resources()                                      
    ptRes.txJust             = "BottomCenter"          
    ptRes.txFontThicknessF   = 1.0                  
    ptRes.txFontHeightF      = CountFontHeightF   

    # plot points for models and observation/reanalysis
    dum10 = []
    dum11 = []
    dum12 = []
    for i in range(nCase):
        if MarkerOn:
            if i<nmodel:
                gsRes.gsMarkerIndex      = Markers[0]
            else:
                gsRes.gsMarkerIndex      = Markers[1] 
            gsRes.gsMarkerColor      = Colors[i] 
            dum10.append(Ngl.add_polymarker(wks, VFE, X[i], Y[i], gsRes))

        if (MarkerOn and CountTxOn) or (not MarkerOn):            
            ptRes.txFontColor =  Colors[i] 
            
            dum11.append(Ngl.add_text(wks, VFE, str(i+1), X[i]+CountTxXOffset, Y[i]+CountTxYOffset, ptRes))

        if hasattr(opt_VFE, "stdline"):    
            xLAstd = [X1[i], X2[i]]
            yLAstd = [Y1[i], Y2[i]]
	
            gsRes2.gsLineColor       = Colors[i]              
            gsRes2.gsLineDashPattern = 0                      
            gsRes2.gsLineThicknessF  = 2.
                 
            dum12.append(Ngl.add_polyline(wks, VFE, xLAstd, yLAstd, gsRes2))

    #====================Part 5:Add case legend and variable labels
    # plot legend of data type
    if hasattr(opt_VFE, "DatatypeLegendOn"):
        if not isinstance(opt_VFE.DatatypeLegendOn,bool):
            print("VFE_diagram error: 'DatatypeLegendOn' of opt_VFE should be bool!")
            os._exit(0)

        if opt_VFE.DatatypeLegendOn:
            if hasattr(opt_VFE, "DatatypeLegend"):
                if (not isinstance(opt_VFE.DatatypeLegend,(list,tuple))) or \
                    (not all(isinstance(x,str) for x in opt_VFE.DatatypeLegend)):
                    print("VFE_diagram error: 'DatatypeLegend' of opt_VFE should be list/tuple of which elements are str!")
                    os._exit(0)
                DatatypeLabels = opt_VFE.DatatypeLegend

                if nCase==nmodel:
                    DatatypeLabels = DatatypeLabels[0]
                else:
                    DatatypeLabels = DatatypeLabels[0:2]
            else:
                DatatypeLabels = np.where(nCase==nmodel,["Models"],["Models","Reanalysis"])

            lgRes = Ngl.Resources()
            lgRes.lgItemType      = "Markers"

            if hasattr(opt_VFE, "DatatypeLegendFontHeightF"):
                if (not isinstance(opt_VFE.DatatypeLegendFontHeightF,(int,float))):
                    print("VFE_diagram error: 'DatatypeLegendFontHeightF' of opt_VFE should be int/float!")
                    os._exit(0)
                lgRes.lgLabelFontHeightF = np.float(opt_VFE.DatatypeLegendFontHeightF)
            else:
                lgRes.lgLabelFontHeightF = 0.03

            if hasattr(opt_VFE, "DatatypeLegendXL"):
                if (not isinstance(opt_VFE.DatatypeLegendXL,(int,float))):
                    print("VFE_diagram error: 'DatatypeLegendXL' of opt_VFE should be int/float!")
                    os._exit(0)
                LegendXL = np.float(opt_VFE.DatatypeLegendXL)
            else:
                LegendXL = 0.8                
      
            if hasattr(opt_VFE, "DatatypeLegendYL"):
                if (not isinstance(opt_VFE.DatatypeLegendYL,(int,float))):
                    print("VFE_diagram error: 'DatatypeLegendYL' of opt_VFE should be int/float!")
                    os._exit(0)
                LegendYL = np.float(opt_VFE.DatatypeLegendYL)
            else:  
                LegendYL = 0.9

            if hasattr(opt_VFE, "DatatypeLegendWidth"):
                if (not isinstance(opt_VFE.DatatypeLegendWidth,(int,float))):
                    print("VFE_diagram error: 'DatatypeLegendWidth' of opt_VFE should be int/float!")
                    os._exit(0)
                lgRes.vpWidthF = np.float(opt_VFE.DatatypeLegendWidth)
            else:
                lgRes.vpWidthF = 0.1

            if hasattr(opt_VFE, "DatatypeLegendHeight"):
                if (not isinstance(opt_VFE.DatatypeLegendHeight,(int,float))):
                    print("VFE_diagram error: 'DatatypeLegendHeight' of opt_VFE should be int/float!")
                    os._exit(0)
                lgRes.vpHeightF = np.float(opt_VFE.DatatypeLegendHeight)/2.
            else:
                lgRes.vpHeightF = 0.1

            if len(DatatypeLabels)==2:
                lgRes.vpHeightF = lgRes.vpHeightF/2.

            if "gsMarkerSizeF" in locals():
                lgRes.lgMarkerSizeF = gsMarkerSizeF*1.3
            else:
                lgRes.lgMarkerSizeF = ptxFontHeightF*1.3
            lgRes.lgMarkerThicknessF = 2.
            lgRes.lgPerimOn          = False

            dum13 = []

            # Model type       
            lgRes.lgMonoMarkerIndex = True
            lgRes.lgMonoMarkerColor = True

            lgRes.lgMarkerColor = Colors[0]
            if MarkerOn:
                lgRes.lgMarkerIndex = Markers[0]
            else:
                lgRes.lgMarkerIndex = 16
            dum13.append(Ngl.legend_ndc(wks, 1, [DatatypeLabels[0],], LegendXL, LegendYL, lgRes))
            # obs/reanalysis type
            if len(DatatypeLabels) == 2:
                lgRes.lgMarkerColor = Colors[nmodel]
                if MarkerOn:
                    lgRes.lgMarkerIndex = Markers[1]
                dum13.append(Ngl.legend_ndc(wks, 1, [DatatypeLabels[1],], LegendXL, \
                    LegendYL-max(lgRes.vpHeightF,lgRes.lgLabelFontHeightF), lgRes))
            
    # plot case labels
    if hasattr(opt_VFE, "caseLabels"):
        caseLabels = opt_VFE.caseLabels

        if (not isinstance(caseLabels,(list,tuple))) or \
            (not all(isinstance(x,str) for x in caseLabels)):
            print("VFE_diagram error: 'caseLabels' of opt_VFE should be list/tuple of which elements are str!")
            os._exit(0)

        if hasattr(opt_VFE, "caseLabelsFontHeightF"):
            if (not isinstance(opt_VFE.caseLabelsFontHeightF,(int,float))):
                print("VFE_diagram error: 'caseLabelsFontHeightF' of opt_VFE should be int/float!")
                os._exit(0)
            caseLabelsFontHeightF = np.float(opt_VFE.caseLabelsFontHeightF)
        else:
            caseLabelsFontHeightF = 0.013

        txres               = Ngl.Resources() 
        txres.txFontHeightF = caseLabelsFontHeightF
        txres.txJust        = "CenterLeft"                                 

        if hasattr(opt_VFE, "caseLabelsYinterval"):
            if (not isinstance(opt_VFE.caseLabelsYinterval,(int,float))):
                print("VFE_diagram error: 'caseLabelsYinterval' of opt_VFE should be int/float!")
                os._exit(0)
            delta_y = np.float(opt_VFE.caseLabelsYinterval)
        else:
            delta_y = 0.06
     
        if hasattr(opt_VFE, "caseLabelsYloc"):
            if (not isinstance(opt_VFE.caseLabelsYloc,(int,float))):
                print("VFE_diagram error: 'caseLabelsYloc' of opt_VFE should be int/float!")
                os._exit(0)
            ys  = np.float(opt_VFE.caseLabelsYloc)                     
        else:
            ys  = max([nCase*delta_y ,0.30])

        if hasattr(opt_VFE, "caseLabelsXloc"):
            if (not isinstance(opt_VFE.caseLabelsXloc,(int,float))):
                print("VFE_diagram error: 'caseLabelsXloc' of opt_VFE should be int/float!")
                os._exit(0)
            xs  = np.float(opt_VFE.caseLabelsXloc)                       
        else:
            xs  = 0.125

        dum14 = []

        for i in range(len(caseLabels)):
            dum14.append(Ngl.add_text(wks, VFE, str(i+1)+" - "+opt_VFE.caseLabels[i], xs, ys, txres))
            ys = ys- delta_y

    # Obs uncertainty
    if hasattr(opt_VFE,"obsUncer"):
        if not isinstance(opt_VFE.obsUncer,(int,float)):
            print("VFE_diagram error: 'obsUncer' of opt_VFE should be int/float!")
            os._exit(0)
        Obs_uncer = np.float(opt_VFE.obsUncer)

        fRes = Ngl.Resources()
        fRes.gsFillColor = "red"
        fRes.gsFillOpacityF = 0.3

        xbox = (1.-Obs_uncer*0.5,1.+Obs_uncer*0.5,1.+Obs_uncer*0.5,1.-Obs_uncer*0.5,1.-Obs_uncer*0.5)
        ybox = (0.,0.,0.01,0.01,0.)

        dum15 = Ngl.add_polygon(wks, VFE, xbox, ybox, fRes)
        
    # plot grid line
    if hasattr(opt_VFE,"draw_grid"):
        if not isinstance(opt_VFE.draw_grid,bool):
            print("VFE_diagram error: 'draw_grid' of opt_VFE should be bool!")
            os._exit(0)
        if opt_VFE.draw_grid:
            Ngl.draw_ndc_grid(wks)

    Ngl.draw(VFE)
    Ngl.frame(wks)
    
    return VFE

Ngl.end()
#################################################################



   