#################################################################
# SCRIPT FOR CALCULATING MULTIPLE STASTICS FOR
# MULTIVARIABLE INTEGRATED EVALUATION (MVIE) of model performance
#  References:
#  Zhang et al., 2020, GMDD, https://doi.org/10.5194/gmd-2020-310
#  Xu et al., 2017, GMD, https://doi.org/10.5194/gmd-10-3805-2017
#  Xu et al., 2016, GMD, https://doi.org/10.5194/gmd-9-4365-2016
#  AFTER CALCULATION, VFE DIAGRAM / METRICS TABLE CAN BE PLOTED.
#################################################################
# The script assumes all variables derived from one model or observation
# are saved in one data file (str). If not, please merge all variables
# into one data file using NCL, Python3 or the third party sofware, e.g., cdo 
# Users can modify the namelist at the beginning of this script
# based on the application.
#################################################################
# Please report bugs to: zhangmengzhuo1996@163.com or xuzhf@tea.ac.cn
#################################################################
#################################################################
import numpy as np 
import xarray as xr 
import os
import Nio
import function.MVIETool_fun_withMIEI as MVIE
#            ####################################
#            #  namelist to be edited by users  #
#            ####################################
#============== user's editing starts here ===========================================================
#------ configuration for input and output data --------------------------
Inputdatadir    = "./data/"                     # Model and observation data input directory (str)
                                
Model_filenames = list(map(lambda x: "example.season.M"+x+".nc", ["1","2","3","4","5","6","7","8","9","10"]))                          
                                                # File names of observation/reanalysis data (list/tuple consisting of str)

Obs_filenames   = list(map(lambda x: "example.season.REA"+x+".nc", ["1","2"]))                          
                                                # File names of observation/reanalysis data (list/tuple consisting of str)

Varname         = "SLP_DJF,SST_DJF,Q600_DJF,T850_DJF,(u850_DJF,v850_DJF),(u200_DJF,v200_DJF)"                          
                                                # Names of Variables to be evaluated (str)           
								                   # Names of variables composing a vector should be written together and put in parentheses seperated by comma
									               # e.g.: "T2m,precip,(u850,v850)"

MVIE_filename   = "example.DJF.centered.areaweight.MIEI"                          
                                                # Output filename (str) [default: "MVIE_stats.nc"]

#------ configuration for data dimension and coordinates info  -------------------------
Var_Coords      = True                          # Evaluating data in a specified part or the whole [False: default]? (bool)

isCoords_time   = False                   	    # Variable has dimension and coordinate for time [False: default]? (bool)
				                                # valid when 'Var_Coords = True'    
#Coords_time     =              		           # Name of time dimension name (str) [default: "time"], valid when 'isCoords_time = True' 
#Range_time      =        		                # Range of time in 'Coords_time'-variable (list/tuple of 2-str or 2-int/float), valid when 'isCoords_time = True'
                                                   # Specify sub-time to be evaluated. e.g.: ["1996-12","2005-12"]
				                                   # When time coordinate var defined with DatetimeIndex, it must set to DatetimeIndex-str; otherwise set to vlues in time coordinate var

isCoords_geo    = True  		                # Variable has dimension and coordinate for latitude/longitude [False: default]? (bool)
                                                   # valid when 'Var_Coords = True'   
Coords_geo      = ["lat","lon"]            	    # Names of latitude/longitude dimension names (list/tuple), valid when 'isCoords_geo = True'  
Range_geo       = {"lat":[0.,90.],"lon":[0.,360.]}       	                
                                                # Range of lat/lon in 'Coords_geo'-variable, valid when 'isCoords_geo = True'
                                                   # Specify sub-region to be evaluated (dic). e.g.: {"lat":[0.,45],"lon":[0,180]}

#hasLevel        =                              # Variable has level dimension or not [False: default]? (str/False)                                               
                                                   # str: name of level dimension, e.g.:"lev", valid when 'Var_Coords = True' 
                                                # The level coordinate is required to read data at specific level.  
#VarLev          =                                 # Specify level for each variable (list/tuple consisting of int/float), valid when setting 'hasLevel' with level name
                                                   # if one variable without level dim, 'VarLev' of it will be omitted.

#------ configuration for area weight and variable weight ---------------
Isarea_wgt      = "lat"          		        # Considering area weighting (str) or not [default: False] in the statistics?
				                                   # str: name of dimension for latitude
  
#Wgt_var         = 		                        # Variable Weighting (1, list/tuple consisting of int/float) [default: 1 - no weight]
				                                   # list/tuple: weighting for the corresponding variables listed in "Varname"

#------ configuration for statistical metrics ---------------------------
#ComMask_On      =                              # Create a common mask for datasets of all models and reference to deal with missing points [True: default] 
                                                   # or unify missing points of each model-reference pair (True)? (bool)
Unify_VarMiss   = True                          # Unify missing points across all variables of one model (True) or not [False: default]? (bool) 

#Typ_stat        = 		                        # Type of calculated statistics ("float32")/["float64": default]    

#Stats_mode      =    			                # Mode of statistics: centered [0: default] / uncentered (1]_ 

#Cal_VME         =    		                    # Calculate VME(ME) [1: dafault] /MEVM (0) in centered mode 

#MISS_F          =       	                    # Parameter F in MISS (int/float) [default: 2]
 
#Print_stats_r   = 		                        # Print the range of statistics to screen [True: default] or not? (bool)

#============== user's editing ends here =============================================================

#########################################
# Check for input information from user #
#########################################

print("#################################")
print("#     MVIETool v1.0 (Python3)   #")
print("#################################")

# 'Model_filenames' check
if not "Model_filenames" in locals():
    print("****Calculate_MVIE error: cannot find 'Model_filenames', and it must be set !")
    os._exit(0)

if (not isinstance(Model_filenames,(list,tuple))) or (not all(isinstance(x,str) for x in Model_filenames)):
    print("****Calculate_MVIE error: 'Model_filenames' should be 'list/tuple' type, and each element of it should be string!")
    os._exit(0)

Nmodel = len(Model_filenames)

if Nmodel < 2:
    print("****Calculate_MVIE error: num of 'Model_filenames' should be greater than 1 !")
    os._exit(0)

# 'Obs_filenames' check
if not "Obs_filenames" in locals():
    print("****Calculate_MVIE error: cannot find 'Obs_filenames', and it must be set !")
    os._exit(0)

if (not isinstance(Obs_filenames,(list,tuple))) or (not all(isinstance(x,str) for x in Obs_filenames)):
    print("****Calculate_MVIE error: 'Obs_filenames' should be 'list/tuple' type, and each element of it should be string!")
    os._exit(0)

Nobs = len(Obs_filenames)

# 'Inputdatadir' check
if not "Inputdatadir" in locals():
    print("****Calculate_MVIE error: cannot find 'Inputdatadir', and it must be set !")
    os._exit(0) 

if not isinstance(Inputdatadir,str):
    print("****Calculate_MVIE error: 'Inputdatadir' should be type of string!")
    os._exit(0) 

if Inputdatadir[-1] != "/":
    Inputdatadir += "/"

# 'Varname' & 'Wgt_var' check 
if not "Varname" in locals():
    print("****Calculate_MVIE error: cannot find 'Varname', and it must be set !")
    os._exit(0)

if not isinstance(Varname,str):
    print("****Calculate_MVIE error: Varname should be a string, and has format as ",\
            "'var1,var2,(var3,var4),var5,(var6,var7,var8)' ")
    os._exit(0)

if not "Wgt_var" in locals():
    Wgt_var = 1.
else:
    if isinstance(Wgt_var,(list,tuple)):
        if (not all(isinstance(x,(int,float)) for x in Wgt_var)):
           print("****Calculate_MVIE error: 'Wgt_var' should be set to 1 (no weight) or list/tuple indicating var weight !")
           os._exit(0)
    elif isinstance(Wgt_var,(int,float)):
        if (Wgt_var!=1) and (Wgt_var!=1.):
            print("****Calculate_MVIE error: 'Wgt_var' should be set to 1 (no weight) or list/tuple indicating var weight !")
            os._exit(0)
    else:
        print("****Calculate_MVIE error: 'Wgt_var' should be set to 1 (no weight) or list/tuple indicating var weight !")
        os._exit(0)

Var_names,Nvar,M,Mvar_D,M_wgt,VarWgt_str = MVIE.Get_Mvars(Varname,Wgt_var)   
Data_D = np.sum(Mvar_D)

# Coordinate info check & read
if (not "Var_Coords" in locals()) or (not isinstance(Var_Coords,bool)):
    print("****Calculate_MVIE error: can not find 'Var_Coords', or 'Var_Coords' is not bool !")
    os._exit(0)

if Var_Coords:
    # define class for coordinate
    class Coords():
        def __init__(self,Cname,Crange,Cspe):
            if Cname != False:
                self.Cname = Cname
            if Crange != False:
                self.Crange = Crange
            if Cspe != False:
                self.Cspe = Cspe

    Sel_str = "("
    Reorder_str = "("

    # Coord 'time'
    if not "isCoords_time" in locals():
        isCoords_time = False
        Ctime = Coords(False,False,False)
    else:
        if not isinstance(isCoords_time,bool):
            print("****Calculate_MVIE error: isCoords_time should be bool (True/False) !")
            os._exit(0)
        
        if isCoords_time:
            if not "Coords_time" in locals():
                Coords_time = "time"
            elif not isinstance(Coords_time,str):
                print("****Calculate_MVIE error: 'Coords_time' should be type of string !")
                os._exit(0)
            
            if (not "Range_time" in locals()) or (not isinstance(Range_time,(list,tuple))) or (len(Range_time)!=2):
                print("****Calculate_MVIE error: cannot find 'Range_time', ",\
                  "or 'Range_time' should be list/tuple of 2 elements! ")
                os._exit(0)

            if (not all(isinstance(x,(int,float)) for x in Range_time)) and (not all(isinstance(x,str) for x in Range_time)):
                print("****Calculate_MVIE error: 'Range_time' should be list/tuple of 2 elements, which are int/float/string !")
                os._exit(0)

            Ctime = Coords(Coords_time,Range_time,False)

            Sel_str += Coords_time+"=time_slice"
            Reorder_str += '"'+Coords_time+'"'
        else:
            Ctime = Coords(False,False,False)
    
    # Coords 'lat'&'lon'
    if not "isCoords_geo" in locals():
        isCoords_geo = False
        Cgeo1 = Coords(False,False,False)
        Cgeo2 = Coords(False,False,False)
    else:
        if not isinstance(isCoords_geo,bool):
            print("****Calculate_MVIE error: isCoords_geo should be bool (True/False) !")
            os._exit(0)

        if isCoords_geo:
            if not "Coords_geo" in locals():
                print("****Calculate_MVIE error: cannot find 'Coords_geo' and it must be set under 'isCoords_geo=True' !")
                os._exit(0)
            if (not isinstance(Coords_geo,(list,tuple))) or (len(Coords_geo)>2) or (not all(isinstance(x,str) for x in Coords_geo)):
                print("****Calculate_MVIE error:'Coords_geo' should be list/tuple with len less than 2, of which element should be string !")
                os._exit(0)
            if (not "Range_geo" in locals()) or (not isinstance(Range_geo,dict)):
                print("****Calculate_MVIE error: cannot find 'Range_geo' and it must be set under 'isCoords_geo=True' or",\
                    "'Range_geo' should be type of dictionary!")
                os._exit(0)
         
            # geo1
            if not Coords_geo[0] in Range_geo:
                print("****Calculate_MVIE error: cannot find",Coords_geo[0],"in the keys of 'Range_geo' !")
                os._exit(0)

            Range_geo1 = Range_geo[Coords_geo[0]]
            if (not isinstance(Range_geo1,(list,tuple))) or (len(Range_geo1)!=2) or (not all(isinstance(x,(int,float)) for x in Range_geo1)):
                print("****Calculate_MVIE error: values in Range_geo should be list/tuple with two elements (int/float), e.g.,",\
                    " {'lat':[0,45]} !")
                os._exit(0)
            Cgeo1 =  Coords(Coords_geo[0],Range_geo1,False)

            if len(Sel_str)>1:
                Sel_str     += ","
                Reorder_str += ','
            Sel_str     += Coords_geo[0]+"=geo1_slice"
            Reorder_str += '"'+Coords_geo[0]+'"'
            
            # geo2
            if len(Coords_geo) == 2:
                if not Coords_geo[1] in Range_geo:
                    print("****Calculate_MVIE error: cannot find",Coords_geo[1],"in the keys of 'Range_geo' !")
                    os._exit(0)

                Range_geo2 = Range_geo[Coords_geo[1]]
                if (not isinstance(Range_geo2,(list,tuple))) or (len(Range_geo2)!=2) or (not all(isinstance(x,(int,float)) for x in Range_geo2)):
                    print("****Calculate_MVIE error: values in Range_geo should be list/tuple with two elements (int/float), e.g.,",\
                        " {'lat':[0,45]} !")
                    os._exit(0)
                Cgeo2 =  Coords(Coords_geo[1],Range_geo2,False)

                if len(Sel_str)>1:
                   Sel_str     += ","
                   Reorder_str += ','
                Sel_str     += Coords_geo[1]+"=geo2_slice"
                Reorder_str += '"'+Coords_geo[1]+'"'
            else:
                Cgeo2 =  Coords(False,False,False)
        else:
            Cgeo1 = Coords(False,False,False)
            Cgeo2 = Coords(False,False,False)

    Reorder_str += ")"

    if (not isCoords_time) and (not isCoords_geo):   
        print("****Calculate_MVIE error: 'Var_Coords' is Ture but with no coordinate info !")
        os._exit(0)

    if not "hasLevel" in locals():
        Clev = Coords(False,False,False)
    else:
        if isinstance(hasLevel,str):
            if not "VarLev" in locals():
                print("****Calculate_MVIE error: 'VarLev' must be set when 'hasLevel' set the name of level dimension !")
                os._exit(0)
            if (not isinstance(VarLev,(list,tuple))) or (len(VarLev)!=Nvar) or (not all(isinstance(x,(int,float)) for x in VarLev)):
                print("****Calculate_MVIE error: 'VarLev' should be type of list/tuple of which elements are int/float, ",\
                    "and len of list/tuple should atch the num of vars in 'Varname' !")
                os._exit(0)
            Clev = Coords(hasLevel,False,VarLev)
        elif isinstance(hasLevel,bool):
            if hasLevel:
                print("****Calculate_MVIE error: 'hasLevel' should be 'False'(bool) or name of level dimension (str) !")
                os._exit(0)
            else:
                Clev = Coords(False,False,False)
        else:
            print("****Calculate_MVIE error: 'hasLevel' should be 'False'(bool) or name of level dimension (str) !")
            os._exit(0)

# 'Isarea_wgt' check
if not "Isarea_wgt" in locals():
    Isarea_wgt = False
else:
    if isinstance(Isarea_wgt,str):
        if not isCoords_geo:
            print("****Calculate_MVIE warning: Isarea_wgt is invalid when 'isCoords_geo' = False !")
            Isarea_wgt = False
        elif not Isarea_wgt in Coords_geo:
            print("****Calculate_MVIE error: cannot find ",Isarea_wgt," in 'Coords_geo' !")
            os._exit(0)
    elif Isarea_wgt != False:
        print("****Calculate_MVIE error: Isarea_wgt should be False or the str of latitude dimension name !")
        os._exit(0)

# 'Stats_mode' check
if not "Stats_mode" in locals():
    Stats_mode = 0
elif (Stats_mode!=0) and (Stats_mode!=1):
    print("****Calculate_MVIE error: 'Stats_mode' should be 0 or 1 !")
    os._exit(0)

# 'ComMask_On' check
if not "ComMask_On" in locals():
    ComMask_On = True
elif not isinstance(ComMask_On,bool):
    print("****Calculate_MVIE error: ComMask_On should be bool !")
    os._exit(0)

# 'Unify_VarMiss' check
if not "Unify_VarMiss" in locals():
    Unify_VarMiss = False
elif not isinstance(Unify_VarMiss,bool):
    print("****Calculate_MVIE error: Unify_VarMiss should be bool !")
    os._exit(0)

# 'Typ_stat' check
if not "Typ_stat" in locals():
    typeS = "d"
else:
    if Typ_stat == "float64":
        typeS = "d"
    elif Typ_stat == "float32":
        typeS = "f"
    else:
        print("****Calculate_MVIE error: Typ_stat should be 'float64' or 'float32'!")
        os._exit(0)
    del Typ_stat 

# 'Cal_VME' check
if Stats_mode == 0:
    if not "Cal_VME" in locals():
        Cal_VME = 1
    elif (Cal_VME!=0) and (Cal_VME!=1):
        print("****Calculate_MVIE error: 'Cal_VME' should be 0 or 1 !")
        os._exit(0)

# 'MISS_F' check
if not "MISS_F" in locals():
    MISS_F = 2.
elif not isinstance(MISS_F,(int,float)):
    print("****Calculate_MVIE error: 'MISS_F' should be int or float !")
    os._exit(0)
    
# 'Print_stats_r' check
if not "Print_stats_r" in locals():
    Print_stats_r = True
elif not isinstance(Print_stats_r,bool):
    print("****Calculate_MVIE error: 'Print_stats_r' should be bool !")
    os._exit(0)

# 'MVIE_filename' check
if not "MVIE_filename" in locals():
    MVIE_filename = "MVIE_stats.nc"
elif not isinstance(MVIE_filename,str):
    print("****Calculate_MVIE error: 'MVIE_filename' should be type of string with format as 'MVIE.nc' !")
    os._exit(0)  
if len(MVIE_filename) < 3:
    MVIE_filename += ".nc"
else:
    if MVIE_filename[-3:] != ".nc":
        MVIE_filename += ".nc"

#################
# Read raw data #
#################

filenames = Model_filenames+Obs_filenames

if Var_Coords:
    for i in range(Nmodel+Nobs):
        idata,iWgt,ishp = MVIE.ReadVars_Coord(Inputdatadir+filenames[i],Var_names,Ctime,\
            Cgeo1,Cgeo2,Clev,Sel_str,Reorder_str,Isarea_wgt)
        
        if i == 0:
            All_data = np.zeros((Nmodel+Nobs,Data_D,idata.shape[1]),float)
            shp = ishp
        else:
            if ishp != shp:
                print("****Calculate_MVIE error: shape of vars in ",filenames[i]," donot match that in",filenames[0]," !")
                os._exit(0)
        
        All_data[i,:,:] = idata
        Area_wgt = iWgt
else:
    for i in range(Nmodel+Nobs):
        idata = MVIE.ReadVars(Inputdatadir+filenames[i],Var_names)

        if i == 0:
            All_data = np.zeros((Nmodel+Nobs,Data_D,idata.shape[1]),float)
            siz = idata.shape[1]
        else:
            if idata.shape[1] != siz:
                print("****Calculate_MVIE error: size of vars in ",filenames[i]," donot match that in",filenames[0]," !")
                os._exit(0)

        All_data[i,:,:] = idata
        Area_wgt = [1,]

################################
# Calculate Reference and Mask #
################################

if Unify_VarMiss:
    if ComMask_On:
        Mask_com = All_data[:,:,:].mean(axis=1).mean(axis=0) -\
            All_data[:,:,:].mean(axis=1).mean(axis=0)
    else:
        if Nobs == 1:
            Mask_com = All_data[Nmodel,:,:].mean(axis=0) -  All_data[Nmodel,:,:].mean(axis=0)
        else:
            Mask_com = All_data[Nmodel:(Nmodel+Nobs),:,:].mean(axis=1).mean(axis=0) -\
                All_data[Nmodel:(Nmodel+Nobs),:,:].mean(axis=1).mean(axis=0)

    Mask_com =  np.tile(Mask_com,(All_data.shape[1],1))
else:
    if ComMask_On:
        Mask_com = All_data[:,:,:].mean(axis=0) - All_data[:,:,:].mean(axis=0)
    else:
        if Nobs == 1:
            Mask_com = All_data[Nmodel,:,:] - All_data[Nmodel,:,:]
        else:
            Mask_com = All_data[Nmodel:(Nmodel+Nobs),:,:].mean(axis=0) -\
                All_data[Nmodel:(Nmodel+Nobs),:,:].mean(axis=0)

if Nobs == 1:
    Reference = All_data[Nmodel,:,:] + Mask_com
else:
    Reference = np.mean(All_data[Nmodel:(Nmodel+Nobs),:,:],axis=0) + Mask_com

###################
# Calsulate stats #
###################

Ncase = np.int(np.where(Nobs==1, Nmodel, Nmodel+Nobs))

if M == 1:   # single variable evaluation
    print("|-------------------------------|") 
    if Data_D == 1:
        print("|          Scalar field         |")
    else:
        print("|          Vector field         |")
    print("|-------------------------------|")

    if Stats_mode == 0:
        print("      --- Centered stats ---     ")
        if Data_D == 1:
            Stats_names = ["CORR","SD","cRMSD","ME"]
        else:
            Stats_names = ["cVSC","cRMSL","cRMSVD",str(np.where(Cal_VME==1,"VME","MEVM"))]  
    else:
        print("     --- Uncentered stats ---    ")
        if Data_D == 1:
            Stats_names = ["uCORR","rms","RMSD"]
        else:
            Stats_names = ["VSC","RMSL","RMSVD"] 

    Nstats = len(Stats_names)
    Stats  = np.zeros((Nstats,Ncase), float)

    Stats_other  = np.zeros((2,Ncase), float)

    Skillscore_names = ["S1_cen","S2_cen","S1_uncen","S2_uncen"]
    SkillS = np.zeros((4,Ncase), float)

    # mean error of vector direction between model and ref
    if (Stats_mode == 0) and (Cal_VME==0) and (Data_D == 2):
        MEVD = np.full(Ncase, np.nan)

    for i in range(Ncase):
        # add mask of Nan for model and reference
        Case_data,Ref_data = MVIE.Unif_nan_pair(All_data[i,:,:],Reference,Unify_VarMiss)

        # calculate stats
        Stats[0,i] = MVIE.VSCor(Case_data, Ref_data, Stats_mode, Area_wgt)
        Stats[1,i] = MVIE.RMSLength(Case_data, Stats_mode, Area_wgt)/MVIE.RMSLength(Ref_data, Stats_mode, Area_wgt)
        Stats[2,i] = MVIE.RMSVDiff(Case_data, Ref_data, Stats_mode, Area_wgt)/\
            MVIE.RMSLength(Ref_data, Stats_mode, Area_wgt)

        Stats_other[0,i] = MVIE.VSCor(Case_data, Ref_data, 1-Stats_mode, Area_wgt)
        Stats_other[1,i] = MVIE.RMSLength(Case_data, 1-Stats_mode, Area_wgt)/\
            MVIE.RMSLength(Ref_data, 1-Stats_mode, Area_wgt)

        if Stats_mode == 0:
            Stats[3,i] = MVIE.VecMError(Case_data, Ref_data, Area_wgt, Cal_VME)/ \
                            MVIE.RMSLength(Ref_data, Stats_mode, Area_wgt)
            if (Cal_VME==0) and (Data_D == 2):
                MEVD[i] = MVIE.MEVecDir(Case_data, Ref_data, Area_wgt)

    # calculate S1&S2
    if Stats_mode == 0:
        SkillS[0,:],SkillS[1,:] = MVIE.Skill_score2(Stats[1,:], Stats[0,:])
        SkillS[2,:],SkillS[3,:] = MVIE.Skill_score2(Stats_other[1,:], Stats_other[0,:])
    else:
        SkillS[2,:],SkillS[3,:] = MVIE.Skill_score2(Stats[1,:], Stats[0,:])
        SkillS[0,:],SkillS[1,:] = MVIE.Skill_score2(Stats_other[1,:], Stats_other[0,:])
   
    # print range of stats
    if Print_stats_r:
        MVIE.Pri_stat_range1(Stats, SkillS, Stats_names)
else:
    # multivariable integrated evaluation
    print("|-------------------------------| \n|      Multivariable field      | \n|-------------------------------|")

    if Stats_mode == 0:
        print("      --- Centered stats ---     ")
        # names of stats for individual variables
        Stats1_names  = ["CORR", "SD", "cRMSD", "ME"]
        # names of stats for multivariable field
        Stats2_names  = ["cVSC", "cRMSL", "SD_std", "cRMSVD", str(np.where(Cal_VME==1,"VME","MEVM"))]

        # mean error of vector direction between model and ref
        if Cal_VME == 0:
            MEVD = np.full((Ncase,M), np.nan)
        
        # rms, VSC for calculating uncentered MISS
        rms = np.zeros((Ncase,M), float)
        VSC = np.zeros(Ncase, float)

        # cMIEI
        cMIEI = np.zeros(Ncase, float)
    else:
        print("     --- Uncentered stats ---    ")
        # names of stats for individual variables
        Stats1_names  = ["uCORR", "rms", "RMSD"]
        # names of stats for multivariable field
        Stats2_names  = ["VSC", "RMSL", "rms_std", "RMSVD"]
        
        # SD, cVSC for calculating centered MISS
        SD   = np.zeros((Ncase,M), float)
        cVSC = np.zeros(Ncase, float)

    Nstats1 = len(Stats1_names)
    Nstats2 = len(Stats2_names)

    Stats1 = np.zeros((Nstats1,Ncase,M), float)
    Stats2 = np.zeros((Nstats2,Ncase), float)

    MISS_names = ["cMISS","uMISS"]
    MISS = np.zeros((2,Ncase), float)

    for i in range(Ncase):
        # add mask of Nan for model and reference
        Case_data,Ref_data = MVIE.Unif_nan_pair(All_data[i,:,:],Reference,Unify_VarMiss)

        # calculate stats for individual variables
        iD = 0
        for j in range(M):
            # CORR/uCORR
            Stats1[0,i,j] = MVIE.VSCor(Case_data[iD:(iD+Mvar_D[j]),:], Ref_data[iD:(iD+Mvar_D[j]),:], \
                                Stats_mode, Area_wgt)
            # SD/rms
            Stats1[1,i,j] = MVIE.RMSLength(Case_data[iD:(iD+Mvar_D[j]),:], Stats_mode, Area_wgt)/ \
                                MVIE.RMSLength(Ref_data[iD:(iD+Mvar_D[j]),:], Stats_mode, Area_wgt)
            # cRMSD/RMSD
            Stats1[2,i,j] = MVIE.RMSVDiff(Case_data[iD:(iD+Mvar_D[j]),:], Ref_data[iD:(iD+Mvar_D[j]),:], \
                                Stats_mode, Area_wgt)/ \
                                    MVIE.RMSLength(Ref_data[iD:(iD+Mvar_D[j]),:], Stats_mode, Area_wgt)
            
            if Stats_mode == 0:
                # ME
                Stats1[3,i,j] = MVIE.VecMError(Case_data[iD:(iD+Mvar_D[j]),:], Ref_data[iD:(iD+Mvar_D[j]),:], \
                                Area_wgt, Cal_VME)/MVIE.RMSLength(Ref_data[iD:(iD+Mvar_D[j]),:], Stats_mode, Area_wgt)                           
                # MEVD
                if (Cal_VME==0) and (Mvar_D[j] == 2):
                    MEVD[i,j] = MVIE.MEVecDir(Case_data[iD:(iD+Mvar_D[j]),:], Ref_data[iD:(iD+Mvar_D[j]),:], Area_wgt)

                # rms
                rms[i,j] = MVIE.RMSLength(Case_data[iD:(iD+Mvar_D[j]),:], 1-Stats_mode, Area_wgt)/ \
                                MVIE.RMSLength(Ref_data[iD:(iD+Mvar_D[j]),:], 1-Stats_mode, Area_wgt)
            else:
                # SD
                SD[i,j] = MVIE.RMSLength(Case_data[iD:(iD+Mvar_D[j]),:], 1-Stats_mode, Area_wgt)/ \
                                MVIE.RMSLength(Ref_data[iD:(iD+Mvar_D[j]),:], 1-Stats_mode, Area_wgt)

            iD = iD+Mvar_D[j]

        # normalization and add variable weighting for multivariable field
        M_nor = MVIE.Nor_matrix(Ref_data, Mvar_D, M_wgt, Area_wgt)
        Case_data *=  M_nor
        Ref_data *= M_nor

        # calculate stats for mutivariable intergrated field
        # cVSC/VSC
        Stats2[0,i] = MVIE.VSCor(Case_data, Ref_data, Stats_mode, Area_wgt)
        # cRMSL/RMSL
        Stats2[1,i] = MVIE.RMSLength(Case_data, Stats_mode, Area_wgt)/MVIE.RMSLength(Ref_data, Stats_mode, Area_wgt)
        # SD_std/rms_std    
        if isinstance(M_wgt,(int,float)):  ##
	        Stats2[2,i] = np.sqrt(np.mean((Stats1[1,i,:]-np.mean(Stats1[1,i,:]))**2))
        else:
            Stats2[2,i] = MVIE.Std_varWgt(Stats1[1,i,:], M_wgt)
        # cRMSVD/RMSVD
        Stats2[3,i] = MVIE.RMSVDiff(Case_data, Ref_data, Stats_mode, Area_wgt)/ \
                                        MVIE.RMSLength(Ref_data, Stats_mode, Area_wgt)

        if Stats_mode == 0:
            # VME/MEVM
            Stats2[4,i] = MVIE.VecMError(Case_data, Ref_data, Area_wgt, Cal_VME)/ \
                                        MVIE.RMSLength(Ref_data, Stats_mode, Area_wgt)
            # VSC
            VSC[i] = MVIE.VSCor(Case_data, Ref_data, 1, Area_wgt)

            # centered MISS
            MISS[0,i] = MVIE.MISScore(Stats1[1,i,:], Stats2[0,i], M_wgt, MISS_F) 
            # uncentered MISS
            MISS[1,i] = MVIE.MISScore(rms[i,:], VSC[i], M_wgt, MISS_F)

            # cMIEI
            cMIEI[i] = MVIE.MIEIndex2(Stats1[1,i,:], Stats2[0,i], M_wgt)
        else:
            # cVSC
            cVSC[i] = MVIE.VSCor(Case_data, Ref_data, 0, Area_wgt)
            
            # centered MISS
            MISS[0,i] = MVIE.MISScore(SD[i,:], cVSC[i], M_wgt, MISS_F)
            # uncentered MISS
            MISS[1,i] = MVIE.MISScore(Stats1[1,i,:], Stats2[0,i], M_wgt, MISS_F) 

    if not isinstance(M_wgt,(int,float)):         
        print("Var Weight:=======================")
        formatVar = '%'+str(max(list(map(len,VarWgt_str))))+'s'
        for i in range(M):
            print("  "+formatVar%VarWgt_str[i]+":",'%4.3f'%M_wgt[i])
    if Print_stats_r:
        if Stats_mode == 0:
            MVIE.Pri_stat_range2(Stats1, Stats2, MISS, opt_ME=Cal_VME)
        else:
            MVIE.Pri_stat_range2(Stats1, Stats2, MISS)

############################################
# Save results of MVIE in MVIE_filename.nc #
############################################
 
# Create MVIE_filename.nc
if os.path.isfile(MVIE_filename):
    print("***** Delete existing "+MVIE_filename+" file, and create "+MVIE_filename+" *****")
    os.system("rm " + MVIE_filename)
else:
    print("***** Create NetCDF file:",MVIE_filename," *****")
    
f_new = Nio.open_file(MVIE_filename, "c")

# Save outputs
if M == 1:
    # single scalar/vector field evaluation
    if Data_D == 1:
        Stats_longname = [["Correlation Coefficient","Standard Deviation",\
            "Centered Root Mean Square Difference","Mean Error"],\
                ["Uncentered Correlation Coefficient", \
                    "Root Mean Square","Root Mean Square Difference"]]
        Scores_longname = ["Centered skill score1 for scalar field", \
            "Centered skill score2 for scalar field", \
                "Uncentered skill score1 for scalar field", \
                    "Uncentered skill score2 for scalar field"]
        opt_isvec = 0
    else:
        Stats_longname = [["Centered Vector Similarity Coefficient","Centered Root Mean Square Length",\
            "Centered Root Mean Square Vector Difference","Vector Mean Error"],\
                    ["Vector Similarity Coefficient","Root Mean Square Length", \
                        "Root Mean Square Vector Difference"]]
        if Stats_mode==0:
            if Cal_VME==0:
                Stats_longname[0][3] = "Mean error of vector magnitude"

        Scores_longname = ["Centered skill score1 for vector field", \
            "Centered skill score2 for vector field", \
                "Uncentered skill score1 for vector field", \
                    "Uncentered skill score2 for vector field"]
        
        opt_isvec = 1
    Units = [["divided by SD of reference","divided by rms of reference"], \
        ["divided by cRMSL of reference","divided by RMSL of reference"]]

    # create dimention variables
    f_new.create_dimension('case',Ncase)
    f_new.create_variable('case','i',('case',))
    f_new.variables['case'].assign_value(np.int32(np.arange(Ncase)))
    f_new.create_dimension('var',1)
    f_new.create_variable('var','i',('var',))
    f_new.variables['var'].assign_value(np.int32(1))

    for i in range(Nstats):
        f_new.create_variable(Stats_names[i],typeS,('case',))

        setattr(f_new.variables[Stats_names[i]],'standard_name',Stats_longname[Stats_mode][i])
        if i != 0:
            setattr(f_new.variables[Stats_names[i]],'units',Units[opt_isvec][Stats_mode])
   
        if typeS == "f":
            f_new.variables[Stats_names[i]][:] = np.float32(Stats[i,:])
        else:
            f_new.variables[Stats_names[i]][:] = Stats[i,:]   
 
    for i in range(4):
        f_new.create_variable(Skillscore_names[i],typeS,('case',))
        setattr(f_new.variables[Skillscore_names[i]],'standard_name',Scores_longname[i])
        if typeS == "f":
            f_new.variables[Skillscore_names[i]][:] = np.float32(SkillS[i,:])
        else:
            f_new.variables[Skillscore_names[i]][:] = SkillS[i,:]

else:
    # MVIE
    Stats1_longname = [["Correlation Coefficient","Standard Deviation", \
        "Centered Root Mean Square Difference","Mean Error"], \
            ["Uncentered Correlation Coefficient", \
                "Root Mean Square","Root Mean Square Difference"]]
    Stats2_longname = [["Centered Vector Similarity Coefficient","Centered Root Mean Square Length", \
        "standard deviation of SD","Centered Root Mean Square Vector Difference", "Vector Mean Error"], \
                ["Vector Similarity Coefficient","Root Mean Square Length", \
                    "standard deviation of rms","Root Mean Square Vector Difference"]]
    if Stats_mode==0:
        if Cal_VME==0:
            Stats2_longname[0][4] = "Mean error of vector magnitude"

    MISS_longname = ["Centered Multivariable Integrated Skill Score", \
        "Uncentered Multivariable Integrated Skill Score"]
    Units = [["divided by SD of reference","divided by rms of reference"], \
        ["divided by cRMSL of reference","divided by RMSL of reference"]]
 
    # create dimention variables
    f_new.create_dimension('case',Ncase)
    f_new.create_variable('case','i',('case',))
    f_new.variables['case'].assign_value(np.int32(np.arange(Ncase)))
    f_new.create_dimension('var',M)
    f_new.create_variable('var','i',('var',))
    f_new.variables['var'].assign_value(np.int32(np.arange(M)))

    # stats for individual variables
    for i in range(Nstats1):
        f_new.create_variable(Stats1_names[i],typeS,('case','var'))

        setattr(f_new.variables[Stats1_names[i]],'standard_name',Stats1_longname[Stats_mode][i])
        if i != 0:
            setattr(f_new.variables[Stats1_names[i]],'units',Units[0][Stats_mode])

        if typeS == "f":
            f_new.variables[Stats1_names[i]][:] = np.float32(Stats1[i,:,:])
        else:
            f_new.variables[Stats1_names[i]][:] = Stats1[i,:,:]

    # stats for multiple fields
    for i in range(Nstats2):
        f_new.create_variable(Stats2_names[i],typeS,('case',))

        setattr(f_new.variables[Stats2_names[i]],'standard_name',Stats2_longname[Stats_mode][i])
        if (i!=0) and (i!=2):
            setattr(f_new.variables[Stats2_names[i]],'units',Units[1][Stats_mode])

        if typeS == "f":
            f_new.variables[Stats2_names[i]][:] = np.float32(Stats2[i,:])
        else:
            f_new.variables[Stats2_names[i]][:] = Stats2[i,:]

    # MISS
    for i in range(2):
        f_new.create_variable(MISS_names[i],typeS,('case',))
        setattr(f_new.variables[MISS_names[i]],'standard_name',MISS_longname[i]) 
        if typeS == "f": 
            f_new.variables[MISS_names[i]][:] = np.float32(MISS[i,:])
        else:
            f_new.variables[MISS_names[i]][:] = MISS[i,:]  

    if "cMIEI" in locals():
        f_new.create_variable("cMIEI",typeS,('case',)) 
        if typeS == "f": 
            f_new.variables["cMIEI"][:] = np.float32(cMIEI)
        else:
            f_new.variables["cMIEI"][:] = cMIEI   

# MEVD
if "MEVD" in locals(): 
    if M == 1:
        f_new.create_variable('MEVD',typeS,('case',))
    else:
        f_new.create_variable('MEVD',typeS,('case','var'))
    setattr(f_new.variables['MEVD'],'standard_name','Mean error of vector direction between model and ref')
    if typeS == "f":
        f_new.variables['MEVD'][:] = np.float32(MEVD)
    else: 
        f_new.variables['MEVD'][:] = MEVD
   
f_new.close()

#################################################################