;***************************************************************************                          
load "$NCARG_ROOT/lib/ncarg/nclscripts/csm/contributed.ncl"
load "$NCARG_ROOT/lib/ncarg/nclscripts/csm/gsn_code.ncl"
load "$NCARG_ROOT/lib/ncarg/nclscripts/csm/gsn_csm.ncl"
;***************************************************************************
; Procedure to draw the Metrics Table in MVIE
;***************************************************************************
; To report bugs or make suggestions, please send emails at:
;  zhangmengzhuo1996@163.com , xuzhf@tea.ac.cn
;***************************************************************************
procedure metrics_table(mfname[1]:string, fnames[*]:string, stats[*]:string, \
                        caseNames[*]:string, varNames[*]:string, \
                        zoom[*]:numeric, opt_metrics:logical)				               		
;***************************************************************************
; Arguments:
;  mfname      : name of output table
;  fnames      : datafile(s) of statistics for plotting
;  stats       : names of statistics chosen to be plotted
;  caseNames   : names of model/observation/reanalysis [Ncase]
;  varNames    : names of vars [Nvar]
;  zoom        : width of the table
;  opt_metrics : plot attribution
; where:
;  [Ncase] - num of cases
;            - if num of obs is 1, case is models
;            - if num of obs is greater than 1, case is models and obs
;  [Nvar]  - num of variables
;***************************************************************************
; Statistics that can be shown in metrics table:
; ---------------------------------------------------------------------------
; |                     | Stats                          | Dimsizes         |
; ---------------------------------------------------------------------------
; | Individual variables| uCORR,rms,RMSD,SD,CORR,cRMSD,  | [Ncase] x [Nvar] |
; |                     | ME(MEVD)                       |                  |
; ---------------------------------------------------------------------------
; | Multivariable field | VSC,RMSL,RMSVD,rms_std,cVSC,   | [Ncase]          |
; |                     | cRMSL,cRMSVD,SD_std,VME/MEVM   |                  |
; ---------------------------------------------------------------------------
; | Index for summary   | cMISS,uMISS                    | [Ncase]          |   
; ---------------------------------------------------------------------------
;***************************************************************************
; The defaults that the user can modify:
; opt_metrics@pltType                 = "eps"            ;# image storage format:"ps"/"eps"/"pdf"/"png" (str)
; opt_metrics@wkOrientation           = "portrait"       ;# PS or PDF output produced in "portrait"/"landscape" (str)
; opt_metrics@RMSlevs                 = (/0.6, 0.65, 0.70, 0.75, 0.80, 0.85, 0.90, 0.95, 0.98, 1.02, 1.05, \
;                                1.10, 1.15, 1.20, 1.25, 1.30, 1.35, 1.40/)
;                                                        ;# RMS levels for SD/rms, cRMSL/RMSL (array of numeric values) 
; opt_metrics@MElevs                  = (/-0.5,-0.45,-0.40,-0.35,-0.3,-0.25,-0.20,-0.15,-0.10,-0.05,0., \
;                                0.05,0.10,0.15,0.20,0.25,0.30,0.35,0.40,0.45,0.5/)
;                                                        ;# ME levels for ME, VME/MEVM (array of numeric values) 
; opt_metrics@RMSDlevs                = (/0.10, 0.15, 0.20, 0.25, 0.30, 0.35, 0.40, 0.45, 0.5/)
;                                                        ;# RMSD levels for cRMSD/RMSD, cRMSVD/RMSVD, SD_std/rms_std (array of numeric values) 
; opt_metrics@CORRlevs                = (/0.6, 0.7, 0.8, 0.84, 0.86, 0.88, 0.90, 0.91, 0.92, 0.93, 0.94, 0.95, 0.96, 0.98/)
;                                                        ;# CORR levels for CORR/uCORR, cVSC/VSC, cMISS, MISS (array of numeric values) 
; opt_metrics@RMScmap                 =                  ;# colormap for RMS levels (valid under setting 'RMSlevs')
; opt_metrics@MEcmap                  =                  ;# colormap for ME levels (valid under setting 'MElevs')
; opt_metrics@CORRcmap                =                  ;# colormap for CORR levels (valid under setting 'CORRlevs')
; opt_metrics@RMSDcmap                =                  ;# colormap for RMSD levels (valid under setting 'RMSDlevs')
;## for setting cmap:
;##  [1] a string with colorfile name
;##  [2] an array of string with named-colors and number-str between two colors, which has format as (/'color-1','N-1','color-2','N-2',
;##     ...'color-(K-1)','N-(K-1)','color-K'/), where sum(N)-K should be equal or greater than the num of levs. 
;##  [3] N x 3 (N x 4) RGB (RGBA) array of numeric values, where N is equal or greater than the num of levs plus 1. 
;##  [4] 1-D array with NhlTColorIndex
;##  [5] 1-D array with named colors
; opt_metrics@plotMEVD               = False             ;# whether to plot MEVD when ME is plotted and MEVD is in files (logical)
; opt_metrics@CaseLocation           = "Top"             ;# location of caselabels "Top"/"Left" (str)
; opt_metrics@titleHeight            = 0.08              ;# height of title box (numeric value) (valid when opt_metrics@CaseLocation='Left')	
; opt_metrics@titleWidth             = 0.1	            ;# width of title box (numeric value) (valid when opt_metrics@CaseLocation='Left')
; opt_metrics@caseHeight             = 0.035             ;# height of case box  (numeric value) (valid when opt_metrics@CaseLocation='Left')
; opt_metrics@varWidth               = 0.07              ;# width of var box (numeric value) (valid when opt_metrics@CaseLocation='Top')	
; opt_metrics@caseWidth              = 0.1	            ;# width of case box (numeric value) (valid when opt_metrics@CaseLocation='Top')	
; opt_metrics@caseHeight             = 0.02              ;# height of case box (numeric value) (valid when opt_metrics@CaseLocation='Top')
; opt_metrics@caseTHeight            = 0.0075            ;# text font height of case-label (numeric value) 
; opt_metrics@titleTHeight           = 0.015             ;# text font height of title (numeric value) 
; opt_metrics@varTHeight             = 0.006             ;# text font height of variable-label (numeric value)  
; opt_metrics@statsTHeight           = 0.0075            ;# text font height of statistic-label (numeric value) 
; opt_metrics@metricsTHeight         = 0.006             ;# text font height of metrics (numeric value) 
; opt_metrics@TitleBackgroundcolor   = "CornflowerBlue"  ;# background color for title 
; opt_metrics@VarBackgroundcolor     = "Gray70"          ;# background color for case/ststs/var  
;## Three ways to set color 
;## [1] a str with named colors, e.g.: opt_metrics@TitleBackgroundcolor = "red"
;## [2] a int value with NhlTColorIndex, e.g.: opt_metrics@TitleBackgroundcolor = 1
;## [3] 1-D array of 3/4 numeric values with RGB/RGBA, e.g.: opt_metrics@TitleBackgroundcolor = [0.5,0.5,0.5]
; opt_metrics@tableTitle             = "METRICS"         ;# title of table (str) 
; opt_metrics@showTextOn             = True              ;# show performance metrics (logical) 
; opt_metrics@decialMetricsN         = "3.2"             ;# number of decimal places for performance metrics (str) (valid under 'opt_metrics@showTextOn = True')
; opt_metrics@decialBarN             = "3.2"             ;# number of decimal places for colorbar marker (str) (valid under 'opt_metrics@showTextOn = False')
; opt_metrics@colorbarFHeight        = opt_metrics@varTHeight
;                                                        ;# colorbar font height (numeric value) (valid under 'opt_metrics@showTextOn = False')
; opt_metrics@box_legend             = False             ;# whether to plot legend for box                          
;                                                ;## [1] False - not draw legend for box
;                                                ;## [2] array of str - content of legend for box, num should match num of files in 'MVIE_filenames2'
;                                                ;##     (valid when the num of files in 'MVIE_filenames2' greater than 1)
; opt_metrics@box_lgFontHeightF      = opt_metrics@varTHeight*1.2;# text font height in box legend (numeric value) (valid under setting 'box_legend' with str)
; opt_metrics@box_lgheight           =                   ;# height of box legend (numeric value) (valid under setting 'box_legend' with str)
; opt_metrics@highlightsummary       = True              ;# enclose stats for summary, i.e.,cMISS,uMISS, with red box for emphasis (True: default) or not (False)?
; opt_metrics@draw_grid              = False             ;# draw grid on wks or not (a logical value) (False: default)
;*************************************************************************** 
begin
;***************************************************************************
   ;open workstation
   if (opt_metrics .and. isatt(opt_metrics,"pltType")) then
      wks_type = opt_metrics@pltType

      if ((.not.isstring(wks_type)) .or. (.not.isscalar(wks_type)) .or. ((wks_type.ne."eps").and. \
         (wks_type.ne."pdf").and.(wks_type.eq."ps").and.(wks_type.eq."png"))) then
         print("metrics_table: opt_metrics@pltType should be 'pdf', 'ps', 'eps', or 'png' !")
         exit
      end if   
   else  
      wks_type = "eps"
   end if
   
   if (((wks_type.eq."ps").or.(wks_type.eq."pdf")) .and. (isatt(opt_metrics,"wkOrientation"))) then
      if ((opt_metrics@wkOrientation.ne."portrait") .and. (opt_metrics@wkOrientation.ne."landscape")) then
         print("metrics_table: opt_metrics@wkOrientation should be 'portrait' or 'landscape' !")
         exit
      else
         wks_type@wkOrientation = opt_metrics@wkOrientation
      end if
   end if

   wks = gsn_open_wks(wks_type,mfname) 

   print_table([/"*Create <"+mfname+"."+wks_type+">*"/],"%s") 

   ;add files
   Nf = dimsizes(fnames)

   if (Nf.eq.1) then
      f = addfile(fnames, "r")

      if (any(.not. isfilevar(f,stats))) then
         print("metrics table error: cannot find all "+str_join(stats, ",")+" in "+fnames+" !")
         exit
      end if
   else
      if (Nf .gt. 4) then
         print("metrics table error: the max of num for fnames is 4 !")
         exit
      else
         f = addfiles(fnames, "r")
         ListSetType(f, "join")
      end if

      do j = 0,(Nf-1)
         if (any(.not. isfilevar(f[j],stats))) then
            print("metrics table error: cannot find all "+str_join(stats, ",")+" in "+fnames(j)+" !")
            exit
         end if
      end do
   end if

   ;box template
   xbox0  = (/0.0,1.0,1.0,0.0,0.0/) 
   ybox0  = (/0.0,0.0,1.0,1.0,0.0/)

   if (Nf .eq. 2) then
      ind_box = new((/2,4/),integer)
      ind_box(0,:) = (/0,2,3,4/)
      ind_box(1,:) = (/0,1,2,4/)
   end if
   if (Nf .gt. 2) then
      ind_box = new((/4,4/),integer)
      ind_box(0,:) = (/0,0,3,4/)
      ind_box(1,:) = (/2,0,3,2/)
      ind_box(2,:) = (/2,0,1,2/)
      ind_box(3,:) = (/0,0,1,0/)
   end if
;------------------------------------------------------------------
; Part 1: Check information of evaluation and read data
;------------------------------------------------------------------
   ;number of stats
   Nstats = dimsizes(stats)

   Stats1 = (/"cRMSD","SD","CORR","ME","RMSD","rms","uCORR"/)
   Stats2 = (/"cRMSVD","SD_std","cRMSL","cVSC","cMISS","uMISS", \
               "VME","MEVM","RMSVD","rms_std","RMSL","VSC","cMIEI","uMIEI"/)

   Nstats1 = 0
   do i = 0,(dimsizes(Stats1)-1)
      if (.not. ismissing(ind(Stats1(i).eq.stats))) then
         Nstats1 = Nstats1 + 1
      end if
   end do
  
   Nstats2 = 0
   do i = 0,(dimsizes(Stats2)-1)
      if (.not. ismissing(ind(Stats2(i).eq.stats))) then
         Nstats2 = Nstats2 + 1
      end if
   end do

   if ((Nstats1+Nstats2) .ne. Nstats) then
      print("metrics table error: some stats in stats_metrics2 (stats) are not provided by metrics table!")
      exit
   end if

   if (opt_metrics .and. isatt(opt_metrics, "plotMEVD")) then
      plotMEVD = opt_metrics@plotMEVD
      if ((.not.islogical(plotMEVD)) .or. (.not.isscalar(plotMEVD))) then
         print("metrics table error: opt_metrics@plotMEVD should be a logical value (True/False) !")
         exit
      end if
   else
      plotMEVD = False
   end if

   if (plotMEVD) then
      if (.not. ismissing(ind("ME".eq.stats))) then
         if (Nf .eq. 1) then
            if (isfilevar(f,"MEVD")) then
               MEVD  = f->MEVD
               nVec = num(.not.ismissing(MEVD(0,:)))
               if (dimsizes(dimsizes(MEVD)).ne.2) then
                  print("metrics_table error: MEVD in file:"+fnames+" should be 2D array!")
                  exit
               end if
               MEVD_dim := dimsizes(MEVD)
               if ((MEVD_dim(0).lt.Ncase).or.(MEVD_dim(1).lt.Nvar)) then
                  print("metrics_table error: dimsize of MEVD in file:"+fnames+" does not match Num(Model_names+Obs_names) X Num(Var_names)!")
                  exit
               end if
            end if
         else
            isfile_MEVD = True
            do i = 0,(Nf-1)
               if (.not. isfilevar(f[i],"MEVD")) then
                  isfile_MEVD = False
                  break
               else
                  if (dimsizes(dimsizes(MEVD)).ne.2) then
                     print("metrics_table error: MEVD in file:"+fnames+" should be 2D array!")
                     exit
                  end if
                  MEVD_dim := dimsizes(MEVD)
                  if ((MEVD_dim(0).lt.Ncase).or.(MEVD_dim(1).lt.Nvar)) then
                     print("metrics_table error: dimsize of MEVD in file:"+fnames+" does not match Num(Model_names+Obs_names) X Num(Var_names)!")
                     exit
                  end if
               end if
            end do

            if (isfile_MEVD) then
               MEVD = new((/Nf,Ncase,Nvar/),float)
               do i = 0,(Nf-1)
                  iMEVD:= f[i]->MEVD
                  MEVD(i,:,:) = iMEVD(0:(Ncase-1),0:(Nvar-1))
               end do
              
               nVec = num(.not.ismissing(MEVD(0,0,:)))
            end if
         end if   
      end if
   end if
   if (.not. isvar("nVec")) then
      nVec = 0
   end if

   ;number of case
   if ((.not.isstring(caseNames)) .or. (dimsizes(dimsizes(caseNames)).ne.1)) then
      print("metrics table error: caseNames should be an array of string !")
      exit
   end if
   Ncase = dimsizes(caseNames)

   ;number of var
   if ((.not.isstring(varNames)) .or. (dimsizes(dimsizes(varNames)).ne.1)) then
      print("metrics table error: varNames should be an array of string !")
      exit
   end if
   Nvar  = dimsizes(varNames)
;------------------------------------------------------------------
; Part 2: Get color levels and create color map
;------------------------------------------------------------------
   With_colormap = new(4, logical)
   With_colormap = False

   do it = 0,3
      if (it .eq. 0) then
         Stats_it := (/"ME","VME","MEVM"/)
      end if
      if (it .eq. 1) then
         Stats_it := (/"cRMSD","RMSD","cRMSVD","RMSVD","SD_std","rms_std","cMIEI","uMIEI"/)
      end if
      if (it .eq. 2) then
         Stats_it := (/"SD","rms","cRMSL","RMSL"/)
      end if
      if (it .eq. 3) then
         Stats_it := (/"CORR","uCORR","cVSC","VSC","cMISS","uMISS"/)
      end if

      do is = 0,(dimsizes(Stats_it)-1)
         if (.not. ismissing(ind(Stats_it(is).eq.stats))) then
            With_colormap(it) = True
            break
         end if
      end do
   end do

   ;get levels and colormap 
   levs_names = (/"ME","RMSD","RMS","CORR"/)+"levs"
   cmap_names = (/"ME","RMSD","RMS","CORR"/)+"cmap"

   do it = 0,3
      if (With_colormap(it)) then
         if (opt_metrics .and. isatt(opt_metrics,levs_names(it))) then
            levs := opt_metrics@$levs_names(it)$
            dim_lev := dimsizes(levs)

            if ((.not.isnumeric(levs)) .or. (dimsizes(dim_lev).ne.1) .or. (.not.isMonotonic(levs))) then
               print("metrics table error: opt_metrics@"+levs_names(it)+" should 1D monotonic array of numeric values !")
               exit
            end if

            if (isatt(opt_metrics,cmap_names(it))) then
               cmap0 := opt_metrics@$cmap_names(it)$

               if (isnumeric(cmap0)) then
                  dim_cmap0 := dimsizes(cmap0)
                  if ((dimsizes(dim_cmap0).eq.1).and.isint(cmap0)) then ;Type-1: 1-D array of NhlTColorIndex
                     if (dim_cmap0.lt.(dim_lev+1)) then
                        print("metrics table error: not enough NhlTColorIndex("+dim_cmap0+")in opt_metrics@"+cmap_names(it)+\
                           " to match "+levs_names(it)+" !")
                        exit
                     end if
                     if (min(cmap0).lt.0) then
                        print("metrics table error: when set opt_metrics@"+cmap_names(it)+" with NhlTColorIndex, it should"+\
                           " be greater than 0!")
                        exit
                     end if
                     cmap := cmap0(0:dim_lev)
                  else if ((dimsizes(dim_cmap0).eq.2).and.((dim_cmap0(1).eq.3).or.(dim_cmap0(1).eq.4))) then ;Type-2: Nx3 RGB (Nx4 RGBA) array
                           if (dim_cmap0(0).lt.(dim_lev+1)) then
                              print("metrics table error: not enough RGB(RGBA) colors in opt_metrics@"+cmap_names(it)+\
                                 " to match "+levs_names(it)+" !")
                              exit
                           end if
                           if (min(cmap0).lt.0) then
                              print("metrics table error: when set opt_metrics@"+cmap_names(it)+" with RGB/RGBA, it should"+\
                                 " be greater than 0!")
                              exit
                           end if
                           cmap := new((/dim_lev+3,dim_cmap0(1)/), float)
                           cmap(0,:)  = 1
                           cmap(1,:)  = 0
                           if (typeof(cmap0) .ne. "float") then
                              cmap(2:,:) = tofloat(cmap0(0:dim_lev,:))
                           else
                              cmap(2:,:) = cmap0(0:dim_lev,:)
                           end if
                        else
                           print("metrics table error: error in opt_metrics@"+cmap_names(it)+"! Please refer"+\
                              " to readme.namelist.NCL-v or User Guide!")
                           exit
                        end if
                  end if
               else if (isstring(cmap0)) then
                        if (dimsizes(cmap0) .eq. 1) then  ;Type-3: Colorcmap name
                           colorbar := read_colormap_file(cmap0)
                           dim_bar  := dimsizes(colorbar)

                           if (dim_bar(0) .lt. (dim_lev+1)) then 
                              print("metrics table error: error in opt_metrics@"+cmap_names(it)+" ! Colors in "\
                                 + cmap0 + ".rgb" + "are not enough for levs.")   
                              exit
                           end if

                           cmap := new((/dim_lev+3,dim_bar(1)/), float)
                           cmap(0,:)  = 1
                           cmap(1,:)  = 0
                           cmap(2:,:) = span_color_rgba(cmap0, dim_lev+1)
                        else if (dimsizes(dimsizes(cmap0)) .eq. 1) then                     
                                 dim_cmap0 := dimsizes(cmap0)

                                 if (all(.not.is_string_numeric(cmap0))) then  ;Type-4: 1-D array with named color
                                    if (dim_cmap0.lt.(dim_lev+1)) then
                                       print("metrics table error: not enough named colors in opt_metrics@"+cmap_names(it)+\
                                          " to match "+levs_names(it)+" !")
                                       exit
                                    end if
                                    cmap := NhlGetNamedColorIndex(wks,cmap0)
                                    if (min(cmap).lt.0) then
                                       print("metrics table error: there is invalid named color in opt_metrics@"+cmap_names(it)+"!")
                                       exit
                                    end if
                                 else
                                    if (tofloat(dim_cmap0)/2 .eq. (dim_cmap0/2)) then
                                       print("metrics table error: error in opt_metrics@"+cmap_names(it)+" ! If cmap " + \
                                          "set with named-colors and number-str, it should format as (/'color-1','N-1','color-2'" + \
                                          ",'N-2',...'color-(K-1)','N-(K-1)','color-K'/).")
                                       exit
                                    end if

                                    colors := cmap0(ispan(0, dim_cmap0-1, 2))
                                    colors_range := cmap0(ispan(1, dim_cmap0-2, 2))

                                    if (any(.not. is_string_numeric(colors_range))) then
                                       print("metrics table error: error in opt_metrics@"+cmap_names(it)+" ! If cmap " + \
                                          "set with named colors, it should format as (/'color-1','N-1','color-2'" + \
                                          ",'N-2',...'color-(K-1)','N-(K-1)','color-K'/).")
                                       exit
                                    end if
                                    if (min(NhlGetNamedColorIndex(wks,colors)).lt.0) then
                                       print("metrics table error: there is invalid named color in opt_metrics@"+cmap_names(it)+"!")
                                       exit
                                    end if

                                    colors_range := toint(colors_range)

                                    if ((sum(colors_range)-(dimsizes(colors_range)-1)) .lt. (dim_lev+1)) then
                                       print("metrics table error: error in opt_metrics@"+cmap_names(it)+" ! If cmap " + \
                                          "set with named colors with format as (/'color-1','N-1','color-2'" + \
                                          ",'N-2',...'color-(K-1)','N-(K-1)','color-K'/), sum(N) - K  should " + \
                                          "be equal or greater than num of levs.")
                                       exit
                                    end if
                              
                                    opt_color := True
                                    opt_color@NumColorsInRange = toint(colors_range)
                                    cmap1 := span_named_colors(colors, opt_color)

                                    cmap := cmap1(0:(dim_lev+2),:)
                                 end if
                              else
                                 print("metrics table error: error in opt_metrics@"+cmap_names(it)+"! Please refer"+\
                                    " to readme.namelist.NCL-v or User Guide!")
                                 exit
                              end if
                        end if
                     else
                        print("metrics table error: error in opt_metrics@"+cmap_names(it)+"! Please refer"+\
                           " to readme.namelist.NCL-v or User Guide!")
                        exit
                     end if
               end if
            else
               cmap := new((/dim_lev+3,4/), float)
               cmap(0,:) = 1
               cmap(1,:) = 0

               if (it .eq. 0) then ;MEcmap
                  colorbar  := read_colormap_file("cmocean_curl")

                  if (max(levs).lt.0.) then
                     cmap(2:,:) = span_color_rgba(colorbar(26:126,:), dim_lev+1)
                  else
                     if (min(levs).gt.0.) then
                        cmap(2:,:) = span_color_rgba(colorbar(127:217,:), dim_lev+1)
                     else
                        N1 := num(levs.lt.0)
                        N2 := num(levs.gt.0)
                        cmap(2:(N1+1),:) = span_color_rgba(colorbar(26:120,:), N1)

                        if (.not.ismissing(ind(0. .eq.levs))) then
                           cmap((N1+2):(N1+3),:) = 1
                           cmap((N1+4):,:)  = span_color_rgba(colorbar(133:217,:), N2)
                        else
                           cmap((N1+2),:) = 1
                           cmap((N1+3):,:)  = span_color_rgba(colorbar(133:217,:), N2)
                        end if
                     end if
                  end if
               end if
                           
               if (it .eq. 1) then ;RMSDcmap
                  colorbar  := read_colormap_file("WhiteYellowOrangeRed")
                  cmap(2:,:) = span_color_rgba(colorbar(10:200,:), dim_lev+1)
               end if

               if (it .eq. 2) then ;RMScmap
                  colorbar1  := read_colormap_file("MPL_Blues")
                  colorbar2  := read_colormap_file("MPL_Reds")

                  if (max(levs) .lt. 1.) then
                     cmap(2:,:) = span_color_rgba(colorbar1(97:10,:), dim_lev+1)
                  else
                     if (min(levs) .gt. 1.) then
                        cmap(2:,:) = span_color_rgba(colorbar2(10:97,:), dim_lev+1)
                     else
                        N1 := num(levs.lt.1)
                        N2 := num(levs.gt.1)
                        cmap(2:(N1+1),:) = span_color_rgba(colorbar1(97:10,:), N1)

                        if (.not. ismissing(ind(1. .eq.levs))) then
                           cmap((N1+4):,:)  = span_color_rgba(colorbar2(10:97,:), N2)
                           cmap((N1+2):(N1+3),:) = 1
                        else 
                           cmap((N1+3):,:)  = span_color_rgba(colorbar2(10:97,:), N2)
                           cmap((N1+2),:) = 1
                        end if
                     end if
                  end if
               end if

               if (it .eq. 3) then ;CORRcmap
                  colorbar  := read_colormap_file("WhiteGreen")
                  cmap(2:,:) = span_color_rgba(colorbar(175:10,:), dim_lev+1)
               end if
            end if
         else   
            if (it .eq. 0) then ;MElevs, MEcmap
               levs := (/-0.5,-0.45,-0.40,-0.35,-0.3,-0.25,-0.20,-0.15,-0.10,-0.05,0., \
                        0.05,0.10,0.15,0.20,0.25,0.30,0.35,0.40,0.45,0.5/)
               cmap := new((/dimsizes(levs)+3,4/), float)
               cmap(0,:) = 1
               cmap(1,:) = 0     

               colorbar  := read_colormap_file("cmocean_curl")
               cmap(2:12,:) = span_color_rgba(colorbar(26:120,:), 11)
               cmap(13:,:)  = span_color_rgba(colorbar(133:217,:), 11)
            end if

            if (it .eq. 1) then ;RMSDlevs, RMSDcmap
               levs := (/0.10, 0.15, 0.20, 0.25, 0.30, 0.35, 0.40, 0.45, 0.5/)
               cmap := new((/dimsizes(levs)+3,4/), float)
               cmap(0,:) = 1
               cmap(1,:) = 0   
               cmap(2:,:) = read_colormap_file("sunshine_9lev")
            end if

            if (it .eq. 2) then ;RMSlevs, RMScmap
               levs := (/0.6, 0.65, 0.70, 0.75, 0.80, 0.85, 0.90, 0.95, 0.98, 1.02,\
                              1.05, 1.10, 1.15, 1.20, 1.25, 1.30, 1.35, 1.40/)
               cmap := new((/dimsizes(levs)+3,4/), float)
               cmap(0,:) = 1
               cmap(1,:) = 0   
               cmap(2:,:) = read_colormap_file("temp_diff_18lev")
            end if

            if (it .eq. 3) then ;CORRlevs, CORRcmap
               levs := (/0.6, 0.7, 0.8, 0.84, 0.86, 0.88, 0.90, 0.91, 0.92, 0.93, 0.94, 0.95, 0.96, 0.98/)
               cmap := new((/dimsizes(levs)+3,4/), float)
               cmap(0,:) = 1
               cmap(1,:) = 0

               colorbar  := read_colormap_file("WhiteGreen")
               cmap(2:,:) = span_color_rgba(colorbar(175:10,:), dimsizes(levs)+1)
            end if
         end if
      end if

      if (With_colormap(it)) then
         if (it .eq. 0) then
            MElevs = levs
            MEcmap = cmap
         end if
         if (it .eq. 1) then
            RMSDlevs = levs
            RMSDcmap = cmap
         end if
         if (it .eq. 2) then
            RMSlevs = levs
            RMScmap = cmap
         end if
         if (it .eq. 3) then
            CORRlevs = levs
            CORRcmap = cmap
         end if
      end if
   end do
;------------------------------------------------------------------
; Part 3: Parameters for metrics table
;------------------------------------------------------------------
   ;whether to show metrics on 
   if (opt_metrics .and. isatt(opt_metrics,"showTextOn")) then
      if ((.not.islogical(opt_metrics@showTextOn)) .or. (.not.isscalar(opt_metrics@showTextOn))) then
         print("metrics table error: opt_metrics@showTextOn should be a logical (True/False) !")
         exit
      else
         Show_T = opt_metrics@showTextOn
      end if
   else
      Show_T = True
   end if
   
   if (Nf .gt. 1) then
      Show_T = False
   end if

   ;structure of metrics table   
   if (opt_metrics .and. isatt(opt_metrics,"CaseLocation")) then
      CaseL = opt_metrics@CaseLocation

      if ((.not.isstring(CaseL)) .or. (.not.isscalar(CaseL)) .or. \
         ((CaseL.ne."Left").and.(CaseL.ne."Top"))) then 
         print("metrics table error: opt_metrics@CaseLocation should be a string ('Top'/'Left') !")
         exit
      end if
   else
      CaseL = "Top"
   end if

   if (CaseL.eq."Top") then
      CaseTopOn = True
   else
      CaseTopOn = False
   end if
  
   ;get size of different modules in table 
   if (CaseTopOn) then ;caselabels on the top 
      if (opt_metrics .and. isatt(opt_metrics,"caseHeight")) then 
         case_height = opt_metrics@caseHeight

         if ((.not.isnumeric(case_height)) .or. (.not.isscalar(case_height))) then
            print("metrics table error: opt_metrics@caseHeight should be a numeric value !")
            exit
         end if
      else
         case_height = 0.02
      end if

      if (opt_metrics .and. isatt(opt_metrics,"caseWidth")) then 
         case_width = opt_metrics@caseWidth

         if ((.not.isnumeric(case_width)) .or. (.not.isscalar(case_width))) then
            print("metrics table error: opt_metrics@caseWidth should be a numeric value !")
            exit
         end if
      else
         case_width = 0.1
      end if
  
      if (opt_metrics .and. isatt(opt_metrics,"varWidth")) then
         var_width = opt_metrics@varWidth
         if ((.not.isnumeric(var_width)) .or. (.not.isscalar(var_width))) then
            print("metrics table error: opt_metrics@varWidth should be a numeric value !")
            exit
         end if
      else
         var_width = 0.07
      end if

      title_height = case_width
      var_height   = (zoom-case_width)/tofloat(Nstats1*Nvar+Nstats2+nVec)    
      stats_height = var_height
    
      stats1_width = var_height*Nvar
      stats2_width = var_width+var_height
      title_width  = var_width+stats_height
   else  ;caselabels in the left
      if (opt_metrics .and. isatt(opt_metrics,"titleHeight")) then 
         title_height = opt_metrics@titleHeight

         if ((.not.isnumeric(title_height)) .or. (.not.isscalar(title_height))) then
            print("metrics table error: opt_metrics@titleHeight should be a numeric value !")
            exit
         end if
      else
         title_height = 0.08
      end if

      if (opt_metrics .and. isatt(opt_metrics,"titleWidth")) then
         title_width = opt_metrics@titleWidth

         if ((.not.isnumeric(title_width)) .or. (.not.isscalar(title_width))) then
            print("metrics table error: opt_metrics@titleWidth should be a numeric value !")
            exit
         end if
      else
         title_width = 0.1
      end if
  
      if (opt_metrics .and. isatt(opt_metrics,"caseHeight")) then
         case_height = opt_metrics@caseHeight

         if ((.not.isnumeric(case_height)) .or. (.not.isscalar(case_height))) then
            print("metrics table error: opt_metrics@caseHeight should be a numeric value !")
            exit
         end if
      else
         case_height = 0.035
      end if

      case_width    = title_width 
      stats1_height = title_height/2.
      var_height    = stats1_height
      var_width     = (zoom-title_width)/(Nvar*Nstats1+Nstats2+nVec)
      stats2_height = var_width
      stats2_width  = title_height
      stats1_width  = Nvar*var_width
   end if
   
   ;get font height in table 
   if (opt_metrics .and. isatt(opt_metrics,"caseTHeight")) then 
      case_theight = opt_metrics@caseTHeight

      if ((.not.isnumeric(case_theight)) .or. (.not.isscalar(case_theight))) then
         print("metrics table error: opt_metrics@caseTHeight should be a numeric value !")
         exit
      end if
   else
      case_theight = 0.0075
   end if

   if (opt_metrics .and. isatt(opt_metrics,"titleTHeight")) then
      title_theight = opt_metrics@titleTHeight

      if ((.not.isnumeric(title_theight)) .or. (.not.isscalar(title_theight))) then
         print("metrics table error: opt_metrics@titleTHeight should be a numeric value !")
         exit
      end if
   else
      title_theight = 0.015
   end if  
  
   if (opt_metrics .and. isatt(opt_metrics,"varTHeight")) then 
      var_theight = opt_metrics@varTHeight

      if ((.not.isnumeric(var_theight)) .or. (.not.isscalar(var_theight))) then
         print("metrics table error: opt_metrics@varTHeight should be a numeric value !")
         exit
      end if
   else
      var_theight = 0.006
   end if  

   if (opt_metrics .and. isatt(opt_metrics,"statsTHeight")) then
      stats_theight = opt_metrics@statsTHeight

      if ((.not.isnumeric(stats_theight)) .or. (.not.isscalar(stats_theight))) then
         print("metrics table error: opt_metrics@statsTHeight should be a numeric value !")
         exit
      end if
   else
      stats_theight = 0.0075
   end if

   if (opt_metrics .and. Show_T) then
      if (opt_metrics .and. isatt(opt_metrics,"metricsTHeight")) then
         metrics_theight = opt_metrics@metricsTHeight

         if ((.not.isnumeric(metrics_theight)) .or. (.not.isscalar(metrics_theight))) then
            print("metrics table error: opt_metrics@metricsTHeight should be a numeric value !")
            exit
         end if
      else
         metrics_theight = 0.006
      end if
   end if

   ;get table title and filling color 
   if (opt_metrics .and. isatt(opt_metrics,"tableTitle")) then
      table_title = opt_metrics@tableTitle

      if ((.not.isstring(table_title)) .or. (.not.isscalar(table_title))) then
         print("metrics table error: opt_metrics@tableTitle should be a string !")
         exit
      end if
   else
      table_title = "METRICS"
   end if

   if (opt_metrics .and. isatt(opt_metrics,"TitleBackgroundcolor")) then
      TitleBackgroundcolor = opt_metrics@TitleBackgroundcolor

      if (isscalar(TitleBackgroundcolor)) then
         if (isstring(TitleBackgroundcolor)) then
            if (NhlGetNamedColorIndex(wks,TitleBackgroundcolor).lt.0) then
               print("metrics table error: invalid named color in opt_metrics@TitleBackgroundcolor!")
               exit
            end if
         else if (isint(TitleBackgroundcolor)) then
                  if (TitleBackgroundcolor.lt.0) then
                     print("metrics table error: NhlTColorIndex in opt_metrics@TitleBackgroundcolor should be greater than 0!")
                     exit
                  end if
               else
                  print("metrics table error: error in opt_metrics@TitleBackgroundcolor! "+\
                     "Please refer to readme.namelist.NCL-v or User Guide !")
                  exit
               end if
         end if
      else if (((dimsizes(dimsizes(TitleBackgroundcolor)).eq.1)).and.(isnumeric(TitleBackgroundcolor))) then
               if (((dimsizes(TitleBackgroundcolor).ne.3).and.(dimsizes(TitleBackgroundcolor).ne.4))\
                  .or.(min(TitleBackgroundcolor).lt.0)) then
                  print("metrics table error: when set opt_metrics@TitleBackgroundcolor with RGB(RGBA) array, its dimsize"+\
                     " should be 3 or 4 and the values should be positive!")
                  exit
               end if
            else
               print("metrics table error: error in opt_metrics@TitleBackgroundcolor! "+\
                  "Please refer to readme.namelist.NCL-v or User Guide !")
               exit
            end if
      end if
   else
      TitleBackgroundcolor = "CornflowerBlue"
   end if
  
   if (opt_metrics .and. isatt(opt_metrics,"VarBackgroundcolor")) then
      VarBackgroundcolor = opt_metrics@VarBackgroundcolor

      if (isscalar(VarBackgroundcolor)) then
         if (isstring(VarBackgroundcolor)) then
            if (NhlGetNamedColorIndex(wks,VarBackgroundcolor).lt.0) then
               print("metrics table error: invalid named color in opt_metrics@VarBackgroundcolor!")
               exit
            end if
         else if (isint(VarBackgroundcolor)) then
                  if (VarBackgroundcolor.lt.0) then
                     print("metrics table error: NhlTColorIndex in opt_metrics@VarBackgroundcolor should be greater than 0!")
                     exit
                  end if
               else
                  print("metrics table error: error in opt_metrics@VarBackgroundcolor! "+\
                     "Please refer to readme.namelist.NCL-v or User Guide !")
                  exit
               end if
         end if
      else if (((dimsizes(dimsizes(VarBackgroundcolor)).eq.1)).and.(isnumeric(VarBackgroundcolor))) then
               if (((dimsizes(VarBackgroundcolor).ne.3).and.(dimsizes(VarBackgroundcolor).ne.4))\
                  .or.(min(VarBackgroundcolor).lt.0)) then
                  print("metrics table error: when set opt_metrics@VarBackgroundcolor with RGB(RGBA) array, its dimsize"+\
                     " should be 3 or 4 and the values should be positive!")
                  exit
               end if
            else
               print("metrics table error: error in opt_metrics@VarBackgroundcolor! "+\
                  "Please refer to readme.namelist.NCL-v or User Guide !")
               exit
            end if
      end if
   else
      VarBackgroundcolor = "Gray70"
   end if

   ;whether to plot legend of box
   if (Nf.gt.1) then
      if (opt_metrics .and. isatt(opt_metrics,"box_legend")) then
         if (islogical(opt_metrics@box_legend) .and. (opt_metrics@box_legend.eq.False)) then
            with_boxleg  = False
         else
            if (isstring(opt_metrics@box_legend)) then
               box_legend   = opt_metrics@box_legend
               with_boxleg  = True

               if ((dimsizes(dimsizes(box_legend)).ne.1) .or. (dimsizes(box_legend).ne.Nf)) then
                  print("metrics table error: opt_metrics@box_legend should be False or an array of string, "+\
                        "or it does not match num of datafiles input ! ")
                  exit
               end if
            else
               print("metrics table error: opt_metrics@box_legend should be False or an array of string !")
               exit
            end if
         end if
      else
         with_boxleg = False
      end if

      if (with_boxleg) then
         if (opt_metrics .and. isatt(opt_metrics,"box_lgFontHeightF")) then 
            boxlgFontHeightF = opt_metrics@box_lgFontHeightF

            if ((.not.isnumeric(boxlgFontHeightF)) .or. (.not.isscalar(boxlgFontHeightF))) then
               print("metrics table error: opt_metrics@box_lgFontHeightF should be a numeric value !")
               exit
            end if        
         end if

         if (opt_metrics .and. isatt(opt_metrics,"box_lgheight")) then
            boxleg_height = opt_metrics@box_lgheight

            if ((.not.isnumeric(boxleg_height)) .or. (.not.isscalar(boxleg_height))) then
               print("metrics table error: opt_metrics@box_lgheight should be a numeric value !")
               exit
            end if
         end if  
      end if

      spl_Res                  = True
      spl_Res@gsLineThicknessF = 1.0
      spl_Res@gsLineColor      = VarBackgroundcolor
   else
      with_boxleg = False
   end if

   ;plot frame for cMISS/uMISS
   if (opt_metrics .and. isatt(opt_metrics,"highlightsummary")) then
      plot_box = opt_metrics@highlightsummary

      if ((.not.islogical(plot_box)).or.(.not.isscalar(plot_box))) then
         print("metrics_table: opt_metrics@highlightsummary should be True or False !")
         exit
      end if
   else
      plot_box = True
   end if

   ;parameters for colorbar
   if (Show_T) then
      mv_tRes               = True
	   mv_tRes@txFontHeightF = metrics_theight

      if (opt_metrics .and. isatt(opt_metrics,"decialMetricsN")) then
         if ((.not.isstring(opt_metrics@decialMetricsN)) .or. (.not.isscalar(opt_metrics@decialMetricsN))) then        
            print("metrics table error: opt_metrics@decialMetricsN should be a string, and has format as '3.2' !")
            exit
         end if
    
         decial_Show = "%"+opt_metrics@decialMetricsN+"f"
      else
         decial_Show = "%3.2f"
      end if
   else

      lt_Res = True

      if (opt_metrics .and. isatt(opt_metrics,"colorbarFHeight")) then 
         if ((.not.isnumeric(opt_metrics@colorbarFHeight)) .or. (.not.isscalar(opt_metrics@colorbarFHeight))) then
            print("metrics table error: opt_metrics@colorbarFHeight should be a numeric value !")
            exit
         end if
         lt_Res@txFontHeightF   = opt_metrics@colorbarFHeight
      else
         lt_Res@txFontHeightF   = var_theight                   
      end if

      if (opt_metrics .and. isatt(opt_metrics,"decialBarN")) then
         if ((.not.isstring(opt_metrics@decialBarN)) .or. (.not.isscalar(opt_metrics@decialBarN))) then       
            print("metrics table error: opt_metrics@decialBarN should be a string, and has format as '3.2' !")
            exit
         end if
         decial_Bar = "%"+opt_metrics@decialBarN+"f"
      else
         decial_Bar = "%3.2f"
      end if   
   end if 
;------------------------------------------------------------------
; Part 3: Plot table
;------------------------------------------------------------------  
   ;plot table title
   xbox = title_width*xbox0
   ybox = 1.-title_height*ybox0

   tt_pRes               = True
   tt_pRes@gsFillColor  := TitleBackgroundcolor
  
   gsn_polygon_ndc(wks,xbox,ybox,tt_pRes)

   tt_tRes               = True
   tt_tRes@txFontHeightF = title_theight

   ixtbox = xbox(0)+0.5*(xbox(1)-xbox(0))
   iytbox = ybox(0)+0.5*(ybox(2)-ybox(0))
 
   gsn_text_ndc(wks,table_title,ixtbox,iytbox,tt_tRes)
   gsn_polyline_ndc(wks,xbox,ybox,False)
  
   ;plot caselabels 
   mn_pRes               = True
   mn_pRes@gsFillColor  := VarBackgroundcolor
  
   mn_tRes               = True
   mn_tRes@txFontHeightF = case_theight

   if (CaseTopOn) then
      do ic = 0,(Ncase-1)
         xbox   = max(xbox)+case_height*xbox0

         gsn_polygon_ndc(wks,xbox,ybox,mn_pRes)

         ixtbox = xbox(0)+0.5*(xbox(1)-xbox(0))
         iytbox = ybox(0)+0.5*(ybox(2)-ybox(0))

         mn_tRes@txAngleF      = 90

         gsn_text_ndc(wks,caseNames(ic),ixtbox,iytbox,mn_tRes)
         gsn_polyline_ndc(wks,xbox,ybox,False)    
      end do
   else
      do ic = 0,(Ncase-1)
         ybox   = min(ybox)-case_height*ybox0

         gsn_polygon_ndc(wks,xbox,ybox,mn_pRes)

         ixtbox = xbox(0)+0.5*(xbox(1)-xbox(0))
         iytbox = ybox(0)+0.5*(ybox(2)-ybox(0))

         mn_tRes@txAngleF      = 0

         gsn_text_ndc(wks,caseNames(ic),ixtbox,iytbox,mn_tRes)
         gsn_polyline_ndc(wks,xbox,ybox,False)    
      end do
   end if

   ;plot metrics
   mv_pRes = True

   colorbar_title  = new(4,string)
   colorbar_title  = "//"
   With_colorbar   = new(4,"logical")

   split_L = new((/4,2/),"float")
   L_str   = where(CaseTopOn, min(ybox), max(xbox))
   NLine   = 0

   do it = 0,3   ; 4 types of colormaps
      if (it .eq. 0) then
         iStats1 := "ME"
         iStats2 := (/"VME","MEVM"/)

         if (isvar("MEcmap")) then
            cmap := MEcmap
            levs := MElevs
         end if
      end if
      if (it .eq. 1) then
         iStats1 := (/"cRMSD","RMSD"/)
         iStats2 := (/"cRMSVD","RMSVD","SD_std","rms_std","cMIEI","uMIEI"/)

         if (isvar("RMSDcmap")) then
            cmap := RMSDcmap
            levs := RMSDlevs
         end if
      end if
      if (it .eq. 2) then
         iStats1 := (/"SD","rms"/)
         iStats2 := (/"cRMSL","RMSL"/)

         if (isvar("RMScmap")) then
            cmap := RMScmap
            levs := RMSlevs
         end if
      end if
      if (it .eq. 3) then
         iStats1 := (/"CORR","uCORR"/)
         iStats2 := (/"cVSC","VSC","cMISS","uMISS"/)

         if (isvar("CORRcmap")) then
            cmap := CORRcmap
            levs := CORRlevs
         end if
      end if
      if (dimsizes(dimsizes(cmap)).eq.1) then
         Cmap_isIndex = True
      else
         Cmap_isIndex = False
      end if

      iN1 = dimsizes(iStats1)
      iN2 = dimsizes(iStats2)

      do is = 0,(iN1-1)
         if (.not. ismissing(ind(iStats1(is).eq.stats))) then
            if (Nf .eq. 1) then
               metrics := f->$iStats1(is)$
            else
               metrics := f[:]->$iStats1(is)$
            end if
            dimM := dimsizes(metrics)
            if (Nf .eq. 1) then
               if (dimM(0).lt.Ncase) then
                  print("metrics table error: dimsizes of var-"+iStats1(is)+" in MVIE_filenames2 does not match num(Model_names+Obs_names)! ")
                  exit
               end if
               if (dimM(1).lt.Nvar) then
                  print("metrics table error: dimsizes of var-"+iStats1(is)+" in MVIE_filenames2 does not match num(Var_names)! ")
                  exit
               end if
            else
               if (dimM(1).lt.Ncase) then
                  print("metrics table error: dimsizes of var-"+iStats1(is)+" in MVIE_filenames2 does not match num(Model_names+Obs_names)! ")
                  exit
               end if
               if (dimM(2).lt.Nvar) then
                  print("metrics table error: dimsizes of var-"+iStats1(is)+" in MVIE_filenames2 does not match num(Var_names)! ")
                  exit
               end if
            end if

            colorbar_title(it) = colorbar_title(it) +"/"+ iStats1(is)

            if (CaseTopOn) then
               xbox = stats_height*xbox0
               ybox = min(ybox) - (stats1_width+where(it.eq.0, nVec*var_height, 0))*ybox0
            else
               xbox = max(xbox) + (stats1_width+where(it.eq.0, nVec*var_width, 0))*xbox0
               ybox = 1. - stats1_height*ybox0
            end if

            mn_tRes@txAngleF = where(CaseTopOn, 90, 0)

            gsn_polygon_ndc(wks,xbox,ybox,mn_pRes)
            gsn_polyline_ndc(wks,xbox,ybox,False)

            ixtbox = xbox(0)+0.5*(xbox(1)-xbox(0))
            iytbox = ybox(0)+0.5*(ybox(2)-ybox(0))

            mn_tRes@txFontHeightF = stats_theight
           
            gsn_text_ndc(wks,iStats1(is),ixtbox,iytbox,mn_tRes)

            do iv = 0,(Nvar-1)
               if ((it.eq.0) .and. (nVec.ne.0)) then
                  if ((Nf.eq.1) .and. (.not.ismissing(MEVD(0,iv)))) then
                     iv_Vec = True
                  else
                     if ((Nf.gt.1) .and. (.not.ismissing(MEVD(0,0,iv)))) then
                        iv_Vec = True
                     else
                        iv_Vec = False
                     end if
                  end if
               else
                  iv_Vec = False
               end if

               if (CaseTopOn) then
                  xbox = stats_height + var_width*xbox0
                  ybox = where(iv.eq.0, max(ybox), min(ybox)) - where(iv_Vec, 2, 1)*var_height*ybox0           
               else
                  xbox = where(iv.eq.0, min(xbox), max(xbox)) + where(iv_Vec, 2., 1.)*var_width*xbox0          
                  ybox = 1. - stats1_height - var_height*ybox0
               end if

               gsn_polygon_ndc(wks,xbox,ybox,mn_pRes)
               gsn_polyline_ndc(wks,xbox,ybox,False)

               ixtbox = xbox(0)+0.5*(xbox(1)-xbox(0))
               iytbox = ybox(0)+0.5*(ybox(2)-ybox(0))

               mn_tRes@txFontHeightF = var_theight
               mn_tRes@txAngleF      = 0

               gsn_text_ndc(wks,varNames(iv),ixtbox,iytbox,mn_tRes)

               if (iv_Vec) then     
                  if (CaseTopOn) then
                     ybox     = max(ybox) - var_height*ybox0
                     ybox_add = min(ybox) - var_height*ybox0
                  else
                     xbox     = min(xbox) + var_width*xbox0
                     xbox_add = max(xbox) + var_width*xbox0
                  end if
               end if

               do ic = 0,(Ncase-1)
                  if (CaseTopOn) then
                     xbox = max(xbox) + case_height*xbox0
                  else
                     ybox = min(ybox) - case_height*ybox0
                  end if

                  if (iv_Vec) then
                     if (CaseTopOn) then
                        xbox_add = xbox
                     else
                        ybox_add = ybox
                     end if
                  end if

                  if (Nf.eq.1) then
                     if (Cmap_isIndex) then
                        mv_pRes@gsFillColor := GetFillColorIndex(levs,cmap,metrics(ic,iv))
                     else
                        mv_pRes@gsFillColor := GetFillColor(levs,cmap,metrics(ic,iv))
                     end if

                     gsn_polygon_ndc(wks,xbox,ybox,mv_pRes)
                     gsn_polyline_ndc(wks,xbox,ybox,False)

                     if (Show_T) then
                        ixtbox = xbox(0)+0.5*(xbox(1)-xbox(0))
                        iytbox = ybox(0)+0.5*(ybox(2)-ybox(0))

                        gsn_text_ndc(wks,sprintf(decial_Show,metrics(ic,iv)),ixtbox,iytbox,mv_tRes)
                     end if
                  else
                     mv_pRes@gsFillColor := VarBackgroundcolor   
                     gsn_polygon_ndc(wks,xbox,ybox,mv_pRes)      

                     ixtbox = xbox(0)+0.5*(xbox(1)-xbox(0))
                     iytbox = ybox(0)+0.5*(ybox(2)-ybox(0))

                     do ig = 0,(Nf-1)
                        tr_xbox := xbox(ind_box(ig,:))
                        tr_ybox := ybox(ind_box(ig,:))

                        if (Nf.gt.2) then
                           tr_xbox(1) = ixtbox
                           tr_ybox(1) = iytbox
                        end if
                        
                        if (Cmap_isIndex) then
                           mv_pRes@gsFillColor := GetFillColorIndex(levs, cmap, metrics(ig,ic,iv))
                        else
                           mv_pRes@gsFillColor := GetFillColor(levs,cmap,metrics(ig,ic,iv))
                        end if
                        gsn_polygon_ndc(wks,tr_xbox,tr_ybox,mv_pRes)
                     end do
   
                     gsn_polyline_ndc(wks,xbox,ybox,spl_Res)
                  end if

                  if (iv_Vec) then
                     if (Nf .eq. 1) then
                        if (Cmap_isIndex) then
                           mv_pRes@gsFillColor := GetFillColorIndex(levs, cmap, MEVD(ic,iv)/180.)
                        else
                           mv_pRes@gsFillColor := GetFillColor(levs, cmap ,MEVD(ic,iv)/180.)
                        end if

                        gsn_polygon_ndc(wks,xbox_add,ybox_add,mv_pRes)
                        gsn_polyline_ndc(wks,xbox_add,ybox_add,False)

                        if (Show_T) then
                           ixtbox = xbox_add(0)+0.5*(xbox_add(1)-xbox_add(0))
                           iytbox = ybox_add(0)+0.5*(ybox_add(2)-ybox_add(0))

                           gsn_text_ndc(wks,sprintf("%4.1f",MEVD(ic,iv)),ixtbox,iytbox,mv_tRes)
	                     end if 
                     else
                        mv_pRes@gsFillColor := VarBackgroundcolor           
                        gsn_polygon_ndc(wks,xbox_add,ybox_add,mv_pRes)      

                        ixtbox = xbox_add(0)+0.5*(xbox_add(1)-xbox_add(0))
                        iytbox = ybox_add(0)+0.5*(ybox_add(2)-ybox_add(0))

                        do ig = 0,(Nf-1)
                           tr_xbox := xbox_add(ind_box(ig,:))
                           tr_ybox := ybox_add(ind_box(ig,:))

                           if (Nf.gt.2) then
                              tr_xbox(1) = ixtbox
                              tr_ybox(1) = iytbox
                           end if

                           if (Cmap_isIndex) then
                              mv_pRes@gsFillColor := GetFillColorIndex(levs, cmap, MEVD(ig,ic,iv)/180.)
                           else 
                              mv_pRes@gsFillColor := GetFillColor(levs,cmap,MEVD(ig,ic,iv)/180.)
                           end if
                           gsn_polygon_ndc(wks,tr_xbox,tr_ybox,mv_pRes)
                        end do

                        gsn_polyline_ndc(wks,xbox_add,ybox_add,spl_Res)  
                     end if
                  end if
               end do

               if (iv_Vec) then
                  if (CaseTopOn) then
                     ybox = ybox_add
                  else
                     xbox = xbox_add
                  end if
               end if
            end do
         end if
      end do

      do is = 0,(iN2-1)
         if (.not. ismissing(ind(iStats2(is).eq.stats))) then
            if (((iStats2(is).eq."cMISS") .or. (iStats2(is).eq."uMISS")) \
               .and. (.not.isvar("MISS_loc"))) then
               MISS_loc = where(CaseTopOn, min(ybox), max(xbox))
            end if
            if (((iStats2(is).eq."cMIEI") .or. (iStats2(is).eq."uMIEI"))) then
               if (.not.isvar("MISS_loc")) then
                  MIEI_loc = where(CaseTopOn, min(ybox), max(xbox))
                  nMIEI = 1
               else
                  nMIEI = nMIEI + 1
               end if
            end if

            if (Nf.eq.1) then
               metrics := f->$iStats2(is)$
            else
               metrics := f[:]->$iStats2(is)$
            end if

            dimM := dimsizes(metrics)
            if (Nf .eq. 1) then
               if (dimM(0).lt.Ncase) then
                  print("metrics table error: dimsizes of var-"+iStats2(is)+" in MVIE_filenames2 does not match num(Model_names+Obs_names)! ")
                  exit
               end if
            else
               if (dimM(1).lt.Ncase) then
                  print("metrics table error: dimsizes of var-"+iStats2(is)+" in MVIE_filenames2 does not match num(Model_names+Obs_names)! ")
                  exit
               end if         
            end if
         
            colorbar_title(it) = colorbar_title(it) +"/"+ iStats2(is)
          
            if (CaseTopOn) then
               xbox = stats2_width*xbox0
               ybox = min(ybox) - stats_height*ybox0
            else
               xbox = max(xbox) + stats2_height*xbox0
               ybox = 1. - stats2_width*ybox0
            end if

            mn_tRes@txAngleF  = where(CaseTopOn, 0, 90)

            gsn_polygon_ndc(wks,xbox,ybox,mn_pRes)
            gsn_polyline_ndc(wks,xbox,ybox,False)

            ixtbox = xbox(0)+0.5*(xbox(1)-xbox(0))
            iytbox = ybox(0)+0.5*(ybox(2)-ybox(0))

            mn_tRes@txFontHeightF = stats_theight
            
            gsn_text_ndc(wks,iStats2(is),ixtbox,iytbox,mn_tRes)

            do ic = 0,(Ncase-1)
               if (CaseTopOn) then      
                  xbox = max(xbox) + case_height*xbox0
               else
                  ybox = min(ybox) - case_height*ybox0
               end if

               if (Nf.eq.1) then
                  if (Cmap_isIndex) then
                     mv_pRes@gsFillColor := GetFillColorIndex(levs,cmap,metrics(ic))
                  else    
                     mv_pRes@gsFillColor := GetFillColor(levs,cmap,metrics(ic))
                  end if

                  gsn_polygon_ndc(wks,xbox,ybox,mv_pRes)
                  gsn_polyline_ndc(wks,xbox,ybox,False)

                  if (Show_T) then
                     ixtbox = xbox(0)+0.5*(xbox(1)-xbox(0))
                     iytbox = ybox(0)+0.5*(ybox(2)-ybox(0))

                     gsn_text_ndc(wks,sprintf(decial_Show,metrics(ic)),ixtbox,iytbox,mv_tRes)
                  end if
               else
                  mv_pRes@gsFillColor := VarBackgroundcolor   ;;;;
                  gsn_polygon_ndc(wks,xbox,ybox,mv_pRes)      ;;;;

                  ixtbox = xbox(0)+0.5*(xbox(1)-xbox(0))
                  iytbox = ybox(0)+0.5*(ybox(2)-ybox(0))

                  do ig = 0,(Nf-1)
                     tr_xbox := xbox(ind_box(ig,:))
                     tr_ybox := ybox(ind_box(ig,:))

                     if (Nf.gt.2) then
                        tr_xbox(1) = ixtbox
                        tr_ybox(1) = iytbox
                     end if

                     if (Cmap_isIndex) then
                        mv_pRes@gsFillColor := GetFillColorIndex(levs,cmap,metrics(ig,ic))
                     else 
                        mv_pRes@gsFillColor := GetFillColor(levs,cmap,metrics(ig,ic))
                     end if
                     gsn_polygon_ndc(wks,tr_xbox,tr_ybox,mv_pRes)
                  end do
            
                  gsn_polyline_ndc(wks,xbox,ybox,spl_Res)
               end if
            end do
         end if
      end do

      if (L_str .ne. where(CaseTopOn, min(ybox), max(xbox))) then
         L_str = where(CaseTopOn, min(ybox), max(xbox))
         split_L(NLine,:) = L_str

         NLine = NLine + 1

         With_colorbar(it) = True
      else
         With_colorbar(it) = False
      end if 
   end do 

   ;plot split lines for colormap
   if (NLine .gt. 1) then
      line_Res                 := True
      line_Res@gsLineThicknessF = 3.0

      if (CaseTopOn) then
         xline = (/0, title_width+Ncase*case_height/)
         yline = split_L
      else
         xline = split_L 
         yline = (/1., 1.-title_height-Ncase*case_height/)
      end if

      do il = 0,(NLine-2)
         if (CaseTopOn) then
            gsn_polyline_ndc(wks,xline,yline(il,:),line_Res) 
         else
            gsn_polyline_ndc(wks,xline(il,:),yline,line_Res) 
         end if
      end do
   end if

   ;plot frame for colorarea
   if (CaseTopOn) then
      xbox_frame = title_width + (Ncase*case_height)*xbox0
      ybox_frame = 1. - title_height - (zoom-title_height)*ybox0
   else
      xbox_frame = case_width + (zoom-case_width)*xbox0
      ybox_frame = 1. - title_height - Ncase*case_height*ybox0
   end if
   gsn_polyline_ndc(wks, xbox_frame, ybox_frame, False)

   ;plot frame for cMISS/uMISS
   if (plot_box .and. isvar("MISS_loc")) then
      if (CaseTopOn) then
         xbox_frame := (Ncase*case_height+title_width)*xbox0                                
         ybox_frame := MISS_loc + (min(ybox)-MISS_loc)*ybox0
      else                             
         xbox_frame := MISS_loc + (max(xbox)-MISS_loc)*xbox0
         ybox_frame := 1. - (title_height+Ncase*case_height)*ybox0
      end if 

      line_Res                 := True
      line_Res@gsLineThicknessF = 3.0
      line_Res@gsLineColor      = "red"

      gsn_polyline_ndc(wks, xbox_frame, ybox_frame, line_Res)
   end if  

   if (plot_box .and. isvar("MIEI_loc")) then
      if (CaseTopOn) then
         xbox_frame := (Ncase*case_height+title_width)*xbox0                                
         ybox_frame := MIEI_loc - nMIEI*stats_height*ybox0
      else                             
         xbox_frame := MIEI_loc + nMIEI*stats2_height*xbox0
         ybox_frame := 1. - (title_height+Ncase*case_height)*ybox0
      end if 

      line_Res                 := True
      line_Res@gsLineThicknessF = 3.0
      line_Res@gsLineColor      = "red"

      gsn_polyline_ndc(wks, xbox_frame, ybox_frame, line_Res)
   end if  

   ;plot color labels and box legend
   if (.not. Show_T) then               
      lt_Res                      = True 
      lt_Res@txJust               = 1   

      lb_height   = case_height*1.3

      if (with_boxleg) then
         if (.not. isvar("boxleg_height")) then
            boxleg_height = lb_height * num(With_colorbar)
         end if
         
         if (CaseTopOn) then 
            boxleg_width  = boxleg_height * (case_height/var_height) 
         else
            boxleg_width  = boxleg_height * (var_width/case_height)  
         end if
      else
         boxleg_height = 0.
         boxleg_width  = 0.
      end if

      table_width  = where(CaseTopOn, title_width+Ncase*case_height, zoom) 
      border_width = case_height*0.15

      lb_x     = where(with_boxleg,border_width,0) + boxleg_width + \
                  lt_Res@txFontHeightF * max(strlen(colorbar_title)-2.) 
      lb_width = table_width - lb_x - border_width

      lb_Res                      = True
      lb_Res@lbPerimOn            = False                              ; no label bar box
      lb_Res@lbOrientation        = "Horizontal"                       ; orientation
      lb_Res@lbLabelFontHeightF   = lt_Res@txFontHeightF*0.9           ; label font height
      lb_Res@lbLabelAlignment     = "InteriorEdges"                    ; where to label             
      lb_Res@lbMonoFillPattern    = True                               ; fill sold
      lb_Res@vpWidthF             = lb_width                           ; size
      lb_Res@vpHeightF            = lb_height
      lb_Res@lbLabelFont          = 22
      
      lb_y   = min(ybox) - where(nVec.eq.0, 0.4*case_height, 0.8*case_height)

      lt_x = border_width + boxleg_width 
      lt_y = lb_y - 0.5*lb_height 

      if (with_boxleg) then
         xbox_leg = border_width + boxleg_width*xbox0

         ybox_leg = lb_y -  abs((lb_height*num(With_colorbar)-boxleg_height)/2.) \
                        - boxleg_height*ybox0

         lg_box = True

         if (isvar("boxlgFontHeightF")) then
            lg_box@txFontHeightF = boxlgFontHeightF
         else
            lg_box@txFontHeightF = var_theight*1.2
         end if

         if (Nf .eq. 2) then
            xtbox2 = (/min(xbox_leg)+0.04*boxleg_width, max(xbox_leg)-0.04*boxleg_width/)          
            ytbox2 = (/min(ybox_leg)+0.2*boxleg_height, max(ybox_leg)-0.2*boxleg_height/)

            txJust2 = (/"bottomleft", "topright"/)

            do ig = 0,1
               ixbox := xbox_leg(ind_box(ig,:))
               iybox := ybox_leg(ind_box(ig,:))
               gsn_polyline_ndc(wks,ixbox,iybox,False)

               lg_box@txJust := txJust2(ig)
               gsn_text_ndc(wks, box_legend(ig), xtbox2(ig), ytbox2(ig), lg_box)             
            end do
         else
            gsn_polyline_ndc(wks,xbox_leg,ybox_leg,False)

            xtbox4 = (/min(xbox_leg)+0.025*boxleg_width, min(xbox_leg)+0.5*boxleg_width, \
                        max(xbox_leg)-0.025*boxleg_width, min(xbox_leg)+0.5*boxleg_width/)
            ytbox4 = (/min(ybox_leg)+0.5*boxleg_height, min(ybox_leg)+0.1*boxleg_height, \
                        min(ybox_leg)+0.5*boxleg_height, max(ybox_leg)-0.1*boxleg_height/)
            txJust4 = (/"centerleft", "bottomcenter", "centerright", "topcenter"/)
         
            do ig = 0,(Nf-1)      
               ixbox := xbox_leg(ind_box(ig,:))
               iybox := ybox_leg(ind_box(ig,:))

               ixbox(1) = min(xbox_leg)+0.5*boxleg_width
               iybox(1) = min(ybox_leg)+0.5*boxleg_height

               gsn_polyline_ndc(wks,ixbox,iybox,False)

               lg_box@txJust := txJust4(ig)
               gsn_text_ndc(wks, box_legend(ig), xtbox4(ig), ytbox4(ig), lg_box)
            end do
         end if 
      end if

      notisfirst = False

      do ib = 0,3
         if (With_colorbar(ib)) then
            if (ib .eq. 0) then
               cmap := MEcmap
               levs := MElevs
            end if
            if (ib .eq. 1) then
               cmap := RMSDcmap
               levs := RMSDlevs
            end if
            if (ib .eq. 2) then
               cmap := RMScmap
               levs := RMSlevs
            end if
            if (ib .eq. 3) then
               cmap := CORRcmap
               levs := CORRlevs
            end if

            if (notisfirst) then
               lb_y = lb_y - lb_height
               lt_y = lb_y - 0.5*lb_height
            else
               notisfirst = True
            end if

            lb_Res@lbFillColors := cmap(2:,:)

            if ((ib.eq.0) .and. (nVec.gt.0)) then
               levs2_lb = sprintf("(%3.0f)",levs*180)

               if (num(abs(levs).gt.1.) .gt. 0) then
                  levs2_lb(ind(abs(levs).gt.1.)) = " "
               end if

               addlb_Res = True
               addlb_Res@txFontColor   = "firebrick"
               addlb_Res@txFontHeightF = lb_Res@lbLabelFontHeightF*0.95
               addlb_Res@txFont        = 22

               sub_width = lb_Res@vpWidthF / (dimsizes(levs)+1.)

               do i = 1,(dimsizes(levs)+1)
                  sub_x = lb_x + i*sub_width
                  gsn_text_ndc(wks, levs2_lb(i-1), lb_x + lb_Res@vpWidthF,lb_y,addlb_Res)
               end do

               addlb_Res@txJust = "CenterRight"
               gsn_text_ndc(wks, "(MEVD)",lb_x + lb_Res@vpWidthF,lb_y,addlb_Res)
            end if

            gsn_text_ndc(wks,str_sub_str(colorbar_title(ib), "///", "  "),lt_x,lt_y,lt_Res)
   
            gsn_labelbar_ndc (wks,dimsizes(levs)+1,sprintf(decial_Bar,levs),lb_x,lb_y,lb_Res)
         end if
      end do

      xbox = table_width*xbox0

      if ((with_boxleg).and. \
         ((min(ybox) - lb_y + lb_height).lt.(abs(lb_height*num(With_colorbar)-boxleg_height)+boxleg_height))) then
         ybox = min(ybox) - 0.2*case_height - (abs(lb_height*num(With_colorbar)-boxleg_height)+boxleg_height)*ybox0
      else
         ybox = min(ybox) - 0.2*case_height - (min(ybox) - lb_y + lb_height)*ybox0
      end if

      line_Res := True
      line_Res@gsLineColor = VarBackgroundcolor

      gsn_polyline_ndc(wks,xbox,ybox,line_Res)
   end if

   ;plot grid line
   if (opt_metrics .and. isatt(opt_metrics,"draw_grid")) then
      if ((.not.isscalar(opt_metrics@draw_grid)).or.(.not.islogical(opt_metrics@draw_grid))) then
         print("metrics table error: 'draw_grid' of opt_metrics should be bool!")
         exit
      end if
      if (opt_metrics@draw_grid) then
         drawNDCGrid(wks)
      end if
   end if

   ;draw table                 
   draw(wks)  
   frame(wks)
;***************************************************************************
end

