#################################################################
import Ngl
import os
import numpy as np
import Nio
#################################################################
# Choose a color to fill in a plot (line/gon/marker) based on 
# secondary scalar field.
#################################################################
def GetFillColor(cnlvls, cmap, data:float):
    if data < cnlvls[0]:
        return(cmap[0])
    elif data > cnlvls[-1]:
        return(cmap[-1])
    else:
        for i in np.arange(1,len(cnlvls)):
            if data < cnlvls[i]:
                return(cmap[i])
#----------------------------------------------------------------
#################################################################
# Judge 1D array is monotonoc
#################################################################
def isMonotonic(array):
    isNotIncrea = True
    isNotDecrea = True
    for i in range(1,len(array)):
        if array[i]>array[i-1]:
            isNotIncrea = False
        if array[i]<array[i-1]:
            isNotDecrea = False

    M = 0

    if (isNotDecrea or isNotIncrea):
        if array[0] < array[1]:
            M = 1
        elif array[0] > array[1]:
            M = -1

    return M
#################################################################
# Span colors from RGB/RGBA array
#################################################################
def Span_color_rgb(color,num,opt=1):
    if opt == 1:
        color_ind = np.int32(Ngl.fspan(0,color.shape[0]-1,num))
    else:
        color_ind = np.int32(Ngl.fspan(color.shape[0]-1,0,num))

    return color[color_ind,:]
#################################################################
# Function to draw the Metrics Table in MVIE
#################################################################
# To report bugs or make suggestions, please send emails at:
#  zhangmengzhuo1996@163.com , xuzhf@tea.ac.cn
#################################################################
# Arguments:
#  mfname      : name of output table
#  fnames      : datafile of statistics for plotting
#  stats       : names of statistics chosen to be plotted
#  caseNames   : names of model/observation/reanalysis [Ncase]
#  varNames    : names of vars [Nvar]
#  zoom        : width of the table
#  opt_metrics : Ngl.Resources()
# where:
#  [Ncase] - num of cases
#            - if num of obs is 1, case is models
#            - if num of obs is greater than 1, case is models and obs
#  [Nvar]  - num of variables
#################################################################
# Statistics that can be shown in metrics table:
# ---------------------------------------------------------------------------
# |                     | Stats                          | Shape            |
# ---------------------------------------------------------------------------
# | Individual variables| uCORR,rms,RMSD,SD,CORR,cRMSD,  | [Ncase] x [Nvar] |
# |                     | ME(MEVD)                        |                 |
# ---------------------------------------------------------------------------
# | Multivariable field | VSC,RMSL,RMSVD,rms_std,cVSC,   | [Ncase]          |
# |                     | cRMSL,cRMSVD,SD_std,VME/MEVM   |                  |
# ---------------------------------------------------------------------------
# | Index for summary   | cMISS,uMISS                    | [Ncase]          |   
# ---------------------------------------------------------------------------
#################################################################
# The defaults that the user can modify:
# opt_metrics.pltType                 = "eps"            ## image storage format:"ps"/"eps"/"pdf"/"png" (str)
# opt_metrics.wkOrientation           = "portrait"       ## PS or PDF output produced in "portrait"/"landscape" 
# opt_metrics.RMSlevs                 = [0.6, 0.65, 0.70, 0.75, 0.80, 0.85, 0.90, 0.95, 0.98, 1.02, 1.05, \
#    1.10, 1.15, 1.20, 1.25, 1.30, 1.35, 1.40]           ## RMS levels for SD/rms, cRMSL/RMSL (list/tuple of int/flt)                                       
# opt_metrics.MElevs                  = [-0.5,-0.45,-0.40,-0.35,-0.3,-0.25,-0.20,-0.15,-0.10,-0.05,0., \
#    0.05,0.10,0.15,0.20,0.25,0.30,0.35,0.40,0.45,0.5]   ## ME levels for ME, VME/MEVM (list/tuple of int/flt)  
# opt_metrics.RMSDlevs                = [0.10, 0.15, 0.20, 0.25, 0.30, 0.35, 0.40, 0.45, 0.5]
#                                                        ## RMSD levels for cRMSD/RMSD, cRMSVD/RMSVD, SD_std/rms_std (list/tuple of int/flt)  
# opt_metrics.CORRlevs                = [0.6, 0.7, 0.8, 0.84, 0.86, 0.88, 0.90, 0.91, 0.92, 0.93, 0.94, 0.95, 0.96, 0.98]
#                                                        ## CORR levels for CORR/uCORR, cVSC/VSC, cMISS, MISS (list/tuple of int/flt)  
# opt_metrics.RMScmap                 =                  ## colormap for RMS levels (valid under setting 'RMSlevs')
# opt_metrics.MEcmap                  =                  ## colormap for ME levels (valid under setting 'MElevs')
# opt_metrics.CORRcmap                =                  ## colormap for CORR levels (valid under setting 'CORRlevs')
# opt_metrics.RMSDcmap                =                  ## colormap for RMSD levels (valid under setting 'RMSDlevs')
### Four ways to set cmap:
### [1] list/tuple consisting of str with named colors, e.g.: ["red","blue","green"]
### [2] list/tuple consisting of int with NhlTColorIndex, e.g.: [1,2,3,4,5]
### [3] str with colormap name, e.g.: "cmocean_curl"
### [4] list/tuple consisting of RGB/RGBA-list/tuple, e.g.:[[0.5,0.5,0.5],[0.1,0.1,0.1],[0.4,0.5,0.3]]
# opt_metrics.plotMEVD               = False             ## whether to plot MEVD when ME is plotted and MEVD is in files (bool)
# opt_metrics.CaseLocation           = "Top"             ## location of caselabels "Top"/"Left" (str)
# opt_metrics.titleHeight            = 0.08              ## height of title box (int/flt) (valid when opt_metrics.CaseLocation='Left')	
# opt_metrics.titleWidth             = 0.1	             ## width of title box (int/flt) (valid when opt_metrics.CaseLocation='Left')
# opt_metrics.caseHeight             = 0.035             ## height of case box  (int/flt) (valid when opt_metrics.CaseLocation='Left')
# opt_metrics.varWidth               = 0.07              ## width of var box (int/flt) (valid when opt_metrics.CaseLocation='Top')	
# opt_metrics.caseWidth              = 0.1	             ## width of case box (int/flt) (valid when opt_metrics.CaseLocation='Top')	
# opt_metrics.caseHeight             = 0.02              ## height of case box (int/flt) (valid when opt_metrics.CaseLocation='Top')
# opt_metrics.caseTHeight            = 0.0075            ## text font height of case-label (int/flt) 
# opt_metrics.titleTHeight           = 0.015             ## text font height of title (int/flt) 
# opt_metrics.varTHeight             = 0.006             ## text font height of variable-label (int/flt)  
# opt_metrics.statsTHeight           = 0.0075            ## text font height of statistic-label (int/flt) 
# opt_metrics.metricsTHeight         = 0.006             ## text font height of metrics (int/flt) 
# opt_metrics.TitleBackgroundcolor   = "CornflowerBlue"  ## background color for title 
# opt_metrics.VarBackgroundcolor     = "Gray70"          ## background color for case/ststs/var 
#### Three ways to set color
###[1] str with named colors, e.g.: opt_metrics.TitleBackgroundcolor = "red"
###[2] int with NhlTColorIndex, e.g.: opt_metrics.TitleBackgroundcolor = 1
###[3] list/tuple consisting of 3/4 int/float with RGB/RGBA, e.g.: opt_metrics.TitleBackgroundcolor = [0.5,0.5,0.5]
# opt_metrics.tableTitle             = "METRICS"         ## title of table (str) 
# opt_metrics.showTextOn             = True              ## show performance metrics (bool) 
# opt_metrics.decialMetricsN         = "3.2"             ## number of decimal places for performance metrics (str) (valid under 'opt_metrics.showTextOn = True')
# opt_metrics.decialBarN             = "3.2"             ## number of decimal places for colorbar marker (str) (valid under 'opt_metrics.showTextOn = False')
# opt_metrics.colorbarFHeight        = opt_metrics.varTHeight
#                                                        ## colorbar font height (int/flt) (valid under 'opt_metrics.showTextOn = False')
# opt_metrics.box_legend             = False             ## whether to plot legend for box                          
#                                                        ### [1] False - not draw legend for box
#                                                        ### [2] array of str - content of legend for box, num should match num of files in 'MVIE_filenames2'
#                                                        ###     (valid when the num of files in 'MVIE_filenames2' greater than 1)
# opt_metrics.box_lgFontHeightF      = opt_metrics.varTHeight*1.2
#                                                        ## text font height in box legend (int/flt) (valid under setting 'box_legend' with str)
# opt_metrics.box_lgheight           =                   ## height of box legend (int/flt) (valid under setting 'box_legend' with str)
# opt_metrics.highlightsummary       = True              ## enclose stats for summary, i.e.,cMISS,uMISS, with red box for emphasis (True: default) or not (False)?
# opt_metrics.draw_grid              = False             ## draw grid on wks or not (a bool value) (False: default)
#################################################################
def metrics_table(mfname:str, fnames:(list,tuple), stats:(list,tuple),caseNames:(list,tuple), varNames:(list,tuple), \
    zoom:(int,float), opt_metrics:object):
    # set plotting format 
    if hasattr(opt_metrics,"pltType"):
        if (getattr(opt_metrics,"pltType")!="eps") and (getattr(opt_metrics,"pltType")!="pdf") and\
            (getattr(opt_metrics,"pltType")!="ps") and (getattr(opt_metrics,"pltType")!="png"):
            print("metrics_table error: opt_metrics.pltType should be 'pdf', 'ps', 'eps', or 'png' !")
            os._exit(0)
        else:
            wks_type = opt_metrics.pltType
    else:
        wks_type = "eps"

    wksRes = Ngl.Resources()
    if ((wks_type == "eps")or(wks_type == "pdf")) and hasattr(opt_metrics,"wkOrientation"):
        if (getattr(opt_metrics,"wkOrientation")!="portrait") and  (getattr(opt_metrics,"wkOrientation")!="landscape"):
            print("metrics_table error: opt_metrics.wkOrientation should be 'portrait' or 'landscape' !")
            os._exit(0)
        else:
            wksRes.wkOrientation = opt_metrics.wkOrientation

    wks = Ngl.open_wks(wks_type, mfname, wksRes) 

    # check stats
    if (not isinstance(stats,(list,tuple))) or (not all(isinstance(x,str) for x in stats)):
        print("metrics_table error: 'stats' should be list/tuple consisting of str!")
        os._exit(0)

    # add files
    if (not isinstance(fnames,(list,tuple))) or (not all(isinstance(x,str) for x in fnames)):
        print("metrics_table error: 'fnames' should be list/tuple consisting of str!")
        os._exit(0)

    Nf = len(fnames)

    if Nf == 1:
        f = Nio.open_file(fnames[0], "r")

        if not all(x in f.variables.keys() for x in stats):
            print("metrics_table error: cannot find all stats:"+",".join(stats)," in ",fnames,"!")
            os._exit(0)
    else:
        if Nf > 4:
            print("metrics_table error: the max of num for fnames is 4 !")
            os._exit(0)

        f = []

        for i in fnames:
            ifile = Nio.open_file(i, "r")
            if not all(x in ifile.variables.keys() for x in stats):
                print("metrics_table error: cannot find all stats:"+",".join(stats)," in ",i,"!")
                os._exit(0)
            f.append(ifile)

    # box template
    xbox0 = np.array((0.0,1.0,1.0,0.0,0.0),float)
    ybox0 = np.array((0.0,0.0,1.0,1.0,0.0),float)

    if Nf == 2:
        ind_box = [[0,2,3,4],[0,1,2,4]]
    if Nf > 2:
        ind_box = [[0,0,3,4],[2,0,3,2],[2,0,1,2],[0,0,1,0]]

    #=========Part 1:Check information of evaluation and read data
    # number of var
    if (not isinstance(varNames,(list,tuple))) or (not all(isinstance(x,str) for x in varNames)):
        print("metrics table error: varNames should be list/tuple of string !")
        os._exit(0)
    Nvar = len(varNames)

    # num of stats
    Nstats = len(stats)
    
    Stats1 = ["cRMSD","SD","CORR","ME","RMSD","rms","uCORR"]
    Stats2 = ["cRMSVD","SD_std","cRMSL","cVSC","cMISS","uMISS","VME", \
              "RMSVD","rms_std","RMSL","VSC","cMISS","uMISS"]

    Nstats1 = sum(x in Stats1 for x in stats)
    Nstats2 = sum(x in Stats2 for x in stats)

    if (Nstats1+Nstats2) != Nstats:
        print("metrics table error: some stats in stats_metrics2 (stats) are not provided by metrics table!")
        os._exit(0)

    if hasattr(opt_metrics,"plotMEVD"):
        plotMEVD = opt_metrics.plotMEVD
        if not isinstance(plotMEVD,bool):
            print("metrics table error: opt_metrics.plotMEVD should be bool!")
            os._exit(0)
    else:
        plotMEVD = False

    if plotMEVD and ("ME" in stats):
        if Nf == 1:
            if "MEVD" in f.variables.keys():
                MEVD = f.variables["MEVD"]
                nVec = Nvar - np.nansum(np.isnan(MEVD[0,:]))
            else:
                nVec = 0.
        else:
            isfile_MEVD = True
            for ifile in f:
                if not "MEVD" in ifile.variables.keys():
                    isfile_MEVD = False
                    break

            if isfile_MEVD:
                for i in range(Nf):
                    if i == 0:
                        shp0 = f[i].variables["MEVD"].shape
                        MEVD = np.zeros((Nf,shp0[0],shp0[1]),float)
                    else:
                        if f[i].variables["MEVD"].shape!=shp0:
                           isfile_MEVD = False
                           break 
                    MEVD[i,:,:] = f[i].variables["MEVD"]
            if isfile_MEVD:
                nVec = Nvar - np.nansum(np.isnan(MEVD[0,0,:]))
            else:
                nVec = 0.
    else:
        nVec = 0.

    # number of case

    if (not isinstance(caseNames,(tuple,list))) or (not all(isinstance(x,str) for x in caseNames)):
        print("metrics table error: caseNames should be list/tuple of string !")
        os._exit(0)
    Ncase = len(caseNames)

    #=================Part 2:Get color levels and create color map

    Stats_all = [["ME","VME","MEVM"],["cRMSD","RMSD","cRMSVD","RMSVD","SD_std","rms_std"],\
        ["SD","rms","cRMSL","RMSL"],["CORR","uCORR","cVSC","VSC","cMISS","uMISS"]]
    With_colormap = []

    for i in range(4):
        if sum(x in Stats_all[i] for x in stats) > 0:
            With_colormap.append(True)
        else:
            With_colormap.append(False)

    levs_names = list(map(lambda x:x+"levs", ["ME","RMSD","RMS","CORR"]))
    cmap_names = list(map(lambda x:x+"cmap", ["ME","RMSD","RMS","CORR"]))

    for i in range(4):
        if With_colormap[i]:
            if hasattr(opt_metrics,levs_names[i]):
                levs = eval("opt_metrics."+levs_names[i])   # get color level

                if (not isinstance(levs,(tuple,list))) or (not all(isinstance(x,(int,float)) for x in levs)):
                    print("metrics table error: opt_metrics.",levs_names[i]," should be list/tuple consisting of int/float values!")
                    os._exit(0)

                levs = np.array(levs, dtype=float)
                if isMonotonic(levs) == 0:
                    print("metrics table error: opt_metrics.",levs_names[i]," should be monotonic!")
                    os._exit(0)
  
                if hasattr(opt_metrics,cmap_names[i]):
                    cmap0 = eval("opt_metrics."+cmap_names[i])    # get color map 

                    if isinstance(cmap0,str): # type1 for cmap: colorfile [str]
                        try:
                            colorbar = Ngl.read_colormap_file(cmap0)
                        except:
                            print("metrics table error: cannot read colorfile in opt_metrics.",cmap_names[i],"!")
                            os._exit(0)
                       
                        if colorbar.shape[0] < (levs.shape[0]+1):
                            print("metrics table error: colors in colorfile of opt_metrics.",cmap_names[i]," are not enough for levs!")
                            os._exit(0)
   
                        cmap = Span_color_rgb(colorbar,levs.shape[0]+1)
                    elif isinstance(cmap0,(tuple,list)):
                        if all(isinstance(tuple,list) for x in cmap0):  # type2 for cmap: RGB/RGBA [list/tuple consisting of RGB(RGBA)-list/tuple]
                            if not all(len(x)==len(cmap[0]) for x in cmap0):
                                print("metrics table error: when setting opt_metrics.",cmap_names[i]," with RGB or RGBA, "\
                                    "it should be tuple/list of which elements should be in the same length!") 
                                os._exit(0)
   
                            try:
                                cmap0 = np.array(cmap0, dtype=float)
                            except:
                                print("metrics table error: when setting opt_metrics.",cmap_names[i]," with RGB or RGBA, "\
                                    "it should be tuple/list of which elements are in the same shape!")       
                                os._exit(0)

                            if (cmap0.ndim != 2) or ((cmap0.shape[1]!=3) and (cmap0.shape[1]!=4)):
                                print("metrics table error: when setting opt_metrics.",cmap_names[i]," with RGB or RGBA,"\
                                    "it should be tuple/list of which element consists of 3 or 4 values!")
                                os._exit(0)
                            if cmap0.shape[0] < (levs.shape[0]+1):
                                print("metrics table error: colors in opt_metrics.",cmap_names[i],\
                                    " are not enough for opt_metrics.",levs_names[i],"!")
                                os._exit(0)
                            if min(cmap0) < 0.:
                                print("metrics table error: when setting opt_metrics.",cmap_names[i]," with RGB or RGBA,"\
                                    "values in it should be larger than 0!")
                                os._exit(0)

                            cmap = cmap0[0:(levs.shape[0]+1),:]
                        elif all(isinstance(x,str) for x in cmap0): # type3 for cmap: colornames [list/tuple consisting of str]
                            if len(cmap0) < (levs.shape[0]+1):
                                print("metrics table error: named colors in opt_metrics.",cmap_names[i]," are not enough for levs!")
                                os._exit(0)
                            for x in cmap0:
                                if Ngl.get_named_color_index(wks,x) < 0:
                                    print("metrics table error: named colors-",x," in opt_metrics",cmap_names[i]," is invalid!")
                                    os._exit(0)
                            cmap = cmap0[0:levs.shape[0]]
                        elif all(isinstance(x,int) for x in cmap0): # type4 for cmap: NhlTColorIndex [list/tuple consisting of int]
                            if len(cmap0) < (levs.shape[0]+1):
                                print("metrics table error: NhlTColorIndex in opt_metrics.",cmap_names[i]," are not enough for levs!")
                                os._exit(0)
                            if min(cmap0) < 0:
                                print("metrics table error: NhlTColorIndex in opt_metrics.",cmap_names[i]," should be greater than 0 !")
                                os._exit(0)
                            cmap = cmap0[0:levs.shape[0]]
                        else:
                            print("metrics table error: when opt_metrics.",cmap_names[i]," set to tuple/list, it should use RGB or RGBA",\
                                "[e.g.:((0.5,0.2,0.3),(0.1,0.5,0.6),(0.6,0.7,0.2))] or named colors [e.g.,('blue','red','black')] or NhlTColorIndex.")
                            os._exit(0)
                    else:
                        print("metrics table error: error in opt_metrics.",cmap_names[i]," !")
                        os._exit(0)
                else:
                    cmap = np.zeros((levs.shape[0]+1,4),float)

                    if isMonotonic(levs) == -1 :
                        levs = levs[::-1]
                    
                    if i == 0:   # ME cmap
                        colorbar = Ngl.read_colormap_file("cmocean_curl")

                        if max(levs) <= 0:     
                            cmap[:,:] = Span_color_rgb(colorbar[26:127,:],levs.shape[0]+1)
                        elif min(levs) >= 0:
                            cmap[:,:] = Span_color_rgb(colorbar[127:217,:],levs.shape[0]+1)
                        else:
                            N1 = sum(levs < 0)
                            N2 = sum(levs > 0)

                            cmap[0:N1,:] = Span_color_rgb(colorbar[26:121,:],N1)
                            if 0 in levs:
                                cmap[N1:(N1+2),:] = 1
                                cmap[(N1+2):,:] = Span_color_rgb(colorbar[133:218,:],N2)
                            else:
                                cmap[N1,:] = 1
                                cmap[(N1+1):,:] = Span_color_rgb(colorbar[133:218,:],N2)
                    
                    if i == 1:   # RMSD cmap
                        colorbar = Ngl.read_colormap_file("WhiteYellowOrangeRed")
                        cmap[:,:] = Span_color_rgb(colorbar[10:201,:],levs.shape[0]+1)

                    if i == 2:   # RMS cmap
                        if max(levs) < 1:
                            colorbar = Ngl.read_colormap_file("MPL_Blues")
                            cmap[:,:] = Span_color_rgb((colorbar[10:98,:])[::-1,:],levs.shape[0]+1)
                        elif min(levs) > 1:
                            colorbar = Ngl.read_colormap_file("MPL_Reds")
                            cmap[:,:] = Span_color_rgb(colorbar[10:98,:],levs.shape[0]+1)
                        else:
                            N1 = sum(levs < 1)
                            N2 = sum(levs > 1)

                            colorbar = Ngl.read_colormap_file("MPL_Blues")
                            cmap[0:N1,:] = Span_color_rgb((colorbar[10:98,:])[::-1,:],N1)

                            colorbar = Ngl.read_colormap_file("MPL_Reds")

                            if 1 in levs:
                                cmap[N1:(N1+2),:] = 1
                                cmap[(N1+2):,:] = Span_color_rgb(colorbar[10:98,:], N2)
                            else:
                                cmap[N1,:] = 1
                                cmap[(N1+1):,:] = Span_color_rgb(colorbar[10:98,:], N2)

                    if i == 3:   # CORR cmap
                        colorbar = Ngl.read_colormap_file("WhiteGreen")
                        cmap[:,:] = Span_color_rgb((colorbar[10:176,:])[::-1,:],levs.shape[0]+1)
            else:
                if i == 0:   # ME levs & cmap
                    levs = np.array((-0.5,-0.45,-0.40,-0.35,-0.3,-0.25,-0.20,-0.15,-0.10,-0.05,0.,\
                        0.05,0.10,0.15,0.20,0.25,0.30,0.35,0.40,0.45,0.5),float)
                    cmap = np.zeros((levs.shape[0]+1,4),float)

                    colorbar = Ngl.read_colormap_file("cmocean_curl")
                    cmap[0:11,:] = Span_color_rgb(colorbar[26:121,:], 11)
                    cmap[11:,:]  = Span_color_rgb(colorbar[133:218,:], 11)

                if i == 1:   # RMSD levs & cmap
                    levs = np.array((0.10,0.15,0.20,0.25,0.30,0.35,0.40,0.45,0.5),float)
                    cmap = Ngl.read_colormap_file("sunshine_9lev")

                if i == 2:   # RMS levs & cmap
                    levs = np.array((0.6,0.65,0.70,0.75,0.80,0.85,0.90,0.95,0.98,1.02,\
                        1.05,1.10,1.15,1.20,1.25,1.30,1.35,1.40),float)
                    cmap = Ngl.read_colormap_file("temp_diff_18lev")
                
                if i == 3:   # CORR levs & cmap
                    levs = np.array((0.6,0.7,0.8,0.84,0.86,0.88,0.90,0.91,0.92,0.93,0.94,0.95,0.96,0.98),float)

                    colorbar = Ngl.read_colormap_file("WhiteGreen")
                    cmap = Span_color_rgb((colorbar[10:176,:])[::-1,:],levs.shape[0]+1)

            # define levs & cmap
            if i == 0:
                MElevs = levs
                MEcmap = cmap
            elif i == 1:
                RMSDlevs = levs
                RMSDcmap = cmap
            elif i == 2:
                RMSlevs = levs
                RMScmap = cmap
            else:
                CORRlevs = levs
                CORRcmap = cmap

    #==========================Part 3:Parameters for metrics table
    # whether to show metrics on 
    if hasattr(opt_metrics,"showTextOn"):
        Show_T = opt_metrics.showTextOn
        if not isinstance(Show_T,bool):
            print("metrics table error: opt_metrics.showTextOn should be bool!")
            os._exit(0)    
    else:
        Show_T = True

    if Nf > 1:
        Show_T = False

    # structure of metrics table
    if hasattr(opt_metrics,"CaseLocation"):
        CaseL = opt_metrics.CaseLocation
        if (CaseL!="Left") and (CaseL!="Top"):
            print("metrics table error: opt_metrics.CaseLocation should be 'Left' or 'Top'!")
            os._exit(0)
    else:
        CaseL = "Top"

    CaseTopOn = bool(np.where(CaseL == "Top", True, False))
 
    # get size of different modules in table
    if CaseTopOn: # case on the top
        if hasattr(opt_metrics,"caseHeight"):
            if not isinstance(opt_metrics.caseHeight,(int,float)):
                print("metrics table error: opt_metrics.caseHeight should be int/float!")
                os._exit(0)            
            case_height = np.float(opt_metrics.caseHeight)
        else:
            case_height = 0.02

        if hasattr(opt_metrics,"caseWidth"):        
            if not isinstance(opt_metrics.caseWidth,(int,float)):
                print("metrics table error: opt_metrics.caseWidth should be int/float!")
                os._exit(0)
            case_width = np.float(opt_metrics.caseWidth)
        else:
            case_width = 0.1

        if hasattr(opt_metrics,"varWidth"):
            if not isinstance(opt_metrics.varWidth,(int,float)):
                print("metrics table error: opt_metrics.varWidth should be int/float!")
                os._exit(0)
            var_width = np.float(opt_metrics.varWidth)
        else:
            var_width = 0.07

        title_height = case_width
        var_height   = (zoom-case_width)/(Nstats1*Nvar+Nstats2+nVec)    
        stats_height = var_height

        stats1_width = var_height*Nvar
        stats2_width = var_width+var_height
        title_width  = var_width+stats_height
    else: # case in the left
        if hasattr(opt_metrics,"titleHeight"):       
            if not isinstance(opt_metrics.titleHeight,(int,float)):
                print("metrics table error: opt_metrics.titleHeight should be int/float!")
                os._exit(0)
            title_height = np.float(opt_metrics.titleHeight)   
        else:
            title_height = 0.08

        if hasattr(opt_metrics,"titleWidth"):         
            if not isinstance(opt_metrics.titleWidth,(int,float)):
                print("metrics table error: opt_metrics.titleWidth should be int/float!")
                os._exit(0)
            title_width = np.float(opt_metrics.titleWidth)
        else:
            title_width = 0.1

        if hasattr(opt_metrics,"caseHeight"):      
            if not isinstance(opt_metrics.caseHeight,(int,float)):
                print("metrics table error: opt_metrics.caseHeight should be int/float!")
                os._exit(0)
            case_height = np.float(opt_metrics.caseHeight)            
        else:
            case_height = 0.035

        case_width    = title_width 
        stats1_height = title_height/2.
        var_height    = stats1_height
        var_width     = (zoom-title_width)/(Nvar*Nstats1+Nstats2+nVec)
        stats2_height = var_width
        stats2_width  = title_height
        stats1_width  = Nvar*var_width

    # get font height in table 
    if hasattr(opt_metrics,"caseTHeight"):
        if not isinstance(opt_metrics.caseTHeight,(int,float)):
                print("metrics table error: opt_metrics.caseTHeight should be int/float!")
                os._exit(0)
        case_theight = np.float(opt_metrics.caseTHeight)
    else:
        case_theight = 0.0075

    if hasattr(opt_metrics,"titleTHeight"):
        if not isinstance(opt_metrics.titleTHeight,(int,float)):
                print("metrics table error: opt_metrics.titleTHeight should be int/float!")
                os._exit(0)
        title_theight = np.float(opt_metrics.titleTHeight)
    else:
        title_theight = 0.015
  
    if hasattr(opt_metrics,"varTHeight"):
        if not isinstance(opt_metrics.varTHeight,(int,float)):
                print("metrics table error: opt_metrics.varTHeight should be int/float!")
                os._exit(0)
        var_theight = np.float(opt_metrics.varTHeight)
    else:
        var_theight = 0.006

    if hasattr(opt_metrics,"statsTHeight"):
        if not isinstance(opt_metrics.statsTHeight,(int,float)):
                print("metrics table error: opt_metrics.statsTHeight should be int/float!")
                os._exit(0)
        stats_theight = opt_metrics.statsTHeight
    else:
        stats_theight = 0.0075
        
    if Show_T:
        if hasattr(opt_metrics,"metricsTHeight"):
            if not isinstance(opt_metrics.metricsTHeight,(int,float)):
                print("metrics table error: opt_metrics.metricsTHeight should be int/float!")
                os._exit(0)
            metrics_theight = np.float(opt_metrics.metricsTHeight)
        else:
            metrics_theight = 0.006

    # get table title and filling color 
    if hasattr(opt_metrics,"tableTitle"):
        table_title = opt_metrics.tableTitle
        if not isinstance(table_title,str):
            print("metrics table error: opt_metrics.tableTitle should be str!")
            os._exit(0)
    else:
        table_title = "METRICS"
    
    if hasattr(opt_metrics,"TitleBackgroundcolor"):
        TitleBackgroundcolor = opt_metrics.TitleBackgroundcolor
        if isinstance(TitleBackgroundcolor,str):
            if Ngl.get_named_color_index(wks,TitleBackgroundcolor) < 0:
               print("metrics table error: named color-",TitleBackgroundcolor,"in opt_metrics.TitleBackgroundcolor is invalid!")
               os._exit(0)
        elif isinstance(TitleBackgroundcolor,int):
            if min(TitleBackgroundcolor)<0:
               print("metrics table error: NhlTColorIndex :",TitleBackgroundcolor,"in opt_metrics.TitleBackgroundcolor should be greater than 0!")
               os._exit(0)
        elif isinstance(TitleBackgroundcolor,(list,tuple)):
            if (not all(isinstance(x,(int,float)) for x in TitleBackgroundcolor)) or\
                ((len(TitleBackgroundcolor)!=3) and (len(TitleBackgroundcolor)!=4)) or\
                    (min(TitleBackgroundcolor)<0):
                print("metrics table error: RGB/RGBA list/tuple in opt_metrics.TitleBackgroundcolor should be in",\
                    "3(4)-length and consisting of positive int/float!")
                os._exit(0)
            TitleBackgroundcolor = np.array(TitleBackgroundcolor,float)
        else:
            print("metrics table error: error in opt_metrics.TitleBackgroundcolor, please refer to readme.namelist.Py-v or User Guide!")
            os._exit(0)    
    else:
        TitleBackgroundcolor = "CornflowerBlue"
  
    if hasattr(opt_metrics,"VarBackgroundcolor"):
        VarBackgroundcolor = opt_metrics.VarBackgroundcolor
        if isinstance(VarBackgroundcolor,str):
            if Ngl.get_named_color_index(wks,VarBackgroundcolor) < 0:
               print("metrics table error: named color-",VarBackgroundcolor,"in opt_metrics.VarBackgroundcolor is invalid!")
               os._exit(0)
        elif isinstance(VarBackgroundcolor,int):
            if min(VarBackgroundcolor)<0:
               print("metrics table error: NhlTColorIndex :",VarBackgroundcolor,"in opt_metrics.VarBackgroundcolor should be greater than 0!")
               os._exit(0)
        elif isinstance(VarBackgroundcolor,(list,tuple)):
            if (not all(isinstance(x,(int,float)) for x in VarBackgroundcolor)) or\
                ((len(VarBackgroundcolor)!=3) and (len(VarBackgroundcolor)!=4)) or\
                    (min(VarBackgroundcolor)<0):
                print("metrics table error: RGB/RGBA list/tuple in opt_metrics.VarBackgroundcolor should in",\
                    "3(4)-length and consisting of positive int/float!")
                os._exit(0)
            VarBackgroundcolor = np.array(VarBackgroundcolor,float)
        else:
            print("metrics table error: error in opt_metrics.VarBackgroundcolor, please refer to readme.namelist.Py-v or User Guide!")
            os._exit(0)    
    else:
        VarBackgroundcolor = "Gray70"

    # whether to plot legend of box
    if Nf > 1:
        if hasattr(opt_metrics,"box_legend"):
            if opt_metrics.box_legend == False:
                with_boxleg  = False
            elif isinstance(opt_metrics.box_legend,(list,tuple)) or (not all(isinstance(x,str) for x in opt_metrics.box_legend)):
                box_legend   = opt_metrics.box_legend
                with_boxleg  = True

                if len(box_legend) != Nf:
                    print("metrics table error: opt_metrics.box_legend does not match the num of datafiles input!")
                    os._exit(0)
            else:
                print("metrics table error: opt_metrics.box_legend should be False or the list of legend str!")
                os._exit(0)
        else:
            with_boxleg = False

        if with_boxleg:
            if hasattr(opt_metrics,"box_lgheight"):
                if not isinstance(opt_metrics.box_lgheight,(int,float)):
                    print("metrics table error: opt_metrics.box_lgheight should be int/float!")
                    os._exit(0)
                boxleg_height = np.float(opt_metrics.box_lgheight)
            if hasattr(opt_metrics,"box_lgFontHeightF"):
                if not isinstance(opt_metrics.box_lgFontHeightF,(int,float)):
                    print("metrics table error: opt_metrics.box_lgFontHeightF should be int/float!")
                    os._exit(0)
                boxlgFontHeightF = np.float(opt_metrics.box_lgFontHeightF)
       
        spl_Res                  = Ngl.Resources()
        spl_Res.gsLineThicknessF = 1.0
        spl_Res.gsLineColor      = VarBackgroundcolor
    else:
        with_boxleg = False

    # whether to plot box for MISS if plotting MISS
    if hasattr(opt_metrics,"highlightsummary"):
        plot_box = opt_metrics.highlightsummary
        if not isinstance(plot_box,bool):
            print("metrics_table: opt_metrics.highlightsummary should be bool!")
            os._exit(0)
    else:
        plot_box = True

    # parameters for colorbar
    if not Show_T:
        if hasattr(opt_metrics,"decialBarN"):
            if not isinstance(opt_metrics.decialBarN,str):
                print("metrics table error: opt_metrics.decialBarN should be str, and has format as '3.2'!")
                os._exit(0)      

            decial_Bar = opt_metrics.decialBarN
            Comp =  decial_Bar.split(".")
            if len(Comp) != 2:
                print("metrics table error: opt_metrics.decialBarN should have format as '3.2'!")
                os._exit(0)
            if (not Comp[0].isdigit()) and (not Comp[1].isdigit()):
                print("metrics table error: opt_metrics.decialBarN should have format as '3.2'!")
                os._exit(0)

            decial_Bar = decial_Bar+"f"
        else: 
            decial_Bar = "3.2f"

        lt_Res = Ngl.Resources()
 
        if hasattr(opt_metrics,"colorbarFHeight"):
            if not isinstance(opt_metrics.colorbarFHeight,(int,float)):
                print("metrics table error: opt_metrics.colorbarFHeight should be int/float!")
                os._exit(0)
            lt_Res.txFontHeightF   = np.float(opt_metrics.colorbarFHeight)
        else:
            lt_Res.txFontHeightF   = var_theight
    else:
        mv_tRes               = Ngl.Resources()
        mv_tRes.txFontHeightF = metrics_theight

        if hasattr(opt_metrics,"decialMetricsN"):
            if not isinstance(opt_metrics.decialMetricsN,str):
                print("metrics table error: opt_metrics.decialMetricsN should be str!")
                os._exit(0)  
            decial_Show = opt_metrics.decialMetricsN
            Comp =  decial_Show.split(".")
            if len(Comp) != 2:
                print("metrics table error: opt_metrics.decialMetricsN should have format as '3.2'!")
                os._exit(0)
            if (not Comp[0].isdigit()) and (not Comp[1].isdigit()):
                print("metrics table error: opt_metrics.decialMetricsN should have format as '3.2'!")
                os._exit(0)

            decial_Show = decial_Show+"f"
        else:
            decial_Show = "%3.2f"

    #============================================Part 4:Plot table
    # plot table title
    xbox = title_width*xbox0
    ybox = 1.-title_height*ybox0

    tt_pRes               = Ngl.Resources()
    tt_pRes.gsFillColor   = TitleBackgroundcolor
    Ngl.polygon_ndc(wks,xbox,ybox,tt_pRes)

    tt_tRes               = Ngl.Resources()
    tt_tRes.txFontHeightF = title_theight

    ixtbox = xbox[0]+0.5*(xbox[1]-xbox[0])
    iytbox = ybox[0]+0.5*(ybox[2]-ybox[0])

    Ngl.text_ndc(wks,table_title,ixtbox,iytbox,tt_tRes)  
    Ngl.polyline_ndc(wks,xbox,ybox,Ngl.Resources())
  
    # plot caselabels 
    mn_pRes               = Ngl.Resources()
    mn_pRes.gsFillColor   = VarBackgroundcolor
    mn_tRes               = Ngl.Resources()
    mn_tRes.txFontHeightF = case_theight

    if CaseTopOn:
        for i in range(Ncase):
            xbox   = max(xbox)+case_height*xbox0
            Ngl.polygon_ndc(wks,xbox,ybox,mn_pRes)

            mn_tRes.txAngleF = 90

            ixtbox = xbox[0]+0.5*(xbox[1]-xbox[0])
            iytbox = ybox[0]+0.5*(ybox[2]-ybox[0])
    
            Ngl.text_ndc(wks,caseNames[i],ixtbox,iytbox,mn_tRes)
            Ngl.polyline_ndc(wks,xbox,ybox,Ngl.Resources())    
    else:
        for i in range(Ncase):
            ybox   = min(ybox)-case_height*ybox0
            Ngl.polygon_ndc(wks,xbox,ybox,mn_pRes)

            mn_tRes.txAngleF      = 0

            ixtbox = xbox[0]+0.5*(xbox[1]-xbox[0])
            iytbox = ybox[0]+0.5*(ybox[2]-ybox[0])
            
            Ngl.text_ndc(wks,caseNames[i],ixtbox,iytbox,mn_tRes)
            Ngl.polyline_ndc(wks,xbox,ybox,Ngl.Resources()) 

    # plot metrics
    mv_pRes = Ngl.Resources()
              
    colorbar_title   = []

    L_str   = np.float(np.where(CaseTopOn, np.min(ybox), np.max(xbox)))
    split_L = []

    iStats1 = [["ME"],["cRMSD","RMSD"],["SD","rms"],["CORR","uCORR"]]
    iStats2 = [["VME","MEVM"],["cRMSVD","RMSVD","SD_std","rms_std"],["cRMSL","RMSL"],["cVSC","VSC","cMISS","uMISS"]]

    for it in range(4):
        if With_colormap[it]:
            if it == 0:
                levs = MElevs 
                cmap = MEcmap
            elif it == 1:
                levs = RMSDlevs
                cmap = RMSDcmap
            elif it == 2:
                levs = RMSlevs
                cmap = RMScmap
            else:
                levs = CORRlevs
                cmap = CORRcmap

        it_title = ""

        for istat in range(len(iStats1[it])):
            if iStats1[it][istat] in stats:
                if Nf == 1:
                    metrics = np.array(f.variables[iStats1[it][istat]])
                    if (metrics.ndim!=2) or (metrics.shape[0]<Ncase) or (metrics.shape[1]<Nvar):
                        print("metrics table error:",iStats1[it][istat]," in file ",fnames," should be 2D array, and ",\
                            "match caseNames and varNames!")
                        os._exit(0)
                else:
                    metrics = np.zeros((Nvar,Ncase,Nvar),float)
                    for j in range(Nf):
                        imetrics = np.array(f[j].variables[iStats1[it][istat]])
                        if (imetrics.ndim!=2) or (imetrics.shape[0]<Ncase) or (imetrics.shape[1]<Nvar):
                            print("metrics table error:",iStats1[it][istat]," in file ",fnames," should 2D array, and ",\
                                "match caseNames and varNames!")
                            os._exit(0)
                        metrics[j,:,:] = imetrics[0:Ncase,0:Nvar]

                it_title += str(np.where(len(it_title)==0, iStats1[it][istat], "/"+iStats1[it][istat]))

                if CaseTopOn:
                    xbox = stats_height*xbox0
                    ybox = np.min(ybox) - (stats1_width+np.where(it==0, nVec*var_height, 0))*ybox0
                else:
                   xbox = np.max(xbox) + (stats1_width+np.where(it==0, nVec*var_width, 0))*xbox0
                   ybox = 1. - stats1_height*ybox0

                mn_tRes.txAngleF = np.float(np.where(CaseTopOn, 90, 0))

                Ngl.polygon_ndc(wks,xbox,ybox,mn_pRes)
                Ngl.polyline_ndc(wks,xbox,ybox,Ngl.Resources())

                ixtbox = xbox[0]+0.5*(xbox[1]-xbox[0])
                iytbox = ybox[0]+0.5*(ybox[2]-ybox[0])

                mn_tRes.txFontHeightF = stats_theight

                Ngl.text_ndc(wks,iStats1[it][istat],ixtbox,iytbox,mn_tRes)

                iv_Vec = False

                for iv in range(Nvar):
                    if (it==0) and (nVec!=0):
                        if (Nf==1) and (not np.isnan(MEVD[0,iv])):
                            iv_Vec = True
                        elif (Nf>1) and (not np.isnan(MEVD[0,0,iv])):
                            iv_Vec = True
                        else:
                            iv_Vec = False

                    if CaseTopOn:
                        xbox = stats_height + var_width*xbox0
                        ybox = np.where(iv==0, np.max(ybox), np.min(ybox)) - np.where(iv_Vec, 2, 1)*var_height*ybox0           
                    else:
                        xbox = np.where(iv==0, np.min(xbox), np.max(xbox)) + np.where(iv_Vec, 2., 1.)*var_width*xbox0          
                        ybox = 1. - stats1_height - var_height*ybox0

                    Ngl.polygon_ndc(wks,xbox,ybox,mn_pRes)
                    Ngl.polyline_ndc(wks,xbox,ybox,Ngl.Resources())

                    ixtbox = xbox[0]+0.5*(xbox[1]-xbox[0])
                    iytbox = ybox[0]+0.5*(ybox[2]-ybox[0])

                    mn_tRes.txFontHeightF = var_theight
                    mn_tRes.txAngleF      = 0

                    Ngl.text_ndc(wks,varNames[iv],ixtbox,iytbox,mn_tRes)

                    if iv_Vec:
                        if CaseTopOn:
                            ybox     = np.max(ybox) - var_height*ybox0
                            ybox_add = np.min(ybox) - var_height*ybox0
                        else:
                            xbox     = np.min(xbox) + var_width*xbox0
                            xbox_add = np.max(xbox) + var_width*xbox0

                    for ic in range(Ncase):
                        if CaseTopOn:
                            xbox = np.max(xbox) + case_height*xbox0
                        else:
                            ybox = np.min(ybox) - case_height*ybox0

                        if iv_Vec:
                            if CaseTopOn:
                                xbox_add = xbox
                            else:
                                ybox_add = ybox

                        if Nf == 1:
                            mv_pRes.gsFillColor = GetFillColor(levs,cmap,metrics[ic,iv])
                            
                            Ngl.polygon_ndc(wks,xbox,ybox,mv_pRes)
                            Ngl.polyline_ndc(wks,xbox,ybox,Ngl.Resources())

                            if Show_T:
                                ixtbox = xbox[0]+0.5*(xbox[1]-xbox[0])
                                iytbox = ybox[0]+0.5*(ybox[2]-ybox[0])

                                Ngl.text_ndc(wks,str(decial_Show%metrics[ic,iv]),ixtbox,iytbox,mv_tRes)
                        else:
                            ixtbox = xbox[0]+0.5*(xbox[1]-xbox[0])
                            iytbox = ybox[0]+0.5*(ybox[2]-ybox[0])

                            for ig in range(Nf):
                                tr_xbox = xbox[ind_box[ig]]
                                tr_ybox = ybox[ind_box[ig]]

                                if Nf > 2:
                                   tr_xbox[1] = ixtbox
                                   tr_ybox[1] = iytbox

                                mv_pRes.gsFillColor = GetFillColor(levs,cmap,metrics[ig,ic,iv])
                                Ngl.polygon_ndc(wks,tr_xbox,tr_ybox,mv_pRes)

                            if Nf == 3:
                                tr_xbox = xbox[ind_box[Nf]]
                                tr_ybox = ybox[ind_box[Nf]]
                                tr_xbox[1] = ixtbox
                                tr_ybox[1] = iytbox

                                mv_pRes.gsFillColor = VarBackgroundcolor
                                Ngl.polygon_ndc(wks,tr_xbox,tr_ybox,mv_pRes)

                            Ngl.polyline_ndc(wks,xbox,ybox,spl_Res)

                        if iv_Vec:
                            if Nf == 1:
                                mv_pRes.gsFillColor = GetFillColor(levs,cmap,MEVD[ic,iv]/180.)

                                Ngl.polygon_ndc(wks,xbox_add,ybox_add,mv_pRes)
                                Ngl.polyline_ndc(wks,xbox_add,ybox_add,Ngl.Resources())

                                if Show_T:
                                    ixtbox = xbox_add[0]+0.5*(xbox_add[1]-xbox_add[0])
                                    iytbox = ybox_add[0]+0.5*(ybox_add[2]-ybox_add[0])
                                    Ngl.text_ndc(wks,str(decial_Show%(MEVD[ic,iv]/180.)),ixtbox,iytbox,mv_tRes)
                            else:
                                ixtbox = xbox_add[0]+0.5*(xbox_add[1]-xbox_add[0])
                                iytbox = ybox_add[0]+0.5*(ybox_add[2]-ybox_add[0])
    
                                for ig in range(Nf):
                                    tr_xbox = xbox_add[ind_box[ig]]
                                    tr_ybox = ybox_add[ind_box[ig]]

                                    if Nf > 2:
                                        tr_xbox[1] = ixtbox
                                        tr_ybox[1] = iytbox

                                    mv_pRes.gsFillColor = GetFillColor(levs,cmap,MEVD[ig,ic,iv]/180.)
                                    Ngl.polygon_ndc(wks,tr_xbox,tr_ybox,mv_pRes)

                                if Nf == 3:
                                    tr_xbox = xbox[ind_box[Nf]]
                                    tr_ybox = ybox[ind_box[Nf]]
                                    tr_xbox[1] = ixtbox
                                    tr_ybox[1] = iytbox
                                   
                                    mv_pRes.gsFillColor = VarBackgroundcolor
                                    Ngl.polygon_ndc(wks,tr_xbox,tr_ybox,mv_pRes) 

                                Ngl.polyline_ndc(wks,xbox_add,ybox_add,spl_Res)
                
                    if iv_Vec:
                        if CaseTopOn:
                            ybox = ybox_add
                        else:
                            xbox = xbox_add

        for istat in range(len(iStats2[it])):
            if iStats2[it][istat] in stats:
                if ((iStats2[it][istat] == "cMISS") or (iStats2[it][istat] == "uMISS"))\
                    and (not "MISS_loc" in locals()):
                    MISS_loc = np.where(CaseTopOn, np.min(ybox), np.max(xbox))

                if Nf == 1:
                    metrics = np.array(f.variables[iStats2[it][istat]])
                    if (metrics.ndim!=1) or (metrics.shape[0]<Ncase):
                        print("metrics table error:",iStats2[it][istat]," in file ",fnames," should 2D array, and ",\
                            "match caseNames !")
                        os._exit(0)
                else:
                    metrics = np.zeros((Nvar,Ncase),float)
                    for j in range(Nf):
                        imetrics = np.array(f[j].variables[iStats2[it][istat]])
                        if (imetrics.ndim!=1) or (imetrics.shape[0]<Ncase):
                            print("metrics table error:",iStats2[it][istat]," in file ",fnames," should 2D array, and ",\
                                "match caseNames and varNames!")
                            os._exit(0)
                        metrics[j,:] = imetrics[0:Ncase]

                it_title += str(np.where(len(it_title)==0, iStats2[it][istat], "/"+iStats2[it][istat]))

                if CaseTopOn:
                    xbox = stats2_width*xbox0
                    ybox = np.min(ybox) - stats_height*ybox0
                else:
                    xbox = np.max(xbox) + stats2_height*xbox0
                    ybox = 1. - stats2_width*ybox0

                mn_tRes.txAngleF  = np.float(np.where(CaseTopOn, 0, 90))

                Ngl.polygon_ndc(wks,xbox,ybox,mn_pRes)
                Ngl.polyline_ndc(wks,xbox,ybox,Ngl.Resources())

                ixtbox = xbox[0]+0.5*(xbox[1]-xbox[0])
                iytbox = ybox[0]+0.5*(ybox[2]-ybox[0])

                mn_tRes.txFontHeightF = stats_theight
            
                Ngl.text_ndc(wks,iStats2[it][istat],ixtbox,iytbox,mn_tRes)

                for ic in range(Ncase):
                    if CaseTopOn:
                        xbox = np.max(xbox) + case_height*xbox0
                    else:
                        ybox = np.min(ybox) - case_height*ybox0

                    if Nf == 1:
                        mv_pRes.gsFillColor = GetFillColor(levs,cmap,metrics[ic])

                        Ngl.polygon_ndc(wks,xbox,ybox,mv_pRes)
                        Ngl.polyline_ndc(wks,xbox,ybox,Ngl.Resources())

                        if Show_T:
                            ixtbox = xbox[0]+0.5*(xbox[1]-xbox[0])
                            iytbox = ybox[0]+0.5*(ybox[2]-ybox[0])
                            Ngl.text_ndc(wks,str(decial_Show%metrics[ic]),ixtbox,iytbox,mv_tRes)
                    else:
                        ixtbox = xbox[0]+0.5*(xbox[1]-xbox[0])
                        iytbox = ybox[0]+0.5*(ybox[2]-ybox[0])

                        for ig in range(Nf):
                            tr_xbox = xbox[ind_box[ig]]
                            tr_ybox = ybox[ind_box[ig]]

                            if Nf > 2:
                                tr_xbox[1] = ixtbox
                                tr_ybox[1] = iytbox

                            mv_pRes.gsFillColor = GetFillColor(levs,cmap,metrics[ig,ic])
                            Ngl.polygon_ndc(wks,tr_xbox,tr_ybox,mv_pRes)

                        if Nf == 3:
                            tr_xbox = xbox[ind_box[Nf]]
                            tr_ybox = ybox[ind_box[Nf]]
                            tr_xbox[1] = ixtbox
                            tr_ybox[1] = iytbox

                            mv_pRes.gsFillColor = VarBackgroundcolor
                            Ngl.polygon_ndc(wks,tr_xbox,tr_ybox,mv_pRes)
                        Ngl.polyline_ndc(wks,xbox,ybox,spl_Res)

        if L_str != np.where(CaseTopOn, np.min(ybox), np.max(xbox)):
            L_str = str(np.where(CaseTopOn, np.min(ybox), np.max(xbox)))
            split_L.append(L_str)
        if len(it_title) > 0:
            colorbar_title.append(it_title)
        else:
            colorbar_title.append(False)

    # plot split lines for colormap
    if len(split_L) > 1:
        line_Res = Ngl.Resources()    
        line_Res.gsLineThicknessF = 3.0

        if CaseTopOn:
            xline = np.array((0, title_width+Ncase*case_height),float)
        else:
            yline = np.array((1., 1.-title_height-Ncase*case_height),float)

        for loc in split_L:
            if CaseTopOn:
                yline = np.array((loc,loc),float)
            else:
                xline = np.array((loc,loc),float)
            Ngl.polyline_ndc(wks,xline,yline,line_Res)

    # plot frame for colorarea
    if CaseTopOn:
        xbox_frame = title_width + (Ncase*case_height)*xbox0
        ybox_frame = 1. - title_height - (zoom-title_height)*ybox0
    else:
        xbox_frame = title_width + (zoom-title_width)*xbox0
        ybox_frame = 1. - title_height - Ncase*case_height*ybox0
    Ngl.polyline_ndc(wks, xbox_frame, ybox_frame, Ngl.Resources())

    # plot frame for cMISS/uMISS
    if plot_box and ("MISS_loc" in locals()):
        if CaseTopOn:
            xbox_frame = (Ncase*case_height+title_width)*xbox0                                
            ybox_frame = MISS_loc + (np.min(ybox)-MISS_loc)*ybox0
        else:
            xbox_frame = MISS_loc + (np.max(xbox)-MISS_loc)*xbox0
            ybox_frame = 1. - (title_height+Ncase*case_height)*ybox0

        line_Res  = Ngl.Resources()
        line_Res.gsLineThicknessF = 3.0
        line_Res.gsLineColor      = "red"

        Ngl.polyline_ndc(wks, xbox_frame, ybox_frame, line_Res)

    # plot color labels and box legend
    if not Show_T:               
        lb_height   = case_height*1.3

        if with_boxleg:
            if not "boxleg_height" in locals():
                boxleg_height = lb_height * np.sum(With_colormap)
                 
            if CaseTopOn: 
                boxleg_width  = boxleg_height * (case_height/var_height) 
            else:
                boxleg_width  = boxleg_height * (var_width/case_height)  
        else:
            boxleg_height = 0.
            boxleg_width  = 0.

        table_width  = np.where(CaseTopOn, title_width+Ncase*case_height, zoom) 
        border_width = case_height*0.15

        lb_x = np.where(with_boxleg,border_width,0) + boxleg_width + \
            lt_Res.txFontHeightF * (max(len(x) for x in colorbar_title)-2.) 
        lb_width = table_width - lb_x - border_width

        lb_Res  = Ngl.Resources()
        lb_Res.lbPerimOn            = False                            
        lb_Res.lbOrientation        = "Horizontal"                      
        lb_Res.lbLabelFontHeightF   = lt_Res.txFontHeightF*0.9         
        lb_Res.lbLabelAlignment     = "InteriorEdges"                             
        lb_Res.lbMonoFillPattern    = True                            
        lb_Res.vpWidthF             = lb_width                         
        lb_Res.vpHeightF            = lb_height
        lb_Res.lbLabelFont          = 22

        lt_Res.txJust               = "CenterRight"
      
        lb_y   = np.min(ybox) - np.where(nVec==0, 0.4*case_height, 0.8*case_height)

        lt_x = border_width + boxleg_width 
        lt_y = lb_y - 0.5*lb_height 

        if with_boxleg:
            xbox_leg = border_width + boxleg_width*xbox0
            ybox_leg = lb_y -  np.abs((lb_height*np.sum(With_colormap)-boxleg_height)/2.) \
                - boxleg_height*ybox0

            lg_box = Ngl.Resources()

            if "boxlgFontHeightF" in locals():
                lg_box.txFontHeightF = boxlgFontHeightF
            else:
                lg_box.txFontHeightF = var_theight*1.2

            if Nf == 2:
                xtbox2 = np.array((np.min(xbox_leg)+0.04*boxleg_width, np.max(xbox_leg)-0.04*boxleg_width),float)          
                ytbox2 = np.array((np.min(ybox_leg)+0.2*boxleg_height, np.max(ybox_leg)-0.2*boxleg_height),float)

                txJust2 = ["bottomleft", "topright"]

                for ig in range(2):
                    ixbox = xbox_leg[ind_box[ig]]
                    iybox = ybox_leg[ind_box[ig]]
                    Ngl.polyline_ndc(wks,ixbox,iybox,Ngl.Resources())

                    lg_box.txJust = txJust2[ig]
                    Ngl.text_ndc(wks, box_legend[ig], xtbox2[ig], ytbox2[ig], lg_box)             
            else:
                Ngl.polyline_ndc(wks,xbox_leg,ybox_leg,Ngl.Resources())

                xtbox4 = np.array((np.min(xbox_leg)+0.025*boxleg_width, np.min(xbox_leg)+0.5*boxleg_width,\
                    np.max(xbox_leg)-0.025*boxleg_width, np.min(xbox_leg)+0.5*boxleg_width),float)
                ytbox4 = np.array((np.min(ybox_leg)+0.5*boxleg_height, np.min(ybox_leg)+0.1*boxleg_height, \
                    np.min(ybox_leg)+0.5*boxleg_height, np.max(ybox_leg)-0.1*boxleg_height),float)
                txJust4 = ["centerleft", "bottomcenter", "centerright", "topcenter"]
         
                for ig in range(Nf):    
                    ixbox = xbox_leg[ind_box[ig]]
                    iybox = ybox_leg[ind_box[ig]]

                    ixbox[1] = np.min(xbox_leg)+0.5*boxleg_width
                    iybox[1] = np.min(ybox_leg)+0.5*boxleg_height

                    Ngl.polyline_ndc(wks,ixbox,iybox,Ngl.Resources())

                    lg_box.txJust = txJust4[ig]
                    Ngl.text_ndc(wks, box_legend[ig], xtbox4[ig], ytbox4[ig], lg_box)

        notisfirst = False
        lt_x = lb_x

        for ib in range(4):
            if With_colormap[ib]:
                if ib == 0:
                    levs = MElevs 
                    cmap = MEcmap
                elif ib == 1:
                    levs = RMSDlevs
                    cmap = RMSDcmap
                elif ib == 2:
                    levs = RMSlevs
                    cmap = RMScmap
                else:
                    levs = CORRlevs
                    cmap = CORRcmap           

                if notisfirst:
                    lb_y = lb_y - lb_height
                    lt_y = lb_y - 0.5*lb_height
                else:
                    notisfirst = True

                lb_Res.lbFillColors = cmap   

                if (ib==0) and (nVec>0):
                    levs2_lb = list(map("{:.0f}".format, (levs*180)))
                    
                    if np.sum(abs(levs)>1.)>0.:   
                        levs2_lb[np.where(abs(levs)>1.)] = " "

                    addlb_Res = Ngl.Resources()
                    addlb_Res.txFontColor   = "firebrick"
                    addlb_Res.txFontHeightF = lb_Res.lbLabelFontHeightF*0.95
                    addlb_Res.txFont        = 22

                    sub_width = lb_Res.vpWidthF / (levs.shape[0]+1.)

                    for i in range(1,levs.shape[0]+1):
                        sub_x = lb_x + i*sub_width
                        Ngl.text_ndc(wks,levs2_lb[i-1],sub_x,lb_y,addlb_Res)

                    addlb_Res.txJust = "CenterRight"
                    Ngl.text_ndc(wks,"(MEVD)",lb_x + lb_Res.vpWidthF,lb_y,addlb_Res)
                    
                Ngl.text_ndc(wks,colorbar_title[ib]+":  ",lt_x,lt_y,lt_Res)

                Ngl.labelbar_ndc (wks,levs.shape[0]+1,list(map(("{:"+decial_Bar+"}").format,levs)),lb_x,lb_y,lb_Res)
        
        xbox = table_width*xbox0

        if with_boxleg and ((np.min(ybox) - lb_y + lb_height)<\
            (np.abs(lb_height*np.sum(With_colormap)-boxleg_height)+boxleg_height)):
            ybox = np.min(ybox) - 0.2*case_height - \
                (np.abs(lb_height*np.sum(With_colormap)-boxleg_height)+boxleg_height)*ybox0
        else:
            ybox = np.min(ybox) - 0.2*case_height - (np.min(ybox) - lb_y + lb_height)*ybox0
        
        line_Res = Ngl.Resources()
        line_Res.gsLineColor = VarBackgroundcolor

        Ngl.polyline_ndc(wks,xbox,ybox,line_Res)

    # plot grid line
    if hasattr(opt_metrics,"draw_grid"):
        if not isinstance(opt_metrics.draw_grid,bool):
            print("metrics table error: 'draw_grid' of opt_metrics should be bool!")
            os._exit(0)
        if opt_metrics.draw_grid:
            Ngl.draw_ndc_grid(wks)

    # draw table  
    Ngl.draw(wks)
    Ngl.frame(wks) 

Ngl.end()              
#----------------------------------------------------------------


         