#--------------------------------------------------------------
# FUNCTIONS FOR MULTIVARIABLE EVALUATION TOOL
#--------------------------------------------------------------
# To report bugs or make suggestions, please send emails at:
#  zhangmengzhuo1996@163.com , xuzhf@tea.ac.cn
#--------------------------------------------------------------
import numpy as np 
import xarray as xr 
import os
import math
import Nio
#--------------------------------------------------------------
###############################################################
# Get the number of scalar/vector variables for evaluation (M),
#  and their dimensions. Check var weighting.
###############################################################
def Get_Mvars(Varname,Wgt_var):
    # get var names
    if not isinstance(Varname,str):
        print("Calculate_MVIE error: Varname should be string!")
        os._exit(0)

    var_names = Varname.split(",")
    Nvar = len(var_names)

    M = 0
    isVec = False
    for i in var_names:
        if isVec:
            if ")" in i:
                isVec = False
        else:
            if "(" in i:
                isVec = True
            M += 1

    Mvar_D = np.zeros(M, "int")
    j      = 0
    isVec  = False

    for i in range(Nvar):
        Mvar_D[j] += 1
        if isVec:
            if ")" in var_names[i]:
                isVec = False
                var_names[i] = var_names[i].strip(")")
                j += 1
        else:
            if "(" in var_names[i]:
                isVec = True
                var_names[i] = var_names[i].strip("(")
            else:
                j += 1

    VarWgt_str = []
    # get var wgt
    if isinstance(Wgt_var,list):
        if (len(Wgt_var)!=1) and (M==1):
            print("Calculate_MVIE warning: 'Wgt_var' is invalid because there is ",\
                    "only one scalar or vector variable for evaluation!")
            M_wgt = 1.
        else:
            if len(Wgt_var) != Nvar:
                print("Calculate_MVIE error: the num of 'Wgt_var' should match ",\
                        "with the num of variables in Varname!")
                os._exit(0)

            if any(x<=0. for x in Wgt_var):
                print("Calculate_MVIE error: each element in 'Wgt_var' should be larger than 0! ")
                os._exit(0)

            Wgt_var = list(map(float,Wgt_var)) 
            Wgt_var = Wgt_var/np.sum(Wgt_var)

            M_wgt = np.zeros(M,float)
            
            i = 0
            for j in range(M):
                M_wgt[j] = Wgt_var[i]
                iVarstr = var_names[i]

                if Mvar_D[j] > 1:
                    for k in range(Mvar_D[j]):
                        if Wgt_var[i] != Wgt_var[i+k]:
                            print("Calculate_MVIE error: weights for components of a vector ",\
                                  "variable in 'Wgt_var' should be the same!")
                            os._exit(0)
                        if k>0:
                            iVarstr += ","
                            iVarstr += var_names[i+k]
                    i += Mvar_D[j]
                else:
                    i += 1  
                VarWgt_str.append(iVarstr)                          
    else:
        M_wgt = 1.

    return var_names,Nvar,M,Mvar_D,M_wgt,VarWgt_str
#--------------------------------------------------------------
###############################################################
# Read variables with no coordinates to fixed format
###############################################################
def ReadVars(filename,varname):
    if not os.path.isfile(filename):
        print("Calculate_MVIE error: cannot find file:",filename,"!")
        os._exit(0)

    try: 
        f = xr.open_dataset(filename)
    except:
        print("Calculate_MVIE error: cannot open ",filename,"!")
        os._exit(0)

    for i in range(len(varname)):
        try:
            ivar = np.array(f[varname[i]])
        except:
            print("Calculate_MVIE error: cannot read ",varname[i]," in file:",filename," !")
            os._exit(0)

        if len(varname)==1:
            data = np.reshape(ivar,(1,ivar.size))
        else:
            if i == 0:          
                data = np.zeros((len(varname),ivar.size), float)
                shp0 = ivar.shape
            else:
                if ivar.shape != shp0:
                    print("Claculate_MVIE error: shape of ",varname[i]," does not match that of ",\
                        varname[0]," in file:",filename," !")
                    os._exit(0)                
            data[i,:] = np.reshape(ivar,ivar.size)
            
    return data
#--------------------------------------------------------------
###############################################################
# Generate area weights for equally spaced grids
###############################################################
def latWgt(lat):   
    if max(lat)>(90.+0.001):
        print("Claculate_MVIE error: max value in latitude-Coordinate is greater than 90 and cannot create area-weighting!")
        os._exit(0)
    if min(lat)<(-90.-0.001):
        print("Claculate_MVIE error: min value in latitude-Coordinate is less than -90 and cannot create area-weighting!")
        os._exit(0)

    dlat = np.abs(lat[1]-lat[0]) 
    
    # check whether is equally spaced latitudes
    lat_diff = np.abs(np.array(lat[1:len(lat)])-np.array(lat[0:(len(lat)-1)]))

    if not (min(lat_diff)>=(dlat-0.001) and max(lat_diff)<=(dlat+0.001)):
        print("Claculate_MVIE error: selected latitudes for area weighting are not equally spaced! ")
        os._exit(0)

    rad = math.pi/180.
    dlat *= rad*0.5

    area_wgt = np.zeros(len(lat),float)

    for i in range(len(lat)):
        area_wgt[i] = np.abs(np.sin(lat[i]*rad+dlat) - np.sin(lat[i]*rad-dlat))

    if np.abs(lat[0]) > 89.9999:
        area_wgt[0] = np.abs(1.-np.sin(math.pi/2.-dlat))

    if np.abs(lat[len(lat)-1]) > 89.9999:
        area_wgt[len(lat)-1] = np.abs(1.-np.sin(math.pi/2.-dlat))

    return area_wgt
#--------------------------------------------------------------
###############################################################
# Conform 1D array to 2D/3D array
###############################################################
def Conform_dims(data,dims,axis):
    if axis == len(dims)-1:
        if len(dims) == 2:
            return np.tile(data,(dims[0],1))
        else:
            return np.tile(data,(dims[0],dims[1],1))
    elif axis == 0:
        return np.reshape(np.repeat(data,np.prod(dims[1:]),axis=0),dims)
    else:
        return np.reshape(np.tile(np.repeat(data,dims[2],axis=0),dims[0]),dims)
#--------------------------------------------------------------
###############################################################
# Read variables with no coordinates to fixed format
###############################################################
def ReadVars_Coord(filename,varname,Ctime,Cgeo1,Cgeo2,Clev,Sel_str,Reorder_str,Isarea_wgt):
    if not os.path.isfile(filename):
        print("Calculate_MVIE error: cannot find file:",filename,"!")
        os._exit(0)

    try: 
        f = xr.open_dataset(filename)
    except:
        print("Calculate_MVIE error: cannot open ",filename,"!")
        os._exit(0)

    Count_Coords = 0

    if hasattr(Ctime,"Cname"):  #time Coord
        Cname = Ctime.Cname
        Crange = Ctime.Crange

        try:
            time = f[Cname]
        except:
            print("Claculate_MVIE error: error in reading ",Cname," in file:",filename," !")
            os._exit(0)

        if isinstance(Crange[0],str):
            try:
                time_index = time.indexes
                del time_index
            except:
                print("Claculate_MVIE error: when 'Range_time' is list/tuple of Time-index str, CoordinateVar-"+Cname,\
                    "in file ("+filename+") should defined with index!")
                os._exit(0)

            try: 
                time_slice = time.loc[Crange[0]:Crange[1]]               
            except:
                print("Claculate_MVIE error: cannot find 'Range_time':"+Crange[0]+":"+Crange[1],"in CoordinateVar-"+Cname,"in file ("+filename+")!")               
                os._exit(0)

            if (len(time_slice)==0) and (not time_slice):
                print("Claculate_MVIE error: cannot find time in 'Range_time' in CoordinateVar-"+Cname,"in file:"+filename+"!")
                os._exit(0)
        else:
            if time.dtype == "datetime64[ns]":
                try:
                    time_slice = time.loc[Crange[0]:Crange[1]]
                except:
                    print("Claculate_MVIE error: when CoordinateVar-"+Cname,"defined with DateTimeIndex(datetime64[ns]), "+\
                        "must set 'Range_time' with DateTimeIndex rather than numeric values!")
                    os._exit(0)
            else:
                try:
                    time_slice = time[np.logical_and(time>=min(Crange),time<=max(Crange))]
                except:
                    print("Claculate_MVIE error: error in 'Range_time' for ",Cname," var of file:",filename," !")
                    os._exit(0)
                
        Count_Coords += 1

    if hasattr(Cgeo1,"Cname"):  #geo1 Coord
        Cname = Cgeo1.Cname
        Crange = Cgeo1.Crange

        try:
            geo1 = f[Cname]
        except:
            print("Claculate_MVIE error: error in reading ",Cname," in file:",filename," !")
            os._exit(0)

        try:
            geo1_slice = geo1[np.logical_and(geo1>=min(Crange),geo1<=max(Crange))]
        except:
            print("Claculate_MVIE error: error in 'Range_geo' for ",Cname," var of file:",filename," !")
            os._exit(0)

        if Isarea_wgt == Cname:
            Area_wgt = latWgt(geo1_slice)
            lat_Coords = Count_Coords

        Count_Coords += 1

    if hasattr(Cgeo2,"Cname"):  #geo2 Coord
        Cname = Cgeo2.Cname
        Crange = Cgeo2.Crange

        try:
            geo2 = f[Cname]
        except:
            print("Claculate_MVIE error: error in reading ",Cname," in file:",filename," !")
            os._exit(0)

        try:
            geo2_slice = geo2[np.logical_and(geo2>=min(Crange),geo2<=max(Crange))]
        except:
            print("Claculate_MVIE error: error in 'Range_geo' for ",Cname," var of file:",filename," !")
            os._exit(0)

        if Isarea_wgt == Cname:
            Area_wgt = latWgt(geo2_slice)
            lat_Coords = Count_Coords

        Count_Coords += 1 

    if hasattr(Clev,"Cname"): # lev Coord
        Levname = Clev.Cname
        lev_Spc = Clev.Cspe

        try:
            Lev = f[Levname]
        except:
            print("Claculate_MVIE error: error in reading ",Levname," in file:",filename," !")
            os._exit(0)
        del(Lev)

    # read
    for i in range(len(varname)):
        try:
            ivar = f[varname[i]]
        except:
            print("Calculate_MVIE error: cannot read ",varname[i]," in file:",filename," !")
            os._exit(0)

        if not hasattr(ivar,"dims"):
            print("Calculate_MVIE error: ",varname[i]," has no dim attribution in file:",filename," !")
            os._exit(0)

        if hasattr(Clev,"Cname") and (Levname in ivar.dims):
            exec_str = "ivar_sel = ivar.sel"+Sel_str+","+Levname+"=lev_Spc[i])"
            if (ivar.ndim!=Count_Coords) and (ivar.ndim!=(Count_Coords+1)):
                print("Calculate_MVIE error: num of dimensions in "+varname[i]+" of file("+filename+") does match dimension&coordinate informantion input !")
                os._exit(0)
            
            try:     
                exec(exec_str)
            except:
                print("Calculate_MVIE error: cannot select ",Levname,"=",lev_Spc[i]," in ",varname[i]," of file:",filename," !")
                os._exit(0)
        else:
            if ivar.ndim != Count_Coords:
                print("Calculate_MVIE error: num of dimensions in "+varname[i]+" of file("+filename+") does match dimension&coordinate informantion input !")
                os._exit(0)

            exec_str = "ivar_sel = ivar.sel"+Sel_str+")"
            try:     
                exec(exec_str)
            except:
                print("Calculate_MVIE error: cannot select sub part in ",varname[i]," of file:",filename," !")
                os._exit(0)
        
        locals()["ivar_sel"].squeeze()

        exec_str = "ivar_sel.transpose"+Reorder_str
        exec(exec_str)

        if len(varname)==1:
            data = np.reshape(np.array(locals()["ivar_sel"]),(1,locals()["ivar_sel"].size))
        else:
            if i == 0:
                data = np.zeros((len(varname),locals()["ivar_sel"].size), float)
            data[i,:] = np.reshape(np.array(locals()["ivar_sel"]),locals()["ivar_sel"].size)

    var_shp = locals()["ivar_sel"].shape

    if Isarea_wgt == False:
        Area_wgt = [1,]
    else:
        if Count_Coords > 1:
            Area_wgt = np.reshape(Conform_dims(Area_wgt, var_shp, lat_Coords),locals()["ivar_sel"].size)

    return data,Area_wgt,var_shp
#--------------------------------------------------------------
###############################################################
# Generate unique Mask of Nan for pairwise model and reference
###############################################################
def Unif_nan_pair( model, Ref, opt_Var):
    model1 = np.squeeze(model)
    Ref1   = np.squeeze(Ref)

    if opt_Var:
        Mask_unq = model1.mean(axis=0) - model1.mean(axis=0) +\
            Ref1.mean(axis=0) - Ref1.mean(axis=0)
        Mask_unq = np.reshape(np.tile(Mask_unq,Ref1.shape[0]),Ref1.shape)
    else:
        Mask_unq = model1 - model1 + Ref1 - Ref1

    model2 = model1 + Mask_unq
    Ref2   = Ref1 + Mask_unq

    return model2,Ref2
#--------------------------------------------------------------
###############################################################
# Calculate mean of 1D or 2D data with Wgt for rightmost dimention
###############################################################
def AvgWgt(data,opt_wgt):
    if data.ndim == 1:   # 1D data
        wgt = opt_wgt + data - data
        Avg = np.nansum(data*wgt)/np.nansum(wgt)
    else:   # 2D data
        Avg = np.zeros(data.shape[0],np.float)

        for i in range(data.shape[0]):
            wgt = opt_wgt + data[i,:] - data[i,:]
            Avg[i] = np.nansum(data[i,:]*wgt)/np.nansum(wgt)
            del wgt
            
    return Avg
#--------------------------------------------------------------
###############################################################
# Calculate anomaly field
###############################################################
def Del_mean( var, opt_wgt):
    if var.ndim == 1:   # 1D data
        if len(opt_wgt) == 1:
            ano_field = var - np.nanmean(var) 
        else:
            ano_field = var - AvgWgt(var,opt_wgt) 
    else:   # 2D data
        if len(opt_wgt) == 1:
            ano_field = var - Conform_dims(np.nanmean(var,axis=1),var.shape,axis=0)
        else:
            Varavg = AvgWgt(var,opt_wgt) 
            ano_field = var - Conform_dims(Varavg,var.shape,axis=0)

    return ano_field
#--------------------------------------------------------------
###############################################################
# Centered (opt=0) / Uncentered (opt=1) Vector Similarity 
#  Correlation
###############################################################  
def VSCor( model, Ref, opt, area_wgt ):
    model1 = np.squeeze(model)
    Ref1   = np.squeeze(Ref)

    if opt == 0:
        model2 = Del_mean(model1, area_wgt)
        Ref2   = Del_mean(Ref1, area_wgt)
    else:
        model2 = model1
        Ref2   = Ref1

    if len(area_wgt) == 1:
        opt_axis = np.where(model1.ndim == 1, 0, 1)
        VSC = np.sum(np.nanmean(model2*Ref2,axis=opt_axis))/\
            (np.sqrt(np.sum(np.nanmean(model2**2,axis=opt_axis)))*\
                np.sqrt(np.sum(np.nanmean(Ref2**2,axis=opt_axis))))
    else:
        VSC = np.sum(AvgWgt(model2*Ref2,area_wgt))/\
            (np.sqrt(np.sum(AvgWgt(model2**2,area_wgt)))*np.sqrt(np.sum(AvgWgt(Ref2**2,area_wgt)))) 

    return VSC
#--------------------------------------------------------------
###############################################################
# Centered (opt=0) / Uncentered (opt=1) Root Mean Square 
#  Length
###############################################################
def RMSLength( model, opt, area_wgt ):
    model1 = np.squeeze(model)

    if opt == 0:
        model2 = Del_mean(model1, area_wgt)
    else:
        model2 = model1

    if len(area_wgt) == 1:
        opt_axis = np.where(model1.ndim == 1, 0, 1)
        RMSL = np.sqrt(np.sum(np.nanmean(model2**2,axis=opt_axis)))
    else:
        RMSL = np.sqrt(np.sum(AvgWgt(model2**2,area_wgt)))

    return RMSL
#--------------------------------------------------------------
###############################################################
# Centered (opt=0) / Uncentered (opt=1) Root Mean Square
#  Vector Difference
###############################################################
def RMSVDiff( model, Ref, opt, area_wgt ):
    model1 = np.squeeze(model)
    Ref1   = np.squeeze(Ref)

    if opt == 0:
        model2 = Del_mean(model1, area_wgt)
        Ref2   = Del_mean(Ref1, area_wgt)
    else:
        model2 = model1
        Ref2   = Ref1

    if len(area_wgt) == 1:
        opt_axis = np.where(model1.ndim == 1, 0, 1)
        RMSVD = np.sqrt(np.sum(np.nanmean((model2-Ref2)**2,axis=opt_axis)))
    else:
        RMSVD = np.sqrt(np.sum(AvgWgt((model2-Ref2)**2,area_wgt)))

    return RMSVD
#--------------------------------------------------------------
###############################################################
# Vector Mean Error(VME) or Mean Error of Vector Magnitude (MEVM)
############################################################### 
def VecMError( model, Ref, area_wgt, opt):
    model1 = np.squeeze(model)
    Ref1   = np.squeeze(Ref)

    if model1.ndim == 1:   # ME for scalar variable
        if len(area_wgt) == 1:
            ME = np.nanmean(model1) - np.nanmean(Ref1)
        else:
            ME = AvgWgt(model1,area_wgt) - AvgWgt(Ref1,area_wgt)
        return ME
    else:
        if opt == 1:   # VME for vector variable
            if len(area_wgt) == 1:
                VME = np.sqrt(np.sum((np.nanmean(model1,axis=1) - np.nanmean(Ref1,axis=1))**2))
            else:
                VME = np.sqrt(np.sum((AvgWgt(model1,area_wgt) - AvgWgt(Ref1,area_wgt))**2))
            return VME
        else:   # MEVM for vector variable
            if len(area_wgt) == 1:
                MEVM = np.nanmean(np.sqrt(np.nanmean(model1**2,axis=0))-np.sqrt(np.nanmean(Ref1**2,axis=0)))
            else:
                MEVM = AvgWgt(np.sqrt(np.nanmean(model1**2,axis=0))-np.sqrt(np.nanmean(Ref1**2,axis=0)),area_wgt)
            return MEVM
#--------------------------------------------------------------
###############################################################
# Azimuth for model referring to ref
############################################################### 
def AzimuthM(model,ref):
    Ang_ref = np.arctan2(ref[1,:],ref[0,:])

    model_toref = np.zeros(model.shape,float)
    model_toref[0,:] = model[0,:]*np.cos(Ang_ref)+model[1,:]*np.sin(Ang_ref)
    model_toref[1,:] = model[1,:]*np.cos(Ang_ref)-model[0,:]*np.sin(Ang_ref)

    Ang = np.arctan2(model_toref[1,:],model_toref[0,:])*180./math.pi

    return Ang
#--------------------------------------------------------------
###############################################################
# Mean Error of Vector direction between model and ref
############################################################### 
def MEVecDir( model, Ref, area_wgt ):
    angle = AzimuthM(model,Ref)
       
    if len(area_wgt) == 1:
        MEVD = np.nanmean(angle)
    else:
        MEVD = AvgWgt(angle,area_wgt)

    return(MEVD)
#--------------------------------------------------------------
###############################################################
# Skill scores (S1&S2) for individual scalar/vector field
############################################################### 
def Skill_score2( SD, CORR ):
    R0 = np.max(CORR)

    S1 = 4.*(1.+CORR)/((SD+1./SD)**2*(1.+R0))
    S2 = 4.*(1.+CORR)**4/((SD+1./SD)**2*(1.+R0)**4)

    return S1,S2
#--------------------------------------------------------------
###############################################################
# Matrix for normalization and adding variable weighting for
#   multivariable field
############################################################### 
def Nor_matrix( ref, Var_D, M_wgt, area_wgt ):
    RMS_nor = np.zeros(len(ref), float)

    iD = 0
    if isinstance(M_wgt,(int,float)):
        for i in range(len(Var_D)):
            RMS_nor[iD:(iD+Var_D[i])] = 1./RMSLength(ref[iD:(iD+Var_D[i]),:],1,area_wgt)
            iD += Var_D[i]
    else:
        for i in range(len(Var_D)):
            RMS_nor[iD:(iD+Var_D[i])] = M_wgt[i]/RMSLength(ref[iD:(iD+Var_D[i]),:],1,area_wgt)
            iD += Var_D[i]

    return Conform_dims(RMS_nor,ref.shape,axis=0)
#--------------------------------------------------------------
###############################################################
# Calculate rms_std / SD_std with variable weighting
############################################################### 
def Std_varWgt( ratio, M_wgt):
    ratio_avg = np.average(ratio, weights=M_wgt)
    ratio_std = np.sqrt(np.average((ratio-ratio_avg)**2, weights=M_wgt))

    return ratio_std
#--------------------------------------------------------------
###############################################################
# Calculate Multivariable Intergrated Evaluation Index (MIEI)
#  Model and Ref data should be normalized and added variable 
#  weighting if choosing
############################################################### 
def MIEIndex( rms, VSC, M_wgt, F):
    R_rms = np.where(rms>1, 1./rms, rms)

    if isinstance(M_wgt,(int,float)):
        MIEI2 = np.mean((R_rms-1.)**2)+F*(1.-VSC)
    else:
        MIEI2 = np.average((R_rms-1.)**2, weights=M_wgt)+F*(1.-VSC)
    MIEI = np.where(MIEI2<0, 0, np.sqrt(MIEI2))
        
    return MIEI
#--------------------------------------------------------------
###############################################################
# Calculate Multivariable Intergrated Skill Score (MISS)
############################################################### 
def MISScore( rms, VSC, M_wgt, F):
    MIEI = MIEIndex(rms,VSC,M_wgt,F)
    MISS = (F+1.-MIEI**2)/(F+1.)

    return MISS
#--------------------------------------------------------------
###############################################################
# Print range of stats for individual scalar/vector field
############################################################### 
def Pri_stat_range1( stats, s, stats_names):
    print("      *********************")
    print("      *  Ranges of stats  *")
    print("      *********************")
    print("----------------------------------")

    for i in range(len(stats_names)):
        print("  ",'%-10s'%stats_names[i],":",'%6.3f'%np.min(stats[i,:]),",",'%6.3f'%np.max(stats[i,:]))
    print("----------------------------------")

    S_names = ["S1_cen","S2_cen","S1_uncen","S2_uncen"]
    for i in range(4):
        print("  ",'%-10s'%S_names[i],":",'%6.3f'%np.min(s[i,:]),",",'%6.3f'%np.max(s[i,:]))
    print("----------------------------------")

    return
#--------------------------------------------------------------
###############################################################
# Print range of stats for multivariable integrated field
############################################################### 
def Pri_stat_range2( stats1, stats2, MISS, opt_ME=1):
    print("      *********************")
    print("      *  Ranges of stats  *")
    print("      *********************")
    print("----------------------------------")

    List_stats1 = [[[np.min(stats1[0,:,:]),np.min(stats2[0,:]),np.min(MISS[0,:]),np.min(MISS[1,:])],\
        [np.min(stats1[1,:,:]),np.min(stats2[1,:])],\
            [np.min(stats2[2,:]),np.min(stats1[2,:,:]),np.min(stats2[3,:])]],\
                [[np.max(stats1[0,:,:]),np.max(stats2[0,:]),np.max(MISS[0,:]),np.max(MISS[1,:])],\
                    [np.max(stats1[1,:,:]),np.max(stats2[1,:])],\
                        [np.max(stats2[2,:]),np.max(stats1[2,:,:]),np.max(stats2[3,:])]]]

    if stats2.shape[0] == 5:
        List_names = [["CORR      ","cVSC      ","MISS_cen  ","MISS_uncen"], \
            ["SD        ","cRMSL     "], \
                ["SD_std    ","cRMSD     ","cRMSVD    "], \
                    ["ME        ","VME       "]]
        if opt_ME==0:
            List_names[3][1] = "MEVM"
            
        List_stats2 = [[np.min(stats1[3,:,:]),np.min(stats2[4,:])], \
            [np.max(stats1[3,:,:]),np.max(stats2[4,:])]]
    else:
        List_names = [["uCORR     ","VSC       ","MISS_cen  ","MISS_uncen"], \
            ["rms       ","RMSL      "], \
                ["rms_std   ","RMSD      ","RMSVD     "]]

    for i in range(3):
        for j in range(len(List_names[i])):
            print("  ",'%-10s'%List_names[i][j],":",'%6.3f'%List_stats1[0][i][j],",",'%6.3f'%List_stats1[1][i][j])
        print("----------------------------------")

    if stats2.shape[0] == 5:
        for j in range(len(List_names[3])):
            print("  ",'%-10s'%List_names[3][j],":",'%6.3f'%List_stats2[0][j],",",'%6.3f'%List_stats2[1][j])
        print("----------------------------------")

    return
#--------------------------------------------------------------