#!/usr/bin/perl -w

use strict;
#
# Defined HARMONIE domains
#
# For each domain the following properties should be set
#
# TSTEP    Time step
# NLON     Number of points (x)
# NLAT     Number of points (y)
# LONC     Longitude of domain centre (degrees)
# LATC     Latitude of domain center (degrees)
# LON0     Reference longitude of the projection (degrees)
# LAT0     Reference latitude of the projection (degrees)
# GSIZE    Grid size in meters (x,y)
#

my $default_ezone=11;

sub Harmonie_domains{
  my $domain=shift;

  my %domains = (
   'REU12_N' =>{
      'TSTEP'  => '300',
      'NLON'   => '324',
      'NLAT'   => '360',
      'LONC'   => '12.0',
      'LATC'   => '60.6',
      'LON0'   => '10.0',
      'LAT0'   => '60.6',
      'GSIZE'  => '12000.',
   },
   'REU12_S' =>{
      'TSTEP'  => '300',
      'NLON'   => '324',
      'NLAT'   => '300',
      'LONC'   => '8.0',
      'LATC'   => '46.0',
      'LON0'   => '14.0',
      'LAT0'   => '46.0',
      'GSIZE'  => '12000.',
   },
   'PALP3' =>{
      'TSTEP'  => '75',
      'NLON'   => '600',
      'NLAT'   => '512',
      'LONC'   => '10.0',
      'LATC'   => '45.5',
      'LON0'   => '16.0',
      'LAT0'   => '45.5',
      'GSIZE'  => '3000.',
   },
   'PALP2.5'=>{
      'TSTEP'  => '60',
      'NLON'   => '800',
      'NLAT'   => '864',
      'LONC'   => '5.25',
      'LATC'   => '48.25',
      'LON0'   => '18.0',
      'LAT0'   => '48.25',
      'GSIZE'  => '2500.',
   },
   'NORWAY2.5'=>{
      'TSTEP'  => '60',
       'NLON'  => '360',
       'NLAT'  => '800',
       'LONC'  => '14.2',
       'LATC'  => '65.2',
       'LON0'  => '-23',
       'LAT0'  => '64.35',
       'GSIZE' => '2500.',
   },
   'IBERIA_2.5'=>{
      'TSTEP'  => '60',
      'NLON'   => '576',
      'NLAT'   => '480',
      'LONC'   => '-2.5',
      'LATC'   => '40.0',
      'LON0'   => '-2.5',
      'LAT0'   => '40.0',
      'GSIZE'  => '2500.',
   },
   'ARCTIC24' =>{
      'TSTEP'  => '600',
      'NLON'   => '320',
      'NLAT'   => '360',
      'LONC'   => '-90.0',
      'LATC'   => '89.0',
      'LON0'   => '0.0',
      'LAT0'   => '90.0',
      'GSIZE'  => '24000.',
   },
   'ARCTIC12'=>{
      'TSTEP'  => '300',
      'NLON'   => '640',
      'NLAT'   => '720',
      'LONC'   => '-90.0',
      'LATC'   => '89.0',
      'LON0'   => '0.0',
      'LAT0'   => '90.0',
      'GSIZE'  => '12000.',
   },
   'AFRICA50' =>{
      'TSTEP'  => '1200',
      'NLON'   => '240',
      'NLAT'   => '270',
      'LONC'   => '18.0',
      'LATC'   => '-3.0',
      'LON0'   => '18.0',
      'LAT0'   => '3.0',
      'GSIZE'  => '50000.',
   },
   'EAFR25' =>{
      'TSTEP'  => '600',
      'NLON'   => '108',
      'NLAT'   => '90',
      'LONC'   => '33.0',
      'LATC'   => '-3.6',
      'LON0'   => '33.0',
      'LAT0'   => '3.6',    
      'GSIZE'  => '25000.',
   },
   'EAFR12.5' =>{
      'TSTEP'  => '360',
      'NLON'   => '200',
      'NLAT'   => '160',
      'LONC'   => '33.0',
      'LATC'   => '-3.6',
      'LON0'   => '33.0',
      'LAT0'   => '3.6',    
      'GSIZE'  => '12500.',
   },
   'LVIC2.5' =>{
      'TSTEP'  => '60',
      'NLON'   => '450',
      'NLAT'   => '486',
      'LONC'   => '33.0',
      'LATC'   => '-3.05',
      'LON0'   => '33.0',
      'LAT0'   => '3.05',    
      'GSIZE'  => '2500.',
   },
  );

  #
  # List available domains if the requested is not recognized
  #

  if ( ! exists $domains{$domain} ) {

    print "\nCould not find the definition of domain ".$domain." in Harmonie_domains.pm\n";
    print "Available domains are:\n\n";

    for my $key ( sort keys %domains ) {
       print "  $key\n";
    } ;
    exit 1 ;
  };

  # Check if all needed values are set and calclulate necessary ones
  foreach my $var ( ('TSTEP','NLON','NLAT','LONC','LATC','LON0','LAT0','GSIZE' )){
    unless ( $domains{$domain}{$var} ) { print "ERROR: $var is not set in Harmonie_domains.pm for domain $domain!\n"; exit 1;}
  };

  # EZONE if not given by user
  unless ( $domains{$domain}{'EZONE'} ) { $domains{$domain}{'EZONE'}=$default_ezone; }

  # NNOEXTZX if not given by user
  unless ( $domains{$domain}{'NNOEXTZX'} ) { $domains{$domain}{'NNOEXTZX'}=0; }
  # NNOEXTZY if not given by user
  unless ( $domains{$domain}{'NNOEXTZY'} ) { $domains{$domain}{'NNOEXTZY'}=0; }

  # Dimension of grid without extenzion zone
  my $NLON=$domains{$domain}{'NLON'};
  my $NLAT=$domains{$domain}{'NLAT'};
  my $EZONE=$domains{$domain}{'EZONE'};
  my $NNOEXTZX=$domains{$domain}{'NNOEXTZX'};
  my $NNOEXTZY=$domains{$domain}{'NNOEXTZY'};
  $domains{$domain}{'NDLUXG'}=$NLON - $EZONE ;
  $domains{$domain}{'NDGUXG'}=$NLAT - $EZONE ;
  unless ( $NNOEXTZX == 0 ) { $domains{$domain}{'NDLUXG'}=$NLON - $NNOEXTZX ;}
  unless ( $NNOEXTZY == 0 ) { $domains{$domain}{'NDGUXG'}=$NLAT - $NNOEXTZY ;}

  # Sinus of LAT0
  my $pi = 3.14159265358979;
  my $LAT0=$domains{$domain}{'LAT0'};
  $domains{$domain}{'SINLAT0'}=sin($LAT0 * $pi/180. ) ;

  # Truncation of grid if not given by user
  unless ( $domains{$domain}{'NMSMAX_LINE'} ) { $domains{$domain}{'NMSMAX_LINE'}=floor(( $NLON -2 ) / 2) ; } ;
  unless ( $domains{$domain}{'NSMAX_LINE'} )  { $domains{$domain}{'NSMAX_LINE'}=floor(( $NLAT -2 ) / 2) ; } ;
  
  # Assign the return hash
  my %dom=();
  for my $att ( sort keys %{ $domains{$domain} } ) {
    $dom{$att}=$domains{$domain}{$att};
  };

  return %dom;
}
1;


