#!/usr/bin/env python2
# -*- coding: utf-8 -*-
"""
Created on Wed May 3 16:56:55 2017

@author: davevanwees

Carbon cycle model for 0.25, 0.10, 0.05 deg and 500-meter input data. Can be calculated globally (tile='global') for Africa (tile='Africa'),
or for a specific MODIS tile (tile='MODIS_tileID, e.g. tile='h19v09').

!!! For main physical calculations see the function 'algorithm' !!!

"""

import numpy as np
import matplotlib.pyplot as plt
from scipy import io
import datetime
from dateutil.relativedelta import relativedelta
import netCDF4 as netcdf
import sys
import os
import glob
import re
import time as timer
import h5py
import gdal, gdalconst
from osgeo import osr
import copy
import scipy.ndimage
import openpyxl as excl
import warnings
from pyhdf.SD import SD, SDC
from configparser import ConfigParser
from collections import OrderedDict
from cdo import *; cdo = Cdo()
plt.rc('image', interpolation='none')       # change default image interpolation
gdal.UseExceptions()
os.environ['GDAL_DATA'] = '/Users/davevanwees/Library/Enthought/Canopy/edm/envs/User/share/gdal'
#del CDF_MOD_NETCDF4, CDF_MOD_SCIPY, CDO_PY_VERSION

#warnings.simplefilter('error')      # this raises Warnings as if an error, stopping execution and showing error location.


#import gc
#gc.collect()        # garbage collection, for freeing memory.


class Model:
    ''' This is a Carbon Cycle model class. Class variables (e.g. modsettings, modparams, etc.) must be set up before class definition.
    All other settings are defined in the instance definition (__init__ method).
    Units = gC/m2
    
    Functions
    ---------
    __init__(self, sdir, tile=None, mode=None, resolution=None, firemode=None, act=None, sampleloc=None, config_mod=None, config_init=None)
    add_fire(self)
    algorithm(self, step, month)
    diagnose(self, step, month)
    annual_diag(self)
    save(self)
    load(self)
    load_spinup_pools(self)
    clima(self, X, *argv)
    plot_sampleloc(self, *argv)
    var_overview(self)
    
    -- Static methods --
    amean(X)
    loadGFED(layer, years)
    multiprocess(num_of_cores)
    plot_dict(times, *argv)
    '''
    
    #print 'Setting up Model ... ' ,
    
    # Load external methods from supportive scripts.
    from carbonmodel_tile3 import main_tile, load_degbox, load_area, loci, border_check, grid_conv, tile_mask, MODIS_tile_corners, load_proj, haversine, pixel_area, tilemap, construct_geolocation, tile_index, MODIS_pixloc
    from carbonmodel_read3 import main_read, readdata_LWMASK, readdata_MODIS, readdata_LCperc, readdata_CDO, loadGFED, readdata_WURbiomass, readdata_WURsoil, readdata_BouvetAGB, readdata_MODISperc, readdata_Hansen
    
    wdir = '/Volumes/Mac_HD/Work/Vici_project/Koolstof_model/'      # model directory
    wdir_geo = '/Volumes/Mac_HD/Work/Data/MODIS_old/MODIS_geolocation_500m/'  # geolocation file directory
    ddir = '/Volumes/Mac_HD/Work/Data/MODIS_old/'                             # data directory

    wdir_geo = '/Volumes/LaCie/Data_backup/MODIS_old/MODIS_geolocation_500m/'
    ddir = '/Volumes/LaCie/Data_backup/MODIS_old/'
    
    # model initialization counters
    loadn = 1
    modcounters = {
        "loadn": loadn,
        "num_of_instances": 0,
        "plotn": 0,
    } 
    
    # model class settings
    modsettings = OrderedDict([
        ("start", 2),  # (default=2)
        ("spinup", 200),  # (default=200)
        ("execut", 15),  # years_savg + execut = len(years) (default=15)
        ("year_savg", 5),  # number of years in spinup average (defualt=5)
        ("fastload", 'on'),  # loads data only for spinup time period.
        ("soil_moisture", 'SWVL1'),  # choose soil moisture dataset, 'CCI', 'SWVL1' (Era-Interim) or 'SWVL1-Land' (Era_Interim-Land).
        ("speedspin", 'off'),  # use spinup outcome estimates to speedup spinup, based on turnover rates.
        ("yrspin", 'on'),   # spinup in annual timesteps.
        ("herbivory", 'off'),    # herbivory
        ("float_dtype", '32'), # run model on float64 or float32 precision.
        ("biome_method", 'majority'),  # ['majority' or 'weighted' or 'perbiome' or 'skip']. Majority biome, or percentage biome weighted average, or each biome seperate for scaling of turnover, f_cw, etc, or no biome-specific parameters at all (constant value).
        ("MBF", 'off'),     # Also works with percentual biome variables now (bp).
    ])
    
    # model class parameters
    modparams = OrderedDict([
        # "f_cw": 0.5,                                # g C per W (conversion of radiation to carbon)
        ("f_cw", 0.3 * 2.64),  # Field et al.(1995): 0.3 gC/MJ. 1 W*month = 2.64 MJ. -> 0.3 gC/MJ = 0.79 gC/W*month (or gC/W/month). See also Potter et al. 1993.
        ("q10", 1.5),  # temperature effect on respiration. Rate of change in case of 10 C degree increase in temperature.
        ("f_resp", 0.5),  # fraction of dead carbon losses that is respired
        ("turnover_leaf", (1 /  0.5) / 12),  # fraction per month   [gC/month ???]
        ("turnover_gras", (1 /  0.5) / 12),
        ("turnover_litt", (1 /  0.5) / 12),
        ("turnover_stem", (1 / 30.0) / 12),
        ("turnover_root", (1 /  1.0) / 12),
        ("turnover_cwd",  (1 /  4.0) / 12),
        ("turnover_soil", (1 /  1.0) / 12),
        ("turnover_slow", (1 / 10.0) / 12),
        ("f_hc", 0.5),  # Herbivore consumption efficiency = 50%
    ])
    
    
    def biome_params(modparams):
        # Adjust parameters to biome-specific values.
        
        modparams['f_cw'] = np.array([0.3, 0.284, 0.354, 0.280, 0.255, 0.283, 0.299, 0.208, np.nan, np.nan, 0.229, 0.242, np.nan, 0.135, np.nan, 0.269])    # Potter 1993 / Field 1995
        modparams['f_cw'][8] = 0.250  # woody savannah: mid-value
        modparams['f_cw'][9] = 0.229  # normal savannah: taken same value as grassland
        modparams['f_cw'][12] = 0.150    # urban
        modparams['f_cw'][14] = 0.250  # boreal sparse forest: mid-value, same as woody savannah
        modparams['f_cw'][2] = 0.3      # latest adjustments
        modparams['f_cw'][12] = 0.3     # urban
        
        # FINE-TUNING FOR AFRICA
        modparams['f_cw'][8] = 0.280
        modparams['f_cw'][9] = 0.208 # normal savannah: taken same value as open shrubland
        modparams['f_cw'][2] = 0.300 # tropical forest.
        
        modparams['f_cw'] = modparams['f_cw'] * 2.64        # unit conversion.
        
        ''' GFED 4 values (Werf et al. 2017)
        decid broadleaf = 40
        decid needleleaf = 65
        everg broadleaf = 55
        everg needleleaf = 52
        '''
        
        #modparams['turnover_stem'] = np.repeat(modparams['turnover_stem'], 16)
        #modparams['turnover_stem'][[1, 2, 3]] = (1 / 50.0) / 12
        #modparams['turnover_stem'][[5]] = (1 / 35.0) / 12
        #modparams['turnover_stem'][[8]] = (1 / 27.0) / 12
        #modparams['turnover_stem'][[10, 11, 14]] = (1 / 18.0) / 12
        #modparams['turnover_stem'][[9]] = (1 / 14.0) / 12
        #modparams['turnover_stem'][[12]] = (1 / 12.0) / 12
        
        modparams['turnover_stem'] = np.repeat(modparams['turnover_stem'], 16)  # inluding water, so 16.
        modparams['turnover_stem'][[1, 2, 3]] = (1 / 60.0) / 12
        modparams['turnover_stem'][[4]] = (1 / 35.0) / 12
        modparams['turnover_stem'][[5]] = (1 / 42.0) / 12
        modparams['turnover_stem'][[8]] = (1 / 32.0) / 12
        modparams['turnover_stem'][[9]] = (1 / 12.0) / 12
        modparams['turnover_stem'][[10]] = (1 / 18.0) / 12
        modparams['turnover_stem'][[11, 12]] = (1 / 15.0) / 12
        modparams['turnover_stem'][[14]] = (1 / 30.0) / 12
        
        # FINE-TUNING FOR AFRICA
        modparams['turnover_litt'] = np.repeat(modparams['turnover_litt'], 16)  # inluding water, so 16.
        modparams['turnover_cwd'] = np.repeat(modparams['turnover_cwd'], 16)  # inluding water, so 16.
        modparams['turnover_litt'][[7, 9, 10, 11]] = (1 / 0.3) / 12
        modparams['turnover_cwd'][[7, 9, 10, 11]] = (1 / 1.0) / 12
        modparams['turnover_gras'] = (1 / 0.3) / 12
        
        # new 10-06-2018
        modparams['turnover_litt'][[6, 7, 8, 9, 10, 11]] = (1 / 0.2) / 12
        modparams['turnover_cwd'][[8]] = (1 / 2.0) / 12
        modparams['turnover_gras'] = (1 / 0.5) / 12
        
        # new 14-06-2018: aiming to rotate the scatter plot
        modparams['turnover_gras'] = np.repeat(modparams['turnover_gras'], 16)  # inluding water, so 16.
        modparams['turnover_gras'][[10, 11]] = (1 / 0.2) / 12            # factor 2.5 reduction
        modparams['turnover_litt'][[7, 10, 11]] = (1 / 0.1) / 12        # factor 2 reduction
        
        modparams['turnover_gras'][[7, 8, 9]] = (1 / 0.3) / 12
        modparams['turnover_litt'][[8]] = (1 / 0.15) / 12
        
        # new 16-06-2018: trying to boost woody_SA FC by a factor 1.5.
        #modparams['turnover_gras'][[8]] = (1 / 0.5) / 12
        #modparams['turnover_litt'][[8]] = (1 / 0.2) / 12
        
        # new 17-06-2018: boost norma_SA a bit.
        #modparams['turnover_gras'][[9]] = (1 / 0.4) / 12
        
        # new 17-06-2018_2: boost norma_SA a bit more.
        #modparams['turnover_gras'][[9]] = (1 / 0.5) / 12
        
        modparams['f_cw'][2] = 0.354 * 2.64 # tropical forest.
        modparams['turnover_stem'][[4]] = (1 / 60.0) / 12
        modparams['turnover_stem'][[8]] = (1 / 40.0) / 12
        modparams['turnover_stem'][[9]] = (1 / 10.0) / 12
        
        
        
        # new 30-07-2018: building on 14-06
        modparams['turnover_stem'][[5]] = (1 / 35.0) / 12
        modparams['turnover_stem'][[8]] = (1 / 35.0) / 12
        modparams['turnover_stem'][[9]] = (1 / 8.0) / 12
        
        modparams['turnover_stem'][[9]] = (1 / 5.0) / 12
        
        modparams['turnover_gras'][[10]] = (1 / 0.1) / 12
        
        #potential option to lower woody savanna:
        #modparams['turnover_cwd'][[8]] = (1 / 1.0) / 12
        
        
        
        
        modparams['turnover_leaf'] = np.repeat(modparams['turnover_leaf'], 16)  # inluding water, so 16.
        modparams['turnover_leaf'][[1, 3]] = (1 / 2.0) / 12     # needleleaf turnover: Agren et al. (2008)
        modparams['turnover_leaf'][[2, 5]] = (1 / 1.0) / 12
        
        return modparams
    
    
    def biome_params_Africa(modparams, setnum=None):
        # Adjust parameters to biome-specific values.
        
        modparams['f_cw'] = np.array([0.3, 0.284, 0.354, 0.280, 0.255, 0.283, 0.299, 0.208, np.nan, np.nan, 0.229, 0.242, np.nan, 0.135, np.nan, 0.269])  # Potter 1993 / Field 1995
        modparams['f_cw'][8] = 0.280  # mid value
        modparams['f_cw'][9] = 0.208  # normal savannah: taken same value as open shrubland
        modparams['f_cw'][12] = 0.3  # urban
        modparams['f_cw'][14] = 0.250  # boreal sparse forest: mid-value, same as woody savannah
        modparams['f_cw'] = modparams['f_cw'] * 2.64        # unit conversion
        
        ''' GFED 4 values (Werf et al. 2017)
        decid broadleaf = 40
        decid needleleaf = 65
        everg broadleaf = 55
        everg needleleaf = 52
        '''
        
        modparams['turnover_stem'] = np.repeat(modparams['turnover_stem'], 16)  # inluding water, so 16.
        modparams['turnover_stem'][[1, 2, 3]] = (1 / 60.0) / 12
        modparams['turnover_stem'][[4]] = (1 / 35.0) / 12
        modparams['turnover_stem'][[5]] = (1 / 42.0) / 12
        modparams['turnover_stem'][[8]] = (1 / 32.0) / 12
        modparams['turnover_stem'][[9]] = (1 / 12.0) / 12
        modparams['turnover_stem'][[10]] = (1 / 18.0) / 12
        modparams['turnover_stem'][[11, 12]] = (1 / 15.0) / 12
        modparams['turnover_stem'][[14]] = (1 / 30.0) / 12
        
        modparams['turnover_stem'][[5]] = (1 / 35.0) / 12
        modparams['turnover_stem'][[8]] = (1 / 35.0) / 12   # was (1 / 40.0) / 12
        modparams['turnover_stem'][[9]] = (1 / 5.0) / 12    # was (1 / 10.0) / 12
        
        modparams['turnover_leaf'] = np.repeat(modparams['turnover_leaf'], 16)  # inluding water, so 16.
        modparams['turnover_leaf'][[1, 3]] = (1 / 2.0) / 12     # needleleaf turnover: Agren et al. (2008)
        modparams['turnover_leaf'][[2, 5]] = (1 / 1.0) / 12
        
        modparams['turnover_litt'] = np.repeat(modparams['turnover_litt'], 16)  # inluding water, so 16.
        modparams['turnover_litt'][[6, 9]] = (1 / 0.2) / 12
        modparams['turnover_litt'][[7, 10, 11]] = (1 / 0.1) / 12
        modparams['turnover_litt'][[8]] = (1 / 0.15) / 12
        
        modparams['turnover_cwd'] = np.repeat(modparams['turnover_cwd'], 16)  # inluding water, so 16.
        modparams['turnover_cwd'][[7, 9, 10, 11]] = (1 / 1.0) / 12
        modparams['turnover_cwd'][[8]] = (1 / 2.0) / 12
        
        modparams['turnover_gras'] = (1 / 0.5) / 12
        modparams['turnover_gras'] = np.repeat(modparams['turnover_gras'], 16)  # inluding water, so 16.
        modparams['turnover_gras'][[7, 8, 9]] = (1 / 0.3) / 12
        modparams['turnover_gras'][[11]] = (1 / 0.2) / 12
        modparams['turnover_gras'][[10]] = (1 / 0.1) / 12
        
        modparams['turnover_cwd'][[8]] = (1 / 1.0) / 12    # lower woody savanna
        
        #new 03-08:
        modparams['turnover_stem'][[8]] = (1 / 30.0) / 12
        modparams['turnover_stem'][[9]] = (1 / 3.0) / 12
        modparams['turnover_gras'][[6]] = (1 / 0.3) / 12
        
        # FOR 17-06_2
        if setnum == 2:
            modparams['turnover_stem'][[4]] = (1 / 60.0) / 12
            modparams['turnover_stem'][[8]] = (1 / 40.0) / 12
            modparams['turnover_stem'][[9]] = (1 / 10.0) / 12    
            
            modparams['turnover_gras'][[8]] = (1 / 0.5) / 12
            modparams['turnover_litt'][[8]] = (1 / 0.2) / 12
            
            modparams['turnover_gras'][[9]] = (1 / 0.5) / 12
            
            modparams['turnover_cwd'][[8]] = (1 / 2.0) / 12
        
        # FOR Bouvet tune
        if setnum == 3:
            
            modparams['f_cw'][8] = 0.280 * 2.64  # mid value
            modparams['f_cw'][9] = 0.280 * 2.64  # normal savannah: taken same value as open shrubland
            
            modparams['turnover_stem'][[4, 5, 6, 7, 8, 9]] = (1 / 60.0) / 12
            modparams['turnover_stem'][[10, 11]] = (1 / 60.0) / 12
            
            modparams['turnover_litt'][[6, 7, 8, 9, 10, 11]] = (1 / 0.2) / 12
            modparams['turnover_cwd'][[7, 8, 9, 10, 11]] = (1 / 4.0) / 12
            
            # modparams['turnover_gras'][[7,8,9]] = (1 / 0.4) / 12
            # modparams['turnover_gras'][[10]] = (1 / 0.1) / 12
            # modparams['turnover_litt'][[9]] = (1 / 0.3) / 12
            # modparams['turnover_cwd'][[8]] = (1 / 4.0) / 12
            
        
        return modparams
    
    
    if modsettings['biome_method'] != 'skip':
        modparams = biome_params_Africa(modparams) #, setnum=3)
    
    # fire parameters
    fireparams = OrderedDict([
        ("settling", 0.3),  # part of fire product that settles on/in ground. the other part is released in smoke.
        ("cc_leaf", [0.8, 1.0]),  # CC, Combustion completeness min and max
        ("cc_stem", [0.2, 0.4]),
        ("cc_litt", [0.9, 1.0]),  # Fine leaf litter
        ("cc_cwd",  [0.4, 0.6]),
        ("cc_soil", [0.9, 1.0]),  # From Werf et al. (2006). Is upgraded to burn depth in Werf et al. (2010)
    ])
    
    regions = np.load(wdir_geo + 'GFED_tiles_per_region.npy')[()]       # per region which tiles overlap fully or partly with region.
    '''
    regions = OrderedDict([         # print result from the load above:
        ('Ocean', (0, ['h00v08', 'h00v09', 'h00v10', 'h01v07', 'h01v08', 'h01v09', 'h01v10', 'h01v11', 'h02v06', 'h02v08', 'h02v09', 'h02v10', 'h02v11', 'h03v05', 'h03v06', 'h03v07', 'h03v08', 'h03v09', 'h03v10', 'h03v11', 'h04v05', 'h04v09', 'h04v10', 'h04v11', 'h05v10', 'h05v11', 'h05v13', 'h06v03', 'h06v11', 'h07v03', 'h07v05', 'h07v06', 'h07v07', 'h08v03', 'h08v04', 'h08v05', 'h08v06', 'h08v07', 'h08v08', 'h08v09', 'h08v11', 'h09v02', 'h09v03', 'h09v04', 'h09v06', 'h09v07', 'h09v08', 'h09v09', 'h10v02', 'h10v03', 'h10v05', 'h10v06', 'h10v07', 'h10v08', 'h10v09', 'h10v10', 'h10v11', 'h11v01', 'h11v02', 'h11v03', 'h11v04', 'h11v05', 'h11v06', 'h11v07', 'h11v08', 'h11v10', 'h11v11', 'h11v12', 'h12v01', 'h12v02', 'h12v03', 'h12v04', 'h12v05', 'h12v07', 'h12v08', 'h12v12', 'h12v13', 'h13v01', 'h13v02', 'h13v03', 'h13v04', 'h13v08', 'h13v09', 'h13v11', 'h13v12', 'h13v13', 'h13v14', 'h14v00', 'h14v01', 'h14v02', 'h14v03', 'h14v04', 'h14v09', 'h14v10', 'h14v11', 'h14v14', 'h14v15', 'h14v16', 'h15v00', 'h15v01', 'h15v02', 'h15v03', 'h15v05', 'h15v07', 'h15v11', 'h15v14', 'h15v15', 'h15v16', 'h16v00', 'h16v01', 'h16v02', 'h16v05', 'h16v06', 'h16v07', 'h16v08', 'h16v09', 'h16v12', 'h16v14', 'h16v16', 'h17v00', 'h17v01', 'h17v02', 'h17v03', 'h17v04', 'h17v05', 'h17v06', 'h17v08', 'h17v10', 'h17v12', 'h17v13', 'h17v15', 'h17v16', 'h18v00', 'h18v01', 'h18v02', 'h18v03', 'h18v04', 'h18v05', 'h18v08', 'h18v09', 'h18v14', 'h18v15', 'h18v16', 'h19v00', 'h19v01', 'h19v02', 'h19v03', 'h19v04', 'h19v05', 'h19v07', 'h19v09', 'h19v10', 'h19v11', 'h19v12', 'h19v15', 'h19v16', 'h20v00', 'h20v01', 'h20v02', 'h20v03', 'h20v04', 'h20v05', 'h20v09', 'h20v11', 'h20v12', 'h20v13', 'h20v15', 'h20v16', 'h21v00', 'h21v01', 'h21v04', 'h21v05', 'h21v06', 'h21v07', 'h21v08', 'h21v09', 'h21v10', 'h21v11', 'h21v13', 'h21v15', 'h21v16', 'h22v01', 'h22v04', 'h22v05', 'h22v06', 'h22v07', 'h22v08', 'h22v09', 'h22v10', 'h22v11', 'h22v13', 'h22v14', 'h22v15', 'h22v16', 'h23v01', 'h23v02', 'h23v03', 'h23v04', 'h23v06', 'h23v07', 'h23v08', 'h23v09', 'h23v10', 'h23v11', 'h23v15', 'h23v16', 'h24v01', 'h24v02', 'h24v03', 'h24v06', 'h24v07', 'h24v10', 'h24v12', 'h24v15', 'h24v16', 'h25v02', 'h25v03', 'h25v05', 'h25v07', 'h25v08', 'h25v09', 'h26v02', 'h26v03', 'h26v05', 'h26v06', 'h26v07', 'h26v08', 'h27v03', 'h27v04', 'h27v05', 'h27v06', 'h27v07', 'h27v08', 'h27v09', 'h27v10', 'h27v11', 'h27v12', 'h27v14', 'h28v03', 'h28v04', 'h28v05', 'h28v06', 'h28v07', 'h28v08', 'h28v09', 'h28v10', 'h28v11', 'h28v12', 'h28v13', 'h28v14', 'h29v03', 'h29v05', 'h29v06', 'h29v07', 'h29v08', 'h29v09', 'h29v10', 'h29v11', 'h29v12', 'h29v13', 'h30v04', 'h30v05', 'h30v06', 'h30v07', 'h30v08', 'h30v09', 'h30v10', 'h30v12', 'h30v13', 'h31v04', 'h31v06', 'h31v07', 'h31v08', 'h31v09', 'h31v10', 'h31v11', 'h31v12', 'h31v13', 'h32v07', 'h32v08', 'h32v09', 'h32v10', 'h32v11', 'h32v12', 'h33v07', 'h33v08', 'h33v09', 'h33v10', 'h33v11', 'h34v07', 'h34v08', 'h34v09', 'h34v10', 'h35v08', 'h35v09', 'h35v10'])), 
        ('BONA', (1, ['h06v03', 'h07v03', 'h08v03', 'h09v02', 'h09v03', 'h09v04', 'h10v02', 'h10v03', 'h10v04', 'h11v02', 'h11v03', 'h11v04', 'h12v01', 'h12v02', 'h12v03', 'h12v04', 'h13v01', 'h13v02', 'h13v03', 'h13v04', 'h14v01', 'h14v02', 'h14v03', 'h14v04', 'h15v01', 'h15v02', 'h16v01'])),
        ('TENA', (2, ['h07v05', 'h08v04', 'h08v05', 'h08v06', 'h09v04', 'h09v05', 'h09v06', 'h10v04', 'h10v05', 'h10v06', 'h11v04', 'h11v05', 'h12v04', 'h12v05', 'h13v04'])),
        ('CEAM', (3, ['h07v05', 'h07v06', 'h07v07', 'h08v05', 'h08v06', 'h08v07', 'h09v05', 'h09v06', 'h09v07', 'h09v08', 'h10v06', 'h10v07', 'h10v08', 'h11v06', 'h11v07', 'h12v05', 'h12v07'])),
        ('NHSA', (4, ['h09v08', 'h10v07', 'h10v08', 'h11v07', 'h11v08', 'h12v07', 'h12v08', 'h13v08'])),
        ('SHSA', (5, ['h09v09', 'h10v09', 'h10v10', 'h11v09', 'h11v10', 'h11v11', 'h11v12', 'h12v09', 'h12v10', 'h12v11', 'h12v12', 'h12v13', 'h13v09', 'h13v10', 'h13v11', 'h13v12', 'h13v13', 'h13v14', 'h14v09', 'h14v10', 'h14v11', 'h14v14'])),
        ('EURO', (6, ['h15v02', 'h15v03', 'h16v01', 'h16v02', 'h17v01', 'h17v02', 'h17v03', 'h17v04', 'h17v05', 'h18v01', 'h18v02', 'h18v03', 'h18v04', 'h18v05', 'h19v01', 'h19v02', 'h19v03', 'h19v04', 'h19v05', 'h20v04', 'h20v05'])),
        ('MIDE', (7, ['h16v06', 'h17v05', 'h17v06', 'h18v05', 'h18v06', 'h19v04', 'h19v05', 'h19v06', 'h20v04', 'h20v05', 'h20v06', 'h21v04', 'h21v05', 'h21v06', 'h21v07', 'h22v05', 'h22v06', 'h22v07', 'h23v05', 'h23v06', 'h23v07'])),
        ('NHAF', (8, ['h16v06', 'h16v07', 'h16v08', 'h17v06', 'h17v07', 'h17v08', 'h18v06', 'h18v07', 'h18v08', 'h19v06', 'h19v07', 'h19v08', 'h20v06', 'h20v07', 'h20v08', 'h21v06', 'h21v07', 'h21v08', 'h21v09', 'h22v07', 'h22v08', 'h23v07', 'h23v08'])),
        ('SHAF', (9, ['h18v09', 'h19v09', 'h19v10', 'h19v11', 'h19v12', 'h20v09', 'h20v10', 'h20v11', 'h20v12', 'h21v09', 'h21v10', 'h21v11', 'h22v09', 'h22v10', 'h22v11'])),
        ('BOAS', (10, ['h08v03', 'h09v02', 'h10v02', 'h11v02', 'h12v01', 'h18v01', 'h18v02', 'h19v01', 'h19v02', 'h19v03', 'h20v01', 'h20v02', 'h20v03', 'h21v01', 'h21v02', 'h21v03', 'h22v01', 'h22v02', 'h22v03', 'h23v01', 'h23v02', 'h23v03', 'h23v04', 'h24v02', 'h24v03', 'h25v02', 'h25v03', 'h25v04', 'h26v02', 'h26v03', 'h26v04', 'h27v03', 'h27v04', 'h28v03', 'h28v04', 'h29v03'])),
        ('CEAS', (11, ['h19v03', 'h19v04', 'h20v03', 'h20v04', 'h21v03', 'h21v04', 'h21v05', 'h22v03', 'h22v04', 'h22v05', 'h23v03', 'h23v04', 'h23v05', 'h24v03', 'h24v04', 'h24v05', 'h25v03', 'h25v04', 'h25v05', 'h25v06', 'h26v03', 'h26v04', 'h26v05', 'h26v06', 'h27v04', 'h27v05', 'h27v06', 'h28v04', 'h28v05', 'h28v06', 'h28v07', 'h29v05', 'h29v06'])),
        ('SEAS', (12, ['h23v05', 'h23v06', 'h24v05', 'h24v06', 'h24v07', 'h25v06', 'h25v07', 'h25v08', 'h26v06', 'h26v07', 'h26v08', 'h27v06', 'h27v07', 'h27v08', 'h28v06', 'h28v07', 'h28v08', 'h29v06', 'h29v07', 'h30v07', 'h30v08', 'h31v07', 'h31v08', 'h32v07'])),
        ('EQAS', (13, ['h27v08', 'h27v09', 'h28v08', 'h28v09', 'h28v10', 'h29v07', 'h29v08', 'h29v09', 'h29v10', 'h30v08', 'h30v09', 'h30v10', 'h31v08', 'h31v09', 'h31v12', 'h32v09', 'h32v10', 'h32v11', 'h33v08', 'h33v09', 'h33v10', 'h33v11', 'h34v08', 'h34v09', 'h34v10', 'h35v10'])),
        ('AUST', (14, ['h27v11', 'h27v12', 'h28v11', 'h28v12', 'h28v13', 'h29v10', 'h29v11', 'h29v12', 'h29v13', 'h30v10', 'h30v11', 'h30v12', 'h30v13', 'h31v10', 'h31v11', 'h31v12', 'h31v13', 'h32v10', 'h32v11', 'h32v12']))
    ])
    '''
    biomes_UMD = OrderedDict([
        ('0',  ('water_W', 'water', 0)),
        ('1',  ('everg_NF', 'evergreen_needleleaf_forest', 1)),
        ('2',  ('everg_BF', 'evergreen_broadleaf_forest', 2)),
        ('3',  ('decid_NF', 'deciduous_needleleaf_forest', 3)),
        ('4',  ('decid_BF', 'deciduous_broadleaf_forest', 4)),
        ('5',  ('mixed_F', 'mixed_forests', 5)),
        ('6',  ('close_SH', 'closed_shrublands', 6)),
        ('7',  ('open_SH', 'open_shrublands', 7)),
        ('8',  ('woody_SA', 'woody_savannas', 8)),
        ('9',  ('norma_SA', 'savannas', 9)),
        ('10', ('grass_G', 'grasslands', 10)),
        ('11', ('crops_C', 'croplands', 12)),
        ('12', ('urban_U', 'urban_and_built-up', 13)),
        ('13', ('barren_B', 'barren_or_sparsely_vegetated', 16)),
        ('14', ('sparse_BO', 'sparse_boreal_forest', 17)),
        ('15', ('tundra_T', 'tundra', 18)),
        ('xx', ('unclassified', -2)),
    ])
    
    biomes_IGBP = OrderedDict([
        ('0',  ('water', 0)),
        ('1',  ('evergreen_needleleaf_forest', 1)),
        ('2',  ('evergreen_broadleaf_forest', 2)),
        ('3',  ('deciduous_needleleaf_forest', 3)),
        ('4',  ('deciduous_broadleaf_forest', 4)),
        ('5',  ('mixed_forests', 5)),
        ('6',  ('closed_shrublands', 6)),
        ('7',  ('open_shrublands', 7)),
        ('8',  ('woody_savannas', 8)),
        ('9',  ('savannas', 9)),
        ('10', ('grasslands', 10)),
        ('11', ('permanent_wetlands', 11)),
        ('12', ('croplands', 12)),
        ('13', ('urban_and_built-up', 13)),
        ('14', ('cropland_natural_vegetation_mosaic', 14)),
        ('15', ('snow_and_ice', 15)),
        ('16', ('barren_or_sparsely_vegetated', 16)),
        ('17', ('sparse_boreal_forest', 17)),
        ('18', ('tundra', 18)),
        ('xx', ('unclassified', -2)),
    ])
    
    # dataset_dict = {dataset: [product, layer, dtype (to be saved), datarange, (optional:n_types)]}
    # datarange is a tuple with: (valid start, valid end, fill value, (unclassified)).    Water is set to 0, use MOD44W for consistent water mask.
    dataset_dict = {
        "AIRT": 'Era-Interim/AIRT/ECMWF_Era-interim-monthlyofdaily_2000-2017_AIRT_grid-025x025deg.nc',
        "SSR": 'Era-Interim/SSR/ECMWF_Era-interim-monthlyaccum_2000-2017_NSSR_Step-0-12_grid-025x025deg.nc',
        "CCI": 'CCI/ESACCI-SOILMOISTURE-L3S-SSMV-COMBINED-monthly_sm.nc',
        "SWVL1": 'Era-Interim/SWVL1/ECMWF_Era-interim-monthly_2000-2017_SWVL1_grid-025x025deg.nc',
        "SWVL1-Land": 'Era-Interim/SWVL1-Land/ECMWF_Era-interim-Land-monthly_2000-2010_SWVL1_grid-025x025deg.nc',
        "FTC": ['MOD44B', 'Percent_Tree_Cover', 'Byte', (0,100,253)],   # 200 = water.
        "NTV": ['MOD44B', 'Percent_NonTree_Vegetation', 'Byte', (0,100,253)],   # 200 = water
        "Fpar": ['MCD15A2H', 'Fpar_500m', 'Byte', (0,100,255)],     # 249-254 are water, urban, snow, unclassified, etc., should be set 0.
        "BA": ['MCD64A1', 'Burn Date', 'Float32', (0,366,-1)],      # -2 = water. Int16 (16-bit signed integer).
        "LC1": ['MCD12Q1', 'Land_Cover_Type_1', 'Byte', (0,16,255,254)],
        "LC2": ['MCD12Q1', 'Land_Cover_Type_2', 'Byte', (0,16,255,254)],
        "LC1_types": ['MCD12Q1', 'Land_Cover_Type_1', 'Float32', (0,16,255,254), 17],  # IGBP classification
        "LC2_types": ['MCD12Q1', 'Land_Cover_Type_2', 'Float32', (0,16,255,254), 17],  # UMD classification. Still use n_types = 17 because class numbers are not changed.
        "LWM": ['MOD44W', 'water_mask', 'Float32', (0,1,250,253)],     # 250 fill value is specifically: Outside of projection, i.e. edge of globe.
        "LWMb": ['ISLSCP II', 'water_mask', 'Float32'],
        "NPP": ['MOD17A3v5', 'Npp_1km', 'Float32', (0,65500,65535)],     #  65529-65534 are water, urban, snow, unclassified, etc., should be set 0. UInt16.
        "MORT": ['mortality', 'Fire_Tree_Mortality', 'Float32'],
        "Hansen": ['Hansen_ForestLoss/lossyear', 'lossyear', 'Float32', (0,17)],
        }
    
    dataset_dict_new = {
        "AIRT": 'Era-Interim/AIRT/ECMWF_Era-interim-monthlyofdaily_2000-2017_AIRT_grid-025x025deg.nc',
        "SSR": 'Era-Interim/SSR/ECMWF_Era-interim-monthlyaccum_2000-2017_NSSR_Step-0-12_grid-025x025deg.nc',
        "CCI": 'CCI/ESACCI-SOILMOISTURE-L3S-SSMV-COMBINED_1997-2018-v04.4_sm_monthly.nc',
        "SWVL1": 'Era-Interim/SWVL1/ECMWF_Era-interim-monthly_2000-2017_SWVL1_grid-025x025deg.nc',
        "SWVL1-Land": 'Era-Interim/SWVL1-Land/ECMWF_Era-interim-Land-monthly_2000-2010_SWVL1_grid-025x025deg.nc',
        "FTC": ['MOD44B', 'Percent_Tree_Cover', 'Byte', (0,100,253)],  # 200 = water.
        "NTV": ['MOD44B', 'Percent_NonTree_Vegetation', 'Byte', (0,100,253)],  # 200 = water
        "Fpar": ['MCD15A2H', 'Fpar_500m', 'Byte', (0,100,255,249)],  # 250-254 are water, urban, snow, etc., should be set 0.
        "BA": ['MCD64A1', 'Burn Date', 'Float32', (0,366,-1)],  # -2 = water. Int16 (16-bit signed integer).
        "LC": ['MCD12Q1', 'LC_Type1', 'Byte', (0,16,255)],
        "LC_types": ['MCD12Q1', 'LC_Type1', 'Float32', (0,16,255), 17],
        "LWM": ['MOD44W', 'water_mask', 'Float32', (0,1,250,253)],  # 250 fill value is specifically: Outside of projection, i.e. edge of globe.
        "NPP": ['MOD17A3v5', 'Npp_1km', 'Float32', (0,65500,65535,65529)],  # 65530-65534 are water, urban, snow, etc., should be set 0. UInt16.
        "GPP": ['MOD17A2H', 'Gpp_500m', 'UInt16', (0,3000,32767,32761)], # 32762-32766 are water, urban, snow, etc., should be set 0. UInt16.
        "PSN": ['MOD17A2H', 'PsnNet_500m', 'Int16', (-3000,3000,32767,32761)], # 32762-32766 are water, urban, snow, etc., should be set 0. Int16.
        "lossyear": ['Hansen', 'lossyear', 'Float32', (0,17)],
        "gain": ['Hansen', 'gain', 'Float32', (0,1)],
        "FRP": ['MCD14ML', 'FireMask', 'UInt32', (0,)],     # uint16 is not enough, FRP values >100000 exist.
        "PER": ['MCD14ML', 'FireMask', 'Byte', (0,)],       # fire persistence; number of fires in month.
    }
    
    allyears = ['2000', '2001', '2002', '2003', '2004', '2005', '2006', '2007', '2008', '2009', '2010', '2011', '2012', '2013', '2014', '2015', '2016', '2017']
    
    region_box = {
        'Africa': ([200, 510], [640, 950]),     # 0.25 degree indices of area.
    }
    
    tiles_box = {
        'Africa': sorted(set(regions['NHAF'][1] + regions['SHAF'][1])),
        'DEFOS': ['h12v09', 'h12v10', 'h13v09', 'h13v10'],
        'SHSA': sorted(set(regions['SHSA'][1])),
    }
    
    def __init__(self, sdir, tile=None, mode=None, resolution=None, firemode=None, act=None, sampleloc=None, s_pixelmode=False, config_mod=None, config_init=None):
        
        '''
        Creates a new model instance
        :param sdir: directory location for unpacking model. Directory is created if not yet existing. log and init files, result datasets, and run-specific plots are stored here. Valid input = 'full dir path' [string]
        :param tile: [optional, default='global']: MODIS tile ID or global (in case of res=0.25deg). Valid input = 'tile ID', 'global' [string]
        :param mode: [optional, default=1]: mode 1 (spinup) or mode 2 (execute). Valid input = 1, 2 [integer]
        :param resolution: [optional, default=0.25]: 0.25 degrees or 500m resolution. Valid input = 0.25, 500 [float]
        :param firemode: [optional, default='on']: incorporate fires. Valid input = 'on', 'off' [string]. firemode can be switched after initialization using .add_fire(). This saves us from loading all data again
        :param act: [optional, default='active']: 'active' or 'passive'. Valid input = 'active', 'passive' [string]. 'active' mode loads all data, 'passive' only loads instance parameters as a frame to load results in, using .load()
        :param sampleloc: [optional, default=(391,796)]: list with sample y and x coordinate as indices. Valid input = [lat, lon] (if int: index, if float: degrees)
        :param s_pixelmode: [optional, defualt=False]: if the used spinup is a per-pixel spinup (True) or a full map spinup (False).
        :param config_mod: [optional,default=class settings]: for changing modsettings during initialization. Valid input = {dict}
        :param config_init: [optional,default=init settings]: for changing init settings during initialization. Valid input = {dict}
        '''
        
        Model.modcounters['num_of_instances'] += 1
        Model.sdir = sdir
        
        if config_mod is not None:
            print 'Loading mod settings from config file: %s' % (sdir+'config/'+config_mod)
            Model.modsettings = Model.config_read(sdir + 'config/' + config_mod, 'modsettings')
            Model.modparams = Model.config_read(sdir + 'config/' + config_mod, 'modparams')
            Model.fireparams = Model.config_read(sdir + 'config/' + config_mod, 'fireparams')
        
        if config_init is not None:
            print 'Loading init settings from config file: %s' % (sdir + 'config/' + config_init)
            configinit = Model.config_read(sdir + 'config/' + config_init, 'initsettings')
            if mode is None:        mode = configinit['mode']
            if resolution is None:  resolution = configinit['resolution']
            if tile is None:        tile = configinit['tile']
            if firemode is None:    firemode = configinit['firemode']
            if sampleloc is None:   sampleloc = configinit['sampleloc']
            if sampleloc == 'None': sampleloc = None
        
        if mode is None:            mode = 1  # assign default values
        if resolution is None:      resolution = 0.25
        if tile is None:            tile = 'global'
        if firemode is None:        firemode = 'on'
        if act is None:             act = 'active'
        
        # update init settings dictionary
        self.initsettings = OrderedDict([
            ('sdir', sdir),
            ('mode', mode),
            ('resolution', resolution),
            ('firemode', firemode),
            ('tile', tile),
            ('sampleloc', sampleloc),
            ('s_pixelmode', s_pixelmode),
            ('config_mod', config_mod),
            ('config_init', config_init),
        ])
        
        if not os.path.exists(sdir):  # make model instance directory
            os.makedirs(sdir)
            os.makedirs(sdir + 'config/')
            os.makedirs(sdir + 'data/')
            os.makedirs(sdir + 'figures/')
            os.makedirs(sdir + 'logs/')
        
        if act == 'active':
            
            # write config files:
            Model.config_write(sdir + 'config/config_mod', {'modsettings': Model.modsettings, 'modparams': Model.modparams, 'fireparams': Model.fireparams})
            filename_init = 'config-init%s-m%s-%s' % (Model.modcounters['num_of_instances'], mode, tile)
            if sampleloc != None:
                filename_init = filename_init + '_s%08.4f-%08.4f-%s' % (abs(sampleloc[0][0]), abs(sampleloc[0][1]), int(sampleloc[1]))
            Model.config_write(sdir + 'config/'+filename_init, {'initsettings': self.initsettings})
            
            print 'Current number of instances: %s' % Model.modcounters['num_of_instances']
            print 'Class called from script: %s' % os.path.abspath(sys.modules['__main__'].__file__)
            print 'Model instance unpacking in directory: %s' % sdir
        
        print 'Class -' + self.__class__.__name__ + '- initializing for mode = %s, res = %s, tile = %s, firemode = %s, act = %s ...' % (mode, resolution, tile, firemode, act),
        
        self.mode = mode  # assign variables to instance
        self.resolution = resolution
        self.tile = tile
        self.firemode = firemode
        self.act = act
        self.sampleloc = sampleloc
        self.mbf = Model.modsettings['MBF']
        self.yrspin = Model.modsettings['yrspin']
        if self.resolution == 500:      
            self.res = '500m'
        else:
            self.res = '%03.0fdeg' % float(('%0.2f' % round(self.resolution, 2)).lstrip('0.'))  # converts e.g. 0.25 -> '025deg'
        
        self.settling = Model.fireparams['settling']
        self.cc_leaf = Model.fireparams['cc_leaf']
        self.cc_stem = Model.fireparams['cc_stem']
        self.cc_litt = Model.fireparams['cc_litt']
        self.cc_cwd = Model.fireparams['cc_cwd']
        self.cc_soil = Model.fireparams['cc_soil']
        
        # create list of years to compute, based on model settings
        Model.years = Model.allyears[Model.modsettings['start']: Model.modsettings['start'] + Model.modsettings['year_savg'] + Model.modsettings['execut']]
        if Model.modsettings['fastload'] == 'on' and self.mode == 1:  # only loads spinup years.
            self.years = Model.years[: Model.modsettings['year_savg']]
        else:
            ''' Choose here: include or exclude spinup years in mode 2'''
            self.years = Model.years[: Model.modsettings['year_savg'] + Model.modsettings['execut']]  # include
            # self.years = Model.years[Model.modsettings['year_savg'] : Model.modsettings['year_savg'] + Model.modsettings['execut']]    # exclude
        
        if self.mode == 1:
            self.n_years = Model.modsettings['spinup']
            self.times = np.arange(0, self.n_years, 1 / 12.0)  # create time vector for plotting
        elif self.mode == 2:
            self.n_years = len(self.years)
            #self.times = np.arange(int(self.years[0]), int(self.years[-1]) + 1, 1 / 12.0)
            dates = [datetime.datetime(int(self.years[0]), 1, 1) + n * relativedelta(months=1) for n in range(self.n_years * 12)]
            self.times = netcdf.date2num(dates, units='days since 1900-01-01 00:00:0.0', calendar='standard')
            
        self.loop = np.arange(0, self.n_years)
        
        # run main_tile script for spatial indexing
        Model.main_tile(self)
        
        # load climatology datasets or spinup pools, dependent on mode.
        if self.act == 'active':
           
            if self.mode == 1:
                
                Model.main_read(self, self.years, 'clima')
                
                if self.sampleloc != None:
                    self.pixelmode()
                
                self.mapping()
                
                self.setup_vars(0)
                
            elif self.mode == 2:
                
                pass
                # self.load_spinup_pools(s_pixelmode=s_pixelmode)
                #self.load_spinup_pools(path=Model.wdir+'results/'+'run8_500m_Africa_fire-on-off_speedspin-off_new-abiotic/data/'        
        
        if self.yrspin == 'on':
            for key in Model.modparams.keys():
                if 'turnover' in key:
                    Model.modparams[key] = 1 - (1 - Model.modparams[key]) ** 12     # convert to annual turnover rates.
        
        print 'Initialization Done.'
    
    
    def pixelmode(self):
        ''' Reduce data to one pixel with surrounding border b '''
        
        b = self.sampleloc[1]
        self.pixel_index2d = np.s_[self.sampley - b: self.sampley + b+1, self.samplex - b: self.samplex + b+1]
        self.pixel_index3d = np.s_[:, self.sampley - b: self.sampley + b + 1, self.samplex - b: self.samplex + b + 1]
        
        self.Fpar = self.Fpar[self.pixel_index3d]
        self.BA = self.BA[self.pixel_index3d]
        self.FTC = self.FTC[self.pixel_index3d]
        self.NTV = self.NTV[self.pixel_index3d]
        self.LC = self.LC[self.pixel_index3d]
        self.LWMASK = self.LWMASK[self.pixel_index2d]
        self.fire_month = self.fire_month[self.pixel_index2d]
        if type(self.resolution) == float and self.area.shape[0] > 1+b*2 and self.area.shape[1] > 1+b*2:
            self.area = self.area[self.pixel_index2d]

        # if self.resolution != 500:
        #     for biome in self.BAperc.keys():
        #         self.LCperc[biome] = self.LCperc[biome][self.pixel_index3d]
        #         self.BAperc[biome] = self.BAperc[biome][self.pixel_index3d]
        #         self.FTCperc[biome] = self.FTCperc[biome][self.pixel_index3d]
        #         self.NTVperc[biome] = self.NTVperc[biome][self.pixel_index3d]
        #         self.Fparperc[biome] = self.Fparperc[biome][self.pixel_index3d]
    
    
    def mapping(self):
        
        if self.resolution == 500:
            self.AIRT = self.AIRT[:, self.ilatc_ar, self.ilonc_ar]  # assign 0.25 deg value to modis pixel
            self.SSR = self.SSR[:, self.ilatc_ar, self.ilonc_ar]
            self.SOILM = self.SOILM[:, self.ilatc_ar, self.ilonc_ar]
            self.turnover_slow_map = self.turnover_slow_map[self.ilatc_ar, self.ilonc_ar]
        
        elif self.resolution not in [500, 0.25]:
            zoom = 0.25 / self.resolution
            self.AIRT = scipy.ndimage.zoom(self.AIRT, (1,zoom,zoom), order=1)  # order=1 is bilinear.
            self.SSR = scipy.ndimage.zoom(self.SSR, (1,zoom,zoom), order=1)
            self.SOILM = scipy.ndimage.zoom(self.SOILM, (1,zoom,zoom), order=1)
            self.turnover_slow_map = scipy.ndimage.zoom(self.turnover_slow_map, (zoom,zoom), order=1)
        
        elif self.resolution == 0.25:
            pass
        
        if self.sampleloc != None:
            self.AIRT = self.AIRT[self.pixel_index3d]
            self.SSR = self.SSR[self.pixel_index3d]
            self.SOILM = self.SOILM[self.pixel_index3d]
            self.turnover_slow_map = self.turnover_slow_map[self.pixel_index2d]
    
    
    def load_spinup_pools(self, path=None, s_years=None, s_pixelmode=False, addtxt=None):
        ''' Loading pre-saved spinup pools into model instance
        Can be used to prepare for a 'mode 2' execute run, or for loeading into a 'passive' instance
        :path [optional, default=sdir/data/]: directory to load spinup data from.
        :s_years : number of years of previous spinup.
        :s_pixelmode : if the used spinup is a per-pixel spinup (True) or a full map spinup (False).
        '''
        
        dt = Model.modsettings['float_dtype']
        if path is None: path = Model.sdir + 'data/'
        if (addtxt is not None) and (addtxt != ''): addtxt = '_' + str(addtxt)
        if addtxt is None: addtxt = ''
        
        time0 = timer.time()
        
        if s_years is None:
            filename = 'mode1_' + str(Model.modsettings['spinup']) + 'yr_' + self.res + '_fmode-' + self.firemode + '_' + self.tile
        elif s_years is not None:
            filename = 'mode1_' + str(s_years) + 'yr_' + self.res + '_fmode-' + self.firemode + '_' + self.tile
        
        if (self.sampleloc != None) and (s_pixelmode == True):      # is s_pixelmode == False, the normal full-map spinup is loaded, using the filename defined above.
            # filename = filename + '_s%s-%s-%s' % (self.sampleloc[0][0], self.sampleloc[0][1], self.sampleloc[1])
            filename = filename + '_s%08.4f-%08.4f-%s' % (abs(self.sampleloc[0][0]), abs(self.sampleloc[0][1]), int(self.sampleloc[1]))
        
        filename = filename + addtxt
        
        # extract file extension.
        filematch = [f for f in glob.glob(path + filename + '.*')]
        if len(filematch) == 1:
            ext = os.path.splitext(filematch[0])[1]
        else:
            raise OSError('From user func load(self): File not found or multiple files.')

        print 'Loading spinup pools from file: ' + filename + ext + ' ... ',
        
        
        if ext == '.npz':
            fh = np.load(path + filename + ext)
            spinup_pools = fh['pools_spinup'][()]
        elif ext == '.h5':
            fh = h5py.File(path + filename + ext)
            spinup_pools = fh['pools_spinup'][:]
        elif ext == '.nc':
            fh = netcdf.Dataset(path + filename + ext, 'r', format='NETCDF4')
            spinup_pools = {}
            for key in fh['pools_spinup'].variables.keys():
                if key not in ['time', 'latitude', 'longitude', 'yindex', 'xindex']:
                    spinup_pools[key] = np.squeeze(fh['pools_spinup'][key][:])
        
        
        self.leaf = spinup_pools['leaf'].astype('float'+dt)
        self.stem = spinup_pools['stem'].astype('float'+dt)
        self.rootc = spinup_pools['rootc'].astype('float'+dt)
        self.rootf = spinup_pools['rootf'].astype('float'+dt)
        self.cwd = spinup_pools['cwd'].astype('float'+dt)
        self.soil = spinup_pools['soil'].astype('float'+dt)
        self.slow = spinup_pools['slow'].astype('float'+dt)
        self.gras = spinup_pools['gras'].astype('float'+dt)
        self.litt = spinup_pools['litt'].astype('float'+dt)
        
        
        if (self.sampleloc != None) and (s_pixelmode == False):
            b = self.sampleloc[1]
            self.pixel_index2d = np.s_[self.sampley - b: self.sampley + b+1, self.samplex - b: self.samplex + b+1]
            
            self.leaf = self.leaf[self.pixel_index2d]
            self.stem = self.stem[self.pixel_index2d]
            self.rootc = self.rootc[self.pixel_index2d]
            self.rootf = self.rootf[self.pixel_index2d]
            self.cwd = self.cwd[self.pixel_index2d]
            self.soil = self.soil[self.pixel_index2d]
            self.slow = self.slow[self.pixel_index2d]
            self.gras = self.gras[self.pixel_index2d]
            self.litt = self.litt[self.pixel_index2d]
        
        
        # If s_years, attach resumed diag time-series to the previous part of the spinup.
        if s_years is not None:
            
            self.times = np.arange(0, s_years + self.n_years, 1 / 12.0)  # create time vector for plotting
            
            if ext == '.npz':
                tempload = fh['diag_dict'][()]
            elif ext == '.h5':
                tempload = fh['diag_dict'][:]
            elif ext == '.nc':
                tempload = {}
                for key in fh['diag_dict'].groups.keys():
                    tempload[key] = {}
                    for key_sub in fh.groups['diag_dict'][key].variables.keys():
                        tempload[key][key_sub] = fh['diag_dict'][key][key_sub][:]
            
            self.diagm_pool = tempload['diagm_pool']
            self.diagm_param = tempload['diagm_param']
            self.diagm_out = tempload['diagm_out']
            self.diagt_pool = tempload['diagt_pool']
            self.diagt_param = tempload['diagt_param']
            self.diagt_out = tempload['diagt_out']
            
            if self.firemode == 'on':
                self.diagm_fire = tempload['diagm_fire']
                self.diagt_fire = tempload['diagt_fire']
                
            self.s_years = s_years
        
        print 'Done, duration = %s (h,m,s)' % (timer.strftime("%H:%M:%S", timer.gmtime(timer.time() - time0)))
    
    
    def add_fire(self):
        ''' Adding fire into model instance, switching firemode 'off' to 'on'. Also spinup pools with firemode 'on' are loaded accordingly
        An instance with firemode 'off' can be copyed and switched to fire mode 'on'. This saves us from loading the data multiple times
        '''
        
        print 'Adding fire instance, firemode = on'
        self.firemode = 'on'
        if (self.act == 'active' and self.mode == 2):
            self.load_spinup_pools()      # reload spinup pools, for changing firemode.
            
        Model.modcounters['num_of_instances'] += 1
        
        self.initsettings['firemode'] = self.firemode
        Model.config_write(Model.sdir + 'config/config_init' + str(Model.modcounters['num_of_instances']), {'initsettings': self.initsettings})
        
        self.diagt_fire = {}
        self.diagm_fire = {}
        
        print 'Initialization Done.'
    
    
    def setup_vars(self, step):
        
        '''
        Setup variables annualy.
        '''
        
        dt = Model.modsettings['float_dtype']
        
        if self.mode == 2:
            
            self.year = [self.years[step]]
            Model.main_read(self, self.year, 'normal')
            
            if self.sampleloc != None:
                self.pixelmode()
                
            self.mapping()
        
        
        
        all_percentual = True
        ''' Adjust turnover rates '''
        if (self.mode == 1 and not self.looptest1) \
                or (self.mode == 2):  # run this portion only in the first algorithm iteration (mode 1) or once every year (mode 2).
            
            self.modparams = Model.modparams.copy()
            
            if self.modsettings['biome_method'] == 'skip':
                pass
            
            elif self.modsettings['biome_method'] == 'majority' or self.resolution == 500:
                # Use the majority biome for biome-specific scaling factors.
                
                for paramkey in Model.modparams.keys():
                    if type(Model.modparams[paramkey]) == np.ndarray:
                        self.modparams[paramkey] = Model.modparams[paramkey][self.LC[0, :, :]].astype('float'+dt)
            
            elif self.modsettings['biome_method'] == 'weighted':
                # Use the percentage biome as a weight for the biome-specific scaling factors.
                LC_biomes = []
                for LC_biome in self.LCperc.values():       # dont remove water, because modparams also includes water!
                    LC_biomes.append(LC_biome[0, : , :])
                LC_biomes = np.array(LC_biomes)
                
                for paramkey in Model.modparams.keys():
                    if type(Model.modparams[paramkey]) == np.ndarray:
                        self.modparams[paramkey] = np.sum(LC_biomes * Model.modparams[paramkey][:, None, None], axis=0).astype('float'+dt)
                
                # Only taking the sum because the sum of weights = 1 (sum of all biome percentages).
                
            elif self.modsettings['biome_method'] == 'perbiome':
                if self.biome != None:
                    for paramkey in Model.modparams.keys():
                        if type(Model.modparams[paramkey]) == np.ndarray:
                            if paramkey == 'f_cw':      # NPP has to be applied percentually per biome.
                                self.modparams[paramkey] = Model.modparams[paramkey][int(self.biome)] * self.LCperc[int(self.biome)][0]
                                # TODO: Use of LC causes diff in mode 1!!!
                            else:                   # Instead of f_cw, turnover has to applied everywhere.
                                self.modparams[paramkey] = Model.modparams[paramkey][int(self.biome)]
                            
                            self.modparams[paramkey] = Model.modparams[paramkey][self.LC[0, :, :]].astype('float'+dt) * self.LCperc[int(self.biome)][0]
                    
                    if all_percentual == True:        
                        self.BA = self.BAperc[int(self.biome)]
                        self.Fpar = self.Fparperc[int(self.biome)]
                        self.FTC = self.FTCperc[int(self.biome)]
                        self.NTV = self.NTVperc[int(self.biome)]
                        
                        # self.mortality4 = self.mortality4perc[int(self.biome)] #/ (self.LCperc[int(self.biome)][0])
                        # self.mortality4[np.isnan(self.mortality4)] = 0
                        # self.mortality4[self.mortality4 > 1] = 1
                
            self.looptest1 = 'looped'
        
        
        
        
        self.temp_scalar = Model.modparams['q10'] ** ((self.AIRT - 30) / 10.0)  # Determine environmental scalars. Place before pixelmode above in order to see full arrays of the scalars.
        self.temp_scalar[self.temp_scalar > 1] = 1  # temp and moist scalars both values 0-1
        
        if self.modsettings['soil_moisture'] == 'SWVL1-Land':
            self.moist_scalar = self.SOILM.copy()
            self.moist_scalar[self.moist_scalar > 0.5] = 0.5
            self.moist_scalar = self.moist_scalar / 0.5
            self.moist_scalar = (self.moist_scalar - 0.1) / 0.9
            self.moist_scalar[self.moist_scalar < 0.1] = 0.1
        elif self.modsettings['soil_moisture'] == 'SWVL1':
            self.moist_scalar = self.SOILM.copy()
            self.moist_scalar[self.moist_scalar > 0.37] = 0.37
            self.moist_scalar = self.moist_scalar / 0.37
            self.moist_scalar = (self.moist_scalar - 0.4) / 0.6
            self.moist_scalar[self.moist_scalar < 0.1] = 0.1
        
        self.abiot_scalar = self.temp_scalar * self.moist_scalar
        # FINE-TUNING FOR AFRICA:
        #self.abiot_scalar = np.sqrt(self.abiot_scalar)       # <- new 07-06-2018: np.sqrt. Boosts turnover, to lower biomass.
        if self.modsettings['biome_method'] == 'perbiome':
            if self.biome != None:
                if (int(self.biome) in [9, 7, 10, 11]):
                    self.abiot_scalar = np.sqrt(self.abiot_scalar)  # if per biome, apply root to all gridcells.
        else:
            self.abiot_scalar[:,self.LC[0,:,:] == 9] = np.sqrt(self.abiot_scalar[:,self.LC[0,:,:] == 9])
            self.abiot_scalar[:,self.LC[0,:,:] == 7] = np.sqrt(self.abiot_scalar[:,self.LC[0,:,:] == 7])
            self.abiot_scalar[:,self.LC[0,:,:] == 10] = np.sqrt(self.abiot_scalar[:,self.LC[0,:,:] == 10])
            self.abiot_scalar[:,self.LC[0,:,:] == 11] = np.sqrt(self.abiot_scalar[:,self.LC[0,:,:] == 11])
            #TODO: Use of LC causes diff in mode 1, but only a very small difference.
        
        self.abiot_scalar = self.abiot_scalar / 0.9
        self.abiot_scalar[self.abiot_scalar < 0.1] = 0.1
        self.abiot_scalar[self.abiot_scalar > 1] = 1.0
        
        
        
        FTC_NTV = self.FTC + self.NTV  # fraction total vegetation.
        FTC_NTV[FTC_NTV == 0] = 1  # make all 0 -> 1 in order to avoid divide by zero.   
        self.FTC_NTVr = self.FTC / FTC_NTV  # fraction tree of total vegetation: FTC / (FTC+NTV)
        
        self.NPP = self.SSR * self.Fpar * self.modparams['f_cw']  # * LWMASK          # STEP 1: calculate NPP [gC/m2/month] as the function of solar radiation and FPAR
        # NOTE: In case of Fparperc per biome, it is first divided by LCperc and then again multiplied because f_cw is multiplied by LCperc. This way this code works for both perbiome and perbiome percentual Fpar. 
        self.NPP[self.AIRT < 0] = 0  # set NPP to zero when freezing
        
        
        if self.firemode == 'on':
                        
            self.mortality = self.FTC_NTVr ** 2  # woody vegetation fire mortality: (FTC / (FTC + NTV))^2 (Werf et al. 2003)
            self.mortality[self.mortality == 1] = 0                  
            
            self.ccs_leaf = self.cc_leaf[0] + (1 - self.moist_scalar) * (self.cc_leaf[1] - self.cc_leaf[0])  # compose CC as linearly scaled by soil moisture
            self.ccs_stem = self.cc_stem[0] + (1 - self.moist_scalar) * (self.cc_stem[1] - self.cc_stem[0])
            self.ccs_litt = self.cc_litt[0] + (1 - self.moist_scalar) * (self.cc_litt[1] - self.cc_litt[0])
            self.ccs_cwd = self.cc_cwd[0] + (1 - self.moist_scalar) * (self.cc_cwd[1] - self.cc_cwd[0])
            
        
        if Model.modsettings['herbivory'] == 'on':
            with warnings.catch_warnings():
                warnings.simplefilter('ignore')
                self.NFP = FTC_NTVr * self.NPP * 1 / 3.0 + (1-FTC_NTVr) * self.NPP * 1 / 2.0  # Foliar NPP, in [gC/m2/month]
                self.NFP2 = (self.NFP / 0.3) * 1000 * 12 / 100  # [gC/m2/month] / 0.3 [gC/MJ/month] = [MJ/m2/month] -> * 1000 * 12 = [kJ/m2/year]
                self.HC2 = 10 ** (2.04 * np.log10(self.NFP2) - 4.8)
                self.HC = (self.HC2 / (1000 * 12)) * 0.3 * 100  # back to [gC/m2/month]
        elif Model.modsettings['herbivory'] == 'off':
            self.HC = np.zeros(12)  # herbivory off.
    
    
    def algorithm(self, step, month):

        ''' Main carbon cycle algorithm. This is where all the calculations happen
        Should be called in a loop, iterating through timesteps

        Parameters
        ----------
        :step: year step of loop. Valid input: int
        :month: month step of loop. Valid input: int
        '''
        
        ''' pools: leaf, stem, root, grass, litt, cwd, soil, slow'''
        
        if self.firemode == 'on':
            area_fire = self.BA[month].copy()
        elif self.firemode == 'off':
            area_fire = 0
        
        FTC_NTVr = self.FTC_NTVr[0]
        
        ''' NPP DISTRIBUTION '''
        self.leaf += FTC_NTVr * self.NPP[month] * (1 / 3.0)  # STEP 2: deliver NPP to live biomass
        self.stem += FTC_NTVr * self.NPP[month] * (1 / 3.0) * 0.8  # 20% of stem to root: underground woody roots.
        self.gras += (1-FTC_NTVr) * self.NPP[month] * (1 / 2.0)
        self.rootf += FTC_NTVr * self.NPP[month] * (1/3.0) + (1-FTC_NTVr) * self.NPP[month] * (1/2.0)
        self.rootc += FTC_NTVr * self.NPP[month] * (1/3.0) * 0.2
        
        ''' PRIMARY NATURAL CYCLE '''
        self.gras_herb = self.HC[month] * (1 - area_fire)  # herbivory in non-fire area.
        self.gras -= self.gras_herb
        self.gras_turn = self.gras * self.modparams['turnover_gras']  # * (1-area_fire)   # non-fire live biomass losses, by natural cycling.
        self.leaf_turn = self.leaf * self.modparams['turnover_leaf']  # * (1-area_fire)   # in GFED natural turnover is always happening, also in fire.
        self.stem_turn = self.stem * self.modparams['turnover_stem']  # * (1-area_fire)
        self.rootf_turn = self.rootf * Model.modparams['turnover_root']
        self.rootc_turn = self.rootc * ((1 /  30.0) / 12)  # Gill et al. (2000)
        
        self.herbivory = self.gras_herb * Model.modparams['f_hc']
        
        self.litt += self.leaf_turn + self.gras_turn + self.gras_herb * (1 - Model.modparams['f_hc'])  # dead leafs and grass litter, and uneaten grazing.
        self.cwd += self.stem_turn  # dead stem due to turnover becomes cwd.
        self.soil += self.rootf_turn + self.rootc_turn
        
        ''' FIRE '''  # _fire means fire product, and assumes all is released into atmosphere (combusted)!
        if self.firemode == 'on':
            
            self.gras_fire = self.gras * self.ccs_leaf[month] * area_fire
            self.leaf_fire = self.leaf * self.ccs_leaf[month] * area_fire  # '* self.mortality' removed. Otherwise leaf-fire is often larger than leaf_nofire, because leaf_turn becomes smaller faster than leaf_fire and leaf_mort become larger.
            self.litt_fire = self.litt * self.ccs_litt[month] * area_fire
            self.stem_fire = self.stem * self.ccs_stem[month] * area_fire * self.mortality[0]
            self.cwd_fire = self.cwd * self.ccs_cwd[month] * area_fire
            # TODO: include soil fires (not necessary for Africa).
            
            self.AGB_fire = self.leaf_fire + self.stem_fire + self.gras_fire
            self.LIT_fire = self.cwd_fire + self.litt_fire
            self.tot_fire = self.AGB_fire + self.LIT_fire           # fire event product
            #self.cum_fire += self.tot_fire  # cumulative fire products
            
            self.gras_mort = self.gras * (1 - self.ccs_leaf[month]) * area_fire  # Guido: all gras not combusted goes to litter, i.e. nothing stays alive after fire.
            self.leaf_mort = self.leaf * (1 - self.ccs_leaf[month]) * area_fire * self.mortality[0]  # Guido: just as stem, not combusted part is divided in part to litter and part untouched.
            self.stem_mort = self.stem * (1 - self.ccs_stem[month]) * area_fire * self.mortality[0] 
            self.rootf_mort = self.rootf * FTC_NTVr * area_fire * self.mortality[0]     # roots of fire-induced dead trees also die, and go to soil.
            self.rootc_mort = self.rootc * area_fire * self.mortality[0]
            
            self.litt += self.gras_mort + self.leaf_mort  # dead gras and leaf due to fire becomes litt.
            self.soil += self.rootf_mort + self.rootc_mort  # dead root due to dying tree becomes soil.
            self.cwd += self.stem_mort  # dead stem due to fire becomes cwd.
        
        
        ''' SECONDARY NATURAL CYCLE '''
        self.litt_turn = self.litt * self.modparams['turnover_litt'] * self.abiot_scalar[month]
        self.cwd_turn = self.cwd * self.modparams['turnover_cwd'] * self.abiot_scalar[month]  # decomposition in soil and debris from dead carbon pools.
        self.soil_turn = self.soil * Model.modparams['turnover_soil'] * self.abiot_scalar[month]
        self.slow_turn = self.slow * Model.modparams['turnover_slow'] * self.abiot_scalar[month]
        
        ''' LOSSES '''
        self.gras_out = self.gras_turn + self.gras_fire + self.gras_mort + self.gras_herb  # total biomass removal, natural+fire
        self.leaf_out = self.leaf_turn + self.leaf_fire + self.leaf_mort  # HERBIVORY ALREADY REMOVED FROM GRAS POOL!!!
        self.stem_out = self.stem_turn + self.stem_fire + self.stem_mort  ### Commenting-out '+ self.stem_mort' here, hugely raises stem in fire regions.
        
        self.rootf_out = self.rootf_turn + self.rootf_mort
        self.rootc_out = self.rootc_turn + self.rootc_mort
        self.cwd_out = self.cwd_turn + self.cwd_fire
        self.litt_out = self.litt_turn + self.litt_fire
        
        self.gras -= (self.gras_out - self.gras_herb)  # remaining live biomass # DONT REMOVE HERBIVORY FROM GRAS POOL, already happened.
        self.leaf -= self.leaf_out
        self.stem -= self.stem_out
        self.rootf -= self.rootf_out
        self.rootc -= self.rootc_out
        
        #self.root = self.rootf + self.rootc
        self.root_mort = self.rootf_mort + self.rootc_mort
        self.root_out = self.rootf_out + self.rootc_out
        
        
        ''' RESPIRATION '''
        self.resp_cwd = self.cwd_turn * Model.modparams['f_resp']  # amount of respiration from decomposition.
        self.resp_litt = self.litt_turn * Model.modparams['f_resp']
        self.resp_soil = self.soil_turn * Model.modparams['f_resp']
        self.resp_slow = self.slow_turn * Model.modparams['f_resp']
        
        self.tot_resp = self.resp_cwd + self.resp_litt + self.resp_soil + self.resp_slow  # total respiration from decomposition.
        
        self.litt += self.cwd_turn * (1 - Model.modparams['f_resp'])  # cwd that moves to litter.
        self.soil += self.litt_turn * (1 - Model.modparams['f_resp'])  # litter that moves to soil.
        self.slow += self.soil_turn * (1 - Model.modparams['f_resp'])  # soil that moves to slow decomposition state.
        self.slow += self.slow_turn * (1 - Model.modparams['f_resp'])  # slow not respirated remains in slow.
        
        self.soil_out = self.soil_turn.copy()
        self.slow_out = self.slow_turn.copy()
        
        self.cwd -= self.cwd_out  # remaining dead carbon pools
        self.litt -= self.litt_out
        self.soil -= self.soil_out
        self.slow -= self.slow_out
        
        self.brut_out1 = self.gras_out + self.leaf_out + self.stem_out + self.root_out  # bruto alive out
        self.brut_out2 = self.cwd_out + self.litt_out + self.soil_out + self.slow_out  # bruto dead out
        
        self.net_out = self.tot_resp + self.tot_fire + self.herbivory
        
        ''' 2D instance variable current time step, to pass to diagnose '''
        self.SSR_2d = self.SSR[month].copy()
        self.AIRT_2d = self.AIRT[month].copy()
        self.SOILM_2d = self.SOILM[month].copy()
        self.turnover_slow_map_2d = self.turnover_slow_map.copy()
        
        self.NPP_2d = self.NPP[month].copy()    # These 4 are extra in new method, see also in diagnose().
        self.temp_scalar_2d = self.temp_scalar[month].copy()
        self.moist_scalar_2d = self.moist_scalar[month].copy()
        self.abiot_scalar_2d = self.abiot_scalar[month].copy()
        
        self.Fpar_2d = self.Fpar[month].copy()
        self.FTC_2d = self.FTC[0].copy()
        self.BA_2d = self.BA[month].copy()
    
    
    def construct_scalars(self):
        
        ''' plot scalars
        T = np.arange(0, 45, 1)
        temp_series = map(lambda x: Model.modparams['q10'] ** ((x - 30) / 10.0), T)
        temp_series2= map(lambda x: np.exp(3.36 * (x - 40) / (x + 31.79)), T)

        M = np.arange(0, 0.5, 0.01)
        def moist_func(x):
            if x < 0.1: y = 0.1
            else: y = x / 0.5 - 0.1
            return y
        moist_series = map(moist_func, M)
        '''
        
        AIRT = Model.readdata_CDO('AIRT', 0.25, self.allyears, self.tile, indexdeg=self.index025deg)
        SOILM = Model.readdata_CDO('SWVL1', 0.25, self.allyears, self.tile, indexdeg=self.index025deg)
        LC = Model.readdata_MODIS('LC2', 0.25, self.allyears[len(self.allyears)/2], self.tile, indexdeg=self.index025deg)
        
        temp_scalar = Model.modparams['q10'] ** ((AIRT - 30) / 10.0)  # Determine environmental scalars. Place before pixelmode above in order to see full arrays of the scalars.
        temp_scalar[temp_scalar > 1] = 1  # temp and moist scalars both values 0-1
        
        # self.temp_scalar = np.exp(3.36 * (self.AIRT_2d - 40) / (self.AIRT_2d + 31.79))
        # self.temp_scalar[self.temp_scalar > 1] = 1
        # self.temp_scalar[self.AIRT_2d < -31] = 0

        # self.moist_scalar_old = 0.1 + SOILM * 0.9                 # old method
        # self.moist_scalar_old[self.moist_scalar_old > 0.5] = 0.5
        # self.moist_scalar_olds = self.moist_scalar_old / self.moisture_max  # standardize to 0-1 (with 0 as minimum implicitly). Based on climatology (mode 1) or full year (mode 2) maximum
        # self.abiot_scalar_old = np.sqrt(self.temp_scalar * self.moist_scalar_olds)  # value for amount of warmth and moist (driving decomposition)

        # self.moist_scalar = self.SOILM_2d.copy()
        # self.moist_scalar[self.moist_scalar > 0.5] = 0.5
        # self.moist_scalar = self.moist_scalar / 0.5
        # self.moist_scalar[self.moist_scalar < 0.1] = 0.1
        
        if self.modsettings['soil_moisture'] == 'SWVL1-Land':
            moist_scalar = SOILM.copy()
            moist_scalar[moist_scalar > 0.5] = 0.5
            moist_scalar = moist_scalar / 0.5
            moist_scalar = (moist_scalar - 0.1) / 0.9
            moist_scalar[moist_scalar < 0.1] = 0.1
        elif self.modsettings['soil_moisture'] == 'SWVL1':
            moist_scalar = SOILM.copy()
            moist_scalar[moist_scalar > 0.37] = 0.37
            moist_scalar = moist_scalar / 0.37
            moist_scalar = (moist_scalar - 0.4) / 0.6
            moist_scalar[moist_scalar < 0.1] = 0.1
        
        # self.abiot_scalar = np.sqrt(self.temp_scalar * self.moist_scalar)
        # self.abiot_scalar = (self.abiot_scalar - 0.2) / 0.8
        # self.abiot_scalar[self.abiot_scalar < 0.1] = 0.1
        
        abiot_scalar = temp_scalar * moist_scalar
        
        # abiot_scalar[:,LC[0, :, :] == 9] = np.sqrt(abiot_scalar[:,LC[0, :, :] == 9])
        # abiot_scalar[:,LC[0, :, :] == 7] = np.sqrt(abiot_scalar[:,LC[0, :, :] == 7])
        # abiot_scalar[:,LC[0, :, :] == 10] = np.sqrt(abiot_scalar[:,LC[0, :, :] == 10])
        # abiot_scalar[:,LC[0, :, :] == 11] = np.sqrt(abiot_scalar[:,LC[0, :, :] == 11])
        
        abiot_scalar = abiot_scalar / 0.9
        abiot_scalar[abiot_scalar < 0.1] = 0.1
        abiot_scalar[abiot_scalar > 1] = 1.0
        
        for paramkey in Model.modparams.keys():
            if type(Model.modparams[paramkey]) == np.ndarray:
                self.modparams[paramkey] = Model.modparams[paramkey][LC[0, :, :]].astype('float32')
        
        litt_eturn = abiot_scalar * self.modparams['turnover_litt']
        cwd_eturn = abiot_scalar * self.modparams['turnover_cwd']
        
        leaf_eturn = self.modparams['turnover_leaf']
        stem_eturn = self.modparams['turnover_stem']
        
        return moist_scalar, abiot_scalar, litt_eturn, cwd_eturn, leaf_eturn, stem_eturn
    
    
    def diagnose_alternative(self, step, month):
        land_area = self.area * self.LWMASK
        land_area_tot = np.nansum(land_area)
        
        length = self.n_years * 12
        i = step*12 + month
        
        for field in ['leaf', 'stem', 'root', 'cwd', 'soil', 'slow', 'litt', 'gras']:   # previously diagn
            self.diagt_pool.setdefault(field, np.zeros(length))[i] = np.nansum(eval('self.' + field) * land_area)
            self.diagm_pool.setdefault(field, np.zeros(length))[i] = self.diagt_pool[field][-1] / land_area_tot
        
        for field in ['leaf_out', 'stem_out', 'root_out', 'cwd_out', 'soil_out', 'slow_out', 'net_out']:    # previously pools_out
            self.diagt_out.setdefault(field, np.zeros(length))[i] = np.nansum(eval('self.' + field) * land_area)
            self.diagm_out.setdefault(field, np.zeros(length))[i] = self.diagt_out[field][-1] / land_area_tot
        
        for field in ['SSR', 'AIRT', 'SOILM', 'Fpar', 'FTC', 'NPP', 'temp_scalar', 'moist_scalar', 'tot_resp']:  # previously params
            if field in ['SSR', 'AIRT', 'SOILM', 'Fpar', 'FTC']:    eval_str = '_2d'
            else:   eval_str = ''
            self.diagt_param.setdefault(field, np.zeros(length))[i] = np.nansum(eval('self.' + field + eval_str) * land_area)
            self.diagm_param.setdefault(field, np.zeros(length))[i] = self.diagt_param[field][-1] / land_area_tot
        
        if self.firemode == 'on':
            
            for field in ['leaf_fire', 'stem_fire', 'cwd_fire', 'tot_fire', 'cum_fire', 'BA', 'stem_mort']: # previously diagn_fire
                if field in ['BA']:     eval_str = '_2d'
                else: eval_str = ''
                self.diagt_fire.setdefault(field, np.zeros(length))[i] = np.nansum(eval('self.' + field + eval_str) * land_area)
                self.diagm_fire.setdefault(field, np.zeros(length))[i] = self.diagt_fire[field][-1] / land_area_tot
        
        
        if self.mode == 1:
            
            # self.full_fields_mode1 = []
            self.full_fields_mode1 = ['leaf', 'stem', 'root', 'cwd', 'soil', 'slow']
            if step == self.n_years - 1:  # save data for the last year (12 months) of spinup.
                for field in self.full_fields_mode1:
                    self.diagf.setdefault(field, []).append(eval('self.' + field).copy())
        
        elif self.mode == 2:
            
            self.full_fields_mode2 = ['NPP']
            for field in self.full_fields_mode2:
                self.diagf.setdefault(field, []).append(eval('self.' + field).copy())
    
    
    def diagnose(self, step, month, diagf='full'):
        ''' Creates diagnostics from the main algorithm results.
        Should be called in the same loop as, and after, algorithm()
        
        Parameters
        ----------
        :step: year step of loop. Valid input: int
        :month: month step of loop. Valid input: int
        '''
        #warnings.simplefilter('error')
        land_area = (self.area * self.LWMASK)                         # if res==500m, 1 value: 463.2 ** 2, otherwise array.
        land_area_tot = np.nansum(land_area)
        
        if self.sampleloc != None:
            #TODO: move this to tile or something.
            landtest = list(~np.isnan(land_area).flatten())     # returns boolean True/False list for land/not land.
            if landtest.count(True) == 0:
                warnings.warn('100% of sample location is water!')
            elif  landtest.count(True) < len(landtest)/2.0:
                warnings.warn('More than 50% of sample location is water!')
        
        # for field in ['NPP', 'respiration', 'leaf', 'stem', 'root', 'cwd', 'soil', 'slow']:
        #     self.diagn[field] = np.append(self.diagn.setdefault(field, []), np.nansum(eval('self.' + field) * area))
        # 
        # for field in ['leaf', 'stem', 'root', 'cwd', 'soil', 'slow']:
        #     self.pools[field] = np.append(self.pools.setdefault(field, []), np.nanmean(eval('self.' + field + str([avg_index2d]))))
        # 
        # for field in ['leaf_out', 'stem_out', 'root_out', 'cwd_out', 'soil_out', 'slow_out']:
        #     self.pools_out[field] = np.append(self.pools_out.setdefault(field, []), np.nanmean(eval('self.' + field + str([avg_index2d]))))
        # 
        # for field in ['temperature_scalar', 'moisture_scalar', 'Fpar', 'SSR', 'AIRT', 'SOILM', 'FTC', 'NPP']:
        #     if ('scalar' or 'NPP') in field:
        #         self.params[field] = np.append(self.params.setdefault(field, []), np.nanmean(eval('self.' + field + str([avg_index2d]))))
        #     elif field in ['SSR', 'AIRT', 'SOILM']:
        #         self.params[field] = np.append(self.params.setdefault(field, []), np.nanmean(eval('self.' + field + '_mod' + str([avg_index2d]))))
        #     elif field in ['Fpar']:
        #         self.params[field] = np.append(self.params.setdefault(field, []), np.nanmean(eval('self.' + field + str([avg_index3d]))))
        #     elif field == 'FTC':
        #         self.params[field] = np.append(self.params.setdefault(field, []), np.nanmean(eval('self.' + field + str([avg_index3d0]))))
        # 
        # self.diagn['meta']     = {'name': 'diagnostics', 'avg': 'global / tile total'}
        # self.pools['meta']     = {'name': 'pools', 'avg': 'local'}
        # self.pools_out['meta'] = {'name': 'pools_out', 'avg': 'local'}
        # self.params['meta']    = {'name': 'params', 'avg': 'local'}
        # 
        # # Fire diagnostics
        # if self.firemode == 'on':
        #      
        #     for field in ['leaf_fire', 'stem_fire', 'cwd_fire', 'fire', 'atmos', 'BA', 'stem_mort']:
        #         if field in ['leaf_fire', 'stem_fire', 'cwd_fire', 'fire', 'atmos', 'stem_mort']:
        #             self.diagn_fire[field] = np.append(self.diagn_fire.setdefault(field, []), np.nansum(eval('self.' + field) * area))
        #         if field in ['BA']:
        #             self.diagn_fire[field] = np.append(self.diagn_fire.setdefault(field, []), np.nanmean(eval('self.' + field + str([month]))*self.LWMASK))
        #     
        #     for field in ['leaf_fire', 'stem_fire', 'cwd_fire', 'fire', 'atmos','BA', 'stem_mort']:
        #         if field in ['leaf_fire', 'stem_fire', 'cwd_fire', 'fire', 'atmos', 'stem_mort']:
        #             self.pools_fire[field] = np.append(self.pools_fire.setdefault(field, []), np.nanmean(eval('self.' + field + str([avg_index2d]))))
        #         elif field in ['BA']:
        #             self.pools_fire[field] = np.append(self.pools_fire.setdefault(field, []), np.nanmean(eval('self.' + field + str([avg_index3d]))))
        # 
        #     self.diagn_fire['meta'] = {'name': 'diagnostics_fire', 'avg': 'global / tile total'}
        #     self.pools_fire['meta'] = {'name': 'pools_fire', 'avg': 'local'}
        
        '''
        diagt = tile/global/sampleloc sum
        diagm = tile/global/sampleloc mean
        diagf = full 3d array (mode1: for last year only)
        '''
        
        # alternative: x.setdefault(key, []).append(y)      (but slightly slower due to list-array conversion afterwards)        
        
        for field in ['leaf', 'stem', 'rootf', 'rootc', 'cwd', 'soil', 'slow', 'litt', 'gras']: # previously diagn
            self.diagt_pool[field] = np.append(self.diagt_pool.setdefault(field, []), np.nansum(eval('self.'+field) * land_area))
            self.diagm_pool[field] = np.append(self.diagm_pool.setdefault(field, []), self.diagt_pool[field][-1] / land_area_tot)

        for field in ['leaf_out', 'stem_out', 'rootf_out', 'rootc_out', 'cwd_out', 'soil_out', 'slow_out', 'net_out']: # previously pools_out
            self.diagt_out[field] = np.append(self.diagt_out.setdefault(field, []), np.nansum(eval('self.'+field) * land_area))
            self.diagm_out[field] = np.append(self.diagm_out.setdefault(field, []), self.diagt_out[field][-1] / land_area_tot)

        for field in ['SSR', 'AIRT', 'SOILM', 'Fpar', 'FTC', 'NPP', 'temp_scalar', 'moist_scalar', 'abiot_scalar', 'tot_resp']:  # previously params
            if field in ['SSR', 'AIRT', 'SOILM', 'Fpar', 'FTC', 'NPP', 'temp_scalar', 'moist_scalar', 'abiot_scalar']:    eval_str = '_2d'
            else: eval_str = ''
            self.diagt_param[field] = np.append(self.diagt_param.setdefault(field, []), np.nansum(eval('self.'+field+eval_str) * land_area))
            self.diagm_param[field] = np.append(self.diagm_param.setdefault(field, []), self.diagt_param[field][-1] / land_area_tot)
        
        if self.firemode == 'on':
            
            for field in ['leaf_fire', 'stem_fire', 'cwd_fire', 'litt_fire', 'gras_fire', 'tot_fire', 'BA', 'leaf_mort', 'gras_mort', 'stem_mort', 'rootf_mort', 'rootc_mort']: # previously diagn_fire
                if field in ['BA']:     eval_str = '_2d'
                else: eval_str = ''
                self.diagt_fire[field] = np.append(self.diagt_fire.setdefault(field, []), np.nansum(eval('self.'+field+eval_str) * land_area))
                self.diagm_fire[field] = np.append(self.diagm_fire.setdefault(field, []), self.diagt_fire[field][-1] / land_area_tot)
        
        
        self.AGB = self.leaf + self.stem + self.gras
        self.LIT = self.cwd + self.litt
        self.AGBw = self.leaf + self.stem
        
        if self.mode == 1:
            if type(self.resolution) == float:
                self.full_fields_mode1 = ['leaf','stem','rootf', 'rootc', 'cwd','soil','slow','litt','gras','NPP','net_out', 'leaf_fire', 'stem_fire', 'gras_fire', 'cwd_fire', 'litt_fire']
            elif self.resolution == 500:
                self.full_fields_mode1 = ['AGB', 'LIT', 'AGB_fire', 'LIT_fire', 'NPP', 'AGBw']
            if self.firemode == 'off':
                self.full_fields_mode1 = [x for x in self.full_fields_mode1 if 'fire' not in x]     # remove fire and mort related variables from list.
                self.full_fields_mode1 = [x for x in self.full_fields_mode1 if 'mort' not in x]
            
            if step == self.n_years-1:                  # save data for the last year (12 months) of spinup.
                for field in self.full_fields_mode1:
                    if field in ['NPP']:    eval_str = '_2d'
                    else: eval_str = ''
                    self.diagf.setdefault(field, []).append(eval('self.'+field+eval_str).copy())     # This copy is necessary, otherwise all list items will have the last month's value.
        
        elif self.mode == 2:
            if self.resolution == 0.25:
                self.full_fields_mode2 = ['leaf','stem','rootf', 'rootc', 'cwd','soil','slow','litt','gras','NPP','net_out', 'AGB_fire', 'LIT_fire']
            elif self.resolution != 0.25:  # includes 500.
                self.full_fields_mode2 = ['AGB', 'LIT', 'AGB_fire', 'LIT_fire', 'NPP', 'AGBw']
            if self.firemode == 'off':
                self.full_fields_mode2 = [x for x in self.full_fields_mode2 if 'fire' not in x]
                self.full_fields_mode2 = [x for x in self.full_fields_mode2 if 'mort' not in x]
            
            for field in self.full_fields_mode2[:]:
                if field in ['NPP']:    eval_str = '_2d'
                else: eval_str = ''
                if diagf == 'full':                 # get full monthly time series of full arrays.
                    self.diagf.setdefault(field, []).append(eval('self.'+field+eval_str).copy())
                    
                elif diagf == 'light':
                    
                    # self.full_fields_mode2.remove(field)
                    # self.full_fields_mode2.append(field+'_sum')
                    # self.full_fields_mode2.append(field+'_avg')
                    # 
                    # # calculate annual sum
                    # self.diagf.setdefault(field+'_sum', [])
                    # if month == 0:
                    #     self.diagf[field+'_sum'].append(0)
                    # self.diagf[field+'_sum'][-1] = self.diagf[field+'_sum'][-1] + eval('self.' + field)
                    # 
                    # # calculate annual mean
                    # self.diagf.setdefault(field+'_avg', [])
                    # if month == 11:
                    #     self.diagf[field+'_avg'].append(self.diagf[field+'_sum'][-1] / 12.0)
                    
                    
                    # calculate annual sum
                    self.diagf.setdefault(field, [[], []])
                    if month == 0:
                        self.diagf[field][0].append(0)
                    self.diagf[field][0][-1] = self.diagf[field][0][-1] + eval('self.'+field+eval_str)

                    # calculate annual mean
                    if month == 11:
                        self.diagf[field][1].append(self.diagf[field][0][-1] / 12.0)
                    
                    
                    #TODO: further develop this in 'load_2dvar'
        
        self.diagf['meta'] = diagf      # diagf setting is 'full' or 'light'
            
            # Store all fields as full array: (very large files of 3 gb per year)
            # for field in ['leaf', 'stem', 'root', 'cwd', 'soil', 'slow']:
            #     self.diagf_pool.setdefault(field, []).append(eval('self.'+field))
            # 
            # for field in ['leaf_out', 'stem_out', 'root_out', 'cwd_out', 'soil_out', 'slow_out']:
            #     self.diagf_out.setdefault(field, []).append(eval('self.'+field))
            # 
            # for field in ['SSR', 'AIRT', 'SOILM', 'Fpar', 'FTC', 'NPP', 'temperature_scalar', 'moisture_scalar', 'respiration']:
            #     if field in ['SSR', 'AIRT', 'SOILM', 'Fpar', 'FTC']: eval_str = '_2d'
            #     else: eval_str = ''
            #     self.diagf_param.setdefault(field, []).append(eval('self.'+field + eval_str))
            # 
            # self.diagf_pool['meta']  = 'pools, spatio-temporal'
            # self.diagf_out['meta']   = 'pools_out, spatio-temporal'
            # self.diagf_param['meta'] = 'parameters, spatio-temporal'
            # 
            # if self.firemode == 'on':
            # 
            #     for field in ['leaf_fire', 'stem_fire', 'cwd_fire', 'fire', 'atmos', 'BA', 'stem_mort']:
            #         if field in ['BA']: eval_str = '_2d'
            #         else: eval_str = ''
            #         self.diagf_fire.setdefault(field, []).append(eval('self.' + field + eval_str))
            # 
            #     self.diagf_fire['meta']  = 'fire pools, spatio-temporal'
    
    
    def save(self, path=None, ext=None, addtxt=None):
        ''' Saves 1: spinup pools, 2: diagnostics time-series, and optionally 3: full fields.
        Should be called after the loop including algorithm() and diagnostics() is finished.
        :path [optional, default=sdir/data/]: alternative directory to save data to.
        :addtxt [optional, default='']: add text flag to filename in order to make it unique
        '''
        
        if path is not None:
            if not os.path.exists(path):
                os.makedirs(path)
        elif path is None: path = Model.sdir + 'data/'
        
        if (addtxt is not None) and (addtxt != ''): addtxt = '_' + str(addtxt)
        if addtxt is None: addtxt = ''
        
        if ext is None: ext = '.npz'
        
        '''
        if Model.modsettings['saves'] == 'on':
            print 'Saving results to %s ... ' %path ,
            self.pools_final = {'leaf':self.leaf, 'stem':self.stem, 'root':self.root, 'cwd':self.cwd, 'soil':self.soil, 'slow':self.slow}
            
            if self.firemode == 'off':
                self.pools_final['meta'] = {'modsettings':Model.modsettings, 'modparams':Model.modparams, 'years':self.years}
                self.diagn_final = {'diagn':self.diagn, 'pools':self.pools, 'pools_out':self.pools_out, 'params':self.params}
            
            elif self.firemode == 'on':
                self.pools_final['meta'] = {'modsettings':Model.modsettings, 'modparams':Model.modparams, 'years':self.years, 'fireparams':Model.fireparams}
                self.diagn_final = {'diagn':self.diagn, 'pools':self.pools, 'pools_out':self.pools_out, 'params':self.params, 'diagn_fire':self.diagn_fire, 'pools_fire':self.pools_fire}
        '''

        print 'Saving results to %s ... ' % path,
                   
        self.diag_dict = {'diagt_pool':self.diagt_pool, 'diagm_pool':self.diagm_pool,
                          'diagt_out':self.diagt_out, 'diagm_out':self.diagm_out,
                          'diagt_param':self.diagt_param, 'diagm_param':self.diagm_param}
        if self.firemode == 'on':
            self.diag_dict['diagt_fire'] = self.diagt_fire
            self.diag_dict['diagm_fire'] = self.diagm_fire
        
        if self.mode == 1:
            self.pools_spinup = {'leaf':self.leaf, 'stem':self.stem, 'rootf':self.rootf, 'rootc':self.rootc, 'cwd':self.cwd, 'soil':self.soil, 'slow':self.slow}
            self.pools_spinup['gras'] = self.gras
            self.pools_spinup['litt'] = self.litt
            for key in self.pools_spinup.keys():
                self.pools_spinup[key] = self.pools_spinup[key].astype('float32')   # change dtype(float32)
                
            self.pools_spinup['meta'] = {'modsettings':Model.modsettings, 'initsettings':self.initsettings, 'modparams':Model.modparams, 'years':self.years}
            
            for field in self.full_fields_mode1:                    # convert list of 2d array to 3d array, and change dtype(float32). Also becomes 3d for only one timestep.
                self.diagf[field] = np.array(self.diagf[field]).astype('float32')
        
        elif self.mode == 2:
            
            for field in self.full_fields_mode2:                    # convert list of 2d array to 3d array, and change dtype(float32). Also becomes 3d for only one timestep.
                if self.diagf['meta'] == 'full':
                    self.diagf[field] = np.array(self.diagf[field]).astype('float32')
                elif self.diagf['meta'] == 'light':
                    self.diagf[field][0] = np.array(self.diagf[field][0]).astype('float32')
                    self.diagf[field][1] = np.array(self.diagf[field][1]).astype('float32')
                
        
        if self.s_years is not None:
            self.n_years = self.s_years + Model.modsettings['spinup']        # in case of resumed spinup, add all spinup years.
        filename = 'mode'+str(self.mode)+'_'+str(self.n_years)+'yr_'+self.res+'_fmode-'+self.firemode+'_'+self.tile
        if self.sampleloc != None:
            #filename = filename + '_s%s-%s-%s' % (self.sampleloc[0][0], self.sampleloc[0][1], self.sampleloc[1])
            filename = filename + '_s%08.4f-%08.4f-%s' % (abs(self.sampleloc[0][0]), abs(self.sampleloc[0][1]), int(self.sampleloc[1]))
        filename = filename + addtxt
        
        if self.mode == 1:
            if ext is '.npz':
                np.savez(path + filename, diag_dict=self.diag_dict, pools_spinup=self.pools_spinup, **self.diagf)
            elif ext == '.nc':
                self.save_netcdf(path, filename, diag_dict=self.diag_dict, pools_spinup=self.pools_spinup, diagf=self.diagf)
            elif ext == '.hdf':
                self.save_hdf(path, filename, diag_dict=self.diag_dict, pools_spinup=self.pools_spinup, diagf=self.diagf)
        elif self.mode == 2:
            if ext is '.npz':
                np.savez(path + filename, diag_dict=self.diag_dict, **self.diagf)
            elif ext == '.nc':
                self.save_netcdf(path, filename, diag_dict=self.diag_dict, diagf=self.diagf)
            elif ext == '.hdf':
                self.save_hdf(path, filename, diag_dict=self.diag_dict, diagf=self.diagf)
        print 'file: %s%s ... ' % (filename, ext) ,
        print 'Done'
    
    
    def save_netcdf(self, path, filename, **kwargs):
        
        '''
        Helper function, called by 'save'
        '''
        
        X = {key: kwargs[key] for key in kwargs.keys()}
        
        if self.resolution == 500:  # define lat lon axes, and projection.
            lat = range(2400)
            lon = range(2400)
            lat_dim = 'yindex'
            lon_dim = 'xindex'
        else:
            _, _, lat, lon = Model.load_degbox(self.resolution, self.tile)
            lat_dim = 'lat'
            lon_dim = 'lon'
        
        dsSRS, dsGEO = Model.load_proj(self.resolution, self.tile, projtype='proj4')
        
        if self.sampleloc != None:
            b = self.sampleloc[1]
            lat = lat[self.sampley - b: self.sampley + b + 1]
            lon = lon[self.samplex - b: self.samplex + b + 1]
        
        nf = netcdf.Dataset(path + filename + '.nc', 'w', format='NETCDF4')
            
        nf.setncattr('Conventions', "CF-1.6")
        nf.spatial_ref = dsSRS
        nf.GeoTransform = dsGEO
        ### NEW ###     # add model settings as meta data in file.
        for key, value in self.initsettings.iteritems():
            nf.setncattr(key, str(value))
        nf.setncatts(Model.modsettings)
        for key, value in Model.modparams.iteritems():
            if 'turnover' in key:
                value = 1 / (value*12.0)  # convert to readable turnover rates.  
            if type(value) is np.ndarray:  # convert to list to maintain [..,..,..] notation in string.
                value = np.around(value, 5).tolist()  # use around combined with tolist to remove abundant zeros due to precision.
            nf.setncattr(key, str(value))
        nf.setncatts(Model.fireparams)
        ###
        
        for group in X.keys():
            g = nf.createGroup(group)
            g.setncattr('Conventions', "CF-1.6")
            
            if group == 'diagf':
                g.setncatts({'diagf': u"%s" % self.diagf['meta'] })
            
            if type(X[group].items()[0][1]) is np.ndarray:  # if group dictionary item is an array (i.e. pools spinup or diagf).
                
                g.createDimension(lat_dim, len(lat))
                g.createDimension(lon_dim, len(lon))
                g.createVariable(lat_dim, 'f4', (lat_dim,))
                g.createVariable(lon_dim, 'f4', (lon_dim,))
                g.variables[lat_dim][:] = tuple(lat)
                g.variables[lon_dim][:] = tuple(lon)
                g.variables[lat_dim].setncatts({'standard_name': u"latitude", 'units': u"degrees_north", 'axis': u"Y"})
                g.variables[lon_dim].setncatts({'standard_name': u"longitude", 'units': u"degrees_east", 'axis': u"X"})
                
                g.createVariable('Projection', 'S1')
                g.createVariable('GeoTransform', 'S1')
                g.variables['Projection'][:] = dsSRS
                g.variables['GeoTransform'][:] = str(dsGEO)
                
                if X[group][X[group].keys()[0]].ndim == 2:  # in case of pools_spinup
                    g.createDimension('time', None)  # time as UNLIMITED dimension.
                    g.createVariable('time', 'f', ('time',))
                    g.variables['time'][:] = None
                elif X[group][X[group].keys()[0]].ndim == 3:  # in case of diagf arrays.
                    g.createDimension('time', None)  # time as UNLIMITED dimension.
                    g.createVariable('time', 'f', ('time',))
                    if self.mode == 1:
                        g.variables['time'][:] = self.times[-12:]  # only last year of diagf is saved for mode 1.
                    elif self.mode == 2:
                        if 'light' in self.diagf['meta']:
                            g.variables['time'][:] = self.times[::12]   # grab yearly dates.
                        else:
                            g.variables['time'][:] = self.times
                
                        g.variables['time'].setncatts({'standard_name': u"time", 'long_name': u"time", \
                                                       'units': u"days since 1900-01-01 00:00:0.0", 'calendar': u"standard", \
                                                       'axis': u"time"})  # these attributes are necessary for Panoply to interpret datetime float values.
                
                for layer in X[group].keys():
                    if layer != 'meta':
                        
                        if X[group][layer].ndim == 2:
                            X[group][layer] = np.expand_dims(X[group][layer], axis=0)  # add time axis to array.
                        
                        g.createVariable(layer, 'f4', ('time', lat_dim, lon_dim), zlib=True, complevel=1)  # f4 = float32
                        g.variables[layer][:] = X[group][layer]
                        g.variables[layer].setncattr('grid_mapping', 'spatial_ref')
            
            
            elif type(X[group].items()[0][1]) is list:  # in case of diagf list of arrays
                
                g.createDimension(lat_dim, len(lat))
                g.createDimension(lon_dim, len(lon))
                g.createVariable(lat_dim, 'f4', (lat_dim,))
                g.createVariable(lon_dim, 'f4', (lon_dim,))
                g.variables[lat_dim][:] = tuple(lat)
                g.variables[lon_dim][:] = tuple(lon)
                g.variables[lat_dim].setncatts({'standard_name': u"latitude", 'units': u"degrees_north", 'axis': u"Y"})
                g.variables[lon_dim].setncatts({'standard_name': u"longitude", 'units': u"degrees_east", 'axis': u"X"})
                
                g.createVariable('Projection', 'S1')
                g.createVariable('GeoTransform', 'S1')
                g.variables['Projection'][:] = dsSRS
                g.variables['GeoTransform'][:] = str(dsGEO)
                
                g.createDimension('time', None)  # time as UNLIMITED dimension.
                g.createVariable('time', 'f', ('time',))
                if self.mode == 1:
                    g.variables['time'][:] = self.times[-12:]  # only last year of diagf is saved for mode 1.
                if self.mode == 2:
                    if 'light' in self.diagf['meta']:
                        g.variables['time'][:] = self.times[::12]  # grab yearly dates.
                    else:
                        g.variables['time'][:] = self.times
                    
                    g.variables['time'].setncatts({'standard_name': u"time", 'long_name': u"time", \
                                                   'units': u"days since 1900-01-01 00:00:0.0", 'calendar': u"standard", \
                                                   'axis': u"time"})  # these attributes are necessary for Panoply to interpret datetime float values.
                
                for layer in X[group].keys():
                    if layer != 'meta':
                        
                        for i in range(len(X[group][layer])):
                            
                            g_i = g.createGroup(str(i))
                            
                            if X[group][layer][i].ndim == 2:
                                X[group][layer][i] = np.expand_dims(X[group][layer][i], axis=0)  # add time axis to array.
                            
                            g_i.createVariable(layer, 'f4', ('time', lat_dim, lon_dim), zlib=True, complevel=1)  # f4 = float32
                            g_i.variables[layer][:] = X[group][layer][i]
                            g_i.variables[layer].setncattr('grid_mapping', 'spatial_ref')
            
            
            elif type(X[group].items()[0][1]) is dict:  # in case of diag_dict
                
                time_len = len(X[group].items()[0][1].items()[0][1])
                g.createDimension('time', None)  # time as UNLIMITED dimension.
                g.createVariable('time', 'f', ('time',))
                g.variables['time'][:] = self.times
                
                if self.mode ==2:
                    g.variables['time'].setncatts({'standard_name': u"time", 'long_name': u"time", \
                                                   'units': u"days since 1900-01-01 00:00:0.0", 'calendar': u"standard", \
                                                   'axis': u"time"})  # these attributes are necessary for Panoply to interpret datetime float values.
                
                for layer in X[group].keys():
                    g_i = g.createGroup(layer)
                    for layer_i in X[group][layer].keys():
                        g_i.createVariable(layer_i, 'f4', ('time'))
                        g_i.variables[layer_i][:] = X[group][layer][layer_i]
            
        nf.close()
    
    
    def save_hdf(self, path, filename, **kwargs):
        
        '''
        Helper function, called by 'save'
        '''
        
        X = {key: kwargs[key] for key in kwargs.keys()}

        if self.resolution == 500:  # define lat lon axes, and projection.
            lat = range(2400)
            lon = range(2400)
            lat_dim = 'yindex'
            lon_dim = 'xindex'
        else:
            _, _, lat, lon = Model.load_degbox(self.resolution, self.tile)
            lat_dim = 'lat'
            lon_dim = 'lon'

        dsSRS, dsGEO = Model.load_proj(self.resolution, self.tile, projtype='proj4')

        if self.sampleloc != None:
            b = self.sampleloc[1]
            lat = lat[self.sampley - b: self.sampley + b + 1]
            lon = lon[self.samplex - b: self.samplex + b + 1]

        time_meta = {'standard_name': u"time", 'long_name': u"time", \
                     'units': u"days since 1900-01-01 00:00:0.0", 'calendar': u"standard", \
                     'axis': u"time"}  # these attributes are necessary for Panoply to interpret datetime float values.
        
        hf = h5py.File(path + filename + '.h5', 'w')
        
        if self.mode == 1:
            times_dim = self.times
        elif self.mode == 2:
            #dates = netcdf.num2date(self.times, units='days since 1900-01-01 00:00:0.0', calendar='standard')
            #times_dim = [daten.strftime("%Y-%m") for daten in dates]
            times_dim = self.times
        
        hf['time'] = times_dim  # monthly time stamps.
        hf['time_yr'] = times_dim[::12]     # annual time stamps.
        hf['timefull'] = times_dim[-12:]    # last year only.
        hf[lat_dim] = lat
        hf[lon_dim] = lon
        
        hf.attrs['spatial_ref'] = dsSRS
        hf.attrs['GeoTransform'] = dsGEO
        ### NEW ###     # add model settings as meta data in file.
        for key, value in self.initsettings.iteritems():
            hf.attrs[key] = str(value)
        for key, valye in Model.modsettings.iteritems():
            hf.attrs[key] = str(value)
        for key, value in Model.modparams.iteritems():
            if 'turnover' in key:
                value = 1 / (value * 12.0)  # convert to readable turnover rates.  
            if type(value) is np.ndarray:  # convert to list to maintain [..,..,..] notation in string.
                value = np.around(value, 5).tolist()  # use around combined with tolist to remove abundant zeros due to precision.
            hf.attrs[key] = str(value)
        for key, value in Model.fireparams.iteritems():
            hf.attrs[key] = str(value)
            
        hf.attrs['time'] = str(time_meta)
        
        Projection = np.string_(''':grid_mapping_name = "sinusoidal";
                                :longitude_of_central_meridian = 0.0; // double
                                :earth_radius = 6371007.181; // double
                                :_CoordinateTransformType = "Projection";
                                :_CoordinateAxisTypes = "GeoX GeoY";''')
        
        for group in X.keys():
            g = hf.create_group(group)
            
            if group == 'diagf':
                g.attrs['diagf'] = self.diagf['meta']
                
            if type(X[group].items()[0][1]) is np.ndarray:  # if group dictionary item is an array (i.e. pools spinup or diagf).

                g.create_dataset('Projection', data=Projection, dtype='S10')
                
                for layer in X[group].keys():
                    
                    if layer != 'meta':
                    
                        #if X[group][layer].ndim == 2:
                        #    X[group][layer] = np.expand_dims(X[group][layer], axis=0)  # add time axis to array.
                        
                        g.create_dataset(layer, data=X[group][layer], dtype='float32', compression='gzip', compression_opts=1)
                        g[layer].dims.create_scale(hf[lat_dim], 'lat')
                        g[layer].dims.create_scale(hf[lon_dim], 'lon')
                        
                        if X[group][layer].ndim == 2:  # in case of pools_spinup
                            g[layer].dims[0].attach_scale(hf[lat_dim])
                            g[layer].dims[1].attach_scale(hf[lon_dim])
                        
                        elif X[group][layer].ndim == 3:     # in case of diagf arrays
                            g[layer].dims[1].attach_scale(hf[lat_dim])
                            g[layer].dims[2].attach_scale(hf[lon_dim])
                            
                            if self.mode == 1:
                                g[layer].dims.create_scale(hf['timefull'], 'time')
                                g[layer].dims[0].attach_scale(hf['timefull'])
                            elif self.mode == 2:
                                if 'light' in self.diagf['meta']:
                                    g[layer].dims.create_scale(hf['time_yr'], 'time')
                                    g[layer].dims[0].attach_scale(hf['time_yr'])
                                else:
                                    g[layer].dims.create_scale(hf['time'], 'time')
                                    g[layer].dims[0].attach_scale(hf['time'])
                                
                            #g[layer].dims[0].label = time_meta['units']
            
            
            elif type(X[group].items()[0][1]) is list:  # in case of diagf list of arrays
                
                for i in range(len(X[group][X[group].keys()[0]])):
                    
                    g_i = g.create_group(str(i))
                    
                    g_i.create_dataset('Projection', data=Projection, dtype='S10')
                    
                    for layer in X[group].keys():
                        if layer != 'meta':
                            
                            g_i.create_dataset(layer, data=X[group][layer][i], dtype='float32', compression='gzip', compression_opts=1)
                            g_i[layer].dims.create_scale(hf[lat_dim], 'lat')
                            g_i[layer].dims.create_scale(hf[lon_dim], 'lon')
                            
                            g_i[layer].dims[1].attach_scale(hf[lat_dim])
                            g_i[layer].dims[2].attach_scale(hf[lon_dim])
                            
                            if self.mode == 1:
                                g_i[layer].dims.create_scale(hf['timefull'], 'time')
                                g_i[layer].dims[0].attach_scale(hf['timefull'])
                            elif self.mode == 2:
                                if 'light' in self.diagf['meta']:
                                    g_i[layer].dims.create_scale(hf['time_yr'], 'time')
                                    g_i[layer].dims[0].attach_scale(hf['time_yr'])
                                else:
                                    g_i[layer].dims.create_scale(hf['time'], 'time')
                                    g_i[layer].dims[0].attach_scale(hf['time'])

                            #g_i[layer].dims[0].label = time_meta['units']
                            #for key, value in time_meta.iteritems():
                            #    g_i['time'].attrs[key] = value
                
                
            elif type(X[group].items()[0][1]) is dict:  # in case of diag_dict
                
                for layer in X[group].keys():
                    g_i = g.create_group(layer)
                    for layer_i in X[group][layer].keys():
                        g_i.create_dataset(layer_i, data=X[group][layer][layer_i], dtype='float32', compression='gzip', compression_opts=1)
                        g_i[layer_i].dims.create_scale(hf['time'], 'time')
                        g_i[layer_i].dims[0].attach_scale(hf['time'])
                        
                        g_i[layer_i].dims[0].label = str(time_meta['units'])
                        # if self.mode == 2:
                        #     for key, value in time_meta.iteritems():
                        #         g_i['time'].attrs[key] = value
                            
        hf.close()
                    
        print 'Done.'
       
    
    def plot_sampleloc(self, *argv):
        ''' Plot a map showing sample point location
        
        Parameters
        ----------
        :*argv: Data background to plot sample location on. Valid input: 2D data array [lat, lon]
        '''
        
        Model.modcounters['plotn'] += 1
        fig = plt.figure(Model.modcounters['plotn'], figsize=(10,10))
        ax = fig.add_subplot(111)
        if not argv:
            if self.act == 'active':
                dat = ax.imshow(self.FTC[0,:,:]);  plt.colorbar(dat)
            elif self.act == 'passive':
                Data = Model.readdata_MODIS(self, 'FTC', ['2002'])
                dat = ax.imshow(Data[0,:,:]);  plt.colorbar(dat)
        elif argv:            dat = ax.imshow(argv[0]); plt.colorbar(dat)
        ax.autoscale(False)                                 # needed to remove white borders
        ax.scatter([self.samplex],[self.sampley], c='white', s=150, edgecolor = 'black')      # mark location of sample
        plt.suptitle(self.tile)
        dat.set_clim(0,1)
        plt.show()
    
    
    def overview(self):
        ''' Gives a summarized overview of all variables in Model instance
        Similar to __dict__, but more compact for a better overview
        '''
        classobj = sorted(set(dir(self)) - set(self.__dict__.keys()))       # all objects - instance objects = class objects

        print 'Functions: \n_________'
        for fun in [c for c in classobj if callable(eval('self.'+c))]:
            print fun
        
        print '\nClass objects: \n_____________'
        for obj in [c for c in classobj if not callable(eval('self.'+c))]:
        #for obj in sorted(set(dir(self)) - set(self.__dict__.keys())):
            print obj
        
        print '\nInstance variables: \n__________________'
        for key, value in sorted(self.__dict__.iteritems()):
            try:
                if value.shape != ():
                    print key+' : '+str(value.shape)            # arrays
                else:
                    print key+' : '+str(value)
            except:
                try:
                    print key+' : '+str(value.keys())       # dictionaries
                except:
                    print key+' : '+str(value)              # others
    
    
    def load(self, path=None, addtxt=None, full='off'):
        ''' Loads previously saved results.
        Particularly for loading data into a 'passive' model instance.
        :path [optional, default=sdir/data/]: alternative directory to save data to.
        :addtxt [optional, default='']: add text flag to filename in order to make it unique
        '''
        
        if path is None: path = Model.sdir + 'data/'
        if (addtxt is not None) and (addtxt != ''): addtxt = '_' + str(addtxt)
        if addtxt is None: addtxt = ''
        
        filename = 'mode'+str(self.mode)+'_'+str(self.n_years)+'yr_'+self.res+'_fmode-'+self.firemode+'_'+self.tile
        if self.sampleloc != None:
            #filename = filename + '_s%s-%s-%s' % (self.sampleloc[0][0], self.sampleloc[0][1], self.sampleloc[1])
            filename = filename + '_s%08.4f-%08.4f-%s' % (abs(self.sampleloc[0][0]), abs(self.sampleloc[0][1]), int(self.sampleloc[1]))
        filename = filename + addtxt
        
        # extract file extension.
        filematch = [f for f in glob.glob(path + filename + '.*')]
        if len(filematch) == 1:
            ext = os.path.splitext(filematch[0])[1]
        else:
            raise OSError('From user func load(self): File not found or multiple files.')
        # filematch = [f for f in os.listdir(path) if f.startswith(filename)]
        # if len(filematch) == 1:
        #     ext = '.' + filematch[0].rsplit('.')[1]
        # else:
        #     print 'File not found or multiple files.'
        
        print 'Loading from file: ' + filename + ext
        
        if ext == '.npz':
            fh = np.load(path + filename + ext)
            diag_dict = fh['diag_dict'][()]
        elif ext == '.h5':
            fh = h5py.File(path + filename + ext)
            diag_dict = fh['diag_dict'][:]
        elif ext == '.nc':
            fh = netcdf.Dataset(path + filename + ext, 'r' , format='NETCDF4')
            diag_dict = {}
            for key in fh['diag_dict'].groups.keys():
                diag_dict[key] = {}
                for key_sub in fh.groups['diag_dict'][key].variables.keys():
                    diag_dict[key][key_sub] = fh['diag_dict'][key][key_sub][:]
        
        self.diag_dict = diag_dict
        self.diagt_pool = diag_dict['diagt_pool']
        self.diagm_pool = diag_dict['diagm_pool']
        
        self.diagt_out = diag_dict['diagt_out']
        self.diagm_out = diag_dict['diagm_out']
        
        self.diagt_param = diag_dict['diagt_param']
        self.diagm_param = diag_dict['diagm_param']
        
        if self.firemode == 'on':
            self.diagt_fire = diag_dict['diagt_fire']
            self.diagm_fire = diag_dict['diagm_fire']
        
        del diag_dict
        
        if self.mode == 1: # and self.act == 'active':
            
            if ext == '.npz':
                self.pools_spinup = fh['pools_spinup'][()]
            elif ext == '.h5':
                self.pools_spinup = fh['pools_spinup'][:]
            elif ext == '.nc':
                self.pools_spinup = {}
                for key in fh['pools_spinup'].variables.keys():
                    if key not in ['time', 'latitude', 'longitude', 'yindex', 'xindex']:
                        self.pools_spinup[key] = fh['pools_spinup'][key][:]
            
            # for key in self.pools_spinup.keys():
            #     if key != 'meta':
            #         self.pools_spinup[key] = self.pools_spinup[key].astype('float64')  # change dtype(float64)
                    #TODO: WHY CHANGE TO FLOAT64 HERE ???!!!
        
        if full != 'off':
            
            if ext == '.npz':
                try: self.diagf['meta'] = fh['meta'][()]
                except: self.diagf['meta'] = 'full'         # if earlier file without meta, diagf was always full.
            elif ext == '.nc':    self.diagf['meta'] = fh['diagf'].getncattr('diagf')
            
            if full == 'all':
                if ext == '.npz':
                    fields = fh.keys()
                    fields.remove('diag_dict')
                    if 'meta' in fields: fields.remove('meta')
                    if self.mode == 1: fields.remove('pools_spinup')
                elif ext == '.nc':
                    if self.diagf['meta'] == 'full':    fields = fh['diagf'].variables.keys()
                    elif self.diagf['meta'] == 'light': fields = fh['diagf']['0'].variables.keys()
                fields = [x for x in fields if x not in [u'GeoTransform', u'Projection', u'time', u'yindex', u'xindex', u'meta']]
                
            elif full != 'all':
                if (type(full) is str) or (type(full) is np.string_): fields = [full]
                else:                                                 fields = full[:]
            
            for field in fields:
                if self.diagf['meta'] == 'full':
                    if ext == '.npz':  self.diagf[field] = fh[field][()] #.astype('float64')  # change dtype(float64)
                    elif ext == '.nc': self.diagf[field] = fh['diagf'][field][:]    #.astype('float64')    
                elif self.diagf['meta'] == 'light':
                    self.diagf[field] = []
                    for i in range(2):
                        if ext == '.npz':   self.diagf[field].append(fh[field][()][i])
                        elif ext == '.nc':  self.diagf[field].append(fh['diagf'][str(i)][field][:])
    
    
    def plot_dict(self, times, dicts, path=None, addtxt=None, dual=None, save='on'):  # input: list of diagnostics dictionaries
        ''' Plots one or several dictionaries composed of result arrays, with every dict key in a subplot
        Different dictionaries with the same keys will be plotted in the same subplot for direct comparison

        Parameters
        ----------
        :times: timeseries for x-axis. Valid input: instance timeseries self.times, or manual timeseries.
        :dicts: dictionaries to plot. Valid input: list of dictionaries.
        :path [optional, default=sdir/data/]: alternative directory to save data to.
        :addtxt [optional, default='']: add text flag to filename in order to make it unique.
        :dual [optional, default=one axis]: combine two axis in one subplot.
        '''
        
        if path is None: path = Model.sdir + 'figures/'
        
        if (addtxt is not None) and (addtxt != ''): addtxt = '_' + str(addtxt)
        if addtxt is None: addtxt = ''

        if dual is not None: ax2 = {}
        
        if type(dicts) is dict:     dicts = [dicts]     # if one dictionary, make it a list.
        
        for dictn in dicts:
            from matplotlib.ticker import ScalarFormatter
            if dictn is dicts[0]:  # this is executed only once per dictionary.
                Model.modcounters['plotn'] += 1
                subplot_rows = len(dictn) - len(dictn) / 2  # determine subplot layout
            dic = dictn.copy()

            fig = plt.figure(Model.modcounters['plotn'], figsize=(10, 10))
            #meta = dic.pop('meta')
            for i, name in enumerate(dic.keys()):  # looping through plot layers.
                
                ax1 = fig.add_subplot(subplot_rows, 2, i + 1)
                ax1.set_title(name, x=0.2, y=1.2)
                if dictn is dicts[0] and dual is not None: 
                    ax2[name] = ax1.twinx().twiny()    # this is executed only once per dictionary.
                times1 = range(len(times) / 12)
                plot1 = Model.amean(dic[name])
                #times1 = times[:]
                #plot1 = dic[name]
                if (np.max(plot1) / 1E12) > 1:
                    plot1 = plot1 / 1E12
                    ax1.set_ylabel('Tg C')
                else:
                    ax1.set_ylabel('g C')
                ax1.plot(times1, plot1, alpha=0.8)                   
                ax1.set_xlabel('Year')
                ax1.ticklabel_format(style='sci', scilimits=(-3,4), useOffset=False)  # remove offset value axis
                ax1.get_yaxis().get_offset_text().set_x(-0.13)
                
                if dual is not None:
                    times2 = times[-12 * dual:]  #range(12 * dual)
                    plot2 = dic[name][-12 * dual:]
                    if (np.max(plot2) / 1E12) > 1:
                        plot2 = plot2 / 1E12
                    ax2[name].plot(times2, plot2, alpha=0.6, linestyle='--')
                    ax2[name].ticklabel_format(style='sci', scilimits=(-3,4))  # remove offset value axis
                    #ax2[name].get_yaxis().get_offset_text().set_x(0.9)
                    
        #plt.suptitle(meta['name'].split('_')[0] + ' - ' + meta['name'])
        #plt.suptitle(meta, x=0.5, y=1)
        plt.tight_layout(rect=[0, 0, 1, 1.2])       # [left, bottom, right, top]
        plt.subplots_adjust(top=0.92)  # make room for fig title (bug in tight_layout)
        
        if save == 'off':
            plt.show()
        if save == 'on':  # save figure
            filename = 'mode%s_%syr_%s_fmode-%s_%s%s_fig%s' % (self.mode, self.n_years, self.res, self.firemode, self.tile, addtxt, Model.modcounters['plotn'])
            plt.savefig(path + filename)
    
    
    @staticmethod
    def config_write(filename, config_dict):
        
        config = ConfigParser()
        cfile = open(filename+'_test.txt', 'w')
        
        for section_name, section in config_dict.iteritems():
            config.add_section(section_name)
            for key, value in section.iteritems():
                if (section_name == 'modparams') and ('turnover' in key):
                    value = 1 / (value*12.0)    # convert to readable turnover rates.
                if type(value) is np.ndarray:  # convert to list to maintain [..,..,..] notation in string.
                    value = np.around(value, 5).tolist()  # use around combined with tolist to remove abundant zeros due to precision.
                config.set(section_name, key, str(value))
        
        config.write(cfile)
        cfile.close()
    
    
    @staticmethod
    def config_read(filename, section):
        
        config = ConfigParser()
        config.read(filename)
        
        dictio = OrderedDict()
        options = config.options(section)
        
        for option in options:
            temp = config.get(section, option)
            if temp[0] == '[' and temp[-1] == ']':  # in case of list input
                listx = []
                # Tip: remove any whitespace character (space, tab, newline, return, formfeed): " ".join(foo.split()). https://stackoverflow.com/questions/1546226/simple-way-to-remove-multiple-spaces-in-a-string
                for x in temp[1:-1].replace(',', ', ').split(', '):  # loop list items by splitting string. add space after comma if not yet the case
                    listx.append(str(x))
                temp = listx[:]
            else:                   # if not already a list, make it a list.
                temp = [temp]
            listy = []
            for x in temp:
                if x.count('.') == 1:
                    y = float(x)  # in case of float
                    if (section == 'modparams') and ('turnover' in option):
                        y = (1 / y) / 12.0      # convert from readable to computable turnover rates.
                elif x.count('.') == 0:
                    try:
                        y = int(x)  # in case of integer
                    except:
                        y = str(x)  # in case of string
                else:
                    raise TypeError('Multiple dots (.) found in string, invalid input')
                listy.append(y)
            if len(listy) == 1: listy = listy[0]
            dictio[str(option)] = listy
            
        return dictio
    
    
    @classmethod
    def latlon_mask(cls, lats, lons, resolution, tile):
        
        if type(resolution) == float:
            lon = np.arange(-180 + resolution / 2.0, 180, resolution)  # cell midpoint indices
            lat = np.arange(90 - resolution / 2.0, -90, -resolution)
            if tile != 'global':
                indexdeg, _, lat, lon = cls.load_degbox(resolution, tile)
        
        elif resolution == 500:
            hdf = SD(cls.wdir_geo + 'MODIS_geoloc_500m_' + tile + '.hdf', SDC.READ)
            lat = hdf.select('Latitude').get()
            lon = hdf.select('Longitude').get()
            hdf = []
        
        lat_arrn = np.zeros((len(lat), len(lon)))
        lon_arrn = np.zeros((len(lat), len(lon)))
        
        if type(resolution) == float:
            lat_arrn[(lat >= lats[0]) & (lat <= lats[1]), :] = 1
            lon_arrn[:, (lon >= lons[0]) & (lon <= lons[1])] = 1
        elif resolution == 500:
            lat_arrn[(lat >= lats[0]) & (lat <= lats[1])] = 1
            lon_arrn[(lon >= lons[0]) & (lon <= lons[1])] = 1
        
        latlon_arrn = lat_arrn * lon_arrn
        
        return latlon_arrn
    
    
    @classmethod
    def region_mask_set(cls, region, resolution):
        region_dict = OrderedDict()
        for tilen in cls.regions[region][1]:
            region_arr = cls.region_mask(region, resolution, tilen)
            region_dict[tilen] = region_arr
        return region_dict
       
    
    @classmethod
    def region_mask(cls, regions, resolution, tile, frac=None):
        
        #TODO: Doesnt work very precise for sub-0.25 degree resolutions. Update masks!
        
        if type(resolution) == float:
            indexdeg, sizedeg, _, _ = cls.load_degbox(resolution, tile)
            GFEDregions = cls.loadGFED(resolution, 'basis_regions', '2010')
        elif resolution == 500:
            index025deg, size025deg, _, _ = cls.load_degbox(0.25, tile)
            GFEDregions = cls.loadGFED(0.25, 'basis_regions', '2010', tile)
        
        def conver(ind):
            return 90 - 0.25 * ind - (0.25/2)  # Dit moet 0.25 blijven!, niet generaliseren naar degrees.
        
        maskGI1 = cls.latlon_mask([conver(125), 90.0], [-60.0, -11.25], resolution, tile)  # Greenland and Iceland.
        maskGI2 = cls.latlon_mask([conver(60), 90.0], [-73.75, -60.0], resolution, tile) # Greenland last corner.
        maskSP = cls.latlon_mask([conver(60), 90.0], [5.0, 35.0], resolution, tile)    # Spitsbergen.
        
        GFEDregions[maskGI1 == 1] = 20      # Exclude Greenland and Iceland from EURO region.
        GFEDregions[maskGI2 == 1] = 20
        GFEDregions[maskSP == 1] = 20       # Exclude Spitsbergen from EURO region.
        
        if (type(regions) is str) or (type(regions) is np.string_): regions = [regions]
        
        region_arr = 0
        for region in regions:
        
            if type(resolution) == float:
                if tile == 'global':
                    region_arrn = np.zeros((int(180/resolution), int(360/resolution)))
                    region_arrn[GFEDregions == cls.regions[region][0]] = 1
                elif tile != 'global':
                    region_arrn = np.zeros((sizedeg['lat'], sizedeg['lon']))
                    region_arrn[GFEDregions[indexdeg['lat'][0]:indexdeg['lat'][-1] + 1, indexdeg['lon'][0]:indexdeg['lon'][-1] + 1] == cls.regions[region][0]] = 1
                    # region_arr = cls.tile_mask(region_arr, tile)
            
            elif resolution == 500:
                region_arrn = np.zeros((2400, 2400))
                ilatc_ar, ilonc_ar = cls.grid_conv(tile, 0.25, index025deg)
                region_arrn[GFEDregions[index025deg['lat'][0]:index025deg['lat'][-1] + 1, index025deg['lon'][0]:index025deg['lon'][-1] + 1][ilatc_ar, ilonc_ar] == cls.regions[region][0]] = 1
            
            LW = cls.readdata_LWMASK('LWM', resolution, tile, frac=frac)
            region_arrn *= LW
            region_arr += region_arrn
        
        region_arr[region_arr == 0] = np.nan
        
        return region_arr
    
    
    @classmethod
    def biome_mask(cls, biomes, resolution, year, tile):
        ''' Usable for everything except full 500m Africa fields, because then the biome_dict becomes too large.
        500m Africa fields are loaded on the spot (e.g. in def biome_calc)
        To make a combined mask of multiple biomes, separate by spaces e.g. ['2', '3 4', '5']
        '''
        if (type(biomes) is str) or (type(biomes) is np.string_): biomes = [biomes]
        LC = cls.readdata_MODIS('LC2', resolution, [str(year)], tile)[0, :, :]
        
        biome_mask = 0
        biome_dict = OrderedDict()
        for biome in biomes:
            
            biome_arr = np.zeros((LC.shape[0], LC.shape[1]))
            if biome == 'total':
                biome_arr[LC > 0] = 1       # all except 0 (water) and -2 (unclassified). Pretty much a land mask.
            elif ' ' in biome:            # if multiple biomes are packed as string, e.g. '2 4 5'.
                subbiomes = biome.split()
                for b in subbiomes: biome_arr[LC == int(b)] = 1    # combined mask of multiple biomes.
            else:
                biome_arr[LC == int(biome)] = 1
            biome_mask += biome_arr
            biome_arr = biome_arr.astype(float)
            biome_arr[biome_arr == 0] = np.nan
            biome_dict[biome] = biome_arr
        
        biome_mask[biome_mask > 1] = 1
        biome_mask = biome_mask.astype(float)
        biome_mask[biome_mask == 0] = np.nan
        return biome_mask, biome_dict
    
    
    @classmethod
    def biome_calc_old(cls, X, biomes, resolution, years, tile, Xref=None):
        ''' Usable for:
        1. 500m,    one tile
        2. 0.25deg, one tile
        3. 500m,    one region (e.g. Africa)
        4. 0.25deg, one region (e.g. Africa)
        5. 0.25deg, global
        NaN values are not counted in average: nanmean() functionality.
        Not for multiple tiles, then use biome_calctile
        '''
        rdir = cls.wdir + 'results/paper_results/'
        tiles_box = cls.tiles_box
        
        if (type(biomes) is str) or (type(biomes) is np.string_): biomes = [biomes]
        
        # load land area
        if (resolution == 500) and (tile in tiles_box.keys()):
            
            tiles = tiles_box[tile]
            lwmask = cls.saver(lambda: cls.tile_paste(lambda tilen: cls.readdata_LWMASK('LWM', 500, tilen), tiles), rdir+'LW500_bypass.tif', save=False)
            lwmask[lwmask == 0] = np.nan
            area, _ = cls.load_area(resolution, tile)
            land_area = lwmask * area
            LC = cls.saver(lambda: cls.tile_paste(lambda tile: cls.readdata_MODIS('LC', 500, years, tile), tiles), rdir + 'LC500_%s_bypass.tif' % years, save=False)
            
        else:
            biome_dict = cls.biome_mask(biomes, resolution, year, tile)[1]
            lwmask = cls.readdata_LWMASK('LWM', resolution, tile, frac='no')
            area, _ = cls.load_area(resolution, tile)
            land_area = lwmask * area
        
        # additional tile masking
        if resolution != 500:
            if tile[::3] == 'hv':       # if tile is a MODIS tile.
                tile_mask = cls.tile_mask(resolution, tile, frac='yes')[0]
                land_area = land_area * tile_mask
            elif tile in tiles_box.keys():
                tiles = tiles_box[tile]
                tile_mask = cls.tile_mask(resolution, tiles, frac='yes', extent=tile)[0]
                land_area = land_area * tile_mask
            # else if tile == 'global': no tile masking needed
        
        if Xref is not None:
            abs = X - Xref
            rel = abs / Xref
            rel, _ = cls.perc_cut(rel * (land_area / area), [2.5, 2.5])
            
            rel_avg = OrderedDict()
            abs_avg = OrderedDict()
            rel_std = OrderedDict()
            abs_std = OrderedDict()
        
        # calculate biome sum and average
        X_sum = OrderedDict()
        X_avg = OrderedDict()
        X_std = OrderedDict()
        X_qua = OrderedDict()
        X_bp = OrderedDict()
        for biome in biomes:
            if resolution == 500:
                print biome
            
            # get biome mask
            if (resolution == 500) and (tile in tiles_box.keys()):
                biome_mask = np.zeros_like(LC)
                if biome == 'total':    biome_mask[LC > 0] = 1  # all except 0 (water) and -2 (unclassified). Pretty much a land mask.
                elif ' ' in biome:      # multiple biomes considered together.
                    subbiomes = biome.split()
                    for b in subbiomes: biome_mask[LC == int(b)] = 1    # combined mask of multiple biomes.
                else:
                    biome_mask[LC == int(biome)] = 1
                biome_mask = biome_mask.astype(float)
                biome_mask[biome_mask == 0] = np.nan
            else:
                biome_mask = biome_dict[biome]
            
            land_area_biome = land_area * biome_mask
            #land_area_biometot = np.nansum(land_area_biome)    # Wrong, remove nans in data first, as below.
            biome_mask = None
            
            '''
            #old method: much slower for large arrays (e.g. 4 times slower for 2400x2400). But more clear.
            X_sum[biome] = np.nansum(X * land_area_biome)       # biome sum (area weighted)
            X_avg[biome] = X_sum[biome] / land_area_biometot    # biome average (area weighted)
            X_std[biome] = np.sqrt(np.nansum((X - X_avg[biome]) ** 2 * land_area_biome) / land_area_biometot)   # biome std (area weighted)
            X_qua[biome] = cls.weighted_quantile(X * land_area_biome/land_area_biome, [0.05,0.25,0.5,0.75,0.95], sample_weight=land_area_biome)

            X_bp[biome] = (X * land_area_biome)
            X_bp[biome] = X_bp[biome][~np.isnan(X_bp[biome])] / np.nanmean(land_area_biome)
            '''
            
            X_biome = X * land_area_biome
            index = ~np.isnan(X_biome)
            land_area_biometot = np.sum(land_area_biome[index])
            X_sum[biome] = np.sum(X_biome[index])
            X_avg[biome] = X_sum[biome] / land_area_biometot
            X_std[biome] = np.sqrt(np.sum((X[index] - X_avg[biome]) ** 2 * land_area_biome[index]) / land_area_biometot)
            X_qua[biome] = cls.weighted_quantile(X[index], [0.05,0.25,0.5,0.75,0.95], sample_weight=land_area_biome[index])
            X_bp[biome] = X_biome[index] / np.mean(land_area_biome[index])
            
            # calculate relative differences to reference.
            if Xref is not None:
                
                '''
                #old method. Slower, but more clear.
                rel_avg[biome] = np.nansum(rel * land_area_biome) / land_area_biometot
                abs_avg[biome] = np.nansum(abs * land_area_biome) / land_area_biometot
                #abs_avg[biome] = np.nansum(abs * area * land_area_biome) / land_area_biometot      # totaal, niet per m2

                rel_std[biome] = np.sqrt(np.nansum((rel - rel_avg[biome]) ** 2 * land_area_biome) / land_area_biometot)
                abs_std[biome] = np.sqrt(np.nansum((abs - abs_avg[biome]) ** 2 * land_area_biome) / land_area_biometot)
                '''
                
                rel_biome = rel * land_area_biome
                abs_biome = abs * land_area_biome
                
                rel_avg[biome] = np.nansum(rel_biome[index]) / land_area_biometot
                abs_avg[biome] = np.nansum(abs_biome[index]) / land_area_biometot
                rel_std[biome] = np.sqrt(np.nansum((rel[index] - rel_avg[biome]) ** 2 * land_area_biome[index]) / land_area_biometot)
                abs_std[biome] = np.sqrt(np.nansum((abs[index] - abs_avg[biome]) ** 2 * land_area_biome[index]) / land_area_biometot)
            
        
        if Xref is not None:
            return X_sum, X_avg, X_std, X_qua, X_bp, rel_avg, abs_avg, rel_std, abs_std
        else:
            return X_sum, X_avg, X_std, X_qua, X_bp
    
    
        
    @classmethod
    def biome_calc(cls, X, biomes, resolution, years, tile, Xref=None, LC=None, lwmask=None):
        ''' Usable for:
        1. 500m,    one tile
        2. 0.25deg, one tile
        3. 500m,    one region (e.g. Africa)
        4. 0.25deg, one region (e.g. Africa)
        5. 0.25deg, global
        NaN values are not counted in average: nanmean() functionality.
        Not for multiple tiles, then use biome_calctile
        '''
        rdir = cls.wdir + 'results/paper_results/'
        tiles_box = cls.tiles_box
        
        if (type(biomes) is str) or (type(biomes) is np.string_): biomes = [biomes]
        
        # load land area
        if (resolution == 500) and (tile in tiles_box.keys()):
            
            tiles = tiles_box[tile]
            if lwmask == None:
                lwmask = cls.save(lambda: cls.tile_paste(lambda tilen: cls.readdata_LWMASK('LWM', 500, tilen), tiles), rdir+'LW500_bypass.tif', save=False)
                lwmask[lwmask == 0] = np.nan
            area, _ = cls.load_area(resolution, tile)
            land_area = lwmask * area
            if LC == None:
                LC = cls.saver(lambda: cls.tile_paste(lambda tile: cls.readdata_MODIS('LC', 500, years, tile), tiles), rdir + 'LC500_%s_bypass.tif' % years, save=False)
            
        else:
            biome_dict = cls.biome_mask(biomes, resolution, years, tile)[1]
            lwmask = cls.readdata_LWMASK('LWM', resolution, tile, frac='no')
            area, _ = cls.load_area(resolution, tile)
            land_area = lwmask * area
        
        # additional tile masking
        if resolution != 500:
            if tile[::3] == 'hv':       # if tile is a MODIS tile.
                tile_mask = cls.tile_mask(resolution, tile, frac='yes')[0]
                land_area = land_area * tile_mask
            elif tile in tiles_box.keys():
                tiles = tiles_box[tile]
                tile_mask = cls.tile_mask(resolution, tiles, frac='yes', extent=tile)[0]
                land_area = land_area * tile_mask
            # else if tile == 'global': no tile masking needed
        
        if Xref is not None:
            abs = X - Xref
            rel = abs / Xref
            rel, _ = cls.perc_cut(rel * (land_area / area), [2.5, 2.5])
            
            rel_avg = OrderedDict()
            abs_avg = OrderedDict()
            rel_std = OrderedDict()
            abs_std = OrderedDict()
        
        # calculate biome sum and average
        X_sum = OrderedDict()
        X_avg = OrderedDict()
        X_std = OrderedDict()
        X_qua = OrderedDict()
        X_bp = OrderedDict()
        for biome in biomes:
            if resolution == 500:
                print biome
            
            # get biome mask
            if (resolution == 500) and (tile in tiles_box.keys()):
                biome_mask = np.zeros_like(LC)
                if biome == 'total':    biome_mask[LC > 0] = 1  # all except 0 (water) and -2 (unclassified). Pretty much a land mask.
                elif ' ' in biome:      # multiple biomes considered together.
                    subbiomes = biome.split()
                    for b in subbiomes: biome_mask[LC == int(b)] = 1    # combined mask of multiple biomes.
                else:
                    biome_mask[LC == int(biome)] = 1
                biome_mask = biome_mask.astype(float)
                biome_mask[biome_mask == 0] = np.nan
            else:
                biome_mask = biome_dict[biome]
            
            land_area_biome = land_area * biome_mask
            #land_area_biometot = np.nansum(land_area_biome)    # Wrong, remove nans in data first, as below.
            biome_mask = None
            
            '''
            #old method: much slower for large arrays (e.g. 4 times slower for 2400x2400). But more clear.
            X_sum[biome] = np.nansum(X * land_area_biome)       # biome sum (area weighted)
            X_avg[biome] = X_sum[biome] / land_area_biometot    # biome average (area weighted)
            X_std[biome] = np.sqrt(np.nansum((X - X_avg[biome]) ** 2 * land_area_biome) / land_area_biometot)   # biome std (area weighted)
            X_qua[biome] = cls.weighted_quantile(X * land_area_biome/land_area_biome, [0.05,0.25,0.5,0.75,0.95], sample_weight=land_area_biome)

            X_bp[biome] = (X * land_area_biome)
            X_bp[biome] = X_bp[biome][~np.isnan(X_bp[biome])] / np.nanmean(land_area_biome)
            '''
            
            X_biome = X * land_area_biome
            index = ~np.isnan(X_biome)
            land_area_biometot = np.sum(land_area_biome[index])
            X_sum[biome] = np.sum(X_biome[index])
            X_avg[biome] = X_sum[biome] / land_area_biometot
            X_std[biome] = np.sqrt(np.sum((X[index] - X_avg[biome]) ** 2 * land_area_biome[index]) / land_area_biometot)
            X_qua[biome] = cls.weighted_quantile(X[index], [0.05,0.25,0.5,0.75,0.95], sample_weight=land_area_biome[index])
            X_bp[biome] = X_biome[index] / np.mean(land_area_biome[index])
            
            # calculate relative differences to reference.
            if Xref is not None:
                
                '''
                #old method. Slower, but more clear.
                rel_avg[biome] = np.nansum(rel * land_area_biome) / land_area_biometot
                abs_avg[biome] = np.nansum(abs * land_area_biome) / land_area_biometot
                #abs_avg[biome] = np.nansum(abs * area * land_area_biome) / land_area_biometot      # totaal, niet per m2

                rel_std[biome] = np.sqrt(np.nansum((rel - rel_avg[biome]) ** 2 * land_area_biome) / land_area_biometot)
                abs_std[biome] = np.sqrt(np.nansum((abs - abs_avg[biome]) ** 2 * land_area_biome) / land_area_biometot)
                '''
                
                rel_biome = rel * land_area_biome
                abs_biome = abs * land_area_biome
                
                rel_avg[biome] = np.nansum(rel_biome[index]) / land_area_biometot
                abs_avg[biome] = np.nansum(abs_biome[index]) / land_area_biometot
                rel_std[biome] = np.sqrt(np.nansum((rel[index] - rel_avg[biome]) ** 2 * land_area_biome[index]) / land_area_biometot)
                abs_std[biome] = np.sqrt(np.nansum((abs[index] - abs_avg[biome]) ** 2 * land_area_biome[index]) / land_area_biometot)
            
        
        if Xref is not None:
            return X_sum, X_avg, X_std, X_qua, X_bp, rel_avg, abs_avg, rel_std, abs_std
        else:
            return X_sum, X_avg, X_std, X_qua, X_bp
    
    
    
    
    @classmethod
    def biome_calctile(cls, readfunc, biomes, resolution, year, tiles):
        
        if (type(biomes) is str) or (type(biomes) is np.string_): biomes = [biomes]
        if (type(tiles) is str) or (type(tiles) is np.string_): tiles = [tiles]

        X_sum = OrderedDict([(biome, 0) for biome in biomes])
        X_avg = OrderedDict()
        X_std = OrderedDict()
        X_sqsum = OrderedDict([(biome, 0) for biome in biomes])
        X_bp = OrderedDict([(biome, np.array([])) for biome in biomes])
        X_land = OrderedDict([(biome, 0) for biome in biomes])
        pixels = OrderedDict([(biome, 0) for biome in biomes])
        # ESS = OrderedDict([(biome, 0) for biome in biomes])
        # tile_avgs = OrderedDict([(biome, {}) for biome in biomes])
        # tile_land = OrderedDict([(biome, {}) for biome in biomes])
        for tile in tiles:
            print tile
            # load land area
            biome_dict = cls.biome_mask(biomes, resolution, year, tile)[1]
            lwmask = cls.readdata_LWMASK('LWM', resolution, tile, frac='no')
            area, _ = cls.load_area(resolution, tile)
            land_area = lwmask * area
            
            # additional tile masking
            if resolution != 500:
                tile_mask = cls.tile_mask(resolution, tile, frac='yes')[0]
                land_area = land_area * tile_mask

            X = readfunc(tile)
            for biome in biomes:
                
                # get biome mask
                biome_mask = biome_dict[biome]

                land_area_biome = land_area * biome_mask
                #land_area_biometot = np.nansum(land_area_biome)    # Wrong, remove nans first, as below.
                
                '''
                #old method. Slower, but more clear.
                # X_sum[biome] += np.nansum(X * land_area_biome)  # biome sum
                # X_land[biome] += land_area_biometot
                # pixels[biome] += np.sum(~np.isnan(land_area_biome))
                # bpadd = (X * land_area_biome).flatten()
                # bpadd = bpadd[~np.isnan(bpadd)]
                # X_bp[biome] = np.hstack((X_bp[biome], bpadd))
                '''
                
                X_biome = X * land_area_biome
                index = ~np.isnan(X_biome)
                land_area_biometot = np.sum(land_area_biome[index])
                X_sum[biome] += np.sum(X_biome[index])
                X_land[biome] += land_area_biometot
                pixels[biome] += np.sum(index)
                X_bp[biome] = np.hstack((X_bp[biome], X_biome[index]))
                X_sqsum[biome] += np.sum(X[index] ** 2 * land_area_biome[index])
                
                # tile_avg = np.sum(X_biome[index]) / land_area_biometot
                # tile_std = np.sqrt(np.sum((X[index] - tile_avg) ** 2 * land_area_biome[index]) / land_area_biometot)
                # if np.isnan(tile_avg): tile_avg = 0
                # if np.isnan(tile_std): tile_std = 0
                # ESS[biome] += tile_std ** 2 * land_area_biometot
                # tile_avgs[biome][tile] = tile_avg
                # tile_land[biome][tile] = land_area_biometot
        
        for biome in biomes:
            X_avg[biome] = X_sum[biome] / X_land[biome]
            X_bp[biome] = X_bp[biome] / (X_land[biome] / pixels[biome])
            X_std[biome] = np.sqrt((X_sqsum[biome] / X_land[biome]) - X_avg[biome] ** 2)
            #same: X_std[biome] = np.sqrt((X_sqsum[biome] - 2*X_avg[biome] * X_sum[biome] + X_avg[biome] ** 2 * X_land[biome]) / X_land[biome])
        
        # TGSS = OrderedDict()    
        # for tile in tiles:
        #     for biome in biomes:
        #         if tile == tiles[0]:
        #             TGSS[biome] = 0
        #         TGSS[biome] += (tile_avgs[biome][tile] - X_avg[biome]) ** 2 * tile_land[biome][tile]
        # 
        # for biome in biomes:
        #     X_std[biome] = np.sqrt((ESS[biome] + TGSS[biome]) / X_land[biome])
        
        #TODO: if possible, also calculate pooled/composite percentiles.
        
        return X_sum, X_avg, X_std, X_bp
    
    
    @staticmethod
    def perc_cut(arr, cutoff):
        # cutoff = alpha/2 in %. using a very low cutoff is equivalent to not using a cutoff, but will make relmu Inf.
        
        arr[np.isinf(arr)] = np.nan
        if type(cutoff) is list:
            index_l = arr < np.nanpercentile(arr, cutoff[0])
            index_r = arr > np.nanpercentile(arr, 100 - cutoff[1])
        else:
            index_l = arr < np.nanpercentile(arr, cutoff)
            index_r = arr > np.nanpercentile(arr, 100 - cutoff)
        index = ~(index_l + index_r)

        perc_mask = np.zeros((index.shape[0], index.shape[1]))
        perc_mask[index == True] = 1
        perc_mask[index == False] = np.nan  # just a mask of the index, for checking.

        return arr * perc_mask, perc_mask
    
    ''' --- '''
    
    @classmethod
    def region_sum(cls, sdir, tiles, res, calc):
        # The previous region_sum did the same, but just per region, instead of a tile list. The new one is more flexible.
        
        time0 = timer.time()

        if (type(tiles) is str) or (type(tiles) is np.string_): tiles = [tiles]
        
        tile_dict = OrderedDict()
        for tilen in tiles:
            n = cls(sdir, mode=1, resolution=res, tile=tilen, firemode='on', act='passive')
            
            masks = [tilen in cls.regions[key][1] for key in cls.regions.keys()[1:]]
            masks = [cls.regions.keys()[1:][ind] for ind in np.where(masks)[0]]  # list of GFED regions in tile.
            num_of_masks = len(masks)
            
            area, _ = cls.load_area(res, tilen)
            
            X_dict = OrderedDict()
            
            if calc == 'AGB_LIT':
                if len(masks) == 1:
                    try:    n.load()
                    except: print 'tile %s not available' % tilen; continue
                    Xsum = np.mean(n.diagt_pool['stem'][-12:]) + np.mean(n.diagt_pool['leaf'][-12:]) + np.mean(n.diagt_pool['gras'][-12:] + n.diagt_pool['cwd'][-12:]) + np.mean(n.diagt_pool['litt'][-12:])
                    Xavg = np.mean(n.diagm_pool['stem'][-12:]) + np.mean(n.diagm_pool['leaf'][-12:]) + np.mean(n.diagm_pool['gras'][-12:] + n.diagm_pool['cwd'][-12:]) + np.mean(n.diagm_pool['litt'][-12:])
                    X_dict['tile'] = (Xsum, Xavg)
                    X_dict[masks[0]] = (Xsum, Xavg)
                else:
                    try:    n.load(full=['AGB', 'LIT'])
                    except: print 'tile %s not available' % tilen; continue
                    Data = np.mean(n.diagf['AGB'] + n.diagf['LIT'], axis=0)
                    
                    # values for full tile:
                    lwmask = cls.readdata_LWMASK('LWM', res, tilen)
                    land_area = area * lwmask
                    land_area_tot = np.nansum(land_area)
                    Xsum = np.nansum(Data * land_area)
                    Xavg = Xsum / land_area_tot
                    X_dict['tile'] = (Xsum, Xavg)
                    
                    # values for independent regions:
                    for region in masks:
                        mask = cls.region_mask(region, res, tilen)
                        if type(res) == float:
                            mask = mask * cls.tile_mask(res, tilen)
                            
                        Xsum = np.nansum((Data * area)[mask == 1])
                        Xavg = np.nanmean(Data[mask == 1])
                        X_dict[region] = (Xsum, Xavg)
            
            
            elif calc == 'AGBf_LITf':
                if len(masks) == 1:
                    try:
                        n.load()
                    except:
                        print 'tile %s not available' % tilen; continue
                    Xsum = np.mean(n.diagt_fire['stem_fire'][-12:]) + np.mean(n.diagt_fire['leaf_fire'][-12:]) + np.mean(n.diagt_fire['gras_fire'][-12:] + n.diagt_fire['cwd_fire'][-12:]) + np.mean(n.diagt_fire['litt_fire'][-12:])
                    Xavg = np.mean(n.diagm_fire['stem_fire'][-12:]) + np.mean(n.diagm_fire['leaf_fire'][-12:]) + np.mean(n.diagm_fire['gras_fire'][-12:] + n.diagm_fire['cwd_fire'][-12:]) + np.mean(n.diagm_fire['litt_fire'][-12:])
                    X_dict['tile'] = (Xsum, Xavg)
                    X_dict[masks[0]] = (Xsum, Xavg)
                else:
                    try:    n.load(full=['AGB_fire', 'LIT_fire'])
                    except: print 'tile %s not available' % tilen; continue
                    Data = np.mean(n.diagf['AGB_fire'] + n.diagf['LIT_fire'], axis=0)
                    
                    # values for full tile:
                    lwmask = cls.readdata_LWMASK('LWM', res, tilen)
                    land_area = area * lwmask
                    land_area_tot = np.nansum(land_area)
                    Xsum = np.nansum(Data * land_area)
                    Xavg = Xsum / land_area_tot
                    X_dict['tile'] = (Xsum, Xavg)

                    # values for independent regions:
                    for region in masks:
                        mask = cls.region_mask(region, res, tilen)
                        if type(res) == float:
                            mask *= cls.tile_mask(res, tilen)

                        Xsum = np.nansum((Data * land_area)[mask == 1])
                        Xavg = np.nanmean(Data[mask == 1])
                        X_dict[region] = (Xsum, Xavg)
            
            
            tile_dict[tilen] = X_dict

        print 'Done, duration = %s s' % (timer.time() - time0)
        return tile_dict, timer.time() - time0
    
    
    @classmethod
    def weighted_avg(cls, data_dict, region, res):
        # The previous weighted_avg did the same, but just per region, instead of a tile list. The new one is more flexible.
        
        time0 = timer.time()
        avg = 0
        perc_tot = 0
        for tilen in cls.regions[region][1]:
            
            if tilen in data_dict.keys():
                mask = cls.region_mask(region, res, tilen)
                # if type(res) == float:
                #    mask *= cls.tile_mask(res, tilen) Do I need this here ?????
                
                perc = (len(np.where(mask == 1)[0]) / float(mask.shape[0] * mask.shape[1]))     # percentage of mask in tile.
                perc_tot = perc_tot + perc
                avg = avg + data_dict[tilen][region][1] * perc
            else:
                print 'WARNING: tile %s not in dict, and not used.' % tilen
    
        wavg = avg / perc_tot
        
        return wavg, timer.time() - time0
    
    
    @classmethod
    def region_avg(cls, readfunc, region, res):
        # Compared to weighted_avg, this is a direct method that doenst require region_sum.
        
        time0 = timer.time()
        avg = 0
        perc_tot = 0
        for tilen in cls.regions[region][1]:
            
            masks = [tilen in cls.regions[key][1] for key in cls.regions.keys()[1:]]
            masks = [cls.regions.keys()[1:][ind] for ind in np.where(masks)[0]]  # list of GFED regions in tile.
            num_of_masks = len(masks)
            
            LW = cls.readdata_LWMASK('LWM', res, tilen)
            # LC = cls.readdata_MODIS('LC2', res, [yearn], tilen)[0, :, :]
            # LC[LC > 20] = np.nan
            mask = cls.region_mask(region, res, tilen)
            mask[mask == 0] = np.nan
            mask = mask * LW
            if type(res) == float:
                mask = mask * cls.tile_mask(res, tilen)
            mask[mask == 0] = np.nan
            
            if num_of_masks == 1:
                result = readfunc(tilen, res, direct='on')
            else:
                Data = readfunc(tilen, res)
                result = np.nanmean(Data[mask == 1])
                # arr = Data * fullmask
                # result = np.nanmean(arr)
            
            if np.isnan(result):    result = 0
            _, area_tile = cls.load_area(res, tilen)
            perc = (len(np.where(mask == 1)[0]) / float(mask.shape[0] * mask.shape[1]))  # * area_tile
            perc_tot = perc_tot + perc
            avg = avg + result * perc
        final = avg / perc_tot
        
        return final, timer.time() - time0
    
    ''' --- '''
    
    
    @staticmethod
    def amean(X):
        '''  Calculates annual mean of monthly time-series
        Parameters
        ----------
        :X: monthly time series
        '''
        if len(X)%12 != 0:     print 'Error in amean method: not a multiple of 12'
        X_mean = np.mean(np.reshape(X,(len(X)/12,12)),1)
        return X_mean
    
    
    @staticmethod
    def multiprocess(function, iterator, num_of_cores):
        ''' Overarging function for multiprocessing. Requires the definition of a 'worker' function that includes the .algorithm() loop
        Parameters
        ----------
        :function: worker function to apply multiprocessing to
        :iterator: 
        :num_of_cores: number of processor cores used. Valid input = 1 - 8 [int]
        '''
        import multiprocessing as mp
        import traceback
        from functools import partial
        print 'Start multiprocessing, using %s processor core(s) %s' % (num_of_cores, timer.strftime("%a %d %b %Y %X"))
        time0 = timer.time()
        pool_grouping = 'off'
        ''' For now pool grouping only works when there are no ghost processes. A ghost process causes the pool to live forever.'''
        
        # def function2(i):
        #     return Model.multiprocess_inner(function, i)
        
        try:
            # ADD queue: (not needed, but nice trick)
            #queue = mp.Queue(3)
            #def function_addqueue(queue):       function.queue = queue
            #pool = mp.Pool(num_of_cores, function_addqueue, [queue])
            #pool.map(function, iterator)
            
            if pool_grouping == 'off':          # Basic pool mapping
                pool = mp.Pool(num_of_cores)
                pool.map(function, iterator)
                pool.close()
                
            if pool_grouping == 'on':          # Alternative: loop through grouped pools, ensuring synchronized execution and printing
                num_of_groups = (len(iterator) + num_of_cores - 1) / num_of_cores       # ceiling divide of iterator length by num_of_cores
                for loopgroup in range(num_of_groups):
                    print 'START poolgroup: %s / %s' % (loopgroup, num_of_groups)
                    pool = mp.Pool(num_of_cores)
                    iter_group = iterator[loopgroup * num_of_cores : loopgroup * num_of_cores + num_of_cores]
                    pool.map(function, iter_group)
                    pool.close()
                    
        
        except (KeyboardInterrupt, SystemExit):
            print '|| From pool: Caught KeyboardInterrupt or sys.exit(), terminating workers'
            pool.terminate()
            raise KeyboardInterruptError()      # fake error

        except Exception, e:
            print '|| From pool: Received error: %r' % (e,)
            exc_type, exc_obj, exc_tb = sys.exc_info()
            fname = os.path.split(exc_tb.tb_frame.f_code.co_filename)[1]
            print(exc_type, fname, exc_tb.tb_lineno)
            traceback.print_exc()
            pool.terminate()                                            # terminates CPU processes
        finally:
            pool.join()
            import gc
            gc.collect()                                                # garbage collection, for freeing memory
            print 'Pool(s) Finished %s' % timer.strftime("%a %d %b %Y %X")
            print 'Duration = %s (h,m,s)' % (timer.strftime("%H:%M:%S", timer.gmtime(timer.time() - time0)))
    
    
    @staticmethod
    def tile_paste(readfunc, tiles, dt=0, mres=500, lwmask=False, dsample=None):
        
        '''
        :param readfunc: function that reads data for MODIS tile and outputs array 2d or 3d.
        :param tiles: list of tiles to paste together
        :param dt: dt=0 is one timestep, dt >1 is multiple timesteps.
        :param mres: standard is 500m MODIS, adjust for 1km (mres=1000) or 250m (mres=250).
        :param lwmask: mask water. Default is no masking.
        :param dsample: downsample final result to coarser resolution, by integer factor.
        :return: return pasted array.
        
        Example readfunc:
        def readfunc_example(tilen):
            add = Model.readdata_MODIS('FTC', 500, ['2003'], tilen)[0,:,:]      # add = 2400x2400 array
            return add       # output should be a 2d-array or 3d-array for that tile.
        '''
        
        time0 = timer.time()
        tiles = sorted(set(tiles))
        hs = [int(t[1:3]) for t in tiles]
        vs = [int(t[4:6]) for t in tiles]
        hall = np.arange(min(hs), max(hs)+1)
        vall = np.arange(min(vs), max(vs)+1)
        
        nindex = (2400*500) / mres
        
        if dt == 0:
            shapen = (nindex,nindex)
        elif dt != 0:
            shapen = (dt,nindex,nindex)
        
        i = 0
        for h in hall:
            for v in vall:
                tilen = 'h%02dv%02d' % (h, v)
                if tilen in tiles:
                    try:
                        add = readfunc(tilen)
                        if lwmask == True:
                            add = add * Model.readdata_LWMASK('LWM', 500, tilen)
                    except Exception, e:
                        print 'Tile %s, Error %s, using zeros array' % (tilen, e)
                        add = np.zeros(shapen)
                        add[add == 0] = np.nan
                    if v == min(vall):   vertical = add
                    else:
                        vertical = np.concatenate((vertical, add), axis=-2)
                else:
                    add = np.zeros(shapen)
                    add[add == 0] = np.nan
                    if v == min(vall):   vertical = add
                    else:
                        vertical = np.concatenate((vertical, add), axis=-2)
            if h == min(hall):  ARR = vertical
            else:
                ARR = np.concatenate((ARR, vertical), axis=-1)
        if dsample is not None:
            if dt == 0:
                ARR = ARR.reshape((ARR.shape[0] // dsample, dsample, ARR.shape[1] // dsample, dsample)).mean(3).mean(1)
            elif dt != 0:
                ARR = ARR.reshape((dt, ARR.shape[1] // dsample, dsample, ARR.shape[2] // dsample, dsample)).mean(4).mean(2)
        
        print 'Done, duration = %s s' % (timer.time() - time0)
        return ARR
    
    
    @classmethod
    def region_paste(cls, readfunc, regions, dsample=None):
        # Similar to tile_paste, but including masking for regions.
        
        # example:
        '''
        def readfunc_example(tilen):
            wdir = '/Volumes/Mac_HD/Work/Vici_project/koolstof_model/'
            sdir = wdir + 'results/' + 'run9_500m_Africa/'
            n = Model(sdir, mode=1, resolution=500, tile=tilen, act='passive')
            n.load(full=['AGB', 'LIT'])
            add = np.nanmean(n.diagf['AGB'] + n.diagf['LIT'], axis=0)
            return add      # output should be a 2d-array for that tile.
        '''
        
        time0 = timer.time()
        if (type(regions) is str) or (type(regions) is np.string_): regions = [regions]
        tiles = [cls.regions[region][1] for region in regions]
        tiles = sum((tiles[i] for i in range(len(tiles))), [])
        tiles = sorted(set(tiles))
        hs = [int(t[1:3]) for t in tiles]
        vs = [int(t[4:6]) for t in tiles]
        hall = np.arange(min(hs), max(hs) + 1)
        vall = np.arange(min(vs), max(vs) + 1)
        
        for h in hall:
            for v in vall:
                
                tilen = 'h%02dv%02d' % (h, v)
                if tilen in tiles:
                    
                    # masks kan ook direct worden geladen uit GFED_regions_per_tile.npy, maar onderstaand is stand-alone.
                    masks = [tilen in cls.regions[key][1] for key in cls.regions.keys()[1:]]
                    masks = [cls.regions.keys()[1:][ind] for ind in np.where(masks)[0]]   # list of GFED regions in tile.
                    num_of_masks = len(masks)   # number of GFED regions in tile.
                    masks = list(set(masks).intersection(regions))
                    
                    fullmask = 0
                    for i in range(len(masks)):
                        regiomask = cls.region_mask(masks[i], 500, tilen)
                        regiomask[np.isnan(regiomask)] = 0
                        fullmask = fullmask + regiomask
                    
                    fullmask = fullmask * cls.readdata_LWMASK('LWM', 500, tilen)
                    fullmask[fullmask>1] = 1
                    fullmask[fullmask==0] = np.nan
                    
                    try:
                        add = readfunc(tilen)
                        add = add * fullmask
                        
                    except Exception, e:
                        print 'Tile %s, Error %s, using zeros array' % (tilen, e)
                        add = np.zeros((2400, 2400))
                        add[add == 0] = np.nan
                    if v == min(vall):
                        vertical = add
                    else:
                        vertical = np.concatenate((vertical, add), axis=0)
                else:
                    add = np.zeros((2400, 2400))
                    add[add == 0] = np.nan
                    if v == min(vall):
                        vertical = add
                    else:
                        vertical = np.concatenate((vertical, add), axis=0)
            if h == min(hall):
                ARR = vertical
            else:
                ARR = np.concatenate((ARR, vertical), axis=1)
        if dsample is not None:
            ARR = ARR.reshape((ARR.shape[0] / dsample, dsample, ARR.shape[1] / dsample, dsample)).mean(3).mean(1)

        print 'Done, duration = %s s' % (timer.time() - time0)
        return ARR
    
    
    @classmethod
    def translate(cls, arr, tile, resolution):
        
        if resolution == 500:
            res = '500m'
        elif type(resolution) == float:
            res = '%03.0fdeg' % float(('%0.2f' % round(resolution,2)).lstrip('0.'))  # converts e.g. 0.25 -> '025deg'
        
        total_pix = np.load(cls.wdir_geo + '00_Total_pixels_' + res + '.npy')     # number of MODIS pixels in every degree grid cell.
        
        if arr.ndim == 2: arr = np.expand_dims(arr, axis=0)
        
        Y = []
        time0 = timer.time()
        print 'Processing tile: %s ... ' % tile ,
        summ = np.zeros((arr.shape[0], 180 * int(1 / resolution), 360 * int(1 / resolution)))
        
        hdf = SD(cls.wdir_geo + 'MODIS_geoloc_500m_' + tile + '.hdf', SDC.READ)
        lats_geo = hdf.select('Latitude').get()  # -90 : + 90
        lons_geo = hdf.select('Longitude').get()  # -180 : +180
        hdf = []
        lat_index = np.floor((lats_geo + 90) * int(1 / float(resolution))).astype(int)
        lon_index = np.floor((lons_geo + 180.) * int(1 / float(resolution))).astype(int)
        
        index = np.where((np.nanmean(arr, axis=0) != 0) & (~np.isnan(np.nanmean(arr, axis=0))))     # 500m pixels with zero data or nan can be skipped. Taking the sum to see if its zero for every time step.
        #OLD (not correct): index = np.where((np.sum(arr, axis=0) != 0) & (~np.isnan(np.nanmean(arr, axis=0))))
        #  use nanmean because nanmean([np.nan]) = np.nan, whereas nansum([np.nan]) = 0.
        #TODO: Isn't index = np.where(np.nansum(arr) > 0) exactly the same? Followed by arr[np.isnan(arr)] = 0
        if arr.shape[0] > 1:                    # much faster for multiple time steps.
            for i in range(0, len(index[0])):
                x = index[0][i]
                y = index[1][i]
                if lat_index[x, y] < (180 * int(1 / resolution)) + 1 and lon_index[x, y] < (360 * int(1 / resolution)):
                    arr[:, x, y][np.isnan(arr[:, x, y])] = 0        # IMPORTANT: in case of arr[:,x,y] = [np.nan, 0.35]. The case e.g. if biome mask changes over years.
                    summ[:, lat_index[x, y], lon_index[x, y]] = summ[:, lat_index[x, y], lon_index[x, y]] + arr[:, x, y]
        
        elif arr.shape[0] == 1:                 # much faster for 1 time step. Indexing with 0 instead of ':'.
            for i in range(0, len(index[0])):
                x = index[0][i]
                y = index[1][i]
                if lat_index[x, y] < (180 * int(1 / resolution)) + 1 and lon_index[x, y] < (360 * int(1 / resolution)):
                    summ[0, lat_index[x, y], lon_index[x, y]] = summ[0, lat_index[x, y], lon_index[x, y]] + arr[0, x, y]
        
        summ = np.fliplr(summ)  # fliplr, because flipping 2nd dimension = lat, (time, lat, lon). Also when arr.shape[0]==1, because 1xlatxlon.
        X = summ / total_pix.astype('float')
        X[np.isnan(X)] = 0
        
        X = np.squeeze(np.array(X))
        print 'Done, duration = %s s.' % (timer.time() - time0)
        return X
    
    
    @staticmethod
    def translate_deg_to_coarser(arr, resin, res, include_zeros=True):
        d = int(res / resin)  # downscale fraction
        X = arr.copy()
        if include_zeros == True:
            X[np.isnan(X)] = 0      # dont use nanmean, zeros have to be included in mean!
            Y = X.reshape((X.shape[0] / d, d, X.shape[1] / d, d)).mean(3).mean(1)
            
        elif include_zeros == False:
            Y = np.nanmean(np.nanmean(X.reshape((X.shape[0] / d, d, X.shape[1] / d, d)), axis=3), axis=1)
            
        return Y
    
    
    @classmethod
    def mosaic_500m_gdal(cls, arr):
        
        driver = gdal.GetDriverByName('GTiff')
        ds = driver.Create(cls.wdir + 'temp.tif', arr.shape[1], arr.shape[0], 1, gdal.GDT_Byte, options=['COMPRESS=LZW', 'INTERLEAVE=BAND', 'TILED=YES'])
        ds.GetRasterBand(1).WriteArray(arr)
        ds = None
        
        inputstr1 = cls.wdir + 'temp.tif'
        outputstr1 = cls.wdir + 'temp_out.vrt'
        cmd1 = 'gdal_translate -of VRT %s %s' % (inputstr1, outputstr1)
        print cmd1
        os.system(cmd1)
        
        inputstr2 = cls.wdir + 'temp_out.vrt'
        outputstr2 = cls.wdir + 'mcd64_336_500.tif'
        cmd2 = 'gdalwarp -srcnodata 0 -r mode -t_srs "+proj=latlong +datum=WGS84" -te -180.0 -90.0 180.0 90.0 -tr 0.05 0.05 %s %s' % (inputstr2, outputstr2)
        print cmd2
        os.system(cmd2)
        
        os.system('rm -r %s') % outputstr1
    
    
    #print 'Done \n'
    
    
    @staticmethod
    def weighted_quantile(values, quantiles, sample_weight=None, values_sorted=True, old_style=False):
        """ Very close to numpy.percentile, but supports weights.
        Source: https://stackoverflow.com/questions/21844024/weighted-percentile-using-numpy
        NOTE: quantiles should be in [0, 1]!
        :param values: numpy.array with data
        :param quantiles: array-like with many quantiles needed
        :param sample_weight: array-like of the same length as `array`
        :param values_sorted: bool, if True, then will avoid sorting of initial array
        :param old_style: if True, will correct output to be consistent with numpy.percentile.
        :return: numpy.array with computed quantiles.
        """
        
        ''' Note:
        For small samples, there is a difference between the boxplot result and percentiles calculated here.
        This is because plt.boxplot does an interpolation, and the function used here uses an existing value in the data.
        Takeaway: dont use small samples!!!. E.g. for biomes only select major biomes.
        '''
        
        if np.all(np.isnan(values)):
            return [np.nan]                       # in case of no data in region.
        else:
            quantiles = np.array(quantiles)
            if sample_weight is None:
                sample_weight = np.ones_like(values)
            assert np.all(quantiles >= 0) and np.all(quantiles <= 1), 'quantiles should be in [0, 1]'
            
            values = values[np.isfinite(values)].flatten()                      # remove NaN and flatten.
            sample_weight = sample_weight[np.isfinite(sample_weight)].flatten()
            
            if values_sorted:
                sorter = np.argsort(values)
                values = values[sorter]
                sample_weight = sample_weight[sorter]
            
            weighted_quantiles = np.cumsum(sample_weight) - 0.5 * sample_weight
            if old_style:
                # To be convenient with numpy.percentile
                weighted_quantiles -= weighted_quantiles[0]
                weighted_quantiles /= weighted_quantiles[-1]
            else:
                weighted_quantiles /= np.sum(sample_weight)
            return np.interp(quantiles, weighted_quantiles, values)
    
    
    @staticmethod
    def custom_boxplot(percentiles, axes, redraw=True, *args, **kwargs):
        """
        Generates a customized boxplot based on the given percentile values
        Source: https://stackoverflow.com/questions/27214537/is-it-possible-to-draw-a-matplotlib-boxplot-given-the-percentile-values-instead
        """
        n_box = len(percentiles)
        box_plot = axes.boxplot([[0, 0, 0, 0, 0],]*n_box, *args, **kwargs) 
        # Creates len(percentiles) no of box plots
    
        min_y, max_y = float('inf'), -float('inf')
    
        for box_no, pdata in enumerate(percentiles):
            if len(pdata) == 6:
                (q1_start, q2_start, q3_start, q4_start, q4_end, fliers_xy) = pdata
            elif len(pdata) == 5:
                (q1_start, q2_start, q3_start, q4_start, q4_end) = pdata
                fliers_xy = None
            else:
                continue
    
            # Lower cap
            box_plot['caps'][2*box_no].set_ydata([q1_start, q1_start])
            # xdata is determined by the width of the box plot
    
            # Lower whiskers
            box_plot['whiskers'][2*box_no].set_ydata([q1_start, q2_start])
    
            # Higher cap
            box_plot['caps'][2*box_no + 1].set_ydata([q4_end, q4_end])
    
            # Higher whiskers
            box_plot['whiskers'][2*box_no + 1].set_ydata([q4_start, q4_end])
    
            # Box
            path = box_plot['boxes'][box_no].get_path()
            path.vertices[0][1] = q2_start
            path.vertices[1][1] = q2_start
            path.vertices[2][1] = q4_start
            path.vertices[3][1] = q4_start
            path.vertices[4][1] = q2_start
    
            # Median
            box_plot['medians'][box_no].set_ydata([q3_start, q3_start])
    
            # Outliers
            if fliers_xy is not None and len(fliers_xy[0]) != 0:
                # If outliers exist
                box_plot['fliers'][box_no].set(xdata = fliers_xy[0],
                                               ydata = fliers_xy[1])
    
                min_y = min(q1_start, min_y, fliers_xy[1].min())
                max_y = max(q4_end, max_y, fliers_xy[1].max())
    
            else:
                min_y = min(q1_start, min_y)
                max_y = max(q4_end, max_y)
    
            # The y axis is rescaled to fit the new box plot completely with 10% 
            # of the maximum value at both ends
            axes.set_ylim([min_y*1.1, max_y*1.1])
    
        # If redraw is set to true, the canvas is updated.
        if redraw:
            axes.figure.canvas.draw()
    
        return box_plot
    
    
    @staticmethod
    def saver(func, filename, save=True):   # , *args):
        # The function arguments can also be passed using *args, but lambda is nicer.
        # Source: https://stackoverflow.com/questions/803616/passing-functions-with-arguments-to-another-function-in-python
        
        dtype = filename[-4:]
        if dtype not in ['.npy', '.npz', '.tif']:
            print 'Incorrect, or no, file extension!'
        
        if os.path.isfile(filename):
            
            if dtype == '.npy':
                X = np.load(filename)[()]
            
            elif dtype == '.npz':
                X = np.load(filename)
                X = {key: X[key] for key in X.keys()}
            
            elif dtype == '.tif':
                X = gdal.Open(filename, gdalconst.GA_ReadOnly).ReadAsArray()
        
        else:
            X = func()      # *args)
            
            if dtype == '.npy' and save is True:
                np.save(filename, X)
                
            elif dtype == '.npz' and save is True:
                np.savez(filename, **X)
                
            elif dtype == '.tif' and save is True:
                driver = gdal.GetDriverByName('GTiff')
                ds = driver.Create(filename, X.shape[1], X.shape[0], 1, gdal.GDT_Float32, options=['COMPRESS=LZW', 'INTERLEAVE=BAND', 'TILED=YES'])
                ds.GetRasterBand(1).WriteArray(X)
                ds = None
        
        return X
    
    
    @classmethod
    def load_2dvar(cls, sdir, res, tile, var, method, mode=2, fmode='on', addtxt=None):
        ''' Calculate average annual sum or average annual average of variable over the full model period, resulting in a 2d array.
        Possible entries for var: var in ['AGB', 'FL', 'FC', 'DB', 'NPP', 'BA', 'FTC', 'NTV', 'Fpar', 'LC'] 
        '''
        
        if tile in tiles_box.keys():
            tiles = tiles_box[tile]
        
        # load variable
        n = cls(sdir, mode=mode, resolution=res, tile=tile, firemode=fmode, act='passive')
        #TODO: should be instance method???
        
        if var == 'AGB':
            if (mode == 1) and (type(res) is float):   varlist = ['stem', 'leaf', 'gras']
            if (mode == 1) and (res == 500):           varlist = ['AGB']
            if (mode == 2) and (res == 0.25):          varlist = ['stem', 'leaf', 'gras']
            if (mode == 2) and (res != 0.25):          varlist = ['AGB']
        elif var == 'AGBw':
            if (mode == 1) and (type(res) is float):   varlist = ['stem', 'leaf']
            if (mode == 1) and (res == 500):           varlist = ['AGBw']
            if (mode == 2) and (res == 0.25):          varlist = ['stem', 'leaf']
            if (mode == 2) and (res != 0.25):          varlist = ['AGBw']
        elif var == 'FL':
            if (mode == 1) and (type(res) is float):    varlist = ['stem', 'leaf', 'gras', 'cwd', 'litt']
            if (mode == 1) and (res == 500):            varlist = ['AGB', 'LIT']
            if (mode == 2) and (res == 0.25):           varlist = ['stem', 'leaf', 'gras', 'cwd', 'litt']
            if (mode == 2) and (res != 0.25):           varlist = ['AGB', 'LIT']
        elif ('DB' in var) or ('FC' in var):
            if (mode == 1) and (type(res) is float):    varlist = ['stem_fire', 'leaf_fire', 'gras_fire', 'cwd_fire', 'litt_fire']
            if (mode == 1) and (res == 500):            varlist = ['AGB_fire', 'LIT_fire']
            if (mode == 2) and (res == 0.25):           varlist = ['AGB_fire', 'LIT_fire']
            if (mode == 2) and (res != 0.25):           varlist = ['AGB_fire', 'LIT_fire']
        elif 'NPP' in var:
            varlist = ['NPP']
        else:
            varlist = [var]     # e.g. if var == stem, or another pure model pool.
        
        
        if var in ['FTC', 'NTV', 'Fpar', 'LC']:
            
            if var in ['FTC', 'NTV', 'Fpar']:
                X = np.nanmean(cls.readdata_MODIS(var, res, n.years, tile), axis=0)
            elif var == 'LC':
                X = cls.readdata_MODIS('LC2', res, n.years[len(n.years) / 2], tile)[0]
        
        elif var == 'BA':
            X = cls.readdata_MODIS('BA', res, n.years, tile) * cls.load_area(res, tile)[0]
            diagf = 'full'
        
        else:
            n.load(full=varlist, addtxt=addtxt)
            diagf = n.diagf['meta']
            
            if diagf == 'full':
                X = 0
                for key in varlist:
                    X = X + n.diagf[key]    # sum up pools in varlist
            
            elif diagf == 'light':
                Xsum = 0
                Xavg = 0
                for key in varlist:
                    Xsum = Xsum + n.diagf[key][0]   # sum up pools in varlist
                    Xavg = Xavg + n.diagf[key][1]
        
        if var not in ['FTC', 'NTV', 'Fpar', 'LC']:
            # start calculation of statistic
            if method == 'avg':
                if diagf == 'light':  X = Xavg  # grab annual mean
                X = np.nanmean(X, axis=0)  # average annual mean
            
            elif method == 'sum':
                if mode == 1:
                    if diagf == 'light':  X = Xsum
                    X = np.nansum(X, axis=0)  # annual sum
                if mode == 2:
                    if diagf == 'full':
                        X = np.nanmean(np.nansum(np.reshape(X, (len(X) / 12, 12, X.shape[1], X.shape[2])), axis=1), axis=0)  # average annual sum
                    elif diagf == 'light':
                        X = Xsum
                        X = np.nanmean(X, axis=0)
            n.diagf = {}
            
        
        if var == 'FC':     # divide DB by BA to get FC.
            BA = cls.readdata_MODIS('BA', res, n.years, tile) * cls.load_area(res, tile)[0]
            if method == 'avg':
                BA = np.nanmean(BA, axis=0)  # average annual mean
            elif method == 'sum':
                BA = np.nanmean(np.nansum(np.reshape(BA, (len(BA) / 12, 12, BA.shape[1], BA.shape[2])), axis=1), axis=0)  # average annual sum
            X = X * cls.load_area(res, tile)[0] / BA
            
        # create mask
        lwmask = cls.readdata_LWMASK('LWM', res, tile)
        tilemask = 1
        if res != 500:
            tilemask = cls.tile_mask(res, tiles, frac='no', extent=tile)[0]
        mask = lwmask * tilemask
        
        return X * mask
    
    
    @classmethod
    def translate_to_deg(cls, sdir, resdeg, mode, fmode='on', extent='Africa', avail=False, addtxt=None):

        ress = '%03.0fdeg' % float(('%0.2f' % round(resdeg, 2)).lstrip('0.'))  # converts e.g. 0.25 -> '025deg'
        
        if avail is True:       # Search for available files, e.g. if run is incomplete.
            if mode == 1:   n_years = 200
            elif mode == 2: n_years = 16
            filename = 'data/' + 'mode'+str(mode)+'_'+str(n_years)+'yr_'+'500m_fmode-'+fmode+'_*'+addtxt+'.nc'
            filelist = [f for f in glob.glob(sdir + filename)]
            tiles = [re.search(r'h..v..', f).group() for f in filelist] 
        else:
            tiles = tiles_box[extent]
            
        for tile in tiles:
            
            AGB = cls.load_2dvar(sdir, 500, tile, 'AGB', 'avg', mode=mode, fmode=fmode, addtxt=addtxt)
            AGBw = cls.load_2dvar(sdir, 500, tile, 'AGBw', 'avg', mode=mode, fmode=fmode, addtxt=addtxt)
            FL = cls.load_2dvar(sdir, 500, tile, 'FL', 'avg', mode=mode, fmode=fmode, addtxt=addtxt)
            NPPavg = cls.load_2dvar(sdir, 500, tile, 'NPP', 'avg', mode=mode, fmode=fmode, addtxt=addtxt)
            NPPsum = cls.load_2dvar(sdir, 500, tile, 'NPP', 'sum', mode=mode, fmode=fmode, addtxt=addtxt)
            FTC = cls.load_2dvar(sdir, 500, tile, 'FTC', 'avg', mode=mode, fmode=fmode, addtxt=addtxt)
            Fpar = cls.load_2dvar(sdir, 500, tile, 'Fpar', 'avg', mode=mode, fmode=fmode, addtxt=addtxt)
            if fmode == 'on':
                DBavg = cls.load_2dvar(sdir, 500, tile, 'DB', 'avg', mode=mode, fmode=fmode, addtxt=addtxt)
                DBsum = cls.load_2dvar(sdir, 500, tile, 'DB', 'sum', mode=mode, fmode=fmode, addtxt=addtxt)
                BAavg = cls.load_2dvar(sdir, 500, tile, 'BA', 'avg', mode=mode, fmode=fmode, addtxt=addtxt)
                BAsum = cls.load_2dvar(sdir, 500, tile, 'BA', 'sum', mode=mode, fmode=fmode, addtxt=addtxt)
            
            if fmode == 'on':
                arr = np.array([AGB, AGBw, FL, DBavg, DBsum, NPPavg, NPPsum, BAavg, BAsum, FTC, Fpar])
            elif fmode == 'off':
                arr = np.array([AGB, AGBw, FL, NPPavg, NPPsum, FTC, Fpar])
            arrt = cls.translate(arr, tile, resdeg)
            
            if tile == tiles[0]:
                arr_deg = np.zeros(arrt.shape)
            
            arr_deg = arr_deg + arrt
            
        # arr_deg = arr_deg * cls.tile_mask(res, tiles, frac='yes', extent='global')[0]
        
        indexdeg, _, _, _ = cls.load_degbox(resdeg, extent)
        arr_deg = arr_deg[:, indexdeg['lat'][0]:indexdeg['lat'][-1] + 1, indexdeg['lon'][0]:indexdeg['lon'][-1] + 1]
        
        lwmask = cls.readdata_LWMASK('LWM', resdeg, extent)
        tilemask = cls.tile_mask(resdeg, tiles, frac='no', extent=extent)[0]
        mask = lwmask * tilemask
        arr_dict = {}
        if fmode == 'on':
            arrlist = ['AGB', 'AGBw', 'FL', 'DBavg', 'DBsum', 'NPPavg', 'NPPsum', 'BAavg', 'BAsum', 'FTC', 'Fpar']
        elif fmode == 'off':
            arrlist = ['AGB', 'AGBw', 'FL', 'NPPavg', 'NPPsum', 'FTC', 'Fpar']
        for i, var in enumerate(arrlist):
            arr_dict[var] = arr_deg[i] * mask
        
        
        if fmode == 'on':
            total_pix = np.load(cls.wdir_geo + '00_Total_pixels_'+ress+'.npy')
            indexdeg = cls.load_degbox(resdeg, extent)[0]
            total_pix = total_pix[indexdeg['lat'][0]:indexdeg['lat'][-1] + 1, indexdeg['lon'][0]:indexdeg['lon'][-1] + 1]
            arr_dict['BAsum'] = arr_dict['BAsum'] * total_pix
            arr_dict['BAavg'] = arr_dict['BAavg'] * total_pix
            
            BAavg = cls.load_2dvar(sdir, resdeg, extent, 'BA', 'avg', mode=mode, fmode=fmode, addtxt=addtxt)
            BAsum = cls.load_2dvar(sdir, resdeg, extent, 'BA', 'sum', mode=mode, fmode=fmode, addtxt=addtxt)
            arr_dict['FCsum'] = arr_dict['DBsum'] * cls.load_area(resdeg, extent)[0] / BAsum
            arr_dict['FCavg'] = arr_dict['DBavg'] * cls.load_area(resdeg, extent)[0] / BAavg
            
            # optie 2 (bijna zelfde resultaat):
            #arr_dict['FCsum'] = arr_dict['DBsum'] * cls.load_area(resdeg, extent)[0] / arr_dict['BAsum']
            #arr_dict['FCavg'] = arr_dict['DBavg'] * cls.load_area(resdeg, extent)[0] / arr_dict['BAavg']
        
        return arr_dict
    
    
    @classmethod
    def load_deg(cls, sdir, resdeg, mode, fmode='on', extent='Africa', addtxt=None):
        
        tiles = tiles_box[extent]
        
        ress = '%03.0fdeg' % float(('%0.2f' % round(resdeg, 2)).lstrip('0.'))  # converts e.g. 0.25 -> '025deg'
        
        AGB = cls.load_2dvar(sdir, resdeg, tile, 'AGB', 'avg', mode=mode, fmode=fmode, addtxt=addtxt)
        AGBw = cls.load_2dvar(sdir, resdeg, tile, 'AGBw', 'avg', mode=mode, fmode=fmode, addtxt=addtxt)
        FL = cls.load_2dvar(sdir, resdeg, tile, 'FL', 'avg', mode=mode, fmode=fmode, addtxt=addtxt)
        NPPavg = cls.load_2dvar(sdir, resdeg, tile, 'NPP', 'avg', mode=mode, fmode=fmode, addtxt=addtxt)
        NPPsum = cls.load_2dvar(sdir, resdeg, tile, 'NPP', 'sum', mode=mode, fmode=fmode, addtxt=addtxt)
        FTC = cls.load_2dvar(sdir, resdeg, tile, 'FTC', 'avg', mode=mode, fmode=fmode, addtxt=addtxt)
        Fpar = cls.load_2dvar(sdir, resdeg, tile, 'Fpar', 'avg', mode=mode, fmode=fmode, addtxt=addtxt)
        LC = cls.load_2dvar(sdir, resdeg, tile, 'LC', 'avg', mode=mode, fmode=fmode, addtxt=addtxt)    # just pick method='avg' here, doesnt influence array.
        if fmode == 'on':
            DBavg = cls.load_2dvar(sdir, resdeg, tile, 'DB', 'avg', mode=mode, fmode=fmode, addtxt=addtxt)
            DBsum = cls.load_2dvar(sdir, resdeg, tile, 'DB', 'sum', mode=mode, fmode=fmode, addtxt=addtxt)
            BAavg = cls.load_2dvar(sdir, resdeg, tile, 'BA', 'avg', mode=mode, fmode=fmode, addtxt=addtxt)
            BAsum = cls.load_2dvar(sdir, resdeg, tile, 'BA', 'sum', mode=mode, fmode=fmode, addtxt=addtxt)
        
        if fmode == 'on':
            var_deg = {'AGB':AGB, 'AGBw':AGBw, 'FL':FL, 'DBavg':DBavg, 'DBsum':DBsum, 'NPPavg':NPPavg, 'NPPsum':NPPsum, 'FTC':FTC, 'Fpar':Fpar, 'BAavg':BAavg, 'BAsum':BAsum, 'LC':LC}
        elif fmode == 'off':
            var_deg = {'AGB':AGB, 'AGBw':AGBw, 'FL':FL, 'NPPavg':NPPavg, 'NPPsum':NPPsum, 'FTC':FTC, 'Fpar':Fpar, 'LC':LC}
        
        lwmask = cls.readdata_LWMASK('LWM', resdeg, tile)
        tilemask = cls.tile_mask(resdeg, tiles, frac='no', extent=tile)[0]
        mask = lwmask * tilemask
        for key in var_deg.keys():
            var_deg[key] = var_deg[key] * mask
        
        if fmode == 'on':
            var_deg['FCsum'] = var_deg['DBsum'] * cls.load_area(resdeg, extent)[0] / var_deg['BAsum']
            var_deg['FCavg'] = var_deg['DBavg'] * cls.load_area(resdeg, extent)[0] / var_deg['BAavg']
        
        return var_deg
    
    
    @classmethod
    def load_deg_perbiome(cls, sdir, resdeg, mode, fmode='on', extent='Africa', addtxt=None):
        
        tiles = tiles_box[extent]
        
        ress = '%03.0fdeg' % float(('%0.2f' % round(resdeg, 2)).lstrip('0.'))  # converts e.g. 0.25 -> '025deg'
        
        biomes = cls.biomes_UMD.keys()[:13+1]
        
        if addtxt is None: addtxt = ''
        addtxt_temp = addtxt[:]
        
        AGB=0; AGBw=0; FL=0; NPPavg=0; NPPsum=0; FTC=0; Fpar=0; LC=0; DBavg=0; DBsum=0; BAavg=0; BAsum=0
        for biome in biomes:
            
            addtxt = addtxt_temp + 'biome%02.f' % int(biome)
            
            AGB += cls.load_2dvar(sdir, resdeg, tile, 'AGB', 'avg', mode=mode, fmode=fmode, addtxt=addtxt)
            AGBw += cls.load_2dvar(sdir, resdeg, tile, 'AGBw', 'avg', mode=mode, fmode=fmode, addtxt=addtxt)
            FL += cls.load_2dvar(sdir, resdeg, tile, 'FL', 'avg', mode=mode, fmode=fmode, addtxt=addtxt)
            NPPavg += cls.load_2dvar(sdir, resdeg, tile, 'NPP', 'avg', mode=mode, fmode=fmode, addtxt=addtxt)
            NPPsum += cls.load_2dvar(sdir, resdeg, tile, 'NPP', 'sum', mode=mode, fmode=fmode, addtxt=addtxt)
            if fmode == 'on':
                DBavg += cls.load_2dvar(sdir, resdeg, tile, 'DB', 'avg', mode=mode, fmode=fmode, addtxt=addtxt)
                DBsum += cls.load_2dvar(sdir, resdeg, tile, 'DB', 'sum', mode=mode, fmode=fmode, addtxt=addtxt)
        
        FTC = cls.load_2dvar(sdir, resdeg, tile, 'FTC', 'avg', mode=mode, fmode=fmode, addtxt=addtxt)
        Fpar = cls.load_2dvar(sdir, resdeg, tile, 'Fpar', 'avg', mode=mode, fmode=fmode, addtxt=addtxt)
        LC = cls.load_2dvar(sdir, resdeg, tile, 'LC', 'avg', mode=mode, fmode=fmode, addtxt=addtxt)  # just pick method='avg' here, doesnt influence array.
        if fmode == 'on':
            BAavg = cls.load_2dvar(sdir, resdeg, tile, 'BA', 'avg', mode=mode, fmode=fmode, addtxt=addtxt)
            BAsum = cls.load_2dvar(sdir, resdeg, tile, 'BA', 'sum', mode=mode, fmode=fmode, addtxt=addtxt)
        
        if fmode == 'on':
            var_deg = {'AGB': AGB, 'AGBw': AGBw, 'FL': FL, 'DBavg': DBavg, 'DBsum': DBsum, 'NPPavg': NPPavg, 'NPPsum': NPPsum, 'FTC': FTC, 'Fpar': Fpar, 'BAavg': BAavg, 'BAsum': BAsum, 'LC': LC}
        elif fmode == 'off':
            var_deg = {'AGB': AGB, 'AGBw': AGBw, 'FL': FL, 'NPPavg': NPPavg, 'NPPsum': NPPsum, 'FTC': FTC, 'Fpar': Fpar, 'LC': LC}

        lwmask = cls.readdata_LWMASK('LWM', resdeg, tile)
        tilemask = cls.tile_mask(resdeg, tiles, frac='no', extent=tile)[0]
        mask = lwmask * tilemask
        for key in var_deg.keys():
            var_deg[key] = var_deg[key] * mask
        
        if fmode == 'on':
            var_deg['FCsum'] = var_deg['DBsum'] * cls.load_area(resdeg, extent)[0] / var_deg['BAsum']
            var_deg['FCavg'] = var_deg['DBavg'] * cls.load_area(resdeg, extent)[0] / var_deg['BAavg']
        
        return var_deg
    
    
    @classmethod
    def translate_to_deg_perbiome(cls, var, resdeg, years, extent='Africa'):
        
        ress = '%03.0fdeg' % float(('%0.2f' % round(resdeg, 2)).lstrip('0.'))  # converts e.g. 0.25 -> '025deg'
        
        product = cls.dataset_dict[var][0]
        layer = cls.dataset_dict[var][1]
        
        if (type(years) is str) or (type(years) is np.string_): years = [years]
        
        biomes = cls.biomes_UMD.keys()[:13+1]
        tiles = tiles_box[extent]
        
        for biome in biomes:
            
            for tile in tiles:
                
                mask = []
                X = []
                for year in years:
                    # Data = cls.readdata_MODIS(var, 500, year, tile)
                    ###
                    FTC = cls.readdata_MODIS('FTC', 500, year, tile)
                    NTV = cls.readdata_MODIS('NTV', 500, year, tile)
                    Data = (FTC / (FTC+NTV)) ** 2
                    ###
                    if len(Data) == 12: monthly = True
                    else:               monthly = False
                    
                    mask = cls.biome_mask(biome, 500, year, tile)[0]
                    if len(years) == 1:
                        X = Data * mask
                    
                    elif len(years) > 1:
                        X.append(Data * mask)
                
                X = np.array(X)
                if len(years) > 1:   X = np.vstack(X)
                
                arrt = cls.translate(X, tile, resdeg)
                
                if tile == tiles[0]:
                    arr_deg = np.zeros(arrt.shape)
                
                arr_deg = arr_deg + arrt
                
            if arr_deg.ndim == 2:
                arr_deg = np.expand_dims(arr_deg, axis=0)
            indexdeg, _, _, _ = cls.load_degbox(resdeg, extent)
            arr_deg = arr_deg[:, indexdeg['lat'][0]:indexdeg['lat'][-1] + 1, indexdeg['lon'][0]:indexdeg['lon'][-1] + 1]
            
            if monthly is True:     arr_deg = np.reshape(arr_deg, (len(arr_deg) / 12, 12, arr_deg.shape[1], arr_deg.shape[2]))
            
            driver = gdal.GetDriverByName('GTiff')
            for i, year in enumerate(years):
                
                arr_degi = arr_deg[i]
                if arr_degi.ndim == 2:
                    arr_degi = np.expand_dims(arr_degi, axis=0)
                
                filename = cls.ddir + product + '_tl/' + product + '_' + var + '_' + ress + '_' + year + '_' + extent + '-biome%02.f' % int(biome)
                ds = driver.Create(filename + '.tif', arr_degi.shape[2], arr_degi.shape[1], len(arr_degi), gdal.GDT_Float32, options=['COMPRESS=LZW', 'INTERLEAVE=BAND', 'TILED=YES'])
                
                for i in range(len(arr_degi)):
                    ds.GetRasterBand(i+1).WriteArray(arr_degi[i, :, :])
                ds = None
            'File saved: %s' % filename
    
    
    @classmethod
    def translate_to_deg_old(cls, sdir, resdeg, mode, fmode='on', extent='Africa'):
        
        tiles = tiles_box[extent]
        
        #tiles = ['h20v10']
        
        for tile in tiles:
            n = cls(sdir, mode=mode, resolution=500, tile=tile, firemode=fmode, act='passive')
            #try:
            n.load(full=['AGB', 'LIT', 'AGB_fire', 'LIT_fire', 'NPP'])
            AGB = np.mean(n.diagf['AGB'], axis=0)
            FL = np.mean(n.diagf['AGB'] + n.diagf['LIT'], axis=0)
            FCavg = np.mean(n.diagf['AGB_fire'] + n.diagf['LIT_fire'], axis=0)
            NPPavg = np.mean(n.diagf['NPP'], axis=0)
            
            if mode == 1:
                FCsum = np.sum(n.diagf['AGB_fire'] + n.diagf['LIT_fire'], axis=0)   # annual sum
                NPPsum = np.sum(n.diagf['NPP'], axis=0)     # annual sum
            elif mode == 2:
                if len(n.diagf['AGB_fire']) >= 12:
                    FCsum = np.mean(np.sum(np.reshape(n.diagf['AGB_fire'] + n.diagf['LIT_fire'], (len(n.diagf['AGB_fire']) / 12, 12, 310, 310)), axis=1), axis=0)  # annual sum
                else:
                    FCsum = np.sum(n.diagf['AGB_fire'] + n.diagf['LIT_fire'], axis=0)
                
                if len(n.diagf['NPP']) >= 12:
                    NPPsum = np.mean(np.sum(np.reshape(n.diagf['NPP'], (len(n.diagf['NPP']) / 12, 12, 310, 310)), axis=1), axis=0)  # annual sum
                else:
                    NPPsum = np.sum(n.diagf['NPP'], axis=0)
                    
            n.diagf = {}
                    
                
            #except:
                # AGB = np.zeros((2400, 2400))
                # FL = np.zeros((2400, 2400))
                # FCavg = np.zeros((2400, 2400))
                # FCsum = np.zeros((2400, 2400))
                # NPPavg = np.zeros((2400, 2400))
                # NPPsum = np.zeros((2400, 2400))
            
            FTC = np.nanmean(cls.readdata_MODIS('FTC', 500, n.years, tile), axis=0)
            Fpar = np.nanmean(cls.readdata_MODIS('Fpar', 500, n.years, tile), axis=0)
            BA = cls.readdata_MODIS('BA', 500, n.years, tile) * cls.load_area(500, tile)[0]
            BA = np.mean(np.sum(np.reshape(BA, (len(n.years), 12, 2400, 2400)), axis=1), axis=0)
            
            arr = np.array([AGB, FL, FCavg, FCsum, NPPavg, NPPsum, BA, FTC, Fpar])
            arrt = cls.translate(arr, n.tile, resdeg)
            
            if tile == tiles[0]:
                arr_deg = np.zeros(arrt.shape)
                
            arr_deg = arr_deg + arrt
            
        # arr_deg = arr_deg * cls.tile_mask(res, tiles, frac='yes', extent='global')[0]
        
        indexdeg, _, _, _ = cls.load_degbox(resdeg, extent)
        arr_deg = arr_deg[:, indexdeg['lat'][0]:indexdeg['lat'][-1] + 1, indexdeg['lon'][0]:indexdeg['lon'][-1] + 1]
        
        lwmask = cls.readdata_LWMASK('LWM', resdeg, extent)
        tilemask = cls.tile_mask(resdeg, tiles, frac='no', extent=extent)[0]
        mask = lwmask * tilemask
        arr_dict = {}
        for i, var in enumerate(['AGB', 'FL', 'FCavg', 'FCsum', 'NPPavg', 'NPPsum', 'BA', 'FTC', 'Fpar']):
            arr_dict[var] = arr_deg[i] * mask
        
        return arr_dict
    
    
    @classmethod
    def load_deg_old(cls, sdir, resdeg, mode, fmode='on', extent='Africa'):

        tiles = tiles_box[extent]
        
        ress = '%03.0fdeg' % float(('%0.2f' % round(resdeg, 2)).lstrip('0.'))  # converts e.g. 0.25 -> '025deg'
        
        n = cls(sdir, mode=mode, resolution=resdeg, tile=extent, act='passive')
        
        if mode == 1:
            n.load(full=['stem', 'leaf', 'gras', 'cwd', 'litt'])
            AGB = np.mean(n.diagf['stem'] + n.diagf['leaf'], axis=0)
            FL = np.mean(n.diagf['stem'] + n.diagf['leaf'] + n.diagf['gras'] + n.diagf['cwd'] + n.diagf['litt'], axis=0)    # annual mean
            n.diagf = {}
            
            n.load(full=['stem_fire', 'leaf_fire', 'gras_fire', 'cwd_fire', 'litt_fire'])
            FCavg = np.mean(n.diagf['stem_fire'] + n.diagf['leaf_fire'] + n.diagf['gras_fire'] + n.diagf['cwd_fire'] + n.diagf['litt_fire'], axis=0)  # annual mean
            FCsum = np.sum(n.diagf['stem_fire'] + n.diagf['leaf_fire'] + n.diagf['gras_fire'] + n.diagf['cwd_fire'] + n.diagf['litt_fire'], axis=0)  # annual sum
            n.diagf = {}
            
            n.load(full=['NPP'])
            NPPavg = np.mean(n.diagf['NPP'], axis=0)   # annual mean
            NPPsum = np.sum(n.diagf['NPP'], axis=0) # annual sum
            n.diagf = {}
        
        elif mode == 2:
            if resdeg == 0.25:
                n.load(full=['stem', 'leaf', 'gras', 'cwd', 'litt'])
                AGB = np.mean(n.diagf['stem'] + n.diagf['leaf'], axis=0)
                FL = np.mean(n.diagf['stem'] + n.diagf['leaf'] + n.diagf['gras'] + n.diagf['cwd'] + n.diagf['litt'], axis=0)  # annual mean
                n.diagf = {}
            else:
                n.load(full=['AGB', 'LIT'])
                AGB = np.mean(n.diagf['AGB'], axis=0)
                FL = np.mean(n.diagf['AGB'] + n.diagf['LIT'], axis=0)  # annual mean
                n.diagf = {}
                
            n.load(full=['AGB_fire', 'LIT_fire'])
            FCavg = np.mean(n.diagf['AGB_fire'] + n.diagf['LIT_fire'], axis=0)  # annual mean
            if len(n.diagf['AGB_fire']) >= 12:
                FCsum = np.mean(np.sum(np.reshape(n.diagf['AGB_fire'] + n.diagf['LIT_fire'], (len(n.diagf['AGB_fire'])/12,12,310,310)), axis=1), axis=0)  # annual sum
            else:
                FCsum = np.sum(n.diagf['AGB_fire'] + n.diagf['LIT_fire'], axis=0)
            n.diagf = {}
            
            n.load(full=['NPP'])
            NPPavg = np.mean(n.diagf['NPP'], axis=0)    # annual mean
            if len(n.diagf['NPP']) >= 12:
                NPPsum = np.mean(np.sum(np.reshape(n.diagf['NPP'], (len(n.diagf['NPP'])/12,12,310,310)), axis=1), axis=0)   # annual sum
            else:
                NPPsum = np.sum(n.diagf['NPP'], axis=0)
            n.diagf = {}
        
        FTC = np.nanmean(cls.readdata_MODIS('FTC', resdeg, n.years, extent), axis=0).astype('float64')
        Fpar = np.nanmean(cls.readdata_MODIS('Fpar', resdeg, n.years, extent), axis=0).astype('float64')
        BA = cls.readdata_MODIS('BA', resdeg, n.years, extent) * cls.load_area(resdeg, extent)[0]
        BA = np.mean(np.sum(np.reshape(BA, (len(n.years),12,310,310)), axis=1), axis=0)
        
        LC = cls.readdata_MODIS('LC2', resdeg, n.years[len(n.years) / 2], extent)[0]
        
        var_deg = {'AGB': AGB, 'FL': FL, 'FCavg': FCavg, 'FCsum': FCsum, 'NPPavg': NPPavg, 'NPPsum': NPPsum, 'FTC': FTC, 'Fpar': Fpar, 'BA': BA, 'LC': LC}
        
        lwmask = cls.readdata_LWMASK('LWM', resdeg, extent)
        tilemask = cls.tile_mask(resdeg, tiles, frac='no', extent=extent)[0]
        mask = lwmask * tilemask
        for key in var_deg.keys():
            var_deg[key] = var_deg[key] * mask
            
        return var_deg
    


if __name__ == '__main__':    
    
    def quick_debug():
        
        foldername = 'debug_testrun/'
        sdir = Model.wdir + 'results/' + foldername
        
        time0 = timer.time()
        
        #n = Model(sdir, mode=1, resolution=500, tile='h19v09', firemode='off', act='active')
        #f = copy.deepcopy(n)
        #f.add_fire()
        n = Model(sdir, mode=1, resolution=0.25, tile='Africa') #, sampleloc=[[40.0,40.0], 200])
        print 'start'
        for step in range(n.n_years):
            time_loop = timer.time()
            for month in np.arange(0, 12):
                
                time_a = timer.time()
                n.algorithm2(step, month)
                print timer.time() - time_a
                
                time_b = timer.time()
                n.diagnose(step, month)
                print timer.time() - time_b
                
            print str(step) + '_' + str(timer.time() - time_loop)

        n.save(ext='.hdf')
        
        print 'Done, duration = %s (h,m,s)' % (timer.strftime("%H:%M:%S", timer.gmtime(timer.time() - time0)))
        
        n.plot_dict(n.times, [n.diagm_pool])
        n.plot_dict(n.times, [n.diagm_out])
        n.plot_dict(n.times, [n.diagm_fire])
    #quick_debug()
    
    
    def worker_example(tilen):
       ''' Example of a worker function as used in .multiprocessing()
       Parameters
       ----------
       :tilen: MODIS tile ID. Used as iterator in multiprocessing. Valid input = tile ID string
       '''
       try:
           txtfile = open(Model.wdir+'logs/log_'+tilen+'.txt', 'w')
           sys.stdout = txtfile
           n = Model(1, 'off', tilen, 500, 'passive')
           filename = Model.wdir+'results/mode'+str(n.mode)+'_'+str(n.n_years)+'yr_'+n.res+'_fmode-'+n.firemode+'_'+n.tile
           if not os.path.isfile(filename+'.npy'):
               del n
               n = Model(1, 'off', tilen, 500, 'active')
               print 'START: mode = %s - tile %s' % (n.mode, tilen)

               time0 = timer.time()

               for step in range(n.n_years):
                   print step
                   for month in np.arange(0,12):

                       n.algorithm(step, month)
                       n.diagnose(step, month)

               print 'Done, duration = %s (h,m,s)' % (timer.strftime("%H:%M:%S", timer.gmtime(timer.time() - time0)))  #            
               n.annual_diag()
               n.save()

           else:
               print 'Tile %s already saved, skipping' % tilen
               del n

       except KeyboardInterrupt:
           raise KeyboardInterruptError()

       finally:
           txtfile.close()

           sys.stdout.close()
           sys.stdout = sys.__stdout__         # restore stdout back to console/terminal
    #worker_example('h19v09')
