#!/usr/bin/env python2
# -*- coding: utf-8 -*-
"""
Created on Thu Nov 10 11:10:55 2016

Supportive script for Model. Contains all functions for reading-in datasets required for modelling (including pre-processed data).
Functions inside this script are imported into the main Model script.

@author: davevanwees
"""

import numpy as np
import matplotlib.pyplot as plt
import sys
import os
import glob
import time as timer
import gdal, gdalconst
import pandas as pd
import h5py
import scipy.ndimage
import tempfile
import netCDF4 as netcdf
from collections import OrderedDict
from cdo import *; cdo = Cdo()
plt.rc('image', interpolation='none')       # change default image interpolation
gdal.UseExceptions()


@classmethod
def readdata_CDO(cls, dataset, resdeg, years, tile, indexdeg=None):
    # AIRT (2 metre air temperature [C degrees Celcius]) [96x720x1440] (2000-2017)
    # SSR (Surface net solar radiation) (= incoming - outgoing longwave [W/m2]) [96x720x1440] (2000-2017)
    # CCI Soil Moisture [96x720x1440] (2000-2017)
    # SWVL1 Soil Moisture [96x720x1440] (2000-2017)

    wdir_geo = cls.wdir_geo
    ddir = cls.ddir
    dataset_dict = cls.dataset_dict
    dt = cls.modsettings['float_dtype']
    
    ress = '%03.0fdeg' % float(('%0.2f' % round(resdeg,2)).lstrip('0.'))  # converts e.g. 0.25 -> '025deg'
    
    if indexdeg is None:
        indexdeg, _, _, _ = cls.load_degbox(resdeg, tile)
    
    indexdeg_y = indexdeg['lat'] + 1;  # CDO selindexbox is 1-based
    indexdeg_x = indexdeg['lon'] + 1;
    
    fdir = dataset_dict[dataset]
    
    if (type(years) is str) or (type(years) is np.string_): years = [years]
    
    if dataset in ['AIRT', 'SSR']:
        #TODO: temporary file has to be written using 'output', otherwise there is a cdo bug regarding tmp files. Move away from cdo in future. https://code.mpimet.mpg.de/boards/1/topics/2699
        #data = cdo.remapbil(ddir+'Era-Interim/' + '025deg-'+ress+'_grid_top.txt', input='-selyear,' + years[0] + '/' + years[-1] + ' ' + ddir + fdir)   # bilinear remap from 721 to 720 lat grid. If the output stream is omitted, a temporary file is written and its name is the return value of the call.
        data = cdo.remapbil(ddir + 'Era-Interim/' + '025deg-025deg_grid_top.txt', input='-selyear,' + years[0] + '/' + years[-1] + ' ' + ddir + fdir)
        data = cdo.selindexbox(str(indexdeg_x[0]) + ',' + str(indexdeg_x[-1]) + ',' + str(indexdeg_y[0]) + ',' + str(indexdeg_y[-1]), input=data, returnCdf=True)
    
    elif dataset in ['SWVL1']:
        #data = cdo.remapbil(ddir + 'Era-Interim/' + '025deg-' + ress + '_grid_top.txt', input='-selyear,' + years[0] + '/' + years[-1] + ' ' + ddir + fdir)
        data = cdo.remapbil(ddir + 'Era-Interim/' + '025deg-025deg_grid_top.txt', input='-selyear,' + years[0] + '/' + years[-1] + ' ' + ddir + fdir)
        data = cdo.selindexbox(str(indexdeg_x[0]) + ',' + str(indexdeg_x[-1]) + ',' + str(indexdeg_y[0]) + ',' + str(indexdeg_y[-1]), input=data, returnCdf=True)
        X = data.variables['swvl1'][:].astype('float'+dt)
    
    elif dataset in ['SWVL1-Land']:
        years_rem = [x for x in years if int(x) <= 2010]  # cutoff at 2010 (later data not available)
        years_cut = list(set(years) - set(years_rem))
        if years_rem:       # years <= 2010 (remaining)
            #data = cdo.remapbil(ddir+'Era-Interim/' + '025deg-'+ress+'_grid_top.txt', input='-selyear,' + years_rem[0] + '/' + years_rem[-1] + ' ' + ddir + fdir)
            data = cdo.remapbil(ddir + 'Era-Interim/' + '025deg-025deg_grid_top.txt', input='-selyear,' + years_rem[0] + '/' + years_rem[-1] + ' ' + ddir + fdir)
            data = cdo.selindexbox(str(indexdeg_x[0]) + ',' + str(indexdeg_x[-1]) + ',' + str(indexdeg_y[0]) + ',' + str(indexdeg_y[-1]), input=data, returnCdf=True)
            X = data.variables['swvl1'][:].astype('float'+dt)
        if years_cut:       # years > 2010 (cut off)
            #data_2010 = cdo.remapbil(ddir+'Era-Interim/' + '025deg-'+ress+'_grid_top.txt', input='-selyear,' + '2010/2010' + ' ' + ddir + fdir)
            data_2010 = cdo.remapbil(ddir + 'Era-Interim/' + '025deg-025deg_grid_top.txt', input='-selyear,' + '2010/2010' + ' ' + ddir + fdir)
            data_2010 = cdo.selindexbox(str(indexdeg_x[0]) + ',' + str(indexdeg_x[-1]) + ',' + str(indexdeg_y[0]) + ',' + str(indexdeg_y[-1]), input=data_2010, returnCdf=True)
            X_rep = np.tile(data_2010.variables['swvl1'][:].astype('float'+dt), (len(years_cut),1,1))
            if not years_rem:
                X = X_rep.copy()
        if years_rem and years_cut:
            X = np.concatenate((X, X_rep), axis=0)
    
    elif dataset in ['CCI']:
        data = cdo.selyear(years[0] + '/' + years[-1], input=ddir + fdir)
        data = cdo.selindexbox(str(indexdeg_x[0]) + ',' + str(indexdeg_x[-1]) + ',' + str(indexdeg_y[0]) + ',' + str(indexdeg_y[-1]), input=data, returnCdf=True)
    
    if dataset == 'AIRT':
        X = data.variables['t2m'][:].astype('float'+dt) - 273
        lat = data.variables['lat'][:]
        lon = data.variables['lon'][:]
    elif dataset == 'SSR':    
        X = data.variables['ssr'][:].astype('float'+dt)/(3600.0*24)    # Convert J/m2 to W/m2 (This Era data is accumulated over 24 hours, so 3600*24 seconds)
    #elif dataset == 'SWVL1':  X = data.variables['swvl1'][:]
    elif dataset == 'CCI':    X = data.variables['sm'][:].astype('float'+dt).filled(np.nan)         # read as masked array, so refill with zeros
    
    return X 


@classmethod
def readdata_MODIS(cls, dataset, res, years, tile, indexdeg=None, sizedeg=None):
    # FTC (MOD44Bv51) (Fraction Tree Cover) [8x720x1440] (2000-2016)
    # NTV (MOD44Bv51) (Non-Tree Vegetation) [8x720x1440] (2000-2016)
    # FPAR (MCD15A2H) (Fraction of Absorbed Photosynthetically Active Radiation) [90x720x1440] (2002-2017) 
    # BA Burned Area (MCD64A1) [96x720x1440] (2000-2017)
    # LC1 Biomes, Land Cover Type 1 (MCD12Q1) (2001-2013) - dominant biome

    wdir_geo = cls.wdir_geo
    ddir = cls.ddir
    dataset_dict = cls.dataset_dict
    latlon_mask = cls.latlon_mask
    dt = cls.modsettings['float_dtype']
    
    if res == 500:     ress = '500m'
    elif type(res) == float:
        ress = '%03.0fdeg' % float(('%0.2f' % round(res,2)).lstrip('0.'))      # converts e.g. 0.25 -> '025deg'

        if indexdeg is None or sizedeg is None:
            indexdeg, sizedeg, _, _ = cls.load_degbox(res, tile)
    
    '''Add part where boreal savannah is replaced with boreal forest (according to Yannis, because of similar soil moisture characteristics)'''

    product = dataset_dict[dataset][0]
    layer = dataset_dict[dataset][1]
    fdir = ddir + product + '_tl/'
    if product == 'MCD12Q1' and dataset == 'LC1':
        fdir = ddir + product + '_tl_IGBP/'
    if product == 'MCD12Q1' and dataset == 'LC2':
        fdir = ddir + product + '_tl_UMD/'
    
    X = []

    if (type(years) is str) or (type(years) is np.string_): years = [years]
    for folder in years:

        make_zero_array = False
        if dataset == 'Fpar':
            if int(folder) < 2003: folder = '2003'  # only first half of 2002 is not available
        elif dataset in ['LC1', 'LC2']:
            if int(folder) < 2001:
                folder = '2001'  # use 2001 for pre-2001 years
            elif int(folder) > 2013:
                folder = '2013'  # use 2013 for post-2013 years
        elif dataset in ['FTC', 'NTV']:
            if int(folder) > 2016:
                folder = '2016'     # use 2016 for post-2016 years

        if type(res) == float:
            filen = fdir + product + '_' + dataset + '_' + ress + '_' + folder
        elif res == 500:
            filen = fdir + folder + '/' + product + '_' + dataset + '_' + ress + '_' + folder + '_' + tile

        if res == 500 and dataset == 'BA':
            try:
                ds = gdal.Open(filen + '.tif', gdalconst.GA_ReadOnly)
            except Exception, e:  # catch all errors and return error type as 'e'
                print 'Error %s, Tile %s not found for dataset %s, skipping tile, using zeros array' % (e, tile, dataset)
                make_zero_array = True
        else:
            ds = gdal.Open(filen + '.tif', gdalconst.GA_ReadOnly)

        if make_zero_array == True:
            # In case gdal is not able to read file, a zero array is made instead, and an error is returned.
            # Ideally, an error message is send to parent and the worker is killed, starting up the next tile in the pool.
            # However, killing a worker while maintaining the other workers and continuing the pool, is not possible in the multiprocessing module.
            if type(res) == float:
                Data = np.zeros((12, sizedeg['lat'], sizedeg['lon']), dtype='float'+dt)
            elif res == 500:
                Data = np.zeros((12, 2400, 2400), dtype='float'+dt)
        else:
            if type(res) == float:
                Data = ds.ReadAsArray(yoff=indexdeg['lat'][0], xoff=indexdeg['lon'][0], ysize=sizedeg['lat'], xsize=sizedeg['lon']).astype('float'+dt)
            elif res == 500:
                Data = ds.ReadAsArray().astype('float'+dt)
        
        # ''' New '''
        # NoDataValue = ds.GetRasterBand(1).GetNoDataValue()
        # Data[Data == NoDataValue] = 0
        # ''' End '''
        
        ds = None
        
        if dataset == 'Fpar':
            Data[Data == 255] = 0
        if dataset in ['FTC', 'NTV', 'Fpar']:
            Data = Data / 100.0
        if dataset == 'FRP':
            Data = Data / 10.0
        if dataset in ['LC1', 'LC2']:       # LC1: IGBP  /   LC2: UMD
            Data = Data.astype(int)
            
            if dataset == 'LC1':
                sparse_boreal_forest = 17
                tundra = 18
            elif dataset == 'LC2':            # LC2 ==> UMD
                if type(res) == float:
                    Data[Data == 255] = 16      # replace some 255 values in antartica with barren.
                elif res == 500:
                    Data[Data == 255] = -2      # unclassified.
                Data[Data == 12] = 11  # IGBP to UMD numbering: croplands 12->11
                Data[Data == 13] = 12  # IGBP to UMD numbering: urban 13->12
                Data[Data == 16] = 13  # IGBP to UMD numbering: barren 16->13            
                sparse_boreal_forest = 14  # UMD: 14   /   IGBP: 17
                tundra = 15               # UMD: 15   /   IGBP: 18
            
            
            def conver(ind):
                return 90 - 0.25 * ind - 0.125      # Dit moet 0.25 blijven!, niet generaliseren naar degrees.
            
            maskVS8 = latlon_mask([conver(172), 90.0], [-180.0, -25.0], res, tile) # VS
            maskEU8 = latlon_mask([conver(140), 90.0], [-25.0, 40.0], res, tile)   # EU
            maskRU8 = latlon_mask([conver(185), 90.0], [40.0, 180.0], res, tile)   # RU
            mask170 = latlon_mask([conver(170), 90.0], [-180.0, 180.0], res, tile)
            maskAL10 = latlon_mask([conver(150), 90.0], [-180.0, -140.0], res, tile) # Alaska
            maskVS10 = latlon_mask([conver(138), 90.0], [-140.0, -25.0], res, tile)  # VS
            maskEU10 = latlon_mask([conver(105), 90.0], [-25.0, 40.0], res, tile)    # EU
            maskRU10 = latlon_mask([conver(115), 90.0], [40.0, 180.0], res, tile) # RU
            
            Data[Data * maskVS8 == 8] = 1   # 8=woody savanna
            Data[Data * maskEU8 == 8] = 1   # 8=woody savanna
            Data[Data * maskRU8 == 8] = 1  # 8=woody savanna
            Data[Data * mask170 == 9] = sparse_boreal_forest  # 9=savanna
            Data[Data * mask170 == 7] = sparse_boreal_forest  # 7=open shrubland
            Data[Data * maskAL10 == 10] = tundra    # grassland
            Data[Data * maskVS10 == 10] = tundra  # grassland
            Data[Data * maskEU10 == 10] = tundra  # grassland
            Data[Data * maskRU10 == 10] = tundra  # grassland
            
        
        if res == 500 and dataset == 'BA':
            Data[Data < 0] = 0
            Data[Data > 0] = 1.0
        
        
        if dataset in ['FTC', 'NTV', 'LC1', 'LC2', 'LC', 'NPP']:      # expand time dimension for yearly datasets.
            Data = np.expand_dims(Data, axis=0)
        
        if len(years) == 1:
            X = Data
        elif len(years) > 1:
            X.append(Data)
    
    del Data
    
    X = np.array(X)
    if len(years) > 1:
        X = np.vstack(X)
    
    return X


@classmethod
def readdata_LCperc(cls, dataset, res, years, tile, indexdeg=None, sizedeg=None):
    # LC1perc, LC2 Biomesperc, Land Cover Type 1 - 2 (MCD12Q1) (2001-2013) - percentual biome
    
    ddir = cls.ddir
    dataset_dict = cls.dataset_dict
    dt = cls.modsettings['float_dtype']
    
    ress = '%03.0fdeg' % float(('%0.2f' % round(res, 2)).lstrip('0.'))  # converts e.g. 0.25 -> '025deg'

    if indexdeg is None or sizedeg is None:
        indexdeg, sizedeg, _, _ = cls.load_degbox(res, tile)
    
    if 'perc' in dataset:
        dataset = dataset.replace('perc', '')       # remove 'perc' from dataset name.    
    
    product = dataset_dict[dataset][0]
    if product == 'MCD12Q1' and dataset == 'LC1':
        fdir = ddir + product + '_tl_IGBP/'
    if product == 'MCD12Q1' and dataset == 'LC2':
        fdir = ddir + product + '_tl_UMD/'
    #X = {typen: [] for typen in range(0,17)}
    X = OrderedDict([typen, []] for typen in range(0,17))

    if (type(years) is str) or (type(years) is np.string_): years = [years]
    
    for folder in years:
        
        if int(folder) < 2001:   folder = '2001'     # use 2001 for pre-2001 years
        elif int(folder) > 2013:  folder = '2013'     # use 2013 for post-2013 years
        
        #filen = fdir + product + '_' + dataset + '_' + ress + '_' + folder + '_types'
        #ds = gdal.Open(filen + '.tif', gdalconst.GA_ReadOnly)
        #ds = ds.ReadAsArray(yoff=indexdeg['lat'][0], xoff=indexdeg['lon'][0], ysize=sizedeg['lat'], xsize=sizedeg['lon']).astype('float' + dt)
        
        for typen in range(0,17):
            ''' Old '''
            filen = fdir + product + '_' + dataset + '_types_' + str(typen) + '_' + ress + '_' + folder
            ds = gdal.Open(filen+'.tif',gdalconst.GA_ReadOnly)
            Data = ds.ReadAsArray(yoff=indexdeg['lat'][0], xoff=indexdeg['lon'][0], ysize=sizedeg['lat'], xsize=sizedeg['lon']).astype('float' + dt)
            ds = None
            X[typen].append(Data)
            ''' End '''
            
            #Data = ds[typen]
            #X[typen].append(Data)
            
    ds = None
    del Data
    
    if dataset == 'LC2':
        del X[11]       # IGBP empty in UMD numbering
        del X[14]       # IGBP empty in UMD numbering
        del X[15]       # IGBP empty in UMD numbering
        
        X[11] = X.pop(12)   # IGBP to UMD numbering: croplands 12->11
        X[12] = X.pop(13)   # IGBP to UMD numbering: urban 13->12
        X[13] = X.pop(16)   # IGBP to UMD numbering: barren 16->13
    
    for typen in X.keys():
        X[typen] = np.array(X[typen])
    
    if dataset == 'LC2':
        X[14] = np.zeros((X[0].shape))  # temporary solution, set new boreal biomes as zero.
        X[15] = np.zeros((X[0].shape))
    
    if dataset == 'LC': # NEW. Fill unprocessed tiles with water.
        X[0] = X[0] + 1 - np.sum(X, axis=0)
    
    return X


@classmethod
def readdata_MODISperc(cls, dataset, res, years, tile):
    
    ddir = cls.ddir
    dataset_dict = cls.dataset_dict
    
    if res == 500:     ress = '500m'
    elif type(res) == float:
        ress = '%03.0fdeg' % float(('%0.2f' % round(res,2)).lstrip('0.'))      # converts e.g. 0.25 -> '025deg'
    
    if 'perc' in dataset:
        dataset = dataset.replace('perc', '')       # remove 'perc' from dataset name.
    
    product = dataset_dict[dataset][0]
    
    fdir = ddir + product + '_tl/'
    
    biomes = range(0,13+1)
    X = OrderedDict([typen, []] for typen in biomes)
    
    if (type(years) is str) or (type(years) is np.string_): years = [years]
    for folder in years:
        
        filen = fdir + product + '_' + dataset + '_' + ress + '_' + folder
        
        for biome in biomes:
            filename = filen + '_Africa-biome%02.f' % int(biome)
            ds = gdal.Open(filename + '.tif', gdalconst.GA_ReadOnly)
            Data = ds.ReadAsArray()
            
            if dataset in ['FTC', 'NTV', 'LC1', 'LC2', 'NPP', 'MORT']:  # expand time dimension for yearly datasets.
                Data = np.expand_dims(Data, axis=0)
            
            if len(years) == 1:
                X[biome] = Data
            elif len(years) > 1:
                X[biome].append(Data)
        
    for typen in X.keys():
        X[typen] = np.array(X[typen])
        if len(years) > 1:
            X[typen]= np.vstack(X[typen])
    
    return X


@classmethod
def readdata_LWMASK(cls, dataset, res, tile, frac='no', indexdeg=None, sizedeg=None):
    # LWM, Land-water mask (MOD44W)
    # LWM2, Land-water mask (ISLSCP II)

    wdir_geo = cls.wdir_geo
    ddir = cls.ddir
    dataset_dict = cls.dataset_dict
    dt = cls.modsettings['float_dtype']
    if res == 500:
        ress = '500m'
    elif type(res) == float:
        ress = '%03.0fdeg' % float(('%0.2f' % round(res,2)).lstrip('0.'))  # converts e.g. 0.25 -> '025deg'
    
    if (indexdeg is None or sizedeg is None) and type(res) == float:
        indexdeg, sizedeg, _, _ = cls.load_degbox(res, tile)
    
    if dataset == 'LWM':
        product = dataset_dict[dataset][0]
        fdir = ddir + product + '_tl/'
        folder = '2000-2015'        # lwmask product v6, average over 2000-2015.
        if type(res) == float:
            filen = fdir + product + '_' + dataset + '_' + ress + '_' + folder
            ds = gdal.Open(filen + '.tif', gdalconst.GA_ReadOnly)
            X = ds.ReadAsArray(yoff=indexdeg['lat'][0], xoff=indexdeg['lon'][0], ysize=sizedeg['lat'], xsize=sizedeg['lon']).astype('float' + dt)
        elif res == 500:
            filen = fdir + folder + '/' + product + '_' + dataset + '_' + ress + '_' + folder + '_' + tile
            ds = gdal.Open(filen + '.tif', gdalconst.GA_ReadOnly)
            X = ds.ReadAsArray().astype('float'+dt)
        
        if type(res) == float:
            X[:,-1] = 0
            #X[X < 0.03] = np.nan  # With cutoff = 0.03, LW and GFED masks are roughly equal.
            #X[X >= 0.03] = 1
        elif res == 500:
            ''' OLD '''
            #X = 1 - X
            ''' End '''
            #X[X < 0.1] = 0.0  # Consider less than 0.5 land as water.
            #X[X >= 0.1] = 1.0  # Consider more than 0.5 land as land.
        if frac == 'no':
            X[X > 0] = 1            # consider all partial land as land.
        else:
            X[X < 0.1] = 0         # if frac is 'yes', keep fractional land, but consider <10% land as water. To avoid spikes when dividing.
        X[X == 0.0] = np.nan  # Consider water as nan.
        
    elif dataset == 'LWM2':
        X = pd.read_csv(ddir + 'MOD44W_tl/ANCILLARY_1200/data/land_water_masks_xdeg/land_water_mask_qd.asc', delim_whitespace=True, skiprows=6, nrows=720, header=None).values[:, :].astype('float')
        # X[X==0] = np.nan
        # X[X<1.0] = 1.0          # Consider partial land (0.25, 0.5, 0.75) as land.
        if type(res) == float:
            X = X[indexdeg['lat'][0]:indexdeg['lat'][-1] + 1, indexdeg['lon'][0]:indexdeg['lon'][-1] + 1]
    
    return X


@classmethod
def readdata_Hansen(cls, dataset, res, years, tile, indexdeg=None, sizedeg=None):
    
    wdir_geo = cls.wdir_geo
    ddir = cls.ddir
    dataset_dict = cls.dataset_dict
    dt = cls.modsettings['float_dtype']
    
    if res == 500:     ress = '500m'
    elif type(res) == float:
        ress = '%03.0fdeg' % float(('%0.2f' % round(res,2)).lstrip('0.'))      # converts e.g. 0.25 -> '025deg'
        
    if indexdeg is None or sizedeg is None:
        indexdeg, sizedeg, _, _ = cls.load_degbox(res, tile)

    product = dataset_dict[dataset][0]
    layer = dataset_dict[dataset][1]
    fdir = ddir + product + '/' + layer + '_tl/'
    
    X = []

    if (type(years) is str) or (type(years) is np.string_): years = [years]
    for folder in years:
        
        if res == 500:
            filen = fdir + folder + '/' + product + '_' + layer + '_' + ress + '_' + folder + '_' + tile

        ds = gdal.Open(filen + '.tif', gdalconst.GA_ReadOnly)
        Data = ds.ReadAsArray().astype('float' + dt)
        ds = None

        Data = np.expand_dims(Data, axis=0)
        
        if len(years) == 1:
            X = Data
        elif len(years) > 1:
            X.append(Data)

    del Data

    X = np.array(X)
    if len(years) > 1:
        X = np.vstack(X)

    return X
        

@classmethod
def readdata_WURbiomass(cls, res, tile):

    ddir = cls.ddir
    wdir_geo = cls.wdir_geo
    grid_conv = cls.grid_conv
    load_degbox = cls.load_degbox
    
    fdir = ddir + 'WUR_ForestBiomass/GEOCARBON_Global_Forest_Biomass/GEOCARBON_Global_Forest_AGB_10072015'
    print 'loading WUR Forest biomass dataset ... ',

    ds = gdal.Open(fdir + '.tif', gdalconst.GA_ReadOnly)
    geot = ds.GetGeoTransform()
    X = ds.ReadAsArray().astype('float')
    X[X < 0] = 0
    X = X[1:-1, :]
    lat = geot[3] + geot[5]/2 + geot[5] * np.arange(X.shape[0])
    lon = geot[0] + geot[1]/2 + geot[1] * np.arange(X.shape[1])
    
    if res == 0.01 or res == 500:

        X = np.concatenate((np.zeros((24 * 25, 36000)), X), axis=0)  # add missing latitudes above and below.
        X = np.concatenate((X, np.zeros((136 * 25, 36000))), axis=0)
        lat_before = 90 - 0.005 + geot[5] * np.arange(24 * 25)
        lat_after = -56 - 0.005 + geot[5] * np.arange(136 * 25)
        lat = np.append(lat_before, lat, axis=0)
        lat = np.append(lat, lat_after, axis=0)
        del lat_before, lat_after
        
        if tile != 'global':
            # overlap_dict = np.load(wdir_geo + '00_Pixel_location_001deg_extremes_top.npy')[()]  # degree grid cells that fall in MODIS tile.
            # overlap = overlap_dict[tile]
            # index001deg = {}
            # index001deg['lat'] = np.arange(overlap['lat'][0], overlap['lat'][1] + 1)
            # index001deg['lon'] = np.arange(overlap['lon'][0], overlap['lon'][1] + 1)
            
            index001deg, _, _, _ = load_degbox(0.01, tile)
            
            if res == 0.01:
                X = X[index001deg['lat'][0]: index001deg['lat'][-1] + 1, index001deg['lon'][0]: index001deg['lon'][-1] + 1]

            elif res == 500:
                ilatc_ar, ilonc_ar = grid_conv(tile, 0.01, index001deg)
                X = X[index001deg['lat'][0] + ilatc_ar, index001deg['lon'][0] + ilonc_ar]

    else:
        dsample = int(res / 0.01)   # downsampling factor
        X = X.reshape((X.shape[0] // dsample, dsample, X.shape[1] // dsample, dsample)).mean(3).mean(1)
        
        X = np.concatenate((np.zeros((int(6/res), int(360/res))), X), axis=0)       # Add top 6 degrees and bottom 34 degrees latitude.
        X = np.concatenate((X, np.zeros((int(34/res), int(360/res)))), axis=0)
        lat = np.arange(90 - res / 2.0, -90, -res)  # cell midpoint indices
        lon = np.arange(-180 + res / 2.0, 180, res)
        
        if tile != 'global':
            indexdeg, _, _, _ = load_degbox(res, tile)
            X = X[indexdeg['lat'][0]: indexdeg['lat'][-1] + 1, indexdeg['lon'][0]: indexdeg['lon'][-1] + 1]

    # Model units: [gC/m2] -> [Mg/ha], 1E-6/1E-4 = /100 -> *2 = /50
    # WUR units: [Mg/ha] -> [gC/m2], 1E6/1E4 = *100 -> /2 = *50       units: Aboveground biomass density Mg/ha
    # And additionally a factor 0.5 for carbon per mass
    X *= 50         # transform from Mg/ha to gC/m2
    
    print 'Done.'
    return X


@classmethod
def readdata_WURsoil(cls, res, tile):
    
    ddir = cls.ddir
    
    fdir = ddir + 'WUR_Wise30sec_v1/Interchangeable_format/wise_30sec_v1'
    print 'loading WUR soil biomass dataset ... ',

    ds = gdal.Open(fdir + '.tif', gdalconst.GA_ReadOnly)
    geot = ds.GetGeoTransform()
    X = ds.ReadAsArray().astype('float')
    
    lat = geot[3] + geot[5]/2 + geot[5] * np.arange(X.shape[0])
    lon = geot[0] + geot[1]/2 + geot[1] * np.arange(X.shape[1])
    
    print 'debug'
    
    return X, geot


@classmethod
def readdata_BouvetAGB(cls, res, tile):
    
    ddir = cls.ddir
    wdir_geo = cls.wdir_geo
    grid_conv = cls.grid_conv
    load_degbox = cls.load_degbox
    
    fdir = ddir + 'Bouvet_AGB/AGB_Africa_nomask_uint16_0.05deg'
    print 'loading Bouvet AGB dataset ... ',
    
    ds = gdal.Open(fdir + '.tif', gdalconst.GA_ReadOnly)
    geot = ds.GetGeoTransform()
    X = ds.ReadAsArray().astype('float')
    lat = geot[3] + geot[5] / 2 + geot[5] * np.arange(X.shape[0])
    lon = geot[0] + geot[1] / 2 + geot[1] * np.arange(X.shape[1])
    
    lat_before = lat[0] + 0.05 - geot[5] * np.arange(2 * 4 * 5)[::-1]   # [::-1], flip 1d array
    lat_after = lat[-1] - 0.05 + geot[5] * np.arange(2.5 * 4 * 5)
    lat = np.append(lat_before, lat, axis=0)
    lat = np.append(lat, lat_after, axis=0)
    
    lon_before = lon[0] - 0.05 - geot[1] * np.arange(2 * 4 * 5)[::-1]   # [::-1], flip 1d array
    lon_after = lon[-1] + 0.05 + geot[1] * np.arange(5.5 * 4 * 5)
    lon = np.append(lon_before, lon, axis=0)
    lon = np.append(lon, lon_after, axis=0)
    
    X = np.concatenate((np.zeros((int(2 / 0.05), X.shape[1])), X), axis=0)  # Add top 2 degrees and bottom 2.5 degrees latitude.
    X = np.concatenate((X, np.zeros((int(2.5 / 0.05), X.shape[1]))), axis=0)
    
    X = np.concatenate((np.zeros((X.shape[0], int(2 / 0.05))), X), axis=1)  # Add left 2 degrees and right 5.5 degrees longitude.
    X = np.concatenate((X, np.zeros((X.shape[0], int(5.5 / 0.05)))), axis=1)
    
    if res == 0.05 or res == 500:
        
        if tile != 'Africa':
            
            index005deg, _, _, _ = load_degbox(0.05, tile)
            index005deg['lat'] -= cls.region_box['Africa'][0][0] * int(0.25/0.05)
            index005deg['lon'] -= cls.region_box['Africa'][1][0] * int(0.25/0.05)
            
            if res == 0.05:
                X = X[index005deg['lat'][0]: index005deg['lat'][-1] + 1, index005deg['lon'][0]: index005deg['lon'][-1] + 1]
                
            elif res == 500:
                ilatc_ar, ilonc_ar = grid_conv(tile, 0.05, index005deg)
                X = X[index005deg['lat'][0] + ilatc_ar, index005deg['lon'][0] + ilonc_ar]
    
    else:
        dsample = int(res / 0.05)   # downsampling factor
        X = X.reshape((X.shape[0] // dsample, dsample, X.shape[1] // dsample, dsample)).mean(3).mean(1)
        
        if tile != 'Africa':
            indexdeg, _, _, _ = load_degbox(res, tile)
            indexdeg['lat'] -= cls.region_box['Africa'][0][0] * int(0.25/res)
            indexdeg['lon'] -= cls.region_box['Africa'][1][0] * int(0.25/res)
            X = X[indexdeg['lat'][0]: indexdeg['lat'][-1] + 1, indexdeg['lon'][0]: indexdeg['lon'][-1] + 1]
    
    X *= 50         # transform from t/ha to gC/m2
    
    print 'Done.'
    return X


@classmethod
def loadGFED(cls, res, layer, years, tile):
    ''' Loads GFED4s data into Model instance, for comparison with model results
    Parameters
    ----------
    :layer: GFED4s dataset name as string. Valid input: 'BB', 'NPP', 'Rh', 'burned_fraction', 'source'
    :years: begin and end year of required GFED dataset. Valid input: list of years from instance variable self.years, or manual list of years [num or string]
    '''
    load_degbox = cls.load_degbox
    
    ddir = '/Volumes/Mac_HD/Work/Data/GFED4s/'

    if layer in ['BB', 'NPP', 'Rh', 'burned_fraction', 'source', 'C', 'DM', 'small_fire_fraction']:
        if layer in ['BB', 'NPP', 'Rh']:                group = 'biosphere'
        if layer in ['burned_fraction', 'source']:      group = 'burned_area'
        if layer in ['C', 'DM', 'small_fire_fraction']: group = 'emissions'
        GFED = np.zeros((0, 720, 1440))
        if (type(years) is str) or (type(years) is np.string_): years = [years]
        print 'loading GFED4s layer %s for years %s - %s ... ' % (layer, years[0], years[-1]),
        for folder in years:
            filen = ddir + 'GFED4.1s_' + str(folder)
            ds = h5py.File(filen + '.hdf5', 'r')
            for m in range(12):
                Data = ds[group]['%02d' % (m + 1)][layer][:]
                GFED = np.append(GFED, [Data], axis=0)
            ds = None
    if layer in ['basis_regions', 'grid_cell_area']:
        years = '2010'  # these layers are time-independent, so just pick a year.
        filen = ddir + 'GFED4.1s_' + str(years)
        ds = h5py.File(filen + '.hdf5', 'r')
        GFED = ds['ancill'][layer][:]
        ds = None
        GFED = scipy.ndimage.zoom(GFED, 0.25/res, order=0)  # order=0 is nearest-neighbor.
        
    if tile != 'global':
        indexdeg, _, _, _ = load_degbox(res, tile)
        if layer in ['BB', 'NPP', 'Rh', 'burned_fraction', 'source', 'C', 'DM', 'small_fire_fraction']:
            GFED = GFED[:, indexdeg['lat'][0]: indexdeg['lat'][-1] + 1, indexdeg['lon'][0]: indexdeg['lon'][-1] + 1]
        if layer in ['basis_regions', 'grid_cell_area']:
            GFED = GFED[indexdeg['lat'][0]: indexdeg['lat'][-1] + 1, indexdeg['lon'][0]: indexdeg['lon'][-1] + 1]
    #print 'Done'
    return GFED


def read_normal(self, dataset, years):
    
    if (type(years) is str) or (type(years) is np.string_): years = [years]
    #if self.resolution == 500:
    #    print '-> Loading %s datasets for year %s ... ' % (dataset, years),
    
    time0 = timer.time()
    
    if dataset in ['AIRT', 'SSR', 'CCI', 'SWVL1', 'SWVL1-Land']:
        if self.resolution == 500:  restemp = 0.25
        else: restemp = float(self.resolution)    # makes copy.
        X = self.readdata_CDO(dataset, restemp, years, self.tile, indexdeg=self.index025deg)
    
    if dataset in ['LWM', 'LWM2']:
        X = self.readdata_LWMASK(dataset, self.resolution, self.tile, indexdeg=self.indexdeg, sizedeg=self.sizedeg)
    
    if dataset in ['FTC', 'NTV', 'Fpar', 'BA', 'LC1', 'LC2', 'LC']:
        X = self.readdata_MODIS(dataset, self.resolution, years, self.tile, indexdeg=self.indexdeg, sizedeg=self.sizedeg)
    
    if dataset in ['LC1perc', 'LC2perc']:
        X = self.readdata_LCperc(dataset, self.resolution, years, self.tile, indexdeg=self.indexdeg, sizedeg=self.sizedeg)
    
    if dataset in ['BAperc', 'FTCperc', 'NTVperc', 'Fparperc', 'MORTperc']:
        X = self.readdata_MODISperc(dataset, self.resolution, years, self.tile)
        LC2perc = self.readdata_LCperc('LC2perc', self.resolution, years, self.tile, indexdeg=self.indexdeg, sizedeg=self.sizedeg)
        X = np.array(X.values()) / np.array(LC2perc.values()[:13+1])
        X[np.isnan(X)] = 0
        X[X > 1] = 1

    ### Modified Burned Fraction (MBF)
    if (dataset == 'BA') and (self.mbf == 'on') and (self.resolution != 500):
        
        BA_prev = self.readdata_MODIS('BA', self.resolution, str(int(years[0]) - 1), self.tile, indexdeg=self.indexdeg, sizedeg=self.sizedeg)   # load previous year.
        BA = np.vstack((BA_prev, X))
        def rolling_sum(a, n):
            ret = np.nancumsum(a, axis=0, dtype=float)
            ret[n:, :, :] = ret[n:, :, :] - ret[:-n, :, :]
            return ret[n - 1:, :, :]
        BAsum14 = rolling_sum(BA, n=4)[12 - 4:12+len(X) - 4]
        
        MBF = X / (1 - BAsum14)
        MBF[BAsum14 > 1] = X[BAsum14 > 1].copy()  # if MBF > 1, use normal BF.
        X = MBF.copy()
    
    if (dataset == 'BAperc') and (self.mbf == 'on') and (self.resolution != 500):
        
        BA_prev = self.readdata_MODISperc('BA', self.resolution, str(int(years[0]) - 1), self.tile)     # load previous year.
        LC2perc = self.readdata_LCperc('LC2perc', self.resolution, str(int(years[0]) - 1), self.tile, indexdeg=self.indexdeg, sizedeg=self.sizedeg)
        BA_prev = np.array(BA_prev.values()) / np.array(LC2perc.values()[:13+1])
        BA_prev[np.isnan(BA_prev)] = 0
        for biome in range(0,13+1):
            BA = np.vstack((BA_prev[biome], X[biome]))
            def rolling_sum(a, n):
                ret = np.nancumsum(a, axis=0, dtype=float)
                ret[n:, :, :] = ret[n:, :, :] - ret[:-n, :, :]
                return ret[n - 1:, :, :]
            BAsum14 = rolling_sum(BA, n=4)[12 - 4:12+len(X[biome]) - 4]
            
            MBF = X[biome] / (1 - BAsum14)
            MBF[X[biome] >= (1 - BAsum14)] = 1.0      # NOTE!!!: not equivalent to MBF[MBF>1] = 1.0, because it also sets MBF to 1 when BAsum14>1.
            #biomefraction = self.readdata_LCperc('LC2', self.resolution, years, self.tile, indexdeg=self.indexdeg, sizedeg=self.sizedeg)
            #MBF[BAsum14 > biomefraction[biome]] = X[biome][BAsum14 > biomefraction[biome]].copy()    # if MBF > biome fraction, use normal BF.
            #MBF[BAsum14 >= 1] = X[biome][BAsum14 >= 1].copy()
            X[biome] = MBF.copy()
    ###

    ### New mortality
    if dataset == 'MORT':
        FTC = self.readdata_MODIS('FTC', self.resolution, years, self.tile, indexdeg=self.indexdeg, sizedeg=self.sizedeg)
        NTV = self.readdata_MODIS('NTV', self.resolution, years, self.tile, indexdeg=self.indexdeg, sizedeg=self.sizedeg)
        X = (FTC / (FTC + NTV)) ** 2
    if dataset == 'MORTperc2':
        FTC = self.readdata_MODISperc('FTC', self.resolution, years, self.tile)
        NTV = self.readdata_MODISperc('NTV', self.resolution, years, self.tile)
        biomes = range(0,13+1)
        X = OrderedDict()
        for i, biome in enumerate(biomes):
            X[biome] = (FTC[i] / (FTC[i] + NTV[i])) ** 2
    ###
    
    if dataset in ['BAperc', 'FTCperc', 'NTVperc', 'Fparperc', 'MORTperc', 'MORTperc2']:
        biomes = range(0,13+1)
        X_temp = OrderedDict()
        for i, biome in enumerate(biomes):
            X_temp[biome] = X[i]
        X = X_temp.copy()
    
    #if self.resolution == 500:
    #    print 'Done, duration = %s (h,m,s)' % (timer.strftime("%H:%M:%S", timer.gmtime(timer.time() - time0)))
    
    return X


def read_clima(self, dataset, years):
    
    '''
    self vars: resolution, res, years
    '''

    dt = self.modsettings['float_dtype']

    if (type(years) is str) or (type(years) is np.string_): years = [years]
    if self.resolution == 500:
        print '-> Loading %s datasets for year %s ... ' % (dataset, years) ,
    
    time0 = timer.time()
    save_clima = False
    
    for i, folder in enumerate(years):
        if (type(folder) is str) or (type(folder) is np.string_): folder = [folder]
        
        if dataset in ['AIRT', 'SSR', 'CCI', 'SWVL1', 'SWVL1-Land']:
            if self.resolution == 500:  restemp = 0.25
            else:   restemp = float(self.resolution)    # makes copy.
            Data = self.readdata_CDO(dataset, restemp, folder, self.tile, indexdeg=self.index025deg)
        
        if dataset in ['LWM', 'LWM2']:
            X = self.readdata_LWMASK(dataset, self.resolution, self.tile, indexdeg=self.indexdeg, sizedeg=self.sizedeg)
            break
        
        if dataset in ['LC1', 'LC2', 'LC']:       # for landcover mode 1, simply pick the middle year. No averaging.
            X = self.readdata_MODIS(dataset, self.resolution, years[len(years) / 2], self.tile, indexdeg=self.indexdeg, sizedeg=self.sizedeg)
            break
        
        if dataset in ['LC1perc', 'LC2perc']:
            X = self.readdata_LCperc(dataset, self.resolution, years[len(years) / 2], self.tile, indexdeg=self.indexdeg, sizedeg=self.sizedeg)
            break
            
        if dataset in ['FTC', 'NTV', 'Fpar', 'BA']:
            # if (self.resolution == 500) and (dataset in ['Fpar', 'BA']):      # look for saved climatology.   
            #     clima_name = 'clima_%s-%s_%s_%s' % (years[0], years[-1], self.tile, dataset)
            #     if os.path.isfile(self.wdir + 'results/Climatologies/' + clima_name + '.tif'):
            #         print 'Climatology found - ' + clima_name + ': loading ... ',
            #         ds = gdal.Open(self.wdir + 'results/Climatologies/' + clima_name + '.tif', gdalconst.GA_ReadOnly)
            #         X = ds.ReadAsArray()
            #         break
            #     else:
            #         save_clima = True       # passed below for saving climatology.
            #         print 'Climatology not found - ' + clima_name + ': calculating ... ',
            # TODO: fix .tif reading when resolution = 500.
            Data = self.readdata_MODIS(dataset, self.resolution, folder, self.tile, indexdeg=self.indexdeg, sizedeg=self.sizedeg)
        
        if dataset in ['BAperc', 'FTCperc', 'NTVperc', 'Fparperc', 'MORTperc']:
            Data = self.readdata_MODISperc(dataset, self.resolution, folder, self.tile)
            LC2perc = self.readdata_LCperc('LC2perc', self.resolution, folder, self.tile, indexdeg=self.indexdeg, sizedeg=self.sizedeg)
            Data = np.array(Data.values()) / np.array(LC2perc.values()[:13+1])
            Data[np.isnan(Data)] = 0
            Data[Data > 1] = 1          # Should not be necessary; by definition all values should be <= 1.
        
        ### Modified Burned Fraction (MBF)
        if (dataset == 'BA') and (self.mbf == 'on') and (self.resolution != 500):
            
            BA_prev = self.readdata_MODIS('BA', self.resolution, str(int(folder[0]) - 1), self.tile, indexdeg=self.indexdeg, sizedeg=self.sizedeg)
            BA = np.vstack((BA_prev, Data))
            def rolling_sum(a, n):              # rolling sum with window size = 4
                ret = np.nancumsum(a, axis=0)
                ret[n:,:,:] = ret[n:,:,:] - ret[:-n,:,:]
                return ret[n - 1:,:,:]
            BAsum14 = rolling_sum(BA, n=4)[12-4:24-4]
            
            MBF = Data / (1 - BAsum14)
            MBF[BAsum14 > 1] = Data[BAsum14 > 1].copy()     # if MBF > 1, use normal BF.
            Data = MBF.copy()

        if (dataset == 'BAperc') and (self.mbf == 'on') and (self.resolution != 500):

            BA_prev = self.readdata_MODISperc('BA', self.resolution, str(int(folder[0]) - 1), self.tile)  # load previous year.
            LC2perc = self.readdata_LCperc('LC2perc', self.resolution, str(int(folder[0]) - 1), self.tile, indexdeg=self.indexdeg, sizedeg=self.sizedeg)
            BA_prev = np.array(BA_prev.values()) / np.array(LC2perc.values()[:13+1])
            BA_prev[np.isnan(BA_prev)] = 0
            for biome in range(0,13+1):
                BA = np.vstack((BA_prev[biome], Data[biome]))
                def rolling_sum(a, n):
                    ret = np.nancumsum(a, axis=0, dtype=float)
                    ret[n:, :, :] = ret[n:, :, :] - ret[:-n, :, :]
                    return ret[n - 1:, :, :]
                BAsum14 = rolling_sum(BA, n=4)[12-4:24- 4]
                
                MBF = Data[biome] / (1 - BAsum14)
                MBF[Data[biome] >= (1 - BAsum14)] = 1.0  # NOTE!!!: not equivalent to MBF[MBF>1] = 1.0, because it also sets MBF to 1 when BAsum14>1.
                #biomefraction = self.readdata_LCperc('LC2', self.resolution, folder, self.tile, indexdeg=self.indexdeg, sizedeg=self.sizedeg)
                #MBF[BAsum14 > biomefraction[biome][0]] = Data[biome][BAsum14 > biomefraction[biome][0]].copy()   # if MBF > biome fraction, use normal BF.
                #MBF[BAsum14 >= 1] = Data[biome][BAsum14 >= 1].copy()
                Data[biome] = MBF.copy()
        ###
        
        ### New mortality
        if dataset == 'MORT':
            FTC = self.readdata_MODIS('FTC', self.resolution, folder, self.tile, indexdeg=self.indexdeg, sizedeg=self.sizedeg)
            NTV = self.readdata_MODIS('NTV', self.resolution, folder, self.tile, indexdeg=self.indexdeg, sizedeg=self.sizedeg)
            Data = (FTC / (FTC+NTV)) ** 2
        if dataset == 'MORTperc2':
            FTC = self.readdata_MODISperc('FTC', self.resolution, folder, self.tile)
            FTC = np.array(FTC.values())
            NTV = self.readdata_MODISperc('NTV', self.resolution, folder, self.tile)
            NTV = np.array(NTV.values())
            Data = (FTC / (FTC + NTV)) ** 2
        ###
        
        if i == 0:
            X = np.zeros(Data.shape, dtype='float'+dt)
        
        X = X + Data / len(years)       # This calculates the climatology stepwise.
        
        # if save_clima:       # saving climatology.
        #     driver = gdal.GetDriverByName('GTiff')
        #     ds = driver.Create(self.wdir + 'results/Climatologies/' + clima_name + '.tif', X.shape[2], X.shape[1], 12, gdal.GDT_Float32, options=['COMPRESS=LZW', 'INTERLEAVE=BAND', 'TILED=YES'])
        #     for i in range(12):
        #         ds.GetRasterBand(i + 1).WriteArray(X[i, :, :])
    
    if dataset in ['BAperc', 'FTCperc', 'NTVperc', 'Fparperc', 'MORTperc', 'MORTperc2']:
        biomes = range(0,13+1)
        X_temp = OrderedDict()
        for i, biome in enumerate(biomes):
            X_temp[biome] = X[i]
        X = X_temp.copy()
        
    if self.resolution == 500:
        print 'Done, duration = %s (h,m,s)' % (timer.strftime("%H:%M:%S", timer.gmtime(timer.time() - time0)))

    return X


def main_read(self, years, calcmode):
    
    time1 = timer.time()
    print
    if type(self.resolution) == float:
        print '-> Loading datasets for years %s ... ' % (years) ,
    
    if calcmode == 'clima':
        temp_func = read_clima
    if calcmode == 'normal':
        temp_func = read_normal
    
    self.AIRT = temp_func(self, 'AIRT', years)
    self.SSR = temp_func(self, 'SSR', years)
    if self.modsettings['soil_moisture'] == 'CCI':          self.SOILM = temp_func(self, 'CCI', years)
    elif self.modsettings['soil_moisture'] == 'SWVL1':      self.SOILM = temp_func(self, 'SWVL1', years)
    elif self.modsettings['soil_moisture'] == 'SWVL1-Land': self.SOILM = temp_func(self, 'SWVL1-Land', years)
    
    self.FTC = temp_func(self, 'FTC', years)
    self.NTV = temp_func(self, 'NTV', years)
    self.Fpar = temp_func(self, 'Fpar', years)
    self.BA = temp_func(self, 'BA', years)
    self.LC = temp_func(self, 'LC', years).astype('Int8')
    self.LWMASK = temp_func(self, 'LWM', years)
    if self.resolution != 500:
        pass
        #self.LCperc   = temp_func(self, 'LC2perc', years) 
        #self.BAperc   = temp_func(self, 'BAperc', years)
        #self.FTCperc  = temp_func(self, 'FTCperc', years)
        #self.NTVperc  = temp_func(self, 'NTVperc', years)
        #self.Fparperc = temp_func(self, 'Fparperc', years)
        
        #self.mortality4 = np.zeros(self.FTC.shape) # placeholder.
        #self.mortality4perc = temp_func(self, 'MORTperc', years)
    
    ''' Calculation of time-averaged (mode 1 = climatology average, mode 2 = year average) wetlands.'''
    
    if self.modsettings['soil_moisture'] == 'SWVL1-Land':
        soilm = self.SOILM.copy()
        soilm[soilm > 0.5] = 0.5
        soilm = soilm / 0.5
        soilm = (soilm - 0.1) / 0.9
        soilm[soilm < 0.1] = 0.1
    elif self.modsettings['soil_moisture'] == 'SWVL1':
        soilm = self.SOILM.copy()
        soilm[soilm > 0.37] = 0.37
        soilm = soilm / 0.37
        soilm = (soilm - 0.4) / 0.6
        soilm[soilm < 0.1] = 0.1
    
    self.wetlands = np.mean(soilm, 0) > 0.90            # moisture > 0.9 for time average means its a wetland.
    self.turnover_slow_map = np.ones((self.size025deg['lat'], self.size025deg['lon'])) * self.modparams['turnover_slow']  # make map
    self.turnover_slow_map[self.wetlands] = self.modparams['turnover_slow'] * 0.1  # create slower rate for wetlands
    
    if self.act == 'active':
        if type(self.resolution) == float:
            self.fire_month = np.zeros([self.sizedeg['lat'], self.sizedeg['lon']], dtype='uint8')
        elif self.resolution == 500:
            self.fire_month = np.zeros([2400, 2400], dtype='uint8')
    
    if type(self.resolution) == float:
        print 'Done, duration = %s (h,m,s)' % (timer.strftime("%H:%M:%S", timer.gmtime(timer.time() - time1)))


if __name__ == '__main__':
    
    pass
