!>
!! @file ocprod.f90
!! @brief compute biological production, settling of debris, and related biogeochemistry
!!
!! Notes:
!!
!!    - kchck=1 can be used to check max/min of bgc arrays on wet/dry cells.
!!
!!    - prosil is used only for k=1,2. It is adressed, however, for
!!      k=1,4 in loop 100. To save memory,  js:note seems dated???
!!
!!    - nocetra is the number of all BGC elements (size of ocetra(,,,l))
!!
!! TODO: nosin* and nodel* are calculated for nothing.
!!
!! called by bgc
!!
!! @author Ernst Maier-Reimer, MPI-Met, HH
!!
!! @par Revision History
!!
!! First version by Ernst Maier-Reimer     (MPI-M)     Apr 10, 2001
!!
!! Revised version by Stephanie Legutke    (MPI-M)     Apr 10, 2001
!!
!! Revised version by S.Lorenz/JO.Beismann (MPI-M,NEC) Aug 24, 2007
!!     - OpenMP parallel
!!
!! Revised version by Rene Redler          (MPI-M)     Apr 25, 2013
!!     - code cleaning, refactored for ICON
!!
!! Revised version by Malte Heinemann      (CAU)       Dec 18, 2016
!!     - added particle ballasting according to 
!!       Gehlen et al., 2006 + LGM dust for ballasting and iron 
!!
!! @par Copyright
!! 2002-2013 by MPI-M
!! This software is provided for non-commercial use only.
!! See the LICENSE and the WARRANTY conditions.
!!
SUBROUTINE OCPROD ( kpie, kpje, kpke, ptho, pddpo, pdpio )

  USE mo_kind, ONLY           : wp
  USE mo_carbch, ONLY         : n2budget, h2obudget,  &
       &                        dustdep, dusty, dusty_lgm, dmspar, satoxy, wpoc, &
       &                        wball, wballc, wballo, wballd, &
       &                        wballuncor, wball, wballdd, &
       &                        wball_poc, wball_cal, wball_opa, wball_dus
  USE mo_planetary_constants, ONLY : rhoref_water
  USE mo_mpi, ONLY            : stop_all
  USE mo_commo1, ONLY         : l_cpl_dust,rhoo,dzw
  USE mo_sedmnt, ONLY         : prorca, prcaca, silpro, produs
  USE mo_biomod, ONLY         : sinkspeed_opal, sinkspeed_cal, sinkspeed_poc,      &
       &                        perc_diron, phytomi, grami, rnoi, riron, pi_alpha, &
       &                        fpar, strahl, bkphy,grazra, bkzoo, epsher,         &
       &                        zinges, drempoc, ro2ut, remido, dyphy, spemor,     &
       &                        gammaz, gammap, ecan, rnit, ropal, bkopal,         &
       &                        rcalc, rcar, dremopal, relaxfe, fesoly,            &
       &                        expoor, expoca, exposi, dphymor, dzoomor,          &
       &                        dremdoc, denitrification, nitdem, dremn2o,         &
       &                        n2prod, sulfate_reduction, kbo, wdust,             &
       &                        remido_cya, dremdoc_cya, Pfac_cya, thresh_aerob,   &
       &                        thresh_sred
  USE mo_param1_bgc, ONLY     : ifdust, idms, icalc, iopal, ian2o, igasnit, &
       &                        iiron, iphy, izoo, isilica, iphosph, &
       &                        iano3, ioxygen, idet, idoc, isco212, &
       &                        ialkali, idoccya

  USE mo_control_bgc, ONLY    : dtb, dtbgc, kchck, icyclibgc, l_cyadyn,l_alkcorr

  USE mo_bgc_diagnostic, ONLY : fill_profile, calc_omz_depth
  USE mo_bgc_diagnostic, ONLY : bgcflux, bgcprod, bgc_o_pro, ocetra
  USE mo_bgc_diagnostic, ONLY : lbgc_diag
  USE mo_bgc_diagnostic, ONLY : kfeatm, kwrbioz, kflim, knlim, kplim, &
       &                        kdmsprod, kdms_bac, kdms_uv, kexport, &
       &                        kn2fix, klysokl, kpho, kpdy_deep, kzdy_deep, &  
       &                        kdelcar, kdelsil, kphosy, kgraz,      &
       &                        krem,kexudp,kpdy,kbacr,kexudz,kzdy,   &
       &                        kremeu, kdenit, kremnit, kremsulf,kbacr_cya
  USE mo_bgc_diagnostic, ONLY : io_stdo_bgc, station_profile, prof,   &
       &                        n90depth, kcoex90, kcaex90, kopex90,  &
       &                        n1000depth, kcoex1000, kopex1000, kcaex1000, &
       &                        n2000depth, kcoex2000, kopex2000, kcaex2000, &
       &                        kprorca, kprcaca, ksilpro, kprodus,kaou

  USE mo_swr_absorption, ONLY : swr_frac  ! 3d array that holds the available light fraction ;
                                          ! (formerly abs_bgc) ; now calculated in mpiom mo_swr_absorption

  IMPLICIT NONE

 !! Arguments

  INTEGER, INTENT(in)  :: kpie                  !< 1st REAL of model grid.
  INTEGER, INTENT(in)  :: kpje                  !< 2nd REAL of model grid.
  INTEGER, INTENT(in)  :: kpke                  !< 3rd (vertical) REAL of model grid.

  REAL(wp), INTENT(in) :: ptho (kpie,kpje,kpke) !< potential temperature [deg C]
  REAL(wp), INTENT(in) :: pddpo(kpie,kpje,kpke) !< size of scalar grid cell (3rd dimension) [m]
  REAL(wp), INTENT(in) :: pdpio(kpie,kpje,kpke) !< inverse thickness of grid cell (3rd dimension)[m]

  !! Local variables

  INTEGER  :: i,j,k,n
  REAL(wp) :: dmsp1,dmsp2,dmsp3,dmsp4,dmsp5,dmsp6
  REAL(wp) :: avphy,avanut,avanfe,xa,xn,ya,yn,phosy,                   &
       &      pho, phofa, temfa,                                       &
       &      avgra,grazing,avsil,graton,		                       &
       &      gratpoc,grawa,bacfra,phymor,zoothresh,zoomor,excdoc,exud,&
       &      export, delsil, delcar, sterph, sterzo, remin,	       &
       &      docrem, opalrem, remin2o, aou, refra,                    &
       &      docrem_cya, bacfra_cya
  REAL(wp) :: r_remin, r_bacfra, r_bacfra_cya, tpremin
  REAL(wp) :: rcyano, avoxy
  REAL(wp) :: sgn_alk,ralk_corr
  REAL(wp) :: wopal,wcal
  REAL(wp) :: dustinp, dustinp_lgm, dustinp_volcano
  REAL(wp) :: dmsprod,dms_bac,dms_uv
  INTEGER :: volcano
#ifdef __c_isotopes
  REAL(wp) :: detref, detrl, rem13,rem14
  REAL(wp) :: rl13, rl14
  REAL(wp) :: rocean13, rocean14, flui13, flui14
  REAL(wp) :: fcal13,fcal14
  REAL(wp) :: d13C, d14C
#endif
#ifdef AGG
  REAL(wp) :: fphy                       ! sinking speed wphy * concentration of phytoplankton (for time series only?)
  REAL(wp) :: wmass(kpie,kpje,kpke)      ! sinking speed for 'mass of aggregates'
  REAL(wp) :: wnumb(kpie,kpje,kpke)      ! sinking speed for 'numbers of aggregates'
  REAL(wp) :: aggregate(kpie,kpje,kpke)  ! aggregation (should be renamed)
  REAL(wp) :: dustagg(kpie,kpje,kpke)    ! aggregation of dust

  !TODO: code calculating nosin*, nodel* and avsize, avsmin1, avsmax1 can be erased as this is no longer used.
  REAL(wp) :: avmass, avsize, avsmin1, avsmax1, avnos, anosloss
  ! nosin = nos increase, nosde = nos decrease
  INTEGER  :: nosin1, nosde1, nosin2, nosde2, nosin3, nosde3
  REAL(wp) :: zmornos, avm, avn, eps, e1, e2, e3, e4, es1, es3
  REAL(wp) :: TopM, TopF, snow, fshear, sagg1, sagg2, sagg4
  REAL(wp) :: sett_agg, shear_agg, effsti, dfirst, dshagg, dsett
  REAL(wp) :: checksize, nacheck, flar1, flar2, flar3
  REAL(wp) :: fTSFac, fTMFac, fTopF, fTopM, wphy, wphyup, wnosup, wnos

#endif
#ifdef BALLASTING
  REAL(wp) :: rhosw  ! seawater density for particle ballasting [g m-3]
  REAL(wp) :: rhox,rhoxp,rhoxc,rhoxo,rhoxd ! excess density [g m-3] (and only POC, +calcite, +opal, +dust)
  REAL(wp) :: wball0 ! sinking speed for POC, without calcite, opal or dust [m/day] * timestep dtb [days] => sinking distance of POC [m]
  REAL(wp) :: pdus   ! critical POC concentration, above which dust is included in ballasting [kmolP/m3]
#endif BALLASTING
  !
  !----------------------------------------------------------------------
  !
  ! volcano=0 no dust from volcano
  ! volcano=1 ashes from volcano in dustinp_volcano
  !
  volcano = 0 
  
  ! check if cyanobacteria are switched on
  rcyano=merge(1._wp,0._wp,l_cyadyn)
  ! when l_alkcorr = true, alkalinty sign fix is active
  ralk_corr=merge(1._wp,0._wp,l_alkcorr)
  sgn_alk=merge(1._wp,-1._wp,l_alkcorr)
  !
  ! Constant parameters
  !
  ! parameter definition in beleg_bgc.f90

  CALL maschk(kpie,kpje,kpke,0)

  dmsp6=dmspar(6)
  dmsp5=dmspar(5)
  dmsp4=dmspar(4)
  dmsp3=dmspar(3)
  dmsp2=dmspar(2)
  dmsp1=dmspar(1)

  !-OLD-OMP PARALLEL PRIVATE(i,j,k,l,dustinp,dustinp_volcano,                 &
  !-OLD-OMP   avphy,avanut,avanfe,xa,xn,phosy,volcell,avsil,avgra,ya,yn,      &
  !-OLD-OMP   grazing,graton,gratpoc,grawa,bacfra,phymor,                     &
  !-OLD-OMP   zoomor,excdoc,exud,export,delsil,delcar,dmsprod,dms_bac,dms_uv, &
  !-OLD-OMP   sterph,sterzo,remin,docrem,opalrem,aou,refra,remin2o,           &
  !-OLD-OMP   pho,phofa,temfa,fopa,fdet,fcal                                  &
#ifdef __c_isotopes
  !-OLD-OMP  ,rem13,rem14,rl13,rl14,rocean13,rocean14,           &
  !-OLD-OMP   flui13,flui14,fcal13,fcal14,d13C,d14C,detref,detrl &
#endif
#ifdef AGG
  !-OLD-OMP   ,fphy,avmass, avsize, avsmin1, avsmax1, avnos, anosloss , &
  !-OLD-OMP    nosin1, nosde1, nosin2, nosde2, nosin3, nosde3,          &
  !-OLD-OMP    zmornos, avm, avn, eps, e1,e2,e3,e4,es1,es3,             &
  !-OLD-OMP    TopM, TopF, snow,fshear,sagg1,sagg2,sagg4,               &
  !-OLD-OMP    sett_agg,shear_agg,effsti,dfirst,dshagg,dsett,           &
  !-OLD-OMP    checksize,nacheck,flar1,flar2,flar3,                     &
  !-OLD-OMP    fTSFac,fTMFac,fTopF,fTopM,wphy,wphyup,wnosup,wnos        &
#endif
  !-OLD-OMP    )

  wopal = sinkspeed_opal * dtb
  wcal  = sinkspeed_cal  * dtb

  ! dust flux from the atmosphere to the surface layer;
  ! dust fields are monthly mean values in units of kg/m2/year
  ! dissolved iron is a fixed fraction (3.5%), and immediately released
  ! 1% of the iron input is dissolved [='bio-available'] (see perc_diron in beleg_bgc)

  dustinp_volcano = 0._wp

  !-OLD-OMP DO
  DO j = 1, kpje
     DO i = 1, kpie
        IF (pddpo(i, j, 1) > 0.5_wp) THEN   ! wet point
           IF (l_cpl_dust) THEN
              dustinp = dustdep(i, j) / 365._wp * dtb * pdpio(i, j, 1)
              IF (volcano.EQ.1) THEN
                 dustinp_volcano = dustdep(i, j) / 365._wp * dtb * pdpio(i, j, 1)
                 dustinp         = dusty(i, j)   / 365._wp * dtb * pdpio(i, j, 1)
              ENDIF
           ELSE
              ! modern or LGM dust deposition for ballasting and iron
              dustinp = dusty(i, j) / 365._wp * dtb * pdpio(i, j, 1)
              dustinp_lgm = dusty_lgm(i, j) / 365._wp * dtb * pdpio(i, j, 1)
           ENDIF
           ! modern dust:
           !ocetra(i,j,1,ifdust) = ocetra(i,j,1,ifdust) + dustinp + dustinp_volcano ! use present-day dust for ballasting
           !bgcflux(i,j,kfeatm) = dustinp*perc_diron*pddpo(i,j,1)/dtbgc  ! bio available iron kmol/(m**2 s)
           !ocetra(i,j,1,iiron)  = ocetra(i,j,1,iiron) + dustinp*perc_diron  &! perc_diron =.035*.01 /55.85 ! why *0.8 in IPCC_HAM?
           !     &                                     + dustinp_volcano*perc_diron*0.8_wp/3.5_wp  ! volcanic ash has lower iron content
           !
           ! or LGM dust:
           ocetra(i,j,1,ifdust) = ocetra(i,j,1,ifdust) + dustinp_lgm + dustinp_volcano
           bgcflux(i,j,kfeatm) = dustinp_lgm*perc_diron*pddpo(i,j,1)/dtbgc  ! bio available iron kmol/(m**2 s)
           ocetra(i,j,1,iiron)  = ocetra(i,j,1,iiron) + dustinp_lgm*perc_diron  &! perc_diron =.035*.01 /55.85 ! why *0.8 in IPCC_HAM?
                &                                     + dustinp_volcano*perc_diron*0.8_wp/3.5_wp  ! volcanic ash has lower iron content
        ENDIF
     ENDDO
  ENDDO

#ifdef BALLASTING
  ! ballasting according to Gehlen et al. 2006 (excess density)
  ! rhox=(mass of particles/volume of water) [g/m3] / 
  ! POC==det
  ! ocetra for POC and calcite [molC/m3 water], for opal [molSi/m3 water], and for ifdust [kg/m3 water]
  wball0 = sinkspeed_poc * dtb ! distance [m] POC sinks per bgc timestep (dtb in days) 

  ! Assume that dust is only aggregated where POC > pdus [kmolP/m3]
  ! pdus=0.5e-9_wp ! default
  pdus=0._wp ! always include dust in ballasting
  ! pdus=999._wp ! no effect of dust on ballasting

  !-OLD-OMP DO
  DO k = 1, kpke
     DO j = 1, kpje
        DO i = 1, kpie
         IF (pddpo(i,j,k) > 0.5_wp) THEN               ! wet cell
           ! excess density rhox = sum(molarmass*concentration)/sum(molarmass/density*concentration)-rho_sw
           ! densities for POC=1.06e6g/m3, calcite=2.71e6g/m3, opal=2.10e6g/m3 
           ! molar masses for POC=32.7g/mol, calcite=100g/mol, opal=72.8g/mol
           ! stoichiometric ratios from Takahashi et al. 1985 => C:P=122; see mo_beleg_bgc.f90 
           ! DUST: assume it is mostly quartz (SiO2) with a density of 2.6e6g/m3 (like claydens; see also https://www.mineralienatlas.de)
           !       while really it is a mix with, e.g., illite (2.79e6g/m3), kaolinite (2.68e6g/m3), calcite... 
           !       compare Table 6 in Pye (1987) "Aeolian Dust and Dust Deposits"

           ! make sure POC concentrations are positive
           IF (ocetra(i,j,k,idet) < 0._wp) THEN
                   WRITE(0,*) "BALLASTING WARNING: POC < 0; setting to 0"
                   WRITE(0,*) "i,j,k,det:",i,j,k,ocetra(i,j,k,idet)
                   ocetra(i,j,k,idet) = 0._wp
           ENDIF

           ! make sure dust concentrations are positive
           IF (ocetra(i,j,k,ifdust) < 0._wp) THEN
                   WRITE(0,*) "BALLASTING WARNING: fdust < 0; setting to 0"
                   WRITE(0,*) "i,j,k,fdust:",i,j,k,ocetra(i,j,k,ifdust)
                   ocetra(i,j,k,ifdust) = 0._wp
           ENDIF


           ! use in-situ density
           IF (rhoo(i,j,k) > 1.0599e3_wp) THEN
                   call stop_all('BALLASTING: in situ density > rho_POC => negative wball; must use rhoref_water.')
           ENDIF
           rhosw=rhoo(i,j,k)*1.e3_wp ! [g m-3]
           ! OR: use seawater reference density
           ! rhosw=rhoref_water*1.e3_wp ! [g m-3]

           ! hypothetical excess density and sinking speed, assuming POC and calcite are the only particles
           rhoxc= ( 3989.4_wp*ocetra(i,j,k,idet)+100.0_wp*ocetra(i,j,k,icalc) )*1.e6_wp &
            &    /( 3763.6_wp*ocetra(i,j,k,idet)+ 36.9_wp*ocetra(i,j,k,icalc) )  &
            &    - rhosw
           wballc(i,j,k)  = wball0 * rhoxc/(1.06e6_wp-rhosw) ! sinking speed assuming POC and calcite are only particles
           ! hypothetical excess density and sinking speed, assuming POC and opal are the only particles
           rhoxo= ( 3989.4_wp*ocetra(i,j,k,idet)+72.8_wp*ocetra(i,j,k,iopal) )*1.e6_wp &
            &    /( 3763.6_wp*ocetra(i,j,k,idet)+34.7_wp*ocetra(i,j,k,iopal) )  &
            &    - rhosw
           wballo(i,j,k)  = wball0 * rhoxo/(1.06e6_wp-rhosw) ! sinking speed assuming POC and opal are only particles
           ! if POC concentration is high enough, dust is assumed to join the aggregate formation
           IF (ocetra(i,j,k,idet) > pdus) THEN
             ! hypothetical excess density and sinking speed, assuming POC and dust are the only particles
             rhoxd= ( 3989.4_wp*ocetra(i,j,k,idet)+ocetra(i,j,k,ifdust) )*1.e6_wp &
              &    /( 3763.6_wp*ocetra(i,j,k,idet)+ocetra(i,j,k,ifdust)/2.6_wp )  &
              &    - rhosw
             wballd(i,j,k)  = wball0 * rhoxd/(1.06e6_wp-rhosw) ! sinking speed assuming POC and dust are only particles
             ! actual excess density, including dust:
             rhox=  ( 3989.4_wp*ocetra(i,j,k,idet)+100.0_wp*ocetra(i,j,k,icalc)+72.8_wp*ocetra(i,j,k,iopal)+ocetra(i,j,k,ifdust) )*1.e6_wp &
              &    /( 3763.6_wp*ocetra(i,j,k,idet)+ 36.9_wp*ocetra(i,j,k,icalc)+34.7_wp*ocetra(i,j,k,iopal)+ocetra(i,j,k,ifdust)/2.6_wp )  &
              &    - rhosw
           ELSE
             rhoxd=0._wp ! if POC does not meet dust, and POC and dust are the only particles, then 
                         ! there is no hypothetical excess density or aggregate formation
             wballd(i,j,k)  = wball0
             ! actual excess density, not including dust:
             ! Assuming that POC is still affected by calcite and opal. Is that OK?
             ! Better to assume that POC meets nothing, if POC concentration is low?
             rhox= ( 3989.4_wp*ocetra(i,j,k,idet)+100.0_wp*ocetra(i,j,k,icalc)+72.8_wp*ocetra(i,j,k,iopal) )*1.e6_wp &
              &    /( 3763.6_wp*ocetra(i,j,k,idet)+ 36.9_wp*ocetra(i,j,k,icalc)+34.7_wp*ocetra(i,j,k,iopal) )  &
              &    - rhosw
           ENDIF
           ! make sure excess density is neither too large (larger than the
           ! density of the heaviest component of the aggregate)...
           IF (rhox > 2.72e6_wp) THEN
                   WRITE(0,*) "BALLASTING WARNING: rhox too large despite postitive POC!"
                   WRITE(0,*) "i,j,k,wball,det,cal,opa,dus,rhoxc,rhoxo,rhoxd,rhox,rhoo:"
                   WRITE(0,*)  i,j,k,wball(i,j,k),ocetra(i,j,k,idet),ocetra(i,j,k,icalc),ocetra(i,j,k,iopal),ocetra(i,j,k,ifdust), &
            &                  rhoxc,rhoxo,rhoxd,rhox,rhoo(i,j,k)
                   call stop_all('BALLASTING: rhox still too large.')
           ENDIF
           ! ... nor negative
           IF (rhox < 0._wp) THEN
                   WRITE(0,*) "BALLASTING WARNING: rhox < 0"
                   WRITE(0,*) "i,j,k,wball,det,cal,opa,dus,rhoxc,rhoxo,rhoxd,rhox,rhoo:"
                   WRITE(0,*)  i,j,k,wball(i,j,k),ocetra(i,j,k,idet),ocetra(i,j,k,icalc),ocetra(i,j,k,iopal),ocetra(i,j,k,ifdust), &
            &                  rhoxc,rhoxo,rhoxd,rhox,rhoo(i,j,k)
                   call stop_all('BALLASTING: negative excess density.')
           ENDIF

           ! compute the sinking speed wball according to Gehlen et al. 2006 + dust effect
           wballuncor(i,j,k) = wball0 * rhox /(1.06e6_wp-rhosw) ! sinking speed of POC, calcite, opal and, if POC high enough, dust
                                                                ! uncorrected for too high speeds (>dzw allowed)
           wball(i,j,k)  = wballuncor(i,j,k)                    ! make sure sinking distance is smaller than grid box height dzw
                                                                ! else, mass loss possible
           IF ( wball(i,j,k) > dzw(k) ) THEN
                   ! for wball0=3.5, this happens a lot, but only in upper 8
                   ! layers / not much below euphotic zone
                   !WRITE(0,*) "BALLASTING WARNING: wball > dzw; set to dzw"
                   !WRITE(0,*)  i,j,k,wball(i,j,k),dzw(k)
                   wball(i,j,k) = dzw(k)
           ENDIF

           ! dust sinks at wball if POC concentration is high enough
           IF (ocetra(i,j,k,idet) > pdus) THEN
                   wballdd(i,j,k) = wball(i,j,k)
           ELSE
                   ! using dust diameter / Stoke's equation, see
                   ! mo_beleg_bgc.f90
                   wballdd(i,j,k) = wdust
           ENDIF

         ENDIF !wet cell
        ENDDO
     ENDDO
  ENDDO

  DO k = 1, kpke
     DO j = 1, kpje
        DO i = 1, kpie
          wball_poc(i,j,k)=wball(i,j,k)
          wball_cal(i,j,k)=wball(i,j,k)
          wball_opa(i,j,k)=wball(i,j,k)
          wball_dus(i,j,k)=wballdd(i,j,k) ! should be =wball, since pdus=0
        ENDDO
     ENDDO
  ENDDO
 



#endif BALLASTING


#ifdef AGG
  !***********************************************************************
  !
  !  special resetting for particle numbers, that sets their concentration
  !  (particles per volume, ocetra(inos)) depending on the mass of marine snow:
  !
  !  Compartments have already been set to 0 in
  !  ADVECTION_BGC.h and OCTDIFF_BGC.h. js: ???
  !
  !  Ensure that if there is no mass, there are no particles, and
  !  that the number of particles is in the right range (this is crude, but
  !  is supposed to be needed only due to numerical errors such as truncation or
  !  overshoots during advection)
  !
  ! (1) avnos<<avmass, such that eps = FractDim + 1: increase numbers
  !     such that eps = FractDim + 1 + safe (currently set to 1.e-6 in BELEG_BGC)
  !
  ! (2) avnos>>avmass, such that  Nbar (=Mass/Nos/cellmass) <=1: decrease numbers
  !     such that Nbar=1.1 (i.e. 1.1 cells per aggregate, set in BELEG_BGC)

  DO k = 1, kpke
     DO j = 1, kpje
        DO i = 1, kpie

           IF (pddpo(i,j,k) > 0.5) THEN               ! wet cell

              avmass = ocetra(i,j,k,iphy) + ocetra(i,j,k,idet)
              snow   = avmass*1.e+6                   ! why *1.e6??

              ! look for max. and min average size = Nbar
              IF (ocetra(i,j,k,inos) > 0.) THEN

                 avsize  = snow/ocetra(i,j,k,inos)/cellmass   ! js: not further used other than below 2 lines
                 avsmin1 = MIN(avsize,avsmin1)                ! js: not used
                 avsmax1 = MAX(avsize,avsmax1)                ! js: not used

              ENDIF

              ! check whether the numbers have to be decreased or increased

              IF (snow*pupper > ocetra(i,j,k,inos)) THEN
                 nosin1 = nosin1 + 1                                  ! counter 'nos increase' (not further used), not set to 0.
              ENDIF

              IF (snow*plower < ocetra(i,j,k,inos)) THEN
                 nosde1 = nosde1 + 1                                  ! counter 'nos decrease' (not further used)
              ENDIF

              ocetra(i,j,k,inos) = MAX( snow*pupper, ocetra(i,j,k,inos))
              ocetra(i,j,k,inos) = MIN( snow*plower, ocetra(i,j,k,inos))    !js (MAX/MIN correct?)

           ENDIF     ! endif wet cell

        ENDDO
     ENDDO
  ENDDO

#endif  /*AGG*/

  CALL maschk(kpie,kpje,kpke,101)
! write(0,*)'here we go'
  !
  ! Biological productivity in the euphotic zone (upper 90m)
  !
  !-OLD-OMP DO
  DO j = 1, kpje
     DO k = 1, kwrbioz
        DO i = 1, kpie

           IF (pddpo(i, j, k) > 0.5_wp) THEN

#ifdef AGG
              avmass = ocetra(i,j,k,iphy)+ocetra(i,j,k,idet)
#endif /*AGG*/


#ifdef __c_isotopes
              rocean13 = ocetra(i,j,k,isco213)/ocetra(i,j,k,isco212)     ! "ratio oceanic 13C/12C" (at least in carchm)
              rocean14 = ocetra(i,j,k,isco214)/ocetra(i,j,k,isco212)     ! ratio oceanic 14C/12C
#endif
              avphy  = MAX(phytomi,ocetra(i,j,k,iphy))                   ! 'available' phytoplankton
              avgra  = MAX(grami,ocetra(i,j,k,izoo))                     ! 'available' zooplankton
              avsil  = MAX(0._wp, ocetra(i,j,k,isilica))                 ! available silicate
              avanut = MAX(0._wp, MIN(ocetra(i,j,k,iphosph),        &    ! available nutrients (phosphate   [kmol P /m3]
                   &          rnoi*ocetra(i,j,k,iano3)))                 !                     + nitrate)
              avanfe = MAX(0._wp, MIN(avanut,ocetra(i,j,k,iiron)/riron)) ! available iron

              IF (lbgc_diag) THEN
                 IF (avanfe == ocetra(i,j,k,iiron)/riron) THEN
                    bgcprod(i,j,k,kflim) = 1._wp
                    bgcprod(i,j,k,knlim) = 0._wp
                    bgcprod(i,j,k,kplim) = 0._wp
                 ELSE IF (avanfe == avanut) THEN
                    IF (ocetra(i,j,k,iphosph) <= rnoi*ocetra(i,j,k,iano3)) THEN
                       bgcprod(i,j,k,kplim) = 1._wp
                       bgcprod(i,j,k,knlim) = 0._wp
                       bgcprod(i,j,k,kflim) = 0._wp
                    ELSE
                       bgcprod(i,j,k,kplim) = 0._wp
                       bgcprod(i,j,k,knlim) = 1._wp
                       bgcprod(i,j,k,kflim) = 0._wp
                    END IF
                 END IF
              ENDIF   ! logical end of bgc_diag

              phofa = pi_alpha*fPAR*strahl(i,j)*swr_frac(i,j,k)
              temfa = 0.6_wp * 1.066_wp**ptho(i,j,k)
              pho   = dtb*phofa*temfa/SQRT(phofa**2 + temfa**2)

              bgcprod(i,j,k,kpho) = pho/dtb

              xa    = avanfe
              xn    = xa / (1._wp + pho*avphy / (xa+bkphy) )                           ! bkphy = half saturation constant
              phosy = MAX(0._wp, xa-xn)                                                ! photo synthesis
              xn    = MAX(xn, 1.e-10_wp)
              ya    = avphy + phosy                                                    ! new phytoplankton concentration before grazing
              yn    = (ya+grazra*avgra*phytomi/(avphy+bkzoo))            &             ! grazing
                   & / (1._wp + grazra * avgra / (avphy + bkzoo))
              grazing = MAX(0._wp, ya-yn)                                              ! what about grazing below euphotic zone?
              graton  = epsher * (1._wp - zinges) * grazing                            ! "grazing to (re-dissolved) nutrients"
              gratpoc = (1._wp - epsher) * grazing                                     ! epsher=0.8 "grazing to POC"
              grawa   = epsher*zinges*grazing                                          ! grazer 'wachstum(?)'

              phymor    = dyphy * MAX(0._wp, (ocetra(i,j,k,iphy) - 2._wp * phytomi))   ! phytoplankton mortality dyphy=.008*dt
              zoothresh = MAX(0._wp, (ocetra(i,j,k,izoo) - 2._wp * grami))
              zoomor    = spemor*zoothresh*zoothresh                                   ! zooplankton mortality
              excdoc    = gammaz*zoothresh                                             ! excretion to DOC (zooplankton)
              exud      = gammap * MAX(0._wp, (ocetra(i,j,k,iphy) - 2._wp*phytomi))    ! exudation to DOC (phytoplankton)
             
              ! Limit O2 consumption by aerob decomposition processes
              ! to thresh_aerob kmol O2 m-3  (default 5.e-8)
              ! check if POC aerob remin occurs not 
              
              ! "available oxygen" for aerob remineralisation
              ! == O2 after PP and oxidation of NH4-> NO3 (graton,
              ! ecan*zoomor)

              avoxy = ocetra(i,j,k,ioxygen)                   &
                   &                + ro2ut*phosy                    &
                   &                - (graton+ecan*zoomor)*ro2ut    
              
  
              IF (avoxy > thresh_aerob) THEN                               ! remineralization of poc12 using oxygen
                ! Total potential decomp. 
                 tpremin = remido*ocetra(i,j,k,idoc)  &
            &         + rcyano *remido_cya*ocetra(i,j,k,idoccya) &
            &         + drempoc * ocetra(i,j,k,idet)

!                write(0,*)tpremin 
                ! Indiv. fractions of decomp.
                 r_bacfra = remido*ocetra(i,j,k,idoc) / tpremin 
                 r_bacfra_cya = rcyano * remido_cya*ocetra(i,j,k,idoccya) / tpremin
                 r_remin  =  drempoc * ocetra(i,j,k,idet) / tpremin
                  
                 xn=ocetra(i,j,k,idet)/(1._wp+drempoc)
                 remin=max(0._wp,ocetra(i,j,k,idet)-xn)
                 remin = MIN(remin,               &
                      &      r_remin*(avoxy-thresh_aerob)/ro2ut)                                  
                 ! 'detritus remineralized fraction' (?)
#ifdef __c_isotopes
                 detref = remin / (ocetra(i, j, k, idet) + 1.e-20_wp)
                 rem13  = detref*ocetra(i,j,k,idet13)                                  ! remineralization of poc13
                 rem14  = detref*ocetra(i,j,k,idet14)                                  !                     poc14
#endif
                ! doc remin only under aerob conditions
                 xn=ocetra(i,j,k,idoc)/(1._wp+remido)
                 bacfra=max(0._wp,ocetra(i,j,k,idoc) - xn)
                 bacfra    = MIN(bacfra,   &                         ! remido = remineralization rate of DOM
                      &     r_bacfra*(avoxy-thresh_aerob)/ro2ut)

                 xn=ocetra(i,j,k,idoccya)/(1._wp+remido_cya)
                 bacfra_cya=max(0._wp,ocetra(i,j,k,idoccya) - xn )
                 bacfra_cya = rcyano * MIN(bacfra_cya, &
                      &     r_bacfra_cya*(avoxy-thresh_aerob)/ro2ut)

                 bgcprod(i,j,k,kremnit) = 0._wp  
              ELSE
                 remin = 0._wp
#ifdef __c_isotopes
                 rem13 = 0._wp
                 rem14 = 0._wp
#endif
                 bacfra = 0._wp 
                 bacfra_cya = 0._wp 
              ENDIF



              ocetra(i,j,k,iphosph) = ocetra(i,j,k,iphosph) &
                   &                + bacfra - phosy + graton  &
                   &                + ecan*zoomor + remin + Pfac_cya*bacfra_cya

              ocetra(i,j,k,iano3) = ocetra(i,j,k,iano3) &
                   &              + (bacfra+bacfra_cya-phosy+graton+ecan*zoomor)*rnit&
                   &              + remin*rnit

              export = zoomor * (1._wp - ecan) + phymor + gratpoc                      ! ecan=.95, gratpoc= .2*grazing [P-units]

              ocetra(i,j,k,idet) = ocetra(i,j,k,idet) + export -remin                  ! k=1,8

              ! new from emr version 4.5.06
#ifdef __c_isotopes
              ocetra(i,j,k,idet13)  = ocetra(i,j,k,idet13)  + export*rcar*bifr13 -rem13
              ocetra(i,j,k,isco213) = ocetra(i,j,k,isco213) - export*rcar*bifr13 +rem13

              ocetra(i,j,k,idet14)  = ocetra(i,j,k,idet14)  + export*rcar*bifr14 -rem14
              ocetra(i,j,k,isco214) = ocetra(i,j,k,isco214) - export*rcar*bifr14 +rem14

              flui13 = MAX(rocean13 - 1._wp, 0._wp)                                    ! only effective if rocean >1 in euphotic layer
              flui14 = MAX(rocean14 - 1._wp, 0._wp)

              ! short term removal from dissolved pool "for gas exchange" (in carchm),
              ! effect is fairly small, 12C in P-units, 13C/14C in C-units (*122)

              ocetra(i,j,k,isco213) = ocetra(i,j,k,isco213) +rem14*rcar   &
                   &                - rcar*export*bifr13*flui13              ! bifr = biogenic fractionation (0.98)  ! *flui13
              ocetra(i,j,k,isco214) = ocetra(i,j,k,isco214)               &
                   &                - rcar*export* bifr14 * flui14           ! *rocean14 (--> flui14?) 4.5.06

              ocetra(i,j,k,idet13)  = ocetra(i,j,k,idet13)                &
                   &                + rcar*export*flui13*bifr13              ! det in P-units, det13 in C-units (*122)
              ocetra(i,j,k,idet14)  = ocetra(i,j,k,idet14)                &  ! (kwrbioz loop)
                   &                + rcar*export*flui14*bifr14
#endif /*__c_isotopes*/

#ifdef AGG
              delsil = MIN(ropal * phosy * avsil / (avsil + bkopal), 0.5_wp * avsil)
              delcar = rcalc * MIN(calmax*phosy,(phosy-delsil/ropal))
#else
              delsil = MIN(ropal * export * avsil / (avsil + bkopal), 0.5_wp * avsil)
              delcar = rcalc * export * bkopal/(avsil+bkopal)                ! 'detritus linked calcium carbonate '
#endif

              ! DMS based on shell production
              dmsprod = (dmsp5*delsil+dmsp4*delcar)                        &
                   &  * (1._wp + 1._wp / (ptho(i, j, k) + dmsp1)**2)
              dms_bac = dmsp3 * dtb * ABS(ptho(i, j, k) + 3._wp) * ocetra(i, j, k, idms) &        ! bacterial consumption
                   &  * (ocetra(i,j,k,idms)/(dmsp6+ocetra(i,j,k,idms)))
              dms_uv  = dmsp2 * 4._wp * dtb * phofa * ocetra(i, j, k, idms)                 ! decay due to UV-radiation
              !!$      dms_uv  = dmsp2*4.*pho*ocetra(i,j,k,idms)                            ! decay due to UV-radiation

              ocetra(i,j,k,idms) = ocetra(i,j,k,idms)                      &
                   &             + dmsprod - dms_bac - dms_uv

              ! end DMS

              ocetra(i,j,k,isco212) = ocetra(i,j,k,isco212)-delcar         &                ! - CACO3 production
                   &                + rcar*( bacfra+bacfra_cya - phosy + graton + ecan*zoomor + remin) ! + remineralization C-units

#ifdef __c_isotopes
              ocetra(i,j,k,isco213) = ocetra(i,j,k,isco213)                &
                   &                - delcar*rocean13                      &
                   &                + bifr13*rcar*( bacfra - phosy + graton + ecan*zoomor) ! bifr=0.98

              ocetra(i,j,k,isco214)=ocetra(i,j,k,isco214)-delcar*rocean14
              ! js: for efficency below line (which should in principle be there)
              !     is neglected (additional tracer field would be needed
              !     to account for radioactive decay of 14C in particles)
              !     &      + bifr14*rcar*( bacfra - phosy + graton + ecan*zoomor)
#endif /*__c_isotopes*/

              ocetra(i, j, k, ialkali) = ocetra(i, j, k, ialkali) - 2._wp * delcar &
                                       - rnit * (bacfra - phosy + graton + ecan*zoomor + remin + bacfra_cya)

              ocetra(i,j,k,iphy) = ocetra(i,j,k,iphy) + phosy - grazing       &
                   &             - phymor - exud
              ocetra(i,j,k,ioxygen) = ocetra(i,j,k,ioxygen)                   &
                   &                + ro2ut*(phosy-bacfra-bacfra_cya)                    &
                   &                - (graton+ecan*zoomor+remin)*ro2ut

              ocetra(i,j,k,izoo)  = ocetra(i,j,k,izoo) + grawa - excdoc - zoomor

              ocetra(i,j,k,idoc)  = ocetra(i,j,k,idoc) - bacfra + excdoc + exud
              ocetra(i,j,k,idoccya) = ocetra(i,j,k,idoccya) - bacfra_cya

              ocetra(i,j,k,icalc) = ocetra(i,j,k,icalc) + delcar

#ifdef __c_isotopes
              ocetra(i,j,k,icalc13) = ocetra(i,j,k,icalc13) + delcar*rocean13

              ocetra(i,j,k,icalc14) = ocetra(i,j,k,icalc14) + delcar*rocean14
#endif

              opalrem = dremopal * 0.1_wp * (ptho(i, j, k) + 3.0_wp) * ocetra(i, j, k, iopal)

              ocetra(i,j,k,isilica) = ocetra(i,j,k,isilica) - delsil + opalrem

              ocetra(i,j,k,iopal) = ocetra(i,j,k,iopal) + delsil - opalrem

              ocetra(i,j,k,iiron) = ocetra(i,j,k,iiron)                            &
                   &              + (bacfra+bacfra_cya-phosy+graton+ecan*zoomor+remin)*riron  &
                   &              - relaxfe * MAX(ocetra(i, j, k, iiron) - fesoly, 0._wp)

#ifdef AGG
              !***********************************************************************
              ! effects of biological processes on number of particles:
              ! photosynthesis creates POM
              ! exudation removes POM
              ! grazing removes POM; but only the fraction that is not egested as
              ! fecal pellets again (grawa remains in zoo, graton goes to po4)
              ! none of the processes at the current time is assumed to change
              ! the size distribution (subject to change)
              ! NOTE that phosy, exud etc. are in kmol/m3!
              ! Thus divide by avmass (kmol/m3)
              !**********************************************************************

              IF (avmass > 0._wp) THEN
                 avnos = ocetra(i,j,k,inos)
                 anosloss = (phosy-exud-graton-grawa)*avnos/avmass
                 ocetra(i,j,k,inos) = ocetra(i,j,k,inos) + anosloss
              ENDIF

              !***********************************************************************
              ! dead zooplankton corpses come with their own, flat distribution
              ! this flow even takes place if there is neither nos nor mass
              ! NOTE: zoomor is in kmol/m3!! Thus multiply flow by 1.e+6
              !***********************************************************************

              zmornos = zoomor * (1._wp-ecan) * zdis * 1.e+6
              ocetra(i,j,k,inos) = ocetra(i,j,k,inos) + zmornos

#endif /*AGG*/
              !
              ! add up over kwrbioz for total inventory (used in inventory_bgc.f90)
              !
              expoor(i,j) = expoor(i,j) + pddpo(i,j,k)*export*rcar
              expoca(i,j) = expoca(i,j) + pddpo(i,j,k)*delcar
              exposi(i,j) = exposi(i,j) + pddpo(i,j,k)*delsil
              !
              ! write output for bgcprod (CDI)

              bgcprod(i,j,k,kdmsprod) = dmsprod/dtbgc       !CDI tinka
              bgcprod(i,j,k,kdms_bac) = dms_bac/dtbgc       !CDI tinka
              bgcprod(i,j,k,kdms_uv)  = dms_uv/dtbgc        !CDI tinka
              bgcprod(i,j,k,kexport)  = export*rcar/dtbgc   !CDI tinka
              bgcprod(i,j,k,kdelcar)  = delcar/dtbgc        !CDI tinka
              bgcprod(i,j,k,kdelsil)  = delsil/dtbgc        !CDI tinka

              bgcprod(i,j,k,kphosy)   = phosy/dtbgc         !CDI tinka
              bgcprod(i,j,k,kgraz)    = grazing/dtbgc       !CDI tinka
              bgcprod(i,j,k,krem)     = graton/dtbgc    
              bgcprod(i,j,k,kpdy)     = phymor/dtbgc    
              bgcprod(i,j,k,kzdy)     = zoomor/dtbgc    
              bgcprod(i,j,k,kexudp)   = exud/dtbgc    
              bgcprod(i,j,k,kexudz)   = excdoc/dtbgc 

             
              bgcprod(i,j,k,kbacr)     = bacfra/dtbgc    
              bgcprod(i,j,k,kbacr_cya) = bacfra_cya/dtbgc    
              bgcprod(i,j,k,kremeu)    = remin/dtbgc    

              bgcprod(i,j,k,kaou)      = satoxy(i,j,k)-ocetra(i,j,k,ioxygen)  
           ENDIF ! pddpo(i,j,k) > 0.5

        ENDDO ! kpie
     ENDDO ! kwrbioz
  ENDDO ! kpje

  CALL maschk(kpie,kpje,kpke,102)

  ! filling of profiles at ocean stations
  IF (station_profile) THEN
     DO n = 1, PROF
        CALL fill_profile(n,pddpo)
     END DO
  ENDIF

#ifdef AGG
  DO  k = 1, kpke
     DO j = 1, kpje
        DO i = 1, kpie

           IF (pddpo(i, j, k) > 0.5_wp) THEN
              avmass = ocetra(i,j,k,iphy) + ocetra(i,j,k,idet)
              snow   = avmass * 1.e+6_wp
              !  check whether the numbers had to be decreased or increased
              IF (snow*pupper > ocetra(i,j,k,inos)) THEN
                 nosin2 = nosin2 + 1
              ENDIF
              IF (snow/cellmass < ocetra(i,j,k,inos)) THEN
                 nosde2 = nosde2 + 1
              ENDIF
           ENDIF

        ENDDO
     ENDDO
  ENDDO
#endif /*AGG*/

  !-----below euphotic zone
  !-OLD-OMP DO
  DO k = kwrbioz+1, kpke
     DO j = 1, kpje
        DO i = 1, kpie

           IF (pddpo(i, j, k) > 0.5_wp) THEN
#ifdef AGG
              avmass = ocetra(i,j,k,iphy) + ocetra(i,j,k,idet)
#endif /*AGG*/
              sterph = dphymor * MAX(0._wp, ocetra(i,j,k,iphy)-phytomi)         ! 'sterberate' phytoplankton
              sterzo = dzoomor * MAX(0._wp, ocetra(i,j,k,izoo)-grami)           ! 'sterberate' zooplankton
              ocetra(i,j,k,iphy) = ocetra(i,j,k,iphy) - sterph
              ocetra(i,j,k,izoo) = ocetra(i,j,k,izoo) - sterzo

              IF(ocetra(i,j,k,ioxygen) > 5.e-8_wp) THEN                         ! remineralization of poc12 using oxygen
                 remin = MIN(drempoc * ocetra(i,j,k,idet),   &
                      &       0.5_wp * ocetra(i,j,k,ioxygen)/ro2ut)

#ifdef __c_isotopes
                 ! FIXME: almzer?
                 detref = remin / (ocetra(i, j, k, idet) + 1.e-20_wp)           ! 'detritus remineralized fraction' (?)
                 rem13  = detref*ocetra(i,j,k,idet13)                           ! remineralization of poc13
                 rem14  = detref*ocetra(i,j,k,idet14)                           !                     poc14
#endif

                 docrem = MIN(dremdoc * ocetra(i,j,k,idoc),                 &   ! remineralization of doc
                      &        0.5_wp * (ocetra(i, j, k, ioxygen) - 5.e-8_wp)/ro2ut)

                 docrem_cya =  rcyano * MIN(dremdoc_cya* ocetra(i,j,k,idoccya),                 &   ! remineralization of doc
                      &        0.5_wp * (ocetra(i, j, k, ioxygen) - 5.e-8_wp)/ro2ut)                
                 bgcprod(i,j,k,kremnit)  = 0._wp


              ELSE                   !changed from =max(remin,0.) etc. to =0. js3.5.2006
                 remin = 0._wp
#ifdef __c_isotopes
                 rem13 = 0._wp
                 rem14 = 0._wp
#endif
                 docrem = 0._wp
                 docrem_cya = 0._wp
              ENDIF

              ocetra(i,j,k,idet)    = ocetra(i,j,k,idet) - remin + sterph + sterzo
#ifdef __c_isotopes
              ocetra(i,j,k,idet13)  = ocetra(i,j,k,idet13)                &
                   &                + rcar*bifr13*(sterph+sterzo) - rem13

              ocetra(i,j,k,idet14)  = ocetra(i,j,k,idet14)-rem14
#endif

              ocetra(i,j,k,ialkali) = ocetra(i,j,k,ialkali)               &
                   &                - rnit*(remin+docrem+docrem_cya)

              ocetra(i,j,k,isco212) = ocetra(i,j,k,isco212)               &
                   &                + rcar*(remin+docrem+docrem_cya)
#ifdef __c_isotopes
              ocetra(i,j,k,isco213) = ocetra(i,j,k,isco213)               &
                   &                + rcar*docrem*bifr13 + rem13       ! rem13 not *rcar as idet13 is in C-units
           !  ocetra(i,j,k,isco214) = ocetra(i,j,k,isco214)            ! no biogenic component for 14C
#endif

              ocetra(i,j,k,idoc)    = ocetra(i,j,k,idoc) - docrem
              ocetra(i,j,k,idoccya)    = ocetra(i,j,k,idoccya) - docrem_cya

              ocetra(i,j,k,ioxygen) = ocetra(i,j,k,ioxygen)               &
                   &                - ro2ut*(remin+docrem+docrem_cya)

              ocetra(i,j,k,iphosph) = ocetra(i,j,k,iphosph)+remin+docrem + Pfac_cya*docrem_cya

              ocetra(i,j,k,iano3)   = ocetra(i,j,k,iano3)+(remin+docrem+docrem_cya)*rnit

              ocetra(i,j,k,iiron)   = ocetra(i,j,k,iiron)                 &
                   &                + (remin+docrem+docrem_cya)*riron                &
                   &                - relaxfe*MAX(ocetra(i,j,k,iiron)-fesoly, 0._wp)

              !***********************************************************************
              ! as ragueneau (2000) notes, Si(OH)4sat is about 1000 umol, but
              ! Si(OH)4 varies only between 0-100 umol
              ! so the expression dremopal*(Si(OH)4sat-Si(OH)4) would change the
              ! rate only from 0 to 100%
              !***********************************************************************

              opalrem = dremopal * 0.1_wp * (ptho(i, j, k) + 3.0_wp) * ocetra(i, j, k, iopal)

              ocetra(i,j,k,iopal)   = ocetra(i,j,k,iopal)   - opalrem

              ocetra(i,j,k,isilica) = ocetra(i,j,k,isilica) + opalrem

              !***********************************************************************
              !           There is about 1.e4 O2 on 1 N2O molecule (Broecker&Peng)
              !           refra : Tim Rixen, pers. communication
              !***********************************************************************

              aou   = satoxy(i,j,k)-ocetra(i,j,k,ioxygen)
              refra = 1._wp + 3._wp * (0.5_wp + SIGN(0.5_wp, aou - 1.97e-4_wp))

              ocetra(i,j,k,ian2o)   = ocetra(i,j,k,ian2o)                  &
                   &                + (remin + docrem + docrem_cya) * 1.e-4_wp * ro2ut * refra

              ocetra(i,j,k,igasnit) = ocetra(i,j,k,igasnit)                &
                   &                - (remin+docrem+docrem_cya) * 1.e-4_wp * ro2ut * refra

              ocetra(i,j,k,ioxygen) = ocetra(i,j,k,ioxygen)               &
                   &                - (remin+docrem+docrem_cya)*1.e-4_wp*ro2ut*refra*0.5_wp

              dms_bac = dmsp3*ABS(ptho(i,j,k)+3._wp) * ocetra(i,j,k,idms)

              ocetra(i,j,k,idms) = ocetra(i,j,k,idms)- dms_bac
#ifdef AGG
              !***********************************************************************
              ! loss of snow aggregates (by numbers) due to remineralization of poc
              ! gain of snow aggregates (by numbers) due to zooplankton mortality
              ! NOTE that remin is in kmol/m3. Thus divide by avmass (kmol/m3)
              !***********************************************************************

              IF (avmass > 0._wp) THEN
                 avnos = ocetra(i,j,k,inos)
                 ocetra(i,j,k,inos) = ocetra(i,j,k,inos)                  &
                      &             - remin * avnos/avmass
              ENDIF

              !***********************************************************************
              ! dead zooplankton corpses come with their own, flat distribution
              ! this flow even takes place if there is neither nos nor mass
              ! NOTE: zoomor is in kmol/m3!! Thus multiply flow by 1.e+6
              !***********************************************************************

              zmornos = sterzo * zdis * 1.e+6_wp
              ocetra(i,j,k,inos) = ocetra(i,j,k,inos) + zmornos
#endif /*AGG*/
           !   bgcprod(i,j,k,kremeu)  = remin/dtbgc
              bgcprod(i,j,k,kpdy_deep)     = sterph/dtbgc   
              bgcprod(i,j,k,kzdy_deep)     = sterzo/dtbgc    

           ENDIF

        END DO
     END DO
  END DO

  !-----below euphotic zone

  CALL maschk(kpie,kpje,kpke,3)

  !---------------------------------------------------------------------
  !-OLD-OMP DO
  DO k = kwrbioz+1, kpke
     DO j = 1,  kpje
        DO i = 1, kpie
           IF (pddpo(i, j, k) > 0.5_wp) THEN    ! needed for masked restart files
           IF (ocetra(i, j, k, ioxygen) < 5.e-7_wp) THEN                        ! denitrification
#ifdef AGG
              avmass = ocetra(i,j,k,iphy) + ocetra(i,j,k,idet)
#endif /*AGG*/
              ! remineralization using NO3 ; proportional to remineralization of poc
              remin  = denitrification*drempoc  &
                   & * MIN(ocetra(i,j,k,idet),0.5_wp*ocetra(i,j,k,iano3)/nitdem)

              ! P-units
#ifdef __c_isotopes
              detref = remin / (ocetra(i, j, k, idet) + 1.e-60_wp)                     
              rem13=detref*ocetra(i,j,k,idet13)                               ! C-units
              rem14  = detref*ocetra(i,j,k,idet14)                              ! C-units
#endif

              remin2o = dremn2o * MIN(ocetra(i,j,k,idet),                    &  ! remineralization using N2O
                   &                  0.003_wp * ocetra(i,j,k,ian2o) / (2._wp*ro2ut))

#ifdef __c_isotopes
              detrl = remin2o/(ocetra(i,j,k,idet) + 1.e-60_wp)                  ! detrl?
              rl13  = detrl*ocetra(i,j,k,idet13)                                ! C-units
              rl14  = detrl*ocetra(i,j,k,idet14)                                ! C-units
#endif
              ocetra(i,j,k,isco212)  = ocetra(i,j,k,isco212)  &
                   &                 + rcar*(remin + remin2o)
              ! proxies 13C, 14C
#ifdef __c_isotopes
              ocetra(i,j,k,isco213)  = ocetra(i,j,k,isco213)  &
                   &                 + (rem13+rl13  )
              !    &                 + rcar* (rem13+rl13  ) ! changed 3.5.2006
              ocetra(i,j,k,isco214)  = ocetra(i,j,k,isco214)  &
                   &                 + (rem14+rl14  )
#endif

              ocetra(i,j,k,idet)    = ocetra(i,j,k,idet)      &
                   &                - (remin + remin2o)
              ! proxies
#ifdef __c_isotopes
              ocetra(i,j,k,idet13)  = ocetra(i,j,k,idet13)    &
                   &                - (rem13+rl13)
              ocetra(i,j,k,idet14)  = ocetra(i,j,k,idet14)    &
                   &                - (rem14+rl14)
#endif
              ocetra(i,j,k,iphosph) = ocetra(i,j,k,iphosph)   &
                   &                + (remin + remin2o)

              ocetra(i,j,k,iano3)   = ocetra(i,j,k,iano3)     &
                   &                - nitdem*remin + rnit*remin2o

              ocetra(i,j,k,igasnit) = ocetra(i,j,k,igasnit)   &
                   &                + n2prod*remin + 2._wp*ro2ut*remin2o

              ocetra(i,j,k,iiron)   = ocetra(i,j,k,iiron)     &
                   &                + riron*(remin+remin2o)

              ocetra(i,j,k,ian2o)   = ocetra(i,j,k,ian2o)     &
                   &                - 2._wp * ro2ut * remin2o
              !alkalinity is increased during denitrification due to consumption of H+ (see Wolf-Gladrow etal,2007)

              ocetra(i,j,k,ialkali) = ocetra(i,j,k,ialkali)   &
                   &                + nitdem*remin - rnit*remin2o

              n2budget(i,j,k)       = n2budget(i,j,k) + 2._wp*n2prod*remin * pddpo(i,j,k)

              !denitrification produces water (H2O), the corresponding O2 uptake is budgeted in h2obudget
              h2obudget(i,j,k)      = h2obudget(i,j,k) + 0.5_wp*n2prod*remin * pddpo(i,j,k)

              bgc_o_pro(i,j,k,kdenit) = nitdem*remin/dtbgc
#ifdef AGG
              !***********************************************************************
              ! loss of snow aggregates (numbers) due to remineralization of poc
              ! NOTE that remin is in kmol/m3. Thus divide by avmass (kmol/m3)
              !***********************************************************************
              IF (avmass > 0._wp) THEN
                 avnos = ocetra(i,j,k,inos)
                 ocetra(i,j,k,inos) = ocetra(i,j,k,inos)         &
                      &             - (remin+remin2o)*avnos/avmass
              ENDIF
#endif /*AGG*/
              bgcprod(i,j,k,kremnit)  = remin/dtbgc


           ENDIF ! oxygen < 5.e-7
              bgcprod(i,j,k,kaou)  = satoxy(i,j,k)-ocetra(i,j,k,ioxygen)    !CDI tinka

           ENDIF ! pddpo < 0.5
        END DO
     END DO
  END DO

  !sulphate reduction   ! introduced 11.5.2007 to improve poc-remineralisation in the
  !                       oxygen minimum zone in the subsurface equatorial Pacific
  !     assumption of endless pool of SO4 (typical concentration are on the order of mmol/l)

  !-OLD-OMP DO
  DO k = kwrbioz+1, kpke
     DO j = 1, kpje
        DO i = 1, kpie
           IF (pddpo(i, j, k) > 0.5_wp) THEN    ! needed for masked restart files
           IF (ocetra(i, j, k, ioxygen) < thresh_sred) THEN
#ifdef AGG
              avmass = ocetra(i,j,k,iphy) + ocetra(i,j,k,idet)
#endif /*AGG*/
              remin = sulfate_reduction * ocetra(i,j,k,idet)
#ifdef __c_isotopes
              detref = sulfate_reduction
              rem13  = detref * ocetra(i,j,k,idet13)
              rem14  = detref * ocetra(i,j,k,idet14)
#endif
              ocetra(i,j,k,idet)    = ocetra(i,j,k,idet)    -        remin
              ocetra(i,j,k,ialkali) = ocetra(i,j,k,ialkali) + sgn_alk * rnit * remin ! sgn_alk= +1 for l_alkcorr=true, -1 as default
              ocetra(i,j,k,isco212) = ocetra(i,j,k,isco212) + rcar * remin
#ifdef __c_isotopes
              ocetra(i,j,k,isco213) = ocetra(i,j,k,isco213) + rem13
              ocetra(i,j,k,isco214) = ocetra(i,j,k,isco214) + rem14
              ocetra(i,j,k,idet13)  = ocetra(i,j,k,idet13)  - rem13
              ocetra(i,j,k,idet14)  = ocetra(i,j,k,idet14)  - rem14
#endif
              ocetra(i,j,k,iphosph) = ocetra(i,j,k,iphosph) +         remin

              ocetra(i,j,k,iano3)   = ocetra(i,j,k,iano3)   + rnit  * remin
              ocetra(i,j,k,iiron)   = ocetra(i,j,k,iiron)   + riron * remin
              !sulphate reduction indirectly effects O2 bugdet, which is budgeted in h2obudget
              h2obudget(i,j,k) = h2obudget(i,j,k) - ro2ut*remin *pddpo(i,j,k)
              n2budget(i,j,k)  = n2budget(i,j,k) + ralk_corr*2._wp*rnit*remin*pddpo(i,j,k) ! ralk_corr==1 when l_alkcorr=true, 0 default

              bgcprod(i,j,k,kremsulf) = remin/dtbgc

              ELSE

               bgcprod(i,j,k,kremsulf) = 0._wp

            ENDIF ! O2
            ENDIF ! pddpo
        END DO
     END DO
  END DO

  !    end sulphate reduction

  CALL maschk(kpie,kpje,kpke,4)

#ifdef AGG
  DO  k = 1, kpke
     DO j = 1, kpje
        DO i = 1, kpie

           IF (pddpo(i, j, k) > 0.5_wp) THEN

              avmass = ocetra(i,j,k,iphy) + ocetra(i,j,k,idet)
              snow   = avmass * 1.e+6_wp

              !           check whether the numbers had to be decreased or increased
              IF (snow*pupper > ocetra(i,j,k,inos)) THEN
                 nosin3 = nosin3 + 1
              ENDIF

              IF (snow/cellmass < ocetra(i,j,k,inos)) THEN
                 nosde3 = nosde3 + 1
              ENDIF

           ENDIF

        ENDDO
     ENDDO
  ENDDO
#endif /*AGG*/

  !js why twice ifdef AGG?
#ifdef AGG

  ! **********************AGGREGATION*****(by Iris Kriest)***************
  ! General:
  ! Sinking speed, size distribution and aggregation are calculated
  ! as in Kriest and Evans, 2000.
  ! I assume that opal and calcium carbonate sink at the same speed as P (mass).
  !
  ! Sinking speed and aggregation: I assume that if there is no phosphorous mass,
  ! the sinking speed is the maximal sinking speed of aggregates. I further
  ! assume that then there are no particles, and that the rate of aggregation
  ! is 0. This scheme removes no P in the absence of P, but still opal and/or
  ! calcium carbonate.
  ! This could or should be changed, because silica as well as carbonate
  ! shells will add to the aggregate mass, and should be considered.
  ! Puh. Does anyone know functional relationships between
  ! size and Si or CaCO3? Perhaps in a later version, I have to
  ! take the relationship between mass and size (i.e., density)?
  !
  ! 1. Size distribution and loss of marine snow aggregates due to aggregation
  ! (aggregate(i,j,k)) and sinking speed of mass and numbers (wmass(i,j,k)
  ! and wnumb(i,j,k) are calculated in a loop over 2-kpke.
  !
  ! 2. The depth of the first layer may change due to ice drift, etc.
  ! This puts a restriction onto the maximum sinking speed.
  ! I currently set the max. size for size dependent sinking onto
  ! one appropriate for this depth.
  !
  ! 3. The fluxes out of the bottom layer are calculated from sinking speed
  ! and mass concentration, and form the boundary condition for the sediment.
  !
  ! 4. The fluxes in layer kpke->2 are calculated from sinking speed and mass
  ! concentration, sinking speed, aggregation and number concentration.

  ! 5. The fluxes in layer 1 are calculated from sinking speed and mass
  ! concentration, sinking speed, aggregation and number concentration.  (??)
  !************************************************************************

  !-OLD-OMP DO
    DO k = 2, kpke
     DO i = 1, kpie
        DO j = 1, kpje

           IF (pddpo(i, j, k) > 0.5_wp) THEN

              avm = ocetra(i,j,k,iphy)+ocetra(i,j,k,idet)

              IF (avm > 0._wp) THEN
                 snow = avm * 1.e+6_wp
                 avn  = ocetra(i,j,k,inos)
                 eps  = ((1.+ FractDim)*snow-avn*cellmass) / (snow-avn*cellmass)

                 ! prevent epsilon from becoming exactly one of the values which are
                 ! needed for the division
                 IF (ABS(eps - 3._wp) < 1.e-15_wp) eps = 3._wp + vsmall
                 IF (ABS(eps - 4._wp) < 1.e-15_wp) eps = 4._wp + vsmall
                 IF (ABS(eps - 3._wp - SinkExp) < 1.e-15_wp) eps = 3._wp + SinkExp + vsmall
                 IF (ABS(eps - 1._wp - SinkExp - FractDim) < 1.e-15_wp)    &
                      eps = 1._wp + SinkExp + FractDim + vsmall

                 e1   = 1._wp - eps
                 e2   = 2._wp - eps
                 e3   = 3._wp - eps
                 e4   = 4._wp - eps
                 es1  = e1 + SinkExp
                 es3  = e3 + SinkExp
                 TopF = (alar1/alow1)**e1     ! alar1 'largest diameter', alow1 'smallest diameter'
                 TopM = TopF*TMFac

                 ! SINKING SPEED FOR THIS LAYER
                 wmass(i,j,k) = cellsink * ( (FractDim+e1)/ (FractDim+es1)    &
                      &       + TopM*TSFac*SinkExp/ (FractDim+es1))
                 wnumb(i,j,k) = cellsink * (e1/es1+TopF*TSFac*SinkExp/es1)

                 ! AGGREGATION

                 ! As a first step, assume that shear in the upper 4 layers is high and
                 ! zero below. Subject to change. js: this is for 20 layer version.
                 !                                    include 40 layer version
                 !                                  should be replaced by check for depth. done 29072005js
                 IF (k.LE. n90depth) THEN
                    fshear = fsh
                 ELSE
                    fshear = 0._wp
                 ENDIF

                 ! shear kernel:
                 sagg1  = (TopF - 1._wp) * (TopF * alar3 - alow3) * e1 / e4          &
                      & + 3._wp * (TopF * alar1 - alow1) * (TopF * alar2 - alow2)    &
                      &       * e1 * e1 / (e2 * e3)
                 sagg2   = TopF &
                      &  * ((alar3 + 3._wp &
                      &  * (alar2 * alow1 * e1 / e2 + alar1 * alow2 * e1 / e3)       &
                      &         + alow3 * e1 / e4) &
                      &  - TopF * alar3 * (1._wp + REAL(3, wp) * (e1 / e2 + e1 / e3) &
                      &                          + e1 / e4))
                 sagg4   = TopF * TopF * 4._wp * alar3

                 shear_agg = (sagg1+sagg2+sagg4)*fshear

                 ! settlement kernel:
                 sagg1  = (TopF * TopF * alar2 * TSFac - alow2)                      &
                      & * SinkExp / (es3 * e3 * (es3 + e1))                          &
                      & + alow2 * ((1. - TopF * TSFac) / (e3 * es1)                  &
                      & - (1._wp - TopF) / (es3*e1))
                 sagg2  = TopF * e1 * (TSFac * ( alow2 - TopF * alar2) / e3          &
                      & - (alow2 - TopF * alar2 * TSFac) / es3)

                 sett_agg =  (e1*e1*sagg1+sagg2)*fse

                 effsti = Stick*(ocetra(i,j,k,iopal)*1.e+6/ropal)/                   &
                      &  ((ocetra(i,j,k,iopal) * 1.e+6_wp / ropal)+snow)

                 aggregate(i,j,k) = (shear_agg+sett_agg)*effsti*avn*avn

                 ! dust aggregation:
                 ! shear kernel:
                 dfirst = dustd3 + 3._wp*dustd2*alar1 + 3._wp*dustd1*alar2+alar3    ! dustd3: dust diameter**3, d2: **2
                 dshagg = e1*fsh*(dfirst*TopF/e1-(                                   &
                      &  (TopF - 1._wp) / e1 * dustd3 + 3._wp * (TopF*alar1-alow1)/e2*dustd2 &
                      &   + 3._wp * (TopF * alar2 - alow2) / e3 * dustd1 + (TopF * alar3 - alow3) / e4))

                 ! settlement kernel:
                 dsett = fse*dustd2*((e1+SinkExp*TopF*TSFac)/es1-dustsink/cellsink)

                 dustagg(i,j,k) = effsti*avn*ocetra(i,j,k,ifdust)*(dshagg+dsett)

              ELSE    ! available mass le 0
                 wmass(i,j,k)       = TSFac*cellsink
                 wnumb(i, j, k)     = 0._wp
                 aggregate(i,j,k)   = 0._wp
                 dustagg(i, j, k)   = 0._wp
                 ocetra(i,j,k,inos) = 0._wp
              ENDIF

           ENDIF   ! wet cell

        ENDDO   ! je
     ENDDO   ! ie
  ENDDO   ! ke

  CALL maschk(kpie,kpje,kpke,5)

  ! EVALUATE SINKING RATE AND AGGREGATION FOR SURFACE LAYER, WHICH MAY BE
  ! LESS DEEP THAN INITIALLY SET BECAUSE OF EVAPORATION, ICE ETC.

  !-OLD-OMP DO
  DO j = 1, kpje
     DO i = 1, kpie

        IF (pddpo(i, j, 1) > 0.5_wp) THEN

           !ik evaluate safe length scale for size dependent sinking and
           !ik aggregation, and the resulting sinking rate and aggregation rate.
           !ik zo may reduce the first layer depth to values that are small and
           !ik may cause the sinking length to exceed the layers depth.
           !ik to be safe, for this upper layer set the upper size such that
           !ik loss due to sinking is at max the whole inventory of this box.
           !ik aggregation will be calculated accordingly.

           checksize = (pddpo(i,j,1)/cellsink)**(1./SinkExp)*alow1

           IF (alar1 > checksize) THEN
              nacheck=nacheck+1              ! js: seems not to be used
           ENDIF

           flar1 = MIN(alar1,checksize)     ! reduce diameter of largest particle
           flar2 = flar1 * flar1
           flar3 = flar2 * flar1
           fTSFac = (flar1/alow1)**SinkExp
           fTMFac = (flar1/alow1)**FractDim

           ! SIZE DITRIBUTION
           avm = ocetra(i,j,1,iphy)+ocetra(i,j,1,idet)       ! available mass   (js: add dust here to account for ballast effect?)

           IF(avm > 0._wp) THEN

              snow = avm*1.e+6_wp
              avn  = ocetra(i,j,1,inos)                      ! available numbers
              eps  = ((1._wp+ FractDim)*snow-avn*cellmass) / (snow-avn*cellmass) ! exponential coefficient of size distribution

              IF (ABS(eps - 3._wp) <  1.e-15_wp) eps=3._wp + vsmall
              IF (ABS(eps - 4._wp) <  1.e-15_wp) eps=4._wp + vsmall
              IF (ABS(eps - 3._wp - SinkExp) <  1.e-15_wp) &
                   &  eps = 3._wp + SinkExp + vsmall
              IF (ABS(eps - 1._wp - SinkExp - FractDim) <  1.e-15_wp) &
                   &  eps = 1._wp + SinkExp + FractDim + vsmall

              e1  = 1._wp - eps
              e2  = 2._wp - eps
              e3  = 3._wp - eps
              e4  = 4._wp - eps
              es1 = e1 + SinkExp
              es3 = e3 + SinkExp

              fTopF = (flar1/alow1)**e1
              fTopM = fTopF*fTMFac

              ! SINKING SPEEDS
              wmass(i,j,1) = cellsink * ( (FractDim+e1)/ (FractDim+es1) &
                   &       + fTopM*fTSFac*SinkExp/ (FractDim+es1))
              wnumb(i,j,1) = cellsink * (e1/es1+fTopF*fTSFac*SinkExp/es1)

              ! AGGREGATION
              sagg1  = (fTopF-1._wp)*(fTopF*flar3-alow3)*e1/e4                         &
                   & + 3._wp*(fTopF*flar1-alow1)*(fTopF*flar2-alow2)*e1*e1/(e2*e3)
              sagg2  = fTopF*(                                                         &
                   &   (flar3+3._wp*(flar2*alow1*e1/e2+flar1*alow2*e1/e3)+alow3*e1/e4) &
                   &   - fTopF*flar3*(1._wp + 3._wp * (e1/e2 + e1/e3) + e1/e4))
              sagg4  = fTopF*fTopF*4._wp*flar3

              shear_agg = (sagg1+sagg2+sagg4)*fsh

              sagg1  = (fTopF * fTopF * flar2 * fTSFac - alow2)              &
                   & * SinkExp / (es3 * e3 * (es3 + e1))                     &
                   & + alow2 * ((1._wp - fTopF * fTSFac) / (e3 * es1)        &
                   & - (1._wp - fTopF) / (es3*e1))
              sagg2  = fTopF * e1 * (fTSFac * ( alow2 - fTopF * flar2) / e3  &
                   & - (alow2 - fTopF * flar2 * fTSFac) / es3)

              sett_agg =  (e1*e1*sagg1+sagg2)*fse


              effsti=Stick*(ocetra(i,j,1,iopal)*1.e+6_wp/ropal)/             &
                   &  ((ocetra(i,j,1,iopal)*1.e+6_wp/ropal)+snow)

              aggregate(i,j,1) = (shear_agg+sett_agg)*effsti*avn*avn

              ! dust aggregation:
              ! shear kernel:
              dfirst=dustd3+3._wp*dustd2*flar1+3._wp*dustd1*flar2+flar3
              dshagg=e1*fsh*(dfirst*fTopF/e1-(                                     &
                   &  (fTopF-1._wp)/e1*dustd3+3._wp*(fTopF*flar1-alow1)/e2*dustd2  &
                   &   +3._wp*(fTopF*flar2-alow2)/e3*dustd1+(fTopF*flar3-alow3)/e4))

              ! settlement kernel:
              dsett=fse*dustd2*((e1+SinkExp*fTopF*fTSFac)/es1-dustsink/cellsink)

              dustagg(i,j,1) = effsti*avn*ocetra(i,j,1,ifdust)*(dshagg+dsett)

           ELSE                            ! available mass le 0.
              wmass(i,j,1)       = fTSFac*cellsink
              wnumb(i,j,1)       = 0._wp
              aggregate(i,j,1)   = 0._wp
              dustagg(i,j,1)     = 0._wp
              ocetra(i,j,1,inos) = 0._wp
           ENDIF

        ENDIF    ! wet cell

     ENDDO
  ENDDO

  ! EVALUATE SINKING RATE AND AGGREGATION FOR BOTTOM LAYER, WHICH MAY BE
  ! LESS THICK THAN THE MINIMUM LAYER THICKNESS

  !-OLD-OMP DO
  DO j = 1, kpje
     DO i = 1, kpie

        IF (pddpo(i,j,1) > 0.5_wp) THEN

           IF (alar1max(i,j) < alar1) THEN

              !ik take safe length scale for size dependent sinking and
              !ik aggregation, and the resulting sinking rate and aggregation rate.

              flar1  = alar1max(i,j)
              flar2  = flar1 * flar1
              flar3  = flar2 * flar1
              fTSFac = TSFmax(i,j)
              fTMFac = TMFmax(i,j)

              ! SIZE DITRIBUTION
              avm = ocetra(i,j,kbo(i,j),iphy)+ocetra(i,j,kbo(i,j),idet)

              IF(avm > 0._wp) THEN

                 snow = avm*1.e+6_wp       ! why *1.e6?
                 avn  = ocetra(i,j,kbo(i,j),inos)
                 eps  = ((1._wp + FractDim)*snow - avn*cellmass) / (snow-avn*cellmass)

                 IF (ABS(eps - 3._wp) < 1.e-15_wp) eps = 3._wp + vsmall
                 IF (ABS(eps - 4._wp) < 1.e-15_wp) eps = 4._wp + vsmall
                 IF (ABS(eps - 3._wp - SinkExp) < 1.e-15_wp) eps = 3._wp + SinkExp + vsmall
                 IF (ABS(eps - 1._wp - SinkExp - FractDim) < 1.e-15_wp)           &
                      &  eps = 1._wp + SinkExp + FractDim + vsmall

                 e1 = 1._wp - eps
                 e2 = 2._wp - eps
                 e3 = 3._wp - eps
                 e4 = 4._wp - eps
                 es1 = e1 + SinkExp
                 es3 = e3 + SinkExp

                 fTopF = (flar1/alow1)**e1
                 fTopM = fTopF*fTMFac

                 ! SINKING SPEEDS
                 wmass(i,j,kbo(i,j)) = cellsink *                              &
                      &        ( (FractDim+e1)/ (FractDim+es1)                 &
                      &          +fTopM*fTSFac*SinkExp/ (FractDim+es1))
                 wnumb(i,j,kbo(i,j)) = cellsink *                              &
                      &          (e1/es1+fTopF*fTSFac*SinkExp/es1)

                 ! AGGREGATION
                 sagg1  = (fTopF-1._wp)*(fTopF*flar3-alow3)*e1/e4                        &
                      & +  3._wp*(fTopF*flar1-alow1)*(fTopF*flar2-alow2)*e1*e1/(e2*e3)
                 sagg2  = fTopF*(                                                        &
                      &  (flar3+3._wp*(flar2*alow1*e1/e2+flar1*alow2*e1/e3)+alow3*e1/e4) &
                      & - fTopF*flar3*(1._wp + 3._wp * (e1/e2 + e1/e3) + e1/e4))
                 sagg4  = fTopF*fTopF*4._wp*flar3

                 shear_agg = (sagg1+sagg2+sagg4)*fsh

                 sagg1  = (fTopF * fTopF * flar2 * fTSFac - alow2)                &
                      & * SinkExp / (es3 * e3 * (es3 + e1))                       &
                      & + alow2 * ((1._wp - fTopF * fTSFac) / (e3 * es1)          &
                      & - (1._wp - fTopF) / (es3*e1))
                 sagg2  = fTopF * e1 * (fTSFac * ( alow2 - fTopF * flar2) / e3    &
                      & - (alow2 - fTopF * flar2 * fTSFac) / es3)
                 sett_agg = (e1*e1*sagg1+sagg2)*fse


                 effsti = Stick*(ocetra(i,j,kbo(i,j),iopal)*1.e+6_wp/ropal)/      &
                      &  ((ocetra(i,j,kbo(i,j),iopal)*1.e+6_wp/ropal)+snow)

                 aggregate(i,j,kbo(i,j)) = (shear_agg+sett_agg)*effsti*avn*avn

                 ! dust aggregation:
                 ! shear kernel:
                 dfirst = dustd3 + 3._wp*dustd2*flar1 + 3._wp*dustd1*flar2 + flar3
                 dshagg = e1*fsh*(dfirst*fTopF/e1 - (                                  &
                      &  (fTopF-1._wp)/e1*dustd3 + 3._wp*(fTopF*flar1-alow1)/e2*dustd2 &
                      & + 3._wp*(fTopF*flar2-alow2)/e3*dustd1+(fTopF*flar3-alow3)/e4))

                 ! settlement kernel:
                 dsett = fse*dustd2*((e1+SinkExp*fTopF*fTSFac)/es1-dustsink/cellsink)

                 dustagg(i,j,kbo(i,j)) = effsti*avn*ocetra(i,j,kbo(i,j),ifdust)   &
                      &                *(dshagg+dsett)

              ELSE
                 wmass(i,j,kbo(i,j))       = fTSFac*cellsink
                 wnumb(i,j,kbo(i,j))       = 0._wp
                 aggregate(i,j,kbo(i,j))   = 0._wp
                 dustagg(i,j,kbo(i,j))     = 0._wp
                 ocetra(i,j,kbo(i,j),inos) = 0._wp
              ENDIF ! avm

           ENDIF ! alar1max

        ENDIF ! pddpo

     ENDDO
  ENDDO

  CALL maschk(kpie,kpje,kpke,6)

  ! diagnose fluxes (fixme : can go into diagnostic subroutine
  !
  !js: here the check should be depth >90 m etc (13.09.2005)
  !    (the original approach worked here (ifdef AGG) because wmass
  !    is zero for depth <kbo. it does not work below (ifndef AGG)
  DO j = 1, kpje
     DO i = 1, kpie

        ! fluxes at 90 m (aggregation part)
        IF (kbo(i,j) > n90depth) THEN
           bgcflux(i,j,kcoex90) = (ocetra(i,j,n90depth,iphy)+ocetra(i,j,n90depth,idet)) &
                &               *wmass(i,j,n90depth)/dtbgc                              !CDI tinka
           bgcflux(i,j,kopex90) = ocetra(i,j,n90depth,iopal)*wmass(i,j,n90depth)/dtbgc  !CDI tinka
           bgcflux(i,j,kcaex90) = ocetra(i,j,n90depth,icalc)*wmass(i,j,n90depth)/dtbgc  !CDI tinka

        ENDIF

        ! fluxes at about 1000 m
        IF (kbo(i,j) > n1000depth) THEN
           bgcflux(i,j,kcoex1000) = (ocetra(i,j,n1000depth,iphy)           &                  !CDI tinka
                &                 +  ocetra(i,j,n1000depth,idet))*wmass(i,j,n1000depth)/dtbgc !CDI tinka
           bgcflux(i,j,kopex1000) =  ocetra(i,j,n1000depth,iopal)*wmass(i,j,n1000depth)/dtbgc !CDI tinka)
           bgcflux(i,j,kcaex1000) =  ocetra(i,j,n1000depth,icalc)*wmass(i,j,n1000depth)/dtbgc !CDI tinka
        ENDIF

        ! fluxes at about 1950 m
        IF (kbo(i,j) > n2000depth) THEN
           bgcflux(i,j,kcoex2000) = (ocetra(i,j,n2000depth,iphy)                      &        !CDI tinka
                &                 +  ocetra(i,j,n2000depth,idet))*wmass(i,j,n2000depth )/dtbgc !CDI tinka
           bgcflux(i,j,kopex2000) =  ocetra(i,j,n2000depth,iopal)*wmass(i,j,n2000depth)/dtbgc  !CDI tinka
           bgcflux(i,j,kcaex2000) =  ocetra(i,j,n2000depth,icalc)*wmass(i,j,n2000depth)/dtbgc  !CDI tinka
        ENDIF

     ENDDO
  ENDDO

  !IK COMPUTE FLUXES FOR BOUNDARY CONDITION/BOTTOM LAYER

  !js fluxes to sediment   (still AGG)

  !-OLD-OMP DO
  DO j = 1, kpje
     DO i = 1, kpie

        IF (pddpo(i,j,kbo(i,j)) > 0.5_wp) THEN
           wphy = wmass(i,j,kbo(i,j))

           prorca(i,j) = ocetra(i,j,kbo(i,j),iphy)  *wphy                &
                &      + ocetra(i,j,kbo(i,j),idet)  *wphy
           prcaca(i,j) = ocetra(i,j,kbo(i,j),icalc) *wphy
           silpro(i,j) = ocetra(i,j,kbo(i,j),iopal) *wphy
           produs(i,j) = ocetra(i,j,kbo(i,j),ifdust)*dustsink            &
                &      + ocetra(i,j,kbo(i,j),iadust)*wphy

        ENDIF

     END DO
  END DO

  ! COMPUTE FLUXES FOR LAYERS 2 TO kpke
  !-OLD-OMP DO
  DO k = kpke, 2, -1
     DO j = 1, kpje
        DO i = 1, kpie

           IF (pddpo(i,j,k) > 0.5_wp) THEN

              ! SINKING SPEED FOR UPPER LAYER
              wphyup = wmass(i,j,k-1)    ! settling velocity of mass
              wnosup = wnumb(i,j,k-1)    ! settling velocity of number of marine snow aggregates

              ! SINKING SPEED FOR ACTUAL LAYER
              wphy = wmass(i,j,k)
              wnos = wnumb(i,j,k)

              ! SUM-UP FLUXES (compute new concentrations)
              ocetra(i,j,k,iphy)    = ocetra(i,j,k,iphy) +                                        &
                   &                + (ocetra(i,j,k-1,iphy)*wphyup-ocetra(i,j,k,iphy)*wphy)       &
                   &                * pdpio(i,j,k)

              ocetra(i,j,k,idet)    = ocetra(i,j,k,idet)                                          &
                   &                + (ocetra(i,j,k-1,idet)*wphyup-ocetra(i,j,k,idet)*wphy)       &
                   &                * pdpio(i,j,k)
#ifdef __c_isotopes
              ocetra(i,j,k,idet13)  = ocetra(i,j,k,idet13)                                        &
                   &                + (ocetra(i,j,k-1,idet13)*wphyup-ocetra(i,j,k,idet13)*wphy)   &
                   &                * pdpio(i,j,k)

              ocetra(i,j,k,idet14)  = ocetra(i,j,k,idet14)                                        &
                   &                + (ocetra(i,j,k-1,idet14)*wphyup-ocetra(i,j,k,idet14)*wphy)   &
                   &                * pdpio(i,j,k)

              ocetra(i,j,k,icalc13) = ocetra(i,j,k,icalc13) +                                     &
                   &                + (ocetra(i,j,k-1,icalc13)*wphyup-ocetra(i,j,k,icalc13)*wphy) &
                   &                * pdpio(i,j,k)

              ocetra(i,j,k,icalc14) = ocetra(i,j,k,icalc14)                                       &
                   &                + (ocetra(i,j,k-1,icalc14)*wphyup-ocetra(i,j,k,icalc14)*wphy) &
                   &                * pdpio(i,j,k)
#endif
              ocetra(i,j,k,icalc)   = ocetra(i,j,k,icalc)                                         &
                   &                + (ocetra(i,j,k-1,icalc)*wphyup-ocetra(i,j,k,icalc)*wphy)     &
                   &                * pdpio(i,j,k)

              ocetra(i,j,k,iopal)   = ocetra(i,j,k,iopal)                                         &
                   &                + (ocetra(i,j,k-1,iopal)*wphyup-ocetra(i,j,k,iopal)*wphy)     &
                   &                * pdpio(i,j,k)

              ocetra(i,j,k,inos)    = ocetra(i,j,k,inos) - aggregate(i,j,k)                       &
                   &                + (ocetra(i,j,k-1,inos)*wnosup-ocetra(i,j,k,inos)*wnos)       &
                   &                * pdpio(i,j,k)

              ! sinking of free dust and loss due to attachment to aggregated dust
              ocetra(i,j,k,ifdust)  = ocetra(i,j,k,ifdust) - dustagg(i,j,k)                       &
                   &                + (ocetra(i,j,k-1,ifdust)-ocetra(i,j,k,ifdust))*dustsink      &
                   &                * pdpio(i,j,k)

              ! sinking of aggregated dust and gain due to attachment of free dust to aggregates
              ocetra(i,j,k,iadust)  = ocetra(i,j,k,iadust) + dustagg(i,j,k)                       &
                   &                + (ocetra(i,j,k-1,iadust)*wphyup-ocetra(i,j,k,iadust)*wphy)   &
                   &                * pdpio(i,j,k)
           ENDIF

        END DO
     END DO
  END DO

  !IK  COMPUTE FLUXES FOR SURFACE LAYER

  !-OLD-OMP DO
  DO j=1,kpje
     DO i=1,kpie

        IF(pddpo(i,j,1) > 0._wp) THEN

           wphy = wmass(i,j,1)
           wnos = wnumb(i,j,1)

           ! SUM-UP FLUXES
           ocetra(i,j,1,iphy)   = ocetra(i,j,1,iphy)                        &
                &               - ocetra(i,j,1,iphy)*wphy*pdpio(i,j,1)

           ocetra(i,j,1,idet)   = ocetra(i,j,1,idet)                        &
                &               - ocetra(i,j,1,idet)*wphy*pdpio(i,j,1)

           ocetra(i,j,1,icalc)  = ocetra(i,j,1,icalc)                       &
                &               - ocetra(i,j,1,icalc)*wphy*pdpio(i,j,1)

           ocetra(i,j,1,iopal)  = ocetra(i,j,1,iopal)                       &
                &               - ocetra(i,j,1,iopal)*wphy*pdpio(i,j,1)

           ocetra(i,j,1,inos)   = ocetra(i,j,1,inos) - aggregate(i,j,1)     &
                &               - ocetra(i,j,1,inos)*wnos*pdpio(i,j,1)

           ! sinking of free dust and loss of free dust to aggregated dust
           ocetra(i,j,1,ifdust) = ocetra(i,j,1,ifdust) - dustagg(i,j,1)     &
                &               - ocetra(i,j,1,ifdust)*dustsink*pdpio(i,j,1)

           ! sinking of aggregated dust and gain due to attachment of free dust to aggregates
           ocetra(i,j,1,iadust) = ocetra(i,j,1,iadust) + dustagg(i,j,1)     &
                &               - ocetra(i,j,1,iadust)*wphy*pdpio(i,j,1)
        ENDIF

     END DO
  END DO

  CALL maschk(kpie,kpje,kpke,8)


#endif /*AGG*/
  !-------------------------------------------end aggregation part

#ifndef AGG

#ifdef BALLASTING
  !
  ! write output for bgcmean
  !
  !js  check against 90, 1000, 2000 m otherwise output is corrupted

  !    were water depth is >0 but less than export depth
  DO j = 1, kpje
     DO i = 1, kpie

        ! write out fluxes at 90 m (fixed settling velocity part)
        IF (kbo(i,j) > n90depth) THEN
           bgcflux(i,j,kcoex90) = ocetra(i,j,n90depth,idet )*wball_poc(i,j,n90depth)/dtbgc    !CDI tinka
           bgcflux(i,j,kopex90) = ocetra(i,j,n90depth,iopal)*wball_opa(i,j,n90depth)/dtbgc    !CDI tinka
           bgcflux(i,j,kcaex90) = ocetra(i,j,n90depth,icalc)*wball_cal(i,j,n90depth)/dtbgc    !CDI tinka
        ENDIF

        ! write out fluxes at about 1000 m
        IF (kbo(i,j) > n1000depth) THEN
           bgcflux(i,j,kcoex1000) = ocetra(i,j,n1000depth,idet )*wball_poc(i,j,n1000depth)/dtbgc !CDI tinka
           bgcflux(i,j,kopex1000) = ocetra(i,j,n1000depth,iopal)*wball_opa(i,j,n1000depth)/dtbgc !CDI tinka
           bgcflux(i,j,kcaex1000) = ocetra(i,j,n1000depth,icalc)*wball_cal(i,j,n1000depth)/dtbgc !CDI tinka
        ENDIF

        ! write out fluxes at about 1950 m
        IF (kbo(i,j) > n2000depth) THEN
           bgcflux(i,j,kcoex2000) = ocetra(i,j,n2000depth,idet )*wball_poc(i,j,n2000depth)/dtbgc !CDI tinka
           bgcflux(i,j,kopex2000) = ocetra(i,j,n2000depth,iopal)*wball_opa(i,j,n2000depth)/dtbgc !CDI tinka
           bgcflux(i,j,kcaex2000) = ocetra(i,j,n2000depth,icalc)*wball_cal(i,j,n2000depth)/dtbgc !CDI tinka
        ENDIF

     ENDDO
  ENDDO

  ! implicit method:
  ! C(k,T+dt)=C(k,T) + (w*dt/ddpo(k))*(C(k-1,T+1)-C(k,T+1))
  ! -->
  ! C(k,T+dt)=(ddpo(k)*C(k,T)+w*dt*C(k-1,T+dt))/(ddpo(k)+w*dt)
  ! sedimentation=w*dt*C(ks,T+dt)
  !

  CALL maschk(kpie,kpje,kpke,-8)


  k = 1                         ! -----------surface layer
  !-OLD-OMP DO
  DO j = 1, kpje
     DO i = 1, kpie

        IF (pddpo(i,j,k) > 0.5_wp) THEN

           ocetra(i,j,k,idet)  = (ocetra(i,j,k,idet)*pddpo(i,j,k))      &
                &              / (pddpo(i,j,k)+wball_poc(i,j,k))

           !proxies
#ifdef __c_isotopes
           ocetra(i,j,k,idet13)  = (ocetra(i,j,k,idet13)*pddpo(i,j,k))  &
                &                / (pddpo(i,j,k)+wball_poc(i,j,k))

           ocetra(i,j,k,idet14)  = (ocetra(i,j,k,idet14)*pddpo(i,j,k))  &
                &                / (pddpo(i,j,k)+wball_poc(i,j,k))
#endif

           ocetra(i,j,k,icalc)   = (ocetra(i,j,k,icalc)*pddpo(i,j,k))   &
                &                / (pddpo(i,j,k)+wball_cal(i,j,k))

           !proxies
#ifdef __c_isotopes
           ocetra(i,j,k,icalc13) = (ocetra(i,j,k,icalc13)*pddpo(i,j,k)) &
                &                / (pddpo(i,j,k)+wball_cal(i,j,k))

           ocetra(i,j,k,icalc14) = (ocetra(i,j,k,icalc14)*pddpo(i,j,k))  &
                &                / (pddpo(i,j,k)+wball_cal(i,j,k))
#endif

           ocetra(i,j,k,iopal)   = (ocetra(i,j,k,iopal)*pddpo(i,j,k))    &
                &                / (pddpo(i,j,k)+wball_opa(i,j,k))

           ocetra(i,j,k,ifdust)  = (ocetra(i,j,k,ifdust)*pddpo(i,j,k))   &
                &                / (pddpo(i,j,k)+wball_dus(i,j,k))

        ENDIF

     ENDDO
  ENDDO


  !-OLD-OMP DO
  DO j = 1, kpje
     DO k = 2, kpke                    ! ---------------------- water column
        DO i = 1, kpie

           IF(pddpo(i,j,k) > 0.5_wp) THEN

              ocetra(i,j,k,idet)    = (ocetra(i,j,k  ,idet)*pddpo(i,j,k)    &
                   &                +  ocetra(i,j,k-1,idet)*wball_poc(i,j,k-1))/          &
                   &                          (pddpo(i,j,k)+wball_poc(i,j,k))
#ifdef __c_isotopes
              ocetra(i,j,k,idet13)  = (ocetra(i,j,k  ,idet13)*pddpo(i,j,k)  &
                   &                +  ocetra(i,j,k-1,idet13)*wball_poc(i,j,k-1))/        &
                   &                            (pddpo(i,j,k)+wball_poc(i,j,k))

              ocetra(i,j,k,idet14)  = (ocetra(i,j,k  ,idet14)*pddpo(i,j,k)  &
                   &                +  ocetra(i,j,k-1,idet14)*wball_poc(i,j,k-1))/        &
                   &                            (pddpo(i,j,k)+wball_poc(i,j,k))
#endif
              ocetra(i,j,k,icalc)   = (ocetra(i,j,k  ,icalc)*pddpo(i,j,k)   &
                   &                +  ocetra(i,j,k-1,icalc)*wball_cal(i,j,k-1))/         &
                   &                           (pddpo(i,j,k)+wball_cal(i,j,k))
#ifdef __c_isotopes
              ocetra(i,j,k,icalc13) = (ocetra(i,j,k  ,icalc13)*pddpo(i,j,k) &
                   &                +  ocetra(i,j,k-1,icalc13)*wball_cal(i,j,k-1))/       &
                   &                             (pddpo(i,j,k)+wball_cal(i,j,k))

              ocetra(i,j,k,icalc14) = (ocetra(i,j,k  ,icalc14)*pddpo(i,j,k) &
                   &                +  ocetra(i,j,k-1,icalc14)*wball_cal(i,j,k-1))/       &
                   &                             (pddpo(i,j,k)+wball_cal(i,j,k))
#endif

              ocetra(i,j,k,iopal)   = (ocetra(i,j,k  ,iopal)*pddpo(i,j,k)   &
                   &                +  ocetra(i,j,k-1,iopal)*wball_opa(i,j,k-1))/        &
                   &                           (pddpo(i,j,k)+wball_opa(i,j,k))

              ocetra(i,j,k,ifdust)  = (ocetra(i,j,k ,ifdust)*pddpo(i,j,k)   &
                   &                +  ocetra(i,j,k-1,ifdust)*wball_dus(i,j,k-1))/       &
                   &                            (pddpo(i,j,k)+wball_dus(i,j,k))

           ENDIF

        ENDDO
     ENDDO
  ENDDO

  CALL maschk(kpie,kpje,kpke,9)

  ! ------------------------------flux to sediment
  !-OLD-OMP DO
  DO j = 1, kpje
     DO i = 1, kpie

        IF (pddpo(i,j,1) > 0.5_wp) THEN

           prorca(i,j) = ocetra(i,j,kbo(i,j),idet )*wball_poc(i,j,kbo(i,j))
           prcaca(i,j) = ocetra(i,j,kbo(i,j),icalc)*wball_cal(i,j,kbo(i,j))
#ifdef __c_isotopes
           pror13(i,j) = ocetra(i,j,kbo(i,j),idet13 )*wball_poc(i,j,kbo(i,j))
           prca13(i,j) = ocetra(i,j,kbo(i,j),icalc13)*wball_cal(i,j,kbo(i,j))
           pror14(i,j) = ocetra(i,j,kbo(i,j),idet14 )*wball_poc(i,j,kbo(i,j))
           prca14(i,j) = ocetra(i,j,kbo(i,j),icalc14)*wball_cal(i,j,kbo(i,j))
#endif
           silpro(i,j) = ocetra(i,j,kbo(i,j),iopal )*wball_opa(i,j,kbo(i,j))
           produs(i,j) = ocetra(i,j,kbo(i,j),ifdust)*wball_dus(i,j,kbo(i,j))
           !
           !  write output
           !
           bgcflux(i,j,kprorca) = prorca(i,j)/dtbgc     !CDI tinka
           bgcflux(i,j,kprcaca) = prcaca(i,j)/dtbgc     !CDI tinka
           bgcflux(i,j,ksilpro) = silpro(i,j)/dtbgc     !CDI tinka
           bgcflux(i,j,kprodus) = produs(i,j)/dtbgc     !CDI tinka

        ENDIF

     END DO
  END DO

  CALL maschk(kpie,kpje,kpke,10)
#endif BALLASTING


#ifndef BALLASTING
  !
  ! write output for bgcmean
  !
  !js  check against 90, 1000, 2000 m otherwise output is corrupted

  !    were water depth is >0 but less than export depth
  DO j = 1, kpje
     DO i = 1, kpie

        ! write out fluxes at 90 m (fixed settling velocity part)
        IF (kbo(i,j) > n90depth) THEN
           bgcflux(i,j,kcoex90) = ocetra(i,j,n90depth,idet)*wpoc(n90depth)/dtbgc      !CDI tinka
           bgcflux(i,j,kopex90) = ocetra(i,j,n90depth,iopal)*wopal/dtbgc    !CDI tinka
           bgcflux(i,j,kcaex90) = ocetra(i,j,n90depth,icalc)*wcal/dtbgc     !CDI tinka
        ENDIF

        ! write out fluxes at about 1000 m
        IF (kbo(i,j) > n1000depth) THEN
           bgcflux(i,j,kcoex1000) = ocetra(i,j,n1000depth,idet)*wpoc(n1000depth)/dtbgc      !CDI tinka
           bgcflux(i,j,kopex1000) = ocetra(i,j,n1000depth,iopal)*wopal/dtbgc    !CDI tinka
           bgcflux(i,j,kcaex1000) = ocetra(i,j,n1000depth,icalc)*wcal/dtbgc     !CDI tinka
        ENDIF

        ! write out fluxes at about 1950 m
        IF (kbo(i,j) > n2000depth) THEN
           bgcflux(i,j,kcoex2000) = ocetra(i,j,n2000depth,idet)*wpoc(n2000depth)/dtbgc      !CDI tinka
           bgcflux(i,j,kopex2000) = ocetra(i,j,n2000depth,iopal)*wopal/dtbgc    !CDI tinka
           bgcflux(i,j,kcaex2000) = ocetra(i,j,n2000depth,icalc)*wcal/dtbgc     !CDI tinka
        ENDIF

     ENDDO
  ENDDO

  ! implicit method:
  ! C(k,T+dt)=C(k,T) + (w*dt/ddpo(k))*(C(k-1,T+1)-C(k,T+1))
  ! -->
  ! C(k,T+dt)=(ddpo(k)*C(k,T)+w*dt*C(k-1,T+dt))/(ddpo(k)+w*dt)
  ! sedimentation=w*dt*C(ks,T+dt)
  !

  CALL maschk(kpie,kpje,kpke,-8)

  k = 1                         ! -----------surface layer
  !-OLD-OMP DO
  DO j = 1, kpje
     DO i = 1, kpie

        IF (pddpo(i,j,k) > 0.5_wp) THEN

           ocetra(i,j,k,idet)  = (ocetra(i,j,k,idet)*pddpo(i,j,k))      &
                &              / (pddpo(i,j,k)+wpoc(k))

           !proxies
#ifdef __c_isotopes
           ocetra(i,j,k,idet13)  = (ocetra(i,j,k,idet13)*pddpo(i,j,k))  &
                &                / (pddpo(i,j,k)+wpoc(k))

           ocetra(i,j,k,idet14)  = (ocetra(i,j,k,idet14)*pddpo(i,j,k))  &
                &                / (pddpo(i,j,k)+wpoc(k))
#endif

           ocetra(i,j,k,icalc)   = (ocetra(i,j,k,icalc)*pddpo(i,j,k))   &
                &                / (pddpo(i,j,k)+wcal)

           !proxies
#ifdef __c_isotopes
           ocetra(i,j,k,icalc13) = (ocetra(i,j,k,icalc13)*pddpo(i,j,k)) &
                &                / (pddpo(i,j,k)+wcal)

           ocetra(i,j,k,icalc14) = (ocetra(i,j,k,icalc14)*pddpo(i,j,k))  &
                &                / (pddpo(i,j,k)+wcal)
#endif

           ocetra(i,j,k,iopal)   = (ocetra(i,j,k,iopal)*pddpo(i,j,k))    &
                &                / (pddpo(i,j,k)+wopal)

           ocetra(i,j,k,ifdust)  = (ocetra(i,j,k,ifdust)*pddpo(i,j,k))   &
                &                / (pddpo(i,j,k)+wdust)

        ENDIF

     ENDDO
  ENDDO

  !-OLD-OMP DO
  DO j = 1, kpje
     DO k = 2, kpke                    ! ---------------------- water column
        DO i = 1, kpie

           IF(pddpo(i,j,k) > 0.5_wp) THEN

              ocetra(i,j,k,idet)    = (ocetra(i,j,k  ,idet)*pddpo(i,j,k)    &
                   &                +  ocetra(i,j,k-1,idet)*wpoc(k-1))/          &
                   &                          (pddpo(i,j,k)+wpoc(k))
#ifdef __c_isotopes
              ocetra(i,j,k,idet13)  = (ocetra(i,j,k  ,idet13)*pddpo(i,j,k)  &
                   &                +  ocetra(i,j,k-1,idet13)*wpoc(k-1))/        &
                   &                            (pddpo(i,j,k)+wpoc(k))

              ocetra(i,j,k,idet14)  = (ocetra(i,j,k  ,idet14)*pddpo(i,j,k)  &
                   &                +  ocetra(i,j,k-1,idet14)*wpoc(k-1))/        &
                   &                            (pddpo(i,j,k)+wpoc(k))
#endif
              ocetra(i,j,k,icalc)   = (ocetra(i,j,k  ,icalc)*pddpo(i,j,k)   &
                   &                +  ocetra(i,j,k-1,icalc)*wcal)/         &
                   &                           (pddpo(i,j,k)+wcal)
#ifdef __c_isotopes
              ocetra(i,j,k,icalc13) = (ocetra(i,j,k  ,icalc13)*pddpo(i,j,k) &
                   &                +  ocetra(i,j,k-1,icalc13)*wcal)/       &
                   &                             (pddpo(i,j,k)+wcal)

              ocetra(i,j,k,icalc14) = (ocetra(i,j,k  ,icalc14)*pddpo(i,j,k) &
                   &                +  ocetra(i,j,k-1,icalc14)*wcal)/       &
                   &                             (pddpo(i,j,k)+wcal)
#endif

              ocetra(i,j,k,iopal)   = (ocetra(i,j,k  ,iopal)*pddpo(i,j,k)   &
                   &                +  ocetra(i,j,k-1,iopal)*wopal)/        &
                   &                           (pddpo(i,j,k)+wopal)

              ocetra(i,j,k,ifdust)  = (ocetra(i,j,k ,ifdust)*pddpo(i,j,k)   &
                   &                +  ocetra(i,j,k-1,ifdust)*wdust)/       &
                   &                            (pddpo(i,j,k)+wdust)

           ENDIF

        ENDDO
     ENDDO
  ENDDO

  CALL maschk(kpie,kpje,kpke,9)

  ! ------------------------------flux to sediment
  !-OLD-OMP DO
  DO j = 1, kpje
     DO i = 1, kpie

        IF (pddpo(i,j,1) > 0.5_wp) THEN

           prorca(i,j) = ocetra(i,j,kbo(i,j),idet )*wpoc(kbo(i,j))
           prcaca(i,j) = ocetra(i,j,kbo(i,j),icalc)*wcal
#ifdef __c_isotopes
           pror13(i,j) = ocetra(i,j,kbo(i,j),idet13 )*wpoc(kbo(i,j))
           prca13(i,j) = ocetra(i,j,kbo(i,j),icalc13)*wcal
           pror14(i,j) = ocetra(i,j,kbo(i,j),idet14 )*wpoc(kbo(i,j))
           prca14(i,j) = ocetra(i,j,kbo(i,j),icalc14)*wcal
#endif
           silpro(i,j) = ocetra(i,j,kbo(i,j),iopal)*wopal
           produs(i,j) = ocetra(i,j,kbo(i,j),ifdust)*wdust
           !
           !  write output
           !
           bgcflux(i,j,kprorca) = prorca(i,j)/dtbgc     !CDI tinka
           bgcflux(i,j,kprcaca) = prcaca(i,j)/dtbgc     !CDI tinka
           bgcflux(i,j,ksilpro) = silpro(i,j)/dtbgc     !CDI tinka
           bgcflux(i,j,kprodus) = produs(i,j)/dtbgc     !CDI tinka

        ENDIF

     END DO
  END DO

  CALL maschk(kpie,kpje,kpke,10)
#endif /* not BALLASTING */




#endif /* not AGG*/

  !-OLD-OMP END PARALLEL

  ! calculate OMZ and write output
  CALL calc_omz_depth(pddpo)

  !
  ! Check maximum/minimum values in wet/dry cells.
  !
  IF (kchck == 1) CALL CHCK_BGC(io_stdo_bgc,icyclibgc,             &
       &'Check values of ocean tracer at exit from SBR OCPROD :',  &
       & kpie,kpje,kpke,pddpo)

END SUBROUTINE OCPROD
