!>
!! @brief Variables for inorganic carbon cycle
!!
!! Definition of variables and allocation of memory
!!
!! @author S.Legutke (MPI-M)
!!
!! @par Revision History
!!
!! First version by S.Legutke            (MPI-M)    Oct 31, 2001
!!
!! Modified version by Patrick Wetzel    (MPI-M)    Apr 16, 2002
!!     - new: atm, atdifv, suppco2
!!     - changed: chemc(:,:,:) to chemcm(:,:,:,:)
!! Modified version by Katharina Six    (MPI-M)     Mar,31, 2014
!!     - removed : chemcm(:,:,:,:) 
!!     - substituted by solubility field and saturation field for O2 and N2
!!     - dissoziation constantes are already written on 3-d files (ak13,ak23,....)
!!     - no field for borate concentration nescessary as already calculated from salinity 
!!
!! @par Copyright
!! 2002-2013 by MPI-M
!! This software is provided for non-commercial use only.
!! See the LICENSE and the WARRANTY conditions.
!!
MODULE mo_carbch

  USE mo_kind, ONLY   : wp
  USE mo_commo1, ONLY : l_cpl_co2, l_cpl_dust

  IMPLICIT NONE

  PUBLIC

  REAL(wp), ALLOCATABLE, TARGET :: atm   (:,:,:)
  REAL(wp), ALLOCATABLE, TARGET :: ocetra(:,:,:,:)
  REAL(wp), ALLOCATABLE         :: solco2(:,:)

  REAL(wp), ALLOCATABLE :: wpoc(:)  ! depth-dependent detritus settling speed
  ! BALLASTING
  REAL(wp), ALLOCATABLE :: wball(:,:,:)  ! particle sinking speed (POC and ballast)
  REAL(wp), ALLOCATABLE :: wballuncor(:,:,:) ! not corrected for wball>dzw
  REAL(wp), ALLOCATABLE :: wballc(:,:,:)   ! hypothetical POC+calcite
  REAL(wp), ALLOCATABLE :: wballo(:,:,:)   ! hypothetical POC+opal
  REAL(wp), ALLOCATABLE :: wballd(:,:,:)   ! hypothetical POC+dust
  REAL(wp), ALLOCATABLE :: wballdd(:,:,:)  ! actual speed of dust
  REAL(wp), ALLOCATABLE :: wball_poc(:,:,:)
  REAL(wp), ALLOCATABLE :: wball_cal(:,:,:)
  REAL(wp), ALLOCATABLE :: wball_opa(:,:,:)
  REAL(wp), ALLOCATABLE :: wball_dus(:,:,:)

  REAL(wp), ALLOCATABLE, TARGET :: co3 (:,:,:)
  REAL(wp), ALLOCATABLE, TARGET :: hi  (:,:,:)
  REAL(wp), ALLOCATABLE, TARGET :: aksp(:,:,:)

  REAL(wp), ALLOCATABLE :: akw3(:,:,:)
  REAL(wp), ALLOCATABLE :: akb3(:,:,:)
  REAL(wp), ALLOCATABLE :: ak13(:,:,:)
  REAL(wp), ALLOCATABLE :: ak23(:,:,:)
  REAL(wp), ALLOCATABLE :: satoxy(:,:,:)
  REAL(wp), ALLOCATABLE :: satn2(:,:)
  REAL(wp), ALLOCATABLE :: satn2o(:,:)
  REAL(wp), ALLOCATABLE :: atdifv(:,:)
  REAL(wp), ALLOCATABLE :: suppco2(:,:)
  REAL(wp), ALLOCATABLE :: sedfluxo(:,:,:)
  REAL(wp), ALLOCATABLE :: aksurf(:,:,:)    !> dissociation constants for DIC,bor and water 
                                            !> at the sea surface, no pressure dependency 

  REAL(wp), ALLOCATABLE :: co2flux(:,:)     !> sea-air C-flux, ingetrated over whole simulation period
  REAL(wp), ALLOCATABLE :: o2flux(:,:)      !> sea-air O2-flux, ingetrated over whole simulation period
  REAL(wp), ALLOCATABLE :: n2flux(:,:)      !> sea-air N2-flux, ingetrated over whole simulation period
  REAL(wp), ALLOCATABLE :: n2oflux(:,:)     !> sea-air N2-flux, ingetrated over whole simulation period

#ifdef __c_isotopes
  REAL(wp), ALLOCATABLE :: c13flux(:,:)     !> sea-air C13-flux, ingetrated over whole simulation period
  REAL(wp), ALLOCATABLE :: c14flux(:,:)     !> sea-air C14-flux, ingetrated over whole simulation period
#endif

  REAL(wp), ALLOCATABLE :: co2trans(:,:)    !> transfer coefficient for CO2 atmosph/ocean
                                            !> exchange, still to be multiplied by v**2
  REAL(wp), ALLOCATABLE :: co2conc(:,:)
  REAL(wp), ALLOCATABLE :: co2flux_cpl(:,:)

  ! for input of dust fields (on-line from ECHAM)
  ! reintroduced js 19062006 */
  REAL(wp), ALLOCATABLE :: dustdep(:,:)     ! + updated allocation
  REAL(wp), ALLOCATABLE, TARGET :: dusty(:,:)
  REAL(wp), ALLOCATABLE, TARGET :: dusty_lgm(:,:) ! BALLASTING: LGM dust for ballasting

  REAL(wp), ALLOCATABLE :: c14pool(:,:,:)

  REAL(wp) :: dmspar(6)

  ! decay coefficient for sco214 plus more for 13C/14C
  REAL(wp) :: c14dec, D14Catm, Ratm, D14Cocn, Roc14
  REAL(wp) :: roc13, atcoa
  REAL(wp) :: ozkoa, c14prod, c14inva, eins, c14ret, c14prosta

  REAL(wp) :: atm_co2, atm_o2, atm_n2
  !                              .35e-3 * 5.1e14*12. .35e-3 * 5.1e14
  REAL(wp) :: globalmean_co2, globalmean_o2, globalmean_n2
  REAL(wp) :: atm_c13, atm_c14, atmacon, atmacmol

  REAL(wp), PARAMETER :: ppm2con=0.35e-3_wp     !> ppm2con: atmospheric weight: ~10000kg/m^2, avrg. ~29 g/mol
                                                !> --> 350 kmol/m^2 --> 1ppm ~ 0.35e-3 kmol/m^2
  REAL(wp), PARAMETER :: contppm=1._wp/ppm2con  !> inverse of conversion factor molC/m**2 to ppm (in AT)

  REAL(wp), PARAMETER :: molw_co2=44.011_wp     !> Total Molecular Weight of CO2 (44.011 g/mol)
 
  REAL(wp), PARAMETER :: molw_dry_air=28.970_wp !> mean pseudo-molecular weight of dry air

  REAL(wp) :: ems_per_step

  REAL(wp) :: totarea

  ! n2budget closes the mass balance for the alkalinity for biogenic induced changes in N2
  REAL(wp), ALLOCATABLE :: n2budget(:,:,:)
  ! h2obudget closes the mass balance for the oxygen for biogenic induced changes in N2
  REAL(wp), ALLOCATABLE :: h2obudget(:,:,:)

CONTAINS

  SUBROUTINE ALLOC_MEM_CARBCH(kpie,kpje,kpke)

    USE mo_control_bgc
    USE mo_param1_bgc

    INTEGER :: kpie,kpje,kpke

    IF ( kchck > 0 ) THEN
      WRITE(io_stdo_bgc,*)'Memory allocation for variable wpoc ...'
    ENDIF
    ALLOCATE (wpoc(kpke))

    IF ( kchck > 0 ) THEN
      WRITE(io_stdo_bgc,*)'Memory allocation for variable ocetra ...'
      WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
      WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
      WRITE(io_stdo_bgc,*)'Third dimension    : ',kpke
      WRITE(io_stdo_bgc,*)'Forth dimension    : ',nocetra
    ENDIF
    ALLOCATE (ocetra(kpie,kpje,kpke,nocetra))

    IF ( kchck > 0 ) THEN
      WRITE(io_stdo_bgc,*)'Memory allocation for variable hi ...'
      WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
      WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
      WRITE(io_stdo_bgc,*)'Third dimension    : ',kpke
    ENDIF
    ALLOCATE (hi(kpie,kpje,kpke))

    IF ( kchck > 0 ) THEN
      WRITE(io_stdo_bgc,*)'Memory allocation for variable co3 ...'
      WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
      WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
      WRITE(io_stdo_bgc,*)'Third dimension    : ',kpke
    ENDIF
    ALLOCATE (co3(kpie,kpje,kpke))
    ALLOCATE (c14pool(kpie,kpje,kpke))

    IF ( kchck > 0 ) THEN
      WRITE(io_stdo_bgc,*)'Memory allocation for variable solco2 ...'
      WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
      WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
    ENDIF
    ALLOCATE (solco2(kpie,kpje))
    solco2(:,:) = rmasko

    IF ( kchck > 0 ) THEN
      WRITE(io_stdo_bgc,*)'Memory allocation for variable satn2o ...'
      WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
      WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
    ENDIF
    ALLOCATE (satn2o(kpie,kpje))
    satn2o(:,:) = rmasko

    IF ( kchck > 0 ) THEN
      WRITE(io_stdo_bgc,*)'Memory allocation for variable satoxy ...'
      WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
      WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
      WRITE(io_stdo_bgc,*)'Third dimension    : ',kpke
    ENDIF
    ALLOCATE (satoxy(kpie,kpje,kpke))
    satoxy(:,:,:) = rmasko

    IF ( kchck > 0 ) THEN
      WRITE(io_stdo_bgc,*)'Memory allocation for variable satn2 ...'
      WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
      WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
    ENDIF
    ALLOCATE (satn2(kpie,kpje))
    satn2(:,:) = rmasko

    IF ( kchck > 0 ) THEN
      WRITE(io_stdo_bgc,*)'Memory allocation for variable sedfluxo ...'
      WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
      WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
      WRITE(io_stdo_bgc,*)'Third dimension    : ',npowtra
    ENDIF
    ALLOCATE (sedfluxo(kpie,kpje,npowtra))
    sedfluxo(:,:,:) = 0._wp

    IF ( kchck > 0 ) THEN
      WRITE(io_stdo_bgc,*)'Memory allocation for variable aksp ...'
      WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
      WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
      WRITE(io_stdo_bgc,*)'Third dimension    : ',kpke
    ENDIF
    ALLOCATE (aksp(kpie,kpje,kpke))
    aksp(:,:,:) = rmasko

    IF ( kchck > 0 ) THEN
      WRITE(io_stdo_bgc,*)'Memory allocation for variable ak23 ...'
      WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
      WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
      WRITE(io_stdo_bgc,*)'Third dimension    : ',kpke
    ENDIF
    ALLOCATE (ak23(kpie,kpje,kpke))

    IF ( kchck > 0 ) THEN
      WRITE(io_stdo_bgc,*)'Memory allocation for variable ak13 ...'
      WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
      WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
      WRITE(io_stdo_bgc,*)'Third dimension    : ',kpke
    ENDIF
    ALLOCATE (ak13(kpie,kpje,kpke))

    IF ( kchck > 0 ) THEN
      WRITE(io_stdo_bgc,*)'Memory allocation for variable akb3 ...'
      WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
      WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
      WRITE(io_stdo_bgc,*)'Third dimension    : ',kpke
    ENDIF
    ALLOCATE (akb3(kpie,kpje,kpke))

    IF ( kchck > 0 ) THEN
      WRITE(io_stdo_bgc,*)'Memory allocation for variable akw3 ...'
      WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
      WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
      WRITE(io_stdo_bgc,*)'Third dimension    : ',kpke
    ENDIF
    ALLOCATE (akw3(kpie,kpje,kpke))

    IF ( kchck > 0 ) THEN
      WRITE(io_stdo_bgc,*)'Memory allocation for variable aksurf ...'
      WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
      WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
      WRITE(io_stdo_bgc,*)'Third dimension    : ',4
    ENDIF
    ALLOCATE (aksurf(kpie,kpje,4))
     aksurf(:,:,:) = rmasko
    IF ( kchck > 0 ) THEN
      WRITE(io_stdo_bgc,*)'Memory allocation for variable atm ...'
      WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
      WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
      WRITE(io_stdo_bgc,*)'Third dimension    : ',natm
    ENDIF
    ALLOCATE (atm(kpie,kpje,natm))
    IF ( kchck > 0 ) THEN
      WRITE(io_stdo_bgc,*)'Memory allocation for variable atdifv ...'
      WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
      WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
    ENDIF
    ALLOCATE (atdifv(kpie,kpje))
    IF ( kchck > 0 ) THEN
      WRITE(io_stdo_bgc,*)'Memory allocation for variable suppco2 ...'
      WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
      WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
    ENDIF
    ALLOCATE (suppco2(kpie,kpje))
    suppco2(:,:) = 0.0_wp

    ! from esm version start
    !       WRITE(io_stdo_bgc,*)'Memory allocation for variable dmsflux ...'
    !       WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
    !       WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje

    !       ALLOCATE (dmsflux(kpie,kpje))
    !       dmsflux(:,:)=0.0

    !       WRITE(io_stdo_bgc,*)'Memory allocation for variable dms ...'
    !       WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
    !       WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje

    !       ALLOCATE (dms(kpie,kpje))
    !       dms(:,:)=0.0
    IF ( kchck > 0 ) THEN
      WRITE(io_stdo_bgc,*)'Memory allocation for variable co2flux ...'
      WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
      WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
    ENDIF
    ALLOCATE (co2flux(kpie,kpje))
    co2flux(:,:) = 0.0_wp
    IF ( kchck > 0 ) THEN
      WRITE(io_stdo_bgc,*)'Memory allocation for variable o2flux ...'
      WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
      WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
    ENDIF
    ALLOCATE (o2flux(kpie,kpje))
    o2flux(:,:) = 0.0_wp
    IF ( kchck > 0 ) THEN
      WRITE(io_stdo_bgc,*)'Memory allocation for variable n2flux ...'
      WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
      WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
    ENDIF
    ALLOCATE (n2flux(kpie,kpje))
    n2flux(:,:) = 0.0_wp
    IF ( kchck > 0 ) THEN
      WRITE(io_stdo_bgc,*)'Memory allocation for variable n2oflux ...'
      WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
      WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
    ENDIF
    ALLOCATE (n2oflux(kpie,kpje))
    n2oflux(:,:) = 0.0_wp

#ifdef __c_isotopes
    IF ( kchck > 0 ) THEN
      WRITE(io_stdo_bgc,*)'Memory allocation for variable c13flux ...'
      WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
      WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
    ENDIF
    ALLOCATE (c13flux(kpie,kpje))
    c13flux(:,:) = 0.0_wp
    IF ( kchck > 0 ) THEN
      WRITE(io_stdo_bgc,*)'Memory allocation for variable c14flux ...'
      WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
      WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
    ENDIF
    ALLOCATE (c14flux(kpie,kpje))
    c14flux(:,:) = 0.0_wp
#endif

    IF (l_cpl_co2) THEN
      IF ( kchck > 0 ) THEN
        WRITE(io_stdo_bgc,*)'Memory allocation for variable co2trans ...'
        WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
        WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
      ENDIF
      ALLOCATE (co2trans(kpie,kpje))
      co2trans(:,:) = 0.0_wp
      IF ( kchck > 0 ) THEN
        WRITE(io_stdo_bgc,*)'Memory allocation for variable co2conc ...'
        WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
        WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
      ENDIF
      ALLOCATE (co2conc(kpie,kpje))
      co2conc(:,:) = 0.0_wp
      IF ( kchck > 0 ) THEN
        WRITE(io_stdo_bgc,*)'Memory allocation for variable co2flux_cpl ...'
        WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
        WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
      ENDIF
      ALLOCATE (co2flux_cpl(kpie,kpje))
      co2flux_cpl(:,:) = 0.0_wp
    ENDIF
    IF (l_cpl_dust) THEN
      IF ( kchck > 0 ) THEN
        WRITE(io_stdo_bgc,*)'Memory allocation for variable dustdep ...'
        WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
        WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
      ENDIF

      ALLOCATE (dustdep(kpie,kpje))
      dustdep(:,:) = 0.0_wp

      ! js changed from else to endif for supervolcano exp.
      ! BALLASTING: added LGM dust separately
    ENDIF
    IF ( kchck > 0 ) THEN
      WRITE(io_stdo_bgc,*)'Memory allocation for variable dusty ...'
      WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
      WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
    ENDIF
    ALLOCATE (dusty(kpie,kpje))
    dusty(:,:) = 0.0_wp
    ALLOCATE (dusty_lgm(kpie,kpje))
    dusty_lgm(:,:) = 0.0_wp
    ! end insert from esm version

    IF ( kchck > 0 ) THEN
      WRITE(io_stdo_bgc,*)'Memory allocation for variable n2budget ...'
      WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
      WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
      WRITE(io_stdo_bgc,*)'Third dimension    : ',kpke
    ENDIF
    ALLOCATE (n2budget(kpie,kpje,kpke))
    n2budget = 0._wp
    IF ( kchck > 0 ) THEN
      WRITE(io_stdo_bgc,*)'Memory allocation for variable h2obudget ...'
      WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
      WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
      WRITE(io_stdo_bgc,*)'Third dimension    : ',kpke
    ENDIF
    ALLOCATE (h2obudget(kpie,kpje,kpke))
    h2obudget(:,:,:) = 0._wp

    ! particle BALLASTING
    IF ( kchck > 0 ) THEN
      WRITE(io_stdo_bgc,*)'Memory allocation for variable wball ...'
      WRITE(io_stdo_bgc,*)'First dimension    : ',kpie
      WRITE(io_stdo_bgc,*)'Second dimension   : ',kpje
      WRITE(io_stdo_bgc,*)'Third dimension    : ',kpke
    ENDIF
    ALLOCATE (wball(kpie,kpje,kpke))
    ALLOCATE (wballuncor(kpie,kpje,kpke))
    ALLOCATE (wballdd(kpie,kpje,kpke))
    ALLOCATE (wballc(kpie,kpje,kpke))
    ALLOCATE (wballo(kpie,kpje,kpke))
    ALLOCATE (wballd(kpie,kpje,kpke))
    ALLOCATE (wball_poc(kpie,kpje,kpke))
    ALLOCATE (wball_cal(kpie,kpje,kpke))
    ALLOCATE (wball_opa(kpie,kpje,kpke))
    ALLOCATE (wball_dus(kpie,kpje,kpke))
    wball(:,:,:) = 0._wp
    wballuncor(:,:,:) = 0._wp
    wballdd(:,:,:) = 0._wp
    wballc(:,:,:) = 0._wp
    wballo(:,:,:) = 0._wp
    wballd(:,:,:) = 0._wp
    wball_poc(:,:,:) = 0._wp
    wball_cal(:,:,:) = 0._wp
    wball_opa(:,:,:) = 0._wp
    wball_dus(:,:,:) = 0._wp

  END SUBROUTINE ALLOC_MEM_CARBCH

END MODULE mo_carbch
