!>
!! @brief Definition of varlists for I/O
!!
!! @author NN MPI-M
!!
!! @par Revision History
!!
!! First version by N:N:
!!
!!
!! @par Copyright
!! 2002-2013 by MPI-M
!! This software is provided for non-commercial use only.
!! See the LICENSE and the WARRANTY conditions.
!!
MODULE mo_bgc_varlist

  USE mo_kind, ONLY     : dp, sp, wp
  USE mo_carbch, ONLY   : dusty,dusty_lgm,wball,wballo,wballc, &
       &                  wballd,wballdd,wballuncor, &
       &                  wball_poc,wball_cal,wball_opa,wball_dus
  USE mo_mpi,   ONLY    : p_io,p_pe
  USE mo_control_bgc, ONLY : dtb, inv_dtbgc, l_cyadyn
  USE mo_varlist, ONLY  : varlist, new_var, varlist_add,   &
       &                  apply_gridids, generate_zaxisid, &
       print_varlist
  USE mo_diagnosis, ONLY: reference_area  ! global_ocean_surface_area


  USE mo_bgc_diagnostic, ONLY : varlist_add_profile, PROF,                       &
       &                              bgcflux, bgcprod, bgcomz, bgc_o_pro,       &
       &                              bgcrate, bgcsedfl,kpho,                    &
       &                              kphosy, kgraz, kexport, kdelsil, kdelcar,  &
       &                              kdmsprod, kdms_bac, kdms_uv, kflim, kplim, &
       &                              krem, kexudp, kpdy, kbacr, kexudz, kzdy,   &
       &                              kremeu, kremnit,kremsulf, kaou, kbacr_cya, &
       &                              knlim, kco2flux, kco214f, ko2flux,  &
       &                              kn2flux, kn2oflux, kdmsflux, kprorca, kprcaca,   &
       &                              ksilpro, kprodus, kkwco2, kpco2, kco2fxd,  &
       &                              kco2fxu, kopex90, kopex1000, kopex2000, kcaex90, &
       &                              kcaex1000, kcaex2000, kcoex90, kcoex1000,  &
       &                              kcoex2000, kc13flux, kc14flux, knh3flux,   &
       &                              kn2fix, klysokl, kfeatm, komz, komz_depth, &
       &                              kdpco2, kdpo2, kdissol, kdenit, ksco212,   &
       &                              kalkali, kphosph, kano3, ksilica, kiron,   &
       &                              ksedic, ksedal, ksedph, ksedox, ksedn2,    &
       &                              ksedno3, ksedsi, ksedfe,koxygen,kdetritus, &
       &                              kn2fix_cya, kphosy_cya, kcyal_det, kcyal_doc,&
       &                              kpdy_deep, kzdy_deep, kcyafac

  IMPLICIT NONE

  PRIVATE

  INTEGER  :: ncode, n, n0

#include "pointer_intent_macro.inc"
  INCLUDE 'cdi.inc'

  TYPE (varlist), POINTER :: bgc_varlist

  PUBLIC :: build_bgc_varlist, bgc_varlist, register_bgc_diagnostic

CONTAINS


  SUBROUTINE generate_sed_zaxisid(this)

#ifndef NOCDI
    USE mo_parallel, ONLY   : p_io, p_pe
#endif

    USE mo_sedmnt, ONLY     : z_sed
    USE mo_param1_bgc, ONLY : ks

    TYPE (varlist), POINTERINTENT(in) :: this
    TYPE (varlist), POINTER :: current

#ifndef NOCDI
    INTEGER :: sed_zaxisid

    REAL(wp) :: sed_zaxis(ks)

    !   sediment
    sed_zaxis(:) = 1000.0_wp * z_sed(:)
    sed_zaxisid = zaxiscreate(ZAXIS_GENERIC, ks)

    CALL zaxisDefName(sed_zaxisid, 'sediment')
    CALL zaxisDefLongname(sed_zaxisid, 'depth_of_sediment')
    CALL zaxisDefUnits(sed_zaxisid, 'mm')

    CALL zaxisdeflevels(sed_zaxisid, sed_zaxis)

    IF ( p_pe == p_io ) THEN

       current => this

       DO WHILE ( ASSOCIATED(current) )

          IF (current%vardata%zaxis == 'sed') THEN

             current%vardata%zaxisid  = sed_zaxisid

          ENDIF

          current => current%next ! make current alias of next varnode

       END DO

    ENDIF

#endif

  END SUBROUTINE generate_sed_zaxisid

  ! ---------------------------------------------------------------------

  SUBROUTINE build_bgc_varlist

    USE mo_parallel, ONLY : p_pe, p_io

    USE mo_param1_bgc,ONLY : isco212, ialkali, iphosph, iano3, isilica, iiron,        &
         &                   ioxygen, idet, icalc, iphy, igasnit, iopal, izoo, idoc,  &
         &                   ian2o, idms, ifdust, iiron, issso12, isssc12, issssil,   &
         &                   issster, ipowaic, ipowaal, ipowaph, ipowaox, ipown2,     &
         &                   isco213, isco214, idet13, idet14, icalc13, icalc14,      &
         &                   issso13, issso14, isssc13, isssc14, ipowc13, ipowc14,    &
         &                   inos, iadust,                                            &
         &                   ipowno3, ipowasi, ipowafe, iatmco2, iatmo2, iatmn2,      &
         &                   idoccya, icya

    USE mo_biomod, ONLY: rcar, riron

    USE mo_carbch,ONLY : ocetra, hi, co3, aksp, atm

    USE mo_sedmnt,ONLY : sedlay, burial, sedhpl, powtra,seddiag,kdenit_sed,ksred_sed,seddenit

    USE mo_inventory, ONLY : totalcarbon, zco2flux, totalcarboce,                         &
         &                   totalcarbsed, calcitesediment, organsed, totalphos, watphos, &
         &                   sedphos, totalsil, watsil, sedsil, silinpglint, orginpglint, &
         &                   calcinpglint

    USE mo_eddydiag, ONLY : vo20, vpo40, vdet0,               &
         uo20, upo40, udet0, wo20, wpo40, wdet0              

    IMPLICIT NONE

    REAL(wp), PARAMETER :: kilo = 1.e3_wp
    REAL(wp), PARAMETER :: s2year = 3.1536E7_wp
    REAL(wp)            :: ptogtc
    REAL(wp), PARAMETER :: ctogtc = 12._wp*1.E-12_wp  ! kmol C to GtC
    REAL(wp), PARAMETER :: ntotgn = 14._wp*1.E-9_wp  ! kmol N to TgN

    INTEGER n,n0,ncode

    REAL(wp) :: inv_reference_area
    inv_reference_area = 1.0_wp / reference_area
    !print*,'bgc varlist reference area ', reference_area

    !fixme: this can become a parameter 
    ptogtc = rcar*12._wp*1.E-12_wp  ! kmol P to GtC

    ! build up the list

    NULLIFY(bgc_varlist) ! initially nullify list (empty)

    CALL varlist_add(bgc_varlist, &
         new_var('sco212', 'dissolved_inorganic_carbon', &
         'kmol C m-3', 7, ocetra, isco212, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('alkali', 'total_alkalinity', &
         'kmol m-3', 10, ocetra, ialkali, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('phosph', 'dissolved_phosphate', &
         'kmol P m-3', 11, ocetra, iphosph, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('ano3', 'dissolved_nitrate', &
         'kmol N m-3', 14, ocetra, iano3, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('silica', 'dissolved_silicate', &
         'kmol Si m-3', 15, ocetra, isilica, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('iron', 'iron_concentration', &
         'kmol Fe m-3', 31, ocetra, iiron, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('oxygen', 'oxygen_concentration', &
         'kmol O2 m-3', 12, ocetra, ioxygen, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('gasnit', 'gaseous_nitrogen', &
         'kmol N2 m-3', 13, ocetra, igasnit, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('doc', 'dissolved_organic_carbon', &
         'kmol P m-3', 16, ocetra, idoc, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('det', 'detrital_organic_carbon', &
         'kmol P m-3', 17, ocetra, idet, 'p', 'c'))

    CALL varlist_add(bgc_varlist, &
         new_var('hi', 'hydrogen ion_concentration', &
         'kmol m-3', 20, hi, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('co3', 'dissolved carbonate', &
         'kmol C m-3', 21, co3, 'p', 'c'))

    CALL varlist_add(bgc_varlist, &
         new_var('phy', 'phytoplankton_concentration', &
         'kmol P m-3', 22, ocetra, iphy, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('zoo', 'zooplankton_concentration', &
         'kmol P m-3', 23, ocetra, izoo, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('calc', 'calcium carbonate', &
         'kmol C m-3', 24, ocetra, icalc, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('opal', 'biogenic silica', &
         'kmol Si m-3', 27, ocetra, iopal, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('an2o', 'laughing gas', &
         'kmol N2 m-3', 28, ocetra, ian2o, 'p', 'c'))

    CALL varlist_add(bgc_varlist, &
         new_var('doccya', 'dissolved_organic_carbon_cyano', &
         'kmol P m-3',144, ocetra, idoccya, 'p', 'c'))

    CALL varlist_add(bgc_varlist, &
         new_var('cya', 'cyanobacteria_concentration', &
         'kmol P m-3',145, ocetra, icya, 'p', 'c'))

    CALL varlist_add(bgc_varlist, &
         new_var('dms', 'dimethylsulfide', &
         'kmol S m-3', 29, ocetra, idms, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('fdust', 'non-aggregated dust', &
         'kg m-3', 30, ocetra, ifdust, 'p', 'c'))
    ! output for particle BALLASTING
    CALL varlist_add(bgc_varlist, &
         new_var('wball', 'particle sinking speed (Gehlen plus dust), not used everywhere', &
         'm/timestep', 288, wball, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('wballuncor', 'wball uncorrected for distances higher than dzw', &
         'm/timestep', 401, wballuncor, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('wballc', 'wball if only POC and calcite were present', &
         'm/timestep', 402, wballc, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('wballo', 'wball if only POC and opal were present', &
         'm/timestep', 403, wballo, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('wballd', 'wball if only POC and dust were present', &
         'm/timestep', 404, wballd, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('wball_poc', 'sinking speed of detritus', &
         'm/timestep', 405, wball_poc, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('wball_cal', 'sinking speed of calcite', &
         'm/timestep', 406, wball_cal, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('wball_opa', 'sinking speed of opal', &
         'm/timestep', 407, wball_opa, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('wball_dus', 'sinking speed of dust', &
         'm/timestep', 408, wball_dus, 'p', 'c'))


    CALL varlist_add(bgc_varlist, &
         new_var('dusty', 'dust', &
         'Fixme', 34, dusty, 'p', 's', input_required=.TRUE.))
    ! LGM dust for BALLASTING
    CALL varlist_add(bgc_varlist, &
         new_var('dusty_lgm', 'dust', &
         'Fixme', 35, dusty_lgm, 'p', 's', input_required=.TRUE.))

    CALL varlist_add(bgc_varlist, &
         new_var('aksp', 'apparent solubility product for calcite', &
         '(kmol m-3)2', 37, aksp, 'p', 'c'))

    CALL varlist_add(bgc_varlist, &
         new_var('ssso12', 'sediment accumulated organic carbon', &
         'kmol P m-3', 38, sedlay, issso12, 'p', 'sed'))
    CALL varlist_add(bgc_varlist, &
         new_var('sssc12', 'sediment accumulated calcium carbonate', &
         'kmol C m-3', 41, sedlay, isssc12, 'p', 'sed'))
    CALL varlist_add(bgc_varlist, &
         new_var('ssssil', 'sediment accumulated opal', &
         'kmol Si m-3', 44, sedlay, issssil, 'p', 'sed'))
    CALL varlist_add(bgc_varlist, &
         new_var('ssster', 'sediment accumulated clay', &
         'kmol m-3', 45, sedlay, issster, 'p', 'sed'))

    CALL varlist_add(bgc_varlist, &
         new_var('bsso12', 'burial layer of organic carbon', &
         'kmol P m-2', 46, burial, issso12, 'p', 'g'))
    CALL varlist_add(bgc_varlist, &
         new_var('bssc12', 'burial layer of calcium carbonate', &
         'kmol C m-2', 47, burial, isssc12, 'p', 'g'))
    CALL varlist_add(bgc_varlist, &
         new_var('bsssil', 'burial layer of opal', &
         'kmol Si m-2', 48, burial, issssil, 'p', 'g'))
    CALL varlist_add(bgc_varlist, &
         new_var('bsster', 'burial layer of clay', &
         'kmol m-2', 49, burial, issster, 'p', 'g'))

    CALL varlist_add(bgc_varlist, &
         new_var('sedhpl', 'sediment accumulated hydrogen ions', &
         'kmol m-3', 50, sedhpl, 'p', 'sed'))

    CALL varlist_add(bgc_varlist, &
         new_var('powaic', 'sediment pore water dic', &
         'kmol C m-3', 51, powtra, ipowaic, 'p', 'sed'))
    CALL varlist_add(bgc_varlist, &
         new_var('powaal', 'sediment pore water alkalinity', &
         'kmol m-3', 54, powtra, ipowaal, 'p', 'sed'))
    CALL varlist_add(bgc_varlist, &
         new_var('powaph', 'sediment pore water phosphate', &
         'kmol P m-3', 55, powtra, ipowaph, 'p', 'sed'))
    CALL varlist_add(bgc_varlist, &
         new_var('powaox', 'sediment pore water oxygen', &
         'kmol O2 m-3', 56, powtra, ipowaox, 'p', 'sed'))
    CALL varlist_add(bgc_varlist, &
         new_var('pown2', 'sediment pore water gaseous nitrogen', &
         'kmol N2 m-3', 57, powtra, ipown2, 'p', 'sed'))
    CALL varlist_add(bgc_varlist, &
         new_var('powno3', 'sediment pore water nitrate (no3)', &
         'kmol N m-3', 58, powtra, ipowno3, 'p', 'sed'))
    CALL varlist_add(bgc_varlist, &
         new_var('powasi', 'sediment pore water silicid acid (si(oh)4)', &
         'kmol Si m-3', 59, powtra, ipowasi, 'p', 'sed'))
    CALL varlist_add(bgc_varlist, &
         new_var('powafe', 'sediment pore water iron ', &
         'kmol Fe m-3', 60, powtra, ipowafe, 'p', 'sed'))

    CALL varlist_add(bgc_varlist, &
         new_var('atmco2', 'atmospheric CO2', &
         'ppm', 61, atm, iatmco2, 'p', 's'))
    CALL varlist_add(bgc_varlist, &
         new_var('atmo2', 'atmospheric O2', &
         'ppm', 62, atm, iatmo2, 'p', 's'))
    CALL varlist_add(bgc_varlist, &
         new_var('atmn2', 'atmospheric N2', &
         'ppm', 64, atm, iatmn2, 'p', 's'))

    !#ifdef __c_isotopes
    CALL varlist_add(bgc_varlist, &
         new_var('sco213', 'dissolved_inorganic_carbon13', &
         'kmol m-3', 8, ocetra, isco213, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('sco214', 'dissolved_inorganic_carbon14', &
         'kmol m-3', 9, ocetra, isco214, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('det13', 'particulate organic carbon13', &
         'kmolP m-3', 18, ocetra, idet13, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('det14', 'particulate organic carbon14', &
         'kmolP m-3', 19, ocetra, idet14, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('calc13', 'calcium carbonate13', &
         'kmolP m-3', 25, ocetra, icalc13, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('calc14', 'calcium carbonate14', &
         'kmolP m-3', 26, ocetra, icalc14, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('ssso13', 'sediment accumulated organic carbon13', &
         'kmol m-2', 39, sedlay, issso13, 'p', 'sed'))
    CALL varlist_add(bgc_varlist, &
         new_var('ssso14', 'sediment accumulated organic carbon14', &
         'kmol m-2', 40, sedlay, issso14, 'p', 'sed'))
    CALL varlist_add(bgc_varlist, &
         new_var('sssc13', 'sediment accumulated calcium carbonate13', &
         'kmol m-2', 42, sedlay, isssc13, 'p', 'sed'))
    CALL varlist_add(bgc_varlist, &
         new_var('sssc14', 'sediment accumulated calcium carbonate14' &
         , 'kmol m-2', 43, sedlay, isssc14, 'p', 'sed'))
    CALL varlist_add(bgc_varlist, &
         new_var('powc13', 'sediment pore water dic13', &
         'kmol m-3', 52, powtra, ipowc13, 'p', 'sed'))
    CALL varlist_add(bgc_varlist, &
         new_var('powc14', 'sediment pore water dic14', &
         'kmol m-3', 53, powtra, ipowc14, 'p', 'sed'))
    CALL varlist_add(bgc_varlist, &
         new_var('nos', 'marine snow aggregates per cm3', &
         'cm-3', 32, ocetra, inos, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('adust', 'aggregated dust', &
         'kg m-3', 33, ocetra, iadust, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('denit_sed', 'sediment denitrification', &
         'kmol N m-3', 167, seddiag, kdenit_sed, 'p', 'sed'))
    CALL varlist_add(bgc_varlist, &
         new_var('sred_sed', 'sediment sulfate reduction', &
         'kmol P m-3', 168, seddiag, ksred_sed, 'p', 'sed'))
    CALL varlist_add(bgc_varlist, &
         new_var('denit_sed_int', 'integrated sediment denitrification', &
         'kmol N  m-2', 169, seddenit, 'p', 'g'))


    ! variables from bgcmean_bioz, vertical resolvled
    CALL varlist_add(bgc_varlist, &
         new_var('phosy', 'photosynthesis', &
         'mol C m-3 s-1', 100, bgcprod, kphosy, 'p', 'c', &
         factor = rcar * kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('EU_pho', 'EU_Tlim_Llim', &
         'd-1', 510, bgcprod, kpho, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('graz', 'zoo_grazing', &
         'mol Fe m-3 s-1', 101, bgcprod, kgraz, 'p', 'c', &
         factor = riron * kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('export', 'detritus_production', &
         'kmol C m-3 s-1', 84, bgcprod, kexport, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('delsil', 'opal_production', &
         'kmol Si m-3 s-1', 86, bgcprod, kdelsil, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('delcar', 'calcium_production', &
         'kmol C m-3 s-1', 85, bgcprod, kdelcar, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('dmsprod', 'dms_production', &
         'kmol S m-3 s-1', 69, bgcprod, kdmsprod, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('dms_bac', 'dms_bacterial_consumption', &
         'kmol S m-3 s-1', 70, bgcprod, kdms_bac, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('dms_uv', 'dms_uv_light_destruction', &
         'kmol S m-3 s-1', 71, bgcprod, kdms_uv, 'p', 'c'))

    CALL varlist_add(bgc_varlist, &
         new_var('flim', 'fe limitation_of_pp', &
         ' ', 138, bgcprod, kflim, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('plim', 'p_limitation_of_pp', &
         ' ', 139, bgcprod, kplim, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('nlim', 'n_limitation_of_pp', &
         ' ', 140, bgcprod, knlim, 'p', 'c'))

    CALL varlist_add(bgc_varlist, &
         new_var('phosy_cya', 'photosynthesis_cyano', &
         'mol C m-3 s-1', 141, bgc_o_pro, kphosy_cya, 'p', 'c', &
         factor = rcar * kilo))

    CALL varlist_add(bgc_varlist, &
         new_var('cyalos_det', 'cyano_loss_det', &
         'mol C m-3 s-1', 142, bgc_o_pro, kcyal_det, 'p', 'c', &
         factor = rcar * kilo))
    
    CALL varlist_add(bgc_varlist, &
         new_var('doccya_fac', 'doc_det_fractionation_cya', &
         '', 298, bgc_o_pro, kcyafac, 'p', 'c'))

    CALL varlist_add(bgc_varlist, &
         new_var('cyalos_doc', 'cyano_loss_doc', &
         'mol C m-3 s-1', 146, bgc_o_pro, kcyal_doc, 'p', 'c', &
         factor = rcar * kilo))

    CALL varlist_add(bgc_varlist, &
         new_var('remin_via_grazer', 'remin_via_grazer', &
         'kmol P m-3 s-1', 292, bgcprod, krem, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('bacterial_activity', 'bacterial_activity', &
         'kmol P m-3 s-1', 293, bgcprod, kbacr, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('remin_of_det_by_O2_', 'remin_of_detritus_by_oxygen', &
         'kmol P m-3 s-1', 294, bgcprod, kremeu, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('remin_of_det_by_N', 'remin_of_detritus_by_nitrogen', &
         'kmol P m-3 s-1', 295, bgcprod, kremnit, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('remin_of_det_by_S', 'remin_of_detritus_by_sulfate', &
         'kmol P m-3 s-1', 296, bgcprod, kremsulf, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('degr_bac_doc_cya', 'microbial degradation of DOC_cya', &
         'kmol P m-3 s-1', 297, bgcprod, kbacr_cya, 'p', 'c'))



    ! variables from bgcmean_2d
    CALL varlist_add(bgc_varlist, &
         new_var('co2flux', 'co2flux ', &
         'kg C m-2 s-1', 92, bgcflux, kco2flux, 'p', 's', factor=12._wp))
    CALL varlist_add(bgc_varlist, &
         new_var('co214f', 'co214flux', &
         'kmol m-2 s-1', 91, bgcflux, kco214f, 'p', 's'))
    CALL varlist_add(bgc_varlist, &
         new_var('o2flux', 'oxygen_flux', &
         'mol O2 m-2 s-1', 72, bgcflux, ko2flux, 'p', 's', factor=kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('n2flux', 'nitrogen_flux', &
         'kmol N2 m-2 s-1', 74, bgcflux, kn2flux, 'p', 's'))
    CALL varlist_add(bgc_varlist, &
         new_var('n2oflux', 'n2oflux', &
         'kmol N2 m-2 s-1', 93, bgcflux, kn2oflux, 'p', 's'))
    CALL varlist_add(bgc_varlist, &
         new_var('dmsflux', 'dms_flux', &
         'mol S m-2 s-1', 68, bgcflux, kdmsflux, 'p', 's', factor=kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('prorca', 'detritus sediment flux', &
         'mol P m-2 s-1', 94, bgcflux, kprorca, 'p', 'g', factor=kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('prcaca', 'caco3 sediment flux', &
         'mol C m-2 s-1', 95, bgcflux, kprcaca, 'p', 'g', factor=kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('silpro', 'opal sediment flux', &
         'kmol Si m-2 s-1', 96, bgcflux, ksilpro, 'p', 'g'))
    CALL varlist_add(bgc_varlist, &
         new_var('produs', 'dust flux to sediment (free+aggregated)', &
         'kmol m-2 s-1', 97, bgcflux, kprodus, 'p', 'g'))
    CALL varlist_add(bgc_varlist, &
         new_var('kwco2', 'co2_exchange_coefficient', &
         'kmol m-2 s-1 ppm-1', 73, bgcflux, kkwco2, 'p', 's'))
    CALL varlist_add(bgc_varlist, &
         new_var('pco2', 'co2_partial_pressure', &
         'ppm', 67, bgcflux, kpco2, 'p', 's'))
    CALL varlist_add(bgc_varlist, &
         new_var('co2fxd', 'co2_flux_down', &
         'kmol C m-2 s-1', 65, bgcflux, kco2fxd, 'p', 's'))
    CALL varlist_add(bgc_varlist, &
         new_var('co2fxu', 'co2_flux_up', &
         'kmol C m-2 s-1', 66, bgcflux, kco2fxu, 'p', 's'))
    CALL varlist_add(bgc_varlist, &
         new_var('opex90', 'opal flux in 90 m', &
         'mol Si m-2 s-1', 75, bgcflux, kopex90, 'p', 's=90', factor = kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('opex1000', 'opal flux in 1000 m', &
         'mol Si m-2 s-1', 76, bgcflux, kopex1000, 'p', 's=1000', &
         factor = kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('opex2000', 'opal flux in 2000 m', &
         'mol Si m-2 s-1', 77, bgcflux, kopex2000, 'p', 's=2000', &
         factor = kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('caex90', 'calc flux in 90 m', &
         'mol C m-2 s-1', 78, bgcflux, kcaex90, 'p', 's=90', factor = kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('caex1000', 'calc flux in 1000 m', &
         'mol C m-2 s-1', 79, bgcflux, kcaex1000, 'p', 's=1000', &
         factor = kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('caex2000', 'calc flux in 2000 m', &
         'mol C m-2 s-1', 80, bgcflux, kcaex2000, 'p', 's=2000', &
         factor = kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('coex90', 'om flux in 90 m', &
         'mol C m-2 s-1', 81, bgcflux, kcoex90, 'p', 's=90', &
         factor = rcar * kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('coex1000', 'om flux in 1000 m', &
         'mol C m-2 s-1', 82, bgcflux, kcoex1000, 'p', 's=1000', &
         factor = rcar * kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('coex2000', 'om flux in 2000 m', &
         'mol C m-2 s-1', 83, bgcflux, kcoex2000, 'p', 's=2000', &
         factor = rcar * kilo))

    ! all optional variables from bgcmean files

    !#ifdef __c_isotopes
    CALL varlist_add(bgc_varlist, &
         new_var('c13flux', 'c13 air-sea-flux', &
         'kmol m-2 s-1', 103, bgcflux, kc13flux, 'p', 's'))
    CALL varlist_add(bgc_varlist, &
         new_var('c14flux', 'c14 air-sea-flux', &
         'kmol m-2 s-1', 104, bgcflux, kc14flux, 'p', 's'))
    !#ifdef AMMO
    CALL varlist_add(bgc_varlist, &
         new_var('nh3flux', 'nh3 sea-air-flux', &
         'kmol N m-2 s-1', 105, bgcflux, knh3flux, 'p', 's'))
    !#endif
    CALL varlist_add(bgc_varlist, &
         new_var('n2fix_diag', 'diagnostic_nitrogen_fixation_rate', &
         'mol N m-2 s-1', 157, bgcflux, kn2fix, 'p', 's', factor = kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('lysokl', 'depth_of_lysokline', &
         'm', 160, bgcflux, klysokl, 'p', 's'))
    CALL varlist_add(bgc_varlist, &
         new_var('featm', 'surface_flux_of_iron', &
         'mol Fe m-2 s-1', 161, bgcflux, kfeatm, 'p', 's', factor = kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('o2min', 'oxygen_minimum_concentration', &
         'mol O2 m-3', 162, bgcomz, komz, 'p', 's', factor = kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('zo2min', 'depth_of_oxygen_minimum_concentration', &
         'm', 163, bgcomz, komz_depth, 'p', 's'))
    CALL varlist_add(bgc_varlist, &
         new_var('dpco2', 'delta_co2_partial_pressure', &
         'ppm',164, bgcflux, kdpco2, 'p', 's'))
    CALL varlist_add(bgc_varlist, &
         new_var('dpo2', 'delta_o2_partial_pressure', &
         'ppm', 165, bgcflux, kdpo2, 'p', 's'))

    CALL varlist_add(bgc_varlist, &
         new_var('caco3_diss', 'dissolution of caco3', &
         'kmol C m-3 s-1', 158, bgc_o_pro, kdissol, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('denitr', 'denitrification', &
         'mol N m-2 s-1', 159, bgc_o_pro, kdenit, 'p', 'si', factor = kilo))

    CALL varlist_add(bgc_varlist, &
         new_var('n2fix_cya_vol', 'cyanobacteria_nitrogen_fixation', &
         'mol N m-3 s-1', 166, bgc_o_pro, kn2fix_cya, 'p', 'c', factor = kilo))



    ! CMIP5 surface tracer concentrations [units mol/m*3]
    CALL varlist_add(bgc_varlist, &
         new_var('surf_sco212', 'surface_dissolved_inorganic_carbon', &
         'mol C m-3', 107, ocetra, isco212, 'p', 's=0', factor = kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('surf_alkali', 'surface_total_alkalinity', &
         'mol m-3', 110, ocetra, ialkali, 'p', 's=0', factor = kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('surf_phosph', 'surface_dissolved_phosphate', &
         'mol P m-3', 111, ocetra, iphosph, 'p', 's=0', factor = kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('surf_ano3', 'surface_dissolved_nitrate', &
         'mol N m-3', 114, ocetra, iano3, 'p', 's=0', factor = kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('surf_silica', 'surface_dissolved_silicate', &
         'mol Si m-3', 115, ocetra, isilica, 'p', 's=0', factor = kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('surf_iron', 'surface_iron_concentration', &
         'mol Fe m-3', 131, ocetra, iiron, 'p', 's=0', factor = kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('surf_oxygen', 'surface_oxygen_concentration', &
         'mol O2 m-3', 112, ocetra, ioxygen, 'p', 's=0', factor = kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('surf_doc', 'surface_dissolved_organic_carbon', &
         'mol C m-3', 116, ocetra, idoc, 'p', 's=0', factor = rcar*kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('surf_det', 'surface_detrital_organic_carbon', &
         'mol C m-3', 117, ocetra, idet, 'p', 's=0', factor = rcar*kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('surf_hi', 'surface_hydrogen_ion_concentration', &
         'kmol m-3', 120, hi, 'p', 's=0'))
    CALL varlist_add(bgc_varlist, &
         new_var('co3', 'surface_dissolved_carbonate_concentration', &
         'mol C m-3', 121, co3, 'p', 's=0', factor = kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('surf_phy', 'surface_phytoplankton_concentration', &
         'mol C m-3', 122, ocetra, iphy, 'p', 's=0', factor = rcar*kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('surf_zoo', 'surface_zooplankton_concentration', &
         'mol C m-3', 123, ocetra, izoo, 'p', 's=0', factor = rcar*kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('surf_calc', 'surface_calcium_carbonate', &
         'mol C m-3', 124, ocetra, icalc, 'p', 's=0', factor = kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('surf_opal', 'surface_biogenic_silica', &
         'mol Si m-3', 127, ocetra, iopal, 'p', 's=0', factor = kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('surf_dms', 'surface_dimethylsulfide', &
         'mol S m-3', 129, ocetra, idms, 'p', 's=0', factor = kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('surf_aksp', 'surface_apparent_solubility_product_for_calcite', &
         '(kmol m-3)2',137, aksp, 'p', 's=0'))

    CALL varlist_add(bgc_varlist, &
         new_var('aou', 'aparent_oxygen_ut', &
         'kmol O2 m-3', 143, bgcprod, kaou, 'p', 'c'))



    !CMIP5 integrated  variables
    CALL varlist_add(bgc_varlist, &
         new_var('intpp', 'integrated_primary_production', &
         'mol C m-2 s-1', 200, bgcprod, kphosy, 'p', 'si', factor = rcar*kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('intdelsil', 'integrated_opal_production', &
         'mol Si m-2 s-1', 186, bgcprod, kdelsil, 'p', 'si', factor = kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('intdelcar', 'integrated_calcium_production', &
         'mol C m-2 s-1', 185, bgcprod, kdelcar, 'p', 'si', factor = kilo))
    CALL varlist_add(bgc_varlist, &
         new_var('intpp_cya', 'integrated_primary_production_cyano', &
         'mol C m-2 s-1', 201, bgc_o_pro, kphosy_cya, 'p', 'si', factor = rcar*kilo))

    ! carbon cycle diagnostic
    CALL varlist_add(bgc_varlist, &
         new_var('calcinpglint', 'total_inorganic_calcium_input_to_sediment', &
         'kmol', 203, calcinpglint, 'g', 'g'))
    CALL varlist_add(bgc_varlist, &
         new_var('orginpglint', 'total_organic_calcium_input_to_sediment', &
         'kmol', 204, orginpglint, 'g', 'g'))
    CALL varlist_add(bgc_varlist, &
         new_var('silinpglint', 'total_silicat_input_to_sediment', &
         'kmol', 205, silinpglint, 'g', 'g'))

    CALL varlist_add(bgc_varlist, &
         new_var('intsco212', 'integrated dissolved_inorganic_carbon', &
         'kg C m-2', 206, ocetra, isco212, 'p', 'si', factor = 12._wp))

    !CMIP5 integrated variables top 100 m
    CALL varlist_add(bgc_varlist, &
         new_var('fddtdic', 'change in dissolved_inorganic_carbon', &
         'mol C m-2 s-1', 207, bgcrate, ksco212, 'p', 'si<100', factor = kilo*inv_dtbgc))
    CALL varlist_add(bgc_varlist, &
         new_var('fddtalk', 'change in total_alkalinity', &
         'mol m-2 s-1', 210, bgcrate, kalkali, 'p', 'si<100', factor = kilo*inv_dtbgc))
    CALL varlist_add(bgc_varlist, &
         new_var('fddtdip', 'change in dissolved_phosphate', &
         'mol P m-2 s-1', 211, bgcrate, kphosph, 'p', 'si<100', factor = kilo*inv_dtbgc))
    CALL varlist_add(bgc_varlist, &
         new_var('fddtdin', 'change in dissolved_nitrate', &
         'mol N m-3', 214, bgcrate, kano3, 'p', 'si<100', factor = kilo*inv_dtbgc))
    CALL varlist_add(bgc_varlist, &
         new_var('fddtdisi', 'change in dissolved_silicate', &
         'mol Si m-2 s-1', 215, bgcrate, ksilica, 'p', 'si<100', factor = kilo*inv_dtbgc))
    CALL varlist_add(bgc_varlist, &
         new_var('fddtdife', 'change in iron_concentration', &
         'mol Fe m-2 s-1', 231, bgcrate, kiron, 'p', 'si<100', factor = kilo*inv_dtbgc))
   CALL varlist_add(bgc_varlist, &
         new_var('fddtdoxy', 'change in oxygen_concentration', &
         'mol O m-2 s-1', 232, bgcrate, koxygen, 'p', 'si<100', factor = kilo*inv_dtbgc))
   CALL varlist_add(bgc_varlist, &
         new_var('fddtddet', 'change in detritus_concentration', &
         'mol P m-2 s-1', 233, bgcrate, kdetritus, 'p', 'si<100', factor = kilo*inv_dtbgc))
   CALL varlist_add(bgc_varlist, &
         new_var('doxydt', 'change in oxygen_concentration', &
         'mol O m-3 s-1', 234, bgcrate, koxygen, 'p', 'c', factor = kilo*inv_dtbgc))
   CALL varlist_add(bgc_varlist, &
         new_var('ddetdt', 'change in detritus_concentration', &
         'mol P m-3 s-1', 235, bgcrate, kdetritus, 'p', 'c', factor = kilo*inv_dtbgc))
    CALL varlist_add(bgc_varlist, &
         new_var('ddipdt', 'change in dissolved_phosphate', &
         'mol P m-3 s-1', 236, bgcrate, kphosph, 'p', 'c', factor = kilo*inv_dtbgc))

    CALL varlist_add(bgc_varlist, &
         new_var('sedfludic', 'sedimentflux_to_ocean_dic', &
         'kmol C m-2 s-1', 280, bgcsedfl, ksedic, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('sedflualk', 'sedimentflux_to_ocean_alk', &
         'kmol m-2 s-1', 281, bgcsedfl, ksedal, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('sedflupo4', 'sedimentflux_to_ocean_phosphate', &
         'kmol P m-2 s-1', 282, bgcsedfl, ksedph, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('sedfluo2', 'sedimentflux_to_ocean_oxygen', &
         'kmol O2 m-2 s-1', 283, bgcsedfl, ksedox, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('sedflun2', 'sedimentflux_to_ocean_nitrogen', &
         'kmol N m-2 s-1', 284, bgcsedfl, ksedn2, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('sedfluno3', 'sedimentflux_to_ocean_nitrate', &
         'kmol N m-2 s-1', 285, bgcsedfl, ksedno3, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('sedflusi', 'sedimentflux_to_ocean_silicate', &
         'kmol Si m-2 s-1', 286, bgcsedfl, ksedsi, 'p', 'c'))
    CALL varlist_add(bgc_varlist, &
         new_var('sedflufe', 'sedimentflux_to_ocean_iron', &
         'kmol Fe m-2 s-1', 287, bgcsedfl, ksedfe, 'p', 'c'))

    CALL varlist_add(bgc_varlist, &
         new_var('totalcarbon', 'total_carbon', &
         'mol C', 301, totalcarbon, 'g', 's', factor = kilo))

    CALL varlist_add(bgc_varlist, &
         new_var('zco2flux', 'global_carbon_flux', &
         'mol C/day', 303, zco2flux, 'g', 's', factor = kilo))

    CALL varlist_add(bgc_varlist, &
         new_var('totalcarboce', 'total_carbon_of_water_column_incl_pore_water', &
         'mol C', 305, totalcarboce, 'g', 's', factor = kilo))

    CALL varlist_add(bgc_varlist, &
         new_var('totalcarbsed', 'total_carbon_of_sediment', &
         'mol C', 306, totalcarbsed, 'g', 's', factor = kilo))

    CALL varlist_add(bgc_varlist, &
         new_var('calcitesediment', 'total_calcite_of_sediment', &
         'mol C', 307, calcitesediment, 'g', 's', factor = kilo))

    CALL varlist_add(bgc_varlist, &
         new_var('organsed', 'organic_carbon_of_sediment', &
         'mol C', 308, organsed, 'g', 's', factor = kilo))

    CALL varlist_add(bgc_varlist, &
         new_var('totalphos', 'total_phosphate', &
         'mol P', 309, totalphos, 'g', 's', factor = kilo))

    CALL varlist_add(bgc_varlist, &
         new_var('watphos', 'total_phosphate_of_water_column_incl_pore_water', &
         'mol P', 310, watphos, 'g', 's', factor = kilo))

    CALL varlist_add(bgc_varlist, &
         new_var('sedphos', 'total_phosphate_of_sediment', &
         'mol P', 311, sedphos, 'g', 's', factor = kilo))

    CALL varlist_add(bgc_varlist, &
         new_var('totalsil', 'total_silicate', &
         'mol Si', 312, totalsil, 'g', 's', factor = kilo))

    CALL varlist_add(bgc_varlist, &
         new_var('watsil', 'total_silicate_of_water_column_incl_pore_water', &
         'mol Si', 313, watsil, 'g', 's', factor = kilo))

    CALL varlist_add(bgc_varlist, &
         new_var('sedsil', 'total_silicate_of_sediment', &
         'mol Si', 314, sedsil, 'g', 's', factor = kilo))

    CALL varlist_add(bgc_varlist, &
         new_var('uO20', &
         'product_of_sea_water_x_velocity_and_oxygen_concentration',&
         'mol O m-2 s-1',315,uO20,'u','c', factor = kilo))

    CALL varlist_add(bgc_varlist, &
         new_var('vO20', &
         'product_of_sea_water_y_velocity_and_oxygen_concentration',&
         'mol O m-2 s-1',316,vO20,'v','c', factor = kilo))

    CALL varlist_add(bgc_varlist, &
         new_var('wO20', &
         'product_of_upward_sea_water_velocity_and_oxygen_concentration',&
         'mol O m-2 s-1',317,wO20,'p','i', factor = kilo))

    CALL varlist_add(bgc_varlist, &
         new_var('uPO40', &
         'product_of_sea_water_x_velocity_and_phosphate_concentration',&
         'mol P m-2 s-1',318,uPO40,'u','c', factor = kilo))

    CALL varlist_add(bgc_varlist, &
         new_var('vPO40', &
         'product_of_sea_water_y_velocity_and_phosphate_concentration',&
         'mol P m-2 s-1',319,vPO40,'v','c', factor = kilo))

    CALL varlist_add(bgc_varlist, &
         new_var('wPO40', &
         'product_of_upward_sea_water_velocity_and_phosphate_concentration',&
         'mol P m-2 s-1',320,wPO40,'p','i', factor = kilo))

    CALL varlist_add(bgc_varlist, &
         new_var('udet0', &
         'product_of_sea_water_x_velocity_and_detritus_concentration',&
         'mol P m-2 s-1',321,udet0,'u','c', factor = kilo))

    CALL varlist_add(bgc_varlist, &
         new_var('vdet0', &
         'product_of_sea_water_y_velocity_and_detritus_concentration',&
         'mol P m-2 s-1',322,vdet0,'v','c', factor = kilo))

    CALL varlist_add(bgc_varlist, &
         new_var('wdet0', &
         'product_of_upward_sea_water_velocity_and_detritus_concentration',&
         'mol P m-2 s-1',323,wdet0,'p','i', factor = kilo))


    !Time series (output in netcdf allows higher code numbers)
    CALL varlist_add(bgc_varlist, &
         new_var('global_primary_production', 'global_primary_production', &
         'GtC yr-1', 500, bgcprod, kphosy, 'g', 'si', factor = ptogtc*s2year))
    CALL varlist_add(bgc_varlist, &
         new_var('global_zooplankton_grazing', 'global_zooplankton_grazing', &
         'GtC yr-1', 501, bgcprod, kgraz, 'g', 'si', factor = ptogtc*s2year))
    CALL varlist_add(bgc_varlist, &
         new_var('global_OM_export_at_90m', 'global_om_export_at_90m', &
         'GtC yr-1', 502, bgcflux, kcoex90, 'g', 's', factor = ptogtc*s2year))
    CALL varlist_add(bgc_varlist, &
         new_var('global_calc_export_at_90m', 'global_calc_export_at_90m', &
         'GtC yr-1', 503, bgcflux, kcaex90, 'g', 's', factor = ctogtc*s2year))
    CALL varlist_add(bgc_varlist, &
         new_var('global_opal_export_at_90m', 'global_opal_export_at_90m', &
         'kmol Si s-1', 504, bgcflux, kopex90, 'g', 's'))
    CALL varlist_add(bgc_varlist, &
         new_var('global_opal_production', 'global_opal_production', &
         'kmol Si s-1', 505, bgcprod, kdelsil, 'g', 'si'))
    CALL varlist_add(bgc_varlist, &
         new_var('global_caco3_production', 'global_caco3_production', &
         'GtC yr-1', 506, bgcprod, kdelcar, 'g', 'si', factor = ctogtc*s2year))
    CALL varlist_add(bgc_varlist, &
         new_var('global_net_co2_flux', 'global_net_co2_flux', &
         'GtC yr-1', 507, bgcflux, kco2flux, 'g', 's', factor = ctogtc*s2year))
    CALL varlist_add(bgc_varlist, &
         new_var('global_surface_alkalinity', 'global_surface_alkalinity', &
         'kmol m-3', 508, ocetra, ialkali, 'g', 's=0', factor = inv_reference_area))
    CALL varlist_add(bgc_varlist, &
         new_var('global_surface_dissolved_inorganic_carbon', & 
         'global_surface_dissolved_inorganic_carbon', &
         'kmol C m-3', 509, ocetra, isco212, 'g', 's=0', factor = inv_reference_area))

    CALL varlist_add(bgc_varlist, &
         new_var('EU_remin_via_grazer', 'EU_remin_via_grazer', &
         'GtC yr-1', 511, bgcprod, krem, 'g', 'si', factor = ptogtc*s2year))
    CALL varlist_add(bgc_varlist, &
         new_var('EU_exdudation_phytoplankton', 'EU_exudation_phytoplankton', &
         'GtC yr-1', 512, bgcprod, kexudp, 'g', 'si', factor = ptogtc*s2year))
    CALL varlist_add(bgc_varlist, &
         new_var('EU_phytoplankton_dying', 'EU_phytoplankton_dying', &
         'GtC yr-1', 513, bgcprod, kpdy, 'g', 'si', factor = ptogtc*s2year))
    CALL varlist_add(bgc_varlist, &
         new_var('EU_bacterial_activity', 'EU_bacterial_activity', &
         'GtC yr-1', 514, bgcprod, kbacr, 'g', 'si', factor = ptogtc*s2year))
    CALL varlist_add(bgc_varlist, &
         new_var('EU_exudation_zooplankton', 'EU_exudation_zooplankton', &
         'GtC yr-1', 515, bgcprod, kexudz, 'g', 'si', factor = ptogtc*s2year))
    CALL varlist_add(bgc_varlist, &
         new_var('EU_zooplankton_dying', 'EU_zooplankton_dying', &
         'GtC yr-1', 516, bgcprod, kzdy, 'g', 'si', factor = ptogtc*s2year))

    IF(l_cyadyn)THEN
      CALL varlist_add(bgc_varlist, &
          new_var('EU_N2_fixation', 'EU_N2_fixation', &
         'TgN yr-1', 517, bgc_o_pro, kn2fix_cya, 'g', 'si', factor = ntotgn*s2year))
     ELSE
      CALL varlist_add(bgc_varlist, &
          new_var('EU_N2_fixation', 'EU_N2_fixation', &
         'TgN yr-1', 517, bgcflux, kn2fix, 'g', 's', factor = ntotgn*s2year))
    ENDIF
    CALL varlist_add(bgc_varlist, &
         new_var('EU_N2_flux', 'EU_N2_flux', &
         'TgN yr-1', 518, bgcflux, kn2flux, 'g', 's', factor = ntotgn*s2year))
    CALL varlist_add(bgc_varlist, &
         new_var('WC_denit', 'denitrification', &
         'TgN yr-1', 519, bgc_o_pro, kdenit, 'g', 'si', factor = ntotgn*s2year))
    CALL varlist_add(bgc_varlist, &
         new_var('global_primary_production_cya', 'global_primary_production_cyano', &
         'GtC yr-1', 520, bgc_o_pro, kphosy_cya, 'g','si',factor=ptogtc*s2year))
    CALL varlist_add(bgc_varlist, &
         new_var('EU_bact_degr_DOC_cya', 'EU_microbial degradation DOCcya', &
         'GtC yr-1', 521, bgcprod, kbacr_cya, 'g', 'si', factor = ptogtc*s2year))
    CALL varlist_add(bgc_varlist, &
         new_var('EU_remin_of_detritus', 'EU_remin_of_detritus', &
         'GtC yr-1', 522, bgcprod, kremeu, 'g', 'si', factor = ptogtc*s2year))
    CALL varlist_add(bgc_varlist, &
         new_var('SED_denit', 'sediment denitrification', &
         'TgN yr-1', 523, seddenit, 'g', 'si',factor=ntotgn*s2year))
    CALL varlist_add(bgc_varlist, &
         new_var('global_cya_loss_det', 'global_cyano_loss_det', &
         'GtC yr-1', 524, bgc_o_pro, kcyal_det, 'g','si',factor=ptogtc*s2year))
    CALL varlist_add(bgc_varlist, &
         new_var('global_cya_loss_doc', 'global_cyano_loss_doc', &
         'GtC yr-1', 525, bgc_o_pro, kcyal_doc, 'g','si',factor=ptogtc*s2year))
    CALL varlist_add(bgc_varlist, &
         new_var('DEEP_phytoplankton_dying', 'DEEP_phytoplankton_dying', &
         'GtC yr-1', 526, bgcprod, kpdy_deep, 'g', 'si', factor = ptogtc*s2year))
    CALL varlist_add(bgc_varlist, &
         new_var('DEEP_zooplankton_dying', 'DEEP_zooplankton_dying', &
         'GtC yr-1', 527, bgcprod, kzdy_deep, 'g', 'si', factor = ptogtc*s2year))





    n0 = 1000

    DO n = 1, PROF ! fixme : use no of profiles
       ncode = n0*n
       CALL varlist_add_profile(bgc_varlist, ncode, n)
    ENDDO

    IF ( p_pe == p_io ) CALL print_varlist('hamocc.partab', bgc_varlist)

    IF ( p_pe == p_io ) CALL apply_gridids(bgc_varlist)

    CALL generate_zaxisid(bgc_varlist)
    CALL generate_sed_zaxisid(bgc_varlist)

  END SUBROUTINE build_bgc_varlist

  ! ---------------------------------------------------------------------

  SUBROUTINE register_bgc_diagnostic

    USE mo_control_bgc, ONLY    : io_stdo_bgc
    USE mo_bgc_diagnostic, ONLY : lbgc_diag

    USE mo_varlist, ONLY        : varlist_is_range_registered
    USE mo_eddydiag, ONLY       : leddydiag


    IF (varlist_is_range_registered(bgc_varlist, 138, 140)) THEN
       lbgc_diag = .TRUE.
       if(p_pe.eq.p_io)WRITE(io_stdo_bgc,*)'lbgc_diag=',lbgc_diag
    END IF


    ! check if eddy diagnostic is registered
    IF (varlist_is_range_registered(bgc_varlist, 315, 323)) THEN
      leddydiag=.TRUE.
      if(p_pe.eq.p_io)WRITE(io_stdo_bgc,*)'leddydiag=',leddydiag
    END IF

  END SUBROUTINE register_bgc_diagnostic


END MODULE mo_bgc_varlist
