!>
!! @brief set start values for bgc variables
!!
!!     - bgc variables are initialized. They might be overwritten
!!       if read from restart by call of AUFR_BGC.
!!     - physical constants are defined
!!     - fields used for budget calculations (should be in extra SBR!)
!!       and as boundary conditions are set to zero.
!!     - POC sinking speed is initialized
!!
!! used by ini_bgc
!!
!! @author Ernst Maier-Reimer     (MPI-M)
!!
!! @par Revision History
!!
!! First version by Ernst Maier-Reimer     (MPI-M)    Apr 10, 2001
!! Revised version by S. Legutke           (MPI-M)    Apr 10, 2001
!! Revised version by Rene Redler          (MPI-M)    Apr 30, 2013
!! Revised version by I.Stemmler           (MPI-M)    Jun 12, 2015
!! 
!! @par Copyright
!! 2002-2013 by MPI-M
!! This software is provided for non-commercial use only.
!! See the LICENSE and the WARRANTY conditions.
!!
MODULE MO_BELEG_BGC

  USE mo_kind, ONLY        : wp
  USE mo_mpi, ONLY         : p_pe, p_io
  USE mo_carbch, ONLY      : hi, co3, totarea, ocetra, atdifv, dmspar, atm, &
       &                     atm_co2, atm_o2, atm_n2, atmacon, atmacmol,    &
       &                     atcoa, ozkoa, c14dec, eins, c14ret, d14catm,   &
       &                     ratm, wpoc
  USE mo_biomod, ONLY      : phytomi, grami, remido, dyphy, zinges,        &
       &                     epsher, grazra, spemor, gammap, gammaz, ecan, &
       &                     pi_alpha, fpar, bkphy, bkzoo, bkopal,         &
       &                     sinkspeed_poc, sinkspeed_opal, sinkspeed_cal, &
       &                     drempoc, dremdoc, dphymor, dzoomor,           &
       &                     dremopal, dremn2o, sulfate_reduction,         &
       &                     dremcalc, n2_fixation, ro2ut, rcar, rnit,     &
       &                     rnoi, nitdem, n2prod, rcalc, ropal, calmax,   &
       &                     gutc, perc_diron, riron, fesoly, relaxfe,     &
       &                     denitrification, kbo, bolay, rn2,             &
       &                     dustd1, dustd2, dustsink, wdust,              &
       &                     mc_fac, mc_eu_speed, mc_depth,                &
       &                     cycdec, pi_alpha_cya,cya_growth_max,          &
       &                     Topt_cya,T1_cya,T2_cya,bkcya_N, bkcya_P, bkcya_fe, &
       &                     remido_cya, dremdoc_cya, buoyancyspeed_cya, &
       &                     Pfac_cya, doccya_fac, thresh_aerob, thresh_sred, &
       &                     ro2bal

  USE mo_sedmnt, ONLY      : powtra, sedlay, sedhpl,disso_po,disso_op,disso_cal,&
       &                     o2ut, rno3, claydens, denit_sed, sred_sed 
  USE mo_control_bgc, ONLY : io_stdo_bgc, ldtbgc, dtb, dtbgc, rmasko, rmasks, &
       &                     deltacalc, deltaorg, deltasil,l_cyadyn, lmartin, &
       &                     l_doccyaf
  USE mo_param1, ONLY      : je1
  USE mo_param1_bgc, ONLY  : diffat, ialkali, ian2o, iatmco2, iatmn2,     &
       &                     iano3, iatmo2, icalc, idet, idoc, igasnit,   &
       &                     iopal, ioxygen,                              &
       &                     iphosph, iphy, ipowaal, ipowaic, ipowaox,    &
       &                     ipowaph, ipowasi, ipown2, ipowno3,           &
       &                     isco212, isilica, isssc12, issso12, issssil, &
       &                     izoo, ks, idms, ifdust, iiron, ipowafe, issster, &
       &                     idoccya, icya
  USE mo_diagnosis , ONLY  : ibek
  USE mo_param1, ONLY      : je1
  USE mo_commo1, ONLY      : l_cpl_co2, ddpo, giph
  USE mo_basin_masks, ONLY : ibek, global_ocean, atlantic_ocean, &
       &                     southern_ocean, tropical_pacific_ocean
  USE mo_planetary_constants, ONLY: g, rhoref_water
  USE mo_parallel, ONLY    : global_sum_2d_pio, &
       &                     je, p_joff, have_g_js, have_g_je
  USE mo_boundsexch, ONLY  : bounds_exch
  USE mo_inventory, ONLY   : calcinp, orginp, silinp

  IMPLICIT NONE

  PRIVATE

  PUBLIC :: ini_aquatic_tracers,            &
       &    ini_pore_water_tracers,         &
       &    ini_continental_carbon_input,   &
       &    ini_atmospheric_concentrations, &
       &    ini_diffat, ini_aggregation,    &
       &    ini_wpoc,           &
       &    print_bgc_variables, beleg_bgc

CONTAINS

  SUBROUTINE BELEG_BGC
    !
    ! Initialize overall time step counter.
    !
    ldtbgc = 0

    IF (diffat) THEN
       ! all concentrations will be calculated in carchm
    ELSE
       atm_co2 = 278._wp
#ifdef __c_isotopes
       atm_c13 = 276.2_wp ! preindustrial atmospheric d13c=-6.5 permil --> 276.2ppm? test js 15082006 ok.
       ! piston velocity ~8m/yr -> equilibration time atmosphere/ocean ~500 years
       atm_c14 = 274._wp
#endif
       atcoa   = atm_co2  ! emr for use in maschk

       atm_o2  = 196800._wp
       atm_n2  = 802000._wp
    ENDIF

    ozkoa    = 3.e15_wp                      ! ozean kohlenstoff alt, start with dummy value in maschk
    atmacon  = 0.35e-3_wp * 5.1e14_wp*12._wp ! factor for atmospheric concentration of CO2 in g C -> ppm
    atmacmol = 0.35e-3_wp * 5.1e14_wp        ! factor for atmospheric concentration of CO2 in mol C -> ppm
    ! 5.1e14 = total surface of Earth in [m^2]

    !
    ! Biology
    !
    !ik note that the unit is kmol/m3!
    phytomi  = 1.e-11_wp ! i.e. 1e-5 mmol P/m3 minimum concentration of phytoplankton
    grami    = 1.e-11_wp ! i.e. 1e-5 mmol P/m3 minimum concentration of zooplankton | test e-11 ->e-8 very slow decay

    !ik addded parameter definition; taken from OCPROD.F (dtb= 1./ steps/day)
    !      remido=0.004*dtb      ! 1/d -remineralization rate (of DOM)
    remido   = 0.008_wp * dtb    ! KS, JS, EMR 12/07/2007
    dyphy    = 0.008_wp * dtb    ! 1/d -mortality rate of phytoplankton
    zinges   = 0.6_wp            ! dimensionless fraction -assimilation efficiency of zooplankton
    epsher   = 0.8_wp            ! dimensionless fraction - (1-epsher)=fraction of grazing egested
    grazra   = 1.0_wp            ! 1/d -grazing rate [emr: 0.6-0.9]
    spemor   = 3.e6_wp * dtb     ! 1/d -mortality rate of zooplankton
    gammap   = 0.03_wp * dtb     ! 1/d -exudation rate
    gammaz   = 0.06_wp * dtb     ! 1/d -excretion rate
    ecan     = 0.95_wp           ! fraction of mortality as PO_4
    pi_alpha = 0.02_wp           ! initial slope of production vs irradiance
    ! curve (alpha) (0.002 for 10 steps per day)
    fPAR     = 0.4_wp            ! fraction of Photosynthetic Active Radiation
    thresh_aerob = 5.e-8_wp      ! kmol m-3,  O2 threshold for aerob remineralization
    thresh_sred = 3.e-6_wp      ! kmol m-3,  O2 threshold for sulfate reduction

    !new emr
#ifdef __c_isotopes
    ! fractionation for photosynthesis plafr13 and plafr14 valid for particles
    ! for avoiding too many tracers, surface gross rates work with reduced
    ! values bifr13 and bifr14
    plafr13 = 1._wp              ! planktonic fractionatio 13C   (never used) (HAMOCC3.1: 0.98)
    plafr14 = 1._wp
    bifr13  = 0.98_wp            ! biogenic fractionation ?
    bifr14  = 0.98_wp
#endif

    ! half sat. constants, note that the units are kmol/m3 ! (conc0 in hamocc3.1)
    bkphy  = 1.e-8_wp !i.e. 0.04 mmol P/m3 |js: 0.01 vs. 0.04? check 0.4 16.9.
    bkzoo  = 4.e-8_wp !i.e. 0.04 mmol P/m3
    bkopal = 1.e-6_wp !i.e. 1.0  mmol Si/m3
    !js: no bkiron? (fitzwater et al 1996) et al. 200 120 nMol/m3, moore et al 80nMol/m3)

    !sinking speed
    sinkspeed_poc  = 5.0_wp  ! daily sinking speed of poc
    sinkspeed_opal = 30.0_wp ! daily sinking speed of opal
    sinkspeed_cal  =  30._wp ! daily sinking speed of cal

    ! Martin curve sinking
    mc_fac = 2.0_wp       !0.858_wp default value from Martin ea 1987 
    mc_eu_speed = 3.5_wp     !1/d sinking speed in depth < mc_depth
    mc_depth = 100._wp

    ! water column remineralisation constants

    !+ist_20150611 drempoc *dtb now done in read_namelist 
    drempoc  = 0.025_wp      ! 1/d      ! 0.75/month. H3.1: 0.2/month k=1, 0.1/month k>1
    !-ist_20150611 
    dremdoc  = 0.004_wp * dtb   ! 1/d
    dphymor  = 0.1_wp * dtb     ! 1/d
    dzoomor  = 0.02_wp * dtb    ! 1/d
!   dremopal = 0.0075_wp * dtb  ! 1/d      ! 0.01 -> 0.001 js10072006 : slightly overdone -->0.0075
!   BALLASTING: default / standard is 0.01*dtb; reduce for smaller sinking speeds to
!   make sure that silicate is removed from euphotic zone to allow calcite
!   production
    dremopal = 0.01_wp / 6._wp * dtb

! ------ cyanobacteria
    buoyancyspeed_cya = 1._wp   ! daily buoyancy speed of cya  
    cycdec            = 0.1_wp 
    pi_alpha_cya      = 0.03_wp      ! m2 W-1 d-1
    cya_growth_max    = 0.2_wp      ! d-1
    Topt_cya          = 28._wp       ! deg C
    T1_cya            = 5.5_wp       ! deg C
    T2_cya            = 1._wp        ! deg C
    bkcya_P           = 1.e-8_wp     ! kmol/m3  
    bkcya_fe          = 90.e-8_wp     ! kmol/m3  
    bkcya_N           = 1.e-9_wp     ! kmol/m3  
    remido_cya        = 0.0008_wp * dtb
    dremdoc_cya       = 0.0008_wp * dtb   
    Pfac_cya          = 1._wp 
    doccya_fac        = 0.1_wp
! ------

    dremn2o  = 0.01_wp*dtb      ! 1/d
    sulfate_reduction = 0.005_wp
    ! dremsul  = 0.005_wp*dtb
    dremcalc = 0.075_wp *dtb    ! 0.2 -> 0.02 js10072006 : slightly overdone --> 0.075

#ifdef AGG
    drempoc  = 0.1_wp *dtb      ! 1/d re-introduced 09062006
    dremopal = 3.3e-3_wp *dtb   ! js 4.7.2006 0.0033 from .01/3. (60/20 m/d)
    dphymor  = 0.2_wp *dtb      ! 1/d
#endif

    ! nitrogen fixation by blue green algae (cyano.f90)

    ! bluefix = 0.005_wp * dtb  ! 1/d
    n2_fixation = 0.005_wp

    ! total denitrification rate is a fraction of aerob remineralisation rate drempoc

    ! dremn3o = 0.05_wp*drempoc ! 1/d
    denitrification = 0.05_wp   ! 1/d

    ! extended redfield ratio declaration
    ! Note: stoichiometric ratios are based on Takahashi etal. (1985)
    ! P:N:C:-O2 + 1:16:122:172

    ro2ut = 172._wp
    rcar  = 122._wp
    rnit  = 16._wp
    rnoi  = 1._wp/rnit
    rn2   = 2._wp
    ro2bal = ro2ut-rcar-2._wp-1.5_wp*rnit ! OM conversion for O2 mass balance
                                          ! OM { rcar C: rnit N : P }
                                          ! C -> CO2, N-> NO3, P -> PO4
                                          ! C[O2] = 1, N[O2] =1.5, P[O2]=2 
                                          ! total O2   = [O2]+[CO2]+2[PO4]+1.5[NO3] - ro2bal[OM]   

    ! N consumption of denitrification corrected after Paulmier etal, 2009)
    nitdem = 121.6_wp      ! nitrate demand to remin. 1 mol P in suboxic water
    n2prod = 68.8_wp       ! N2 production for 1 mol P remineralized in suboxic water

    rcalc = 35._wp ! iris 40 !calcium carbonate to organic phosphorous production ratio
    IF (l_cpl_co2) THEN
       rcalc = 20._wp ! emr !calcium carbonate to organic phosphorous production ratio
    END IF

    ropal  = 25._wp ! iris 25 !opal to organic phosphorous production ratio
    calmax = 0.20_wp ! max fraction of CaCO3 production (AGG) (max rain ratio)
    gutc   = 0._wp   ! fraction of caco3 that dissolves during passage through zooplankton guts (not used yet)

    ! for interaction with sediment module
    o2ut = 172._wp
    rno3 = 16._wp

    ! denitrification & sulfate reduction rates in the sediment
    denit_sed = 0.01_wp  ! 1/d  sediment denitrification rate 
    sred_sed = 0.0001_wp *dtb ! sediment sulfate reduction rate
    disso_op = 0.002592_wp
    disso_cal = 0.00864_wp
    disso_po = 0.01_wp

    ! weight percent iron in dust deposition (0.035) times Fe solubility (0.01) /55.85 g--> mol
    perc_diron = 0.035_wp * 0.01_wp / 55.85_wp

    ! the three values below are from Johnson et al., 1997 Mar. Chemistry 57, 137-161
    ! riron   = 5.*rcar*1.e-6       ! 'Q'=5.*122.*1.e-6 = 6.1e-4   (iron to phosphate stoichiometric ratio * umol->mol)
    ! riron   = 2.*rcar*1.e-6       ! 14.2.06: 5.->2. (2.5umol P/nmol Fe) emr: 3.*rcar*1.e-6
    riron   = 3._wp * rcar*1.e-6_wp ! 06.3.06: 2.->3. coex90=12.2GTC/a before change

    fesoly  = 0.6_wp *1.e-9_wp      ! global mean/max (?) dissolved iron concentration in deep water (for relaxation) [mol/m3]
    ! 'fesoly' stands for 'apparent solubility value' of iron

    relaxfe = 0.05_wp/365._wp*dtb       ! relaxation time for iron to fesoly corresponds to 20 yrs
    !  relaxfe = 0.005_wp/365._wp*dtb       ! changed as suggested by emr
    ! (should probably be named scavenge_fe or remove_fe)
    ! (relaxfe is only used to remove iron, corresponding to Johnson's 200yr residence time)
    ! but 200 yrs might be too long, as iron concentrations in the Atlantic are too high.

    ! decay parameter for sco214, half life time = 5730years
    c14dec = (LOG(2._wp)/(5730._wp*365._wp))*dtb
    eins   = 1._wp
    c14ret = eins-c14dec

    ! Ratm: normalized atmospheric ratio of C-14/C-12, D14Catm: atmospheric Delta C-14
    D14Catm = 0._wp             ! D14Catm=0. only for equilibrium runs
    Ratm = 1._wp + D14Catm/1000._wp

    ! Set constants for calculation of DMS ( mo_carbch )
    ! Parameters are a result from Kettle optimisation (fit to Kettle data, version emr, not tinka)

    dmspar(6) = 0.100000000E-07_WP            !0 half saturation microbial
    dmspar(5) = 1.25_wp * 0.109784522E-01_wp  ! production with delsil(diatoms)
    dmspar(4) = 1.25_wp * 0.107638502E+00_wp  ! production with delcar(coccolithoporides)

    dmspar(3) = 0.0096_wp  !4.8e-5       !2*1.6e-3 microbial consumption
    dmspar(2) = 0.0075_wp  !0.0003       !2*0.005  photolysis (uv-destruction)
    dmspar(1) = 10._wp                   !2*5.     temperature dependent release by phytoplankton

  END SUBROUTINE BELEG_BGC

  ! ---------------------------------------------------------------------

  SUBROUTINE print_bgc_variables
    IF (p_pe == p_io) THEN
      WRITE(io_stdo_bgc,*)                                             &
           &'****************************************************************'
      WRITE(io_stdo_bgc,*)                                             &
           &'* '
      WRITE(io_stdo_bgc,*)                                             &
           &'* Values of BELEG_BGC variables : '
      IF (.NOT. diffat) THEN
        WRITE(io_stdo_bgc,*)                                             &
             &'*                              atm_co2      = ',atm_co2
        WRITE(io_stdo_bgc,*)                                             &
             &'*                              atm_o2       = ',atm_o2
        WRITE(io_stdo_bgc,*)                                             &
             &'*                              atm_n2       = ',atm_n2
      ENDIF
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              phytomi      = ',phytomi
      WRITE(io_stdo_bgc,*)                                             &
         &'*                              grami        = ',grami
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              remido       = ',remido/dtb
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              dyphy        = ',dyphy/dtb
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              zinges       = ',zinges
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              epsher       = ',epsher
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              grazra       = ',grazra/dtb
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              spemor       = ',spemor/dtb
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              gammap       = ',gammap/dtb
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              gammaz       = ',gammaz/dtb
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              ecan         = ',ecan
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              bkphy        = ',bkphy
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              bkzoo        = ',bkzoo
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              bkopal       = ',bkopal
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              sinkspeed_poc  = ',sinkspeed_poc
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              sinkspeed_cal  = ',sinkspeed_cal
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              sinkspeed_opal = ',sinkspeed_opal
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              drempoc      = ', drempoc/dtb
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              denitrification      = ',denitrification
      WRITE(io_stdo_bgc,*)                                             &
           &'*                           sulfate_reduction     = ',sulfate_reduction
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              dremdoc      = ',dremdoc/dtb
      WRITE(io_stdo_bgc,*)                                             &
           &'*                           thresh_aerob  = ',thresh_aerob
      WRITE(io_stdo_bgc,*)                                             &
           &'*                           thresh_sred  = ',thresh_sred
      WRITE(io_stdo_bgc,*)                                             &
         &'*                              dremopal     = ',dremopal/dtb
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              dphymor      = ',dphymor/dtb
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              dzoomor      = ',dzoomor/dtb
      WRITE(io_stdo_bgc,*)                                             &
         &'*                              n2_fixation  = ',n2_fixation
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              ro2ut        = ',ro2ut
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              rcar         = ',rcar
      WRITE(io_stdo_bgc,*)                                             &
         &'*                              rnit         = ',rnit
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              rnoi         = ',rnoi
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              rcalc        = ',rcalc
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              ropal        = ',ropal
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              gutc         = ',gutc
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              o2ut         = ',o2ut
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              rno3         = ',rno3
      WRITE(io_stdo_bgc,*)                                             &
         &'*                              denit_sed      = ',denit_sed/dtb
      WRITE(io_stdo_bgc,*)                                             &
         &'*                              sred_sed      = ',sred_sed/dtb
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              perc_diron   = ',perc_diron
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              riron        = ',riron
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              fesoly       = ',fesoly
      WRITE(io_stdo_bgc,*)                                             &
         &'*                              relaxfe      = ',relaxfe/dtb
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              c14dec       = ',c14dec/dtb
      WRITE(io_stdo_bgc,*)                                             &
         &'*                              D14Catm      = ',D14Catm
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              Ratm         = ',Ratm
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              dmspar(1)    = ',dmspar(1)
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              dmspar(2)    = ',dmspar(2)
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              dmspar(3)    = ',dmspar(3)
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              dmspar(4)    = ',dmspar(4)
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              dmspar(5)    = ',dmspar(5)
     IF(lmartin)then
       WRITE(io_stdo_bgc,*)                                             &
           &'*                              mc_fac       = ', mc_fac
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              mc_eu_speed (m/d)    = ', mc_eu_speed
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              mc_depth (m)   = ', mc_depth

      ENDIF
      IF(l_cyadyn)then
        WRITE(io_stdo_bgc,*)                                              &
           &'*                              l_cyadyn     =', l_cyadyn
        WRITE(io_stdo_bgc,*)                                              &
           &'*                              l_doccyaf     =', l_doccyaf
        WRITE(io_stdo_bgc,*)                                              &
           &'*                     buoyancyspedd_cya    =',buoyancyspeed_cya/dtb  
        WRITE(io_stdo_bgc,*)                                              &
           &'*                                cycdec    =',cycdec 
        WRITE(io_stdo_bgc,*)                                              &    
           &'*                          pi_alpha_cya    =',pi_alpha_cya
        WRITE(io_stdo_bgc,*)                                              &
           &'*                        cya_growth_max    =',cya_growth_max
        WRITE(io_stdo_bgc,*)                                              &                   
           &'*                              Topt_cya    =',Topt_cya
        WRITE(io_stdo_bgc,*)                                              &
           &'*                                T1_cya    =',T1_cya
        WRITE(io_stdo_bgc,*)                                              &
           &'*                                T2_cya    =',T2_cya
        WRITE(io_stdo_bgc,*)                                              &
           &'*                               bkcya_P    =',bkcya_P
        WRITE(io_stdo_bgc,*)                                              &
           &'*                              bkcya_fe    =',bkcya_fe
        WRITE(io_stdo_bgc,*)                                              &
           &'*                               bkcya_N    =',bkcya_N
        WRITE(io_stdo_bgc,*)                                              &
           &'*                            remido_cya    =', remido_cya/dtb
        WRITE(io_stdo_bgc,*)                                              &
           &'*                           dremdoc_cya    =', dremdoc_cya/dtb   
        WRITE(io_stdo_bgc,*)                                              &
           &'*                              PFAC_cya    =', Pfac_cya 
        WRITE(io_stdo_bgc,*)                                              &
           &'*                             doccya_fac   =', doccya_fac 
      ENDIF
    ENDIF

#ifndef AGG
    dustd1 = 0.0001_wp !cm = 1 um, boundary between clay and silt; 
    dustd2 = dustd1*dustd1
    dustsink = (g * 86400._wp / 18._wp                          &  ! g * sec per day / 18. | js: Stoke's law for small particles
         &   * (claydens - rhoref_water) / 1.567_wp * 1000._wp  &  ! excess density / dyn. visc. | -> cm/s to m/day
         &   * dustd2 * 1.e-4_wp)*dtb                              ! *diameter**2 |*1000 *1.e-4?
    wdust = dustsink

    IF ( p_pe == P_io ) THEN
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              dustd1       = ',dustd1
      WRITE(io_stdo_bgc,*)                                             &
         &'*                              dustd2       = ',dustd2
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              dustsink     = ',dustsink/dtb
      WRITE(io_stdo_bgc,*)                                             &
           &'*                              wdust        = ',wdust/dtb
    ENDIF
#endif

    IF ( p_pe == P_io ) THEN
      WRITE(io_stdo_bgc,*)                                             &
           &'****************************************************************'
    ENDIF
  END SUBROUTINE print_bgc_variables

  ! ---------------------------------------------------------------------
  SUBROUTINE ini_wpoc(kpke,ptiestw)
  ! initialize wpoc
  ! if lmartin==TRUE (mo_control_bgc, nml)
  ! wpoc increases linearly with depth below mc_depth (beleg, nml)
  ! otherwise the constant sinkspeed_poc (beleg, nml) is used
   INTEGER, INTENT(in) :: kpke
   REAL(wp),INTENT(in) :: ptiestw(kpke+1)

   INTEGER :: k
   REAL(wp) :: at_mc_depth
   
   ! default case: constant sinking speed
   wpoc = sinkspeed_poc * dtb

   IF(lmartin)then
    DO k = 1,kpke
      ! Are we at z0 (below which the Martin curve starts)?
      at_mc_depth=merge(0._wp,1._wp,ptiestw(k+1)<=mc_depth)
      ! w=w0 + a*(z-z0)
      ! z0= mc_depth
      ! a=remin_rate/b  with F(z)=F(z0)(z/zo)**(-b) 
      wpoc(k) = mc_eu_speed*dtb + at_mc_depth * drempoc/mc_fac * (ptiestw(k+1) - mc_depth) 
      IF (p_pe == p_io) WRITE(io_stdo_bgc,*)'wpoc:',ptiestw(k+1), wpoc(k) /dtb                     
    ENDDO
   ENDIF

  END SUBROUTINE ini_wpoc
  
  SUBROUTINE ini_aquatic_tracers ( kpie, kpje, kpke )

    INTEGER, INTENT(in)  :: kpie                  !< 1st REAL of model grid.
    INTEGER, INTENT(in)  :: kpje                  !< 2nd REAL of model grid.
    INTEGER, INTENT(in)  :: kpke                  !< 3rd (vertical) REAL of model grid.

    INTEGER :: i,j,k,m

    REAL(wp) :: phosat, phosmed, phospac
    REAL(wp) :: oxyat, oxymed, oxypac
    REAL(wp) :: silat, silmed, silpac

    !  Initial values for aquatic (advected) ocean tracers (for new start)

    phosat  = 1.e-6_wp
    phospac = 2.5e-6_wp
    phosmed = 2.e-7_wp
    silat   = 2.e-5_wp
    silmed  = 5.e-6_wp
    silpac  = 1.e-4_wp
    oxypac  = 1.5e-4_wp
    oxyat   = 2.5e-4_wp
    oxymed  = 2.e-4_wp

    DO k = 1, kpke
       DO j = 1, kpje
          DO i = 1, kpie

             IF (ddpo(i,j,k) > 0.5_wp) THEN     ! wet points

                m = ibek(i,j)

                ocetra(i,j,k,isco212) = 2.27e-3_wp     ! [kmol/m3]
                ocetra(i,j,k,ialkali) = 2.37e-3_wp

                SELECT CASE(m)
                CASE (southern_ocean:tropical_pacific_ocean)
                   ocetra(i,j,k,iphosph) = phospac
                   ocetra(i,j,k,ioxygen) = oxypac
                   ocetra(i,j,k,isilica) = silpac
                CASE (0:atlantic_ocean)
                   ocetra(i,j,k,iphosph) = phosat
                   ocetra(i,j,k,ioxygen) = oxyat
                   ocetra(i,j,k,isilica) = silat
                CASE (global_ocean)
                   ocetra(i,j,k,iphosph) = phosmed
                   ocetra(i,j,k,ioxygen) = oxymed
                   ocetra(i,j,k,isilica) = silmed
                END SELECT

                ocetra(i,j,k,igasnit)= 0.0_wp
                ocetra(i,j,k,iano3)  = rnit*ocetra(i,j,k,iphosph)
                ocetra(i,j,k,idoc)   = 1.e-10_wp
                ocetra(i,j,k,iphy)   = 1.e-8_wp
                ocetra(i,j,k,izoo)   = 1.e-8_wp
                ocetra(i,j,k,idet)   = 1.e-8_wp
                ocetra(i,j,k,icalc)  = 0._wp
                ocetra(i,j,k,iopal)  = 1.e-8_wp
                ocetra(i,j,k,idoccya)= 1.e-10_wp
                ocetra(i,j,k,icya)   = 1.e-10_wp
#ifdef __c_isotopes
                ocetra(i,j,k,isco214)=0.75_wp*2.27e-3_wp !Paddy: oldest water: 1600y --> X0.83
#endif
                ocetra(i,j,k,ian2o)    = 0._wp
                ocetra(i,j,k,idms)     = 0._wp
                ocetra(i,j,k,ifdust)   = 0._wp
                ocetra(i,j,k,iiron)    = 0.6_wp*1.e-9_wp
                ! ocetra(i,j,k,ibeten) = 0.6*1.e-12
                hi(i,j,k)              = 3.e-9_wp
                co3(i,j,k)             = 0._wp       ! this good for initialisation -> 2.e-4?
#ifdef __c_isotopes
                ocetra(i,j,k,isco213) = 2.27e-3_wp     ! adjusted to reference ratio 13C/12C=1 (*100)!
                ocetra(i,j,k,isco214) = 2.27e-3_wp
                ocetra(i,j,k,idet13)  = 1.e-8_wp
                ocetra(i,j,k,idet14)  = 1.e-8_wp
                ocetra(i,j,k,icalc13) = 0._wp
                ocetra(i,j,k,icalc14) = 0._wp
#endif

#ifdef AGG
                ! calculate initial numbers from mass, to start with appropriate size distribution
                snow = (ocetra(i,j,k,iphy)+ocetra(i,j,k,idet))*1.e+6_wp
                ocetra(i,j,k,inos)   = snow / cellmass / (FractDim+1._wp)
                ocetra(i,j,k,iadust) = 0._wp
#endif /*AGG*/

             ELSE                              ! dry points
                ocetra(i,j,k,iphosph) = rmasko
                ocetra(i,j,k,isilica) = rmasko
                ocetra(i,j,k,ioxygen) = rmasko
                ocetra(i,j,k,ialkali) = rmasko
                ocetra(i,j,k,isco212) = rmasko
                ocetra(i,j,k,iano3)   = rmasko
                ocetra(i,j,k,igasnit) = rmasko
                ocetra(i,j,k,idoc)    = rmasko
                ocetra(i,j,k,iphy)    = rmasko
                ocetra(i,j,k,izoo)    = rmasko
                ocetra(i,j,k,idet)    = rmasko
                ocetra(i,j,k,icalc)   = rmasko
                ocetra(i,j,k,iopal)   = rmasko
                ocetra(i,j,k,ian2o)   = rmasko
                ocetra(i,j,k,idms)    = rmasko
                ocetra(i,j,k,ifdust)  = rmasko
                ocetra(i,j,k,iiron)   = rmasko
                ocetra(i,j,k,idoccya) = rmasko
                ocetra(i,j,k,icya)    = rmasko
                ! ocetra(i,j,k,ibeten) = rmasko
                hi(i,j,k)             = rmasko
                co3(i,j,k)            = rmasko
#ifdef __c_isotopes
                ocetra(i,j,k,isco213) = rmasko
                ocetra(i,j,k,isco214) = rmasko
                ocetra(i,j,k,idet13)  = rmasko
                ocetra(i,j,k,icalc13) = rmasko
                ocetra(i,j,k,idet14)  = rmasko
                ocetra(i,j,k,icalc14) = rmasko
#endif
#ifdef AGG
                ocetra(i,j,k,inos)    = rmasko
                ocetra(i,j,k,iadust)  = rmasko
#endif /*AGG*/
             ENDIF

          ENDDO
       ENDDO
    ENDDO

  END SUBROUTINE ini_aquatic_tracers

  ! ---------------------------------------------------------------------

  SUBROUTINE ini_pore_water_tracers(grid_shape)

    INTEGER, INTENT(in) :: grid_shape(3)

    INTEGER :: i, j, k
    INTEGER :: kpie, kpje
    !  Initial values for sediment pore water tracers. (solid components?)

    kpie = grid_shape(1)
    kpje = grid_shape(2)

    DO k = 1, ks
       DO j = 1, kpje
          DO i = 1, kpie

             IF(bolay(i,j) > 0._wp) THEN
                powtra(i,j,k,ipowaic) = ocetra(i,j,kbo(i,j),isco212)
                powtra(i,j,k,ipowaal) = ocetra(i,j,kbo(i,j),ialkali)
                powtra(i,j,k,ipowaph) = ocetra(i,j,kbo(i,j),iphosph)
                powtra(i,j,k,ipowaox) = ocetra(i,j,kbo(i,j),ioxygen)
                powtra(i,j,k,ipown2)  = 0._wp
                powtra(i,j,k,ipowno3) = ocetra(i,j,kbo(i,j),iano3)
                powtra(i,j,k,ipowasi) = ocetra(i,j,kbo(i,j),isilica)
                powtra(i,j,k,ipowafe) = ocetra(i,j,kbo(i,j),iiron)

                sedlay(i,j,k,issso12) = 1.e-8_wp
                sedlay(i,j,k,isssc12) = 1.e-8_wp
#ifdef __c_isotopes
                sedlay(i,j,k,issso13) = 1.e-8_wp
                sedlay(i,j,k,isssc13) = 1.e-8_wp
                sedlay(i,j,k,issso14) = 1.e-8_wp
                sedlay(i,j,k,isssc14) = 1.e-8_wp
#endif

                sedlay(i,j,k,issster) = 30._wp
                sedlay(i,j,k,issssil) = 3._wp

                sedhpl(i,j,k)         = hi(i,j,kbo(i,j))
             ELSE
                powtra(i,j,k,ipowno3) = rmasks   ! pore water
                powtra(i,j,k,ipown2)  = rmasks
                powtra(i,j,k,ipowaic) = rmasks
                powtra(i,j,k,ipowaal) = rmasks
                powtra(i,j,k,ipowaph) = rmasks
                powtra(i,j,k,ipowaox) = rmasks
                powtra(i,j,k,ipowasi) = rmasks
                powtra(i,j,k,ipowafe) = rmasks
                sedlay(i,j,k,issso12) = rmasks   ! solid sediment
                sedlay(i,j,k,isssc12) = rmasks
#ifdef __c_isotopes
                sedlay(i,j,k,issso13) = rmasks
                sedlay(i,j,k,isssc13) = rmasks
                sedlay(i,j,k,issso14) = rmasks
                sedlay(i,j,k,isssc14) = rmasks
#endif
                sedlay(i,j,k,issssil) = rmasks
                sedlay(i,j,k,issster) = rmasks
                sedhpl(i,j,k)         = rmasks
             ENDIF

          ENDDO
       ENDDO
    ENDDO

  END SUBROUTINE ini_pore_water_tracers

  ! ---------------------------------------------------------------------

  SUBROUTINE ini_continental_carbon_input

    USE mo_param1, ONLY: ie,je
    USE mo_commo1, ONLY: area, weto

    ! Rate of continental carbon input per unit area
    ! Compute total ocean area

    REAL(wp) :: ocean_area(ie,je)

    ocean_area(:,:) = area(:,:)*weto(:,:,1)

    CALL global_sum_2d_pio(ocean_area,totarea)

    calcinp = deltacalc * dtbgc / totarea
    orginp  = deltaorg  * dtbgc / totarea
    silinp  = deltasil  * dtbgc / totarea

  END SUBROUTINE ini_continental_carbon_input

  SUBROUTINE ini_atmospheric_concentrations

    !  atmospheric concentrations (overwritten from restart file)

    atm(:,:,iatmco2) = atm_co2
    atm(:,:,iatmo2)  = atm_o2
    atm(:,:,iatmn2)  = atm_n2
#ifdef __c_isotopes
    atm(:,:,iatmc13) = atm_co2 - (atm_co2 * 0.0065_wp)
    atm(:,:,iatmc14) = atm_co2 - (atm_co2 * 0.0065_wp)**2
#endif

  END SUBROUTINE ini_atmospheric_concentrations

  ! ---------------------------------------------------------------------

  SUBROUTINE ini_diffat(grid_shape)

    INTEGER, INTENT(in) :: grid_shape(3)

    INTEGER  :: i, j, jj
    INTEGER  :: kpie, kpje
    REAL(wp) :: north
    REAL(wp) :: south

    kpie = grid_shape(1)
    kpje = grid_shape(2)

    IF (diffat) THEN
       atdifv(:,:) = 1._wp

       !  finding the equator   (p_ioff defined in ../src_oce/mo_parallel.f90)
       !js this could be within IFDEF DIFFAT? (seems to be used only in atmotr, which is called only if DIFFAT (bgc.f90))
       WRITE(io_stdo_bgc,*)'setting equatorial diffusion'

       DO  i = 1, kpie
          !ii=1+(i+p_ioff-1)*2 ! global i-index for giph_g   | giph_g global latitude (parallelization)
          north = 1._wp
          south = 1._wp
          DO  j = 1, kpje

             jj = 1+(j+p_joff-1)*2 ! global j-index for giph_g

             IF (jj<=2) CYCLE

             ! was once north = giph_g(ii,jj-2), see doc/giph_and_giph_g.txt
             north = giph(2*i - 1, 2*j - 3)

             ! was once south=giph_g(ii,jj), see doc/giph_and_giph_g.txt
             south=giph(2*i - 1, 2*j - 1)

             IF ((north >= 0._wp).AND.(south.LE.0._wp)) THEN
                atdifv(i,j) = 0.01_wp
                IF(j<=je-1) atdifv(i,j+1) = 0.02_wp
                IF(j>=   2) atdifv(i,j-1) = 0.02_wp
                IF(j<=je-2) atdifv(i,j+2) = 0.05_wp
                IF(j>=   3) atdifv(i,j-2) = 0.05_wp
             ENDIF

             IF ((north >= 30._wp) .AND. (south <= 30._wp)) THEN
                atdifv(i,j) = 0.1_wp
                IF(j<=je-1) atdifv(i,j+1) = 0.2_wp
                IF(j>=   2) atdifv(i,j-1) = 0.2_wp
             ENDIF
             IF ((north >= -30._wp).AND.(south <= -30._wp)) THEN
                atdifv(i,j) = 0.1_wp
                IF(j<=je-1) atdifv(i,j+1) = 0.2_wp
                IF(j>=   2) atdifv(i,j-1) = 0.2_wp
             ENDIF

          ENDDO
       ENDDO

       ! tjahns: hope atdifv contains p-points, if not please correct
       CALL bounds_exch(1, 'p', atdifv, 'beleg_bgc diffat') ! for safety only

       ! put ENDIF for DIFFAT here?
       !
       !     no diffusion into the poles
       !
       DO i = 1, kpie
          IF (have_g_js) atdifv(i,1)    = 0._wp
          ! IF (have_g_js) atdifv(i,2)  = 0._wp
          IF (have_g_je) atdifv(i,kpje) = 0._wp
          IF (have_g_je) atdifv(i,je1)  = 0._wp
       ENDDO

    ENDIF

  END SUBROUTINE ini_diffat

  ! ---------------------------------------------------------------------

  SUBROUTINE ini_aggregation(grid_shape)

    INTEGER, INTENT(in) :: grid_shape(3)

#ifdef AGG
    INTEGER  :: kpie, kpje
    INTEGER  :: i, j, k
    REAL(wp) :: shear, zmini, talar1, snow, checksink

    kpie = grid_shape(1)
    kpje = grid_shape(2)

    ! parameters needed for the aggregation module (see Kriest 2002, DSR I vol.49, p. 2133-2162)

    SinkExp  = 0.62_wp               ! exponent of the sinking speed vs. diameter relationship
    FractDim = 1.62_wp               ! exponent of the diameter vs. phosphorous content relationship
    Stick    = 0.40_wp               ! maximum stickiness
    cellmass = 0.012_wp/rnit         ! [nmol P]   minimum mass of a particle in phosphorous units (rnit=16)
    !ik   cellmass = 0.0039_wp/rnit  ! [nmol P] for 10 um diameter
    cellsink = 1.40_wp *dtb          ! [m/d] see Kriest 2002, Table 2 Ref 8 (from Stokes' formula, delta rho 0.052 g/cm3)
    !ik      cellsink = 0.911 *dtb   ! [m/d]  for a 10 um diameter
    shear    = 86400._wp             ! wind induced shear in upper 100m , 1 d^-1
    fsh      = 0.163_wp * shear *dtb ! turbulent shear (used for aggregation)
    fse      = 0.125_wp * 3.1415927_wp * cellsink * 100._wp ! differential settling (used for aggregation) (100=10**2 [d**2])
    alow1    = 0.002_wp              ! diameter of smallest particle [cm]
    !ik      alow1 = 0.001_wp        ! diameter of smallest particle [cm]
    alow2    = alow1 * alow1
    alow3    = alow2 * alow1
    alar1    = 1.0_wp                ! diameter of largest particle for size dependend aggregation and sinking [cm]
    vsmall   = 1.e-9_wp
    safe     = 1.e-6_wp
    pupper   = safe/((FractDim+safe)*cellmass) ! upper boundary for cells per aggregate (?)
    plower   = 1._wp/(1.1_wp*cellmass)         ! lower   --------------"------------------
    zdis     = 0.01_wp / ((FractDim + 0.01_wp)*cellmass)

    !ik check max possible sinking speed in relation to min.
    !ik layer thinkness and time step for all standard layers, except
    !ik the bottom layer.
    !ik if max possible sinking speed (per time step) is greater
    !ik than min layer thickness, decrease max. length for sinking and
    !ik aggregation

    zmini = 8000._wp

    DO  j = 1, kpje
       DO  i = 1, kpie
          DO  k = 1, kbo(i,j)-1
             IF (ddpo(i,j,k) > 0.5_wp) THEN
                zmini = MIN(ddpo(i,j,k),zmini)
             ENDIF
          ENDDO
       ENDDO
    ENDDO

    CALL global_min(zmini)

    checksink =(zmini/cellsink)**(1._wp/SinkExp)*alow1

    IF (checksink < alar1) THEN

       WRITE(io_stdo_bgc,*) 'Allowed max. length for sinking'              &
            & ,' with min. depth of '                                      &
            & , zmini, ' m for layers 1-(kbo-1) and time step of ',dtb     &
            & ,' days is' , checksink                                      &
            & ,'cm, which is smaller than prescribed value of', alar1, ' cm'

       talar1 = alar1
       alar1  = checksink

       WRITE(io_stdo_bgc,*) 'Set max. length for sinking and aggregation &
            &  from ',talar1,' to ', alar1

    ENDIF

    alar2 = alar1 * alar1
    alar3 = alar2 * alar1
    TSFac = (alar1/alow1)**SinkExp
    TMFac = (alar1/alow1)**FractDim

    !ik check the maximum possible sinking speed for the bottom layer (which
    !ik may be smaller than zmini, and write to array alar1max, tsfmax, tmfmax

    DO j = 1, kpje
       DO i = 1, kpie

          alar1max(i,j) = alar1
          TSFmax(i,j)   = TSFac
          TMFmax(i,j)   = TMFac

          IF (ddpo(i,j,kbo(i,j)) > 0.5_wp) THEN

             !ik evaluate safe length scale for size dependent sinking and
             !ik aggregation, and the resulting sinking rate and aggregation rate.

             checksink = (ddpo(i,j,kbo(i,j))/cellsink)**(1._wp/SinkExp)        &
                  &                    *alow1
             IF(checksink < alar1) THEN
                alar1max(i,j) = checksink
                TSFmax(i,j)   = (checksink/alow1)**SinkExp
                TMFmax(i,j)   = (checksink/alow1)**FractDim
                WRITE(io_stdo_bgc,*) 'resetting alar1 to',checksink,'at i =',  &
                     &     i,' j = ',j,' k = ', kbo(i,j), ' with dz = ',       &
                     &     ddpo(i,j,kbo(i,j))
             ENDIF

          ENDIF
       ENDDO
    ENDDO

    ! for shear aggregation of dust:
    dustd1   = 0.0001_wp ![cm] = 1 um, boundary between clay and silt
    dustd2   = dustd1*dustd1
    dustd3   = dustd2*dustd1
    dustsink = (g * 86400._wp / 18._wp                          & ! g * sec per day / 18.
         &   * (claydens - rhoref_water) / 1.567_wp * 1000._wp  & !excess density / dyn. visc.
         &   * dustd2 * 1.e-4_wp)*dtb                             ! --> 4.73e-2 m/d

    WRITE(io_stdo_bgc,*) 'dust diameter (cm)', dustd1
    WRITE(io_stdo_bgc,*) 'dust sinking speed (m/d)', dustsink / dtb

    IF(dustsink > cellsink) THEN
       WRITE(io_stdo_bgc,*) 'dust sinking speed greater than cellsink'
       dustsink = cellsink
       WRITE(io_stdo_bgc,*) 'set dust sinking speed to cellsink'
    ENDIF

#endif /*AGG*/
  END SUBROUTINE ini_aggregation

END MODULE mo_beleg_bgc
