! =======================================================================
!
! Copyright Centre National de la Recherche Scientifique (CNRS)
!
! [Institut des Geosciences de l'Environnement (IGE) & Laboratoire des
! Sciences du climat et de l'Environnement (LSCE)]
!
! contributor(s): C. Ritz, A. Quiquet, C. Dumas, V. Peyaud, V. Rommelaere,
! A. Fabre, N. Lhomme (2018)
!
! contacts:
! C.R.: catherine.ritz@univ-grenoble-alpes.fr
! C.D.: christophe.dumas@lsce.ipsl.fr
! A.Q.: aurelien.quiquet@lsce.ipsl.fr
!
! This file: [remplimat-shelves-tabTu.f90]
! is part of the GRISLI (GRenoble Ice-Shelf and Land-Ice model)
! software, a computer program whose purpose is to compute the evolution
! of ice thickness coupled to three-dimensional ice temperature and
! velocity fields given a mass-balance scenario.
!
! This software is governed by the CeCILL license under French law and
! abiding by the rules of distribution of free software.  You can  use,
! modify and/ or redistribute the software under the terms of the CeCILL
! license as circulated by CEA, CNRS and INRIA at the following URL
! "http://www.cecill.info".
!
! As a counterpart to the access to the source code and  rights to copy,
! modify and redistribute granted by the license, users are provided only
! with a limited warranty  and the software's author,  the holder of the
! economic rights,  and the successive licensors  have only  limited
! liability.
!
! In this respect, the user's attention is drawn to the risks associated
! with loading,  using,  modifying and/or developing or reproducing the
! software by the user in light of its specific status of free software,
! that may mean  that it is complicated to manipulate,  and  that  also
! therefore means  that it is reserved for developers  and  experienced
! professionals having in-depth computer knowledge. Users are therefore
! encouraged to load and test the software's suitability as regards their
! requirements in conditions enabling the security of their systems and/or
! data to be ensured and,  more generally, to use and operate it in the
! same conditions as regards security.
!
! The fact that you are presently reading this means that you have had
! knowledge of the CeCILL license and that you accept its terms.
!
! ======================================================================


! La matrice du systeme elliptique est nommee "L2" dans ce qui suit 
! mais n'est pas dimensionnee car elle serait trop grande (2 nx ny x 2 nx ny)


! Au lieu de cela on ecrit les elements non nuls de chaque ligne 
! Tu et Tv pour les lignes de l'equation en U
! Su et Sv pour les lignes de l'equation en V


! dans cette version les Tuij sont sous forme de tableau.
! le conditions aux limites sont donnees sur les bords de la grille
! Il y a un nettoyage de la matrice pour enlever les noeuds qui ne servent pas.


!--------------------------------------------------------------------------
subroutine rempli_L2(nx1,nx2,ny1,ny2,uxprec,uyprec,uxnew,uynew,imx,imy,ifail_L2)    
! 
! - ecrit l'equation elliptique des vitesses sur le domaine nx1,nx2 et ny1,ny2
! - appelle la routine de resolution  : call  resol_ellipt
! - renvoie les nouvelles vitesses uxnew, uynew
!
!
!      nx1,nx2       bornes du domaine
!      ny1,ny2
!                                                                   
!      uxprex(nx1:nx2,ny1:ny2)                                                 
!      uyprec(nx1:nx2,ny1:ny2) vitesses de l'iteration precedente              
!                                                                    
!      uxnew(nx1:nx2,ny1:ny2)                                                   
!      uynew(nx1:nx2,ny1:ny2) uynew resultat de cette iteration 
!
!      imx(nx1:nx2,ny1:ny2) masque pour imposer les vitesses ou leur drivee    
!      imy(nx1:nx2,ny1:ny2) masque pour imposer les vitesses ou leur drive           
!
! eventuellement le domaine n1,n2 peut etre un sous-domaine de nx,ny
! attention, dans ce cas l'appel devra tre 
! call rempli-L2 (nx1,nx2,ny1,ny2,ux1(nx1:nx2,ny1:ny2),uy1(nx1:nx2,ny1:ny2), &
!                                 ux2(nx1:nx2,ny1:ny2),uy2(nx1:nx2,ny1:ny2),  &
!                                 imx(nx1:nx2,ny1:ny2,imy(nx1:nx2,ny1:ny2),ifail_L2)
!                     
!-----------------------------------------------------------------------

!$ USE OMP_LIB
use module3d_phy
use remplimat_declar        ! les tableaux tuij, ....
use module_choix

! use eq_ellip_sgbsv_mod

implicit none

! declarations variables dummy

integer,intent(in) :: nx1             ! bornes du domaine en x (noeuds majeurs)
integer,intent(in) :: nx2
integer,intent(in) :: ny1             ! bornes du domaine en y (noeuds majoeurs)
integer,intent(in) :: ny2

!integer :: n1            ! dimension selon x
!integer :: n2            ! dimension selon y
integer :: ifail_L2         ! pour les rapports d'erreur
  
real, dimension(nx1:nx2,ny1:ny2),intent(in)  :: uxprec    ! vitesse en entree routine
real, dimension(nx1:nx2,ny1:ny2),intent(in)  :: uyprec    ! vitesse en entree routine

! masques vitesses.
! Pour donner de la souplesse dans les zones qu'on traite
! 

! imx(i,j)=0 -> ne pas traiter cette vitesse
! imx(i,j)=1 -> uxnew(i,j)=uxprec(i,j)                    ! pour imposer les vitesses
! imx(i,j)=2 -> traitement gnral equation elliptique    ! 
! imx(i,j) < 0 condition aux limites                      ! voir routine rempli_Tuij
!------------------------------------------------------------------------------------

  

integer, dimension(nx1:nx2,ny1:ny2),intent(in)  :: imx    ! masque en entree routine
integer, dimension(nx1:nx2,ny1:ny2),intent(in)  :: imy    ! masque en entree routine



real, dimension(nx1:nx2,ny1:ny2),intent(out) :: uxnew     ! vitesse en sortie de la routine
real, dimension(nx1:nx2,ny1:ny2),intent(out) :: uynew     ! vitesse en sortie de la routine


! variables locales.
!--------------------

real    :: dx2=dx*dx     ! variable de travail
real    :: beta          ! pour le frottement
real    :: scal          ! pour le conditionnement (diagonale=1)

! pour les fronts, on suppose que l'epaisseur est celle du noeud amont
! dans opposx et opposy

real, dimension(nx,ny) :: Hmx_oppos      !
real, dimension(nx,ny) :: Hmy_oppos      ! 

if (itracebug.eq.1)  call tracebug(' Subroutine rempli_L2')


!-----------------------------
!$OMP PARALLEL
!$OMP WORKSHARE
Tu(:,:,:,:) = 0. ; Tv(:,:,:,:) = 0. ; Su(:,:,:,:) = 0. ; Sv(:,:,:,:) = 0.
opposx(:,:) = 0. ; opposy(:,:) = 0.
Mu(:,:,:,:) = 0. ; Mv(:,:,:,:) = 0. ; Nu(:,:,:,:) = 0. ; Nv(:,:,:,:) = 0.

ligu_L2(:,:) = 0 ; ligv_L2(:,:) = 0

ok_umat(:,:) = .true.  ; ok_vmat(:,:) = .true.
ghost_x(:,:) = .false. ; ghost_y(:,:) = .false.

pos_ligu(:,:)=-9999 ; pos_ligv(:,:)=-9999

Hmx_oppos(:,:) = Hmx(:,:)
Hmy_oppos(:,:) = Hmy(:,:)
!$OMP END WORKSHARE
!$OMP END PARALLEL

! calcul de Hmx_oppos et Hmy_oppos dans le cas de fronts
! ce calcul pourrait tre dans diagno et faire passer hmx_oppos par module

! 4 avril 2012 : Finalement il me semble que c'est faux, il suffit de garder le Hmx habituel.
! cat

!!$do j=ny1,ny2
!!$   do i=nx1,nx2
!!$
!!$      il=max(1,i-1)
!!$      jl=max(1,j-1)
!!$
!!$
!!$         if ((H(i,j).gt.1.).and.(H(il,j).le.1.)) then   ! Bord West : on garde H(i,j)
!!$            Hmx_oppos(i,j)=H(i,j)
!!$
!!$         else if ((H(i,j).le.1.).and.(H(il,j).gt.1.)) then   ! Bord Est : on garde H(il,j)
!!$            Hmx_oppos(i,j)=H(il,j)
!!$
!!$         end if
!!$
!!$         if ((H(i,j).gt.1.).and.(H(i,jl).le.1.)) then   ! Bord Sud : on garde H(i,j)
!!$            Hmy_oppos(i,j)=H(i,j)
!!$
!!$         else if ((H(i,j).le.1.).and.(H(i,jl).gt.1.)) then   ! Bord Nord on garde H(i,jl)
!!$            Hmy_oppos(i,j)=H(i,jl)
!!$
!!$         end if
!!$
!!$
!!$      end do
!!$   end do


! mailles mineures en dehors du domaine
!imy(:,ny1) = 0
!imx(nx1,:) = 0           


! limite la valeur de frotmx a betamax
! devrait plutot etre fait dans dragging
!------------------------------------------
call limit_frotm


! remplissage des Tuij, .... 
!----------------------------
call rempli_Tuij

! debug : impression des Tuij en un point
!write(166,*)
!write(166,*) 'impression des Tuij'

!j = 12

!do i = 244,245

!!$do i = 244,246
!!$   write(166,801) i,imx(i,j),frotmx(i,j),hmx_oppos(i,j),sdx(i,j)
!!$   do k= 1,-1,-1
!!$      write(166,800)  k,Tu(i,j,-1:1,k),opposx(i,j)
!!$   end do
!!$   write(166,*)
!!$   do k= 1,-1,-1
!!$      write(166,800)  k,Tv(i,j,-1:1,k)
!!$   end do
!!$   write(166,*)
!!$   write(166,*) 'impression des Svij'
!!$   write(166,801) i,imy(i,j),frotmy(i,j),hmy_oppos(i,j),sdy(i,j)
!!$   do k= 1,-1,-1
!!$      write(166,800)  k,Su(i,j,-1:1,k),opposy(i,j)
!!$   end do
!!$   write(166,*)
!!$   do k= 1,-1,-1
!!$      write(166,800)  k,Sv(i,j,-1:1,k)
!!$   end do
!!$   
!!$
!!$
!!$800 format(' k=',(i0,1x),': ',4(es12.4,1x) )
!!$801 format(' i,im,frotm,hm,sd   ',2(i0.1,x),3(es12.4,1x))
!!$   write(166,*)
!!$end do
!!$write(166,*)


! debug : differents termes de l equation MacAyeal
!---------------------------------------------------


!!$do j=ny1+1,ny2-1 
!!$   do i=nx1+1,nx2-1
!!$
!!$      if (gzmx(i,j)) then  
!!$         do jl=-1,1
!!$            do il=-1,1
!!$               debug_3D(i,j,82) =debug_3D(i,j,82)+(Tu(i,j,il,jl)*Vcol_x(i+il,j+jl) & 
!!$                    +Tv(i,j,il,jl)*Vcol_y(i+il,j+jl))             
!!$
!!$            end do
!!$         end do
!!$!         debug_3D(i,j,83)=frotmx(i,j)*dx2*Vcol_x(i,j)
!!$      end if
!!$   end do
!!$end do


!!$do j=ny1,ny2
!!$   do i=nx1,nx2
!!$      debug_3D(i,j,84) = imx(i,j)
!!$   end do
!!$end do

!   call rempli_Tuij                    ! pour reutiliser le nouveau beta


! Pour determiner les colonnes Mu, Mv, Nu, Nv
!---------------------------------------------
call Mu_Mv


! pour faire une sortie graphique de la matrice L2 et bande couchee avant tout rangement
!---------------------------------------------------------------------------------------
!call graphique_L2(2*ny+2,2*ny+2,nx1,nx2,ny1,ny2,imx(nx1:nx2,ny1:ny2),imy(nx1:nx2,ny1:ny2))  


! pour determiner les noeuds qui participent a l'equation elliptique
!-------------------------------------------------------------------
call okmat0

! pour avoir la diagonale=1 a faire imperativement APRES okmat
!--------------------------------------------------------------
!$OMP PARALLEL
!$OMP DO PRIVATE(scal)
do j=ny1,ny2
   do i=nx1,nx2


      if (imx(i,j).ne.0) then                        ! test pour eviter les noeuds non traites  

         scal=Tu(i,j,0,0) 
         if(abs(scal).lt.1.e-30) then
            write(num_tracebug,*)'                      i,j,imx,pvi',i,j,imx(i,j),pvi(i,j)
!            write(num_tracebug,*) Tu(i,j,:,:)
         end if


         Tu(i,j,:,:)=Tu(i,j,:,:)/scal                ! qui produiraient une division par 0
         Tv(i,j,:,:)=Tv(i,j,:,:)/scal                ! Tu(:,:,0,0) est la diagonale
         opposx(i,j)=opposx(i,j)/scal

!        if ((i.eq.200).and.(j.eq.251)) then 
!            write(num_tracebug,*)'      i,j,imx',i,j,imx(i,j)
!            write(num_tracebug,*)'     pvi,scal',pvi(i,j),scal
!            write(num_tracebug,*) Tu(i,j,:,-1)
!            write(num_tracebug,*) Tu(i,j,:,0)
!            write(num_tracebug,*) Tu(i,j,:,1)
!            write(num_tracebug,*)'       opposx',opposx(i,j)
!         end if



      end if

      if (imy(i,j).ne.0) then                        ! test pour eviter les noeuds non traites 
         scal=Sv(i,j,0,0)
         if(abs(scal).lt.1.e-30) then
            write(num_tracebug,*)'                      i,j,Svij,imy,pvi',i,j,imy(i,j),pvi(i,j)
!            write(num_tracebug,*) Sv(i,j,:,:)
         end if

         Su(i,j,:,:)=Su(i,j,:,:)/scal                ! qui produiraient une division par 0
         Sv(i,j,:,:)=Sv(i,j,:,:)/scal                ! Sv(:,:,0,0) est la diagonale
         opposy(i,j)=opposy(i,j)/scal
      end if

 

   end do
end do
!$OMP END DO
!$OMP END PARALLEL

! mise a identite des noeuds fantomes
call ghost_identite

call ghost_remove

! call graphique_L2(kl,ku,nx1,nx2,ny1,ny2,imx(nx1:nx2,ny1:ny2),imy(nx1:nx2,ny1:ny2))   


!debug_3d(:,:,35) = opposx(:,:)
!debug_3d(:,:,36) = opposy(:,:)



call resol_ellipt(nx1,nx2,ny1,ny2,                                       &   ! bornes du domaine

                    uxprec(nx1:nx2,ny1:ny2), uyprec(nx1:nx2,ny1:ny2),    &   ! vitesses precedentes


                    uxnew(nx1:nx2,ny1:ny2),  uynew(nx1:nx2,ny1:ny2),     &   ! nouvelles vitesses

                    ifail_L2)                                                ! masques

if (itracebug.eq.1) call tracebug ('apres subroutine resol_ellipt')

if (itracebug.eq.1) write(num_tracebug,*) '(ifail_L2 (si 0 pas de probleme)', ifail_L2

! remise a 0 des noeuds fantomes
!$OMP PARALLEL
!$OMP WORKSHARE
where ((Hmx(nx1:nx2,ny1:ny2).le.1.).and.(flgzmx(nx1:nx2,ny1:ny2)))
   uxnew(nx1:nx2,ny1:ny2)=0.
end where

where ((Hmy(nx1:nx2,ny1:ny2).le.1.).and.(flgzmy(nx1:nx2,ny1:ny2)))
   uynew(nx1:nx2,ny1:ny2)=0.
end where
!$OMP END WORKSHARE
!$OMP END PARALLEL

! call graphique_L2(kl,ku,nx1,nx2,ny1,ny2,imx(nx1:nx2,ny1:ny2),imy(nx1:nx2,ny1:ny2))   



! pour appeler la routine de resolution du systeme lineaire

!!$i = 244
!!$do j=11,11,-1
!!$
!!$write(166,*) 'j=',j
!!$   write(166,803) i,uxprec(i,j),i+1,uxprec(i+1,j),i+2,uxprec(i+2,j)
!!$   write(166,802) i,uxnew(i,j),i+1,uxnew(i+1,j),i+2,uxnew(i+2,j)
!!$802 format(3('uxnew ','   i=',(i0,1x,es12.4,1x) ))
!!$803 format(3('uxprec','   i=',(i0,1x,es12.4,1x) ))
!!$write(166,*)

!!$end do

! limitation des nouvelles vitesses

!!$uxnew(:,:)=max(-3900.,uxnew(:,:))
!!$uxnew(:,:)=min( 3900.,uxnew(:,:))
!!$uynew(:,:)=max(-3900.,uynew(:,:))
!!$uynew(:,:)=min( 3900.,uynew(:,:))
!!$




888 return




contains                        ! pour que les subroutines qui suivent partagent les variables de rempli_L2


!-----------------------------------------------------------------------
! routines internes au module
!
! tout ce qui suite est ecrit sous forme de routines
! pour rendre plus lisible remplidom


subroutine limit_frotm

!$USE OMP_LIB
!-------------------------
! limite la valeur de frotmx a betamax
! devrait plutot etre fait dans dragging

if (itracebug.eq.1)  call tracebug(' Subroutine limit_frotm')
if (itracebug.eq.1) write(Num_tracebug,*)'betamax = ',betamax

!$OMP PARALLEL
!$OMP WORKSHARE
where (flgzmx(:,:))
   frotmx(:,:)=min(abs(betamx(:,:)),betamax_2d(:,:))
elsewhere
   frotmx(:,:)=0
end where

where (flgzmy(:,:))
   frotmy(:,:)=min(abs(betamy(:,:)),betamax_2d(:,:))
elsewhere
   frotmy(:,:)=0
end where
!$OMP END WORKSHARE
!$OMP END PARALLEL

end subroutine limit_frotm
!------------------------------------------------------------------


!------------------------------------------------------------------
!
! subroutines calcul des Tuij                         Cat juin 2008
!

subroutine rempli_Tuij                                    ! appelle tous les cas

!$ USE OMP_LIB

! imx(i,j)=0 -> ne pas traiter cette vitesse
! imx(i,j)=1 -> uxnew(i,j)=uxprec(i,j)                    ! pour imposer les vitesses
! imx(i,j)=2 -> traitement gnral equation elliptique    ! 

! imx(i,j) < 0 condition aux limites                    
!------------------------------------
! numerotation en tournant dans le sens direct.
! -1 bord Sud (en bas), -2 bord Est (a droite)
! -3 bord Nord (en haut), -4 bord West (a gauche).
! les coins ont l'indice combine -12 coin en bas a gauche

!   -34                -3   Nord              -23    
!     !----------------------------------------!
!     !                                        !
!     !              1 (prescrite)             !
!  -4 !                   ou                   ! -2
! West!              2 (L2)                    !  Est
!     !                                        !
!     !                                        !
!     !----------------------------------------!
!    -41               -1  Sud                -12 


if (itracebug.eq.1)  call tracebug(' Rempli Tuij')




count_line=1                                 ! pour numeroter les lignes
!------------------------------------------------------------------
!$OMP PARALLEL
!$OMP DO
lignes_UV: do j=ny1,ny2 
   do i=nx1,nx2

      if (i.gt.nx1) then                      ! Vitesses U
         ligu_L2(i,j)=count_line
         !$OMP ATOMIC
         count_line=count_line+1
          
         case_imx: select case (imx(i,j))     ! la routine appelee depend d'imx
         !------------------------------------

         case(0)                              ! ne pas traiter ce point
         ligu_L2(i,j)=0
         count_line=count_line-1

         case(1)                              ! vitesse imposee
            call vel_U_presc(uxprec(i,j),Tu(i,j,0,0),opposx(i,j))    

         case(2)                              ! cas general
            call vel_U_general(frotmx(i,j),dx2,pvi(i,j),pvi(i-1,j),pvm(i,j),pvm(i,j+1),sdx(i,j),rog,hmx_oppos(i,j), &
            Tu(i,j,:,:),Tv(i,j,:,:),opposx(i,j))

         case(-1)                             ! bord sud
            call vel_U_South(Tu(i,j,:,:),Tv(i,j,:,:),opposx(i,j))          

         case(-2)                             ! bord Est
            call vel_U_East(pvi(i,j),pvi(i-1,j),rog,H(i-1,j),rowg,slv(i-1,j),B(i-1,j),dx,Tu(i,j,:,:),Tv(i,j,:,:),opposx(i,j))
       
         case(-3)                             ! bord nord
            call vel_U_North(Tu(i,j,:,:),Tv(i,j,:,:),opposx(i,j))   

         case(-4)                             ! bord West
            call vel_U_West(pvi(i,j),pvi(i-1,j),rog,H(i,j),rowg,slv(i,j),B(i,j),dx,Tu(i,j,:,:),Tv(i,j,:,:),opposx(i,j))

         case(-12)                            ! coin SE
            call vel_U_SE(Tu(i,j,:,:),opposx(i,j))

         case(-23)                            ! coin NE
            call vel_U_NE(Tu(i,j,:,:),opposx(i,j))

         case(-34)                            ! coin NW
            call vel_U_NW(Tu(i,j,:,:),opposx(i,j))

         case(-41)                            ! coin SW
            call vel_U_SW(Tu(i,j,:,:),opposx(i,j))

         end select case_imx
      end if

      if (j.gt.ny1) then                      ! Vitesses V

         ligv_L2(i,j)=count_line
         !$OMP ATOMIC
         count_line=count_line+1

         case_imy: select case (imy(i,j))     ! la routine appelee depend d'imy
        !------------------------------------

         case(0)                              ! ne pas traiter ce point
         ligv_L2(i,j)=0
         count_line=count_line-1

         case(1)                              ! vitesse imposee
            call vel_V_presc(uyprec(i,j),Sv(i,j,0,0),opposy(i,j))
            
         case(2)                              ! cas general
            call vel_V_general(frotmy(i,j),dx2,pvi(i,j),pvi(i,j-1),pvm(i,j),pvm(i+1,j),sdy(i,j),rog,hmy_oppos(i,j), &
            Su(i,j,:,:),Sv(i,j,:,:),opposy(i,j))
            
         case(-1)                             ! bord sud
            call vel_V_South(pvi(i,j),pvi(i,j-1),rog,H(i,j),rowg,slv(i,j),B(i,j),dx,Sv(i,j,:,:),Su(i,j,:,:),opposy(i,j))
            
         case(-2)                             ! bord Est
            call vel_V_East(Sv(i,j,:,:),Su(i,j,:,:),opposy(i,j))
            
         case(-3)                             ! bord nord
            call vel_V_North(pvi(i,j),pvi(i,j-1),rog,H(i-1,j),rowg,slv(i-1,j),B(i-1,j),dx,Sv(i,j,:,:),Su(i,j,:,:),opposy(i,j))
            
         case(-4)                             ! bord West
            call vel_V_West(Sv(i,j,:,:),Su(i,j,:,:),opposy(i,j))
            
         case(-12)                            ! coin SE
            call vel_V_SE(Sv(i,j,:,:),opposy(i,j)) 
            
         case(-23)                            ! coin NE
            call vel_V_NE(Sv(i,j,:,:),opposy(i,j))
            
         case(-34)                            ! coin NW
            call vel_V_NW(Sv(i,j,:,:),opposy(i,j)) 
            
         case(-41)                            ! coin SW

            call vel_V_SW(Sv(i,j,:,:),opposy(i,j))
            
         end select case_imy
      end if
   end do
end do lignes_UV
!$OMP END DO
!$OMP END PARALLEL

end subroutine rempli_Tuij
!------------------------------------------------------------------

! Les differents cas :
! Vitesses U, puis vitesses V les coins sont traites a la fin


! vitesses U
!------------------------------------------------------------------

subroutine vel_U_presc(uxprec,Tu,opposx)     !                     ! vitesse imposee

implicit none

real,intent(in)  :: uxprec
real,intent(out) :: Tu
real,intent(out) :: opposx

  Tu=1.
  opposx=uxprec

end subroutine vel_U_presc
!------------------------------------------------------------------
                   
subroutine vel_U_general(frotmx,dx2,pvi,pvi_imoinsun,pvm,pvm_jplusun,sdx,rog,hmx_oppos,Tu,Tv,opposx)                   ! cas general

implicit none

real,intent(in) :: frotmx
real,intent(in) :: dx2
real,intent(in) :: pvi
real,intent(in) :: pvi_imoinsun
real,intent(in) :: pvm
real,intent(in) :: pvm_jplusun
real,intent(in) :: sdx
real,intent(in) :: rog
real,intent(in) :: hmx_oppos
real,dimension(-1:1,-1:1),intent(inout) :: Tu
real,dimension(-1:1,-1:1),intent(inout) :: Tv
real,intent(out) :: opposx

   beta=frotmx*dx2                       !  Terme en u(i,j)

   Tu(0,0)  = -4.*pvi - 4.*pvi_imoinsun - pvm_jplusun - pvm - beta

   Tu(-1,0) =  4.*pvi_imoinsun              !  Terme en u(i-1,j)

   Tu(1,0)  =  4.*pvi                !  Terme en u(i+1,j)

   Tu(0,-1) =  pvm                   !  Terme en u(i,j-1)

   Tu(0,1)  =  pvm_jplusun                 !  Terme en u(i,j+1)

   Tv(0,0)  = -2.*pvi - pvm       !  Terme en v(i,j)

   Tv(-1,0) =  2.*pvi_imoinsun + pvm     !  Terme en v(i-1,j)

   Tv(-1,1) = -2.*pvi_imoinsun - pvm_jplusun   !  Terme en v(i-1,j+1)

   Tv(0,1)  =  2.*pvi + pvm_jplusun     !  Terme en v(i,j+1)

   opposx   =  rog*hmx_oppos*sdx*dx2  !  Terme en opposx(i,j)

   return
end subroutine vel_U_general

!------------------------------------------------------------------
! voir explications dans vel_U_West

subroutine vel_U_South(Tu,Tv,opposx)       ! bord sud non cisaillement           

implicit none                       
real,dimension(-1:1,-1:1),intent(inout) :: Tu
real,dimension(-1:1,-1:1),intent(inout) :: Tv
real, intent(out) :: opposx

  Tu(0,0)  =  1.
  Tu(0,1)  = -1.
  Tv(0,1)  = -1.
  Tv(-1,1) =  1.
  opposx   =  0.

   return
end subroutine vel_U_South
!------------------------------------------------------------------

subroutine vel_U_North(Tu,Tv,opposx)     ! bord nord   non cisaillement       

implicit none                       
real,dimension(-1:1,-1:1),intent(inout) :: Tu
real,dimension(-1:1,-1:1),intent(inout) :: Tv
real, intent(out) :: opposx

  Tu(0,0)  =  1.
  Tu(0,-1) = -1.
  Tv(0,0)  =  1.
  Tv(-1,0) = -1.
  opposx   =  0.

   return
end subroutine vel_U_North
!------------------------------------------------------------------
! voir explications dans vel_U_West

subroutine vel_U_East(pvi,pvi_imoinsun,rog,H_imoinsun,rowg,slv_imoinsun,B_imoinsun,dx,Tu,Tv,opposx) ! bord Est   pression

implicit none
real,intent(in) :: pvi
real,intent(in) :: pvi_imoinsun
real,intent(in) :: rog
real,intent(in) :: H_imoinsun
real,intent(in) :: rowg
real,intent(in) :: slv_imoinsun
real,intent(in) :: B_imoinsun
real,intent(in) :: dx                       
real,dimension(-1:1,-1:1),intent(inout) :: Tu
real,dimension(-1:1,-1:1),intent(inout) :: Tv
real, intent(out) :: opposx
       
  Tu(0,0)   =  4.*pvi_imoinsun
  Tu(-1,0)  = -4.*pvi_imoinsun
  Tv(0,1)   =  2.*pvi
  Tv(0,0)   = -2.*pvi
  opposx    =  0.5*(rog*H_imoinsun**2-rowg*(max(slv_imoinsun-B_imoinsun,0.))**2)*dx

   return
end subroutine vel_U_East
!------------------------------------------------------------------

subroutine vel_U_West(pvi,pvi_imoinsun,rog,H,rowg,slv,B,dx,Tu,Tv,opposx) ! bord West pression
                                             ! tous les coef * -1
implicit none
real,intent(in) :: pvi
real,intent(in) :: pvi_imoinsun
real,intent(in) :: rog
real,intent(in) :: H
real,intent(in) :: rowg
real,intent(in) :: slv
real,intent(in) :: B
real,intent(in) :: dx                       
real,dimension(-1:1,-1:1),intent(inout) :: Tu
real,dimension(-1:1,-1:1),intent(inout) :: Tv
real, intent(out) :: opposx


!  Tu(i,j,0,0)    =  4.*pvi(i-1,j)
!  Tu(i,j,1,0)    = -4.*pvi(i-1,j)

  Tu(0,0)    =  4.*pvi              ! quand le bord est epais, il faut prendre 
  Tu(1,0)    = -4.*pvi              ! le noeud du shelf pas celui en H=1
                                             ! s'il est fin, pas de difference

!!$  Tv(i-1,j,-1,0) =  2.*pvi(i-1,j)         ! ????
!!$  Tv(i-1,j,-1,1) = -2.*pvi(i-1,j)

  Tv(-1,0) =  2.*pvi_imoinsun
  Tv(-1,1) = -2.*pvi_imoinsun
 opposx    = -0.5*(rog*H**2-rowg*(max(slv-B,0.))**2)*dx

 
   return
end subroutine vel_U_West
!------------------------------------------------------------------


! vitesses V
!------------------------------------------------------------------

subroutine vel_V_presc(uyprec,Sv,opposy)          ! vitesse imposee

implicit none
real,intent(in)  :: uyprec
real,intent(out) :: Sv
real,intent(out) :: opposy

  Sv=1.
  opposy=uyprec

   return
end subroutine vel_V_presc
!------------------------------------------------------------------
                    
subroutine vel_V_general(frotmy,dx2,pvi,pvi_jmoinsun,pvm,pvm_iplusun,sdy,rog,hmy_oppos,Su,Sv,opposy)! cas general

implicit none
real,intent(in) :: frotmy
real,intent(in) :: dx2
real,intent(in) :: pvi
real,intent(in) :: pvi_jmoinsun
real,intent(in) :: pvm
real,intent(in) :: pvm_iplusun
real,intent(in) :: sdy
real,intent(in) :: rog
real,intent(in) :: hmy_oppos
real,dimension(-1:1,-1:1),intent(inout) :: Su
real,dimension(-1:1,-1:1),intent(inout) :: Sv
real,intent(out) :: opposy

real :: beta


   beta          =  frotmy*dx2           !  Terme en v(i,j)
   Sv(0,0)   = -4.*pvi - 4.*pvi_jmoinsun - pvm_iplusun - pvm-beta

   Sv(0,-1)  =  4.*pvi_jmoinsun             !  Terme en v(i,j-1)

   Sv(0,1)   =  4.*pvi               !  Terme en v(i,j+1)

   Sv(-1,0)  =  pvm                  !  Terme en v(i-1,j)

   Sv(1,0)   =  pvm_iplusun                !  Terme en v(i+1,j)

   Su(0,0)   = -2.*pvi - pvm      !  Terme en u(i,j)

   Su(0,-1)  =  2.*pvi_jmoinsun + pvm    !  Terme en u(i,j-1)

   Su(1,-1)  = -2.*pvi_jmoinsun - pvm_iplusun  !  Terme en u(i+1,j-1)

   Su(1,0)   =  2.*pvi + pvm_iplusun    !  Terme en u(i+1,j)

   opposy    =  rog*hmy_oppos*sdy*dx2 !  Terme en opposy(i,j)

   return
 end subroutine vel_V_general

!------------------------------------------------------------------

subroutine vel_V_West(Sv,Su,opposy)     ! bord West non cisaillement           

implicit none
real,dimension(-1:1,-1:1),intent(inout) :: Sv
real,dimension(-1:1,-1:1),intent(inout) :: Su
real,intent(out) :: opposy

  Sv(0,0)  =  1.
  Sv(1,0)  = -1.
  Su(1,0)  = -1.
  Su(1,-1) =  1.
  opposy   =  0.

   return
end subroutine vel_V_West
!------------------------------------------------------------------


subroutine vel_V_East(Sv,Su,opposy)         ! bord Est   non cisaillement       

implicit none
real,dimension(-1:1,-1:1),intent(inout) :: Sv
real,dimension(-1:1,-1:1),intent(inout) :: Su
real,intent(out) :: opposy

  Sv(0,0)  =  1.
  Sv(-1,0) = -1.
  Su(0,0)  =  1.
  Su(0,-1) = -1.
  opposy   =  0.

   return
end subroutine vel_V_East
!------------------------------------------------------------------
!-----------------------------------------------------------------------
! voir explications dans vel_U_West

subroutine vel_V_North(pvi,pvi_jmoinsun,rog,H_imoinsun,rowg,slv_imoinsun,B_imoinsun,dx,Sv,Su,opposy) ! bord Nord   pression

implicit none
real,intent(in) :: pvi
real,intent(in) :: pvi_jmoinsun
real,intent(in) :: rog
real,intent(in) :: H_imoinsun
real,intent(in) :: rowg
real,intent(in) :: slv_imoinsun
real,intent(in) :: B_imoinsun
real,intent(in) :: dx
real,dimension(-1:1,-1:1),intent(inout) :: Sv
real,dimension(-1:1,-1:1),intent(inout) :: Su
real,intent(out) :: opposy
       
  Sv(0,0)  =  4.*pvi_jmoinsun
  Sv(0,-1) = -4.*pvi_jmoinsun
  Su(1,0)  =  2.*pvi
  Su(0,0)  = -2.*pvi
  opposy   = 0.5*(rog*H_imoinsun**2-rowg*(max(slv_imoinsun-B_imoinsun,0.))**2)*dx

   return
end subroutine vel_V_North
!------------------------------------------------------------------
! voir explications dans vel_U_West

subroutine vel_V_South(pvi,pvi_jmoinsun,rog,H,rowg,slv,B,dx,Sv,Su,opposy) ! bord sud  pression
                                              ! tous les coef * -1
implicit none
real,intent(in) :: pvi
real,intent(in) :: pvi_jmoinsun
real,intent(in) :: rog
real,intent(in) :: H
real,intent(in) :: rowg
real,intent(in) :: slv
real,intent(in) :: B
real,intent(in) :: dx
real,dimension(-1:1,-1:1),intent(inout) :: Sv
real,dimension(-1:1,-1:1),intent(inout) :: Su
real,intent(out) :: opposy

  Sv(0,0)  = 4.*pvi
  Sv(0,1)  =-4.*pvi
  Su(0,-1) = 2.*pvi_jmoinsun
  Su(1,-1) =-2.*pvi_jmoinsun
  opposy   = -0.5*(rog*H**2-rowg*(max(slv-B,0.))**2)*dx

   return
end subroutine vel_V_South
!------------------------------------------------------------------


! coins
!------------------------------------------------------------------

! Dans les coins, la condition non-cisaillement est la meme
! pour les noeuds en U et en V. Il ne faut donc l'utiliser qu'une seule fois
! On ajoute alors comme condition la symetrie
! 
! du coup la condition s'ecrit dU/dy=0 en U et dV/dx=0 en V
! la condition de non cisaillement est automatiquement satisfaite
! 


!Coin SE
!----------
subroutine vel_U_SE(Tu,opposx)                        

implicit none
real,dimension(-1:1,-1:1),intent(inout) :: Tu
real,intent(out) :: opposx

  Tu(0,0) =  1.    !*pvi(i,j)
  Tu(0,1) = -1.    !*pvi(i,j)
  opposx  =  0.

   return
end subroutine vel_U_SE
!------------------------------------------------------------------

subroutine vel_V_SE(Sv,opposy)                        

implicit none
real,dimension(-1:1,-1:1),intent(inout) :: Sv
real,intent(out) :: opposy

  Sv(0,0) =  1.   !*pvi(i,j)
  Sv(-1,0)= -1.   !*pvi(i,j)
  opposy  =  0.

   return
end subroutine vel_V_SE
!------------------------------------------------------------------


!Coin SW
!----------
subroutine vel_U_SW(Tu,opposx)                        

implicit none
real,dimension(-1:1,-1:1),intent(inout) :: Tu
real,intent(out) :: opposx

  Tu(0,0) =  1.   ! *pvi(i,j)
  Tu(0,1) = -1.   ! *pvi(i,j)
  opposx  =  0.

   return
end subroutine vel_U_SW
!------------------------------------------------------------------

subroutine vel_V_SW(Sv,opposy)

implicit none
real,dimension(-1:1,-1:1),intent(inout) :: Sv
real,intent(out) :: opposy                    

  Sv(0,0) =  1.   !*pvi(i,j)
  Sv(1,0) = -1.   !*pvi(i,j)
  opposy  =  0.

   return
end subroutine vel_V_SW
!------------------------------------------------------------------

!Coin NE
!----------
subroutine vel_U_NE(Tu,opposx)

implicit none
real,dimension(-1:1,-1:1),intent(inout) :: Tu
real,intent(out) :: opposx                     

  Tu(0,0)  =  1.  ! *pvi(i,j)
  Tu(0,-1) = -1.  ! *pvi(i,j)
  opposx   =  0.

   return
end subroutine vel_U_NE
!------------------------------------------------------------------

subroutine vel_V_NE(Sv,opposy)                        

implicit none
real,dimension(-1:1,-1:1),intent(inout) :: Sv
real,intent(out) :: opposy

  Sv(0,0)  =  1.  ! *pvi(i,j)
  Sv(-1,0) = -1.  ! *pvi(i,j)
  opposy   =  0.

   return
end subroutine vel_V_NE
!------------------------------------------------------------------

!Coin NW
!----------
subroutine vel_U_NW(Tu,opposx)

implicit none                        
real,dimension(-1:1,-1:1),intent(inout) :: Tu
real,intent(out) :: opposx 

  Tu(0,0)  =  1.  ! *pvi(i,j)
  Tu(0,-1) = -1.  ! *pvi(i,j)
  opposx   =  0.

   return
end subroutine vel_U_NW
!------------------------------------------------------------------

subroutine vel_V_NW(Sv,opposy)                        

implicit none                        
real,dimension(-1:1,-1:1),intent(inout) :: Sv
real,intent(out) :: opposy 

  Sv(0,0)  =  1.  !*pvi(i,j)
  Sv(1,0)  = -1.  !*pvi(i,j)
  opposy   =  0.

   return
end subroutine vel_V_NW
!------------------------------------------------------------------


! fin des routines Tuij,...
!------------------------------------------------------------------



subroutine Mu_Mv

!$ USE OMP_LIB
!---------------------------------------------------------  
! Pour dterminer  Mu, Mv, Nu, Nv les colonnes de L2
! A appeler avant la division des Tu, Sv par la diagonale
!---------------------------------------------------------

! pourrait tre dans un autre fichier

implicit none

if (itracebug.eq.1)  call tracebug(' Mu_Mv')

!$OMP PARALLEL
!$OMP DO PRIVATE(ilmin,ilmax,jlmin,jlmax)
Col_U: do j=ny1,ny2   ! balaye tous les noeuds U
   do i=nx1+1,nx2

!     ilmin et jlmin boucle U
      ilmin=max(-1,nx1+1-i)  ! pour avoir i+il entre nx1+1 et nx2
      ilmax=min(1,nx2-i)

      jlmin=max(-1,ny1-j)    ! pour avoir j+jl entre ny1 et ny2
      jlmax=min(1,ny2-j)



      do jl=jlmin,jlmax      ! balaye tous les coefficients d'un noeud U
         do il=ilmin,ilmax
           
            Mu(i+il,j+jl,-il,-jl)=abs(Tu(i,j,il,jl))   ! abs pour faciliter les tests.
            Mv(i+il,j+jl,-il,-jl)=abs(Tv(i,j,il,jl))


         end do
      end do
      diagU(i,j)=Tu(i,j,0,0)

   end do
end do Col_U
!$OMP END DO

!$OMP DO PRIVATE(ilmin,ilmax,jlmin,jlmax)
Col_V: do j=ny1+1,ny2   ! balaye tous les noeuds V
   do i=nx1,nx2

!     ilmin et jlmin boucle V
      ilmin=max(-1,nx1-i)      ! pour avoir i+il entre nx1 et nx2
      ilmax=min(1,nx2-i)

      jlmin=max(-1,ny1+1-j)    ! pour avoir j+jl entre ny1+1 et ny2
      jlmax=min(1,ny2-j)

      do jl=jlmin,jlmax        ! balaye tous les coefficients d'un noeud U
         do il=ilmin,ilmax
            
            Nu(i+il,j+jl,-il,-jl)=abs(Su(i,j,il,jl))   ! abs pour faciliter les tests.
            Nv(i+il,j+jl,-il,-jl)=abs(Sv(i,j,il,jl))

         end do
      end do

      diagV(i,j)=Sv(i,j,0,0)


   end do
end do Col_V
!$OMP END DO
!$OMP END PARALLEL

return
end subroutine Mu_Mv
!----------------------------------------------------------------------------------

!-------------------------------------------------------------------------
subroutine okmat0       

!$ USE OMP_LIB 
! pourrais tre dans un autre fichier avec un appel (nx1,nx2,ny1,ny2)

! determine les noeuds qui participent a l'equation elliptique

! ce sont les noeuds dont aucun autre noeud ne dpend (colonne Mu,Nu nulle)
! et qui ne dpendent d'aucun noeud (ligne Tu,Tv nulle sauf diagonale)

implicit none

! tableaux de travail
real, dimension(-1:1,-1:1) :: Mloc   ! sous tableau local de M
real, dimension(-1:1,-1:1) :: Nloc   ! sous tableau local de N
real, dimension(-1:1,-1:1) :: Tuloc  ! sous tableau local de Tu
real, dimension(-1:1,-1:1) :: Tvloc  ! sous tableau local de Tv
real, dimension(-1:1,-1:1) :: Suloc  ! sous tableau local de Su
real, dimension(-1:1,-1:1) :: Svloc  ! sous tableau local de Sv

! test pour les noeuds fantomes
real :: pvi_ghost


if (itracebug.eq.1)  call tracebug(' Okmat')

! initialisation

pvi_ghost=40.*pvimin
!pvi_ghost=10.*pvimin   ! condition trop faible  : pas de ghost
!$OMP PARALLEL
!$OMP WORKSHARE
ok_umat(:,:) =.false.
ok_vmat(:,:) =.false.
ghost_x(:,:) =.false. 
ghost_y(:,:) =.false.
!$OMP END WORKSHARE

!$OMP DO PRIVATE(Mloc,Nloc,Tuloc,Tvloc,Suloc,Svloc)
do j=ny1,ny2
   do i=nx1,nx2

! okumat     vitesse U
!---------------------------


! condition colonne.
!---------------------------
      Mloc(:,:)=abs(Mu(i,j,:,:))
      Nloc(:,:)=abs(Nu(i,j,:,:))
      Mloc(0,0)=0.                 ! diagonale

! okumat vrai si au moins un terme de la colonne est > eps_col et imx non 0
      ok_umat(i,j)=((any(Mloc.gt.eps_col)).or.(any(Nloc.gt.eps_col))).and.(imx(i,j).ne.0)


! ghost_x est vrai si tous les elements sont inferieurs a pvi_ghost
      ghost_x(i,j)=((all(Mloc.lt.pvi_ghost)).and.(all(Nloc.lt.pvi_ghost)))

       

! condition ligne
!---------------------------
      Tuloc(:,:)=abs(Tu(i,j,:,:))
      Tvloc(:,:)=abs(Tv(i,j,:,:))

! ghost_x est vrai si tous les elements y compris diagonale sont inferieurs a pvi_ghost
      ghost_x(i,j)=(ghost_x(i,j).and.((all(Tuloc.lt.pvi_ghost)).and.(all(Tvloc.lt.pvi_ghost))))


      Tuloc(0,0)=0.                ! mettre cette ligne en commentaire pour avoir
                                   ! aussi les noeuds identite

! ok_umat est vrai si au moins un des termes de la ligne  est > eps_col
      ok_umat(i,j)=ok_umat(i,j).or.((any(Tuloc.gt.eps_col)).or.(any(Tvloc.gt.eps_col)))



! on elimine les noeuds en imx=0
      ok_umat(i,j)=ok_umat(i,j).and.(imx(i,j).ne.0)


! on elimine les noeuds en ghost
      ok_umat(i,j)= ok_umat(i,j).and.(.not.ghost_x(i,j))

! okvmat     vitesse V
!---------------------------

! condition colonne.
!-------------------------------
      Mloc(:,:)=abs(Mv(i,j,:,:))
      Nloc(:,:)=abs(Nv(i,j,:,:))
      Nloc(0,0)=0.

! okvmat vrai si au moins un terme de la colonne est > eps_col et imy non 0
      ok_vmat(i,j)=((any(Mloc.gt.eps_col)).or.(any(Nloc.gt.eps_col))).and.(imy(i,j).ne.0)


! ghost_y est vrai si tous les elements sont inferieurs a pvi_ghost
      ghost_y(i,j)=((all(Mloc.lt.pvi_ghost)).and.(all(Nloc.lt.pvi_ghost)))


! condition ligne
!--------------------------------
      Suloc(:,:)=abs(Su(i,j,:,:))
      Svloc(:,:)=abs(Sv(i,j,:,:))

! ghost_y est vrai si tous les elements y compris diagonale sont inferieurs a pvi_ghost
      ghost_y(i,j)=(ghost_y(i,j).and.((all(Suloc.lt.pvi_ghost)).and.(all(Svloc.lt.pvi_ghost))))

      Svloc(0,0)=0.                ! mettre cette ligne en commentaire pour avoir
                                   ! aussi les noeuds identite

! okvmat est vrai si au moins un des termes de la ligne  est > eps_col
      ok_vmat(i,j)=ok_vmat(i,j).or.((any(Suloc.gt.eps_col)).or.(any(Svloc.gt.eps_col)))


! on elimine les noeuds en imy=0
      ok_vmat(i,j)=ok_vmat(i,j).and.(imy(i,j).ne.0)

! on elimine les noeuds en ghost
       ok_vmat(i,j)=ok_vmat(i,j).and.(.not.ghost_y(i,j))

! test 
       if ((.not.ghost_y(i,j)).and.(.not.ok_vmat(i,j))) then
          write(6,*)'pb ok_vmat',i,j
          write(6,*) 'Mloc',Mloc
          write(6,*) 'Nloc',Nloc
          write(6,*) 'Suloc',Suloc
          write(6,*) 'Svloc',Svloc
       end if
 

   end do
end do
!$OMP END DO 

! on enleve les lignes 1 decalees
!$OMP WORKSHARE
ok_umat(1,:)=.false.
ok_vmat(:,1)=.false.
!$OMP END WORKSHARE
!$OMP END PARALLEL

! sortie Netcdf pour verifier ok_umat
!~ where (ok_umat(:,:))
!~    debug_3D(:,:,39)=1
!~ elsewhere
!~    debug_3D(:,:,39)=0
!~ end where
!~ 
!~ where (ok_vmat(:,:))
!~    debug_3D(:,:,40)=1
!~ elsewhere
!~    debug_3D(:,:,40)=0
!~ end where
!~ 
!~ ! sortie Netcdf pour verifier ghost
!~ where (ghost_x(:,:))
!~    debug_3D(:,:,41)=1
!~ elsewhere
!~    debug_3D(:,:,41)=0
!~ 
!~ end where
!~ 
!~ where (ghost_y(:,:))
!~    debug_3D(:,:,42)=1
!~ elsewhere
!~    debug_3D(:,:,42)=0
!~ end where

if (itracebug.eq.1)  call tracebug(' Sortie  Okmat')
return
end subroutine okmat0

!--------------------------------------------------------------------------------------
subroutine ghost_identite                           ! mise a identite des noeuds fantomes
!$ USE OMP_LIB

implicit none

!$OMP PARALLEL
!$OMP DO
do j=ny1,ny2
   do i=nx1,nx2


      if (ghost_x(i,j)) then                        ! noeuds fantomes
         Tu(i,j,:,:)=0.
         Tv(i,j,:,:)=0.
         Tu(i,j,0,0)=1.
         opposx(i,j)=-3.33333
      end if

      if (ghost_y(i,j)) then                        ! noeuds fantomes
         Su(i,j,:,:)=0.
         Sv(i,j,:,:)=0.
         Sv(i,j,0,0)=1.
         opposy(i,j)=-3.33333
      end if

   end do
end do
!$OMP END DO
!$OMP END PARALLEL

return
end subroutine ghost_identite
!------------------------------------------------------------------------------

subroutine ghost_remove                   ! met a 0 les Tuij ... qui corresponde a des ghost
!$ USE OMP_LIB

implicit none

!$OMP PARALLEL
!$OMP DO PRIVATE(ilmin,ilmax,jlmin,jlmax)
do j=ny1,ny2
  do i=nx1,nx2

!u_ghost:       if (ok_umat(i,j)) then


    ilmin=max(-1,nx1+1-i)  ! pour avoir i+il entre nx1+1 et nx2
    ilmax=min(1,nx2-i)

    jlmin=max(-1,ny1-j)    ! pour avoir j+jl entre ny1 et ny2
    jlmax=min(1,ny2-j)

    do jl = jlmin , jlmax      ! balaye tous les coefficients d'un noeud U de -1 a 1
      do il = ilmin , ilmax   !                                           de -1 a 1 
        if (ghost_x(i+il,j+jl)) then
          Tu(i,j,il,jl)=0.
          Su(i,j,il,jl)=0.
        end if
    end do
    end do
!            end if u_ghost

!v_ghost:   if (ok_vmat(i,j)) then

    ilmin=max(-1,nx1-i)      ! pour avoir i+il entre nx1 et nx2
    ilmax=min(1,nx2-i)

    jlmin=max(-1,ny1+1-j)    ! pour avoir j+jl entre ny1+1 et ny2
    jlmax=min(1,ny2-j)

    do jl = jlmin , jlmax      ! balaye tous les coefficients d'un noeud U de -1 a 1
      do il = ilmin , ilmax   !                                           de -1 a 1 
        if (ghost_y(i+il,j+jl)) then
          Tv(i,j,il,jl)=0.
          Sv(i,j,il,jl)=0.
        end if
      end do
    end do
!            end if v_ghost
  end do
end do
!$OMP END DO
!$OMP END PARALLEL
end subroutine ghost_remove
!----------------------------------------------------------------------------------------
subroutine calc_beta(uxgiven,uygiven)         ! calcule betamx et betamy
!$USE OMP_LIB                                 ! a partir du champ de vitesse
                                              ! uxprec, uyprec
implicit none
real, dimension(nx,ny) :: uxgiven             ! prescribed velocity
real, dimension(nx,ny) :: uygiven             ! 
real                   :: maxbeta = 1.e6      ! beta will take maxbeta  value when 
real                   :: limgliss = 0.1      ! ugiven < limgliss (maxbeta in Pa an /m)
real                   :: betamin = 10.       ! minimum value on grounded point (in Pa an /m)

!debug_3D(:,:,69)=uxgiven(:,:)
!debug_3D(:,:,70)=uygiven(:,:)

                   
if (itracebug.eq.1)  call tracebug(' Subroutine calc_beta')

!$OMP PARALLEL
!$OMP DO
do j=ny1+1,ny2-1 
   do i=nx1+1,nx2-1

x_calc:    if (flgzmx(i,j)) then

      betamx(i,j)=-opposx(i,j) 
      debug_3D(i,j,35) = opposx(i,j)/dx/dx       ! driving stress ou pression hydrostatique (Pa)
      do jl=-1,1
         do il=-1,1
            betamx(i,j) = betamx(i,j)+(Tu(i,j,il,jl)*uxgiven(i+il,j+jl) & 
                                      +Tv(i,j,il,jl)*uygiven(i+il,j+jl)) 

!            debug_3D(i,j,82)=debug_3D(i,j,82) + (Tu(i,j,il,jl)*uxgiven(i+il,j+jl) & 
!                                      +Tv(i,j,il,jl)*uygiven(i+il,j+jl)) 

         end do
      end do

!      debug_3D(i,j,82)=debug_3D(i,j,82)/dx/dx   ! longitudinal stress computed from velocities


!  fonctionne mme quand frotmx n'est pas nul

      betamx(i,j) = betamx(i,j) + frotmx(i,j)*dx2 *uxgiven(i,j)
      
      if (abs(uxgiven(i,j)).gt.limgliss) then
         betamx(i,j)=betamx(i,j)/dx2/uxgiven(i,j)
         betamx(i,j)=betamx(i,j)*drag_mx(i,j)
         
      else
         if (flotmx(i,j)) then 
            betamx(i,j)=0.
         else
            betamx(i,j)= maxbeta   ! il faudrait mettre la limitation a l'aide du driving stress
         end if
      endif

   else                            ! les noeuds non flgzmx sont d'office a maxbeta
      betamx(i,j)=maxbeta     
   end if x_calc

y_calc:    if (flgzmy(i,j)) then
  
      betamy(i,j)=-opposy(i,j)
!      debug_3D(i,j,36) = opposy(i,j)/dx/dx      ! driving stress ou pression hydrostatique (Pa)
      do jl=-1,1
         do il=-1,1

            betamy(i,j) = betamy(i,j)+(Su(i,j,il,jl)*uxgiven(i+il,j+jl) & 
                                      +Sv(i,j,il,jl)*uygiven(i+il,j+jl))

!            debug_3D(i,j,83)=debug_3D(i,j,83) + (Su(i,j,il,jl)*uxgiven(i+il,j+jl) & 
!                                      +Sv(i,j,il,jl)*uygiven(i+il,j+jl))

         
         end do
      end do

!      debug_3D(i,j,83)=debug_3D(i,j,83) /dx/dx  ! longitudinal stress computed from velocities

!  fonctionne mme quand frotmx n'est pas nul

      betamy(i,j) = betamy(i,j) + frotmy(i,j)*dx2 *uygiven(i,j)

      if (abs(uygiven(i,j)).gt.limgliss) then
         betamy(i,j)=betamy(i,j)/dx2/uygiven(i,j)
         betamx(i,j)=betamx(i,j)*drag_my(i,j)
      else
         if (flotmy(i,j)) then 
            betamy(i,j)=0.
         else
            betamy(i,j)= maxbeta   ! il faudrait mettre la limitation a l'aide du driving stress
         end if
      endif

   else
      betamy(i,j)=maxbeta       ! les noeuds non flgzmx sont d'office a maxbeta
   end if y_calc

   end do
end do
!$OMP END DO
!$OMP END PARALLEL
! loop to spread negative beta on the neighbours

do j=2,ny-1
   do i=2,nx-1
      if (betamx(i,j).lt.0.) then
         betamx(i-1,j) = max(betamin, (betamx(i-1,j) - betamx(i,j)) * 0.5)
         betamx(i+1,j) = max(betamin,(betamx(i+1,j) - betamx(i,j)) * 0.5)
         betamx(i,j)   = max(betamin,betamx(i,j))
      end if
      if (betamy(i,j).lt.0.) then
         betamy(i,j-1) = max(betamin,betamy(i,j-1) - betamy(i,j) * 0.5)
         betamy(i,j+1) = max(betamin,betamy(i,j+1) - betamy(i,j) * 0.5)
         betamy(i,j)   = max(betamin,betamy(i,j))
      end if
   end do
end do

! average
!$OMP PARALLEL
!$OMP WORKSHARE
beta_centre(:,:)=0.
!$OMP END WORKSHARE
!$OMP DO
do j=2,ny-1
  do i=2,nx-1
    beta_centre(i,j) = ((betamx(i,j)+betamx(i+1,j)) &
          + (betamy(i,j)+betamy(i,j+1)))*0.25
  end do
end do
!$OMP END DO
!$OMP END PARALLEL


! les noeuds negatifs veulent dire qu'il faudrait ajouter un moteur pour aller aussi vite
! que les vitesses de bilan. 
! il faudrait repartir plutot que mettre a 0

! a mettre ailleurs
! betamx(:,:)=max(betamx(:,:),0.)
! betamy(:,:)=max(betamy(:,:),0.)
! beta_centre(:,:)=max(beta_centre(:,:),0.)

!  iter_beta=2


if (itracebug.eq.1)  call tracebug(' Fin calc_beta')
end subroutine calc_beta

end subroutine rempli_L2
