! =======================================================================
!
! Copyright Centre National de la Recherche Scientifique (CNRS)
!
! [Institut des Geosciences de l'Environnement (IGE) & Laboratoire des
! Sciences du climat et de l'Environnement (LSCE)]
!
! contributor(s): C. Ritz, A. Quiquet, C. Dumas, V. Peyaud, V. Rommelaere,
! A. Fabre, N. Lhomme (2018)
!
! contacts:
! C.R.: catherine.ritz@univ-grenoble-alpes.fr
! C.D.: christophe.dumas@lsce.ipsl.fr
! A.Q.: aurelien.quiquet@lsce.ipsl.fr
!
! This file: [eq_ellipt_sgbsv_mod-0.2.f90]
! is part of the GRISLI (GRenoble Ice-Shelf and Land-Ice model)
! software, a computer program whose purpose is to compute the evolution
! of ice thickness coupled to three-dimensional ice temperature and
! velocity fields given a mass-balance scenario.
!
! This software is governed by the CeCILL license under French law and
! abiding by the rules of distribution of free software.  You can  use,
! modify and/ or redistribute the software under the terms of the CeCILL
! license as circulated by CEA, CNRS and INRIA at the following URL
! "http://www.cecill.info".
!
! As a counterpart to the access to the source code and  rights to copy,
! modify and redistribute granted by the license, users are provided only
! with a limited warranty  and the software's author,  the holder of the
! economic rights,  and the successive licensors  have only  limited
! liability.
!
! In this respect, the user's attention is drawn to the risks associated
! with loading,  using,  modifying and/or developing or reproducing the
! software by the user in light of its specific status of free software,
! that may mean  that it is complicated to manipulate,  and  that  also
! therefore means  that it is reserved for developers  and  experienced
! professionals having in-depth computer knowledge. Users are therefore
! encouraged to load and test the software's suitability as regards their
! requirements in conditions enabling the security of their systems and/or
! data to be ensured and,  more generally, to use and operate it in the
! same conditions as regards security.
!
! The fact that you are presently reading this means that you have had
! knowledge of the CeCILL license and that you accept its terms.
!
! ======================================================================


module eq_ellip_sgbsv_mod

!   module pour la resolution des equations elliptiques
!   de l'ice shelf et ice stream en utilisant une resolution
!   de systeme bande  sgbsv (lapack)

!   voir le rappel des argument de SGBSV en fin de fichier

! declaration des variables
!$ USE OMP_LIB
use remplimat_declar   ! toutes les routines de ce module connaissent les Tuij....


implicit none

integer,parameter :: klmax=2*nlmin           ! largeur max demi-bande inferieure
integer,parameter :: kumax=2*nlmin           ! largeur max demi-bande superieure
integer,parameter :: ldbmx=2*klmax+kumax+1   ! largeur de bande maximum (taille de la matrice) >= 2kl+ku+1

integer :: lbande        ! pour appel sgbsv (largeur de bande) = kl+ku+1 (ne pas confondre avec ldbmx)
integer :: ldb           ! pour appel sgbsv (largeur de bande+ travail) = 2*kl+ku+1 
integer :: ldb_old       ! pour garder la valeur du tour precedent (allocation de mmat)
integer :: nrhs = 1      ! en gnral 1


integer :: ku            ! largeur demi-bande superieure effective
integer :: kl            ! largeur demi-bande inferieure effective
integer :: nblig         ! nombre d'equations traitees effectivement
integer :: nblig_old     ! pour garder la valeur du tour precedent (allocation de mmat) 

integer :: ldiag         ! compteur du numero de ligne
 
integer :: iii,jjj       ! position du noeud de l'echec de sgbsv 
integer :: ifail_sgbsv   ! recuperation d'erreur



! tableaux pour passer a sgbsv

real,dimension(nflmax,1)         :: bdr       ! vecteur membre de droite de l'equation
integer,dimension(nflmax)        :: ipiv      ! pivot


! seul mmat est allocatable parce que la premiere dimension est variable

real,dimension(:,:), allocatable ::  mmat     ! AB, matrice appelee par sgbsv contenant les diagonales
                                              ! sous forme "couchees" sera dimensionnee (ldab,nblig)

! tableaux de travail
integer,dimension(nflmax) :: lu_band          ! largeur de bande 
integer,dimension(nflmax) :: ll_band          ! largeur de bande inferieure 



contains


!----------------------------
subroutine  redim_matrice

! allocation dynamique de mmat

if (allocated(mmat)) then 
   deallocate(mmat,stat=err)
     if (err/=0) then
        print *,"Erreur  la desallocation de mmat",err
        stop 
     end if
end if


allocate(mmat(ldb,nblig),stat=err)
if (err/=0) then
   print *,"erreur a l'allocation du tableau mmat",err
   print *,"ldb,nblig",ldb,nblig
   stop 
end if


!write(6,*) "allocation mmat ok, ldb,nblig",ldb,nblig

end subroutine redim_matrice

!_________________________________________________________________
subroutine initial_matrice   ! pour compatibilites vieilles version
end subroutine initial_matrice

! -------------------------------------------------------------------------------

subroutine resol_ellipt(nx1,nx2,ny1,ny2,uxprec,uyprec,uxnew,uynew,ifail)

!  prepare la matrice mmat (version "couchee" de L2)
!  resoud l'equation elliptique par un appel a sgbsv
!  renvoie les nouvelles vitesses
!
!                                                                   
!      uxprex(n1,n2)                                                 
!      uyprec(n1,n2) vitesses de l'iteration precedente              
!                                                                    
!      uxnew(n1,n2)                                                   
!      uynew(n1,n2) uynew resultat de cette iteration 
!
!---------------------------------------------------------------------------

integer,intent(in) :: nx1             ! bornes du domaine en x (noeuds majeurs)
integer,intent(in) :: nx2
integer,intent(in) :: ny1             ! bornes du domaine en y (noeuds majoeurs)
integer,intent(in) :: ny2

integer :: ifail                      ! pour les rapports d'erreur

real, dimension(nx1:nx2,ny1:ny2),intent(in)  :: uxprec    ! vitesse en entree routine
real, dimension(nx1:nx2,ny1:ny2),intent(in)  :: uyprec    ! vitesse en entree routine

real, dimension(nx1:nx2,ny1:ny2),intent(out) :: uxnew     ! vitesse en sortie de la routine
real, dimension(nx1:nx2,ny1:ny2),intent(out) :: uynew     ! vitesse en sortie de la routine

nrhs = 1
nblig_old=0
ldb_old=0

call prepare_mat(nx1,nx2,ny1,ny2)         ! prepare la matrice pour appel a SGBSV

! quelques outils de debug ci-dessous
!---------------------------------------

! write(6,'(a20,4(i0,2x))') 'kl, ku ,ldb, nblig   ',kl, ku ,ldb, nblig 
!---------------------------------------------------------

! routine qui permet de visualiser les mmatrices L2 et mmat
! call graphique_L2(kl,ku,nx1,nx2,ny1,ny2,imx(nx1:nx2,ny1:ny2),imy(nx1:nx2,ny1:ny2))   
!---------------------------------------------------------

! routine qui permet de tester l'appel sgbsv et de visualiser la matrice
! call graph_sgbsv(nblig,kl,ku,nrhs,mmat,ldb,ipiv,bdr,nblig,ifail_sgbsv)
!---------------------------------------------------------



! appel a la resolution du systeme (voir doc a la fin du fichier)


if (nblig.ne.0) then
   call sgbsv(nblig,kl,ku,nrhs,mmat,ldb,ipiv,bdr,nblig,ifail_sgbsv)

   if (ifail_sgbsv.eq.0) then                ! la resolution s'est bien passee
      !   write(6,*)'-----------------------'
      !   write(6,*) 'sortie ok de sgbsv'
      !   write(6,*)'-----------------------'
      !$OMP PARALLEL
      !$OMP DO
      do j = ny1,ny2
         do i = nx1,nx2

            if (ok_umat(i,j)) then 
               uxnew(i,j) = bdr(ligu_L2(i,j),1)
            else
               uxnew(i,j) = uxprec(i,j)
            end if

            if (ok_vmat(i,j)) then 
               uynew(i,j) = bdr(ligv_L2(i,j),1)
            else
               uynew(i,j) = uyprec(i,j)
            end if
         end do
      end do
      !$OMP END DO
      !$OMP END PARALLEL

   else

      call erreur_sgbsv(ifail_sgbsv)
      ifail=ifail_sgbsv
   end if

else                 ! nblig=0, pas de passage par sgbsv
   !$OMP PARALLEL
   !$OMP WORKSHARE
   uxnew(nx1:nx2,ny1:ny2)=uxprec(nx1:nx2,ny1:ny2)
   uynew(nx1:nx2,ny1:ny2)=uyprec(nx1:nx2,ny1:ny2)
   !$OMP END WORKSHARE
   !$OMP END PARALLEL
end if

debug_3D(nx1:nx2,ny1:ny2,37)=uxnew(nx1:nx2,ny1:ny2)
debug_3D(nx1:nx2,ny1:ny2,38)=uynew(nx1:nx2,ny1:ny2)

return
end subroutine resol_ellipt

!--------------------------------------------------------------------------------

subroutine prepare_mat(nx1,nx2,ny1,ny2)         ! prepare la matrice pour appel a SGBSV

integer :: nx1,nx2        ! bornes du domaine traite
integer :: ny1,ny2        ! bornes du domaine traite


integer :: ligne          ! dans L2
integer :: colonne        ! dans L2
integer :: ll             ! dans le calcul de la largeur de bande

! tableau de travail pour calculer la largeur de bande
integer,dimension(20)     :: pos


! numerotation des lignes 
!--------------------------
!$OMP PARALLEL
!$OMP WORKSHARE
ligu_L2(:,:) = -9999
ligv_L2(:,:) = -9999
pos_ligu(:,:) = -9999
pos_ligv(:,:) = -9999
!$OMP END WORKSHARE
!$OMP END PARALLEL
! Boucle sur les noeuds avec la dimension la plus grande a l'interieur

count_line = 1                                 ! pour numeroter les lignes

count: do j = ny1,ny2
   do i = nx1,nx2

      if (ok_umat(i,j)) then
         ligu_L2(i,j) = count_line             ! ligne de U(i,j) dans L2

         pos_ligu(count_line,1) = i            ! i,j d'un noeud U de ligne count_line
         pos_ligu(count_line,2) = j

         count_line = count_line+1


                                        
      end if
 
      if (ok_vmat(i,j)) then
         ligv_L2(i,j) = count_line             ! ligne de V(i,j) dans L2

         pos_ligv(count_line,1) = i            ! i,j d'un noeud V de ligne count_line
         pos_ligv(count_line,2) = j

         count_line = count_line+1

      end if


   end do
end do count


nblig = count_line-1


! calcul de la largeur de bande. depend de la numerotation des lignes
!--------------------------------------------------------------------
!$OMP PARALLEL
!$OMP WORKSHARE
lu_band(:) = 0
ll_band(:) = 0
!$OMP END WORKSHARE
!$OMP END PARALLEL

larg_band: do j = ny1,ny2
   do i = nx1,nx2

      lband_U: if (ok_umat(i,j)) then         ! pour chaque ligne uij, on rempli pos avec les numeros de colonne

!     ilmin et jlmin boucle U
      ilmin = max(-1,nx1+1-i)    ! pour avoir i+il entre nx1+1 et nx2
      ilmax = min(1,nx2-i)

      jlmin = max(-1,ny1-j)      ! pour avoir j+jl entre ny1 et ny2
      jlmax = min(1,ny2-j)  

   
         ll = 1 
         ldiag = ligu_L2(i,j)            ! numero colonne de la diagonale  
         pos(:) = ldiag              

         do jl = jlmin , jlmax      ! balaye tous les coefficients de la colonne Uij de -1 a 1
            do il = ilmin , ilmax   !                                                de -1 a 1 

               if (Tu(i,j,il,jl).ne.0.) then 
                  pos(ll) = ligu_L2(i+il,j+jl)
                  ll = ll+1
               end if
               
               if (Tv(i,j,il,jl).ne.0.) then
                  pos(ll) = ligv_L2(i+il,j+jl)
                  ll = ll+1
               end if
            end do
         end do


         pos(:) = pos(:)-ldiag    ! on soustrait le numero de colonne de celui de la diagonale
         ll_band(ldiag) = -minval(pos)
         lu_band(ldiag) = maxval(pos)

      end if lband_U

      lband_V: if (ok_vmat(i,j)) then         ! pour chaque ligne uij, on rempli pos avec les numeros de colonne

!     ilmin et jlmin boucle V
      ilmin = max(-1,nx1-i)    ! pour avoir i+il entre nx1 et nx2
      ilmax = min(1,nx2-i)

      jlmin = max(-1,ny1+1-j)    ! pour avoir j+jl entre ny1+1 et ny2
      jlmax = min(1,ny2-j)     

       
         ll = 1 
         ldiag = ligv_L2(i,j)            ! numero colonne de la diagonale  
         pos(:) = ldiag              

         do jl = jlmin , jlmax      ! balaye tous les coefficients de la colonne Uij de -1 a 1
            do il = ilmin , ilmax   !                                                de -1 a 1 

               if (Su(i,j,il,jl).ne.0.) then 
                  pos(ll) = ligu_L2(i+il,j+jl)
                  ll = ll+1
               end if
               
               if (Sv(i,j,il,jl).ne.0) then
                  pos(ll) = ligv_L2(i+il,j+jl)
                  ll = ll+1
               end if
            end do
         end do


         pos(:) = pos(:)-ldiag    ! on soustrait le numero colonne de la diagonale
         ll_band(ldiag) = -minval(pos)
         lu_band(ldiag) = maxval(pos)

      end if lband_V
   end do
end do larg_band

ku = maxval(lu_band)        ! largeur de bande sur-diagonale
kl = maxval(ll_band)        ! largeur de bande sous-diagonale

lbande = kl+ku+1            ! largeur de bande
ldb    = 2*kl+ku+1          ! pour le dimensionnement de mmat

! eventuellement pour imposer les valeurs de kl et ku
!kl = klmax
!ku = kumax

lbande = kl+ku+1            ! largeur de bande
ldb    = 2*kl+ku+1          ! pour le dimensionnement de mmat


! ecriture de la largeur de bande pour en faire eventuellement le trace
!----------------------------------------------------------------------
!open(124,file = 'largeur-bande')

!do l = 1,nblig
!   write(124,*) l,ll_band(l),lu_band(l)
!end do

!close(124)

! si la matrice mmat  change de taille on la re allocate.

if ((ldb.ne.ldb_old).and.(nblig.ne.nblig_old)) call redim_matrice


ldb_old=ldb                 ! pour garder en memoire les anciennes valeurs 
nblig_old=nblig

! ecriture de la matrice proprement dite et du vecteur bdr
!------------------------------------------------------------
!$OMP PARALLEL
!$OMP WORKSHARE
mmat(:,:) = 0.
bdr(:,:) = 0.
!$OMP END WORKSHARE

!$OMP DO PRIVATE(ligne,ilmin,ilmax,jlmin,jlmax,colonne)
ij_loop : do j = ny1,ny2         ! attention, on rempli 2 lignes par 2 lignes (u,v)
   do i = nx1,nx2

u_line: if (ok_umat(i,j)) then

   ligne = ligu_L2(i,j)

   if (ligne.lt.0) then          ! teste certains problemes de numeros de ligne
      write(6,*) 'U_line,i,j',i,j,ligne
   end if

   bdr(ligne,1) = opposx(i,j)


! boucle U : les definitions de ilmin et jlmin sont differentes dans les boucles u et v

      ilmin = max(-1,nx1+1-i)    ! pour avoir i+il entre nx1+1 et nx2
      ilmax = min(1,nx2-i)

      jlmin = max(-1,ny1-j)      ! pour avoir j+jl entre ny1 et ny2
      jlmax = min(1,ny2-j)



iljl_lineU:      do jl = jlmin , jlmax      ! balaye tous les coefficients d'un noeud U de -1 a 1
         do il = ilmin , ilmax              !                                           de -1 a 1 


! ecriture des Tu
               colonne = ligu_L2(i+il,j+jl)

               if (abs(Tu(i,j,il,jl)).gt.eps_col) then

                  if (colonne.lt.0) then    ! trace si un noeud ghost est appele par erreur
                     write(6,*) 'U_line,i,j,il,jl,Tu',i,j,il,jl,Tu(i,j,il,jl),ghost_x(i+il,j+jl),ghost_y(i+il,j+jl)
                  end if


                  mmat(lbande+ligne-colonne,colonne) = Tu(i,j,il,jl)        ! ecriture des Tu

               endif

! ecriture des Tv
               colonne = ligv_L2(i+il,j+jl)

               if (abs(Tv(i,j,il,jl)).gt.eps_col) then    

                  if (colonne.lt.0) then  ! trace si un noeud ghost est appele par erreur
                     write(6,*) 'U_line,i,j,il,jl,Tv',i,j,il,jl,Tv(i,j,il,jl),ghost_x(i+il,j+jl),ghost_y(i+il,j+jl)
                  end if
   
                  mmat(lbande+ligne-colonne,colonne) = Tv(i,j,il,jl)       ! ecriture des Tv

               end if

            end do
         end do iljl_lineU

      end if u_line

v_line:  if (ok_vmat(i,j)) then

   ligne = ligv_L2(i,j)
   bdr(ligne,1) = opposy(i,j)


! boucle V : les definitions de ilmin et jlmin sont differentes dans les boucles u et v

      ilmin = max(-1,nx1-i)        ! pour avoir i+il entre nx1 et nx2
      ilmax = min(1,nx2-i)

      jlmin = max(-1,ny1+1-j)      ! pour avoir j+jl entre ny1+1 et ny2
      jlmax = min(1,ny2-j)


iljl_lineV:      do jl = jlmin , jlmax      ! balaye tous les coefficients d'un noeud U de -1 a 1
         do il = ilmin , ilmax              !                                           de -1 a 1 

! ecriture des Su    
               colonne = ligu_L2(i+il,j+jl)

               if (abs(Su(i,j,il,jl)).gt.eps_col) then

                  if (colonne.lt.0) then  ! trace si un noeud ghost est appele par erreur
                     write(6,*) 'V_line,i,j,il,jl,Su',i,j,il,jl,Su(i,j,il,jl),ghost_x(i+il,j+jl),ghost_y(i+il,j+jl)
                  end if

                  mmat(lbande+ligne-colonne,colonne) = Su(i,j,il,jl)      ! ecriture des Su
              
               end if

! ecriture des Sv
               colonne = ligv_L2(i+il,j+jl)

               if (abs(Sv(i,j,il,jl)).gt.eps_col) then

                  if (colonne.lt.0) then  ! trace si un noeud ghost est appele par erreur
                     write(6,*) 'V_line,i,j,il,jl,Sv',i,j,il,jl,Sv(i,j,il,jl),ghost_x(i+il,j+jl),ghost_y(i+il,j+jl)
                  end if


                  mmat(lbande+ligne-colonne,colonne) = Sv(i,j,il,jl)      ! ecriture des Sv

               end if

            end do
      end do iljl_lineV
   end if v_line
 end do
end do ij_loop
!$OMP END DO
!$OMP END PARALLEL


return
end subroutine prepare_mat

!-----------------------------------------------------------------------------
subroutine erreur_sgbsv(ifail_sgbsv)

! donne quelques infos en cas d'erreur sgbsv

integer  :: ifail_sgbsv
integer  :: li,lj      ! coordonnees du point a probleme

if (ifail_sgbsv.gt.0) then

   write(*,*) ' ---------------------------------------'
   write(*,*) '  ERREUR DANS L''ELIMINATION GAUSSIENNE '
   write(*,*) '  ERREUR Numero',ifail_sgbsv
   write(*,*) 
   write(*,*) ' vecteurs colineaires                   '
   write(*,*) ' diagonale nulle dans decomposition LU  '

! recuperation du noeud de l'erreur

   if (pos_ligu(ifail_sgbsv,1).gt.0) then                              ! pb sur une vitesse U
      write(*,*) ' vitesse U        i =', pos_ligu(ifail_sgbsv,1), &
                       '            j =', pos_ligu(ifail_sgbsv,2)

 
      li=pos_ligu(ifail_sgbsv,1)
      lj=pos_ligu(ifail_sgbsv,2)

! balaye les noeuds autour  pb avec le format ci-dessous
997 format(3(i0,2x),'a4',2x,es12.2,2x,L3)

      do j=max(lj-1,1),min(lj+1,ny)
         do i=max(li-1,1),min(li+1,ny)

            write(6,*)
            write(6,*)'Tu, Tv, i,j,' ,i,j
            write(6,*)'--------------------'

               do jl = -1 , 1      ! balaye tous les coefficients de la colonne Uij de -1 a 1
                  do il = -1 , 1   !                                                de -1 a 1 
 
                     if ((Tu(i,j,il,jl)).ne.0.) then
                        write(6,997) il,jl,ligu_L2(i+il,j+jl),'Tu=',Tu(i,j,il,jl), ok_umat(i+il,j+jl)
                     end if
                     if ((Tv(i,j,il,jl)).ne.0.) then
                        write(6,997) il,jl,ligv_L2(i+il,j+jl),'Tv=',Tv(i,j,il,jl), ok_vmat(i+il,j+jl)
                     end if
                  end do
               end do

            end do
         end do

   else
      write(*,*) ' vitesse V        i =', pos_ligv(ifail_sgbsv,1), &
                       '            j =', pos_ligv(ifail_sgbsv,2)


      li=pos_ligv(ifail_sgbsv,1)
      lj=pos_ligv(ifail_sgbsv,2)

      do j=max(lj-1,1),min(lj+1,ny)
         do i=max(li-1,1),min(li+1,ny)
            
            write(6,*)
            write(6,*)'Su, Sv, i,j,' ,i,j

               do jl = -1 , 1       ! balaye tous les coefficients de la colonne Uij de -1 a 1
                  do il = -1 , 1    !                                                de -1 a 1 
                     if ((Su(i,j,il,jl)).ne.0.) then
                       write(6,*) il,jl,ligu_L2(i+il,j+jl),'Su=',Su(i,j,il,jl), ok_umat(i+il,j+jl)
!                        write(6,997) il,jl,ligu_L2(i+il,j+jl),'Su=',Su(i,j,il,jl), ok_umat(i+il,j+jl)

                     end if
                     if ((Sv(i,j,il,jl)).ne.0.) then
                       write(6,*) il,jl,ligv_L2(i+il,j+jl),'Sv=',Sv(i,j,il,jl), ok_vmat(i+il,j+jl)
!                        write(6,997) il,jl,ligv_L2(i+il,j+jl),'Sv=',Sv(i,j,il,jl), ok_vmat(i+il,j+jl)
                     end if

                  end do
               end do
            end do
         end do

   endif

   write(*,*) 
   write(*,*) ' ---------------------------------------'
  

else if (ifail_sgbsv.lt.0) then

   write(*,*) ' ---------------------------------------'
   write(*,*) '  ERREUR DANS L''ELIMINATION GAUSSIENNE '
   write(*,*) '  ERREUR Numero',ifail_sgbsv
   write(*,*) 
   write(*,*) ' argument a une valeur illegale         '
   write(*,*) 
   write(*,*) ' ---------------------------------------'
   
      
end if
end subroutine erreur_sgbsv
!-----------------------------------------------------------------------------


end module eq_ellip_sgbsv_mod

!-----------------------------------------------------------------------------
!
! rappel arguments sgbsv
!
!      SUBROUTINE SGBSV( N, KL, KU, NRHS, AB, LDAB, IPIV, B, LDB, INFO )
!
!  Arguments
!  =========
!
!  N       (input) INTEGER
!          The number of linear equations, i.e., the order of the
!          matrix A.  N >= 0.
!
!  KL      (input) INTEGER
!          The number of subdiagonals within the band of A.  KL >= 0.
!
!  KU      (input) INTEGER
!          The number of superdiagonals within the band of A.  KU >= 0.
!
!  NRHS    (input) INTEGER
!          The number of right hand sides, i.e., the number of columns
!          of the matrix B.  NRHS >= 0.
!
!  AB      (input/output) REAL array, dimension (LDAB,N)
!          On entry, the matrix A in band storage, in rows KL+1 to
!          2*KL+KU+1; rows 1 to KL of the array need not be set.
!          The j-th column of A is stored in the j-th column of the
!          array AB as follows:
!          AB(KL+KU+1+i-j,j) = A(i,j) for max(1,j-KU)<=i<=min(N,j+KL)
!          On exit, details of the factorization: U is stored as an
!          upper triangular band matrix with KL+KU superdiagonals in
!          rows 1 to KL+KU+1, and the multipliers used during the
!          factorization are stored in rows KL+KU+2 to 2*KL+KU+1.
!          See below for further details.
!
!  LDAB    (input) INTEGER
!          The leading dimension of the array AB.  LDAB >= 2*KL+KU+1.
!
!  IPIV    (output) INTEGER array, dimension (N)
!          The pivot indices that define the permutation matrix P;
!          row i of the matrix was interchanged with row IPIV(i).
!
!  B       (input/output) REAL array, dimension (LDB,NRHS)
!          On entry, the N-by-NRHS right hand side matrix B.
!          On exit, if INFO = 0, the N-by-NRHS solution matrix X.
!
!  LDB     (input) INTEGER
!          The leading dimension of the array B.  LDB >= max(1,N).
!
!  INFO    (output) INTEGER
!          = 0:  successful exit
!          < 0:  if INFO = -i, the i-th argument had an illegal value
!          > 0:  if INFO = i, U(i,i) is exactly zero.  The factorization
!                has been completed, but the factor U is exactly
!                singular, and the solution has not been computed.
!-----------------------------------------------------------------------------
