%% General Settings %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

clc;                % Refresh Command Window
clear;              % clear Workspace
close all;          % close all Figures
format long g       % 14 decimal places
addpath(genpath('../Functions')) % load Functions
% time units (in seconds)
sec = 1; minute = 60*sec; hour = 60*minute; day = 24*hour; year = 365*day;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('-------------------------------------------------')
disp('Prepare forcing data from DICE for JSBACH schemes')
disp('-------------------------------------------------')
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%



%% Specification Part (by User)


% start/end date initialization/real run (in JSBACH forcing files have to
% be entire days mandatory)
startdate_ini = '19900101';
enddate_ini   = '19991024';
startdate_run = '19991023';
enddate_run   = '19991027';

% original timestep's length in minutes
dt_ini_orig = 180 * minute;
dt_run_orig = 10 * minute;

% new timestep's length in minutes
dt_ini = 60 * minute;
dt_run = 10 * minute;

% List of variables to read
var_str = {'Tair','Wind','Rainf','SWdown','LWdown','Qair'}; 

% minimal wind speed (wind speed ought not drop on zero in numeric models)
eps_wind = 0.01;

% latitude/longitude of experiment site

lat_exp = 37.6500;
lon_exp = 263.2650;

% Output folder
output_folder = 'Output';

% experiment ID
expid_ini = 'dice.10years';
expid_run = 'dice.3days';

% read data
read_data = true;

%% Specific Settings

% load forcing data
if(read_data)
    suffix = 'spinup';
    % NC-file einlesen
    filename = './Input/DICE_LSM_spinup.nc';
    output   = './Input/spinup.mat';
    Text     = ['Read ', filename(9:end),'...'];
    disp(' ');disp(Text)
    read_routine  % run subroutine
    
    suffix = 'screen';
    % NC-file einlesen
    filename = './Input/DICE_LSM_screen.nc';
    output   = './Input/screen.mat';
    Text     = ['Read ', filename(9:end),'...'];
    disp(' ');disp(Text)
    read_routine  % run subroutine

else
    load('./Input/spinup.mat')
    load('./Input/screen.mat')
end

data_ini = [Tair_spinup',Wind_spinup',Rainf_spinup',SWdown_spinup',LWdown_spinup',Qair_spinup'];
data_run = [Tair_screen,Wind_screen,Rainf_screen,SWdown_screen,LWdown_screen,Qair_screen];
% data_run contains one too many timesteps
data_run = data_run(1:end-1,:);

% number of gridpoint in x-/y-direction
nx = 1;                 
ny = 2; 

% number of timesteps
nt_ini = (datenum(enddate_ini,'yyyymmdd') - datenum(startdate_ini,'yyyymmdd')) * day/dt_ini;
nt_run = (datenum(enddate_run,'yyyymmdd') - datenum(startdate_run,'yyyymmdd')) * day/dt_run;

% output filename
ncfilename_ini = strcat('./',output_folder,'/forcing.',expid_ini,'.',num2str(dt_ini/minute),'min.nc');
ncfilename_run = strcat('./',output_folder,'/forcing.',expid_run,'.',num2str(dt_run/minute),'min.nc');



%% Change Timestep by Interpolation or Averaging
disp(' ');
disp('Interpolation/Averaging for Initial Run...')
data_ini = changedt(dt_ini,dt_ini_orig,data_ini);
disp(' ');
disp('Interpolation/Averaging for Initial Run...')
data_run = changedt(dt_run,dt_run_orig,data_run);


%% Create entire days for real run
% number of entire days
ndays  = 4;
indata = nan(ndays*day/dt_run,size(data_run,2));
% hour of day data begins
hod = 19;
ii = hod*hour/dt_run;
indata(1:ii,:) = repmat(data_run(1,:),ii,1);
indata(ii+1:ii+length(data_run),:) = data_run;
indata(ii+length(data_run)+1:end,:) = repmat(data_run(end,:),length(indata)-(ii+length(data_run)+1)+1,1);
data_run = indata;

%% Physical contraints
time_ini=(1:nt_ini)*dt_ini/day;
time_run=(1:nt_run)*dt_run/day;

for nv=1:length(var_str)
    figure
    y=data_ini(:,nv);
    plot(time_ini,y)
    hold on
    title(var_str{nv})
end
for nv=1:length(var_str)
    figure
    y=data_run(:,nv);
    plot(time_run,y)
    hold on
    title(var_str{nv})
end


%% NETCDF Output

dt = dt_ini;
nt = nt_ini;
indata = data_ini;
startdate = startdate_ini;
ncfilename = ncfilename_ini;

create_netcdf

dt = dt_run;
nt = nt_run;
indata = data_run;
startdate  = startdate_run;
ncfilename = ncfilename_run;

create_netcdf

disp(' ');
disp('Forcing data successfully created.');




