%PLOT_SW_VEGETATION
%  Script to plot the comparison of the SPARTACUS SW radiation scheme
%  with reference Monte Carlo calculations for the RAMI4PILPS
%  intercomparison cases. This script is normally called from the
%  "do_all_plots" script.

% Do we do the near infrared?
if ~exist('do_nir','var')
  do_nir = 0; % default VIS
end

% Select scene: Shrubland='SHR', Open forest canopy='OFC'
if ~exist('scene','var')
  scene = 'OFC';
end

% Do we just do selected surface albedos
if ~exist('do_selected','var')
  do_selected = 0;
end

% Do we include calculations with different numbers of regions?
if ~exist('do_extras','var')
  do_extras = 0;
end

% Do we do fast calculations for the vegetation-free sub-canopy layer?
if ~exist('do_fast','var')
  do_fast = 0;
end

% Specify properties that depend on spectral region
if do_nir
  disp('Near infrared')
  spec = 'NIR';
  leaf_ref = 0.3912;
  leaf_trans = 0.4146;
  albedos = [0.2142 0.5568];
else
  disp('Visible');
  spec = 'VIS';
  leaf_ref = 0.0735;
  leaf_trans = 0.0566;
  albedos = [0.1217 0.9640];
end

% We don't do the black surface case from RAMI4PILPS, so when loading
% the reference Monte Carlo data, we need to offset the albedo index
% by 1 to skip over this case (see plot_ref.m)
alb_offset = 1;

% Run SPARTACUS every degree of solar zenith angle
sza = [0:89 89.5 89.9];

% Number of sun angles
nsza = length(sza);

% If all leaf-area-index values are used we don't need an offset when
% loading the reference Monte Carlo data (see plot_ref.m)
lai_offset = 0;

% Specify properties that depend on the scene
if strcmp(scene,'OFC')
  disp('Open forest canopy')
  % Open forest canopy: trees of diameter 10 m
  sphere_radius = 5;  % m
  sphere_lai = 5;
  base_z = 4; % Height of tree-canopy base (m)
  spheres_per_hectare_list = [12.8 38.2400 63.6800];
  if do_selected
    % Only one selected vegetation density
    spheres_per_hectare_list = [38.2400];
    lai_offset = 1;
  end
else
  disp('Shrubland')
  % Shrubland
  spheres_per_hectare_list = [1274 2547 5093];
  sphere_radius = 0.5; % m
  sphere_lai = 2.5;
  base_z = 0.01; % Height of shrub base (m)
end

% Number of rows/columns of sub-plots
nalb = length(albedos);
nlai = length(spheres_per_hectare_list);

% Reference (Monte Carlo) solar zenith angles
szas_mc = plot_ref;

% Plot the data
clf
set(gcf,'units','inches','paperposition',[0.5 0.5 25 16]);
if ~do_selected
  set(gcf,'defaultlinelinewidth',1,'defaultaxesfontsize',10);
else
  set(gcf,'defaultlinelinewidth',1,'defaultaxesfontsize',8);
end

% Loop over leaf-area-index (equivalent to vegetation cover) and
% surface albedo
for ilai = 1:nlai
  spheres_per_hectare = spheres_per_hectare_list(ilai);
  for ialb = 1:nalb
    albedo = albedos(ialb);

    % Zero the arrays to store the results for every sun angle
    ref   = zeros(1,nsza);
    sw_dn = ref;
    sw_dn_direct = ref;
    absor = ref;

    for ii = 1:nsza
      [ref(ii),sw_dn(ii),sw_dn_direct(ii),absor(ii),sp,out] = calc_sw_vegetation(sza(ii),albedo,sphere_lai,leaf_ref,leaf_trans,...
										 sphere_radius,spheres_per_hectare,base_z,3,do_fast);
    end

    % Orientation of the subplots depends on whether we are just looking at selected case
    if ~do_selected
      subplot(2,3,(ialb-1).*3+ilai)
    else
      subplot(2,3,(ilai-1).*3+ialb)
    end

    % Plot some dummy lines so that the legend makes sense in the extras case
    if do_extras & ialb == 1 & ilai == 1
       plot(-1,-1,'r');
       hold on
       plot(-1,-1,'g');
       plot(-1,-1,'b');
       plot(-1,-1,'k-.','linewidth',0.5);
       plot(-1,-1,'k--','linewidth',0.5);
       plot(-1,-1,'k-');
       plot(-1,-1,'ko');
    end

    % Plot SPARTACUS reflectance, absorptance and surface downwelling (transmittance)
    plot(sza,ref,'r');
    hold on
%    plot(sza,sw_dn_direct,'b--');
    plot(sza,absor,'g');
    plot(sza,sw_dn,'b');

    % Save values at solar zenith angles where Monte Carlo
    % calculations have been performed
    if ~exist('spartacus_save','var')
      spartacus_save.ref = [];
      spartacus_save.absor = [];
      spartacus_save.sw_dn = [];
    end
    spartacus_save.ref(end+1:end+length(szas_mc)) = interp1(sza,ref,szas_mc);
    spartacus_save.absor(end+1:end+length(szas_mc)) = interp1(sza,absor,szas_mc);
    spartacus_save.sw_dn(end+1:end+length(szas_mc)) = interp1(sza,sw_dn,szas_mc);

    if do_extras
      % Also plot the 1-region case (Sellers-like) 
      for ii = 1:nsza
	[ref(ii),sw_dn(ii),sw_dn_direct(ii),absor(ii),sp,out] = calc_sw_vegetation(sza(ii),albedo,sphere_lai,leaf_ref,leaf_trans,...
										   sphere_radius,spheres_per_hectare,base_z,1);
      end

      plot(sza,ref,'r-.','linewidth',0.5);
      hold on
      plot(sza,absor,'g-.','linewidth',0.5);
      plot(sza,sw_dn,'b-.','linewidth',0.5);

      % Save values at solar zenith angles where Monte Carlo
      % calculations have been performed
      if ~exist('homog_save','var')
	homog_save.ref = [];
	homog_save.absor = [];
	homog_save.sw_dn = [];
      end
      homog_save.ref(end+1:end+length(szas_mc)) = interp1(sza,ref,szas_mc);
      homog_save.absor(end+1:end+length(szas_mc)) = interp1(sza,absor,szas_mc);
      homog_save.sw_dn(end+1:end+length(szas_mc)) = interp1(sza,sw_dn,szas_mc);

      % ...and the 2-region case
      for ii = 1:nsza
	[ref(ii),sw_dn(ii),sw_dn_direct(ii),absor(ii),sp,out] = calc_sw_vegetation(sza(ii),albedo,sphere_lai,leaf_ref,leaf_trans,...
										   sphere_radius,spheres_per_hectare,base_z,2,do_fast);
      end

      plot(sza,ref,'r--','linewidth',0.5);
      hold on
      plot(sza,absor,'g--','linewidth',0.5);
      plot(sza,sw_dn,'b--','linewidth',0.5);

      % Save values at solar zenith angles where Monte Carlo
      % calculations have been performed
      if ~exist('spartacus2_save','var')
	spartacus2_save.ref = [];
	spartacus2_save.absor = [];
	spartacus2_save.sw_dn = [];
      end
      spartacus2_save.ref(end+1:end+length(szas_mc)) = interp1(sza,ref,szas_mc);
      spartacus2_save.absor(end+1:end+length(szas_mc)) = interp1(sza,absor,szas_mc);
      spartacus2_save.sw_dn(end+1:end+length(szas_mc)) = interp1(sza,sw_dn,szas_mc);

    end

    plot(-1,-1,'ko');
    % Plot Monte Carlo results
    [sza_mc,sw_dn_mc,absor_mc,ref_mc] = plot_ref(ilai+lai_offset,ialb+alb_offset,spec,scene);

    % Store Monte Carlo results
    if ~exist('mc_save','var')
      mc_save.ref = [];
      mc_save.absor = [];
      mc_save.sw_dn = [];
    end
    mc_save.ref(end+1:end+length(szas_mc)) = ref_mc;
    mc_save.absor(end+1:end+length(szas_mc)) = absor_mc;
    mc_save.sw_dn(end+1:end+length(szas_mc)) = sw_dn_mc;

    axis([0 90 0 1]);
    xlabel('Solar zenith angle (\circ)')
    ylabel('Normalized flux');
    set(gca,'xtick',[0:30:90]);
    grid off
    if ilai==1 & ialb==1
      if do_extras
	h=legend('Reflectance','Absorptance','Transmittance','Homogeneous','SPARTACUS-2','SPARTACUS-3',...
		 'Monte Carlo','location','west');
	legend boxoff
	set(h,'fontsize',10)
      else
	h=legend('SPARTACUS reflectance','SPARTACUS absorptance','SPARTACUS transmittance',...
		 'Monte Carlo','location','west');
	legend boxoff
	set(h,'fontsize',10)

      end
    end
    myfrac = sp.frac;

    title(['\bf(' 'a'-1+(ialb-1).*nlai+ilai ') ' spec ', \alpha=', ...
	   num2str(albedo) ', c_v=' num2str(myfrac,1)]);
  end
end
