function M = fast_expm(in)
%FAST_EXPM  Compute matrix exponential for 2x2 and 3x3 matrices of a specific form
%  M = fast_expm(in)
%  If "in" is a vector of length 2 then the matrix to be exponentiated is
%    { -in(1)   in(2) }
%    {  in(1)  -in(2) }
%  If "in" is a vector of length 3 then the matrix to be exponentiated is
%    { -in(1)   in(2)           0   }
%    {  in(1)  -in(2)-in(3)   in(3) }
%    {    0           in(3)  -in(3) }
%
%  Author: Robin Hogan <r.j.hogan@ecmwf.int>
%  Copyright: 2017 European Centre for Medium Range Weather Forecasts
%
%  Copying and distribution of this file, with or without modification,
%  are permitted in any medium without royalty provided the copyright
%  notice and this notice are preserved.

if sum(double(in==0)) > 0
  % Exponential of a matrix containing zeros is just the identity matrix
  M = eye(length(in));

elseif length(in) == 2
  % Putzer's algorithm
  a = in(1);
  b = in(2);
  Gamma = [-a  b;a -b];
  M = eye(2) + (1 - exp(-(a+b)))./(a+b) .* Gamma;

elseif length(in) == 3
  % Diagonalization method
  a = in(1);
  b = in(2);
  c = in(3);
  gamma1 = -(a+b+2*c)/2 + sqrt(a*a+b*b+4*c*c+2*a*b-4*a*c)/2;
  gamma2 = -(a+b+2*c)/2 - sqrt(a*a+b*b+4*c*c+2*a*b-4*a*c)/2;
  V = [b/(a+gamma1)  b/(a+gamma2)  b/a; ...
       1             1             1; ...
       c/(c+gamma1)  c/(c+gamma2)  1];
%  cond(V)
  M = V * diag([exp(gamma1) exp(gamma2) 1]) / V;
else
  error(['fast_expm not defined for input of length ' num2str(length(in))]);

end
