function [ref,surf_dn,surf_dn_direct,absor,sp,out] ...
    = calc_sw_vegetation_3d(sza,albedo,sphere_lai,leaf_ref,leaf_trans,...
			    sphere_radius,spheres_per_hectare,base_z,nreg,...
			    fast)
%CALC_SW_VEGETATION_3D  Compute 3D shortwave properties of an idealized vegetation 
%                       canopy from the RAMI4PILPS intercomparison
%  [ref,surf_dn,surf_dn_direct,absor,sp,out] ...
%      = calc_sw_vegetation_3d(sza,albedo,sphere_lai,leaf_ref,leaf_trans,...
%                              sphere_radius,spheres_per_hectare,base_z,nreg,...
%                              approx)
%  Input arguments:
%    sza              solar zenith angle (degrees)
%    albedo           surface albedo
%    sphere_lai       leaf-area-index of spherical vegetation elements
%    leaf_ref         leaf reflectance
%    leaf_trans       leaf transmittance
%    sphere_radius    radius of spherical vegetation elements
%    spheres_per_hectare  number of spherical vegetation elements per hectare
%    base_z           height of base of vegetation elements above the ground (m)
%    nreg             number of regions to use (1, 2 or 3, default 3)
%    fast             do we use fast sub-canopy-layer calc? (0 or 1, default 0)
%    eff_diam         override effective diameter (default = 2*sphere_radius)
%    
%  Output arguments:
%    ref              scene reflectance
%    surf_dn          normalized transmittance of canopy (transmittance)
%    surf_dn_direct   direct part of surf_dn
%    absor            canopy absorptance
%    sp               data structure input to SPARTACUS algorithm
%    out              data structure output from SPARTACUS algorithm
%
%  Author: Robin Hogan <r.j.hogan@ecmwf.int>
%  Copyright: 2017 European Centre for Medium Range Weather Forecasts
%
%  Copying and distribution of this file, with or without modification,
%  are permitted in any medium without royalty provided the copyright
%  notice and this notice are preserved.

% Normalization area (m2)
hectare_area = 100.^2;

% Fraction of area covered by spheres
sphere_frac = spheres_per_hectare.*pi.*sphere_radius.^2./hectare_area;

% Total sphere perimeter length divided by area (m-1)
norm_perimeter = spheres_per_hectare.*2.*pi.*sphere_radius./hectare_area;

% Default is to use three regions
if ~exist('nreg','var')
  nreg = 3;
end

sp.sza = sza;    % Solar zenith angle (degrees)
sp.n   = 2;      % Number of layers

% If only one region is specified (Sellers-like scheme) then we
% actually use two regions but shrink the clear region to be very
% small
if nreg == 1
  sp.m = 2;
else
  sp.m = nreg;
end

% Heights of the 3 half-levels (m). In effect, the spheres are treated
% as cylinders with base "base_z" metres above the ground, then adding
% the sphere diameter to get the top of the vegetation canopy
sp.z   = [0 base_z base_z+2.*sphere_radius];
%sp.z(2:3) = sp.z(2:3) + (1-1/sqrt(2)).*sphere_radius.*[1 -1];

% Compute optical depth of each region in each layer
if sp.m == 2
  % Two regions: LAI is multiplied by 0.5 to get optical depth, which
  % is appropriate for randomly oriented leaves
  sp.od  = [0 0;0 sphere_lai.*0.5];
else
  % Three regions: we have two vegetated regions of equal area, and
  % need to specify the percentile of the distribution represented by
  % the sparser of the two regions
  %lower_percentile = 0.25; % Incorrect value in version 1.0
  % The following is specific to spheres: if the zenith-optical-depth
  % distribution of a homogeneous sphere is split into two halves,
  % then the appropriate percentile to use for the lower half is as
  % follows (around 35%). This implies that the mean optical depth of
  % the lower half of the distribution is 0.7 times the mean optical
  % depth of the whole sphere.
  lower_percentile = 2.0.^(-1.5);
  sp.od  = [0 0;0 sphere_lai.*0.5.*(2.*lower_percentile);0 sphere_lai.*0.5.*(2.0-2.0.*lower_percentile)];
end

% Pass leaf properties in
sp.leaf_r = [0 0; 0 leaf_ref];
sp.leaf_t = [0 0; 0 leaf_trans];

% Add third region if needed
if sp.m == 3
  sp.leaf_r(3,1:2) = sp.leaf_r(2,1:2);
  sp.leaf_t(3,1:2) = sp.leaf_t(2,1:2);
end

% Fraction of domain occupied by cylinders
if nreg == 1
  % In 1-region case the vegetation expands to fill entire domain
  sp.frac = 0.9999;
  % We also need to scale-down the optical depth
  sp.od = sp.od .* sphere_frac;
else
  sp.frac = sphere_frac;
end

% Surface albedo
sp.surf_albedo = albedo;

% So that the result is normalized by the incoming flux, we scale the
% incoming top-of-atmosphere value at this point
sp.top_flux = 1./cosd(sza);

% Optionally override effective diameter, which is used to compute the
% rate of exchange of radiation between regions
if exist('eff_diam','var')
  sp.eff_diam = eff_diam;
elseif nreg == 1
  % Turn off 3D effects if representing just one region
  sp.eff_diam = inf;
else
  sp.eff_diam = 2.*sphere_radius;
end

% If we want to use fast sub-canopy layer calculation appropriate if
% there is no vegetation, then pass this variable on
if exist('fast','var')
  sp.fast = fast;
end

% Run SPARTACUS shortwave algorithm
out = spartacus_sw_vegetation(sp);

% Store results

% Reflectance
ref = out.flux_up(end);
% Total downwelling
surf_dn = out.flux_dn(1);
% Direct component of downwelling
surf_dn_direct = out.flux_dn_direct(1);
% Absorption is the difference in net flux at top and bottom of layer
absor = out.flux_dn(end)-out.flux_up(end)-out.flux_dn(1)+out.flux_up(1);
